/* IEEE 802.11 SoftMAC layer
 * Copyright (c) 2005 Andrea Merello <andreamrl@tiscali.it>
 *
 * Mostly extracted from the rtl8180-sa2400 driver for the 
 * in-kernel generic ieee802.11 stack.
 *
 * Few lines might be stolen from other part of the ieee80211
 * stack. Copyright who own it's copyright
 *
 * WPA code stolen from the ipw2200 driver.
 * Copyright who own it's copyright. 
 *
 * released under the GPL
 */


#include "ieee80211.h"

#include <linux/random.h>
#include <linux/delay.h>
#include <linux/version.h>
#include <asm/uaccess.h>
#ifdef ENABLE_DOT11D
#include "dot11d.h"
#endif

#ifdef RTK_DMP_PLATFORM
#include <linux/usb_setting.h> 
#endif
extern void _setup_timer( struct timer_list*, void*, unsigned long );
u8 rsn_authen_cipher_suite[16][4] = {
	{0x00,0x0F,0xAC,0x00}, 
	{0x00,0x0F,0xAC,0x01}, 
	{0x00,0x0F,0xAC,0x02}, 
	{0x00,0x0F,0xAC,0x03}, 
	{0x00,0x0F,0xAC,0x04}, 
	{0x00,0x0F,0xAC,0x05}, 
};

short ieee80211_is_54g(struct ieee80211_network *net)
{
	return ((net->rates_ex_len > 0) || (net->rates_len > 4));
}

short ieee80211_is_shortslot(struct ieee80211_network net)
{
	return (net.capability & WLAN_CAPABILITY_SHORT_SLOT);
}

/* returns the total length needed for pleacing the RATE MFIE
 * tag and the EXTENDED RATE MFIE tag if needed.
 * It encludes two bytes per tag for the tag itself and its len
 */
unsigned int ieee80211_MFIE_rate_len(struct ieee80211_device *ieee)
{
	unsigned int rate_len = 0;
	
	if (ieee->modulation & IEEE80211_CCK_MODULATION)
		rate_len = IEEE80211_CCK_RATE_LEN + 2;
		
	if (ieee->modulation & IEEE80211_OFDM_MODULATION)
		
		rate_len += IEEE80211_OFDM_RATE_LEN + 2;
	
	return rate_len;
}

/* pleace the MFIE rate, tag to the memory (double) poined. 
 * Then it updates the pointer so that
 * it points after the new MFIE tag added.
 */  
void ieee80211_MFIE_Brate(struct ieee80211_device *ieee, u8 **tag_p)
{
	u8 *tag = *tag_p; 
	
	if (ieee->modulation & IEEE80211_CCK_MODULATION){
		*tag++ = MFIE_TYPE_RATES;
		*tag++ = 4;
		*tag++ = IEEE80211_BASIC_RATE_MASK | IEEE80211_CCK_RATE_1MB;
		*tag++ = IEEE80211_BASIC_RATE_MASK | IEEE80211_CCK_RATE_2MB;
		*tag++ = IEEE80211_BASIC_RATE_MASK | IEEE80211_CCK_RATE_5MB;
		*tag++ = IEEE80211_BASIC_RATE_MASK | IEEE80211_CCK_RATE_11MB;
	}
	
	/* We may add an option for custom rates that specific HW might support */
	*tag_p = tag;
}

void ieee80211_MFIE_Grate(struct ieee80211_device *ieee, u8 **tag_p)
{	
	u8 *tag = *tag_p; 
	
		if (ieee->modulation & IEEE80211_OFDM_MODULATION){
		
		*tag++ = MFIE_TYPE_RATES_EX;
		*tag++ = 8;
		*tag++ = IEEE80211_BASIC_RATE_MASK | IEEE80211_OFDM_RATE_6MB;
		*tag++ = IEEE80211_BASIC_RATE_MASK | IEEE80211_OFDM_RATE_9MB;
		*tag++ = IEEE80211_BASIC_RATE_MASK | IEEE80211_OFDM_RATE_12MB;
		*tag++ = IEEE80211_BASIC_RATE_MASK | IEEE80211_OFDM_RATE_18MB;
		*tag++ = IEEE80211_BASIC_RATE_MASK | IEEE80211_OFDM_RATE_24MB;
		*tag++ = IEEE80211_BASIC_RATE_MASK | IEEE80211_OFDM_RATE_36MB;
		*tag++ = IEEE80211_BASIC_RATE_MASK | IEEE80211_OFDM_RATE_48MB;
		*tag++ = IEEE80211_BASIC_RATE_MASK | IEEE80211_OFDM_RATE_54MB;
		
	}
	
	/* We may add an option for custom rates that specific HW might support */
	*tag_p = tag;
}


void ieee80211_WMM_Info(struct ieee80211_device *ieee, u8 **tag_p) {
	u8 *tag = *tag_p;

	*tag++ = MFIE_TYPE_GENERIC; 
	*tag++ = 7;
	*tag++ = 0x00;
	*tag++ = 0x50;
	*tag++ = 0xf2;
	*tag++ = 0x02;
	*tag++ = 0x00;
	*tag++ = 0x01;
#ifdef SUPPORT_USPD	
	if(ieee->current_network.wmm_info & 0x80) {
		*tag++ = 0x0f|MAX_SP_Len;
	} else {
		*tag++ = MAX_SP_Len;
	}
#else 
	*tag++ = MAX_SP_Len;
#endif
	*tag_p = tag;
}

#ifdef THOMAS_TURBO
void ieee80211_TURBO_Info(struct ieee80211_device *ieee, u8 **tag_p) {
	u8 *tag = *tag_p;

        *tag++ = MFIE_TYPE_GENERIC; 
        *tag++ = 7;
        *tag++ = 0x00;
        *tag++ = 0xe0;
        *tag++ = 0x4c;
        *tag++ = 0x01;
        *tag++ = 0x02;
        *tag++ = 0x11;
	*tag++ = 0x00;

	*tag_p = tag;
	printk(KERN_ALERT "This is enable turbo mode IE process\n");
}
#endif

void enqueue_mgmt(struct ieee80211_device *ieee, struct sk_buff *skb)
{
	int nh;
	nh = (ieee->mgmt_queue_head +1) % MGMT_QUEUE_NUM;
			
/*
 * if the queue is full but we have newer frames then
 * just overwrites the oldest.
 *	
 * if (nh == ieee->mgmt_queue_tail)
 *		return -1;
 */		
	ieee->mgmt_queue_head = nh;
	ieee->mgmt_queue_ring[nh] = skb;
	
}

struct sk_buff *dequeue_mgmt(struct ieee80211_device *ieee)
{
	struct sk_buff *ret;
	
	if(ieee->mgmt_queue_tail == ieee->mgmt_queue_head)
		return NULL;
		
	ret = ieee->mgmt_queue_ring[ieee->mgmt_queue_tail];
	
	ieee->mgmt_queue_tail = 
		(ieee->mgmt_queue_tail+1) % MGMT_QUEUE_NUM;
	
	return ret;
}

void init_mgmt_queue(struct ieee80211_device *ieee)
{
	ieee->mgmt_queue_tail = ieee->mgmt_queue_head = 0;
}


u8
MgntQuery_TxRateExcludeCCKRates(struct ieee80211_device *ieee)
{
	u16	i;
	u8	QueryRate = 0;
	u8	BasicRate;

	
	for( i = 0; i < ieee->current_network.rates_len; i++)
	{
		BasicRate = ieee->current_network.rates[i]&0x7F;
		if(!ieee80211_is_cck_rate(BasicRate))
		{
			if(QueryRate == 0)
			{
				QueryRate = BasicRate;
			}
			else
			{
				if(BasicRate < QueryRate)
				{
					QueryRate = BasicRate;
				}
			}
		}		
	}

	if(QueryRate == 0)
	{
		QueryRate = 12;	
		printk("No BasicRate found!!\n");
	}
	return QueryRate;
}

u8 MgntQuery_MgntFrameTxRate(struct ieee80211_device *ieee)
{
	PRT_HIGH_THROUGHPUT      pHTInfo = ieee->pHTInfo;
	u8 rate;

#if defined RTL8190P || defined RTL8192E || defined RTL8192U
	if(pHTInfo->IOTAction & HT_IOT_ACT_MGNT_USE_CCK_6M)
		rate = 0x0c;    
	else
		rate = ieee->basic_rate & 0x7f;
#elif defined RTL8192SE || defined RTL8192SU
	if(pHTInfo->IOTAction & HT_IOT_ACT_WA_IOT_Broadcom)
	{
		rate = MgntQuery_TxRateExcludeCCKRates(ieee);
	}
	else
		rate = ieee->basic_rate & 0x7f;
#endif

	if(rate == 0){
		if(ieee->mode == IEEE_A||
		   ieee->mode== IEEE_N_5G||
		   (ieee->mode== IEEE_N_24G&&!pHTInfo->bCurSuppCCK))
			rate = 0x0c;
		else
			rate = 0x02;
	}

	/*
	if( pMgntInfo->bScanInProgress || (pMgntInfo->bDualModeScanStep!=0) )
	{
	if(pMgntInfo->dot11CurrentWirelessMode==WIRELESS_MODE_A)
	rate = 0x0c;
	else
	rate = 0x02;
	}
	 */
	return rate;
}


void ieee80211_sta_wakeup(struct ieee80211_device *ieee, short nl);

inline void softmac_mgmt_xmit(struct sk_buff *skb, struct ieee80211_device *ieee)
{
	unsigned long flags;
	short single = ieee->softmac_features & IEEE_SOFTMAC_SINGLE_QUEUE;
	struct ieee80211_hdr_3addr  *header=
		(struct ieee80211_hdr_3addr  *) skb->data;

	cb_desc *tcb_desc = (cb_desc *)(skb->cb + 8);
	spin_lock_irqsave(&ieee->lock, flags);
	
	/* called with 2nd param 0, no mgmt lock required */
	ieee80211_sta_wakeup(ieee,0);
	
	tcb_desc->queue_index = MGNT_QUEUE;
	tcb_desc->data_rate = MgntQuery_MgntFrameTxRate(ieee);	
	tcb_desc->RATRIndex = 7;
	tcb_desc->bTxDisableRateFallBack = 1;
	tcb_desc->bTxUseDriverAssingedRate = 1;
#ifdef _RTL8192_EXT_PATCH_
	tcb_desc->macId = 0;
#endif	
	if(single){
		if(ieee->queue_stop){
			enqueue_mgmt(ieee,skb);
		}else{
			header->seq_ctl = cpu_to_le16(ieee->seq_ctrl[0]<<4);

			if (ieee->seq_ctrl[0] == 0xFFF)
				ieee->seq_ctrl[0] = 0;
			else
				ieee->seq_ctrl[0]++;
			
			/* avoid watchdog triggers */
			ieee->softmac_data_hard_start_xmit(skb,ieee->dev,ieee->basic_rate);
		}
		
		spin_unlock_irqrestore(&ieee->lock, flags);
	}else{
		spin_unlock_irqrestore(&ieee->lock, flags);
		spin_lock_irqsave(&ieee->mgmt_tx_lock, flags);

		header->seq_ctl = cpu_to_le16(ieee->seq_ctrl[0] << 4);

		if (ieee->seq_ctrl[0] == 0xFFF)
			ieee->seq_ctrl[0] = 0;
		else
			ieee->seq_ctrl[0]++;

		/* check wether the managed packet queued greater than 5 */
		if(!ieee->check_nic_enough_desc(ieee->dev,tcb_desc->queue_index)||\
				(skb_queue_len(&ieee->skb_waitQ[tcb_desc->queue_index]) != 0)||\
				(ieee->queue_stop) ) {
			/* insert the skb packet to the management queue */
			/* as for the completion function, it does not need 
			 * to check it any more.
			 * */
			printk("%s():insert to waitqueue!\n",__FUNCTION__);
			skb_queue_tail(&ieee->skb_waitQ[tcb_desc->queue_index], skb);
		} else {
			ieee->softmac_hard_start_xmit(skb,ieee->dev);
		}
		spin_unlock_irqrestore(&ieee->mgmt_tx_lock, flags);
	}
}

inline void softmac_ps_mgmt_xmit(struct sk_buff *skb, struct ieee80211_device *ieee)
{
	
	short single = ieee->softmac_features & IEEE_SOFTMAC_SINGLE_QUEUE;
	struct ieee80211_hdr_3addr  *header =
		(struct ieee80211_hdr_3addr  *) skb->data;
	u16 fc,type,stype;
        cb_desc *tcb_desc = (cb_desc *)(skb->cb + 8);
	
	fc = header->frame_ctl;
	type = WLAN_FC_GET_TYPE(fc);
	stype = WLAN_FC_GET_STYPE(fc);
	
	
	if(stype != IEEE80211_STYPE_PSPOLL) 
	tcb_desc->queue_index = MGNT_QUEUE;
	else
		tcb_desc->queue_index = HIGH_QUEUE;

	tcb_desc->data_rate = MgntQuery_MgntFrameTxRate(ieee);
	tcb_desc->RATRIndex = 7;
	tcb_desc->bTxDisableRateFallBack = 1;
	tcb_desc->bTxUseDriverAssingedRate = 1;
	if(single){
#ifdef _RTL8192_EXT_PATCH_
		if(!(stype == IEEE80211_STYPE_PSPOLL)&&(type == IEEE80211_FTYPE_CTL)) {
#else	
		if(!(type == IEEE80211_FTYPE_CTL)) {
#endif
		header->seq_ctl = cpu_to_le16(ieee->seq_ctrl[0] << 4);

		if (ieee->seq_ctrl[0] == 0xFFF)
			ieee->seq_ctrl[0] = 0;
		else
			ieee->seq_ctrl[0]++;
			
		}
		/* avoid watchdog triggers */
		ieee->softmac_data_hard_start_xmit(skb,ieee->dev,ieee->basic_rate);
	
	}else{
#ifdef _RTL8192_EXT_PATCH_
		if(!(stype == IEEE80211_STYPE_PSPOLL)&&(type == IEEE80211_FTYPE_CTL)) {
#else	
		if(!(type == IEEE80211_FTYPE_CTL)) {
#endif
		header->seq_ctl = cpu_to_le16(ieee->seq_ctrl[0] << 4);
	
		if (ieee->seq_ctrl[0] == 0xFFF)
			ieee->seq_ctrl[0] = 0;
		else
			ieee->seq_ctrl[0]++;
		}
		ieee->softmac_hard_start_xmit(skb,ieee->dev);
		
	}
}

inline struct sk_buff *ieee80211_probe_req(struct ieee80211_device *ieee)
{
	unsigned int len,rate_len;
	u8 *tag;
	struct sk_buff *skb;
	struct ieee80211_probe_request *req;
	
#ifdef _RTL8192_EXT_PATCH_
	short extMore = 0;
	if(ieee->ext_patch_ieee80211_probe_req_1)
		extMore = ieee->ext_patch_ieee80211_probe_req_1(ieee);
#endif

	len = ieee->current_network.ssid_len;
	
	rate_len = ieee80211_MFIE_rate_len(ieee);
	
#ifdef _RTL8192_EXT_PATCH_
	if(!extMore)
		skb = dev_alloc_skb(sizeof(struct ieee80211_probe_request) +
			    2 + len + rate_len + ieee->tx_headroom);
	else
		skb = dev_alloc_skb(sizeof(struct ieee80211_probe_request) +
			    2 + len + rate_len+128+ieee->tx_headroom); 
#else
#ifdef USB_USE_ALIGNMENT
        u32 Tmpaddr;
        int alignment;
        skb = dev_alloc_skb(sizeof(struct ieee80211_probe_request) +
                            2 + len + rate_len + ieee->tx_headroom + USB_512B_ALIGNMENT_SIZE);
#else	
	skb = dev_alloc_skb(sizeof(struct ieee80211_probe_request) +
			    2 + len + rate_len + ieee->tx_headroom);
#endif
#endif

	if (!skb) 
		return NULL;

#ifdef USB_USE_ALIGNMENT
        Tmpaddr = (u32)skb->data;
        alignment = Tmpaddr & 0x1ff;
        skb_reserve(skb,(USB_512B_ALIGNMENT_SIZE - alignment));
#endif
	
	skb_reserve(skb, ieee->tx_headroom);
	
	req = (struct ieee80211_probe_request *) skb_put(skb,sizeof(struct ieee80211_probe_request));
	req->header.frame_ctl = cpu_to_le16(IEEE80211_STYPE_PROBE_REQ);
	req->header.duration_id = 0; 
	
	memset(req->header.addr1, 0xff, ETH_ALEN);
	memcpy(req->header.addr2, ieee->dev->dev_addr, ETH_ALEN);
	memset(req->header.addr3, 0xff, ETH_ALEN);
	
	tag = (u8 *) skb_put(skb,len+2+rate_len);
	
	*tag++ = MFIE_TYPE_SSID;
	*tag++ = len;
	memcpy(tag, ieee->current_network.ssid, len);
	tag += len;
	
	ieee80211_MFIE_Brate(ieee,&tag);
	ieee80211_MFIE_Grate(ieee,&tag);

#ifdef _RTL8192_EXT_PATCH_
	if(extMore)
		ieee->ext_patch_ieee80211_probe_req_2(ieee, skb, tag);
#endif

	return skb;
}

struct sk_buff *ieee80211_get_beacon_(struct ieee80211_device *ieee);

#ifdef _RTL8192_EXT_PATCH_
void ext_ieee80211_send_beacon_wq(void *data)
{
	struct ieee80211_device *ieee = (struct ieee80211_device *)container_of_work_rsl((struct work_struct *)data, struct ieee80211_device, ext_send_beacon_wq);
	struct sk_buff *skb;
	
	skb = ieee80211_get_beacon_(ieee);

	if (skb){
		softmac_mgmt_xmit(skb, ieee);
		ieee->softmac_stats.tx_beacons++;
	}
	

	
}
#endif
void ieee80211_send_beacon(struct ieee80211_device *ieee)
{
	struct sk_buff *skb;
	if(!ieee->ieee_up)
		return;		
	skb = ieee80211_get_beacon_(ieee);
	
	if (skb){
		softmac_mgmt_xmit(skb, ieee);
		ieee->softmac_stats.tx_beacons++;
	}
	
	if(ieee->beacon_txing && ieee->ieee_up){
		mod_timer(&ieee->beacon_timer,jiffies+(MSECS(ieee->current_network.beacon_interval-5)));
	}
}


void ieee80211_send_beacon_cb(unsigned long _ieee)
{
	struct ieee80211_device *ieee =
		(struct ieee80211_device *) _ieee;
	unsigned long flags;

	spin_lock_irqsave(&ieee->beacon_lock, flags);
	ieee80211_send_beacon(ieee);
	spin_unlock_irqrestore(&ieee->beacon_lock, flags);
}

#ifdef _RTL8192_EXT_PATCH_

inline struct sk_buff *ieee80211_probe_req_with_SSID(struct ieee80211_device *ieee, char *ssid, int len_ssid)
{
	unsigned int len,rate_len;
	u8 *tag;
	struct sk_buff *skb;
	struct ieee80211_probe_request *req;
	
#ifdef _RTL8192_EXT_PATCH_
	short extMore = 0;
	if(ieee->ext_patch_ieee80211_probe_req_1)
		extMore = ieee->ext_patch_ieee80211_probe_req_1(ieee);
#endif

	len = len_ssid;
	
	rate_len = ieee80211_MFIE_rate_len(ieee);
	
#ifdef _RTL8192_EXT_PATCH_
	if(!extMore)
#endif
		skb = dev_alloc_skb(sizeof(struct ieee80211_probe_request) +
			    2 + len + rate_len + ieee->tx_headroom);
#ifdef _RTL8192_EXT_PATCH_
	else
		skb = dev_alloc_skb(sizeof(struct ieee80211_probe_request) +
			    2 + len + rate_len+128+ieee->tx_headroom); 
#endif
	
	if (!skb) 
		return NULL;
	
	req = (struct ieee80211_probe_request *) skb_put(skb,sizeof(struct ieee80211_probe_request));
	req->header.frame_ctl = cpu_to_le16(IEEE80211_STYPE_PROBE_REQ);
	req->header.duration_id = 0; 
	
	memset(req->header.addr1, 0xff, ETH_ALEN);
	memcpy(req->header.addr2, ieee->dev->dev_addr, ETH_ALEN);
	memset(req->header.addr3, 0xff, ETH_ALEN);
	
	tag = (u8 *) skb_put(skb,len+2+rate_len);
	
	*tag++ = MFIE_TYPE_SSID;
	*tag++ = len;
	if(len)
	{
		memcpy(tag, ssid, len);
		tag += len;
	}
	
	ieee80211_MFIE_Brate(ieee,&tag);
	ieee80211_MFIE_Grate(ieee,&tag);
	
#ifdef _RTL8192_EXT_PATCH_
	if(extMore)
		ieee->ext_patch_ieee80211_probe_req_2(ieee, skb, tag);
#endif
	return skb;
}

#endif 

#if defined(RTL8192U) || defined(RTL8192SU) || defined(RTL8192SE)
void ieee80211_ibss_wait_timeout(unsigned long _ieee)
{
	struct ieee80211_device *ieee =
		(struct ieee80211_device *) _ieee;
	printk("======>%s():oh oh ibss wait beacon time out, search a new ibss now\n",__FUNCTION__);	
	ieee80211_stop_send_beacons(ieee);
#if LINUX_VERSION_CODE > KERNEL_VERSION(2,5,0)	
	cancel_delayed_work(&ieee->start_ibss_wq);
	cancel_delayed_work(&ieee->link_change_wq);
#endif	
	ieee80211_stop_scan(ieee);
	queue_delayed_work_rsl(ieee->wq, &ieee->start_ibss_wq, MSECS(150)); 
}
#endif

#ifdef _RTL8192_EXT_PATCH_
void ieee80211_send_probe(struct ieee80211_device *ieee, u8 is_mesh)
#else
void ieee80211_send_probe(struct ieee80211_device *ieee)
#endif
{
	struct sk_buff *skb;
#ifdef _RTL8192_EXT_PATCH_
	if(is_mesh) 
		skb = ieee80211_probe_req_with_SSID(ieee, NULL, 0);
	else
#endif
	skb = ieee80211_probe_req(ieee);
	if (skb){
		softmac_mgmt_xmit(skb, ieee);
		ieee->softmac_stats.tx_probe_rq++;
	}
}

#ifdef _RTL8192_EXT_PATCH_
void ieee80211_send_probe_requests(struct ieee80211_device *ieee, u8 is_mesh)
{
	if (ieee->active_scan && (ieee->softmac_features & IEEE_SOFTMAC_PROBERQ)){
		ieee80211_send_probe(ieee,is_mesh);
		ieee80211_send_probe(ieee,is_mesh);
	}
}
#else
void ieee80211_send_probe_requests(struct ieee80211_device *ieee)
{
	if (ieee->active_scan && (ieee->softmac_features & IEEE_SOFTMAC_PROBERQ)){
		ieee80211_send_probe(ieee);
		ieee80211_send_probe(ieee);
	}
}
#endif

/* this performs syncro scan blocking the caller until all channels
 * in the allowed channel map has been checked. 
 */
#ifdef _RTL8192_EXT_PATCH_
void ieee80211_softmac_scan_syncro(struct ieee80211_device *ieee, u8 is_mesh)
#else
void ieee80211_softmac_scan_syncro(struct ieee80211_device *ieee)
#endif
{
	short ch = 0;
#ifdef ENABLE_DOT11D
	u8 channel_map[MAX_CHANNEL_NUMBER+1];
	memcpy(channel_map, GET_DOT11D_INFO(ieee)->channel_map, MAX_CHANNEL_NUMBER+1);
#endif
#ifdef _RTL8192_EXT_PATCH_
	if(is_mesh)
		ieee->meshscanning = true;
	else
#endif
	ieee->be_scan_inprogress = true;
	down(&ieee->scan_sem);
	
	while(1)
	{
		
		do{
			ch++;
			if (ch > MAX_CHANNEL_NUMBER) 
				goto out; /* scan completed */
#ifdef ENABLE_DOT11D
		}while(!channel_map[ch]);
#else
		}while(!ieee->channel_map[ch]);
#endif
	
		/* this fuction can be called in two situations
		 * 1- We have switched to ad-hoc mode and we are
		 *    performing a complete syncro scan before conclude
		 *    there are no interesting cell and to create a 
		 *    new one. In this case the link state is 
		 *    IEEE80211_NOLINK until we found an interesting cell.
		 *    If so the ieee8021_new_net, called by the RX path
		 *    will set the state to IEEE80211_LINKED, so we stop
		 *    scanning
		 * 2- We are linked and the root uses run iwlist scan.
		 *    So we switch to IEEE80211_LINKED_SCANNING to remember
		 *    that we are still logically linked (not interested in
		 *    new network events, despite for updating the net list,
		 *    but we are temporarly 'unlinked' as the driver shall
		 *    not filter RX frames and the channel is changing.
		 * So the only situation in witch are interested is to check
		 * if the state become LINKED because of the #1 situation
		 */    
		    
		if (ieee->state == IEEE80211_LINKED)
			goto out;
		if (ieee->sync_scan_hurryup){
			printk("============>sync_scan_hurryup out\n");
			goto out;
		}
		ieee->set_chan(ieee->dev, ch);
#ifdef ENABLE_DOT11D
		if(channel_map[ch] == 1)
#endif		
#ifdef _RTL8192_EXT_PATCH_
		ieee80211_send_probe_requests(ieee, is_mesh);
#else
		ieee80211_send_probe_requests(ieee);
#endif
		
		/* this prevent excessive time wait when we
		 * need to wait for a syncro scan to end..
		 */  		


		msleep_interruptible_rsl(IEEE80211_SOFTMAC_SCAN_TIME);
		
	}
out:
	if(ieee->state < IEEE80211_LINKED){
	        ieee->sync_scan_hurryup = 0;
		ieee->actscanning = false;
		up(&ieee->scan_sem);
#ifdef _RTL8192_EXT_PATCH_
		if(is_mesh)
			ieee->meshscanning=false;
		else
#endif		
		ieee->be_scan_inprogress = false;
	}
	else{
		ieee->actscanning = false;
		ieee->sync_scan_hurryup = 0;
#ifdef ENABLE_DOT11D
		if(IS_DOT11D_ENABLE(ieee))
			DOT11D_ScanComplete(ieee);
#endif	
		up(&ieee->scan_sem);
#ifdef _RTL8192_EXT_PATCH_
		if(is_mesh)
			ieee->meshscanning=false;
		else
#endif		
		ieee->be_scan_inprogress = false;
	}
}

void ieee80211_softmac_scan_wq(void *data)
{
	struct ieee80211_device *ieee = container_of_dwork_rsl(data, struct ieee80211_device, softmac_scan_wq);

#ifndef _RTL8192_EXT_PATCH_
	u8 last_channel = ieee->current_network.channel; 
#endif
#ifdef ENABLE_DOT11D
	u8 channel_map[MAX_CHANNEL_NUMBER+1];
	memcpy(channel_map, GET_DOT11D_INFO(ieee)->channel_map, MAX_CHANNEL_NUMBER+1);
#endif
	if(!ieee->ieee_up)
		return;	
        if(ieee->eRFPowerState == eRfOff)
        {
            printk("======>%s():rf state is eRfOff, return\n",__FUNCTION__);
            ieee->actscanning = false;
            ieee->scan_watch_dog = 0;
            ieee->scanning = 0;
            return;
        }
#ifdef _RTL8192_EXT_PATCH_
	if((ieee->be_scan_inprogress) || (ieee->meshscanning))
		return;
#else
	if(ieee->be_scan_inprogress)
		return;
#endif
	down(&ieee->scan_sem);
	do{
		ieee->current_network.channel = 
			(ieee->current_network.channel + 1) % MAX_CHANNEL_NUMBER;
		if (ieee->scan_watch_dog++ > MAX_CHANNEL_NUMBER) 
		{
		#ifdef ENABLE_DOT11D
      			if (!channel_map[ieee->current_network.channel]);
		#else
			if (!ieee->channel_map[ieee->current_network.channel]);
		#endif
				ieee->current_network.channel = 6;
				goto out; /* no good chans */
		}
	}
#ifdef ENABLE_DOT11D
	while(!channel_map[ieee->current_network.channel]);
#else
	while(!ieee->channel_map[ieee->current_network.channel]);
#endif				
	if (ieee->scanning == 0 )
		goto out;
	ieee->set_chan(ieee->dev, ieee->current_network.channel);
#ifdef ENABLE_DOT11D
	if(channel_map[ieee->current_network.channel] == 1)
#endif
#ifdef _RTL8192_EXT_PATCH_
	ieee80211_send_probe_requests(ieee, 0);
#else
	ieee80211_send_probe_requests(ieee);
#endif				

#if LINUX_VERSION_CODE > KERNEL_VERSION(2,6,20)	
	queue_delayed_work_rsl(ieee->wq, &ieee->softmac_scan_wq, MSECS(IEEE80211_SOFTMAC_SCAN_TIME));
#else
	if (ieee->scanning == 1) 
		mod_timer(&ieee->scan_timer,(jiffies + MSECS(IEEE80211_SOFTMAC_SCAN_TIME)));
#endif

	up(&ieee->scan_sem);
	return;
out:
#ifdef ENABLE_DOT11D
	if(IS_DOT11D_ENABLE(ieee))
		DOT11D_ScanComplete(ieee);
#endif	
#ifdef _RTL8192_EXT_PATCH_
	ieee->current_network.channel = ieee->backup_channel;
	ieee->set_chan(ieee->dev, ieee->current_network.channel);
#else
	ieee->current_network.channel = last_channel;
#endif
	ieee->actscanning = false;
	ieee->scan_watch_dog = 0;
	ieee->scanning = 0;
	up(&ieee->scan_sem);
}

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,20)	
void ieee80211_softmac_scan_cb(unsigned long _dev)
{
	unsigned long flags;
	struct ieee80211_device *ieee = (struct ieee80211_device *)_dev;
	
	spin_lock_irqsave(&ieee->lock, flags);
	ieee80211_start_scan(ieee);
	spin_unlock_irqrestore(&ieee->lock, flags);
}
#endif


void ieee80211_beacons_start(struct ieee80211_device *ieee)
{
	unsigned long flags;	
	spin_lock_irqsave(&ieee->beacon_lock,flags);

	ieee->beacon_txing = 1;
	ieee80211_send_beacon(ieee);
	
	spin_unlock_irqrestore(&ieee->beacon_lock,flags);
}

void ieee80211_beacons_stop(struct ieee80211_device *ieee)
{
	unsigned long flags;

	spin_lock_irqsave(&ieee->beacon_lock,flags);

	ieee->beacon_txing = 0;
 	del_timer_sync(&ieee->beacon_timer);

	spin_unlock_irqrestore(&ieee->beacon_lock,flags);

}


void ieee80211_stop_send_beacons(struct ieee80211_device *ieee)
{
	if(ieee->stop_send_beacons)
		ieee->stop_send_beacons(ieee->dev);
	if (ieee->softmac_features & IEEE_SOFTMAC_BEACONS)
		ieee80211_beacons_stop(ieee);
}


void ieee80211_start_send_beacons(struct ieee80211_device *ieee)
{
	if(ieee->start_send_beacons)
		ieee->start_send_beacons(ieee->dev);
	if(ieee->softmac_features & IEEE_SOFTMAC_BEACONS)
		ieee80211_beacons_start(ieee); 
}


void ieee80211_softmac_stop_scan(struct ieee80211_device *ieee)
{
	
	
	down(&ieee->scan_sem);
	ieee->scan_watch_dog = 0;
	if (ieee->scanning == 1){
		ieee->scanning = 0;

#if LINUX_VERSION_CODE > KERNEL_VERSION(2,6,20)	
		cancel_delayed_work(&ieee->softmac_scan_wq);
#else
		del_timer_sync(&ieee->scan_timer);
#endif
	}
	
	up(&ieee->scan_sem);
}

void ieee80211_stop_scan(struct ieee80211_device *ieee)
{
	if (ieee->softmac_features & IEEE_SOFTMAC_SCAN)
		ieee80211_softmac_stop_scan(ieee);
	else
		ieee->stop_scan(ieee->dev);
}

/* called with ieee->lock held */
void ieee80211_start_scan(struct ieee80211_device *ieee)
{
	printk("===>%s()\n",__FUNCTION__);
#ifdef ENABLE_IPS
	if(ieee->ieee80211_ips_leave_wq != NULL)
	ieee->ieee80211_ips_leave_wq(ieee->dev);
#endif


#ifdef ENABLE_DOT11D
	if(IS_DOT11D_ENABLE(ieee) )
	{			
		if(IS_COUNTRY_IE_VALID(ieee))
		{
			RESET_CIE_WATCHDOG(ieee); 
		}
	}
#endif
	if (ieee->softmac_features & IEEE_SOFTMAC_SCAN){	
		if (ieee->scanning == 0){
			ieee->scanning = 1;
#ifdef _RTL8192_EXT_PATCH_
			ieee->backup_channel = ieee->current_network.channel;
			printk("===>backup_channel is %d\n",ieee->backup_channel);
#endif
			queue_delayed_work_rsl(ieee->wq, &ieee->softmac_scan_wq, 0);
		}
	}else
		ieee->start_scan(ieee->dev);
	
}

/* called with wx_sem held */
#ifdef _RTL8192_EXT_PATCH_
void ieee80211_start_scan_syncro(struct ieee80211_device *ieee, u8 is_mesh)
#else
void ieee80211_start_scan_syncro(struct ieee80211_device *ieee)
#endif
{
#ifdef ENABLE_DOT11D
	if(IS_DOT11D_ENABLE(ieee) )
	{			
		if(IS_COUNTRY_IE_VALID(ieee))
		{
			RESET_CIE_WATCHDOG(ieee); 
		}
	}
#endif
	ieee->sync_scan_hurryup = 0;
	if (ieee->softmac_features & IEEE_SOFTMAC_SCAN)
#ifdef _RTL8192_EXT_PATCH_
		ieee80211_softmac_scan_syncro(ieee, is_mesh);
#else
		ieee80211_softmac_scan_syncro(ieee);
#endif
	else
		ieee->scan_syncro(ieee->dev);
		
}

#ifdef _RTL8192_EXT_PATCH_
inline struct sk_buff *ieee80211_authentication_req(struct ieee80211_network *beacon, 
	struct ieee80211_device *ieee, int challengelen,u8 * daddr)
#else
inline struct sk_buff *ieee80211_authentication_req(struct ieee80211_network *beacon, 
	struct ieee80211_device *ieee, int challengelen)
#endif
{
	struct sk_buff *skb;	
	struct ieee80211_authentication *auth;
	int len = sizeof(struct ieee80211_authentication) + challengelen + ieee->tx_headroom;
#ifdef _RTL8192_EXT_PATCH_
	bool is_mesh = false;
	is_mesh = ieee->ext_patch_ieee80211_is_mesh(ieee,daddr);
#endif
#ifdef USB_USE_ALIGNMENT
        u32 Tmpaddr;
        int alignment;
        skb = dev_alloc_skb(len + USB_512B_ALIGNMENT_SIZE);
#else
	skb = dev_alloc_skb(len); 
#endif

	if (!skb) return NULL;
	
#ifdef USB_USE_ALIGNMENT
        Tmpaddr = (u32)skb->data;
        alignment = Tmpaddr & 0x1ff;
        skb_reserve(skb,(USB_512B_ALIGNMENT_SIZE - alignment));
#endif
	
	skb_reserve(skb, ieee->tx_headroom);
	
	auth = (struct ieee80211_authentication *)
		skb_put(skb, sizeof(struct ieee80211_authentication));
	
	auth->header.frame_ctl = IEEE80211_STYPE_AUTH;
	if (challengelen) auth->header.frame_ctl |= IEEE80211_FCTL_WEP;
	
	auth->header.duration_id = 0x013a; 
#ifdef _RTL8192_EXT_PATCH_
	if(is_mesh)
		memcpy(auth->header.addr1,daddr,ETH_ALEN);
	else
#endif
		memcpy(auth->header.addr1, beacon->bssid, ETH_ALEN);
	memcpy(auth->header.addr2, ieee->dev->dev_addr, ETH_ALEN);
#ifdef _RTL8192_EXT_PATCH_
	if(is_mesh)
		memset(auth->header.addr3, 0, ETH_ALEN);
	else
		memcpy(auth->header.addr3, beacon->bssid, ETH_ALEN);
#else
	memcpy(auth->header.addr3, beacon->bssid, ETH_ALEN);
#endif	
	if(ieee->auth_mode == 0)
		auth->algorithm = WLAN_AUTH_OPEN;
	else if(ieee->auth_mode == 1)
		auth->algorithm = WLAN_AUTH_SHARED_KEY;
	else if(ieee->auth_mode == 2)
		auth->algorithm = WLAN_AUTH_OPEN;
	printk("=================>%s():auth->algorithm is %d\n",__FUNCTION__,auth->algorithm);
	auth->transaction = cpu_to_le16(ieee->associate_seq);
	ieee->associate_seq++;
	
	auth->status = cpu_to_le16(WLAN_STATUS_SUCCESS);
	
	return skb;
	
}

void constructWMMIE(u8* wmmie, u8* wmm_len,u8 oui_subtype)
{
	u8	szQoSOUI[] ={221, 0, 0x00, 0x50, 0xf2, 0x02, 0, 1};
	
	if (oui_subtype == OUI_SUBTYPE_QOS_CAPABI)
	{
		szQoSOUI[0] = 46;
		szQoSOUI[1] = *wmm_len;
		memcpy(wmmie,szQoSOUI,3);
		*wmm_len = 3;
	}	
	else
	{
		szQoSOUI[1] = *wmm_len + 6;
		szQoSOUI[6] = oui_subtype;
		memcpy(wmmie, szQoSOUI, 8);
		*(wmmie+8) = 0;
		*wmm_len = 9;
	}	
}

static struct sk_buff* ieee80211_probe_resp(struct ieee80211_device *ieee, u8 *dest)
{
	u8 *tag;
	int beacon_size;
	struct ieee80211_probe_response *beacon_buf;
	struct sk_buff *skb = NULL;
	int encrypt;
	int atim_len,erp_len;
	struct ieee80211_crypt_data* crypt;
	
	char *ssid = ieee->current_network.ssid;
	int ssid_len = ieee->current_network.ssid_len;
	int rate_len = ieee->current_network.rates_len+2;
	int rate_ex_len = ieee->current_network.rates_ex_len;
	int wpa_ie_len = ieee->wpa_ie_len;
	u8 erpinfo_content = 0;

	u8* tmp_ht_cap_buf=NULL;
	u8 tmp_ht_cap_len=0;
	u8* tmp_ht_info_buf=NULL;
	u8 tmp_ht_info_len=0;
	PRT_HIGH_THROUGHPUT	pHTInfo = ieee->pHTInfo;
	u8* tmp_generic_ie_buf=NULL;
	u8 tmp_generic_ie_len=0;
#if defined(RTL8192U) || defined(RTL8192SU) || defined(RTL8192SE)
	u8 wmmie[9] = {0};
	u8 wmm_len = 0;
#endif
	if(rate_ex_len > 0) rate_ex_len+=2;
	
	if(ieee->current_network.capability & WLAN_CAPABILITY_IBSS)
		atim_len = 4;
	else
		atim_len = 0;
	
#if 0
	if(ieee80211_is_54g(&ieee->current_network)) 
		erp_len = 3;
	else
		erp_len = 0;
#else
      if((ieee->current_network.mode == IEEE_G) 
	  	||( ieee->current_network.mode == IEEE_N_24G && ieee->pHTInfo->bCurSuppCCK)) {
	  	erp_len = 3;
		erpinfo_content = 0;
		if(ieee->current_network.buseprotection)
			erpinfo_content |= ERP_UseProtection;
      	}
	else
		erp_len = 0;
#endif

#ifdef _RTL8192_EXT_PATCH_
	crypt = ieee->sta_crypt[ieee->tx_keyidx];
#else
	crypt = ieee->crypt[ieee->tx_keyidx];
#endif	
	encrypt = ieee->host_encrypt && crypt && crypt->ops && 
		((0 == strcmp(crypt->ops->name, "WEP") || wpa_ie_len));
#if 1	
	if(ieee->pHTInfo->bCurrentHTSupport){
		tmp_ht_cap_buf =(u8*) &(ieee->pHTInfo->SelfHTCap);
		tmp_ht_cap_len = sizeof(ieee->pHTInfo->SelfHTCap);
		tmp_ht_info_buf =(u8*) &(ieee->pHTInfo->SelfHTInfo);
		tmp_ht_info_len = sizeof(ieee->pHTInfo->SelfHTInfo);
#ifdef _RTL8192_EXT_PATCH_
		HTConstructCapabilityElement(ieee, tmp_ht_cap_buf, &tmp_ht_cap_len,encrypt, 1);
#else
		HTConstructCapabilityElement(ieee, tmp_ht_cap_buf, &tmp_ht_cap_len,encrypt);
#endif	
		HTConstructInfoElement(ieee,tmp_ht_info_buf,&tmp_ht_info_len, encrypt);
		

		if(pHTInfo->bRegRT2RTAggregation)
		{
			tmp_generic_ie_buf = ieee->pHTInfo->szRT2RTAggBuffer;
			tmp_generic_ie_len = sizeof(ieee->pHTInfo->szRT2RTAggBuffer);
			HTConstructRT2RTAggElement(ieee, tmp_generic_ie_buf, &tmp_generic_ie_len);
		}
	}
#endif
#if defined(RTL8192U) || defined(RTL8192SU) || defined(RTL8192SE)
	if(ieee->qos_support){

		if(ieee->iw_mode == IW_MODE_ADHOC)
		{
			wmm_len = 1;
			constructWMMIE(wmmie,&wmm_len,OUI_SUBTYPE_WMM_INFO);
		}
	}
#endif

	beacon_size = sizeof(struct ieee80211_probe_response)+2+
		ssid_len
		+3 
		+rate_len
		+rate_ex_len
		+atim_len
		+erp_len
		+wpa_ie_len
#if defined(RTL8192U) || defined(RTL8192SU) || defined(RTL8192SE)
		+tmp_ht_cap_len
		+tmp_ht_info_len
		+tmp_generic_ie_len
		+wmm_len
#endif
		+ieee->tx_headroom;
#ifdef USB_USE_ALIGNMENT        
        u32 Tmpaddr=0;
        int alignment=0;
        skb = dev_alloc_skb(beacon_size + USB_512B_ALIGNMENT_SIZE);
#else
	skb = dev_alloc_skb(beacon_size);
#endif
	if (!skb) 
		return NULL;

#ifdef USB_USE_ALIGNMENT
        Tmpaddr = (u32)skb->data;
        alignment = Tmpaddr & 0x1ff;
        skb_reserve(skb,(USB_512B_ALIGNMENT_SIZE - alignment));
#endif
	
	skb_reserve(skb, ieee->tx_headroom);
	
	beacon_buf = (struct ieee80211_probe_response*) skb_put(skb, (beacon_size - ieee->tx_headroom));
	memcpy (beacon_buf->header.addr1, dest,ETH_ALEN);
	memcpy (beacon_buf->header.addr2, ieee->dev->dev_addr, ETH_ALEN);
	memcpy (beacon_buf->header.addr3, ieee->current_network.bssid, ETH_ALEN);

	beacon_buf->header.duration_id = 0; 
	beacon_buf->beacon_interval = 
		cpu_to_le16(ieee->current_network.beacon_interval);
	beacon_buf->capability = 
		cpu_to_le16(ieee->current_network.capability & WLAN_CAPABILITY_IBSS);
	beacon_buf->capability |= 
		cpu_to_le16(ieee->current_network.capability & WLAN_CAPABILITY_SHORT_PREAMBLE); 

	if(ieee->short_slot && (ieee->current_network.capability & WLAN_CAPABILITY_SHORT_SLOT))
		cpu_to_le16((beacon_buf->capability |= WLAN_CAPABILITY_SHORT_SLOT));		
	
#ifndef _RTL8192_EXT_PATCH_
	crypt = ieee->crypt[ieee->tx_keyidx];
#endif
	if (encrypt)	
		beacon_buf->capability |= cpu_to_le16(WLAN_CAPABILITY_PRIVACY);
	
		
	beacon_buf->header.frame_ctl = cpu_to_le16(IEEE80211_STYPE_PROBE_RESP);
	beacon_buf->info_element[0].id = MFIE_TYPE_SSID;	
	beacon_buf->info_element[0].len = ssid_len;
	
	tag = (u8*) beacon_buf->info_element[0].data;
	
	memcpy(tag, ssid, ssid_len);
	
	tag += ssid_len;
	
	*(tag++) = MFIE_TYPE_RATES;
	*(tag++) = rate_len-2; 
	memcpy(tag,ieee->current_network.rates,rate_len-2);
	tag+=rate_len-2;
	
	*(tag++) = MFIE_TYPE_DS_SET;
	*(tag++) = 1;
	*(tag++) = ieee->current_network.channel;
	
	if(atim_len){
	u16 val16;
		*(tag++) = MFIE_TYPE_IBSS_SET;
		*(tag++) = 2;
		 val16 = cpu_to_le16(ieee->current_network.atim_window);
		memcpy((u8 *)tag, (u8 *)&val16, 2);
		tag+=2;
	}
	
	if(erp_len){
		*(tag++) = MFIE_TYPE_ERP;
		*(tag++) = 1;
		*(tag++) = erpinfo_content; 
	}
#if defined(RTL8192U) || defined(RTL8192SU) || defined(RTL8192SE)
	if(tmp_ht_cap_len){
		*(tag++) = MFIE_TYPE_HT_CAP;
		*(tag++) = tmp_ht_cap_len - 2;
		memcpy(tag, tmp_ht_cap_buf, tmp_ht_cap_len - 2);	
		tag += tmp_ht_cap_len - 2;
	}
#endif
	if(rate_ex_len){
		*(tag++) = MFIE_TYPE_RATES_EX;
		*(tag++) = rate_ex_len-2; 
		memcpy(tag,ieee->current_network.rates_ex,rate_ex_len-2);
		tag+=rate_ex_len-2;
	}

#if defined(RTL8192U) || defined(RTL8192SU) || defined(RTL8192SE)
	if(tmp_ht_info_len){
		*(tag++) = MFIE_TYPE_HT_INFO;
		*(tag++) = tmp_ht_info_len - 2;
		memcpy(tag, tmp_ht_info_buf, tmp_ht_info_len -2);
		tag += tmp_ht_info_len - 2;
	}
#endif
	
	if (wpa_ie_len)
	{	
		if (ieee->iw_mode == IW_MODE_ADHOC)
		{
			memcpy(&ieee->wpa_ie[14], &ieee->wpa_ie[8], 4);
		}
		memcpy(tag, ieee->wpa_ie, ieee->wpa_ie_len);
		tag += ieee->wpa_ie_len;
	}

#if defined(RTL8192U) || defined(RTL8192SU) || defined(RTL8192SE)
	if(tmp_generic_ie_len)
	{
		(*tag++) = 0xdd;
		(*tag++) = tmp_generic_ie_len - 2;
		memcpy(tag,tmp_generic_ie_buf,tmp_generic_ie_len -2);
		tag += tmp_generic_ie_len -2;
		
	}
#endif

#if defined(RTL8192U) || defined(RTL8192SU) || defined(RTL8192SE)
	if(ieee->qos_support)
	{
		memcpy(tag,wmmie,wmm_len);
	}
#endif
	return skb;
}

#ifdef _RTL8192_EXT_PATCH_
void ConstructHostNameIE(struct ieee80211_device* ieee, 
		u8* hostname_ie, u8* hostname_ie_len)
{
	memset(hostname_ie, 0, MAX_HOST_NAME_LENGTH);
	*hostname_ie++ = 0x48;
	*hostname_ie++ = 0x4F;
	*hostname_ie++ = 0x53;
	*hostname_ie++ = 0x54;

	memcpy(hostname_ie, ieee->hostname, ieee->hostname_len);

	*hostname_ie_len = ieee->hostname_len + 4 + 2;

	return;
}
#endif

#ifdef _RTL8192_EXT_PATCH_
struct sk_buff* ieee80211_ext_probe_resp_by_net(struct ieee80211_device *ieee, u8 *dest, struct ieee80211_network *net)
{
	u8 *tag;
	int beacon_size;
	struct ieee80211_probe_response *beacon_buf;
	struct sk_buff *skb;
	int encrypt;
	int atim_len = 0,erp_len = 0;
	struct ieee80211_crypt_data* crypt;
	u8 broadcast_addr[] = {0xff,0xff,0xff,0xff,0xff,0xff};
	
	char *ssid = net->ssid;
	int ssid_len = net->ssid_len;

	int rate_len = ieee->current_mesh_network.rates_len+2;
	int rate_ex_len = ieee->current_mesh_network.rates_ex_len;
	int wpa_ie_len = 0;
	u8 erpinfo_content = 0;
#if 0
	u8* tmp_ht_cap_buf=NULL;
	u8 tmp_ht_cap_len=0;
	u8* tmp_ht_info_buf=NULL;
	u8 tmp_ht_info_len=0;
	PRT_HIGH_THROUGHPUT	pHTInfo = ieee->pHTInfo;
	u8* tmp_generic_ie_buf=NULL;
	u8 tmp_generic_ie_len=0;
#endif
	u8 wmmie[9] = {0};
	u8 wmm_len = 0;
#ifdef _RTL8192_EXT_PATCH_		
	u8 hostname_ie_buf[MAX_HOST_NAME_LENGTH+4];
	u8 hostname_ie_len=0;
#endif
	if(rate_ex_len > 0) rate_ex_len+=2;
	if( ieee->meshScanMode&4){
		ieee->current_mesh_network.channel = ieee->ext_patch_ieee80211_ext_stop_scan_wq_set_channel(ieee);
		if(ieee->current_mesh_network.channel == 0)
			ieee->current_mesh_network.channel = 1;
	}
	if( ieee->meshScanMode&6)
	{
		queue_work_rsl(ieee->wq, &ieee->ext_stop_scan_wq);
	}	
	if(ieee->current_mesh_network.capability & WLAN_CAPABILITY_IBSS) 
		atim_len = 4;
	else
		atim_len = 0;
	
	
	if((ieee->current_mesh_network.mode == IEEE_G) 
	  	||( ieee->mode == IEEE_N_24G && ieee->pHTInfo->bCurSuppCCK)){ 
		erp_len = 3;
		erpinfo_content = 0;
		if(ieee->bUseProtection)
			erpinfo_content |= ERP_UseProtection;
	}	
	else
		erp_len = 0;
	
	if ((IW_MODE_MESH==ieee->iw_mode))
	{
		wpa_ie_len = ieee->wpa_ie_len; 
	}
		
#ifdef _RTL8192_EXT_PATCH_
        crypt = ieee->cryptlist[0]->crypt[ieee->mesh_txkeyidx];
#else
	
	crypt = ieee->crypt[ieee->tx_keyidx];
#endif  
	
	
	encrypt = ieee->host_encrypt && crypt && crypt->ops && 
		((0 == strcmp(crypt->ops->name, "WEP")||wpa_ie_len));
#if 1	
#if 0
	if(ieee->pHTInfo->bCurrentHTSupport){
		tmp_ht_cap_buf =(u8*) &(ieee->pHTInfo->SelfHTCap);
		tmp_ht_cap_len = sizeof(ieee->pHTInfo->SelfHTCap);
		tmp_ht_info_buf =(u8*) &(ieee->pHTInfo->SelfHTInfo);
		tmp_ht_info_len = sizeof(ieee->pHTInfo->SelfHTInfo);
		HTConstructCapabilityElement(ieee, tmp_ht_cap_buf, &tmp_ht_cap_len,encrypt, 1);
		HTConstructInfoElement(ieee,tmp_ht_info_buf,&tmp_ht_info_len, encrypt);


		if(pHTInfo->bRegRT2RTAggregation)
		{
			tmp_generic_ie_buf = ieee->pHTInfo->szRT2RTAggBuffer;
			tmp_generic_ie_len = sizeof(ieee->pHTInfo->szRT2RTAggBuffer);
			HTConstructRT2RTAggElement(ieee, tmp_generic_ie_buf, &tmp_generic_ie_len);
		}
	}
#endif
#ifdef _RTL8192_EXT_PATCH_		
	{
		ConstructHostNameIE(ieee, hostname_ie_buf, &hostname_ie_len);
#if 0
		int i = 0;
		printk("%s: HOST NAME IE!!!!!!!!!\n", __FUNCTION__);
		for(i=0; i< (hostname_ie_len-2); i++)
			printk(" %2.2x-", *(hostname_ie_buf+i));
		printk("\n\n");
#endif
	}
#endif
#endif
	if(ieee->qos_support){
#ifdef _RTL8192_EXT_PATCH_
		if((ieee->iw_mode == IW_MODE_MESH) || (ieee->iw_mode == IW_MODE_ADHOC))
#else
		if(ieee->iw_mode == IW_MODE_ADHOC)
#endif
		{
			wmm_len = 1;
			constructWMMIE(wmmie,&wmm_len,OUI_SUBTYPE_WMM_INFO);
		}
	}
	beacon_size = sizeof(struct ieee80211_probe_response)+2+
		ssid_len
		+3 
		+rate_len
		+rate_ex_len
		+atim_len
		+erp_len
                +wpa_ie_len
#ifdef _RTL8192_EXT_PATCH_		
		+hostname_ie_len
#endif
		+wmm_len
		+ieee->tx_headroom;
	skb = dev_alloc_skb(beacon_size+100);
	
	if (!skb) 
 		return NULL;
	skb_reserve(skb, ieee->tx_headroom);
	beacon_buf = (struct ieee80211_probe_response*) skb_put(skb, (beacon_size - ieee->tx_headroom));
	
	memcpy (beacon_buf->header.addr1, dest,ETH_ALEN);
	memcpy (beacon_buf->header.addr2, ieee->meshdev->dev_addr, ETH_ALEN);
	memcpy (beacon_buf->header.addr3, ieee->current_mesh_network.bssid, ETH_ALEN);

	beacon_buf->header.duration_id = 0; 
	
	beacon_buf->beacon_interval = 
		cpu_to_le16(ieee->current_mesh_network.beacon_interval);  
	beacon_buf->capability = 
		cpu_to_le16(ieee->current_mesh_network.capability & WLAN_CAPABILITY_IBSS);
	
	if(ieee->short_slot && (ieee->current_mesh_network.capability & WLAN_CAPABILITY_SHORT_SLOT))
		cpu_to_le16((beacon_buf->capability |= WLAN_CAPABILITY_SHORT_SLOT));		

	if (encrypt)	
		beacon_buf->capability |= cpu_to_le16(WLAN_CAPABILITY_PRIVACY);
	
		
	beacon_buf->header.frame_ctl = cpu_to_le16(IEEE80211_STYPE_PROBE_RESP);
	
	beacon_buf->info_element[0].id = MFIE_TYPE_SSID;	
	beacon_buf->info_element[0].len = ssid_len;
	
	tag = (u8*) beacon_buf->info_element[0].data;
	
	if(memcmp(dest, broadcast_addr, ETH_ALEN ))
		memcpy(tag, ssid, ssid_len);	
	else	
		ssid_len=0;
	
	tag += ssid_len;
	
		
	*(tag++) = MFIE_TYPE_RATES;
	*(tag++) = rate_len-2; 
	memcpy(tag,ieee->current_mesh_network.rates,rate_len-2);
	tag+=rate_len-2;
	
	*(tag++) = MFIE_TYPE_DS_SET;
	*(tag++) = 1;
	*(tag++) = ieee->current_mesh_network.channel;  
	
	
	if(atim_len){
		u16 val16;
		*(tag++) = MFIE_TYPE_IBSS_SET;
		*(tag++) = 2;
		 val16 = cpu_to_le16(ieee->current_mesh_network.atim_window);
		memcpy((u8 *)tag, (u8 *)&val16, 2);
		tag+=2;
	}
	
	if(erp_len){
		*(tag++) = MFIE_TYPE_ERP;
		*(tag++) = 1;
		*(tag++) = erpinfo_content; 
	}
#if 0	
	if(tmp_ht_cap_len){
	*(tag++) = MFIE_TYPE_HT_CAP;
	*(tag++) = tmp_ht_cap_len - 2;
	memcpy(tag, tmp_ht_cap_buf, tmp_ht_cap_len - 2);	
	tag += tmp_ht_cap_len - 2;
	}
#endif

	if(rate_ex_len){
		*(tag++) = MFIE_TYPE_RATES_EX;
		*(tag++) = rate_ex_len-2; 
		memcpy(tag,ieee->current_mesh_network.rates_ex,rate_ex_len-2);
		tag+=rate_ex_len-2;
	}

#if 0 
	if(tmp_ht_info_len){
	*(tag++) = MFIE_TYPE_HT_INFO;
	*(tag++) = tmp_ht_info_len - 2;
	memcpy(tag, tmp_ht_info_buf, tmp_ht_info_len -2);
	tag += tmp_ht_info_len - 2;
	}
#endif
	if (wpa_ie_len) {
		memcpy(tag, ieee->wpa_ie, ieee->wpa_ie_len);
		tag+=ieee->wpa_ie_len;
	}

#if 0	
	if(tmp_generic_ie_len)
	{
		(*tag++) = 0xdd;
		(*tag++) = tmp_generic_ie_len - 2;
		memcpy(tag,tmp_generic_ie_buf,tmp_generic_ie_len -2);
		tag += tmp_generic_ie_len -2;
		
	}
#endif	
#ifdef _RTL8192_EXT_PATCH_	
	if(hostname_ie_len)
	{
		(*tag++) = 0xdd;
		(*tag++) = hostname_ie_len - 2;
		memcpy(tag, hostname_ie_buf, hostname_ie_len - 2);
		tag += hostname_ie_len -2;
	}
#endif

	if(ieee->qos_support)
	{
		memcpy(tag,wmmie,wmm_len);
	}

	skb->dev = ieee->dev;
	return skb;
}
#endif 

struct sk_buff* ieee80211_assoc_resp(struct ieee80211_device *ieee, u8 *dest)
{
	struct sk_buff *skb;
	u8* tag;
	
	struct ieee80211_crypt_data* crypt;
	struct ieee80211_assoc_response_frame *assoc;
	short encrypt;
	
	unsigned int rate_len = ieee80211_MFIE_rate_len(ieee);
	int len = sizeof(struct ieee80211_assoc_response_frame) + rate_len + ieee->tx_headroom;
	
#ifdef USB_USE_ALIGNMENT
        u32 Tmpaddr=0;
        int alignment=0;
        skb = dev_alloc_skb(len + USB_512B_ALIGNMENT_SIZE);
#else	
	skb = dev_alloc_skb(len); 	
#endif
	
	if (!skb) 
		return NULL;
	
#ifdef USB_USE_ALIGNMENT
        Tmpaddr = (u32)skb->data;
        alignment = Tmpaddr & 0x1ff;
        skb_reserve(skb,(USB_512B_ALIGNMENT_SIZE - alignment));
#endif
	
	skb_reserve(skb, ieee->tx_headroom);
	
	assoc = (struct ieee80211_assoc_response_frame *)
		skb_put(skb,sizeof(struct ieee80211_assoc_response_frame));
	
	assoc->header.frame_ctl = cpu_to_le16(IEEE80211_STYPE_ASSOC_RESP);
	memcpy(assoc->header.addr1, dest,ETH_ALEN);
	memcpy(assoc->header.addr3, ieee->dev->dev_addr, ETH_ALEN);
	memcpy(assoc->header.addr2, ieee->dev->dev_addr, ETH_ALEN);
	assoc->capability = cpu_to_le16(ieee->iw_mode == IW_MODE_MASTER ? 
		WLAN_CAPABILITY_BSS : WLAN_CAPABILITY_IBSS);
	
		
	if(ieee->short_slot)
		assoc->capability |= cpu_to_le16(WLAN_CAPABILITY_SHORT_SLOT);
		
	if (ieee->host_encrypt){
#ifdef _RTL8192_EXT_PATCH_
		crypt = ieee->cryptlist[0]->crypt[ieee->tx_keyidx];
#else
		crypt = ieee->crypt[ieee->tx_keyidx];
#endif

	}		
	else crypt = NULL;
	
	encrypt = ( crypt && crypt->ops);
	   
	if (encrypt)
		assoc->capability |= cpu_to_le16(WLAN_CAPABILITY_PRIVACY);
	
	assoc->status = 0;
	assoc->aid = cpu_to_le16(ieee->assoc_id);
	if (ieee->assoc_id == 0x2007) 
		ieee->assoc_id=0;
	else 
		ieee->assoc_id++;
	
	tag = (u8*) skb_put(skb, rate_len);
	ieee80211_MFIE_Brate(ieee, &tag);
	ieee80211_MFIE_Grate(ieee, &tag);
	
	return skb;
}

struct sk_buff* ieee80211_auth_resp(struct ieee80211_device *ieee,int status, u8 *dest)
{
	struct sk_buff *skb;
	struct ieee80211_authentication *auth;
	int len = ieee->tx_headroom + sizeof(struct ieee80211_authentication)+1;
#ifdef _RTL8192_EXT_PATCH_
	bool is_mesh = ieee->ext_patch_ieee80211_is_mesh(ieee,dest);
#endif
#ifdef USB_USE_ALIGNMENT
        u32 Tmpaddr=0;
        int alignment=0;
        skb = dev_alloc_skb(len + USB_512B_ALIGNMENT_SIZE);
#else	
	skb = dev_alloc_skb(len); 	
#endif
	if (!skb) 
		return NULL;
	
	skb->len = sizeof(struct ieee80211_authentication);
	
#ifdef USB_USE_ALIGNMENT
        Tmpaddr = (u32)skb->data;
        alignment = Tmpaddr & 0x1ff;
        skb_reserve(skb,(USB_512B_ALIGNMENT_SIZE - alignment));
#endif
	
	skb_reserve(skb, ieee->tx_headroom);

	auth = (struct ieee80211_authentication *)
		skb_put(skb, sizeof(struct ieee80211_authentication));
	
	auth->status = cpu_to_le16(status);
	auth->transaction = cpu_to_le16(2);
	auth->algorithm = cpu_to_le16(WLAN_AUTH_OPEN);
	
#ifdef _RTL8192_EXT_PATCH_
	if(is_mesh)
		memset(auth->header.addr3, 0, ETH_ALEN);
	else
		memcpy(auth->header.addr3, ieee->dev->dev_addr, ETH_ALEN);
#else
	memcpy(auth->header.addr3, ieee->dev->dev_addr, ETH_ALEN);
#endif
	memcpy(auth->header.addr2, ieee->dev->dev_addr, ETH_ALEN);
	memcpy(auth->header.addr1, dest, ETH_ALEN);
	auth->header.frame_ctl = cpu_to_le16(IEEE80211_STYPE_AUTH); 
	return skb;
	
	
}

struct sk_buff* ieee80211_null_func(struct ieee80211_device *ieee,short pwr)
{
	struct sk_buff *skb;
	struct ieee80211_hdr_3addr* hdr;
	
#ifdef USB_USE_ALIGNMENT
        u32 Tmpaddr=0;
        int alignment=0;
        skb = dev_alloc_skb(sizeof(struct ieee80211_hdr_3addr) + ieee->tx_headroom + USB_512B_ALIGNMENT_SIZE);
#else	
	skb = dev_alloc_skb(sizeof(struct ieee80211_hdr_3addr)+ieee->tx_headroom); 
#endif	
	if (!skb) 
		return NULL;
	
#ifdef USB_USE_ALIGNMENT
        Tmpaddr = (u32)skb->data;
        alignment = Tmpaddr & 0x1ff;
        skb_reserve(skb,(USB_512B_ALIGNMENT_SIZE - alignment));
#endif	
	skb_reserve(skb, ieee->tx_headroom);

	hdr = (struct ieee80211_hdr_3addr*)skb_put(skb,sizeof(struct ieee80211_hdr_3addr));
	
	memcpy(hdr->addr1, ieee->current_network.bssid, ETH_ALEN);
	memcpy(hdr->addr2, ieee->dev->dev_addr, ETH_ALEN);
	memcpy(hdr->addr3, ieee->current_network.bssid, ETH_ALEN);
	
	hdr->frame_ctl = cpu_to_le16(IEEE80211_FTYPE_DATA | 
		IEEE80211_STYPE_NULLFUNC | IEEE80211_FCTL_TODS | 
		(pwr ? IEEE80211_FCTL_PM:0)); 
	
	return skb;
	
	
}

struct sk_buff* ieee80211_pspoll_func(struct ieee80211_device *ieee)
{
	struct sk_buff *skb;
	struct ieee80211_pspoll_hdr* hdr;
	
#ifdef USB_USE_ALIGNMENT
        u32 Tmpaddr=0;
        int alignment=0;
        skb = dev_alloc_skb(sizeof(struct ieee80211_pspoll_hdr) + ieee->tx_headroom + USB_512B_ALIGNMENT_SIZE);
#else	
	skb = dev_alloc_skb(sizeof(struct ieee80211_pspoll_hdr)+ieee->tx_headroom); 
#endif	
	if (!skb) 
		return NULL;
	
#ifdef USB_USE_ALIGNMENT
        Tmpaddr = (u32)skb->data;
        alignment = Tmpaddr & 0x1ff;
        skb_reserve(skb,(USB_512B_ALIGNMENT_SIZE - alignment));
#endif	
	skb_reserve(skb, ieee->tx_headroom);

	hdr = (struct ieee80211_pspoll_hdr*)skb_put(skb,sizeof(struct ieee80211_pspoll_hdr));

	memcpy(hdr->bssid, ieee->current_network.bssid, ETH_ALEN);
	memcpy(hdr->ta, ieee->dev->dev_addr, ETH_ALEN);

	hdr->aid = cpu_to_le16(ieee->assoc_id | 0xc000); 
	hdr->frame_ctl = cpu_to_le16(IEEE80211_FTYPE_CTL |IEEE80211_STYPE_PSPOLL | IEEE80211_FCTL_PM); 
	
	return skb;	
	
}

void ieee80211_resp_to_assoc_rq(struct ieee80211_device *ieee, u8* dest)
{
	struct sk_buff *buf = ieee80211_assoc_resp(ieee, dest);
	
	if (buf)
		softmac_mgmt_xmit(buf, ieee);
}


void ieee80211_resp_to_auth(struct ieee80211_device *ieee, int s, u8* dest)
{
	struct sk_buff *buf = ieee80211_auth_resp(ieee, s, dest);
	
	if (buf)
		softmac_mgmt_xmit(buf, ieee);
}


void ieee80211_resp_to_probe(struct ieee80211_device *ieee, u8 *dest)
{
	

	struct sk_buff *buf = ieee80211_probe_resp(ieee, dest);
	if (buf) 
		softmac_mgmt_xmit(buf, ieee);
}


inline int SecIsInPMKIDList(struct ieee80211_device *ieee, u8 *bssid)
{
	int i = 0;

	do 
	{
		if ((ieee->PMKIDList[i].bUsed) && (memcmp(ieee->PMKIDList[i].Bssid, bssid, ETH_ALEN) == 0))
		{
			break;
		}
		else
		{	
			i++;
		}
	} while (i < NUM_PMKID_CACHE);

	if (i == NUM_PMKID_CACHE)
	{ 
		i = -1;
	}
	else
	{ 
	}

	return (i);
	
}


inline struct sk_buff *ieee80211_association_req(struct ieee80211_network *beacon,struct ieee80211_device *ieee)
{
	struct sk_buff *skb;
	
	struct ieee80211_assoc_request_frame *hdr;
	u8 *tag;
	u8* ht_cap_buf = NULL;
	u8 ht_cap_len=0;
	u8* realtek_ie_buf=NULL;
	u8 realtek_ie_len=0;
	int wpa_ie_len= ieee->wpa_ie_len;
	int wps_ie_len = ieee->wps_ie_len;
	unsigned int ckip_ie_len=0;
	unsigned int ccxrm_ie_len=0;
	unsigned int cxvernum_ie_len=0;
	struct ieee80211_crypt_data* crypt;
	int encrypt;
	int	PMKCacheIdx;
	
	unsigned int rate_len = ieee80211_MFIE_rate_len(ieee);
	unsigned int wmm_info_len = beacon->qos_data.supported?9:0;
#ifdef THOMAS_TURBO	
	unsigned int turbo_info_len = beacon->Turbo_Enable?9:0;
#endif

	int len = 0; 
                             
#ifdef _RTL8192_EXT_PATCH_
	crypt = ieee->sta_crypt[ieee->tx_keyidx];
#else
	crypt = ieee->crypt[ieee->tx_keyidx];
#endif
	if(crypt != NULL) {
		encrypt = ieee->host_encrypt && crypt && crypt->ops && ((0 == strcmp(crypt->ops->name,"WEP") || wpa_ie_len));
	} else {
		encrypt = 0;
	}

#ifdef ENABLE_TKIP11N
	if (ieee->bForcedBgMode == true) {
#else
	if ((ieee->ieee80211_ap_sec_type && (ieee->ieee80211_ap_sec_type(ieee)&SEC_ALG_TKIP)) ||(ieee->bForcedBgMode == true)) {
#endif
		ieee->pHTInfo->bEnableHT = 0;
		ieee->mode = WIRELESS_MODE_G;
	}
	
	if(ieee->pHTInfo->bCurrentHTSupport&&ieee->pHTInfo->bEnableHT)
	{
		ht_cap_buf = (u8*)&(ieee->pHTInfo->SelfHTCap);
		ht_cap_len = sizeof(ieee->pHTInfo->SelfHTCap);
#ifdef _RTL8192_EXT_PATCH_
		HTConstructCapabilityElement(ieee, ht_cap_buf, &ht_cap_len, encrypt, 0);
#else
		HTConstructCapabilityElement(ieee, ht_cap_buf, &ht_cap_len, encrypt);
#endif
		if(ieee->pHTInfo->bCurrentRT2RTAggregation)
		{
			realtek_ie_buf = ieee->pHTInfo->szRT2RTAggBuffer;
			realtek_ie_len = sizeof( ieee->pHTInfo->szRT2RTAggBuffer);
			HTConstructRT2RTAggElement(ieee, realtek_ie_buf, &realtek_ie_len);
       	 			
		}
	}
	if(ieee->qos_support){
		wmm_info_len = beacon->qos_data.supported?9:0;
	}
	

	if(beacon->bCkipSupported)
	{
		ckip_ie_len = 30+2;
	}
	if(beacon->bCcxRmEnable)
	{	
		ccxrm_ie_len = 6+2;
	}
	if( beacon->BssCcxVerNumber >= 2 )
	{	
		cxvernum_ie_len = 5+2;
	}
	
	PMKCacheIdx = SecIsInPMKIDList(ieee, ieee->current_network.bssid);
	if (PMKCacheIdx >= 0)
	{
		wpa_ie_len += 18;
		printk("[PMK cache]: WPA2 IE length: %x\n", wpa_ie_len);
	}	
	
#ifdef THOMAS_TURBO		
	len = sizeof(struct ieee80211_assoc_request_frame)+ 2
		+ beacon->ssid_len
		+ rate_len
		+ wpa_ie_len + wps_ie_len
		+ wmm_info_len
		+ turbo_info_len
                + ht_cap_len
		+ realtek_ie_len
		+ ckip_ie_len
		+ ccxrm_ie_len
		+ cxvernum_ie_len
		+ ieee->tx_headroom;
#else
	len = sizeof(struct ieee80211_assoc_request_frame)+ 2
		+ beacon->ssid_len
		+ rate_len
		+ wpa_ie_len + wps_ie_len
		+ wmm_info_len
                + ht_cap_len
		+ realtek_ie_len
		+ ckip_ie_len
		+ ccxrm_ie_len
		+ cxvernum_ie_len
		+ ieee->tx_headroom;
#endif
	
#ifdef USB_USE_ALIGNMENT
	u32 Tmpaddr=0;
	int alignment=0;
	skb = dev_alloc_skb(len + USB_512B_ALIGNMENT_SIZE);
#else	
	skb = dev_alloc_skb(len);
#endif
	
	if (!skb) 
		return NULL;

#ifdef USB_USE_ALIGNMENT
        Tmpaddr = (u32)skb->data;
        alignment = Tmpaddr & 0x1ff;
        skb_reserve(skb,(USB_512B_ALIGNMENT_SIZE - alignment));
#endif

	skb_reserve(skb, ieee->tx_headroom);
	
	hdr = (struct ieee80211_assoc_request_frame *)
		skb_put(skb, sizeof(struct ieee80211_assoc_request_frame)+2);
	
	
	hdr->header.frame_ctl = IEEE80211_STYPE_ASSOC_REQ;
	hdr->header.duration_id= 37; 
	memcpy(hdr->header.addr1, beacon->bssid, ETH_ALEN);
	memcpy(hdr->header.addr2, ieee->dev->dev_addr, ETH_ALEN);
	memcpy(hdr->header.addr3, beacon->bssid, ETH_ALEN);

	memcpy(ieee->ap_mac_addr, beacon->bssid, ETH_ALEN);
	
	hdr->capability = cpu_to_le16(WLAN_CAPABILITY_BSS);
	if (beacon->capability & WLAN_CAPABILITY_PRIVACY ) 
		hdr->capability |= cpu_to_le16(WLAN_CAPABILITY_PRIVACY);

	if (beacon->capability & WLAN_CAPABILITY_SHORT_PREAMBLE)
		hdr->capability |= cpu_to_le16(WLAN_CAPABILITY_SHORT_PREAMBLE); 

	if(ieee->short_slot)
		hdr->capability |= cpu_to_le16(WLAN_CAPABILITY_SHORT_SLOT);
 	if (wmm_info_len) 
	hdr->capability |= cpu_to_le16(WLAN_CAPABILITY_QOS);

	hdr->listen_interval = beacon->listen_interval;
	
	hdr->info_element[0].id = MFIE_TYPE_SSID;

	hdr->info_element[0].len = beacon->ssid_len;
	tag = skb_put(skb, beacon->ssid_len);
	memcpy(tag, beacon->ssid, beacon->ssid_len);
	
	tag = skb_put(skb, rate_len); 
	
	ieee80211_MFIE_Brate(ieee, &tag);
	ieee80211_MFIE_Grate(ieee, &tag);
	if( beacon->bCkipSupported )
	{
		static u8	AironetIeOui[] = {0x00, 0x01, 0x66}; 
		u8	CcxAironetBuf[30];
		OCTET_STRING	osCcxAironetIE;

		memset(CcxAironetBuf, 0,30);
		osCcxAironetIE.Octet = CcxAironetBuf;
		osCcxAironetIE.Length = sizeof(CcxAironetBuf);
		memcpy(osCcxAironetIE.Octet, AironetIeOui, sizeof(AironetIeOui));

		osCcxAironetIE.Octet[IE_CISCO_FLAG_POSITION] |=  (SUPPORT_CKIP_PK|SUPPORT_CKIP_MIC) ;
		tag = skb_put(skb, ckip_ie_len);
		*tag++ = MFIE_TYPE_AIRONET;
		*tag++ = osCcxAironetIE.Length;
		memcpy(tag,osCcxAironetIE.Octet,osCcxAironetIE.Length);
		tag += osCcxAironetIE.Length;
	}

	if(beacon->bCcxRmEnable)
	{
		static u8 CcxRmCapBuf[] = {0x00, 0x40, 0x96, 0x01, 0x01, 0x00};
		OCTET_STRING osCcxRmCap;

		osCcxRmCap.Octet = CcxRmCapBuf;
		osCcxRmCap.Length = sizeof(CcxRmCapBuf);
		tag = skb_put(skb,ccxrm_ie_len);
		*tag++ = MFIE_TYPE_GENERIC;
		*tag++ = osCcxRmCap.Length;
		memcpy(tag,osCcxRmCap.Octet,osCcxRmCap.Length);
		tag += osCcxRmCap.Length;
	}

	if( beacon->BssCcxVerNumber >= 2 )
	{
		u8			CcxVerNumBuf[] = {0x00, 0x40, 0x96, 0x03, 0x00};
		OCTET_STRING	osCcxVerNum;
		CcxVerNumBuf[4] = beacon->BssCcxVerNumber;
		osCcxVerNum.Octet = CcxVerNumBuf;
		osCcxVerNum.Length = sizeof(CcxVerNumBuf);
		tag = skb_put(skb,cxvernum_ie_len);
		*tag++ = MFIE_TYPE_GENERIC;
		*tag++ = osCcxVerNum.Length;
		memcpy(tag,osCcxVerNum.Octet,osCcxVerNum.Length);
		tag += osCcxVerNum.Length;
	}
	if(ieee->pHTInfo->bCurrentHTSupport&&ieee->pHTInfo->bEnableHT){
		if(ieee->pHTInfo->ePeerHTSpecVer != HT_SPEC_VER_EWC)
		{
			tag = skb_put(skb, ht_cap_len);
			*tag++ = MFIE_TYPE_HT_CAP;
			*tag++ = ht_cap_len - 2;
			memcpy(tag, ht_cap_buf,ht_cap_len -2);
			tag += ht_cap_len -2;
		}
	}	         

	
	tag = skb_put(skb, ieee->wpa_ie_len);	
	if (wpa_ie_len){
		memcpy(tag, ieee->wpa_ie, ieee->wpa_ie_len);
		
		if (PMKCacheIdx >= 0)
		{
			tag = skb_put(skb, 18);		
			*tag = 1;
			*(tag + 1) = 0;	
			memcpy((tag + 2), &ieee->PMKIDList[PMKCacheIdx].PMKID, 16);
		}
	}

	tag = skb_put(skb,wmm_info_len);
	if(wmm_info_len) {
	  ieee80211_WMM_Info(ieee, &tag);
	}

	if(wps_ie_len && ieee->wps_ie)
	{
		tag = skb_put(skb, wps_ie_len);		
		memcpy(tag, ieee->wps_ie, wps_ie_len);		
	}	
	
#ifdef THOMAS_TURBO
	tag = skb_put(skb,turbo_info_len);
        if(turbo_info_len) {
                ieee80211_TURBO_Info(ieee, &tag);
        }
#endif

	if(ieee->pHTInfo->bCurrentHTSupport&&ieee->pHTInfo->bEnableHT){
		if(ieee->pHTInfo->ePeerHTSpecVer == HT_SPEC_VER_EWC)
		{
			tag = skb_put(skb, ht_cap_len);
			*tag++ = MFIE_TYPE_GENERIC;
			*tag++ = ht_cap_len - 2;
			memcpy(tag, ht_cap_buf,ht_cap_len - 2);
			tag += ht_cap_len -2;
		}

		if(ieee->pHTInfo->bCurrentRT2RTAggregation){
			tag = skb_put(skb, realtek_ie_len);
			*tag++ = MFIE_TYPE_GENERIC;
			*tag++ = realtek_ie_len - 2;
			memcpy(tag, realtek_ie_buf,realtek_ie_len -2 );
		}
	}
	return skb;
}

void ieee80211_associate_abort(struct ieee80211_device *ieee)
{
	
	unsigned long flags;
	spin_lock_irqsave(&ieee->lock, flags);
	
	ieee->associate_seq++;
	
	/* don't scan, and avoid to have the RX path possibily
	 * try again to associate. Even do not react to AUTH or
	 * ASSOC response. Just wait for the retry wq to be scheduled.
	 * Here we will check if there are good nets to associate
	 * with, so we retry or just get back to NO_LINK and scanning
	 */
	if (ieee->state == IEEE80211_ASSOCIATING_AUTHENTICATING){
		IEEE80211_DEBUG_MGMT("Authentication failed\n"); 
		ieee->softmac_stats.no_auth_rs++;
	}else{
		IEEE80211_DEBUG_MGMT("Association failed\n"); 
		ieee->softmac_stats.no_ass_rs++;
	}
		
	ieee->state = IEEE80211_ASSOCIATING_RETRY;
		
	queue_delayed_work_rsl(ieee->wq, &ieee->associate_retry_wq, \
                           IEEE80211_SOFTMAC_ASSOC_RETRY_TIME);
	
	spin_unlock_irqrestore(&ieee->lock, flags);
}

void ieee80211_associate_abort_cb(unsigned long dev)
{
	ieee80211_associate_abort((struct ieee80211_device *) dev);
}


#ifdef _RTL8192_EXT_PATCH_
void ieee80211_associate_step1(struct ieee80211_device *ieee,u8 * daddr)
#else
void ieee80211_associate_step1(struct ieee80211_device *ieee)
#endif
{
	struct ieee80211_network *beacon = &ieee->current_network;
	struct sk_buff *skb;
#ifdef _RTL8192_EXT_PATCH_
	bool is_mesh = false;
#endif
	IEEE80211_DEBUG_MGMT("Stopping scan\n");
	
	ieee->softmac_stats.tx_auth_rq++;
#ifdef _RTL8192_EXT_PATCH_
	skb=ieee80211_authentication_req(beacon, ieee, 0,daddr);
#else
	skb=ieee80211_authentication_req(beacon, ieee, 0);
#endif
#ifdef _RTL8192_EXT_PATCH_
	is_mesh = ieee->ext_patch_ieee80211_is_mesh(ieee,daddr);
	if(is_mesh)	{
		if(skb)	
			softmac_mgmt_xmit(skb, ieee);
		return;
	}else
#endif		
	if (!skb) 
		ieee80211_associate_abort(ieee);
	else{ 
		ieee->state = IEEE80211_ASSOCIATING_AUTHENTICATING ;
		IEEE80211_DEBUG_MGMT("Sending authentication request\n");
		softmac_mgmt_xmit(skb, ieee);
		if(!timer_pending(&ieee->associate_timer)){
			ieee->associate_timer.expires = jiffies + (HZ / 2);
			add_timer(&ieee->associate_timer);
		}
	}
}

void ieee80211_auth_challenge(struct ieee80211_device *ieee, u8 *challenge, int chlen)
{
	u8 *c;	
	struct sk_buff *skb;
	struct ieee80211_network *beacon = &ieee->current_network;
	
	ieee->associate_seq++;
	ieee->softmac_stats.tx_auth_rq++;
	
#ifdef _RTL8192_EXT_PATCH_
	skb = ieee80211_authentication_req(beacon, ieee, chlen+2,beacon->bssid);
#else
	skb = ieee80211_authentication_req(beacon, ieee, chlen+2);
#endif		
	if (!skb) 
		ieee80211_associate_abort(ieee);
	else{
		c = skb_put(skb, chlen+2);
		*(c++) = MFIE_TYPE_CHALLENGE;
		*(c++) = chlen;
		memcpy(c, challenge, chlen);
		
		IEEE80211_DEBUG_MGMT("Sending authentication challenge response\n");
		
#ifdef _RTL8192_EXT_PATCH_
		ieee80211_encrypt_fragment(ieee, skb, sizeof(struct ieee80211_hdr_3addr  ), 0);
#else
		ieee80211_encrypt_fragment(ieee, skb, sizeof(struct ieee80211_hdr_3addr  ));
#endif		
			
		softmac_mgmt_xmit(skb, ieee);
		mod_timer(&ieee->associate_timer, jiffies + (HZ/2));
#if 0
		ieee->associate_timer.expires = jiffies + (HZ / 2);
		add_timer(&ieee->associate_timer);
#endif
	}	
	kfree(challenge);
}

void ieee80211_associate_step2(struct ieee80211_device *ieee)
{
	struct sk_buff* skb;
	struct ieee80211_network *beacon = &ieee->current_network;
	
	del_timer_sync(&ieee->associate_timer);
	
	IEEE80211_DEBUG_MGMT("Sending association request\n");
	
	ieee->softmac_stats.tx_ass_rq++;
	skb=ieee80211_association_req(beacon, ieee);
	if (!skb) 
		ieee80211_associate_abort(ieee);
	else{
		softmac_mgmt_xmit(skb, ieee);
		mod_timer(&ieee->associate_timer, jiffies + (HZ/2));
#if 0
		ieee->associate_timer.expires = jiffies + (HZ / 2);
		add_timer(&ieee->associate_timer);
#endif
	}	
}

#define CANCELLED  2
void ieee80211_associate_complete_wq(void *data)
{
	struct ieee80211_device *ieee = (struct ieee80211_device *)container_of_work_rsl(data, struct ieee80211_device, associate_complete_wq);
	PRT_POWER_SAVE_CONTROL pPSC = (PRT_POWER_SAVE_CONTROL)(&(ieee->PowerSaveControl));
	printk(KERN_INFO "Associated successfully\n");
        if(ieee->is_silent_reset == 0){
            printk("============>normal associate\n");
            notify_wx_assoc_event(ieee); 
        }

	netif_carrier_on(ieee->dev);
	ieee->is_roaming = false;
	if(ieee80211_is_54g(&ieee->current_network) && 
		(ieee->modulation & IEEE80211_OFDM_MODULATION)){
		
		ieee->rate = 108;
		printk(KERN_INFO"Using G rates:%d\n", ieee->rate);
	}else{
		ieee->rate = 22;
		printk(KERN_INFO"Using B rates:%d\n", ieee->rate);
	}
	if (ieee->pHTInfo->bCurrentHTSupport&&ieee->pHTInfo->bEnableHT)
	{
		printk("Successfully associated, ht enabled\n");
		HTOnAssocRsp(ieee);
	} else {
		printk("Successfully associated, ht not enabled(%d, %d)\n", 
				ieee->pHTInfo->bCurrentHTSupport, ieee->pHTInfo->bEnableHT);
		memset(ieee->dot11HTOperationalRateSet, 0, 16);
	}
	ieee->LinkDetectInfo.SlotNum = 2 * (1 + ieee->current_network.beacon_interval/500);
	if(ieee->LinkDetectInfo.NumRecvBcnInPeriod==0||ieee->LinkDetectInfo.NumRecvDataInPeriod==0 )
	{
		ieee->LinkDetectInfo.NumRecvBcnInPeriod = 1;
		ieee->LinkDetectInfo.NumRecvDataInPeriod= 1;	
	}
	pPSC->LpsIdleCount = 0;
	ieee->link_change(ieee->dev);

#ifdef _RTL8192_EXT_PATCH_	
	if(ieee->set_key_for_AP)
		ieee->set_key_for_AP(ieee);
	/* Synchronize mesh channel to wlan channel in MSTA mode.*/
	{
		if(ieee->current_mesh_network.channel != ieee->current_network.channel)
		{
			printk("^^^^^^^^^^^^^^^^Change mesh channel %d with wlan channel %d\n", 
					ieee->current_mesh_network.channel, ieee->current_network.channel);
			ieee->ext_patch_ieee80211_close_all_peerlink(ieee, CANCELLED);
			ieee->current_mesh_network.channel = ieee->current_network.channel;
			if(ieee->ext_patch_r819x_wx_set_channel)
			{
				ieee->ext_patch_r819x_wx_set_channel(ieee, ieee->current_network.channel);
				ieee->ext_patch_r819x_wx_set_mesh_chan(ieee->dev,ieee->current_network.channel);
			}
		}
	}
#endif	
        if(ieee->is_silent_reset == 0){
        } else if(ieee->is_silent_reset == 1) {
            printk("==================>silent reset associate\n");
            ieee->is_silent_reset = 0;
        }

	if (ieee->data_hard_resume)
		ieee->data_hard_resume(ieee->dev);

#ifdef RTK_DMP_PLATFORM
	kobject_hotplug(&ieee->dev->class_dev.kobj, KOBJ_LINKUP);
#endif
}

void ieee80211_associate_complete(struct ieee80211_device *ieee)
{
	del_timer_sync(&ieee->associate_timer);

#if 0
	for(i = 0; i < 6; i++) {
	  ieee->seq_ctrl[i] = 0;
	}
#endif	
	ieee->state = IEEE80211_LINKED;
#if 0
	if (ieee->pHTInfo->bCurrentHTSupport)
	{
		printk("Successfully associated, ht enabled\n");
		queue_work_rsl(ieee->wq, &ieee->ht_onAssRsp);
	}
	else
	{
		printk("Successfully associated, ht not enabled\n");
		memset(ieee->dot11HTOperationalRateSet, 0, 16);
		HTSetConnectBwMode(ieee, HT_CHANNEL_WIDTH_20, HT_EXTCHNL_OFFSET_NO_EXT);
	}
#endif
	queue_work_rsl(ieee->wq, &ieee->associate_complete_wq);
}

void ieee80211_associate_procedure_wq(void *data)
{
	struct ieee80211_device *ieee = container_of_dwork_rsl(data, struct ieee80211_device, associate_procedure_wq);
	ieee->sync_scan_hurryup = 1;
#ifdef ENABLE_IPS
	if(ieee->ieee80211_ips_leave != NULL)
        	ieee->ieee80211_ips_leave(ieee->dev);
#endif
	down(&ieee->wx_sem);
	
	if (ieee->data_hard_stop)
		ieee->data_hard_stop(ieee->dev);
	
	ieee80211_stop_scan(ieee);
	printk("===>%s(), chan:%d\n", __FUNCTION__, ieee->current_network.channel);	
	HTSetConnectBwMode(ieee, HT_CHANNEL_WIDTH_20, HT_EXTCHNL_OFFSET_NO_EXT);
	if(ieee->eRFPowerState == eRfOff)
	{
            printk("=============>%s():Rf state is eRfOff, schedule ipsleave wq again,return\n",__FUNCTION__);
#ifdef ENABLE_IPS
		if(ieee->ieee80211_ips_leave_wq != NULL)
			ieee->ieee80211_ips_leave_wq(ieee->dev);
#endif
		up(&ieee->wx_sem);
		return;
	}
	ieee->associate_seq = 1;
#ifdef _RTL8192_EXT_PATCH_
	ieee80211_associate_step1(ieee, ieee->current_network.bssid);
#else
	ieee80211_associate_step1(ieee);
#endif	
	up(&ieee->wx_sem);
}

#ifdef _RTL8192_EXT_PATCH_

void ieee80211_ext_stop_scan_wq(void *data)
{
	struct ieee80211_device *ieee = (struct ieee80211_device *)container_of_work_rsl(data, struct ieee80211_device, ext_stop_scan_wq);
	ieee->sync_scan_hurryup = 1;
	
	down(&ieee->wx_sem);
	
	if (ieee->data_hard_stop)
		ieee->data_hard_stop(ieee->dev);
	
	ieee80211_stop_scan(ieee);

	up(&ieee->wx_sem);
}


void ieee80211_ext_send_11s_beacon(struct ieee80211_device *ieee)
{	
	queue_work_rsl(ieee->wq, &ieee->ext_send_beacon_wq);
}

#endif 


inline void ieee80211_softmac_new_net(struct ieee80211_device *ieee, struct ieee80211_network *net)
{
	u8 tmp_ssid[IW_ESSID_MAX_SIZE+1];
	int tmp_ssid_len = 0;
	
	short apset,ssidset,ssidbroad,apmatch,ssidmatch;
	
	/* we are interested in new new only if we are not associated 
	 * and we are not associating / authenticating
	 */
	if (ieee->state != IEEE80211_NOLINK)
		return; 

	if ((ieee->iw_mode == IW_MODE_INFRA) && !(net->capability & WLAN_CAPABILITY_BSS))
		return;
	
	if ((ieee->iw_mode == IW_MODE_ADHOC) && !(net->capability & WLAN_CAPABILITY_IBSS))
		return;

	if ((ieee->iw_mode == IW_MODE_ADHOC) && (net->channel > ieee->ibss_maxjoin_chal)) {
		return;
	}
#ifdef _RTL8192_EXT_PATCH_
	if (ieee->iw_mode == IW_MODE_INFRA || ieee->iw_mode == IW_MODE_ADHOC 
	    || ((ieee->iw_mode == IW_MODE_MESH) && (ieee->only_mesh == 0)))
#else
	if (ieee->iw_mode == IW_MODE_INFRA || ieee->iw_mode == IW_MODE_ADHOC) 
#endif
		{
		/* if the user specified the AP MAC, we need also the essid
		 * This could be obtained by beacons or, if the network does not
		 * broadcast it, it can be put manually.
		 */
		apset = ieee->wap_set;
		ssidset = ieee->ssid_set;
		ssidbroad =  !(net->ssid_len == 0 || net->ssid[0]== '\0');
		apmatch = (memcmp(ieee->current_network.bssid, net->bssid, ETH_ALEN)==0);
		if(!ssidbroad){
		    	ssidmatch = (ieee->current_network.ssid_len == net->hidden_ssid_len)&&\
					(!strncmp(ieee->current_network.ssid, net->hidden_ssid, net->hidden_ssid_len));
			if(net->hidden_ssid_len > 0)
                        {
			        strncpy(net->ssid, net->hidden_ssid, net->hidden_ssid_len);
			        net->ssid_len = net->hidden_ssid_len;
                                ssidbroad = 1;
                        }
		}
		else
			ssidmatch = (ieee->current_network.ssid_len == net->ssid_len)&&\
					(!strncmp(ieee->current_network.ssid, net->ssid, net->ssid_len));

			
		if (	/* if the user set the AP check if match.
		         * if the network does not broadcast essid we check the user supplyed ANY essid
			 * if the network does broadcast and the user does not set essid it is OK
			 * if the network does broadcast and the user did set essid chech if essid match
			 */
			( apset && apmatch && 
				((ssidset && ssidbroad && ssidmatch) || (ssidbroad && !ssidset) || (!ssidbroad && ssidset)) )
			/* if the ap is not set, check that the user set the bssid
			 * and the network does bradcast and that those two bssid matches
			 */ 
			 ||  (!apset && ssidset && ssidbroad && ssidmatch) 
			){
				/* if the essid is hidden replace it with the
				* essid provided by the user.
				*/
				if (!ssidbroad){
					strncpy(tmp_ssid, ieee->current_network.ssid, IW_ESSID_MAX_SIZE);
					tmp_ssid_len = ieee->current_network.ssid_len;
				}
				memcpy(&ieee->current_network, net, sizeof(struct ieee80211_network));
				if (!ssidbroad){
					strncpy(ieee->current_network.ssid, tmp_ssid, IW_ESSID_MAX_SIZE);
					ieee->current_network.ssid_len = tmp_ssid_len;
				}
				printk(KERN_INFO"Linking with %s,channel:%d, qos:%d, myHT:%d, networkHT:%d, mode:%x cur_net.flags:0x%x\n",ieee->current_network.ssid,ieee->current_network.channel, ieee->current_network.qos_data.supported, ieee->pHTInfo->bEnableHT, ieee->current_network.bssht.bdSupportHT, ieee->current_network.mode, ieee->current_network.flags);

				HTResetIOTSetting(ieee->pHTInfo);
                ieee->wmm_acm = 0;
#ifdef _RTL8192_EXT_PATCH_
				if ((ieee->iw_mode == IW_MODE_INFRA) || 
					((ieee->iw_mode == IW_MODE_MESH) && (ieee->only_mesh == 0)))  
#else
				if (ieee->iw_mode == IW_MODE_INFRA)
#endif
				{
					/* Join the network for the first time */
					ieee->AsocRetryCount = 0;	
					if((ieee->current_network.qos_data.supported == 1) &&
					   ieee->current_network.bssht.bdSupportHT)
/*WB, 2008.09.09:bCurrentHTSupport and bEnableHT two flags are going to put together to check whether we are in HT now, so needn't to check bEnableHT flags here. That's is to say we will set to HT support whenever joined AP has the ability to support HT. And whether we are in HT or not, please check bCurrentHTSupport&&bEnableHT now please.*/
					{
#ifdef ENABLE_AMSDU   
						if((ieee->mode == IEEE_N_24G) && (ieee->mode == IEEE_N_5G))
							HTUseDefaultSetting(ieee);
#endif
						HTResetSelfAndSavePeerSetting(ieee, &(ieee->current_network));
					}
					else
					{
						ieee->pHTInfo->bCurrentHTSupport = false;
					}

					ieee->state = IEEE80211_ASSOCIATING;
					if(ieee->LedControlHandler != NULL)
					        ieee->LedControlHandler(ieee->dev, LED_CTL_START_TO_LINK); 
					queue_delayed_work_rsl(ieee->wq, &ieee->associate_procedure_wq, 0);
				}else{
#if defined(RTL8192U) || defined(RTL8192SU) || defined(RTL8192SE)
					ieee->state = IEEE80211_LINKED;
					ieee->SetWirelessMode(ieee->dev, ieee->current_network.mode);
					mod_timer(&ieee->ibss_wait_timer,jiffies+(MSECS(20000)));
#else
					if(ieee80211_is_54g(&ieee->current_network) && 
						(ieee->modulation & IEEE80211_OFDM_MODULATION)){
						ieee->rate = 108;
						ieee->SetWirelessMode(ieee->dev, IEEE_G);
						printk(KERN_INFO"Using G rates\n");
					}else{
						ieee->rate = 22;
						ieee->SetWirelessMode(ieee->dev, IEEE_B);
						printk(KERN_INFO"Using B rates\n");
					}
					memset(ieee->dot11HTOperationalRateSet, 0, 16);
					ieee->state = IEEE80211_LINKED;
#endif
				}
			
		}
	}

}

void ieee80211_softmac_check_all_nets(struct ieee80211_device *ieee)
{
	unsigned long flags;
	struct ieee80211_network *target;

	spin_lock_irqsave(&ieee->lock, flags);
			
	list_for_each_entry(target, &ieee->network_list, list) {
		
		/* if the state become different that NOLINK means
		 * we had found what we are searching for
		 */

		if (ieee->state != IEEE80211_NOLINK) 
			break;
	
		if (ieee->scan_age == 0 || time_after(target->last_scanned + ieee->scan_age, jiffies))
		ieee80211_softmac_new_net(ieee, target);
	}
	
	spin_unlock_irqrestore(&ieee->lock, flags);
	
}


static inline u16 auth_parse(struct sk_buff *skb, u8** challenge, int *chlen)
{
	struct ieee80211_authentication *a;
	u8 *t;
	if (skb->len <  (sizeof(struct ieee80211_authentication)-sizeof(struct ieee80211_info_element))){ 
		IEEE80211_DEBUG_MGMT("invalid len in auth resp: %d\n",skb->len);
		return 0xcafe;
	}
	*challenge = NULL;
	a = (struct ieee80211_authentication*) skb->data;
	if(skb->len > (sizeof(struct ieee80211_authentication) +3)){
		t = skb->data + sizeof(struct ieee80211_authentication);
		
		if(*(t++) == MFIE_TYPE_CHALLENGE){
			*chlen = *(t++);
			*challenge = (u8*)kmalloc(*chlen, GFP_ATOMIC);
			memcpy(*challenge, t, *chlen);
		}
	}
	
	return cpu_to_le16(a->status);
	
}


int auth_rq_parse(struct sk_buff *skb,u8* dest)
{
	struct ieee80211_authentication *a;
	
	if (skb->len <  (sizeof(struct ieee80211_authentication)-sizeof(struct ieee80211_info_element))){ 
		IEEE80211_DEBUG_MGMT("invalid len in auth request: %d\n",skb->len);	
		return -1;
	}
	a = (struct ieee80211_authentication*) skb->data;
	
	memcpy(dest,a->header.addr2, ETH_ALEN);
	
	if (le16_to_cpu(a->algorithm) != WLAN_AUTH_OPEN) 
		return  WLAN_STATUS_NOT_SUPPORTED_AUTH_ALG;
	
	return WLAN_STATUS_SUCCESS;
}

static short probe_rq_parse(struct ieee80211_device *ieee, struct sk_buff *skb, u8 *src)
{
	u8 *tag;
	u8 *skbend;
	u8 *ssid=NULL;
	u8 ssidlen = 0;
	
	struct ieee80211_hdr_3addr   *header =
		(struct ieee80211_hdr_3addr   *) skb->data;
	
	if (skb->len < sizeof (struct ieee80211_hdr_3addr  )) 
		return -1; /* corrupted */
        if((memcmp(header->addr3,ieee->current_network.bssid,ETH_ALEN) != 0)&&
                (memcmp(header->addr3,"\xff\xff\xff\xff\xff\xff",ETH_ALEN) != 0)) {
            return -1;
        }

        if(memcmp(header->addr3,ieee->current_network.bssid,ETH_ALEN) == 0) {
        }
	
        if(memcmp(header->addr3,"\xff\xff\xff\xff\xff\xff",ETH_ALEN) == 0) {
        }
	memcpy(src,header->addr2, ETH_ALEN);
	
	skbend = (u8*)skb->data + skb->len;
	
	tag = skb->data + sizeof (struct ieee80211_hdr_3addr  );
	
	while (tag+1 < skbend){
		if (*tag == 0){ 
			ssid = tag+2;
			ssidlen = *(tag+1);
			break;
		}
		tag++; /* point to the len field */
		tag = tag + *(tag); /* point to the last data byte of the tag */
		tag++; /* point to the next tag */
	}
	
	if (ssidlen == 0) return 1;
	
	if (!ssid) return 1; /* ssid not found in tagged param */
	return (!strncmp(ssid, ieee->current_network.ssid, ssidlen));
		
}

int assoc_rq_parse(struct sk_buff *skb,u8* dest)
{
	struct ieee80211_assoc_request_frame *a;
	
	if (skb->len < (sizeof(struct ieee80211_assoc_request_frame) - 
		sizeof(struct ieee80211_info_element))) { 
		
		IEEE80211_DEBUG_MGMT("invalid len in auth request:%d \n", skb->len);
		return -1;
	}
	
	a = (struct ieee80211_assoc_request_frame*) skb->data;
		
	memcpy(dest,a->header.addr2,ETH_ALEN);
	
	return 0;
}

static inline u16 assoc_parse(struct ieee80211_device *ieee, struct sk_buff *skb, int *aid)
{
	struct ieee80211_assoc_response_frame *response_head;
	u16 status_code;

	if (skb->len <  sizeof(struct ieee80211_assoc_response_frame)){ 
		IEEE80211_DEBUG_MGMT("invalid len in auth resp: %d\n", skb->len);
		return 0xcafe;
	}
	
	response_head = (struct ieee80211_assoc_response_frame*) skb->data;
	*aid = le16_to_cpu(response_head->aid) & 0x3fff;

	status_code = le16_to_cpu(response_head->status);
	if((status_code==WLAN_STATUS_ASSOC_DENIED_RATES || \
	   status_code==WLAN_STATUS_CAPS_UNSUPPORTED)&&
	   ((ieee->mode == IEEE_G) && 
	    (ieee->current_network.mode == IEEE_N_24G) &&
            (ieee->AsocRetryCount++ < (RT_ASOC_RETRY_LIMIT-1)))) {
                 ieee->pHTInfo->IOTAction |= HT_IOT_ACT_PURE_N_MODE;
	}else {
		 ieee->AsocRetryCount = 0;
	}

	return le16_to_cpu(response_head->status);
}

void ieee80211_rx_probe_rq(struct ieee80211_device *ieee, struct sk_buff *skb)
{
	u8 dest[ETH_ALEN];
#if defined(RTL8192U) || defined(RTL8192SU) || defined(RTL8192SE)
	struct sta_info *psta = NULL;
#endif
	ieee->softmac_stats.rx_probe_rq++;
	if (probe_rq_parse(ieee, skb, dest) > 0){
		ieee->softmac_stats.tx_probe_rs++;
		ieee80211_resp_to_probe(ieee, dest);
#if defined(RTL8192U) || defined(RTL8192SU) || defined(RTL8192SE)
		if(ieee->iw_mode == IW_MODE_ADHOC){
			psta = GetStaInfo(ieee, dest);
			if(NULL != psta)
				psta->LastActiveTime = jiffies;
		}
#endif
	}
}

static inline void
ieee80211_rx_auth_rq(struct ieee80211_device *ieee, struct sk_buff *skb)
{
	u8 dest[ETH_ALEN];
	int status;
	ieee->softmac_stats.rx_auth_rq++;
	
	if ((status = auth_rq_parse(skb, dest))!= -1){
		ieee80211_resp_to_auth(ieee, status, dest);
	}
	
}

static inline void
ieee80211_rx_assoc_rq(struct ieee80211_device *ieee, struct sk_buff *skb)
{
	
	u8 dest[ETH_ALEN];
	
	ieee->softmac_stats.rx_ass_rq++;
	if (assoc_rq_parse(skb,dest) != -1){
		ieee80211_resp_to_assoc_rq(ieee, dest);
	}
	
	printk(KERN_INFO"New client associated: "MAC_FMT"\n", MAC_ARG(dest));
	#if 0
	spin_lock_irqsave(&ieee->lock,flags);
	add_associate(ieee,dest);
	spin_unlock_irqrestore(&ieee->lock,flags);
	#endif
}


void ieee80211_sta_ps_send_null_frame(struct ieee80211_device *ieee, short pwr)
{
	
	struct sk_buff *buf = ieee80211_null_func(ieee, pwr);
	
	if (buf)
		softmac_ps_mgmt_xmit(buf, ieee);

} 

void ieee80211_sta_ps_send_pspoll_frame(struct ieee80211_device *ieee)
{
	
	struct sk_buff *buf = ieee80211_pspoll_func(ieee);
	
	if (buf)
		softmac_ps_mgmt_xmit(buf, ieee);

} 

short ieee80211_sta_ps_sleep(struct ieee80211_device *ieee, u32 *time_h, u32 *time_l)
{	
	int timeout = ieee->ps_timeout;
	u8 dtim;
	PRT_POWER_SAVE_CONTROL	pPSC = (PRT_POWER_SAVE_CONTROL)(&(ieee->PowerSaveControl));
	/*if(ieee->ps == IEEE80211_PS_DISABLED ||
		ieee->iw_mode != IW_MODE_INFRA || 
		ieee->state != IEEE80211_LINKED)
		
		return 0;
	*/

	if(ieee->LPSDelayCnt)
	{
		ieee->LPSDelayCnt --;
		return 0;
	}
		
	dtim = ieee->current_network.dtim_data;
	if(!(dtim & IEEE80211_DTIM_VALID))
		return 0;
	timeout = ieee->current_network.beacon_interval; 
	ieee->current_network.dtim_data = IEEE80211_DTIM_INVALID;
	if(dtim & ((IEEE80211_DTIM_UCAST | IEEE80211_DTIM_MBCAST)& ieee->ps))
		return 2;
	
	if(!time_after(jiffies, ieee->dev->trans_start + MSECS(timeout))){
		return 0;
	}
	if(!time_after(jiffies, ieee->last_rx_ps_time + MSECS(timeout))){
		return 0;
	}
	if((ieee->softmac_features & IEEE_SOFTMAC_SINGLE_QUEUE ) &&
		(ieee->mgmt_queue_tail != ieee->mgmt_queue_head))
		return 0;
	
	if(time_l){
		if(ieee->bAwakePktSent == true)
		{
			pPSC->LPSAwakeIntvl = 1;
		}
#if 1
		else
		{
			u8		MaxPeriod = 1;

			if(pPSC->LPSAwakeIntvl == 0)
				pPSC->LPSAwakeIntvl = 1;
			if(pPSC->RegMaxLPSAwakeIntvl == 0) 
				MaxPeriod = 1; 
			else if(pPSC->RegMaxLPSAwakeIntvl == 0xFF) 
				MaxPeriod = ieee->current_network.dtim_period;
			else
				MaxPeriod = pPSC->RegMaxLPSAwakeIntvl;
			pPSC->LPSAwakeIntvl = (pPSC->LPSAwakeIntvl >= MaxPeriod) ? MaxPeriod : (pPSC->LPSAwakeIntvl + 1);
		}
#endif
		{
			u8 LPSAwakeIntvl_tmp = 0;
			u8 period = ieee->current_network.dtim_period;
			u8 count = ieee->current_network.tim.tim_count;
			if(count == 0 ) {
				if(pPSC->LPSAwakeIntvl > period)
					LPSAwakeIntvl_tmp = period + (pPSC->LPSAwakeIntvl - period) -((pPSC->LPSAwakeIntvl-period)%period);
				else
					LPSAwakeIntvl_tmp = pPSC->LPSAwakeIntvl;

			} else {
				if(pPSC->LPSAwakeIntvl > ieee->current_network.tim.tim_count)
					LPSAwakeIntvl_tmp = count + (pPSC->LPSAwakeIntvl - count) -((pPSC->LPSAwakeIntvl-count)%period);
				else
					LPSAwakeIntvl_tmp = pPSC->LPSAwakeIntvl;
			}
		
		*time_l = ieee->current_network.last_dtim_sta_time[0] 
			+ MSECS(ieee->current_network.beacon_interval * LPSAwakeIntvl_tmp); 
	}
	}
	
	if(time_h){
#ifdef _RTL8192_EXT_PATCH_
		if(time_l && *time_l < ieee->current_network.last_dtim_sta_time[0])
			*time_h =1;
		else
			*time_h =0;
#else
		*time_h = ieee->current_network.last_dtim_sta_time[1];
		if(time_l && *time_l < ieee->current_network.last_dtim_sta_time[0])
			*time_h += 1;
#endif
	}
	
	return 1;
	
	
}

inline void ieee80211_sta_ps(struct ieee80211_device *ieee)
{

	u32 th,tl;
	short sleep;
	
	unsigned long flags,flags2;
	
	spin_lock_irqsave(&ieee->lock, flags);
	
	if((ieee->ps == IEEE80211_PS_DISABLED ||
		ieee->iw_mode != IW_MODE_INFRA || 
		ieee->state != IEEE80211_LINKED)){
		
		printk("=====>%s(): no need to ps,wake up!! ieee->ps is %d,ieee->iw_mode is %d,ieee->state is %d\n",
			__FUNCTION__,ieee->ps,ieee->iw_mode,ieee->state);
		spin_lock_irqsave(&ieee->mgmt_tx_lock, flags2);
		
		ieee80211_sta_wakeup(ieee, 1);	
		
		spin_unlock_irqrestore(&ieee->mgmt_tx_lock, flags2);
	}
	
	sleep = ieee80211_sta_ps_sleep(ieee,&th, &tl);
	/* 2 wake, 1 sleep, 0 do nothing */
	if(sleep == 0)
	{
		goto out;
	}
	if(sleep == 1){
		if(ieee->sta_sleep == 1){
			ieee->enter_sleep_state(ieee->dev,th,tl);
		}
		
		else if(ieee->sta_sleep == 0){
			spin_lock_irqsave(&ieee->mgmt_tx_lock, flags2);
			
			if(ieee->ps_is_queue_empty(ieee->dev)){
				
			
				ieee->sta_sleep = 2;
				
				ieee->ack_tx_to_ieee = 1;
				
				ieee80211_sta_ps_send_null_frame(ieee,1);
				
				ieee->ps_th = th;
				ieee->ps_tl = tl;
			}		
			spin_unlock_irqrestore(&ieee->mgmt_tx_lock, flags2);
			
		}
		
		ieee->bAwakePktSent = false;
		
	}else if(sleep == 2){
		spin_lock_irqsave(&ieee->mgmt_tx_lock, flags2);
			
		ieee80211_sta_wakeup(ieee,1);
		
		spin_unlock_irqrestore(&ieee->mgmt_tx_lock, flags2);
	}

out:	
	spin_unlock_irqrestore(&ieee->lock, flags);
	
}

void ieee80211_sta_wakeup(struct ieee80211_device *ieee, short nl)
{
	if(ieee->sta_sleep == 0){
		if(nl){
			if(ieee->pHTInfo->IOTAction & HT_IOT_ACT_NULL_DATA_POWER_SAVING)
			{
			ieee->ack_tx_to_ieee = 1;
			ieee80211_sta_ps_send_null_frame(ieee, 0);
		}
			else 
			{
				ieee->ack_tx_to_ieee = 1;
				ieee80211_sta_ps_send_pspoll_frame(ieee);
			}
		}
		return;
		
	}
	
	if(ieee->sta_sleep == 1) 
	{
		ieee->sta_wake_up(ieee->dev);
	}
	
	ieee->sta_sleep = 0;
	
	if(nl){
/*		
		ieee->ack_tx_to_ieee = 1;
		ieee80211_sta_ps_send_null_frame(ieee, 0);
*/
			if(ieee->pHTInfo->IOTAction & HT_IOT_ACT_NULL_DATA_POWER_SAVING)
			{
				ieee->ack_tx_to_ieee = 1;
				ieee80211_sta_ps_send_null_frame(ieee, 0);
			}
			else 
			{
				ieee->ack_tx_to_ieee = 1;
				ieee80211_sta_ps_send_pspoll_frame(ieee);
			}

	}
}

void ieee80211_ps_tx_ack(struct ieee80211_device *ieee, short success)
{
	unsigned long flags,flags2;
	
	spin_lock_irqsave(&ieee->lock, flags);
	
	if(ieee->sta_sleep == 2){
		/* Null frame with PS bit set */
		if(success){
			ieee->sta_sleep = 1;
			ieee->enter_sleep_state(ieee->dev,ieee->ps_th,ieee->ps_tl);
		}
		/* if the card report not success we can't be sure the AP
		 * has not RXed so we can't assume the AP believe us awake
		 */
	} else {/* 21112005 - tx again null without PS bit if lost */
	
		if((ieee->sta_sleep == 0) && !success){
			spin_lock_irqsave(&ieee->mgmt_tx_lock, flags2);
			if(ieee->pHTInfo->IOTAction & HT_IOT_ACT_NULL_DATA_POWER_SAVING)
			{
			ieee80211_sta_ps_send_null_frame(ieee, 0);
			}
			else 
			{
				ieee80211_sta_ps_send_pspoll_frame(ieee);
			}
			spin_unlock_irqrestore(&ieee->mgmt_tx_lock, flags2);
		}
	}
	spin_unlock_irqrestore(&ieee->lock, flags);
}
#ifdef _RTL8192_EXT_PATCH_
void ieee80211_process_action(struct ieee80211_device* ieee, struct sk_buff* skb,
			      struct ieee80211_rx_stats *rx_stats)
#else
void ieee80211_process_action(struct ieee80211_device* ieee, struct sk_buff* skb)
#endif
{
	struct ieee80211_hdr* header = (struct ieee80211_hdr*)skb->data;
	u8* act = ieee80211_get_payload(header);
	u8 category = 0;

	if (act == NULL) {
		IEEE80211_DEBUG(IEEE80211_DL_ERR, "error to get payload of action frame\n");
		return;
	}
#ifdef _RTL8192_EXT_PATCH_
	if ((ieee->iw_mode != IW_MODE_MESH) && (ieee->iw_mode != IW_MODE_INFRA)) {
		IEEE80211_DEBUG(IEEE80211_DL_ERR, "not in mesh mode\n");
		return;
	}
#endif

	category = *act;
	act ++;
	switch (category) {
		case ACT_CAT_BA:
		switch (*act) {
		case ACT_ADDBAREQ:
			ieee80211_rx_ADDBAReq(ieee, skb);
			break;
		case ACT_ADDBARSP:
			ieee80211_rx_ADDBARsp(ieee, skb);
			break;
		case ACT_DELBA:
			ieee80211_rx_DELBA(ieee, skb);
			break;
		}
		break;
#ifdef _RTL8192_EXT_PATCH_      
	case ACT_CAT_MESH_PEERLINK_MGNT:
		if(ieee->iw_mode != IW_MODE_MESH) {
			IEEE80211_DEBUG(IEEE80211_DL_ERR, "peerlink received not in mesh mode\n");
			return;
		}
		switch (*act) {
		case ACT_PEERLINK_OPEN:
			if (ieee->ext_patch_ieee80211_rx_frame_softmac_on_peerlink_open) {
				ieee->ext_patch_ieee80211_rx_frame_softmac_on_peerlink_open(ieee, skb, rx_stats);
			}
			break;
		case ACT_PEERLINK_CONFIRM:
			if (ieee->ext_patch_ieee80211_rx_frame_softmac_on_peerlink_confirm) {
				ieee->ext_patch_ieee80211_rx_frame_softmac_on_peerlink_confirm(ieee, skb);
			}
			break;
		case ACT_PEERLINK_CLOSE:
			if ( ieee->ext_patch_ieee80211_rx_frame_softmac_on_peerlink_close) {
				ieee->ext_patch_ieee80211_rx_frame_softmac_on_peerlink_close(ieee, skb);
			}
			break;
		}
		break;
	case ACT_CAT_MESH_LINK_METRIC:
		if (ieee->iw_mode != IW_MODE_MESH) {
			IEEE80211_DEBUG(IEEE80211_DL_ERR, "link metric received not in mesh mode\n");
			return;
		}
		switch (*act) {
		case ACT_LINKMETRIC_REQ:
			if (ieee->ext_patch_ieee80211_rx_frame_softmac_on_linkmetric_req) {
				ieee->ext_patch_ieee80211_rx_frame_softmac_on_linkmetric_req(
						ieee, skb);
			}
			break;
		case ACT_LINKMETRIC_RSP:
			if (ieee->ext_patch_ieee80211_rx_frame_softmac_on_linkmetric_report) {
				ieee->ext_patch_ieee80211_rx_frame_softmac_on_linkmetric_report(
						ieee, skb);
			}
			break;
		}

		break;
	case ACT_CAT_MESH_PATH_SELECT:
		if (ieee->iw_mode != IW_MODE_MESH) {
			IEEE80211_DEBUG(IEEE80211_DL_ERR, "not in mesh mode\n");
			return;
		}

		switch (*act) {
		case ACT_PATH_REQ:
			if( ieee->ext_patch_ieee80211_rx_frame_softmac_on_pathselect_preq) {
				ieee->ext_patch_ieee80211_rx_frame_softmac_on_pathselect_preq(
						ieee, skb, rx_stats);
			}
			break;
		case ACT_PATH_REPLY:
			if(ieee->ext_patch_ieee80211_rx_frame_softmac_on_pathselect_prep) {
				ieee->ext_patch_ieee80211_rx_frame_softmac_on_pathselect_prep(
						ieee, skb, rx_stats);
			}
			break;
		case ACT_PATH_ERR:
			if(ieee->ext_patch_ieee80211_rx_frame_softmac_on_pathselect_perr) {
				ieee->ext_patch_ieee80211_rx_frame_softmac_on_pathselect_perr(
						ieee, skb, rx_stats);
			}
			break;
		case ACT_RANN:
			if(ieee->ext_patch_ieee80211_rx_frame_softmac_on_pathselect_rann) {
				ieee->ext_patch_ieee80211_rx_frame_softmac_on_pathselect_rann(
						ieee, skb, rx_stats);
			}
			break;
		}
		break;
	case ACT_CAT_MESH_INTERWORKING:
		if (ieee->iw_mode != IW_MODE_MESH) {
			IEEE80211_DEBUG(IEEE80211_DL_ERR, "not in mesh mode\n");
			return;
		}
		if (*act == 0) {
			if (ieee->ext_patch_ieee80211_rx_frame_softmac_on_pathselect_pann) {
				ieee->ext_patch_ieee80211_rx_frame_softmac_on_pathselect_pann(
						ieee, skb, rx_stats);
			}
		}
		break;
#endif
		default:
			break;
	}
	return;
}
inline int
ieee80211_rx_frame_softmac(struct ieee80211_device *ieee, struct sk_buff *skb,
			struct ieee80211_rx_stats *rx_stats, u16 type,
			u16 stype)
{
	struct ieee80211_hdr_3addr *header = (struct ieee80211_hdr_3addr *) skb->data;
	u16 errcode;
	u8* challenge;
	int chlen=0;
	int aid;
	struct ieee80211_assoc_response_frame *assoc_resp;
	bool bSupportNmode = true, bHalfSupportNmode = false; 
	
#ifdef _RTL8192_EXT_PATCH_
	if((!ieee->proto_started)&&(!ieee->mesh_started))
#else
	if(!ieee->proto_started)
#endif
		return 0;
		
	switch (WLAN_FC_GET_STYPE(header->frame_ctl)) {
	
		case IEEE80211_STYPE_ASSOC_RESP:
		case IEEE80211_STYPE_REASSOC_RESP:
		
			IEEE80211_DEBUG_MGMT("received [RE]ASSOCIATION RESPONSE (%d)\n",
					WLAN_FC_GET_STYPE(header->frame_ctl));
			if ((ieee->softmac_features & IEEE_SOFTMAC_ASSOCIATE) &&
				ieee->state == IEEE80211_ASSOCIATING_AUTHENTICATED && 
#ifdef _RTL8192_EXT_PATCH_
				((ieee->iw_mode == IW_MODE_INFRA) || 
				(ieee->iw_mode == IW_MODE_MESH && ieee->only_mesh == 0)))
#else
				(ieee->iw_mode == IW_MODE_INFRA)) 
#endif
			{
				
				if (0 == (errcode=assoc_parse(ieee,skb, &aid))){
#if (LINUX_VERSION_CODE > KERNEL_VERSION(2,6,13))
					struct ieee80211_network *network = kzalloc(sizeof(struct ieee80211_network), GFP_ATOMIC);
#else
					struct ieee80211_network *network = kmalloc(sizeof(struct ieee80211_network), GFP_ATOMIC);
                                        memset( network, 0x00, sizeof( struct ieee80211_network) );
#endif
					if (!network) {
						return 1;
					}

					ieee->state=IEEE80211_LINKED;
					ieee->assoc_id = aid;
					ieee->softmac_stats.rx_ass_ok++;
					/* station support qos */
					/* Let the register setting defaultly with Legacy station */
					if(ieee->qos_support) {
						assoc_resp = (struct ieee80211_assoc_response_frame*)skb->data;
						if (ieee80211_parse_info_param(ieee,assoc_resp->info_element,\
									rx_stats->len - sizeof(*assoc_resp),\
									network,rx_stats)){
							kfree(network);
							return 1;
						}
						else
						{	
							memcpy(ieee->pHTInfo->PeerHTCapBuf, network->bssht.bdHTCapBuf, network->bssht.bdHTCapLen);
							memcpy(ieee->pHTInfo->PeerHTInfoBuf, network->bssht.bdHTInfoBuf, network->bssht.bdHTInfoLen);
						}
						if (ieee->handle_assoc_response != NULL)
							ieee->handle_assoc_response(ieee->dev, (struct ieee80211_assoc_response_frame*)header, network);
						kfree(network);
					}
					ieee80211_associate_complete(ieee);
				} else {
					/* aid could not been allocated */
					ieee->softmac_stats.rx_ass_err++;
					printk(
						"Association response status code 0x%x\n",
						errcode);
					IEEE80211_DEBUG_MGMT(
						"Association response status code 0x%x\n",
						errcode);
					if(ieee->AsocRetryCount < RT_ASOC_RETRY_LIMIT) {
						queue_delayed_work_rsl(ieee->wq, &ieee->associate_procedure_wq, 0);
					} else {
						ieee80211_associate_abort(ieee); 
					}
				}
			}
			break;
		
		case IEEE80211_STYPE_ASSOC_REQ:
		case IEEE80211_STYPE_REASSOC_REQ:
		
			if ((ieee->softmac_features & IEEE_SOFTMAC_ASSOCIATE) &&
				ieee->iw_mode == IW_MODE_MASTER)
					
				ieee80211_rx_assoc_rq(ieee, skb);
			break;
			
		case IEEE80211_STYPE_AUTH:
			if (ieee->softmac_features & IEEE_SOFTMAC_ASSOCIATE){
				if (ieee->state == IEEE80211_ASSOCIATING_AUTHENTICATING && 
#ifdef _RTL8192_EXT_PATCH_
					((ieee->iw_mode == IW_MODE_INFRA)||
					((ieee->iw_mode == IW_MODE_MESH) && (ieee->only_mesh == 0))))
#else
					(ieee->iw_mode == IW_MODE_INFRA))
#endif
				{
						IEEE80211_DEBUG_MGMT("Received authentication response");
						
						if (0 == (errcode=auth_parse(skb, &challenge, &chlen))){
							if(ieee->open_wep || !challenge){
								ieee->state = IEEE80211_ASSOCIATING_AUTHENTICATED;
								ieee->softmac_stats.rx_auth_rs_ok++;
								if(!(ieee->pHTInfo->IOTAction&HT_IOT_ACT_PURE_N_MODE))
								{
									if (!ieee->GetNmodeSupportBySecCfg(ieee->dev))
									{
										if(IsHTHalfNmodeAPs(ieee))
										{	
											bSupportNmode = true;
											bHalfSupportNmode = true;
										}
										else
										{
											bSupportNmode = false;
											bHalfSupportNmode = false;
										}
									printk("==========>to link with AP using SEC(%d, %d)", bSupportNmode, bHalfSupportNmode);
									}
								}								
								/* Dummy wirless mode setting to avoid encryption issue */
								if(bSupportNmode) {
									ieee->SetWirelessMode(ieee->dev, \
											ieee->current_network.mode);
								}else{
									/*TODO*/
									ieee->SetWirelessMode(ieee->dev, IEEE_G);
								}

								if (ieee->current_network.mode == IEEE_N_24G && bHalfSupportNmode == true)
								{
									printk("===============>entern half N mode\n");
									ieee->bHalfWirelessN24GMode = true;
								}
								else
									ieee->bHalfWirelessN24GMode = false;
								
								ieee80211_associate_step2(ieee);
							}else{
								ieee80211_auth_challenge(ieee, challenge, chlen);
							}
						}else{
							ieee->softmac_stats.rx_auth_rs_err++;
							IEEE80211_DEBUG_MGMT("Authentication respose status code 0x%x",errcode);

							printk("Authentication respose status code 0x%x",errcode);
							ieee80211_associate_abort(ieee);
						}
						
					}else if (ieee->iw_mode == IW_MODE_MASTER){
						ieee80211_rx_auth_rq(ieee, skb);
					}
				}
			break;
#if 0				
		case IEEE80211_STYPE_PROBE_REQ:

			if ((ieee->softmac_features & IEEE_SOFTMAC_PROBERS) && 
				((ieee->iw_mode == IW_MODE_ADHOC || 
				ieee->iw_mode == IW_MODE_MASTER) &&
				ieee->state == IEEE80211_LINKED)){
				ieee80211_rx_probe_rq(ieee, skb);
			}
			break;
#endif			
		case IEEE80211_STYPE_DISASSOC:
		case IEEE80211_STYPE_DEAUTH:

			if(memcmp(header->addr3, ieee->current_network.bssid, ETH_ALEN) != 0)
				break;
			/* FIXME for now repeat all the association procedure 
			* both for disassociation and deauthentication
			*/
			if ((ieee->softmac_features & IEEE_SOFTMAC_ASSOCIATE) &&
				ieee->state == IEEE80211_LINKED && 
#ifdef _RTL8192_EXT_PATCH_
				((ieee->iw_mode == IW_MODE_INFRA) ||
				((ieee->iw_mode == IW_MODE_MESH) && (ieee->only_mesh == 0))))
#else
				(ieee->iw_mode == IW_MODE_INFRA))
#endif
			{
				printk("==========>received disassoc/deauth(%x) frame, reason code:%x\n",WLAN_FC_GET_STYPE(header->frame_ctl), ((struct ieee80211_disassoc*)skb->data)->reason);	
				ieee->state = IEEE80211_ASSOCIATING;
				ieee->softmac_stats.reassoc++;
				ieee->is_roaming = true;
				ieee->LinkDetectInfo.bBusyTraffic = false;
				ieee80211_disassociate(ieee);
				RemovePeerTS(ieee, header->addr2); 	
				if(ieee->LedControlHandler != NULL)
				        ieee->LedControlHandler(ieee->dev, LED_CTL_START_TO_LINK); 
				queue_delayed_work_rsl(ieee->wq, &ieee->associate_procedure_wq, 5);
			}
			break;
		case IEEE80211_STYPE_MANAGE_ACT:
#ifdef _RTL8192_EXT_PATCH_
			ieee80211_process_action(ieee,skb,rx_stats);
#else
			ieee80211_process_action(ieee,skb);
#endif
			break;
		default: 
			return -1;
			break;
	}
	
	return 0;
}

/* following are for a simplier TX queue management.
 * Instead of using netif_[stop/wake]_queue the driver
 * will uses these two function (plus a reset one), that
 * will internally uses the kernel netif_* and takes
 * care of the ieee802.11 fragmentation.
 * So the driver receives a fragment per time and might
 * call the stop function when it want without take care
 * to have enought room to TX an entire packet.
 * This might be useful if each fragment need it's own
 * descriptor, thus just keep a total free memory > than
 * the max fragmentation treshold is not enought.. If the
 * ieee802.11 stack passed a TXB struct then you needed  
 * to keep N free descriptors where 
 * N = MAX_PACKET_SIZE / MIN_FRAG_TRESHOLD
 * In this way you need just one and the 802.11 stack
 * will take care of buffering fragments and pass them to 
 * to the driver later, when it wakes the queue.
 */ 
void ieee80211_softmac_xmit(struct ieee80211_txb *txb, struct ieee80211_device *ieee)
{

	unsigned int queue_index = txb->queue_index;
	unsigned long flags;
	int  i;
	cb_desc *tcb_desc = NULL;
	unsigned long queue_len = 0;

	spin_lock_irqsave(&ieee->lock,flags);

	/* called with 2nd parm 0, no tx mgmt lock required */
	ieee80211_sta_wakeup(ieee,0);

	/* update the tx status */
	tcb_desc = (cb_desc *)(txb->fragments[0]->cb + MAX_DEV_ADDR_SIZE);
	if(tcb_desc->bMulticast) {
		ieee->stats.multicast++;
	}
#if 1
	/* if xmit available, just xmit it immediately, else just insert it to the wait queue */
	for(i = 0; i < txb->nr_frags; i++) {
#ifdef USB_TX_DRIVER_AGGREGATION_ENABLE
		queue_len = skb_queue_len(&ieee->skb_drv_aggQ[queue_index]);
#else
		queue_len = skb_queue_len(&ieee->skb_waitQ[queue_index]);
#endif
		if((queue_len  != 0) ||\
			(!ieee->check_nic_enough_desc(ieee->dev,queue_index))||\
		       (ieee->queue_stop)) {
			/* insert the skb packet to the wait queue */
			/* as for the completion function, it does not need 
			 * to check it any more.
			 * */
#ifdef WIFI_TEST
			if (1) 
#else
			if(queue_len < 200)
#endif
			{
#ifdef USB_TX_DRIVER_AGGREGATION_ENABLE
				skb_queue_tail(&ieee->skb_drv_aggQ[queue_index], txb->fragments[i]);
#else
				skb_queue_tail(&ieee->skb_waitQ[queue_index], txb->fragments[i]);
#endif
			}else{
				kfree_skb(txb->fragments[i]);
			}
		}else{
			ieee->softmac_data_hard_start_xmit(
					txb->fragments[i],
					ieee->dev,ieee->rate);
		}
	}	
#endif
	ieee80211_txb_free(txb);

	spin_unlock_irqrestore(&ieee->lock,flags);

}

/* called with ieee->lock acquired */
void ieee80211_resume_tx(struct ieee80211_device *ieee)
{
	int i;
	for(i = ieee->tx_pending.frag; i < ieee->tx_pending.txb->nr_frags; i++) {
		
		if (ieee->queue_stop){
			ieee->tx_pending.frag = i;
			return;
		}else{
		
			ieee->softmac_data_hard_start_xmit( 
				ieee->tx_pending.txb->fragments[i],
				ieee->dev,ieee->rate);
			ieee->stats.tx_packets++;
		}
	}
	
	
	ieee80211_txb_free(ieee->tx_pending.txb);
	ieee->tx_pending.txb = NULL;
}


void ieee80211_reset_queue(struct ieee80211_device *ieee)
{
	unsigned long flags;
	
	spin_lock_irqsave(&ieee->lock,flags);
	init_mgmt_queue(ieee);
	if (ieee->tx_pending.txb){
		ieee80211_txb_free(ieee->tx_pending.txb);
		ieee->tx_pending.txb = NULL;
	}
	ieee->queue_stop = 0;
	spin_unlock_irqrestore(&ieee->lock,flags);

}

void ieee80211_wake_queue(struct ieee80211_device *ieee)
{

	unsigned long flags;
	struct sk_buff *skb;
	struct ieee80211_hdr_3addr  *header;
	
	spin_lock_irqsave(&ieee->lock,flags);
	if (! ieee->queue_stop) goto exit;
	
	ieee->queue_stop = 0;
	
	if(ieee->softmac_features & IEEE_SOFTMAC_SINGLE_QUEUE){
		while (!ieee->queue_stop && (skb = dequeue_mgmt(ieee))){
			
			header = (struct ieee80211_hdr_3addr  *) skb->data;
			
			header->seq_ctl = cpu_to_le16(ieee->seq_ctrl[0] << 4);

			if (ieee->seq_ctrl[0] == 0xFFF)
				ieee->seq_ctrl[0] = 0;
			else
				ieee->seq_ctrl[0]++;

			ieee->softmac_data_hard_start_xmit(skb,ieee->dev,ieee->basic_rate);
		}
	}
	if (!ieee->queue_stop && ieee->tx_pending.txb)
		ieee80211_resume_tx(ieee);
	
	if (!ieee->queue_stop && netif_queue_stopped(ieee->dev)){
		ieee->softmac_stats.swtxawake++;
		netif_wake_queue(ieee->dev);
	}
	
exit :
	spin_unlock_irqrestore(&ieee->lock,flags);
}


void ieee80211_stop_queue(struct ieee80211_device *ieee)
{

	if (! netif_queue_stopped(ieee->dev)){
		netif_stop_queue(ieee->dev);
		ieee->softmac_stats.swtxstop++;
	}
	ieee->queue_stop = 1;
	
}


inline void ieee80211_randomize_cell(struct ieee80211_device *ieee)
{
	
	get_random_bytes(ieee->current_network.bssid, ETH_ALEN);
	
	/* an IBSS cell address must have the two less significant
	 * bits of the first byte = 2 
	 */
	ieee->current_network.bssid[0] &= ~0x01;
	ieee->current_network.bssid[0] |= 0x02;
}

/* called in user context only */
void ieee80211_start_master_bss(struct ieee80211_device *ieee)
{
	ieee->assoc_id = 1;
	
	if (ieee->current_network.ssid_len == 0){
		strncpy(ieee->current_network.ssid, 
			IEEE80211_DEFAULT_TX_ESSID,
			IW_ESSID_MAX_SIZE);
			
		ieee->current_network.ssid_len = strlen(IEEE80211_DEFAULT_TX_ESSID);
		ieee->ssid_set = 1;
	}
	
	memcpy(ieee->current_network.bssid, ieee->dev->dev_addr, ETH_ALEN);
	 
	ieee->set_chan(ieee->dev, ieee->current_network.channel);
	ieee->state = IEEE80211_LINKED;
	ieee->link_change(ieee->dev);
	notify_wx_assoc_event(ieee);
	
	if (ieee->data_hard_resume)
		ieee->data_hard_resume(ieee->dev);
	
	netif_carrier_on(ieee->dev);
}

void ieee80211_start_monitor_mode(struct ieee80211_device *ieee)
{
	if(ieee->raw_tx){
		
		if (ieee->data_hard_resume)
			ieee->data_hard_resume(ieee->dev);
	
		netif_carrier_on(ieee->dev);
	}
}
void ieee80211_start_ibss_wq(void *data)
{
	struct ieee80211_device *ieee = container_of_dwork_rsl(data, struct ieee80211_device, start_ibss_wq);
	/* iwconfig mode ad-hoc will schedule this and return
	 * on the other hand this will block further iwconfig SET
	 * operations because of the wx_sem hold.
	 * Anyway some most set operations set a flag to speed-up
	 * (abort) this wq (when syncro scanning) before sleeping 
	 * on the semaphore
	 */
	if(!ieee->proto_started){
		printk("==========oh driver down return\n");
		return;
	}
	down(&ieee->wx_sem);
	
	if (ieee->current_network.ssid_len == 0){
		strcpy(ieee->current_network.ssid,IEEE80211_DEFAULT_TX_ESSID);
		ieee->current_network.ssid_len = strlen(IEEE80211_DEFAULT_TX_ESSID);
		ieee->ssid_set = 1;
	} 
	
	/* check if we have this cell in our network list */
	ieee80211_softmac_check_all_nets(ieee);
	

	/* if not then the state is not linked. Maybe the user swithced to
	 * ad-hoc mode just after being in monitor mode, or just after
	 * being very few time in managed mode (so the card have had no
	 * time to scan all the chans..) or we have just run up the iface
	 * after setting ad-hoc mode. So we have to give another try..
	 * Here, in ibss mode, should be safe to do this without extra care
	 * (in bss mode we had to make sure no-one tryed to associate when
	 * we had just checked the ieee->state and we was going to start the
	 * scan) beacause in ibss mode the ieee80211_new_net function, when
	 * finds a good net, just set the ieee->state to IEEE80211_LINKED,
	 * so, at worst, we waste a bit of time to initiate an unneeded syncro
	 * scan, that will stop at the first round because it sees the state
	 * associated.
	 */
	if (ieee->state == IEEE80211_NOLINK)
#ifdef _RTL8192_EXT_PATCH_
		ieee80211_start_scan_syncro(ieee, 0);
#else
		ieee80211_start_scan_syncro(ieee);
#endif

	/* the network definitively is not here.. create a new cell */
	if (ieee->state == IEEE80211_NOLINK){
		printk("creating new IBSS cell\n"); 
#ifdef ENABLE_DOT11D 
		ieee->current_network.channel = ieee->IbssStartChnl;
#endif		
		if(!ieee->wap_set)
			ieee80211_randomize_cell(ieee);
		
		if(ieee->modulation & IEEE80211_CCK_MODULATION){
		
			ieee->current_network.rates_len = 4;
			
			ieee->current_network.rates[0] = IEEE80211_BASIC_RATE_MASK | IEEE80211_CCK_RATE_1MB;
			ieee->current_network.rates[1] = IEEE80211_BASIC_RATE_MASK | IEEE80211_CCK_RATE_2MB;
			ieee->current_network.rates[2] = IEEE80211_BASIC_RATE_MASK | IEEE80211_CCK_RATE_5MB;
			ieee->current_network.rates[3] = IEEE80211_BASIC_RATE_MASK | IEEE80211_CCK_RATE_11MB;
				
		}else
			ieee->current_network.rates_len = 0;
		
		if(ieee->modulation & IEEE80211_OFDM_MODULATION){
			ieee->current_network.rates_ex_len = 8;
			
			ieee->current_network.rates_ex[0] = IEEE80211_BASIC_RATE_MASK | IEEE80211_OFDM_RATE_6MB;
			ieee->current_network.rates_ex[1] = IEEE80211_BASIC_RATE_MASK | IEEE80211_OFDM_RATE_9MB;
			ieee->current_network.rates_ex[2] = IEEE80211_BASIC_RATE_MASK | IEEE80211_OFDM_RATE_12MB;
			ieee->current_network.rates_ex[3] = IEEE80211_BASIC_RATE_MASK | IEEE80211_OFDM_RATE_18MB;
			ieee->current_network.rates_ex[4] = IEEE80211_BASIC_RATE_MASK | IEEE80211_OFDM_RATE_24MB;
			ieee->current_network.rates_ex[5] = IEEE80211_BASIC_RATE_MASK | IEEE80211_OFDM_RATE_36MB;
			ieee->current_network.rates_ex[6] = IEEE80211_BASIC_RATE_MASK | IEEE80211_OFDM_RATE_48MB;
			ieee->current_network.rates_ex[7] = IEEE80211_BASIC_RATE_MASK | IEEE80211_OFDM_RATE_54MB;
			
			ieee->rate = 108;
		}else{
			ieee->current_network.rates_ex_len = 0;
			ieee->rate = 22;
		}

		ieee->current_network.QoS_Enable = 0;
#if defined(RTL8192U) || defined(RTL8192SU) || defined(RTL8192SE)
#ifdef ADHOC_11N
		ieee->SetWirelessMode(ieee->dev, ieee->mode);
#else
		ieee->SetWirelessMode(ieee->dev, IEEE_G);	
#endif
#else
		ieee->SetWirelessMode(ieee->dev, IEEE_G);	
#endif
		ieee->current_network.atim_window = 0;
		ieee->current_network.capability = WLAN_CAPABILITY_IBSS;
		if(ieee->short_slot)
			ieee->current_network.capability |= WLAN_CAPABILITY_SHORT_SLOT;
	}

	printk("%s(): ieee->mode = %d\n", __FUNCTION__, ieee->mode);
	if((ieee->mode == IEEE_N_24G) || (ieee->mode == IEEE_N_5G))
		HTUseDefaultSetting(ieee);
	ieee->state = IEEE80211_LINKED;
		
#ifdef _RTL8192_EXT_PATCH_
	ieee->set_chan(ieee->dev, ieee->current_network.channel);  
#endif
	ieee->link_change(ieee->dev);

#if defined(RTL8192U) || defined(RTL8192SU) || defined(RTL8192SE)
	ieee->SetBeaconRelatedRegistersHandler(ieee->dev);

	if(ieee->pHTInfo->bCurBW40MHz)
		HTSetConnectBwMode(ieee, HT_CHANNEL_WIDTH_20_40, (ieee->current_network.channel<=6)?HT_EXTCHNL_OFFSET_UPPER:HT_EXTCHNL_OFFSET_LOWER);  
	else
		HTSetConnectBwMode(ieee, HT_CHANNEL_WIDTH_20, (ieee->current_network.channel<=6)?HT_EXTCHNL_OFFSET_UPPER:HT_EXTCHNL_OFFSET_LOWER);  
#else
	HTSetConnectBwMode(ieee, HT_CHANNEL_WIDTH_20, HT_EXTCHNL_OFFSET_NO_EXT);
#endif
	if(ieee->LedControlHandler != NULL)
	        ieee->LedControlHandler(ieee->dev,LED_CTL_LINK);
	notify_wx_assoc_event(ieee);
	
	ieee80211_start_send_beacons(ieee);
	
	if (ieee->data_hard_resume)
		ieee->data_hard_resume(ieee->dev);
	netif_carrier_on(ieee->dev);
	
	up(&ieee->wx_sem);
}

inline void ieee80211_start_ibss(struct ieee80211_device *ieee)
{
	queue_delayed_work_rsl(ieee->wq, &ieee->start_ibss_wq, MSECS(150));
}

/* this is called only in user context, with wx_sem held */
void ieee80211_start_bss(struct ieee80211_device *ieee)
{
	unsigned long flags;
#ifdef ENABLE_DOT11D
	if(IS_DOT11D_ENABLE(ieee) && !IS_COUNTRY_IE_VALID(ieee))
	{
		if(! ieee->bGlobalDomain)
		{
			return;
		}
	}
#endif	
	/* check if we have already found the net we
	 * are interested in (if any).
	 * if not (we are disassociated and we are not
	 * in associating / authenticating phase) start the background scanning.
	 */
	ieee80211_softmac_check_all_nets(ieee);
	
	/* ensure no-one start an associating process (thus setting
	 * the ieee->state to ieee80211_ASSOCIATING) while we
	 * have just cheked it and we are going to enable scan.
	 * The ieee80211_new_net function is always called with
	 * lock held (from both ieee80211_softmac_check_all_nets and
	 * the rx path), so we cannot be in the middle of such function
	 */
	spin_lock_irqsave(&ieee->lock, flags);
	
	if (ieee->state == IEEE80211_NOLINK){
		ieee->actscanning = true;
		ieee80211_start_scan(ieee);
	}
	spin_unlock_irqrestore(&ieee->lock, flags);
}

void ieee80211_link_change_wq(void *data)
{
	struct ieee80211_device *ieee = container_of_dwork_rsl(data, struct ieee80211_device, link_change_wq);
	ieee->link_change(ieee->dev);
}
/* called only in userspace context */
void ieee80211_disassociate(struct ieee80211_device *ieee)
{
	netif_carrier_off(ieee->dev);
	if (ieee->softmac_features & IEEE_SOFTMAC_TX_QUEUE)
			ieee80211_reset_queue(ieee);
	
	if (ieee->data_hard_stop)
			ieee->data_hard_stop(ieee->dev);
#ifdef ENABLE_DOT11D
	if(IS_DOT11D_ENABLE(ieee))
		Dot11d_Reset(ieee);
#endif
	ieee->state = IEEE80211_NOLINK;
	ieee->is_set_key = false;
	ieee->wap_set = 0;

	queue_delayed_work_rsl(ieee->wq, &ieee->link_change_wq, 0);

	notify_wx_assoc_event(ieee);
	
}

void ieee80211_associate_retry_wq(void *data)
{
	struct ieee80211_device *ieee = container_of_dwork_rsl(data, struct ieee80211_device, associate_retry_wq);
	unsigned long flags;
	
	down(&ieee->wx_sem);
	if(!ieee->proto_started)
		goto exit;
		
	if(ieee->state != IEEE80211_ASSOCIATING_RETRY)
		goto exit;
		
	/* until we do not set the state to IEEE80211_NOLINK 
	* there are no possibility to have someone else trying
	* to start an association procdure (we get here with
	* ieee->state = IEEE80211_ASSOCIATING).
	* When we set the state to IEEE80211_NOLINK it is possible
	* that the RX path run an attempt to associate, but
	* both ieee80211_softmac_check_all_nets and the
	* RX path works with ieee->lock held so there are no
	* problems. If we are still disassociated then start a scan.
	* the lock here is necessary to ensure no one try to start
	* an association procedure when we have just checked the 
	* state and we are going to start the scan.
	*/
	ieee->beinretry = true;
	ieee->state = IEEE80211_NOLINK;

	ieee80211_softmac_check_all_nets(ieee);
	
	spin_lock_irqsave(&ieee->lock, flags);
	
	if(ieee->state == IEEE80211_NOLINK)
	{
		ieee->actscanning = true;
		ieee80211_start_scan(ieee);
	}
	spin_unlock_irqrestore(&ieee->lock, flags);

	ieee->beinretry = false;
exit:
	up(&ieee->wx_sem);
}

struct sk_buff *ieee80211_get_beacon_(struct ieee80211_device *ieee)
{
	u8 broadcast_addr[] = {0xff,0xff,0xff,0xff,0xff,0xff};
	
	struct sk_buff *skb;
	struct ieee80211_probe_response *b;
#ifdef _RTL8192_EXT_PATCH_
	if((ieee->iw_mode == IW_MODE_MESH)&&(ieee->ext_patch_get_beacon_get_probersp ))
		skb = ieee->ext_patch_get_beacon_get_probersp(ieee, broadcast_addr, &(ieee->current_mesh_network));
	else
		skb = ieee80211_probe_resp(ieee, broadcast_addr);
#else
	skb = ieee80211_probe_resp(ieee, broadcast_addr);
#endif
	
	if (!skb) 
		return NULL;
	
	b = (struct ieee80211_probe_response *) skb->data;
	b->header.frame_ctl = cpu_to_le16(IEEE80211_STYPE_BEACON);
		
	return skb;
	
}

struct sk_buff *ieee80211_get_beacon(struct ieee80211_device *ieee)
{
	struct sk_buff *skb;
	struct ieee80211_probe_response *b;
	
	skb = ieee80211_get_beacon_(ieee);
	if(!skb) 
		return NULL;
		
	b = (struct ieee80211_probe_response *) skb->data;	
	b->header.seq_ctl = cpu_to_le16(ieee->seq_ctrl[0] << 4);
	
	if (ieee->seq_ctrl[0] == 0xFFF)
		ieee->seq_ctrl[0] = 0;
	else
		ieee->seq_ctrl[0]++;
	
	return skb;
}

#ifdef _RTL8192_EXT_PATCH_	
void ieee80211_softmac_stop_protocol(struct ieee80211_device *ieee, u8 mesh_flag, u8 shutdown)
#else
void ieee80211_softmac_stop_protocol(struct ieee80211_device *ieee,u8 shutdown)
#endif
{
	ieee->sync_scan_hurryup = 1; 
	down(&ieee->wx_sem);
#ifdef _RTL8192_EXT_PATCH_	
	if(mesh_flag) {
		ieee80211_stop_mesh_protocol(ieee);
	}
	else
#endif	
		ieee80211_stop_protocol(ieee,shutdown);
	up(&ieee->wx_sem);
}


void ieee80211_stop_protocol(struct ieee80211_device *ieee, u8 shutdown)
{
	if (!ieee->proto_started)
		return;
	
	if(shutdown){
	ieee->proto_started = 0;
		ieee->proto_stoppping = 1;
#ifdef ENABLE_IPS
		if(ieee->ieee80211_ips_leave != NULL)
			ieee->ieee80211_ips_leave(ieee->dev);
#endif
	}
	
	ieee80211_stop_send_beacons(ieee);
	del_timer_sync(&ieee->associate_timer);
#if defined(RTL8192U) || defined(RTL8192SU) || defined(RTL8192SE)
	del_timer_sync(&ieee->ibss_wait_timer);
#endif
#if LINUX_VERSION_CODE > KERNEL_VERSION(2,5,0)	
	cancel_delayed_work(&ieee->associate_retry_wq);	
	cancel_delayed_work(&ieee->start_ibss_wq);
	cancel_delayed_work(&ieee->link_change_wq);
#endif	
	ieee80211_stop_scan(ieee);

	ieee80211_disassociate(ieee);
	if(shutdown){
	RemoveAllTS(ieee); 
		ieee->proto_stoppping = 0;
	}
}

#ifdef _RTL8192_EXT_PATCH_	
void ieee80211_stop_mesh_protocol(struct ieee80211_device *ieee)
{
	if (!ieee->mesh_started)
		return;
	ieee->mesh_started = 0;

	if(ieee->ext_patch_ieee80211_stop_protocol)
		ieee->ext_patch_ieee80211_stop_protocol(ieee,0);
	
	ieee80211_stop_send_beacons(ieee);
}
#endif	

#ifdef _RTL8192_EXT_PATCH_	
void ieee80211_softmac_start_protocol(struct ieee80211_device *ieee, u8 mesh_flag)
#else
void ieee80211_softmac_start_protocol(struct ieee80211_device *ieee)
#endif	
{
	ieee->sync_scan_hurryup = 0;  
	down(&ieee->wx_sem);
#ifdef _RTL8192_EXT_PATCH_	
	if (mesh_flag) {
		ieee80211_start_mesh_protocol(ieee);
	}
	else
#endif	
		ieee80211_start_protocol(ieee);
	up(&ieee->wx_sem);
}

void ieee80211_start_protocol(struct ieee80211_device *ieee)
{
	short ch = 0;
 	int i = 0;	
	
	if (ieee->proto_started)
		return;
		
	ieee->proto_started = 1;
	
	if (ieee->current_network.channel == 0){
		do{
			ch++;
			if (ch > MAX_CHANNEL_NUMBER) 
				return; /* no channel found */
#ifdef ENABLE_DOT11D
		}while(!GET_DOT11D_INFO(ieee)->channel_map[ch]);
#else
		}while(!ieee->channel_map[ch]);
#endif	
		ieee->current_network.channel = ch;
	}
	
	if (ieee->current_network.beacon_interval == 0)
		ieee->current_network.beacon_interval = 100;
	
       	for(i = 0; i < 17; i++) {
	  ieee->last_rxseq_num[i] = -1;
	  ieee->last_rxfrag_num[i] = -1;
	  ieee->last_packet_time[i] = 0;
	}

	ieee->wmm_acm = 0;
	/* if the user set the MAC of the ad-hoc cell and then
	 * switch to managed mode, shall we  make sure that association
	 * attempts does not fail just because the user provide the essid
	 * and the nic is still checking for the AP MAC ??
	 */
#ifdef _RTL8192_EXT_PATCH_
	if ((ieee->iw_mode == IW_MODE_INFRA) || ((ieee->iw_mode == IW_MODE_MESH) && (ieee->only_mesh == 0)))  
#else
	if (ieee->iw_mode == IW_MODE_INFRA)  
#endif		
		ieee80211_start_bss(ieee);
		
	else if (ieee->iw_mode == IW_MODE_ADHOC)
		ieee80211_start_ibss(ieee);
		
	else if (ieee->iw_mode == IW_MODE_MASTER)
		ieee80211_start_master_bss(ieee);
		
	else if(ieee->iw_mode == IW_MODE_MONITOR)
		ieee80211_start_monitor_mode(ieee);	
}

#ifdef _RTL8192_EXT_PATCH_
void ieee80211_start_mesh_protocol(struct ieee80211_device *ieee)
{
	short ch = 0;
	if (ieee->mesh_started)
		return;

	ieee->mesh_started = 1;
	
	if (ieee->current_mesh_network.channel == 0){
		do{
			ch++;
			if (ch > MAX_CHANNEL_NUMBER) 
				return; /* no channel found */
#ifdef ENABLE_DOT11D
		}while(!GET_DOT11D_INFO(ieee)->channel_map[ch]);
#else
		}while(!ieee->channel_map[ch]);
#endif	
		ieee->current_mesh_network.channel = ch;
	}
	
	if (ieee->current_mesh_network.beacon_interval == 0)
		ieee->current_mesh_network.beacon_interval = 100;

	ieee->wmm_acm = 0;

	if(ieee->ext_patch_ieee80211_start_protocol)
	{
		 ieee->ext_patch_ieee80211_start_mesh(ieee);
	}
}
#endif


#define DRV_NAME  "Ieee80211"
void ieee80211_softmac_init(struct ieee80211_device *ieee)
{
	int i;
	memset(&ieee->current_network, 0, sizeof(struct ieee80211_network));
	
	ieee->state = IEEE80211_NOLINK;
#ifdef _RTL8192_EXT_PATCH_
	ieee->mesh_state = IEEE80211_NOLINK;
#endif
	ieee->sync_scan_hurryup = 0;
	for(i = 0; i < 5; i++) {
	  ieee->seq_ctrl[i] = 0;
	}
#ifdef ENABLE_DOT11D
	ieee->pDot11dInfo = kmalloc(sizeof(RT_DOT11D_INFO), GFP_ATOMIC);
	if (!ieee->pDot11dInfo)
		IEEE80211_DEBUG(IEEE80211_DL_ERR, "can't alloc memory for DOT11D\n");
	memset(ieee->pDot11dInfo, 0, sizeof(RT_DOT11D_INFO));
#endif
	ieee->LinkDetectInfo.SlotIndex = 0;
	ieee->LinkDetectInfo.SlotNum = 2;
	ieee->LinkDetectInfo.NumRecvBcnInPeriod=0;
        ieee->LinkDetectInfo.NumRecvDataInPeriod=0;
	ieee->LinkDetectInfo.NumTxOkInPeriod =0;
	ieee->LinkDetectInfo.NumRxOkInPeriod =0;
	ieee->LinkDetectInfo.NumRxUnicastOkInPeriod=0;
#ifdef _RTL8192_EXT_PATCH_
	ieee->LinkDetectInfo.LastNumRxUnicast = 0;
	ieee->LinkDetectInfo.LastNumTxUnicast = 0;
	ieee->LinkDetectInfo.IdleCount = 0;
#endif
	ieee->bIsAggregateFrame = false;
	ieee->assoc_id = 0;
	ieee->queue_stop = 0;
	ieee->scanning = 0;
	ieee->softmac_features = 0; 
	ieee->wap_set = 0;
	ieee->ssid_set = 0;
	ieee->proto_started = 0;
	ieee->proto_stoppping = 0;
	ieee->basic_rate = IEEE80211_DEFAULT_BASIC_RATE;
	ieee->rate = 22;
	ieee->ps = IEEE80211_PS_DISABLED;
	ieee->sta_sleep = 0;

#ifdef _RTL8192_EXT_PATCH_
	ieee->mesh_started = 0;
#endif
	ieee->Regdot11HTOperationalRateSet[0]= 0xff;
	ieee->Regdot11HTOperationalRateSet[1]= 0xff;
	ieee->Regdot11HTOperationalRateSet[4]= 0x01;

	ieee->Regdot11TxHTOperationalRateSet[0]= 0xff;
	ieee->Regdot11TxHTOperationalRateSet[1]= 0xff;
	ieee->Regdot11TxHTOperationalRateSet[4]= 0x01;
	
	ieee->actscanning = false;
	ieee->beinretry = false;
	ieee->is_set_key = false;
	init_mgmt_queue(ieee);

	ieee->sta_edca_param[0] = 0x0000A403;
	ieee->sta_edca_param[1] = 0x0000A427;
	ieee->sta_edca_param[2] = 0x005E4342;
	ieee->sta_edca_param[3] = 0x002F3262;
	ieee->aggregation = true;
	ieee->enable_rx_imm_BA = 1;
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,20)	
	_setup_timer(&ieee->scan_timer,
		    ieee80211_softmac_scan_cb,
		    (unsigned long) ieee);
#endif
	ieee->tx_pending.txb = NULL;
	
	_setup_timer(&ieee->associate_timer,
		    ieee80211_associate_abort_cb,
		    (unsigned long) ieee);

	_setup_timer(&ieee->beacon_timer,
		    ieee80211_send_beacon_cb,
		    (unsigned long) ieee);

#if defined(RTL8192U) || defined(RTL8192SU) || defined(RTL8192SE)
	_setup_timer(&ieee->ibss_wait_timer,
		    ieee80211_ibss_wait_timeout,
		    (unsigned long) ieee);
#endif

#if LINUX_VERSION_CODE > KERNEL_VERSION(2,5,0)	
#ifdef PF_SYNCTHREAD
	ieee->wq = create_workqueue(DRV_NAME,0);
#else	
	ieee->wq = create_workqueue(DRV_NAME);
#endif
#endif

	INIT_DELAYED_WORK_RSL(&ieee->link_change_wq,(void*)ieee80211_link_change_wq,ieee);
	INIT_DELAYED_WORK_RSL(&ieee->start_ibss_wq,(void*)ieee80211_start_ibss_wq,ieee);
	INIT_WORK_RSL(&ieee->associate_complete_wq, (void*)ieee80211_associate_complete_wq,ieee);
	INIT_DELAYED_WORK_RSL(&ieee->associate_procedure_wq, (void*)ieee80211_associate_procedure_wq,ieee);
	INIT_DELAYED_WORK_RSL(&ieee->softmac_scan_wq,(void*)ieee80211_softmac_scan_wq,ieee);
	INIT_DELAYED_WORK_RSL(&ieee->associate_retry_wq, (void*)ieee80211_associate_retry_wq,ieee);
	INIT_WORK_RSL(&ieee->wx_sync_scan_wq,(void*)ieee80211_wx_sync_scan_wq,ieee);

#ifdef _RTL8192_EXT_PATCH_
	INIT_WORK_RSL(&ieee->ext_stop_scan_wq,(void*) ieee80211_ext_stop_scan_wq,ieee);
	INIT_WORK_RSL(&ieee->ext_send_beacon_wq,(void*) ext_ieee80211_send_beacon_wq,ieee);
#endif 

	sema_init(&ieee->wx_sem, 1);
	sema_init(&ieee->scan_sem, 1);
	sema_init(&ieee->ips_sem,1);
	
	spin_lock_init(&ieee->mgmt_tx_lock);
	spin_lock_init(&ieee->beacon_lock);
	
	tasklet_init(&ieee->ps_task,
	     (void(*)(unsigned long)) ieee80211_sta_ps,
	     (unsigned long)ieee);

}

void ieee80211_softmac_free(struct ieee80211_device *ieee)
{
	down(&ieee->wx_sem);
#ifdef ENABLE_DOT11D
	if(NULL != ieee->pDot11dInfo)
	{
		kfree(ieee->pDot11dInfo);
		ieee->pDot11dInfo = NULL;
	}
#endif
	del_timer_sync(&ieee->associate_timer);

#if LINUX_VERSION_CODE > KERNEL_VERSION(2,5,0)	
	cancel_delayed_work(&ieee->associate_retry_wq);
	destroy_workqueue(ieee->wq);
#endif
	
	up(&ieee->wx_sem);
}

/******************************************************** 
 * Start of WPA code.                                   *
 * this is stolen from the ipw2200 driver               *
 ********************************************************/

 
static int ieee80211_wpa_enable(struct ieee80211_device *ieee, int value)
{
	/* This is called when wpa_supplicant loads and closes the driver
	 * interface. */
	printk("%s WPA\n",value ? "enabling" : "disabling");
	ieee->wpa_enabled = value;
	memset(ieee->ap_mac_addr, 0, 6); 
	return 0;
}

 
void ieee80211_wpa_assoc_frame(struct ieee80211_device *ieee, char *wpa_ie, int wpa_ie_len)
{
	/* make sure WPA is enabled */
	ieee80211_wpa_enable(ieee, 1);

	ieee80211_disassociate(ieee);
}


static int ieee80211_wpa_mlme(struct ieee80211_device *ieee, int command, int reason)
{
	
	int ret = 0;

	switch (command) {
	case IEEE_MLME_STA_DEAUTH:
		break;

	case IEEE_MLME_STA_DISASSOC:
		ieee80211_disassociate(ieee);
		break;

	default:
		printk("Unknown MLME request: %d\n", command);
		ret = -EOPNOTSUPP;
	}

	return ret;
}


static int ieee80211_wpa_set_wpa_ie(struct ieee80211_device *ieee,
			      struct ieee_param *param, int plen)
{
	u8 *buf;

	if (param->u.wpa_ie.len > MAX_WPA_IE_LEN ||
	    (param->u.wpa_ie.len && param->u.wpa_ie.data == NULL))
		return -EINVAL;

	if (param->u.wpa_ie.len) {
		buf = kmalloc(param->u.wpa_ie.len, GFP_KERNEL);
		if (buf == NULL)
			return -ENOMEM;

		memcpy(buf, param->u.wpa_ie.data, param->u.wpa_ie.len);
		kfree(ieee->wpa_ie);
		ieee->wpa_ie = buf;
		ieee->wpa_ie_len = param->u.wpa_ie.len;
	} else {
		kfree(ieee->wpa_ie);
		ieee->wpa_ie = NULL;
		ieee->wpa_ie_len = 0;
	}

	ieee80211_wpa_assoc_frame(ieee, ieee->wpa_ie, ieee->wpa_ie_len);
	return 0;
}

#define AUTH_ALG_OPEN_SYSTEM			0x1
#define AUTH_ALG_SHARED_KEY			0x2
#define AUTH_ALG_LEAP				0x4
static int ieee80211_wpa_set_auth_algs(struct ieee80211_device *ieee, int value)
{
	
	struct ieee80211_security sec = {
		.flags = SEC_AUTH_MODE,
	};
	int ret = 0;

	if (value & AUTH_ALG_SHARED_KEY) {
		sec.auth_mode = WLAN_AUTH_SHARED_KEY;
		ieee->open_wep = 0;
		ieee->auth_mode = 1;
	} else if (value & AUTH_ALG_OPEN_SYSTEM){
		sec.auth_mode = WLAN_AUTH_OPEN;
		ieee->open_wep = 1;
		ieee->auth_mode = 0;
	}
	else if (value & AUTH_ALG_LEAP){
		sec.auth_mode = WLAN_AUTH_LEAP;
		ieee->open_wep = 1;
		ieee->auth_mode = 2;
	}


	if (ieee->set_security)
		ieee->set_security(ieee->dev, &sec);

	return ret;
}

static int ieee80211_wpa_set_param(struct ieee80211_device *ieee, u8 name, u32 value)
{
	int ret=0;
	unsigned long flags;

	switch (name) {
	case IEEE_PARAM_WPA_ENABLED:
		ret = ieee80211_wpa_enable(ieee, value);
		break;

	case IEEE_PARAM_TKIP_COUNTERMEASURES:
		ieee->tkip_countermeasures=value;
		break;

	case IEEE_PARAM_DROP_UNENCRYPTED: {
		/* HACK:
		 *
		 * wpa_supplicant calls set_wpa_enabled when the driver
		 * is loaded and unloaded, regardless of if WPA is being
		 * used.  No other calls are made which can be used to
		 * determine if encryption will be used or not prior to
		 * association being expected.  If encryption is not being
		 * used, drop_unencrypted is set to false, else true -- we
		 * can use this to determine if the CAP_PRIVACY_ON bit should
		 * be set.
		 */
		struct ieee80211_security sec = {
			.flags = SEC_ENABLED,
			.enabled = value,
		};
 		ieee->drop_unencrypted = value;
		/* We only change SEC_LEVEL for open mode. Others
		 * are set by ipw_wpa_set_encryption.
		 */
		if (!value) {
			sec.flags |= SEC_LEVEL;
			sec.level = SEC_LEVEL_0;
		}
		else {
			sec.flags |= SEC_LEVEL;
			sec.level = SEC_LEVEL_1;
		}
		if (ieee->set_security)
			ieee->set_security(ieee->dev, &sec);
		break;
	}

	case IEEE_PARAM_PRIVACY_INVOKED:
		ieee->privacy_invoked=value;
		break;

	case IEEE_PARAM_AUTH_ALGS:
		ret = ieee80211_wpa_set_auth_algs(ieee, value);
		break;

	case IEEE_PARAM_IEEE_802_1X:
		ieee->ieee802_1x=value;
		break;
	case IEEE_PARAM_WPAX_SELECT:
		spin_lock_irqsave(&ieee->wpax_suitlist_lock,flags);
		spin_unlock_irqrestore(&ieee->wpax_suitlist_lock,flags);
		break;

	default:
		printk("Unknown WPA param: %d\n",name);
		ret = -EOPNOTSUPP;
	}

	return ret;
}

/* implementation borrowed from hostap driver */

#ifdef _RTL8192_EXT_PATCH_
static int ieee80211_wpa_set_encryption(struct ieee80211_device *ieee,
				  struct ieee_param *param, int param_len, u8 is_mesh)
#else
static int ieee80211_wpa_set_encryption(struct ieee80211_device *ieee,
				  struct ieee_param *param, int param_len)
#endif
{
	int ret = 0;
#ifdef _RTL8192_EXT_PATCH_
	u8 i = 0;
#endif
	struct ieee80211_crypto_ops *ops;
	struct ieee80211_crypt_data **crypt;

	struct ieee80211_security sec = {
		.flags = 0,
	};

	param->u.crypt.err = 0;
	param->u.crypt.alg[IEEE_CRYPT_ALG_NAME_LEN - 1] = '\0';

	if (param_len !=
	    (int) ((char *) param->u.crypt.key - (char *) param) +
	    param->u.crypt.key_len) {
		printk("Len mismatch %d, %d\n", param_len,
			       param->u.crypt.key_len);
		return -EINVAL;
	}
	if (param->sta_addr[0] == 0xff && param->sta_addr[1] == 0xff &&
	    param->sta_addr[2] == 0xff && param->sta_addr[3] == 0xff &&
	    param->sta_addr[4] == 0xff && param->sta_addr[5] == 0xff) {
		if (param->u.crypt.idx >= WEP_KEYS)
			return -EINVAL;
#ifdef _RTL8192_EXT_PATCH_
		if(is_mesh)
			crypt = &ieee->cryptlist[0]->crypt[param->u.crypt.idx];
		else
			crypt = &ieee->sta_crypt[param->u.crypt.idx];
#else
		crypt = &ieee->crypt[param->u.crypt.idx];
#endif
	} else {
		return -EINVAL;
	}

	if (strcmp(param->u.crypt.alg, "none") == 0) {
		if (crypt) {
			sec.enabled = 0;
			sec.level = SEC_LEVEL_0;
			sec.flags |= SEC_ENABLED | SEC_LEVEL;
			ieee80211_crypt_delayed_deinit(ieee, crypt);
		}
		goto done;
	}
	sec.enabled = 1;
	sec.flags |= SEC_ENABLED;

	/* IPW HW cannot build TKIP MIC, host decryption still needed. */
	if (!(ieee->host_encrypt || ieee->host_decrypt) &&
	    strcmp(param->u.crypt.alg, "TKIP"))
		goto skip_host_crypt;

	ops = ieee80211_get_crypto_ops(param->u.crypt.alg);
	if (ops == NULL && strcmp(param->u.crypt.alg, "WEP") == 0) {
		request_module("ieee80211_crypt_wep");
		ops = ieee80211_get_crypto_ops(param->u.crypt.alg);
	} else if (ops == NULL && strcmp(param->u.crypt.alg, "TKIP") == 0) {
		request_module("ieee80211_crypt_tkip");
		ops = ieee80211_get_crypto_ops(param->u.crypt.alg);
	} else if (ops == NULL && strcmp(param->u.crypt.alg, "CCMP") == 0) {
		request_module("ieee80211_crypt_ccmp");
		ops = ieee80211_get_crypto_ops(param->u.crypt.alg);
	}
	if (ops == NULL) {
		printk("unknown crypto alg '%s'\n", param->u.crypt.alg);
		param->u.crypt.err = IEEE_CRYPT_ERR_UNKNOWN_ALG;
		ret = -EINVAL;
		goto done;
	}
#ifdef _RTL8192_EXT_PATCH_
	if(is_mesh)  
	{
		for (i=0; i<MAX_MP; i++){
			crypt = &ieee->cryptlist[i]->crypt[param->u.crypt.idx];

			*crypt = ieee->cryptlist[i]->crypt[param->u.crypt.idx];
		}
#endif

		if (*crypt == NULL || (*crypt)->ops != ops) {
			struct ieee80211_crypt_data *new_crypt;

			ieee80211_crypt_delayed_deinit(ieee, crypt);

			new_crypt = (struct ieee80211_crypt_data *)
				kmalloc(sizeof(*new_crypt), GFP_KERNEL);
			if (new_crypt == NULL) {
				ret = -ENOMEM;
				goto done;
			}
			memset(new_crypt, 0, sizeof(struct ieee80211_crypt_data));
			new_crypt->ops = ops;
#ifdef BUILT_IN_IEEE80211
			if (new_crypt->ops)
#else
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,5,0)
			if (new_crypt->ops && try_module_get(new_crypt->ops->owner))
#else
			if (new_crypt->ops && try_inc_mod_count(new_crypt->ops->owner))
#endif			
#endif			
					new_crypt->priv =
						new_crypt->ops->init(param->u.crypt.idx);

			if (new_crypt->priv == NULL) {
				kfree(new_crypt);
				param->u.crypt.err = IEEE_CRYPT_ERR_CRYPT_INIT_FAILED;
				ret = -EINVAL;
				goto done;
			}

			*crypt = new_crypt;
		}

		if (param->u.crypt.key_len > 0 && (*crypt)->ops->set_key &&
				(*crypt)->ops->set_key(param->u.crypt.key,
					param->u.crypt.key_len, param->u.crypt.seq,
					(*crypt)->priv) < 0) {
			printk("key setting failed\n");
			param->u.crypt.err = IEEE_CRYPT_ERR_KEY_SET_FAILED;
			ret = -EINVAL;
			goto done;
		}
#ifdef _RTL8192_EXT_PATCH_
	}
#endif

 skip_host_crypt:
	if (param->u.crypt.set_tx) {
		ieee->tx_keyidx = param->u.crypt.idx;
		sec.active_key = param->u.crypt.idx;
		sec.flags |= SEC_ACTIVE_KEY;
	} else
		sec.flags &= ~SEC_ACTIVE_KEY;

	if (param->u.crypt.alg != NULL) {
		memcpy(sec.keys[param->u.crypt.idx],
		       param->u.crypt.key,
		       param->u.crypt.key_len);
		sec.key_sizes[param->u.crypt.idx] = param->u.crypt.key_len;
		sec.flags |= (1 << param->u.crypt.idx);

		if (strcmp(param->u.crypt.alg, "WEP") == 0) {
			sec.flags |= SEC_LEVEL;
			sec.level = SEC_LEVEL_1;
		} else if (strcmp(param->u.crypt.alg, "TKIP") == 0) {
			sec.flags |= SEC_LEVEL;
			sec.level = SEC_LEVEL_2;
		} else if (strcmp(param->u.crypt.alg, "CCMP") == 0) {
			sec.flags |= SEC_LEVEL;
			sec.level = SEC_LEVEL_3;
		}
	}
 done:
	if (ieee->set_security)
		ieee->set_security(ieee->dev, &sec);

#ifdef _RTL8192_EXT_PATCH_
	if (ret != 0)
	{
		if(is_mesh)   
		{
			for (i=0; i<MAX_MP; i++)
			{
				if (ieee->cryptlist[i]->crypt[param->u.crypt.idx]==NULL){
					break;
				}
				else{
					kfree(ieee->cryptlist[i]->crypt[param->u.crypt.idx]);
					ieee->cryptlist[i]->crypt[param->u.crypt.idx] = NULL;
				}
			}
		}
		else
		{
			kfree(ieee->sta_crypt[param->u.crypt.idx]);
			ieee->sta_crypt[param->u.crypt.idx] = NULL;
		}
	}
#endif
	

	/* Do not reset port if card is in Managed mode since resetting will
	 * generate new IEEE 802.11 authentication which may end up in looping
	 * with IEEE 802.1X.  If your hardware requires a reset after WEP
	 * configuration (for example... Prism2), implement the reset_port in
	 * the callbacks structures used to initialize the 802.11 stack. */
	if (ieee->reset_on_keychange &&
	    ieee->iw_mode != IW_MODE_INFRA &&
	    ieee->reset_port &&
	    ieee->reset_port(ieee->dev)) {
		printk("reset_port failed\n");
		param->u.crypt.err = IEEE_CRYPT_ERR_CARD_CONF_FAILED;
		return -EINVAL;
	}

	return ret;
}

inline struct sk_buff *ieee80211_disauth_skb( struct ieee80211_network *beacon,
		struct ieee80211_device *ieee, u16 asRsn)
{
	struct sk_buff *skb;
	struct ieee80211_disauth *disauth;
#ifdef USB_USE_ALIGNMENT
        u32 Tmpaddr=0;
        int alignment=0;
	int len = sizeof(struct ieee80211_disauth) + ieee->tx_headroom + USB_512B_ALIGNMENT_SIZE;
#else
	int len = sizeof(struct ieee80211_disauth) + ieee->tx_headroom;

#endif
	skb = dev_alloc_skb(len);
	if (!skb) {
		return NULL;
	}

#ifdef USB_USE_ALIGNMENT
        Tmpaddr = (u32)skb->data;
        alignment = Tmpaddr & 0x1ff;
        skb_reserve(skb,(USB_512B_ALIGNMENT_SIZE - alignment));
#endif
	skb_reserve(skb, ieee->tx_headroom);
	
	disauth = (struct ieee80211_disauth *) skb_put(skb,sizeof(struct ieee80211_disauth));
	disauth->header.frame_ctl = cpu_to_le16(IEEE80211_STYPE_DEAUTH);
	disauth->header.duration_id = 0;
	
	memcpy(disauth->header.addr1, beacon->bssid, ETH_ALEN);
	memcpy(disauth->header.addr2, ieee->dev->dev_addr, ETH_ALEN);
	memcpy(disauth->header.addr3, beacon->bssid, ETH_ALEN);
	
	disauth->reason = cpu_to_le16(asRsn);
	return skb;
}

inline struct sk_buff *ieee80211_disassociate_skb( struct ieee80211_network *beacon,
		struct ieee80211_device *ieee, u16 asRsn)
{
	struct sk_buff *skb;
	struct ieee80211_disassoc *disass;
#ifdef USB_USE_ALIGNMENT
        u32 Tmpaddr=0;
        int alignment=0;
	int len = sizeof(struct ieee80211_disassoc) + ieee->tx_headroom + USB_512B_ALIGNMENT_SIZE;
#else
	int len = sizeof(struct ieee80211_disassoc) + ieee->tx_headroom;
#endif
	skb = dev_alloc_skb(len);

	if (!skb) {
		return NULL;
	}
	
#ifdef USB_USE_ALIGNMENT
        Tmpaddr = (u32)skb->data;
        alignment = Tmpaddr & 0x1ff;
        skb_reserve(skb,(USB_512B_ALIGNMENT_SIZE - alignment));
#endif
	skb_reserve(skb, ieee->tx_headroom);

	disass = (struct ieee80211_disassoc *) skb_put(skb,sizeof(struct ieee80211_disassoc));
	disass->header.frame_ctl = cpu_to_le16(IEEE80211_STYPE_DISASSOC);
	disass->header.duration_id = 0;
	
	memcpy(disass->header.addr1, beacon->bssid, ETH_ALEN);
	memcpy(disass->header.addr2, ieee->dev->dev_addr, ETH_ALEN);
	memcpy(disass->header.addr3, beacon->bssid, ETH_ALEN);
	
	disass->reason = cpu_to_le16(asRsn);
	return skb;
}

void SendDisassociation(struct ieee80211_device *ieee, bool deauth, u16 asRsn)
{
	struct ieee80211_network *beacon = &ieee->current_network;
	struct sk_buff *skb;

	if(deauth) {
		skb = ieee80211_disauth_skb(beacon,ieee,asRsn);
	} else {
		skb = ieee80211_disassociate_skb(beacon,ieee,asRsn);
	}

	if (skb){
		softmac_mgmt_xmit(skb, ieee);
	}
}

u8 ieee80211_ap_sec_type(struct ieee80211_device *ieee)
{
	static u8 ccmp_ie[4] = {0x00,0x50,0xf2,0x04};
	static u8 ccmp_rsn_ie[4] = {0x00, 0x0f, 0xac, 0x04};
	int wpa_ie_len= ieee->wpa_ie_len;
	struct ieee80211_crypt_data* crypt;
	int encrypt;

#ifdef _RTL8192_EXT_PATCH_
	crypt = ieee->sta_crypt[ieee->tx_keyidx];
#else
	crypt = ieee->crypt[ieee->tx_keyidx];
#endif
	encrypt = (ieee->current_network.capability & WLAN_CAPABILITY_PRIVACY) ||\
		  (ieee->host_encrypt && crypt && crypt->ops && \
		   (0 == strcmp(crypt->ops->name,"WEP")));

	/* simply judge  */
	if(encrypt && (wpa_ie_len == 0)) {
		return SEC_ALG_WEP;
	} else if((wpa_ie_len != 0)) {
		if (((ieee->wpa_ie[0] == 0xdd) && (!memcmp(&(ieee->wpa_ie[14]),ccmp_ie,4))) || 
				((ieee->wpa_ie[0] == 0x30) && (!memcmp(&ieee->wpa_ie[10],ccmp_rsn_ie, 4))))
			return SEC_ALG_CCMP;
		else
			return SEC_ALG_TKIP;
	} else {
		return SEC_ALG_NONE;
	}
}

#ifdef _RTL8192_EXT_PATCH_
int ieee80211_wpa_supplicant_ioctl(struct ieee80211_device *ieee, struct iw_point *p, u8 is_mesh)
#else
int ieee80211_wpa_supplicant_ioctl(struct ieee80211_device *ieee, struct iw_point *p)
#endif
{
	struct ieee_param *param;
	int ret=0;

	down(&ieee->wx_sem);

	if (p->length < sizeof(struct ieee_param) || !p->pointer){
		ret = -EINVAL;
		goto out;
	}
	
	param = (struct ieee_param *)kmalloc(p->length, GFP_KERNEL);
	if (param == NULL){
		ret = -ENOMEM;
		goto out;
	}
	if (copy_from_user(param, p->pointer, p->length)) {
		kfree(param);
		ret = -EFAULT;
		goto out;
	}

	switch (param->cmd) {

	case IEEE_CMD_SET_WPA_PARAM:
		ret = ieee80211_wpa_set_param(ieee, param->u.wpa_param.name,
					param->u.wpa_param.value);
		break;

	case IEEE_CMD_SET_WPA_IE:
		ret = ieee80211_wpa_set_wpa_ie(ieee, param, p->length);
		break;

	case IEEE_CMD_SET_ENCRYPTION:
#ifdef _RTL8192_EXT_PATCH_
		ret = ieee80211_wpa_set_encryption(ieee, param, p->length, is_mesh);
#else
		ret = ieee80211_wpa_set_encryption(ieee, param, p->length);
#endif
		break;

	case IEEE_CMD_MLME:
		ret = ieee80211_wpa_mlme(ieee, param->u.mlme.command,
				   param->u.mlme.reason_code);
		break;

	default:
		printk("Unknown WPA supplicant request: %d\n",param->cmd);
		ret = -EOPNOTSUPP;
		break;
	}

	if (ret == 0 && copy_to_user(p->pointer, param, p->length))
		ret = -EFAULT;

	kfree(param);
out:
	up(&ieee->wx_sem);
	
	return ret;
}

void notify_wx_assoc_event(struct ieee80211_device *ieee)
{
	union iwreq_data wrqu;
	wrqu.ap_addr.sa_family = ARPHRD_ETHER;
	if (ieee->state == IEEE80211_LINKED)
		memcpy(wrqu.ap_addr.sa_data, ieee->current_network.bssid, ETH_ALEN);
	else
		memset(wrqu.ap_addr.sa_data, 0, ETH_ALEN);
	wireless_send_event(ieee->dev, SIOCGIWAP, &wrqu, NULL);
}

#ifndef BUILT_IN_IEEE80211
EXPORT_SYMBOL_RSL(ieee80211_get_beacon);
EXPORT_SYMBOL_RSL(ieee80211_wake_queue);
EXPORT_SYMBOL_RSL(ieee80211_stop_queue);
EXPORT_SYMBOL_RSL(ieee80211_reset_queue);
EXPORT_SYMBOL_RSL(ieee80211_softmac_stop_protocol);
EXPORT_SYMBOL_RSL(ieee80211_softmac_start_protocol);
EXPORT_SYMBOL_RSL(ieee80211_is_shortslot);
EXPORT_SYMBOL_RSL(ieee80211_is_54g);
EXPORT_SYMBOL_RSL(ieee80211_wpa_supplicant_ioctl);
EXPORT_SYMBOL_RSL(ieee80211_ps_tx_ack);
EXPORT_SYMBOL_RSL(ieee80211_softmac_xmit);
EXPORT_SYMBOL_RSL(ieee80211_stop_send_beacons);
EXPORT_SYMBOL_RSL(notify_wx_assoc_event);
EXPORT_SYMBOL_RSL(SendDisassociation);
EXPORT_SYMBOL_RSL(ieee80211_disassociate);
EXPORT_SYMBOL_RSL(ieee80211_start_send_beacons);
EXPORT_SYMBOL_RSL(ieee80211_stop_scan);
EXPORT_SYMBOL_RSL(ieee80211_send_probe_requests);
EXPORT_SYMBOL_RSL(ieee80211_softmac_scan_syncro);
EXPORT_SYMBOL_RSL(ieee80211_start_scan_syncro);
EXPORT_SYMBOL_RSL(ieee80211_sta_ps_send_null_frame);
EXPORT_SYMBOL_RSL(ieee80211_sta_ps_send_pspoll_frame);
EXPORT_SYMBOL_RSL(ieee80211_sta_wakeup);
EXPORT_SYMBOL_RSL(ieee80211_ap_sec_type);
#ifdef _RTL8192_EXT_PATCH_
EXPORT_SYMBOL_RSL(ieee80211_MFIE_rate_len);
EXPORT_SYMBOL_RSL(ieee80211_MFIE_Brate);
EXPORT_SYMBOL_RSL(ieee80211_MFIE_Grate);
EXPORT_SYMBOL_RSL(ieee80211_WMM_Info);
EXPORT_SYMBOL_RSL(ieee80211_TURBO_Info);
EXPORT_SYMBOL_RSL(ieee80211_ext_probe_resp_by_net);
EXPORT_SYMBOL_RSL(softmac_mgmt_xmit);
EXPORT_SYMBOL_RSL(ieee80211_start_scan);
EXPORT_SYMBOL_RSL(ieee80211_ext_send_11s_beacon);
EXPORT_SYMBOL_RSL(ieee80211_associate_step1);
EXPORT_SYMBOL_RSL(ieee80211_stop_protocol);
EXPORT_SYMBOL_RSL(ieee80211_start_protocol);
#endif 
#endif
