/*
 * Copyright © 2005 Atheros Communications, Inc., All Rights Reserved
 *
 * iwath.c -- Utility to display information from USB Wlan Driver
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/ioctl.h>
#include <linux/wireless.h>
#include <linux/if_ether.h>
#include <linuxioctl.h>

#ifdef __GNU__
static const char athId[] __attribute__ ((unused)) = "$Id: //depot/sw/branches/1.3_USB_LINUX_port/apps/iwath.c#1 $";
#endif /* __GNUC__ */

#define LISTBUF_SIZE    2046
static const char *iwath_version = "iwath v0.1 - Copyright (c) 2005, Atheros Communications";

char *ifname = "ath0";
int network_list = 0;
int device_info = 0;
int bssid_list = 0;

static void print_supported_networks (void *buf);
static void print_device_info (void *buf);
static int drv_get_data(unsigned int command, void *buf, int bufLen);
static void print_bssid_list (void *buf, int length);

static void usage (int option)
{
    printf ("%s\n", iwath_version);
    if (option) {
        printf ("One of [-nbd] options should be used\n");
    }
    printf( "usage:\n"
            "   iwath [-hv] <-nbd> -i<ifname>\n"
            "options:\n"
            "   -n = list netowork types supported\n"
            "   -d = list driver information\n"
            "   -b = list atheros BSS\n"
            "   -v = show version\n"
            "   -h = show this help text\n"
            "\n");
}

int main (int argc, char *argv[])
{ 
    int option;

    for (;;) {
        option = getopt(argc, argv, "i:ndbhv");
        if (option < 0) {
            break;
        }
        switch (option) {
        case 'i' :
            ifname = optarg;
            break;
        case 'n' :
            network_list = 1;
            break;
        case 'd' :
            device_info = 1;
            break;
        case 'b' :
            bssid_list = 1;
            break;
        case 'v' :
            printf ("%s\n", iwath_version);
            return (0);
        case 'h' :
        default:
            usage (0);
            return (-1);
        }
    }

    if (!(network_list | device_info | bssid_list)) {
        usage(1);
        return (-1);
    }
    char *listbuf = malloc (LISTBUF_SIZE);
    if (listbuf == NULL) {
        printf ("Not enough memory\n");
        exit (-1);
    }
    
    printf ("\n");
    if (network_list &&
            (drv_get_data(ATHUSB_IOCTL_GET_NETSUPPORT,listbuf,LISTBUF_SIZE)>=0))
    {
        print_supported_networks(listbuf);
    }

    if (device_info &&
            (drv_get_data(ATHUSB_IOCTL_GET_DEVICE_INFO,listbuf,LISTBUF_SIZE)>0))
    {
        print_device_info(listbuf);
    }

    if (bssid_list) {
        int length;
        length = drv_get_data(ATHUSB_IOCTL_GET_ATH_LIST,listbuf,LISTBUF_SIZE);
        if (length >= 0) {
            print_bssid_list(listbuf, length);
        }
    }
    return (0);
}

static void print_supported_networks (void *buf)
{
    int count;
    PATH_802_11_NETWORK_TYPE_LIST pList = (PATH_802_11_NETWORK_TYPE_LIST)buf;

    printf ("Network Types Supported\n");
    for (count = 0; count < pList->NumberOfItems; count++) {
        if (count) {
            printf (", ");
        }
        switch (pList->NetworkType[count]) {
        case Ath802_11FH:
            printf ("802_11FH");
            break;
        case Ath802_11DS:
            printf ("802_11DS");
            break;
        case Ath802_11OFDM5:
            printf ("802_11OFDM5");
            break;
        case Ath802_11OFDM24:
            printf ("802_11OFDM24");
            break;
        default:
            printf ("Unknown");
        }
    }
    printf ("\n\n");
}

static void print_device_info (void *buf)
{
    PAtheros5000_DeviceInfo pE = (PAtheros5000_DeviceInfo)pE;

    printf ("Device Information :\n");
    printf ("  VendorID             = 0x%04x\n", pE->AthVendorID);
    printf ("  DeviceID             = 0x%04x\n", pE->AthDeviceID);
    printf ("  SubVendorID          = 0x%04x\n", pE->AthSubVendorID);
    printf ("  SubVendor DeviceID   = 0x%04x\n", pE->AthSubVendor_DeviceID);
    printf ("  MAC Revision         = 0x%04x\n", pE->AthMacRev);
    printf ("  PHY Revision         = 0x%04x\n", pE->AthPhyRev);
    printf ("  Analog 5GHz Revision = 0x%04x\n", pE->AthAnalog5GhzRev);
    printf ("  Analog 2GHz Revision = 0x%04x\n", pE->AthAnalog2GhzRev);
    printf ("  Wireless Modes       = 0x%08lx\n", pE->AthHwCapWirelessModes);
    printf ("  BSS Wireless Modes   = 0x%08lx\n", pE->AthInfraWirelessModes);
    printf ("  IBSS Wireless Modes  = 0x%08lx\n", pE->AthAdhocWirelessModes);
    printf ("  NIC SKU              = 0x%08lx\n\n", pE->AthNicSKU);
}

static void print_bssid_list (void *buf, int length)
{
    int             count = (length / sizeof (Atheros5000_BssInfo));
    int             i;

    if (count) {
        printf ("%s\tScan completed:\n", ifname);
    } else {
        printf ("%s\tNo scan results\n", ifname);
    }
    for (i = 0; i < count; i++) {
        PAtheros5000_BssInfo pInfo = (PAtheros5000_BssInfo)buf+i;
        unsigned char   *bssid = pInfo->Bssid;
        unsigned int   channel=(pInfo->Channel & 0xFFFF);
        int j;
        printf("\tCell %02d - Address : ", i+1);
        printf("%02X:%02X:%02X:%02X:%02X:%02X\n",bssid[0], bssid[1], bssid[2],
                bssid[3], bssid[4], bssid[5]);
        printf("\t\tESSID:\"%s\"\n", pInfo->Ssid);
        printf("\t\tMode:%s\n", (pInfo->NetworkType ? "Ad-Hoc" : "Managed"));
        printf("\t\tFrequency:%d.%dGHz\n", channel/1000,channel%1000);
        printf("\t\tEncryption key:%s\n", (pInfo->Privacy) ? "on" : "off");
        for (j = 1; j < 8; j++) {
            unsigned int bitrate = pInfo->SupportedRates[j] & 0x7F;
            if (bitrate) {
//                bitrate = (bitrate * 1000000) / 2;
                printf("\t\tBit Rate:%d.%dMb/s\n", bitrate/2, (bitrate%2)*5);
            }
        }
        printf("\t\tRssi:%ld\n", pInfo->Rssi);
        printf("\t\tVersion:%ld\n", pInfo->version);
        printf("\t\txrCapable:%s\n", pInfo->xrCapable ? "true" : "false");
        printf("\t\txrCurrent:%s\n", pInfo->xrCurrent ? "true" : "false");
        printf("\t\tsuperCap:%ld\n", pInfo->superCap);
        printf("\t\tBeaconInterval:%ld\n", pInfo->BeaconInterval);
        printf("\t\tAtimWindow:%ld\n", pInfo->AtimWindow);
    }
}

static int drv_get_data(unsigned int command, void *buf, int bufLen)
{
    int             status  = 0;
    int             sd;
	struct iwreq    wrq;

	if ((sd = socket(PF_INET, SOCK_DGRAM, 0)) < 0) {
        perror("socket");
        return (-1);
    }

    memset(&wrq, 0, sizeof(struct iwreq));
	wrq.u.mode = command;
    strncpy(wrq.ifr_name, ifname, IFNAMSIZ);
    wrq.u.data.pointer = buf;
    wrq.u.data.length = bufLen;

    status = ioctl(sd, command, &wrq);
    close(sd);

    if (status < 0) {
        perror(ifname);
        return status;
    }

    return (wrq.u.data.length);
}

