/*
 * $Id: //depot/sw/branches/1.3_USB_LINUX_port/src/USB/include/ecosdrv.h#3 $
 *
 * Copyright  2000-2002 Atheros Communications, Inc.,  All Rights Reserved.
 *
 * ECOS specific function wrappers.
 */

#include "wlantype.h"

#ifndef __ECOSDRV_H__
#define __ECOSDRV_H__

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

#include <cyg/infra/cyg_type.h>
#include <cyg/infra/cyg_trac.h>
#include <cyg/infra/diag.h>
#include <cyg/hal/drv_api.h>
#include <cyg/hal/hal_arch.h>
#include <cyg/hal/hal_io.h>
#include <cyg/hal/hal_cache.h>
#include <cyg/error/codes.h>
#include <assert.h>


/* 
 * Types mapping to ECOS types
 */
typedef cyg_handle_t            A_HANDLE;
typedef cyg_interrupt           A_INTERRUPT_OBJECT ;
typedef cyg_mutex_t             A_SEM_TYPE;
typedef cyg_sem_t               A_CNTSEM_TYPE;

typedef struct {
    cyg_handle_t alarm_hdl;
    cyg_alarm    alarm_obj;
} A_TIMER;             


#define LOCAL                   static
#define INLINE                  __inline__


/* 
 *  Memory routines are included only if the ECOS build includes them in which case
 *  CYGPKG_MEMALLOC_MALLOC_ALLOCATORS will be defined
 */
#define A_DRIVER_MALLOC(a)              (malloc(a))
#define A_DRIVER_FREE(a, b)             (free(a))
#define A_MEM_SET(addr, value, size)    memset((char *)(addr), (int)(value), (int)(size))
#define A_MEM_ZERO(addr, size)          memset((char *)(addr), (int)0, (int)(size))
#define A_MEM_CPY(dst, src, size)       memcpy((char *)(dst), (char *)(src), (int)(size))

#if !defined(A_DRIVER_BCOPY)
#define A_DRIVER_BCOPY(src, dst, len)    A_MEM_CPY(dst, src, len)
#endif

/*
 * Words (16b) should be aligned correctly on all platforms, so it
 * can be used even on platforms (mips) with strict alignment
 * requirements.
 */
#define A_MACADDR_COPY(from, to)              \
    do {                                      \
        (to)->words[0] = (from)->words[0];    \
        (to)->words[1] = (from)->words[1];    \
        (to)->words[2] = (from)->words[2];    \
    } while (0)

#define A_MACADDR_COMP(m1, m2)                \
    ((((m1)->words[2] == (m2)->words[2]) &&   \
      ((m1)->words[1] == (m2)->words[1]) &&   \
      ((m1)->words[0] == (m2)->words[0])) == 0)

/* Satisfy overly-picky compilers */
#define CONTRIVED_REFERENCE(target) \
    if (0) {                   \
        target;                \
    }


/*
 * Get the number of millisecond ticks since the system was started.
 */
/* XXX */
static INLINE A_UINT32
A_MS_TICKGET(void)
{
    /* XXX: TBD: Call cyg_clock_get_resolution(cyg_real_time_clock)
       to ensure we're returning millisecond ticks and not just 
       ticks */
    return(cyg_current_time());
}

/* XXX ? */
#define ATH_OSPKTREF_DUP(_pOrig, _pDup)
#define ATH_OSPKTREF_FREE(_pDesc)

#define ATH_OSBUFREF_DUP(_pOrig, _pDup)     do {            \
    if ((_pOrig)->pOSDescPtr) {                             \
        (_pOrig)->pOSDescPtr->clRefCnt++;                   \
        (_pDup)->pOSDescPtr = (_pOrig)->pOSDescPtr;         \
    }                                                       \
} while (0)

/*
 * Define isrPrintf() for printing from ISR context. Just do the diag_printf which 
 * prints on the serial port now, could be made to use CYG_TRACE0() calls when a 
 * proper stratgey for the reporting the logs has been finalized. CYG_TRACE_PRINT()
 * will report the contents of the Trace buffer then 
 */
#define isrPrintf                       diag_printf

#if defined(DEBUG)
#define uiPrintf isrPrintf
#else
static __inline void emptyfn(const char * format, ...) { }
#define uiPrintf 1 ? (void)0 : emptyfn
#endif

// This should be placed elsewhere
#define rxPrintf uiPrintf
#define rxPrintf uiPrintf

/* Timer macros - XXX slightly different from windows macros */
#define A_INIT_TIMER(pd, pt, pf, par)               \
    cyg_alarm_create((pd)->rtcCounterHandle, (pf),  \
                     (cyg_addrword_t) par,          \
                     &(pt)->alarm_hdl,              \
                     &(pt)->alarm_obj) 

#define A_TIMEOUT(pt, per, rep)                       \
     cyg_alarm_initialize((pt)->alarm_hdl,            \
                          cyg_current_time() + (per), \
                          (rep?(per):0))

#define A_UNTIMEOUT(pt)          cyg_alarm_disable((pt)->alarm_hdl)            
#define A_DELETE_TIMER(pt)       cyg_alarm_delete((pt)->alarm_hdl)

// XXX: is HAL_DELAY the right one? Might use cyg_thread_delay() */
#define udelay                      HAL_DELAY_US
#define FIELD_OFFSET(type, field)   ((int)(&((type *)0)->field))


/* a_assert saves code image space by not stringifying #_bool */
#define a_assert( __bool )                                                \
    do {                                                                  \
        if (0 == (__bool)) {                                              \
            diag_printf( "assertion failed: at %s:%d\n",                  \
                         __FILE__, __LINE__);                             \
        for (;;);                                                         \
        }                                                                 \
    } while(0)

#if defined(NO_ASSERT)
#define ASSERT(x) 
#else
#define ASSERT(x) a_assert(x)
#endif

#define A_STRCPY    strcpy
#define A_STRNCPY   strncpy
#define A_STRLEN    strlen
#define A_STRCMP    strcmp
#define A_STRNCMP   strncmp
#define A_RAND      rand

/*
 * Locking macros
 */
#define CREATE_LOCKED   1
#define CREATE_UNLOCKED 0
#define WAIT_FOREVER    0   /* XXX: not implemented */ 
    
#ifdef DEBUG
#define A_SEM_INIT(sem_val, sem_type, make_locked)      \
     do {                                               \
       cyg_mutex_init(&sem_val);                         \
       if ((make_locked == CREATE_LOCKED) &&            \
	   (cyg_mutex_trylock(&sem_val) == 0)) {         \
	    uiPrintf("Cound not allocate lock.");       \
            ASSERT(0);                                  \
	   }                                            \
      } while (0)               
#else
#define A_SEM_INIT(sem_val, sem_type, make_locked)      \
     do {                                               \
       cyg_mutex_init(&sem_val);                         \
       if ((make_locked == CREATE_LOCKED) &&            \
	   (cyg_mutex_trylock(&sem_val) == 0))           \
	    uiPrintf("Cound not allocate lock.");       \
      } while (0)               
#endif /* not DEBUG */


#define A_SEM_VALID(_sem)           (1)
/* XXX timeout discarded for now */
#define A_SEM_LOCK(sem, timeout)    do {                                     \
                                         ASSERT(timeout == WAIT_FOREVER);    \
                                         A_TASK_ASSERT_UNLOCKED();           \
                                         cyg_mutex_lock(&sem);               \
                                    } while(0)
#define A_SEM_TRYLOCK(sem)          cyg_mutex_trylock(&sem)
#define A_SEM_UNLOCK(sem)           cyg_mutex_unlock(&sem)
#define A_SEM_DELETE(sem)           cyg_mutex_destroy(&sem)

#define A_SIB_TAB_LOCK(sibTab)      A_SEM_LOCK(sibTab->sibTableSem, WAIT_FOREVER)
#define A_SIB_TAB_UNLOCK(sibTab)    A_SEM_UNLOCK(sibTab->sibTableSem)
#define A_SIB_ENTRY_LOCK(sib)       A_SEM_LOCK(sib->sibLock, WAIT_FOREVER)
#define A_SIB_ENTRY_UNLOCK(sib)     A_SEM_UNLOCK(sib->sibLock)

#define A_TASK_ASSERT_LOCKED()      ASSERT(cyg_scheduler_read_lock() >  0)
#define A_TASK_ASSERT_UNLOCKED()    ASSERT(cyg_scheduler_read_lock() <= 0)
#define A_TASK_LOCK()               cyg_scheduler_lock()
#define A_TASK_UNLOCK()             do {                       \
                                       A_TASK_ASSERT_LOCKED(); \
                                       cyg_scheduler_unlock(); \
                                    } while(0)
#define A_TASK_YIELD()              cyg_thread_yield()

#define A_TASK_STACKSZ              4096

#define A_TASK_SET_HIGH_PRIORITY()  cyg_thread_set_priority( \
                                            cyg_thread_self(), 0)
          
#define A_TASK_SET_LOW_PRIORITY()  cyg_thread_set_priority( \
                                            cyg_thread_self(), 1)


/* Start a task/thread that runs the function "name" with parameter "param". */

#define A_TASK_CREATE_STACKSZ(name, param, stacksz)     \
do {                                                    \
static char name##stack[stacksz];                       \
static cyg_handle_t name##handle;                       \
static cyg_thread   name##thread;                       \
    cyg_thread_create(1,                                \
		      (cyg_thread_entry_t *)name,               \
		      (CYG_ADDRWORD)(param),                    \
		      #name,                                    \
		      name##stack,                              \
              stacksz,                                  \
		      &name##handle,                            \
		      &name##thread);                           \
                                                        \
    cyg_thread_resume(name##handle);                    \
} while (0)

#define A_TASK_CREATE(name, param)                      \
     A_TASK_CREATE_STACKSZ(name, param, A_TASK_STACKSZ)




#define A_CNTSEM_INIT(sem_val, init_val)                \
     do {                                               \
       cyg_semaphore_init(&sem_val, init_val);          \
     } while (0)               

#define A_CNTSEM_POST(sem_val)                          \
     do {                                               \
       cyg_semaphore_post(&sem_val);                    \
     } while (0)

#define A_CNTSEM_WAIT(sem_val)                          \
     do {                                               \
       A_TASK_ASSERT_UNLOCKED();                        \
       cyg_semaphore_wait(&sem_val);                    \
     } while (0)

#define A_CNTSEM_TRYWAIT(sem_val)                       \
     cyg_semaphore_trywait(&sem_val)

#define A_CNTSEM_PEEK(sem_val, val)                     \
     cyg_semaphore_peek(&sem_val, val)

#define A_CNTSEM_TIMED_WAIT(sem_val, timeout)           \
     do {                                               \
       A_TASK_ASSERT_UNLOCKED();                        \
       cyg_semaphore_timed_wait(&sem_val, (timeout));   \
     } while (0)


/* 
 * Interrupt Handling Macros 
 */


/* Enable & Disable all Interrupts */
#define  A_DISABLE_ALL_INTERRUPTS()                     cyg_interrupt_disable()
#define  A_ENABLE_ALL_INTERRUPTS()                      cyg_interrupt_enable()

/* Enable & Disable the specied interrupts */
#define  A_DISABLE_INTERRUPT(_vector)                   cyg_interrupt_mask(_vector)
#define  A_ENABLE_INTERRUPT(_vector)                    cyg_interrupt_unmask(_vector)
#define  A_ACK_INTERRUPT(_vector)                       cyg_interrupt_acknowledge(_vector)

/* ISR Management Macros */
#define  A_CREATE_INTERRUPT(_vector, _context, _isr, _dsr, _handle, _intr)            \
                    cyg_interrupt_create(                                             \
                                         _vector,   /* Vector                    */   \
                                               0,   /* Queue priority            */   \
                        (cyg_addrword_t)_context,   /* Data pointer              */   \
                               (cyg_ISR_t *)_isr,   /* Interrupt Service Routine */   \
                               (cyg_DSR_t *)_dsr,   /* Deferred Service Routine  */   \
                                         _handle,   /* returned handle           */   \
                                            _intr   /* interrupt                 */   \
                    )


#define  A_DELETE_INTERRUPT(_handle)                    cyg_interrupt_delete(_handle)
#define  A_ATTACH_INTERRUPT(_handle)                    cyg_interrupt_attach(_handle)
#define  A_DETACH_INTERRUPT(_handle)                    cyg_interrupt_detach(_handle)


/*
 * Cache flushing and virtual address translation support.  vxWorks macros
 * handle things cleanly but with some overhead, so optimize for supported
 * platforms.
 *   - A_DATA_() routines are for frame data buffers.
 *   - A_DESC_() macros are for DMA descriptors.  Currently
 *     hardware descriptors are cached as they share memory
 *     with the software frame description.
 *
 * SWCOHERENCY   - descriptors and data are cached
 * SWCOHERENCYWB - SWCOHERENCY with a write back cache
 *
 * Systems with cache line sizes larger than 16 bytes need
 * to access the descriptor uncached as cache flushing is
 * not practical with the DMA control and status on the same
 * cache line.
 */
#define ROUND_UP(x, align)      (((int) (x) + (align - 1)) & ~(align - 1))
#define ROUND_DOWN(x, align)    ((int)(x) & ~(align - 1))
#define CACHE_LINE_SIZE         16

/* ECOS definition to flush the data Cache */
#define ECOS_DATA_CACHE_INVAL(addr, len) \
        HAL_DCACHE_INVALIDATE((void *)ROUND_DOWN(addr, CACHE_LINE_SIZE), \
            ROUND_UP(len+((long)(addr) & (CACHE_LINE_SIZE-1)), CACHE_LINE_SIZE))

/* ECOS call to flush Write Buffer */
#define ECOS_WB_FLUSH() \
    asm volatile ("sync")

#if defined(AR531X) 
/*
 * AR531X uses a write back cache, with it's own lower overhead
 * cache invalidation routine.  V != P, but upper bits are
 * ignored by the DMA bus so a null V2P/P2V macros are used.
 */
#define SWCOHERENCY
#define A_DATA_CACHE_INVAL(addr, len)   ECOS_DATA_CACHE_INVAL(addr, len)
#define A_DATA_CACHE_FLUSH(addr, len)

#define A_DATA_V2P(addr)                ((A_UINT32)(addr))
#define A_DATA_P2V(addr)                (addr)

#define A_DESC_CACHE_INVAL(addr)        A_DATA_CACHE_INVAL(((void *)(&(addr)->hw.word[0])), sizeof(addr->hw))
#define A_DESC_CACHE_FLUSH(addr)
#define A_DESC_V2P(addr)                ((A_UINT32)(addr))
#define A_DESC_P2V(addr)                (addr)
#define A_PIPEFLUSH()                   ECOS_WB_FLUSH()

#elif defined (AR5523)
#define SWCOHERENCY
#define A_DATA_CACHE_INVAL(addr, len)   ECOS_DATA_CACHE_INVAL(addr, len)
#define A_DATA_CACHE_FLUSH(addr, len)

#define A_DATA_V2P(addr)                ((A_UINT32)(addr))
#define A_DATA_P2V(addr)                ((A_UINT32)(addr))

#define A_DESC_CACHE_INVAL(addr)        A_DATA_CACHE_INVAL(((void *)(&(addr)->hw.word[0])), sizeof(addr->hw))
#define A_DESC_CACHE_FLUSH(addr)
#define A_DESC_V2P(addr)                ((A_UINT32)(addr))
#define A_DESC_P2V(addr)                ((A_UINT32)(addr))
#define A_PIPEFLUSH()                   ECOS_WB_FLUSH()

#elif defined(CACHE_PROTO_COPYBACK) || (USER_D_CACHE_MODE & CACHE_COPYBACK)

#define SWCOHERENCY
#define SWCOHERENCYWB
#define A_DATA_CACHE_INVAL(addr, len)   ECOS_DATA_CACHE_INVAL(addr, len)
#define A_DATA_CACHE_FLUSH(addr, len)   
#define A_DATA_V2P(addr)                
#define A_DATA_P2V(addr)                
#define A_DESC_CACHE_INVAL(addr)        A_DATA_CACHE_INVAL(((void *)(&(addr)->hw.word[0])), sizeof(addr->hw))
#define A_DESC_CACHE_FLUSH(addr)        
#define A_DESC_V2P(addr)                A_DATA_V2P(addr)
#define A_DESC_P2V(addr)                A_DATA_P2V(addr)
#define A_PIPEFLUSH()                   ECOS_WB_FLUSH()

#else

/*
 * Sample code to use cacheDmaAlloc() which is uncached on non-coherent
 * platforms.  This is the default case for non-coherent copyback
 * caches.  This code has not been extensively tested, but should
 * be safe on most platforms.
 */
#define VX_CACHE_DMA_MALLOC
#define SWCOHERENCY
#define A_DATA_CACHE_INVAL(addr, len)   CACHE_DRV_INVAL(&cacheDmaFuncs, addr, len)
#define A_DATA_CACHE_FLUSH(addr, len)   CACHE_DRV_FLUSH(&cacheDmaFuncs, addr, len)
#define A_DATA_V2P(addr)                ((A_UINT32)CACHE_DRV_VIRT_TO_PHYS(&cacheDmaFuncs, addr))
#define A_DATA_P2V(addr)                CACHE_DRV_PHYS_TO_VIRT(&cacheDmaFuncs, addr)
#define A_DESC_CACHE_INVAL(addr)        A_DATA_CACHE_INVAL(addr, 32)
#define A_DESC_CACHE_FLUSH(addr)        A_DATA_CACHE_FLUSH(addr, 32)
#define A_DESC_V2P(addr)                A_DATA_V2P(addr)
#define A_DESC_P2V(addr)                A_DATA_P2V(addr)
#define A_PIPEFLUSH()                   CACHE_PIPE_FLUSH()

#endif /* PLATFORM */

#if defined(ECOS_NOTDONE)

A_CHAR * basename(A_CHAR *filename);

#define ECOS_NOTDONE_XXX isrPrintf("! %-30s: %-20s:%-5d ECOS_NOTDONE\n", \
				  __FUNCTION__, basename(__FILE__), __LINE__)
#define ECOS_NOTDONE_XXX_C(x) isrPrintf("! %-30s: %-20s:%-5d ECOS_NOTDONE: %s\n", \
				  __FUNCTION__, basename(__FILE__), __LINE__, x)
#else
#define ECOS_NOTDONE_XXX
#define ECOS_NOTDONE_XXX_C
#endif

#ifdef __cplusplus
}
#endif

#endif /* not __ECOSDRV_H__ */

