/*
 * $Id: //depot/sw/branches/1.3_USB_LINUX_port/src/USB/include/linuxdefs.h#5 $
 *
 * Linux-specific declarations
 *
 * Copyright (c) 2000-2003 Atheros Communications, Inc., All Rights Reserved
 */

#ifndef _LINUXDEFS_H_
#define _LINUXDEFS_H_

#ifdef __cplusplus
extern "C" {
#endif

#define ROUND_UP(x, align)      (((int) (x) + (align - 1)) & ~(align - 1))
#define ROUND_DOWN(x, align)    ((int)(x) & ~(align - 1))

#if DBG
extern int asserts; // defined in wlanglobal.c
#undef ASSERT
#define oldAssert( exp ) \
    ((!(exp)) ? \
     (printk("Assertion failed %s,%s,%s,line=%d\n", \
        #exp,__FILE__,__FUNCTION__,__LINE__),FALSE) : TRUE)
#define ASSERT(x) if (asserts) oldAssert(x)
#else
#define ASSERT(x)
#endif // DBG

static __inline__ char *A_STRCPY(char *dest, const char *src)
{
    char *tmp = dest;

    while ((*dest++ = *src++) != '\0')
        /* nothing */;
    return tmp;
}

static __inline__ char *A_STRNCPY(char *dest, const char *src, int count)
{
    char *tmp = dest;

    while (count-- && (*dest++ = *src++) != '\0')
        /* nothing */;

    return tmp;
}

static __inline__ int A_STRLEN(const char *s)
{
    const char *sc;

    for (sc = s; *sc != '\0'; ++sc)
        /* nothing */;
    return sc - s;
}

static __inline__ int A_STRCMP(const char* cs, const char* ct)
{
    char __res;

    while (1) {
        if ((__res = *cs - *ct++) != 0 || !*cs++)
            break;
    }

    return __res;
}

static __inline__ int A_STRNCMP(const char* cs, const char* ct, int count)
{
    char __res = 0;

    while (count) {
        if ((__res = *cs - *ct++) != 0 || !*cs++)
            break;
        count--;
    }

    return __res;
}

#define A_POOL_TAG              'k5rA'

static __inline__ void * A_DRIVER_MALLOC(unsigned long int bufSize)
{
    PVOID       bufAddr;
#if (LINUX_VERSION_CODE > KERNEL_VERSION(2,6,0))
    bufAddr = kmalloc (bufSize, (in_interrupt() || irqs_disabled()) ? GFP_ATOMIC : GFP_KERNEL);
#else
    bufAddr = kmalloc (bufSize, (in_interrupt()) ? GFP_ATOMIC : GFP_KERNEL);
#endif
    return bufAddr;
}

static __inline__ void A_DRIVER_FREE(void *bufAddr, unsigned long int bufSize)
{
    kfree(bufAddr);
}

#define A_DRIVER_BCOPY(from, to, len)                     \
        (memcpy((void *)(to), (void *)(from), (len)))
#define A_BCOPY(from, to, len)          (memcpy(to, from, len))
#define A_BCOMP(s1, s2, len)            (memcmp(s1, s2, len))
#define A_MACADDR_COPY(from, to)                          \
        (memcpy((void *)&((to)->octets[0]),               \
        (void *)&((from)->octets[0]),WLAN_MAC_ADDR_SIZE))
#define A_MACADDR_COMP(m1, m2)                            \
        (memcmp((void *)&((m1)->octets[0]),                 \
        (void *)&((m2)->octets[0]),                       \
        WLAN_MAC_ADDR_SIZE))
#define A_MEM_ZERO(addr, len)           (memset((addr), 0, ((ULONG)len)))

/* Atomic Functions */
#define A_ATOMIC_INC(pAtomicVar)         atomic_inc(pAtomicVar)
#define A_ATOMIC_DEC(pAtomicVar)         atomic_dec(pAtomicVar)
#define A_ATOMIC_SET(pAtomicVar, Value)  atomic_set(pAtomicVar, Value)
#define A_ATOMIC_READ(pAtomicVar)        atomic_read(pAtomicVar)
#define A_ATOMIC_SUB(Value,pAtomicVar)   atomic_sub(Value, pAtomicVar)

/* Getting the current Interrupt level; Will be maintaining the
 * KeGetCurrentIrql instead of this finally */
#define athGetCurrentIrql(oldIrql)                      \
        do { oldIrql = (in_interrupt() ? DISPATCH_LEVEL : PASSIVE_LEVEL);}while(0)

/* Linux provides a neat way of getting some random data - use it over
 * Tick/time information */
static __inline__ unsigned long A_RAND(void) {
    unsigned long RandomValue;
    get_random_bytes (&RandomValue, sizeof(RandomValue));
    return (RandomValue);
}

/* Linux Event Implmentation */

#define EVENT_INITIALIZED       0
#define SIGNAL_BIT              0

/*
 * The default Linux wait/wake-up mechanism is not very safe to use directly.
 * This can result in a potential race condition.  So emulate the windows
 * style Event mechanisms
 */

/* We dont expect a race condition to occur during initialization, no need
 * to do any atomic operations */
static __inline__ VOID NdisInitializeEvent (PNDIS_EVENT pEvent)
{
    /* Clear signaled state */
    pEvent->SignalState = EVENT_INITIALIZED;
    /* Initialize wait queue head */
    init_waitqueue_head( &pEvent->NdisEvent);
}

/* Event is explicitly cleared using this function. */
static __inline__ VOID NdisResetEvent (PNDIS_EVENT pEvent)
{
    /* Clear Signaled state */
    clear_bit(SIGNAL_BIT, &pEvent->SignalState);
}

static __inline__ VOID NdisSetEvent (PNDIS_EVENT pEvent)
{
    /* We dont wake up the threads if the signal is set already ; assumption
     * is that no thread can wait on an event in a signaled state */
    if (test_and_set_bit(SIGNAL_BIT, &pEvent->SignalState) == 0) {
        /* Event is not set earlier - Wake up the threads */
        wake_up (&pEvent->NdisEvent);
    }
}

/* Wait for a event signalling with timeout, if the event is already set
 * return immediately. */
static __inline__ A_BOOL NdisWaitEvent (PNDIS_EVENT pEvent, A_UINT32 MsToWait)
{
    A_INT32 TimeInJiffies;
    DECLARE_WAITQUEUE(wait,current);
    BOOLEAN status = 0;

    /* Convert milliseconds to Jiffies */
    TimeInJiffies = (MsToWait * HZ) / 1000;

    if ((MsToWait == 0) || (TimeInJiffies < 0)) {
        /* Wait indefinitely */
        TimeInJiffies = MAX_SCHEDULE_TIMEOUT;
    } else if (TimeInJiffies == 0) {
        TimeInJiffies = 1;
    }

    /* Place ourselves in wait queue if signal not present / until timeout */
    add_wait_queue(&pEvent->NdisEvent,&wait);
    set_current_state(TASK_INTERRUPTIBLE);
    while (TimeInJiffies && 
            ((status=test_bit(SIGNAL_BIT,&pEvent->SignalState)) == 0))
    {
        /* schedule timeout returns the remaining time if woken up by somebody
         * or zero on timeout */
        TimeInJiffies = schedule_timeout(TimeInJiffies);
    }
    set_current_state(TASK_RUNNING);
    remove_wait_queue(&pEvent->NdisEvent, &wait);
    
    /* Do not clear the event - this should be done explicitely by ResetEvent */
    return (status);
}

/* Event functions are required for the work item to be implemented */
#define KeSetEvent(pEvent,Increment,Wait)    NdisSetEvent(pEvent);
#define KeClearEvent(pEvent)                 NdisResetEvent(pEvent);
#define KeInitializeEvent(pEvent,Type,State)    \
    do {                                        \
        NdisInitializeEvent(pEvent);            \
        if (State) {                            \
            NdisSetEvent(pEvent);               \
        }                                       \
    }while(0)

/* Crippled version of the windows function - If the time is set to zero,
 * we should return immediately just checking the condition, this is not
 * implemented yet */
static __inline__ int KeWaitForSingleObject(PVOID pObject, A_UINT32 WaitReason, A_UINT32 WaitMode, A_BOOL Alertable, PLARGE_INTEGER Timeout)
{
    A_UINT32 TimeInMs = 0;
    A_INT32 Status;
    if (Timeout == NULL) {
        TimeInMs = 0;
    } else if (Timeout->QuadPart < 0) {
        TimeInMs = -(Timeout->LowPart/10000);
    }
    Status = NdisWaitEvent(pObject, TimeInMs);

    if (Status == 0) {
        return STATUS_TIMEOUT;
    }
    return STATUS_SUCCESS;
}

/*
 * Workaround for the Microsoft compiler to allow undefining the uiPrintf
 * function, because it has a variable argument list.
 */
#if defined(DEBUG_PRINTF)
#define uiPrintf printk
#else
static __inline__ void drvPrintf(const char * format, ...) { }
#undef uiPrintf
#define uiPrintf 1 ? (void)0 : drvPrintf
#endif

/* Debug messages per component */
#define CSERV       "@0"
#define RX_PATH     "@1"
#define TX_PATH     "@2"
#define POWER_MGMT  "@3"
#define USER_REQ    "@4"
     
/* Log level */
#define L_VERBOSE  "1"      /* lots of debug messages     */
#define L_DEBUG    "2"      /* debugging output           */
#define L_NORMAL   "3"      /* informative                */
#define L_WARNING  "4"      /* non-critical errors        */
#define L_CRITICAL "5"      /* only critical errors       */
/* temporary workaround for Unix definition of FILE */

#define FILE    void

/* LOCAL keyword doesn't exist under VS C++ */
#define LOCAL   

/* Driver specific data types */
struct osDevInfo;
typedef struct osDevInfo OS_DEV_INFO;
typedef struct {
    int         initialized;    /* just need to track initialized or not */
    atomic_t    num;
    unsigned long flags;
    spinlock_t  spinLock;       /* used to protect the queue */
} A_SEM_TYPE;

/* Buffer management */

#undef DEBUG_CORRUPTION
#ifdef DEBUG_CORRUPTION
#define LARGE_BUFF_COOKIE               0xdeadbeef
#define LARGE_BUFF_COOKIE_SIZE          (sizeof(A_UINT32))
#define _THIS_BUFF_COOKIE(_p)           (*((A_UINT32 *)(((char *)(_p)) + LARGE_BUFF_SIZE - LARGE_BUFF_COOKIE_SIZE)))
#define A_CORRUPTION_BUG_CHECK(_myVirt) ASSERT_ALWAYS(_THIS_BUFF_COOKIE(_myVirt) == LARGE_BUFF_COOKIE)
#define A_CORRUPTION_BUG_INIT(_myVirt)  do { _THIS_BUFF_COOKIE(_myVirt) = LARGE_BUFF_COOKIE; } while (0)

#ifdef _DEBUG
#define ASSERT_ALWAYS                   ASSERT
#else
#define ASSERT_ALWAYS                   drvAssert
#pragma optimize("", off)
__inline void drvAssert(BOOLEAN cond)
{
    if (!cond) {
        int *x = 0;
        int i  = *x;
    }
}
#pragma optimize("", on)
#endif
#else
#define LARGE_BUFF_COOKIE_SIZE          0
#define A_CORRUPTION_BUG_CHECK(_myVirt)
#define A_CORRUPTION_BUG_INIT(_myVirt)
#endif

/*
 * ethernet 1500 + llc/snap 8 + mac 34 + qos 2
 *   + use a generous crypto allocation of 32
 *   + DRV_BUF_DESC of 8
 * round it all up to a cacheline multiple of 64
 */
#define LARGE_BUFF_SIZE         4096
#define SMALL_BUFF_SIZE         256     /* size of small buffers */
#define AR_BUF_SIZE             (4096 - sizeof(DRV_BUF_DESC) - LARGE_BUFF_COOKIE_SIZE)
#define AR_HEADER_SIZE          96

#define NUM_RX_BLOCKS           128     // Bufs reserved for rx queue
#define NUM_GENERIC_BLOCKS      46      // Bufs reserved for other things, mlme
                                        // Use 46 to accommodate 65500 byte ping (44 + 2 for margin)
#define NUM_LARGE_BUFF          (NUM_RX_BLOCKS + NUM_GENERIC_BLOCKS)
#define NUM_SMALL_BUFF          128     /* small buffer pool for headers and fragments */

#define NUM_TX_DESC             192     /* Atheros descriptors for tx queue, bumped up from 128 */
#define NUM_DESCS               (NUM_SMALL_BUFF + NUM_LARGE_BUFF + NUM_TX_DESC)
#define DESC_SIZE               sizeof(ATHEROS_DESC)

/* Power management modes specified by user */
#define POWERMGT_CAM            0     // Continuously active mode
#define POWERMGT_SLEEP          1     // aggressive sleep mode
#define POWERMGT_PERF           2     // aggressive performance mode
                                        
/* Power management events */
typedef enum pwrMgmtEventType {
    WAKE_UP,
    HIBERNATE,
    RADIO_ONLINE,
    AC_ONLINE,
    FORCE_SLEEP,
    FAKE_SLEEP
} PWR_MGMT_EVENT_TYPE;

/* Power management values for the HIBERNATE, RADIO_ONLINE, AC_ONLINE FORCE_ and FAKE_SLEEP */
typedef enum pwrMgmtValueType {
    ENABLE,
    ENABLED,
    DISABLE,
    DISABLED
} PWR_MGMT_VALUE_TYPE;

/* Power management values for WAKE_UP */
typedef enum pwrMgmtWakeupType {
    SET_PWR_TO_1,
    REDUCE_PWR_FROM_1,
    SET_PWR_TO_2,
    REDUCE_PWR_FROM_2,
    SET_PWR_TO_3,
    REDUCE_PWR_FROM_3
} PWR_MGMT_WAKEUP_TYPE;

/* Device power states */
typedef enum pwrMgmtDevicePowerType {
    D0_STATE,
    D1_STATE,
    D2_STATE,
    D3_STATE
} PWR_MGMT_DEVICE_POWER_TYPE;

/* Power management sub-states of D1 state */
typedef enum {
    STATE_AWAKE,
    STATE_PENDING_SLEEP,
    STATE_SLEEP,
    STATE_PENDING_BLOCKING_SLEEP,
    STATE_BLOCKING_SLEEP
}PWR_MGMT_SUB_STATE;

#define MAX_MAP_REGISTERS        512     /* map regs for dma mappings */
#define MAX_RX_PACKETS           NUM_RX_BLOCKS

#define CONNECTION_RESTORE_TIME  15000   // Overall wait for connection to restore after reset (ms)
#define CONNECT_CHECK_TIME       200     // Check-for-connection interval (ms)
#define ATH_CHECKFORHANG_TIMEOUT 4       // check for hang timeout

// Buffer size passed in NdisMQueryAdapterResources                            
// We should only need 2 adapter resources (interrupt and memory),
// Some devices get extra resources, so have room for 10 resources 
#define ATH_RESOURCE_BUF_SIZE   (sizeof(NDIS_RESOURCE_LIST) + (10 * sizeof(CM_PARTIAL_RESOURCE_DESCRIPTOR)))


#ifdef __cplusplus
}
#endif

#endif // _LINUXDEFS_H_

