/*
 * Copyright (c) 2004 Atheros Communications, Inc., All Rights Reserved
 *
 * athusbrxtx.c -- Atheros USB Driver send and receive file
 * 
 */
#include "athusbapi.h"
#include "athusbdrv.h"
#include "athusbconfig.h"
#include "athusbRxTx.h"

#ifdef __GNU__
static const char athId[] __attribute__ ((unused)) = "$Id: //depot/sw/branches/1.3_USB_LINUX_port/src/USB/transport/usb/host/drv/athusbRxTx.c#3 $";
#endif /* __GNUC__ */

/************************************************************************
* Routine Description:
*
*    This routine allocate ATHUSB_RXTX_OBJECT pool (as well as Irp/Urb pair).
*
*
* Arguments:
*
*    pUsbAdapter - Pointer to usb adapter handle
*
* Return Value:
*
*    NT status value
************************************************************************/
NTSTATUS
athUsbAllocIrpUrbPool(IN PATHUSB_USB_ADAPTER   pUsbAdapter)
{
    A_UINT32    i;
    NTSTATUS    ntStatus;

    athUsbDbgPrint(ATHUSB_LOUD, ("athUsbAllocIrpUrbPool - begins\n"));

    for (i = 0; i < ATHUSB_MAX_IRP_NUM; i++) {

        ntStatus = athUsbAllocTransferObj(pUsbAdapter,FALSE);

        if (!NT_SUCCESS(ntStatus)) {
            
            athUsbDbgPrint(ATHUSB_ERROR, ("athUsbAllocIrpUrbPool [%d] - failed\n", i));

            athUsbFreeIrpUrbPool(pUsbAdapter);
            return ntStatus;            
        }

        pUsbAdapter->totalWriteIrp ++;

        ntStatus = athUsbAllocTransferObj(pUsbAdapter,TRUE);

        if (!NT_SUCCESS(ntStatus)) {
            
            athUsbDbgPrint(ATHUSB_ERROR, ("athUsbAllocIrpUrbPool [%d] - failed\n", i));

            athUsbFreeIrpUrbPool(pUsbAdapter);                
            return ntStatus;
        }

        pUsbAdapter->totalReadIrp ++;
    }

    athUsbDbgPrint(ATHUSB_LOUD, ("athUsbAllocIrpUrbPool - ends\n"));
    return ntStatus;
}

/************************************************************************ 
* Routine Description:
*
*    This routine creates a transfer object for each irp/urb pair.
*
*
* Arguments:
*
*    pUsbAdapter - Pointer to usb adapter handle
*    StreamObject - pointer to stream object
*
* Return Value:
*
*    NT status value
************************************************************************/
NTSTATUS
athUsbAllocTransferObj(IN PATHUSB_USB_ADAPTER   pUsbAdapter,
                       IN BOOLEAN               bRead)
{
    PATHUSB_RXTX_OBJECT     pTransferObject;

    athUsbDbgPrint(ATHUSB_EXTRA_LOUD, ("athUsbAllocTransferObj - begins\n"));

    pTransferObject = AthExAllocatePool(NonPagedPool,
                                    sizeof(ATHUSB_RXTX_OBJECT));

    if (pTransferObject == NULL) {

        athUsbDbgPrint(ATHUSB_ERROR, ("Failed to alloc mem for transferObject\n"));

        return STATUS_INSUFFICIENT_RESOURCES;
    }

    RtlZeroMemory(pTransferObject,sizeof(ATHUSB_RXTX_OBJECT));

    InitializeListHead(&pTransferObject->link);

    pTransferObject->pUsbAdapter = pUsbAdapter;    

    pTransferObject->bRead = bRead;

	/* Linux has specialized functions for allocating URBs. Use a wrapper */
    pTransferObject->urb = AthExAllocateURB(NonPagedPool,
                                   sizeof(struct _URB_BULK_OR_INTERRUPT_TRANSFER));

    if (pTransferObject->urb == NULL) {

        athUsbDbgPrint(ATHUSB_ERROR, ("failed to alloc mem for Urb\n"));        
        IoFreeIrp(irp);
        ExFreePool(pTransferObject);
        return STATUS_INSUFFICIENT_RESOURCES;
    }    

    if (bRead) {
        InsertTailList(&pUsbAdapter->readIdleIrpQueue,
                       &pTransferObject->link);
    } else {
        InsertTailList(&pUsbAdapter->writeIdleIrpQueue,
                                    &pTransferObject->link);
    }
    
    athUsbDbgPrint(ATHUSB_EXTRA_LOUD, ("athUsbAllocTransferObj - ends\n"));

    return STATUS_SUCCESS;
}

/************************************************************************ 
* Routine Description:
*
*    Put the idle IRP into the queue.
*
*
* Arguments:
*
*    pUsbAdapter - Pointer to usb adapter handle
*    pTransferObject - pointer to transfer object
*    bRead - Read/Write flag
*
* Return Value:
*
*    None
************************************************************************/
VOID
athUsbQueueIdleIrp(IN PATHUSB_USB_ADAPTER     pUsbAdapter,
                   IN PATHUSB_RXTX_OBJECT     pTransferObject,
                   IN BOOLEAN                 bRead)
{
    KIRQL                   oldIrql;

    athUsbDbgPrint(ATHUSB_EXTRA_LOUD, ("athUsbQueueIdleIrp - begins\n"));

    IoReuseIrp(pTransferObject->irp,STATUS_SUCCESS);

    athGetCurrentIrql(oldIrql);

    if (bRead) {
        athAcquireSpinLock(&pUsbAdapter->readQueueLock, &oldIrql);        
        RemoveEntryList(&pTransferObject->link);
        InsertTailList(&pUsbAdapter->readIdleIrpQueue,&pTransferObject->link);
        athReleaseSpinLock(&pUsbAdapter->readQueueLock, oldIrql);
    } else {
        athAcquireSpinLock(&pUsbAdapter->writeQueueLock, &oldIrql);        
        RemoveEntryList(&pTransferObject->link);
        InsertTailList(&pUsbAdapter->writeIdleIrpQueue,&pTransferObject->link);
        athReleaseSpinLock(&pUsbAdapter->writeQueueLock, oldIrql);
    }

    athUsbDbgPrint(ATHUSB_EXTRA_LOUD, ("athUsbQueueIdleIrp - ends\n"));
}

/************************************************************************
* Routine Description:
*
*    Put the pending IRP into the queue.
*
*
* Arguments:
*
*    pUsbAdapter - Pointer to usb adapter handle
*    pTransferObject - pointer to transfer object
*    bRead - Read/Write flag
*
* Return Value:
*
*    None
************************************************************************/
VOID
athUsbInsertPendingIrp(IN PATHUSB_USB_ADAPTER     pUsbAdapter,
                      IN PATHUSB_RXTX_OBJECT     pTransferObject,
                      IN BOOLEAN                 bRead)
{
    KIRQL                   oldIrql;

    athUsbDbgPrint(ATHUSB_EXTRA_LOUD, ("athUsbInsertPendingIrp - begins\n"));

    athGetCurrentIrql(oldIrql);

    if (bRead) {
        athAcquireSpinLock(&pUsbAdapter->readQueueLock, &oldIrql);    
        InsertTailList(&pUsbAdapter->readPendingIrpQueue,&pTransferObject->link);
        athReleaseSpinLock(&pUsbAdapter->readQueueLock, oldIrql);
        InterlockedIncrement(&pUsbAdapter->readPendingIrps);
        InterlockedIncrement(&pUsbAdapter->pipeReadPending[pTransferObject->pipeNum]);
    } else {
        athAcquireSpinLock(&pUsbAdapter->writeQueueLock, &oldIrql);    
        InsertTailList(&pUsbAdapter->writePendingIrpQueue,&pTransferObject->link);
        athReleaseSpinLock(&pUsbAdapter->writeQueueLock, oldIrql);
        InterlockedIncrement(&pUsbAdapter->writePendingIrps);
        InterlockedIncrement(&pUsbAdapter->pipeWritePending[pTransferObject->pipeNum]);
    }

    athUsbDbgPrint(ATHUSB_EXTRA_LOUD, ("athUsbInsertPendingIrp - ends\n"));
}

/************************************************************************
* Routine Description:
*
*    Remove the Idle IRP from the queue.
*
*
* Arguments:
*
*    pUsbAdapter - Pointer to usb adapter handle
*    pTransferObject - pointer to transfer object
*    bRead - Read/Write flag
*
* Return Value:
*
*    None
************************************************************************/
NTSTATUS
athUsbRemoveIdleIrp(IN PATHUSB_USB_ADAPTER      pUsbAdapter,
                    IN OUT PATHUSB_RXTX_OBJECT  *ppTransferObject,
                    IN BOOLEAN                  bRead)
{
    KIRQL                   oldIrql;
    PATHUSB_RXTX_OBJECT     pTransferObject;
    PLIST_ENTRY             listEntry;
    NTSTATUS                ntStatus = STATUS_SUCCESS;

    athUsbDbgPrint(ATHUSB_EXTRA_LOUD, ("athUsbRemoveIdleIrp - begins\n"));

    athGetCurrentIrql(oldIrql);

    if (bRead) {
        athAcquireSpinLock(&pUsbAdapter->readQueueLock, &oldIrql); 
        
        if (IsListEmpty(&pUsbAdapter->readIdleIrpQueue)) {
            ntStatus = athUsbAllocTransferObj(pUsbAdapter,TRUE);
            if (!NT_SUCCESS(ntStatus)) {
                athReleaseSpinLock(&pUsbAdapter->readQueueLock, oldIrql);
                athUsbDbgPrint(ATHUSB_ERROR, ("athUsbAllocTransferObj - failed\n"));
                return ntStatus;
            }
            pUsbAdapter->totalReadIrp ++;
            athUsbDbgPrint(ATHUSB_INFO,("total Read IRP = %d\n",pUsbAdapter->totalReadIrp));
        }
        
        listEntry = RemoveHeadList(&pUsbAdapter->readIdleIrpQueue);        
        pTransferObject = CONTAINING_RECORD(listEntry, ATHUSB_RXTX_OBJECT, link);   

        (*ppTransferObject) = pTransferObject;

        athReleaseSpinLock(&pUsbAdapter->readQueueLock, oldIrql);
        
    } else {
        athAcquireSpinLock(&pUsbAdapter->writeQueueLock, &oldIrql);        
        
        if (IsListEmpty(&pUsbAdapter->writeIdleIrpQueue)) {
            ntStatus = athUsbAllocTransferObj(pUsbAdapter,FALSE);
            if (!NT_SUCCESS(ntStatus)) {
                athReleaseSpinLock(&pUsbAdapter->writeQueueLock, oldIrql);
                athUsbDbgPrint(ATHUSB_ERROR, ("athUsbAllocTransferObj - failed\n"));
                return ntStatus;
            }

            pUsbAdapter->totalWriteIrp ++;
            athUsbDbgPrint(ATHUSB_INFO,("total Write IRP = %d\n",pUsbAdapter->totalWriteIrp));
        }
        
        listEntry = RemoveHeadList(&pUsbAdapter->writeIdleIrpQueue);        
        pTransferObject = CONTAINING_RECORD(listEntry, ATHUSB_RXTX_OBJECT, link);   
        
        (*ppTransferObject) = pTransferObject;

        athReleaseSpinLock(&pUsbAdapter->writeQueueLock, oldIrql);        
    }

    athUsbDbgPrint(ATHUSB_EXTRA_LOUD, ("athUsbRemoveIdleIrp - ends\n"));

    return ntStatus;
}

/************************************************************************ 
* Routine Description:
*
*    This routine initializes the urb in the transfer object.
*
* Arguments:
*
*    DeviceObject - Pointer to usb adapter handle
*    pTransferObject - pointer to transfer object
*
* Return Value:
*
*    NT status value
*
************************************************************************/
NTSTATUS
athUsbInitTransferUrb(IN PATHUSB_USB_ADAPTER     pUsbAdapter,
                      IN PATHUSB_RXTX_OBJECT     pTransferObject)

{
    PURB                    urb;        
    PUSBD_PIPE_INFORMATION  pipeInformation;

    athUsbDbgPrint(ATHUSB_EXTRA_LOUD, ("athUsbInitTransferUrb - begins\n"));

    urb = pTransferObject->urb;
    pipeInformation = pTransferObject->pPipeInformation;

#ifdef Linux
    {
    int pipe;
    /* Fill in bulk pipe information and other properties in the URB but dont
     * submit it - FILL_BULK macros are not used here, since they are 
     * mentioned as deprecated by the Linux header files */
    if (pTransferObject->bRead) {
        pipe = usb_rcvbulkpipe(pUsbAdapter->pNextDeviceObject->pUsbDevice,pipeInformation->bEndpointAddress);
    } else {
        pipe = usb_sndbulkpipe(pUsbAdapter->pNextDeviceObject->pUsbDevice,pipeInformation->bEndpointAddress);
    }
    usb_fill_bulk_urb(urb,
            pUsbAdapter->pNextDeviceObject->pUsbDevice,
            pipe,
            pTransferObject->pDataBuffer,
            pTransferObject->dataBufferLen,
            (pTransferObject->bRead) ? athUsbRxUrbComplete:athUsbTxUrbComplete,
            pTransferObject);
    }
#endif

    athUsbDbgPrint(ATHUSB_EXTRA_LOUD, ("athUsbInitTransferUrb - ends\n"));

    return STATUS_SUCCESS;
}

/************************************************************************ 
* Routine Description:
*
*    This routine initializes the Irp in the transfer object.
*
* Arguments:
*
*    DeviceObject - Pointer to usb adapter handle
*    pTransferObject - pointer to transfer object
*
* Return Value:
*
*    NT status value
*
************************************************************************/
NTSTATUS
athUsbInitTransferIrp(IN PATHUSB_USB_ADAPTER     pUsbAdapter,
                      IN PATHUSB_RXTX_OBJECT     pTransferObject)
{
    return STATUS_SUCCESS;
}

/************************************************************************
* Routine Description:
*
*    This routine free ATHUSB_RXTX_OBJECT pool (as well as Irp/Urb pair).
*
*
* Arguments:
*
*    pUsbAdapter - Pointer to usb adapter handle
*
* Return Value:
*
*    none
************************************************************************/
VOID
athUsbFreeIrpUrbPool(IN PATHUSB_USB_ADAPTER     pUsbAdapter)
{
    KIRQL                   oldIrql;
    PATHUSB_RXTX_OBJECT     xferObject;    
    PLIST_ENTRY             listEntry;

    athUsbDbgPrint(ATHUSB_LOUD, ("athUsbFreeIrpUrbPool - begins\n"));

    athGetCurrentIrql(oldIrql);

    athAcquireSpinLock(&pUsbAdapter->readQueueLock, &oldIrql);
    
    while (!IsListEmpty(&pUsbAdapter->readIdleIrpQueue)) {        
        listEntry = RemoveHeadList(&pUsbAdapter->readIdleIrpQueue);        
        xferObject = CONTAINING_RECORD(listEntry, ATHUSB_RXTX_OBJECT, link);           
      
        if (xferObject) { 
            
            if (xferObject->urb) {
				/* Remember, we did special allocations for Linux URB */
                AthExFreeURB(xferObject->urb);
                xferObject->urb = NULL;
            }
            
            if (xferObject->irp) {
                athReleaseSpinLock(&pUsbAdapter->readQueueLock, oldIrql);
                IoFreeIrp(xferObject->irp);
                athAcquireSpinLock(&pUsbAdapter->readQueueLock, &oldIrql);
            }
                
            ExFreePool(xferObject);
        }
    }
    
    athReleaseSpinLock(&pUsbAdapter->readQueueLock, oldIrql);

    athAcquireSpinLock(&pUsbAdapter->writeQueueLock, &oldIrql);

    while (!IsListEmpty(&pUsbAdapter->writeIdleIrpQueue)) {        
        listEntry = RemoveHeadList(&pUsbAdapter->writeIdleIrpQueue);        
        xferObject = CONTAINING_RECORD(listEntry, ATHUSB_RXTX_OBJECT, link);           
    
        if (xferObject) { 
            
            if (xferObject->urb) {
				/* Remember, we did special allocations for Linux URB */
                AthExFreeURB(xferObject->urb);
                xferObject->urb = NULL;
            }            

            if (xferObject->irp) {
                athReleaseSpinLock(&pUsbAdapter->writeQueueLock, oldIrql);
                IoFreeIrp(xferObject->irp);
                athAcquireSpinLock(&pUsbAdapter->writeQueueLock, &oldIrql);
            }
                
            ExFreePool(xferObject);
        }
    }
    
    athReleaseSpinLock(&pUsbAdapter->writeQueueLock, oldIrql);

    athUsbDbgPrint(ATHUSB_LOUD, ("athUsbFreeIrpUrbPool - ends\n"));
}

/************************************************************************
* Routine Description:
*
*    This routine cancel all transaction pending IRP.
*
*
* Arguments:
*
*    pUsbAdapter - Pointer to usb adapter handle
*    bAbort      - abort pipe or not
*
* Return Value:
*
*    none
************************************************************************/
VOID
athUsbCancelAllIrp(IN PATHUSB_USB_ADAPTER     pUsbAdapter,
                IN A_BOOL                     bAbort)
{
    KIRQL                   oldIrql;
    PATHUSB_RXTX_OBJECT     pTransferObject;    
    PLIST_ENTRY             listEntry;
    ULONG                   numIrps;

    athUsbDbgPrint(ATHUSB_LOUD, ("athUsbCancelAllIrp - begins\n"));

    pUsbAdapter->bCancel = TRUE;

    athGetCurrentIrql(oldIrql);

    athAcquireSpinLock(&pUsbAdapter->readQueueLock, &oldIrql);

    if (athReadAtomic(pUsbAdapter->readPendingIrps) != 0) {
        KeClearEvent(&pUsbAdapter->noPendingReadIrpEvent);
    }

    numIrps = 0;
    if (bAbort) {
        while (!IsListEmpty(&pUsbAdapter->readPendingIrpQueue)) {
            listEntry = RemoveHeadList(&pUsbAdapter->readPendingIrpQueue);
            pTransferObject = CONTAINING_RECORD(listEntry, ATHUSB_RXTX_OBJECT, link);
            listEntry->Flink = listEntry->Blink = listEntry;

            //
            // cancel transferobject irps/urb pair
            // safe to touch these irps because the 
            // completion routine always returns 
            // STATUS_MORE_PRCESSING_REQUIRED
            // 
            //
            /* Wrap the IRP/URB cancellation */
            athCancelIrp(pUsbAdapter, pTransferObject, oldIrql, 1);
            numIrps ++;
            
        }   
    }

    athUsbDbgPrint(ATHUSB_LOUD, ("Total %ld Read IRPs cancel\n",numIrps));

    athReleaseSpinLock(&pUsbAdapter->readQueueLock, oldIrql);

    athAcquireSpinLock(&pUsbAdapter->writeQueueLock, &oldIrql);

    if (athReadAtomic(pUsbAdapter->writePendingIrps) != 0) {
        KeClearEvent(&pUsbAdapter->noPendingWriteIrpEvent);
    }

    numIrps = 0;
    if (bAbort) {
        while (!IsListEmpty(&pUsbAdapter->writePendingIrpQueue)) {        
            listEntry = RemoveHeadList(&pUsbAdapter->writePendingIrpQueue);         
            pTransferObject = CONTAINING_RECORD(listEntry, ATHUSB_RXTX_OBJECT, link);
            listEntry->Flink = listEntry->Blink = listEntry;
        
            //
            // cancel transferobject irps/urb pair
            // safe to touch these irps because the 
            // completion routine always returns 
            // STATUS_MORE_PRCESSING_REQUIRED
            // 
            //
            athCancelIrp(pUsbAdapter, pTransferObject, oldIrql, 0);
            
            numIrps ++;
        }   
    }

    athUsbDbgPrint(ATHUSB_LOUD, ("Total %ld Write IRPs cancel\n",numIrps));
    athReleaseSpinLock(&pUsbAdapter->writeQueueLock, oldIrql);

    athUsbDbgPrint(ATHUSB_LOUD, ("athUsbCancelAllIrp - ends\n"));
}

/************************************************************************
* Routine Description:
*
*    This routine cancel all transaction pending IRP.
*
*
* Arguments:
*
*    pUsbAdapter - Pointer to usb adapter handle
*    bAbort      - abort pipe or not
*
* Return Value:
*
*    none
************************************************************************/
VOID
athUsbCancelIrp(IN PATHUSB_USB_ADAPTER     pUsbAdapter,
                IN A_UINT8                 epNum,
                IN A_BOOL                  bRead)
{
    KIRQL                   oldIrql;
    PATHUSB_RXTX_OBJECT     pTransferObject;    
    PLIST_ENTRY             listEntry,tempEntry,pQueue;
    PATHUSB_SPIN_LOCK       pAthLock;

    athUsbDbgPrint(ATHUSB_EXTRA_LOUD, ("athUsbCancelIrp - begins\n"));    

    if (bRead) {
        if (athReadAtomic(pUsbAdapter->pipeReadPending[epNum])) {
            KeClearEvent(&pUsbAdapter->noPendingPipeReadEvent[epNum]);
        }

        pQueue = &pUsbAdapter->readPendingIrpQueue;
        pAthLock = &pUsbAdapter->readQueueLock;
    } else {
        if (athReadAtomic(pUsbAdapter->pipeWritePending[epNum])) {
            KeClearEvent(&pUsbAdapter->noPendingPipeWriteEvent[epNum]);
        }

        pQueue = &pUsbAdapter->writePendingIrpQueue;
        pAthLock = &pUsbAdapter->writeQueueLock;
    }

    athGetCurrentIrql(oldIrql);
    athAcquireSpinLock(pAthLock ,&oldIrql);

    listEntry = pQueue->Flink;
    while (listEntry != pQueue) {

        pTransferObject = CONTAINING_RECORD(listEntry, ATHUSB_RXTX_OBJECT, link);

        if (pTransferObject && (pTransferObject->bRead == bRead)
            && (pTransferObject->pipeNum == epNum))
        {
            tempEntry = listEntry->Flink;  

            RemoveEntryList(listEntry);

            listEntry->Flink = listEntry->Blink = listEntry;
            listEntry = tempEntry;

            //
            // cancel transferobject irps/urb pair
            // safe to touch these irps because the
            // completion routine always returns
            // STATUS_MORE_PRCESSING_REQUIRED
            // 
            //
#ifdef Linux
            if (pTransferObject->urb) {
                athReleaseSpinLock(pAthLock, oldIrql);
                usb_unlink_urb(pTransferObject->urb);
                athAcquireSpinLock(pAthLock ,&oldIrql);
            }
#endif
        } else {
            listEntry = listEntry->Flink;
        }
    }

    athReleaseSpinLock(pAthLock, oldIrql);

    athUsbDbgPrint(ATHUSB_EXTRA_LOUD, ("athUsbCancelIrp - ends\n"));
}
/************************************************************************ 
* Routine Description:
*
*    This routine is invoked from the completion routine to check the status
*    of the irp, urb and the bulk packets.
*
*    updates statistics
*
* Arguments:
*
*    TranferObject - pointer to transfer object for the irp/urb pair 
*                    which completed.
*
* Return Value:
*
*    NT status value
************************************************************************/
NTSTATUS
athUsbProcessTransfer(IN PATHUSB_RXTX_OBJECT     pTransferObject,
                      IN OUT USBD_STATUS         *pUsbdStatus)

{
    PIRP        irp;
    PURB        urb;    
    NTSTATUS    ntStatus;
    USBD_STATUS usbdStatus;

    athUsbDbgPrint(ATHUSB_EXTRA_LOUD, ("athUsbProcessTransfer - begins\n"));

    irp = pTransferObject->irp;
    urb = pTransferObject->urb;
    ntStatus = athUsbGetIoStatus(irp);

    if (!NT_SUCCESS(ntStatus)) {
        athUsbDbgPrint(ATHUSB_ERROR, 
                      ("Bulk irp failed with status = %X, irp =%p\n", 
                      ntStatus,irp))
//        ASSERT (0);
    }

    usbdStatus = athUsbGetUsbdStatus(urb);

    if (!USBD_SUCCESS(usbdStatus)) {
        /*
         * During the driver unload / device removal, URBs are cancelled.
         * This might result in throwing up some error messages, this is 
         * normal and should not be treated as errors
         */
        if (usbdStatus != USBD_STATUS_CANCELED) {
            athUsbDbgPrint(ATHUSB_ERROR, 
                    ("urb failed with status = %X, pData = %p\n", 
                     usbdStatus,pTransferObject->pDataBuffer));
        }
//        ASSERT (0);
        if (NT_SUCCESS(ntStatus)) {
            ntStatus = STATUS_UNSUCCESSFUL;
        }
    }

    *pUsbdStatus = usbdStatus;
    
    athUsbDbgPrint(ATHUSB_EXTRA_LOUD, ("athUsbProcessTransfer - ends\n"));

    return ntStatus;
}

/************************************************************************ 
*Routine Description:
*
*    Send all waiting IRP to the bus driver
*
*Arguments:
*
*    pUsbAdapter - Pointer to usb adapter handle
*
*Return Value:
*
*    None
************************************************************************/
VOID
athUsbDrainWaitQueue(IN PATHUSB_USB_ADAPTER     pUsbAdapter)
{    
    PATHUSB_RXTX_OBJECT             pTransferObject;
    PLIST_ENTRY                     listEntry;
    NTSTATUS                        ntStatus;

    while (!IsListEmpty(&pUsbAdapter->writeWaitingQueue)) {
        listEntry = ExInterlockedRemoveHeadList(&pUsbAdapter->writeWaitingQueue,
                                                &(pUsbAdapter->writeQueueLock.spinLock));
        pTransferObject = CONTAINING_RECORD(listEntry,ATHUSB_RXTX_OBJECT, link);
        athUsbInsertPendingIrp(pUsbAdapter,pTransferObject,FALSE);
        ntStatus = athIoCallDriver(pUsbAdapter->pNextDeviceObject,pTransferObject);
        if(!NT_SUCCESS(ntStatus)) {
            return;
        }
    }

    while (!IsListEmpty(&pUsbAdapter->readWaitingQueue)) {
        listEntry = ExInterlockedRemoveHeadList(&pUsbAdapter->readWaitingQueue,
                                                &(pUsbAdapter->readQueueLock.spinLock));
        pTransferObject = CONTAINING_RECORD(listEntry,ATHUSB_RXTX_OBJECT, link);
        athUsbInsertPendingIrp(pUsbAdapter,pTransferObject,TRUE);
        ntStatus = athIoCallDriver(pUsbAdapter->pNextDeviceObject,pTransferObject);
        if(!NT_SUCCESS(ntStatus)) {            
            return;
        }
    }
}

/************************************************************************ 
* Routine Description:
*
*    DPC routine triggered by the timer to schdule work item 
*    which will do pipe reset
*
* Arguments:
*
*    DeferredContext - context for the dpc routine.
*                      ATHUSB_USB_ADAPTER in our case.
*
* Return Value:
*
*    None
************************************************************************/
VOID
athUsbDpcProc(
              IN   PKDPC    dpc,
              IN   PVOID    deferredContex,
              IN   PVOID    systemContex1,
              IN   PVOID    systemContex2)
{
    PATHUSB_USB_ADAPTER             pUsbAdapter;
    LARGE_INTEGER                   newTime;
    ULONG                           timeDiff;

    athUsbDbgPrint(ATHUSB_INFO, ("DpcRoutine - begins\n"));

    pUsbAdapter = (PATHUSB_USB_ADAPTER)deferredContex;

    if (pUsbAdapter->bStop || pUsbAdapter->bCancel) {
        athUsbDbgPrint(ATHUSB_INFO, ("DpcRoutine - ends\n"));
        return;
    }

    KeClearEvent(&pUsbAdapter->usbAdptDpcEvent);

    pUsbAdapter->bQueueTimer = FALSE;    

    /*if (!pUsbAdapter->bCount) {
        getPentiumCycleCounter(&pUsbAdapter->oldTime);
        pUsbAdapter->bCount = TRUE;
    } else {
        getPentiumCycleCounter(&newTime);
        timeDiff = (ULONG)((newTime.QuadPart - pUsbAdapter->oldTime.QuadPart) /1331);

        if (pUsbAdapter->minTime > timeDiff) {
            pUsbAdapter->minTime = timeDiff;
        }       

        if (pUsbAdapter->maxTime < timeDiff) {
            pUsbAdapter->maxTime = timeDiff;
        }
        pUsbAdapter->totalCycle ++;
        pUsbAdapter->totalTime += timeDiff;
        pUsbAdapter->bCount = FALSE;
    }*/
#if defined(DEBUG) && !defined (Linux)
    getPentiumCycleCounter(&pUsbAdapter->oldTime);
#endif

    athUsbDrainWaitQueue(pUsbAdapter);

#if defined(DEBUG) && !defined (Linux)
    getPentiumCycleCounter(&newTime);
    timeDiff = (ULONG)((newTime.QuadPart - pUsbAdapter->oldTime.QuadPart) /1331);

    if (pUsbAdapter->minTime > timeDiff) {
        pUsbAdapter->minTime = timeDiff;
    }       

    if (pUsbAdapter->maxTime < timeDiff) {
        pUsbAdapter->maxTime = timeDiff;
    }
    pUsbAdapter->totalCycle ++;
    pUsbAdapter->totalTime += timeDiff;
#endif

    KeSetEvent(&pUsbAdapter->usbAdptDpcEvent,
               IO_NO_INCREMENT,
               FALSE);

    athUsbDbgPrint(ATHUSB_INFO, ("DpcRoutine - ends\n"));
}

