/*
 * Copyright (c) 2004 Atheros Communications, Inc., All Rights Reserved
 *
 * osfunc.c -- Atheros USB Driver configuration file
 * 
 * This is a clone of athusbconfig.c.  There was too much OS dependent
 * functions in it to provide any wrappers
 */

#ifdef __GNU__
static const char athId[] __attribute__ ((unused)) = "$Id: //depot/sw/branches/1.3_USB_LINUX_port/src/USB/transport/usb/host/drv/osfunc.c#3 $";
#endif /* __GNUC__ */

#include <athusbapi.h>
#include <athusbdrv.h>
#include "athusbconfig.h"
#include "athusbRxTx.h"

/*****************************************************************************
* Routine Description:
*
*    Send Vendor Specific reset over USB bus to tell target device host
*    want to re-synchronize with device
*
* Arguments:
*
*    pUsbAdapter - Pointer to usb adapter handle
*
* Return Value:
*
*    NT status value
*****************************************************************************/
NTSTATUS
athUsbVendorReset(IN PATHUSB_USB_ADAPTER  pUsbAdapter)
{
    ATHUSB_VENDOR_REQUEST         vendorRequest;
    
    //reset the devie to clear all buffer
    vendorRequest.bRequest = ATHUSB_RESET_FEATURE;
    vendorRequest.direction = 0;
    vendorRequest.wValue = ATHUSB_RESET_DEVICE;
    vendorRequest.wIndex = 0;
    vendorRequest.wLength = 0;

    return athUsbVendorRequest(pUsbAdapter, &vendorRequest);
}

/***************************************************************************** 
* Routine Description:
*
*    This routine returns all the memory allocations acquired during
*    device startup. 
*    
* Arguments:
*
*    pUsbAdapter - Pointer to usb adapter handle
*        
*    
* Return Value:
*
*    STATUS_SUCCESS if the device can be safely removed, an appropriate 
*    NT Status if not.
*******************************************************************************/
NTSTATUS
athUsbReleaseMemory(
    IN PATHUSB_USB_ADAPTER          pUsbAdapter)
{
    //
    // Disconnect from the interrupt and unmap any I/O ports
    //    
    A_UINT32                 i,j;
    PATHUSB_STRING_DESC      pMyStringDesc;        

    if (pUsbAdapter->pUsbDeviceDescriptor) {

        pUsbAdapter->pUsbDeviceDescriptor = NULL;
    }

    if (pUsbAdapter->pUsbConfigurationDescriptor) {

        pUsbAdapter->pUsbConfigurationDescriptor = NULL;
    }

    if (pUsbAdapter->pUsbInterface) {
        
        pUsbAdapter->pUsbInterface = NULL;
    }

    if (pUsbAdapter->pPipeState) {

        ExFreePool(pUsbAdapter->pPipeState);
        pUsbAdapter->pPipeState = NULL;
    }

    for (i = 0; i < pUsbAdapter->stringDescNum; i ++) {
        pMyStringDesc = &pUsbAdapter->stringDesc[i];
        for (j = 0; j < pUsbAdapter->languageNum; j ++) {
            if (pMyStringDesc->UnicodeString[j].Buffer) {
                ExFreePool(pMyStringDesc->UnicodeString[j].Buffer);
                pMyStringDesc->UnicodeString[j].Buffer = NULL;
            }
        }
    }    

    return STATUS_SUCCESS;
}

/***************************************************************************** 
* Routine Description:
*
*    This routine retrieves the status value
*
* Arguments:
*
*    pUsbAdapter - Pointer to usb adapter handle
*    PortStatus - port status
*
* Return Value:
*
*    NT status value
*****************************************************************************/
NTSTATUS
athUsbGetPortStatus(
    IN PATHUSB_USB_ADAPTER      pUsbAdapter,
    IN OUT PULONG               PortStatus
    )
{
    *PortStatus = (USBD_PORT_ENABLED | USBD_PORT_CONNECTED);
    return (STATUS_SUCCESS);
}

NTSTATUS
athUsbResetAllPipes(IN PATHUSB_USB_ADAPTER      pUsbAdapter)
{
    A_UINT32                    i;          
    PUSBD_INTERFACE_INFORMATION pInterfaceInfo;
    NTSTATUS                    ntStatus;

    //
    // initialize variables
    //        
    pInterfaceInfo = pUsbAdapter->pUsbInterface;
    
    athUsbDbgPrint(ATHUSB_LOUD, ("athUsbResetAllPipes - begins\n"));
    
    if (pInterfaceInfo == NULL) {

        return STATUS_SUCCESS;
    }

    for (i = 0; i < pInterfaceInfo->bNumEndpoints; i ++) {
        int pipe;
        struct usb_endpoint_descriptor *endpoint = athUsbGetPipeInformation(pUsbAdapter,i);
        if (endpoint) {
            if (USB_ENDPOINT_DIRECTION_IN(endpoint)) {
                pipe = usb_rcvbulkpipe(pUsbAdapter->pNextDeviceObject->pUsbDevice,endpoint->bEndpointAddress);
            } else {
                pipe = usb_sndbulkpipe(pUsbAdapter->pNextDeviceObject->pUsbDevice,endpoint->bEndpointAddress);
            }
            ntStatus = usb_clear_halt(pUsbAdapter->pNextDeviceObject->pUsbDevice,pipe);
    
            if (!NT_SUCCESS(ntStatus)) {
            
                athUsbDbgPrint(ATHUSB_ERROR, ("athUsbResetPipe failed\n"));
                ntStatus = usb_reset_device(pUsbAdapter->pNextDeviceObject->pUsbDevice);
            }
        }
    }

    athUsbDbgPrint(ATHUSB_LOUD, ("athUsbResetAllPipes - ends\n"));


    return STATUS_SUCCESS;
}

/*****************************************************************************
* Routine Description:
*
*    Send Vendor Specific request over USB bus
*
* Arguments:
*
*    pUsbAdapter - Pointer to usb adapter handle
*    pVendorRequest - vendor request
*
* Return Value:
*
*    NT status value
*****************************************************************************/

NTSTATUS
athUsbVendorRequest(IN PATHUSB_USB_ADAPTER      pUsbAdapter,
                    IN PATHUSB_VENDOR_REQUEST   pVendorRequest)
{
    NTSTATUS            ntStatus        = STATUS_SUCCESS;
    PUCHAR              buffer = NULL;
    unsigned int dwControlPipe;
    unsigned char bRequestType;
    
    athUsbDbgPrint(ATHUSB_LOUD, ("athUsbVendorRequest - begins\n"));

    if (pVendorRequest->wLength) {
        buffer = pVendorRequest->pData;
    }

    if (pVendorRequest->direction) {
        dwControlPipe = usb_sndctrlpipe(pUsbAdapter->pNextDeviceObject->pUsbDevice,0);
        bRequestType = USB_TYPE_VENDOR | USB_RECIP_DEVICE | USB_DIR_IN;
    } else {
        dwControlPipe = usb_rcvctrlpipe(pUsbAdapter->pNextDeviceObject->pUsbDevice,0);
        bRequestType = USB_TYPE_VENDOR | USB_RECIP_DEVICE | USB_DIR_OUT;
    }
    ntStatus = usb_control_msg(pUsbAdapter->pNextDeviceObject->pUsbDevice,
            dwControlPipe,
            pVendorRequest->bRequest,
            bRequestType,
            pVendorRequest->wValue,
            pVendorRequest->wIndex,
            buffer,
            pVendorRequest->wLength,
            HZ);
    athUsbDbgPrint(ATHUSB_LOUD, ("athUsbVendorRequest - ends\n"));
    return ntStatus;
}

/***************************************************************************** 
* Routine Description:
*
*    This routine configures the USB device.
*    In this routines we get the device descriptor, 
*    the configuration descriptor and select the
*    configuration descriptor.
*
* Arguments:
*
*    pUsbAdapter - Pointer to usb adapter handle
*
* Return Value:
*
*    NTSTATUS - NT status value.
*****************************************************************************/
NTSTATUS
athUsbReadandSelectDescriptors(
                               IN PATHUSB_USB_ADAPTER  pUsbAdapter)
{
    NTSTATUS               ntStatus;
    PUSB_DEVICE_DESCRIPTOR deviceDescriptor;    
    PUSB_DEVICE usb_dev;
    
    //
    // initialize variables
    //

    deviceDescriptor = NULL;    

    //
    // 1. Read the device descriptor
    //

    usb_dev = (PUSB_DEVICE)(pUsbAdapter->pNextDeviceObject->pUsbDevice);
    deviceDescriptor = &usb_dev->descriptor;
    pUsbAdapter->pUsbDeviceDescriptor = deviceDescriptor;
    
    ASSERT(deviceDescriptor->bNumConfigurations);

    athUsbDbgPrint(ATHUSB_INFO, ("Atheros USB Device Descriptor:\n"));
    athUsbDbgPrint(ATHUSB_INFO, ("-------------------------\n"));
    athUsbDbgPrint(ATHUSB_INFO, ("bLength %d\n", deviceDescriptor->bLength));
    athUsbDbgPrint(ATHUSB_INFO, ("bDescriptorType 0x%x\n", deviceDescriptor->bDescriptorType));
    athUsbDbgPrint(ATHUSB_INFO, ("bcdUSB 0x%x\n", deviceDescriptor->bcdUSB));
    athUsbDbgPrint(ATHUSB_INFO, ("bDeviceClass 0x%x\n", deviceDescriptor->bDeviceClass));
    athUsbDbgPrint(ATHUSB_INFO, ("bDeviceSubClass 0x%x\n", deviceDescriptor->bDeviceSubClass));
    athUsbDbgPrint(ATHUSB_INFO, ("bDeviceProtocol 0x%x\n", deviceDescriptor->bDeviceProtocol));
    athUsbDbgPrint(ATHUSB_INFO, ("bMaxPacketSize0 0x%x\n", deviceDescriptor->bMaxPacketSize0));
    athUsbDbgPrint(ATHUSB_INFO, ("idVendor 0x%x\n", deviceDescriptor->idVendor));
    athUsbDbgPrint(ATHUSB_INFO, ("idProduct 0x%x\n", deviceDescriptor->idProduct));
    athUsbDbgPrint(ATHUSB_INFO, ("bcdDevice 0x%x\n", deviceDescriptor->bcdDevice));
    athUsbDbgPrint(ATHUSB_INFO, ("iManufacturer 0x%x\n", deviceDescriptor->iManufacturer));
    athUsbDbgPrint(ATHUSB_INFO, ("iProduct 0x%x\n", deviceDescriptor->iProduct));
    athUsbDbgPrint(ATHUSB_INFO, ("iSerialNumber 0x%x\n", deviceDescriptor->iSerialNumber));
    athUsbDbgPrint(ATHUSB_INFO, ("bNumConfigurations 0x%x\n", deviceDescriptor->bNumConfigurations));
    pUsbAdapter->pUsbDeviceDescriptor = deviceDescriptor;

    ntStatus = athUsbConfigureDevice(pUsbAdapter);    

    //We can start to download firmware from here.
    return ntStatus;
}

/*****************************************************************************
* Routine Description:
*
*    This helper routine reads the configuration descriptor
*    for the device in couple of steps.
*
* Arguments:
*
*    pUsbAdapter - Pointer to usb adapter handle
*
* Return Value:
*
*    NTSTATUS - NT status value
*****************************************************************************/
NTSTATUS
athUsbConfigureDevice(
                      IN PATHUSB_USB_ADAPTER  pUsbAdapter)
{
    NTSTATUS                      ntStatus = STATUS_SUCCESS;    
    PUSB_CONFIGURATION_DESCRIPTOR configurationDescriptor;
    PUSB_DEVICE                     usb_dev;

    //
    // initialize the variables
    //

    configurationDescriptor = NULL;    

    //
    // Read the first configuration descriptor
    // This requires two steps:
    // 1. Read the fixed sized configuration desciptor (CD)
    // 2. Read the CD with all embedded interface and endpoint descriptors
    //

    usb_dev = (PUSB_DEVICE)(pUsbAdapter->pNextDeviceObject->pUsbDevice);
#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0))
    configurationDescriptor = usb_dev->actconfig;
#else
    configurationDescriptor = &usb_dev->actconfig->desc;
#endif

    if (configurationDescriptor) {

        //
        // save a copy of configurationDescriptor in deviceExtension
        // remember to free it later.
        //
        pUsbAdapter->pUsbConfigurationDescriptor = configurationDescriptor;

        if (configurationDescriptor->bmAttributes & ATHUSB_REMOTE_WAKEUP_MASK) {
            //
            // this configuration supports remote wakeup
            //
            pUsbAdapter->bWaitWakeEnable = 1;
        } else {
            pUsbAdapter->bWaitWakeEnable = 0;
        }

        ntStatus = athUsbSelectInterfaces(pUsbAdapter, configurationDescriptor);
    } else {

        pUsbAdapter->pUsbConfigurationDescriptor = NULL;
    }

    return ntStatus;
}

/***************************************************************************** 
* Routine Description:
*
*    This helper routine selects the configuration
*
* Arguments:
*
*    pUsbAdapter - Pointer to usb adapter handle
*    ConfigurationDescriptor - pointer to the configuration
*    descriptor for the device
*
* Return Value:
*
*    NT status value
*****************************************************************************/
NTSTATUS
athUsbSelectInterfaces(
    IN PATHUSB_USB_ADAPTER           pUsbAdapter,
    IN PUSB_CONFIGURATION_DESCRIPTOR configurationDescriptor)
{
    A_INT32                     numberOfInterfaces, 
                                interfaceNumber, 
                                interfaceindex;
    A_UINT32                    i;
    PUCHAR                      pInf;
    NTSTATUS                    ntStatus = STATUS_SUCCESS;
    PUSB_INTERFACE_DESCRIPTOR   interfaceDescriptor;
    PUSB_INTERFACE              interface;
    struct usb_interface        *ifp;
    struct usb_endpoint_descriptor *endpoint;
    struct usb_host_interface   *host_ifp;


    //
    // initialize the variables
    //

    interfaceDescriptor = NULL;
    numberOfInterfaces = configurationDescriptor->bNumInterfaces;
    interfaceindex = interfaceNumber = 0;

    //
    // Parse the configuration descriptor for the interface;
    //

#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0))
    ifp = configurationDescriptor->interface;
    interfaceDescriptor = ifp->altsetting + ifp->act_altsetting;
#else
    interface=pUsbAdapter->pNextDeviceObject->pUsbInterface;
    host_ifp = interface->cur_altsetting;
    interfaceDescriptor = &host_ifp->desc;
#endif


#if 0
    for (i = 0; i < interfaceDescriptor->bNumEndpoints; i++) { 
        //
        // perform pipe initialization here
        // set the transfer size and any pipe flags we use
        // USBD sets the rest of the Interface struct members
        //

        interfaceDescriptor->endpoint[i].wMaxPacketSize = 
            ATHUSB_DEFAULT_MAXIMUM_SIZE;
    }
#endif

    // Save the configuration handle for this device
    pUsbAdapter->configurationHandle = configurationDescriptor;

    //
    // save a copy of interface information in the device extension.
    //
    pUsbAdapter->pUsbInterface = interfaceDescriptor;

    //
    // Dump the interface to the debugger
    //

    athUsbDbgPrint(ATHUSB_INFO, ("---------\n"));
    athUsbDbgPrint(ATHUSB_INFO, ("NumberOfPipes 0x%x\n", 
                interfaceDescriptor->bNumEndpoints));
    athUsbDbgPrint(ATHUSB_INFO, ("Length 0x%x\n", 
                interfaceDescriptor->bLength));
    athUsbDbgPrint(ATHUSB_INFO, ("Alt Setting 0x%x\n", 
                interfaceDescriptor->bAlternateSetting));
    athUsbDbgPrint(ATHUSB_INFO, ("Interface Number 0x%x\n", 
                interfaceDescriptor->bInterfaceNumber));
    athUsbDbgPrint(ATHUSB_INFO, ("Class, subclass, protocol 0x%x 0x%x 0x%x\n",
                interfaceDescriptor->bInterfaceClass,
                interfaceDescriptor->bInterfaceSubClass,
                interfaceDescriptor->bInterfaceProtocol));
    //
    // Initialize the pipeState
    // Dump the pipe info
    //

    if (interfaceDescriptor->bNumEndpoints) {
        pUsbAdapter->pPipeState = AthExAllocatePool(NonPagedPool,
                interfaceDescriptor->bNumEndpoints *
                sizeof(ATHUSB_PIPE_STATE)); 
        
        if (pUsbAdapter->pPipeState) { 
            
            for (i=0; i<interfaceDescriptor->bNumEndpoints; i++) {

                pUsbAdapter->pPipeState[i].pipeOpen = FALSE;
            }
        } else { 
            ntStatus = STATUS_INSUFFICIENT_RESOURCES;
            athUsbDbgPrint(ATHUSB_ERROR, ("memory alloc for UsbInterface failed\n"));
        }
    } 
    
    for (i = 0; i < interfaceDescriptor->bNumEndpoints; i ++) {
#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0))
    endpoint = &interfaceDescriptor->endpoint[i];
#else
    endpoint = &host_ifp->endpoint[i].desc;
#endif

        athUsbDbgPrint(ATHUSB_INFO, ("---------\n"));
        athUsbDbgPrint(ATHUSB_INFO, ("PipeType 0x%x\n", 
                    endpoint[i].bDescriptorType));
        athUsbDbgPrint(ATHUSB_INFO, ("EndpointAddress 0x%x\n", 
                    endpoint[i].bEndpointAddress));
        athUsbDbgPrint(ATHUSB_INFO, ("MaxPacketSize 0x%x\n", 
                    endpoint[i].wMaxPacketSize));
        athUsbDbgPrint(ATHUSB_INFO, ("Interval 0x%x\n", 
                    endpoint[i].bInterval));
    }

    return ntStatus;
}
/***************************************************************************** 
* Routine Description:
*
*    This routine is invoked when the device is removed or stopped.
*    This routine de-configures the usb device.
*
*Arguments:
*
*    pUsbAdapter - Pointer to usb adapter handle
*
*Return Value:
*
*    NT status value
*
*****************************************************************************/
NTSTATUS
athUsbDeconfigureDevice(
    IN PATHUSB_USB_ADAPTER           pUsbAdapter)
{
    NTSTATUS     ntStatus = STATUS_SUCCESS;
#if 0
    
    //
    // initialize variables
    //

    siz = sizeof(struct _URB_SELECT_CONFIGURATION);
    urb = AthExAllocatePool(NonPagedPool, siz);

    if (urb) {

        UsbBuildSelectConfigurationRequest(urb, (USHORT)siz, NULL);

        ntStatus = athUsbCallUSBD(pUsbAdapter, urb);

        if (!NT_SUCCESS(ntStatus)) {

            athUsbDbgPrint(ATHUSB_ERROR, ("Failed to deconfigure device\n"));
        }

        ExFreePool(urb);
    } else {

        athUsbDbgPrint(ATHUSB_ERROR, ("Failed to allocate urb\n"));
        ntStatus = STATUS_INSUFFICIENT_RESOURCES;
    }

#endif
    return ntStatus;
}

/***************************************************************************** 
* Routine Description:
*
*    This routine try to get all string descriptor
*
*Arguments:
*
*    pUsbAdapter - Pointer to usb adapter handle
*
*Return Value:
*
*    NT status value
*
*****************************************************************************/
VOID
athUsbGetAllStringDesciptor(
    IN PATHUSB_USB_ADAPTER           pUsbAdapter)
{    
    PUSB_DEVICE_DESCRIPTOR          pDeviceDescriptor;
    PUSB_CONFIGURATION_DESCRIPTOR   pConfigurationDescriptor;
    PUSB_INTERFACE_DESCRIPTOR       pInterfaceDescriptor;
    
    pDeviceDescriptor = pUsbAdapter->pUsbDeviceDescriptor;
    pConfigurationDescriptor = pUsbAdapter->pUsbConfigurationDescriptor; 
    pInterfaceDescriptor = pUsbAdapter->pUsbInterface;

    if (pDeviceDescriptor->iManufacturer) {
        athUsbGetString(pUsbAdapter,pDeviceDescriptor->iManufacturer);
    }

    if (pDeviceDescriptor->iProduct) {
        athUsbGetString(pUsbAdapter,pDeviceDescriptor->iProduct);
    }

    if (pDeviceDescriptor->iSerialNumber) {
        athUsbGetString(pUsbAdapter,pDeviceDescriptor->iSerialNumber);
    }

    if (pConfigurationDescriptor->iConfiguration) {
        athUsbGetString(pUsbAdapter,pConfigurationDescriptor->iConfiguration);
    }

    if (pInterfaceDescriptor->iInterface) {
        athUsbGetString(pUsbAdapter,pInterfaceDescriptor->iInterface);
    }
}

/***************************************************************************** 
* Routine Description:
*
*    This routine try to get specific string descriptor
*
*Arguments:
*
*    pUsbAdapter - Pointer to usb adapter handle
*    index       - string descriptor index
*
*Return Value:
*
*    NT status value
*
*****************************************************************************/
NTSTATUS
athUsbGetString(
    IN PATHUSB_USB_ADAPTER          pUsbAdapter,
    A_UCHAR                         index)
{    
    UCHAR                 buf[256], *pch;
    A_UINT32              length,i;
    PATHUSB_STRING_DESC   pMyStringDesc;
    BOOLEAN               bFound = FALSE;

    pUsbAdapter->languageNum = 0;
    length = athUsbGetStringDescriptor(pUsbAdapter,0,English,buf, sizeof(buf));
    if (length) {
        pUsbAdapter->languageNum = (length - 2) / 2;
        pch = buf + 2;
        for (i = 0; i < pUsbAdapter->languageNum; i ++) {
            RtlCopyMemory(&pUsbAdapter->languageID[i],pch, 2);
            pch += 2;            
        }
    }

    if (pUsbAdapter->languageNum == 0) {
        return STATUS_UNSUCCESSFUL;    
    }

    pMyStringDesc = &pUsbAdapter->stringDesc[pUsbAdapter->stringDescNum];
    pMyStringDesc->Index = index;

    for (i = 0; i < pUsbAdapter->languageNum; i ++) {
        length = athUsbGetStringDescriptor(pUsbAdapter,
                                           index,
                                           pUsbAdapter->languageID[i],
                                           buf, sizeof(buf));
        if (length == 0) {
            continue;
        }

        bFound = TRUE;

        pMyStringDesc->UnicodeString[i].Length = (USHORT)length;
        pMyStringDesc->UnicodeString[i].Buffer = AthExAllocatePool(PagedPool,
                                    pMyStringDesc->UnicodeString[i].Length+1);

        if (pMyStringDesc->UnicodeString[i].Buffer == NULL) {
            return STATUS_INSUFFICIENT_RESOURCES;
        }

        RtlZeroMemory(pMyStringDesc->UnicodeString[i].Buffer, 
                      pMyStringDesc->UnicodeString[i].Length);

        RtlCopyMemory(pMyStringDesc->UnicodeString[i].Buffer, 
                      &buf[2],length - 2);
    }

    if (bFound) {
        pUsbAdapter->stringDescNum ++;
    }

    return STATUS_SUCCESS;
}
                                    

/***************************************************************************** 
* Routine Description:
*    Gets the specified string descriptor from the given device object
*    
* Arguments:
*    DeviceObject    - pointer to the sample device object
*    Index           - Index of the string descriptor
*    LanguageId      - Language ID of the string descriptor
*    pvOutputBuffer  - pointer to the buffer where the data is to be placed
*    ulLength        - length of the buffer
*
* Return Value:
*    Number of valid bytes in data buffer
******************************************************************************/
A_UINT32
athUsbGetStringDescriptor(
    IN PATHUSB_USB_ADAPTER          pUsbAdapter,
    IN A_UCHAR                      index,
    IN USHORT                       languageId,
    IN OUT PVOID                    pvOutputBuffer,
    IN A_UINT32                     ulLength
    )
{
   NTSTATUS            ntStatus        = STATUS_SUCCESS;
   A_UINT32            length          = 0;

   athUsbDbgPrint(ATHUSB_EXTRA_LOUD, ("Enter AthUsb_GetStringDescriptor\n"));

   if (pvOutputBuffer) {

       ntStatus = usb_get_string(pUsbAdapter->pNextDeviceObject->pUsbDevice,
               languageId, index, pvOutputBuffer, ulLength); 

       // If successful, get the length from the Urb, otherwise set length to 0
       if (ntStatus < 0) {
           athUsbDbgPrint(ATHUSB_ERROR, ("\n Failed to Get stringDescriptor %d!!!\n",index));
           length = 0;
       } else {
           length = ntStatus;
       }
       athUsbDbgPrint(ATHUSB_EXTRA_LOUD, ("%d bytes of string descriptor received\n",length));
   } else {
       ntStatus = STATUS_NO_MEMORY;
   } 
   
   athUsbDbgPrint(ATHUSB_EXTRA_LOUD, ("Leaving Ezusb_GetStringDescriptor\n"));
   return length;
}

/* Linux wrappers for Timers - Not being used anywhere really */

void KeInitializeTimer (PKTIMER pTimerObj)
{
    return;
    init_timer (&pTimerObj->osTimer);
    pTimerObj->timeVal = 0;
}

void KeSetTimer(PKTIMER pTimerObj, LARGE_INTEGER dueTime, void *usbAdptDpc)
{
    unsigned long diffTime;

    return;
    diffTime = ((unsigned long)(-dueTime.LowPart) * HZ) / 1000;

    if (diffTime == 0) {
        diffTime = 5;
    }
    pTimerObj->timeVal = diffTime;
    mod_timer(&pTimerObj->osTimer,jiffies+pTimerObj->timeVal);
}

void KeInitializeDpc(void *pDpc,PFN_USBTIMER_CALLBACK athUsbDpcProc, PATHUSB_USB_ADAPTER pUsbAdapter)
{
    struct timer_list *pOsTimer = &pUsbAdapter->usbAdptTimer.osTimer;
    return;
    pOsTimer->function = athUsbDpcProc;
    pOsTimer->data = (unsigned long)pOsTimer;
    pUsbAdapter->usbAdptTimer.pAthUsbAdapter = pUsbAdapter;
}

int KeCancelTimer(PKTIMER pTimerObj)
{
//    del_timer(&pTimerObj->osTimer);
    return TRUE;
}

NTSTATUS
athUsbQueueWorkItem(IN PATHUSB_USB_ADAPTER     pUsbAdapter)
{
    
    if (!pUsbAdapter->bReset && pUsbAdapter->bStop) {
        //device already stop/remove, so just return
        return STATUS_SUCCESS;
    }    

    if (pUsbAdapter->bQueueWorkItem) {
        //already queue one work item, so just return
        return STATUS_SUCCESS;
    }

    athUsbDbgPrint(ATHUSB_INFO, ("athsUsbQueueWorkItem - begins\n"));

    pUsbAdapter->bQueueWorkItem = TRUE;

    KeClearEvent(&pUsbAdapter->noWorkItemPendingEvent);

    athUsbDbgPrint(ATHUSB_INFO, ("athsUsbQueueWorkItem - ends\n"));

    ATHUSB_SCHEDULE_TASK(&pUsbAdapter->athUsbWorkItem);

    return STATUS_SUCCESS;
}

VOID
athUsbWorkItem(PVOID           pContext)
{
    PATHUSB_USB_ADAPTER            pUsbAdapter = pContext;
    NTSTATUS                       ntStatus, ntStatusWrite;   
    ULONG                          portStatus;
    A_UINT8                        i;
    LARGE_INTEGER                  timeOut;

    athUsbDbgPrint(ATHUSB_INFO, ("athUsbWorkerItemCallback - begins\n"));

    pUsbAdapter->bQueueWorkItem = FALSE;

    if (!pUsbAdapter->bReset && pUsbAdapter->bStop) {
        //Device is gone!
        KeSetEvent(&pUsbAdapter->noWorkItemPendingEvent,
                    IO_NO_INCREMENT,
                    FALSE);
        athUsbDbgPrint(ATHUSB_INFO, ("athUsbWorkerItemCallback - ends\n"));
        return;
    }

    ntStatus = athUsbGetPortStatus(pUsbAdapter, &portStatus);
    if (!(NT_SUCCESS(ntStatus)) || 
        ((portStatus & USBD_PORT_ENABLED) != USBD_PORT_ENABLED) ||
        ((portStatus & USBD_PORT_CONNECTED) != USBD_PORT_CONNECTED))
    {
        //Device is gone!
        athUsbDbgPrint(ATHUSB_INFO, ("athUsbWorkerItemCallback - ends\n"));
        KeSetEvent(&pUsbAdapter->noWorkItemPendingEvent,
                    IO_NO_INCREMENT,
                    FALSE);
        pUsbAdapter->bReset = FALSE;
        pUsbAdapter->busState = ATHUSB_BUS_STATE_FATAL;
        pUsbAdapter->statIndHandler((VOID *)pUsbAdapter->pStubHandle,ATHUSB_BUS_STATE_FATAL);
        return;
    }

    if (pUsbAdapter->bReset) {
        ntStatus = usb_reset_device(pUsbAdapter->pNextDeviceObject->pUsbDevice);
        pUsbAdapter->bReset = FALSE;
    } else if (pUsbAdapter->bDeviceReset) {
        KIRQL                   oldIrql;

        athGetCurrentIrql(oldIrql);


        athAcquireSpinLock(&pUsbAdapter->readQueueLock, &oldIrql);

        if (A_ATOMIC_READ(&pUsbAdapter->readPendingIrps) != 0) {
            KeClearEvent(&pUsbAdapter->noPendingReadIrpEvent);
        }


        athReleaseSpinLock(&pUsbAdapter->readQueueLock, oldIrql);

        athAcquireSpinLock(&pUsbAdapter->writeQueueLock, &oldIrql);

        if (A_ATOMIC_READ(&pUsbAdapter->writePendingIrps) != 0) {
            KeClearEvent(&pUsbAdapter->noPendingWriteIrpEvent);
        }

        athReleaseSpinLock(&pUsbAdapter->writeQueueLock, oldIrql);

#define TIMEOUT_FOR_ENDPOINT    10
        /*
         * Give bus sometime to return the IRP back, if we cancel IRP in the
         * middle of bus return IRP back, we may run into trouble
         */
        timeOut.QuadPart = -(A_INT64) 10l*1000l*TIMEOUT_FOR_ENDPOINT;

        ntStatus = KeWaitForSingleObject(&pUsbAdapter->noPendingReadIrpEvent,
                                Executive,
                                KernelMode,
                                FALSE,
                                &timeOut);

        ntStatusWrite=KeWaitForSingleObject(&pUsbAdapter->noPendingWriteIrpEvent,
                                Executive,
                                KernelMode,
                                FALSE,
                                &timeOut);

        if ((ntStatus == STATUS_TIMEOUT) || (ntStatusWrite==STATUS_TIMEOUT)) {
            athUsbDbgPrint(ATHUSB_INFO, ("We need cancel all IRP by ourselves\n"));
            //cancel and free all revious IRP, you can't send the same IRP down
            //if this IRP failed before.
            athUsbCancelAllIrp(pUsbAdapter, TRUE);
        }
         
        ntStatus = athUsbVendorReset(pUsbAdapter);

        if (NT_SUCCESS(ntStatus)) {
            ntStatus = athUsbResetAllPipes(pUsbAdapter);
        }

        pUsbAdapter->bDeviceReset = FALSE;
    } else {
        A_UINT8                      pipeNum;
        NTSTATUS                     ntStatus;
        PUSBD_INTERFACE_INFORMATION  pInterfaceInfo;

        pInterfaceInfo = pUsbAdapter->pUsbInterface;

        for (i = 0; i < MAX_PIPE_NUM; i ++) {

            if (pUsbAdapter->bResetPipe[0][i]) {

                athUsbCancelIrp(pUsbAdapter,i,TRUE);

                //
                // wait for receive irps to complete.
                //
                KeWaitForSingleObject(&pUsbAdapter->noPendingPipeReadEvent[i],
                                      Executive,
                                      KernelMode,
                                      FALSE,
                                      NULL);

                pipeNum = pUsbAdapter->recvEndPoints[i];
                {
                int pipe;
                struct usb_endpoint_descriptor *endpoint = athUsbGetPipeInformation(pUsbAdapter,pipeNum);

                if (USB_ENDPOINT_DIRECTION_IN(endpoint)) {
                    pipe = usb_rcvbulkpipe(pUsbAdapter->pNextDeviceObject->pUsbDevice,endpoint->bEndpointAddress);
                } else {
                    pipe = usb_sndbulkpipe(pUsbAdapter->pNextDeviceObject->pUsbDevice,endpoint->bEndpointAddress);
                }
                ntStatus = usb_clear_halt(
                        pUsbAdapter->pNextDeviceObject->pUsbDevice,pipe);
                }
    
                if (!NT_SUCCESS(ntStatus)) {
            
                    athUsbDbgPrint(ATHUSB_ERROR, ("athUsbResetPipe failed\n"));

                    ntStatus = usb_reset_device(pUsbAdapter->pNextDeviceObject->pUsbDevice);
                }

                pUsbAdapter->bResetPipe[0][i] = FALSE;
            }

            if (pUsbAdapter->bResetPipe[1][i]) {
                athUsbCancelIrp(pUsbAdapter,i,FALSE);

                //
                // wait for receive irps to complete.
                //
                KeWaitForSingleObject(&pUsbAdapter->noPendingPipeWriteEvent[i],
                                      Executive,
                                      KernelMode,
                                      FALSE,
                                      NULL);
                pipeNum = pUsbAdapter->sendEndPoints[i];
                {
                int pipe;
                struct usb_endpoint_descriptor *endpoint = athUsbGetPipeInformation(pUsbAdapter,pipeNum);

                if (USB_ENDPOINT_DIRECTION_IN(endpoint)) {
                    pipe = usb_rcvbulkpipe(pUsbAdapter->pNextDeviceObject->pUsbDevice,endpoint->bEndpointAddress);
                } else {
                    pipe = usb_sndbulkpipe(pUsbAdapter->pNextDeviceObject->pUsbDevice,endpoint->bEndpointAddress);
                }
                ntStatus = usb_clear_halt(
                        pUsbAdapter->pNextDeviceObject->pUsbDevice,pipe);
                }
                if (!NT_SUCCESS(ntStatus)) {
            
                    athUsbDbgPrint(ATHUSB_ERROR, ("athUsbResetPipe failed\n"));

                    ntStatus = usb_reset_device(pUsbAdapter->pNextDeviceObject->pUsbDevice);
                }

                pUsbAdapter->bResetPipe[1][i] = FALSE;
            }
        }
    }

    pUsbAdapter->bCancel = FALSE;

    if (NT_SUCCESS(ntStatus)) {
        pUsbAdapter->bStop = FALSE;
        pUsbAdapter->busState = ATHUSB_BUS_STATE_NORMAL;
        pUsbAdapter->statIndHandler((VOID *)pUsbAdapter->pStubHandle,ATHUSB_BUS_STATE_NORMAL);
    } else {
        pUsbAdapter->busState = ATHUSB_BUS_STATE_FATAL;
        pUsbAdapter->statIndHandler((VOID *)pUsbAdapter->pStubHandle,ATHUSB_BUS_STATE_FATAL);
    }

    //
    // Cleanup before exiting from the worker thread.
    //

    KeSetEvent(&pUsbAdapter->noWorkItemPendingEvent,
               IO_NO_INCREMENT,
               FALSE);
    athUsbDbgPrint(ATHUSB_INFO, ("athUsbWorkerItemCallback - ends\n"));
}


