/*++

Module Name:

    athfmw.c

Abstract:

    This file contains firmware download

Environment:

    Kernel mode

Notes:

--*/

#ifdef __GNU__
static const char athId[] __attribute__ ((unused)) = "$Id: //depot/sw/branches/1.3_USB_LINUX_port/src/USB/transport/usb/host/linux/athfmw.c#3 $";
#endif /* __GNUC__ */

#include "precomp.h"
#include "athusbapi.h"
#include "athusbdrv.h"
#include "athusbconfig.h"
#include "athusbRxTx.h"
#include "athfmw.h"
#include "ar5523.h"
#include <wlantype.h>
#include <linuxdrv.h>
#include <linuxdefs.h>

#define RETRY_LIMIT             3
#define MAX_WAIT_TIME           5   // 20

//#undef FMDLDATA_MAX_LENGTH
//#define FMDLDATA_MAX_LENGTH     256

VOID
athUsbDownloadFirmware(IN PDEVICE_OBJECT  DeviceObject)
{
    A_STATUS                    status;
    A_UINT32                    totalLength,leftLength,curLength;
    PDEVICE_EXTENSION           deviceExtension;
    PATHUSB_FMDL_OBJECT         firmwareObj;
    PATHUSB_FMDL_MSG            pMsgHeader;
    LARGE_INTEGER               readPos;
    A_UINT32                    retryTime = 0;    
    A_UINT32                    timeOut;
    A_BOOL                      bQuit;
    PUCHAR                      fwData;

    totalLength = AR5005_FWLEN;
    fwData = (PUCHAR)ar5005_fw;

    deviceExtension = &DeviceObject->DeviceExtension;
    firmwareObj = AthExAllocatePool(NonPagedPool, 
                                 sizeof(ATHUSB_FMDL_OBJECT));

    if(firmwareObj == NULL) {
        athUsbDbgPrint(ATHUSB_ERROR, ("firmwareObj allocation failed, no memory\n"));
        return;
    }

    RtlZeroMemory(firmwareObj, sizeof(ATHUSB_FMDL_OBJECT));

    firmwareObj->fmData = AthExAllocatePool(NonPagedPool, 
                                         FMDLDATA_MAX_LENGTH);

    if(firmwareObj->fmData == NULL) {
        athUsbDbgPrint(ATHUSB_ERROR, ("fmData allocation failed, no memory\n"));
        fmwFreeMemory(firmwareObj);
        return;
    }

    firmwareObj->fmInMsg = AthExAllocatePool(NonPagedPool, 
                                          FMDLMSG_MAX_LENGTH);

    if(firmwareObj->fmInMsg == NULL) {
        athUsbDbgPrint(ATHUSB_ERROR, ("fmInMsg allocation failed, no memory\n"));
        fmwFreeMemory(firmwareObj);
        return;
    }

    firmwareObj->fmOutMsg = AthExAllocatePool(NonPagedPool, 
                                           FMDLMSG_MAX_LENGTH);

    if(firmwareObj->fmOutMsg == NULL) {
        athUsbDbgPrint(ATHUSB_ERROR, ("fmOutMsg allocation failed, no memory\n"));
        fmwFreeMemory(firmwareObj);
        return;
    }

    deviceExtension->FirmwareObject = firmwareObj;

    NdisInitializeEvent(&firmwareObj->readEvent);
    NdisInitializeEvent(&firmwareObj->writeEvent);
    NdisInitializeEvent(&firmwareObj->quitEvent);

    /*
     * Initialize the USB device
     */
    status = athUsbDrvInit(1,
                           (void *)DeviceObject,
                           (void *)DeviceObject,
                           fmwDldRecvIndication,
                           fmwDldSendConfirm,
                           fmwDldStatusIndication,
                           &firmwareObj->pUsbAdapt,
                           &firmwareObj->totalInPipe,
                           &firmwareObj->totalOutPipe);

    if(status != A_OK) {
        athUsbDbgPrint(ATHUSB_ERROR, ("athUsbDrvInit failed\n"));
        ExFreePool(firmwareObj);
        return;
    }

    leftLength = totalLength;
    readPos.QuadPart = 0;

    firmwareObj->totalLength = totalLength;
    firmwareObj->bFirst = TRUE;

    while (leftLength) {
        if (leftLength <= FMDLDATA_MAX_LENGTH) {
            curLength = leftLength;
            leftLength = 0;
        } else {
            curLength = FMDLDATA_MAX_LENGTH;
            leftLength -= FMDLDATA_MAX_LENGTH;
        }

        RtlZeroMemory(firmwareObj->fmData, FMDLDATA_MAX_LENGTH);
        RtlCopyMemory(firmwareObj->fmData, fwData+(totalLength - leftLength - curLength), curLength);

        RtlZeroMemory(firmwareObj->fmOutMsg, FMDLMSG_MAX_LENGTH);

        pMsgHeader = (PATHUSB_FMDL_MSG) firmwareObj->fmOutMsg;
        pMsgHeader->msgLength = FMDLMSG_HEADER_LENGTH;
        pMsgHeader->dataLegnth = curLength;
        pMsgHeader->totalLength = totalLength;
        pMsgHeader->leftLength = leftLength;
        
        fmwSwapHeader(pMsgHeader);
        /*
         * Send data information over message pipe
         */
        if (athUsbFirmwareWrite(firmwareObj,MESSAGE_PIPE,firmwareObj->fmOutMsg) == FALSE) {
            athUsbDbgPrint(ATHUSB_ERROR,("Write message failed or driver quit\n"));
            break;
        }        

        /*
         * Read data information back from message pipe, there are two case
         * if the image file is too big, this request will complete immediately
         * since target will tell us not to download, otherwise, this request will
         * be pending until the follow image data packet finish transfer.
         */
        RtlZeroMemory(firmwareObj->fmInMsg, FMDLMSG_MAX_LENGTH);
        if (athUsbDrvReceive(firmwareObj->pUsbAdapt,
                             MESSAGE_PIPE,
                             firmwareObj->fmInMsg) != A_OK)
        {
            athUsbDbgPrint(ATHUSB_ERROR,("Read message failed 1\n"));
            break;        
        }
        
        NdisResetEvent (&firmwareObj->readEvent);
        /*
         * Send data over data pipe
         */
        if (athUsbFirmwareWrite(firmwareObj,DATA_PIPE/*MESSAGE_PIPE*/,firmwareObj->fmData) == FALSE) {
            athUsbDbgPrint(ATHUSB_ERROR,("Write data failed or driver quit\n"));
            break;
        }
        
        /* 
         * Check read back message to determine the status
         */
        timeOut = MAX_WAIT_TIME * HZ;     // 5 second
        
        status = NdisWaitEvent(&firmwareObj->readEvent, (MAX_WAIT_TIME * 1000));

        if (status == 1) {
            /*
             * USB write finished event signaled
             */
            if (firmwareObj->bytesReceived) {
                bQuit = FALSE;
            } else {
                bQuit = TRUE;
            }
        } else {
            /*
             * USB write timeout
             */
            bQuit = TRUE;
        }

        if (bQuit == TRUE) {
            athUsbDbgPrint(ATHUSB_ERROR,("Read message failed 2\n"));
            break; 
        }

#if 0
        if (athUsbFirmwareRead(firmwareObj,MESSAGE_PIPE,firmwareObj->fmInMsg) == FALSE) {
            athUsbDbgPrint(ATHUSB_ERROR,("Write data failed or driver quit\n"));
            break;
        }
#endif

        pMsgHeader = (PATHUSB_FMDL_MSG) firmwareObj->fmInMsg;
        fmwSwapHeader(pMsgHeader);

        if (pMsgHeader->msgLength < FMDLMSG_HEADER_LENGTH) {
            athUsbDbgPrint(ATHUSB_ERROR,("Read back message length is too short\n"));
            break;
        }        

        if ((pMsgHeader->dataLegnth != curLength) ||
            (pMsgHeader->leftLength != leftLength) ||
            (pMsgHeader->totalLength != totalLength))
        {
            if (retryTime > RETRY_LIMIT) {
                athUsbDbgPrint(ATHUSB_ERROR,("Over retry limit, give up downloading now\n"));
                break;
            }

            retryTime ++;

            leftLength = totalLength;
            readPos.QuadPart = 0;
            continue;
        }

        readPos.QuadPart += curLength;
    }
    athUsbDrvExit(1,firmwareObj->pUsbAdapt);    
    fmwFreeMemory(firmwareObj);
    deviceExtension->FirmwareObject = NULL;
}

VOID
fmwFreeMemory(IN PATHUSB_FMDL_OBJECT  firmwareObj)
{
    if (firmwareObj->fmData) {
        ExFreePool(firmwareObj->fmData);
    }

    if (firmwareObj->fmInMsg) {
        ExFreePool(firmwareObj->fmInMsg);
    }

    if (firmwareObj->fmOutMsg) {
        ExFreePool(firmwareObj->fmOutMsg);
    }

    ExFreePool(firmwareObj);
}

VOID
fmwSwapHeader(PATHUSB_FMDL_MSG fwmMsg)
{
    fwmMsg->dataLegnth = A_swab32(fwmMsg->dataLegnth);
    fwmMsg->leftLength = A_swab32(fwmMsg->leftLength);
    fwmMsg->msgLength = A_swab32(fwmMsg->msgLength);
    fwmMsg->totalLength = A_swab32(fwmMsg->totalLength);
}

VOID
athUsbAbortDownload(IN PATHUSB_FMDL_OBJECT         firmwareObj)
{
}

BOOLEAN
athUsbFirmwareWrite(IN PATHUSB_FMDL_OBJECT         firmwareObj,
                    IN A_UINT8                     pipeType,
                    IN A_UINT8                     *pBuf)
{
    A_UINT32                    timeOut;
    NTSTATUS                    status;
    A_BOOL                      bRet = FALSE;

        NdisResetEvent (&firmwareObj->writeEvent);
    if (athUsbDrvSend(firmwareObj->pUsbAdapt,
                      pipeType,
                      pBuf) != A_OK)
    {
        return bRet;
    }

    timeOut = MAX_WAIT_TIME * HZ;     // 20 second
        
    status = NdisWaitEvent(&firmwareObj->writeEvent,(MAX_WAIT_TIME * 1000));

    if (status == 0) {
        /*
         * USB write timeout
         */
        bRet = FALSE;
    } else {
        /*
         * USB write finished event signaled
         */
        if (firmwareObj->bytesSent) {
            bRet = TRUE;
        } else {
            bRet = FALSE;
        }
    }

    return bRet;
}

BOOLEAN
athUsbFirmwareRead(IN PATHUSB_FMDL_OBJECT         firmwareObj,
                   IN A_UINT8                     pipeType,
                   IN A_UINT8                     *pBuf)
{
    A_UINT32                    timeOut;
    NTSTATUS                    status;
    A_BOOL                      bRet = FALSE;

    NdisResetEvent (&firmwareObj->readEvent);
    if (athUsbDrvReceive(firmwareObj->pUsbAdapt,
                         pipeType,
                         pBuf) != A_OK)
    {
        return bRet;
    }

    timeOut= MAX_WAIT_TIME * HZ;     // 5 second
        
    status = NdisWaitEvent(&firmwareObj->readEvent, MAX_WAIT_TIME * 1000);

    if (status == 1) {
        /*
         * USB write finished event signaled
         */
        if (firmwareObj->bytesReceived) {
            bRet = TRUE;
        } else {
            bRet = FALSE;
        }
    } else {
        /*
         * USB write timeout
         */
        bRet = FALSE;
    }

    return bRet;
}

VOID
fmwDldRecvIndication(IN  void     *pStubHandle,
                     IN  A_UINT8   epNum,
                     IN  A_UINT8  *buffer,
                     IN  A_UINT32  bytesReceived)
{
    PDEVICE_OBJECT              pDeviceObject;
    PDEVICE_EXTENSION           deviceExtension;
    PATHUSB_FMDL_OBJECT         firmwareObj;
    A_UINT32                    maxRamSize;

    pDeviceObject = (PDEVICE_OBJECT) pStubHandle;
    deviceExtension = &pDeviceObject->DeviceExtension;
    firmwareObj = (PATHUSB_FMDL_OBJECT) deviceExtension->FirmwareObject;

    /*
     * Signal the read completion event
     */
    firmwareObj->bytesReceived = bytesReceived;

    if (firmwareObj->bFirst) {
        firmwareObj->bFirst = FALSE;

        if (bytesReceived) {
            maxRamSize = *((A_UINT32 *)(firmwareObj->fmInMsg + FMDLMSG_HEADER_LENGTH));
            maxRamSize = A_swab32(maxRamSize);

            if (maxRamSize < firmwareObj->totalLength) {
                athUsbDbgPrint(ATHUSB_ERROR,("Image file size is too large\n"));
                KeSetEvent(&firmwareObj->quitEvent,
                           1,
                           FALSE);
            }
        }
    }

    NdisSetEvent(&firmwareObj->readEvent);
}

VOID
fmwDldSendConfirm(IN  void     *pStubHandle,
                  IN  A_UINT8   epNum,
                  IN  A_UINT8  *buffer,
                  IN  A_UINT32  bytesSent)
{
    PDEVICE_OBJECT              pDeviceObject;
    PDEVICE_EXTENSION           deviceExtension;
    PATHUSB_FMDL_OBJECT         firmwareObj;

    pDeviceObject = (PDEVICE_OBJECT)pStubHandle;
    deviceExtension = &pDeviceObject->DeviceExtension;
    firmwareObj = (PATHUSB_FMDL_OBJECT) deviceExtension->FirmwareObject;

    /*
     * Signal the write completion event
     */
    firmwareObj->bytesSent = bytesSent;

    NdisSetEvent(&firmwareObj->writeEvent);
}

VOID
fmwDldStatusIndication(IN  void     *pStubHandle,                            
                       IN  A_UINT32  status)
{
}


