/*
 * Copyright (c) 2004 Atheros Communications, Inc., All Rights Reserved
 *
 * bin2h.c - Binary File To Header File Conversion
 *
 */

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>

#ifdef __GNU__
static const char athId[] __attribute__ ((unused)) = "$Id: //depot/sw/branches/1.3_USB_LINUX_port/src/USB/transport/usb/host/linux/bin2h.c#1 $";
#endif /* __GNUC__ */

/* Buffer size should be in mupliples of Line length, for proper formated output */
#define LINE_SIZE           (6 * 4)         /* Number of words per line =6 */
#define READ_BUFFER_SIZE    (LINE_SIZE*100) /* Number of lines per read =100*/

/* Error Codes on exit */
#define SUCCESS         0
#define FILE_OPEN       (-1)
#define INVALID_ARG     (-2)

int main (int argc, char *argv[])
{
    unsigned char   ReadBuffer[READ_BUFFER_SIZE];
    unsigned char   writebuf[256];
    unsigned int    TotalReadLength;
    unsigned int    TotalWriteLength;
    ssize_t         ReadLength;
    int             BinFile;
    int             HeaderFile;
    int             i;

    /*
     * Sanity check - We should have both the input binary and output header
     * file names passed as parameters
     * TODO - Auto create Header file name from Binary file name
     */
    if (argc <= 2) {
        printf ("\nUsage : %s FW_BINARY FW_HEADER\n\n", argv[0]);
        exit (INVALID_ARG);
    }


    /*
     * If input binary file cannot be opened for reading, exit with error
     */
    if ((BinFile = open (argv[1], O_RDONLY)) < 0) {
        printf ("Opening Binary File : %s\n", argv[1]);
        perror ("open");
        exit (FILE_OPEN);
    }

    /*
     * If output header file cannot be created for writing, exit with error
     */
    if ((HeaderFile = open (argv[2], O_WRONLY|O_CREAT, S_IRUSR | S_IRGRP | S_IROTH)) < 0) {
        printf ("Opening Header File : %s\n", argv[2]);
        close (BinFile);
        perror ("open");
        exit (FILE_OPEN);
    }

    TotalReadLength = 0;
    TotalWriteLength = 0;

    /* Write copyright information and array start */
    sprintf(writebuf, "/*\n * Copyright (c) 2004 Atheros Communications, Inc., All Rights Reserved\n * \n * Atheros AR5523 USB Station Firmware\n * \n */\n\nunsigned int ar5005_fw[] = {");

    /*
     * We are doing double word writes to the header file, if the file length
     * is not in double word boundary we might end up writing some extra
     * bytes to the header file, which should not be a problem for the f/w
     */
    write (HeaderFile, writebuf, strlen(writebuf));
    while ((ReadLength = read (BinFile, ReadBuffer, READ_BUFFER_SIZE)) > 0) {
        unsigned int *pWrite = (unsigned int *)ReadBuffer;
        for (i = 0; i < ReadLength; i+=4) {
            sprintf (writebuf, "%s0x%08x,", (i % 24) ? "":"\n\t", *pWrite++);
            if (write (HeaderFile, writebuf, strlen(writebuf)) < 0) {
                perror ("write");
                close (BinFile);
                close (HeaderFile);
            }
            TotalWriteLength += 4;
        }
        TotalReadLength += ReadLength;
    }
    /* Close the array in the header file */
    write (HeaderFile, "\n};\n", 4);
    sprintf (writebuf, "\n#define AR5005_FWLEN\t0x%x\n\n", TotalWriteLength);
    write (HeaderFile, writebuf, strlen(writebuf));

    /*
     * Print some read write info for the user to understand that we are
     * doing double word write
     */
    printf ("Read %d bytes, Written %d bytes\n", TotalReadLength, TotalWriteLength);
    close (BinFile);
    close (HeaderFile);
    return (SUCCESS);
}

