/*
 * Copyright (c) 2004 Atheros Communications, Inc., All Rights Reserved
 *
 * athmodule.c -- Atheros USB Driver Module
 * 
 */

#define __ATHUSB_KERNEL_MODULE__
#include <wlantype.h>
#include <linux/usb.h>
#include <linuxdrv.h>
#include <athfmw.h>

#ifdef __GNU__
static const char athId[] __attribute__ ((unused)) = "$Id: //depot/sw/branches/1.3_USB_LINUX_port/src/USB/transport/usb/host/linux/fwmodule.c#6 $";
#endif /* __GNUC__ */

int asserts = 1;
#define DRIVER_DESC     "AR5005 USB WirelessLAN Firmware Download Driver"
#define DRIVER_VERSION  "0.02"

/*
 * List of Devices that work with this driver
 */
static const struct usb_device_id ar5005_idtable [] = {
    { USB_DEVICE(0x0CF3,0x0002) },
    { USB_DEVICE(0x413C,0x8113) },
    { USB_DEVICE(0x2001,0x3A01) },
    { USB_DEVICE(0x2001,0x3A03) },
    { USB_DEVICE(0x07D1,0x3A08) },
    { }
};

/************************************************************************ 
* Routine Description:
*
*    This routine is called by the usb core when a new device supported
*    by this driver is connected
*
*
* Arguments:
*
*    dev - Pointer to the usb device object
*    interface - usb interface number of the attached device
*    id - usb device id entry
*
* Return Value:
*
*    Private Driver object
************************************************************************/

#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0))
static void *ar5005_probe (struct usb_device *dev, unsigned interface, const struct usb_device_id *id)
{
    PDEVICE_OBJECT pDeviceObject;
    int status;

    /*
     * If the module is loaded after device insertion, the device needs
     * some delay for proper operation.  But if the device insertion is
     * later, things work even without delay. I have no scientific reason
     * for this delay.
     */
    mdelay(200);
    pDeviceObject = kmalloc(sizeof(DEVICE_OBJECT), GFP_KERNEL);
    if (pDeviceObject == NULL) {
        return NULL;
    }

    pDeviceObject->pUsbDevice = dev;

    athUsbDownloadFirmware(pDeviceObject);

    return pDeviceObject;
}
#else
static int ar5005_probe (struct usb_interface *interface, const struct usb_device_id *id)
{
    PDEVICE_OBJECT pDeviceObject;

    pDeviceObject = kmalloc(sizeof(DEVICE_OBJECT), GFP_KERNEL);
    if (pDeviceObject == NULL) {
        return -ENOMEM;
    }

    pDeviceObject->pUsbDevice = interface_to_usbdev(interface);
    pDeviceObject->pUsbInterface = interface;
    usb_set_intfdata(interface,pDeviceObject);

    athUsbDownloadFirmware(pDeviceObject);

    return 0;
}
#endif


/************************************************************************ 
* Routine Description:
*
*    This routine is called by the usb core when the device is removed
*    from the system. If the firmware download is successful the device
*    disconnection happens automatically even if it is physically
*    connected
*
*
* Arguments:
*
*    dev - Pointer to the usb device object
*    context - Private Driver Object context
*
* Return Value:
*
*    None
************************************************************************/

#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0))
static void ar5005_disconnect (struct usb_device *dev, void *context)
{
    printk (KERN_INFO "ar5005 disconnect %p\n", context);
    if (context) {
        kfree(context);
    }
}
#else
static void ar5005_disconnect (struct usb_interface *interface)
{
    void *context = usb_get_intfdata(interface);
    printk (KERN_INFO "ar5005 disconnect %p\n", context);
    if (context) {
        kfree(context);
    }
}
#endif

static struct usb_driver ar5005_driver = {
    .name =         "AR5005FW",
    .probe =        ar5005_probe,
    .disconnect =   ar5005_disconnect,
    .id_table =     ar5005_idtable,
};

/************************************************************************ 
* Routine Description:
*
*    This routine is called by the linux driver subsystem when the 
*    module is loaded
*
* Arguments:
*
*    None
*
* Return Value:
*
*    None
************************************************************************/


static int __init ar5005_init (void)
{
    int status;

    printk (KERN_INFO "%s v%s\n", DRIVER_DESC, DRIVER_VERSION);

    /* Register the driver with USB subsystem */
    status = usb_register(&ar5005_driver);
    if (status < 0) {
        printk (KERN_ERR __FILE__ ": USB driver registration error %d\n", status);
        return (-1);
    }

    return (status);
}

static void __exit ar5005_exit (void)
{
    usb_deregister(&ar5005_driver);
}

module_init(ar5005_init);
module_exit(ar5005_exit);

#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0))
EXPORT_NO_SYMBOLS;
#endif
MODULE_AUTHOR("Atheros Communications Inc,");
MODULE_DESCRIPTION(DRIVER_DESC);
MODULE_LICENSE("Proprietary");
MODULE_DEVICE_TABLE(usb,ar5005_idtable);

