#ifdef BOOT_LOADER_PLUS
/*****************************************************************************
**                                                                          **
**                          ATHEROS Communications                          **
**                                                                          **
** **************************************************************************/

/**************************** COPYRIGHT INFORMATION ***************************
**                                                                           **
**   This program contains proprietary information which is a trade          **
**   secret of ATHEROS Communications and also is protected as an            **
**   unpublished work under applicable Copyright laws. Recipient is to       **
**   retain this program in confidence and is not permitted to use or make   **
**   copies thereof other than as permitted in a written agreement with      **
**   ATHEROS Communications.                                                 **
**                                                                           **
** ***************************************************************************/

/******************************************************************************
**                                                                           ** 
** Project       : ATC_FDISK                                                 ** 
**                                                                           ** 
** Module        : FLASH DISK                                                ** 
**                                                                           ** 
** File name     : FlashDisk.c                                               ** 
**                                                                           ** 
** Authors       : Bharathi Thillai-Bharathi.Thillai@wipro.com               ** 
**                 Sajith Kumar M.A-sajith.kumar@wipro.com                   ** 
**                                                                           ** 
** Created On    : Thu June 03 2004                                          ** 
**                                                                           ** 
** Description   : This file contains the USB Mass storage disk Handlers     ** 
**                                                                           ** 
** Reference(s)  :                                                           ** 
**                                                                           ** 
**                                                                           ** 
** ***************************************************************************/

/******************************************************************************
**                                                                           **
** History                                                                   **
** Date          Revision     Comment                                        **
** 03-June-2004      0         Created                                       **
**                                                                           **
**                                                                           **
*******************************************************************************/


/******************************************************************************
**                                                                           **
**   HEADER (INCLUDE) SECTION   (check protection before include)            **
**                                                                           **
** ****************************************************************************/

#include "usb.h"
#include "platform.h"
#include "devapi.h"
#include "athusb.h"
#include "stdlib.h"
#include "string.h"
#include "FlashDisk.h"
#include "BootRom.h"
#include "Flash.h"
#include "EcosDrv.h"
#define BOOTROM_PLUS_VERSION 0x0100
extern void sendBuffer(A_UINT8 *Ptr,A_UINT16 length);
extern A_UINT32 getAvailableHeap( void );


/******************************************************************************
**                                                                           **
**                            MACROS                                         **
**                                                                           **
** ****************************************************************************/


/******************************************************************************
**                                                                           **
**                        External Declaration                               **
**                                                                           **
** ***************************************************************************/

extern volatile uchar      speed;
extern void BootRom(A_UINT8 flashDisk);

/******************************************************************************
**                                                                           **
**                        GLOBAL DATA                                        **
**                                                                           **
******************************************************************************/

A_UINT32  ATC_TOTAL_DISK_SECTORS=(ATC_DISK_CAPACITY/LENGTH_OF_EACH_LAB);
FDISK_CMD_INFO_STRUCT fdiskCmdInfo;
FLASH_PROG_STRUCT flashProg;
DUMMY_DISK_STRUCT dFlashDisk;
TEST_SETUP_STRUCT testSetup;
A_UINT8 TEST_MODE=0;
A_UINT8 specialPurposeCmdAllowed=0;
A_INT8 deviceReady=1;
A_INT8 previousState=0;
/***************************************** 
* Device descriptors are always 18 bytes *
*****************************************/
const A_UINT8  FD_DevDesc[18] =
{
    sizeof(FD_DevDesc),         // Length of FD_DevDesc 
    1,                          // "Device" Type of descriptor 
    0, 2,                       // BCD USB version 
    0,                          // Device Class is indicated in the interface descriptors
    0,                          // Device Subclass is indicated in the interface descriptors 
    0,                          // Mass storage devices do not use class-specific protocols 
    APP_CONTROL_MAX_PKT_SIZE,    // Max packet size 
    USB_uint_16_low(DEFAULT_USB_MASS_VENDORID),    // Vendor ID 
    USB_uint_16_high(DEFAULT_USB_MASS_VENDORID),
    USB_uint_16_low(DEFAULT_USB_MASS_PRODUCTID),  //Product ID 
    USB_uint_16_high(DEFAULT_USB_MASS_PRODUCTID),
    USB_uint_16_low(0x0002),    // BCD Device version 
    USB_uint_16_high(0x0002),
    0x1,                        // Manufacturer string index 
    0x2,                        // Product string index 
    0x3,                        // Serial number string index 
    0x1                         // Number of configurations available 
};


/***************************************** 
* USB 2.0 specific descriptor            *
*****************************************/
const A_UINT8  FD_DevQualifierDesc[10] =
{
    sizeof(FD_DevQualifierDesc),// bLength Length of this descriptor 
    6,                          // bDescType This is a DEVICE Qualifier descr 
    0,2,                        // bcdUSB USB revision 2.0 
    0,                          // bDeviceClass 
    0,                          // bDeviceSubClass 
    0,                          // bDeviceProtocol 
    APP_CONTROL_MAX_PKT_SIZE,   // bMaxPacketSize0 
    0x01,                       // bNumConfigurations
    0
};

A_UINT8 FD_ConfigDesc[FD_CONFIG_DESC_SIZE] =
{

    9,                                           // Configuration Descriptor - always 9 bytes                                        
    2,                                           // "Configuration" type of descriptor                                               
    USB_uint_16_low(FD_CONFIG_DESC_SIZE),        // Total length of the Configuration descriptor                                     
    USB_uint_16_high(FD_CONFIG_DESC_SIZE),                                                                                                
    1,                                           // NumInterfaces                                                                    
    1,                                           // Configuration Value                                                              
    4,                                           // Configuration Description String Index                                           
    0xC0,                                        // Attributes.  Self-powered.                                                       
    0x32,                                        // Current draw from bus                                                            
    9,                                           // Interface 0 Descriptor - always 9 bytes                                          
    4,                                           // "Interface" type of descriptor                                                   
    0,                                           // Number of this interface                                                         
    0,                                           // Alternate Setting                                                                
    MAX_HW_ENDPOINTS,                            // Number of endpoints on this interface                                            
    0x08,                                        // Interface Class                                                                  
    ATC_PROTOCOL,                                // Interface Subclass: SCSI transparent command set                                 
    0x50,                                        // Interface Protocol: Bulk only protocol                                           
    0,                                           // Interface Description String Index                                               
    7,                                           // Endpoint 1 (Bulk In Endpoint), Interface 0 Descriptor - always 7 bytes           
    5,                                           // "Endpoint" type of descriptor                                                    
    0x01,                                        //  Endpoint address..                                                              
    0x02,                                        // Attributes.  0=Control 1=Isochronous 2=Bulk 3=Interrupt                          
    USB_uint_16_low(EP1_FS_MAX_PACKET_SIZE),     // Max Packet Size for this endpoint 
    USB_uint_16_high(EP1_FS_MAX_PACKET_SIZE),
    0,                                           // Polling Interval (ms) 
    7,                                           // Endpoint 2 (Bulk Out Endpoint), Interface 0 Descriptor - always 7 bytes
    5,                                           // Endpoint type of descriptor 
    0x82,                                        // Endpoint address.. 
    0x02,                                        // Attributes.  0=Control 1=Isochronous 2=Bulk 3=Interrupt 
    USB_uint_16_low(EP1_FS_MAX_PACKET_SIZE),     // Max Packet Size for this endpoint 
    USB_uint_16_high(EP1_FS_MAX_PACKET_SIZE),
    0, 
    7,
    5,                                           // Polling Interval (ms) 
    0x83,                                        // Endpoint address..
    0x02,                                        // Attributes.  0=Control 1=Isochronous 2=Bulk 3=Interrupt 
    USB_uint_16_low(EP1_FS_MAX_PACKET_SIZE),     // Max Packet Size for this endpoint 
    USB_uint_16_high(EP1_FS_MAX_PACKET_SIZE),
    0,                                           // Polling Interval (ms) 
    7,                                           // Endpoint 2 (Bulk Out Endpoint), Interface 0 Descriptor - always 7 bytes
    5,                                           // Endpoint type of descriptor 
    0x04,                                        // Endpoint address.. 
    0x02,                                        // Attributes.  0=Control 1=Isochronous 2=Bulk 3=Interrupt 
    USB_uint_16_low(EP1_FS_MAX_PACKET_SIZE),     // Max Packet Size for this endpoint 
    USB_uint_16_high(EP1_FS_MAX_PACKET_SIZE),
    0                                            // Polling Interval (ms) 
};

A_UINT8  FD_OtherSpeed_Config[FD_CONFIG_DESC_SIZE] =
{
    9,                                           // Configuration Descriptor - always 9 bytes                                        
    7,                                           // This is a Other speed config descr 
    USB_uint_16_low(FD_CONFIG_DESC_SIZE),        // Total length of the Configuration descriptor                                     
    USB_uint_16_high(FD_CONFIG_DESC_SIZE),                                                                                                
    1,                                           // NumInterfaces                                                                    
    1,                                           // Configuration Value                                                              
    0,                                           // Configuration Description String Index                                           
    0xC0,                                        // Attributes.  Self-powered.                                                       
    0,                                           // Current draw from bus                                                            
    9,                                           // Interface 0 Descriptor - always 9 bytes                                          
    4,                                           // "Interface" type of descriptor                                                   
    0,                                           // Number of this interface                                                         
    0,                                           // Alternate Setting                                                                
    MAX_HW_ENDPOINTS,                            // Number of endpoints on this interface                                            
    0x08,                                        // Interface Class                                                                  
    ATC_PROTOCOL,                                // Interface Subclass: SCSI transparent command set                                 
    0x50,                                        // Interface Protocol: Bulk only protocol                                           
    0,                                           // Interface Description String Index                                               
    7,                                           // Endpoint 1 (Bulk In Endpoint), Interface 0 Descriptor - always 7 bytes           
    5,                                           // "Endpoint" type of descriptor                                                    
    0x01,                                        //  Endpoint address..                                                              
    0x02,                                        // Attributes.  0=Control 1=Isochronous 2=Bulk 3=Interrupt                          
    USB_uint_16_low(EP1_HS_MAX_PACKET_SIZE),     // Max Packet Size for this endpoint 
    USB_uint_16_high(EP1_HS_MAX_PACKET_SIZE),
    0,                                           // Polling Interval (ms) 
    7,                                           // Endpoint 2 (Bulk Out Endpoint), Interface 0 Descriptor - always 7 bytes
    5,                                           // "Endpoint" type of descriptor 
    0x82,                                        //  Endpoint address.. 
    0x02,                                        // Attributes.  0=Control 1=Isochronous 2=Bulk 3=Interrupt 
    USB_uint_16_low(EP1_HS_MAX_PACKET_SIZE),     // Max Packet Size for this endpoint 
    USB_uint_16_high(EP1_HS_MAX_PACKET_SIZE),
    0,                                           // Polling Interval (ms) 
    7,
    5,                                           // Polling Interval (ms) 
    0x83,                                        //  Endpoint address..
    0x02,                                        // Attributes.  0=Control 1=Isochronous 2=Bulk 3=Interrupt 
    USB_uint_16_low(EP1_HS_MAX_PACKET_SIZE),     // Max Packet Size for this endpoint 
    USB_uint_16_high(EP1_HS_MAX_PACKET_SIZE),
    0,                                           // Polling Interval (ms) 
    7,                                           // Endpoint 2 (Bulk Out Endpoint), Interface 0 Descriptor - always 7 bytes
    5,                                           // "Endpoint" type of descriptor 
    0x04,                                        //  Endpoint address.. 
    0x02,                                        // Attributes.  0=Control 1=Isochronous 2=Bulk 3=Interrupt 
    USB_uint_16_low(EP1_HS_MAX_PACKET_SIZE),     // Max Packet Size for this endpoint 
    USB_uint_16_high(EP1_HS_MAX_PACKET_SIZE),
    0                                            // Polling Interval (ms) 
};

// number of strings in the table not including 0 or n.,if the 
// number of strings changes,look for MS_USB_STR_0 everywhere 
// and make the obvious changes. It should be found in 3 places. 
A_UINT8 MS_USB_STR_0[5]  = {0x5,0x03, 0x09, 0x04,0x0};
A_UINT8 MS_USB_STR_1[46] = {46,0x03,'A',0,'T',0,'H',0,'E',0,'R',0,'O',0,'S',0,' ',0,'C',0,'o',0,'m',0,'m',0,'u',0,'n',0,'i',0,'c',0,'a',0,'t',0,'i',0,'o',0,'n',0,'s',0};
A_UINT8 MS_USB_STR_2[40] = {40,0x03,'A',0,'T',0,'C',0,' ',0,'F',0,'l',0,'a',0,'s',0,'h',0,' ',0,'D',0,'i',0,'s',0,'k',0,'-',0,'W',0,'L',0,'A',0,'N',0};
A_UINT8 MS_USB_STR_3[30] = {30,0x03,'5',0,'9',0,'1',0,'9',0,'7',0,'2',0,'4',0,'5',0,'3',0,'1',0,'2',0,'9',0,'1',0,'2',0};
A_UINT8 MS_USB_STR_4[8]  = {8,0x03,'#',0,'0',0,'2',0};

A_UINT8 MS_USB_STR_n[] =
{
    35,
    0x03,
    'B',
    0,
    'a',
    0,
    'd',
    0,
    ' ',
    0,
    'S',
    0,
    'T',
    0,
    'R',
    ' ',
    0,
    'I',
    'D',
    0
};


uint_8_ptr MS_USB_STRING_DESC[] =
{
    MS_USB_STR_0,
    MS_USB_STR_1,
    MS_USB_STR_2,
    MS_USB_STR_3,
    MS_USB_STR_4,
    MS_USB_STR_n
};

boolean        CBW_PROCESSED = FALSE;
CSW_STRUCT     csw;
A_UINT8        globalDataBuffer[100];

MASS_STORAGE_FORMAT_CAPACITY_STRUCT fDiskFormatInfo =
{
    {
        0,0,0
    },
    8,
    /* Data for the capacity */
    {
        0x00, 0x00,UINT16_HIGH(TOTAL_FDISK_SECTORS),UINT16_LOW(TOTAL_FDISK_SECTORS) 

    }, 
    0x2, /*Formatted Media - Current media capacity*/
    {
        0x00,UINT16_HIGH(LENGTH_OF_EACH_LAB),UINT16_LOW(LENGTH_OF_EACH_LAB)
    }

};

A_UINT8   bootLoaderUpdateStatus=0;
A_UINT16  FDISK_START_SECTOR  =5;
A_UINT16  TOTAL_NO_OF_DISK_SECTORS =32;
A_UINT32  FLASH_SECTOR_SIZE   =0x10000;
A_UINT32  FDISK_OFFSET        =0x50000; 
A_UINT32  FDISK_CONFIG_SECTOR         = 0x3;
A_UINT32  FDISK_PASSWORD_OFFSET       = (0x10000-0x20);
A_UINT32  FDISK_DRIVER_VERSION_OFFSET = (0x10000-0x10);
A_UINT32  FDISK_CONFIG_VALID_OFFSET   = (0x10000-0x2);
A_UINT8   *FDiskBase;
A_UINT8   *FDiskConfigBase;
FLASH_DISK_INFO fDisk ={0,0};

#ifndef ATC_CDROM_DISK
MASS_STORAGE_READ_CAPACITY_STRUCT fDiskCapacity = 
{
    /* Data for the capacity */
    {
        0x00, 0x00,UINT16_HIGH(TOTAL_FDISK_SECTORS),UINT16_LOW(TOTAL_FDISK_SECTORS) 

    }, 
    {
        0x00, 0x00,UINT16_HIGH(LENGTH_OF_EACH_LAB),UINT16_LOW(LENGTH_OF_EACH_LAB)
    }
};
MASS_STORAGE_DEVICE_INFO_STRUCT fDiskDeviceInfo =
{
    //RMB must be set to 0 for Fixed Disk to enable AUTRUN feature
    ATC_FIXED_DISK, 0x0, 0x0, 0x01, 0x1F, {0x0, 0x0, 0x0},   

    /* Vendor information: "ATC     " */
    {'A', 'T', 'H','E','R','O','S', 0x20}, 

    /* Product information: "Wlan Flash Disk " */
    {'W', 'L','A', 'N', ' ', 'F','l' , 'a',

        's', 'h', 0x20, 'D', 'i', 's', 'k',0x20}, 

    /* Product Revision level: "1.0 " */
    {'1', '.','0', 0x20},
}; 

A_UINT8 mode_sense_data[4]={3,0x0,0x80,0x0};
/* Block 0 is the boot sector. Following is the data in the boot sector */
A_UINT8 fDiskBootSector[LENGTH_OF_EACH_LAB] =
{
    0xEB,                                   /* 80x86 "short: jump instruction, indicating that the disk is formatted */
    0x3C,                                   /* 8-bit displacement */
    0x90,                                   /* NOP OPCode */
    0x2C,0x54,0x56,0x6C,0x3C,0x49,0x48,43,  /* 8-bytes for OEM identification: "ATC 1.0 " */
    UINT16_LOW(LENGTH_OF_EACH_LAB),        /* bytes/sector: 512 bytes (0x0200) */
    UINT16_HIGH(LENGTH_OF_EACH_LAB),
    0x01,                                   /* Sectors/allocation unit */
    0x02, 0x00,                             /* Reserved sectors: 0x0002 -One Sector for Boot and another one 
                                               for Saving Disk Related Information like Password and Driver Version Number*/  
    0x02,                                   /* Number of File Allocation Tables (FATs): 2 */
    0x70, 0x00,                             /* Number of root directory entries */
    UINT16_LOW(TOTAL_FDISK_SECTORS),        /* Total sectors in logical volume */ 
    UINT16_HIGH(TOTAL_FDISK_SECTORS),
    0xF8,                                   /* Media descriptor byte: 0xF8: Fixed disk */
    0x08, 0x00,                             /* Sectors/FAT: 4 (Each FAT starts at a new sector) */
    0x01, 0x00,                             /* Sectors/track: 1 */
    0x01, 0x00,                             /* Number of heads */
    0x00, 0x00, 0x00, 0x00,                 /* Number of hidden sectors: 0 */
    0x00, 0x00, 0x00, 0x00,                 /* Total sectors in logical volume */
    0x80,                                   /* Physical drive number */
    0x00,                                   /* Reserved */
    0x29,                                   /* Extended boot signature record: 0x29 */
    0x04, 0x05, 0x09, 0x03,                 /* 32-bit binary volume ID */
    'A', 'T', 'C', '_',                     /* Volume label */
    'F', 'D', 'i', 's', 'k',' ',' ', 
    'F','A','T','1','6',' ',' ',' ',        /* Syetm Identifier */
    0xF1,0x7D,
    0xFA,0x33,0xC9,0x8E,0xD1,0xBC,0xFC,0x7B,0x16,0x07,0xBD,0x78,0x00,0xC5,0x76,0x00,
    0x1E,0x56,0x16,0x55,0xBF,0x22,0x05,0x89,0x7E,0x00,0x89,0x4E,0x02,0xB1,0x0B,0xFC,
    0xF3,0xA4,0x06,0x1F,0xBD,0x00,0x7C,0xC6,0x45,0xFE,0x0F,0x8B,0x46,0x18,0x88,0x45,
    0xF9,0xFB,0x38,0x66,0x24,0x7C,0x04,0xCD,0x13,0x72,0x3C,0x8A,0x46,0x10,0x98,0xF7,
    0x66,0x16,0x03,0x46,0x1C,0x13,0x56,0x1E,0x03,0x46,0x0E,0x13,0xD1,0x50,0x52,0x89,
    0x46,0xFC,0x89,0x56,0xFE,0xB8,0x20,0x00,0x8B,0x76,0x11,0xF7,0xE6,0x8B,0x5E,0x0B,
    0x03,0xC3,0x48,0xF7,0xF3,0x01,0x46,0xFC,0x11,0x4E,0xFE,0x5A,0x58,0xBB,0x00,0x07,
    0x8B,0xFB,0xB1,0x01,0xE8,0x94,0x00,0x72,0x47,0x38,0x2D,0x74,0x19,0xB1,0x0B,0x56,
    0x8B,0x76,0x3E,0xF3,0xA6,0x5E,0x74,0x4A,0x4E,0x74,0x0B,0x03,0xF9,0x83,0xC7,0x15,
    0x3B,0xFB,0x72,0xE5,0xEB,0xD7,0x2B,0xC9,0xB8,0xD8,0x7D,0x87,0x46,0x3E,0x3C,0xD8,
    0x75,0x99,0xBE,0x80,0x7D,0xAC,0x98,0x03,0xF0,0xAC,0x84,0xC0,0x74,0x17,0x3C,0xFF,
    0x74,0x09,0xB4,0x0E,0xBB,0x07,0x00,0xCD,0x10,0xEB,0xEE,0xBE,0x83,0x7D,0xEB,0xE5,
    0xBE,0x81,0x7D,0xEB,0xE0,0x33,0xC0,0xCD,0x16,0x5E,0x1F,0x8F,0x04,0x8F,0x44,0x02,
    0xCD,0x19,0xBE,0x82,0x7D,0x8B,0x7D,0x0F,0x83,0xFF,0x02,0x72,0xC8,0x8B,0xC7,0x48,
    0x48,0x8A,0x4E,0x0D,0xF7,0xE1,0x03,0x46,0xFC,0x13,0x56,0xFE,0xBB,0x00,0x07,0x53,
    0xB1,0x04,0xE8,0x16,0x00,0x5B,0x72,0xC8,0x81,0x3F,0x4D,0x5A,0x75,0xA7,0x81,0xBF,
    0x00,0x02,0x42,0x4A,0x75,0x9F,0xEA,0x00,0x02,0x70,0x00,0x50,0x52,0x51,0x91,0x92,
    0x33,0xD2,0xF7,0x76,0x18,0x91,0xF7,0x76,0x18,0x42,0x87,0xCA,0xF7,0x76,0x1A,0x8A,
    0xF2,0x8A,0x56,0x24,0x8A,0xE8,0xD0,0xCC,0xD0,0xCC,0x0A,0xCC,0xB8,0x01,0x02,0xCD,
    0x13,0x59,0x5A,0x58,0x72,0x09,0x40,0x75,0x01,0x42,0x03,0x5E,0x0B,0xE2,0xCC,0xC3,
    0x03,0x18,0x01,0x27,0x0D,0x0A,0x49,0x6E,0x76,0x61,0x6C,0x69,0x64,0x20,0x73,0x79,
    0x73,0x74,0x65,0x6D,0x20,0x64,0x69,0x73,0x6B,0xFF,0x0D,0x0A,0x44,0x69,0x73,0x6B,
    0x20,0x49,0x2F,0x4F,0x20,0x65,0x72,0x72,0x6F,0x72,0xFF,0x0D,0x0A,0x52,0x65,0x70,
    0x6C,0x61,0x63,0x65,0x20,0x74,0x68,0x65,0x20,0x64,0x69,0x73,0x6B,0x2C,0x20,0x61,
    0x6E,0x64,0x20,0x74,0x68,0x65,0x6E,0x20,0x70,0x72,0x65,0x73,0x73,0x20,0x61,0x6E,
    0x79,0x20,0x6B,0x65,0x79,0x0D,0x0A,0x00,0x49,0x4F,0x20,0x20,0x20,0x20,0x20,0x20,
    0x53,0x59,0x53,0x4D,0x53,0x44,0x4F,0x53,0x20,0x20,0x20,0x53,0x59,0x53,0x80,0x01,
    0x00,0x57,0x49,0x4E,0x42,0x4F,0x4F,0x54,0x20,0x53,0x59,0x53,0x00,0x00,0x55,0xAA,
};
A_UINT8 FAT16_SPECIAL_BYTES[3] = 
{
    0xFD, 0xFF, 0xFF     /* FAT ID: Same as Media descriptor */
};
#else
MASS_STORAGE_DEVICE_INFO_STRUCT fDiskDeviceInfo =
{
    /*RMB must be set to 1 for CDROM*/
    ATC_CDROM, 0x80, 0x0, 0x21, 0x1F, {0x0, 0x0, 0x0},   

    /* Vendor information: "ATC     " */
    {'A', 'T', 'H','E','R','O','S', 0x20}, 

    /* Product information: "Wlan Flash Disk " */
    {'W', 'L','A', 'N', ' ', 'C','D' , 'R',

        'O', 'M', 0x20, 'D', 'i', 's', 'k',0x20}, 

    /* Product Revision level: "1.0 " */
    {'1', '.','0', 0x20}

}; 

MASS_STORAGE_READ_CAPACITY_STRUCT fDiskCapacity = 
{
    /* Data for the capacity */
    {
        0x00, 0x00,UINT16_HIGH(TOTAL_FDISK_SECTORS),UINT16_LOW(TOTAL_FDISK_SECTORS) 

    }, 
    {
        0x00, 0x00,UINT16_HIGH(LENGTH_OF_EACH_LAB),UINT16_LOW(LENGTH_OF_EACH_LAB)
    }
};

//const A_UINT8 ModeSenseDataWithoutBD[]={0x0,0x1A,0x01,0x00,0x00,0x00,0x00,0x00,0x2A,0x12,0x03,0x00,0x75,0x63,0x29,0x02,0x1B,0x90,0x00,0xFF,0x00,0x80,0x1B,0x90,0x00,0x34,0x00,0x00};
const A_UINT8 ModeSenseDataCDROM_MPAGE[]     ={0x0,0x1A,0x70,0x00,0x00,0x00,0x00,0x00,0x2A,0x12,0x03,0x00,0x00,0x01,0x29,0x00,0x1B,0x90,0x00,0x00,0x00,0x40,0x1B,0x90,0x00,0x34,0x00,0x00};
const A_UINT8 ModeSenseDataCDROM_MPAGE_WBD[] ={0x1F,0x71,0x00,0x08,0x01,0x00,0x02,0x00,0x00,0x00,0x08,0x00,0x2A,0x12,0x03,0x00,0x00,0x01,0x29,0x00,0x1B,0x90,0x00,0x00,0x00,0x40,0x1B,0x90,0x00,0x34,0x00,0x00};

const A_UINT8 ModeSenseDataErrorRecoverPage[]={0x0,0x0E,0x70,0x00,0x00,0x00,0x00,0x00,0x01,0x06,0x10,0x03,0x00,0x00,0x00,0x00};
const A_UINT8 ModeSenseNoData[]={0x0,0x06,0x70,0x00,0x00,0x00,0x00,0x00};

const A_UINT8 CD_TOC[] = {0,0xA,1,1,0,0x0,0x0,0x00,0x00,0x00,0x00,00}; 
const A_UINT8 CD_TOC2[] = {0,24,1,1,
    0x01,0x16,0x00,0xA0,0x00,0x00,0x00,0x00,0x0,0x00,0x00,
    0x01,0x16,0x00,0xA1,0x00,0x00,0x00,0x00,0x0,0x00,0x00};
const A_UINT8 senseDataNotPresent[]     = {0xF0,0x00,0x06,0x00,0x00,0x00,0x00,0x06,0x00,0x00,0x00,0x00,0x5A,0x01};
const A_UINT8 senseDataTRAY_OPEN[]      = {0xF0,0x00,0x02,0x00,0x00,0x00,0x00,0x06,0x00,0x00,0x00,0x00,0x3A,0x02};
const A_UINT8 senseDataTRAY_BREADY[]    = {0xF0,0x00,0x02,0x00,0x00,0x00,0x00,0x06,0x00,0x00,0x00,0x00,0x04,0x01};
const A_UINT8 senseDataINVALID_COMMAND[]= {0xF0,0x00,0x05,0x00,0x00,0x00,0x00,0x06,0x00,0x00,0x00,0x00,0x20,0x00};
const A_UINT8 senseDataNoAudioPlay[]={0xF0,0x00,0x05,0x00,0x00,0x00,0x00,0x5,0x00,0x00,0x00,0x00,0x64};
const A_UINT8 noSense[]={0x80,0x00,0x00,0x00,0x00,0x00,0x00,0x06,0x00,0x00,0x00,0x00,0x00,0x00};
const A_UINT8 currentProfile[]={0x00,0x00,0x00,0x04,0x00,0x00,0x00,0x00};
#endif //ATC_CDROM_DISK

A_UINT8 senseKey=2;
const A_UINT8 defaultPassWord[MAX_PASSWORD_SIZE]={"Atheros"};
const A_UINT8 defaultSwVersion[MAX_DRIVER_VERSION_SIZE]={1,0};
const A_UINT8 configValidData[2]={0x55,0xAA};
A_UINT32       flashDiskDebug=(FD_DEBUG|FLASH_DEBUG);
DEV_CONTEXT    devCtxtTimer;
DEV_CONTEXT    devCtxtSpTimer;
A_UINT8 otherCommandsReceived=0;


/******************************************************************************
**                                                                           **
**   LOCAL FUNCTION PROTOTYPES (file scope only)                             **
**                                                                           **
** ***************************************************************************/

/******************************************************************************
*                                                                             * 
* Function Name  : updateCommandStatusSuccess                                 * 
*                                                                             *
* Parameters     :  None                                                      *
*                                                                             *
* Returned Value : None                                                       * 
*                                                                             *
* Comments       : Update Command Status as Success                           * 
*                                                                             * 
******************************************************************************/

void updateCommandStatusSuccess(void)
{
    csw.DCSWDATARESIDUE = 0;
    csw.BCSWSTATUS = 0; //Update command status as Success
}

/******************************************************************************
*                                                                             * 
* Function Name  : sendCommandStatus                                          * 
*                                                                             *
* Parameters     :  handle -handle to the usb Device                          *
*                                                                             *
* Returned Value : None                                                       * 
*                                                                             *
* Comments       : Send the status of command execution                       * 
*                                                                             * 
******************************************************************************/
void sendCommandStatus(USB_HANDLE handle)
{
    CBW_PROCESSED = FALSE;
    FD_DBG_PRINTF(FD_INFO,("CSW1 %X %X %X %X\n\n",csw.DCSWDATARESIDUE,
                           csw.DCSWSIGNATURE,csw.BCSWSTATUS,csw.DCSWTAG));
    ASSERT(sendUsbData(handle, MS_DATA_IN_PIPE, (A_UINT8 *)&csw, 13)==USB_OK);
}

/******************************************************************************
*                                                                             * 
* Function Name  : cleanTestSupport                                           * 
*                                                                             *
* Parameters     :  None                                                      *
*                                                                             *
* Returned Value : None                                                       * 
*                                                                             *
* Comments       : Remove support provided for Testing the commands           * 
*                                                                             * 
******************************************************************************/
void cleanTestSupport(void)
{
    TEST_MODE=0;                    //Inactivate the Test Mode
    if ( testSetup.buffer != NULL ) //If buffer is not 0 release the buffer
    {
        A_DRIVER_FREE(testSetup.buffer,testSetup.bufferSize);
        testSetup.buffer = 0;
        testSetup.bufferSize = 0;
    }
}

#ifdef ATC_CDROM_DISK
/******************************************************************************
*                                                                             * 
* Function Name  : sendCommandFailed                                          * 
*                                                                             *
* Parameters     :  handle -handle to the usb Device                          *
*                   senseInfo-Sense information for the Failure               *
*                                                                             *
* Returned Value : None                                                       * 
*                                                                             *
* Comments       : Send the command failure status to the Host                * 
*                                                                             * 
******************************************************************************/
void sendCommandFailed(USB_HANDLE handle,A_UINT8 senseInfo)
{
    csw.DCSWDATARESIDUE = 0;
    csw.BCSWSTATUS = 1;
    sendCommandStatus(handle);
    FD_DBG_PRINTF(FD_INFO,("Sense Key -%d\n",senseInfo));
    senseKey=senseInfo;
}

/******************************************************************************
*                                                                             * 
* Function Name  : sendCommandFailedWithData                                  * 
*                                                                             *
* Parameters     :  handle -handle to the usb Device                          *
*                   senseInfo-Sense information for the Failure               *
*                                                                             *
* Returned Value : None                                                       * 
*                                                                             *
* Comments       : Send the command failure status to the Host for the        *
*                  command which expect data                                  * 
*                                                                             * 
******************************************************************************/
void sendCommandFailedWithData(USB_HANDLE handle,A_UINT8 senseInfo)
{
    csw.DCSWDATARESIDUE = 0;
    csw.BCSWSTATUS = 1;
    sendUsbData(handle, MS_DATA_IN_PIPE, globalDataBuffer, 0);
    senseKey=senseInfo;
}
#endif


/******************************************************************************
*                                                                             * 
* Function Name  : sendCommandSuccess                                         * 
*                                                                             *
* Parameters     :  handle -handle to the usb Device                          *
*                                                                             *
* Returned Value : None                                                       * 
*                                                                             *
* Comments       : Update and Send the command success to the Host            * 
*                                                                             * 
******************************************************************************/
void sendCommandSuccess(USB_HANDLE handle)
{
    updateCommandStatusSuccess(); //Update the command status as success
    sendCommandStatus(handle);    //Send the command status     
}

/******************************************************************************
*                                                                             * 
* Function Name  : sendSpCommandStatus                                        * 
*                                                                             *
* Parameters     :  handle -Handle to the usb Device                          *
*                   status -Status of the SP command execution                *
*                                                                             *
* Returned Value : None                                                       * 
*                                                                             *
* Comments       : Send the status of Special Purpose command  to the Host    * 
*                                                                             * 
******************************************************************************/
void sendSpCommandStatus(USB_HANDLE handle,A_UINT8 status)
{
    updateCommandStatusSuccess();
    globalDataBuffer[0]=status;
    ASSERT(sendUsbData(handle, MS_DATA_IN_PIPE, (A_UINT8 *)&globalDataBuffer, 1)==USB_OK);
}
/******************************************************************************
*                                                                             * 
* Function Name  : handleErrorCase                                            * 
*                                                                             *
* Parameters     :  cmdInfo    -Pointer to CMD_STRUCT                         *
*                                                                             *
* Returned Value : None                                                       * 
*                                                                             *
* Comments       : Handles the Error cases depends on the command             * 
*                                                                             * 
******************************************************************************/
void handleErrorCase(CMD_STRUCT_PTR cmdInfo)
{
    FD_DBG_PRINTF(FD_INFO,("Error Case is called\n"));
    handleScsiUnsupportedCommand(cmdInfo); //Indicate, this command is not supported
}

/******************************************************************************
*                                                                             * 
* Function Name  : handleMassStorageReset                                     * 
*                                                                             *
* Parameters     :  None                                                      *
*                                                                             *
* Returned Value : None                                                       * 
*                                                                             *
* Comments       : Handles the Mass Storage Reset condition                   * 
*                                                                             * 
******************************************************************************/
void handleMassStorageReset(void)
{
    //Not implememnted  because it never occurred. Also the traget Driver
    //will take care of resettiing the pipes if required
    FLASHDISK_INFO(("MASS Storage Reset is called\n"));
}

/******************************************************************************
*                                                                             * 
* Function Name  : get32BitWordFromBytes                                      * 
*                                                                             *
* Parameters     : ptr-Pointer to unsigned char data                          *
*                                                                             *
* Returned Value : 32 Bit Data after combining all the four bytes             * 
*                                                                             *
* Comments       : String to Long convertor                                   * 
*                                                                             * 
******************************************************************************/
A_UINT32 get32BitWordFromBytes(A_UINT8 *ptr)
{
    A_UINT32 wordData=0;
    wordData  = ((A_UINT32)ptr[0] << 24);
    wordData |= ((A_UINT32)ptr[1] << 16);
    wordData |= ((A_UINT32)ptr[2] << 8);
    wordData |= (A_UINT32)ptr[3];
    return(wordData);
}

/******************************************************************************
*                                                                             * 
* Function Name  : get16BitWordFromBytes                                      * 
*                                                                             *
* Parameters     : ptr-Pointer to unsigned char data                          *
*                                                                             *
* Returned Value : 16 Bit Data after combining the two bytes                  * 
*                                                                             *
* Comments       : String to Short convertor                                  * 
*                                                                             * 
******************************************************************************/
A_UINT16 get16BitWordFromBytes(A_UINT8 *ptr)
{
    A_UINT16 wordData=0;
    wordData |= ((A_UINT16)ptr[0] << 8);
    wordData |= (A_UINT16)ptr[1];
    return(wordData);
}

/******************************************************************************
*                                                                             * 
* Function Name  : sendBuffer                                                 * 
*                                                                             *
* Parameters     : Ptr-Pointer to unsigned char data                          *
*                  length-Number of bytes to send                             *
*                                                                             *
* Returned Value : None                                                       * 
*                                                                             *
* Comments       : Send buffer through serial port                            * 
*                                                                             * 
******************************************************************************/
void sendBuffer(A_UINT8 *Ptr,A_UINT16 length)
{    
    int i,j;
    A_UINT8 array[16];

    FLASHDISK_INFO(("\n"));
    for ( i=0;i<(length/16);i++ )
    {
        FLASHDISK_INFO(("%08X :",Ptr))
        for ( j=0;j<16;j++ )
        {
            FLASHDISK_INFO(("%02X ",Ptr[j]));
        }
        memcpy(array,Ptr,16);
        for ( j=0;j<16;j++ )
        {
            if ( (array[j]<0x20) )
            {
                array[j]='.';
            }
        }
        FLASHDISK_INFO((": %c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c\n",
                        array[0],array[1],array[2],array[3],array[4],array[5],array[6],array[7],array[8],array[9],
                        array[10],array[11],array[12],array[13],array[14],array[15]));
        Ptr+=16;
    }
    if ( (length%16)!=0 )
    {
        FLASHDISK_INFO(("%08X :",Ptr));
        for ( j=0;j<(length%16);j++ )
        {
            FLASHDISK_INFO(("%02X ",Ptr[j]));
        }
    }
}

/******************************************************************************
*                                                                             * 
* Function Name  : getFlashSectorSize                                         * 
*                                                                             *
* Parameters     :  None                                                      *
*                                                                             *
* Returned Value : Return the Flash Sector Size                               * 
*                                                                             *
* Comments       : To retreive the Flash Sector size                          * 
*                                                                             * 
******************************************************************************/
A_UINT32 getFlashSectorSize(void ) //Return the Size of one Sector
{
    return(FLASH_SECTOR_SIZE);

}

/******************************************************************************
*                                                                             * 
* Function Name  : updatePendingFDiskWrite                                    * 
*                                                                             *
* Parameters     : None                                                       *
*                                                                             *
* Returned Value : None                                                       * 
*                                                                             *
* Comments       : To Update the pending Flash Disk Write                     * 
*                                                                             * 
******************************************************************************/
void updatePendingFDiskWrite(void)
{
    A_UINT32 sectorSize=getFlashSectorSize(); 
    if ( dFlashDisk.state == TRUE )  //Check if there is any Flash Write pending 
    {
        FD_DBG_PRINTF(FD_INFO, ("Updating the Flash Disk from Dummy Disk\n"));
        eraseSector(dFlashDisk.flashSector);
        if ( writeFlash((dFlashDisk.flashSector*sectorSize),(A_UINT32 *)dFlashDisk.buffPtr,sectorSize)==TRUE )
        {
            FLASHDISK_INFO(("Sector %d is programmed successfully\n",dFlashDisk.flashSector));
        }
        else
        {
            FLASHDISK_INFO(("Programming Sector %d Failed\n",dFlashDisk.flashSector));

        }
        //Release the buffer allocated for the Flash Write
        A_DRIVER_FREE(dFlashDisk.buffPtr,sectorSize);
        dFlashDisk.buffPtr = NULL;
        dFlashDisk.flashSector = 0xFFF;
        dFlashDisk.state = FALSE;
    }

}

/******************************************************************************
*                                                                             * 
* Function Name  : updateFlashDisk                                            * 
*                                                                             *
* Parameters     : startSector -Starting Sector Number                        *
*                  Offset      -Offset in number of bytes from the sector     *
*                               starting                                      *
*                  dataPtr     -Buffer pointer                                *
*                  dataLength  -Buffer length                                 *
*                                                                             *
* Returned Value : None                                                       * 
*                                                                             *
* Comments       : To Update Flash Sector with the new data                   * 
*                                                                             * 
******************************************************************************/
void updateFlashDisk(A_UINT16 startSector,A_UINT16 Offset,
                     A_UINT8 *dataPtr,A_UINT16 dataLength)
{
    A_UINT8 *ptr;
    A_UINT16 flashSector;
    A_UINT32 sectorSize=getFlashSectorSize();

    flashSector = (startSector);
    if ( dFlashDisk.state == FALSE ) //Is already flash Disk write is active
    {
        //Setup the Dummy Disk
        FD_DBG_PRINTF(FD_INFO, ("Setting up Dummy Disk for Sector %d\n",startSector));
        ptr = A_DRIVER_MALLOC(sectorSize);
        ASSERT(ptr);
        A_DATA_CACHE_INVAL(ptr,sectorSize);
        memcpy(ptr,(A_UINT8 *)(SPI_FLASH_BASE_ADDRESS+(startSector*sectorSize)),sectorSize);
        dFlashDisk.buffPtr = ptr;
        dFlashDisk.flashSector = flashSector;
        dFlashDisk.state = TRUE;
    }
    else
    {
        ptr =dFlashDisk.buffPtr;
    }
    if ( dFlashDisk.flashSector != flashSector ) //Requested sector is not the same sector which is pending for updating 
    {
        //Perform the pending Flash write, before update the data in the Dummy Disk
        eraseSector(dFlashDisk.flashSector);
        if ( writeFlash((dFlashDisk.flashSector*sectorSize),(A_UINT32 *)ptr,sectorSize) ==TRUE )
        {
            FLASHDISK_INFO(("Sector %d is programmed successfully\n",dFlashDisk.flashSector));
        }
        else
        {
            FLASHDISK_INFO(("Programming Sector %d Failed\n",dFlashDisk.flashSector));

        }
        A_DATA_CACHE_INVAL(ptr,sectorSize);
        dFlashDisk.flashSector = flashSector;
        memcpy(ptr,(FDiskBase+(startSector*sectorSize)),sectorSize);
    }
    //Update the data in the Dummy Disk
    memcpy((ptr+Offset),dataPtr,dataLength);
}

/******************************************************************************
*                                                                             * 
* Function Name  : releaseFDISK                                               * 
*                                                                             *
* Parameters     :  None                                                      *
*                                                                             *
* Returned Value :  None                                                      * 
*                                                                             *
* Comments       : Release the memory allocated for  BootLoader Plus          * 
*                                                                             * 
******************************************************************************/
void releaseFDISK(void)
{
    if ( fdiskCmdInfo.rxBuffer != NULL )
    {
        FD_DBG_PRINTF(FD_INFO,("Releasing Memory allocated for Rx Buffer\n"));
        A_DRIVER_FREE(fdiskCmdInfo.rxBuffer,fdiskCmdInfo.rxLength);
        fdiskCmdInfo.rxBuffer=NULL;
        fdiskCmdInfo.rxLength=0;
    }
    if ( fdiskCmdInfo.txBuffer != NULL )
    {
        FD_DBG_PRINTF(FD_INFO,("Releasing Memory allocated for Tx Buffer\n"));
        A_DRIVER_FREE(fdiskCmdInfo.txBuffer,fdiskCmdInfo.txLength);
        fdiskCmdInfo.txBuffer=NULL;
        fdiskCmdInfo.txLength=0;
    }
    //Clean the test Support
    cleanTestSupport();
    //Perform the pending flash Write
    updatePendingFDiskWrite();
}

/******************************************************************************
*                                                                             * 
* Function Name  : resetDriverConfig                                          * 
*                                                                             *
* Parameters     :  resetDisk                                                 *
*                   TRUE -Reset the configuartion with the deafult Driver info*
*                   FALSE-Reset the configuartion with the deafult Driver     *
*                         if not done already                                 *
*                                                                             *
* Returned Value :  None                                                      * 
*                                                                             *
* Comments       : Format the Disk depends on the request                     * 
*                                                                             * 
******************************************************************************/
void resetDriverConfig(A_UINT8 resetDisk)
{
    A_UINT8 *tempBuffer;
    A_UINT32 flashSectorSize;
    //if the configuration is not yet updated
    if ( (FDiskConfigBase[FDISK_CONFIG_VALID_OFFSET] != 0x55) || (FDiskConfigBase[FDISK_CONFIG_VALID_OFFSET+1] != 0xAA) ||(resetDisk == TRUE) )
    {
        FLASHDISK_INFO(("Configuring the Default Disk information\n"));
        flashSectorSize=getFlashSectorSize();
        tempBuffer=(A_UINT8 *)A_DRIVER_MALLOC(flashSectorSize);
        ASSERT(tempBuffer);
        //Save the old configuartion
        memcpy(tempBuffer,(A_UINT8 *)(SPI_FLASH_BASE_ADDRESS+(FDISK_CONFIG_SECTOR*flashSectorSize)),flashSectorSize);
        //Reset the Driver version and Password
        memset((tempBuffer+FDISK_PASSWORD_OFFSET),0,32);
        eraseSector(FDISK_CONFIG_SECTOR);
        //Update with the default Driver version and Default Password
        memcpy((tempBuffer+FDISK_PASSWORD_OFFSET),defaultPassWord, sizeof(defaultPassWord));
        memcpy((tempBuffer+FDISK_DRIVER_VERSION_OFFSET),defaultSwVersion, sizeof(defaultSwVersion));

        //Update the valid configuration Bytes
        memcpy((tempBuffer+FDISK_CONFIG_VALID_OFFSET),configValidData,sizeof(configValidData));
        //Update the configuration sector with the updated configuration
        if ( writeFlash((FDISK_CONFIG_SECTOR*flashSectorSize),(A_UINT32 *)tempBuffer,flashSectorSize)==FALSE )
        {
            FLASHDISK_INFO(("Failed\n"));
        }
        else
        {
            FLASHDISK_INFO(("Success\n"));
        }
        //Release the buffer allocated
        A_DRIVER_FREE(tempBuffer,flashSectorSize);
    }
}

#ifndef ATC_CDROM_DISK
/******************************************************************************
*                                                                             * 
* Function Name  : checkFDisk                                                 * 
*                                                                             *
* Parameters     :  resetDisk                                                 *
*                   TRUE -Format the Disk                                     *
*                   FALSE-Format the Disk if not done already                 *
*                                                                             *
* Returned Value :  None                                                      * 
*                                                                             *
* Comments       : Reset the Driver Configuration depends on the request      * 
*                                                                             * 
******************************************************************************/
void checkFDisk(A_UINT8 resetDisk)
{
    A_UINT8 *tempBuffer;
    A_UINT32 flashSectorSize;
    int i=0;
    int errorCount=0;
    for ( i=0;i<LENGTH_OF_EACH_LAB;i++ )
    {
        if ( FDiskBase[i] != fDiskBootSector[i] )  //Compare with the default Boot Sector
        {
            errorCount++;
            if ( errorCount<10 )
            {
                FD_DBG_PRINTF(FD_MAIN_INFO,("MisMatch @%d A:%X E:%X\n",i,FDiskBase[i],fDiskBootSector[i]));
            }
            else
            {
                break;
            }
        }
    }
    //If Mismatch is more than one or request is forced format the Disk
    if ( (errorCount > 1) | (resetDisk == TRUE) )
    {
#ifdef DEBUG_BOOTROM_PLUS
        sendBuffer(FDiskBase,LENGTH_OF_EACH_LAB);
#endif 
        FLASHDISK_INFO(("Formating the Flash Disk...\n"));
        eraseSector(FDISK_START_SECTOR);
        /* Format the "disk" */
        flashSectorSize=getFlashSectorSize();
        tempBuffer=(A_UINT8 *)A_DRIVER_MALLOC(flashSectorSize);
        ASSERT(tempBuffer);
        memset(tempBuffer,0,flashSectorSize);
        memcpy(tempBuffer,(A_UINT8 *)&fDiskBootSector[0], LENGTH_OF_EACH_LAB);
        memcpy((tempBuffer+2*LENGTH_OF_EACH_LAB),FAT16_SPECIAL_BYTES, 4);
        memcpy((tempBuffer+LENGTH_OF_EACH_LAB*6),FAT16_SPECIAL_BYTES, 4);

        if ( writeFlash(FDISK_OFFSET,(A_UINT32 *)tempBuffer,flashSectorSize)==FALSE )
        {
            FLASHDISK_INFO(("Failed\n"));
        }
        else
        {
            FLASHDISK_INFO(("Success\n"));
        }
        A_DRIVER_FREE(tempBuffer,flashSectorSize);
    }
    resetDriverConfig(resetDisk);
}
#endif 

/******************************************************************************
*                                                                             * 
* Function Name  : stopSpTimer                                                * 
*                                                                             *
* Parameters     :  None                                                      *
*                                                                             *
* Returned Value :  None                                                      * 
*                                                                             *
* Comments       : Stop the Timer                                             * 
*                                                                             * 
******************************************************************************/
void stopSpTimer(void)
{
    FD_DBG_PRINTF(FD_INFO,("Stopping Special Purpose Timer[%d]\n",devCtxtSpTimer.option));
    A_DELETE_TIMER(&devCtxtSpTimer.timer);
    devCtxtSpTimer.option=0; //Reset the timer ID
}
/******************************************************************************
*                                                                             * 
* Function Name  : updateDummyDisk                                            * 
*                                                                             *
* Parameters     :  None                                                      *
*                                                                             *
* Returned Value :  None                                                      * 
*                                                                             *
* Comments       : Update the Data in Dummy Disk before writing to Flash      * 
*                                                                             * 
******************************************************************************/
void updateDummyDisk(void)
{
    A_UINT32 sectorSize;
    A_UINT16 startSector;
    A_UINT16 Offset;
    A_UINT16 dataLength;
    A_UINT8 *ptr;

    sectorSize=getFlashSectorSize();
    startSector=(fdiskCmdInfo.startAddress/sectorSize);
    Offset=(fdiskCmdInfo.startAddress-(startSector*sectorSize));

    ptr = fdiskCmdInfo.rxBuffer;
    if ( TEST_MODE == 0 )
    {
        //If Buffer size crosses the sector size update the Flash before moving to next sector
        if ( (Offset+fdiskCmdInfo.dataLength)>sectorSize )
        {
            dataLength= (sectorSize-Offset);
            FD_DBG_PRINTF(FD_MAIN_INFO,("Update Flash in %d Sector-Length %d\n",startSector,dataLength));
            updateFlashDisk((startSector+FDISK_START_SECTOR),Offset,ptr,dataLength);
            startSector++;
            ptr+=dataLength;
            dataLength=(fdiskCmdInfo.dataLength-dataLength);
            FD_DBG_PRINTF(FD_MAIN_INFO, ("Update Flash in %d Sector-Length %d\n",startSector,dataLength));
            updateFlashDisk(startSector+FDISK_START_SECTOR,0,ptr,dataLength);
        }
        else
        {
            FD_DBG_PRINTF(FD_MAIN_INFO,("Updating Flash %X %X\n",startSector,Offset));
            updateFlashDisk((startSector+FDISK_START_SECTOR),Offset,ptr,fdiskCmdInfo.dataLength);
        }
    }
    else
    {
        ASSERT(testSetup.bufferSize >= fdiskCmdInfo.dataLength);
        memcpy(testSetup.buffer ,fdiskCmdInfo.rxBuffer,fdiskCmdInfo.dataLength);
    }
    A_DRIVER_FREE(fdiskCmdInfo.rxBuffer,fdiskCmdInfo.rxLength);
    fdiskCmdInfo.rxBuffer=NULL;
    fdiskCmdInfo.rxLength=0;
    fdiskCmdInfo.dataLength =0;
    fdiskCmdInfo.dataPtr =0;
    //Start one timer to perform the pending flash write if this is the last write request from the Host 
    startSpTimer(ATC_UPDATE_PENDING_FDISK_WRITE,1000);
}
/******************************************************************************
*                                                                             * 
* Function Name  : spTimerHandler                                             * 
*                                                                             *
* Parameters     :  alarm_handle -Alarm handle                                *
*                   data         -Identifier                                  *
*                                                                             *
* Returned Value :  None                                                      * 
*                                                                             *
* Comments       : Special Purpose Timer Handler                              * 
*                                                                             * 
******************************************************************************/
void spTimerHandler(cyg_handle_t alarm_handle,cyg_addrword_t data)
{
    FD_DBG_PRINTF(FD_INFO,("Special Purpose Timer Expired\n"));
    switch ( devCtxtSpTimer.option ) //Check the timer type
    {
        case ATC_START_BOOTLOADER: //Switch from Mass Storage to BootLoader
            {
                FLASHDISK_INFO(("Switching from Flash Disk to Wlan Boot Loader\n"));
                releaseFDISK();
                BootRom(FALSE); 
            }
            break;

        case ATC_UPDATE_BOOTLOADER: //Update the Boot Loader with the BootLoader received from Host
            {
                updateBootLoader();
            }
            break;
        case ATC_UPDATE_PENDING_FDISK_WRITE: //Update pending flash Write
            {
                FLASHDISK_INFO(("Updating the Pending Flash Disk Operation\n"));
                updatePendingFDiskWrite();
            }
            break;
        case ATC_MASS_STORAGE_SUPPORT_TEST: //Mass Storage support not available in the Host start with BootLoader
            {
                FLASHDISK_INFO(("Mass Storage Support not availble in the Host\nBooting with Wlan Support\n"));
                BootRom(FALSE); //Boot with WLAN Support
            }
            break;
        case ATC_AUTORUN_TIMER: //Used autorun simulation of CD insertion
            {
                if ( previousState==3 )
                {
                    //If CD insertion simulation is over Disk has invalid application or Autorun is disabled by Host[On restart], 
                    //So boot with BootLoader
                    stopSpTimer();
                    FLASHDISK_INFO(("Application is not running in the Host\nBooting with Wlan Support\n"));
                    BootRom(FALSE); //Boot with BootLoader 
                }
                else if ( previousState ==100 )
                {
                    deviceReady=-1;
                    if ( otherCommandsReceived == 0 )
                    {
                        previousState=0;
                    }
                    otherCommandsReceived=0;
                }
                else
                {
                    deviceReady=-1; //Move on to next stage of CD insertion simulation
                }
            }
            break;

        case ATC_FORMAT_FDISK: 
            {
#ifndef ATC_CDROM_DISK
                checkFDisk(TRUE);  //Format the Disk
#else
                resetDriverConfig(TRUE);  //Reset the Configuaration
#endif

            }
            break;
        case ATC_EXECUTE_BOOTLOADER: //Execute the Firmware downloaded
            {
                unsigned long       oldints;
                FLASHDISK_INFO(("Executing BootLoader from @ 0x%08x\n\n",flashProg.dataPtr));
                HAL_DISABLE_INTERRUPTS(oldints);
                HAL_DCACHE_SYNC();
                HAL_ICACHE_DISABLE();
                HAL_DCACHE_DISABLE();
                HAL_DCACHE_SYNC();
                HAL_ICACHE_INVALIDATE_ALL();
                HAL_DCACHE_INVALIDATE_ALL();
                ((void (*)(void))flashProg.dataPtr)();
            }
            break;
        default:
            FD_DBG_PRINTF(FD_ERROR,("Invalid Timer Epired\n"));
            break;
    }
    stopSpTimer();
}

/******************************************************************************
*                                                                             * 
* Function Name  : startSpTimer                                               * 
*                                                                             *
* Parameters     :  option-Timer ID                                           *
*                   timeout-Time Out                                          *
*                                                                             *
* Returned Value :  None                                                      * 
*                                                                             *
* Comments       : Start one Timer                                            * 
*                                                                             * 
******************************************************************************/

void startSpTimer(A_UINT16 option,A_UINT16 timeout)
{
    if ( devCtxtSpTimer.option != 0 )
    {
        stopSpTimer();
        FLASHDISK_INFO(("Some timer got reused\n"));
    }
    devCtxtSpTimer.option=option;
    FD_DBG_PRINTF(FD_INFO,("Starting Special Purpose Timer[%d]-Timeout-%d\n",devCtxtSpTimer.option,timeout));
    cyg_clock_to_counter(cyg_real_time_clock(),&devCtxtSpTimer.rtcCounterHandle);
    A_INIT_TIMER(&devCtxtSpTimer, &devCtxtSpTimer.timer,spTimerHandler,&devCtxtSpTimer);
    A_TIMEOUT(&devCtxtSpTimer.timer,timeout, FALSE);

}

/******************************************************************************
*                                                                             * 
 * Function Name  : sendData                                                  * 
*                                                                             *
* Parameters     :  handle     -Usb handler                                   *
*                   bufferPtr  -Buffer Pointer                                *
*                   dataLength -Buffer Length                                 *
*                   requestLength-Buffer Length allocated by the Host         *
* Returned Value :  None                                                      * 
*                                                                             *
* Comments       : Send data through USB                                      * 
*                                                                             * 
******************************************************************************/

void sendData(USB_HANDLE handle,A_UINT8 *bufferPtr, A_UINT32 dataLength,A_UINT32 requestLength)
{
    csw.DCSWDATARESIDUE = 0;
    csw.BCSWSTATUS = 0;
    if ( requestLength < dataLength )
    {
        FD_DBG_PRINTF(FD_MAIN_INFO,("Actual Length- %d Available buffer Length-%d\n",dataLength,requestLength));
        ASSERT(sendUsbData(handle,MS_DATA_IN_PIPE,bufferPtr,requestLength)==USB_OK);
    }
    else
    {
        ASSERT(sendUsbData(handle,MS_DATA_IN_PIPE,bufferPtr,dataLength)==USB_OK);
    }
}


/******************************************************************************
*                                                                             * 
* Function Name  : updateDiskConfiguration                                    * 
*                                                                             *
* Parameters     : deviceSize -Flash Size                                     *
*                                                                             *
* Returned Value :  None                                                      * 
*                                                                             *
* Comments       : Update the Flash size related Parameters                   * 
*                                                                             * 
******************************************************************************/
void updateDiskConfiguration(A_UINT32 deviceSize)
{
    ATC_TOTAL_DISK_SECTORS=(deviceSize/LENGTH_OF_EACH_LAB);
    //Update the Format Capacity fields 
    fDiskFormatInfo.NUMBER_OF_BLOCKS[0] = 0;
    fDiskFormatInfo.NUMBER_OF_BLOCKS[1] = 0;
    fDiskFormatInfo.NUMBER_OF_BLOCKS[2]=UINT16_HIGH(ATC_TOTAL_DISK_SECTORS); 
    fDiskFormatInfo.NUMBER_OF_BLOCKS[3]=UINT16_LOW(ATC_TOTAL_DISK_SECTORS);

    //Update the Inquiry data fields 
    fDiskCapacity.LAST_LOGICAL_BLOCK_ADDRESS[0] = 0;
    fDiskCapacity.LAST_LOGICAL_BLOCK_ADDRESS[1] = 0;
    fDiskCapacity.LAST_LOGICAL_BLOCK_ADDRESS[2] = UINT16_HIGH(ATC_TOTAL_DISK_SECTORS); 
    fDiskCapacity.LAST_LOGICAL_BLOCK_ADDRESS[3] = UINT16_LOW(ATC_TOTAL_DISK_SECTORS);
    //Update the Boot Sector
#ifndef ATC_CDROM_DISK
    fDiskBootSector[0x13]=UINT16_LOW(ATC_TOTAL_DISK_SECTORS); 
    fDiskBootSector[0x14]=UINT16_HIGH(ATC_TOTAL_DISK_SECTORS);
#endif
}

/******************************************************************************
*                                                                             * 
* Function Name  : updateDiskInfo                                             * 
*                                                                             *
* Parameters     : None                                                       *
*                                                                             *
* Returned Value : None                                                       * 
*                                                                             *
* Comments       : Identify the Flash size and update related Parameters      * 
*                                                                             * 
******************************************************************************/
void updateDiskInfo(void)
{
    FLASH_INFO_STRUCT *flashInfo;
    //To provide flexible Disk size depends on the Serial Flash Availble in the Board
    flashInfo=getDeviceInfo();
    
    switch ( flashInfo->DeviceMID )
    { 
        case NEX_MANUFACTURER:  //STM and NEX programming interface are same
        case STM_MANUFACTURER:
            {
                //print the device manufacturer ID         
                if(flashInfo->DeviceMID==NEX_MANUFACTURER)
                {
                   FLASHDISK_INFO(("NexFlash "));
                }
                else
                {
                   FLASHDISK_INFO(("ST Microelectronics "));
                }
                
                
                
                if ( (flashInfo->DeviceType == 0x20) && (flashInfo->DeviceSize == 0x14) )
                {
                    FLASHDISK_INFO(("Serial Flash of size 1MB Found\n")); 
                    TOTAL_NO_OF_DISK_SECTORS =16;       //Total Number of sectors in the Serial Flash
                }
                else if ( (flashInfo->DeviceType == 0x20) && (flashInfo->DeviceSize == 0x15) )
                {
                    FLASHDISK_INFO(("Serial Flash of size 2MB Found\n")); 
                    TOTAL_NO_OF_DISK_SECTORS =32;       //Total Number of sectors in the Serial Flash
                }
                else if ( (flashInfo->DeviceType == 0x20) && (flashInfo->DeviceSize == 0x16) )
                {
                    FLASHDISK_INFO(("Serial Flash of size 4MB Found\n")); 
                    TOTAL_NO_OF_DISK_SECTORS =64;       //Total Number of sectors in the Serial Flash
                }
                
                else
                {
                    FLASHDISK_INFO(("Unsupported Serial Flash Found please upgrade the Firmware to support the same\n")); 
                    ASSERT(0);
                   
                }
                /* First 5 sectors are reserved for Boot Loader and other configurations.*/
                FDISK_START_SECTOR  =5;             //Flash Disk Start Sector 
                FLASH_SECTOR_SIZE   =0x10000;       //Flash Sector size
                FDISK_OFFSET        =(FDISK_START_SECTOR*FLASH_SECTOR_SIZE); //Starting offset in Flash where DISK content start
                FDiskBase=(A_UINT8 *)(SPI_FLASH_BASE_ADDRESS+FDISK_OFFSET);  //Base address of Disk
                //Configuration Base address
                FDiskConfigBase = (A_UINT8 *)(SPI_FLASH_BASE_ADDRESS+(FDISK_CONFIG_SECTOR*FLASH_SECTOR_SIZE));
                //Update other Disk related information
                updateDiskConfiguration((TOTAL_NO_OF_DISK_SECTORS-FDISK_START_SECTOR)*FLASH_SECTOR_SIZE);  
            }
            break;
            
                                
        default:
            FLASHDISK_INFO(("Unsupported Serial Flash Found please upgrade the Firmware to support the same\n")); 
            ASSERT(0);
            break;                   
    }

}
#ifdef ATC_CDROM_DISK
/******************************************************************************
*                                                                             * 
* Function Name  : handleScsiInquiry                                          * 
*                                                                             *
* Parameters     :  cmdInfo    -Pointer to CMD_STRUCT                         *
*                                                                             *
* Returned Value : None                                                       * 
*                                                                             *
* Comments       : Responds appropriately to Request Sense command            * 
*                                                                             * 
******************************************************************************/

void handleRequestSense(CMD_STRUCT_PTR cmdInfo)
{
    CBW_STRUCT_PTR  cbwPtr=cmdInfo->cmdPtr;

    if ( cbwPtr->BMCBWFLAGS & USB_CBW_DIRECTION_BIT )
    {
        switch ( senseKey )
        {
            case 2: //Notify  Device Not ready because of eject is in progress
                sendData(cmdInfo->handle, (A_UINT8 *)&senseDataTRAY_OPEN ,sizeof(senseDataTRAY_OPEN),cbwPtr->DCBWDATALENGTH);
                break;
            case 3:  //Notify  Device Not ready because of Tray is Open
                sendData(cmdInfo->handle, (A_UINT8 *)&senseDataNotPresent ,sizeof(senseDataNotPresent),cbwPtr->DCBWDATALENGTH);
                break;
            case 4:  //Notify  Device Not ready but it becoming ready
                sendData(cmdInfo->handle, (A_UINT8 *)&senseDataTRAY_BREADY ,sizeof(senseDataTRAY_BREADY),cbwPtr->DCBWDATALENGTH);
                break;
            case 5:  
                sendData(cmdInfo->handle, (A_UINT8 *)&senseDataNoAudioPlay ,sizeof(senseDataNoAudioPlay),cbwPtr->DCBWDATALENGTH);
                break;
            case 6:  //Notify  this command is not supported
                sendData(cmdInfo->handle, (A_UINT8 *)&senseDataINVALID_COMMAND ,sizeof(senseDataINVALID_COMMAND),cbwPtr->DCBWDATALENGTH);
                break;
            default:
                FD_DBG_PRINTF(FD_MAIN_INFO,("No Sense\n"));
                sendData(cmdInfo->handle, (A_UINT8 *)&noSense ,sizeof(noSense),cbwPtr->DCBWDATALENGTH);
                break;

        }
        senseKey=0;
    }
    else
    {
        handleErrorCase(cmdInfo); //Invalid option requested inform command failure
    }

}
/******************************************************************************
*                                                                             * 
* Function Name  : handleScsiInquiry                                          * 
*                                                                             *
* Parameters     :  cmdInfo    -Pointer to CMD_STRUCT                         *
*                                                                             *
* Returned Value : None                                                       * 
*                                                                             *
* Comments       : Responds appropriately to Event Status Notification command* 
*                                                                             * 
******************************************************************************/
void handleEventStatusNotification(CMD_STRUCT_PTR cmdInfo) //Only require in XP
{
    CBW_STRUCT_PTR  cbwPtr=cmdInfo->cmdPtr;
    if ( (cbwPtr->BMCBWFLAGS & USB_CBW_DIRECTION_BIT) )
    {
        fdiskCmdInfo.txLength=cbwPtr->DCBWDATALENGTH;
        fdiskCmdInfo.txBuffer=A_DRIVER_MALLOC(fdiskCmdInfo.txLength);  
        memset(fdiskCmdInfo.txBuffer,0,fdiskCmdInfo.txLength);         //Not implemented zero's will be sent
        sendData(cmdInfo->handle, fdiskCmdInfo.txBuffer, fdiskCmdInfo.txLength,cbwPtr->DCBWDATALENGTH);
    }
    else
    {
        handleErrorCase(cmdInfo); //Invalid option requested inform command failure
    } 

}
#endif 
/******************************************************************************
*                                                                             * 
* Function Name  : handleScsiInquiry                                          * 
*                                                                             *
* Parameters     :  cmdInfo    -Pointer to CMD_STRUCT                         *
*                                                                             *
* Returned Value : None                                                       * 
*                                                                             *
* Comments       : Process a Mass storage class Inquiry command               * 
*                                                                             * 
******************************************************************************/
void handleScsiInquiry(CMD_STRUCT_PTR cmdInfo)
{
    CBW_STRUCT_PTR  cbwPtr=cmdInfo->cmdPtr;
    MASS_STORAGE_INQUIRY_PTR   inquiry_cmd_ptr =(MASS_STORAGE_INQUIRY_PTR)((pointer)cbwPtr->CBWCB);
    if ( cbwPtr->BMCBWFLAGS & USB_CBW_DIRECTION_BIT )
    {
        /* Send the device information */
        sendData(cmdInfo->handle,(A_UINT8 *)&fDiskDeviceInfo,
                 sizeof(fDiskDeviceInfo),inquiry_cmd_ptr->ALLOCATION_LENGTH);
    }
    else
    {
        handleErrorCase(cmdInfo); //Invalid option requested inform command failure
    }

} 

/******************************************************************************
*                                                                             * 
* Function Name  : handleScsiUnsupportedCommand                               * 
*                                                                             *
* Parameters     : cmdInfo    -Pointer to CMD_STRUCT                          *
*                                                                             *
* Returned Value : None                                                       * 
*                                                                             *
* Description    : Responds appropriately to unsupported commands             * 
*                                                                             * 
*                                                                             * 
******************************************************************************/
void handleScsiUnsupportedCommand(CMD_STRUCT_PTR cmdInfo)
{ 
    CBW_STRUCT_PTR  cbwPtr=cmdInfo->cmdPtr;

    if ( (cbwPtr->BMCBWFLAGS & USB_CBW_DIRECTION_BIT) )
    {
        /* Send a zero-length packet */
        if ( cbwPtr->DCBWDATALENGTH != 0 )
        {
            sendCommandFailedWithData(cmdInfo->handle,6);
        }
        else
        {
            sendCommandFailed(cmdInfo->handle,6);
        }
    }
    else
    {
        /* Send the command status information */
        csw.DCSWDATARESIDUE = 0;
        csw.BCSWSTATUS = 1;
        fdiskCmdInfo.rxLength=cbwPtr->DCBWDATALENGTH;
        if ( fdiskCmdInfo.rxLength != 0 )
        {
            FD_DBG_PRINTF(FD_MAIN_INFO,("Unsupported command-%Xh\nReceiving Buffer of size %d and ignore it\n",cbwPtr->CBWCB[0],fdiskCmdInfo.rxLength));
            fdiskCmdInfo.rxBuffer=A_DRIVER_MALLOC(fdiskCmdInfo.rxLength);
            receiveUsbData(cmdInfo->handle, MS_DATA_OUT_PIPE, fdiskCmdInfo.rxBuffer,fdiskCmdInfo.rxLength);
        }
        else
        {
            FD_DBG_PRINTF(FD_MAIN_INFO,("Unsupported command-%Xh\nSending Failure\n",cbwPtr->CBWCB[0]));
            sendCommandStatus(cmdInfo->handle);
        }
    } 

} 

/******************************************************************************
*                                                                             * 
* Function Name  : handleScsiReportCapacity                                   * 
*                                                                             *
* Parameters     : cmdInfo    -Pointer to CMD_STRUCT_PTR                      *
*                                                                             *
* Returned Value : None                                                       * 
*                                                                             *
* Description    : Reports the media capacity as a response to READ CAPACITY  *
*                  Command.                                                   * 
*                                                                             * 
******************************************************************************/
void handleScsiReportCapacity(CMD_STRUCT_PTR cmdInfo)
{ 
    CBW_STRUCT_PTR  cbwPtr=cmdInfo->cmdPtr;

    FD_DBG_PRINTF(FD_INFO,("Process Read Capacity[%X-%X]\n",
                           cbwPtr->BMCBWFLAGS,
                           USB_CBW_DIRECTION_BIT));
    if ( cbwPtr->BMCBWFLAGS & USB_CBW_DIRECTION_BIT )
    {
        if ( previousState < 3 )
        {
            sendCommandFailedWithData(cmdInfo->handle,0x2+previousState);
        }
        else
        {
            sendData(cmdInfo->handle,(A_UINT8 *)&fDiskCapacity, 8,cbwPtr->DCBWDATALENGTH);
        }
    }
    else
    {
        handleErrorCase(cmdInfo); //Invalid option requested inform command failure
    }

} 
#ifdef ATC_CDROM_DISK
void handleScsiGetConfig(CMD_STRUCT_PTR cmdInfo)
{ 
    CBW_STRUCT_PTR  cbwPtr=cmdInfo->cmdPtr;

    FD_DBG_PRINTF(FD_INFO,("Process Read Capacity[%X-%X]\n",
                           cbwPtr->BMCBWFLAGS,
                           USB_CBW_DIRECTION_BIT));

    if ( cbwPtr->BMCBWFLAGS & USB_CBW_DIRECTION_BIT )
    {
        if ( senseKey == 0 )
        {
            sendData(cmdInfo->handle,(A_UINT8 *)&currentProfile,
                     sizeof(currentProfile),cbwPtr->DCBWDATALENGTH);
        }
        else
        {
            sendCommandFailedWithData(cmdInfo->handle,0x2);
        }
    }
    else
    {
        handleErrorCase(cmdInfo); //Invalid option requested inform command failure
    }

} 


/******************************************************************************
*                                                                             * 
* Function Name  : handleScsiRead                                             * 
*                                                                             *
* Parameters     :  cmdInfo    -Pointer to CMD_STRUCT                         *
*                                                                             *
* Returned Value : None                                                       * 
*                                                                             *
* Description    : Responds appropriately to Play Audio Command               * 
*                                                                             * 
*                                                                             * 
******************************************************************************/


void handlePlayAudio(CMD_STRUCT_PTR cmdInfo)
{
    CBW_STRUCT_PTR  cbwPtr=cmdInfo->cmdPtr;

    if ( cbwPtr->BMCBWFLAGS & USB_CBW_DIRECTION_BIT )
    {
        handleErrorCase(cmdInfo); //Invalid option requested inform command failure      
    }
    else
    {
        sendCommandFailed(cmdInfo->handle,6); //Inform Command not supported
    }

}
#endif 


/******************************************************************************
*                                                                             * 
* Function Name  : handleScsiModeSense                                        * 
*                                                                             *
* Parameters     :  cmdInfo    -Pointer to CMD_STRUCT                         *
*                                                                             *
* Returned Value : None                                                       * 
*                                                                             *
* Description    : Sends data as a response to Mode Sense Command.            * 
*                                                                             * 
*******************************************************************************/
void handleScsiModeSense(CMD_STRUCT_PTR cmdInfo)
{ 
    CBW_STRUCT_PTR  cbwPtr=cmdInfo->cmdPtr;
    A_UINT8 *senseDataPtr=0;
    A_UINT8 length;


    FD_DBG_PRINTF(FD_INFO,("Process Mode Sense[%X-%X]\n",
                           cbwPtr->BMCBWFLAGS,USB_CBW_DIRECTION_BIT));
#ifndef ATC_CDROM_DISK
    if ( fDisk.writeEnable==TRUE )
    {
        mode_sense_data[2]=0x0;
    }
    else
    {
        mode_sense_data[2]=0x80;
    }
    senseDataPtr=mode_sense_data;
    length=sizeof(mode_sense_data);
#else
    switch ( cbwPtr->CBWCB[2] )
    {
        
        case 0x2A: //CDROM Capability  list
            {
                if ( cbwPtr->CBWCB[1] == 8 )
                {
                    senseDataPtr=(A_UINT8 *)ModeSenseDataCDROM_MPAGE;
                    length=sizeof(ModeSenseDataCDROM_MPAGE);
                }
                else
                {
                    senseDataPtr=(A_UINT8 *)ModeSenseDataCDROM_MPAGE;
                    length=sizeof(ModeSenseDataCDROM_MPAGE);

                }
            }
            break;
        case 0x01: //Error recovery page
            senseDataPtr=(A_UINT8 *)ModeSenseDataErrorRecoverPage;
            length=sizeof(ModeSenseDataErrorRecoverPage);
            break;
        default:     //Default page
            senseDataPtr=(A_UINT8 *)ModeSenseNoData;
            length=sizeof(ModeSenseNoData);
            break;

    }
#endif
    if ( cbwPtr->BMCBWFLAGS & USB_CBW_DIRECTION_BIT )
    {
        sendData(cmdInfo->handle,senseDataPtr,length,cbwPtr->DCBWDATALENGTH);
    }
    else
    {
        handleErrorCase(cmdInfo); //Invalid option requested inform command failure
    }
} 
/******************************************************************************
*                                                                             * 
* Function Name  : handleScsiReadFormatCapacity                               * 
*                                                                             *
* Parameters     :  cmdInfo    -Pointer to CMD_STRUCT                         *
*                                                                             *
* Returned Value : None                                                       * 
*                                                                             *
* Description    : Sends data as a response to READ Format Capacity           * 
*                                                                             * 
******************************************************************************/

void handleScsiReadFormatCapacity(CMD_STRUCT_PTR cmdInfo)
{ 

    CBW_STRUCT_PTR  cbwPtr=cmdInfo->cmdPtr;
    FD_DBG_PRINTF(FD_INFO,("Process Format Capacity[%X-%X]\n",
                           cbwPtr->BMCBWFLAGS,USB_CBW_DIRECTION_BIT));
    if ( cbwPtr->BMCBWFLAGS & USB_CBW_DIRECTION_BIT )
    {
        sendData(cmdInfo->handle,  (A_UINT8 *)&fDiskFormatInfo,
                 sizeof(fDiskFormatInfo),cbwPtr->DCBWDATALENGTH);
    }
    else
    {
        handleErrorCase(cmdInfo); //Invalid option requested inform command failure
    }

} 

/******************************************************************************
*                                                                             * 
* Function Name  : handleScsiVerifyCommand                                             * 
*                                                                             *
* Parameters     :  cmdInfo    -Pointer to CMD_STRUCT                         *
*                                                                             *
* Returned Value : None                                                       * 
*                                                                             *
* Description    : Responds appropriately to Verify Command                   * 
*                                                                             * 
*                                                                             * 
******************************************************************************/

void handleScsiVerifyCommand(CMD_STRUCT_PTR cmdInfo)
{
    CBW_STRUCT_PTR  cbwPtr=cmdInfo->cmdPtr;
    /*Since Data will not be corrupted in the Disk if we don't erase it,
     This command always send Success*/
    if ( cbwPtr->BMCBWFLAGS & USB_CBW_DIRECTION_BIT )
    {
        handleErrorCase(cmdInfo); //Invalid option requested inform command failure
    }
    else
    {
        sendCommandStatus(cmdInfo->handle);
    }
}
/******************************************************************************
*                                                                             * 
* Function Name  : handleScsiRead                                             * 
*                                                                             *
* Parameters     :  cmdInfo    -Pointer to CMD_STRUCT                         *
*                                                                             *
* Returned Value : None                                                       * 
*                                                                             *
* Description    : Sends data as a response to READ Command.                  * 
*                                                                             * 
*                                                                             * 
******************************************************************************/
void handleScsiRead(CMD_STRUCT_PTR cmdInfo)
{
    CBW_STRUCT_PTR  cbwPtr=cmdInfo->cmdPtr;
    uint_32 offset = 0, length = 0;
    A_UINT8 *ptr;

    updatePendingFDiskWrite();
    if ( cbwPtr->BMCBWFLAGS & USB_CBW_DIRECTION_BIT )
    {
        offset  = get16BitWordFromBytes(&cbwPtr->CBWCB[4]);
        length  = get16BitWordFromBytes(&cbwPtr->CBWCB[7]);
        length  = ( length *LENGTH_OF_EACH_LAB);
        offset  = (offset*LENGTH_OF_EACH_LAB);
        if ( TEST_MODE == 0 )
        {
            if ( fdiskCmdInfo.txBuffer == NULL )
            {
                fdiskCmdInfo.txBuffer = A_DRIVER_MALLOC(length);
                ASSERT(fdiskCmdInfo.txBuffer);
            }
            else
            {
                A_DRIVER_FREE(fdiskCmdInfo.txBuffer,fdiskCmdInfo.txLength);
                fdiskCmdInfo.txBuffer = A_DRIVER_MALLOC(length);
                ASSERT(fdiskCmdInfo.txBuffer);
            }
            fdiskCmdInfo.txLength = length;
            A_DATA_CACHE_INVAL(fdiskCmdInfo.txBuffer,length);
            ptr=fdiskCmdInfo.txBuffer;
            memcpy(ptr,(FDiskBase + offset),length);
            sendData(cmdInfo->handle,  ptr, length,cbwPtr->DCBWDATALENGTH);
        }
        else
        {
            ASSERT(testSetup.bufferSize >= length);
            sendData(cmdInfo->handle,  testSetup.buffer, length,cbwPtr->DCBWDATALENGTH);
        }
    }
    else
    {
        handleErrorCase(cmdInfo); //Invalid option requested inform command failure
    }
} 

/******************************************************************************
*                                                                             *
* Function Name  : handleScsiWrite                                            *
*                                                                             *
* Parameters     :  cmdInfo    -Pointer to CMD_STRUCT                         *
*                                                                             *
* Returned Value : None                                                       *
*                                                                             *
* Description    : Sends data as a response to WRITE Command.                 *
*                                                                             *
*                                                                             *
*******************************************************************************/
void handleScsiWrite(CMD_STRUCT_PTR cmdInfo)
{
    uint_32 index1 = 0, index2 = 0;
    A_UINT8 *ptr;
    A_UINT32 length=0;
    A_UINT32 availableRamSpace;
    CBW_STRUCT_PTR  cbwPtr=cmdInfo->cmdPtr;
    if ( (cbwPtr->BMCBWFLAGS & USB_CBW_DIRECTION_BIT) == 0 )
    {
        index1  = get16BitWordFromBytes(&cbwPtr->CBWCB[4]);
        index2  = get16BitWordFromBytes(&cbwPtr->CBWCB[7]);
        fdiskCmdInfo.startAddress=(index1*LENGTH_OF_EACH_LAB);
        length=(LENGTH_OF_EACH_LAB * index2);
        availableRamSpace=(A_UINT32)getAvailableHeap();
        FD_DBG_PRINTF(FD_MAIN_INFO,("Write @%d %d[Length-%d,Available Memory-%d]\n",
                                    index1,index2,length,availableRamSpace));
        if ( fdiskCmdInfo.rxBuffer == NULL )
        {
            fdiskCmdInfo.rxBuffer = A_DRIVER_MALLOC(length);
            ASSERT(fdiskCmdInfo.rxBuffer);
        }
        else
        {
            A_DRIVER_FREE(fdiskCmdInfo.rxBuffer,fdiskCmdInfo.rxLength);
            fdiskCmdInfo.rxBuffer = A_DRIVER_MALLOC(length);
            ASSERT(fdiskCmdInfo.rxBuffer);
        }
        fdiskCmdInfo.rxLength = length;

        ptr=fdiskCmdInfo.rxBuffer;
        if ( getUsbTransferStatus(cmdInfo->handle,MS_DATA_IN_PIPE,USB_RECV) != USB_OK )
        {
            FD_DBG_PRINTF(FD_INFO, ("Write Failed[Transfer status not OK\n"));
            cancelUsbTransfer(cmdInfo->handle,MS_DATA_IN_PIPE,USB_RECV);
        }
        fdiskCmdInfo.cmdState =  WRITE_FLASH_DISK_STATE;
        receiveUsbData(cmdInfo->handle, MS_DATA_OUT_PIPE, ptr,length);
        updateCommandStatusSuccess();
        if ( devCtxtSpTimer.option== ATC_UPDATE_PENDING_FDISK_WRITE )
        {
            stopSpTimer();
        }
    }
    else
    {
        handleErrorCase(cmdInfo); //Invalid option requested inform command failure
    }

} 

/******************************************************************************
*                                                                             *
* Function Name  : handleScsiTestUnitReady                                    *
*                                                                             *
* Parameters     :  cmdInfo    -Pointer to CMD_STRUCT                         *
*                                                                             *
* Returned Value : None                                                       *
* Description    :  Responds appropriately to unit ready query                *
*                                                                             *
*******************************************************************************/
void handleScsiTestUnitReady(CMD_STRUCT_PTR cmdInfo)
{
    CBW_STRUCT_PTR  cbwPtr=cmdInfo->cmdPtr;
    if ( cbwPtr->BMCBWFLAGS & USB_CBW_DIRECTION_BIT )
    {
        handleErrorCase(cmdInfo); //Invalid option requested inform command failure
    }
    else
    {
#ifdef ATC_CDROM_DISK
        if ( previousState < 3 )
        {
            sendCommandFailed(cmdInfo->handle,0x2+previousState);
        }
        else
        {
            sendCommandSuccess(cmdInfo->handle);
        }
        if ( devCtxtSpTimer.option == ATC_MASS_STORAGE_SUPPORT_TEST )
        {
            previousState = 100;
            stopSpTimer();
            startSpTimer(ATC_AUTORUN_TIMER,300);
            otherCommandsReceived=0;
            FLASHDISK_INFO(("Mass Storage Support available in the HOST\n"));
        }
        if ( deviceReady == -1 )
        {
            if ( previousState != 100 )
            {
                stopSpTimer();
                previousState ++;
                deviceReady=1;
                if ( previousState == 3 )
                {
                    startSpTimer(ATC_AUTORUN_TIMER,2000);
                }
                else
                {
                    startSpTimer(ATC_AUTORUN_TIMER,400);
                }
            }
            else
            {
                stopSpTimer();
                deviceReady=1;
                startSpTimer(ATC_AUTORUN_TIMER,200);
            }
        }
#else
        sendCommandSuccess(cmdInfo->handle);
#endif
    }
} 

/******************************************************************************
*                                                                             *   
* Function Name  : handleScsiPreventAllowMediumRemoval                        *   
*                                                                             *
* Parameters     :  cmdInfo    -Pointer to CMD_STRUCT                         *
*                                                                             *
* Returned Value : None                                                       *   
* Description    : Responds appropriately to Prevenet Allow Medium Removal    *   
*                                                                             *   
*******************************************************************************/
void handleScsiPreventAllowMediumRemoval(CMD_STRUCT_PTR cmdInfo)
{ 
    CBW_STRUCT_PTR  cbwPtr=cmdInfo->cmdPtr;
    if ( cbwPtr->BMCBWFLAGS & USB_CBW_DIRECTION_BIT )
    {
        handleErrorCase(cmdInfo); //Invalid option requested inform command failure
    }
    else
    {
        sendCommandSuccess(cmdInfo->handle);
    }
} 

/******************************************************************************
*                                                                             *   
* Function Name  : stringCompare                                              *   
*                                                                             *
* Parameters     :  str1    -First String                                     *
*                   str2    -Second String                                    *
*                   length  -Maximum Length                                   *
*                                                                             *
* Returned Value : Return Maximum Length if both strings matches              *   
*                  else number of matches                                     *
* Description    : Comapre Strings                                            *   
*                                                                             *   
*******************************************************************************/

A_UINT8 stringCompare(A_UINT8 *str1,A_UINT8 *str2,A_UINT8 length)
{
    int i;
    for ( i=0;i<length;i++ )
    {
        if ( str1[i] != str2[i] )
        {
            return(i);
        }
        if ( str1[i] == 0 )
        {
            return(length);
        }

    }
    return(i);
}
/******************************************************************************
*                                                                             *   
* Function Name  : setPassword                                                *   
*                                                                             *
* Parameters     :  None                                                      *
*                                                                             *
* Returned Value : None                                                       *   
*                                                                             *
* Description    : Change Password                                            *   
*                                                                             *   
*******************************************************************************/
void setPassword(void)
{
    if ( stringCompare(&globalDataBuffer[0],&FDiskConfigBase[FDISK_PASSWORD_OFFSET],MAX_PASSWORD_SIZE)==12 )
    { //Verify current password
        if ( TEST_MODE == 0 )
        {
            updateFlashDisk(FDISK_CONFIG_SECTOR,FDISK_PASSWORD_OFFSET,&globalDataBuffer[MAX_PASSWORD_SIZE],MAX_PASSWORD_SIZE);
            updatePendingFDiskWrite();
        }
        else
        {
            ASSERT(testSetup.bufferSize >= MAX_PASSWORD_SIZE);
            memcpy(testSetup.buffer,&globalDataBuffer[MAX_PASSWORD_SIZE],MAX_PASSWORD_SIZE);
        }
    }
    else
    {
        FD_DBG_PRINTF(FD_MAIN_INFO,("Current password is incorrect\n"));
    }
}
void activateSpCommand(CMD_STRUCT_PTR cmdInfo)
{
    CBW_STRUCT_PTR  cbwPtr=cmdInfo->cmdPtr;
    A_UINT8 *passPtr;

    if ( cbwPtr->BMCBWFLAGS & USB_CBW_DIRECTION_BIT )
    {
        //Verify current password
        if ( TEST_MODE == 0 )
        {
            passPtr =&FDiskConfigBase[FDISK_PASSWORD_OFFSET];
        }
        else
        {
            passPtr = globalDataBuffer;
        }
        if ( stringCompare(&cbwPtr->CBWCB[5],&FDiskConfigBase[FDISK_PASSWORD_OFFSET],MAX_PASSWORD_SIZE)==12 )
        {
            specialPurposeCmdAllowed=1;
            FLASHDISK_INFO(("Special Purpose commands can be accessed\n"));
            sendSpCommandStatus(cmdInfo->handle,FDISK_CMD_SUCCESS);

        }
        else
        {
            specialPurposeCmdAllowed=0;
            FLASHDISK_INFO(("Invalid Password Please Contact Admin\n"));
            sendSpCommandStatus(cmdInfo->handle,FDISK_CMD_FAILED_INVALID_PASSWORD);
        }
    }
    else
    {
        handleErrorCase(cmdInfo); //Invalid option requested inform command failure
    }
}
/******************************************************************************
*                                                                             *   
* Function Name  : switchToBootLoader                                         *   
*                                                                             *
* Parameters     :  cmdInfo    -Pointer to CMD_STRUCT                         *
*                                                                             *
* Returned Value : None                                                       *   
*                                                                             *
* Description    : Swicth From CDROM functionality to BootLoader              *    
*                                                                             *   
*******************************************************************************/

void switchToBootLoader(CMD_STRUCT_PTR cmdInfo)
{
    CBW_STRUCT_PTR  cbwPtr=cmdInfo->cmdPtr;
    int timeOut;
    timeOut=get16BitWordFromBytes(&cbwPtr->CBWCB[5]);
    if ( timeOut != 0 ) //If time ourt is no zero
    {
        if ( devCtxtSpTimer.option != ATC_START_BOOTLOADER )
        {
            sendSpCommandStatus(cmdInfo->handle,FDISK_CMD_SUCCESS);
            FLASHDISK_INFO(("Switching to Boot Loader [Time out - %d]\n",timeOut));
            startSpTimer(ATC_START_BOOTLOADER,timeOut);
        }
        else
        {
            sendSpCommandStatus(cmdInfo->handle,FDISK_CMD_FAILED_CMD_ALREADY_ACTIVE);
        }
    }
    else
    {
        FLASHDISK_INFO(("Switching to Boot Loader immediately\n"));
        sendSpCommandStatus(cmdInfo->handle,FDISK_CMD_SUCCESS);
        startSpTimer(ATC_START_BOOTLOADER,5);
    }
}


/******************************************************************************
*                                                                             *   
* Function Name  : getSwVersion                                               *   
*                                                                             *
* Parameters     :  cmdInfo    -Pointer to CMD_STRUCT                         *
*                                                                             *
* Returned Value : None                                                       *   
*                                                                             *
* Description    : To Retreive Driver Version                                 *    
*                                                                             *   
*******************************************************************************/
void getSwVersion(CMD_STRUCT_PTR cmdInfo)
{
    CBW_STRUCT_PTR  cbwPtr=cmdInfo->cmdPtr;
    FD_DBG_PRINTF(FD_INFO, ("Sending Version\n"));
    if ( cbwPtr->BMCBWFLAGS & USB_CBW_DIRECTION_BIT )
    {
        if ( TEST_MODE == 0 )
        {
            memcpy(globalDataBuffer,&FDiskConfigBase[FDISK_DRIVER_VERSION_OFFSET],MAX_DRIVER_VERSION_SIZE);
            sendData(cmdInfo->handle,  globalDataBuffer,MAX_DRIVER_VERSION_SIZE,cbwPtr->DCBWDATALENGTH);
        }
        else
        {
            memcpy(globalDataBuffer,testSetup.buffer,MAX_DRIVER_VERSION_SIZE);
            sendBuffer(globalDataBuffer,2);
            sendData(cmdInfo->handle,  globalDataBuffer,MAX_DRIVER_VERSION_SIZE,cbwPtr->DCBWDATALENGTH);
        }
    }
    else
    {
        handleErrorCase(cmdInfo); //Invalid option requested inform command failure
    }
}

/******************************************************************************
*                                                                             *   
* Function Name  : handleSetPasswordCommand                                   *   
*                                                                             *
* Parameters     :  cmdInfo    -Pointer to CMD_STRUCT                         *
*                                                                             *
* Returned Value : None                                                       *   
*                                                                             *
* Description    : Handler for Change Password Command                        *    
*                                                                             *   
*******************************************************************************/

void handleSetPasswordCommand(CMD_STRUCT_PTR cmdInfo)
{
    CBW_STRUCT_PTR  cbwPtr=cmdInfo->cmdPtr;
    FD_DBG_PRINTF(FD_MAIN_INFO,("Set password\n"));
    if ( (cbwPtr->BMCBWFLAGS & USB_CBW_DIRECTION_BIT) )
    {
        handleErrorCase(cmdInfo); //Invalid option requested inform command failure
    }
    else
    {
        fdiskCmdInfo.cmdState =  SET_PASSWORD_STATE;
        receiveUsbData(cmdInfo->handle, MS_DATA_OUT_PIPE,globalDataBuffer,MAX_PASSWORD_SIZE*2);
        updateCommandStatusSuccess();
    }
}
/******************************************************************************
*                                                                             *   
* Function Name  : setSwVersion                                               *   
*                                                                             *
* Parameters     :  cmdInfo    -Pointer to CMD_STRUCT                         *
*                                                                             *
* Returned Value : None                                                       *   
*                                                                             *
* Description    : Handler to Change the Driver Version                       *    
*                                                                             *   
*******************************************************************************/
void setSwVersion(CMD_STRUCT_PTR cmdInfo)
{
    CBW_STRUCT_PTR  cbwPtr=cmdInfo->cmdPtr;
    A_UINT8 *swVer=&cbwPtr->CBWCB[5];
    if ( cbwPtr->BMCBWFLAGS & USB_CBW_DIRECTION_BIT )
    {
        if ( (swVer[0] == 0) && (swVer[1] == 0) )
        {
            sendSpCommandStatus(cmdInfo->handle,FDISK_CMD_FAILED_INVALID_VERSION);
            return;
        }
        if ( TEST_MODE == 0 )
        {
            updateFlashDisk(FDISK_CONFIG_SECTOR,FDISK_DRIVER_VERSION_OFFSET,swVer,MAX_DRIVER_VERSION_SIZE);
            startSpTimer(ATC_UPDATE_PENDING_FDISK_WRITE,10);
        }
        else
        {
            ASSERT(testSetup.bufferSize >= MAX_DRIVER_VERSION_SIZE);
            memcpy(testSetup.buffer,swVer,MAX_DRIVER_VERSION_SIZE);
            sendBuffer(testSetup.buffer,2);
        }
        sendSpCommandStatus(cmdInfo->handle,FDISK_CMD_SUCCESS);
    }
    else
    {
        handleErrorCase(cmdInfo); //Invalid option requested inform command failure
    }

}

/******************************************************************************
*                                                                             *   
* Function Name  : setupTestSupport                                           *   
*                                                                             *
* Parameters     :  bufferSize  -Buffer Size required                         *
*                                                                             *
* Returned Value : None                                                       *   
*                                                                             *
* Description    : Setup Test Mode                                            *    
*                                                                             *   
*******************************************************************************/
void setupTestSupport(A_UINT32 bufferSize)
{
    TEST_MODE =1;
    ASSERT(testSetup.buffer==NULL);
    if ( testSetup.buffer == NULL )
    {
        testSetup.buffer=A_DRIVER_MALLOC(bufferSize);
        testSetup.bufferSize = bufferSize;
    }

}

/******************************************************************************
*                                                                             *   
* Function Name  : handleTestMode                                             *   
*                                                                             *
* Parameters     :  cmdInfo    -Pointer to CMD_STRUCT                         *
*                                                                             *
* Returned Value : None                                                       *   
*                                                                             *
* Description    : Handler for test Mode Command                              *    
*                                                                             *   
*******************************************************************************/
void handleTestMode(CMD_STRUCT_PTR cmdInfo)
{
    A_UINT32 bufferSize;
    CBW_STRUCT_PTR  cbwPtr=cmdInfo->cmdPtr;
    A_UINT8 testMode=cbwPtr->CBWCB[5];
    FD_DBG_PRINTF(FD_MAIN_INFO,("Test Mode Request ID-%d[Previous status-%d]\n",testMode,TEST_MODE));
    if ( cbwPtr->BMCBWFLAGS & USB_CBW_DIRECTION_BIT )
    {
        switch ( testMode )
        {
            case 1:  //Enble Test Mode
                if ( (TEST_MODE == 0) )
                {
                    bufferSize=get32BitWordFromBytes(&cbwPtr->CBWCB[6]);
                    FD_DBG_PRINTF(FD_MAIN_INFO,("Test Mode ACTIVE-Buffer Size-%d\n",bufferSize));
                    if ( bufferSize > getAvailableHeap() )
                    {
                        sendSpCommandStatus(cmdInfo->handle,ATC_TEST_MODE_UNABALE_TO_ALLOCATE);
                        return;
                    }
                    sendSpCommandStatus(cmdInfo->handle,ATC_TEST_MODE_ACTIVE);
                    setupTestSupport(bufferSize);
                }
                else
                {
                    sendSpCommandStatus(cmdInfo->handle,ATC_TEST_MODE_ALREADY_ACTIVE);
                }
                break;
            case 2: //Disable Test Mode
                if ( (TEST_MODE == 1) )
                {
                    FD_DBG_PRINTF(FD_MAIN_INFO,("Test Mode INACTIVE\n"));
                    sendSpCommandStatus(cmdInfo->handle,ATC_TEST_MODE_INACTIVE);
                    cleanTestSupport();
                }
                else
                {
                    sendSpCommandStatus(cmdInfo->handle,ATC_TEST_MODE_ALREADY_INACTIVE);
                }
                break;
            default:
                sendSpCommandStatus(cmdInfo->handle,ATC_TEST_MODE_UNSUPPORTED);
                break;
        }
    }
    else
    {
        handleErrorCase(cmdInfo); //Invalid option requested inform command failure

    }
}
/******************************************************************************
*                                                                             *   
* Function Name  : resetFDISK_Config                                          *   
*                                                                             *
* Parameters     :  cmdInfo    -Pointer to CMD_STRUCT                         *
*                                                                             *
* Returned Value : None                                                       *   
*                                                                             *
* Description    : Handler for reseting FDISK Config                          *    
*                                                                             *   
*******************************************************************************/

void resetFDISK_Config(CMD_STRUCT_PTR cmdInfo)
{
    CBW_STRUCT_PTR  cbwPtr=cmdInfo->cmdPtr;
    if ( cbwPtr->BMCBWFLAGS & USB_CBW_DIRECTION_BIT )
    {
        handleErrorCase(cmdInfo); //Invalid option requested inform command failure
    }
    else
    {
        sendCommandSuccess(cmdInfo->handle);
        startSpTimer(ATC_FORMAT_FDISK,20);
    }

}

/******************************************************************************
*                                                                             *   
* Function Name  : acknowledgeApplication                                     *   
*                                                                             *
* Parameters     :  cmdInfo    -Pointer to CMD_STRUCT                         *
*                                                                             *
* Returned Value : None                                                       *   
*                                                                             *
* Description    : Handler for Application Sync Command                       *    
*                                                                             *   
*******************************************************************************/
void acknowledgeApplication(CMD_STRUCT_PTR cmdInfo)
{

    CBW_STRUCT_PTR  cbwPtr=cmdInfo->cmdPtr;
    if ( cbwPtr->BMCBWFLAGS & USB_CBW_DIRECTION_BIT )
    {
        sendSpCommandStatus(cmdInfo->handle,FDISK_CMD_SUCCESS);
        if ( devCtxtSpTimer.option== ATC_AUTORUN_TIMER )
        {
            FLASHDISK_INFO( ("Application is running in the Host\n"));
            previousState = 3; 
            stopSpTimer();
        }
    }
    else
    {
        handleErrorCase(cmdInfo); //Invalid option requested inform command failure
    }

}

/******************************************************************************
*                                                                             *   
* Function Name  : acknowledgeApplication                                     *   
*                                                                             *
* Parameters     :  cmdInfo    -Pointer to CMD_STRUCT                         *
*                                                                             *
* Returned Value : None                                                       *   
*                                                                             *
* Description    : Handler for SPecial Purpose Command                        *    
*                                                                             *   
*******************************************************************************/
void handleSpecialPurposeCommand(CMD_STRUCT_PTR cmdInfo)
{   
    A_UINT32 availableRamSpace;
    CBW_STRUCT_PTR  cbwPtr=cmdInfo->cmdPtr;
   
    if ( stringCompare(&cbwPtr->CBWCB[1],(A_UINT8 *)"ATC",3) ==3 )
    {
        if ( specialPurposeCmdAllowed == 1 )
        {
            switch ( cbwPtr->CBWCB[4] )
            {
                
                case ATC_ACIVATE_SP_CMD:
                    {
                        activateSpCommand(cmdInfo);
                    }
                    break;
                case ATC_APPLICATION_NOTIFICATION:
                    {
                        acknowledgeApplication(cmdInfo);
                    }
                    break;
                case ATC_SWITCH_TO_BOOTLOADER:
                    { 
                        switchToBootLoader(cmdInfo);
                    }
                    break;
                case ATC_ENABLE_FLASH_WRITE:
                    {
                        availableRamSpace=(A_UINT32)getAvailableHeap();
                        FLASHDISK_INFO(("Enabling Flash Write[Memory availble-%d]\n",availableRamSpace));
                        /* Send the command status information */
                        if ( fDisk.writeEnable==TRUE )
                        {
                            updatePendingFDiskWrite ();
                            sendSpCommandStatus(cmdInfo->handle,FDISK_CMD_SUCCESS);
                        }
                        else
                        {
                            if ( availableRamSpace >=0x20000 )
                            {
                                sendSpCommandStatus(cmdInfo->handle,FDISK_CMD_SUCCESS);
                                fDisk.writeEnable=TRUE;
                            }
                            else
                            {
                                sendSpCommandStatus(cmdInfo->handle,FDISK_CMD_FAILED_INSUFFICIENT_MEMORY);
                            }
                        }
                        return;
                    }
                    break;
                case ATC_SET_PASSWORD:
                    {
                        handleSetPasswordCommand(cmdInfo);
                    }
                    break;
                case ATC_GET_VERSION:
                    {
                        getSwVersion(cmdInfo);
                    }
                    break;
                case ATC_SET_VERSION:
                    {
                        setSwVersion(cmdInfo);
                    }
                    break;
                case ATC_SET_TEST_MODE:
                    {
                        handleTestMode(cmdInfo);
                    }
                    break;
                case ATC_FORMAT_DISK:
                    {
                        resetFDISK_Config(cmdInfo);
                    }
                    break;
                default:
                    FD_DBG_PRINTF(FD_INFO, ("Unexpected Special Purpose Command\n"));
                    handleErrorCase(cmdInfo); //Invalid option requested inform command failure
                    break;
            }
        }
        else
        {
            switch ( cbwPtr->CBWCB[4] )
            {
                case ATC_ACIVATE_SP_CMD:
                    {
                        activateSpCommand(cmdInfo);
                    }
                    break;
                case ATC_SWITCH_TO_BOOTLOADER:
                    { 
                        switchToBootLoader(cmdInfo);
                    }
                    break;
                case ATC_APPLICATION_NOTIFICATION:
                    {
                        acknowledgeApplication(cmdInfo);
                    }
                    break;
                default:
                    FLASHDISK_INFO(("Special Purpose commands are not activated\n"));
                    handleErrorCase(cmdInfo); //Invalid option requested inform command failure
                    break;
            }

        }
    }
    else
    {
        handleErrorCase(cmdInfo); //Invalid option requested inform command failure
    }
}

/******************************************************************************
*                                                                             *   
* Function Name  : updateFlashBuffer                                          *   
*                                                                             *
* Parameters     :  None                                                      *
*                                                                             *
* Returned Value : None                                                       *   
*                                                                             *
* Description    : Handler for Update Flash sub command                       *    
*                                                                             *   
*******************************************************************************/
void updateFlashBuffer(void)
{
    A_UINT8 *ptr;
    A_UINT32 checkSum=0;
    int i=0;

    ptr= flashProg.dataPtr;
    if ( flashProg.totalLength == flashProg.currentLength ) //If all the data is received
    {
        for ( i=0;i<flashProg.totalLength;i++ )
        {
            checkSum+=*ptr;  //Calculate the checksum
            ptr++;
        }
        flashProg.checkSum=checkSum;
        FLASHDISK_INFO( ("Check Sum=%X\n",checkSum));

    }
}
/******************************************************************************
*                                                                             *   
* Function Name  : resetFlashProgInfo                                          *   
*                                                                             *
* Parameters     :  None                                                      *
*                                                                             *
* Returned Value : None                                                       *   
*                                                                             *
* Description    : Reset Flash Programming status                             *    
*                                                                             *   
*******************************************************************************/
void resetFlashProgInfo(void)
{
    flashProg.checkSum = 0;
    flashProg.currentLength = 0;
    flashProg.currentPacketNo = 0;
    if ( flashProg.dataPtr != 0 )
    {
        A_DRIVER_FREE(flashProg.dataPtr,(flashProg.totalLength));
    }
    flashProg.dataPtr = 0;
    flashProg.state = 0;
    flashProg.totalLength = 0;
    flashProg.startSector = 0xFFFF;
}

/******************************************************************************
*                                                                             *   
* Function Name  : verifyCheckSum                                             *   
*                                                                             *
* Parameters     :  None                                                      *
*                                                                             *
* Returned Value : None                                                       *   
*                                                                             *
* Description    : Handler for Verify CheckSum sub command                    *    
*                                                                             *   
*******************************************************************************/

void  verifyCheckSum(USB_HANDLE handle,A_UINT32 checkSum)
{
    if ( flashProg.checkSum==checkSum )
    {
        globalDataBuffer[0]=1;
        flashProg.state =2;
    }
    else
    {
        globalDataBuffer[0]=2;
        resetFlashProgInfo();
    }
    sendData(handle, globalDataBuffer,1,1);
}

/******************************************************************************
*                                                                             *   
* Function Name  : updateBootLoader                                           *   
*                                                                             *
* Parameters     :  None                                                      *
*                                                                             *
* Returned Value : None                                                       *   
*                                                                             *
* Description    : Handler for Update Boot loader/Boot Loader Plus            *    
*                                                                             *   
*******************************************************************************/
void updateBootLoader(void)
{
    A_UINT8 *ptr;
    A_UINT32 checkSum=0;
    A_UINT32 i;
    A_UINT32 sectorSize;

    sectorSize=getFlashSectorSize();

    for ( i=0;i<flashProg.noSectors;i++ )
    {
        if ( eraseSector((flashProg.startSector+i))==FALSE ) //Erase the sectors
        {
            bootLoaderUpdateStatus=FDISK_CMD_FAILED_SECTOR_ERASE_FAILED;
            return;
        }
    }
    FLASHDISK_INFO( ("Programming Flash...\n"));
    if ( writeFlash((flashProg.startSector*sectorSize),(A_UINT32 *)flashProg.dataPtr,flashProg.totalLength)==TRUE )
    {
        FLASHDISK_INFO( ("Success\n"));
    }
    else
    {
        FLASHDISK_INFO( ("Failed\n"));

        bootLoaderUpdateStatus=FDISK_CMD_FAILED_SECTOR_WRITE_FAILED;
        return;
    }
    ptr=(A_UINT8 *)(SPI_FLASH_BASE_ADDRESS+(flashProg.startSector*sectorSize));
    for ( i=0;i<flashProg.totalLength;i++ )
    {
        checkSum+=(*ptr);
        ptr++;
    }
    FLASHDISK_INFO( ("Verifying CheckSum..."));
    if ( checkSum == flashProg.checkSum )
    {
        bootLoaderUpdateStatus=FDISK_CMD_SUCCESS;
        FLASHDISK_INFO( ("Success\n"));
    }
    else
    {
        bootLoaderUpdateStatus=FDISK_CMD_FAILED_CHECKSUM_WRONG;
        FLASHDISK_INFO( ("Failed\n"));
    }
    resetFlashProgInfo();
}


/******************************************************************************
*                                                                             *   
* Function Name  : getDataFromHost                                            *   
*                                                                             *
* Parameters     :  handle-USb Device Handler                                 *
*                   packetNo-Current data Packet                              *
*                   length-  Data Packet Length                               *
*                                                                             *
* Returned Value : None                                                       *   
*                                                                             *
* Description    : Handler for Get data from Host Sub Command                 *    
*                                                                             *   
*******************************************************************************/
void getDataFromHost(USB_HANDLE handle,
                     A_UINT16 packetNo,
                     A_UINT32 length)
{
    FD_DBG_PRINTF(FD_INFO,("Get data From Host [Size-%d]\n",length));
    updateCommandStatusSuccess();
    if ( flashProg.state == 1 )
    {
        if ( flashProg.currentPacketNo != packetNo )
        {
            resetFlashProgInfo();
            return;
        }
        flashProg.currentPacketNo++;
        if ( getUsbTransferStatus(handle,MS_DATA_IN_PIPE, USB_RECV) != USB_OK )
        {
            FD_DBG_PRINTF(FD_INFO, ("Write Failed[Transfer status not OK\n"));
            cancelUsbTransfer(handle,MS_DATA_IN_PIPE, USB_RECV);
        }
        fdiskCmdInfo.cmdState =  BOOT_LOADER_UPDATE_STATE;
        fdiskCmdInfo.dataLength =length;
        fdiskCmdInfo.dataPtr =(A_UINT8 *)(flashProg.dataPtr+flashProg.currentLength);
        receiveUsbData(handle, MS_DATA_OUT_PIPE, (A_UINT8 *)(flashProg.dataPtr+flashProg.currentLength),length);
    }
}


/******************************************************************************
*                                                                             *   
* Function Name  : handleBootLoaderUpdateCommand                              *   
*                                                                             *
* Parameters     :  cmdInfo    -Pointer to CMD_STRUCT                         *
*                                                                             *
* Returned Value : None                                                       *   
*                                                                             *
* Description    : Main Handler for Falsh Programming Commands                *    
*                                                                             *   
*******************************************************************************/

void handleBootLoaderUpdateCommand(CMD_STRUCT_PTR cmdInfo)
{
    A_UINT32 length=0;
    A_UINT32 sectorSize=0;
    A_UINT16 startSector=0;
    CBW_STRUCT_PTR  cbwPtr=cmdInfo->cmdPtr;

    if ( stringCompare(&cbwPtr->CBWCB[1],(A_UINT8 *)"ATC",3) ==3 )
    {
        if ( specialPurposeCmdAllowed == 1 )
        {
            length  = get32BitWordFromBytes(&cbwPtr->CBWCB[5]);
            switch ( cbwPtr->CBWCB[4] )
            {
                case FLASH_PROG_INIT:  //Initialize the programming setup
                    {
                        if ( flashProg.state == 0 )
                        {
                            startSector=get16BitWordFromBytes(&cbwPtr->CBWCB[9]);
                            FLASHDISK_INFO(("Initializing for programing flash[Size-%d,Start Sector-%d]\n",length,startSector));
                            if ( length > 0x10000 )
                            {
                                FLASHDISK_INFO(("Invalid Total Size Requested\n")); 
                                sendSpCommandStatus(cmdInfo->handle,FDISK_CMD_FAILED_INSUFFICIENT_MEMORY);
                                return;
                            }
                            if ( startSector>=TOTAL_NO_OF_DISK_SECTORS )
                            {
                                FLASHDISK_INFO(("Total Number of Sectors is only %d\n",TOTAL_NO_OF_DISK_SECTORS));
                                sendSpCommandStatus(cmdInfo->handle,FDISK_CMD_FAILED_INSUFFICIENT_MEMORY);
                                return;
                            }
                            sectorSize=getFlashSectorSize();

                            if ( (length%sectorSize)==0 )
                            {
                                flashProg.noSectors=(length/sectorSize);
                            }
                            else
                            {
                                flashProg.noSectors=((length/sectorSize)+1);
                            }
                            resetFlashProgInfo();

                            flashProg.state=1;
                            flashProg.totalLength =length;
                            flashProg.startSector= startSector;
                            flashProg.dataPtr=(A_UINT8 *)A_DRIVER_MALLOC(length);

                            if ( flashProg.dataPtr !=NULL )
                            {
                                A_DATA_CACHE_INVAL(flashProg.dataPtr,length);
                                sendSpCommandStatus(cmdInfo->handle,FDISK_CMD_SUCCESS);
                            }
                            else
                            {
                                sendSpCommandStatus(cmdInfo->handle,FDISK_CMD_FAILED_INSUFFICIENT_MEMORY);
                            }
                        }
                        else
                        {
                            sendSpCommandStatus(cmdInfo->handle,FDISK_CMD_FAILED_WRONG_SEQUENCE);
                        }
                    }
                    break;
                case FLASH_PROG_DATA:  //Data Stage
                    {
                        FD_DBG_PRINTF(FD_INFO, ("Data size-%d\n",length));
                        getDataFromHost(cmdInfo->handle,get16BitWordFromBytes(&cbwPtr->CBWCB[9]),length);
                    }
                    break;
                case FLASH_PROG_VERIFY_CHECKSUM:   //Verify CheckSum command
                    {
                        FD_DBG_PRINTF(FD_INFO,("Verify CheckSum-%X\n",length));
                        verifyCheckSum(cmdInfo->handle,length);
                    }
                    break;
                case FLASH_PROG_BOOTLOADER:        //Update Flash 
                    {
                        if ( flashProg.state == 2 )
                        {
                            sendSpCommandStatus(cmdInfo->handle,FDISK_CMD_SUCCESS); 
                            startSpTimer(ATC_UPDATE_BOOTLOADER,20); //Start the timer to execute immediately
                        }
                        else
                        {
                            sendSpCommandStatus(cmdInfo->handle,FDISK_CMD_FAILED_WRONG_SEQUENCE); 
                            FD_DBG_PRINTF(FD_INFO,("Invalid Flash Programming Command\n"));
                        }
                    }
                    break;
                case EXECUTE_BOOTLOADER:           //Run the software downloaded
                    {
                        if ( flashProg.state == 2 )
                        {
                            sendSpCommandStatus(cmdInfo->handle,FDISK_CMD_SUCCESS); 
                            startSpTimer(ATC_EXECUTE_BOOTLOADER,20); //Start the timer to execute immediately
                        }
                        else
                        {
                            sendSpCommandStatus(cmdInfo->handle,FDISK_CMD_FAILED_WRONG_SEQUENCE); 
                            FD_DBG_PRINTF(FD_INFO,("Invalid Flash Programming Command\n"));
                        }
                    }
                    break;
                case FLASH_UPDATE_STATUS:  //Return the status of Flash Porgramming
                    {
                        if ( flashProg.state == 0 )
                        {
                            sendSpCommandStatus(cmdInfo->handle,bootLoaderUpdateStatus); 
                        }
                        else
                        {
                            sendSpCommandStatus(cmdInfo->handle,FDISK_CMD_FAILED_WRONG_SEQUENCE); 
                        }
                    }
                    break;
                    break;
                default:
                    ASSERT(0);
                    break;
            }
        }
        else
        {
            FLASHDISK_INFO(("Special Purpose commands are not activated\n"));
            handleErrorCase(cmdInfo); //Invalid option requested inform command failure
        }
    }
    else
    {
        handleErrorCase(cmdInfo); //Invalid option requested inform command failure
    }

}
#ifdef ATC_CDROM_DISK
/******************************************************************************
*                                                                             *   
* Function Name  : handleAtapiToc                                             *   
*                                                                             *
* Parameters     :  cmdInfo    -Pointer to CMD_STRUCT                         *
*                                                                             *
* Returned Value : None                                                       *   
*                                                                             *
* Description    : Handler for Read TOC command                               *    
*                                                                             *   
*******************************************************************************/
void handleAtapiToc(CMD_STRUCT_PTR cmdInfo)
{
    CBW_STRUCT_PTR  cbwPtr=cmdInfo->cmdPtr;
    FD_DBG_PRINTF(FD_INFO,("Process Read Toc Command\n"));

    if ( cbwPtr->BMCBWFLAGS & USB_CBW_DIRECTION_BIT )
    {
        switch ( cbwPtr->CBWCB[1] )
        {
            case 0:
                sendData(cmdInfo->handle,(A_UINT8 *)&CD_TOC,sizeof(CD_TOC),cbwPtr->DCBWDATALENGTH);
                break;
            case 2:
                sendData(cmdInfo->handle,(A_UINT8 *)&CD_TOC2, sizeof(CD_TOC2),cbwPtr->DCBWDATALENGTH);
                break;
            default:
                handleErrorCase(cmdInfo); //Invalid option requested inform command failure
                break;
        }
        csw.DCSWDATARESIDUE =0;
        csw.BCSWSTATUS = 0;
    }
    else
    {
        handleErrorCase(cmdInfo); //Invalid option requested inform command failure
    }
}

/******************************************************************************
*                                                                             *   
* Function Name  : handleReadSubChannelInfo                                   *   
*                                                                             *
* Parameters     :  cmdInfo    -Pointer to CMD_STRUCT                         *
*                                                                             *
* Returned Value : None                                                       *   
*                                                                             *
* Description    : Handler for Read Sub Channel Info                          *    
*                                                                             *   
*******************************************************************************/

void handleReadSubChannelInfo(CMD_STRUCT_PTR cmdInfo)

{
    CBW_STRUCT_PTR  cbwPtr=cmdInfo->cmdPtr;

    if ( cbwPtr->BMCBWFLAGS & USB_CBW_DIRECTION_BIT )
    {
        sendCommandFailedWithData(cmdInfo->handle,0x2);
    }
    else
    {
        handleErrorCase(cmdInfo); //Invalid option requested inform command failure  
    }

}
#endif 

/******************************************************************************
*                                                                             *   
* Function Name  : handleScsiMassStorageCommand                               *   
*                                                                             *
* Parameters     : cmdInfo    -Pointer to CMD_STRUCT                          *
*                                                                             *
* Returned Value : None                                                       *   
* Description    : Main handler for Mass storage class command                *   
*                                                                             * 
*                                                                             *   
*******************************************************************************/
void handleScsiMassStorageCommand(CMD_STRUCT_PTR cmdInfo)
{
    CBW_STRUCT_PTR cbwPtr=cmdInfo->cmdPtr;

    cbwPtr->DCBWDATALENGTH=A_swab32(cbwPtr->DCBWDATALENGTH); //Convert to little endian


    if ( cbwPtr->CBWCB[0] != 0 )
    {
        otherCommandsReceived=1;

        FLASHDISK_INFO(("MS CMD[%05d]-%02X[%02X %02X %02X %02X %02X %02X %02X %02X %02X %02X %02X %02X %02X %02X %02X\n",
                        cbwPtr->DCBWDATALENGTH,cbwPtr->CBWCB[0],
                        cbwPtr->CBWCB[1],cbwPtr->CBWCB[2],cbwPtr->CBWCB[3],cbwPtr->CBWCB[4],
                        cbwPtr->CBWCB[5],cbwPtr->CBWCB[6],cbwPtr->CBWCB[7],cbwPtr->CBWCB[8],
                        cbwPtr->CBWCB[9],cbwPtr->CBWCB[10],cbwPtr->CBWCB[11],cbwPtr->CBWCB[12],
                        cbwPtr->CBWCB[13],cbwPtr->CBWCB[14],cbwPtr->CBWCB[15])); 

        //FLASHDISK_INFO(("MS CMD[%05d]-%02X\n",cbwPtr->DCBWDATALENGTH,cbwPtr->CBWCB[0]));

    }



    switch ( cbwPtr->CBWCB[0] )
    {
        case 0x0: /* Request the device to report if it is ready */
            {
                handleScsiTestUnitReady(cmdInfo);

            }
            break;
        case 0x03: /* Transfer status sense data to the host */
            {
                handleRequestSense(cmdInfo);
            }
            break;
        case 0x12: /* Inquity command. get device information */
            {
                handleScsiInquiry(cmdInfo);
            }
            break;
        case 0x5A: //Mode Sense Command
        case 0x1A:
            {
                handleScsiModeSense(cmdInfo);

            }
            break;
        case 0x1E: /* Prevent or allow the removal of media from a removable media device */
            {
                handleScsiPreventAllowMediumRemoval(cmdInfo);
            }

            break;
        case 0x23: /* Read Format Capacities. Report current media capacity */
            {
                handleScsiReadFormatCapacity(cmdInfo);

            }
            break;
        case 0x25: /* Report current media capacity */
            {
                handleScsiReportCapacity(cmdInfo);
            }
            break;
        case 0x28:/*Read(10) Transfer binary data from media to the host */
            {
                handleScsiRead(cmdInfo);
            }
            break;
        case 0x2A:/*Write(10) Transfer binary data from the host to the media*/
            {   if ( fDisk.writeEnable==TRUE )
                {
                    handleScsiWrite(cmdInfo);
                }
                else
                {
                    handleErrorCase(cmdInfo); //Invalid option requested inform command failure
                }
            }
            break;
        case 0x1B: /* Request a request a removable-media device to load or unload its media */
            {
                sendCommandSuccess(cmdInfo->handle);
            }
            break;
        case 0xDD:  //Special Command
            {
                handleSpecialPurposeCommand(cmdInfo);
            }
            break;
        case 0xCC:  //Special Command for Flash Programing
            {
                handleBootLoaderUpdateCommand(cmdInfo);
            }
            break;
        case 0x2F: /* Verify data on the media */
            {
                handleScsiVerifyCommand(cmdInfo);
            }
            break;
#ifdef ATC_CDROM_DISK
        case 0x42:  // Read Sub Channel Command
            {
                handleReadSubChannelInfo(cmdInfo);
            }
            break;
        case 0x43: // Read TOC Command
            {
                handleAtapiToc(cmdInfo);
            }
            break;

        case 0x45: // Play Audio Command
            {
                handlePlayAudio(cmdInfo);
            }
            break;
        case 0x46: // Report current media cnfiguration 
            {
                handleScsiGetConfig(cmdInfo);
            }
            break;

        case 0x4A: // Report current status of media
            {
                handleEventStatusNotification(cmdInfo);
            }
            break;
#endif
        case 0x01: // Position a head of the drive to zero track 
        case 0x04: // Format unformatted media 
        case 0x1D: // Perform a hard reset and execute diagnostics 
        case 0x2B: // Seek the device to a specified address
        case 0x2E: // Transfer binary data from the host to the media and verify data 
        case 0x55: // Mode Select Command
        case 0xA8: // Read (12) Transfer binary data from the media to the host 
        case 0xAA: // Write (12) Transfer binary data from the host to the media
        default:
            {
                handleErrorCase(cmdInfo); //Invalid option requested inform command failure
            }
            break;
    } 

} 


/******************************************************************************
*                                                                             * 
* Function Name  : initMassStorage                                            * 
*                                                                             *
* Parameters     : None                                                       *
*                                                                             *
* Returned Value : None                                                       * 
*                                                                             *
* Description    : Initialises the Mass Storage Support                       * 
*                                                                             *
*******************************************************************************/
void initMassStorage(void)
{ 
    fdiskCmdInfo.cmdState=DEFAULT_STATE;
    fdiskCmdInfo.dataLength = 0;
    fdiskCmdInfo.dataPtr = 0;
    fdiskCmdInfo.rxBuffer = NULL;
    fdiskCmdInfo.rxLength = 0;
    fdiskCmdInfo.txLength = 0;
    dFlashDisk.buffPtr = NULL;
    dFlashDisk.flashSector = 0xFFF;
    dFlashDisk.state = FALSE;
    fdiskCmdInfo.txBuffer = NULL;
    flashProg.dataPtr = 0;
    testSetup.buffer = NULL;
    testSetup.bufferSize = 0;
    resetFlashProgInfo();
    updateDiskInfo();
#ifndef ATC_CDROM_DISK
    checkFDisk(FALSE);
#else
    resetDriverConfig(FALSE);
#endif
    return;
}


/******************************************************************************
*******************************************************************************
**                                                                           **
**                                                                           **
**                                                                           **
**   EXPORTED FUNCTIONS      (prototypes in header file)                     **
**                                                                           **
**                                                                           **
**                                                                           **
** ****************************************************************************
** ***************************************************************************/

/******************************************************************************
*                                                                             * 
* Function Name  : flashDiskRecvIndHandler                                    * 
*                                                                             *
* Parameters     : appHandle -Application Handler                             *
*                  pipeNum   -Data Pipe Number                                *
*                  pBuffer   -Buffer Pointer                                  *
*                  bytesReceived   -Number of Bytes received                  *
*                                                                             *
* Returned Value : None                                                       * 
*                                                                             *
* Description    : Receive interrupt Handler                                  * 
*                                                                             *
*******************************************************************************/
void flashDiskRecvIndHandler(IN APP_HANDLE   appHandle,
                             IN A_UINT8      pipeNum,
                             IN A_UINT8      *pBuffer,
                             IN A_UINT32     bytesReceived)
{
    FLASHDISK_CONTEXT      *pFlashDiskCtxt;
    CBW_STRUCT_PTR          cbwPtr;
    PUSBDRV_HANDLE          pUsbDrvHandle;
    CMD_STRUCT              msCmdInfo;

    pFlashDiskCtxt = (FLASHDISK_CONTEXT *)appHandle;
    pUsbDrvHandle=(PUSBDRV_HANDLE)pFlashDiskCtxt->drvHandle;
    cbwPtr=(CBW_STRUCT_PTR)pBuffer;

    ASSERT (pFlashDiskCtxt);
    A_DATA_CACHE_INVAL(pBuffer, sizeof(A_UINT32)); 
    A_DATA_CACHE_INVAL(pBuffer, bytesReceived);    
    switch ( fdiskCmdInfo.cmdState )
    {
        case WRITE_FLASH_DISK_STATE:  //Flash Write
            {
                fdiskCmdInfo.dataLength =bytesReceived;
                if ( fdiskCmdInfo.rxLength != bytesReceived )
                {
                    FD_DBG_PRINTF(FD_MAIN_INFO,("Rx Command Processed E:%d A:%d\n",fdiskCmdInfo.rxLength,bytesReceived));
                }
                ASSERT(fdiskCmdInfo.rxBuffer==pBuffer);
                sendCommandStatus(pUsbDrvHandle->deviceHandle);
                updateDummyDisk();
                fdiskCmdInfo.cmdState = DEFAULT_STATE;
                return;
            }
            break;

        case BOOT_LOADER_UPDATE_STATE: //Update Boot Loader is in progress
            {
                ASSERT(fdiskCmdInfo.dataLength==bytesReceived);
                fdiskCmdInfo.dataLength =0;
                ASSERT(fdiskCmdInfo.dataPtr==pBuffer);
                fdiskCmdInfo.dataPtr =0;
                fdiskCmdInfo.cmdState = DEFAULT_STATE;
                sendCommandStatus(pUsbDrvHandle->deviceHandle);
                flashProg.currentLength+=bytesReceived;
                updateFlashBuffer();
                return;
            }
            break;
        case SET_PASSWORD_STATE: //Set Passwrod state
            {
                sendCommandStatus(pUsbDrvHandle->deviceHandle);
                setPassword();
                fdiskCmdInfo.dataLength =0;
                fdiskCmdInfo.dataPtr =0;
                fdiskCmdInfo.cmdState = DEFAULT_STATE;
                return;
            }
            break;
        default:
            if ( fdiskCmdInfo.rxBuffer == pBuffer )
            {
                if ( fdiskCmdInfo.rxBuffer != NULL )
                {
                    FD_DBG_PRINTF(FD_MAIN_INFO,("Releasing Rx Buffer allocated for Unsupported commands\n"));
                    A_DRIVER_FREE(fdiskCmdInfo.rxBuffer,fdiskCmdInfo.rxLength);
                    fdiskCmdInfo.rxBuffer=NULL;
                    fdiskCmdInfo.rxLength=0;
                }
            }
            break;
    }
    FD_DBG_PRINTF(FD_INFO, ("Rx : pipe - %d size - %d[Sig-%X]\n",pipeNum, bytesReceived,cbwPtr->DCBWSIGNATURE));
    if ( (!CBW_PROCESSED) && (bytesReceived == 31) && (cbwPtr->DCBWSIGNATURE == A_swab32(USB_DCBWSIGNATURE)) )
    {
        /* A valid CBW was received */
        FD_DBG_PRINTF(FD_INFO,("Valid CBW Received\n"));
        csw.DCSWSIGNATURE = A_swab32(USB_DCSWSIGNATURE);
        csw.DCSWTAG = cbwPtr->DCBWTAG;
        CBW_PROCESSED = TRUE;

        // Process the command 
        msCmdInfo.cmdPtr=cbwPtr;
        msCmdInfo.handle=pUsbDrvHandle->deviceHandle;
        handleScsiMassStorageCommand(&msCmdInfo);
    }
    else if ( CBW_PROCESSED )
    {
        // Send the command status information 
        sendCommandStatus(pUsbDrvHandle->deviceHandle);
    }
    else
    {
        A_DATA_CACHE_INVAL(pFlashDiskCtxt->pMessageBuffer, sizeof(A_UINT32)); 
        athUsbDrvRecv( pFlashDiskCtxt->drvHandle, MS_DATA_OUT_PIPE-1, 
                       (A_UINT8 *)pFlashDiskCtxt->pMessageBuffer);
    }
}

/******************************************************************************
*                                                                             * 
* Function Name  : flashDiskSendCfmHandler                                    * 
*                                                                             *
* Parameters     : appHandle -Application Handler                             *
*                  pipeNum   -Data Pipe Number                                *
*                  pBuffer   -Buffer Pointer                                  *
*                  bytesSent   -Number of Bytes Sent                          *
*                                                                             *
* Returned Value : None                                                       * 
*                                                                             *
* Description    : Transmit interrupt Handler                                 * 
*                                                                             *
*******************************************************************************/
void flashDiskSendCfmHandler(IN APP_HANDLE   appHandle,
                             IN A_UINT8      pipeNum,
                             IN A_UINT8      *pBuffer,
                             IN A_UINT32     bytesSent)
{
    FLASHDISK_CONTEXT   *pFlashDiskCtxt= (FLASHDISK_CONTEXT *)appHandle;
    PUSBDRV_HANDLE pUsbDrvHandle=(PUSBDRV_HANDLE)pFlashDiskCtxt->drvHandle;

    ASSERT (pFlashDiskCtxt);
    FD_DBG_PRINTF(FD_INFO, ("Tx : pipe - %d size - %d\n",pipeNum, bytesSent));

    if ( CBW_PROCESSED )
    {
        // Send the command status information 
        if ( pBuffer == fdiskCmdInfo.txBuffer )
        {
            if ( fdiskCmdInfo.txBuffer != NULL )
            {
                A_DRIVER_FREE(fdiskCmdInfo.txBuffer,fdiskCmdInfo.txLength);
                fdiskCmdInfo.txBuffer=NULL;
                fdiskCmdInfo.txLength=0;
            }
        }
        sendCommandStatus(pUsbDrvHandle->deviceHandle);
    }
    else
    {
        A_DATA_CACHE_INVAL(pFlashDiskCtxt->pMessageBuffer, sizeof(A_UINT32)); 
        athUsbDrvRecv( pFlashDiskCtxt->drvHandle, MS_DATA_OUT_PIPE-1, 
                       (A_UINT8 *)pFlashDiskCtxt->pMessageBuffer);
    }

}

/******************************************************************************
*                                                                             * 
* Function Name  : getDescription                                             * 
*                                                                             *
* Parameters     : handle     -Handle of USB device                           *
*                  setup      -Setup or data Stage                            *
*                  setup_ptr  -Pointer to Setup Structure                     *
*                                                                             *
* Returned Value : None                                                       * 
* Description    : getDescription command                                     * 
*                  The Device Request can ask for Device/Config/string/       *
*                  interface/endpoint descriptors (via wValue).This will post *
*                  an IN response to return the requested descriptor and then *
*                  wait for the OUT which terminates the control transfer.    *
*                                                                             *
******************************************************************************/
void getDescription(USB_HANDLE handle,
                    boolean setup,
                    SETUP_STRUCT_PTR setup_ptr)
{ 
    uint_32  max_pkt_size;
    A_UINT8 i;

    if ( setup )
    {
        // Load the appropriate string depending on the descriptor requested.
        switch ( setup_ptr->VALUE & 0xFF00 )
        {
            
            case 0x0100:
                {
                    sendUsbData(handle, 0, (A_UINT8 *)((pointer)&FD_DevDesc),
                                MIN(setup_ptr->LENGTH, sizeof(FD_DevDesc)));
                }
                break;

            case 0x0200:
                {
                    // Set the Max Packet Size in the config and other speed config 
                    if ( speed == USB_SPEED_HIGH )
                    {
                        max_pkt_size = EP1_HS_MAX_PACKET_SIZE;
                    }
                    else
                    {
                        max_pkt_size = EP1_FS_MAX_PACKET_SIZE;
                    }

                    FD_ConfigDesc[CFG_DESC_EP1TX_MAX_PACKET_SIZE] = 
                    UINT16_LOW(max_pkt_size);
                    FD_ConfigDesc[CFG_DESC_EP1TX_MAX_PACKET_SIZE+1] = 
                    UINT16_HIGH(max_pkt_size);
                    FD_ConfigDesc[CFG_DESC_EP1RX_MAX_PACKET_SIZE] = 
                    UINT16_LOW(max_pkt_size);
                    FD_ConfigDesc[CFG_DESC_EP1RX_MAX_PACKET_SIZE+1] = 
                    UINT16_HIGH(max_pkt_size);

                    sendUsbData(handle, 0, (A_UINT8 *)((pointer)&FD_ConfigDesc),
                                MIN(setup_ptr->LENGTH, sizeof(FD_ConfigDesc)));
                }
                break;

            case 0x0300:
                {
                    i= (setup_ptr->VALUE & 0x00FF);
                    if ( i > MS_USB_STR_NUM )
                    {
                        FD_DBG_PRINTF(FD_INFO,("Invalid String ID\n"));
                        sendUsbData(handle, 0, MS_USB_STRING_DESC[MS_USB_STR_NUM+1],
                                    MIN(setup_ptr->LENGTH, MS_USB_STRING_DESC[MS_USB_STR_NUM+1][0]));
                    }
                    else
                    {
                        FD_DBG_PRINTF(FD_INFO,("String ID %d  Length %d %d\n",i,(setup_ptr->LENGTH),MS_USB_STRING_DESC[i][0]));
                        sendUsbData(handle, 0,MS_USB_STRING_DESC[i], 
                                    MIN(setup_ptr->LENGTH,MS_USB_STRING_DESC[i][0]));
                    } 
                }
                break;

            case 0x600:
                {
                    sendUsbData(handle, 0, (A_UINT8 *)FD_DevQualifierDesc, 
                                MIN(setup_ptr->LENGTH, sizeof(FD_DevQualifierDesc)));
                }
                break;

            case 0x700:
                {
                    if ( speed == USB_SPEED_HIGH )
                    {
                        max_pkt_size = EP1_FS_MAX_PACKET_SIZE;
                    }
                    else
                    {
                        max_pkt_size = EP1_HS_MAX_PACKET_SIZE;
                    } 

                    FD_OtherSpeed_Config[CFG_DESC_EP1TX_MAX_PACKET_SIZE] = 
                    UINT16_LOW(max_pkt_size);
                    FD_OtherSpeed_Config[CFG_DESC_EP1TX_MAX_PACKET_SIZE+1] = 
                    UINT16_HIGH(max_pkt_size);
                    FD_OtherSpeed_Config[CFG_DESC_EP1RX_MAX_PACKET_SIZE] = 
                    UINT16_LOW(max_pkt_size);
                    FD_OtherSpeed_Config[CFG_DESC_EP1RX_MAX_PACKET_SIZE+1] = 
                    UINT16_HIGH(max_pkt_size);
                    sendUsbData(handle, 0, (A_UINT8 *)FD_OtherSpeed_Config, 
                                MIN(setup_ptr->LENGTH, sizeof(FD_OtherSpeed_Config)));
                }
                break;

            default:
                {
                    _usb_device_stall_endpoint(handle, 0, 0);
                    return;
                }
        }
        receiveUsbData(handle, 0, 0, 0);
    }
    return;
} 

/******************************************************************************
*                                                                             * 
* Function Name  : MassStorageClass                                           * 
*                                                                             *
* Parameters     : handle     -Handle of USB device                           *
*                  setup      -Setup or data Stage                            *
*                  setup_ptr  -Pointer to Setup Structure                     *
*                                                                             *
* Returned Value :  None                                                      * 
* Description    : Mass Storage Class Specific request                        *                               
*                                                                             *                                                                              * 
*                                                                             * 
*******************************************************************************/
void MassStorageClass(USB_HANDLE handle,boolean setup,SETUP_STRUCT_PTR setup_ptr)
{ 
    FD_DBG_PRINTF(FD_INFO,("Class Specific Request\n"));
    if ( setup )
    {
        switch ( setup_ptr->REQUEST )
        {
            case 0xFF:/* Bulk-Only Mass Storage Reset: Ready the device for the next CBW from the host*/
                {
                    FD_DBG_PRINTF(FD_INFO,("Bulk Only Mass Storage Reset\n"));
                    handleMassStorageReset();
                }
                break;
            case 0xFE: /* get Max LUN */
                {
                    sendUsbData(handle, 0, &(fDisk.maxLUN),1);
                }
                break;
            default :
                {
                    FD_DBG_PRINTF(FD_ERROR,("Invalid Class Specific request\n"));
                    _usb_device_stall_endpoint(handle,0, 0);
                }
                return;
        } 
        receiveUsbData(handle, 0, 0, 0);
    }

} 


/******************************************************************************
*                                                                             * 
* Function Name  : cyg_user_start                                             * 
*                                                                             *
* Parameters     : None                                                       *
*                                                                             *
* Returned Value : None                                                       * 
* Description    :  First function called by the eCos, Initialises the USB    *
*                   and Flash Disk and registers the  callback functions.     * 
*                                                                             *
*******************************************************************************/
void cyg_user_start(void)
{
    FLASHDISK_INFO(("\n   ----------------------------------\n   ATHEROS Boot Loader Plus  Abhijit\n"));
    FLASHDISK_INFO(("   Version : %d.%d\n   Created on %s at %s\n   ----------------------------------\n", 
                    ((BOOTROM_PLUS_VERSION >> 8 ) & 0xFF), 
                    ((BOOTROM_PLUS_VERSION) & 0xFF), 
                    __DATE__,__TIME__));
    FLASHDISK_INFO(("Starting the Flash Disk...\n"));
    initMassStorage(); //Initialize the Mass storage Device
    BootRom(TRUE);      //Start Flash Disk;
    /* Start one timer to switch from Mass Storage Support to Boot Loader 
       if the Hist Doe not provide Mass storage support                  */
    startSpTimer(ATC_MASS_STORAGE_SUPPORT_TEST,5000);  
}
#endif //BOOT_LOADER_PLUS

