/*
 * $Id: //depot/sw/branches/1.3_USB_LINUX_port/src/USB/transport/usb/target/app/bootrom/bootrom.c#1 $
 *
 * Copyright (c) 2000-2003 Atheros Communications, Inc., All Rights Reserved
 *
 * Boot over USB built for the USB driver
 */
#include <cyg/hal/plf_bcfg.h>
#include "stdio.h"
#include "stdlib.h"
#include "string.h"
#include "ecosdrv.h"
#include "athusbdrv.h"
#include "bootrom.h"

int asserts = 1; 

#define BOOTROM_INIT_DEBUG         (0x00000001)
#define BOOTROM_RECV_DEBUG         (0x00000002)
#define BOOTROM_SEND_DEBUG         (0x00000004)
#define BOOTROM_MEM_DEBUG          (0x00000008)
#define BOOTROM_MESSAGE_DEBUG      (0x00000010)

A_UINT32 bootromDebug       = 0;
A_UINT32 bootRomVerbose     = 1;
BOOTROM_CONTEXT  *pFlashPromCtxt=0;

#define BOOTROM_DEBUG_PRINTF(FLAG, ARG)     \
    if (bootromDebug & (FLAG)) {            \
        diag_printf ARG;                    \
    }                                   

#define BOOTROM_VERBOSE_PRINTF(ARG)         \
    if (bootRomVerbose) {                   \
        diag_printf ARG;                    \
    }                                   

#define MESSAGE_BUFFER_SIZE     (recvPipePktSize[0])
#define DATA_BUFFER_SIZE        (recvPipePktSize[1])
#define BOOT_USB_DEVICE_NUM     0
#define MESSAGE_PIPE            0
#define DATA_PIPE               1
#define IMAGE_ALIGN_BYTES       256


extern void 
bootRomSendCfmHandler(APP_HANDLE appHandle, A_UINT8 pipeNum, 
                      A_UINT8 *pBuffer,A_UINT32 bytesSent);

extern void 
bootRomRecvIndHandler(APP_HANDLE appHandle, A_UINT8 pipeNum, 
                      A_UINT8 *pBuffer,A_UINT32 bytesReceived);

#ifdef BOOT_LOADER_PLUS
extern void flashDiskRecvIndHandler(
                                   IN APP_HANDLE   appHandle,
                                   IN A_UINT8      pipeNum,
                                   IN A_UINT8      *pBuffer,
                                   IN A_UINT32     bytesReceived
                                   );

extern void flashDiskSendCfmHandler(
                                   IN APP_HANDLE   appHandle,
                                   IN A_UINT8      pipeNum,
                                   IN A_UINT8      *pBuffer,
                                   IN A_UINT32     bytesSent
                                   );

extern A_UINT8 ZG_flashDisk;
#define SPI_FLASH_BASE_ADDRESS           0x01000000
#define ATC_BOOT_LOADER_SECTOR           0x2 //Sector 2 is reserved for Normal BootLoader 
#endif

/******************************************************************************
* printMemInfo
*
* Returns: void
*/
void printMemInfo( void )
{
    struct mallinfo mem_info;
    mem_info = mallinfo();
    BOOTROM_DEBUG_PRINTF(BOOTROM_MEM_DEBUG ,
                          ("Memory Used : %d Free : %d\n", mem_info.uordblks,  mem_info.fordblks));
}

/******************************************************************************
* printMemInfo
*
* Returns: A_UINT32
*/
A_UINT32 getAvailableHeap( void )
{
    struct mallinfo mem_info;
    mem_info = mallinfo();
    return (mem_info.fordblks);
}

/******************************************************************************
* usbBootromClean
*
* Returns: A_OK, A_EERROR
*/

void usbBootromClean( BOOTROM_CONTEXT  *pBootromCtxt)
{
    A_STATUS    status;

    if (pBootromCtxt) {

        if (pBootromCtxt->drvHandle)  {
            status =  athUsbDrvShutdown( pBootromCtxt->drvHandle);
            ASSERT (status == A_OK);
        }
        if (pBootromCtxt->pMessageBuffer) {
            A_DRIVER_FREE(pBootromCtxt->pMessageBuffer, sizeof(A_UINT8) * 
                          MESSAGE_BUFFER_SIZE);
        }
        A_DRIVER_FREE(pBootromCtxt, sizeof(BOOTROM_CONTEXT));
    }
}


/******************************************************************************
* usbBootromInit
*
* Returns: A_OK, A_EERROR
*/

A_STATUS usbBootromInit( BOOTROM_CONTEXT  **ppBootromCtxt)
{
    A_STATUS                status;
    BOOTROM_CONTEXT        *pBootromCtxt;
    A_UINT16                bootRomVersion;


    if (bootromDebug & BOOTROM_MEM_DEBUG) {
        printMemInfo();
    }
    /* Read the bootrom version*/
#ifdef BOOT_LOADER_PLUS
    if (ZG_flashDisk == FALSE) {
#endif //BOOT_LOADER_PLUS
    hal_ar5523_bcfgdata_brom_vers(&bootRomVersion);
    BOOTROM_VERBOSE_PRINTF(("\nAR5523 USB Bootloader - Version : %d.%d , %s\n", 
                            ((bootRomVersion >> 8 ) & 0xFF), 
                            ((bootRomVersion) & 0xFF), 
                            __DATE__));
#ifdef BOOT_LOADER_PLUS
    }
#endif //BOOT_LOADER_PLUS

    /* Allocate Memory for the device context */
    pBootromCtxt = (BOOTROM_CONTEXT *) A_DRIVER_MALLOC(sizeof(BOOTROM_CONTEXT));
    if (pBootromCtxt == NULL) {
        BOOTROM_DEBUG_PRINTF(BOOTROM_INIT_DEBUG ,("Failed to allocate Memory\n"));
        return A_NO_MEMORY;
    }
    *ppBootromCtxt = pBootromCtxt;
    A_MEM_ZERO(pBootromCtxt, sizeof (BOOTROM_CONTEXT));

    /* Allocate the buffer for receiving Messages */
    pBootromCtxt->pMessageBuffer = (A_UINT8 *) A_DRIVER_MALLOC(sizeof(A_UINT8) * 
                                                              MESSAGE_BUFFER_SIZE);
    if (pBootromCtxt->pMessageBuffer == NULL) {
        BOOTROM_DEBUG_PRINTF(BOOTROM_INIT_DEBUG ,("Failed to allocate Memory\n"));
        return A_NO_MEMORY;
    }
    status = athUsbDrvInit( BOOT_USB_DEVICE_NUM, (void *)pBootromCtxt,
                         bootRomRecvIndHandler, bootRomSendCfmHandler,
                         &pBootromCtxt->drvHandle,
                         &pBootromCtxt->numSendPipes,&pBootromCtxt->numRecvPipes);

    if (status != A_OK) {
        BOOTROM_DEBUG_PRINTF(BOOTROM_INIT_DEBUG ,("Failed to Init USB Driver\n"));
        return A_NO_MEMORY;
    }
    if (bootromDebug & BOOTROM_MEM_DEBUG) {
        printMemInfo();
    }
    BOOTROM_VERBOSE_PRINTF(("Available Memory : %d bytes\n", 
                            (A_UINT32)getAvailableHeap()));
    
    pBootromCtxt->bootRomState = BOOTROM_INIT_COMMAND_STATE;
    /* Post the Message Buffer for receiving Bootrom Command */
    status = athUsbDrvRecv( pBootromCtxt->drvHandle, MESSAGE_PIPE, 
                            (A_UINT8 *)pBootromCtxt->pMessageBuffer);
    if (status != A_OK ) {
        BOOTROM_DEBUG_PRINTF(BOOTROM_INIT_DEBUG ,("Receive Failed\n"));
        return A_ERROR;
    }
    return A_OK;
}

/******************************************************************************
* initStateHandler
*
* Returns: void
*/
void initStateHandler (
    IN BOOTROM_CONTEXT  *pBootromCtxt,
    IN A_UINT8          *pBuffer
    )
{
    BOOTROM_MESSAGE     *pBootRomMsg;
    IMAGE_INFO          *pImageInfo;
    A_STATUS            status;
    A_UINT32            memAvail;
    /* 
     * Check the available memory, temporily done 
     * through the mallinfo call, has to be changed later
     * with a better meachanism 
     */
    memAvail = getAvailableHeap();

    pBootRomMsg = (BOOTROM_MESSAGE *)pBuffer;
    A_DATA_CACHE_INVAL(pBuffer, sizeof(A_UINT32)); 

    ASSERT (pBootRomMsg->msgLen <= MESSAGE_BUFFER_SIZE);
    A_DATA_CACHE_INVAL(pBuffer, pBootRomMsg->msgLen); 
    BOOTROM_DEBUG_PRINTF(BOOTROM_MESSAGE_DEBUG ,("%d:%d:%d:%d\n",
                         pBootRomMsg->msgLen, pBootRomMsg->pktLen,
                         pBootRomMsg->imageLen, pBootRomMsg->bytesLeft));

    if ((ROUND_UP(pBootRomMsg->imageLen, DATA_BUFFER_SIZE)) > 
        (ROUND_DOWN(memAvail, DATA_BUFFER_SIZE) - IMAGE_ALIGN_BYTES)) {

        /* Available Memory less than requested by the host 
         * so reply back to the host with the available memory 
         */
        pBootRomMsg->msgLen += sizeof(A_UINT32);
        /* Set the status value */
        pBootRomMsg->opStatus = ROUND_DOWN(memAvail, DATA_BUFFER_SIZE) - IMAGE_ALIGN_BYTES;
        /* Send the response back */
        status = athUsbDrvSend( pBootromCtxt->drvHandle, MESSAGE_PIPE, 
                                pBootromCtxt->pMessageBuffer);
        if (status != A_OK ) {
            BOOTROM_DEBUG_PRINTF(BOOTROM_INIT_DEBUG ,("Send Failed\n"));
            ASSERT (FALSE);
        }
        BOOTROM_VERBOSE_PRINTF(("Request from Host Failed, Size Requested - %d Available - %d\n", 
                                pBootRomMsg->imageLen, (A_UINT32)getAvailableHeap()));

    } else {

        pImageInfo = &pBootromCtxt->imageInfo;
        /* Allocate place holder for the Image from the Heap */
        pImageInfo->imageAddress = A_DRIVER_MALLOC (sizeof(A_UINT8) * 
                                                    ROUND_UP(pBootRomMsg->imageLen, DATA_BUFFER_SIZE) + 
                                                    IMAGE_ALIGN_BYTES);
        if (pImageInfo->imageAddress) {

            /* Align the address to the 256 byte memory */
            pImageInfo->imageAlignAddr = (A_UINT8 *)ROUND_UP((A_UINT32) pBootromCtxt->imageInfo.imageAddress,
                                                             IMAGE_ALIGN_BYTES);
            BOOTROM_VERBOSE_PRINTF(("Downloading image @ 0x%08x size - %d bytes\n",
                                    pImageInfo->imageAlignAddr, pBootRomMsg->imageLen));

            /* Post a receive for the first buffer and move into the data state */
            status = athUsbDrvRecv( pBootromCtxt->drvHandle, DATA_PIPE, 
                                    pImageInfo->imageAlignAddr);
            if (status != A_OK ) {
                BOOTROM_DEBUG_PRINTF(BOOTROM_INIT_DEBUG ,("Recv Failed\n"));
                ASSERT (FALSE);
            }
            pImageInfo->imagePtr = pImageInfo->imageAlignAddr + pBootRomMsg->pktLen;
            pImageInfo->imageLen = pBootRomMsg->imageLen;
            pBootromCtxt->bootRomState = BOOTROM_DATA_STATE;
        } else {
            /* 
             * meminfo reported memory is available but allocating that
             * memory failed.
             */
            pBootRomMsg->msgLen += sizeof(A_UINT32);
            pBootRomMsg->opStatus = 0xFFFFFFFF;
            status = athUsbDrvSend( pBootromCtxt->drvHandle, MESSAGE_PIPE, 
                                    pBootromCtxt->pMessageBuffer);
            if (status != A_OK ) {
                BOOTROM_DEBUG_PRINTF(BOOTROM_INIT_DEBUG ,("Send Failed\n"));
                ASSERT (FALSE);
            }
            BOOTROM_VERBOSE_PRINTF(("Request from Host Failed, Memory not available\n"));
        }
    }
    return;
}

/******************************************************************************
* commandStateHandler
*
* Returns: void
*/
void commandStateHandler (
    IN BOOTROM_CONTEXT  *pBootromCtxt,
    IN A_UINT8          *pBuffer
    )
{
    BOOTROM_MESSAGE     *pBootRomMsg;
    A_STATUS            status;
    
    A_DATA_CACHE_INVAL(pBuffer, sizeof(A_UINT32)); 
    pBootRomMsg = (BOOTROM_MESSAGE *)pBuffer;

    ASSERT (pBootRomMsg->msgLen <= MESSAGE_BUFFER_SIZE);
    A_DATA_CACHE_INVAL(pBuffer, pBootRomMsg->msgLen); 

    BOOTROM_DEBUG_PRINTF(BOOTROM_MESSAGE_DEBUG ,("%d:%d:%d:%d\n",
                         pBootRomMsg->msgLen ,pBootRomMsg->pktLen,
                         pBootRomMsg->imageLen, pBootRomMsg->bytesLeft));

    if (pBootRomMsg->imageLen > pBootromCtxt->imageInfo.imageLen) {
        
        /* Image length seems to have changed send a -ve resp */
        pBootRomMsg->msgLen += sizeof(A_UINT32);
        pBootRomMsg->opStatus = pBootromCtxt->imageInfo.imageLen;
        status = athUsbDrvSend( pBootromCtxt->drvHandle, MESSAGE_PIPE, 
                                pBootromCtxt->pMessageBuffer);
        if (status != A_OK ) {
            BOOTROM_DEBUG_PRINTF(BOOTROM_INIT_DEBUG ,("Send Failed\n"));
            ASSERT(FALSE);
        }
    } else {

        /* Post a receive on the Data pipe */
        status = athUsbDrvRecv( pBootromCtxt->drvHandle, DATA_PIPE, 
                                pBootromCtxt->imageInfo.imagePtr);
        if (status != A_OK ) {
            BOOTROM_DEBUG_PRINTF(BOOTROM_INIT_DEBUG ,("Recv Failed\n"));
            ASSERT(FALSE);
        }
        pBootromCtxt->imageInfo.imagePtr = pBootromCtxt->imageInfo.imagePtr + pBootRomMsg->pktLen;
        pBootromCtxt->bootRomState = BOOTROM_DATA_STATE;
    }
    return;
}

/******************************************************************************
* dataStateHandler
*
* Returns: void
*/
void dataStateHandler (
    IN BOOTROM_CONTEXT  *pBootromCtxt,
    IN A_UINT8          *pBuffer
    )
{
    BOOTROM_MESSAGE     *pBootRomMsg;
    A_STATUS            status;
    unsigned long       oldints;
    
    /* Received data on the Data Pipe, Sending Bootrom Cfm Command */
    pBootRomMsg = (BOOTROM_MESSAGE *)pBootromCtxt->pMessageBuffer;
    pBootRomMsg->msgLen += sizeof(A_UINT32);
    pBootRomMsg->opStatus = pBootRomMsg->imageLen;
    
    status = athUsbDrvSend( pBootromCtxt->drvHandle, MESSAGE_PIPE, 
                            pBootromCtxt->pMessageBuffer);
    ASSERT (status == A_OK);
    if (pBootRomMsg->bytesLeft == 0) {
        BOOTROM_VERBOSE_PRINTF(("Download complete, executing image @ 0x%08x\n\n",
                                pBootromCtxt->imageInfo.imageAlignAddr));
        HAL_DISABLE_INTERRUPTS(oldints);
        HAL_DCACHE_SYNC();
        HAL_ICACHE_DISABLE();
        HAL_DCACHE_DISABLE();
        HAL_DCACHE_SYNC();
        HAL_ICACHE_INVALIDATE_ALL();
        HAL_DCACHE_INVALIDATE_ALL();

        ((void (*)(void))pBootromCtxt->imageInfo.imageAlignAddr)();
    }
    pBootromCtxt->bootRomState = BOOTROM_COMMAND_STATE;
    /* Jump to the downloaded Image */

    return;
}

/******************************************************************************
* bootRomEventHandler
*
* Returns: void
*/

void bootRomEventHandler(
    IN BOOTROM_CONTEXT *pBootromCtxt,
    IN A_UINT8          bootromEvent,
    IN A_UINT8          pipeNum,
    IN A_UINT8          *pBuffer,
    IN A_UINT32         rxtxbytes
    )
{
    A_STATUS            status;

    if (bootromEvent == BOOTROM_RECV) {
        if (rxtxbytes) {
            switch (pBootromCtxt->bootRomState) {

                case BOOTROM_INIT_COMMAND_STATE : 
                    ASSERT(pipeNum == MESSAGE_PIPE);
                    initStateHandler (pBootromCtxt, pBuffer);
                    break;
                
                case BOOTROM_DATA_STATE : 
                    ASSERT(pipeNum == DATA_PIPE);
                    dataStateHandler (pBootromCtxt, pBuffer);
                    break;

                case BOOTROM_COMMAND_STATE : 
                    ASSERT(pipeNum == MESSAGE_PIPE);
                    commandStateHandler (pBootromCtxt, pBuffer);
                    break;

                default:
                    ASSERT(FALSE);
                    break;
            }
        } else {
            /* Received Empty buffer on the Pipe, post the buffer back 
             * again for receive 
             */
            status = athUsbDrvRecv( pBootromCtxt->drvHandle, pipeNum, pBuffer);
            ASSERT (status == A_OK);
        }
    } else {
        ASSERT(rxtxbytes);
        ASSERT(pipeNum == MESSAGE_PIPE);
        /* On Getting a Send Confirm just post back the buffer */
        status = athUsbDrvRecv( pBootromCtxt->drvHandle, pipeNum, pBuffer);
        ASSERT (status == A_OK);
    }
}

/******************************************************************************
* BootRom
*
* Returns: void
*/
void BootRom(A_UINT8 flashDisk)
{
    A_STATUS status;
    BOOTROM_CONTEXT  *pBootromCtxt;
    if (pFlashPromCtxt != 0) {
        BOOTROM_DEBUG_PRINTF(BOOTROM_INIT_DEBUG, ("\nClean Up Existing connection\n")); 
        usbBootromClean(pFlashPromCtxt);
        pFlashPromCtxt=0;
    }
#ifdef BOOT_LOADER_PLUS
    ZG_flashDisk=flashDisk;
    #if SPLIT_BOOT_LOADER   //If Normal Boot Loader is available in the sector  ATC_BOOT_LOADER_SECTOR
    if ( ZG_flashDisk == FALSE )
    {
        A_UINT8         *imageAlignAddr;
        unsigned long       oldints;
        imageAlignAddr=(A_UINT8 *)(SPI_FLASH_BASE_ADDRESS+ATC_BOOT_LOADER_SECTOR*0x10000);//Boot Loader available in Sector 3
        BOOTROM_VERBOSE_PRINTF(("Executing BootLoader from @ 0x%08x\n\n",imageAlignAddr));
        HAL_DISABLE_INTERRUPTS(oldints);
        HAL_DCACHE_SYNC();
        HAL_ICACHE_DISABLE();
        HAL_DCACHE_DISABLE();
        HAL_DCACHE_SYNC();
        HAL_ICACHE_INVALIDATE_ALL();
        HAL_DCACHE_INVALIDATE_ALL();
        ((void (*)(void))imageAlignAddr)();
        return;
    }
    #endif //If Boot Loader is not part of Boot Loader Plus
#endif //BOOT_LOADER_PLUS

    status = usbBootromInit(&pBootromCtxt);
    if (status != A_OK) {
        usbBootromClean(pBootromCtxt);
    }
    pFlashPromCtxt=pBootromCtxt;
    /* Let the control flow to ECOS to initialize the 
     * scheduler and further to the idle Task, only then we 
     * would be able to run the DSRs. 
     * The bootrom SM are run from the DSR context, since 
     * there is no other task running in the bootrom.
     */
}

void bootRomRecvIndHandler( 
    IN APP_HANDLE   appHandle,
    IN A_UINT8      pipeNum,
    IN A_UINT8      *pBuffer,
    IN A_UINT32     bytesReceived
    )
{
    BOOTROM_CONTEXT        *pBootromCtxt;
    
    pBootromCtxt = (BOOTROM_CONTEXT *)appHandle;
    ASSERT (pBootromCtxt);

    /* Invoke the Bootloader SM to handle Recv */
#ifdef BOOT_LOADER_PLUS
    if (ZG_flashDisk == FALSE) {
#endif //BOOT_LOADER_PLUS
        BOOTROM_DEBUG_PRINTF(BOOTROM_RECV_DEBUG, ("BootRom::Rx : pipe - %d buf - 0x%x size - %d\n", pipeNum, pBuffer, bytesReceived)); 
        bootRomEventHandler(pBootromCtxt, BOOTROM_RECV, pipeNum, pBuffer, bytesReceived);
#ifdef BOOT_LOADER_PLUS
    } else {
        BOOTROM_DEBUG_PRINTF(BOOTROM_RECV_DEBUG, ("Falsh Disk::Rx : pipe - %d buf - 0x%x size - %d\n", pipeNum, pBuffer, bytesReceived));
        flashDiskRecvIndHandler(appHandle,pipeNum,pBuffer,bytesReceived);
    }
#endif //BOOT_LOADER_PLUS

}


void bootRomSendCfmHandler(
    IN APP_HANDLE   appHandle,
    IN A_UINT8      pipeNum,
    IN A_UINT8      *pBuffer,
    IN A_UINT32     bytesSent
    )
{
    BOOTROM_CONTEXT        *pBootromCtxt;

    pBootromCtxt = (BOOTROM_CONTEXT *)appHandle;
    ASSERT (pBootromCtxt);

    /* Invoke the Bootloader SM to handle send */
#ifdef BOOT_LOADER_PLUS
    if (ZG_flashDisk == TRUE) {
        BOOTROM_DEBUG_PRINTF(BOOTROM_SEND_DEBUG, 
                             ("Falsh Disk::Tx : pipe - %d buf - 0x%x size - %d\n", 
                              pipeNum, pBuffer, bytesSent));
        flashDiskSendCfmHandler(appHandle,pipeNum,pBuffer,bytesSent);
    } else {
#endif //BOOT_LOADER_PLUS
        BOOTROM_DEBUG_PRINTF(BOOTROM_SEND_DEBUG, 
                             ("BootRom::Tx : pipe - %d buf - 0x%x size - %d\n", 
                              pipeNum, pBuffer, bytesSent));
        bootRomEventHandler(pBootromCtxt, BOOTROM_SEND, pipeNum, pBuffer, bytesSent);        
#ifdef BOOT_LOADER_PLUS
    }
#endif //BOOT_LOADER_PLUS
}

#ifndef BOOT_LOADER_PLUS
/******************************************************************************
* cyg_user_start
*
* Entry for the ECOS 
*
*/
void cyg_user_start(void)
{
    BootRom(FALSE);
}
#endif //BOOT_LOADER_PLUS


