/*
 * $Id: //depot/sw/branches/1.3_USB_LINUX_port/src/USB/transport/usb/target/arcusb/arcusb.c#1 $
 *
 * Copyright (c) 2000-2003 Atheros Communications, Inc., All Rights Reserved
 *
 * Atheros USB target driver for the AR5523 Chip
 */

#include "usb.h"
#include "devapi.h"
#include "athusb.h"
#include "stdlib.h"
#include "string.h"

#ifdef USB_DRV_BOOTROM_CONFIG

/* 
 * Only two hardware endpoints are 
 * exposed to the software 
 */
#define MAX_HW_RECV_ENDPOINTS           (2)
#define MAX_HW_SEND_ENDPOINTS           (2)

#else

/* 
 * All the available endpoints are 
 * exposed to the software 
 */
#define MAX_HW_RECV_ENDPOINTS           (2)
#define MAX_HW_SEND_ENDPOINTS           (2)

#endif /* USB_DRV_BOOTROM_CONFIG */

#ifdef DO_ZERO_TERM
#define ZERO_TERMINATE                  (1)
#else
#define ZERO_TERMINATE                  (0)
#endif

#define EP_FS_MAX_PACKET_SIZE           (64)
#define EP_HS_MAX_PACKET_SIZE           (512)
#define APP_CONTROL_MAX_PKT_SIZE        (64)
#define MAX_HW_ENDPOINTS                (MAX_HW_SEND_ENDPOINTS + MAX_HW_RECV_ENDPOINTS)


volatile uchar      speed = 0;
volatile boolean    ENTER_TEST_MODE = FALSE;
volatile uint_16    test_mode_index = 0;

/* Device Descriptor Offsets */          
#define  DEV_DESC_LENGTH_OFFSET         (0)
#define  DEV_DESC_TYPE_OFFSET           (1)
#define  DEV_DESC_USBVER_LO_OFFSET      (2)
#define  DEV_DESC_USBVER_HI_OFFSET      (3)
#define  DEV_DESC_DEV_CLASS_OFFSET      (4)
#define  DEV_DESC_DEV_SUBCLASS_OFFSET   (5)
#define  DEV_DESC_DEVPROTOCOL_OFFSET    (6)
#define  DEV_DESC_MAX_PKT_SIZE_OFFSET   (7)
#define  DEV_DESC_VENDOR_LO_OFFSET      (8)
#define  DEV_DESC_VENDOR_HI_OFFSET      (9)
#define  DEV_DESC_PRODUCT_LO_OFFSET     (10)
#define  DEV_DESC_PRODUCT_HI_OFFSET     (11)
#define  DEV_DESC_DEV_RELNUM_LO_OFFSET  (12)
#define  DEV_DESC_DEV_RELNUM_HI_OFFSET  (13)
#define  DEV_DESC_MFT_STR_str_OFFSET    (14)
#define  DEV_DESC_PRODUCT_STR_OFFSET    (15)
#define  DEV_DESC_SERIALNUM_STR_OFFSET  (16)
#define  DEV_DESC_NUMCFG_OFFSET         (17)
#define  DEV_DESC_LEN                   (18)
          
/* Device Qualifier desc offsets */
#define  DEVQUAL_DESC_LENGTH_OFFSET     (0)
#define  DEVQUAL_DESC_TYPE_OFFSET       (1)
#define  DEVQUAL_DESC_USBVER_LO_OFFSET  (2)
#define  DEVQUAL_DESC_USBVER_HI_OFFSET  (3)
#define  DEVQUAL_DEV_CLASS_OFFSET       (4)
#define  DEVQUAL_DEV_SUBCLASS_OFFSET    (5)
#define  DEVQUAL_DEVPROTOCOL_OFFSET     (6)
#define  DEVQUAL_MAX_PKT_SIZE_OFFSET    (7)
#define  DEVQUAL_NUMCFG_OFFSET          (8)
#define  DEVQUAL_DESC_LEN               (10)

/* Config Part of Cfg Descriptor Offsets */
#define  CONFIG_DESC_LENGTH_OFFSET      (0)  
#define  CONFIG_DESC_TYPE_OFFSET        (1)
#define  CONFIG_DESC_TOT_LEN_LO         (2)
#define  CONFIG_DESC_TOT_LEN_HI         (3)
#define  CONFIG_DESC_NUM_IF_OFFSET      (4)
#define  CONFIG_DESC_CFG_ID_OFFSET      (5)
#define  CONFIG_DESC_CFG_STR_OFFSET     (6)
#define  CONFIG_DESC_ATTRIB_OFFSET      (7)
#define  CONFIG_DESC_MAX_PWR_OFFSET     (8)     
#define  CONFIG_DESC_LEN                (9)

/* Interface Descriptor Offsets */
#define  IF_DESC_LENGTH_OFFSET          (0)  
#define  IF_DESC_TYPE_OFFSET            (1)
#define  IF_DESC_ID_OFFSET              (2)
#define  IF_DESC_ALT_OFFSET             (3)
#define  IF_DESC_NUM_EP_OFFSET          (4)
#define  IF_DESC_IF_CLASS_OFFSET        (5)
#define  IF_DESC_IF_SUBCLASS_OFFSET     (6)
#define  IF_DESC_IF_PROTOCOL_OFFSET     (7)
#define  IF_DESC_IF_STR_OFFSET          (8)
#define  INTERFACE_DESC_LEN             (9)

/* Endpoint Descriptor Offsets */
#define  EP_DESC_LENGTH_OFFSET          (0)  
#define  EP_DESC_TYPE_OFFSET            (1)
#define  EP_DESC_ENDPOINT_OFFSET        (2)
#define  EP_DESC_ATTRIBUTES_OFFSET      (3)
#define  EP_DESC_MAX_PKT_LO_OFFSET      (4)
#define  EP_DESC_MAX_PKT_HI_OFFSET      (5)
#define  EP_DESC_INTERVAL_OFFSET        (6)
#define  EP_DESC_LEN                    (7)

/* Config Descriptor Attribs */
#define  CONFIG_DESC_SIZE               (CONFIG_DESC_LEN + INTERFACE_DESC_LEN + MAX_HW_ENDPOINTS * EP_DESC_LEN)
#define  CONFIG_DESC_EP_DESC_OFFSET     (CONFIG_DESC_LEN + INTERFACE_DESC_LEN)
#define  CONFIG_DESC_IF_DESC_OFFSET     (CONFIG_DESC_LEN)
#define  CONFIG_DESC_NUM_INTERFACES     (1)

#define  DEFAULT_USB_VERSION            (0x0200)
#define  DEFAULT_USB_VENDORID           (0x0CF3)
#define  DEFAULT_USB_PRODUCTID          (0x0001)
#define  DEFAULT_USB_BROM_PRODUCTID     (0x0002)
#define  DEFAULT_USB_DEVRELNUM          (0x0001)

/* number of strings in the table not including 0 or n. */
const uint_8 USB_STR_NUM = 3;

#define  DEFAULT_MFG_STR_INDEX          1
#define  DEFAULT_PRODUCT_STR_INDEX      2
#define  DEFAULT_SERIALNUM_STR_INDEX    3

uchar  USB_IF_ALT[4] = { 0, 0, 0, 0};
uint_8 USB_STR_0[4] = { sizeof(USB_STR_0), 0x03, 0x09, 0x04};
uint_8 USB_STR_n[4] = { sizeof(USB_STR_n), 0x03, ' ', 0};

#define USBSTR(_x)      USB_STR_##_x
#define USBSTRN(_x)     USBSTR(_x)

/*Added for Mass Storage Device Support*/
#ifdef BOOT_LOADER_PLUS
A_UINT8 ZG_flashDisk=0;
extern const A_UINT8  FD_DevDesc[18];
extern A_UINT8 FD_ConfigDesc[];
extern void MassStorageClass(_usb_device_handle handle,boolean setup,SETUP_STRUCT_PTR setup_ptr);
extern void getDescription(_usb_device_handle handle,boolean setup,SETUP_STRUCT_PTR setup_ptr);
#endif
/*End of Mass Storage Device Support declaration*/

uint_8 USBSTRN(DEFAULT_MFG_STR_INDEX)[USB_MANUFACTURER_STR_SIZE + 2] = { 54, 0x03,
    'A',0,'t',0,'h',0,'e',0,'r',0,'o',0,'s',0,' ',0,
    'C',0,'o',0,'m',0,'m',0,'u',0,'n',0,'i',0,'c',0,'a',0,'t',0,
    'i',0,'o',0,'n',0,'s',0,' ',0,'I',0,'n',0,'c',0};
    
uint_8 USBSTRN(DEFAULT_PRODUCT_STR_INDEX)[USB_PRODUCT_STR_SIZE + 2] = { 14, 0x03,
    'A',0,'R',0,'5',0,'5',0,'2',0,'3',0};

uint_8 USBSTRN(DEFAULT_SERIALNUM_STR_INDEX)[USB_SERIALNUM_STR_SIZE + 2] = { 8, 0x03,
    '1',0,'.',0,'0',0};

const uint_8_ptr USB_STRING_DESC[] =
{
    (uint_8_ptr)((pointer)USB_STR_0),
    (uint_8_ptr)((pointer)USB_STR_1),
    (uint_8_ptr)((pointer)USB_STR_2),
    (uint_8_ptr)((pointer)USB_STR_3),
    (uint_8_ptr)((pointer)USB_STR_n),
};



/* Device descriptors are always 18 bytes */
uint_8  DevDesc[DEV_DESC_LEN] =
{
    /* Length of DevDesc */
    sizeof(DevDesc),
    /* "Device" Type of descriptor */
    1,
    /* BCD USB version */
    USB_uint_16_low(DEFAULT_USB_VERSION), USB_uint_16_high(DEFAULT_USB_VERSION),
    /* Device Class  */
    0xFF,
    /* Device Subclass  */
    0x00,
    /* Device protocol */
    0x00,
    /* Max packet size */
    APP_CONTROL_MAX_PKT_SIZE,
    /* Vendor ID */
    USB_uint_16_low(DEFAULT_USB_VENDORID), USB_uint_16_high(DEFAULT_USB_VENDORID),
    /* Product ID */
    USB_uint_16_low(DEFAULT_USB_PRODUCTID), USB_uint_16_high(DEFAULT_USB_PRODUCTID),
    /* BCD Device version */
    USB_uint_16_low(DEFAULT_USB_DEVRELNUM), USB_uint_16_high(DEFAULT_USB_DEVRELNUM),
    /* Manufacturer string index */
    DEFAULT_MFG_STR_INDEX,
    /* Product string index */
    DEFAULT_PRODUCT_STR_INDEX,
    /* Serial number string index */
    DEFAULT_SERIALNUM_STR_INDEX,
    /* Number of configurations available */
    0x1
};

/* USB 2.0 specific descriptor */
uint_8  DevQualifierDesc[DEVQUAL_DESC_LEN] =
{
    /* Length of this descriptor */
    sizeof(DevQualifierDesc),  
    /* This is a DEVICE Qualifier descr */
    6,                         
    /* USB revision */
    USB_uint_16_low(DEFAULT_USB_VERSION), USB_uint_16_high(DEFAULT_USB_VERSION),  
    /* Device Class */
    0xFF,                      
    /* Device SubClass */
    0,                         
    /* Device Protocol */
    0,                         
    /* Max packet size  */
    APP_CONTROL_MAX_PKT_SIZE,  
    /* Number of configurations available */
    0x01,
    /* Reserved */
    0
};


uint_8 ConfigDesc[] =
{
    /***************  Configuration Descriptor ***************/
    /* Configuration Descriptor length - always 9 bytes */
    9,
    /* "Configuration" type of descriptor */
    2,
    /* Total length of the Configuration descriptor */
    USB_uint_16_low(CONFIG_DESC_SIZE), USB_uint_16_high(CONFIG_DESC_SIZE),
    /* NumInterfaces */
    CONFIG_DESC_NUM_INTERFACES,
    /* Configuration Value */
    1,
    /* Configuration Description String Index*/
    0,
#ifdef BUS_POWERED
    /* Attributes.  Bus-powered. */
    0x80,
#else
    /* Attributes.  Bus-powered. */
    0xC0,
#endif 
#ifdef BUS_POWERED
    /* Current draw from bus - 500 mA*/
    250, 
#else
    /* Current draw from bus */
    0, 
#endif
    /***************  Interface Descriptor ******************/

    /* Interface 0 Descriptor - always 9 bytes */
    9,
    /* "Interface" type of descriptor */
    4,
    /* Number of this interface */
    0,
    /* Alternate Setting */
    0,
    /* Number of endpoints on this interface */
    MAX_HW_ENDPOINTS,
    /* Interface Class - Vendor Specific */
    0xFF,
    /* Interface Subclass:  Unknown */
    0x00,
    /* Interface Protocol:  Unknown */
    0x00,
    /* Interface Description String Index */
    0,

    /* ************ Endpoint 1 (Bulk In), Interface 0 Descriptor **************/
    /* Endpoint length - 7 bytes */
    7,
    /* "Endpoint" type of descriptor */
    5,
    /*
    ** Endpoint address.  The low nibble contains the endpoint number and the
    ** high bit indicates TX(1) or RX(0).
    */
    0x81,
    /* Attributes.  0=Control 1=Isochronous 2=Bulk 3=Interrupt */
    0x02,
    /* Max Packet Size for this endpoint */
    USB_uint_16_low(EP_FS_MAX_PACKET_SIZE), 
    USB_uint_16_high(EP_FS_MAX_PACKET_SIZE),
    /* Polling Interval (ms) */
    0,

    /* ************ Endpoint 2 (Bulk Out) ***********************************/
    /* Endpoint length - 7 bytes */
    7,
    /* "Endpoint" type of descriptor */
    5,
    /*
    ** Endpoint address.  The low nibble contains the endpoint number and the
    ** high bit indicates TX(1) or RX(0).
    */
    0x01,
    /* Attributes.  0=Control 1=Isochronous 2=Bulk 3=Interrupt */
    0x02,
    /* Max Packet Size for this endpoint */
    USB_uint_16_low(EP_FS_MAX_PACKET_SIZE), 
    USB_uint_16_high(EP_FS_MAX_PACKET_SIZE),
    /* Polling Interval (ms) */
    0,
    /* ************   Endpoint 3 (Interrupt/Bulk IN) Descriptor *******************/
    /* Endpoint length - 7 bytes */
    7,

    /* "Endpoint" type of descriptor */
    5,
    /*
    ** Endpoint address.  The low nibble contains the endpoint number and the
    ** high bit indicates TX(1) or RX(0).
    */
    0x82,
    /* Attributes.  0=Control 1=Isochronous 2=Bulk 3=Interrupt */
#ifdef INTERRUPT_ENDPOINT
    0x03,
#else
    0x02,
#endif
    /* Max Packet Size for this endpoint */
    USB_uint_16_low(EP_FS_MAX_PACKET_SIZE), 
    USB_uint_16_high(EP_FS_MAX_PACKET_SIZE),
    /* Polling Interval (ms) */
    0,
    /* ************   Endpoint 4 (Bulk Out) Descriptor *******************/
    /* Endpoint length - 7 bytes */
    7,

    /* "Endpoint" type of descriptor */
    5,
    /*
    ** Endpoint address.  The low nibble contains the endpoint number and the
    ** high bit indicates TX(1) or RX(0).
    */
    0x02,
    /* Attributes.  0=Control 1=Isochronous 2=Bulk 3=Interrupt */
    0x02,
    /* Max Packet Size for this endpoint */
    USB_uint_16_low(EP_FS_MAX_PACKET_SIZE), 
    USB_uint_16_high(EP_FS_MAX_PACKET_SIZE),
    /* Polling Interval (ms) */
    0,
    /**************   Endpoint 5 (Bulk Out) Descriptor *******************/
    /* Endpoint length - 7 bytes */
    7,
    /* "Endpoint" type of descriptor */
    5,
    /*
    ** Endpoint address.  The low nibble contains the endpoint number and the
    ** high bit indicates TX(1) or RX(0).
    */
    0x03,
    /* Attributes.  0=Control 1=Isochronous 2=Bulk 3=Interrupt */
    0x02,
    /* Max Packet Size for this endpoint */
    USB_uint_16_low(EP_FS_MAX_PACKET_SIZE), 
    USB_uint_16_high(EP_FS_MAX_PACKET_SIZE),
    /* Polling Interval (ms) */
    0,
    /**************   Endpoint 6 (Bulk Out) Descriptor *******************/
    /* Endpoint length - 7 bytes */
    7,
    /* "Endpoint" type of descriptor */
    5,
    /*
    ** Endpoint address.  The low nibble contains the endpoint number and the
    ** high bit indicates TX(1) or RX(0).
    */
    0x04,
    /* Attributes.  0=Control 1=Isochronous 2=Bulk 3=Interrupt */
    0x02,
    /* Max Packet Size for this endpoint */
    USB_uint_16_low(EP_FS_MAX_PACKET_SIZE), 
    USB_uint_16_high(EP_FS_MAX_PACKET_SIZE),
    /* Polling Interval (ms) */
    0,
    /**************   Endpoint 7 (Bulk Out) Descriptor *******************/
    /* Endpoint length - 7 bytes */
    7,
    /* "Endpoint" type of descriptor */
    5,
    /*
    ** Endpoint address.  The low nibble contains the endpoint number and the
    ** high bit indicates TX(1) or RX(0).
    */
    0x05,
    /* Attributes.  0=Control 1=Isochronous 2=Bulk 3=Interrupt */
    0x02,
    /* Max Packet Size for this endpoint */
    USB_uint_16_low(EP_FS_MAX_PACKET_SIZE), 
    USB_uint_16_high(EP_FS_MAX_PACKET_SIZE),
    /* Polling Interval (ms) */
    0,
    /**************   Endpoint 8 (Bulk Out) Descriptor *******************/
    /* Endpoint length - 7 bytes */
    7,
    /* "Endpoint" type of descriptor */
    5,
    /*
    ** Endpoint address.  The low nibble contains the endpoint number and the
    ** high bit indicates TX(1) or RX(0).
    */
    0x06,
    /* Attributes.  0=Control 1=Isochronous 2=Bulk 3=Interrupt */
    0x02,
    /* Max Packet Size for this endpoint */
    USB_uint_16_low(EP_FS_MAX_PACKET_SIZE), 
    USB_uint_16_high(EP_FS_MAX_PACKET_SIZE),
    /* Polling Interval (ms) */
    0,
    /**************   Endpoint 9 (Bulk Out) Descriptor *******************/
    /* Endpoint length - 7 bytes */
    7,
    /* "Endpoint" type of descriptor */
    5,
    /*
    ** Endpoint address.  The low nibble contains the endpoint number and the
    ** high bit indicates TX(1) or RX(0).
    */
    0x07,
    /* Attributes.  0=Control 1=Isochronous 2=Bulk 3=Interrupt */
    0x02,
    /* Max Packet Size for this endpoint */
    USB_uint_16_low(EP_FS_MAX_PACKET_SIZE), 
    USB_uint_16_high(EP_FS_MAX_PACKET_SIZE),
    /* Polling Interval (ms) */
    0,
    /**************   Endpoint 10 (Bulk Out) Descriptor *******************/
    /* Endpoint length - 7 bytes */
    7,
    /* "Endpoint" type of descriptor */
    5,
    /*
    ** Endpoint address.  The low nibble contains the endpoint number and the
    ** high bit indicates TX(1) or RX(0).
    */
    0x08,
    /* Attributes.  0=Control 1=Isochronous 2=Bulk 3=Interrupt */
    0x02,
    /* Max Packet Size for this endpoint */
    USB_uint_16_low(EP_FS_MAX_PACKET_SIZE), 
    USB_uint_16_high(EP_FS_MAX_PACKET_SIZE),
    /* Polling Interval (ms) */
    0,
    /**************   Endpoint 11 (Bulk Out) Descriptor *******************/
    /* Endpoint length - 7 bytes */
    7,
    /* "Endpoint" type of descriptor */
    5,
    /*
    ** Endpoint address.  The low nibble contains the endpoint number and the
    ** high bit indicates TX(1) or RX(0).
    */
    0x09,
    /* Attributes.  0=Control 1=Isochronous 2=Bulk 3=Interrupt */
    0x02,
    /* Max Packet Size for this endpoint */
    USB_uint_16_low(EP_FS_MAX_PACKET_SIZE), 
    USB_uint_16_high(EP_FS_MAX_PACKET_SIZE),
    /* Polling Interval (ms) */
    0,
    /**************   Endpoint 12 (Bulk Out) Descriptor *******************/
    /* Endpoint length - 7 bytes */
    7,
    /* "Endpoint" type of descriptor */
    5,
    /*
    ** Endpoint address.  The low nibble contains the endpoint number and the
    ** high bit indicates TX(1) or RX(0).
    */
    0x0a,
    /* Attributes.  0=Control 1=Isochronous 2=Bulk 3=Interrupt */
    0x02,
    /* Max Packet Size for this endpoint */
    USB_uint_16_low(EP_FS_MAX_PACKET_SIZE), 
    USB_uint_16_high(EP_FS_MAX_PACKET_SIZE),
    /* Polling Interval (ms) */
    0,
};

uint_8  other_speed_config[] =
{
    /* Descriptor length - always 9 bytes */
    9,
    /* Descriptor Type - This is a Other speed config descr */
    7,                         
    /* Length of this descriptor */
    USB_uint_16_low(CONFIG_DESC_SIZE), USB_uint_16_high(CONFIG_DESC_SIZE),
    /* NumInterfaces */
    CONFIG_DESC_NUM_INTERFACES,
    /* Configuration Value */
    1,
    /* Configuration Description String Index*/
    0,
#ifdef BUS_POWERED
#ifdef BOOTROM
    /* Attributes.  Bus-powered. */
    0x80,
#else
    /* Attributes.  Bus-powered. */
    0xA0,
#endif
#else
#ifdef BOOTROM
    /* Attributes.  Bus-powered. */
    0xC0,
#else
    /* Attributes.  Self-powered. */
    0xE0, 
#endif
#endif 
#ifdef BUS_POWERED
    /* Current draw from bus */
    250,             
#else
    /* Current draw from bus */
    0,              
#endif
    /***************  Interface Descriptor ******************/

    /* Interface 0 Descriptor - always 9 bytes */
    9,
    /* "Interface" type of descriptor */
    4,
    /* Number of this interface */
    0,
    /* Alternate Setting */
    0,
    /* Number of endpoints on this interface -  Intr Endpoint to be added later */
    MAX_HW_ENDPOINTS,
    /* Interface Class - Vendor Specific */
    0xFF,
    /* Interface Subclass:  Unknown */
    0x00,
    /* Interface Protocol:  Unknown */
    0x00,
    /* Interface Description String Index */
    0,

    /* ************   Endpoint 1 (Bulk IN) Descriptor *******************/
    /* Endpoint 1 (Bulk In Endpoint), Interface 0 Descriptor - always 7 bytes*/
    7,
    /* "Endpoint" type of descriptor */
    5,
    /*
    ** Endpoint address.  The low nibble contains the endpoint number and the
    ** high bit indicates TX(1) or RX(0).
    */
    0x81,
    /* Attributes.  0=Control 1=Isochronous 2=Bulk 3=Interrupt */
    0x02,
    /* Max Packet Size for this endpoint */
    USB_uint_16_low(EP_FS_MAX_PACKET_SIZE), 
    USB_uint_16_high(EP_FS_MAX_PACKET_SIZE),
    /* Polling Interval (ms) */
    0,
    /* ************   Endpoint 2 (Bulk Out) Descriptor *******************/
    /* Endpoint 2 (Bulk Out Endpoint), Interface 0 Descriptor - always 7 bytes*/
    7,
    /* "Endpoint" type of descriptor */
    5,
    /*
    ** Endpoint address.  The low nibble contains the endpoint number and the
    ** high bit indicates TX(1) or RX(0).
    */
    0x01,
    /* Attributes.  0=Control 1=Isochronous 2=Bulk 3=Interrupt */
    0x02,
    /* Max Packet Size for this endpoint */
    USB_uint_16_low(EP_FS_MAX_PACKET_SIZE), 
    USB_uint_16_high(EP_FS_MAX_PACKET_SIZE),
    /* Polling Interval (ms) */
    0,
    /* ************   Endpoint 3 (Interrupt / Bulk IN) Descriptor *******************/
    /* Endpoint length - 7 bytes */
    7,

    /* "Endpoint" type of descriptor */
    5,
    /*
    ** Endpoint address.  The low nibble contains the endpoint number and the
    ** high bit indicates TX(1) or RX(0).
    */
    0x82,
    /* Attributes.  0=Control 1=Isochronous 2=Bulk 3=Interrupt */
#ifdef INTERRUPT_ENDPOINT
    0x03,
#else
    0x02,
#endif
    /* Max Packet Size for this endpoint */
    USB_uint_16_low(EP_FS_MAX_PACKET_SIZE), 
    USB_uint_16_high(EP_FS_MAX_PACKET_SIZE),
    /* Polling Interval (ms) */
    0,
    /* ************   Endpoint 4 (Bulk Out) Descriptor *******************/
    /* Endpoint 3 (Bulk Out Endpoint), Interface 0 Descriptor - always 7 bytes*/
    7,
    /* "Endpoint" type of descriptor */
    5,
    /*
    ** Endpoint address.  The low nibble contains the endpoint number and the
    ** high bit indicates TX(1) or RX(0).
    */
    0x02,
    /* Attributes.  0=Control 1=Isochronous 2=Bulk 3=Interrupt */
    0x02,
    /* Max Packet Size for this endpoint */
    USB_uint_16_low(EP_FS_MAX_PACKET_SIZE), 
    USB_uint_16_high(EP_FS_MAX_PACKET_SIZE),
    /* Polling Interval (ms) */
    0,
        /**************   Endpoint 5 (Bulk Out) Descriptor *******************/
    /* Endpoint 4 (Bulk Out Endpoint), Interface 0 Descriptor - always 7 bytes*/
    7,
    /* "Endpoint" type of descriptor */
    5,
    /*
    ** Endpoint address.  The low nibble contains the endpoint number and the
    ** high bit indicates TX(1) or RX(0).
    */
    0x03,
    /* Attributes.  0=Control 1=Isochronous 2=Bulk 3=Interrupt */
    0x02,
    /* Max Packet Size for this endpoint */
    USB_uint_16_low(EP_FS_MAX_PACKET_SIZE), 
    USB_uint_16_high(EP_FS_MAX_PACKET_SIZE),
    /* Polling Interval (ms) */
    0,
    /**************   Endpoint 6 (Bulk Out) Descriptor *******************/
    /* Endpoint 5 (Bulk Out Endpoint), Interface 0 Descriptor - always 7 bytes*/
    7,
    /* "Endpoint" type of descriptor */
    5,
    /*
    ** Endpoint address.  The low nibble contains the endpoint number and the
    ** high bit indicates TX(1) or RX(0).
    */
    0x04,
    /* Attributes.  0=Control 1=Isochronous 2=Bulk 3=Interrupt */
    0x02,
    /* Max Packet Size for this endpoint */
    USB_uint_16_low(EP_FS_MAX_PACKET_SIZE), 
    USB_uint_16_high(EP_FS_MAX_PACKET_SIZE),
    /* Polling Interval (ms) */
    0,
    /**************   Endpoint 7 (Bulk Out) Descriptor *******************/
    /* Endpoint 6 (Bulk Out Endpoint), Interface 0 Descriptor - always 7 bytes*/
    7,
    /* "Endpoint" type of descriptor */
    5,
    /*
    ** Endpoint address.  The low nibble contains the endpoint number and the
    ** high bit indicates TX(1) or RX(0).
    */
    0x05,
    /* Attributes.  0=Control 1=Isochronous 2=Bulk 3=Interrupt */
    0x02,
    /* Max Packet Size for this endpoint */
    USB_uint_16_low(EP_FS_MAX_PACKET_SIZE), 
    USB_uint_16_high(EP_FS_MAX_PACKET_SIZE),
    /* Polling Interval (ms) */
    0,
    /**************   Endpoint 8 (Bulk Out) Descriptor *******************/
    /* Endpoint 7 (Bulk Out Endpoint), Interface 0 Descriptor - always 7 bytes*/
    7,
    /* "Endpoint" type of descriptor */
    5,
    /*
    ** Endpoint address.  The low nibble contains the endpoint number and the
    ** high bit indicates TX(1) or RX(0).
    */
    0x06,
    /* Attributes.  0=Control 1=Isochronous 2=Bulk 3=Interrupt */
    0x02,
    /* Max Packet Size for this endpoint */
    USB_uint_16_low(EP_FS_MAX_PACKET_SIZE), 
    USB_uint_16_high(EP_FS_MAX_PACKET_SIZE),
    /* Polling Interval (ms) */
    0,
    /**************   Endpoint 9 (Bulk Out) Descriptor *******************/
    /* Endpoint 8 (Bulk Out Endpoint), Interface 0 Descriptor - always 7 bytes*/
    7,
    /* "Endpoint" type of descriptor */
    5,
    /*
    ** Endpoint address.  The low nibble contains the endpoint number and the
    ** high bit indicates TX(1) or RX(0).
    */
    0x07,
    /* Attributes.  0=Control 1=Isochronous 2=Bulk 3=Interrupt */
    0x02,
    /* Max Packet Size for this endpoint */
    USB_uint_16_low(EP_FS_MAX_PACKET_SIZE), 
    USB_uint_16_high(EP_FS_MAX_PACKET_SIZE),
    /* Polling Interval (ms) */
    0,
    /**************   Endpoint 10 (Bulk Out) Descriptor *******************/
    /* Endpoint 9 (Bulk Out Endpoint), Interface 0 Descriptor - always 7 bytes*/
    7,
    /* "Endpoint" type of descriptor */
    5,
    /*
    ** Endpoint address.  The low nibble contains the endpoint number and the
    ** high bit indicates TX(1) or RX(0).
    */
    0x08,
    /* Attributes.  0=Control 1=Isochronous 2=Bulk 3=Interrupt */
    0x02,
    /* Max Packet Size for this endpoint */
    USB_uint_16_low(EP_FS_MAX_PACKET_SIZE), 
    USB_uint_16_high(EP_FS_MAX_PACKET_SIZE),
    /* Polling Interval (ms) */
    0,
    /**************   Endpoint 11 (Bulk Out) Descriptor *******************/
    /* Endpoint 10 (Bulk Out Endpoint), Interface 0 Descriptor - always 7 bytes*/
    7,
    /* "Endpoint" type of descriptor */
    5,
    /*
    ** Endpoint address.  The low nibble contains the endpoint number and the
    ** high bit indicates TX(1) or RX(0).
    */
    0x09,
    /* Attributes.  0=Control 1=Isochronous 2=Bulk 3=Interrupt */
    0x02,
    /* Max Packet Size for this endpoint */
    USB_uint_16_low(EP_FS_MAX_PACKET_SIZE), 
    USB_uint_16_high(EP_FS_MAX_PACKET_SIZE),
    /* Polling Interval (ms) */
    0,
    /**************   Endpoint 12 (Bulk Out) Descriptor *******************/
    /* Endpoint 11 (Bulk Out Endpoint), Interface 0 Descriptor - always 7 bytes*/
    7,
    /* "Endpoint" type of descriptor */
    5,
    /*
    ** Endpoint address.  The low nibble contains the endpoint number and the
    ** high bit indicates TX(1) or RX(0).
    */
    0x0a,
    /* Attributes.  0=Control 1=Isochronous 2=Bulk 3=Interrupt */
    0x02,
    /* Max Packet Size for this endpoint */
    USB_uint_16_low(EP_FS_MAX_PACKET_SIZE), 
    USB_uint_16_high(EP_FS_MAX_PACKET_SIZE),
    /* Polling Interval (ms) */
    0,
};

uint_16        usb_status;
uint_8         endpoint, if_status;
uint_8         data_to_send;
uint_16        sof_count;
uint_8         setup_packet[sizeof(SETUP_STRUCT)];
SETUP_STRUCT   local_setup_packet;
static uint_8  serviceRegistered=0;


/*FUNCTION*----------------------------------------------------------------
* 
* Function Name  : cancel_ep_xfer
* Returned Value : None
* Comments       :
*   Cancels pending transfer in  a specified endpoint.
* 
*END*--------------------------------------------------------------------*/
uint_8 cancel_ep_xfers
    (
    /* [IN] Handle of the USB device */
    _usb_device_handle   handle,

    /* [IN] Endpoint */
    uint_16              ep_num,
    
    /* [IN] Direction */
    uint_8               direction
    )
{ /* Body */
    uint_8  status;

    if (_usb_device_get_transfer_status(handle, ep_num, direction) != USB_OK) {
        
        status = _usb_device_cancel_pend_xfers(handle, ep_num, direction);
        if (status != USB_OK) {
            return status;
        } else {
            if (_usb_device_get_transfer_status(handle, ep_num, direction) != USB_OK) {
                return USBERR_TRANSFER_IN_PROGRESS;
            }
        }
    }
    return USB_OK;
} /* EndBody */

/*FUNCTION*----------------------------------------------------------------
* 
* Function Name  : reset_ep0
* Returned Value : None
* Comments       :
*   Initializes a specified endpoint.
* 
*END*--------------------------------------------------------------------*/
uint_8 init_ep0
    (
    /* [IN] Handle of the USB device */
    _usb_device_handle   handle,

    /* [IN] Direction */
    uint_8               direction
    )
{ /* Body */
    uint_8 status;

    status = _usb_device_init_endpoint(handle, 0, DevDesc[DEV_DESC_MAX_PKT_SIZE_OFFSET], 
                                       direction, USB_CONTROL_ENDPOINT, 0);
    ASSERT (status == USB_OK );
    
    return status;
} /* EndBody */

/*FUNCTION*----------------------------------------------------------------
* 
* Function Name  : deinit_ep0
* Returned Value : None
* Comments       :
*   DeInitializes a specified endpoint.
* 
*END*--------------------------------------------------------------------*/
uint_8 deinit_ep0
    (
    /* [IN] Handle of the USB device */
    _usb_device_handle   handle,

    /* [IN] Direction */
    uint_8               direction
    )
{ /* Body */
    uint_8 status;

    status = _usb_device_deinit_endpoint(handle, 0, direction);
    
    ASSERT (status == USB_OK );
    
    return status;
}
/*FUNCTION*----------------------------------------------------------------
* 
* Function Name  : reset_ep0
* Returned Value : None
* Comments       :
*   Resets the control endpoint.
* 
*END*--------------------------------------------------------------------*/
void reset_ep0
    (
    /* [IN] Handle of the USB device */
    _usb_device_handle   handle,

    /* [IN] direction */
    uint_8               direction
    )
{ /* Body */
    uint_8  status;

    /* De Initialize the Endpoint */
    status = deinit_ep0(handle, direction);
    ASSERT (status == USB_OK);

    /* Cancel all the pending Transfer */
    status = cancel_ep_xfers(handle, 0, direction);
    ASSERT (status == USB_OK);

    /* Initialize the Endpoint back */
    status = init_ep0(handle, direction);
    ASSERT (status == USB_OK);
} /* EndBody */

/*FUNCTION*----------------------------------------------------------------
* 
* Function Name  : init_ep
* Returned Value : None
* Comments       :
*   Initializes a specified endpoint.
* 
*END*--------------------------------------------------------------------*/
uint_8 init_ep
    (
    /* [IN] Handle of the USB device */
    _usb_device_handle   handle,

    /* [IN] Endpoint */
    uint_16              ep_num,
    
    /* [IN] Direction */
    uint_8               direction
    )
{ /* Body */
    uint_8         status;
    uint_32        max_pkt_size;

    if (speed == USB_SPEED_HIGH) {
        max_pkt_size = EP_HS_MAX_PACKET_SIZE;
    } else {
        max_pkt_size = EP_FS_MAX_PACKET_SIZE;
    } /* Endif */
    
    status = _usb_device_init_endpoint(handle, ep_num, max_pkt_size,
                                       direction, USB_BULK_ENDPOINT, 
                                       (ZERO_TERMINATE ? 
                                        (!USB_DEVICE_DONT_ZERO_TERMINATE) : 
                                        USB_DEVICE_DONT_ZERO_TERMINATE));
    ASSERT (status == USB_OK );
    
    return status;
} /* EndBody */

/*FUNCTION*----------------------------------------------------------------
* 
* Function Name  : deinit_ep
* Returned Value : None
* Comments       :
*   DeInitializes a specified endpoint.
* 
*END*--------------------------------------------------------------------*/
uint_8 deinit_ep
    (
    /* [IN] Handle of the USB device */
    _usb_device_handle   handle,

    /* [IN] Endpoint */
    uint_16              ep_num,
    
    /* [IN] Direction */
    uint_8               direction
    )
{ /* Body */
    uint_8 status;

    status = _usb_device_deinit_endpoint(handle, ep_num, direction);
    
    ASSERT (status == USB_OK );
    
    return status;
}


/*FUNCTION*----------------------------------------------------------------
* 
* Function Name  : reset_ep
* Returned Value : None
* Comments       :
*   Resets a specified endpoint.
* 
*END*--------------------------------------------------------------------*/
void reset_ep
    (
    /* [IN] Handle of the USB device */
    _usb_device_handle   handle,

    /* [IN] Endpoint */
    uint_16              ep_num,

    /* [IN] direction */
    uint_8               direction
    )
{ /* Body */
    uint_8  status;

    status = deinit_ep(handle, ep_num, direction);
    ASSERT (status == USB_OK);

    status = cancel_ep_xfers(handle, ep_num, direction);
    ASSERT (status == USB_OK);

    status = init_ep(handle, ep_num, direction);
    ASSERT (status == USB_OK);
} /* EndBody */


/*FUNCTION*----------------------------------------------------------------
* 
* Function Name  : ch9GetStatus
* Returned Value : None
* Comments       :
*     Chapter 9 GetStatus command
*     wValue=Zero
*     wIndex=Zero
*     wLength=1
*     DATA=bmERR_STAT
*     The GET_STATUS command is used to read the bmERR_STAT register.
*     
*     Return the status based on the bRrequestType bits:
*     device (0) = bit 0 = 1 = self powered
*                  bit 1 = 0 = DEVICE_REMOTE_WAKEUP which can be modified
*     with a SET_FEATURE/CLEAR_FEATURE command.
*     interface(1) = 0000.
*     endpoint(2) = bit 0 = stall.
*     static uint_8_ptr pData;
*
*     See section 9.4.5 (page 190) of the USB 1.1 Specification.
* 
*END*--------------------------------------------------------------------*/
void ch9GetStatus
    (
    /* USB handle */
    _usb_device_handle handle,

    /* Is it a Setup phase? */
    boolean setup,

    /* The setup packet pointer */
    SETUP_STRUCT_PTR setup_ptr
    )
{ /* Body */
    if (setup) {
        switch (setup_ptr->REQUESTTYPE) {
            
            case 0x80:
                /* Device request */
                _usb_device_get_status(handle, USB_STATUS_DEVICE, &usb_status);
                usb_status = A_swab16(usb_status);
                /* Send the requested data */
                _usb_device_send_data(handle, 0, (uchar_ptr)((pointer)&usb_status), 
                                      sizeof(usb_status));
                break;

            case 0x81:
                /* Interface request */
                if_status = USB_IF_ALT[setup_ptr->INDEX & 0x00FF];
                if_status = A_swab16(if_status);
                /* Send the requested data */
                _usb_device_send_data(handle, 0, &if_status, sizeof(if_status));
                break;

            case 0x82:
                /* Endpoint request */
                endpoint = setup_ptr->INDEX & USB_STATUS_ENDPOINT_NUMBER_MASK;
                _usb_device_get_status(handle,
                                       USB_STATUS_ENDPOINT | endpoint, &usb_status);
                usb_status = A_swab16(usb_status);
                /* Send the requested data */
                _usb_device_send_data(handle, 0, (uchar_ptr)((pointer)&usb_status), 
                                      sizeof(usb_status));      
                break;

            default:
                /* Unknown request */
                _usb_device_stall_endpoint(handle, 0, 0);
                return;

        } /* Endswitch */

        /* status phase */
        _usb_device_recv_data(handle, 0, 0, 0);
    } /* Endif */
    return;
} /* Endbody */

/*FUNCTION*----------------------------------------------------------------
* 
* Function Name  : ch9ClearFeature
* Returned Value : None
* Comments       :
*     Chapter 9 ClearFeature command
* 
*END*--------------------------------------------------------------------*/
void ch9ClearFeature
    (
    /* USB handle */
    _usb_device_handle handle,

    /* Is it a Setup phase? */
    boolean setup,

    /* The setup packet pointer */
    SETUP_STRUCT_PTR setup_ptr
    )
{ /* Body */
    static uint_8           endpoint;
    uint_16                 usb_status;

    _usb_device_get_status(handle, USB_STATUS_DEVICE_STATE, &usb_status);

    if ((usb_status != USB_STATE_CONFIG) && (usb_status != USB_STATE_ADDRESS)) {
        _usb_device_stall_endpoint(handle, 0, 0);
        return;
    } /* Endif */

    if (setup) {
        switch (setup_ptr->REQUESTTYPE) {
            
            case 0:
                /* DEVICE */
                if (setup_ptr->VALUE == 1) {
                    /* clear remote wakeup */
                    _usb_device_get_status(handle, USB_STATUS_DEVICE, &usb_status);
                    usb_status &= ~USB_REMOTE_WAKEUP;
                    _usb_device_set_status(handle, USB_STATUS_DEVICE, usb_status);
                } else {
                    _usb_device_stall_endpoint(handle, 0, 0);
                    return;
                } /* Endif */
                break;

            case 2:
                /* ENDPOINT */
                if (setup_ptr->VALUE != 0) {
                    _usb_device_stall_endpoint(handle, 0, 0);
                    return;
                } /* Endif */
                endpoint = setup_ptr->INDEX & USB_STATUS_ENDPOINT_NUMBER_MASK;
                _usb_device_get_status(handle, USB_STATUS_ENDPOINT | endpoint,
                                       &usb_status);
                /* unstall */
                _usb_device_set_status(handle, USB_STATUS_ENDPOINT | endpoint,
                                       0);
                break;
            default:
                _usb_device_stall_endpoint(handle, 0, 0);
                return;
        } /* Endswitch */

        /* status phase */
        _usb_device_send_data(handle, 0, 0, 0);
    } /* Endif */
    return;
} /* Endbody */

/*FUNCTION*----------------------------------------------------------------
* 
* Function Name  : ch9SetFeature
* Returned Value : None
* Comments       :
*     Chapter 9 SetFeature command
* 
*END*--------------------------------------------------------------------*/
void ch9SetFeature
    (
    /* USB handle */
    _usb_device_handle handle,

    /* Is it a Setup phase? */
    boolean setup,

    /* The setup packet pointer */
    SETUP_STRUCT_PTR setup_ptr
    )
{ /* Body */
    uint_16           usb_status;
    uint_8            endpoint;

    if (setup) {
        switch (setup_ptr->REQUESTTYPE) {
            
            case 0:
                /* DEVICE */
                switch (setup_ptr->VALUE) {
                    case 1:
                        /* set remote wakeup */
                        _usb_device_get_status(handle, USB_STATUS_DEVICE, &usb_status);
                        usb_status |= USB_REMOTE_WAKEUP;
                        _usb_device_set_status(handle, USB_STATUS_DEVICE, usb_status);
                        break;
                    case 2:
                        /* Test Mode */
                        if ((setup_ptr->INDEX & 0x00FF) || 
                            (speed != USB_SPEED_HIGH)) {
                            _usb_device_stall_endpoint(handle, 0, 0);
                            return;
                        } /* Endif */
                        _usb_device_get_status(handle, USB_STATUS_DEVICE_STATE, 
                                               &usb_status);
                        if ((usb_status == USB_STATE_CONFIG) || 
                            (usb_status == USB_STATE_ADDRESS) || 
                            (usb_status == USB_STATE_DEFAULT)) {
                            ENTER_TEST_MODE = TRUE;
                            test_mode_index = (setup_ptr->INDEX & 0xFF00);
                        } else {
                            _usb_device_stall_endpoint(handle, 0, 0);
                            return;
                        } /* Endif */
                        break;
                    default:
                        _usb_device_stall_endpoint(handle, 0, 0);
                        return;
                } /* Endswitch */
                break;

            case 2:
                /* ENDPOINT */
                if (setup_ptr->VALUE != 0) {
                    _usb_device_stall_endpoint(handle, 0, 0);
                    return;
                } /* Endif */
                endpoint = setup_ptr->INDEX & USB_STATUS_ENDPOINT_NUMBER_MASK;
                _usb_device_get_status(handle, USB_STATUS_ENDPOINT | endpoint,
                                       &usb_status);
                /* set stall */
                _usb_device_set_status(handle, USB_STATUS_ENDPOINT | endpoint,
                                       1);
                break;

            default:
                _usb_device_stall_endpoint(handle, 0, 0);
                return;
        } /* Endswitch */

        /* status phase */
        _usb_device_send_data(handle, 0, 0, 0);
    } else {
        if (ENTER_TEST_MODE) {
            /* Enter Test Mode */
            _usb_device_set_status(handle, USB_STATUS_TEST_MODE, 
                                   test_mode_index);
        } /* Endif */
    } /* Endif */
    return;
} /* Endbody */

/*FUNCTION*----------------------------------------------------------------
* 
* Function Name  : ch9SetAddress
* Returned Value : None
* Comments       :
*     Chapter 9 SetAddress command
*     We setup a TX packet of 0 length ready for the IN token
*     Once we get the TOK_DNE interrupt for the IN token, then
*     we change the ADDR register and go to the ADDRESS state.
* 
*END*--------------------------------------------------------------------*/
void ch9SetAddress
    (
    /* USB handle */
    _usb_device_handle handle,

    /* Is it a Setup phase? */
    boolean setup,

    /* The setup packet pointer */
    SETUP_STRUCT_PTR setup_ptr
    )
{ /* Body */
    static uint_8  new_address;
    uint_8         i = 0;

    if (setup) {
        new_address = setup_ptr->VALUE;
        /* ack */
        _usb_device_send_data(handle, 0, 0, 0);
    } else {
        _usb_device_set_status(handle, USB_STATUS_ADDRESS, new_address);
        _usb_device_set_status(handle, USB_STATUS_DEVICE_STATE,
                               USB_STATE_ADDRESS);

        /* Reset all the Send EndPoints */
        for (i = 1; i <= MAX_HW_SEND_ENDPOINTS; i++) {
            reset_ep(handle, i, USB_SEND);
        }

        /* Reset all the Receive EndPoints */
        for (i = 1; i <= MAX_HW_RECV_ENDPOINTS; i++) {
            reset_ep(handle, i, USB_RECV);
        }
    } /* Endif */
    return;
} /* Endbody */

/*FUNCTION*----------------------------------------------------------------
* 
* Function Name  : ch9GetDescription
* Returned Value : None
* Comments       :
*     Chapter 9 GetDescription command
*     The Device Request can ask for Device/Config/string/interface/endpoint
*     descriptors (via wValue). We then post an IN response to return the
*     requested descriptor.
*     And then wait for the OUT which terminates the control transfer.
* 
*END*--------------------------------------------------------------------*/
void ch9GetDescription
    (
    /* USB handle */
    _usb_device_handle handle,

    /* Is it a Setup phase? */
    boolean setup,

    /* The setup packet pointer */
    SETUP_STRUCT_PTR setup_ptr
    )
{ /* Body */
    uint_32  max_pkt_size;
    uint_8   i = 0;

    if (setup) {
        /* Load the appropriate string depending on the descriptor requested.*/
        switch (setup_ptr->VALUE & 0xFF00) {
            
            case 0x0100:
                _usb_device_send_data(handle, 0, (uchar_ptr)((pointer)&DevDesc),
                                      MIN(setup_ptr->LENGTH, sizeof(DevDesc)));
                break;

            case 0x0200:
                /* Set the Max Packet Size in the config and other speed config */
                if (speed == USB_SPEED_HIGH) {
                    max_pkt_size = EP_HS_MAX_PACKET_SIZE;
                } else {
                    max_pkt_size = EP_FS_MAX_PACKET_SIZE;
                } /* Endif */
                for (i = 0; i < MAX_HW_ENDPOINTS; i++) {
                    ConfigDesc[CONFIG_DESC_EP_DESC_OFFSET + EP_DESC_LEN * i +
                               EP_DESC_MAX_PKT_LO_OFFSET] = USB_uint_16_low(max_pkt_size);
                    ConfigDesc[CONFIG_DESC_EP_DESC_OFFSET + EP_DESC_LEN * i +
                               EP_DESC_MAX_PKT_HI_OFFSET] = USB_uint_16_high(max_pkt_size);
                }
                _usb_device_send_data(handle, 0, (uchar_ptr)((pointer)&ConfigDesc),
                                      MIN(setup_ptr->LENGTH, CONFIG_DESC_SIZE));
                
                break;

            case 0x0300:
                if ((setup_ptr->VALUE & 0x00FF) > USB_STR_NUM) {
                    _usb_device_send_data(handle, 0, USB_STRING_DESC[USB_STR_NUM+1],
                                          MIN(setup_ptr->LENGTH, USB_STRING_DESC[USB_STR_NUM+1][0]));
                } else {
                    _usb_device_send_data(handle, 0,
                                          USB_STRING_DESC[setup_ptr->VALUE & 0x00FF],
                                          MIN(setup_ptr->LENGTH,
                                              USB_STRING_DESC[setup_ptr->VALUE & 0x00FF][0]));
                } /* Endif */      
                break;

            case 0x600:
                _usb_device_send_data(handle, 0, (uchar_ptr)DevQualifierDesc, 
                                      MIN(setup_ptr->LENGTH, sizeof(DevQualifierDesc)));
                break;

            case 0x700:
                if (speed == USB_SPEED_HIGH) {
                    max_pkt_size = EP_FS_MAX_PACKET_SIZE;
                } else {
                    max_pkt_size = EP_HS_MAX_PACKET_SIZE;
                } /* Endif */

                for (i = 0; i < MAX_HW_ENDPOINTS; i++) {
                    other_speed_config[CONFIG_DESC_EP_DESC_OFFSET + EP_DESC_LEN * i +
                               EP_DESC_MAX_PKT_LO_OFFSET] = USB_uint_16_low(max_pkt_size);
                    other_speed_config[CONFIG_DESC_EP_DESC_OFFSET + EP_DESC_LEN * i +
                               EP_DESC_MAX_PKT_HI_OFFSET] = USB_uint_16_high(max_pkt_size);
                }
                _usb_device_send_data(handle, 0, (uchar_ptr)other_speed_config, 
                                      MIN(setup_ptr->LENGTH, CONFIG_DESC_SIZE));
                
                break;

            default:
                _usb_device_stall_endpoint(handle, 0, 0);
                return;
        } /* Endswitch */
        /* status phase */
        _usb_device_recv_data(handle, 0, 0, 0);
    } /* Endif */
    return;
} /* Endbody */

/*FUNCTION*----------------------------------------------------------------
* 
* Function Name  : ch9SetDescription
* Returned Value : None
* Comments       :
*     Chapter 9 SetDescription command
* 
*END*--------------------------------------------------------------------*/
void ch9SetDescription
    (
    /* USB handle */
    _usb_device_handle handle,

    /* Is it a Setup phase? */
    boolean setup,

    /* The setup packet pointer */
    SETUP_STRUCT_PTR setup_ptr
    )
{ /* Body */
    _usb_device_stall_endpoint(handle, 0, 0);
    return;
} /* Endbody */

/*FUNCTION*----------------------------------------------------------------
* 
* Function Name  : ch9GetConfig
* Returned Value : None
* Comments       :
*     Chapter 9 GetConfig command
* 
*END*--------------------------------------------------------------------*/
void ch9GetConfig
    (
    /* USB handle */
    _usb_device_handle handle,

    /* Is it a Setup phase? */
    boolean setup,

    /* The setup packet pointer */
    SETUP_STRUCT_PTR setup_ptr
    )
{ /* Body */
    uint_16 current_config;
    /* Return the currently selected configuration */
    if (setup) {
        _usb_device_get_status(handle, USB_STATUS_CURRENT_CONFIG,
                               &current_config);
        data_to_send = current_config;      
        _usb_device_send_data(handle, 0, &data_to_send, sizeof(data_to_send));
        /* status phase */
        _usb_device_recv_data(handle, 0, 0, 0);
    } /* Endif */
    return;
} /* Endbody */

/*FUNCTION*----------------------------------------------------------------
* 
* Function Name  : ch9SetConfig
* Returned Value : None
* Comments       :
*     Chapter 9 SetConfig command
* 
*END*--------------------------------------------------------------------*/
void ch9SetConfig
    (
    /* USB handle */
    _usb_device_handle handle,

    /* Is it a Setup phase? */
    boolean setup,

    /* The setup packet pointer */
    SETUP_STRUCT_PTR setup_ptr
    )
{ /* Body */
    uint_16 usb_state;

    if (setup) {
        if ((setup_ptr->VALUE & 0x00FF) > 1) {
            /* generate stall */
            _usb_device_stall_endpoint(handle, 0, 0);
            return;
        } /* Endif */
        /* 0 indicates return to unconfigured state */
        if ((setup_ptr->VALUE & 0x00FF) == 0) {
            _usb_device_get_status(handle, USB_STATUS_DEVICE_STATE, &usb_state);
            if ((usb_state == USB_STATE_CONFIG) || 
                (usb_state == USB_STATE_ADDRESS)) {
                /* clear the currently selected config value */
                _usb_device_set_status(handle, USB_STATUS_CURRENT_CONFIG, 0);
                _usb_device_set_status(handle, USB_STATUS_DEVICE_STATE,
                                       USB_STATE_ADDRESS);
                /* status phase */      
                _usb_device_send_data(handle, 0, 0, 0);
            } else {
                _usb_device_stall_endpoint(handle, 0, 0);
            } /* Endif */
            return;
        } /* Endif */

        /*
        ** If the configuration value (setup_ptr->VALUE & 0x00FF) differs
        ** from the current configuration value, then endpoints must be
        ** reconfigured to match the new device configuration
        */
        _usb_device_get_status(handle, USB_STATUS_CURRENT_CONFIG,
                               &usb_state);
        if (usb_state != (setup_ptr->VALUE & 0x00FF)) {
            /* Reconfigure endpoints here */
            switch (setup_ptr->VALUE & 0x00FF) {

                default:
                    break;
            } /* Endswitch */
            _usb_device_set_status(handle, USB_STATUS_CURRENT_CONFIG,
                                   setup_ptr->VALUE & 0x00FF);
            _usb_device_set_status(handle, USB_STATUS_DEVICE_STATE,
                                   USB_STATE_CONFIG);      
            /* status phase */      
            _usb_device_send_data(handle, 0, 0, 0);
            return;
        } /* Endif */
        _usb_device_set_status(handle, USB_STATUS_DEVICE_STATE,
                               USB_STATE_CONFIG);
        /* status phase */
        _usb_device_send_data(handle, 0, 0, 0);

    } /* Endif */
    return;
} /* Endbody */

/*FUNCTION*----------------------------------------------------------------
* 
* Function Name  : ch9GetInterface
* Returned Value : None
* Comments       :
*     Chapter 9 GetInterface command
* 
*END*--------------------------------------------------------------------*/
void ch9GetInterface
    (
    /* USB handle */
    _usb_device_handle handle,

    /* Is it a Setup phase? */
    boolean setup,

    /* The setup packet pointer */
    SETUP_STRUCT_PTR setup_ptr
    )
{ /* Body */
    uint_16 usb_state;

    _usb_device_get_status(handle, USB_STATUS_DEVICE_STATE, &usb_state);
    if (usb_state != USB_STATE_CONFIG) {
        _usb_device_stall_endpoint(handle, 0, 0);
        return;
    } /* Endif */

    if (setup) {
        _usb_device_send_data(handle, 0, &USB_IF_ALT[setup_ptr->INDEX & 0x00FF],
                              MIN(setup_ptr->LENGTH, sizeof(uint_8)));
        /* status phase */      
        _usb_device_recv_data(handle, 0, 0, 0);
    } /* Endif */
    return;
} /* Endbody */

/*FUNCTION*----------------------------------------------------------------
* 
* Function Name  : ch9SetInterface
* Returned Value : None
* Comments       :
*     Chapter 9 SetInterface command
* 
*END*--------------------------------------------------------------------*/
void ch9SetInterface
    (
    /* USB handle */
    _usb_device_handle handle,

    /* Is it a Setup phase? */
    boolean setup,

    /* The setup packet pointer */
    SETUP_STRUCT_PTR setup_ptr
    )
{ /* Body */
    if (setup) {
        if (setup_ptr->REQUESTTYPE != 0x01) {
            _usb_device_stall_endpoint(handle, 0, 0);
            return;
        } /* Endif */

        /*
        ** If the alternate value (setup_ptr->VALUE & 0x00FF) differs
        ** from the current alternate value for the specified interface,
        ** then endpoints must be reconfigured to match the new alternate
        */
        if (USB_IF_ALT[setup_ptr->INDEX & 0x00FF]
            != (setup_ptr->VALUE & 0x00FF)) {
            USB_IF_ALT[setup_ptr->INDEX & 0x00FF] = (setup_ptr->VALUE & 0x00FF);
            /* Reconfigure endpoints here. */

        } /* Endif */

        /* status phase */
        _usb_device_send_data(handle, 0, 0, 0);
    } /* Endif */
    return;
} /* Endbody */

/*FUNCTION*----------------------------------------------------------------
* 
* Function Name  : ch9SynchFrame
* Returned Value : 
* Comments       :
*     Chapter 9 SynchFrame command
* 
*END*--------------------------------------------------------------------*/
void ch9SynchFrame
    (
    /* USB handle */
    _usb_device_handle handle,

    /* Is it a Setup phase? */
    boolean setup,

    /* The setup packet pointer */
    SETUP_STRUCT_PTR setup_ptr
    )
{ /* Body */

    if (setup) {
        if (setup_ptr->REQUESTTYPE != 0x02) {
            _usb_device_stall_endpoint(handle, 0, 0);
            return;
        } /* Endif */

        if ((setup_ptr->INDEX & 0x00FF) >=
            ConfigDesc[CONFIG_DESC_NUM_INTERFACES]) {
            _usb_device_stall_endpoint(handle, 0, 0);
            return;
        } /* Endif */

        _usb_device_get_status(handle, USB_STATUS_SOF_COUNT, &sof_count);
        sof_count = A_swab16(sof_count);
        _usb_device_send_data(handle, 0, (uchar_ptr)((pointer)&sof_count),
                              MIN(setup_ptr->LENGTH, sizeof(sof_count)));
        /* status phase */      
        _usb_device_recv_data(handle, 0, 0, 0);
    } /* Endif */
    return;
} /* Endbody */


#define VENDOR_RESET_FEATURE    0x00
#define VENDOR_RESET_DEVICE     0x01
#define VENDOR_RESET_ENDPOINT   0x02
/*FUNCTION*----------------------------------------------------------------
* 
* Function Name  : vendorReset
* Returned Value : 
* Comments       :
* 
*END*--------------------------------------------------------------------*/

void vendorReset
    (
    /* USB handle */
    _usb_device_handle  handle,

    /* Is it a Setup phase? */
    boolean             setup,

    /* The setup packet pointer */
    SETUP_STRUCT_PTR    setup_ptr
    )
{ /* Body */
    uint_16 ep_num = 0;
    uint_8  i = 0;
    static uint_32 vendorResetCount;

    isrPrintf("USB:%s [%d]\n", __FUNCTION__, vendorResetCount);
    vendorResetCount++;

    if (setup) {
        switch (setup_ptr->VALUE) {

            case VENDOR_RESET_DEVICE: 
                /* Reset all the Send EndPoints */
                for (i = 1; i <= MAX_HW_SEND_ENDPOINTS; i++) {
                    reset_ep(handle, i, USB_SEND);
                }
                /* Reset all the Receive EndPoints */
                for (i = 1; i <= MAX_HW_RECV_ENDPOINTS; i++) {
                    reset_ep(handle, i, USB_RECV);
                }
                break;

            case VENDOR_RESET_ENDPOINT:
                ep_num = (setup_ptr->INDEX & USB_STATUS_ENDPOINT_NUMBER_MASK);
                if ((ep_num > 0) && (ep_num <= MAX_HW_SEND_ENDPOINTS) && (setup_ptr->INDEX & 0x80)) {
                    /* Reset the Send Endpoint */
                    reset_ep(handle, ep_num, USB_SEND);
                } else if ((ep_num > 0) && (ep_num <= MAX_HW_RECV_ENDPOINTS) && !(setup_ptr->INDEX & 0x80)) {
                    /* Reset the Receive Endpoint */
                    reset_ep(handle, ep_num, USB_RECV);
                }
                break;

            default:
                break;
        }
    }
}

/*FUNCTION*----------------------------------------------------------------
* 
* Function Name  : ch9Vendor
* Returned Value : 
* Comments       :
* 
*END*--------------------------------------------------------------------*/
void ch9Vendor
    (
    /* USB handle */
    _usb_device_handle handle,

    /* Is it a Setup phase? */
    boolean setup,

    /* The setup packet pointer */
    SETUP_STRUCT_PTR setup_ptr
    )
{ /* Body */
    uint_16 ep_num = 0;

    if (setup) {
        switch (setup_ptr->REQUEST) {
            /* Add the Vendor specific Commands here */
            case VENDOR_RESET_FEATURE:
                vendorReset( handle, setup, setup_ptr);
                break;

            default :
                _usb_device_stall_endpoint(handle, ep_num, 0);
                return;
        } /* EndSwitch */

        /* status phase */
        _usb_device_send_data(handle, 0, 0, 0);
    } /* Endif */

} /* Endbody */


/*FUNCTION*----------------------------------------------------------------
* 
* Function Name  : service_speed
* Returned Value : None
* Comments       :
*     Called upon a speed detection event.
* 
*END*--------------------------------------------------------------------*/
void service_speed
    (
    /* [IN] Handle of the USB driver */
    _usb_drv_handle      handle,

    /* [IN] Unused */
    boolean              setup,

    /* [IN] Unused */
    uint_8               direction,

    /* [IN] Unused */
    uint_8_ptr           buffer,

    /* [IN] Unused */
    uint_32              length      
    )
{ /* EndBody */
    if (!serviceRegistered) return;

    USB_DEBUG_PRINTF(USB_SPEED_DEBUG, ("Speed : %d\n", length));

    speed = length;
    return;
} /* EndBody */


/*FUNCTION*----------------------------------------------------------------
* 
* Function Name  : service_reset
* Returned Value : None
* Comments       :
*     Called upon a bus reset event.  Initialises the control endpoint.
* 
*END*--------------------------------------------------------------------*/
void service_reset
    (
    /* [IN] Handle of the USB driver */
    _usb_drv_handle      handle,

    /* [IN] Unused */
    boolean              setup,

    /* [IN] Unused */
    uint_8               direction,

    /* [IN] Unused */
    uint_8_ptr           buffer,

    /* [IN] Unused */
    uint_32              length      
    )
{ /* Body */
    PUSBDRV_HANDLE       drv_handle;

    if (!serviceRegistered) return;
    
    drv_handle = (PUSBDRV_HANDLE)handle; 
    USB_DEBUG_PRINTF(USB_RESET_DEBUG, ("USB:Reset\n"));

    /* Initialize the endpoint 0 in both directions */
    reset_ep0(drv_handle->deviceHandle, USB_RECV);
    reset_ep0(drv_handle->deviceHandle, USB_SEND);
    
    athUsbDrvCtrlEvent(drv_handle, DRV_RESET_EVENT);

    return;
} /* EndBody */

/*FUNCTION*----------------------------------------------------------------
* 
* Function Name  : service_suspend
* Returned Value : None
* Comments       :
*     Called upon a Suspend event.
* 
*END*--------------------------------------------------------------------*/
void service_suspend
    (
    /* [IN] Handle of the USB driver */
    _usb_drv_handle      handle,

    /* [IN] Unused */
    boolean              setup,

    /* [IN] Unused */
    uint_8               direction,

    /* [IN] Unused */
    uint_8_ptr           buffer,

    /* [IN] Unused */
    uint_32              length      
    )
{ /* EndBody */
    PUSBDRV_HANDLE       drv_handle;
    
    if (!serviceRegistered) return;

    drv_handle =  (PUSBDRV_HANDLE)handle; 
    USB_DEBUG_PRINTF(USB_SUSPEND_DEBUG, ("USB:Suspend\n"));

    athUsbDrvCtrlEvent(drv_handle, DRV_SUSPEND_EVENT);

} /* EndBody */


void service_port_resume
    (
    /* [IN] Handle of the USB driver */
    _usb_drv_handle      handle,

    /* [IN] Unused */
    boolean              setup,

    /* [IN] Unused */
    uint_8               direction,

    /* [IN] Unused */
    uint_8_ptr           buffer,

    /* [IN] Unused */
    uint_32              length      
    )
{ /* EndBody */
    if (!serviceRegistered) return;
    
    USB_DEBUG_PRINTF(USB_SUSPEND_DEBUG, ("USB:Port Resume\n"));

    return;
} /* EndBody */

void service_port_suspend
    (
    /* [IN] Handle of the USB driver */
    _usb_drv_handle      handle,

    /* [IN] Unused */
    boolean              setup,

    /* [IN] Unused */
    uint_8               direction,

    /* [IN] Unused */
    uint_8_ptr           buffer,

    /* [IN] Unused */
    uint_32              length      
    )
{ /* EndBody */
    if (!serviceRegistered) return;

    USB_DEBUG_PRINTF(USB_SUSPEND_DEBUG, ("USB:Port Suspend\n"));

    return;
} /* EndBody */

/*FUNCTION*----------------------------------------------------------------
* 
* Function Name  : service_ep0
* Returned Value : None
* Comments       :
*     Called upon a completed endpoint 0 (USB 1.1 Chapter 9) transfer
* 
*END*--------------------------------------------------------------------*/
          
void service_ep0
    (
    /* [IN] Handle of the USB driver */
    _usb_drv_handle      handle,

    /* [IN] Is it a setup packet? */
    boolean              setup,

    /* [IN] Direction of the transfer.  Is it transmit? */
    uint_8               direction,

    /* [IN] Pointer to the data buffer */
    uint_8_ptr           buffer,

    /* [IN] Length of the transfer */
    uint_32              length      
    )
{ /* Body */
    boolean  class_request = FALSE;
    PUSBDRV_HANDLE       drv_handle;
    uint_16              pre_cfg_state;
    uint_16              post_cfg_state;

    drv_handle = (PUSBDRV_HANDLE)handle; 
    if (setup) {
        _usb_device_read_setup_data(drv_handle->deviceHandle, 0, 
                                    (uchar_ptr)&local_setup_packet);
        USB_DEBUG_PRINTF(USB_CONTROL_DEBUG, ("%02x:%02x:%04x:%04x:%04x\n",
                                             local_setup_packet.REQUESTTYPE,
                                             local_setup_packet.REQUEST,
                                             local_setup_packet.VALUE,
                                             local_setup_packet.INDEX,
                                             local_setup_packet.LENGTH));
    } else if (class_request) {
        class_request = FALSE;
        /* Finish your class or vendor request here */

        return;
    } /* Endif */

    switch (local_setup_packet.REQUESTTYPE & 0x60) {
        
        case 0x00:
            /* Standard request */
            switch (local_setup_packet.REQUEST) {
                
                case 0x0:
                    ch9GetStatus(drv_handle->deviceHandle, setup, 
                                    &local_setup_packet);
                    break;

                case 0x1:
                    ch9ClearFeature(drv_handle->deviceHandle, setup, 
                                    &local_setup_packet);
                    break;

                case 0x3:
                    ch9SetFeature(drv_handle->deviceHandle, setup, 
                                    &local_setup_packet);
                    break;

                case 0x5:
                    ch9SetAddress(drv_handle->deviceHandle, setup, 
                                    &local_setup_packet);
                    break;

                case 0x6:
#ifdef BOOT_LOADER_PLUS
                    if ( ZG_flashDisk ==TRUE )
                    {
                        getDescription(drv_handle->deviceHandle, setup, &local_setup_packet);
                    }
                    else
#endif //BOOT_LOADER_PLUS
                    {
                        ch9GetDescription(drv_handle->deviceHandle, setup,&local_setup_packet);
                    }
                    break;

                case 0x7:
                    ch9SetDescription(drv_handle->deviceHandle, setup, 
                                    &local_setup_packet);
                    break;

                case 0x8:
                    ch9GetConfig(drv_handle->deviceHandle, setup, 
                                    &local_setup_packet);
                    break;

                case 0x9:

                    _usb_device_get_status(drv_handle->deviceHandle, 
                                           USB_STATUS_DEVICE_STATE, &pre_cfg_state);

                    ch9SetConfig(drv_handle->deviceHandle, setup, 
                                    &local_setup_packet);
                    _usb_device_get_status(drv_handle->deviceHandle, 
                                           USB_STATUS_DEVICE_STATE, &post_cfg_state);
                    if ((pre_cfg_state != USB_STATE_CONFIG) && 
                        (post_cfg_state == USB_STATE_CONFIG)) 
                    {
                        /* Do the notification of the event to the app */
                        athUsbDrvCtrlEvent(drv_handle, DRV_INIT_EVENT);
                    }

                    break;

                case 0xa:
                    ch9GetInterface(drv_handle->deviceHandle, setup, 
                                    &local_setup_packet);
                    break;

                case 0xb:
                    ch9SetInterface(drv_handle->deviceHandle, setup, 
                                    &local_setup_packet);
                    break;

                case 0xc:
                    ch9SynchFrame(drv_handle->deviceHandle, setup, 
                                    &local_setup_packet);
                    break;

                default:
                    _usb_device_stall_endpoint(drv_handle->deviceHandle, 0, 0);
                    break;

            } /* Endswitch */

            break;

        case 0x20:
            /* class specific request */
            class_request = TRUE;
#ifdef BOOT_LOADER_PLUS
            if ( ZG_flashDisk ==TRUE )
            {
                MassStorageClass(drv_handle->deviceHandle,setup,&local_setup_packet);
            }
#endif
            return;

        case 0x40:
            /* vendor specific request */
            ch9Vendor(drv_handle->deviceHandle, setup, &local_setup_packet);
            break;

        default:
            _usb_device_stall_endpoint(drv_handle->deviceHandle, 0, 0);
            break;

    } /* Endswitch */

    return;
} /* Endbody */

/*FUNCTION*----------------------------------------------------------------
* 
* Function Name  : service_ep
* Returned Value : None
* Comments       :
*     Called upon a completed endpoint transfer
* 
*END*--------------------------------------------------------------------*/
void service_ep
    (
    /* [IN] Handle of the USB driver */
    _usb_drv_handle      handle,
    
    /* end point numbe */
    uint_8               epNum,

    /* [IN] Is it a setup packet? */
    boolean              setup,

    /* [IN] Direction of the transfer.  Is it transmit? */
    uint_8               direction,

    /* [IN] Pointer to the data buffer */
    uint_8_ptr           buffer,

    /* [IN] Length of the transfer */
    uint_32              length      
     
    )
{ 
#define MAX_EPNUM MAX(MAX_HW_RECV_ENDPOINTS, MAX_HW_SEND_ENDPOINTS)
    if (!serviceRegistered) return;
          
    if (epNum) {
        if (epNum <= MAX_EPNUM) {
            athUsbDrvSendRecv( (PUSBDRV_HANDLE) handle, epNum, direction, buffer, length);         
        }
    } else {
        service_ep0(handle,setup,direction,buffer,length);
    }
}

A_STATUS 
arcInit(
    /* [IN] Handle of the USB driver */
    PUSBDRV_HANDLE  drv_handle,

    /* [IN] usb device number */
    A_UINT8         usb_dev_num
    )
{
    A_UINT8                 error;
    A_INT8                  i=0;

    USB_DEBUG_PRINTF(USB_INIT_DEBUG, ("USB:Init\n"));
    /* Disable the USB Interrupt */
    _disable_interrupts (usb_dev_num);

    /* Initialize the USB enumeration values with the driver parameters */
    /* Set the Vendor ID */

#ifdef BOOT_LOADER_PLUS
    if ( ZG_flashDisk == FALSE ) {
#endif //BOOT_LOADER_PLUS
    if (sysBoardData.usbParam[usb_dev_num].usbVendorID && 
        (sysBoardData.usbParam[usb_dev_num].usbVendorID < 0xFFFF)) {
        DevDesc[DEV_DESC_VENDOR_LO_OFFSET] = USB_uint_16_low(
                                                sysBoardData.usbParam[usb_dev_num].usbVendorID);
        DevDesc[DEV_DESC_VENDOR_HI_OFFSET] = USB_uint_16_high(
                                                sysBoardData.usbParam[usb_dev_num].usbVendorID);
    }

    /* Set the Product ID */
    if (sysBoardData.usbParam[usb_dev_num].usbProductID && 
        (sysBoardData.usbParam[usb_dev_num].usbProductID < 0xFFFF)) {

#ifdef BOOTROM
        sysBoardData.usbParam[usb_dev_num].usbProductID  += 1;
#endif
#ifdef WLAN_TARGET
        sysBoardData.usbParam[usb_dev_num].usbProductID  += 2;
#endif
        DevDesc[DEV_DESC_PRODUCT_LO_OFFSET] = USB_uint_16_low(
                                                sysBoardData.usbParam[usb_dev_num].usbProductID);
        DevDesc[DEV_DESC_PRODUCT_HI_OFFSET] = USB_uint_16_high(
                                                sysBoardData.usbParam[usb_dev_num].usbProductID);
    }  else {
#ifdef BOOTROM
        DevDesc[DEV_DESC_PRODUCT_LO_OFFSET] = USB_uint_16_low(DEFAULT_USB_BROM_PRODUCTID);
#endif
#ifdef WLAN_TARGET
        DevDesc[DEV_DESC_PRODUCT_LO_OFFSET] = USB_uint_16_low(3);
#endif
    }
    diag_printf("Overide : %d en : %d\n", sysConfigData.usbConfig[usb_dev_num].rwakeOverride,
             sysConfigData.usbConfig[usb_dev_num].rwakeEn);
    if (sysConfigData.usbConfig[usb_dev_num].rwakeOverride) {
        ConfigDesc[CONFIG_DESC_ATTRIB_OFFSET] = (ConfigDesc[CONFIG_DESC_ATTRIB_OFFSET] & ~(0x20))
                                                | (sysConfigData.usbConfig[usb_dev_num].rwakeEn ? 0x20 : 0);
        diag_printf("Config : %x\n", ConfigDesc[CONFIG_DESC_ATTRIB_OFFSET]);
    }
    
    if (sysBoardData.usbParam[usb_dev_num].usbDeviceRelNum && 
        (sysBoardData.usbParam[usb_dev_num].usbDeviceRelNum < 0xFFFF)) {
        DevDesc[DEV_DESC_DEV_RELNUM_LO_OFFSET] = USB_uint_16_low(
                                                sysBoardData.usbParam[usb_dev_num].usbDeviceRelNum);
        DevDesc[DEV_DESC_DEV_RELNUM_HI_OFFSET] = USB_uint_16_high(
                                                sysBoardData.usbParam[usb_dev_num].usbDeviceRelNum);
    }
    
    /* Set the Driver Strings in the Strings Array */
    if (sysBoardData.usbParam[usb_dev_num].usbManfStrLen && 
        (sysBoardData.usbParam[usb_dev_num].usbManfStrLen <= USB_MANUFACTURER_STR_SIZE)) {
        USBSTRN(DEFAULT_MFG_STR_INDEX)[0] = sysBoardData.usbParam[usb_dev_num].usbManfStrLen + 2;
        
        memcpy(&USBSTRN(DEFAULT_MFG_STR_INDEX)[2], sysBoardData.usbParam[usb_dev_num].usbManufacturerStr, 
                                                USB_MANUFACTURER_STR_SIZE);
    }
    
    if (sysBoardData.usbParam[usb_dev_num].usbPdtStrLen && 
        (sysBoardData.usbParam[usb_dev_num].usbPdtStrLen <= USB_PRODUCT_STR_SIZE)) {
        USBSTRN(DEFAULT_PRODUCT_STR_INDEX)[0] = sysBoardData.usbParam[usb_dev_num].usbPdtStrLen + 2;
        memcpy(&USBSTRN(DEFAULT_PRODUCT_STR_INDEX)[2], sysBoardData.usbParam[usb_dev_num].usbProductStr, 
                                            USB_PRODUCT_STR_SIZE);
    }
    
    if (sysBoardData.usbParam[usb_dev_num].usbNumStrLen && 
        (sysBoardData.usbParam[usb_dev_num].usbNumStrLen <= USB_SERIALNUM_STR_SIZE)) {
        USBSTRN(DEFAULT_SERIALNUM_STR_INDEX)[0] = sysBoardData.usbParam[usb_dev_num].usbNumStrLen + 2;
        memcpy(&USBSTRN(DEFAULT_SERIALNUM_STR_INDEX)[2], sysBoardData.usbParam[usb_dev_num].usbSerialNumStr, 
                                            USB_SERIALNUM_STR_SIZE);
    }
#ifdef BOOT_LOADER_PLUS
    }
#endif //BOOT_LOADER_PLUS

    /* Initialize the USB interface */
    error = _usb_device_init(usb_dev_num, drv_handle, 
                             &drv_handle->deviceHandle, 
                             (MAX(MAX_HW_RECV_ENDPOINTS,MAX_HW_SEND_ENDPOINTS) + 1));
    if (error != USB_OK) {
        return A_ERROR;
    } /* Endif */

    serviceRegistered = 1;

    init_ep0(drv_handle->deviceHandle, USB_RECV);
    init_ep0(drv_handle->deviceHandle, USB_SEND);
    
    /* Initialize the Endpoints */
    for (i = 1; i <= MAX_HW_SEND_ENDPOINTS; i++) {
        init_ep(drv_handle->deviceHandle, i, USB_SEND);
    }
    /* Reset all the Receive EndPoints */
    for (i = 1; i <= MAX_HW_RECV_ENDPOINTS; i++) {
        init_ep(drv_handle->deviceHandle, i, USB_RECV);
    }

    _enable_interrupts(usb_dev_num);
    return A_OK;
}

A_STATUS 
arcShutdown(
    /* [IN] Handle of the USB driver */
    PUSBDRV_HANDLE  drv_handle
    )
{
    
    /* Disable the USB Interrupt */
    _disable_interrupts (drv_handle->deviceNum);
    
    serviceRegistered = 0;
    
    _usb_device_shutdown(drv_handle->deviceHandle);

    _enable_interrupts(drv_handle->deviceNum);
    return A_OK;
}

A_STATUS 
arcResume (
    /* [IN] Handle of the USB driver */
    PUSBDRV_HANDLE  drv_handle
    )
{
    _usb_device_assert_resume (drv_handle->deviceHandle);
    return A_OK;
}


