/*HEADER******************************************************************
**************************************************************************
*** 
*** Copyright (c) 2000-2002 ARC International.
*** All rights reserved                                          
***                                                              
*** This software embodies materials and concepts which are      
*** confidential to ARC International and is made
*** available solely pursuant to the terms of a written license   
*** agreement with ARC International             
***
*** $Workfile:dev_main.c$
*** $Revision: #1 $
*** $Date: 2005/08/24 $
***
*** Description:      
***  This file contains the main USB device API functions that will be 
***  used by most applications.
***                                                               
**************************************************************************
*END*********************************************************************/
#include "devapi.h"
#include "usb.h"
#include "usbprv.h"
#include "usbprv_dev.h"

#ifdef __USB_OS_MQX__
   #include "mqx_arc.h"
#endif

/*FUNCTION*-------------------------------------------------------------
*
*  Function Name  : _usb_device_free_XD
*  Returned Value : void
*  Comments       :
*        Enqueues a XD onto the free XD ring.
*
*END*-----------------------------------------------------------------*/

void _usb_device_free_XD
   (
      /* [IN] the dTD to enqueue */
      pointer  xd_ptr
   )
{ /* Body */
   USB_DEV_STATE_STRUCT_PTR   usb_dev_ptr = (USB_DEV_STATE_STRUCT_PTR)
      (((XD_STRUCT_PTR)xd_ptr)->SCRATCH_PTR->PRIVATE);

   /*
   ** This function can be called from any context, and it needs mutual
   ** exclusion with itself.
   */
   USB_lock(usb_dev_ptr->DEV_NUM);

   /*
   ** Add the XD to the free XD queue (linked via PRIVATE) and
   ** increment the tail to the next descriptor
   */
   USB_XD_QADD(usb_dev_ptr->XD_HEAD, usb_dev_ptr->XD_TAIL, 
      (XD_STRUCT_PTR)xd_ptr);
   usb_dev_ptr->XD_ENTRIES++;

   USB_unlock(usb_dev_ptr->DEV_NUM);

} /* Endbody */


/*FUNCTION*-------------------------------------------------------------
*
*  Function Name  : _usb_device_init
*  Returned Value : USB_OK or error code
*  Comments       :
*        Initializes the USB device specific data structures and calls 
*  the low-level device controller chip initialization routine.
*
*END*-----------------------------------------------------------------*/
uint_8 _usb_device_init
   (
      /* [IN] the USB device controller to initialize */
      uint_8                    devnum,
      
      /* [IN] the USB driver context */
      _usb_drv_handle           drv_handle,

      /* [OUT] the USB_USB_dev_initialize state structure */
      _usb_device_handle _PTR_  handle,
            
      /* [IN] number of endpoints to initialize */
      uint_8                    endpoints 
   )
{ /* Body */
   USB_DEV_STATE_STRUCT_PTR         usb_dev_ptr;
   XD_STRUCT_PTR                    xd_ptr;
   uint_8                           temp, i, error;
   SCRATCH_STRUCT_PTR               temp_scratch_ptr;
#ifdef __USB_OS_MQX__   
   USB_CALLBACK_FUNCTIONS_STRUCT_PTR call_back_table_ptr;
#endif
   
   if (devnum > MAX_USB_DEVICES) {
      return USBERR_INVALID_DEVICE_NUM;
   } /* Endif */
   
   /* Allocate memory for the state structure */
   usb_dev_ptr = (USB_DEV_STATE_STRUCT_PTR)
      USB_memalloc(sizeof(USB_DEV_STATE_STRUCT));
      
   if (usb_dev_ptr == NULL) {
      return USBERR_ALLOC_STATE;
   } /* Endif */
   
   /* Zero out the internal USB state structure */
   USB_memzero(usb_dev_ptr,sizeof(USB_DEV_STATE_STRUCT));
   
#ifdef __USB_OS_MQX__   
   call_back_table_ptr = (USB_CALLBACK_FUNCTIONS_STRUCT_PTR)
      _mqx_get_io_component_handle(IO_USB_COMPONENT);
      
   usb_dev_ptr->CALLBACK_STRUCT_PTR = (call_back_table_ptr + devnum);
      
   if (!usb_dev_ptr->CALLBACK_STRUCT_PTR) {
      return USBERR_DRIVER_NOT_INSTALLED;
   } /* Endif */
#endif
   
   /* Store the device Number */
   usb_dev_ptr->DEV_NUM =  devnum;
   /* Multiple devices will have different base addresses and 
   ** interrupt vectors (For future)
   */ 
   usb_dev_ptr->DEV_PTR = _bsp_get_usb_base(devnum);
   usb_dev_ptr->DEV_VEC = _bsp_get_usb_vector(devnum);
   usb_dev_ptr->USB_STATE = USB_STATE_UNKNOWN;
   
   usb_dev_ptr->MAX_ENDPOINTS = endpoints;
   
   temp = (usb_dev_ptr->MAX_ENDPOINTS * 2);

   /* Allocate MAX_XDS_FOR_TR_CALLS */
   xd_ptr = (XD_STRUCT_PTR)
      USB_memalloc(sizeof(XD_STRUCT) * MAX_XDS_FOR_TR_CALLS);
      
   if (xd_ptr == NULL) {
      return USBERR_ALLOC_TR;
   } /* Endif */
   
   usb_dev_ptr->XD_BASE = xd_ptr;
   
   USB_memzero(xd_ptr, sizeof(XD_STRUCT) * MAX_XDS_FOR_TR_CALLS);

   /* Allocate memory for internal scratch structure */   
   usb_dev_ptr->XD_SCRATCH_STRUCT_BASE = (SCRATCH_STRUCT_PTR)
      USB_memalloc(sizeof(SCRATCH_STRUCT)*MAX_XDS_FOR_TR_CALLS);
   
   if (usb_dev_ptr->XD_SCRATCH_STRUCT_BASE == NULL) {
      return USBERR_ALLOC;
   } /* Endif */

   temp_scratch_ptr = usb_dev_ptr->XD_SCRATCH_STRUCT_BASE;
   usb_dev_ptr->XD_HEAD = NULL;
   usb_dev_ptr->XD_TAIL = NULL;
   usb_dev_ptr->XD_ENTRIES = 0;

   /* Enqueue all the XDs */   
   for (i=0;i<MAX_XDS_FOR_TR_CALLS;i++) {
      xd_ptr->SCRATCH_PTR = temp_scratch_ptr;
      xd_ptr->SCRATCH_PTR->FREE = _usb_device_free_XD;
      xd_ptr->SCRATCH_PTR->PRIVATE = (pointer)usb_dev_ptr;
      _usb_device_free_XD((pointer)xd_ptr);
      xd_ptr++;
      temp_scratch_ptr++;
   } /* Endfor */

   usb_dev_ptr->TEMP_XD_PTR = (XD_STRUCT_PTR)USB_memalloc(sizeof(XD_STRUCT));
   USB_memzero(usb_dev_ptr->TEMP_XD_PTR, sizeof(XD_STRUCT));

   /* Initialize the USB controller chip */
   
#ifdef __USB_OS_MQX__   
   error = ((USB_CALLBACK_FUNCTIONS_STRUCT_PTR)\
      usb_dev_ptr->CALLBACK_STRUCT_PTR)->DEV_INIT(devnum, usb_dev_ptr);
#else
   error = _usb_dci_vusb20_init(devnum, usb_dev_ptr);
#endif

   if (error) {
      return USBERR_INIT_FAILED;
   } /* Endif */
   
   usb_dev_ptr->DRV_HANDLE = drv_handle;
   *handle = usb_dev_ptr;
   
   return USB_OK;
   
} /* EndBody */

/*FUNCTION*-------------------------------------------------------------
*
*  Function Name  : _usb_device_init_endpoint
*  Returned Value : USB_OK or error code
*  Comments       :
*     Initializes the endpoint and the data structures associated with the 
*  endpoint
*
*END*-----------------------------------------------------------------*/
uint_8 _usb_device_init_endpoint
   (
      /* [IN] the USB_USB_dev_initialize state structure */
      _usb_device_handle         handle,
            
      /* [IN] the Endpoint number */
      uint_8                     ep_num,
            
      /* [IN] MAX Packet size for this endpoint */
      uint_16                    max_pkt_size,
            
      /* [IN] Direction */
      uint_8                     direction,
            
      /* [IN] Type of Endpoint */
      uint_8                     type,
            
      /* [IN] After all data is transfered, should we terminate the transfer 
      ** with a zero length packet if the last packet size == MAX_PACKET_SIZE? 
      */
      uint_8                     flag   
   )
{ /* Body */
   uint_8         error = 0;
   USB_DEV_STATE_STRUCT_PTR      usb_dev_ptr;
   
   usb_dev_ptr = (USB_DEV_STATE_STRUCT_PTR)handle;

   /* Initialize the transfer descriptor */
   usb_dev_ptr->TEMP_XD_PTR->EP_NUM = ep_num;
   usb_dev_ptr->TEMP_XD_PTR->BDIRECTION = direction;
   usb_dev_ptr->TEMP_XD_PTR->WMAXPACKETSIZE = max_pkt_size;
   usb_dev_ptr->TEMP_XD_PTR->EP_TYPE = type;
   usb_dev_ptr->TEMP_XD_PTR->DONT_ZERO_TERMINATE = flag;

#ifdef __USB_OS_MQX__   
    ((USB_CALLBACK_FUNCTIONS_STRUCT_PTR)\
      usb_dev_ptr->CALLBACK_STRUCT_PTR)->DEV_INIT_ENDPOINT(handle, usb_dev_ptr->TEMP_XD_PTR);
#else
   error = _usb_dci_vusb20_init_endpoint(handle, usb_dev_ptr->TEMP_XD_PTR);
#endif
   
   return error;
   
} /* EndBody */

/*FUNCTION*-------------------------------------------------------------
*
*  Function Name  : _usb_device_get_transfer_status
*  Returned Value : Status of the transfer
*  Comments       :
*        returns the status of the transaction on the specified endpoint.
*
*END*-----------------------------------------------------------------*/
uint_8 _usb_device_get_transfer_status
   (
      /* [IN] the USB_USB_dev_initialize state structure */
      _usb_device_handle         handle,
            
      /* [IN] the Endpoint number */
      uint_8                     ep_num,
            
      /* [IN] direction */
      uint_8                     direction
   )
{ /* Body */
   uint_8   status;
   USB_DEV_STATE_STRUCT_PTR      usb_dev_ptr;
   
   usb_dev_ptr = (USB_DEV_STATE_STRUCT_PTR)handle;
   
   USB_lock(usb_dev_ptr->DEV_NUM);

#ifdef __USB_OS_MQX__   
   status = ((USB_CALLBACK_FUNCTIONS_STRUCT_PTR)\
      usb_dev_ptr->CALLBACK_STRUCT_PTR)->DEV_GET_TRANSFER_STATUS(handle, ep_num, direction);
#else
   status = _usb_dci_vusb20_get_transfer_status(handle, ep_num, direction);
#endif

   USB_unlock(usb_dev_ptr->DEV_NUM);

   /* Return the status of the last queued transfer */
   return (status);

} /* EndBody */

/*FUNCTION*-------------------------------------------------------------
*
*  Function Name  : _usb_device_read_setup_data
*  Returned Value : USB_OK or error code
*  Comments       :
*        Reads the setup data from the hardware
*
*END*-----------------------------------------------------------------*/
void _usb_device_read_setup_data
   (
      /* [IN] the USB_USB_dev_initialize state structure */
      _usb_device_handle         handle,
            
      /* [IN] the Endpoint number */
      uint_8                     ep_num,
            
      /* [IN] buffer for receiving Setup packet */
      uchar_ptr                  buff_ptr
   )
{ /* Body */
   USB_DEV_STATE_STRUCT_PTR      usb_dev_ptr;
   
   usb_dev_ptr = (USB_DEV_STATE_STRUCT_PTR)handle;

#ifdef __USB_OS_MQX__   
    ((USB_CALLBACK_FUNCTIONS_STRUCT_PTR)\
      usb_dev_ptr->CALLBACK_STRUCT_PTR)->DEV_GET_SETUP_DATA(handle, ep_num, buff_ptr);
#else
   _usb_dci_vusb20_get_setup_data(handle, ep_num, buff_ptr);
#endif

} /* EndBody */

/*FUNCTION*-------------------------------------------------------------
*
*  Function Name  : _usb_device_set_address
*  Returned Value : USB_OK or error code
*  Comments       :
*        Sets the device address as assigned by the host during enumeration
*
*END*-----------------------------------------------------------------*/
void _usb_device_set_address
   (
      /* [IN] the USB_USB_dev_initialize state structure */
      _usb_device_handle         handle,
      
      /* [IN] the USB address to be set in the hardware */
      uint_8                     address
   )
{ /* Body */
   USB_DEV_STATE_STRUCT_PTR      usb_dev_ptr;
   
   usb_dev_ptr = (USB_DEV_STATE_STRUCT_PTR)handle;

#ifdef __USB_OS_MQX__   
    ((USB_CALLBACK_FUNCTIONS_STRUCT_PTR)\
      usb_dev_ptr->CALLBACK_STRUCT_PTR)->DEV_SET_ADDRESS(handle, address);
#else
   _usb_dci_vusb20_set_address(handle, address);
#endif
   
} /* EndBody */

/*FUNCTION*-------------------------------------------------------------
*
*  Function Name  : _usb_device_cancel_data
*  Returned Value : USB_OK or error code
*  Comments       :
*        Cacnel pednign data on a specified endpoint.
*
*END*-----------------------------------------------------------------*/
uint_8 _usb_device_cancel_pend_xfers
   (
      /* [IN] the USB_USB_dev_initialize state structure */
      _usb_device_handle         handle,
            
      /* [IN] the Endpoint number */
      uint_8                     ep_num,

      /* [IN] direction */
      uint_8                     direction
   )
{
    uint_8                        error = USB_OK;
    USB_DEV_STATE_STRUCT_PTR      usb_dev_ptr;

    usb_dev_ptr = (USB_DEV_STATE_STRUCT_PTR)handle;
    
    USB_lock(usb_dev_ptr->DEV_NUM);
   
    /* Cancel pending transfer on the specified endpoint for the specified 
    ** direction 
    */
    error = _usb_dci_vusb20_cancel_pend_xfers(handle, ep_num, direction);

    USB_unlock(usb_dev_ptr->DEV_NUM);
   
    return error;
}


