/*HEADER******************************************************************
**************************************************************************
*** 
*** Copyright (c) 2000-2002 ARC International.
*** All rights reserved                                          
***                                                              
*** This software embodies materials and concepts which are      
*** confidential to ARC International and is made
*** available solely pursuant to the terms of a written license   
*** agreement with ARC International             
***
*** $Workfile:vusbhs_dev_main.c$
*** $Revision: #1 $
*** $Date: 2005/08/24 $
***
*** Description:      
***  This file contains the main VUSB_HS Device Controller interface 
***  functions.
***                                                               
**************************************************************************
*END*********************************************************************/
#include "devapi.h"
#include "usb.h"
#include "usbprv.h"
#include "usbprv_dev.h"

#ifdef __USB_OS_MQX__
   #include "mqx_arc.h"
#endif

/* in the  OTG mode this need to be a global */
USB_DEV_STATE_STRUCT_PTR global_usb_device_state_struct_ptr;
#ifndef __USB_OS_MQX__
extern volatile    boolean                 IN_ISR[];
#endif

/*FUNCTION*-------------------------------------------------------------
*
*  Function Name  : _usb_dci_vusb20_init
*  Returned Value : USB_OK or error code
*  Comments       :
*        Initializes the USB device controller.
*
*END*-----------------------------------------------------------------*/
uint_8 _usb_dci_vusb20_init
   (
      /* [IN] the USB device controller to initialize */
      uint_8                     devnum,

      /* [OUT] the USB_dev_initialize state structure */
      _usb_device_handle         handle
   )
{ /* Body */
   USB_DEV_STATE_STRUCT_PTR                     usb_dev_ptr;
   volatile VUSB20_REG_STRUCT _PTR_             dev_ptr;
   volatile VUSB20_REG_STRUCT _PTR_             cap_dev_ptr;
   uint_32                                      temp;
   uint_8                                       max_ep_supported;
   uint_8                                       max_ep_configured;
   
   usb_dev_ptr = (USB_DEV_STATE_STRUCT_PTR)handle;

   cap_dev_ptr = (volatile VUSB20_REG_STRUCT _PTR_)
      _bsp_get_usb_capability_register_base(devnum);
      
   /* Get the base address of the VUSB_HS registers */
   usb_dev_ptr->DEV_PTR = 
      (VUSB20_REG_STRUCT_PTR)((uint_32)cap_dev_ptr + 
      (cap_dev_ptr->REGISTERS.CAPABILITY_REGISTERS.CAPLENGTH_HCIVER & 
      EHCI_CAP_LEN_MASK));

   /* Get the maximum number of endpoints supported by this USB controller */
   max_ep_supported = 
      (cap_dev_ptr->REGISTERS.CAPABILITY_REGISTERS.DCC_PARAMS & 
       VUSB20_MAX_ENDPTS_SUPPORTED);
   max_ep_configured = usb_dev_ptr->MAX_ENDPOINTS;
   
   usb_dev_ptr->MAX_ENDPOINTS = MIN (max_ep_configured, max_ep_supported);
   usb_dev_ptr->MAX_ENDPOINTS = MIN (USB_MAX_ENDPOINTS, usb_dev_ptr->MAX_ENDPOINTS);

   dev_ptr = (volatile VUSB20_REG_STRUCT _PTR_)usb_dev_ptr->DEV_PTR;
   
   temp = (usb_dev_ptr->MAX_ENDPOINTS * 2);
   
   /* Allocate memory for the endpoint queue heads */
   /* 
   ** Allocate additional 2K bytes so that we can align it at 2K-byte 
   ** boundary and still have enough room 
   */
   /* The EP_QUEUE_HEAD_PTR in the usb_dev_ptr will be used by shutdown */ 
   usb_dev_ptr->EP_QUEUE_HEAD_BASE = (VUSB20_EP_QUEUE_HEAD_STRUCT_PTR)
      USB_memalloc((temp * sizeof(VUSB20_EP_QUEUE_HEAD_STRUCT)) + 2048);
   
   if (usb_dev_ptr->EP_QUEUE_HEAD_BASE == NULL) {
      return USBERR_ALLOC_EP_QUEUE_HEAD;
   } /* Endif */

   /* Align the endpoint queue head to 2K boundary */   
   usb_dev_ptr->EP_QUEUE_HEAD_PTR = (VUSB20_EP_QUEUE_HEAD_STRUCT_PTR)
      USB_MEM2048_ALIGN((uint_32)usb_dev_ptr->EP_QUEUE_HEAD_BASE);

   /* Allocate the Device Transfer Descriptors: 32 bytes aligned */   
   usb_dev_ptr->DTD_BASE_PTR = (VUSB20_EP_TR_STRUCT_PTR)
      USB_memalloc((MAX_EP_TR_DESCRS * sizeof(VUSB20_EP_TR_STRUCT)) + 32);
   
   if (usb_dev_ptr->DTD_BASE_PTR == NULL) {
      return USBERR_ALLOC_DTD_BASE;
   } /* Endif */

   /* Align the dTD base to 32 byte boundary */   
   usb_dev_ptr->DTD_ALIGNED_BASE_PTR = (VUSB20_EP_TR_STRUCT_PTR)
      USB_MEM32_ALIGN((uint_32)usb_dev_ptr->DTD_BASE_PTR);

   /* Allocate memory for internal scratch structure */   
   usb_dev_ptr->SCRATCH_STRUCT_BASE = (SCRATCH_STRUCT_PTR)
      USB_memalloc(sizeof(SCRATCH_STRUCT)*MAX_EP_TR_DESCRS);
   
   if (usb_dev_ptr->SCRATCH_STRUCT_BASE == NULL) {
      return USBERR_ALLOC;
   } /* Endif */   

#ifndef __USB_OTG__
   /* Install the interrupt service routine */
#ifndef __USB_OS_MQX__   
   USB_install_isr(usb_dev_ptr->DEV_NUM, _usb_dci_vusb20_isr, usb_dev_ptr, 
                   &usb_dev_ptr->INTR_HANDLE);
#else
   if (!(USB_install_isr(_bsp_get_usb_vector(usb_dev_ptr->DEV_NUM), 
      _usb_dci_vusb20_isr, (pointer)usb_dev_ptr, &usb_dev_ptr->INTR_HANDLE)))
   {
      return USBERR_INSTALL_ISR;
   } /* Endbody */
#endif
#endif   /* __USB_OTG__*/
   
   usb_dev_ptr->USB_STATE = USB_STATE_UNKNOWN;
#ifdef BUS_POWERED
   usb_dev_ptr->USB_DEVICE_STATE = 0;
#else
   usb_dev_ptr->USB_DEVICE_STATE = USB_SELF_POWERED;
#endif 

   /* Initialize the VUSB_HS controller */   
   _usb_dci_vusb20_chip_initialize((pointer)usb_dev_ptr);
   
   global_usb_device_state_struct_ptr = usb_dev_ptr;

   return USB_OK;   
} /* EndBody */

/*FUNCTION*-------------------------------------------------------------
*
*  Function Name  : _usb_dci_vusb20_chip_initialize
*  Returned Value : USB_OK or error code
*  Comments       :
*        Initializes the USB device controller.
*
*END*-----------------------------------------------------------------*/
void _usb_dci_vusb20_chip_initialize
   (
      /* [IN] the USB_dev_initialize state structure */
      _usb_device_handle         handle
   )
{ /* Body */
   USB_DEV_STATE_STRUCT_PTR                     usb_dev_ptr;
   volatile VUSB20_REG_STRUCT _PTR_             dev_ptr;
   volatile VUSB20_REG_STRUCT _PTR_             cap_dev_ptr;
   VUSB20_EP_QUEUE_HEAD_STRUCT_PTR              ep_queue_head_ptr;
   VUSB20_EP_TR_STRUCT_PTR                      dTD_ptr;
   uint_32                                      temp, i, port_control;
   SCRATCH_STRUCT_PTR                           temp_scratch_ptr;
   
   usb_dev_ptr = (USB_DEV_STATE_STRUCT_PTR)handle;

   dev_ptr = (volatile VUSB20_REG_STRUCT _PTR_)usb_dev_ptr->DEV_PTR;
   cap_dev_ptr = (volatile VUSB20_REG_STRUCT _PTR_)
      _bsp_get_usb_capability_register_base(usb_dev_ptr->DEV_NUM);
   
   /* Stop the controller */
   dev_ptr->REGISTERS.OPERATIONAL_DEVICE_REGISTERS.USB_CMD &= 
      ~EHCI_CMD_RUN_STOP;

   /* Reset the controller to get default values */
   dev_ptr->REGISTERS.OPERATIONAL_DEVICE_REGISTERS.USB_CMD = EHCI_CMD_CTRL_RESET;

   while (dev_ptr->REGISTERS.OPERATIONAL_DEVICE_REGISTERS.USB_CMD & 
      EHCI_CMD_CTRL_RESET) 
   {
      /* Wait for the controller reset to complete */
   } /* EndWhile */

   /* Program the controller to be the USB device controller */ 
   dev_ptr->REGISTERS.OPERATIONAL_DEVICE_REGISTERS.USB_MODE = 
      VUSBHS_MODE_CTRL_MODE_DEV 
#if BIG_ENDIAN 
       | VUSBHS_MODE_BIG_ENDIAN;
#else 
       ;
#endif
   
   temp = (usb_dev_ptr->MAX_ENDPOINTS * 2);

   /* Zero out the Endpoint queue heads */
   USB_memzero(usb_dev_ptr->EP_QUEUE_HEAD_BASE,((temp * 
      sizeof(VUSB20_EP_QUEUE_HEAD_STRUCT)) + 2048));
   
   /* Initialize the internal dTD head and tail to NULL */   
   usb_dev_ptr->DTD_HEAD = NULL;
   usb_dev_ptr->DTD_TAIL = NULL;
   
   /* Zero out the device transfer descriptors */
   USB_memzero(usb_dev_ptr->DTD_BASE_PTR,((MAX_EP_TR_DESCRS * 
      sizeof(VUSB20_EP_TR_STRUCT)) + 32));
      
   /* Zero out the internal device transfer descriptor heads and tails */
   USB_memzero(usb_dev_ptr->EP_DTD_HEADS, (sizeof(VUSB20_EP_TR_STRUCT_PTR) * 
      usb_dev_ptr->MAX_ENDPOINTS * 2));

   /* Zero out the endpoint transaction descriptor heads and tails */
   USB_memzero(usb_dev_ptr->EP_DTD_TAILS, (sizeof(VUSB20_EP_TR_STRUCT_PTR) * 
      usb_dev_ptr->MAX_ENDPOINTS * 2));

   /* Make sure the 16 MSBs of this register are 0s */
   dev_ptr->REGISTERS.OPERATIONAL_DEVICE_REGISTERS.ENDPT_SETUP_STAT = 0;
   
   ep_queue_head_ptr = usb_dev_ptr->EP_QUEUE_HEAD_PTR;

   /* Initialize all device queue heads */
   for (i=0;i<temp;i++) {
      /* Interrupt on Setup packet */
      (ep_queue_head_ptr + i)->MAX_PKT_LENGTH = 
         (((uint_32)USB_MAX_CTRL_PAYLOAD << 
         VUSB_EP_QUEUE_HEAD_MAX_PKT_LEN_POS) | VUSB_EP_QUEUE_HEAD_IOS);
      (ep_queue_head_ptr + i)->STATUS = VUSB_EP_DEINIT;
      (ep_queue_head_ptr + i)->NEXT_DTD_PTR = 
         VUSB_EP_QUEUE_HEAD_NEXT_TERMINATE;
   } /* Endfor */

   /* Configure the Endpoint List Address */
   dev_ptr->REGISTERS.OPERATIONAL_DEVICE_REGISTERS.EP_LIST_ADDR = 
      (uint_32)ep_queue_head_ptr;
      
   if (cap_dev_ptr->REGISTERS.CAPABILITY_REGISTERS.HCS_PARAMS & 
 	   VUSB20_HCS_PARAMS_PORT_POWER_CONTROL_FLAG) 
   {
      port_control = dev_ptr->REGISTERS.OPERATIONAL_DEVICE_REGISTERS.PORTSCX[0];
      port_control &= (~EHCI_PORTSCX_W1C_BITS | ~EHCI_PORTSCX_PORT_POWER);
      dev_ptr->REGISTERS.OPERATIONAL_DEVICE_REGISTERS.PORTSCX[0] = port_control;
   } /* Endif */
   
   dTD_ptr = usb_dev_ptr->DTD_ALIGNED_BASE_PTR;

   temp_scratch_ptr = usb_dev_ptr->SCRATCH_STRUCT_BASE;

   /* Enqueue all the dTDs */   
   for (i=0;i<MAX_EP_TR_DESCRS;i++) {
      dTD_ptr->SCRATCH_PTR = temp_scratch_ptr;
      dTD_ptr->SCRATCH_PTR->FREE = _usb_dci_vusb20_free_dTD;
      /* Set the dTD to be invalid */
      dTD_ptr->NEXT_TR_ELEM_PTR = VUSBHS_TD_NEXT_TERMINATE;
      /* Set the Reserved fields to 0 */
      dTD_ptr->SIZE_IOC_STS &= ~VUSBHS_TD_RESERVED_FIELDS;
      dTD_ptr->SCRATCH_PTR->PRIVATE = (pointer)usb_dev_ptr;
      _usb_dci_vusb20_free_dTD((pointer)dTD_ptr);
      dTD_ptr++;
      temp_scratch_ptr++;
   } /* Endfor */

   /* Initialize the endpoint 0 properties */
   dev_ptr->REGISTERS.OPERATIONAL_DEVICE_REGISTERS.ENDPTCTRLX[0] = 
      (EHCI_EPCTRL_TX_DATA_TOGGLE_RST | EHCI_EPCTRL_RX_DATA_TOGGLE_RST);
   dev_ptr->REGISTERS.OPERATIONAL_DEVICE_REGISTERS.ENDPTCTRLX[0] &= 
      ~(EHCI_EPCTRL_TX_EP_STALL | EHCI_EPCTRL_RX_EP_STALL);
   
   /* Enable interrupts */
   dev_ptr->REGISTERS.OPERATIONAL_DEVICE_REGISTERS.USB_INTR = 
      (EHCI_INTR_INT_EN | EHCI_INTR_ERR_INT_EN | 
      EHCI_INTR_PORT_CHANGE_DETECT_EN | EHCI_INTR_RESET_EN |
      EHCI_INTR_DEVICE_SUSPEND 
#if VUSBHS_SOF_INTR_EN
      | EHCI_INTR_SOF_UFRAME_EN
#endif
      );
   
   usb_dev_ptr->USB_STATE = USB_STATE_UNKNOWN;
#ifdef BUS_POWERED
   usb_dev_ptr->USB_DEVICE_STATE = 0;
#else
   usb_dev_ptr->USB_DEVICE_STATE = USB_SELF_POWERED;
#endif 
   
   /* Set the Run bit in the command register */
   dev_ptr->REGISTERS.OPERATIONAL_DEVICE_REGISTERS.USB_CMD = EHCI_CMD_RUN_STOP;
   
} /* EndBody */

/*FUNCTION*-------------------------------------------------------------
*
*  Function Name  : _usb_dci_vusb20_free_dTD
*  Returned Value : void
*  Comments       :
*        Enqueues a dTD onto the free DTD ring.
*
*END*-----------------------------------------------------------------*/

void _usb_dci_vusb20_free_dTD
   (
      /* [IN] the dTD to enqueue */
      pointer  dTD_ptr
   )
{ /* Body */
   USB_DEV_STATE_STRUCT_PTR   usb_dev_ptr = (USB_DEV_STATE_STRUCT_PTR)
      (((VUSB20_EP_TR_STRUCT_PTR)dTD_ptr)->SCRATCH_PTR->PRIVATE);

   /*
   ** This function can be called from any context, and it needs mutual
   ** exclusion with itself.
   */
   USB_lock(usb_dev_ptr->DEV_NUM);

   /*
   ** Add the dTD to the free dTD queue (linked via PRIVATE) and
   ** increment the tail to the next descriptor
   */
   EHCI_DTD_QADD(usb_dev_ptr->DTD_HEAD, usb_dev_ptr->DTD_TAIL, 
      (VUSB20_EP_TR_STRUCT_PTR)dTD_ptr);
   usb_dev_ptr->DTD_ENTRIES++;

   USB_unlock(usb_dev_ptr->DEV_NUM);

} /* Endbody */


/*FUNCTION*-------------------------------------------------------------
*
*  Function Name  : _usb_dci_vusb20_flush_endpoint
*  Returned Value : boolean
*  Comments       :
*        Flushes the Endpoint.
*
*END*-----------------------------------------------------------------*/
uint_8 _usb_dci_vusb20_flush_endpoint
   (
       /* [IN] the USB_dev_initialize state structure */
       USB_DEV_STATE_STRUCT_PTR         usb_dev_ptr,
       
       /* [IN] EP Bit Postion */
       uint_32                          bit_pos
   )
{
    uint_32                                     pollcount = 0;
    volatile VUSB20_REG_STRUCT _PTR_            dev_ptr;

    dev_ptr = (volatile VUSB20_REG_STRUCT _PTR_)usb_dev_ptr->DEV_PTR;
    /* Write 1 to the Flush register */
    dev_ptr->REGISTERS.OPERATIONAL_DEVICE_REGISTERS.ENDPTFLUSH = bit_pos;

    /* Wait until flushing completed */
    while ((dev_ptr->REGISTERS.OPERATIONAL_DEVICE_REGISTERS.ENDPTFLUSH & bit_pos) && 
           (pollcount++ < HW_OP_POLL_TIMEOUT)) {
       /* ENDPTFLUSH bit should be cleared to indicate this 
       ** operation is complete 
       */
    } /* EndWhile */
    ASSERT (pollcount < HW_OP_POLL_TIMEOUT); 
    return USB_OK;
}


/*FUNCTION*-------------------------------------------------------------
*
*  Function Name  : _usb_dci_vusb20_is_endpoint_primed
*  Returned Value : boolean
*  Comments       :
*        Checks whether the endpoint is primed ot not.
*
*END*-----------------------------------------------------------------*/
boolean _usb_dci_vusb20_is_endpoint_primed
   (
       /* [IN] the USB_dev_initialize state structure */
       USB_DEV_STATE_STRUCT_PTR         usb_dev_ptr,
       
       /* [IN] EP Bit Postion */
       uint_32                          bit_pos
   )
{
    uint_32                                     pollcount = 0;
    uint_32                                     prime_status;
    volatile VUSB20_REG_STRUCT _PTR_            dev_ptr;

    dev_ptr = (volatile VUSB20_REG_STRUCT _PTR_)usb_dev_ptr->DEV_PTR;
    do {
        /* Set ATDTW bit in USBCMD register to 1 */
        dev_ptr->REGISTERS.OPERATIONAL_DEVICE_REGISTERS.USB_CMD |= EHCI_CMD_DTD_TRIPWIRE;
        
        /* correct status bit in ENDPTSTAT */
        prime_status  = (dev_ptr->REGISTERS.OPERATIONAL_DEVICE_REGISTERS.ENDPTSTATUS & bit_pos);
        prime_status |= (dev_ptr->REGISTERS.OPERATIONAL_DEVICE_REGISTERS.ENDPTPRIME  & bit_pos);
    } while ((!(dev_ptr->REGISTERS.OPERATIONAL_DEVICE_REGISTERS.USB_CMD & EHCI_CMD_DTD_TRIPWIRE)) && 
             (pollcount++ < HW_OP_POLL_TIMEOUT));
    ASSERT (pollcount < HW_OP_POLL_TIMEOUT); 

    dev_ptr->REGISTERS.OPERATIONAL_DEVICE_REGISTERS.USB_CMD &= ~EHCI_CMD_DTD_TRIPWIRE;

    return ((prime_status != 0));
}

/*FUNCTION*-------------------------------------------------------------
*
*  Function Name  : _usb_dci_vusb20_attach_to_qhead
*  Returned Value : boolean
*  Comments       :
*        Attaches the DTD to the Qhead.
*
*END*-----------------------------------------------------------------*/
boolean _usb_dci_vusb20_attach_to_qhead
    (
       /* [IN] the USB_dev_initialize state structure */
       USB_DEV_STATE_STRUCT_PTR                 usb_dev_ptr,

       /* [IN] the Queue head */
       volatile VUSB20_EP_QUEUE_HEAD_STRUCT_PTR ep_queue_head_ptr,

       /* [IN] The transfer descriptor address */
       volatile VUSB20_EP_TR_STRUCT_PTR         dTD_ptr,
       
       /* [IN] EP Bit Postion */
       uint_32                                  bit_pos
    )
{
    uint_32                                     pollcount = 0;
    volatile VUSB20_REG_STRUCT _PTR_            dev_ptr;

    dev_ptr = (volatile VUSB20_REG_STRUCT _PTR_)usb_dev_ptr->DEV_PTR;

    ep_queue_head_ptr->NEXT_DTD_PTR = (uint_32)dTD_ptr;
    ep_queue_head_ptr->SIZE_IOC_INT_STS = 0;

    /* Set ATDTW bit in USBCMD register to 1 */
    dev_ptr->REGISTERS.OPERATIONAL_DEVICE_REGISTERS.USB_CMD |= EHCI_CMD_DTD_TRIPWIRE;
    dev_ptr->REGISTERS.OPERATIONAL_DEVICE_REGISTERS.ENDPTPRIME = bit_pos;
    /* Wait until the endpoint is primed */
    while ((!(dev_ptr->REGISTERS.OPERATIONAL_DEVICE_REGISTERS.ENDPTSTATUS & bit_pos)) 
           && (pollcount++ < HW_OP_POLL_TIMEOUT) && 
           (dev_ptr->REGISTERS.OPERATIONAL_DEVICE_REGISTERS.USB_CMD & EHCI_CMD_DTD_TRIPWIRE)) 
    {
       /* If ENDPTSTATUS is set then endpoint is primed */
       /* This could take longer time for Isochronous endpoint */
       /* The user can customize this according to the need */
    } /* EndWhile */

    dev_ptr->REGISTERS.OPERATIONAL_DEVICE_REGISTERS.USB_CMD &= ~EHCI_CMD_DTD_TRIPWIRE;
    ASSERT (pollcount < HW_OP_POLL_TIMEOUT); 
    return ((pollcount < HW_OP_POLL_TIMEOUT));
}


/*FUNCTION*-------------------------------------------------------------
*
*  Function Name  : _usb_dci_vusb20_add_dTD_emptyhwq
*  Returned Value : uint_8
*  Comments       :
*        Attach DTDs to a Queuehead in which hardware is not acting on.
*
*END*-----------------------------------------------------------------*/
uint_8 _usb_dci_vusb20_add_dTD_emptyhwq
   (
       /* [IN] the USB_dev_initialize state structure */
      USB_DEV_STATE_STRUCT_PTR          usb_dev_ptr,
     
      /* [IN] The transfer descriptor address */
      volatile VUSB20_EP_TR_STRUCT_PTR  first_dTD_ptr,

      /* [IN] The transfer descriptor address */
      volatile VUSB20_EP_TR_STRUCT_PTR  last_dTD_ptr,

      /* [IN] The transfer descriptor address */
      XD_STRUCT_PTR                     xd_ptr
   )
{
    uint_32                                     temp, bit_pos;
    volatile VUSB20_EP_QUEUE_HEAD_STRUCT_PTR    ep_queue_head_ptr;
    volatile VUSB20_REG_STRUCT _PTR_            dev_ptr;

    dev_ptr = (volatile VUSB20_REG_STRUCT _PTR_)usb_dev_ptr->DEV_PTR;
    temp = (2*xd_ptr->EP_NUM + xd_ptr->BDIRECTION);
    bit_pos = (1 << (16 * xd_ptr->BDIRECTION + xd_ptr->EP_NUM));
    
    ep_queue_head_ptr = (VUSB20_EP_QUEUE_HEAD_STRUCT_PTR)
       dev_ptr->REGISTERS.OPERATIONAL_DEVICE_REGISTERS.EP_LIST_ADDR + 
       temp;
    
    _usb_dci_vusb20_attach_to_qhead(usb_dev_ptr, ep_queue_head_ptr, 
                                    first_dTD_ptr, bit_pos);
    if (!usb_dev_ptr->EP_DTD_HEADS[temp]) 
    {
        usb_dev_ptr->EP_DTD_HEADS[temp] = first_dTD_ptr;
    } 
    else 
    {
        (usb_dev_ptr->EP_DTD_TAILS[temp])->NEXT_TR_ELEM_PTR = (uint_32)first_dTD_ptr;
    }
    /* New tail */
    usb_dev_ptr->EP_DTD_TAILS[temp] = last_dTD_ptr;
    return USB_OK;
}

/*FUNCTION*-------------------------------------------------------------
*
*  Function Name  : _usb_dci_vusb20_add_dTD_livehwq
*  Returned Value : uint_8
*  Comments       :
*        Attach DTDs to a Queuehead in which hardware acting on.
*
*END*-----------------------------------------------------------------*/
uint_8 _usb_dci_vusb20_add_dTD_livehwq
   (
      /* [IN] the USB_dev_initialize state structure */
      USB_DEV_STATE_STRUCT_PTR          usb_dev_ptr,
     
      /* [IN] The transfer descriptor address */
      volatile VUSB20_EP_TR_STRUCT_PTR  first_dTD_ptr,

      /* [IN] The transfer descriptor address */
      volatile VUSB20_EP_TR_STRUCT_PTR  last_dTD_ptr,

      /* [IN] The transfer descriptor address */
      XD_STRUCT_PTR                     xd_ptr
   )
{
    uint_32                                     pollcount = 0;
    volatile VUSB20_EP_TR_STRUCT_PTR            old_tail = NULL;
    uint_32                                     temp, bit_pos;
    volatile VUSB20_EP_QUEUE_HEAD_STRUCT_PTR    ep_queue_head_ptr;
    volatile VUSB20_REG_STRUCT _PTR_            dev_ptr;
    uint_32                                     prime_status;

    dev_ptr = (volatile VUSB20_REG_STRUCT _PTR_)usb_dev_ptr->DEV_PTR;
    temp = (2*xd_ptr->EP_NUM + xd_ptr->BDIRECTION);
    bit_pos = (1 << (16 * xd_ptr->BDIRECTION + xd_ptr->EP_NUM));

    ep_queue_head_ptr = (VUSB20_EP_QUEUE_HEAD_STRUCT_PTR)
       dev_ptr->REGISTERS.OPERATIONAL_DEVICE_REGISTERS.EP_LIST_ADDR + 
       temp;
    
    pollcount = 0;
    old_tail  = usb_dev_ptr->EP_DTD_TAILS[temp];

    do {
        /* Set ATDTW bit in USBCMD register to 1 */
        dev_ptr->REGISTERS.OPERATIONAL_DEVICE_REGISTERS.USB_CMD |= EHCI_CMD_DTD_TRIPWIRE;
        /* correct status bit in ENDPTSTAT */
        prime_status = dev_ptr->REGISTERS.OPERATIONAL_DEVICE_REGISTERS.ENDPTSTATUS & bit_pos;
        if (prime_status) {
            /* New tail */
            usb_dev_ptr->EP_DTD_TAILS[temp] = old_tail;
            (usb_dev_ptr->EP_DTD_TAILS[temp])->NEXT_TR_ELEM_PTR = (uint_32)first_dTD_ptr;
            usb_dev_ptr->EP_DTD_TAILS[temp] = last_dTD_ptr;
        }

    } while ((!(dev_ptr->REGISTERS.OPERATIONAL_DEVICE_REGISTERS.USB_CMD & EHCI_CMD_DTD_TRIPWIRE)) && (pollcount++ < HW_OP_POLL_TIMEOUT));

    ASSERT (pollcount < HW_OP_POLL_TIMEOUT); 
    dev_ptr->REGISTERS.OPERATIONAL_DEVICE_REGISTERS.USB_CMD &= ~EHCI_CMD_DTD_TRIPWIRE;

    if (!prime_status) {
        /* The endpoint was not not primed so no other transfers on 
        ** the queue 
        */
        usb_dev_ptr->EP_DTD_TAILS[temp] = old_tail;
        _usb_dci_vusb20_add_dTD_emptyhwq( usb_dev_ptr, first_dTD_ptr, last_dTD_ptr, xd_ptr);
    }
    return USB_OK;
}

/*FUNCTION*-------------------------------------------------------------
*
*  Function Name  : _usb_dci_vusb20_add_dTD
*  Returned Value : USB_OK or error code
*  Comments       :
*        Adds a device transfer desriptor(s) to the queue.
*
*END*-----------------------------------------------------------------*/
uint_8 _usb_dci_vusb20_add_dTD
   (
      /* [IN] the USB_dev_initialize state structure */
      _usb_device_handle         handle,
     
      /* [IN] The transfer descriptor address */
      XD_STRUCT_PTR              xd_ptr
   )
{ /* Body */
   USB_DEV_STATE_STRUCT_PTR                     usb_dev_ptr;
   volatile VUSB20_REG_STRUCT _PTR_             dev_ptr;
   VUSB20_EP_TR_STRUCT_PTR                      dTD_ptr;
   VUSB20_EP_TR_STRUCT_PTR                      temp_dTD_ptr = NULL, first_dTD_ptr = NULL;
   volatile VUSB20_EP_QUEUE_HEAD_STRUCT_PTR     ep_queue_head_ptr;
   uint_32                                      curr_pkt_len, remaining_len; 
   uint_32                                      curr_offset, temp, bit_pos;
   uint_32                                      prime_status = 0;
   
   usb_dev_ptr = (USB_DEV_STATE_STRUCT_PTR)handle;
   dev_ptr = (volatile VUSB20_REG_STRUCT _PTR_)usb_dev_ptr->DEV_PTR;

   remaining_len = xd_ptr->WTOTALLENGTH;
   curr_offset = 0;
   temp = (2*xd_ptr->EP_NUM + xd_ptr->BDIRECTION);
   bit_pos = (1 << (16 * xd_ptr->BDIRECTION + xd_ptr->EP_NUM));
   
   ep_queue_head_ptr = (VUSB20_EP_QUEUE_HEAD_STRUCT_PTR)
      dev_ptr->REGISTERS.OPERATIONAL_DEVICE_REGISTERS.EP_LIST_ADDR + 
      temp;

   /* Invalidate the cache for the the Queue Head Pointer */
   if (ep_queue_head_ptr) {
       DATA_CACHE_INVAL(ep_queue_head_ptr, sizeof(VUSB20_EP_QUEUE_HEAD_STRUCT)); 
   }

   if (ep_queue_head_ptr->STATUS != VUSB_EP_INIT) {
       return USBERR_TX_FAILED;
   }

   do {
      /* Check if we need to split the transfer into multiple dTDs */
      if (remaining_len > VUSB_EP_MAX_LENGTH_TRANSFER) {
         curr_pkt_len = VUSB_EP_MAX_LENGTH_TRANSFER;
      } else {
         curr_pkt_len = remaining_len;
      } /* Endif */
   
      remaining_len -= curr_pkt_len;

      /* Get a dTD from the queue */   
      EHCI_DTD_QGET(usb_dev_ptr->DTD_HEAD, usb_dev_ptr->DTD_TAIL, dTD_ptr);
   
      if (!dTD_ptr) {
         return USBERR_TR_FAILED;
      } /* Endif */
   
      usb_dev_ptr->DTD_ENTRIES--;

      if (curr_offset == 0) {
         first_dTD_ptr = dTD_ptr;
      } /* Endif */
   
      /* Zero the dTD. Leave the last 4 bytes as that is the scratch pointer */
      // USB_memzero(dTD_ptr,(sizeof(VUSB20_EP_TR_STRUCT) - 4));

      /* Initialize the dTD */
      dTD_ptr->SCRATCH_PTR->PRIVATE = handle;
   
      /* Set the Terminate bit */
      dTD_ptr->NEXT_TR_ELEM_PTR = VUSB_EP_QUEUE_HEAD_NEXT_TERMINATE;
   
      /* Fill in the transfer size */
      if (!remaining_len) {
         dTD_ptr->SIZE_IOC_STS = ((curr_pkt_len << 
            VUSBHS_TD_LENGTH_BIT_POS) | (VUSBHS_TD_IOC) | 
            (VUSBHS_TD_STATUS_ACTIVE));
      } else {
         dTD_ptr->SIZE_IOC_STS = ((curr_pkt_len << VUSBHS_TD_LENGTH_BIT_POS) 
            | VUSBHS_TD_STATUS_ACTIVE);
      } /* Endif */
   
      /* Set the reserved field to 0 */
      dTD_ptr->SIZE_IOC_STS &= ~VUSBHS_TD_RESERVED_FIELDS;

      /* 4K apart buffer page pointers */
      dTD_ptr->BUFF_PTR0 = (uint_32)(xd_ptr->WSTARTADDRESS + curr_offset);
      dTD_ptr->BUFF_PTR1 = (dTD_ptr->BUFF_PTR0 + 4096);
      dTD_ptr->BUFF_PTR2 = (dTD_ptr->BUFF_PTR1 + 4096);
      dTD_ptr->BUFF_PTR3 = (dTD_ptr->BUFF_PTR2 + 4096);
      dTD_ptr->BUFF_PTR4 = (dTD_ptr->BUFF_PTR3 + 4096);
   
      curr_offset += curr_pkt_len;

      /* Remember which XD to use for this dTD */
      dTD_ptr->SCRATCH_PTR->XD_FOR_THIS_DTD = (pointer)xd_ptr;
      
      if (temp_dTD_ptr == NULL) 
      {
         temp_dTD_ptr = dTD_ptr;
      } 
      else 
      {
         /* Should not do |=. The Terminate bit should be zero */
         temp_dTD_ptr->NEXT_TR_ELEM_PTR = (uint_32)dTD_ptr;
         temp_dTD_ptr = dTD_ptr;
      } /* Endif */

   } while (remaining_len); /* EndWhile */
   
   /* Check if the endpoint is primed */
   prime_status  = _usb_dci_vusb20_is_endpoint_primed(usb_dev_ptr, bit_pos);
  
   if (!prime_status) {
      /* The HW Endpoint Queue is Empty */
      _usb_dci_vusb20_add_dTD_emptyhwq( usb_dev_ptr, first_dTD_ptr, temp_dTD_ptr, xd_ptr);
   } else {
      /* The HW Endpint Queue is live */
      _usb_dci_vusb20_add_dTD_livehwq( usb_dev_ptr, first_dTD_ptr, temp_dTD_ptr, xd_ptr);
   }
   return USB_OK;
} /* EndBody */

/*FUNCTION*-------------------------------------------------------------
*
*  Function Name  : _usb_dci_vusb20_send_data
*  Returned Value : USB_OK or error code
*  Comments       :
*        Sends data by adding and executing the dTD. Non-blocking.
*
*END*-----------------------------------------------------------------*/
uint_8 _usb_dci_vusb20_send_data
   (
      /* [IN] the USB_dev_initialize state structure */
      _usb_device_handle         handle,
     
      /* [IN] The transfer descriptor address */
      XD_STRUCT_PTR              xd_ptr
   )
{ /* Body */
   /* Add and execute the device transfer descriptor */
   if (xd_ptr->EP_NUM == 0) {
       USB_DEBUG_PRINTF(USB_CONTROL_DEBUG, ("*%d\n", xd_ptr->WTOTALLENGTH));
   } else {
       USB_DEBUG_PRINTF(USB_SEND_DEBUG, ("*%d\n", xd_ptr->WTOTALLENGTH));
   }

   return(_usb_dci_vusb20_add_dTD(handle, xd_ptr));
} /* EndBody */

/*FUNCTION*-------------------------------------------------------------
*
*  Function Name  : _usb_dci_vusb20_recv_data
*  Returned Value : USB_OK or error code
*  Comments       :
*        Receives data by adding and executing the dTD. Non-blocking.
*
*END*-----------------------------------------------------------------*/
uint_8 _usb_dci_vusb20_recv_data
   (
      /* [IN] the USB_dev_initialize state structure */
      _usb_device_handle         handle,
     
      /* [IN] The transfer descriptor address */
      XD_STRUCT_PTR              xd_ptr
   )
{ /* Body */
   /* Add and execute the device transfer descriptor */
   return(_usb_dci_vusb20_add_dTD(handle, xd_ptr));
} /* EndBody */

/*FUNCTION*-------------------------------------------------------------
*
*  Function Name  : _usb_dci_vusb20_process_tr_complete
*  Returned Value : None
*  Comments       :
*        Services transaction complete interrupt
*
*END*-----------------------------------------------------------------*/
void _usb_dci_vusb20_process_tr_complete
   (
      /* [IN] the USB_dev_initialize state structure */
      _usb_device_handle         handle
   )
{ /* Body */
   USB_DEV_STATE_STRUCT_PTR                     usb_dev_ptr;
   volatile VUSB20_REG_STRUCT _PTR_             dev_ptr;
   volatile VUSB20_EP_TR_STRUCT _PTR_           dTD_ptr; 
   volatile VUSB20_EP_TR_STRUCT _PTR_           temp_dTD_ptr;
   volatile VUSB20_EP_QUEUE_HEAD_STRUCT _PTR_   ep_queue_head_ptr;
   uint_32                             temp, i, j, ep_num, direction, bit_pos;
   uint_32                             remaining_length = 0, actual_transfer_length = 0;
   volatile XD_STRUCT _PTR_                     xd_ptr = NULL;
   volatile XD_STRUCT _PTR_                     temp_xd_ptr = NULL;
   uchar_ptr                                    buff_start_address = NULL;
   
   usb_dev_ptr = (USB_DEV_STATE_STRUCT_PTR)handle;
   dev_ptr = (volatile VUSB20_REG_STRUCT _PTR_)usb_dev_ptr->DEV_PTR;
   
   /* We use separate loops for ENDPTSETUPSTAT and ENDPTCOMPLETE because the 
   ** setup packets are to be read ASAP 
   */
   
   /* Process all Setup packet received interrupts */
   bit_pos = dev_ptr->REGISTERS.OPERATIONAL_DEVICE_REGISTERS.ENDPT_SETUP_STAT;
   
   if (bit_pos) {
      for (i=0;i<usb_dev_ptr->MAX_ENDPOINTS;i++) {
         if (bit_pos & (1 << i)) {
            service_ep(usb_dev_ptr->DRV_HANDLE, i, TRUE, 0, 0, 8);
         } /* Endif */
      } /* Endfor */
   } /* Endif */
   
   /* Don't clear the endpoint setup status register here. It is cleared as a 
   ** setup packet is read out of the buffer 
   */

   /* Process non-setup transaction complete interrupts */   
   bit_pos = dev_ptr->REGISTERS.OPERATIONAL_DEVICE_REGISTERS.ENDPTCOMPLETE;

   /* Clear the bits in the register */
   dev_ptr->REGISTERS.OPERATIONAL_DEVICE_REGISTERS.ENDPTCOMPLETE = bit_pos;
   
   if (bit_pos) {
      /* Get the endpoint number and the direction of transfer */
      for (j=0;j<2;j++) {
          if (j == 0) {
              direction = USB_RECV;
          } else {
              direction = USB_SEND;
          }
        for (i=0;i<usb_dev_ptr->MAX_ENDPOINTS;i++) {
             if (bit_pos & (1 << ((direction == USB_RECV) ? i : (16 + i)))) {
            
                ep_num = i;
                temp = (2*ep_num + direction);

                /* Get the first dTD */      
                dTD_ptr = usb_dev_ptr->EP_DTD_HEADS[temp];
            
                ep_queue_head_ptr = (VUSB20_EP_QUEUE_HEAD_STRUCT_PTR)
                    dev_ptr->REGISTERS.OPERATIONAL_DEVICE_REGISTERS.EP_LIST_ADDR + temp;

                /* Invalidate the cache for the the Queue Head Pointer */
                if (ep_queue_head_ptr) {
                    DATA_CACHE_INVAL(ep_queue_head_ptr, sizeof(VUSB20_EP_QUEUE_HEAD_STRUCT)); 
                }
                /* Process all the dTDs for respective transfers */
                while (dTD_ptr) {
                    /* Flush the cache of the DTD PTR */
                    DATA_CACHE_INVAL(dTD_ptr, sizeof(VUSB20_EP_TR_STRUCT));

                    if (dTD_ptr->SIZE_IOC_STS & VUSBHS_TD_STATUS_ACTIVE) {
                        /*
                         * Temporary workaround for hardware issue
                         * Please see bug 12175 for detail
                         */
                        if ((ep_num == 0) && 
                            (((dTD_ptr->SIZE_IOC_STS >> VUSBHS_TD_LENGTH_BIT_POS) & 0x7f) == 64))
                        {
                            if ((dTD_ptr->SIZE_IOC_STS & VUSBHS_TD_IOC) != VUSBHS_TD_IOC) {
                                break;
                            } else {
                                dTD_ptr->SIZE_IOC_STS &= ~VUSBHS_TD_STATUS_ACTIVE;
                            }
                        } else {               
                            /* No more dTDs to process. Next one is owned by VUSB */
                            break;
                        }
                    } /* Endif */               
                    /* Get the correct internal transfer descriptor */
                    xd_ptr = (volatile XD_STRUCT _PTR_)dTD_ptr->SCRATCH_PTR->XD_FOR_THIS_DTD;
                    if (xd_ptr) {
                        buff_start_address = xd_ptr->WSTARTADDRESS;
                        actual_transfer_length = xd_ptr->WTOTALLENGTH;
                        temp_xd_ptr = xd_ptr;
                    } /* Endif */
               
                    /* Get the address of the next dTD */
                    temp_dTD_ptr = (VUSB20_EP_TR_STRUCT_PTR)
                        (dTD_ptr->NEXT_TR_ELEM_PTR & VUSBHS_TD_ADDR_MASK);
                  
                    if (!(dTD_ptr->SIZE_IOC_STS & VUSBHS_TD_ERROR_MASK)) {
                        /* Get the length of transfer from the current dTD */   
                        remaining_length += ((dTD_ptr->SIZE_IOC_STS & VUSB_EP_TR_PACKET_SIZE) >> 16);
                        actual_transfer_length -= remaining_length;
                    } else {
                        /* Clear the Halt condition */
                        ep_queue_head_ptr->SIZE_IOC_INT_STS = 0;
                    } /* Endif */
                    /* Retire the processed dTD */
                    _usb_dci_vusb20_cancel_transfer(handle, ep_num, direction);
                    if (temp_dTD_ptr) {
                        if ((uint_32)temp_dTD_ptr->SCRATCH_PTR->\
                            XD_FOR_THIS_DTD != (uint_32)temp_xd_ptr) 
                        {
                             /* Transfer complete. Call the register service function for the 
                              ** endpoint 
                              */
                            service_ep(usb_dev_ptr->DRV_HANDLE, ep_num, FALSE, direction, 
                            buff_start_address, actual_transfer_length);
                            remaining_length = 0;
                        } /* Endif */
                    } else {
                        /* Transfer complete. Call the register service function for the 
                        ** endpoint 
                        */
                        service_ep(usb_dev_ptr->DRV_HANDLE, ep_num, FALSE, direction, 
                        buff_start_address, actual_transfer_length);
                    } /* Endif */
                    dTD_ptr = temp_dTD_ptr;
                } /* Endwhile */
            } /* Endif */
         } /* Endfor */
      } /* Endfor */
   } /* Endif */
} /* EndBody */

/*FUNCTION*-------------------------------------------------------------
*
*  Function Name  : _usb_dci_vusb20_isr
*  Returned Value : None
*  Comments       :
*        Services all the VUSB_HS interrupt sources
*
*END*-----------------------------------------------------------------*/
void _usb_dci_vusb20_isr
   (
      _usb_device_handle handle
   )
{ /* Body */
   USB_DEV_STATE_STRUCT_PTR                     usb_dev_ptr;
   volatile VUSB20_REG_STRUCT _PTR_             dev_ptr;
   uint_32                                      status;

   usb_dev_ptr = (USB_DEV_STATE_STRUCT_PTR)handle;
#ifndef __USB_OS_MQX__
#ifndef __USB_OTG__   
   IN_ISR[usb_dev_ptr->DEV_NUM] = TRUE;
#endif
#endif

   dev_ptr = (volatile VUSB20_REG_STRUCT _PTR_)usb_dev_ptr->DEV_PTR;

   /* Flush the caches */
   for (;;) {
      status = dev_ptr->REGISTERS.OPERATIONAL_DEVICE_REGISTERS.USB_STS;
      
      if (!(status & dev_ptr->REGISTERS.OPERATIONAL_DEVICE_REGISTERS.USB_INTR)) {
         break;
      } /* Endif */
   
      /* Clear all the interrupts occured */
      dev_ptr->REGISTERS.OPERATIONAL_DEVICE_REGISTERS.USB_STS = status;
   
      if (status & EHCI_STS_RESET) {
         _usb_dci_vusb20_process_reset((pointer)usb_dev_ptr);
      } /* Endif */
      
      if (status & EHCI_STS_PORT_CHANGE) {
         _usb_dci_vusb20_process_port_change((pointer)usb_dev_ptr);
      } /* Endif */
      
      if (status & EHCI_STS_INT) {
         _usb_dci_vusb20_process_tr_complete((pointer)usb_dev_ptr);
      } /* Endif */

      if (status & EHCI_STS_ERR) {
         _usb_dci_vusb20_process_error((pointer)usb_dev_ptr);
      } /* Endif */
     
#if 0 
      if (status & EHCI_STS_SOF) {
         _usb_dci_vusb20_process_SOF((pointer)usb_dev_ptr);
      } /* Endif */
#endif
     if (status & EHCI_STS_SUSPEND) {
          /* Fix for the false suspend detected if line is not SE0 (reset) 
           * then recognize the suspend event */
          if (dev_ptr->REGISTERS.OPERATIONAL_DEVICE_REGISTERS.PORTSCX[0] & 
              EHCI_PORTSCX_LINE_STATUS_BITS) 
          {
              _usb_dci_vusb20_process_suspend((pointer)usb_dev_ptr);
          }
      }
   } /* Endfor */
#ifndef __USB_OS_MQX__   
#ifndef __USB_OTG__    
   IN_ISR[usb_dev_ptr->DEV_NUM] = FALSE;
#endif
#endif   
} /* EndBody */

/*FUNCTION*-------------------------------------------------------------
*
*  Function Name  : _usb_dci_vusb20_process_reset
*  Returned Value : None
*  Comments       :
*        Services reset interrupt
*
*END*-----------------------------------------------------------------*/
void _usb_dci_vusb20_process_reset
   (
      /* [IN] the USB_dev_initialize state structure */
      _usb_device_handle         handle
   )
{ /* Body */
   USB_DEV_STATE_STRUCT_PTR                     usb_dev_ptr;
   volatile VUSB20_REG_STRUCT _PTR_             dev_ptr;
   uint_32                                      temp;
   
   usb_dev_ptr = (USB_DEV_STATE_STRUCT_PTR)handle;
   dev_ptr = (volatile VUSB20_REG_STRUCT _PTR_)usb_dev_ptr->DEV_PTR;
   
   /* Inform the application so that it can cancel all previously queued transfers */
   service_reset(usb_dev_ptr->DRV_HANDLE,  0, 0, 0, 0);
   
   /* The address bits are past bit 25-31. Set the address */
   dev_ptr->REGISTERS.OPERATIONAL_DEVICE_REGISTERS.DEVICE_ADDR &= ~0xFE000000;
   
   /* Clear all the setup token semaphores */
   temp = dev_ptr->REGISTERS.OPERATIONAL_DEVICE_REGISTERS.ENDPT_SETUP_STAT;
   dev_ptr->REGISTERS.OPERATIONAL_DEVICE_REGISTERS.ENDPT_SETUP_STAT = temp;

   /* Clear all the endpoint complete status bits */   
   temp = dev_ptr->REGISTERS.OPERATIONAL_DEVICE_REGISTERS.ENDPTCOMPLETE;
   dev_ptr->REGISTERS.OPERATIONAL_DEVICE_REGISTERS.ENDPTCOMPLETE = temp;
   
   while (dev_ptr->REGISTERS.OPERATIONAL_DEVICE_REGISTERS.ENDPTPRIME & 0xFFFFFFFF) {
      /* Wait until all ENDPTPRIME bits cleared */
   } /* Endif */
   
   /* Write 1s to the Flush register */
   dev_ptr->REGISTERS.OPERATIONAL_DEVICE_REGISTERS.ENDPTFLUSH = 0xFFFFFFFF;
   
   if (dev_ptr->REGISTERS.OPERATIONAL_DEVICE_REGISTERS.PORTSCX[0] & 
      EHCI_PORTSCX_PORT_RESET) 
   {
      usb_dev_ptr->BUS_RESETTING = TRUE;
      usb_dev_ptr->USB_STATE = USB_STATE_POWERED;
   }
#if 0
   else { 
      /* re-initialize */      
      /* 
       * assert here to see the real reason why it so much 
       * time to reach here
       */
       ASSERT(FALSE);
      _usb_dci_vusb20_chip_initialize((pointer)usb_dev_ptr);
      return;
   } /* Endif */
#endif
   service_reset(usb_dev_ptr->DRV_HANDLE,  0, 0, 0, 0);

} /* EndBody */

/*FUNCTION*-------------------------------------------------------------
*
*  Function Name  : _usb_dci_vusb20_process_suspend
*  Returned Value : None
*  Comments       :
*        Services suspend interrupt
*
*END*-----------------------------------------------------------------*/
void _usb_dci_vusb20_process_suspend
   (
      /* [IN] the USB_dev_initialize state structure */
      _usb_device_handle         handle
   )
{ /* Body */
   USB_DEV_STATE_STRUCT_PTR                     usb_dev_ptr;
   
   usb_dev_ptr = (USB_DEV_STATE_STRUCT_PTR)handle;
   
   usb_dev_ptr->USB_DEV_STATE_B4_SUSPEND = usb_dev_ptr->USB_STATE;
   
   usb_dev_ptr->USB_STATE = USB_STATE_SUSPEND;

   /* Inform the upper layers */
   service_suspend(usb_dev_ptr->DRV_HANDLE, 0, 0, 0, 0);

} /* EndBody */

/*FUNCTION*-------------------------------------------------------------
*
*  Function Name  : _usb_dci_vusb20_process_SOF
*  Returned Value : None
*  Comments       :
*        Services SOF interrupt
*
*END*-----------------------------------------------------------------*/
#if 0
void _usb_dci_vusb20_process_SOF
   (
      /* [IN] the USB_dev_initialize state structure */
      _usb_device_handle         handle
   )
{ /* Body */
   USB_DEV_STATE_STRUCT_PTR                     usb_dev_ptr;
   VUSB20_REG_STRUCT_PTR                        dev_ptr;
   
   usb_dev_ptr = (USB_DEV_STATE_STRUCT_PTR)handle;
   dev_ptr = (VUSB20_REG_STRUCT_PTR)usb_dev_ptr->DEV_PTR;

   /* Inform the upper layer */   
   _usb_device_call_service(usb_dev_ptr, USB_SERVICE_SOF, 0, 0, 0, 
      dev_ptr->REGISTERS.OPERATIONAL_DEVICE_REGISTERS.USB_FRINDEX);
} /* EndBody */
#endif


/*FUNCTION*-------------------------------------------------------------
*
*  Function Name  : _usb_dci_vusb20_process_port_change
*  Returned Value : None
*  Comments       :
*        Services port change detect interrupt
*
*END*-----------------------------------------------------------------*/
void _usb_dci_vusb20_process_port_change
   (
      /* [IN] the USB_dev_initialize state structure */
      _usb_device_handle         handle
   )
{ /* Body */
   USB_DEV_STATE_STRUCT_PTR                     usb_dev_ptr;
   volatile VUSB20_REG_STRUCT _PTR_             dev_ptr;
   
   usb_dev_ptr = (USB_DEV_STATE_STRUCT_PTR)handle;
   dev_ptr = (volatile VUSB20_REG_STRUCT _PTR_)usb_dev_ptr->DEV_PTR;

   if (usb_dev_ptr->BUS_RESETTING) {
      /* Bus reset operation complete */
      usb_dev_ptr->BUS_RESETTING = FALSE;
   } /* Endif */
   
   if (!(dev_ptr->REGISTERS.OPERATIONAL_DEVICE_REGISTERS.PORTSCX[0] & 
         EHCI_PORTSCX_PORT_RESET)) 
   {
      /* Get the speed */
      if (dev_ptr->REGISTERS.OPERATIONAL_DEVICE_REGISTERS.PORTSCX[0] & 
         EHCI_PORTSCX_PORT_HIGH_SPEED) 
      {
         usb_dev_ptr->SPEED = USB_SPEED_HIGH;
      } else {
         usb_dev_ptr->SPEED = USB_SPEED_FULL;
      } /* Endif */

      /* Inform the upper layers of the speed of operation */      
      service_speed(usb_dev_ptr->DRV_HANDLE,  0, 0, 0, usb_dev_ptr->SPEED);
   
   } /* Endif */
      
   if (dev_ptr->REGISTERS.OPERATIONAL_DEVICE_REGISTERS.PORTSCX[0] & 
      EHCI_PORTSCX_PORT_SUSPEND) 
   {

      usb_dev_ptr->USB_DEV_STATE_B4_SUSPEND = usb_dev_ptr->USB_STATE;
      usb_dev_ptr->USB_STATE = USB_STATE_SUSPEND;

      /* Inform the upper layers */
      service_port_suspend(usb_dev_ptr->DRV_HANDLE,  0, 0, 0, 0);
   } /* Endif */
   
   if (!(dev_ptr->REGISTERS.OPERATIONAL_DEVICE_REGISTERS.PORTSCX[0] & 
      EHCI_PORTSCX_PORT_SUSPEND) && 
      (usb_dev_ptr->USB_STATE == USB_STATE_SUSPEND)) 
   {
      usb_dev_ptr->USB_STATE = usb_dev_ptr->USB_DEV_STATE_B4_SUSPEND;
      /* Inform the upper layers */
      service_port_resume(usb_dev_ptr->DRV_HANDLE, 0, 0, 0, 0);
      return;
   } /* Endif */
   
   usb_dev_ptr->USB_STATE = USB_STATE_DEFAULT;
} /* EndBody */

/*FUNCTION*-------------------------------------------------------------
*
*  Function Name  : _usb_dci_vusb20_process_error
*  Returned Value : None
*  Comments       :
*        Services error interrupt
*
*END*-----------------------------------------------------------------*/
void _usb_dci_vusb20_process_error
   (
      /* [IN] the USB_dev_initialize state structure */
      _usb_device_handle         handle
   )
{ /* Body */
   USB_DEV_STATE_STRUCT_PTR                     usb_dev_ptr;
   
   usb_dev_ptr = (USB_DEV_STATE_STRUCT_PTR)handle;
   
   /* Increment the error count */
   usb_dev_ptr->ERRORS++;
   
} /* EndBody */

/*FUNCTION*-------------------------------------------------------------
*
*  Function Name  : _usb_dci_vusb20_set_address
*  Returned Value : None
*  Comments       :
*        Sets the newly assigned device address
*
*END*-----------------------------------------------------------------*/
void _usb_dci_vusb20_set_address
   (
      /* [IN] the USB_dev_initialize state structure */
      _usb_device_handle         handle,
     
      /* Address of the device assigned by the host */
      uint_8                     address
   )
{ /* Body */
   USB_DEV_STATE_STRUCT_PTR                     usb_dev_ptr;
   volatile VUSB20_REG_STRUCT _PTR_             dev_ptr;
   
   usb_dev_ptr = (USB_DEV_STATE_STRUCT_PTR)handle;
   dev_ptr = (volatile VUSB20_REG_STRUCT _PTR_)usb_dev_ptr->DEV_PTR;
   
   /* The address bits are past bit 25-31. Set the address */
   dev_ptr->REGISTERS.OPERATIONAL_DEVICE_REGISTERS.DEVICE_ADDR = 
      ((uint_32)address << VUSBHS_ADDRESS_BIT_SHIFT);
   
   usb_dev_ptr->USB_STATE = USB_STATE_ADDRESS;
} /* EndBody */

/*FUNCTION*-------------------------------------------------------------
*
*  Function Name  : _usb_dci_vusb20_get_setup_data
*  Returned Value : None
*  Comments       :
*        Reads the Setup data from the 8-byte setup buffer
*
*END*-----------------------------------------------------------------*/
void _usb_dci_vusb20_get_setup_data
   (
      /* [IN] the USB_dev_initialize state structure */
      _usb_device_handle         handle,
            
      /* [IN] the Endpoint number */
      uint_8                     ep_num,
            
      /* [OUT] address of the buffer to read the setup data into */
      uchar_ptr                  buffer_ptr
   )
{ /* Body */
   USB_DEV_STATE_STRUCT_PTR                     usb_dev_ptr;
   volatile VUSB20_REG_STRUCT _PTR_             dev_ptr;
   VUSB20_EP_QUEUE_HEAD_STRUCT_PTR              ep_queue_head_ptr;
   
   usb_dev_ptr = (USB_DEV_STATE_STRUCT_PTR)handle;
   dev_ptr = (volatile VUSB20_REG_STRUCT _PTR_)usb_dev_ptr->DEV_PTR;

   /* Get the endpoint queue head */
   ep_queue_head_ptr = (VUSB20_EP_QUEUE_HEAD_STRUCT_PTR)
      dev_ptr->REGISTERS.OPERATIONAL_DEVICE_REGISTERS.EP_LIST_ADDR + 2*ep_num + 
      USB_RECV;

   /* Invalidate the cache for the the Queue Head Pointer */
   if (ep_queue_head_ptr) {
       DATA_CACHE_INVAL(ep_queue_head_ptr, sizeof(VUSB20_EP_QUEUE_HEAD_STRUCT)); 
   }
   /* Copy the setup packet to private buffer */
   USB_memcopy((uchar_ptr)ep_queue_head_ptr->SETUP_BUFFER, buffer_ptr, 8);
   
   /* Clear the bit in the ENDPTSETUPSTAT */
   dev_ptr->REGISTERS.OPERATIONAL_DEVICE_REGISTERS.ENDPT_SETUP_STAT = 
      (1 << ep_num);
   
} /* EndBody */

/*FUNCTION*-------------------------------------------------------------
*
*  Function Name  : _usb_dci_vusb20_init_endpoint
*  Returned Value : None
*  Comments       :
*        Initializes the specified endpoint and the endpoint queue head
*
*END*-----------------------------------------------------------------*/
uint_8 _usb_dci_vusb20_init_endpoint
   (
      /* [IN] the USB_dev_initialize state structure */
      _usb_device_handle         handle,
            
      /* [IN] the transaction descriptor address */
      XD_STRUCT_PTR              xd_ptr
   )
{ /* Body */
   USB_DEV_STATE_STRUCT_PTR                     usb_dev_ptr;
   volatile VUSB20_REG_STRUCT _PTR_             dev_ptr;
   VUSB20_EP_QUEUE_HEAD_STRUCT _PTR_            ep_queue_head_ptr;
   uint_32                                      bit_pos;
   
   usb_dev_ptr = (USB_DEV_STATE_STRUCT_PTR)handle;
   dev_ptr = (volatile VUSB20_REG_STRUCT _PTR_)usb_dev_ptr->DEV_PTR;

   /* Get the endpoint queue head address */
   ep_queue_head_ptr = (VUSB20_EP_QUEUE_HEAD_STRUCT_PTR)
      dev_ptr->REGISTERS.OPERATIONAL_DEVICE_REGISTERS.EP_LIST_ADDR + 
      2*xd_ptr->EP_NUM + xd_ptr->BDIRECTION;
      
   if (ep_queue_head_ptr) {
       DATA_CACHE_INVAL(ep_queue_head_ptr, sizeof(VUSB20_EP_QUEUE_HEAD_STRUCT)); 
   }
   bit_pos = (1 << (16 * xd_ptr->BDIRECTION + xd_ptr->EP_NUM));
   
   if (ep_queue_head_ptr->STATUS != VUSB_EP_DEINIT) {
       /* Endpoint Already Initialized */
       return USB_OK;
   }
   /* Check if the Endpoint is Primed */
   if (_usb_dci_vusb20_is_endpoint_primed(usb_dev_ptr, bit_pos)) {
       if (_usb_dci_vusb20_flush_endpoint(usb_dev_ptr, bit_pos) != USB_OK) {
           return USBERR_EP_INIT_FAILED;
       }
   }
   
   /* Set the max packet length, interrupt on Setup and Mult fields */
   if (xd_ptr->EP_TYPE == USB_ISOCHRONOUS_ENDPOINT) {
       /* Mult bit should be set for isochronous endpoints */
       ep_queue_head_ptr->MAX_PKT_LENGTH = ((xd_ptr->WMAXPACKETSIZE << 16) | 
                                            (1 << VUSB_EP_QUEUE_HEAD_MULT_POS));
   } else {
       if (xd_ptr->EP_TYPE != USB_CONTROL_ENDPOINT) {
           ep_queue_head_ptr->MAX_PKT_LENGTH = ((xd_ptr->WMAXPACKETSIZE << 16) | 
                                                (xd_ptr->DONT_ZERO_TERMINATE ? 
                                                 VUSB_EP_QUEUE_HEAD_ZERO_LEN_TER_SEL : 0));
       } else {
           ep_queue_head_ptr->MAX_PKT_LENGTH = ((xd_ptr->WMAXPACKETSIZE << 16) | 
                                                VUSB_EP_QUEUE_HEAD_IOS);
       } /* Endif */
   } /* Endif */
      
   /* Enable the endpoint for Rx and Tx and set the endpoint type */
   dev_ptr->REGISTERS.OPERATIONAL_DEVICE_REGISTERS.ENDPTCTRLX[xd_ptr->EP_NUM] |= 
                     ((xd_ptr->BDIRECTION ? (EHCI_EPCTRL_TX_ENABLE | 
                     EHCI_EPCTRL_TX_DATA_TOGGLE_RST) : 
                     (EHCI_EPCTRL_RX_ENABLE | EHCI_EPCTRL_RX_DATA_TOGGLE_RST)) | 
                     (xd_ptr->EP_TYPE << (xd_ptr->BDIRECTION ? 
                     EHCI_EPCTRL_TX_EP_TYPE_SHIFT : EHCI_EPCTRL_RX_EP_TYPE_SHIFT)));
   /* Set the Endpoint Status to Initialized */
   ep_queue_head_ptr->STATUS = VUSB_EP_INIT;
      
   return USB_OK;
   
} /* EndBody */

/*FUNCTION*-------------------------------------------------------------
*
*  Function Name  : _usb_dci_vusb20_get_transfer_status
*  Returned Value : USB_OK or error code
*  Comments       :
*        Gets the status of a transfer
*
*END*-----------------------------------------------------------------*/
uint_8 _usb_dci_vusb20_get_transfer_status
   (
      /* [IN] the USB_dev_initialize state structure */
      _usb_device_handle         handle,
     
      /* [IN] the Endpoint number */
      uint_8                     ep_num,
            
      /* [IN] direction */
      uint_8                     direction
   )
{ /* Body */
   USB_DEV_STATE_STRUCT_PTR                     usb_dev_ptr;
   VUSB20_EP_TR_STRUCT_PTR                      dTD_ptr;
   XD_STRUCT_PTR                                xd_ptr;
   uint_8                                       status;
   
   usb_dev_ptr = (USB_DEV_STATE_STRUCT_PTR)handle;
   
   /* Unlink the dTD */
   dTD_ptr = usb_dev_ptr->EP_DTD_HEADS[2*ep_num + direction];

   if (dTD_ptr) {
      /* Get the transfer descriptor for the dTD */
      xd_ptr = (XD_STRUCT_PTR)dTD_ptr->SCRATCH_PTR->XD_FOR_THIS_DTD;
      status = xd_ptr->BSTATUS;
   } else {
      status = USB_STATUS_IDLE;
   } /* Endif */
   
   return (status);

} /* EndBody */



/*FUNCTION*-------------------------------------------------------------
*
*  Function Name  : _usb_dci_vusb20_cancel_data
*  Returned Value : USB_OK or error code
*  Comments       :
*        Cancel pending transfers in a Endpoint
*
*END*-----------------------------------------------------------------*/
uint_8 _usb_dci_vusb20_cancel_pend_xfers
   (
      /* [IN] the USB_dev_initialize state structure */
      _usb_device_handle         handle,
      
      /* [IN] the Endpoint number */
      uint_8                     ep_num,
            
      /* [IN] direction */
      uint_8                     direction
   )
{ /* Body */
    USB_DEV_STATE_STRUCT_PTR                     usb_dev_ptr;
    volatile VUSB20_REG_STRUCT _PTR_             dev_ptr;
    volatile VUSB20_EP_TR_STRUCT _PTR_           dTD_ptr; 
    volatile VUSB20_EP_TR_STRUCT _PTR_           temp_dTD_ptr;
    volatile VUSB20_EP_QUEUE_HEAD_STRUCT _PTR_   ep_queue_head_ptr;
    volatile uint_32                             temp, bit_pos;
    volatile uint_32                             remaining_length = 0, actual_transfer_length = 0;
    volatile XD_STRUCT _PTR_                     xd_ptr = NULL;
    volatile XD_STRUCT _PTR_                     temp_xd_ptr = NULL;
    uchar_ptr                                    buff_start_address = NULL;

    usb_dev_ptr = (USB_DEV_STATE_STRUCT_PTR)handle;
    dev_ptr = (volatile VUSB20_REG_STRUCT _PTR_)usb_dev_ptr->DEV_PTR;

    temp = (2* ep_num + direction);
    bit_pos = (1 << (16 * direction + ep_num));

    ep_queue_head_ptr = (VUSB20_EP_QUEUE_HEAD_STRUCT_PTR)
       dev_ptr->REGISTERS.OPERATIONAL_DEVICE_REGISTERS.EP_LIST_ADDR + 
       temp;

    /* Invalidate the cache for the the Queue Head Pointer */
    if (ep_queue_head_ptr) {
        DATA_CACHE_INVAL(ep_queue_head_ptr, sizeof(VUSB20_EP_QUEUE_HEAD_STRUCT)); 
    }
    if (ep_queue_head_ptr->STATUS != VUSB_EP_DEINIT) {
        /* 
         * Endpoint not deintialialzed, this will cause the next up descriptors 
         * to be sent while we cancel the head of the line descriptor.
         */
        return USBERR_TR_CANCEL_FAILED;
    }

    /* Get the first dTD */      
    dTD_ptr = usb_dev_ptr->EP_DTD_HEADS[temp];

    /* Process all the dTDs for respective transfers */
    while (dTD_ptr) {

       /* Flush the cache of the DTD PTR */
       DATA_CACHE_INVAL(dTD_ptr, sizeof(VUSB20_EP_TR_STRUCT)); 

       /* Get the correct internal transfer descriptor */
       xd_ptr = (volatile XD_STRUCT _PTR_)dTD_ptr->SCRATCH_PTR->XD_FOR_THIS_DTD;
       if (xd_ptr) {
          buff_start_address = xd_ptr->WSTARTADDRESS;
          actual_transfer_length = xd_ptr->WTOTALLENGTH;
          temp_xd_ptr = xd_ptr;
       } /* Endif */

       /* Get the address of the next dTD */
       temp_dTD_ptr = (VUSB20_EP_TR_STRUCT_PTR)
          (dTD_ptr->NEXT_TR_ELEM_PTR & VUSBHS_TD_ADDR_MASK);

       if (!(dTD_ptr->SIZE_IOC_STS & VUSBHS_TD_ERROR_MASK)) {
          /* Get the length of transfer from the current dTD */   
          remaining_length += ((dTD_ptr->SIZE_IOC_STS & VUSB_EP_TR_PACKET_SIZE) >> 16);
          actual_transfer_length -= remaining_length;
       } 

       /* Retire the processed dTD */
       _usb_dci_vusb20_cancel_transfer(handle, ep_num, direction);
       if (temp_dTD_ptr) {
          if ((uint_32)temp_dTD_ptr->SCRATCH_PTR->\
             XD_FOR_THIS_DTD != (uint_32)temp_xd_ptr) 
          {
             /* Transfer complete. Call the register service function for the 
             ** endpoint 
             */
             service_ep(usb_dev_ptr->DRV_HANDLE, ep_num, FALSE, direction, 
                buff_start_address, actual_transfer_length);
             remaining_length = 0;
          } /* Endif */
       } else {
          /* Transfer complete. Call the register service function for the 
          ** endpoint 
          */
            service_ep(usb_dev_ptr->DRV_HANDLE, ep_num, FALSE, direction, 
             buff_start_address, actual_transfer_length);
       } /* Endif */
       dTD_ptr = temp_dTD_ptr;
    } /* Endwhile */
    return USB_OK;
} /* EndBody */
/* EOF */


/*
 * This function really returns the following:
 *
 * !0 - device is in suspend
 *  0 - device is resuming
 *
 */

unsigned int
_athr_usb_dci_vusb20_get_linestate(_usb_device_handle handle) {
   USB_DEV_STATE_STRUCT_PTR                     usb_dev_ptr;
   volatile VUSB20_REG_STRUCT _PTR_             dev_ptr;
   unsigned int                                 reg;

   usb_dev_ptr = (USB_DEV_STATE_STRUCT_PTR)handle;
   dev_ptr = (volatile VUSB20_REG_STRUCT _PTR_)usb_dev_ptr->DEV_PTR;

   reg = dev_ptr->REGISTERS.OPERATIONAL_DEVICE_REGISTERS.PORTSCX[0] ;

   /* Redundant checks here:
    *
    * FORCE_PORT_RESUME becomes 1 if a J/K transition is detected on the
    * bus.
    *
    * SUSPEND is 1 when the device is in SUSPEND mode.
    *
    * USB_STATE is the last known STATE.
    */

   if (!(reg & EHCI_PORTSCX_PORT_FORCE_RESUME) &&
       (reg & EHCI_PORTSCX_PORT_SUSPEND) &&
       (usb_dev_ptr->USB_STATE == USB_STATE_SUSPEND))
       return(1);
   else
       return(0);
}
