/*
 * $Id: //depot/sw/branches/1.3_USB_LINUX_port/src/USB/transport/usb/target/arcusb/platform.c#1 $
 *
 * Copyright (c) 2000-2004 Atheros Communications, Inc., All Rights Reserved
 *
 * platform.c : Platform Abstraction routines for the ARC USB SW, includes 
 * part of the ARC defined platform abstraction 
 */
#include "platform.h"
#include "athusb.h"
#include "stdlib.h"
#include "string.h"


typedef     uint_32                 (*ISR_FUNCTION)(void *);

typedef struct {
    A_INTERRUPT_OBJECT      usbIntrObj;
    A_HANDLE                usbIntHandle;
    ISR_FUNCTION            usbIsrPtr;
    DEV_HANDLE              devHandle;
    A_UINT8                 dsrFlag;
    ISR_FUNCTION            usbDsrPtr;
    DRV_HANDLE              drvHandle;
}INTR_STATE_HANDLE, *INTR_STATE_HANDLE_PTR;

static      uint_8                  disable_count[USB_MAX_DEV];
volatile    boolean                 IN_ISR[USB_MAX_DEV];
INTR_STATE_HANDLE_PTR               INTR_CTXT[USB_MAX_DEV];


/*
 *  Prototypes
 */

static A_UINT32 USB_isr(A_UINT32 usbIntvector, A_UINT32 usbHandle);
static void     USB_dsr(A_UINT32 usbIntvector, A_UINT32 usbIntcount, A_UINT32 intrHandle);

/**************************************************************
 * USB_int_uninstall_isr
 *
 * API to Initialize the Interrupt related Context  
 */

A_STATUS 
USB_int_init (
    A_UINT8 deviceNum, 
    void    (*isrFunction)(void *), 
    void    (*dsrFunction)(void *), 
    void    *driverHandle,
    void    **intrHandle
    )
{
    INTR_STATE_HANDLE     *handle;

    handle = malloc(sizeof(INTR_STATE_HANDLE));
    if (handle  == NULL) {
        return A_NO_MEMORY;
    }
    memset(handle, 0, sizeof(INTR_STATE_HANDLE));
    /* Store the ARCUSB ISR handler */
    handle->usbIsrPtr = (ISR_FUNCTION)isrFunction;
    handle->usbDsrPtr = (ISR_FUNCTION)dsrFunction;
    handle->drvHandle = driverHandle;

    INTR_CTXT[deviceNum] = handle;
    *intrHandle = handle;
    return A_OK;
}

/**************************************************************
 * USB_int_defer_status
 *
 * API to set the defer to DSR status called in the ISR 
 * context to schedule a DSR, after the completion of the
 * interrupt.
 */
void
USB_int_defer_status (
    void    *intrHandle,
    A_UINT8 status
    )
{
    INTR_STATE_HANDLE     *handle;

    handle = (INTR_STATE_HANDLE *)intrHandle;
    ASSERT (handle);
    handle->dsrFlag |= status;
}

/**************************************************************
 * USB_int_deinit
 *
 * API to free the Interrupt related Context  
 */
A_STATUS 
USB_int_deinit (
    A_UINT8 deviceNum, 
    void    *intrHandle
    )
{
    INTR_STATE_HANDLE     *handle;

    handle = (INTR_STATE_HANDLE *)intrHandle;
    if (handle != NULL) {
        A_DRIVER_FREE(handle, sizeof(INTR_STATE_HANDLE));
        INTR_CTXT[deviceNum] = 0;
    }
    return A_OK;
}

/**************************************************************
 * USB_int_install_isr
 *
 * API to Intall the USB ISR 
 */

void 
USB_int_install_isr (A_UINT8 deviceNum, 
                     void (*isrFunction)(void *), 
                     void *usbHandle, 
                     void **intrHandle
    )
{
    INTR_STATE_HANDLE  *handle;
    A_UINT8             vectorNumber = sysBoardData.usbParam[deviceNum].iVector;
    
    handle = INTR_CTXT[deviceNum];
    ASSERT (handle);
    /* Store the ARCUSB ISR handler */
    handle->usbIsrPtr = (ISR_FUNCTION)isrFunction;
    handle->devHandle = usbHandle;

    /* Create the Interrupt Obj */
    A_CREATE_INTERRUPT (
        vectorNumber,                   /* Vector to attach to       */
        (A_UINT32)handle,               /* Data pointer              */
        USB_isr,                        /* Interrupt Service Routine */
        USB_dsr,                        /* Deferred Service Routine  */
        &handle->usbIntHandle,          /* returned handle           */
        &handle->usbIntrObj             /* interrupt                 */
        );
 
    /* Connect to the Interupt Object */
    if (handle->usbIntHandle) {
        A_ATTACH_INTERRUPT(handle->usbIntHandle);
    }
    *intrHandle = handle;
}

/**************************************************************
 * USB_int_uninstall_isr
 *
 * API to UnInstall the USB ISR 
 */

void 
USB_int_uninstall_isr (A_UINT8 deviceNum, void *intrHandle)
{ 
    INTR_STATE_HANDLE     *handle;

    handle = (INTR_STATE_HANDLE *)intrHandle;
    if (handle != NULL) {
        if (handle->usbIntHandle) {
    
            /* Detach the Interrupt and then delete it */
            A_DETACH_INTERRUPT(handle->usbIntHandle);
            A_DELETE_INTERRUPT(handle->usbIntHandle);
            handle->usbIntHandle = (A_HANDLE)NULL;
        }
    }
} 

/**************************************************************
 * USB_isr 
 *
 * USB Interrupt Hander 
 */

static A_UINT32 
USB_isr(A_UINT32 usbIntvector, A_UINT32 intrHandle)
{
    INTR_STATE_HANDLE     *handle;
    A_UINT8                dsrSched;

    handle = (INTR_STATE_HANDLE *)intrHandle;
    /* Mask the vector */
    A_DISABLE_INTERRUPT(usbIntvector);                   
    
    /* Call the real handler */
    (*(handle->usbIsrPtr))((void *)handle->devHandle);

    /* Clear the Interrupt */
    A_ACK_INTERRUPT(usbIntvector);                   

    /* Reenable the Interrupt */
    A_ENABLE_INTERRUPT(usbIntvector);

    dsrSched = handle->dsrFlag ;
    handle->dsrFlag = 0;

    return  ((dsrSched) ? CYG_ISR_CALL_DSR : CYG_ISR_HANDLED);
}

/**************************************************************
 * USB_isr 
 *
 * USB DSR procedure only invoked is DSR processing is 
 * required
 */
static void 
USB_dsr(A_UINT32 usbIntvector, A_UINT32 usbIntcount, A_UINT32 intrHandle)
{
    INTR_STATE_HANDLE     *handle;

    handle = (INTR_STATE_HANDLE *)intrHandle;
    
    (*(handle->usbDsrPtr))((void *)handle->drvHandle);
}

/**************************************************************
 * _bsp_get_usb_vector 
 *
 * Returns the Vector Number 
 */

A_UINT8 
_bsp_get_usb_vector (A_UINT8 deviceNumber)
{
    if (deviceNumber < USB_MAX_DEV) {
        return (sysBoardData.usbParam[deviceNumber].iVector);
    } else {
       return 0;
    }
}

/**************************************************************
 * _bsp_get_usb_base 
 *
 * Returns the USB base address
 */

void *
_bsp_get_usb_base (A_UINT8 deviceNumber)
{
    if (deviceNumber < USB_MAX_DEV) {
        return (void *)(sysBoardData.usbParam[deviceNumber].usbMemBase);
    } else {
        return ((void *)0);
    }
}

/**************************************************************
 * _bsp_get_usb_capability_register_base
 *
 * Returns the USB capability register base address
 */

void * 
_bsp_get_usb_capability_register_base (A_UINT8 deviceNumber)
{ 
    if (deviceNumber < USB_MAX_DEV) {
        return ((void *)(sysBoardData.usbParam[deviceNumber].usbMemBase + 
                         USB_CAPABILITY_REG_OFFSET));
    } else {
        return 0;
    }
}

/**************************************************************
 * _platform_lock
 *
 * Disables the USB interrupt & locks the scheduler from 
 * preempting the caller
 */
void 
_platform_lock(A_UINT8 deviceNumber)
{
    if (deviceNumber < USB_MAX_DEV) {
        if (!IN_ISR[deviceNumber]) {
            LOCK_TASKS();
            _disable_interrupts (deviceNumber);
        }
    }
}

/**************************************************************
 * _platform_unlock
 *
 * Enables the USB interrupt & unlocks the scheduler  so that 
 * called task can be preemted
 */
void 
_platform_unlock(A_UINT8 deviceNumber)
{
    if (deviceNumber < USB_MAX_DEV) {
        if (!IN_ISR[deviceNumber]) {
            UNLOCK_TASKS();
            _enable_interrupts (deviceNumber);
        }
    }
}

/**************************************************************
 * _disable_interrupts
 *
 * Disables the USB Interrupt
 */
void 
_disable_interrupts (A_UINT8 deviceNumber)
{ 
    A_TASK_LOCK();
    if (deviceNumber < USB_MAX_DEV) {
        if (!disable_count[deviceNumber]) {
            A_DISABLE_INTERRUPT(sysBoardData.usbParam[0].iVector);
        }
        disable_count[deviceNumber]++;
    }
    A_TASK_UNLOCK();
}

/**************************************************************
 * _enable_interrupts
 *
 * Enables the USB interrupt
 */
void 
_enable_interrupts (A_UINT8 deviceNumber)
{
    A_TASK_LOCK();
    if (deviceNumber < USB_MAX_DEV) {
        ASSERT(disable_count[deviceNumber] || IN_ISR[deviceNumber]);
        disable_count[deviceNumber]--;
        if (!disable_count[deviceNumber]) {
            A_ENABLE_INTERRUPT(sysBoardData.usbParam[0].iVector);
        }
    }
    A_TASK_UNLOCK();
}
