/*
 * $Id: //depot/sw/branches/1.3_USB_LINUX_port/src/USB/transport/usb/target/drv/athusbdrv.c#1 $
 *
 * Copyright (c) 2000-2003 Atheros Communications, Inc., All Rights Reserved
 *
 * Atheros USB target driver for the AR5523 Chip
 */

#include "usb.h"
#include "devapi.h"
#include "athusb.h"
#include "stdlib.h"
#include "string.h"

#ifdef DEBUG
A_UINT32 athUsbDebug =  0;
#endif
USB_PARAM  usbParam[USB_MAX_DEV];

#define NUM_SEND_PIPES      (2)
#define NUM_RECV_PIPES      (10)


A_UINT32  sendPipePktSize[NUM_SEND_PIPES] = {
    512, 2048, };

A_UINT32  recvPipePktSize[NUM_RECV_PIPES] = {
    512, 2048, 512, 512, 512, 512, 512, 512, 512, 512, };

#define USB_DATA_SEND_PIPE_TO_EP(_pipe)  (_pipe + 1)
#define USB_DATA_RECV_PIPE_TO_EP(_pipe)  (_pipe + 1)
#define USB_DATA_SEND_EP_TO_PIPE(_pipe)  (_pipe - 1)
#define USB_DATA_RECV_EP_TO_PIPE(_pipe)  (_pipe - 1)

/******************************************************************************
* athUsbDrvInit
*
* This routine initializes the USB Driver and returns the handle to the driver
* which is to be used in subsequent calls to the USB Driver. The function also 
* returns the number of data pipes available to the caller of the function 
*
* Returns: A_OK, A_EERROR
*/
A_UINT8 poll = 0;

A_STATUS 
athUsbDrvInit( 
    IN  A_UINT8                  usbDeviceNum,
    IN  APP_HANDLE               appHandle,
    IN  RECV_INDICATION_HANDLER  recvIndHandler,
    IN  SEND_CONFIRM_HANDLER     sendCfmHandler,
    OUT DRV_HANDLE              *pDrvHandle,
    OUT A_UINT8                 *pSendPipes,
    OUT A_UINT8                 *pRecvPipes
    )
{
    PUSBDRV_HANDLE              pUsbDrvHandle;
    A_STATUS                    status;
    DrvEventList                *pFreeEventList = NULL;
    USB_DRV_OS_CTXT             *pOsCtxt;
    drvEventNode                *pDrvEvent;
    A_UINT32                    i;
    
    /* Validate the device number */
    if (usbDeviceNum >= USB_MAX_DEV) {
        return A_DEVICE_NOT_FOUND;
    }

    /* Allocate the USB driver structure */
    pUsbDrvHandle = (PUSBDRV_HANDLE)A_DRIVER_MALLOC(sizeof(USBDRV_HANDLE));
    if (pUsbDrvHandle == NULL) {
        return A_NO_MEMORY;
    }
    A_MEM_ZERO(pUsbDrvHandle, sizeof(USBDRV_HANDLE));
    *pDrvHandle = pUsbDrvHandle;

    /* Store the Handlers */
    pUsbDrvHandle->deviceNum      = usbDeviceNum;
    pUsbDrvHandle->appHandle      = appHandle;
    pUsbDrvHandle->recvIndHandler = recvIndHandler;
    pUsbDrvHandle->sendCfmHandler = sendCfmHandler,

    /* Store the Number of pipes available to the caller */
    *pSendPipes = NUM_SEND_PIPES;
    *pRecvPipes = NUM_RECV_PIPES;

    status = USB_int_init(usbDeviceNum, NULL, athUsbProcessXferEvents, 
                         pUsbDrvHandle, &pUsbDrvHandle->pIntHandle);
    ASSERT(status == A_OK);

    pUsbDrvHandle->pOsContext = (PUSB_DRV_OS_CTXT)A_DRIVER_MALLOC(sizeof(USB_DRV_OS_CTXT));
    if (pUsbDrvHandle->pOsContext == NULL) {
        /* Free the driver handle */
        A_DRIVER_FREE(pUsbDrvHandle, sizeof(USBDRV_HANDLE));
        return A_NO_MEMORY;
    }
    pOsCtxt = pUsbDrvHandle->pOsContext;
    A_MEM_ZERO(pOsCtxt, sizeof(USB_DRV_OS_CTXT));
    /* 
     * Initialize the Event list for storing Events and list 
     * for storing free Event Nodes 
     */
    STAILQ_INIT((DrvEventList *)&pOsCtxt->freeEventList);
    STAILQ_INIT((DrvEventList *)&pOsCtxt->dsrEventList);
    pFreeEventList = (DrvEventList *)&pOsCtxt->freeEventList;

    for (i = 0; i < MAX_NUM_DSR_EVENTS; i++) {
        /* Add them to the Free List */
        pDrvEvent = &pOsCtxt->eventNodeList[i];
        ASSERT(pDrvEvent);
        STAILQ_INSERT_TAIL(pFreeEventList, pDrvEvent, nextEvent);
    }
    status = arcInit(pUsbDrvHandle, usbDeviceNum);

    if (status != A_OK) {
        /* Arc Initialization Part Failed. Clean the installation */
        arcShutdown(pUsbDrvHandle);

        status = USB_int_deinit (usbDeviceNum, pUsbDrvHandle->pIntHandle);
        ASSERT(status == A_OK);
        /* Free the allocated OS context Structure */
        A_DRIVER_FREE(pUsbDrvHandle->pOsContext, sizeof(USB_DRV_OS_CTXT));
        /* Free the driver handle */
        A_DRIVER_FREE(pUsbDrvHandle, sizeof(USBDRV_HANDLE));
        *pDrvHandle = 0;
        return status;
    }
    return A_OK;
}

/******************************************************************************
* athUsbDrvShutdown
*
* This routine shut the USB Driver.
*
* Returns: A_OK, A_EERROR
*/
A_STATUS 
athUsbDrvShutdown( 
    IN DRV_HANDLE           pDrvHandle
    )
{
    PUSBDRV_HANDLE      pUsbDrvHandle;
    A_STATUS            status;

    pUsbDrvHandle = (PUSBDRV_HANDLE) pDrvHandle;
    
    status = arcShutdown(pUsbDrvHandle);
    ASSERT(status == A_OK);

    status = USB_int_deinit (pUsbDrvHandle->deviceNum, pUsbDrvHandle->pIntHandle);
    ASSERT(status == A_OK);

    ASSERT(pUsbDrvHandle->pOsContext);
    A_DRIVER_FREE(pUsbDrvHandle->pOsContext, sizeof(USB_DRV_OS_CTXT));
    
    /* Free the driver handle */
    ASSERT(pUsbDrvHandle);
    A_DRIVER_FREE(pUsbDrvHandle, sizeof(USBDRV_HANDLE));
    
    return A_OK;
}

/******************************************************************************
* athUsbDrvRegEventHndlr
*
* This routine shut the USB Driver.
*
* Returns: A_OK, A_EERROR
*/

A_STATUS 
athUsbDrvRegEventHndlr( 
    IN DRV_HANDLE           pDrvHandle,
    IN USBDRV_EVENT         usbEvent,
    IN DRV_EVENT_HANDLER    eventHandler
    )
{
    PUSBDRV_HANDLE      pUsbDrvHandle;

    pUsbDrvHandle = (PUSBDRV_HANDLE)pDrvHandle;

    /* Event types handled currently are 
     * DRV_SUSPEND_EVENT & USB_RESUME_EVENT */
    if (usbEvent < DRV_MAX_EVENTS) {
        pUsbDrvHandle->usbEventHandler[usbEvent] = eventHandler;
        return A_OK;
    } else {
        return A_ERROR;
    }
}

/******************************************************************************
* athUsbDrvRecv
*
* This function is called to post buffer for receiving data in a specified 
* receive pipe.
* 
* Returns: A_OK, A_EERROR
*/

A_STATUS 
athUsbDrvRecv(              
    IN DRV_HANDLE         drvHandle,
    IN A_UINT8            pipeNum,
    IN A_UINT8           *pBuffer
    )
{
    A_UINT8             status;
    PUSBDRV_HANDLE      pUsbDrvHandle;

    pUsbDrvHandle = (PUSBDRV_HANDLE)drvHandle;
    /* Validate the Pipe Number */
    if ((pipeNum >= NUM_RECV_PIPES) || (!pUsbDrvHandle)) {
        return A_ERROR;
    }
    status = _usb_device_recv_data(pUsbDrvHandle->deviceHandle, 
                                   USB_DATA_RECV_PIPE_TO_EP(pipeNum), 
                                   pBuffer,
                                   recvPipePktSize[pipeNum]);
    if (status != USB_OK ) {
        /* Recv call failed */
        return A_ERROR;
    } 
    return A_OK;
}

/******************************************************************************
* athUsbDrvSend
*
* This function is called to send data in a specified send pipe.
* 
* Returns: A_OK, A_EERROR
*/
#ifdef DO_ZERO_TERM
A_STATUS 
athUsbDrvSend(    
    IN DRV_HANDLE         drvHandle,
    IN A_UINT8            pipeNum,
    IN A_UINT8           *pBuffer,
    IN A_UINT32           bufSize
   )
#else
A_STATUS 
athUsbDrvSend(    
    IN DRV_HANDLE         drvHandle,
    IN A_UINT8            pipeNum,
    IN A_UINT8           *pBuffer
   )
#endif
{
    A_UINT8             status;
    PUSBDRV_HANDLE      pUsbDrvHandle;
#ifndef DO_ZERO_TERM
    A_UINT32           bufSize;
    bufSize = sendPipePktSize[pipeNum];
#endif

    pUsbDrvHandle = (PUSBDRV_HANDLE)drvHandle;

    /* Validate the Pipe Number */
    if ((pipeNum >= NUM_SEND_PIPES) || (!pUsbDrvHandle)) {
        return A_ERROR;
    }
    status = _usb_device_send_data(pUsbDrvHandle->deviceHandle, 
                                   USB_DATA_SEND_PIPE_TO_EP(pipeNum), 
                                   pBuffer,
                                   bufSize);
    if (status != USB_OK ) {
        /* Send call failed */
        return A_ERROR;
    } 
    return A_OK;
}

/******************************************************************************
* athUsbDrvSendRecv
*
* This function is called to send cfm & Ind to the USB driver layer 
* 
* Returns: A_OK, A_EERROR
*/

void 
athUsbDrvSendRecv(    
    IN PUSBDRV_HANDLE       pUsbDrvHandle,
    IN A_UINT8              epNum,
    IN A_UINT8              xferDir,
    IN A_UINT8              *pBuffer,
    IN A_UINT32             length      
   )
{
    
    if (poll) {
        A_TASK_UNLOCK();
        if (xferDir == USB_RECV) {
            /* Receive Indication Event */
            (pUsbDrvHandle->recvIndHandler)(pUsbDrvHandle->appHandle,
                                            USB_DATA_RECV_EP_TO_PIPE(epNum),
                                            pBuffer, length);
        } else {
           /* Send Confirm Event */
           if (xferDir == USB_SEND){
                (pUsbDrvHandle->sendCfmHandler)(pUsbDrvHandle->appHandle,
                                            USB_DATA_SEND_EP_TO_PIPE(epNum),
                                            pBuffer, length);
           } else {
                ASSERT(0);
           }
        } 
       A_TASK_LOCK();
    } else {
        DrvEventList           *pFreeEventList;
        DrvEventList           *pDsrEventList;
        drvEventNode           *pEvent;

        pFreeEventList = (DrvEventList *)
                        &pUsbDrvHandle->pOsContext->freeEventList;
        pDsrEventList  = (DrvEventList *)
                     &pUsbDrvHandle->pOsContext->dsrEventList;
    
     
        /* Remove a Node from the Free List */
	    pEvent = STAILQ_FIRST(pFreeEventList);
	    if (pEvent != NULL) {
            /* Remove the Node from Queue */
		    STAILQ_REMOVE_HEAD(pFreeEventList, nextEvent);

            /* Add the Event to the DSR Event Queue */
        
            pEvent->eventType = XFER_EVENT;
            pEvent->event.xferEvent.pipeNum = USB_DATA_SEND_EP_TO_PIPE(epNum);
            pEvent->event.xferEvent.xferDir = ((xferDir == USB_SEND) ? DRV_EVENT_SEND : DRV_EVENT_RECV);
            pEvent->event.xferEvent.pBuffer = pBuffer;
            pEvent->event.xferEvent.xferLen = length;
        
            STAILQ_INSERT_TAIL(pDsrEventList, pEvent, nextEvent);
            USB_int_defer_status ( pUsbDrvHandle->pIntHandle, 1);
        }
        ASSERT (pEvent);
    }
}

/******************************************************************************
* athUsbDrvSendRecv
*
* This function is called to send cfm & Ind to the USB driver layer 
* 
* Returns: A_OK, A_EERROR
*/
void 
athUsbDrvCtrlEvent(    
    IN PUSBDRV_HANDLE       pUsbDrvHandle,
    IN USBDRV_EVENT         usbEvent
   )
{
    if (poll) {
       A_TASK_UNLOCK();
       if (pUsbDrvHandle->usbEventHandler[usbEvent]) {
           (pUsbDrvHandle->usbEventHandler[usbEvent])(pUsbDrvHandle->appHandle);
       }
       A_TASK_LOCK();
    } else {
        DrvEventList           *pFreeEventList;
        DrvEventList           *pDsrEventList;
        drvEventNode           *pEvent;

        pFreeEventList = (DrvEventList *)
                     &pUsbDrvHandle->pOsContext->freeEventList;
        pDsrEventList  = (DrvEventList *)
                     &pUsbDrvHandle->pOsContext->dsrEventList;
     
        /* Remove a Node from the Free List */
	    pEvent = STAILQ_FIRST(pFreeEventList);
	    if (pEvent != NULL) {
            /* Remove the Node from Queue */
		    STAILQ_REMOVE_HEAD(pFreeEventList, nextEvent);

            pEvent->eventType = CTRL_EVENT;
            pEvent->event.ctrlEvent.drvEvent = usbEvent;
        
            /* Add the Event to the DSR Event Queue */
            STAILQ_INSERT_TAIL(pDsrEventList, pEvent, nextEvent);
            USB_int_defer_status ( pUsbDrvHandle->pIntHandle, 1);
        }
        ASSERT (pEvent);
    }
}

/******************************************************************************
* athUsbProcessXferEvents
*
* This function is called to process the Events queued during the ISR
* processing of packets. Eventually sends the Indications & Cfm to the
* registered handler.
* 
*/
void 
athUsbProcessXferEvents(
    IN DRV_HANDLE       drvHandle
    )
{
    DrvEventList           *pFreeEventList;
    DrvEventList           *pDsrEventList;
    drvEventNode           *pEvent;
    drvEventNode           *pEventNext;
    PUSBDRV_HANDLE          pUsbDrvHandle;

    pUsbDrvHandle = (PUSBDRV_HANDLE)drvHandle;
    ASSERT (pUsbDrvHandle);

	if (poll)	 return;    

    pFreeEventList = (DrvEventList *)
                     &pUsbDrvHandle->pOsContext->freeEventList;
    pDsrEventList  = (DrvEventList *)
                     &pUsbDrvHandle->pOsContext->dsrEventList;
    
    _disable_interrupts(pUsbDrvHandle->deviceNum);
    /* Remove a Node from the DSR List */
	pEvent = STAILQ_FIRST(pDsrEventList);
    _enable_interrupts(pUsbDrvHandle->deviceNum);
	
    while (pEvent != NULL) {

        _disable_interrupts(pUsbDrvHandle->deviceNum);
        
        pEventNext = STAILQ_NEXT(pEvent, nextEvent);
		STAILQ_REMOVE_HEAD(pDsrEventList, nextEvent);
        
        _enable_interrupts(pUsbDrvHandle->deviceNum);
		
        ASSERT(pEvent->eventType <= CTRL_EVENT);
        if (pEvent->eventType == CTRL_EVENT) {
            
            /* call the control event handler */
            if (pUsbDrvHandle->usbEventHandler[pEvent->event.ctrlEvent.drvEvent]) {
                (pUsbDrvHandle->usbEventHandler[pEvent->event.ctrlEvent.drvEvent])(pUsbDrvHandle->appHandle);
            }

        } else if (pEvent->eventType == XFER_EVENT) {
            if (pEvent->event.xferEvent.xferDir == DRV_EVENT_RECV) {
                /* Receive Indication Event */
                (pUsbDrvHandle->recvIndHandler)(pUsbDrvHandle->appHandle,
                                                pEvent->event.xferEvent.pipeNum,
                                                pEvent->event.xferEvent.pBuffer,
                                                pEvent->event.xferEvent.xferLen);
            } else if (pEvent->event.xferEvent.xferDir == DRV_EVENT_SEND){
                /* Send Confirm Event */
                (pUsbDrvHandle->sendCfmHandler)(pUsbDrvHandle->appHandle,
                                                pEvent->event.xferEvent.pipeNum,
                                                pEvent->event.xferEvent.pBuffer,
                                                pEvent->event.xferEvent.xferLen);
            }
        }

        _disable_interrupts(pUsbDrvHandle->deviceNum);
        /* Add the processed Event to the Free List */
        STAILQ_INSERT_TAIL(pFreeEventList, pEvent, nextEvent);

        _enable_interrupts(pUsbDrvHandle->deviceNum);
        pEvent = pEventNext;

	}
    return;
}


/******************************************************************************
* athUsbDrvDisable
*
* This function is called to Disable the USB device
* 
* Returns: A_OK, A_EERROR
*/

A_STATUS 
athUsbDrvDisable (
    IN DRV_HANDLE               pDrvHandle
    )
{
    PUSBDRV_HANDLE      pUsbDrvHandle;

    pUsbDrvHandle = (PUSBDRV_HANDLE)pDrvHandle;
    _disable_interrupts(pUsbDrvHandle->deviceNum);
    return A_OK;
}

/******************************************************************************
* athUsbDrvEnable
*
* This function is called to Enable the USB device 
* 
* Returns: A_OK, A_EERROR
*/

A_STATUS 
athUsbDrvEnable (
    IN DRV_HANDLE               pDrvHandle
    )
{
    PUSBDRV_HANDLE      pUsbDrvHandle;

    pUsbDrvHandle = (PUSBDRV_HANDLE)pDrvHandle;
    _enable_interrupts(pUsbDrvHandle->deviceNum);
    return A_OK;
}

/* XXX */

A_STATUS
athUsbDrvPoll (
    IN DRV_HANDLE               pDrvHandle
    ) 
{
    PUSBDRV_HANDLE      pUsbDrvHandle;
    extern void _usb_dci_vusb20_isr(_usb_device_handle handle);

    pUsbDrvHandle = (PUSBDRV_HANDLE)pDrvHandle;

    A_TASK_LOCK();
    athUsbDrvDisable(pDrvHandle);
    _usb_dci_vusb20_isr(pUsbDrvHandle->deviceHandle);
	athUsbProcessXferEvents(pDrvHandle);
    athUsbDrvEnable(pDrvHandle);
    A_TASK_UNLOCK();

    return A_OK;
}

unsigned int
athUsbDrvGetLineState(
    IN DRV_HANDLE               pDrvHandle
    )
{
    PUSBDRV_HANDLE      pUsbDrvHandle;
    extern unsigned int _athr_usb_dci_vusb20_get_linestate(_usb_device_handle);

    pUsbDrvHandle = (PUSBDRV_HANDLE)pDrvHandle;

    return(_athr_usb_dci_vusb20_get_linestate(pUsbDrvHandle->deviceHandle)); 
}

