#ifdef BOOT_LOADER_PLUS
/*****************************************************************************
**                                                                          **
**                          ATHEROS Communications                          **
**                                                                          **
** **************************************************************************/

/**************************** COPYRIGHT INFORMATION ***************************
**                                                                           **
**   This program contains proprietary information which is a trade          **
**   secret of ATHEROS Communications and also is protected as an            **
**   unpublished work under applicable Copyright laws. Recipient is to       **
**   retain this program in confidence and is not permitted to use or make   **
**   copies thereof other than as permitted in a written agreement with      **
**   ATHEROS Communications.                                                 **
**                                                                           **
** ***************************************************************************/

/******************************************************************************
**                                                                           ** 
** Project       : ATC_FDISK                                                 ** 
**                                                                           ** 
** Module        : FLASH DISK                                                ** 
**                                                                           ** 
** File name     : Flash.c                                                   ** 
**                                                                           ** 
** Authors       : Bharathi Thillai-Bharathi.Thillai@wipro.com               ** 
**                 Sajith Kumar M.A-sajith.kumar@wipro.com                   ** 
**                                                                           ** 
** Created On    : Thu June 03 2004                                          ** 
**                                                                           ** 
** Description   : This file contains the Falsh Utilities                    ** 
**                                                                           ** 
** Reference(s)  :                                                           ** 
**                                                                           ** 
**                                                                           ** 
** ***************************************************************************/

/******************************************************************************
**                                                                           **
** History                                                                   **
** Date          Revision     Comment                                        **
** 03-June-2004      0         Created                                       **
**                                                                           **
**                                                                           **
*******************************************************************************/


/******************************************************************************
**                                                                           **
**   HEADER (INCLUDE) SECTION   (check protection before include)            **
**                                                                           **
** ****************************************************************************/

#include "usb.h"
#include "platform.h"
#include "devapi.h"
#include "athusb.h"
#include "stdlib.h"
#include "string.h"
#include "FlashDisk.h"
#include "Flash.h"

volatile A_UINT32   *spiCSregPtr = (A_UINT32 *)SPI_CONTROL_STATUS_REG;   
volatile A_UINT32   *spiAOregPtr = (A_UINT32 *)SPI_ADDRESS_OPCODE_REG;    
volatile A_UINT32   *spiDATAregPtr = (A_UINT32 *)SPI_DATA_REG;    
volatile A_UINT32   *flashSelctPtr = (A_UINT32 *)SPI_FLASH_SELECT;    
FLASH_INFO_STRUCT flashInfo;

/******************************************************************************
*                                                                             *   
* Function Name  : waitTillDeviceBusy                                         *   
*                                                                             *
* Parameters     : None                                                       *
*                                                                             *
* Returned Value : TRUE  -If success                                          *   
*                  FALSE -If Failed                                           *
* Description    : Wait until the previous opetaion is over                   *    
*                                                                             *   
*******************************************************************************/
A_UINT8 waitTillDeviceBusy(void)
{
    A_UINT32 count=0;
    while ( 1 )
    {
        if ( ((*spiCSregPtr) & SPI_BUSY_MASK)==0 )
        {
            return(TRUE);
        }
        count++;
        if ( count > 0xF0000 )
        {
            FLASHDISK_INFO(("Timeout\n"));
            return(FALSE);
        }
    }
}
/******************************************************************************
*                                                                             *   
* Function Name  : selectFlash                                                *   
*                                                                             *
* Parameters     : None                                                       *
*                                                                             *
* Returned Value : None                                                       *   
* Description    : Select the Flash Type as SPI based                         *    
*                                                                             *   
*******************************************************************************/

void selectFlash(void)
{
    (*flashSelctPtr) |= 0x00000001;
}


/******************************************************************************
*                                                                             *   
* Function Name  : wait_on_spi_device_status                                  *   
*                                                                             *
* Parameters     : None                                                       *
*                                                                             *
* Returned Value : TRUE  -If success                                          *   
*                  FALSE -If Failed                                           *
* Description    : Wait until the previous SPI opetaion is over               *    
*                                                                             *   
*******************************************************************************/

A_UINT8 wait_on_spi_device_status ( void )
{
    A_UINT32 count=0;
    while ( 1 )
    {
        (*spiAOregPtr) = SPI_DISABLE_WRITE;
        (*spiCSregPtr) = SPI_SEND_RECEIVE_ONE_BYTE;

        // Poll
        waitTillDeviceBusy();
        if ( ( (*spiDATAregPtr) & 0x1 )==0 )
        {
            return(TRUE);
        }
        count++;
        if ( count > 0xF0000 )
        {
            FD_DBG_PRINTF(FD_INFO,("Timeout\n"));
            return(FALSE);
        }

    } 
}
/******************************************************************************
*                                                                             *   
* Function Name  : write_enable_spi_device                                    *   
*                                                                             *
* Parameters     : None                                                       *
*                                                                             *
* Returned Value : None                                                       *   
*                                                                             *
* Description    : Enable SPI device write                                    *    
*                                                                             *   
*******************************************************************************/
void write_enable_spi_device ( void )
{
    // Code the Enable Write enable opcode into opcode register
    (*spiAOregPtr) = SPI_ENABLE_WRITE;

    // Send the One byte command   
    (*spiCSregPtr) = SPI_SEND_ONE_BYTE;

    waitTillDeviceBusy();
}

/******************************************************************************
*                                                                             *   
* Function Name  : eraseSector                                                *  
*                                                                             *
* Parameters     : sectorNo-Sector Number                                     *
*                                                                             *
* Returned Value : TRUE  -If success                                          *   
*                  FALSE -If Failed                                           *
* Description    : Erase particular sector                                    *    
*                                                                             *   
*******************************************************************************/
A_UINT8 eraseSector ( A_UINT16 sectorNo ) //Erase one sector
{
    A_UINT8 *ptr;
    A_UINT8 error=FALSE;
    A_UINT32 sectorSize=getFlashSectorSize();
    A_UINT32 startAddress=((A_UINT32)sectorNo*sectorSize);
    ptr=(A_UINT8 *)(SPI_FLASH_BASE_ADDRESS + startAddress);
    startAddress=(startAddress<<8);
    selectFlash();
    FLASHDISK_INFO( ("Erasing Sector %d...",sectorNo));
    write_enable_spi_device();
    (*spiAOregPtr) =(A_UINT32)(startAddress | SPI_SECTOR_ERASE );
    (*spiCSregPtr) = SPI_SEND_FOUR_BYTES;
    if ( waitTillDeviceBusy() == FALSE )
    {
        error = TRUE;
    }
    if ( wait_on_spi_device_status() == FALSE )
    {
        error = TRUE;
    }
    if ( error==FALSE )
    {
        FLASHDISK_INFO(("Success\n"));
    }
    else
    {
        FLASHDISK_INFO(("Failed\n"));
        return(FALSE);
    }
    return(TRUE);
}

/******************************************************************************
*                                                                             *   
* Function Name  : write_disable_spi_device                                   *   
*                                                                             *
* Parameters     : None                                                       *
*                                                                             *
* Returned Value : None                                                       *   
*                                                                             *
* Description    : Disable SPI device write                                   *    
*                                                                             *   
*******************************************************************************/

void write_disable_spi_device ( void )
{
    // Code the Enable Write enable opcode into opcode register
    (*spiAOregPtr) = SPI_DISABLE_WRITE;

    // Send the One byte command   
    (*spiCSregPtr) = SPI_SEND_ONE_BYTE;
    waitTillDeviceBusy();
}

/******************************************************************************
*                                                                             *   
* Function Name  : erase_spi_device                                           *  
*                                                                             *
* Parameters     : sectorNo-Sector Number                                     *
*                                                                             *
* Returned Value : None                                                       *   
*                                                                             *
* Description    : Erase complete Flash                                       *    
*                                                                             *   
*******************************************************************************/
void erase_spi_device ( void )
{
    // Erase the device 
    (*spiAOregPtr) = SPI_BULK_ERASE;

    // Program
    (*spiCSregPtr) = SPI_SEND_FOUR_BYTES;

    waitTillDeviceBusy();
    wait_on_spi_device_status();

}

/******************************************************************************
*                                                                             *   
* Function Name  : getDeviceInfo                                              *  
*                                                                             *
* Parameters     : None                                                       *
*                                                                             *
* Returned Value : FLASH_INFO_STRUCT -Flash Device Info                       *   
* Description    : Retrieve available Device Info                             *    
*                                                                             *   
*******************************************************************************/
FLASH_INFO_STRUCT *getDeviceInfo (void)
{
    A_UINT32 data;
    // Erase the device 
    selectFlash();
    (*spiAOregPtr) = SPI_READ_IDENTIFICATION;

    // Program
    (*spiCSregPtr) = SPI_TX_1BYTE_RX_3BYTES;

    waitTillDeviceBusy();
    data=*spiDATAregPtr;
    FD_DBG_PRINTF(FD_INFO,("MID-%X,Mem Type-%X, Size-%X\n",(data&0xFF),((data>>8)&0xFF),((data>>16)&0xFF)));
    flashInfo.DeviceMID=(data&0xFF);
    flashInfo.DeviceType=((data>>8)&0xFF);
    flashInfo.DeviceSize=((data>>16)&0xFF);
    return(&flashInfo);
}

/******************************************************************************
*                                                                             *   
* Function Name  : writeFlashWord                                             *  
*                                                                             *
* Parameters     : offset-Offset address                                      *
*                  data  -32 Bit data to be written                           *
*                                                                             *
* Returned Value : TRUE  -If success                                          *   
*                  FALSE -If Failed                                           *
*                                                                             *
* Description    : Retrieve available Device Info                             *    
*                                                                             *   
*******************************************************************************/
A_UINT8 writeFlashWord(A_UINT32 offset,A_UINT32 data)
{
    static A_UINT32 count=0;
    write_enable_spi_device();
    //Start Program
    //Setup address stage
    (*spiAOregPtr) = (A_UINT32)((offset << 8) + SPI_WRITE_DATA);

    (*spiDATAregPtr) = data;

    (*spiCSregPtr) = SPI_SEND_EIGHT_BYTES;

    waitTillDeviceBusy();

    wait_on_spi_device_status();

    (*spiDATAregPtr) = 0;

    write_disable_spi_device();

    //Setup address stage
    (*spiAOregPtr) = (((A_UINT32)offset) << 8) + SPI_READ_DATA;

    // Program
    (*spiCSregPtr) = SPI_SEND_RECEIVE_FOUR_BYTES;

    // Poll
    waitTillDeviceBusy();

    // Program
    if ( (*spiDATAregPtr) != data )
    {
        count++;
        return(FALSE);
    }
    return(TRUE);
}

/******************************************************************************
*                                                                             *   
* Function Name  : writeFlash                                                 *  
*                                                                             *
* Parameters     : offset-Offset address                                      *
*                  srcPtr-Source Buffer Pointer                               *
*                  length-Length of Source Buffer                             *
*                                                                             *
* Returned Value : TRUE  -If success                                          *   
*                  FALSE -If Failed                                           *
*                                                                             *
* Description    : Write Flash from buffer                                    *    
*                                                                             *   
*******************************************************************************/
A_UINT8 writeFlash(A_UINT32 offset,A_UINT32 *srcPtr,A_UINT32 length)
{
    A_UINT32 i;
    A_UINT32 percentLength;
    A_UINT32 *ptr=srcPtr;
    A_UINT32 curPercent=0;

    if ( srcPtr == NULL )
    {
        return(FALSE);
    }
    selectFlash();
    if ( (length%4) != 0 )
    {
        length+=(4-(length%4));
    }
    percentLength=(length/10);
    FLASHDISK_INFO(("00%% Completed"));
    for ( i=0;i<length;i+=4 )
    {
        if ( writeFlashWord((offset+i),*ptr) == FALSE )
        {
            return(FALSE);
        }
        ptr++;
        if ( (i%percentLength) < 4 )
        {
            FLASHDISK_INFO(("\r%d%%",curPercent*10));
            curPercent++;
        }
    }
    FLASHDISK_INFO(("\r                \r"));
    return(TRUE);
}
/******************************************************************************
*                                                                             *   
* Function Name  : setFlash                                                   *  
*                                                                             *
* Parameters     : offset-Offset address                                      *
*                  data  -Data to be set in all location                      *
*                  length-Length                                              *
*                                                                             *
* Returned Value : TRUE  -If success                                          *   
*                  FALSE -If Failed                                           *
*                                                                             *
* Description    : Set particular data in Flash memory range of requested size*    
*                                                                             *   
*******************************************************************************/

A_UINT8 setFlash(A_UINT32 offset,A_UINT32 data,A_UINT32 length)
{
    int i;
    A_UINT32 percentLength;
    A_UINT32 curPercent=0;
    selectFlash();
    if ( (length%4) != 0 )
    {
        length+=(4-(length%4));
    }
    percentLength=(length/100);
    FLASHDISK_INFO(("\n"));
    for ( i=0;i<length;i+=4 )
    {
        if ( writeFlashWord((offset+i),data) == FALSE )
        {
            return(FALSE);
        }
        if ( (i%percentLength) < 4 )
        {
            FLASHDISK_INFO(("\r%d%% Completed",curPercent));
            curPercent++;
        }
    }
    FLASHDISK_INFO(("\n"));
    return(TRUE);
}
#endif //BOOT_LOADER_PLUS



