/*
 *
 * encapsulation/de-capsulation functions
 *
 * Copyright  2000-2003 Atheros Communications, Inc.,  All Rights Reserved.
 */

/*
  This module implements the encapsulation/de-capsulation functions.

  Based on the WECA Interoperatability test requirement only two
  Ethernet protocols are supported:

  AppleTalk ARP - assigned number 80F3
  Novell IPX - assigned number 8137

  they are the members of the Selective Translation Table

  The implementation is based on ANSI/IEEE Std 802.1H and RFC1042

  The frame translation and forwarding flows are:

  from DSM to WM:
  a) Ethernet MAC frames -
  check the protocol type field in the MAC against the Selective
  Translation Table
  1) protocol type is not in the table: the received Ethernet
  MAC is replaced by its RFC1042 representation

  2) protocol type is in the table: the received Ethernet MAC
  is replaced by its Bridge-Tunnel Encapsulation representation

  b) Frames with ISO/IEC 8802.3 MAC -
  replaced by the appropriate outgoing MAC below the LLC

  The translation from Ethernet to Bridge-Tunnel Encapsulation Protocol

  a) the source and Destination addresses of the Ethernet MAC are used to create
  a MAC header appropriate for the LAN type

  b) The lLC header is formatted as an Unnumbered Information Unit (UI)
  command with the SNAP/SAP values for the DSAP and SSAP fields. The protocol
  identifier is formatted using the Bridge-Tunnel Ethernet OUI as octets 0, 1,
  and 2 and the Ethernet protocol type as octets 3 and 4; the protocol
  identifier forms a part of the LLC data within the ISO/IEC 8802 MAC frame

  c) The MAC data is copied to the LLC data field beyond the protocol identifier
  data. Should the copied MAC data exceed the maximum size of the ISO/IEC 8802
  MAC frame, the frame is discarded by the bridge

  d) The FCS is recalculated

  The translation from Ethernet to RFC1042 Encapsulation Protocol

  a) the source and Destination addresses of the Ethernet MAC are used to create
  a MAC header appropriate for the LAN type

  b) The LLC header is formatted as an Unnumbered Information Unit (UI)
  command with the SNAP/SAP values for the DSAP and SSAP fields. The protocol
  identifier is formatted using the RFC1042 OUI as octets 0, 1,
  and 2 and the Ethernet protocol type as octets 3 and 4; the protocol
  identifier forms a part of the LLC data within the ISO/IEC 8802 MAC frame

  c) The MAC data is copied to the LLC data field beyond the protocol identifier
  data. Should the copied MAC data exceed the maximum size of the ISO/IEC 8802
  MAC frame, the frame is discarded by the bridge

  d) The FCS is recalculated

  From WM to DSM:
  a) LLC is Bridge-Tunnel Encapsulation protocol -
  The Bridge-Tunnel MAC and LLC is replaced by the Ethernet MAC frame

  b) LLC is RFC1042 protocol -
  The protocol identifier in the SNAP header is checked against the
  Selective Translation Table
  1) protocol identifer is not in the table: the RFC1042 MAC and
  LLC is replaced by its Ethernet MAC representation
  2) protocol identifier is in the table: the received MAC is
  replaced by the the ISO/IEC8802-3 MAC below the LLC
  c) Other LLC -
  The received MAC is replaced by the ISO/IEC 8802-3 MAC below the LLC
*/



#include "wlandrv.h"
#include "wlanext.h"
#include "wlanframe.h"
#include "wlanCapDe.h"
#include "wlanSend.h"
#include "wlanchannel.h"
#include "display.h"
#include "ui.h"
#include "ccx.h"


#ifndef BUILD_AP
#include "stacserv.h"
#endif

#undef  LOCAL
#if defined(DEBUG) || defined(_DEBUG)
#define LOCAL
#undef  INLINE
#define INLINE
#else
#define LOCAL static
#endif

#if defined(VPRINT)
void 
print_hex_string(char* buf, int len)
{
    int i;

    if (len==0) { uiPrintf("<empty string>"); return; }

    for (i = 0; i < len; i++) {
        uiPrintf("%02x ", *((unsigned char *)buf + i));
	if ((i&0xf)==15) uiPrintf("\n");
    }
    if ((i&0xf))
	uiPrintf("\n");
}


void 
print_c_hex_string(char* buf, int len)
{
    int i;

    if (len==0) { uiPrintf("<>"); return; }

    for (i = 0; i < len; i++) {
	if (i<len-1)
        	uiPrintf("0x%02x,", *((unsigned char *)buf + i));
	else
        	uiPrintf("0x%02x", *((unsigned char *)buf + i));
	if ((i&0xf)==15) uiPrintf("\n");
    }
    if ((i&0xf)!=1)
	uiPrintf("\n");
}


void 
print_vera_hex_string(char* cname, char *buf, int len)
{
    int i;

    if (len==0) { uiPrintf("<>"); return; }

    for (i = 0; i < len; i++) {
	if (i<len-1)
        	uiPrintf("%s[%d]='h%02x, ", cname, i, *((unsigned char *)buf + i));
	else
        	uiPrintf("%s[%d]='h%02x", cname, i, *((unsigned char *)buf + i));
	if ((i&0x3)==3) uiPrintf("\n");
    }
    if ((i&0xf)!=1)
	uiPrintf("\n");
}

void
mpx(m, buf, len)
char *m, *buf; int len;
{
	uiPrintf("%s", m);
	print_hex_string(buf, len);
}
#endif
#ifdef VPRINT
int vprintflag = 0;
int txprintflag = 0;
int
vprint(int flag, int param)
{
    uiPrintf("vprint %d %d\n", flag, param);
    vprintflag = flag;
    return(vprintflag);
}
int
txprint(int flag, int param)
{
    txprintflag = flag;
    return(txprintflag);
}
#endif

static const CAP_CONST bridgeTunnel = ETHERNET_TUNNEL;
static const CAP_CONST vrfc1042     = RFC1042OUI;
static const CAP_CONST aironetSnap  = AIRONET_SNAP;

LOCAL INLINE A_STATUS recvFastFrame(WLAN_DEV_INFO *pDev, ATHEROS_DESC *pDesc, ATHL2P_802_3_TUNNEL_HDR *pFFHdr);

#if (defined(PCLASS) && defined(WME))
/*
 * packet classifier V0.1:
 *    if an IP frame and tos is set, then it's a dscp frame.
 *    if tos==0x28 then it's interpreted as video
 *    if tos==0x30 then it's interpreted as voip (by WME rules).
 *
 * returns: qnum in lower bits, mode flags in upper bits (e.g. use QoSDATA)
 * assumes use with new encap routines
 */
LOCAL INLINE A_UINT16
pclass(WLAN_DEV_INFO *pdevInfo, SIB_ENTRY *pDestSib, struct ip *ip)
{
    A_UINT16 qmode;

    qmode = TXQ_ID_FOR_AC1 | (1<<4);	/* default WME queue and mode */

    if (pdevInfo->staConfig.WmeEnabled == 0) {
        return (qmode);
    }



    if ((unsigned char)ip->ip_tos) {
#ifdef TEST_QOSDATA
        if (ip->ip_tos==0xbc) {         /* dscp control to enable qosDATA to a station */
            pDestSib->qosdataEnabled = 1;
            uiPrintf("-- qosdata enabled\n");
        } else
        if (ip->ip_tos==0xad) {
            pDestSib->qosdataEnabled = 0;
            uiPrintf("-- qosdata disabled\n");
        } else
#endif
        if (ip->ip_tos==0xef) {         /* Motorola dscp trigger for UPSD */
            /* if (pdevInfo->staConfig.UpsdEnabled) */
            qmode = TXQ_ID_FOR_AC3 | (3<<4) | TX_IS_UPSD;
        } else if (ip->ip_tos==0x30) {
            qmode = TXQ_ID_FOR_AC3 | (3<<4);
        } else if (ip->ip_tos==0x28) {
            qmode = TXQ_ID_FOR_AC2 | (2<<4);
        } else if (ip->ip_tos==0x18) {
            qmode = TXQ_ID_FOR_AC1 | (1<<4);
        } else if (ip->ip_tos==0x08) {
            qmode = TXQ_ID_FOR_AC0;
        }

#ifdef VPRINT
        if (vprintflag) {
            uiPrintf("-- tos(%x) q(%d) p(%d) m(%x)\n", (unsigned char)ip->ip_tos, 
                     qmode&0xf, (qmode>>4)&0xf, qmode&0xff00);
        }
#endif
    }

    return (qmode);
}

LOCAL INLINE A_UINT16
vclass(WLAN_DEV_INFO *pdevInfo, SIB_ENTRY *pDestSib, VLAN_HDR *pVlanHdr)
{
    A_UINT16 qmode = TXQ_ID_FOR_DATA;

    if (pdevInfo->staConfig.WmeEnabled) {
        switch(pVlanHdr->userPri & 0x7) {
        case 0:
        case 2:
            qmode = TXQ_ID_FOR_AC1 | (1<<4);             /* best effort */
            break;
        case 1:
        case 3:
            qmode = TXQ_ID_FOR_AC0;             /* less than best effort */
            break;
        case 4:
        case 5:
            qmode = TXQ_ID_FOR_AC2 | (2<<4);         /* video and controlled load */
            break;
        case 6:
        case 7:
            qmode = TXQ_ID_FOR_AC3 | (3<<4);         /* voice and network control */
        }
    }
#ifdef VPRINT
    if (vprintflag) {
        uiPrintf("vlan pri (%d) wme acc(%d)\n", pVlanHdr->userPri&0x7, (qmode>>4)&0xf);
    }
#endif
    return qmode;
}
#endif

/*******************************************************************************
 * Buffer management utilties - helpful for encap/decap code
 *
 */

INLINE A_UINT32
bufferHeadroom(ATHEROS_DESC *pDesc)
{
    A_UCHAR *pHead = pDesc->pOrigBufferVirtPtr;

    if (!pHead) {
        return 0;
    }
    return (pDesc->pBufferVirtPtr.byte - pHead);
}

INLINE void *
bufferPushHdr(ATHEROS_DESC *pDesc, A_UINT32 len)
{
    ASSERT(bufferHeadroom(pDesc) >= len);

    pDesc->pBufferVirtPtr.byte -= len;
    pDesc->bufferPhysPtr       -= len;
    pDesc->bufferLength        += len;
    pDesc->frameLength         += len;

    return pDesc->pBufferVirtPtr.ptr;
}

INLINE void *
bufferPopHdr(ATHEROS_DESC *pDesc, A_UINT32 len)
{
    A_UCHAR *ptr;
   
    ASSERT(pDesc->frameLength >= len);
    pDesc->frameLength -= len;

    while (pDesc->bufferLength == 0) {
        pDesc = pDesc->pNextVirtPtr;
        ASSERT(pDesc);
        if (!pDesc) {
            return NULL;
        }
    }
    ptr = pDesc->pBufferVirtPtr.byte;

    ASSERT(pDesc->bufferLength >= len);
    pDesc->pBufferVirtPtr.byte       += len;
    pDesc->bufferPhysPtr             += len;
    pDesc->bufferLength -= len;

    return ptr;
}

INLINE void *
bufferGetHdr(ATHEROS_DESC *pDesc, A_UINT32 len)
{
    while (pDesc->bufferLength == 0) {
        pDesc = pDesc->pNextVirtPtr;
        ASSERT(pDesc);
        if (!pDesc) {
            return NULL;
        }
    }

    ASSERT(pDesc->bufferLength >= len);
    if (pDesc->bufferLength < len) {
        return NULL;
    }

    return pDesc->pBufferVirtPtr.ptr;
}

#ifdef VXWORKS
#define bufferRefcount(_pDesc)              \
    ((_pDesc)->pOSDescPtr ? (_pDesc)->pOSDescPtr->clRefCnt : 1)
#else
#define bufferRefcount(_pDesc)              1
#endif

LOCAL A_STATUS
bufferPrepForEncap(WLAN_DEV_INFO *pDev, ATHEROS_DESC **ppDesc, A_UINT32 maxHdrLen)
{
    ATHEROS_DESC *pDesc      = *ppDesc;
    ATHEROS_DESC *pEnCapDesc = NULL;

    /*
     * Optimization: Can return the same buffer if it has
     * sufficient headroom and no additional outstanding
     * references; the swcrypto code requires flat buffers
     * and relies on this optimization! NDIS code ensures
     * enough headroom for the swcrypto case
     */
    if ((bufferHeadroom(pDesc) >= maxHdrLen) &&
        (bufferRefcount(pDesc) <= 1))
    {
        return A_OK;
    }

    CREATE_HEADER_BUFF_AND_DESC(pDev, (A_UINT16)maxHdrLen, &pEnCapDesc);
    if (!pEnCapDesc) {
        return A_NO_MEMORY;
    }

    /*
     * Make it so that we can push headers into it. Copy
     * over useful fields from pDesc; Link it in..
     */
    pEnCapDesc->pBufferVirtPtr.byte += maxHdrLen;
    pEnCapDesc->bufferPhysPtr       += maxHdrLen;
    pEnCapDesc->bufferLength        = 0;
    pEnCapDesc->frameLength         = pDesc->frameLength;
    A_DESC_COPY(pDesc, pEnCapDesc);
    ATH_OSPKTREF_DUP(pDesc, pEnCapDesc);

    if (pDesc->bufferLength) {
        pEnCapDesc->pNextVirtPtr = pDesc;
    } else {
        pEnCapDesc->pNextVirtPtr = pDesc->pNextVirtPtr;
        freeBuffandDesc(pDev, pDesc);
    }

    if (pEnCapDesc->pNextVirtPtr) {
        pEnCapDesc->more = 1;
    }

    *ppDesc = pEnCapDesc;

    return A_OK;
}

LOCAL void *   
bufferTruncateTail(WLAN_DEV_INFO *pDev, ATHEROS_DESC *pDesc, A_UINT32 len)
{
    A_UINT32 buffLen = 0;
    A_UINT32 newLen  = 0;
    ATHEROS_DESC *pHeadDesc = pDesc;
    
    if (pHeadDesc->frameLength <= len) {
        freeBuffandDescChain(pDev, pDesc);
        return NULL;
    }
    newLen = pHeadDesc->frameLength - len;
    while (pDesc && ((buffLen + pDesc->bufferLength) < newLen)) {
        buffLen += pDesc->bufferLength;
        pDesc    = pDesc->pNextVirtPtr;
    }
    ASSERT(pDesc);

    if (pDesc){
        pDesc->bufferLength = newLen - buffLen;
        pHeadDesc->frameLength = newLen;
        
        if (pDesc->pNextVirtPtr) {
            pDesc->more = 0; 
            pDesc->pTxFirstDesc = pHeadDesc;
            pHeadDesc->pTxLastDesc = pDesc;
            
            /* Free the rest of the descriptors and buffers */ 
            freeBuffandDescChain(pDev, pDesc->pNextVirtPtr);
            pDesc->pNextVirtPtr = NULL;
        }
    }
    return pHeadDesc;
}

/*************************************************************************
 * Transmit Queue Policy related routines
 * Figures out which queue to use to transmit the frame!
 * This is where QoS intelligence should develop!!
 */

LOCAL INLINE QUEUE_DETAILS *
txQueuePolicy(WLAN_DEV_INFO *pDev, ATHEROS_DESC *pDesc, A_UINT16 tcId)
{
    QUEUE_DETAILS *pTxQueue = &pDesc->pOpBss->txQueue;

#if (defined(WME) && defined(PCLASS))
    qmode = tcId;
    if (pDev->staConfig.WmeEnabled) {
        if (qmode & TX_IS_UPSD) {

            pTxQueue = &pDesc->pDestSibEntry->upsdQueue;
            if (!A_SEM_VALID(pTxQueue->qSem)) {
                A_STATUS r = descQInit(pTxQueue);
#ifdef DEBUGUPSD
                uiPrintf("INIT UPSD Q(%x)", (unsigned)pTxQueue);
#endif
                if (r != A_OK) {
                    uiPrintf("descQInit failed????\n");
                    pTxQueue = &pDesc->pOpBss->wmequeues[TXQ_ID_FOR_AC1];
                }
            }
        } else {
            pTxQueue = &pDesc->pOpBss->wmequeues[qmode & txQMask];
        }
    }
#endif
    if (IS_CHAN_G(pDev->staConfig.pChannel->channelFlags) &&
        pDesc->pDestSibEntry && 
        (pDesc->pDestSibEntry->wlanMode == STA_MODE_G) && 
        (pDev->staConfig.abolt & ABOLT_BURST))
    {
        pTxQueue = &pDesc->pOpBss->burstQueue;
    }

    pDesc->targetTxQId = pTxQueue->targetQueueId;

#ifdef VPRINT
    if (vprintflag) {
        uiPrintf("qmode(%x) qnum(%d)\n", qmode, pTxQueue->descQueueAttribs.priority);
    }
#endif

    return pTxQueue;
}


/*************************************************************************
 * Fast Frame Policy related routines
 *
 * Use Fast Frame threshold: i.e. try coalescing frames only after
 * ensuring at least ffThresh number of frames on the hw queue!
 * Keep it as an unsigned int so that value of 0 gets treated
 * as infinity - usage of ffThresh is structured to do that.
 */

LOCAL INLINE A_BOOL
ffIsHwQueueDeepEnough(WLAN_DEV_INFO *pDev)
{
    A_UINT32 qDepth;

    wdcTargetGetStatus(pDev->targetHandle,
                       ST_TX_QUEUE_DEPTH,
                       NULL,
                       (TARGET_STATUS_DATA *)&qDepth,
                       NULL);

    return (qDepth > (A_UINT32)(pDev->ffThresh-1));
}

/* Initialize Fast Frame support. */
void
ffInit(WLAN_DEV_INFO *pDev)
{
    pDev->ffThresh = 1;
}


LOCAL INLINE A_STATUS
txFFPolicy(WLAN_DEV_INFO *pDev, ATHEROS_DESC *pDesc, A_UINT16 etherType)
{
    if (pDev->bssDescr->bsstype != INFRASTRUCTURE_BSS) {
        return A_ENOTSUP;
    }

    if (!(pDev->staConfig.fastFramesSupport)) {
        return A_ENOTSUP;
    }

    if (pDesc->pOpBss != &pDev->baseBss) {
        return A_ENOTSUP;
    }

    if (!pDesc->pDestSibEntry) {
        return A_ENOTSUP;
    }

    if (!pDesc->pDestSibEntry->athAdvCapElement.info.useFastFrame) {
        return A_ENOTSUP;
    }

    if (pDev->localSta->serviceType == WLAN_AP_SERVICE) {
        /* check for both 3 and 4 address frame case */ 
        if (isGrp(&pDesc->pBufferVirtPtr.lanHeader->destAddr) &&
            pDesc->pDestSibEntry->pBridgeApSibEntry == NULL)
        {
            return A_ENOTSUP;
        }
    }

    /*
     * EAPOL frames have a different encryption policy - so
     * we can't combine them with other frames! Deny them
     * access to fast frame facilities!!
     */
    if (etherType == EAPOL_TYPEORLEN) {
        return A_EACCES;
    }

    return A_OK;
}


/*******************************************************************************
 * Encryption policy related utilities
 */

LOCAL INLINE A_STATUS
txEnPolicy(WLAN_DEV_INFO *pDev, ATHEROS_DESC *pDesc, A_UINT16 etherType)
{
    SIB_ENTRY *pSib = pDesc->pDestSibEntry;

    pDesc->pKey = NULL;

    /*
     * The following piece of code does not need depend on
     * being an AP or a STA i.e. if we have the data structures
     * set correctly, and the configuration managed at the
     * right place(s). Since that is unfortunately not the
     * case right now, a run time conditional for it being
     * AP or STA is hypocritical; it is consequently compile
     * time conditional - as its a compile time decision as
     * to what data structure captures the encryption state
     */

#ifdef BUILD_AP
    if (!pSib) {
        if (etherType == EAPOL_TYPEORLEN) {
            /* Mr Security says chuck them */
            return A_ERROR;
        }
        if (pDev->staConfig.privacyInvoked) {
            pDesc->pKey = &pDev->keyTable[pDev->staConfig.defaultKey];
            ASSERT(pDesc->pKey->keyLength);
        }
        return A_OK;
    }
#endif

    if (pDev->staConfig.wpaEnabled) {
        WLAN_PRIV_RECORD *pKey = NULL;
#ifdef BUILD_AP
        pKey = pSib->pPriv;
#else
        if (pDev->staConfig.defaultKey != KEYINDEX_INVALID) {
            pKey = &pDev->keyTable[pDev->staConfig.defaultKey];
        }
#endif
        if (pKey && pKey->keyLength) {
            pDesc->pKey = pKey;
            return A_OK;
        }
        if (etherType == EAPOL_TYPEORLEN) {
            return A_OK;
        }
        return A_ERROR;
    }

    if (etherType == EAPOL_TYPEORLEN) {
        return A_OK;
    }

    if (pDev->localSta->serviceType == WLAN_AP_SERVICE) {
        if (pDev->staConfig.privacyInvoked && pSib->privacyInvoked) {
            pDesc->pKey = pSib->pPriv;
            ASSERT(pDesc->pKey);
            return A_OK;
        }
    } else {
        if (pDev->staConfig.wpaEnabled || pDev->staConfig.privacyInvoked) {
            if (pDev->staConfig.defaultKey == KEYINDEX_INVALID) {
                return A_ERROR;
            }
            pDesc->pKey = &pDev->keyTable[pDev->staConfig.defaultKey];
            if (pDesc->pKey && pDesc->pKey->keyLength) {
                return A_OK;
            }
            return A_ERROR;
        }
    }

    return A_OK;
}

LOCAL INLINE A_BOOL
txEnUseMMH(WLAN_DEV_INFO *pDev, ATHEROS_DESC *pDesc)
{
    LLC_SNAP_HEADER *pLLCHdr;
    WLAN_PRIV_RECORD *pKey = pDesc->pKey;

    ASSERT(pKey);

    if (pKey->keyFlags & PRIV_CKIP_MIC) {
        pLLCHdr = bufferGetHdr(pDesc, sizeof(LLC_SNAP_HEADER));
        /* Don't use MMH on DDP frames */
        if (A_BCOMP(pLLCHdr, &aironetSnap, CAP_TYPE_SIZE) == 0 &&
            pLLCHdr->etherType == be2cpu16(DDP_SNAP)) {
            return FALSE;
        }
        return TRUE;
    }
    return FALSE;
}

LOCAL INLINE A_STATUS
rxEnPolicy(WLAN_DEV_INFO *pDev, ATHEROS_DESC *pDesc, A_UINT16 etherType,
    A_UINT16 encrypted, WLAN_MACADDR *pDestAddr)
{
    SIB_ENTRY *pSib = pDesc->pSrcSibEntry;

    /*
     * Is it OK for EAPOL frames to be encrypted also???
     * or should be enforce non-encryption for them...
     */
    if (etherType == EAPOL_TYPEORLEN) {
#ifdef BUILD_AP
        /*
         *  The AP shall not forward any EAPOL frames.
         *  The current platform architectural models do not permit
         *  an AP to passively forward such frames; the AP 
         *  must be a participant in any EAPOL authentication it sees.
         */
        if (A_MACADDR_COMP(pDestAddr, &pDesc->pOpBss->bssId)) {
#ifdef DEBUG
            uiPrintf("Misdirected 802.1x frame to ");
            printMacAddress(*pDestAddr);
            uiPrintf("\n");
#endif
            return A_EACCES;
        }
#endif
        return A_OK;
    }

    /*
     * could/should we successfully receive an encrypted frame
     * if encryption was not enabled???
     */
    if (encrypted) {
        return A_OK;
    }

    ASSERT(pSib);

#ifdef BUILD_AP
    if (!pSib->privacyInvoked) {
        return A_OK;
    }
    pSib->stats.RxDiscardFrames++;
#else
    if (!pDev->staConfig.privacyInvoked ||
        ccxMixedPrivacyPolicyCheck(pDev) == A_OK)
    {
        return A_OK;
    }
    pDev->localSta->stats.RcvWEPExcludedCount++;
    pSib->stats.RcvWEPExcludedCount++;
#endif

    return A_EACCES;
}

/*******************************************************************************
 * CKIP does parts of the crypto underneath the 802.11 layer using
 * a special etherType. The transmit side routine expects an 802.2
 * encapsulated frame and works by overwriting the original 802.2
 * CAP_CONST by the MMH 802.2 hdr i.e. CKIP LLC/SNAP and CKIP MIC.
 * The receive side routine does the reverse - by creating a RFC1042
 * CAP_CONST
 */

LOCAL INLINE A_STATUS
xformDot2ToMMH(WLAN_DEV_INFO *pDev, ATHEROS_DESC *pDesc)
{
    LLC_SNAP_HEADER *pLLCHdr;

    ASSERT(pDev->localSta->swMicEnabled);

    pLLCHdr = bufferPushHdr(pDesc, ETHER_TYPE_SIZE + CKIP_MIC_FIELD_SIZE);
    A_BCOPY(&aironetSnap, pLLCHdr, sizeof(CAP_CONST));
    pLLCHdr->etherType = cpu2be16(CKIP_TYPE_FIELD);

    A_MEM_ZERO((void *)(pLLCHdr + 1), CKIP_MIC_FIELD_SIZE);

    return A_OK;
}

/*
 *  Transform a frame to 802.2 encapsulation, taking the Cisco MMH
 *  encapsulation into account.  To do so, we 
 *  check its MIC value, then overwrite its SNAP/LLC with an RFC1042
 *  format header.  Other encapsulations are just passed thru unchanged,
 *  but everybody gets de-MIC'd here.
 */  
LOCAL INLINE A_STATUS
xformMicToDot2(WLAN_DEV_INFO* pDev, ATHEROS_DESC *pDesc,
               WLAN_MACADDR *src, WLAN_MACADDR *dest, void *qosInfo)
{
    LLC_SNAP_HEADER *pLLCHdr;

    pLLCHdr = bufferGetHdr(pDesc, sizeof(LLC_SNAP_HEADER));

    /* Check to see if we need MMH check */
    if ((pDesc->pKey->keyFlags & PRIV_CKIP_MIC) &&
        A_BCOMP(pLLCHdr, &aironetSnap, CAP_TYPE_SIZE) == 0)
    {
        /* Only do header rewrite for MMH encaps, not DDP */
        if (pLLCHdr->etherType == be2cpu16(CKIP_TYPE_FIELD)) {

            if (swDemic(pDev, pDesc, src, dest, qosInfo) != A_OK) {
                return A_ERROR;
            }

            /*
             *  We're going to overwrite the MMH header with a garden-variety
             *  802.2 header now.  We effectively keep the LLC/SNAP space,
             *  but discard the MMH MIC, the MMH sequence number, and the MMH 
             *  ethertype (0x0002).
             *  Note the ethertype is already included in the defn of the 
             *  LLC_SNAP_HEADER, so we're just avoiding a double-allocation.
             */
            bufferPopHdr(pDesc, ETHER_TYPE_SIZE + CKIP_MIC_FIELD_SIZE);
            pLLCHdr = bufferGetHdr(pDesc, sizeof(LLC_SNAP_HEADER));
            A_BCOPY(&vrfc1042, pLLCHdr, sizeof(CAP_CONST));
        }
        return A_OK;
    }

    /* All other non-MMH protocols get run thru the general demic */
    if (swDemic(pDev, pDesc, src, dest, qosInfo) != A_OK) {
        return A_ERROR;
    }

    return A_OK;
}

/*******************************************************************************
 * Its a valid 802.11 data frame; send it on its way
 */

LOCAL INLINE A_STATUS
xmitDot11Frame(WLAN_DEV_INFO *pDev, QUEUE_DETAILS *pTxQueue,
               ATHEROS_DESC *pDesc)
{
    A_STATUS status;

#if defined(DEBUG)
    ipDecode(pDev, pDesc, FALSE);
#endif

    /* Apply MIC function at MSDU level before fragmentation */
    if (pDesc->pKey) {
        status = swEnmic(pDev, pDesc->pKey, &pDesc);
        if (status != A_OK) {
            freeBuffandDescChain(pDev, pDesc);
            return status;
        }
    }

    return wlanFrameTransmit(pDev, pTxQueue, pDesc);
}


/*******************************************************************************
 * Convert a 802.3/2 (possible Fast) Frame to a 802.11 frame
 * while taking care of encryption policy and send it on its
 * way
 * Not INLINE since its called from a couple of places
 */

LOCAL A_STATUS
xmitDot3FastFrame(WLAN_DEV_INFO *pDev, QUEUE_DETAILS *pTxQueue,
                  ATHEROS_DESC *pDesc)
{
    SIB_ENTRY         *pSib  = pDesc->pDestSibEntry;
    LAN_FRAME_HEADER  *pLanHdr;
    WLAN_FRAME_HEADER *pWlanHdr;
    LLC_SNAP_HEADER   *pLLCHdr = NULL;
    WLAN_MACADDR       srcAddr, destAddr;
    QOS_CNTL          *pQos = 0;
    A_STATUS           status;
    A_UINT32           headersize;
    A_UINT32           pad;
    LAN_LLC_FRAME_HEADER *pLanLLCHdr= NULL;
    A_BOOL             isSNAPLANHdr;

    pLanHdr = bufferGetHdr(pDesc, sizeof(LAN_FRAME_HEADER));

    ASSERT(!IS_ETHERTYPE(be2cpu16(pLanHdr->lanTypeOrLen)));
    pLanLLCHdr = bufferGetHdr(pDesc, sizeof(LAN_LLC_FRAME_HEADER));
    isSNAPLANHdr = (LLC_SAP_SNAP == pLanLLCHdr->dsap);

    /*
     * Cache the src and dst addresses and pull out
     * the ethernet header
     */
    A_MACADDR_COPY(&pLanHdr->srcAddr, &srcAddr);
    A_MACADDR_COPY(&pLanHdr->destAddr, &destAddr);
    bufferPopHdr(pDesc, sizeof(LAN_FRAME_HEADER));

    /* enforce the encryption policy and lookup the key */
    if(isSNAPLANHdr) {
        pLLCHdr = bufferGetHdr(pDesc, sizeof(LLC_SNAP_HEADER));
    }

    status = txEnPolicy(pDev, pDesc, 
                    (pLLCHdr==NULL) ? 0 : be2cpu16(pLLCHdr->etherType));
    if (status != A_OK) {
        freeBuffandDescChain(pDev, pDesc);
        return status;
    }

    /*
     * A conservative estimate of the max encap header size
     * needed here would be 32 bytes for WLAN - assume four
     * addr bridge mode with QoS (or just word aligned); max
     * 8 bytes of IV, and another 10 bytes of overhead for
     * CKIP encapsulation - all rounded up
     */
#define MAX_ENCAP_HDRLEN 52
    status = bufferPrepForEncap(pDev, &pDesc, MAX_ENCAP_HDRLEN);
    if (status != A_OK) {
        freeBuffandDescChain(pDev, pDesc);
        return status;
    }

    if (pDesc->pKey) {
        A_UINT32 ivLen = WEP_IV_FIELD_SIZE;

        if (txEnUseMMH(pDev, pDesc)) {
            status = xformDot2ToMMH(pDev, pDesc);
            if (status != A_OK) {
                freeBuffandDescChain(pDev, pDesc);
                return status;
            }
        }

        if ((pDesc->pKey->keyType == PRIV_KEY_TYPE_AES_CCM) ||
            (pDesc->pKey->keyType == PRIV_KEY_TYPE_TKIP) ||
            (pDesc->pKey->keyType == PRIV_KEY_TYPE_TKIP_SW))
        {
            ivLen += EXT_IV_FIELD_SIZE;
        }
        bufferPushHdr(pDesc, ivLen);

        A_MEM_ZERO(pDesc->pBufferVirtPtr.ptr, ivLen);
        if (ivLen > WEP_IV_FIELD_SIZE) {
            pDesc->pBufferVirtPtr.word[0] |= EXT_IV_BIT;
        }
    }

    /* Clear isEthFrm field */
    pDesc->isEthFrm = FALSE;

    /*
     * Set default header size.
     * Extend it for header syntax options.
     */
    headersize = sizeof(WLAN_DATA_MAC_HEADER3);
    if (pSib) {
        if (pSib->pBridgeApSibEntry) {
            headersize += sizeof(WLAN_MACADDR);
        }
        if (pSib->qosdataEnabled) {
            headersize += sizeof(QOS_CNTL);
        }
    }

    /*
     * determine if alignment pad is needed.
     */
    if ((pad = A_ROUNDUP_PAD(headersize, sizeof(A_UINT32)))) {

        bufferPushHdr(pDesc, pad);
        pDesc->frameLength -= pad;
    }

    if (pSib && pSib->qosdataEnabled) {
        pQos = bufferPushHdr(pDesc, sizeof(QOS_CNTL));
        pQos->txop = 0;
        pQos->ackpolicy = 0;
        pQos->esop = 0;
        pQos->reserved = 0;
        pQos->tid = (A_UINT8)pTxQueue->descQueueAttribs.priority;
        pDesc->targetTxQId = pTxQueue->targetQueueId;
        pDesc->qosMode  |= TX_ISQOSDATA;
    }

    if (pSib && pSib->pBridgeApSibEntry) {
        WLAN_MACADDR *pAddr4 = bufferPushHdr(pDesc, sizeof(WLAN_MACADDR));
        A_MACADDR_COPY(&srcAddr, pAddr4);
    }

    pWlanHdr = bufferPushHdr(pDesc, sizeof(WLAN_DATA_MAC_HEADER3));
    *(A_UINT16 *)(&pWlanHdr->frameControl) = 0;
    pWlanHdr->frameControl.protoVer = PROTOCOL_VER_0;
    pWlanHdr->frameControl.fType    = FRAME_DATA;
    pWlanHdr->frameControl.fSubtype = (pQos)? SUBT_QOS:SUBT_DATA;
    if (pDesc->pKey) {
        pWlanHdr->frameControl.wep = 1;
    }

    if (pSib && pSib->pBridgeApSibEntry) {
        pWlanHdr->frameControl.ToDS   = 1;
        pWlanHdr->frameControl.FromDS = 1;

        A_MACADDR_COPY(&pSib->pBridgeApSibEntry->macAddr, &pWlanHdr->address1);
        A_MACADDR_COPY(&pDev->localSta->macAddr, &pWlanHdr->address2);
        A_MACADDR_COPY(&destAddr, &pWlanHdr->address3);
    } else if (pDev->localSta->serviceType == WLAN_AP_SERVICE) {
        pWlanHdr->frameControl.ToDS   = 0;
        pWlanHdr->frameControl.FromDS = 1;

        A_MACADDR_COPY(&destAddr, &pWlanHdr->address1);
        A_MACADDR_COPY(&pDesc->pOpBss->bssId, &pWlanHdr->address2);
        A_MACADDR_COPY(&srcAddr, &pWlanHdr->address3);
    } else if (pDev->bssDescr->bsstype == INFRASTRUCTURE_BSS) {
        pWlanHdr->frameControl.ToDS   = 1;
        pWlanHdr->frameControl.FromDS = 0;

        A_MACADDR_COPY(&pDesc->pOpBss->bssId, &pWlanHdr->address1);
        A_MACADDR_COPY(&srcAddr, &pWlanHdr->address2);
        A_MACADDR_COPY(&destAddr, &pWlanHdr->address3);
    } else {
        ASSERT(pDev->bssDescr->bsstype == INDEPENDENT_BSS);

        pWlanHdr->frameControl.ToDS   = 0;
        pWlanHdr->frameControl.FromDS = 0;

        A_MACADDR_COPY(&destAddr, &pWlanHdr->address1);
        A_MACADDR_COPY(&srcAddr, &pWlanHdr->address2);
        A_MACADDR_COPY(&pDesc->pOpBss->bssId, &pWlanHdr->address3);
    }

    pWlanHdr->seqControl.data = 0;

    return xmitDot11Frame(pDev, pTxQueue, pDesc);
}

static unsigned int ffTxCount = 0; /* debug */

/*******************************************************************************
 * Fast frame encapsulation - requires Dot3 frames underneath!
 * Uses ATHL2P_802_3_TUNNEL protocol
 * This routine will munge an incoming Dot3 frame into a fast
 * frame (if possible) and send it on its way
 */

static CAP_CONST atherosCapConst  = ATHEROS_CAP_CONST;

LOCAL INLINE A_STATUS
xmitDot3Frame(WLAN_DEV_INFO *pDev, ATHEROS_DESC *pDesc)
{
    SIB_ENTRY               *pSib    = pDesc->pDestSibEntry;
    ATHEROS_DESC            *pFFDesc = NULL;
    A_UINT16                 tcId    = 0;
    QUEUE_DETAILS           *pTxQueue;
    ATHL2P_802_3_TUNNEL_HDR *pFFHdr;
    LAN_LLC_FRAME_HEADER    *pLanLLCHdr;
    LLC_SNAP_HEADER         *pLLCHdr = NULL;
    LAN_FRAME_HEADER        *pLanHdr;
    A_UINT32                 pad;
    A_STATUS                 status;

    /* discard runt frames */
    if (pDesc->frameLength < sizeof(LAN_LLC_FRAME_HEADER)) {
        freeBuffandDescChain(pDev, pDesc);
        return A_EPROTO;
    }
    pLanLLCHdr = bufferGetHdr(pDesc, sizeof(LAN_LLC_FRAME_HEADER));
#if (defined(PCLASS) && defined(WME))
    pLLCHdr = NULL;

    if(pLanLLCHdr->dsap == LLC_SAP_SNAP) {
        pLanHdr = bufferGetHdr(pDesc, 
                    sizeof(LAN_FRAME_HEADER) + sizeof(LLC_SNAP_HEADER));
        pLLCHdr = (LLC_SNAP_HEADER* ) (pLanHdr + 1);

        if(pLLCHdr->etherType == cpu2be16(ETHERTYPE_802_1Q))
        {
            /*
             * Header manipulation is hacky here - but should work for
             * all practical cases
             */
            bufferPopHdr(pDesc, 
                         sizeof(LLC_SNAP_HEADER) + sizeof(LAN_FRAME_HEADER));
            tcId = vclass(pDev, pDesc->pDestSibEntry, 
                          bufferGetHdr(pDesc, sizeof(VLAN_HDR)));
            A_BCOPY(pLLCHdr,
            bufferPushHdr(pDesc, 
                sizeof(LLC_SNAP_HEADER) + sizeof(LAN_FRAME_HEADER) - sizeof(VLAN_HDR)),
                sizeof(LLC_SNAP_HEADER) + sizeof(LAN_FRAME_HEADER) - ETHER_TYPE_SIZE);

        } else if (pLLCHdr->etherType == cpu2be16(IPV4)) {
            bufferPopHdr(pDesc, sizeof(LLC_SNAP_HEADER) + sizeof(LAN_FRAME_HEADER));
            tcId = spectraclass(pDev, bufferGetHdr(pDesc, sizeof(struct ip)));
            bufferPushHdr(pDesc, sizeof(LLC_SNAP_HEADER) + sizeof(LAN_FRAME_HEADER));
        }
        else {
            tcId = TXQ_ID_FOR_AC1; /* Put it in Best effort Q; */
        }
    }
#endif

    pTxQueue = txQueuePolicy(pDev, pDesc, tcId);
    A_SEM_LOCK(pTxQueue->qSem, WAIT_FOREVER);

    if (pSib) {
        pFFDesc = getPktFromQueue(pTxQueue->pStagingQueue, pSib->pTxFFDesc);
        ASSERT(pFFDesc == pSib->pTxFFDesc);
        pSib->pTxFFDesc = NULL;
    }

    status = txFFPolicy(pDev, pDesc, pLLCHdr ? be2cpu16(pLLCHdr->etherType) : 0 );
    switch (status) {
    case A_OK:
        break;
    case A_EACCES:
        if (pFFDesc) {
            xmitDot3FastFrame(pDev, pTxQueue, pFFDesc);
            pFFDesc = NULL;
        }
        /* fall through */
    case A_ENOTSUP:
        ASSERT(!pFFDesc);
        status = xmitDot3FastFrame(pDev, pTxQueue, pDesc);
        A_SEM_UNLOCK(pTxQueue->qSem);
        return status;
    default:
        uiPrintf("xmitDot3Frame: Unknown error code from txFFPolicy %d\n", status);
        A_SEM_UNLOCK(pTxQueue->qSem);
        freeBuffandDescChain(pDev, pDesc);
        return status;
    }

    if (!pFFDesc) {
#if 0
        /*
         * Opportunistically get the latest state of the HW queue!
         * it might have been a while since we've reaped it! it could
         * end up draining some other SIB's staged frame - but that's
         * OK - its better for this SIB to take the latency hit in
         * that case anyway!
         * Later we may want to do tx reaping at a higher priority
         * so that we don't have to do this here!
         */
        processTxQueueLocked(pDev, pTxQueue);
#endif

        /*
         * we may want to consider if this frame is going to
         * end up in the PS or swretry queue anyway
         */
        if (!ffIsHwQueueDeepEnough(pDev)) {
            status = xmitDot3FastFrame(pDev, pTxQueue, pDesc);
            A_SEM_UNLOCK(pTxQueue->qSem);
            return status;
        }

        A_TX_DESC_CHAIN_FORMAT(pDesc);
        pDesc->isEthFrm  = TRUE;
        pDesc->timeStamp = A_MS_TICKGET();

        descQPushTail(pTxQueue->pStagingQueue, pDesc, pDesc->pTxLastDesc);
        A_ATOMIC_INC(&pTxQueue->pStagingQueue->qFrameCount);
        pSib->pTxFFDesc = pDesc;
        A_SEM_UNLOCK(pTxQueue->qSem);

        return A_OK;
    }

    ffTxCount++; /* debug */

    pad = A_ROUNDUP_PAD(pFFDesc->frameLength, sizeof(A_UINT32));
    pFFDesc->frameLength += (pDesc->frameLength + pad);
    pFFDesc->pTxLastDesc->bufferLength += pad;
    pFFDesc->pTxLastDesc->more = 1;
    pFFDesc->pTxLastDesc->pNextVirtPtr = pDesc;

    /*
     * A conservative estimate of the max encap header size
     * needed here would be 32 bytes for WLAN - assume four
     * addr bridge mode with QoS (or just word aligned); max
     * 8 bytes of IV, and another 10 bytes of overhead for
     * MMH encapsulation; the header we're going to construct
     * is LLC/SNAP + 4 bytes of Atheros Layer 2 header - all
     * rounded up followed by the pad
     */
    pad = A_ROUNDUP_PAD(sizeof(ATHL2P_802_3_TUNNEL_HDR) + 2, sizeof(A_UINT32));
    status = bufferPrepForEncap(pDev, &pFFDesc, 64 + pad);
    if (status != A_OK) {
        A_SEM_UNLOCK(pTxQueue->qSem);
        freeBuffandDescChain(pDev, pFFDesc);
        return status;
    }

    pFFHdr = bufferPushHdr(pFFDesc, sizeof(ATHL2P_802_3_TUNNEL_HDR) + pad);
    pFFHdr->proto       = ATHL2P_802_3_TUNNEL;
    pFFHdr->frameType   = FRAME_TYPE_FAST_FRAME;
    pFFHdr->optHdrLen32 = 0;
    pFFHdr->offset      = 0;
    pFFHdr->seqNum      = 0;
    CPU2BE_ATHL2P(pFFHdr);

    pLLCHdr = bufferPushHdr(pFFDesc, sizeof(LLC_SNAP_HEADER));
    A_BCOPY(&atherosCapConst, pLLCHdr, sizeof(CAP_CONST));
    pLLCHdr->etherType = cpu2be16(ETHERTYPE_ATHL2P);

    pLanHdr = bufferPushHdr(pFFDesc, sizeof(LAN_FRAME_HEADER));
    A_MACADDR_COPY(&pSib->macAddr, &pLanHdr->destAddr);
    A_MACADDR_COPY(&pDev->staConfig.macAddr, &pLanHdr->srcAddr);
    /*
     * set pseudo length parameter - its going to be discarded! real length
     * i.e. (pFFDesc->frameLength - sizeof(LAN_FRAME_HEADER))
     * is possibly greater than 0x600
     */
    pLanHdr->lanTypeOrLen = 0;

#if defined(DEBUG) || defined(_DEBUG)
    ++pSib->stats.RetryBins[9];
    ++pSib->stats.TxDmaUnderrun;
    ++pDev->localSta->stats.RetryBins[9];
    ++pDev->localSta->stats.TxDmaUnderrun;
#endif
    pFFDesc->ffFlag = 1;

    status = xmitDot3FastFrame(pDev, pTxQueue, pFFDesc);
    A_SEM_UNLOCK(pTxQueue->qSem);

    return status;
}


/*******************************************************************************
 * Most arriving ethernet frames are DIX encapsulated (legacy rules!).
 * IEEE802.11 needs the IEEE802.2 encapsulation underneath to get the
 * etherType information across.
 * Some vague RFCs (comments at beginning of this file) define the
 * transformation of a DIX encapsulated ethernet frame to a 
 * 802.3/802.2 encapsulated frame
 * so that the 802.11 layer can deal with them appropriately
 * This routine will munge it an arriving DIX frame into a Dot3 frame
 * and send it on its way
 */

LOCAL INLINE A_STATUS
xmitDixFrame(WLAN_DEV_INFO *pDev, ATHEROS_DESC *pDesc)
{
    LAN_FRAME_HEADER *pLanHdr;
    LAN_FRAME_HEADER  cachedLanHdr;
    LLC_SNAP_HEADER  *pLLCHdr;
    const CAP_CONST  *pCapConst;
    A_UINT16          typeOrLen;
    A_STATUS          status;

    pLanHdr   = bufferGetHdr(pDesc, sizeof(LAN_FRAME_HEADER));
    if (pLanHdr == NULL) {
        freeBuffandDescChain(pDev, pDesc);
        return A_ERROR;
    }
    typeOrLen = be2cpu16(pLanHdr->lanTypeOrLen);

    /*
     * Nothing to do if got the length field in the ethernet
     * header! i.e. already using 802.3/802.2 encapsulation
     */
    if (!IS_ETHERTYPE(typeOrLen)) {
#ifdef VPRINT
        if (vprintflag>1) {
            uiPrintf(".3 frame\n");
        }
#endif
        
        ASSERT((pDesc->frameLength - sizeof(LAN_FRAME_HEADER)) 
                                                            >= typeOrLen);
        if ((pDesc = bufferTruncateTail(pDev, pDesc, 
                                        pDesc->frameLength - 
                                        sizeof(LAN_FRAME_HEADER) -
                                        typeOrLen)) != NULL)
        {
            return xmitDot3Frame(pDev, pDesc);
        } else {
            return A_EPROTO;
        }
    }

    /*
     * Pull out and cache the ethernet header - the intent
     * is to leave the data in these buffers untouched incase
     * there are other references to it - the LLC/SNAP header
     * will be created in a prepended buffer in that case
     */
#ifdef VPRINT
    if (vprintflag>1) {
        mpx("dix ", pLanHdr, sizeof(LAN_FRAME_HEADER));
        uiPrintf("VLAN %x %x\n", typeOrLen, (unsigned)pLanHdr);
    }
#endif
    A_BCOPY(pLanHdr, &cachedLanHdr, sizeof(LAN_FRAME_HEADER) - ETHER_TYPE_SIZE);
    bufferPopHdr(pDesc, sizeof(LAN_FRAME_HEADER));

    /*
     * use a conservative estimate equal to our transmit
     * header buffer size; it is supposed to accomodate
     * the LLC/SNAP, MMH encapsulation, crypto IV, and
     * the WLAN mac header!
     */
    status = bufferPrepForEncap(pDev, &pDesc, AR_HEADER_SIZE);
    if (status != A_OK) {
        freeBuffandDescChain(pDev, pDesc);
        return status;
    }

    /* Create the LLC/SNAP header depending on the etherType */
    pLLCHdr = bufferPushHdr(pDesc, sizeof(LLC_SNAP_HEADER));
    pCapConst = ((typeOrLen == APPLE_TALK_ARP) || (typeOrLen == NOVELL_IPX))
              ? &bridgeTunnel
              : &vrfc1042;
    A_BCOPY(pCapConst, pLLCHdr, sizeof(CAP_CONST));
    pLLCHdr->etherType = cpu2be16(typeOrLen);
    cachedLanHdr.lanTypeOrLen = cpu2be16(pDesc->frameLength);

    /* Create the (temporary) ethernet hdr */
    pLanHdr = bufferPushHdr(pDesc, sizeof(LAN_FRAME_HEADER));
    A_BCOPY(&cachedLanHdr, pLanHdr, sizeof(LAN_FRAME_HEADER));

    return xmitDot3Frame(pDev, pDesc);
}


/*************************************************************************
 * wlanDataFrameTransmit - entry point for transmit of a single data
 *   frame; called by the AP or STA code when the incoming buffers
 *   from the OS have been converted to ATHEROS_DESC representation.
 *   The sib corresponding to the destination should have been looked
 *   up and validated for a connection/association
 *   The frame still looks like a 802.3/Ethernet frame
 * Is a part of the encapsulation API since its still dealing with
 *   802.3 type of frames; the new generation stuff will directly deal
 *   with native 802.11 frames and would therefore directly call the
 *   core API in wlanSend.c
 *
 * RETURNS: error status
 *   Discards the frame on error
 */

A_STATUS
wlanDataFrameTransmit(WLAN_DEV_INFO *pDev, ATHEROS_DESC *pDesc)
{
    A_STATUS status;

    /*
     *  Drop all outgoing frames if lower layers have decided that we should 
     *  play the countermeasures game.
     */
    if (pDev->trafficBlockTimestamp) {
#ifndef BUILD_AP
        /* XXX this will probably choke off the 2nd EAPOL-Key msg */
        pDev->bssDescr->apState |= BAD_AP;
        pDev->bssDescr->badAPTimeout = A_MS_TICKGET() +
            MIC_COUNTERMEASURE_TIMEOUT;
        cservStopBss(pDev, DISCONNECT_NOW, DONT_REMOVE_BSS, SEND_DEAUTH);
        cservChooseApAndSwitch(pDev, 0);
        pDev->micErrorM2Sent = FALSE;
#endif
        /* XXX Careful with rollover! */
        if (A_MS_TICKGET() - pDev->trafficBlockTimestamp <
            MIC_COUNTERMEASURE_TIMEOUT)
        {
#ifdef DEBUG
            uiPrintf("MIC Countermeasure Tx block: Curtime %d ",
                     A_MS_TICKGET());
            uiPrintf("Will unblock at %d ms\n",
                     pDev->trafficBlockTimestamp + MIC_COUNTERMEASURE_TIMEOUT);
#endif
            freeBuffandDescChain(pDev, pDesc);
            return A_ERROR;
        }
        pDev->trafficBlockTimestamp = 0;
    }

    /* no WLAN xmit status indication is needed for data frame */
    pDesc->bIndicateXmitStatus = FALSE;

    status = xmitDixFrame(pDev, pDesc);

    return status;
}


/*************************************************************************
 * Fast frame queue drain related transmit entry point
 */

A_STATUS
txFFDrain(WLAN_DEV_INFO *pDev, QUEUE_DETAILS *pTxQueue)
{
    ATHEROS_DESC *pDesc;
    A_UINT32     age;

    if (!pTxQueue->pStagingQueue) {
        return A_OK;
    }

    while (1) {
#ifndef BUILD_AP
        A_SEM_LOCK(pTxQueue->pStagingQueue->qSem, WAIT_FOREVER);
#endif
        pDesc = pTxQueue->pStagingQueue->pDescQueueHead;
#ifndef BUILD_AP
        A_SEM_UNLOCK(pTxQueue->pStagingQueue->qSem);
#endif
        if(!pDesc) {
            break;
        }

        age = A_MS_TICKGET() - pDesc->timeStamp;

        ASSERT(pDev->staConfig.fastFramesSupport);

        if ((age < 10) && ffIsHwQueueDeepEnough(pDev)) {
            return A_OK;
        }

#ifndef BUILD_AP
        A_SEM_LOCK(pTxQueue->pStagingQueue->qSem, WAIT_FOREVER);
#endif
        pTxQueue->pStagingQueue->pDescQueueHead =
            pDesc->pTxLastDesc->pNextVirtPtr;
        pDesc->pTxLastDesc->pNextVirtPtr = NULL;
        if (!pTxQueue->pStagingQueue->pDescQueueHead) {
            pTxQueue->pStagingQueue->pDescQueueTail = NULL;
        }
#ifdef BUILD_AP
        --pTxQueue->pStagingQueue->qFrameCount;
#else
        NdisInterlockedDecrement(&pTxQueue->pStagingQueue->qFrameCount);
#endif

#ifndef BUILD_AP
        A_SEM_UNLOCK(pTxQueue->pStagingQueue->qSem);
#endif
        if (pDesc->pDestSibEntry->pTxFFDesc == pDesc) {
            pDesc->pDestSibEntry->pTxFFDesc = NULL;
            return xmitDot3FastFrame(pDev, pTxQueue, pDesc);
        }

        freeBuffandDescChain(pDev, pDesc);
    }

    return A_OK;
}

/*************************************************************************
 * Flush Fast frame queue 
 */
void
txFfFlush(WLAN_DEV_INFO *pDev)
{
    UINT32 tempFFThresh;            
            
    tempFFThresh = pDev->ffThresh;
    pDev->ffThresh = 0;            

    txFFDrain(pDev, &pDev->baseBss.txQueue);
    txFFDrain(pDev, &pDev->baseBss.burstQueue);            

    pDev->ffThresh = tempFFThresh;   
}

/*************************************************************************
 * recvDot3Frame - converts a Dot3 encapsulated frame to a DIX frame,
 *   if needed, and passes it up the stack
 *
 * RETURNS: error status
 *   Discards the frame on error
 */

LOCAL INLINE A_STATUS
recvDot3Frame(WLAN_DEV_INFO *pDev, ATHEROS_DESC *pDesc)
{
    LAN_FRAME_HEADER *pLanHdr;
    LLC_SNAP_HEADER  *pLLCHdr;
    A_BOOL            doDix;
    LAN_LLC_FRAME_HEADER* pLanLLCHdr = 
                        bufferGetHdr(pDesc,sizeof(LAN_LLC_FRAME_HEADER));

    if(LLC_SAP_SNAP == pLanLLCHdr->dsap) {
        pLanHdr = bufferGetHdr(pDesc, sizeof(LAN_FRAME_HEADER) +
                           sizeof(LLC_SNAP_HEADER));
        if (pLanHdr == NULL) {
            freeBuffandDescChain(pDev, pDesc);
            return A_EPROTO;
        }

        pLLCHdr = (LLC_SNAP_HEADER *)(pLanHdr + 1);

        doDix   = (A_BCOMP(pLLCHdr, &bridgeTunnel, sizeof(CAP_CONST)) == 0);
        if ((!doDix) && (A_BCOMP(pLLCHdr, &vrfc1042, sizeof(CAP_CONST)) == 0)) {
            doDix = (pLLCHdr->etherType != cpu2be16(APPLE_TALK_ARP)) &&
                    (pLLCHdr->etherType != cpu2be16(NOVELL_IPX));
        }

        /* the following transformation is hacky - but it works */
        if (doDix) {
            LAN_FRAME_HEADER *pNewHdr;

            bufferPopHdr(pDesc, sizeof(LLC_SNAP_HEADER));
            pNewHdr = bufferGetHdr(pDesc, sizeof(LAN_FRAME_HEADER));

            A_MACADDR_COPY(&pLanHdr->srcAddr, &pNewHdr->srcAddr);
            A_MACADDR_COPY(&pLanHdr->destAddr, &pNewHdr->destAddr);
        }
    }

    pDev->localSta->stats.GoodReceives++;
    if (pDesc->pSrcSibEntry) {
        pDesc->pSrcSibEntry->stats.GoodReceives++;
    }
    athFrameReceive(pDev, pDesc);

    return A_OK;
}

static unsigned int ffRxCount = 0; /* debug */

/*************************************************************************
 * recvFastFrame - munges a IEEE802.11 data frame to a Dot3 frame
 *   and passes it up the stack
 *
 * RETURNS: error status
 *   Discards the frame on error
 */

LOCAL INLINE A_STATUS
recvFastFrame(WLAN_DEV_INFO *pDev, ATHEROS_DESC *pDesc, ATHL2P_802_3_TUNNEL_HDR *pFFHdr)
{
    SIB_ENTRY               *pSib = pDesc->pSrcSibEntry;
    ATHEROS_DESC            *pRxDesc;
    A_UINT32                 len;

    if (!pSib) {
        /* fast frame link state is not available */
        freeBuffandDescChain(pDev, pDesc);
        return A_EPROTO;
    }

    len = sizeof(ATHL2P_802_3_TUNNEL_HDR) + (pFFHdr->optHdrLen32 * sizeof(A_UINT32));

    if (pSib->pRxFFDesc &&
        ((pSib->pRxFFDesc->ffSeqNum + 1) == pFFHdr->seqNum) &&
        (pSib->pRxFFDesc->frameLength ==
            (pSib->pRxFFDesc->bufferLength + pFFHdr->offset)) &&
        ((pRxDesc = memAllocateDescriptor(pDev))))
    {
        ASSERT(pDesc->pNextVirtPtr == NULL);        /* for now */
        pRxDesc->pSrcSibEntry              = pDesc->pSrcSibEntry;
        pRxDesc->pOpBss                    = pDesc->pOpBss;
        pRxDesc->pBufferVirtPtr.byte       = pDesc->pBufferVirtPtr.byte + len;
        pRxDesc->bufferPhysPtr             = pDesc->bufferPhysPtr + len;
        pRxDesc->is4AddrFrm                = pDesc->is4AddrFrm;
        pRxDesc->bufferLength = pFFHdr->offset;

        ATH_OSBUFREF_DUP(pDesc, pRxDesc);
#ifndef VXWORKS
        ++pDesc->ffRefHack;
        pRxDesc->ffRefHack = (A_UINT32)pDesc;
#endif
        pSib->pRxFFDesc->pNextVirtPtr      = pRxDesc;
        pSib->pRxFFDesc->more = 1;

        pRxDesc = pSib->pRxFFDesc;
        pSib->pRxFFDesc = NULL;

        recvDot3Frame(pDev, pRxDesc);
    }

    pDesc->ffSeqNum = pFFHdr->seqNum;
    len += pFFHdr->offset;
    len += A_ROUNDUP_PAD(len + 2, sizeof(A_UINT32));

    if (NULL == bufferPopHdr(pDesc, len)) {
        freeBuffandDescChain(pDev, pDesc);
        return A_EPROTO;
    }

    rxFFDrain(pDev, pSib);

    while (pDesc->frameLength >= sizeof(LAN_FRAME_HEADER)) {
        LAN_FRAME_HEADER *pLanHdr =
            bufferGetHdr(pDesc, sizeof(LAN_FRAME_HEADER));

        if (!pLanHdr) {
            uiPrintf("Misbehaving FFTx - runt(0x%x)\n",
                     pDesc->frameLength);
            freeBuffandDescChain(pDev, pDesc);
            return A_EPROTO;
        }

        len = be2cpu16(pLanHdr->lanTypeOrLen);
        if (IS_ETHERTYPE(len)) {
            uiPrintf("Misbehaving FFTx - etherType(0x%x)\n", len);
            freeBuffandDescChain(pDev, pDesc);
            return A_EPROTO;
        }
        if (len < sizeof(LLC_SNAP_HEADER)) {
            uiPrintf("Runt frame in FF - length(0x%x)\n", len);
            freeBuffandDescChain(pDev, pDesc);
            return A_EPROTO;
        } 

        len += sizeof(LAN_FRAME_HEADER);
        if (pDesc->frameLength == len) {
            return recvDot3Frame(pDev, pDesc);
        }

        if (pDesc->frameLength < len) {
            pDesc->frameLength = len;
            pSib->pRxFFDesc = pDesc;
            return A_OK;
        }

        if (pDesc->frameLength < A_ROUNDUP(len, sizeof(A_UINT32))) {
            uiPrintf("Misbehaving FFTx - short(0x%x)\n",
                    A_ROUNDUP(len, sizeof(A_UINT32)) -
                     pDesc->frameLength);
            freeBuffandDescChain(pDev, pDesc);
            return A_EPROTO;
        }

        pRxDesc = memAllocateDescriptor(pDev);
        if (!pRxDesc) {
            freeBuffandDescChain(pDev, pDesc);
            return A_NO_MEMORY;
        }

        ASSERT(pDesc->pNextVirtPtr == NULL);        /* for now */

        pRxDesc->pSrcSibEntry              = pDesc->pSrcSibEntry;
        pRxDesc->pOpBss                    = pDesc->pOpBss;
        pRxDesc->pBufferVirtPtr.ptr        = pDesc->pBufferVirtPtr.ptr;
        pRxDesc->bufferPhysPtr             = pDesc->bufferPhysPtr;
        pRxDesc->is4AddrFrm                = pDesc->is4AddrFrm;
        pRxDesc->frameLength  = len;
        pRxDesc->bufferLength = len;
        bufferPopHdr(pDesc, A_ROUNDUP(len, sizeof(A_UINT32)));

        ATH_OSBUFREF_DUP(pDesc, pRxDesc);
#ifndef VXWORKS
        ++pDesc->ffRefHack;
        pRxDesc->ffRefHack = (A_UINT32)pDesc;
#endif
        recvDot3Frame(pDev, pRxDesc);
    }

    uiPrintf("Misbehaving FFTx - runt(0x%x)\n",
             pDesc->frameLength);
    freeBuffandDescChain(pDev, pDesc);
    return A_EPROTO;
}


/*************************************************************************
 * recvDot11Frame - munges a IEEE802.11 data frame to a Dot3 frame
 *   and passes it up the stack
 *
 * RETURNS: error status
 *   Discards the frame on error
 */

LOCAL INLINE A_STATUS
recvDot11Frame(WLAN_DEV_INFO *pDev, ATHEROS_DESC *pDesc)
{
    WLAN_FRAME_HEADER *pWlanHdr;
    LAN_FRAME_HEADER   cachedLanHdr;
    A_UINT32           cachedQoSInfo = 0;
    LAN_FRAME_HEADER  *pLanHdr;
    LLC_SNAP_HEADER   *pLLCHdr;
    WLAN_MACADDR      *pSrc, *pDest;
    FRAME_CONTROL      fc;
    A_UINT16           etherType = 0;
    A_STATUS           status;
    A_UINT32           ivLen = 0;
    LAN_LLC_HEADER*    pLanLLCHdr;
#ifdef WME
    A_UINT32           sizetoPop = 0;
    A_UINT16           qosCntl;
#endif 

    pWlanHdr = bufferGetHdr(pDesc, sizeof(WLAN_DATA_MAC_HEADER3));
    if (!pWlanHdr) {
        freeBuffandDescChain(pDev, pDesc);
        return A_OK;
    }
    fc       = pWlanHdr->frameControl;

    ASSERT(fc.fType == FRAME_DATA);
#ifdef WME
    if ((fc.fSubtype != SUBT_DATA) && (fc.fSubtype != SUBT_QOS)) {
        /* silently discard non data nor qos subtype frames */
#else
    if (fc.fSubtype != SUBT_DATA) {
        /* silently discard non data subtype frames */
#endif
        freeBuffandDescChain(pDev, pDesc);
        return A_OK;
    }

    /*
     * The following two step copy using temp variables is necessary
     * as we're copying over the same place in memory.
     */
    pSrc  = fc.FromDS ? (fc.ToDS ? &pWlanHdr->address4 : &pWlanHdr->address3) :
                        &pWlanHdr->address2;
    pDest = fc.ToDS ? &pWlanHdr->address3 : &pWlanHdr->address1;
    A_MACADDR_COPY(pSrc,  &cachedLanHdr.srcAddr);
    A_MACADDR_COPY(pDest, &cachedLanHdr.destAddr);

#ifdef VPRINT
    if (vprintflag>1 /*&& (fc.fSubtype&SUBT_QOS)*/) {
        unsigned int len = sizeof(WLAN_DATA_MAC_HEADER3)+2;

        if (pDesc->bufferLength >len) {
            len = min(48, pDesc->bufferLength);
        }
        mpx("R\n", pWlanHdr, len);
    }
#endif

    pDesc->is4AddrFrm = 0;
    bufferPopHdr(pDesc, sizeof(WLAN_DATA_MAC_HEADER3));

    if (fc.FromDS && fc.ToDS) {
        pDesc->is4AddrFrm = 1;
#ifdef WME
        sizetoPop += sizeof(WLAN_MACADDR);
        if (fc.fSubtype & SUBT_QOS) {
            void *tmp = bufferGetHdr(pDesc, sizeof(WLAN_MACADDR) + sizeof(QOS_CNTL));
            A_UINT8 *tmpQosCntl = (A_UINT8 *)((WLAN_MACADDR *)tmp + 1);
            qosCntl =  (A_UINT16) ((*tmpQosCntl) + (*(tmpQosCntl + 1) << 8));
            cachedQoSInfo |= qosCntl;
            sizetoPop += sizeof(QOS_CNTL);
            pDesc->txQandMode = qosCntl;
        }
    } else {
        if (fc.fSubtype & SUBT_QOS) {
            A_UINT8 *tmp = (A_UINT8 *) bufferGetHdr(pDesc, sizeof(QOS_CNTL));
            qosCntl = (A_UINT16) ((*tmp) + (*(tmp+1) << 8));
            cachedQoSInfo |= qosCntl;
            sizetoPop += sizeof(QOS_CNTL);
            pDesc->txQandMode = qosCntl;
        }
    }
    
    bufferPopHdr (pDesc, A_ROUNDUP(sizetoPop, sizeof(A_UINT32)));

#else 
         bufferPopHdr(pDesc, A_ROUNDUP(sizeof(WLAN_MACADDR), sizeof(A_UINT32)));
    }

#endif

    if (fc.wep) {
        ivLen = WEP_IV_FIELD_SIZE;
        if (pDesc->pBufferVirtPtr.word[0] & EXT_IV_BIT) {
            ivLen += EXT_IV_FIELD_SIZE;
        }
        bufferPopHdr(pDesc, ivLen);
    }

    if (pDesc->frameLength < sizeof(LAN_LLC_HEADER)) {
        /* silently discard runt frames */
        freeBuffandDescChain(pDev, pDesc);
        return A_OK;
    }

    if (fc.wep) {
        status = xformMicToDot2(pDev, pDesc, 
                                &cachedLanHdr.srcAddr,
                                &cachedLanHdr.destAddr, &cachedQoSInfo);
        if (status == A_ERROR) {
            freeBuffandDescChain(pDev, pDesc);
            return status;
        }
        if (pDesc->frameLength < sizeof(LAN_LLC_HEADER)) {
            /* Discard frames which are still runt */            
            freeBuffandDescChain(pDev, pDesc);
            return A_OK;
        }
    }

    pLanLLCHdr = bufferGetHdr(pDesc, sizeof(LAN_LLC_HEADER));

    if(LLC_SAP_SNAP == pLanLLCHdr->dsap)
    {
        pLLCHdr   = bufferGetHdr(pDesc, sizeof(LLC_SNAP_HEADER));
        etherType = be2cpu16(pLLCHdr->etherType);
    }

    status = rxEnPolicy(pDev, pDesc, etherType, fc.wep,
                        &cachedLanHdr.destAddr);
    if (status != A_OK) {
        freeBuffandDescChain(pDev, pDesc);
        return status;
    }

    if (etherType == ETHERTYPE_ATHL2P) {
        ATHL2P_802_3_TUNNEL_HDR *pAthL2PHdr;
        if (NULL == bufferPopHdr(pDesc, sizeof(LLC_SNAP_HEADER))) {
            freeBuffandDescChain(pDev, pDesc);
            return A_OK;
        }

        pAthL2PHdr = bufferGetHdr(pDesc, sizeof(ATHL2P_802_3_TUNNEL_HDR));
        if (!pAthL2PHdr) {
            uiPrintf("Bad ATHL2P - runt(0x%x)\n",
                 pDesc->frameLength);
            freeBuffandDescChain(pDev, pDesc);
            return A_EPROTO;
        }

        BE2CPU_ATHL2P(pAthL2PHdr);

        if (pAthL2PHdr->proto == ATHL2P_802_3_TUNNEL) {
            switch (pAthL2PHdr->frameType) {
            case FRAME_TYPE_FAST_FRAME:
                return recvFastFrame(pDev, pDesc, pAthL2PHdr);
                break;
            default:
                /* let upper layer handle it */
                break;
            }
        } /* else let upper layer handle it */
    }
    rxFFDrain(pDev, pDesc->pSrcSibEntry);

    cachedLanHdr.lanTypeOrLen = cpu2be16(pDesc->frameLength);

    pLanHdr = bufferPushHdr(pDesc, sizeof(LAN_FRAME_HEADER));
    A_BCOPY(&cachedLanHdr, pLanHdr, sizeof(LAN_FRAME_HEADER));

    return recvDot3Frame(pDev, pDesc);
}


/*************************************************************************
 * wlanDataFrameReceive - entry point for receiving data frames and
 *   pushing them up the stack! Looks like a IEEE802.11 data frame
 *   when it comes in
 *
 * RETURNS: error status
 *   Discards the frame on error
 */

A_STATUS
wlanDataFrameReceive(WLAN_DEV_INFO *pDev, ATHEROS_DESC *pDesc)
{
    /*
     * The following is rather hacky - it maintains the receive
     * buffer length(s) and frame length(s) within HW TxControl
     * part of the descriptor!! Its the current strategy never
     * the less because our Atheros Descriptors don't have the
     * corresponding software fields. So current strategy of
     * treating the corresponding TX CONTROL fields as generic
     * allows us to keep the Tx code as such, perform the DS
     * service as such, and use the buffer Get/Push/Pop Hdr 
     * utilities as such!
     */
    A_RX_DESC_CHAIN_TO_TX(pDesc);

    return recvDot11Frame(pDev, pDesc);
}

