/*
 * $Id: //depot/sw/branches/1.3_USB_LINUX_port/src/USB/wlan/host/common/crypto.c#1 $
 *
 * Copyright  2000-2003 Atheros Communications, Inc.,  All Rights Reserved.
 *
 * Software encryption code.  HW support is in the HAL.
 *
 * The sw encryption support is split into 2 levels.  The sw encryption
 * algorithm engine and hw encryption/decryption engine control.
 *
 * hw encryption engine control:
 * enable/disable hw encryption/decryption engine based on the
 *
 * sw encryption engine:
 * implement the WEP, AES algorithms in software
 *      sw WEP & AES encryption - swEnCipher();
 *      sw WEP & AES decryption - swDecipher();
 *
 */

#include "wlandrv.h"
#include "wlanext.h"
#include "wlansme.h"
#include "wlansmeext.h"
#include "wlanframe.h"
#include "wlanext.h"
#include "display.h"
#include "ui.h"

#include "crypto/ccm.h"
#include "crypto/ocb.h"
#include "crypto/wep.h"
#include "crypto/tkip.h"

#define AES_OCB_NONCE_SIZE 16
#define WEP_LOCAL_KEY_SIZE 19

INLINE void
warpNonceSet(WLAN_DATA_MAC_HEADER *, A_UCHAR *, A_UINT32);

INLINE A_STATUS
warp_encipher(WLAN_PRIV_RECORD *, A_UCHAR *, A_UCHAR *, A_UINT16, A_UINT16);

INLINE A_STATUS
warp_decipher(WLAN_PRIV_RECORD *, A_UCHAR *, A_UCHAR *, A_UINT16, A_UINT16);

static INLINE A_STATUS
inOutBufCopy(WLAN_DEV_INFO *, ATHEROS_DESC *, void **, void **);

INLINE void
wepLocalKeySet(WLAN_DATA_MAC_HEADER *, A_UCHAR *, WLAN_PRIV_RECORD *, A_UINT32);

INLINE A_STATUS
wep_encipher(WLAN_PRIV_RECORD *, A_UCHAR *, A_UINT16);

INLINE A_STATUS
wep_decipher(WLAN_PRIV_RECORD *, A_UCHAR *, A_UINT16);

static INLINE A_STATUS
inPlaceBufCopy(WLAN_DEV_INFO *, ATHEROS_DESC *, void **);

static A_STATUS
singleFlatBufCopy(WLAN_DEV_INFO *pDevInfo, ATHEROS_DESC **pClearDesc);

#ifndef BUILD_AP
int ckip_encipher(WLAN_PRIV_RECORD *pk, A_UINT8 *pkt, int pktlen);
int ckip_decipher(WLAN_PRIV_RECORD *pk, A_UINT8 *pkt, int pktlen);
#endif

#ifdef VERBOSE
void drvMpx(unsigned char *buf, int len);
#endif

#ifdef DEBUG
int micStride = 0;
int replayDebug = 0;
#else
const int micStride = 0;
const int replayDebug = 0;
#endif

A_BOOL
micFailNow(void)
{
#ifdef DEBUG
    static int micCount = 0;

    if (micStride) {
        if (micCount++ > 0 && (micCount % micStride) == 0) {
            /* One-shot operation */
            micStride = 0;
            return TRUE;
        }
    }
#endif
    return FALSE;
}

/*
 *  Enforce key usage policies, and return a valid pKey.
 */
A_STATUS
rxKeyLookup(WLAN_DEV_INFO *pdevInfo, ATHEROS_DESC *pDesc)
{
    WLAN_PRIV_RECORD            *pKey;
    WLAN_MACADDR                *dest =
        &pDesc->pBufferVirtPtr.header->address1;
    A_BOOL                      isGroup = isGrp(dest);
    SIB_ENTRY                   *pSib = pDesc->pSrcSibEntry;

    pDesc->pKey = NULL;

    /* Gotta allow clear to pass at all times */
    if (!pDesc->pBufferVirtPtr.header->frameControl.wep || !pSib) {
        return A_ENOENT;
    }

    if (pSib) {
        A_UINT32 hwRecvIndex = pDesc->status.rx.hwIndex;

        if (!isGroup) {
            if (pSib->wdcConnId == hwRecvIndex) {
                pKey = pSib->pPriv;

                if (pKey == NULL) {
                    uiPrintf("rxKeyLookup: NULL SIB's key\n");
                }

            } else {
                /* Nasty asymetric static WEP key case */
                if ((pdevInfo->staConfig.privacyInvoked) &&
                    (!pdevInfo->staConfig.wpaEnabled)) {
                    /* The AP may use another transmit key than ours */
                    pKey = &pdevInfo->keyTable[hwRecvIndex];
                    if (pKey->keyLength == 0) {
                        uiPrintf("rxLookup: invalid key\n");
                        return A_ERROR;
                    }
                } else {
                    uiPrintf("rxLookup: mismatch connId (%d) != rxhwIndex (%d)\n",
                             pSib->wdcConnId, pDesc->status.rx.hwIndex);
                    return A_ERROR;
                }
            }
        } else {
            ASSERT(pDesc->status.rx.hwIndex < MAX_SHARED_KEYS);
            pKey = &pdevInfo->keyTable[pDesc->status.rx.hwIndex];
            if (pKey->keyLength == 0) {
                uiPrintf("rxKeyLookup: null BSS key at index %d\n",
                         pDesc->status.rx.hwIndex);
            }
        }
    }
    /*
     *  Policy check:
     *      Must have a valid key
     *      Multicast frames must not do a keysrch (XXX until we support
     *                  ad hoc with unique broadcast keys)
     */
    if (pKey == NULL ||
        !(pKey->keyFlags & PRIV_KEY_LOCKED) ||
        pKey->keyLength == 0)
    {
        pdevInfo->localSta->stats.RcvWEPExcludedCount++;
        if (pSib) {
            pSib->stats.RcvWEPExcludedCount++;
        }
        uiPrintf("rxLookupKey: reject frame\n");
        return A_ERROR;
    }

    ASSERT(isGroup || !pSib || pSib->pPriv == pKey ||
           (pKey->keyFlags & PRIV_UNIQUEKEY) == 0);

    pDesc->pKey = pKey;

    return A_OK;
}

/*
 * Determine which ciphers are supported by Target.
 * Called during startup in order to initialize
 * cipher capabilities that are cached on the Host.
 */
void
cryptCapInit(WLAN_DEV_INFO *pdev)
{
    DEV_CAPABILITIES *pCap = &pdev->devCap;
    A_UINT32         boolVal;

    /* Cipher */
    wdcTargetGetCapability(pdev->targetHandle,
                           CAP_CIPHER_WEP,
                           NULL,
                           (TARGET_CAP_DATA *)&boolVal,
                           NULL);
    pCap->supportCipherWEP = boolVal ? TRUE : FALSE;



    wdcTargetGetCapability(pdev->targetHandle,
                           CAP_CIPHER_AES_OCB,
                           NULL,
                           (TARGET_CAP_DATA *)&boolVal,
                           NULL);
    pCap->supportCipherAES_OCB = boolVal ? TRUE : FALSE;



    wdcTargetGetCapability(pdev->targetHandle,
                           CAP_CIPHER_AES_CCM,
                           NULL,
                           (TARGET_CAP_DATA *)&boolVal,
                           NULL);
    pCap->supportCipherAES_CCM = boolVal ? TRUE : FALSE;



    wdcTargetGetCapability(pdev->targetHandle,
                           CAP_CIPHER_TKIP,
                           NULL,
                           (TARGET_CAP_DATA *)&boolVal,
                           NULL);
    pCap->supportCipherTKIP = boolVal ? TRUE : FALSE;



    wdcTargetGetCapability(pdev->targetHandle,
                           CAP_CIPHER_CKIP,
                           NULL,
                           (TARGET_CAP_DATA *)&boolVal,
                           NULL);
    pCap->supportCipherCKIP = boolVal ? TRUE : FALSE;



    /* Mic */
    wdcTargetGetCapability(pdev->targetHandle,
                           CAP_MIC_AES_CCM,
                           NULL,
                           (TARGET_CAP_DATA *)&boolVal,
                           NULL);
    pCap->supportMicAES_CCM = boolVal ? TRUE : FALSE;



    wdcTargetGetCapability(pdev->targetHandle,
                           CAP_MIC_TKIP,
                           NULL,
                           (TARGET_CAP_DATA *)&boolVal,
                           NULL);
    pCap->supportMicTKIP = boolVal ? TRUE : FALSE;



    wdcTargetGetCapability(pdev->targetHandle,
                           CAP_MIC_CKIP,
                           NULL,
                           (TARGET_CAP_DATA *)&boolVal,
                           NULL);
    pCap->supportMicCKIP = boolVal ? TRUE : FALSE;

}

A_BOOL
cipherSupported(WLAN_DEV_INFO *pdevInfo, A_UINT32 keyType)
{
    DEV_CAPABILITIES *pCap = &pdevInfo->devCap;

    switch(keyType) {
    case PRIV_KEY_TYPE_WEP:
    case PRIV_KEY_TYPE_WEP_COMP:
        return pCap->supportCipherWEP;
        break;

    case PRIV_KEY_TYPE_AES_OCB:
        return pCap->supportCipherAES_OCB;
        break;

    case PRIV_KEY_TYPE_AES_CCM:
        return pCap->supportCipherAES_CCM;
        break;

    case PRIV_KEY_TYPE_NULL:
        return TRUE;
        break;

    case PRIV_KEY_TYPE_TKIP:
        return pCap->supportCipherTKIP;
        break;

    case PRIV_KEY_TYPE_TKIP_SW:
        return FALSE;
        break;

    case PRIV_KEY_TYPE_CKIP:
        return pCap->supportCipherCKIP;
        break;

    default:
        ASSERT(0); /* Unknown Cipher type */
        return FALSE;
        break;
    }
}

A_BOOL
micSupported(WLAN_DEV_INFO *pdevInfo, A_UINT32 keyType)
{
    DEV_CAPABILITIES *pCap = &pdevInfo->devCap;

    switch(keyType) {
    case PRIV_KEY_TYPE_WEP:
    case PRIV_KEY_TYPE_WEP_COMP:
        return TRUE;
        break;

    case PRIV_KEY_TYPE_AES_OCB:
        return TRUE;
        break;

    case PRIV_KEY_TYPE_AES_CCM:
        return pCap->supportMicAES_CCM;
        break;

    case PRIV_KEY_TYPE_NULL:
        return TRUE;
        break;

    case PRIV_KEY_TYPE_TKIP:
        return pCap->supportMicTKIP;
        break;

    case PRIV_KEY_TYPE_TKIP_SW:
        return FALSE;
        break;

    case PRIV_KEY_TYPE_CKIP:
        return pCap->supportMicCKIP;
        break;

    default:
        ASSERT(0); /* Unknown MIC type */
        return FALSE;
        break;
    }
}

/*
 * If this is a WEP frame we need to:
 *- get the key from sib entry
 *- sw decrypt the frame if needed
 *- update decrypt error counter
 */
A_STATUS
setupDecrypt(WLAN_DEV_INFO *pdevInfo, ATHEROS_DESC *pDesc)
{
    WLAN_FRAME_3_DATA_ENCRYPT   *pHdr = pDesc->pBufferVirtPtr.header3Iv;
    WLAN_PRIV_RECORD            *pKey = pDesc->pKey;
    A_STATUS                    retVal = A_OK;
    SIB_ENTRY                   *pSib = pDesc->pSrcSibEntry;

    if (!pHdr->macHeader.frameControl.wep) {
        return A_OK;
    }

    cryptoEndianChange(pDesc->pBufferVirtPtr.header, LE2CPU);

    /* if sw decryption is needed */
    if (!cipherSupported(pdevInfo, pKey->keyType)
#ifdef BUILD_AP
        || pdevInfo->localSta->swEncryptionEnabled == TRUE ||
        (pSib && pSib->swEncryptionEnabled)
#endif
        )
    {
        if ((retVal = swDeCipher(pdevInfo, pKey, pDesc)) != A_OK) {
            return retVal;
        }
    }

    /* check IV+EIV for replay counter violations here */
    if (!pDesc->status.rx.decryptError) {
        if ((retVal = wlanReplayCheck(pdevInfo, pKey, pDesc)) != A_OK) {
            return retVal;
        }
    }

    /* Subtract the ICV field length */
    pDesc->status.rx.dataLength -= icvLengthGet(pKey);

    return retVal;
}

/*
 * This routine calculate values for WARP's nonce
 */
INLINE void
warpNonceSet(WLAN_DATA_MAC_HEADER *pWlanHdr,
             A_UCHAR *pNonce,
             A_UINT32 keyIV)
{
    /* Ignore the TCID for now until we have QoS support */
    /* Concatenate the Src and Dst MAC addresses with the 28bits of IV */
    A_MEM_ZERO(pNonce, sizeof(AES_OCB_NONCE_SIZE));

    /* Copy the full 4 IV bytes */
    *(pNonce + 0) = (A_UCHAR)(keyIV & 0xff);   /* 28 bit IV from payload */
    *(pNonce + 1) = (A_UCHAR)((keyIV >> 8) & 0xff);
    *(pNonce + 2) = (A_UCHAR)((keyIV >> 16) & 0xff);
    *(pNonce + 3) = (A_UCHAR)((keyIV >> 24) & 0xff);

    /* Zero out the 4 msb's that would normally be the TCID */
    *(pNonce + 3) &= 0xf;

    /* get address info */
    A_MACADDR_COPY(((pWlanHdr->frameControl.ToDS == 0) ?
                    &pWlanHdr->address1 : &pWlanHdr->address3),
                   (WLAN_MACADDR *)(pNonce + AES_IV_FIELD_SIZE +
                                    WLAN_MAC_ADDR_SIZE));
    A_MACADDR_COPY(((pWlanHdr->frameControl.FromDS == 0) ?
                    &pWlanHdr->address2 : ((pWlanHdr->frameControl.ToDS == 0) ?
                                           &pWlanHdr->address3 :
                                           &pWlanHdr->address4)),
                   (WLAN_MACADDR *)(pNonce + AES_IV_FIELD_SIZE));
}

/*
 * This routine calculate values for WEP's local key
 */
INLINE void
wepLocalKeySet(WLAN_DATA_MAC_HEADER *pWlanHdr,
               A_UCHAR *pLocalKey,
               WLAN_PRIV_RECORD *pKey,
               A_UINT32 iv)
{
    A_UINT32 nBytesKeyLen;
#ifndef BUILD_AP
    A_UINT32 i, temp = KEY_XOR;
#endif

    *(pLocalKey + 0) = (A_UCHAR)(iv & 0xff);  /* 24 bit IV from payload */
    *(pLocalKey + 1) = (A_UCHAR)((iv >> 8) & 0xff);
    *(pLocalKey + 2) = (A_UCHAR)((iv >> 16) & 0xff);
    nBytesKeyLen = pKey->keyLength >> 3;

    A_BCOPY(pKey->keyVal, (pLocalKey + 3), nBytesKeyLen);

    /* unmask the key values for station */
#ifndef BUILD_AP
    for (i = 3; i < (nBytesKeyLen + 3); i++) {
        *(pLocalKey + i) ^= temp;
    }
#endif

    if (nBytesKeyLen < 13) {
        A_BCOPY(pLocalKey, (pLocalKey + nBytesKeyLen + 3), 13-nBytesKeyLen);
    }
}

static INLINE A_STATUS
inPlaceBufCopy(WLAN_DEV_INFO *pdevInfo, ATHEROS_DESC *pInChainDesc,
               void **ppClearBufPtr)
{
    ATHEROS_DESC         *pPayLoadDesc, *pDesc;
    A_UCHAR              *pBufPtr;           /* pointer to payload buffer */
    A_UINT32             hdrLen, dataLen = 0;
    A_UINT16             ivLen;

    hdrLen = headerLengthGet(pInChainDesc);
    ivLen = ivLengthGet(pInChainDesc);

    /*
     * put clear data into one buffer, get the clear buffer pointer
     */

    /* first check whether the payload is already in a contiguous buffer */
    if (pInChainDesc->more) {
        /* allocate a new desc/buffer for payload */
        /* TODO: some optimization could be done here */
        if (createBuffandDesc(pdevInfo, AR_BUF_SIZE, &pPayLoadDesc) != A_OK) {
            return A_NO_MEMORY;
        }

        /* copy all payload data into new payload buffer */
        pBufPtr = pPayLoadDesc->pBufferVirtPtr.byte;

        /* header buffer may contain some payload bytes too */
        dataLen = pInChainDesc->bufferLength - hdrLen - ivLen;
        if (dataLen) {
            A_BCOPY(pInChainDesc->pBufferVirtPtr.byte + hdrLen + ivLen,
                    pBufPtr, dataLen);
            pInChainDesc->bufferLength -= dataLen;
            pBufPtr += dataLen;
        }

        pDesc = pInChainDesc;
        while (pDesc->more) {
            pDesc = pDesc->pNextVirtPtr;
            ASSERT(pDesc);

            A_BCOPY(pDesc->pBufferVirtPtr.ptr,
                    (A_UCHAR *)pBufPtr,
                    pDesc->bufferLength);

            pBufPtr += pDesc->bufferLength;
        }

        /* release the old buffer chain */
        pPayLoadDesc->pNextVirtPtr  = pInChainDesc->pTxLastDesc->pNextVirtPtr;

        pDesc = pInChainDesc->pNextVirtPtr;
        pInChainDesc->pTxLastDesc->pNextVirtPtr = NULL;

        ATH_OSPKTREF_DUP(pDesc, pPayLoadDesc);

        freeBuffandDescChain(pdevInfo, pDesc);

        /* re-construct the desc chain */
        pInChainDesc->pNextVirtPtr              = pPayLoadDesc;
        pInChainDesc->pTxLastDesc               = pPayLoadDesc;

        pPayLoadDesc->pTxFirstDesc              = pInChainDesc;
        pPayLoadDesc->pTxLastDesc               = pPayLoadDesc;
        pPayLoadDesc->bufferLength =
            pInChainDesc->frameLength -
            hdrLen - ivLen - FCS_FIELD_SIZE;

        *ppClearBufPtr = (A_UCHAR *)pPayLoadDesc->pBufferVirtPtr.ptr;

    } else {
        *ppClearBufPtr = (A_UCHAR *)pInChainDesc->pBufferVirtPtr.ptr +
            hdrLen + ivLen;
    }

    return A_OK;
}

static INLINE A_STATUS
inOutBufCopy(WLAN_DEV_INFO *pdevInfo, ATHEROS_DESC *pInChainDesc,
             void **ppClearBufPtr, void **ppCipherBufPtr)
{
    ATHEROS_DESC            *pPayLoadDesc, *pDesc;
    A_UCHAR                 *pBufPtr;           /* pointer to payload buffer */
    WLAN_DATA_MAC_HEADER    *pWlanHdr;          /* WLAN frame header */
    A_UINT32                hdrLen;
    A_UINT16                ivLen;

    pWlanHdr = pInChainDesc->pBufferVirtPtr.header;

    hdrLen = headerLengthGet(pInChainDesc);
    ivLen = ivLengthGet(pInChainDesc);

    /*
     * put clear data into one buffer, get the clear buffer pointer
     */

    /* first to check whether the payload is already in a continous buffer */
    if (pInChainDesc->more) {

        /* copy all data into new payload buffer */
        pBufPtr = (A_UCHAR *)pdevInfo->virtEncryptBuf;

        /* header buffer may contain some payload bytes too */
        A_BCOPY(pInChainDesc->pBufferVirtPtr.ptr, pBufPtr,
                pInChainDesc->bufferLength);
        pBufPtr += pInChainDesc->bufferLength;

        pDesc = pInChainDesc;
        while (pDesc->more) {
            pDesc = pDesc->pNextVirtPtr;
            ASSERT(pDesc);

            A_BCOPY(pDesc->pBufferVirtPtr.ptr, pBufPtr,
                    pDesc->bufferLength);

            pBufPtr += pDesc->bufferLength;
        }

        *ppClearBufPtr = pdevInfo->virtEncryptBuf;

    } else {

        *ppClearBufPtr = (A_UCHAR *)pInChainDesc->pBufferVirtPtr.ptr;
    }

    /* allocate a new desc/buffer for output payload */
    if (createBuffandDesc(pdevInfo, AR_BUF_SIZE, &pPayLoadDesc) != A_OK) {
        return A_NO_MEMORY;
    }

    *ppCipherBufPtr = pPayLoadDesc->pBufferVirtPtr.ptr;

    /* link the allocated desc/buff after the header desc/buffer */
    pPayLoadDesc->pNextVirtPtr  = pInChainDesc->pTxLastDesc->pNextVirtPtr;

    if (pInChainDesc->more) {
        pDesc = pInChainDesc->pNextVirtPtr;
        pInChainDesc->pTxLastDesc->pNextVirtPtr = NULL;

        freeBuffandDescChain(pdevInfo, pDesc);
    } else {
        pInChainDesc->more = 1;
    }

    /* re-construct the desc chain */
    pInChainDesc->pNextVirtPtr              = pPayLoadDesc;
    pInChainDesc->pTxLastDesc               = pPayLoadDesc;
    pPayLoadDesc->pTxFirstDesc              = pInChainDesc;
    pPayLoadDesc->pTxLastDesc               = pPayLoadDesc;
    pInChainDesc->bufferLength = hdrLen + ivLen;
    pPayLoadDesc->bufferLength =
        pInChainDesc->frameLength - hdrLen - ivLen - FCS_FIELD_SIZE;

    return A_OK;
}

INLINE A_STATUS
wep_encipher(WLAN_PRIV_RECORD *pKey, A_UCHAR *pBuf, A_UINT16 pLen)
{
    A_UCHAR           localKey[WEP_LOCAL_KEY_SIZE];
    A_UINT32          nBytesKeyLen;
    A_UINT16          hdrLen, len;
    WLAN_FRAME_HEADER *pWlanHdr;
#ifndef BUILD_AP
    A_UINT32          i, temp = KEY_XOR;
#endif

    hdrLen = WLAN_HDR_SIZE;

    /* If to/fromDS are both 1, we're bridging */
    pWlanHdr = (WLAN_FRAME_HEADER *)pBuf;
    if (pWlanHdr->frameControl.ToDS == 1 &&
       pWlanHdr->frameControl.FromDS == 1)
    {
        hdrLen += WLAN_MAC_ADDR_SIZE;
    }
    hdrLen = A_ROUNDUP(hdrLen, sizeof(A_UINT32));

    pBuf += hdrLen;

    /* build the localkey from the IV and key */
    localKey[0] = *pBuf++;
    localKey[1] = *pBuf++;
    localKey[2] = *pBuf++;
    pBuf++;
    /* pBuf now points to start of payload */

    /* trim off header and IV */
    len = pLen - hdrLen - 4;

#if 0
    localKey[0] = (A_UCHAR)(pKey->keyIV & 0xff);/* 24 bit IV from payload */
    localKey[1] = (A_UCHAR)((pKey->keyIV >> 8) & 0xff);
    localKey[2] = (A_UCHAR)((pKey->keyIV >> 16) & 0xff);
#endif

    nBytesKeyLen = pKey->keyLength >> 3;

    A_BCOPY(pKey->keyVal, &localKey[3], nBytesKeyLen);

    /* unmask the key values for station */
#ifndef BUILD_AP
    for (i = 3; i < (nBytesKeyLen + 3); i++) {
        localKey[i] ^= temp;
    }
#endif

    if (nBytesKeyLen < 13) {
        A_BCOPY(localKey, &localKey[nBytesKeyLen+3], 13-nBytesKeyLen);
    }

    switch(pKey->keyLength) {
    case 128:
        WepEncipher128(localKey, pBuf, len);
        break;

    case 104:
    case 40:
        WepEncipher(localKey, pBuf, len);
        break;

    default:
#ifdef DEBUG
        uiPrintf("unsupported WEP key length! \n");
#endif
        return A_ERROR;
    }

    return A_OK;

}

INLINE A_STATUS
warp_encipher(WLAN_PRIV_RECORD *pKey, A_UCHAR *pClearBuf, A_UCHAR *pCipherBuf, A_UINT16 hdrLen,
              A_UINT16 pLen)
{
    A_UCHAR nonce[AES_OCB_NONCE_SIZE];

    warpNonceSet((WLAN_DATA_MAC_HEADER *)pClearBuf, nonce, pKey->keyIV);

    if (pKey->keyLength == 128) {
        ocb_aes_encrypt((keystruct *)pKey->initValue, nonce,
                        (pClearBuf + hdrLen),
                        (A_UINT16)pLen, pCipherBuf, (pCipherBuf + pLen));
    } else {
#ifdef DEBUG
        uiPrintf("unsupported AES key length! \n");
#endif
        return A_ERROR;
    }

    return A_OK;

}

A_STATUS
swEnCipher(struct wlanDevInfo *pdevInfo, WLAN_PRIV_RECORD *pKey,
           ATHEROS_DESC *pClearDesc)
{
    A_UCHAR                 *pClearBuf, *pCipherBuf;
    A_UINT16                payLoadLen, hdrLen;
    WLAN_DATA_MAC_HEADER    *pWlanHdr;
    A_STATUS                status = A_ERROR;
    A_UINT16                ivLen;
    A_UINT8                 mask;

    mask = (pdevInfo->localSta->serviceType == WLAN_STA_SERVICE) ? KEY_XOR : 0;

    if ((pKey == NULL) || (pClearDesc == NULL)) {
        return A_ERROR;
    }

    pWlanHdr = pClearDesc->pBufferVirtPtr.header;

    hdrLen = headerLengthGet(pClearDesc);
    ivLen = ivLengthGet(pClearDesc);

    /* We use this to tell us how much to encrypt */
    payLoadLen = (A_UINT16)pClearDesc->frameLength -
                 hdrLen - ivLen - FCS_FIELD_SIZE;

    /* do nothing if no payload data */
    if (payLoadLen == 0) {
        return A_OK;
    }

    switch(pKey->keyType) {
    case PRIV_KEY_TYPE_WEP:
        /* We use hw wep when we have  MIC but no CKIP */
        ASSERT((pKey->keyFlags & PRIV_CKIP_MIC) == 0);

        /* first to make sure the payload is already in a contiguous buffer */
        if (inPlaceBufCopy(pdevInfo, pClearDesc,
                           (void **)&pClearBuf) != A_OK)
        {
            return A_NO_MEMORY;
        }

        status = wep_encipher(pKey, pClearBuf, payLoadLen);
        break;

#ifndef BUILD_AP
    case PRIV_KEY_TYPE_CKIP:
        /*
         * Sigh.  This interface requires the WLAN header, so we pass
         * a ptr to the entire (contiguous) frame.  It needs to
         * calc the proper len too, so similarly pass the len of whole frame.
         */
        status = ckip_encipher(pKey,
                               pClearDesc->pBufferVirtPtr.byte,
                               payLoadLen + hdrLen + ivLen);
        break;

#endif
    case PRIV_KEY_TYPE_TKIP:
    case PRIV_KEY_TYPE_TKIP_SW:
        keyMask(pKey->keyVal, pKey->keyVal, mask, pKey->keyLength/8);

        if (pClearDesc->more) {
            /* This will usually be done at enMic() time */
            if ((status = singleFlatBufCopy(pdevInfo, &pClearDesc)) != A_OK) {
                freeBuffandDescChain(pdevInfo, pClearDesc);
                return status;
            }
        }

        status = tkipEncipher(pKey, pClearDesc->pBufferVirtPtr.byte,
                              payLoadLen + hdrLen + ivLen);
        keyMask(pKey->keyVal, pKey->keyVal, mask, pKey->keyLength/8);
        break;

    case PRIV_KEY_TYPE_AES_OCB:

        /* first to make sure the payload is already in a contiguous buffer */
        if (inOutBufCopy(pdevInfo, pClearDesc, (void **)&pClearBuf,
                         (void **)&pCipherBuf) != A_OK) {
            return A_NO_MEMORY;
        }

        status = warp_encipher(pKey, pClearBuf, pCipherBuf,
                               hdrLen + ivLen, payLoadLen);

        break;

    case PRIV_KEY_TYPE_AES_CCM:

        /* first to make sure the payload is already in a contiguous buffer */
        if (inPlaceBufCopy(pdevInfo, pClearDesc, (void **)&pClearBuf) != A_OK)
        {
            return A_NO_MEMORY;
        }

        cryptoEndianChange(pClearDesc->pBufferVirtPtr.header, CPU2LE);
        status = ccm_encipher(pKey, pClearDesc->pBufferVirtPtr.ptr,
                              hdrLen + ivLen,
                              pClearBuf, payLoadLen);
        cryptoEndianChange(pClearDesc->pBufferVirtPtr.header, LE2CPU);

        break;

    default:
#ifdef DEBUG
        uiPrintf("swEncipher: unsupported key type! \n");
#endif
        break;
    }

    return status;
}

A_STATUS
swDeCipher(struct wlanDevInfo *pdevInfo,
           WLAN_PRIV_RECORD *pKey,
           ATHEROS_DESC *pCipherDesc)
{
    A_UCHAR              localkey[WEP_LOCAL_KEY_SIZE];
    A_UCHAR              *fp;
    A_INT32              r = 0, payLoadLen;
    A_UINT16             hdrLen, ivLen, frameLen;
    A_UINT32             ivMask, rxKeyIV = 0;
    WLAN_DATA_MAC_HEADER *pWlanHdr;
    A_UINT32             buffPhysAddr;
    void                 *pBuffVirtAddr;
    void                 *pOSDescAddr;
    A_UINT16             icvLen;
    A_UCHAR              nonce[AES_OCB_NONCE_SIZE];
    void                 *clearBuf, *pCipherBuf;
    void                 *micBuf;
    A_UINT8              mask;

    mask = (pdevInfo->localSta->serviceType ==
            WLAN_STA_SERVICE) ? KEY_XOR : 0;

    pWlanHdr = pCipherDesc->pBufferVirtPtr.header;

    ASSERT(pWlanHdr->frameControl.wep);

    ASSERT(pKey);

    /* IV length is included in headerLength */
    hdrLen = headerLengthGet(pCipherDesc);
    ivLen = ivLengthGet(pCipherDesc);
    icvLen = icvLengthGet(pKey);
    frameLen = (A_UINT16)pCipherDesc->status.rx.dataLength;
    fp = (A_UCHAR *)pCipherDesc->pBufferVirtPtr.ptr + hdrLen + ivLen;
    payLoadLen = frameLen - hdrLen - ivLen - FCS_FIELD_SIZE;
    rxKeyIV = pCipherDesc->pBufferVirtPtr.header3Iv->iv;

    if (payLoadLen <= icvLen) {
        return A_OK;
    }

    switch(pKey->keyType) {
    case PRIV_KEY_TYPE_WEP:
    case PRIV_KEY_TYPE_WEP_COMP:
        ivMask  = 0x00ffffff;           /* 24 bit IV */
        rxKeyIV &= ivMask;
        switch (pKey->keyLength) {
        case 128:
        case 104:
        case 40:
            /* 24 bit IV from payload */
            localkey[0] = (A_UCHAR)(rxKeyIV & 0xff);
            localkey[1] = (A_UCHAR)((rxKeyIV >> 8) & 0xff);
            localkey[2] = (A_UCHAR)((rxKeyIV >> 16) & 0xff);
            r = pKey->keyLength >> 3;
            A_BCOPY(pKey->keyVal, &localkey[3], r);
            keyMask(&localkey[3], &localkey[3], mask, (A_UINT16)r);
            if (r < 13) {
                A_BCOPY(&localkey[0], &localkey[r+3], 13-r);
            }

            if (r == 16) {                  // 128-bit key, r will be 10
                r = WepDecipher128(localkey, fp, (A_UINT16)payLoadLen);
            } else {
                r = WepDecipher(localkey, fp, (A_UINT16)payLoadLen);
            }
#ifdef DEBUG
            if (r == 0) {
                uiPrintf("swDeCipher: WEP decipher failed for ");
                printMacAddress(pWlanHdr->address2);
                uiPrintf("\n");
            }
#endif
            break;

        default:
#ifdef DEBUG
            uiPrintf("unsupported WEP key length! \n");
#endif
            break;
        }
        break;

#ifndef BUILD_AP
    case PRIV_KEY_TYPE_CKIP:
        r = (ckip_decipher(pKey, (A_UINT8 *)pWlanHdr, frameLen -
                          FCS_FIELD_SIZE) == A_OK) ? 1 : 0;
        break;

#endif
    case PRIV_KEY_TYPE_TKIP:
    case PRIV_KEY_TYPE_TKIP_SW:
        keyMask(pKey->keyVal, pKey->keyVal, mask, pKey->keyLength/8);
        r = (tkipDecipher(pKey, (A_UINT8 *)pWlanHdr, frameLen -
                         FCS_FIELD_SIZE) == A_OK) ? 1 : 0;
        keyMask(pKey->keyVal, pKey->keyVal, mask, pKey->keyLength/8);
        break;

    case PRIV_KEY_TYPE_AES_OCB:
        payLoadLen -= AES_ICV_FIELD_SIZE;
        ivMask  = 0x0fffffff;           /* 28 bit IV */
        rxKeyIV &= ivMask;
        ASSERT(payLoadLen >= 0);

        warpNonceSet(pWlanHdr, &nonce[0], rxKeyIV);

        /* allocate a new clear buffer for AES decipher engine output */
#ifdef BUILD_AP
        if (hwAllocatePhysBuffer(pdevInfo,
                                 AR_BUF_SIZE,
                                 &buffPhysAddr,
                                 &pBuffVirtAddr,
                                 (CL_BLK_ID *)&pOSDescAddr) != OK) {
            return A_MEMORY_NOT_AVAIL;
        }
#else
        buffPhysAddr = pdevInfo->physDecryptBuf;
        pBuffVirtAddr = pdevInfo->virtDecryptBuf;
#endif

        switch(pKey->keyLength) {
        case 128:
            clearBuf = (A_UCHAR *)pBuffVirtAddr + hdrLen + ivLen;
            micBuf = (void *)(fp + payLoadLen);
            r = ocb_aes_decrypt((keystruct *)pKey->initValue, &nonce,
                                fp, (A_UINT16)payLoadLen, clearBuf, micBuf);
            if (r) {
                /* copy the header to the clear buffer */
                A_BCOPY(pCipherDesc->pBufferVirtPtr.ptr, pBuffVirtAddr,
                        hdrLen + ivLen);

                /* free the cipher buffer */
#ifdef BUILD_AP
                hwFreePhysBuffer (pdevInfo,
                                  pCipherDesc->pOrigBufferVirtPtr,
                                  pCipherDesc->pOSDescPtr);
#else
                pdevInfo->physDecryptBuf = pCipherDesc->pOrigbufferPhysPtr;
                pdevInfo->virtDecryptBuf = pCipherDesc->pOrigBufferVirtPtr;
                pOSDescAddr = 0;
#endif

                /* link the clear buffer to the descriptor */
                pCipherDesc->bufferPhysPtr = pCipherDesc->pOrigbufferPhysPtr = buffPhysAddr;
                pCipherDesc->pBufferVirtPtr.ptr = pBuffVirtAddr;
                pCipherDesc->pOrigBufferVirtPtr = pBuffVirtAddr;
#ifdef BUILD_AP
                pCipherDesc->pOSDescPtr = pOSDescAddr;
#endif
            } else {
#ifdef BUILD_AP
                /* free the clear buffer */
                hwFreePhysBuffer (pdevInfo, pBuffVirtAddr, pOSDescAddr);
#else
                // couldn't decrypt; no need to swap ptrs
#endif
#ifdef DEBUG
                uiPrintf("swDeCipher: AES decipher failed for ");
                printMacAddress(pWlanHdr->address2);
                uiPrintf("\n");
#endif
            }

            break;
        default:
#ifdef DEBUG
            uiPrintf("swDecipher: unsupported AES key length for ");
            printMacAddress(pWlanHdr->address2);
            uiPrintf("\n");
#endif
            break;
        }

        break;

    case PRIV_KEY_TYPE_AES_CCM:
        pCipherBuf = pCipherDesc->pBufferVirtPtr.byte;

        cryptoEndianChange(pCipherDesc->pBufferVirtPtr.header, CPU2LE);
        r = (ccm_decipher(pKey, pCipherBuf, (frameLen - FCS_FIELD_SIZE))
             == A_OK) ? 1 : 0;
        cryptoEndianChange(pCipherDesc->pBufferVirtPtr.header, LE2CPU);

        break;

    default:
#ifdef DEBUG
        uiPrintf("swDecipher: unsupported key type for ");
        printMacAddress(pWlanHdr->address2);
        uiPrintf("\n");
#endif
        break;
    }

    if (r) {
        return A_OK;
    }
    return A_DECRYPT_ERROR;
}

/*
 *  This is intended to be called on an MSDU basis right now.
 *  Do not free ppDesc on failure.  Caller will do it.
 */
A_STATUS
swEnmic(WLAN_DEV_INFO *pDevInfo, WLAN_PRIV_RECORD *pKey, ATHEROS_DESC **ppDesc)
{
    ATHEROS_DESC *pDesc = *ppDesc;
    A_STATUS      ret = A_OK;

    switch (pKey->keyType) {
    case PRIV_KEY_TYPE_WEP:
    case PRIV_KEY_TYPE_CKIP:
        if (pKey->keyFlags & PRIV_CKIP_MIC) {
            if (pDesc->more) {
                /*
                 * XXX Divy.
                 * Why does the previous code require a single Tx desc ?
                 */
                if ((ret = singleFlatBufCopy(pDevInfo, &pDesc)) != A_OK) {
                    /* Calling routine does the buffer free */
                    return ret;
                }
            }
#ifndef BUILD_AP
            /* frameLength/bufferLength already include space for the MIC */
            ret = ckip_enmic(pKey, pDesc->pBufferVirtPtr.byte,
                              pDesc->bufferLength);
#endif
        }
        break;

    case PRIV_KEY_TYPE_TKIP:
    case PRIV_KEY_TYPE_TKIP_SW:
        pDesc->frameLength += TKIP_MIC_FIELD_SIZE;

        if (!micSupported(pDevInfo, pKey->keyType) ||
            ((!pDesc->ffFlag) &&
             (pDesc->frameLength >= pDevInfo->staConfig.userFragThreshold)))
        {
            if (pDesc->more) {
                /*  This will be done for AP or if we're fraggin */
                if ((ret = singleFlatBufCopy(pDevInfo, &pDesc)) != A_OK) {
                    /* Calling routine does the buffer free */
                    return ret;
                }
            }

            /*
             *  It's important to incr bufferLen _after_ the coalescing
             *  is done so that we're operating on a single bufferLength,
             *  instead of the first of a chain.
             */
            pDesc->bufferLength += TKIP_MIC_FIELD_SIZE;

            ret = tkipEnmic(pKey, pDesc->pBufferVirtPtr.byte,
                            pDesc->bufferLength);
        }
        break;

    default:
        break;
    }

    *ppDesc = pDesc;
    return ret;
}

static A_STATUS
singleFlatBufCopy(WLAN_DEV_INFO *pDevInfo, ATHEROS_DESC **pClearDesc)
{
    ATHEROS_DESC *pDesc = *pClearDesc, *pFlatDesc = NULL, *pLoopDesc,
        *pLastDesc = NULL;
    A_UINT32     bufLen = 0;

    if (pClearDesc == NULL || *pClearDesc == NULL) {
        return A_ERROR;
    }

    /* boogie if single buf already */
    if (!pDesc->more) {
        return A_OK;
    }

    if (createBuffandDesc(pDevInfo, AR_BUF_SIZE, &pFlatDesc) != A_OK) {
        return A_NO_MEMORY;
    }

    pLoopDesc = pDesc;
    do {
        A_BCOPY(pLoopDesc->pBufferVirtPtr.byte,
                pFlatDesc->pBufferVirtPtr.byte + bufLen,
                pLoopDesc->bufferLength);
        bufLen += pLoopDesc->bufferLength;
        pLastDesc = pLoopDesc;
        pLoopDesc = pLoopDesc->pNextVirtPtr;
    } while (pLastDesc->more);

    /* copy over useful fields from pDesc to pFlatDesc */
    A_DESC_COPY(pDesc, pFlatDesc);

    pFlatDesc->frameLength  = pDesc->frameLength;
    pFlatDesc->bufferLength = bufLen;

    pFlatDesc->pNextVirtPtr   = pLastDesc->pNextVirtPtr;

    ATH_OSPKTREF_DUP(pDesc->pNextVirtPtr, pFlatDesc);

    freeBuffandDescChain(pDevInfo, pDesc);

    *pClearDesc = pFlatDesc;

    return A_OK;
}

A_STATUS
swDemic(WLAN_DEV_INFO *pDevInfo, ATHEROS_DESC *pDesc, WLAN_MACADDR *src,
        WLAN_MACADDR *dest, void *qosInfo)
{
    WLAN_PRIV_RECORD *pKey = pDesc->pKey;
    A_STATUS         ret = A_OK;
    WLAN_STATS       *pStats = &pDevInfo->localSta->stats;
    SIB_ENTRY        *pSib = pDesc->pSrcSibEntry;
#ifdef DEBUG
    A_BOOL           poisonMic = FALSE;
#else
    const A_BOOL     poisonMic = FALSE;
#endif

#ifdef DEBUG
    if (micFailNow()) {
        poisonMic = TRUE;
        pDesc->status.rx.decryptError |= DECRYPTERROR_MIC;
        ret = A_ERROR;
    }
#endif

    ASSERT(pSib);

    ASSERT(pKey);
    if (pKey == NULL) {
        return A_ERROR;
    }

    switch (pKey->keyType) {
    case PRIV_KEY_TYPE_WEP:
    case PRIV_KEY_TYPE_WEP_COMP:
    case PRIV_KEY_TYPE_CKIP:
#ifndef BUILD_AP
        if (pKey->keyFlags & PRIV_CKIP_MIC) {
            ASSERT(pDesc->more == 0);
            ret = ckip_demic(pKey, pDesc->pBufferVirtPtr.byte,
                             pDesc->frameLength, src, dest);
            if (ret != A_OK) {
                pStats->RcvDeMMHErrors++;
                pSib->stats.RcvDeMMHErrors++;
            }
        }
#endif
        break;

    case PRIV_KEY_TYPE_TKIP:
    case PRIV_KEY_TYPE_TKIP_SW:
        if (poisonMic == FALSE &&
            (!micSupported(pDevInfo, pKey->keyType) ||
             (pDesc->status.rx.decryptError & DECRYPTERROR_MIC)))
        {
            ret = tkipDemic(pKey, pDesc->pBufferVirtPtr.byte,
                            pDesc->frameLength, src, dest, qosInfo);
        }

        if (ret == A_OK) {
#ifdef DEBUG
            if (pDesc->status.rx.decryptError & DECRYPTERROR_MIC) {
                uiPrintf("WAR 6903 triggered.  Framelen = %d\n",
                         pDesc->status.rx.dataLength);
            }
#endif
            pDesc->status.rx.decryptError &= ~DECRYPTERROR_MIC;
        } else {
            pDesc->status.rx.decryptError |= DECRYPTERROR_MIC;
            pStats->RcvDemicErrors++;
            pSib->stats.RcvDemicErrors++;

#ifdef DEBUG
            printMacAddress(*src);
            uiPrintf(": MIC Error! Framelen = %d\n",
                     pDesc->status.rx.dataLength);
#endif /* DEBUG */

            A_TASK_LOCK(); // lock out dot1xTask
            micErrorProcess(pDevInfo, pDesc, MICERR_RXFRAME);
            A_TASK_UNLOCK();

            break;
        }

        /* Discard the MIC */
        pDesc->frameLength -= TKIP_MIC_FIELD_SIZE;
        pDesc->bufferLength -= TKIP_MIC_FIELD_SIZE;
        break;

    default:
        break;
    }

    return ret;
}

/* enable/disable local station's hw encryption engine */

void
setupSwEncryption(WLAN_DEV_INFO *pdevInfo)
{
    SIB_ENTRY *lSib = pdevInfo->localSta;

    /*
     *  Normal case -- enable the hardware encryption engine
     */
    lSib->swEncryptionEnabled = FALSE;
    lSib->swMicEnabled        = FALSE;

    if (lSib->serviceType == WLAN_STA_SERVICE) {
        /* Init the sw encryption state */
        if (((IS_CIPHER_AES_CCM(lSib->uCipher) ||
              IS_CIPHER_AES_CCM(lSib->mCipher)) &&
             !cipherSupported(pdevInfo, PRIV_KEY_TYPE_AES_CCM)))
        {
            lSib->swEncryptionEnabled = TRUE;

        } else if (lSib->ckipEnabled &&
            !cipherSupported(pdevInfo, PRIV_KEY_TYPE_CKIP))
        {
            lSib->swEncryptionEnabled = TRUE;

        } else if ((IS_CIPHER_TKIP(lSib->uCipher) ||
                    IS_CIPHER_TKIP(lSib->mCipher)) &&
                   !cipherSupported(pdevInfo, PRIV_KEY_TYPE_TKIP))
        {
            lSib->swEncryptionEnabled = TRUE;
        }

        /* Init the sw mic state */
        if (lSib->ckipUseMic &&
            !micSupported(pdevInfo, PRIV_KEY_TYPE_CKIP))
        {
            lSib->swMicEnabled = TRUE;

        } else if ((IS_CIPHER_TKIP(lSib->uCipher) ||
            IS_CIPHER_TKIP(lSib->mCipher)) &&
          !micSupported(pdevInfo, PRIV_KEY_TYPE_TKIP))
        {
            lSib->swMicEnabled = TRUE;
        }
   }

    if (lSib->serviceType == WLAN_AP_SERVICE) {
        if (((pdevInfo->staConfig.encryptionAlg == ENCRYPTION_AES_CCM) ||
            (pdevInfo->staConfig.encryptionAlg == ENCRYPTION_AUTO)))
        {
            /* Crete AES support requires HW encryption engine disabled */
            if (!cipherSupported(pdevInfo, PRIV_KEY_TYPE_AES_OCB)) {
                lSib->swEncryptionEnabled = TRUE;
            }
        }
    }
}

void
keyPrivateInit(WLAN_PRIV_RECORD *pKey)
{
    switch (pKey->keyType) {
    case PRIV_KEY_TYPE_WEP:
    case PRIV_KEY_TYPE_WEP_COMP:
        if (pKey->keyFlags & PRIV_CKIP_MIC) {
            ckip_setkey(pKey);
        }
        break;

    case PRIV_KEY_TYPE_AES_CCM:
        ccm_init(pKey);
        break;

    case PRIV_KEY_TYPE_TKIP:
    case PRIV_KEY_TYPE_TKIP_SW:
        if (tkipIVSet(pKey, NULL, TKIP_REKEY) != A_OK) {
            uiPrintf("Could not alloc mem for tkip rekey\n");
            return;
        }
        break;

    case PRIV_KEY_TYPE_CKIP:
        ckip_setkey(pKey);
        break;

    default:
        break;
    }
}

/*
 *  This gets used in the transmit path.
 *  We fill in the IV and Extended IV parts of the frame.
 *  The encaps routine will have already setup the EXTIV bit, if needed.
 */
void
frameIVSetup(WLAN_DEV_INFO *pDevInfo, WLAN_PRIV_RECORD *pKey,
             WLAN_FRAME_HEADER *pWlanHdr)
{
    A_UINT32    iv;
    A_UINT32    ivMask, ivShift = 0;
    A_UINT32    extBit = 0;
    A_UINT32    *pHdrIv, *pHdrExtIv;

    pHdrIv      = (A_UINT32 *)(GETIVPTR(pWlanHdr));
    pHdrExtIv   = (A_UINT32 *)(GETEXTIVPTR(pWlanHdr));

    /* Increment (randomize) the per key IV */
    iv = pKey->keyIV + 1;

    switch (pKey->keyType) {
    case PRIV_KEY_TYPE_AES_CCM:
        ivMask  = 0x0000ffff;               /* 16 bit IV, 32 bit ext IV */
        ivShift = 16;

        /* if lower IV DWORD overflowed */
        if ((iv & ivMask) == 0) {
            pKey->extKeyIV++;
        }
        *pHdrExtIv = pKey->extKeyIV;
        pKey->keyIV = iv;
        *pHdrIv = (iv & ivMask) | EXT_IV_BIT |
            ((pKey->keyFlags & PRIV_FRAMEKEY) << 30);
        break;

    case PRIV_KEY_TYPE_AES_OCB:
        ivMask = 0x0fffffff;                /* 28 bit IV */
        ivShift = 4;
        pKey->keyIV = iv;
        /*
         *  The IV is xmitted from lsb to msb, so we byteswap it
         *  so the presentation on a sniffer is msb to lsb.
         */
#ifdef BIG_ENDIAN
        *pHdrIv = (iv & ivMask) |
            ((pKey->keyFlags & PRIV_FRAMEKEY) << 30) | extBit;
#else
        *pHdrIv = ((cpu2be32(iv & ivMask) >> ivShift) & ivMask) |
            ((pKey->keyFlags & PRIV_FRAMEKEY) << 30) | extBit;
#endif
        break;

    case PRIV_KEY_TYPE_TKIP:
    case PRIV_KEY_TYPE_TKIP_SW:
        ivMask  = 0x00ffffff;               /* 16 bit IV, 32 bit ext IV */
        ivShift = 8;          /* but we split the IV16 across another byte */
        extBit = EXT_IV_BIT;

        /* if lower IV overflowed */
        if ((iv & 0xFFFF) == 0) {
            pKey->extKeyIV++;
        }
        pKey->keyIV = iv;
        *pHdrExtIv = pKey->extKeyIV;

        /* this is done in wlanEncaps() */
        ASSERT(*pHdrIv & EXT_IV_BIT);
        tkipIVSet(pKey, (A_UINT8 *)pHdrIv, TKIP_IV_UPDATE);

        /*
         *  tkipIVSet() has formed the IV in LE order.  We want to
         *  use our same code in frameEndianChange() which expects
         *  the IV in host order, so we swap here.  XXX we should
         *  really just fix tkipIVSet() to put the IV into host order.
         */
        *pHdrIv     = cpu2le32(*pHdrIv);
        *pHdrExtIv  = cpu2le32(*pHdrExtIv);
        break;

    default:  // WEP
        ivMask = 0x00ffffff;                /* 24 bit IV */
        ivShift = 8;

        /*
         * Filter keys that are weak according to "Weaknesses in the
         * Key Scheduling Algorithm of RC4" by Scott Fluhrer, Itsik
         * Martin and Adi Shimir.  We do not filter other "weak"
         * keys, so this is only a patch.
         *
         * Form: any, 0xff, 0x3-0x8, 0xf, 0x13 (40, 104, 128b keys)
         */
        if (((iv & 0xffff) >= 0xff03) &&
            ((iv & 0xffff) <= (0xff03 + ((A_UINT32)
                                         pKey->keyLength >> 3))))
        {
            iv = (iv & 0xffff00) | (4 + ((A_UINT32)pKey->keyLength >> 3));
        }
        pKey->keyIV = iv;
        /*
         *  The IV is xmitted from lsb to msb, so we byteswap it
         *  so the presentation on a sniffer is msb to lsb.
         */
        /* XXX Divy. Predator target is big endian */
        *pHdrIv = (iv & ivMask) |
            ((pKey->keyFlags & PRIV_FRAMEKEY) << 30) | extBit;
        break;
    }

    return;
}

/*
 *  This routine is used in the receive path to verify that a frame
 *  is not being replayed, by an attacker or otherwise.  As usual,
 *  we don't simply cast to a WLAN_FRAME_3_DATA_ENCRYPT_EXTIV * since
 *  we have to acct for the payload alignment reqd by Atheros hw.
 *  Frame is assumed to be in host order by now, and contiguous.
 */
A_STATUS
wlanReplayCheck(WLAN_DEV_INFO *pDev, WLAN_PRIV_RECORD *pKey,
                   ATHEROS_DESC *pDesc)
{
    A_INT64 pktNum = 0;
    A_UINT32 iv = *(A_UINT32 *)(pDesc->pBufferVirtPtr.byte +
                                headerLengthGet(pDesc));
    /* extIv may not be used by some algorithms */
    A_UINT32 extIv = *(A_UINT32 *)(pDesc->pBufferVirtPtr.byte +
                                   headerLengthGet(pDesc) + sizeof(iv));

    switch (pKey->keyType) {

    case PRIV_KEY_TYPE_AES_CCM:
        pktNum = extIv << 16;
        pktNum |= (iv & 0x0000ffff);

        if (pktNum <= pKey->keyRSC) {
#ifdef DEBUG
            if (replayDebug) {
                WLAN_DATA_MAC_HEADER *pWlanHdr;

                pWlanHdr = pDesc->pBufferVirtPtr.header;
                printMacAddress(pWlanHdr->address2);
                uiPrintf(" replayCheck failed: obs %d, exp %d (or more)\n",
                     (A_UINT32)pktNum, (A_UINT32)pKey->keyRSC);
            }
#endif
            return A_ERROR;
        }
        pKey->keyRSC = pktNum;
        break;

    case PRIV_KEY_TYPE_TKIP:
    case PRIV_KEY_TYPE_TKIP_SW:

        /* Take all of extIv, make room for iv16 */
        pktNum = extIv << 16;

        /*
         *  IV is arranged as A B C D where
         *  A = KeyID bits
         *  B = LSB of IV16
         *  C = FMS protection
         *  D = MSB of IV16
         */
        pktNum |= ((iv & 0x00ff0000) >> 16);
        pktNum |= ((iv & 0x000000ff) <<  8);

        if (pktNum <= pKey->keyRSC) {
#ifdef DEBUG
            if (replayDebug) {
                WLAN_DATA_MAC_HEADER *pWlanHdr;

                pWlanHdr = pDesc->pBufferVirtPtr.header;
                printMacAddress(pWlanHdr->address2);
                uiPrintf(" replayCheck failed: obs %d, exp %d (or more)\n",
                     (A_UINT32)pktNum, (A_UINT32)pKey->keyRSC);
            }
#endif
            return A_ERROR;
        }

        pKey->keyRSC = pktNum;
        break;

    default:
        break;
    }
    return A_OK;
}

/*
 * Processes the tkip mic errors and executes countermeasures
 * if neccesary.  This function gets called for every packet.
 * The error parameter will be set to the type of mic error,
 * otherwise it will be set to zero.  Function needs to be called
 * for every packet in order to reset the mic error counter if
 * we have NOT received mic errors in MIC_COUNTERMEASURE_TIMEOUT.
 * Returns 0 if countermeasures were not neccesary.
 * Returns 1 if countermeasures were performed.
 */

int
micErrorProcess(WLAN_DEV_INFO *pdevInfo, ATHEROS_DESC *pDesc,
                MIC_ERROR_TYPE error)
{
    A_UINT32 i;
    int cmeasures;
    A_UINT32 ticks;
#ifndef BUILD_AP
    A_UINT32 indicationType;
    WLAN_FRAME_HEADER *pHeader = pDesc->pBufferVirtPtr.header;

    ASSERT(error != MICERR_KEYMSG);

    if (error) {
        indicationType = isGrp(&pHeader->address1) ?
            NDIS_802_11_AUTH_REQUEST_GROUP_ERROR :
            NDIS_802_11_AUTH_REQUEST_PAIRWISE_ERROR;

        osIndicateAuth(pdevInfo, indicationType, &pdevInfo->baseBss.bssId);
        uiPrintf("Sent indicateAuth MIC error %x framelen %d\n",
                 indicationType,
                 pDesc->status.rx.dataLength);

    }
#endif

    ticks = A_MS_TICKGET();
    if (!error) {
        if (pdevInfo->lastMicError && ((ticks - pdevInfo->lastMicError) >
            MIC_COUNTERMEASURE_TIMEOUT))
        {
            pdevInfo->lastMicError = 0;
        }
        return (0);
    }

    /*
     *  Check a MIC Failure (or report) occurred within 60 seconds
     *  of the previous one. Second check handles the wrap around case.
     */
    if (pdevInfo->lastMicError &&
        (((ticks - pdevInfo->lastMicError) < MIC_COUNTERMEASURE_TIMEOUT) ||
        ((ticks + pdevInfo->lastMicError) < MIC_COUNTERMEASURE_TIMEOUT)))
    {
        uiPrintf("Got 2nd MIC error %s within 60sec at %d\n",
                 ((error == MICERR_KEYMSG) ?  "key message" : "in rx frame"),
                 ticks);

        if (pdevInfo->localSta->serviceType == WLAN_AP_SERVICE) {
            for (i = 0; i < pdevInfo->pSibTable->sibTableSize; i++) {
                SIB_ENTRY *tSib = &pdevInfo->pSibTable->sta[i];
                if (tSib->staState & STATE_ASSOC) {
#ifdef DEBUG
                    uiPrintf("Sending deauth to STA: ");
                    printMacAddress(tSib->macAddr);
                    uiPrintf("\n");
#endif
                    wlanMlmeDeauthRequest(
                        pdevInfo,
                        &pdevInfo->baseBss,
                        &tSib->macAddr,
                        REASON_MIC_FAILURE,
                        TRUE);
                }
            }
        }
        uiPrintf("Countermeasures now activated at %d\n", ticks);

        if (pdevInfo->trafficBlockTimestamp == 0) {
            pdevInfo->trafficBlockTimestamp = ticks;
        }
        cmeasures = 1;
    } else {
        uiPrintf("Got first MIC error %s at %d\n",
                 ((error == MICERR_KEYMSG) ? "key message" : "in rx frame"),
                 ticks);

        pdevInfo->lastMicError = ticks;
        cmeasures = 0;
    }

    return cmeasures;
}
