/*
 *  Copyright  2003 Atheros Communications, Inc.,  All Rights Reserved.
 */

static const char athId[] __attribute__ ((unused)) = "$Id: //depot/sw/branches/1.3_USB_LINUX_port/src/USB/wlan/host/common/crypto/aes_keywrap.c#1 $";

/*
 * aes_keywrap.c
 *
 * This file is non-optimized implementation of the AES Keywrap algorithm as
 * specified in RFC-3394.
 * The following description is taken from RFC-3394.
 *
 * The following notation is used in the description of the key wrapping
 * algorithms:
 *
 *    AES(K, W)     Encrypt W using the AES codebook with key K
 *    AES-1(K, W)   Decrypt W using the AES codebook with key K
 *    MSB(j, W)     Return the most significant j bits of W
 *    LSB(j, W)     Return the least significant j bits of W
 *    B1 ^ B2       The bitwise exclusive or (XOR) of B1 and B2
 *    B1 | B2       Concatenate B1 and B2
 *    K             The key-encryption key K
 *    n             The number of 64-bit key data blocks
 *    s             The number of steps in the wrapping process, s = 6n
 *    P[i]          The ith plaintext key data block
 *    C[i]          The ith ciphertext data block
 *    A             The 64-bit integrity check register
 *    R[i]          An array of 64-bit registers where
 *                     i = 0, 1, 2, ..., n
 *    A[t], R[i][t] The contents of registers A and R[i] after encryption
 *                     step t.
 *    IV            The 64-bit initial value used during the wrapping
 *                     process.
 *
 * In the key wrap algorithm, the concatenation function will be used to
 * concatenate 64-bit quantities to form the 128-bit input to the AES
 * codebook.  The extraction functions will be used to split the 128-bit
 * output from the AES codebook into two 64-bit quantities.
 *
 * Inputs:  Plaintext, n 64-bit values {P1, P2, ..., Pn}, and
 *          Key, K (the KEK).
 * Outputs: Ciphertext, (n+1) 64-bit values {C0, C1, ..., Cn}.
 *
 * 1) Initialize variables.
 *
 *     Set A = IV, an initial value (see 2.2.3)
 *     For i = 1 to n
 *         R[i] = P[i]
 *
 * 2) Calculate intermediate values.
 *
 *     For j = 0 to 5
 *         For i=1 to n
 *             B = AES(K, A | R[i])
 *             A = MSB(64, B) ^ t where t = (n*j)+i
 *             R[i] = LSB(64, B)
 *
 * 3) Output the results.
 *
 *     Set C[0] = A
 *     For i = 1 to n
 *         C[i] = R[i]
 *
 * A standalone mode is supported for ease of algorithm verification.
 * While in the src directory do:
 * gcc -g -DSTANDALONE -Iinclude -Iinclude/crypto -L/usr/lib -lc -o aeskw
 *      common/crypto/aes_keywrap.c common/crypto/rijndael-alg-fst.c
 */
#ifdef STANDALONE
#include <stdio.h>
#include <string.h>
typedef char                    A_CHAR;
typedef unsigned char           A_UCHAR;
typedef A_CHAR                  A_INT8;
typedef A_UCHAR                 A_UINT8;
typedef short                   A_INT16;
typedef unsigned short          A_UINT16;
typedef int                     A_INT32;
typedef unsigned int            A_UINT32;
typedef unsigned int            A_UINT;
typedef A_UCHAR                 A_BOOL;
typedef unsigned long long      A_UINT64;
#define A_BCOPY(x,y,z)          bcopy(x,y,z)
#define A_BCOMP(x,y,z)          bcmp(x,y,z)
#include "wlantype.h"
#else
#include "wlandrv.h"
#endif /* STANDALONE */

#include "crypto/rijndael-alg-fst.h"


#define AESKW_IV        0xa6a6a6a6a6a6a6a6ULL

/*
 * Prototypes
 */
static void aeskw_xor64(A_UINT8 *srcA, A_UINT8 *srcB, A_UINT8 *dst);

/*
 *  Performs aes keywrap on src and returns cipher on dest.
 *  buflen is length in bytes of input buffer and should be a multiple
 *      of 64 bits.
 *  dest is the output buffer and should be 8 bytes longer than the input
 *      buffer
 *  keylen is length of key in bytes
 *
 */

A_STATUS
aes_keywrap_encrypt(char *dest, const char *src, unsigned long buflen,
                    const char *keyptr, unsigned long keylen)
{
        A_UINT64 xort;
        A_UINT64 iv = AESKW_IV;
        A_UINT8 aes_data[16];
        int i, j, n, rounds;
        A_UINT32 aes_rk[80];    // aes key schedule

        if ((buflen & 0x7) || (keylen & 0x7)) {
            /*
             * buflen or keylen are not a multiple of 64 bits
             */
            return (A_EINVAL);
        }

        n = buflen >> 3;        // n is number of 64-bit data blocks
        if (n < 2) {
            /*
             * AES Keywrap is specified for a minimum of n=2
             */
            return (A_EINVAL);
        }

        keylen = keylen << 3;
        rounds = rijndaelKeySetupEnc(aes_rk, keyptr, keylen);

        A_BCOPY(src, dest + sizeof(A_UINT64), buflen);
        A_BCOPY(&iv, aes_data, sizeof(A_UINT64));

        for (j=0; j < 6; j++) {
            for (i=1; i <= n; i++) {
                A_BCOPY(dest + (sizeof(A_UINT64) * i),
                        aes_data + sizeof(A_UINT64), sizeof(A_UINT64));
                rijndaelEncrypt(aes_rk, rounds, aes_data, aes_data);
                xort = cpu2be64(((A_UINT64)n * j) + i);
                aeskw_xor64(aes_data, (A_UINT8 *)&xort, aes_data);
                A_BCOPY(aes_data + sizeof(A_UINT64),
                        dest + (sizeof(A_UINT64) * i), sizeof(A_UINT64));
            }
        }
        A_BCOPY(aes_data, dest, sizeof(A_UINT64));

        return (A_OK);
}

/*
 *  Performs aes key unwrap on cipher src and returns clear text on dest.
 *  buflen is length in bytes of input buffer and should be a multiple
 *      of 64 bits.
 *  dest is the output buffer and will be 8 bytes shorter than src.
 *  keylen is length of key in bytes.
 */
A_STATUS
aes_keywrap_decrypt(char *dest, const char *src, unsigned long buflen,
                    const char *keyptr, unsigned long keylen)
{
        A_UINT64 xort;
        A_UINT8 aes_data[16];
        int i, j, n, rounds;
        A_UINT32 aes_rk[80];    // aes key schedule
        int status = A_OK;
        A_UINT64 iv = AESKW_IV;

        if ((buflen & 0x7) || (keylen & 0x7)) {
            /*
             * buflen or keylen are not a multiple of 64 bits
             */
            return (A_EINVAL);
        }

        buflen -= sizeof (A_UINT8);     // get size of clear data
        n = (buflen >> 3);   // n is number of 64-bit data blocks
        if (n < 2) {
            /*
             * AES Keywrap is specified for a minimum of n=2
             */
            return (A_EINVAL);
        }

        keylen = keylen << 3;
        rounds = rijndaelKeySetupDec(aes_rk, keyptr, keylen);

        A_BCOPY(src, aes_data, sizeof(A_UINT64));
        A_BCOPY(src + sizeof(A_UINT64), dest, buflen);

        for (j=5; j >= 0; j--) {
            for (i=n; i > 0; i--) {
                xort = cpu2be64(((A_UINT64)n * j) + i);
                aeskw_xor64(aes_data, (A_UINT8 *)&xort, aes_data);
                A_BCOPY(dest + (sizeof(A_UINT64) * (i-1)),
                        aes_data + sizeof(A_UINT64), sizeof(A_UINT64));
                rijndaelDecrypt(aes_rk, rounds, aes_data, aes_data);
                A_BCOPY(aes_data + sizeof(A_UINT64),
                       dest + (sizeof(A_UINT64) * (i-1)), sizeof(A_UINT64));
            }
        }
        if (A_BCOMP(aes_data, &iv, sizeof(A_UINT64)) != 0) {
                status = A_ERROR;
        }

        return (status);
}

/*
 * Performs an xor over a 64 bit value without any alignment restrictions
 * XXX Available somewhere else?
 */
static void
aeskw_xor64(A_UINT8 *srcA, A_UINT8 *srcB, A_UINT8 *dst)
{
    dst[0] = srcA[0] ^ srcB[0];
    dst[1] = srcA[1] ^ srcB[1];
    dst[2] = srcA[2] ^ srcB[2];
    dst[3] = srcA[3] ^ srcB[3];
    dst[4] = srcA[4] ^ srcB[4];
    dst[5] = srcA[5] ^ srcB[5];
    dst[6] = srcA[6] ^ srcB[6];
    dst[7] = srcA[7] ^ srcB[7];
}


#ifdef STANDALONE
/*
 * This routine verifies the AES Keywrap implementation using the
 * test vectors provided in RFC-3394.  It is meant to run on a UNIX
 * environment.
 */
int
main()
{
    int testnum = 1;
    int i, n, error;
    char cypherout[512], clearout[512];
    u_int *outputhex;
    struct aeskw_test {
        int akt_datalen;                // bytes of data
        int akt_keylen;                 // bytes of keylen
        char akt_key[64];
        char akt_input[64];
        char akt_cypher[64];
    } aktTestDefinition[] = {
        /* 128 bits of Key Data with 128 bits of KEK */
        {
        16,
        16,
        {0x00,0x01,0x02,0x03,0x04,0x05,0x06,0x07,0x08,0x09,0x0a,0x0b,0x0c,
            0x0d,0x0e,0x0f},
        {0x00,0x11,0x22,0x33,0x44,0x55,0x66,0x77,0x88,0x99,0xaa,0xbb,0xcc,
            0xdd,0xee,0xff},
        {0x1f,0xa6,0x8b,0x0a,0x81,0x12,0xb4,0x47,0xae,0xf3,0x4b,0xd8,0xfb,
            0x5a,0x7b,0x82,0x9d,0x3e,0x86,0x23,0x71,0xd2,0xcf,0xe5},
        },
        /* 128 bits of Key Data with 192 bits of KEK */
        {
         16,
         24,
        {0x00,0x01,0x02,0x03,0x04,0x05,0x06,0x07,0x08,0x09,0x0a,0x0b,0x0c,
            0x0d,0x0e,0x0f,0x10,0x11,0x12,0x13,0x14,0x15,0x16,0x17},
        {0x00,0x11,0x22,0x33,0x44,0x55,0x66,0x77,0x88,0x99,0xaa,0xbb,0xcc,
            0xdd,0xee,0xff},
        {0x96,0x77,0x8b,0x25,0xae,0x6c,0xa4,0x35,0xf9,0x2b,0x5b,0x97,
            0xc0,0x50,0xae,0xd2,0x46,0x8a,0xb8,0xa1,0x7a,0xd8,0x4e,0x5d},
        },
        /* 128 bits of Key Data with 256 bits of KEK */
        {16,
         32,
        {0x00,0x01,0x02,0x03,0x04,0x05,0x06,0x07,0x08,0x09,0x0a,0x0b,0x0c,
            0x0d,0x0e,0x0f,0x10,0x11,0x12,0x13,0x14,0x15,0x16,0x17,0x18,
            0x19,0x1a,0x1b,0x1c,0x1d,0x1e,0x1f},
        {0x00,0x11,0x22,0x33,0x44,0x55,0x66,0x77,0x88,0x99,0xaa,0xbb,0xcc,
            0xdd,0xee,0xff},
        {0x64,0xe8,0xc3,0xf9,0xce,0x0f,0x5b,0xa2,0x63,0xe9,0x77,0x79,
            0x05,0x81,0x8a,0x2a,0x93,0xc8,0x19,0x1e,0x7d,0x6e,0x8a,0xe7},
        },
        /* 192 bits of Key Data with 192 bits of KEK */
        {24,
         24,
        {0x00,0x01,0x02,0x03,0x04,0x05,0x06,0x07,0x08,0x09,0x0a,0x0b,0x0c,
            0x0d,0x0e,0x0f,0x10,0x11,0x12,0x13,0x14,0x15,0x16,0x17},
        {0x00,0x11,0x22,0x33,0x44,0x55,0x66,0x77,0x88,0x99,0xaa,0xbb,0xcc,
            0xdd,0xee,0xff,0x00,0x01,0x02,0x03,0x04,0x05,0x06,0x07},
        {0x03,0x1d,0x33,0x26,0x4e,0x15,0xd3,0x32,0x68,0xf2,0x4e,0xc2,
            0x60,0x74,0x3e,0xdc,0xe1,0xc6,0xc7,0xdd,0xee,0x72,0x5a,0x93,
            0x6b,0xa8,0x14,0x91,0x5c,0x67,0x62,0xd2},
        },
        {0},
    };
    struct aeskw_test *kw_test;

    for (testnum=0; aktTestDefinition[testnum].akt_datalen != 0; testnum++) {
        kw_test = &aktTestDefinition[testnum];

        printf("\n");
        bzero(cypherout, sizeof (cypherout));
        /*
         * Encrypt Test
         */
        error = aes_keywrap_encrypt(cypherout, kw_test->akt_input,
                    kw_test->akt_datalen, kw_test->akt_key,
                    kw_test->akt_keylen);
        if (error != A_OK) {
            printf("AES Key Wrap Test #%d failed with return code %d\n",
                       testnum, error);
            continue;

        } else {
            printf("AES Key Wrap Test #%d cipher data:\n", testnum);
            outputhex = (u_int *)cypherout;
            for (i = 0; i < (kw_test->akt_datalen >> 2) + 2; i++) {
                printf("0x%.8x ", *outputhex);
                outputhex++;
            }
            printf("\n");
            error = memcmp(cypherout, kw_test->akt_cypher,
                          kw_test->akt_datalen + 8);
            if (!error) {
                printf("AES Key wrap Test #%d passed\n", testnum);
            } else {
                printf("AES Key Wrap Test #%d failed\n", testnum);
            }
        }

        /*
         * Decrypt Test
         */
        bzero(clearout, sizeof (clearout));
        if (error == A_OK) {
            error = aes_keywrap_decrypt(clearout, cypherout,
                    kw_test->akt_datalen + sizeof (A_UINT64),
                    kw_test->akt_key, kw_test->akt_keylen);
        }
        if (error != A_OK) {
            printf("AES Key UnWrap Test #%d failed with return code %d\n",
                       testnum, error);
        } else {
            printf("AES Key UnWrap Test #%d returned with clear data:\n",
                   testnum);
            outputhex = (u_int *)clearout;
            for (i = 0; i < (kw_test->akt_datalen >> 2); i++) {
                printf("0x%.8x ", *outputhex);
                outputhex++;
            }
            printf("\n");
            error = memcmp(clearout, kw_test->akt_input,
                          kw_test->akt_datalen);
            if (!error) {
                printf("AES Key UnWrap Test #%d passed\n", testnum);
            } else {
                printf("AES Key UnWrap Test #%d failed\n", testnum);
            }
        }
    }

    return (0);
}

#endif /* STANDALONE */
