/*
 *  Copyright (c) 2002 Atheros Communications Inc.  All rights reserved
 */

#ifdef STANDALONE
#include <stddef.h>
#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#define STATIC  static
#define INLINE
#define ASSERT(x)       assert(x)
#include "assert.h"
#include "ckip.h"
#else /* STANDALONE */
#define dumpbytes(a,b,c)
#define mpx(a,b,c)
#define STATIC
#endif /* STANDALONE */

#define nonalignedbuffers
#if defined(VXWORKS) || defined(PCI_NDIS) || defined (Linux)
#include "crypto/swcrypto_common.h"
#include "wlanproto.h"
#include "wlanext.h"
#include "wlanframe.h"
#include "ui.h"
#else
#include "swcrypto_common.h"
#include "wlanproto.h"
#endif

#ifndef STANDALONE
/* Ckip private state and interfaces */
#include "crypto/ckip.h"
#ifdef VERBOSE
extern void drvMpx(unsigned char *, int);
#endif
#endif


static void crc32_init();
extern void Crc32Init();
extern int WepDecipher();
extern void WepEncipher();

#define FASTERWEP

/*===========================================================================*/
/*=========== SUPPORT ROUTINES ==============================================*/
/*===========================================================================*/

#ifdef STANDALONE
void
dumpbytes(char *header, const void *pb, int len)
{
    cu8 *pc;
    int i;

    fprintf(stdout, "%s", header);
    pc = pb;
    for (i=0; i<len; i++) {
        fprintf(stdout, "%02X ", *pc++);
    }
    fprintf(stdout, "\n");
}
#endif

/*
 * Initialize the crc32 table and wep key vector.
 * Called once per boot
 *
 */
void
ckip_internal_init()
{
    crc32_init();                       // internal crc32
    Crc32Init();                        // external (fast) crc32
}

/*===========================================================================*/
/*=================== WEP ICV ===============================================*/
/*===========================================================================*/
#define CRC_TABSZ         256         /* no of entries in table */
u32 ct_crctab[CRC_TABSZ];
#define CRC_32            0xedb88320L /* CRC-32 polynomial */

static union {
    u32 lg;
} crc;

static void
crc32_init(void)
{
    u32 i, j, item, accum;

    for (i = 0;  i < CRC_TABSZ;  i++) {
        for (accum = 0, item = i << 1, j = 8;  j > 0;  j--) {
            item >>= 1;
            if ((item ^ accum) & 0x0001) {
                accum = (accum >> 1) ^ CRC_32;
            } else {
                accum >>= 1;
            }
        }
        ct_crctab[i] = accum;
    }
    crc.lg = 0xffffffff;
}

static u32
crc32_get(cu8 *buf, int size)
{
    int i;

    for (i = 0;  i < size;  i++) {
        crc.lg = ct_crctab[(crc.lg ^ buf[i]) & 0xff] ^
            ((crc.lg >> 8) & 0xFFFFFFL);
    }
    crc.lg = ~crc.lg;

    return crc.lg;
}

static void
wep_calculateICV(cu8 *buf, int n, u8 *icv)
{
    u32 crc32;

    //crc32_init();
    crc.lg = 0xffffffff;
    crc32 = crc32_get(buf, n);

    icv[0] = (crc32 >>  0) & 0xFF;
    icv[1] = (crc32 >>  8) & 0xFF;
    icv[2] = (crc32 >> 16) & 0xFF;
    icv[3] = (crc32 >> 24) & 0xFF;
}

/*===========================================================================*/
/*=================== WEP CIPHER=============================================*/
/*===========================================================================*/
/* basic 802.11 wep key initialization routine and en/decipher               */

typedef struct {
    u8 S[256];
    u8 i;
    u8 j;
} wep_context;

#define swap_byte(x,y) { u8 t; t = *(x); *(x) = *(y); *(y) = t; }

static void
wep_keyinit(wep_context *ctx, cu8 *key, int keylen)
{
    register u8  i, j, k;
    u8           *S;

    ctx->i = 0;
    ctx->j = 0;

    S = &ctx->S[0];

    i = 0;
    do {
        S[i] = i;
    } while (++i != 0);

    i = j = k = 0;
    do {
        j += S[i] + key[k];
        swap_byte(&S[i], &S[j]);
        if (++k >= keylen) {
            k = 0;
        }
    } while (++i != 0);
}

static void
wep_cipher(wep_context *ctx, u8 *buf, int buflen)
{
    u8 i,j;
    u8 *S;

    i = ctx->i;
    j = ctx->j;
    S = &ctx->S[0];

    while (buflen--) {
        i++;
        j += S[i];
        swap_byte(&S[i], &S[j]);
        *buf ^= S[(u8)(S[i] + S[j])];
        buf++;
    }

    ctx->i = i;
    ctx->j = j;
}

static void
CKIP_key_permute(
    u8  *PK,           /* output permuted key */
    cu8 *CK,           /* input CKIP key */
    u8  toDsFromDs,    /* input toDs/FromDs bits */
    cu8 *piv           /* input pointer to IV */
)
{
    int i;
    u16 H[2], tmp;          /* H=32-bits of per-packet hash value */
    u16 L[8], R[8];         /* L=u16 array of CK, R=u16 array of PK */

//#define DEBUGKEYMIX
#ifdef DEBUGKEYMIX
mpx("CK in\t", CK, 16);
printf("DS(%x) \n", toDsFromDs);
mpx("piv\t", piv, 16);
#elif defined(VERBOSE)
 uiPrintf("CK in\t"); drvMpx((u8 *)CK, 16);
 uiPrintf("DS(%x)\n", toDsFromDs);
 uiPrintf("piv\t"); drvMpx((u8 *)piv, 16);
#endif

    /* build L from input key */
    memset(L, 0, sizeof(L));
    for (i = 0; i < 16; i++) {
        L[i >> 1] |= ( ((u16)(CK[i])) << ( i & 1 ? 8 : 0) );
    }

    H[0] = (((u16)piv[0]) << 8) + piv[1];
    H[1] = ( ((u16)toDsFromDs) << 8) | piv[2];

    for (i = 0; i < 8; i++) {
        H[0] ^= L[i];           /* 16-bits of key material */
        tmp   = _S_(H[0]);      /* 16x16 permutation */
        H[0]  = tmp ^ H[1];     /* set up for next round */
        H[1]  = tmp;
        R[i]  = H[0];           /* store into key array  */
    }

    /* sweep in the other direction */
    tmp=L[0];
    for (i = 7; i > 0; i--) {
        R[i] = tmp = rotLeft_1(tmp) + R[i];
    }

    /* IV of the permuted key is unchanged */
    PK[0] = piv[0];
    PK[1] = piv[1];
    PK[2] = piv[2];

    /* key portion of the permuted key is changed */
    for (i = 3; i < 16; i++) {
        PK[i] = (u8)(R[i >> 1] >> (i & 1 ? 8 : 0));
    }

#ifdef DEBUGKEYMIX
mpx("CK out\t", PK, 16);
#elif defined(VERBOSE)
 uiPrintf("CK out\t"); drvMpx(PK, 16);
#endif
}    

/*===========================================================================*/
/*=================== CKIP MIC MMH FUNCTION =================================*/
/*===========================================================================*/

#ifdef STANDALONE
void
debugmic(mic_context *cp)
{
    printf("[%d]\n", cp->position);
    mpx("coeff\t= ", cp->coefficient, 16);
    mpx("accum\t= ", &cp->accum, 8);
}
#endif

/*===========================================================================*/
/*========== CKIP MIC MMH COEFFICIENTS ======================================*/
/*===========================================================================*/

/* AES returns 16-bytes of coefficient which are used in 4-byte (u32) units */
/* aes_encrypt() is used to generate the MMH coefficients in counter mode    */
/* aes_encrypt(K,IN,OUT) -- applies aes to IN using key K, generating OUT    */
/* K: input key, IN: input data to encrypt, OUT: output encrypted data       */
/* all of the parameters K, IN, OUT are 16-byte in length                    */


void static
aes_encrypt(unsigned char *key, unsigned char *input, unsigned char *output)
{
    unsigned int rk[44];
    int          r;

    r = rijndaelKeySetupEnc(rk, key, 128);

    /* XXX should check value of r */

    rijndaelEncrypt(rk, 10, input, output);
}

void
aes_precompute(mic_context *cp)
{
    u8  aes_counter[16];
    u32 bytep, counter;
    int r, rk[44];

    r = rijndaelKeySetupEnc(rk, &cp->CK[0], 128);

    /* XXX should check value of r */

    for (bytep = 0; bytep < sizeof(cp->aes_precomp); bytep += 16) {
        memset(&aes_counter[0], 0, sizeof(aes_counter));
        counter = bytep >> 4;

        aes_counter[15] = (u8)(counter >> 0);
        aes_counter[14] = (u8)(counter >> 8);
        aes_counter[13] = (u8)(counter >> 16);
        aes_counter[12] = (u8)(counter >> 24);

        //aes_encrypt(&cp->CK[0], &aes_counter[0], &cp->aes_precomp[bytep]);
        rijndaelEncrypt(rk, 10, &aes_counter[0], &cp->aes_precomp[bytep]);
    }
}

// MIC_GETCOEFF replaces the subroutine call mic_getcoefficient(),
// using precomputed aes coefficients
//#define       MIC_GETCOEFF(cp)    (GETBIG32(&cp->aes_precomp[(cp->position-1)& ~3]))
#define MIC_GETCOEFF(aes, cp)    (GETBIG32(&aes[(cp->position - 1) & ~3]))

static u32
mic_getcoefficient(mic_context *context)
{
    u8   aes_counter[16];
    int  coeff_position;
    u8   *p;
    u8   *q, *qq;

    coeff_position = (context->position - 1) >> 2;
    if ((coeff_position & 3) == 0) {
        /*
         * fetching the first coefficient
         * -- get new 16-byte aes counter output
         */
        u32 counter = (coeff_position >> 2);

        /* new counter value */
        memset(&aes_counter[0], 0, sizeof(aes_counter));
        aes_counter[15] = (u8)(counter >> 0);
        aes_counter[14] = (u8)(counter >> 8);
        aes_counter[13] = (u8)(counter >> 16);
        aes_counter[12] = (u8)(counter >> 24);

        aes_encrypt(&context->CK[0], &aes_counter[0], context->coefficient);
    }

    p = &(context->coefficient[(coeff_position & 3) << 2]);

#ifdef TESTPRECOMP
    q = &context->aes_precomp[(context->position-1)& ~3];
    qq = (u8 *)MIC_GETCOEFF(context);
    printf("[%d] p %x  q %x  qq  %x\n",context->position,  *p, *q, *qq);
#endif

    return GETBIG32(p);
}

/*===========================================================================*/
/*========== MIC MMH ROUTINES  ==============================================*/
/*===========================================================================*/

/* prepare for calculation of a new mic */
void
mic_init(mic_context *context, const u8 *CK)
{
    /* prepare for new mic calculation */
    context->accum = 0;
    context->position = 0;
}

/* add some bytes to the mic calculation */
void
mic_update(mic_context *context, const u8 *pOctets, int len, u8 *aes)
{
    int byte_position;
    u32 val;
    u64 val64;

    byte_position = context->position & 3;
    while (len > 0) {
        /* build a 32-bit word for MIC multiply accumulate */
        do {
            if (len == 0) {
                return;
            }
            context->part[byte_position++] = *pOctets++;
            context->position++;
            len--;
        } while (byte_position < 4);

        /* have a full 32-bit word to process */
        val = GETBIG32(&context->part[0]);
        //MIC_ACCUM(val);
        val64 = (u64)val;
        FAST_MIC_ACCUM(val64, context, aes);

        byte_position = 0;
    }
}

/* calculate the mic */
void
mic_final(mic_context *context, u8 digest[4], u8 *aes)
{
    int byte_position;
    u32 val;
    u64 sum, utmp;
    s64 stmp;
    u64 val64;

    /* deal with partial 32-bit word left over from last update */
    if ((byte_position = (context->position & 3)) != 0) {
        /* have a partial word in part to deal with -- zero unused bytes */
        do {
            context->part[byte_position++] = 0;
            context->position++;
        } while (byte_position < 4);
        val = GETBIG32(&context->part[0]);
        //MIC_ACCUM(val);
        val64 = (u64)val;
        FAST_MIC_ACCUM(val64, context, aes);
    }
    
    /* reduce the accumulated u64 to a 32-bit MIC */
    sum = context->accum;

    stmp = (sum  & 0x00000000ffffffff) - ((sum >> 32)  * 15);

    utmp = (stmp & 0x00000000ffffffff) - ((stmp >> 32) * 15);

    sum  = utmp & 0x00000000ffffffff;

    if (utmp > 0x0000000010000000f) {
        sum -= 15;
    }

    val = (u32)sum;

    digest[0] = (val >> 24) & 0xFF;
    digest[1] = (val >> 16) & 0xFF;
    digest[2] = (val >> 8) & 0xFF;
    digest[3] = (val >> 0) & 0xFF;
}

/*===========================================================================*/
/*=================== CKIP MIC PACKET CALCULATION ===========================*/
/*===========================================================================*/


static void
ckip_mic(struct ckip_private *ckip, cu8 *pDA, cu8 *pSA,
         int seq, u8 *payload, int payloadlen)
{
    u8       *pSEQ, *pMIC, *pOrigPayload, *aes;
    A_UINT16 etherLen;
    int      OrigPayloadLen;

    ASSERT(ckip->valid_aes_precomp);

    aes = (u8 *)ckip->ctx.aes_precomp;

    etherLen       = cpu2be16(payloadlen);
    pMIC           = payload + (sizeof(mic_snap));
    pSEQ           = pMIC + 4;
    pOrigPayload   = pSEQ + 4;
    OrigPayloadLen = payloadlen - (sizeof(mic_snap) + 4 + 4);
    *(u32 *)pSEQ   = cpu2be32(seq);

#ifdef BUILDMIC
    printf("CKIP_MIC parameters\n");
    printf("seq(%d)\n", seq);
    mpx("DA\t", pDA, 6);
    mpx("SA\t", pSA, 6);
    mpx("payload\n", payload, payloadlen);
    mpx("aes pre\n", aes, 16);
#elif defined(VERBOSE)
    uiPrintf("CKIP_MIC parameters\n");
    uiPrintf("DA ="); drvMpx((u8 *)pDA, 6);
    uiPrintf("SA ="); drvMpx((u8 *)pSA, 6);
    uiPrintf("etherLen ="); drvMpx((u8 *)&etherLen, 2);
    uiPrintf("*pSEQ = %08X\n", *(u32 *)pSEQ);
    uiPrintf("payload\n"); drvMpx(payload, payloadlen);
    uiPrintf("aes ="); drvMpx((u8 *)aes, 16);
#endif

    mic_init(&ckip->ctx, ckip->ctx.CK);   /* initialize for calculating the mic */
    mic_update(&ckip->ctx, pDA, WLAN_MAC_ADDR_SIZE, aes); /* Mic <-- Destination address (DA) */
    mic_update(&ckip->ctx, pSA, WLAN_MAC_ADDR_SIZE, aes); /* Mic <-- Source address (SA) */
    mic_update(&ckip->ctx, (A_UINT8 *)&etherLen, sizeof(etherLen), aes); /* Mic <-- Length field (big endian) */
    mic_update(&ckip->ctx, payload, sizeof(mic_snap), aes);    /* MIC <-- snap header */
    mic_update(&ckip->ctx, pSEQ, sizeof(seq), aes);            /* Mic <-- SEQ field */
    mic_update(&ckip->ctx, pOrigPayload, OrigPayloadLen, aes); /* Mic <-- original payload */

    mic_final(&ckip->ctx, pMIC, aes);                       /* Calculate/Store the MIC */

#ifdef STANDALONE
    mpx("final mic\t", pMIC, 4);
#elif defined(VERBOSE)
    uiPrintf("final MIC"); drvMpx((u8 *)pMIC, 4);
    uiPrintf("payload"); drvMpx(payload, payloadlen);
#endif
}

/*
 * Check the mic on a received packet
 * Returns A_ERROR or A_OK
 * Caller is responsible for seq number checking
 */
static int
ckip_chkmic(struct ckip_private *ckip, cu8 *pDA, cu8 *pSA,
            u8 *payload, int payloadlen)
{
    u8       *pSEQ, *pMIC, *pOrigPayload, *aes;
    int      OrigPayloadLen;
    u8       calcmic[4];
    A_UINT16 etherLen;

    ASSERT(ckip->valid_aes_precomp);
    aes            = (u8 *)ckip->ctx.aes_precomp;
    pMIC           = payload + (sizeof(mic_snap));
    pSEQ           = pMIC + 4;
    pOrigPayload   = pSEQ + 4;
    OrigPayloadLen = payloadlen - (sizeof(mic_snap) + 4 + 4);
    etherLen       = cpu2be16(payloadlen);

#ifdef CHKMIC
    printf("CKIP_MIC parameters\n");
    mpx("DA\t", pDA, 6);
    mpx("SA\t", pSA, 6);
    mpx("CK\t", CK, 16);
    mpx("payload\n", payload, payloadlen);
#elif defined(VERBOSE)
    uiPrintf("ChkMIC: DA"); drvMpx((u8 *)pDA, 6);
    uiPrintf("SA"); drvMpx((u8 *)pSA, 6);
    uiPrintf("CKIP key"); drvMpx(ckip->ctx.CK, 16);
    uiPrintf("Initial AES key:"); drvMpx(aes, 16);
    uiPrintf("Frame to demic:"); drvMpx(payload, payloadlen);
#endif

compute_mic:

    /* initialize for calculating the mic */
    mic_init(&ckip->ctx, ckip->ctx.CK);
    /* Mic <-- Destination address (DA) */
    mic_update(&ckip->ctx, pDA, WLAN_MAC_ADDR_SIZE, aes);
    /* Mic <-- Source address (SA) */
    mic_update(&ckip->ctx, pSA, WLAN_MAC_ADDR_SIZE, aes);
    /* Mic <-- Length field (big endian) */
    mic_update(&ckip->ctx, (u8 *)&etherLen, sizeof(etherLen), aes);
    /* MIC <-- snap header */
    mic_update(&ckip->ctx, payload, sizeof(mic_snap), aes);
    /* Mic <-- SEQ field */
    mic_update(&ckip->ctx, pSEQ, 4, aes);
    /* Mic <-- original payload */
    mic_update(&ckip->ctx, pOrigPayload, OrigPayloadLen, aes);

    /* Calculate/Store the MIC */
    mic_final(&ckip->ctx, calcmic, aes);
    if (A_BCOMP(pMIC, calcmic, 4) == 0) {
        return A_OK;
    }

    /*
     * Try a second time with saved key/aes on failure
     */
    if ((aes == (u8 *)ckip->ctx.aes_precomp) && ckip->valid_aes_savcomp) {
        aes = (u8 *)ckip->ctx.aes_savcomp;
        goto compute_mic;
    }

    return A_ERROR;
}

int
ckip_init(WLAN_PRIV_RECORD *pk)
{
    struct ckip_private *ckip;
    int r;

    if (!pk) {
        return A_ERROR;
    }

    ckip      = pk->initValue;

    if (!ckip) {
        ckip = (struct ckip_private *)A_DRIVER_MALLOC(sizeof(*ckip));
        if (!ckip) {
            return A_NO_MEMORY;
        }
        pk->initValue = (void *)ckip;
    }

    ckip->valid_aes_precomp = ckip->valid_aes_savcomp = 0;
    return A_OK;
}

void
ckip_close(WLAN_PRIV_RECORD *pk)
{
    struct ckip_private *ckip;

    if (!pk) {
        return;
    }

    ckip = pk->initValue;
    if (!ckip) {
        return;
    }

    A_DRIVER_FREE(ckip, sizeof(*ckip));
}

/*
 * setup ckip internal key by copying (and extending if needed)
 * the key contained in pk->keyVal.   If key is non-null,
 * it will also clobber whatever is in pk->keyVal with an
 * (assumed) 16-byte key.
 */
int
ckip_setkey(WLAN_PRIV_RECORD *pk)
{
    struct ckip_private *ckip;
    mic_context         *ctxp;
    int                 keyLen, r, i;

    ckip = pk->initValue;                   // get ckip private struct
    if (!ckip) {
        r = ckip_init(pk);
        if (r != A_OK) {
            return r;
        }
        ckip = pk->initValue;
    }

    // make (extended) copy of the keystruct key for ckip internal use.
    // funky, but allows ckip->CK to be longer than pk->keyVal
    // for the unusual case where the MIC is enabled, but KP is not.
    ctxp = &ckip->ctx;
    keyLen = pk->keyLength/8; /* Note:  keyLength is in bits */

    /*
     * copy incoming key into ckip private area
     */
    A_DRIVER_BCOPY(pk->keyVal, ctxp->CK, keyLen);

    /* expand to 16 bytes if necessary */
    for (i = 0; keyLen < MAX_KEY_LEN_BYTES; i++, keyLen++) {
        ctxp->CK[keyLen] = ctxp->CK[i];
    }

#ifdef STANDALONE
#ifdef SETKEYS
printf("SETKEYS\n");
mpx("WEP\t", pk->keyVal, pk->keyLength/8);
mpx("CKIP\t", ctxp->CK, 16);
#endif
#elif defined(VERBOSE)
 uiPrintf("SETKEYS\npKey\t"); drvMpx(pk->keyVal, pk->keyLength/8);
 uiPrintf("CKIP key:\t"); drvMpx(ctxp->CK, 16);
#endif

    // save current aes precomp if there is a valid one
    if (ckip->valid_aes_precomp) {
        A_DRIVER_BCOPY(ckip->ctx.aes_precomp, ckip->ctx.aes_savcomp,
		       MAX_KEY_LEN_BYTES);
        ckip->valid_aes_savcomp = 1;
    }

    /* precompute the aes MMH coefficients and mark the buffer valid */
    aes_precompute(ctxp);
    ckip->valid_aes_precomp = 1;

    // initialize the ckip SEQ numbers
    ckip->seq_uplink = 0x00000000;  // even for uplink packets to AP
    ckip->seq_dnlink = 0x00000001;  // odd for dnlink packets from AP

    return A_OK;
}

#ifdef STANDALONE
void
ckip_set_sequence(WLAN_PRIV_RECORD *pk, int up, int dn)
{
    struct ckip_private *ckip;

    ckip = pk->initValue;
    if (!ckip) {
        return;
    }

    ckip->seq_uplink = up;
    ckip->seq_dnlink = dn;
}
#endif

/*
 * external driver interface for ckip encipher
 */
int
ckip_encipher(WLAN_PRIV_RECORD *pk, u8 *pkt, int pktlen)
{
    u8 ds, *ecr, *pDA, *pSA;
    int hdrlen, ecrlen, r;
    struct ckip_private *ckip;
    u8 *pSEQ, *pMIC, *payload;
    u8 ckip_pk[16];

    ckip = pk->initValue;
    ASSERT(ckip != NULL);

    hdrlen = WLAN_HDR_SIZE + WEP_IV_FIELD_SIZE;

    switch (pkt[1] & 3) {
    case 0: /* FromDs=0, ToDs=0 */
        pDA = &pkt[4];
        pSA = &pkt[10];
        break;
    case 1: /* FromDs=0, ToDs=1 */
        pDA = &pkt[16];
        pSA = &pkt[10];
        break;
    case 2: /* FromDs=1, ToDs=0 */
        pDA = &pkt[4];
        pSA = &pkt[16];
        break;
    case 3: /* FromDs=1, ToDs=1 */
        hdrlen += WLAN_MAC_ADDR_SIZE;
        pDA = &pkt[16];
        pSA = &pkt[24];
        break;
    }

    hdrlen = A_ROUNDUP(hdrlen, sizeof(A_UINT32));

    ecr = pkt + hdrlen;
    /* Trim off hdr, iv, and fcs...does NOT include ICV since caller did not */
    ecrlen = pktlen - hdrlen;

#ifdef VERBOSE
    uiPrintf("CKIP encipher input:"); drvMpx(pkt, pktlen);
#endif

    /* 
     * keymix routine needs the iv (at ecr-4).
     * wep routine is given the mixed key (ckip_pk)
     * plus the exact bytes to wep (ecr, ecrlen).
     * ecrlen must not include the icv.
     * WepEncipher adds icv to the end of the buffer.
     */
    CKIP_key_permute(ckip_pk, ckip->ctx.CK, pkt[1] & 3,
                     ecr - WEP_IV_FIELD_SIZE);
    WepEncipher(ckip_pk, ecr, ecrlen);
    return A_OK;
}

int
ckip_decipher(WLAN_PRIV_RECORD *pk, u8 *pkt, int pktlen)
{
    u8 ds, *ecr, *pDA, *pSA;
    int hdrlen, ecrlen, r;
    struct ckip_private *ckip;
    u8 *pSEQ, *pMIC, *payload;
    u8 ckip_pk[16];

    ckip = pk->initValue;
    ASSERT(ckip != NULL);

    hdrlen = WLAN_HDR_SIZE + WEP_IV_FIELD_SIZE;
    switch (pkt[1] & 3) {
    case 0: /* FromDs=0, ToDs=0 */
        pDA = &pkt[4];
        pSA = &pkt[10];
        break;
    case 1: /* FromDs=0, ToDs=1 */
        pDA = &pkt[16];
        pSA = &pkt[10];
        break;
    case 2: /* FromDs=1, ToDs=0 */
        pDA = &pkt[4];
        pSA = &pkt[16];
        break;
    case 3: /* FromDs=1, ToDs=1 */
        hdrlen += WLAN_MAC_ADDR_SIZE;
        pDA = &pkt[16];
        pSA = &pkt[24];
        break;
    }

    hdrlen = A_ROUNDUP(hdrlen, sizeof(A_UINT32));

    ecr    = pkt + hdrlen;
    ecrlen = pktlen - hdrlen;

    /*
     * same calling conventions here as in ckip_encipher()
     */
    CKIP_key_permute(ckip_pk, ckip->ctx.CK, pkt[1] & 3,
                     ecr - WEP_IV_FIELD_SIZE);  
    if (WepDecipher(ckip_pk, ecr, ecrlen)) {
        return A_OK;
    } else {
        return A_ERROR;
    }
}

/*
 *  This interface is assumed to be called with all fields in a single buffer.
 *  The buffer will look like:
 *
 *   24    4     3       3        2      4     4      2
 * +-----+----+-----+---------+-------+-----+-----+-------+
 * |WLAN | IV | LLC | Aironet | Proto | MIC | SEQ | ether | Data...
 * |hdr  |    |     |  OUI    |       |     |     |  Type |
 * +-----+----+-----+---------+-------+-----+-----+-------+
 *
 */
/*
 *  pkt - ptr to start of WLAN header
 *  pktlen - length of entire frame, does NOT include ICV
 */
A_STATUS
ckip_enmic(WLAN_PRIV_RECORD *pk, u8 *pkt, int pktlen)
{
    struct ckip_private *ckip;
    u8   *ecr, *pDA, *pSA;
    int  hdrlen, ecrlen;

    ckip = pk->initValue;
    ASSERT(ckip != NULL);

    hdrlen = WLAN_HDR_SIZE + WEP_IV_FIELD_SIZE;

    switch (pkt[1] & 3) {
    case 0: /* FromDs=0, ToDs=0 */
        pDA = &pkt[4];
        pSA = &pkt[10];
        break;
    case 1: /* FromDs=0, ToDs=1 */
        pDA = &pkt[16];
        pSA = &pkt[10];
        break;
    case 2: /* FromDs=1, ToDs=0 */
        pDA = &pkt[4];
        pSA = &pkt[16];
        break;
    case 3: /* FromDs=1, ToDs=1 */
        hdrlen += WLAN_MAC_ADDR_SIZE;
        pDA = &pkt[16];
        pSA = &pkt[24];
        break;
    }

    hdrlen = A_ROUNDUP(hdrlen, sizeof(A_UINT32));
    ecr = pkt + hdrlen;                 /* point past IV */
    ecrlen = pktlen - hdrlen;           /* trim off hdr and IV */
    ckip_mic(ckip, pDA, pSA, ckip->seq_uplink, ecr, ecrlen);

    ckip->seq_uplink += 2;

    return A_OK;
}

/*
 *  pkt - ptr to start of WLAN header
 *  pktlen - length of whole frame, not including ICV (it was already stripped)
 */
int
ckip_demic(WLAN_PRIV_RECORD *pk, u8 *pkt, int pktlen,
           WLAN_MACADDR *src, WLAN_MACADDR *dest)
{
    struct ckip_private *ckip;
    u8   *ecr;
    int  hdrlen, ecrlen, r;

    ckip = pk->initValue;                   // get ckip private struct
    ASSERT(ckip != NULL);

    ASSERT(ckip->valid_aes_precomp);

    /* ecr points to the beginning of the LLC/SNAP hdr */
    ecr    = pkt;
    ecrlen = pktlen;

#ifdef VERBOSE
    uiPrintf("DEMIC this frame:"); drvMpx(pkt, pktlen);
    uiPrintf("WEP key:"); drvMpx(pk->keyVal, pk->keyLength/8);
#endif

    r = ckip_chkmic(ckip, (cu8 *)dest, (cu8 *)src, ecr, ecrlen);
    if (r == A_OK) {
        ckip->seq_dnlink =
            be2cpu32(*(u32 *)(ecr + sizeof(mic_snap) + sizeof(int)));
        /* XXX check for monotonicity */
    }

    return r;
}
