/*
 *  Copyright (c) 2002 Atheros Communications Inc.  All rights reserved
 */
#ifdef SELFTEST
#define STANDALONE
#endif

#ifdef STANDALONE
#include <stddef.h>
#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#define STATIC  static
#define INLINE
#define ASSERT(x)       assert(x)
#include <swcrypto_common.h>
#include "assert.h"
#include <tkip.h>
#include <wep.h>
#else /* STANDALONE */
#define dumpbytes(a,b,c)
#define mpx(a,b,c)
#define STATIC
#endif /* STANDALONE */

#define nonalignedbuffers

#if defined(VXWORKS) || defined(PCI_NDIS) || defined (Linux)
#include "crypto/swcrypto_common.h"
#include "crypto/tkip.h"
#include "crypto/wep.h"
#include "wlanproto.h"
#include "wlanext.h"
#include "wlanframe.h"
#include "ui.h"
#endif

#ifndef STANDALONE
#ifdef VERBOSE
extern void drvMpx(unsigned char *, int);
#endif
#endif

struct tkip_private {
    A_UINT8 p1k[10];
    A_UINT8 rc4key[16];
};

static A_UINT8 zero[4] = { 0,0,0,0 };           // zero word for non-qos michael encapsulation
static A_UINT8 qosh[4];

#define ICVLEN      4                   // wep icv
#define IVLEN       8                   // extended iv
#define MICLEN      8                   // michael mic


/*
 * tkip_init
 *      called automagically on the first entry to tkip encipher
 *      the tkip encipher call can be removed if tkip_init is called
 *      before tkip encipher is invoked.
 */
static A_STATUS
tkip_init(pk)
    WLAN_PRIV_RECORD *pk;
{
    struct tkip_private *tkip;

    // init IV counters
    pk->keyIV = pk->extKeyIV = 0;

    // malloc tkip private data if needed
    tkip = pk->initValue;
    if (!tkip) {
        tkip = (struct tkip_private *)
            A_DRIVER_MALLOC(sizeof(struct tkip_private));
        if (!tkip) {
            return A_NO_MEMORY;
        }
    }
    pk->initValue = (void *)tkip;

    return A_OK;
}

void
tkip_set_replay(pk, lo, hi)
    WLAN_PRIV_RECORD *pk;
{
    struct tkip_private *tkip;

    tkip = pk->initValue;
    pk->keyIV = lo;
    pk->extKeyIV = hi;
}

void
tkip_get_replay(pk, ivp)
    WLAN_PRIV_RECORD *pk;
    A_UINT8 *ivp;
{
    A_UINT16 iv16;
    A_UINT32 iv32;

    iv16 = 0;
    iv16 = (ivp[0]<<8) | (ivp[2]);
    iv32 = 0;
    iv32 |= (ivp[4]<<0);
    iv32 |= (ivp[5]<<8);
    iv32 |= (ivp[6]<<16);
    iv32 |= (ivp[7]<<24);
    pk->keyIV = iv16;
    pk->extKeyIV = iv32;
}


/*  This assumes the frame header is in host order. */
static A_UINT16
tkipHeaderLengthGet(WLAN_FRAME_3_DATA_ENCRYPT_EXTIV *pWlanHdr)
{
    FRAME_CONTROL *fc = &pWlanHdr->macHeader.frameControl;
    A_UINT16       hlen;

    hlen = 24;
    if (fc->ToDS && fc->FromDS) {
        hlen += WLAN_MAC_ADDR_SIZE;     // frame contains A4
    }
    if (fc->fSubtype & SUBT_QOS) {      // check for qos frame
        hlen += 2;
    }
    hlen = A_ROUNDUP(hlen, sizeof(A_UINT32));   // align for hw
    return hlen;
}

void 
tkip_close(pk)
    WLAN_PRIV_RECORD *pk;
{
    if (pk->initValue) {
        A_DRIVER_FREE((void *)pk->initValue, sizeof(struct tkip_private));
    }
}

/*
 * Set the IV for tx packets
 *      takes TSC from pk
 * Calculate cached P1K when rekeying
 *      controlled by op==TKIP_REKEY
 */
A_STATUS
tkipIVSet(WLAN_PRIV_RECORD *pk, A_UINT8 *ivp, TKIP_OP op)
{
    A_UINT32 iv16;
    A_UINT32 iv32;

    switch (op) {
    case TKIP_REKEY:
        return tkip_init(pk);
        break;

    case TKIP_IV_UPDATE:
        if (ivp) {
            iv16 = pk->keyIV;
            iv32 = pk->extKeyIV;

            ivp[0] = (A_UINT8)(iv16>>8);
            ivp[1] = (A_UINT8)(((iv16>>8) | 0x20) & 0x7f);
            ivp[2] = (A_UINT8)iv16;
            /* keyIndex | extIV flag */
            ivp[3] = (A_UINT8)(0x20 | ((pk->keyFlags & 0x3) << 6));

            ivp[4] = (A_UINT8)iv32;         
            ivp[5] = (A_UINT8)(iv32>>8);
            ivp[6] = (A_UINT8)(iv32>>16);
            ivp[7] = (A_UINT8)(iv32>>24);       // 32-bits of extended IV, little-endian
        }
    }

    return A_OK;
}

/*
 * extract Sequence Counter from iv
 * return as an int64 for replay countermeasures
 */
A_UINT64
tkipSeqCtrGet(A_UINT8 *ivp)
{
    A_UINT64 val;

    val = 0;
    val = ivp[7];
    val <<= 8;
    val |= ivp[6];
    val <<= 8;
    val |= ivp[5];
    val <<= 8;
    val |= ivp[4];
    val <<= 8;
    val |= ivp[0];
    val <<= 8;
    val |= ivp[2];

    return (val);
}

/*
 *  tkipEncipher API
 *  Frame is assumed to have its .11hdr and IV/EXTIV all filled in by now.
 *  pk     ptr to key struct
 *  mpdu   frame header
 *  pktlen length of frame including IV, MIC;  excluding ICV and FCS
*/
A_STATUS
tkipEncipher(WLAN_PRIV_RECORD *pk, A_UINT8 *mpdu, int pktlen)
{
    struct tkip_private *tk;            // tkip local data
    A_UINT16 iv16;                              // 16-bit iv16
    A_UINT32 iv32;                              // 32-bit iv32
    A_UINT8 *datap, *ivp;
    A_UINT32 hlen, dlen;
    WLAN_FRAME_3_DATA_ENCRYPT_EXTIV *pWlanHdr =
        (WLAN_FRAME_3_DATA_ENCRYPT_EXTIV *)mpdu;

    ASSERT(pk);                 // pk is non-null

    tk = (struct tkip_private *)pk->initValue;
    ASSERT(tk);

    hlen = tkipHeaderLengthGet(pWlanHdr);

    ivp = &mpdu[hlen];
    datap = ivp + IVLEN;
    dlen = pktlen - hlen - IVLEN;

    iv16 = (A_UINT16)pk->keyIV;
    iv32 = pk->extKeyIV;

    // call Phase1 when iv16 carries out (or always)
    Phase1(tk->p1k, pk->keyVal, mpdu+10, iv32);

    // Phase2 creates rc4key from keyval, p1k, and iv16
    Phase2(tk->rc4key, pk->keyVal, tk->p1k, iv16);

    /* adds ICV after dlen */
    WepEncipher(tk->rc4key, datap, (unsigned short)dlen);

    return(A_OK);
}

/*
 *  tkipDecipher API
 *  pk     pKey
 *  mpdu   frame 
 *  flen   frame length, excluding FCS
*/
A_STATUS
tkipDecipher(WLAN_PRIV_RECORD *pk, A_UINT8 *mpdu, A_UINT32 flen)
{
    struct tkip_private *tk;
    A_UINT16            iv16;
    A_UINT32            iv32;
    A_UINT8             *ivp, *datap;
    A_UINT32            hlen, dlen, tmp32;
    WLAN_FRAME_3_DATA_ENCRYPT_EXTIV *pWlanHdr =
        (WLAN_FRAME_3_DATA_ENCRYPT_EXTIV *)mpdu;

    ASSERT(pk);                 // assume pk is non-null

    tk = (struct tkip_private *)pk->initValue;
    ASSERT(tk);

    hlen  = tkipHeaderLengthGet(pWlanHdr);

    /* Don't use the pWlanHdr->iv since it may not line up */
    ivp   = &mpdu[hlen];        // calc start of iv field
    datap = ivp + IVLEN;
    dlen  = flen - hlen - IVLEN;

    /* Extract the IV and ExtIV from rx pkt */
    /* This copy should be 4-byte aligned given hlen, above */
    tmp32 = *(A_UINT32 *)&ivp[0];
    iv16  = (A_UINT16)(tmp32 & 0xFF) << 8;
    iv16 |= (tmp32 & 0x00FF0000) >> 16;

    iv32  = *(A_UINT32 *)&ivp[sizeof(tmp32)];

    // call Phase1 when iv16 carries out (or always)
    Phase1(tk->p1k, pk->keyVal, mpdu+10, iv32);

    // Phase2 creates rc4key from keyval, p1k, and iv16
    Phase2(tk->rc4key, pk->keyVal, tk->p1k, iv16);

    if (WepDecipher(tk->rc4key, datap, (unsigned short)dlen)) {
        return A_OK;
    } else {
        return A_ERROR;
    }
}

/*
 * mpdu - ptr to start of WLAN header
 * flen - length of frame excluding ICV and FCS
 */
A_STATUS
tkipEnmic(WLAN_PRIV_RECORD *pk, A_UINT8 *mpdu, A_UINT32 flen)
{
    WLAN_MACADDR *da, *sa;
    A_UINT8 mhdr[12];                   // local hdr input to michael
    A_UINT8 *ivp, *datap, *micp;
    A_UINT8 qos;
    A_UINT8 *qosp = zero;
    A_UINT32 hlen, dlen;
    WLAN_FRAME_3_DATA_ENCRYPT_EXTIV *pWlanHdr =
        (WLAN_FRAME_3_DATA_ENCRYPT_EXTIV *)mpdu;
    FRAME_CONTROL *fc = &pWlanHdr->macHeader.frameControl;

    ASSERT(pk);                 // assume pk is non-null

    hlen = tkipHeaderLengthGet(pWlanHdr);

    if (fc->fSubtype & SUBT_QOS) {      // check for qos frame
        qos = mpdu[hlen] & 0xf;           // 4-bits of qos TID
        qosh[0] = qos;
        qosp = qosh;
    }

    ivp   = &mpdu[hlen];                  // calc start of iv field
    datap = ivp + IVLEN;
    dlen  = flen - hlen - IVLEN - MICLEN;
    micp  = datap + dlen;                  // remember mic address

    if (! fc->ToDS && ! fc->FromDS) {
        da = &pWlanHdr->macHeader.address1;
        sa = &pWlanHdr->macHeader.address2;
    } else if (! fc->ToDS && fc->FromDS) {
        da = &pWlanHdr->macHeader.address1;
        sa = &pWlanHdr->macHeader.address3;
    } else if ( fc->ToDS && ! fc->FromDS) {
        da = &pWlanHdr->macHeader.address3;
        sa = &pWlanHdr->macHeader.address2;
    } else { /* ToDS = FromDS = 1 */
        da = &pWlanHdr->macHeader.address3;
        sa = &((WLAN_FRAME_DATA_ENCRYPT_EXTIV *)pWlanHdr)->macHeader.address4;
    }

    A_BCOPY(da, mhdr, WLAN_MAC_ADDR_SIZE);
    A_BCOPY(sa, &mhdr[WLAN_MAC_ADDR_SIZE], WLAN_MAC_ADDR_SIZE);

    //qosp = 0;
    michael(pk->micTxKeyVal, datap, dlen, mhdr, sizeof(mhdr), qosp);
    return A_OK;
}

/*
 * mpdu - ptr to start of WLAN header
 * flen - length of frame excluding ICV and FCS
 * Note:  Assumes fields are in host order!
 */
A_STATUS
tkipDemic(WLAN_PRIV_RECORD *pk, A_UINT8 *mpdu, A_UINT32 flen,
          WLAN_MACADDR *src, WLAN_MACADDR *dest, void *qosInfo)
{
    A_UINT8 smic[MICLEN];
    A_UINT8 mhdr[2*WLAN_MAC_ADDR_SIZE];  // local hdr input to michael
    A_UINT8 *micp;
    A_UINT32 dlen, r;

    ASSERT(pk);

    dlen = flen - MICLEN;
    micp = mpdu + dlen;

    /* build a mic-buffer */
    A_BCOPY(dest, mhdr, WLAN_MAC_ADDR_SIZE);
    A_BCOPY(src, &mhdr[WLAN_MAC_ADDR_SIZE], WLAN_MAC_ADDR_SIZE);

    /* remember recv'd mic */
    A_BCOPY(micp, smic, TKIP_MIC_FIELD_SIZE);

    michael(pk->micRxKeyVal, mpdu, dlen, mhdr, sizeof(mhdr), qosInfo);
    r = A_BCOMP(smic, micp, TKIP_MIC_FIELD_SIZE);

    /* Map the return type */
    if (r == 0) {
        return A_OK;
    } else {
        return A_ERROR;
    }
}

// XXX this just resolves the refs
int
tkip_unused()
{
    unsigned long a[1], b[1], c[1];
    xor_block(a, b);
    xor3_block(a, b, c);
    cmp_mic_err(a, b);
    return 0;
}
