/***********************************************************************

   Contents:    Generate 802.11 per-packet RC4 key hash test vectors
   Date:        April 19, 2002
   Authors:     Doug Whiting,   Hifn
                Russ Housley,   RSA Labs
                Niels Ferguson, MacFergus
                Doug Smith,     Cisco
   Notes:

   This code is released to the public domain use, built solely out of
   the goodness of our hearts for the benefit of all mankind. As such,
   there are no warranties of any kind given on the correctness or
   usefulness of this code.

   This code is written for pedagogical purposes, NOT for performance.

************************************************************************/

#ifdef TKIPSELFTEST
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>
#include <time.h>
#endif

#define	XWEP_LE	1
#define	ASSERT(p)	assert(p)

typedef unsigned char  byte;   /*  8-bit byte (octet)  */
typedef unsigned short u16b;   /* 16-bit unsigned word */
typedef unsigned long  u32b;   /* 32-bit unsigned word */

/* macros for extraction/creation of byte/u16b values  */
#define RotR1(v16)   ((((v16) >> 1) & 0x7FFF) ^ (((v16) & 1) << 15))
#define   Lo8(v16)   ((byte)( (v16)       & 0x00FF))
#define   Hi8(v16)   ((byte)(((v16) >> 8) & 0x00FF))
#define  Lo16(v32)   ((u16b)( (v32)       & 0xFFFF))
#define  Hi16(v32)   ((u16b)(((v32) >>16) & 0xFFFF))
#define  Mk16(hi,lo) ((lo) ^ (((u16b)(hi)) << 8))

/* select the Nth 16-bit word of the temporal key byte array TK[]   */
#define  TK16(N)     Mk16(TK[2*(N)+1],TK[2*(N)])

/* S-box lookup: 16 bits --> 16 bits */
#define _S_(v16)     (Sbox[0][Lo8(v16)] ^ Sbox[1][Hi8(v16)])

/* fixed algorithm "parameters" */
#define PHASE1_LOOP_CNT   8    /* this needs to be "big enough"     */
#define TA_SIZE           6    /*  48-bit transmitter address       */
#define TK_SIZE          16    /* 128-bit temporal key              */
#define P1K_SIZE         10    /*  80-bit Phase1 key                */
#define RC4_KEY_SIZE     16    /* 128-bit RC4KEY (104 bits unknown) */

/* configuration settings */
//#define DO_SANITY_CHECK   1    /* validate properties of S-box?     */


/* 2-byte by 2-byte subset of the full AES S-box table */
const u16b Sbox[2][256]=       /* Sbox for hash (can be in ROM)     */
{ {
   0xC6A5,0xF884,0xEE99,0xF68D,0xFF0D,0xD6BD,0xDEB1,0x9154,
   0x6050,0x0203,0xCEA9,0x567D,0xE719,0xB562,0x4DE6,0xEC9A,
   0x8F45,0x1F9D,0x8940,0xFA87,0xEF15,0xB2EB,0x8EC9,0xFB0B,
   0x41EC,0xB367,0x5FFD,0x45EA,0x23BF,0x53F7,0xE496,0x9B5B,
   0x75C2,0xE11C,0x3DAE,0x4C6A,0x6C5A,0x7E41,0xF502,0x834F,
   0x685C,0x51F4,0xD134,0xF908,0xE293,0xAB73,0x6253,0x2A3F,
   0x080C,0x9552,0x4665,0x9D5E,0x3028,0x37A1,0x0A0F,0x2FB5,
   0x0E09,0x2436,0x1B9B,0xDF3D,0xCD26,0x4E69,0x7FCD,0xEA9F,
   0x121B,0x1D9E,0x5874,0x342E,0x362D,0xDCB2,0xB4EE,0x5BFB,
   0xA4F6,0x764D,0xB761,0x7DCE,0x527B,0xDD3E,0x5E71,0x1397,
   0xA6F5,0xB968,0x0000,0xC12C,0x4060,0xE31F,0x79C8,0xB6ED,
   0xD4BE,0x8D46,0x67D9,0x724B,0x94DE,0x98D4,0xB0E8,0x854A,
   0xBB6B,0xC52A,0x4FE5,0xED16,0x86C5,0x9AD7,0x6655,0x1194,
   0x8ACF,0xE910,0x0406,0xFE81,0xA0F0,0x7844,0x25BA,0x4BE3,
   0xA2F3,0x5DFE,0x80C0,0x058A,0x3FAD,0x21BC,0x7048,0xF104,
   0x63DF,0x77C1,0xAF75,0x4263,0x2030,0xE51A,0xFD0E,0xBF6D,
   0x814C,0x1814,0x2635,0xC32F,0xBEE1,0x35A2,0x88CC,0x2E39,
   0x9357,0x55F2,0xFC82,0x7A47,0xC8AC,0xBAE7,0x322B,0xE695,
   0xC0A0,0x1998,0x9ED1,0xA37F,0x4466,0x547E,0x3BAB,0x0B83,
   0x8CCA,0xC729,0x6BD3,0x283C,0xA779,0xBCE2,0x161D,0xAD76,
   0xDB3B,0x6456,0x744E,0x141E,0x92DB,0x0C0A,0x486C,0xB8E4,
   0x9F5D,0xBD6E,0x43EF,0xC4A6,0x39A8,0x31A4,0xD337,0xF28B,
   0xD532,0x8B43,0x6E59,0xDAB7,0x018C,0xB164,0x9CD2,0x49E0,
   0xD8B4,0xACFA,0xF307,0xCF25,0xCAAF,0xF48E,0x47E9,0x1018,
   0x6FD5,0xF088,0x4A6F,0x5C72,0x3824,0x57F1,0x73C7,0x9751,
   0xCB23,0xA17C,0xE89C,0x3E21,0x96DD,0x61DC,0x0D86,0x0F85,
   0xE090,0x7C42,0x71C4,0xCCAA,0x90D8,0x0605,0xF701,0x1C12,
   0xC2A3,0x6A5F,0xAEF9,0x69D0,0x1791,0x9958,0x3A27,0x27B9,
   0xD938,0xEB13,0x2BB3,0x2233,0xD2BB,0xA970,0x0789,0x33A7,
   0x2DB6,0x3C22,0x1592,0xC920,0x8749,0xAAFF,0x5078,0xA57A,
   0x038F,0x59F8,0x0980,0x1A17,0x65DA,0xD731,0x84C6,0xD0B8,
   0x82C3,0x29B0,0x5A77,0x1E11,0x7BCB,0xA8FC,0x6DD6,0x2C3A,
  },


  {  /* second half of table is byte-reversed version of first! */
   0xA5C6,0x84F8,0x99EE,0x8DF6,0x0DFF,0xBDD6,0xB1DE,0x5491,
   0x5060,0x0302,0xA9CE,0x7D56,0x19E7,0x62B5,0xE64D,0x9AEC,
   0x458F,0x9D1F,0x4089,0x87FA,0x15EF,0xEBB2,0xC98E,0x0BFB,
   0xEC41,0x67B3,0xFD5F,0xEA45,0xBF23,0xF753,0x96E4,0x5B9B,
   0xC275,0x1CE1,0xAE3D,0x6A4C,0x5A6C,0x417E,0x02F5,0x4F83,
   0x5C68,0xF451,0x34D1,0x08F9,0x93E2,0x73AB,0x5362,0x3F2A,
   0x0C08,0x5295,0x6546,0x5E9D,0x2830,0xA137,0x0F0A,0xB52F,
   0x090E,0x3624,0x9B1B,0x3DDF,0x26CD,0x694E,0xCD7F,0x9FEA,
   0x1B12,0x9E1D,0x7458,0x2E34,0x2D36,0xB2DC,0xEEB4,0xFB5B,
   0xF6A4,0x4D76,0x61B7,0xCE7D,0x7B52,0x3EDD,0x715E,0x9713,
   0xF5A6,0x68B9,0x0000,0x2CC1,0x6040,0x1FE3,0xC879,0xEDB6,
   0xBED4,0x468D,0xD967,0x4B72,0xDE94,0xD498,0xE8B0,0x4A85,
   0x6BBB,0x2AC5,0xE54F,0x16ED,0xC586,0xD79A,0x5566,0x9411,
   0xCF8A,0x10E9,0x0604,0x81FE,0xF0A0,0x4478,0xBA25,0xE34B,
   0xF3A2,0xFE5D,0xC080,0x8A05,0xAD3F,0xBC21,0x4870,0x04F1,
   0xDF63,0xC177,0x75AF,0x6342,0x3020,0x1AE5,0x0EFD,0x6DBF,
   0x4C81,0x1418,0x3526,0x2FC3,0xE1BE,0xA235,0xCC88,0x392E,
   0x5793,0xF255,0x82FC,0x477A,0xACC8,0xE7BA,0x2B32,0x95E6,
   0xA0C0,0x9819,0xD19E,0x7FA3,0x6644,0x7E54,0xAB3B,0x830B,
   0xCA8C,0x29C7,0xD36B,0x3C28,0x79A7,0xE2BC,0x1D16,0x76AD,
   0x3BDB,0x5664,0x4E74,0x1E14,0xDB92,0x0A0C,0x6C48,0xE4B8,
   0x5D9F,0x6EBD,0xEF43,0xA6C4,0xA839,0xA431,0x37D3,0x8BF2,
   0x32D5,0x438B,0x596E,0xB7DA,0x8C01,0x64B1,0xD29C,0xE049,
   0xB4D8,0xFAAC,0x07F3,0x25CF,0xAFCA,0x8EF4,0xE947,0x1810,
   0xD56F,0x88F0,0x6F4A,0x725C,0x2438,0xF157,0xC773,0x5197,
   0x23CB,0x7CA1,0x9CE8,0x213E,0xDD96,0xDC61,0x860D,0x850F,
   0x90E0,0x427C,0xC471,0xAACC,0xD890,0x0506,0x01F7,0x121C,
   0xA3C2,0x5F6A,0xF9AE,0xD069,0x9117,0x5899,0x273A,0xB927,
   0x38D9,0x13EB,0xB32B,0x3322,0xBBD2,0x70A9,0x8907,0xA733,
   0xB62D,0x223C,0x9215,0x20C9,0x4987,0xFFAA,0x7850,0x7AA5,
   0x8F03,0xF859,0x8009,0x171A,0xDA65,0x31D7,0xC684,0xB8D0,
   0xC382,0xB029,0x775A,0x111E,0xCB7B,0xFCA8,0xD66D,0x3A2C,
  }
};


#if DO_SANITY_CHECK
#include <assert.h>
/*
**********************************************************************
* Routine: SanityCheckTable -- verify Sbox properties
*
* Inputs:  Sbox
* Output:  None, but an assertion fails if the tables are wrong
* Notes:
*   The purpose of this routine is solely to illustrate and
*   verify the following properties of the Sbox table:
*      - the Sbox is a "2x2" subset of the AES table:
*            Sbox + affine transform + MDS.
*      - the Sbox table can be easily designed to fit in a
*            512-byte table, using a byte swap
*      - the Sbox table can be easily designed to fit in a
*            256-byte table, using some shifts and a byte swap
**********************************************************************
*/
void SanityCheckTable(void)
    {
    const static int  M_x = 0x11B;   /* AES irreducible polynomial */
    const static byte Sbox8[256] = { /* AES 8-bit Sbox */
        0x63,0x7c,0x77,0x7b,0xf2,0x6b,0x6f,0xc5,
        0x30,0x01,0x67,0x2b,0xfe,0xd7,0xab,0x76,
        0xca,0x82,0xc9,0x7d,0xfa,0x59,0x47,0xf0,
        0xad,0xd4,0xa2,0xaf,0x9c,0xa4,0x72,0xc0,
        0xb7,0xfd,0x93,0x26,0x36,0x3f,0xf7,0xcc,
        0x34,0xa5,0xe5,0xf1,0x71,0xd8,0x31,0x15,
        0x04,0xc7,0x23,0xc3,0x18,0x96,0x05,0x9a,
        0x07,0x12,0x80,0xe2,0xeb,0x27,0xb2,0x75,
        0x09,0x83,0x2c,0x1a,0x1b,0x6e,0x5a,0xa0,
        0x52,0x3b,0xd6,0xb3,0x29,0xe3,0x2f,0x84,
        0x53,0xd1,0x00,0xed,0x20,0xfc,0xb1,0x5b,
        0x6a,0xcb,0xbe,0x39,0x4a,0x4c,0x58,0xcf,
        0xd0,0xef,0xaa,0xfb,0x43,0x4d,0x33,0x85,
        0x45,0xf9,0x02,0x7f,0x50,0x3c,0x9f,0xa8,
        0x51,0xa3,0x40,0x8f,0x92,0x9d,0x38,0xf5,
        0xbc,0xb6,0xda,0x21,0x10,0xff,0xf3,0xd2,
        0xcd,0x0c,0x13,0xec,0x5f,0x97,0x44,0x17,
        0xc4,0xa7,0x7e,0x3d,0x64,0x5d,0x19,0x73,
        0x60,0x81,0x4f,0xdc,0x22,0x2a,0x90,0x88,
        0x46,0xee,0xb8,0x14,0xde,0x5e,0x0b,0xdb,
        0xe0,0x32,0x3a,0x0a,0x49,0x06,0x24,0x5c,
        0xc2,0xd3,0xac,0x62,0x91,0x95,0xe4,0x79,
        0xe7,0xc8,0x37,0x6d,0x8d,0xd5,0x4e,0xa9,
        0x6c,0x56,0xf4,0xea,0x65,0x7a,0xae,0x08,
        0xba,0x78,0x25,0x2e,0x1c,0xa6,0xb4,0xc6,
        0xe8,0xdd,0x74,0x1f,0x4b,0xbd,0x8b,0x8a,
        0x70,0x3e,0xb5,0x66,0x48,0x03,0xf6,0x0e,
        0x61,0x35,0x57,0xb9,0x86,0xc1,0x1d,0x9e,
        0xe1,0xf8,0x98,0x11,0x69,0xd9,0x8e,0x94,
        0x9b,0x1e,0x87,0xe9,0xce,0x55,0x28,0xdf,
        0x8c,0xa1,0x89,0x0d,0xbf,0xe6,0x42,0x68,
        0x41,0x99,0x2d,0x0f,0xb0,0x54,0xbb,0x16 };


    int i,k,k2,k3;
    byte bitmap[0x2000];

    /* show that smaller tables can be used, if desired */
    for (i=0;i<256;i++)
        {
        k  = Sbox8[i];
        k2 = (k << 1) ^ ((k & 0x80) ? M_x : 0);
        k3 =  k ^ k2;
        ASSERT(Sbox[0][i] == ((k2 << 8) ^ k3));
        ASSERT(Sbox[1][i] == ((k3 << 8) ^ k2));
        }

    /* now make sure that it's a 16-bit permutation */
    memset(bitmap,0,sizeof(bitmap));
    for (i=0;i<0x10000;i++)
        {
        k = _S_(i); /* do an S-box lookup: 16 --> 16 bits */
        ASSERT(k < (1 << 16));
        ASSERT((bitmap[k >> 3] & (1 << (k & 7))) == 0);
        bitmap[k >> 3] |= 1 << (k & 7);
        }
    for (i=0;i<sizeof(bitmap);i++)
        ASSERT(bitmap[i] == 0xFF);

    /* if we reach here, the 16-bit Sbox is ok */
    //printf("Table sanity check successful\n");
    }
#endif


/*
**********************************************************************
* Routine: Phase 1 -- generate P1K, given TA, TK, IV32
*
* Inputs:
*     TK[]      = temporal key                         [128 bits]
*     TA[]      = transmitter's MAC address            [ 48 bits]
*     IV32      = upper 32 bits of IV                  [ 32 bits]
* Output:
*     P1K[]     = Phase 1 key                          [ 80 bits]
*
* Note:
*     This function only needs to be called every 2**16 packets,
*     although in theory it could be called every packet.
*
**********************************************************************
*/
void Phase1(u16b *P1K,const byte *TK,const byte *TA,u32b IV32)
    {
    int  i;

    /* Initialize the 80 bits of P1K[] from IV32 and TA[0..5]     */
    P1K[0]      = Lo16(IV32);
    P1K[1]      = Hi16(IV32);
    P1K[2]      = Mk16(TA[1],TA[0]); /* use TA[] as little-endian */
    P1K[3]      = Mk16(TA[3],TA[2]);
    P1K[4]      = Mk16(TA[5],TA[4]);

    /* Now compute an unbalanced Feistel cipher with 80-bit block */
    /* size on the 80-bit block P1K[], using the 128-bit key TK[] */
    for (i=0; i < PHASE1_LOOP_CNT ;i++)
        {                 /* Each add operation here is mod 2**16 */
        P1K[0] += _S_(P1K[4] ^ TK16((i&1)+0));
        P1K[1] += _S_(P1K[0] ^ TK16((i&1)+2));
        P1K[2] += _S_(P1K[1] ^ TK16((i&1)+4));
        P1K[3] += _S_(P1K[2] ^ TK16((i&1)+6));
        P1K[4] += _S_(P1K[3] ^ TK16((i&1)+0));
        P1K[4] +=  (u16b)i;                    /* avoid "slide attacks" */
        }
    }


/*
**********************************************************************
* Routine: Phase 2 -- generate RC4KEY, given TK, P1K, IV16
*
* Inputs:
*     TK[]      = Temporal key                         [128 bits]
*     P1K[]     = Phase 1 output key                   [ 80 bits]
*     IV16      = low 16 bits of IV counter            [ 16 bits]
* Output:
*     RC4KEY[]  = the key used to encrypt the packet   [128 bits]
*
* Note:
*     The value {TA,IV32,IV16} for Phase1/Phase2 must be unique
*     across all packets using the same key TK value. Then, for a
*     given value of TK[], this TKIP48 construction guarantees that
*     the final RC4KEY value is unique across all packets.
*
* Suggested implementation optimization: if PPK[] is "overlaid"
*     appropriately on RC4KEY[], there is no need for the final
*     for loop below that copies the PPK[] result into RC4KEY[].
*
**********************************************************************
*/
void Phase2(byte *RC4KEY,const byte *TK,const u16b *P1K,u16b IV16)
    {
    int  i;
    u16b PPK[6];                          /* temporary key for mixing    */

    /* Note: all adds in the PPK[] equations below are mod 2**16         */
    for (i=0;i<5;i++) PPK[i]=P1K[i];      /* first, copy P1K to PPK      */
    PPK[5]  =  P1K[4] + IV16;             /* next,  add in IV16          */

    /* Bijective non-linear mixing of the 96 bits of PPK[0..5]           */
    PPK[0] +=    _S_(PPK[5] ^ TK16(0));   /* Mix key in each "round"     */
    PPK[1] +=    _S_(PPK[0] ^ TK16(1));
    PPK[2] +=    _S_(PPK[1] ^ TK16(2));
    PPK[3] +=    _S_(PPK[2] ^ TK16(3));
    PPK[4] +=    _S_(PPK[3] ^ TK16(4));
    PPK[5] +=    _S_(PPK[4] ^ TK16(5));   /* Total # S-box lookups == 6  */

    /* Final sweep: bijective, "linear". Rotates kill LSB correlations   */
    PPK[0] +=  RotR1(PPK[5] ^ TK16(6));
    PPK[1] +=  RotR1(PPK[0] ^ TK16(7));   /* Use all of TK[] in Phase2   */
    PPK[2] +=  RotR1(PPK[1]);
    PPK[3] +=  RotR1(PPK[2]);
    PPK[4] +=  RotR1(PPK[3]);
    PPK[5] +=  RotR1(PPK[4]);
    /* Note: At this point, for a given key TK[0..15], the 96-bit output */
    /*       value PPK[0..5] is guaranteed to be unique, as a function   */
    /*       of the 96-bit "input" value   {TA,IV32,IV16}. That is, P1K  */
    /*       is now a keyed permutation of {TA,IV32,IV16}.               */

    /* Set RC4KEY[0..3], which includes "cleartext" portion of RC4 key   */
    RC4KEY[0] = Hi8(IV16);                /* RC4KEY[0..2] is the WEP IV  */
    RC4KEY[1] =(Hi8(IV16) | 0x20) & 0x7F; /* Help avoid weak (FMS) keys  */
    RC4KEY[2] = Lo8(IV16);
    RC4KEY[3] = Lo8((PPK[5] ^ TK16(0)) >> 1);


    /* Copy 96 bits of PPK[0..5] to RC4KEY[4..15]  (little-endian)       */
    for (i=0;i<6;i++)
        {
        RC4KEY[4+2*i] = Lo8(PPK[i]);
        RC4KEY[5+2*i] = Hi8(PPK[i]);
        }
    }




#ifdef TKIPSELFTEST


/*
**********************************************************************
* Routine: doTestCase -- execute a test case, and print results
**********************************************************************
*/
void DoTestCase(byte *RC4KEY,u32b IV32,u16b IV16,const byte *TA,const byte *TK)
    {
    int  i;
    u16b P1K[P1K_SIZE/2];  /* "temp" copy of phase1 key */

    printf("\nTK    =");
    for (i=0;i<TK_SIZE;i++) printf(" %02X",TK[i]);
    printf("\nTA    =");
    for (i=0;i<TA_SIZE;i++) printf(" %02X",TA[i]);
    printf("\nIV32  = %08X   [transmitted LE as",IV32);  /* show byte order */
//  IF BIG ENDIAN TRANSMISSION
    //for (i=0;i<4;i++) printf(" %02X",(IV32 >> (24-8*i)) & 0xFF);
//  IF LITTLE ENDIAN TRANSMISSION
    for (i=3;i>=0;i--) printf(" %02X",(IV32 >> (24-8*i)) & 0xFF);
    printf("]");
    printf("\nIV16  = %04X",IV16);


    Phase1(P1K,TK,TA,IV32);

    printf("\nP1K   =");
    for (i=0;i<P1K_SIZE/2;i++) printf(" %04X ",P1K[i] & 0xFFFF);

	mpx("\np1K\t= ", P1K, 10);
	mpx("tk\t= ", TK, 16);
	mpx("ta\t= ", TA, 6);
	mpx("iv32\t= ", &IV32, 4);

    Phase2(RC4KEY,TK,P1K,IV16);

    printf("\nRC4KEY= ");
    for (i=0;i<RC4_KEY_SIZE;i++) printf("%02X ",RC4KEY[i]);
    }


/*
**********************************************************************
* Static (Repeatable) Test Cases
**********************************************************************
*/
void DoStaticTestCases(int testCnt)
    {
    int  i,j;
    byte TA[TA_SIZE],TK[TK_SIZE],RC4KEY[RC4_KEY_SIZE];
    u16b IV16=0;
    u32b IV32=0;

    /* set a fixed starting point */
    for (i=0;i<TK_SIZE;i++) TK[i]=i;
    for (i=0;i<TA_SIZE;i++) TA[i]=(i+1)*17;
    TA[0] = TA[0] & 0xFC;               /* Clear I/G and U/L bits in OUI */

    /* now generate tests, feeding results back into new tests */
    for (i=0; i<testCnt/2; i++)
        {
        printf("\n\nTest vector #%d:",2*i+1);
        DoTestCase(RC4KEY,IV32,IV16,TA,TK);
        IV16++;                        /* emulate per-packet "increment" */
        if (IV16 == 0) IV32++;
        printf("\n\nTest vector #%d:",2*i+2);
        DoTestCase(RC4KEY,IV32,IV16,TA,TK);
        /* feed these results back to seed the next test input values    */
        IV16 = (i) ? Mk16(RC4KEY[15],RC4KEY[4]) : 0xFFFF;  /* force wrap */
        IV32 =       Mk16(RC4KEY[14],RC4KEY[5]);
        IV32 =       Mk16(RC4KEY[13],RC4KEY[7]) + (IV32 << 16);
        for (j=0;j<TA_SIZE;j++) TA[j]^=RC4KEY[12-j];
        for (j=0;j<TK_SIZE;j++) TK[j]^=RC4KEY[(j+i+1) % RC4_KEY_SIZE] ^
                                       RC4KEY[(j+i+7) % RC4_KEY_SIZE] ;
        TA[0] = TA[0] & 0xFC;           /* Clear I/G and U/L bits in OUI */
        }
    /* note: comparing the final output is a good check of correctness   */
    printf("\n");
    }


/*
**********************************************************************
* Test Cases Generated at Random
**********************************************************************
*/
void DoRandomTestCases(int testCnt)
    {
    int  i,j;
    u16b IV16;
    u32b IV32;
    byte TA[TA_SIZE],RC4KEY[RC4_KEY_SIZE],TK[TK_SIZE];

    printf("Random tests:\n");
    /* now generate tests "recursively" */
    for (i=0; i<testCnt; i++)
        {
        IV16 = rand() & 0xFFFF;
        IV32 = rand() + (rand() << 16);
        for (j=0;j<TK_SIZE;j++) TK[j]=rand() & 0xFF;
        for (j=0;j<TA_SIZE;j++) TA[j]=rand() & 0xFF;
        TA[0] = TA[0] & 0xFC;         /* Clear I/G and U/L bits in OUI */
        printf("\n\nRandom test vector #%d:",i+1);
        DoTestCase(RC4KEY,IV32,IV16,TA,TK);
        }
    printf("\n");
    }





/*
**********************************************************************
* Usage text
**********************************************************************
*/
#define NUM_TEST_CNT  8
void Usage(void)
    {
    printf(
        "Usage:   TKIP48 [options]\n"
        "Purpose: Generate test vectors for 802.11 TKIP48\n"
        "Options  -?   -- output this usage text\n"
        "         -r   -- generate test vectors at random\n"
        "         -sN  -- init random seed to N\n"
        "         -tN  -- generate N tests (default = %d)\n",
        NUM_TEST_CNT
        );
    exit(0);
    }

/*
**********************************************************************
* Main
**********************************************************************
*/
int 
main(int argc, char **argv)
    {
    char *parg;
    int   i,doRand = 0;
    int   testCnt  = NUM_TEST_CNT;
    u32b  seed     = (u32b) time(NULL);
    int vera = 1;

#if DO_SANITY_CHECK
    SanityCheckTable();
#endif

    for (i=1; i<argc; i++)
        {
        parg = argv[i];
        switch (parg[0])
            {
            case '-':
                switch (parg[1])
                    {
                    case '?':
                    case 'H':
                    case 'h':
                        Usage();
                        return 0;
                    case 'R':
                    case 'r':  /* generate some random test vectors */
                        doRand  = 1;
                        break;
                    case 'S':
                    case 's':
                        seed    = atoi(parg+2);
                        break;
                    case 'T':
                    case 't':
                        testCnt = atoi(parg+2);
                        break;
                    default:
                        break;
                    }
                break;
            case '?':
                Usage();
                return 0;
            default:
                printf("Invalid argument: \"%s\"\n", parg);
                return 1;
            }
        }

	if (vera) {
		dotest();
	}
    srand(seed);
    if (doRand) printf("Seed = %u\n",seed);


    /* generate some test vectors */
    if (doRand) DoRandomTestCases(testCnt);
    else        DoStaticTestCases(testCnt);

    return 0;
}

dotest()
{
unsigned char ta[TA_SIZE], rc4key[16], tk[TK_SIZE];
unsigned short p1k[P1K_SIZE/2];
unsigned short iv16;
unsigned long iv32;
int i;

	for(i=0; i<TK_SIZE; i++) tk[i]=i;
	for(i=0; i<TA_SIZE; i++) ta[i]=(i+1)*17;
	ta[0] &= 0xfc;
	iv16 = 0;
	iv32 = 0;

	Phase1(p1k, tk, ta, iv32);
	Phase2(rc4key, tk, p1k, iv16);

	printf("vera test\n");
	for(i=0; i<16; i++) printf("%2x ", rc4key[i]);

	
}
#endif

#ifdef VERA

/*
 * interface for Vera
 */
eva_tkip_rc4key(tk, tklen, aiv16, iv16len, aiv32, iv32len, ta, talen, rc4key, rc4keylen) 
unsigned char *tk, *aiv16, *aiv32, *ta, *rc4key;
{
unsigned short iv16;
unsigned long iv32;
unsigned short p1k[P1K_SIZE/2];
int i;

	//iv16 = 0;
	//iv16 = aiv16[0];
	//iv16 |= aiv16[1]<<8;
	iv16 = 0;			// transform 3 bytes of rc4 key
	iv32 = 0;
#ifdef XWEP_LE
	iv16 |= aiv16[2];		// rc4key[2]
	iv16 |= aiv16[0]<<8;		// rc4key[0]
	iv32 |= aiv32[0]; 		// iv32 in LE format
	iv32 |= aiv32[1]<<8;
	iv32 |= aiv32[2]<<16;
	iv32 |= aiv32[3]<<24;
#endif
#ifdef XWEP_BE
	iv16 |= aiv16[0];		// rc4key[0]
	iv16 |= aiv16[2]<<8;		// rc4key[1]
	iv32 |= aiv32[3]; 		// iv32 in BE format
	iv32 |= aiv32[2]<<8;
	iv32 |= aiv32[1]<<16;
	iv32 |= aiv32[0]<<24;
#endif

	Phase1(p1k, tk, ta, iv32);
	Phase2(rc4key, tk, p1k, iv16);
}
#endif
