/*
 * $Id: //depot/sw/branches/1.3_USB_LINUX_port/src/USB/wlan/host/common/desc.c#1 $
 *
 * Copyright (c) 2000-2003 Atheros Communications, Inc., All Rights Reserved
 *
 * Functions to operate on ATHEROS descriptor queues
 */

#include "wlanext.h"
#include "ui.h"


/**************************************************************************
 * descQInit - initialize the data structure corresponding to a descriptor
 *  queue
 *
 * RETURNS: success or failure
 */
A_STATUS
descQInit(QUEUE_DETAILS *pQueue)
{
    A_SEM_INIT(pQueue->qSem, 0, CREATE_UNLOCKED);
    pQueue->qBurstCount      = 0;
    pQueue->pBurstHeadDesc   = NULL;
    pQueue->pBurstTailDesc   = NULL;
    pQueue->queuingBurst     = FALSE;
    pQueue->pendBFrameCount  = 0;
    pQueue->burstCTSDur      = 0;

    pQueue->pDescQueueHead = pQueue->pDescQueueTail = NULL;
    A_ATOMIC_SET(&pQueue->qFrameCount,0);

    pQueue->descQueueAttribs.aifs           = 0;
    pQueue->descQueueAttribs.burstTime      = 0;
    pQueue->descQueueAttribs.compression    = FALSE;
    pQueue->descQueueAttribs.logCwMax       = 0;
    pQueue->descQueueAttribs.logCwMin       = 0;
    pQueue->descQueueAttribs.priority       = 0;

    return A_SEM_VALID(pQueue->qSem) ? A_OK : A_ERROR;
}

/**************************************************************************
 * descQGetHead - get pointer to first descriptor on the queue,
 *                don't remove it
 *
 * RETURNS: ATHEROS_DESC *
 */
ATHEROS_DESC *
descQGetHead(QUEUE_DETAILS *pQueue)
{
    return pQueue->pDescQueueHead;
}

/**************************************************************************
 * descQGetTail - Get pointer to last descriptor on the queue,
 *                don't remove it
 *
 * RETURNS: ATHEROS_DESC *
 */
ATHEROS_DESC *
descQGetTail(QUEUE_DETAILS *pQueue)
{
    return pQueue->pDescQueueTail;
}

/**************************************************************************
* descQPopHead - get a descriptor from the given queue
*
* RETURNS: ATHEROS_DESC *
*/
ATHEROS_DESC *
descQPopHead(QUEUE_DETAILS *pQueue)
{
    ATHEROS_DESC *pDesc;

    pDesc = pQueue->pDescQueueHead;
    if (pDesc) {
        pQueue->pDescQueueHead = pDesc->pNextVirtPtr;

        /*
         * See if the new head is the same as the one we just popped;
         * this should only occur if we just popped the tail desc of
         * a self-linked queue.
         */
        if (pQueue->pDescQueueHead == pDesc) {
            ASSERT(pQueue->isTailSelfLinked);
            pQueue->pDescQueueHead = NULL;
        }

        if (pQueue->pDescQueueHead == NULL) {
            pQueue->pDescQueueTail = NULL;
        }

        pDesc->pNextVirtPtr = NULL;
    }

    return pDesc;
}

/**************************************************************************
* descQPushTail - add the descriptor to the tail of queue struct
*
* RETURNS: N/A
*/
void
descQPushTail(QUEUE_DETAILS *pQueue, ATHEROS_DESC *pDesc, ATHEROS_DESC *pTail)
{
    if (!pDesc) {
        return;
    }
    ASSERT(pQueue->pDescQueueTail != pDesc);

    if (pQueue->isTailSelfLinked) {
        pTail->pNextVirtPtr = pTail;
    }

    /* check to see if anything is in the queue */
    if (pQueue->pDescQueueTail == NULL) {
        ASSERT(pQueue->pDescQueueHead == NULL);
        pQueue->pDescQueueHead = pDesc;
    } else {
        pQueue->pDescQueueTail->pNextVirtPtr = pDesc;
    }
    pQueue->pDescQueueTail = pTail;

    return;
}

/**************************************************************************
* descQPushHead - add the descriptor to the head of queue struct
*                 (non conventional!)
*
* RETURNS: N/A
*/
void
descQPushHead(QUEUE_DETAILS *pQueue, ATHEROS_DESC *pDesc)
{
    if (!pDesc) {
        return;
    }
    ASSERT(pQueue->pDescQueueHead != pDesc);
    ASSERT(!pQueue->isTailSelfLinked);

    /* check to see if anything is in the queue */
    if (pQueue->pDescQueueHead == NULL) {
        ASSERT(pQueue->pDescQueueTail == NULL);
        pQueue->pDescQueueTail = pDesc;
    } else {
        pDesc->pNextVirtPtr = pQueue->pDescQueueHead;
    }
    pQueue->pDescQueueHead = pDesc;

    return;
}

/**************************************************************************
* getPktFromQueueStart - get a packet from the start of queue
*
* Removes the packet's descriptor chain from the head of the queue.
* Sets the new head of the queue to the next packet's descriptor chain
* In case of the fragmented packet, the whole packet is given out
*
* RETURNS: Pointer to the descriptor chain of the packet if successful
*      NULL if failed
*/
ATHEROS_DESC *
getPktFromQueueStart(QUEUE_DETAILS *pQueue)
{
    ATHEROS_DESC *pDesc, *pVarDesc;

    A_SEM_LOCK(pQueue->qSem, WAIT_FOREVER);

    /* don't do anything if the queue is empty */
    if (pQueue->pDescQueueHead == NULL) {
        A_SEM_UNLOCK(pQueue->qSem);
        return NULL;
    }

    /* get descriptor from head of queue */
    pDesc = pQueue->pDescQueueHead;

    /* walk through all the fragments */
    pVarDesc = pDesc;
    while (pVarDesc->pBufferVirtPtr.header->frameControl.moreFrag) {
        pVarDesc = pVarDesc->pTxLastDesc->pNextVirtPtr;
        ASSERT(pVarDesc);

        NdisInterlockedDecrement(&pQueue->qFrameCount);
    }
    pVarDesc = pVarDesc->pTxLastDesc;
    NdisInterlockedDecrement(&pQueue->qFrameCount);

    /* set the new queue head */
    pQueue->pDescQueueHead = pVarDesc->pNextVirtPtr;
    if (pQueue->pDescQueueHead == NULL) {
        pQueue->pDescQueueTail = NULL;
    }

    pVarDesc->pNextVirtPtr = NULL;

    A_SEM_UNLOCK(pQueue->qSem);

    return (pDesc);
}

/**************************************************************************
* getPktFromQueue - get the given packet from within the queue
*
* Removes the packet's descriptor chain from where ever its in
* the queue. In case of the fragmented packet, only one fragment
* is given out.. Doesn't deal with physical ptrs - is considered
* for software queues only
*
* RETURNS: Pointer to the descriptor chain of the packet if successful
*      NULL if failed
*/
ATHEROS_DESC *
getPktFromQueue(QUEUE_DETAILS *pQueue, ATHEROS_DESC *pDesc)
{
    A_SEM_LOCK(pQueue->qSem, WAIT_FOREVER);
    if (pDesc) {
        ATHEROS_DESC **ppTail = &pQueue->pDescQueueHead;
        ATHEROS_DESC  *pTail  = pDesc->pTxLastDesc;

        while (*ppTail != pDesc) {
            if ((*ppTail) == NULL) {
                A_SEM_UNLOCK(pQueue->qSem);
                return NULL;
            }
            if (!pTail->pNextVirtPtr) {
                pQueue->pDescQueueTail = *ppTail;
            }
            ppTail = &((*ppTail)->pNextVirtPtr);
        }
        *ppTail = pTail->pNextVirtPtr;
        if (!pQueue->pDescQueueHead) {
            pQueue->pDescQueueTail = NULL;
        }
        NdisInterlockedDecrement(&pQueue->qFrameCount);

        pTail->pNextVirtPtr = NULL;
    }

    A_SEM_UNLOCK(pQueue->qSem);
    return pDesc;
}

/**************************************************************************
 * memAllocateDescriptor - allocate a single descriptor from the
 *  free descriptor pool
 *
 * RETURNS: ATHEROS_DESC *
 */
ATHEROS_DESC *
memAllocateDescriptor(WLAN_DEV_INFO *pdevInfo)
{
    QUEUE_DETAILS *pQueue = &pdevInfo->emptyDescQueue;
    ATHEROS_DESC *pDesc;

    A_SEM_LOCK(pQueue->qSem, WAIT_FOREVER);
    pDesc = descQPopHead(pQueue);

    if (pDesc) {
        NdisInterlockedDecrement(&pdevInfo->freeDescCount);
    }

    A_SEM_UNLOCK(pQueue->qSem);

    if (pDesc) {
        /* clean up the descriptor */
        A_DESC_INIT(pDesc);
    }

    return pDesc;
}

/**************************************************************************
* memFreeDescriptor - free a descriptor
*
* This routine will free a descriptor that was already created by
* memAllocateDescriptor()
*
* RETURNS: N/A
*/
void
memFreeDescriptor(WLAN_DEV_INFO *pdevInfo, ATHEROS_DESC *pDesc)
{
    QUEUE_DETAILS *pQueue = &pdevInfo->emptyDescQueue;

    ASSERT(pDesc->freeQueue == pQueue);

    pDesc->pNextVirtPtr = NULL;

    pDesc->pOrigBufferVirtPtr = NULL;
    pDesc->pOrigbufferPhysPtr = 0;

    A_SEM_LOCK(pQueue->qSem, WAIT_FOREVER);
    descQPushTail(pQueue, pDesc, pDesc);

    NdisInterlockedIncrement(&pdevInfo->freeDescCount);

    A_SEM_UNLOCK(pQueue->qSem);

    return;
}

/**************************************************************************
* freeBuffandDescChain - free buffer and descriptor chain
*
* Free the buffer and descriptor chain, created with createBuffandDesc()
*
* RETURNS: N/A
*/
void
freeBuffandDescChain(WLAN_DEV_INFO *pdevInfo, ATHEROS_DESC *pDesc)
{
    ATHEROS_DESC *pNextDesc;

    while (pDesc != NULL) {
        /* clean the current buffer and descriptor */
        pNextDesc = pDesc->pNextVirtPtr;
        freeBuffandDesc(pdevInfo, pDesc);
        pDesc = pNextDesc;
    }
    return;
}
