/*
 * $Id: //depot/sw/branches/1.3_USB_LINUX_port/src/USB/wlan/host/common/display.c#1 $
 *
 * Copyright  2000-2003 Atheros Communications, Inc.,  All Rights Reserved.
 *
 * This contains functions to display descriptor and packets etc.
 */

#if defined(VXWORKS)
#include "VxWorks.h"
#elif defined (PCI_NDIS)
#include <memory.h>
#include <stdio.h>
#include <stdlib.h>
#endif

#include "wlantype.h"
#include "wlanproto.h"
#include "wlansme.h"
#include "wlandrv.h"
#include "wlanext.h"
#include "diags/display.h"
#include "diags/ui.h"

/* Name strings for the frame types */

char *mgtTypeStr[] = {
    "ASSOC REQ",            /* 0 */
    "ASSOC RESP",           /* 1 */
    "REASSOC REQ",          /* 2 */
    "REASSOC RESP",         /* 3 */
    "PROBE REQ",            /* 4 */
    "PROBE RESP",           /* 5 */
    "",                     /* 6 */
    "",                     /* 7 */
    "BEACON",               /* 8 */
    "ATIM",                 /* 9 */
    "DISASSOC",             /* a */
    "AUTHENTICATE",         /* b */
    "DEAUTHENTICATE"        /* c */
};

char *ctrlTypeStr[] = {
    "", "", "", "", "",     /* 0 - 4 */
    "", "", "", "", "",     /* 5 - 9 */
    "PS POLL",              /* a */
    "RTS",                  /* b */
    "CTS",                  /* c */
    "ACK",                  /* d */
    "CF END",               /* e */
    "CF END+ACK"            /* f */
};

char    *dataTypeStr[] = {
    "DATA",                 /* 0 */
    "DATA+CF ACK",          /* 1 */
    "DATA+CF POLL",         /* 2 */
    "DATA+CF ACK+POLL",     /* 3 */
    "NULL",                 /* 4 */
    "NULL+CF ACK",          /* 5 */
    "NULL+CF POLL",         /* 6 */
    "NULL+CF ACK+POLL"      /* 7 */
};

/* Name strings for type of service */
char *servTypeStr[] = {
    "STA",  /* Station WLAN_STA_SERVICE = 0 */
    "AP"    /* Access point WLAN_AP_SERVICE = 1 */
};


/**************************************************************************
* printSsid - Print an SSID from an SSID struct
*
*
* RETURNS: N/A
*/
void
printSsid(SSID *pSsid)
{
    A_CHAR tmp[ELE_SSID_SIZE + 1], *pFrom, *pTo;
    int    i, bytesToCopy;

    ASSERT(pSsid);
    
    A_MEM_ZERO(tmp, sizeof(tmp));

    bytesToCopy = A_MIN(pSsid->length, sizeof(tmp) - 1);
    pFrom       = pSsid->ssid;
    pTo         = tmp;

    for (i = 0; i < bytesToCopy; i++) {
        if (*pFrom < 0x20 || *pFrom < 0) {
            *pTo++ = '?';
            pFrom++;
        } else {
            if (*pFrom == '%') {
                *pTo++ = *pFrom;
            }
            *pTo++ = *pFrom++;
        }
    }

    uiPrintf(tmp);
}
 
/**************************************************************************
* printMacAddress - Print a MAC address
*
*
* RETURNS: N/A
*/
void
printMacAddress(WLAN_MACADDR macAddress)
{
    A_UINT16 i;

    for (i = 0; i < WLAN_MAC_ADDR_SIZE; i++) {
        uiPrintf("%02x%s", macAddress.octets[i], (i+1) < WLAN_MAC_ADDR_SIZE ? ":" : "" );
    }
}

/**************************************************************************
* displayFrameControl - Print the Frame control fields of header
*
*
* RETURNS: N/A
*/
void
printFrameControl(FRAME_CONTROL *frameControl)
{
    uiPrintf(PROTO_VER              "%01x  ", frameControl->protoVer);
    switch (frameControl->fType) {
    case FRAME_MGT:
        uiPrintf(MGT_TYPE "  ");
        uiPrintf(SUBTYPE                "%s\n", mgtTypeStr[frameControl->fSubtype]);
        break;
    case FRAME_CTRL:
        uiPrintf(CTRL_TYPE "  ");
        uiPrintf(SUBTYPE                "%s\n", ctrlTypeStr[frameControl->fSubtype]);
        break;
    case FRAME_DATA:
        uiPrintf(DATA_TYPE "  ");
        uiPrintf(SUBTYPE                "%s\n", dataTypeStr[frameControl->fSubtype]);
        break;
    default:
        uiPrintf(TYPE                   "%01x  ", frameControl->fType);
        uiPrintf(SUBTYPE                "%01x\n", frameControl->fSubtype);
    }

    uiPrintf(TO_DS                  "%01x  ", frameControl->ToDS);
    uiPrintf(FROM_DS                "%01x  ", frameControl->FromDS);
    uiPrintf(MORE_FRAG              "%01x\n", frameControl->moreFrag);
    uiPrintf(RETRY                  "%01x  ", frameControl->retry);
    uiPrintf(PWR_MGT                "%01x  ", frameControl->pwrMgt);
    uiPrintf(MORE_DATA              "%01x\n", frameControl->moreData);
    uiPrintf(WEP                    "%01x  ", frameControl->wep);
    uiPrintf(ORDER                  "%01x  ", frameControl->order);
    return;
}

/**************************************************************************
 * displayCapInfo - displays various fields of Capabilities Info structure
 *
 * Takes pointer to the CAP_INFO structure (which is only 2 bytes) and prints
 * the bit fields.
 *
 * RETURNS: N/A
 */
void
displayCapInfo(CAP_INFO *capInfo)
{
    uiPrintf("Capability Information\n");
    uiPrintf(CF_POLLABLE "%01x  ", capInfo->cfPollable);
    uiPrintf(CF_POLLREQ  "%01x  ", capInfo->cfPollReq);
    uiPrintf(ESS_CAP     "%01x  ", capInfo->ess);
    uiPrintf(IBSS_CAP    "%01x\n", capInfo->ibss);
    uiPrintf(PRIVACY     "%01x  ", capInfo->privacy);
    uiPrintf(ALLOW_SHORT "%01x  ", capInfo->shortPreamble);
    uiPrintf(ALLOW_SSLOT "%01x\n", capInfo->shortSlotTime);
}

/**************************************************************************
 * displayInfoElement - displays Information elements in mgt frame
 *
 * Takes pointer to the INFO_ELEMENT structure and prints the element ID,
 * length and variable length value in hex dump.
 *
 * RETURNS: N/A
 */
void
displayInfoElement(INFO_ELEMENT *infoEle)
{
    int     i, j;

    uiPrintf("INFO ELEMENT  ");
    if (infoEle == NULL) {
        uiPrintf(" is NULL\n");
        return;
    }
    switch (infoEle->elementID) {
    case ELE_SSID:
        uiPrintf(SSID_VALUE);
        break;

    case ELE_SUPPORTED_RATES:
        uiPrintf(SUPP_RATES);
        break;

    case ELE_EXT_SUPPORTED_RATES:
        uiPrintf(EXT_SUPP_RATES);
        break;

    case ELE_FH_PARAM_SET:
        uiPrintf(FH_SET);
        break;

    case ELE_DS_PARAM_SET:
        uiPrintf(DS_SET);
        break;

    case ELE_CF_PARAM_SET:
        uiPrintf(CF_SET);
        break;

    case ELE_TIM:
        uiPrintf(TIM_DISP);
        break;

    case ELE_IBSS_PARAM_SET:
        uiPrintf(IBSS_DISP);
        break;

    case ELE_CHLG_TEXT:
        uiPrintf(CHLG_DISP);
        break;


    case ELE_COUNTRY_INFO:
        uiPrintf(COUNTRY_INFO_DISP);
        break;

    case ELE_NONERP:
        uiPrintf(NONERP_DISP);
        break;

    case ELE_VENDOR_PRIVATE:
        uiPrintf(VENPRIV_DISP);
        break;

    default:
        uiPrintf("%15s %4d", UNKNOWN_DISP, infoEle->elementID);
        break;
    }
    uiPrintf("Length  0x%02x bytes\n", infoEle->length);

    /* Now dump the value field */
    uiPrintf("Value   ");
    i = 0;
    while(i < infoEle->length) {
        for (j = 0; i < infoEle->length && j < 16; j++, i++) {
            uiPrintf("%02x ", infoEle->value[i]);
        }
        if (i < (infoEle->length - 1)) {
            uiPrintf("\n        ");
        }
    }
    uiPrintf("\n");
}

/**************************************************************************
 * displayInfoElementShort - displays Information elements (one per line)
 *
 * Takes pointer to the INFO_ELEMENT structure and prints the element ID,
 * length and variable length value in hex dump, all in one line
 *
 * RETURNS: N/A
 */
void
displayInfoElementShort(INFO_ELEMENT *infoEle)
{
    int     i;

    uiPrintf("ELEM ");
    switch (infoEle->elementID) {
    case ELE_SSID:
        uiPrintf(SSID_VALUE);
        break;

    case ELE_SUPPORTED_RATES:
        uiPrintf(SUPP_RATES);
        break;

    case ELE_EXT_SUPPORTED_RATES:
        uiPrintf(EXT_SUPP_RATES);
        break;

    case ELE_FH_PARAM_SET:
        uiPrintf(FH_SET);
        break;

    case ELE_DS_PARAM_SET:
        uiPrintf(DS_SET);
        break;

    case ELE_CF_PARAM_SET:
        uiPrintf(CF_SET);
        break;

    case ELE_TIM:
        uiPrintf(TIM_DISP);
        break;

    case ELE_IBSS_PARAM_SET:
        uiPrintf(IBSS_DISP);
        break;

    case ELE_CHLG_TEXT:
        uiPrintf(CHLG_DISP);
        break;

    default:
        uiPrintf("%15s %4d", UNKNOWN_DISP, infoEle->elementID);
        break;
    }
    uiPrintf("0x%02x = ", infoEle->length);

    /* Now dump the value field */
    i = 0;
    while(i < (infoEle->length) && i < 14) {
        uiPrintf("%02x ", infoEle->value[i++]);
    }
    if (infoEle->length >= 14) {
        uiPrintf("...");
    }
    uiPrintf("\n");
}

/**************************************************************************
 * displayMgtFrameHdr - Display header of a management frame
 *
 * Displays only the frame header of a management frame.
 *
 * RETURNS: N/A
 */
void
displayMgtFrameHdr(WLAN_MGT_MAC_HEADER *pmacHdr)
{
    printFrameControl( &pmacHdr->frameControl );
    uiPrintf(DUR_ID                 "%04x\n",
             WLAN_GET_DURATION(pmacHdr->durationNav));
    uiPrintf(DEST_ADDR ); printMacAddress(pmacHdr->destAddr); uiPrintf("   ");
    uiPrintf(SRC_ADDR  ); printMacAddress(pmacHdr->srcAddr); uiPrintf("\n");
    uiPrintf(BSSID_ADDR); printMacAddress(pmacHdr->bssId); uiPrintf("\n");
    uiPrintf(SEQ_NUM                "%04x  ", WLAN_GET_SEQNUM(pmacHdr->seqControl));
    uiPrintf(FRAG_NUM               "%01x  \n", WLAN_GET_FRAGNUM(pmacHdr->seqControl));
}

/**************************************************************************
 * displayMgtFrame - Display a complete management frame
 *
 * Displays a complete management frame. It prints the standard mgt frame
 * header, followed by the known fixed size fields. Then it positions p1
 * at the beginning of the variable length INFO_ELEMENT field. A loop at the
 * end steps thru all the info elements and prints them.
 * PARAMETERS:
 * pmacHdr - pointer to frame
 * frameSize - length of the frame, not including fcs, but may include iv.
 *
 * RETURNS: N/A
 */
void
displayMgtFrame(WLAN_MGT_MAC_HEADER *pmacHdr, A_UINT16 frameSize)
{
    A_UINT16  subtype;
    A_UINT8   *p1;            /* Pointer to beginning of info element field */
    A_UINT16  len, elen;

    if (pmacHdr->frameControl.fType != FRAME_MGT)
        return;

    displayMgtFrameHdr(pmacHdr);

    subtype = (A_UINT16) pmacHdr->frameControl.fSubtype;
    switch (subtype) {
    case SUBT_ASSOC_REQ: {
            /* p2 is conveniently typecast pointer */
            WLAN_FRAME_ASSOC_REQ *p2 = (WLAN_FRAME_ASSOC_REQ *) pmacHdr;
            displayCapInfo(&(p2->capInfo));
            uiPrintf(LISTEN_INT "%04x\n", p2->listenInterval);
            p1 = (A_UINT8 *) &(((WLAN_FRAME_ASSOC_REQ *)pmacHdr)->ssid);
        }
        break;

    case SUBT_ASSOC_RESP:
    case SUBT_REASSOC_RESP: {
            WLAN_FRAME_ASSOC_RESP *p2 = (WLAN_FRAME_ASSOC_RESP *) pmacHdr;
            displayCapInfo(&(p2->capInfo));
            uiPrintf(STATUS_CODE "%04x  ", p2->statusCode);
            uiPrintf(ASSOC_ID "%04x\n", p2->assocId & 0x3fff);
            p1 = (A_UINT8 *) &(((WLAN_FRAME_ASSOC_RESP *)pmacHdr)->rateSet);
        }
        break;

    case SUBT_REASSOC_REQ: {
            WLAN_FRAME_REASSOC_REQ *p2 = (WLAN_FRAME_REASSOC_REQ *) pmacHdr;
            displayCapInfo(&(p2->capInfo));
            uiPrintf(LISTEN_INT "%04x", p2->listenInterval);
            uiPrintf(CURAP_ADDR); printMacAddress(p2->curAPAddr); uiPrintf("\n");
            p1 = (A_UINT8 *) &(((WLAN_FRAME_REASSOC_REQ *)pmacHdr)->ssid);
        }
        break;

    case SUBT_PROBE_REQ:
        p1 = (A_UINT8 *) &(((WLAN_FRAME_PROBE_REQ *)pmacHdr)->ssid);
        break;

    case SUBT_PROBE_RESP:
    case SUBT_BEACON: {
            WLAN_FRAME_PROBE_RESP *p2 = (WLAN_FRAME_PROBE_RESP *)pmacHdr;
            A_UINT32              val;

            A_DRIVER_BCOPY(&(p2->timestamp.high), &val, 4);
            uiPrintf(TIMESTAMP_STR "%08x", val);
            A_DRIVER_BCOPY(&(p2->timestamp.low), &val, 4);
            uiPrintf("%08x  ", val);
            uiPrintf(BEACON_INTRV "%04x\n", p2->beaconInterval);
            displayCapInfo(&(p2->capInfo));
            p1 = (A_UINT8 *) &(((WLAN_FRAME_PROBE_RESP *)pmacHdr)->buffer);
            break;
        }
    case SUBT_AUTH: {
            if (pmacHdr->frameControl.wep) {
                /* IV field in encrypted frame, use different frame */
                WLAN_FRAME_AUTH_ENCRYPT *p2 =
                    (WLAN_FRAME_AUTH_ENCRYPT *)pmacHdr;
                uiPrintf(ALGO_NO "%04x ", p2->algNo);
                uiPrintf(TRANS_SEQ "%04x ", p2->transSeqNo);
                uiPrintf(STATUS_CODE "%04x\n", p2->statusCode);
                p1 = (A_UINT8 *) &(p2->challenge);
            } else {
                /* regular AUTH frame */
                WLAN_FRAME_AUTH_CHLG *p2 = (WLAN_FRAME_AUTH_CHLG *)pmacHdr;
                uiPrintf(ALGO_NO "%04x ", p2->algNo);
                uiPrintf(TRANS_SEQ "%04x ", p2->transSeqNo);
                uiPrintf(STATUS_CODE "%04x\n", p2->statusCode);
                p1 = (A_UINT8 *) &(p2->challenge);
            }
            break;
        }
    case SUBT_DEAUTH:
    case SUBT_DISASSOC: {
            WLAN_FRAME_DEAUTH *p2 = (WLAN_FRAME_DEAUTH *)pmacHdr;
            uiPrintf(REASON_CODE "%04x\n", p2->reasonCode);
        }
    default:
        return;
    }

    /* Now, p1 is the beginning of the variable size fields */
    /* len is the offset into the frame */
    len = (int) (p1 - (A_UINT8 *)pmacHdr);

    /* A valid element should have at least 3 bytes - elementID,
     * length and one value byte. If not, we stop. Last 4 bytes of frame are FCS.
     */
    while (len <= (frameSize - 3)) {
        displayInfoElement((INFO_ELEMENT *) p1);
        /* Increment len by no. of bytes in this element.
         * Element Id and Length byte are added to the value
         * of length field. Even if length field is 0,
         * this will increment len by 2. Don't use INFO_ELEMENT_SIZE
         * macro, it will return 0 if length is 0.
         */
        elen = ((INFO_ELEMENT *)p1)->length +
               sizeof(A_UINT8) + sizeof(A_UINT8);
        len += elen;
        p1  += elen;
    }
}

/**************************************************************************
 * displayMgtFrameShort - Display a complete management frame, short display
 *
 * Displays a complete management frame but data is not expanded much
 *
 * RETURNS: N/A
 */
void
displayMgtFrameShort(WLAN_MGT_MAC_HEADER *pmacHdr, A_UINT16 frameSize)
{
    A_UINT16 subtype;
    A_UINT8  *p1 = NULL;
    A_UINT16 len, elen;

    if (pmacHdr->frameControl.fType != FRAME_MGT) {
        return;
    }

    uiPrintf("MGT ");
    uiPrintf("%20s  ", mgtTypeStr[pmacHdr->frameControl.fSubtype]);
    uiPrintf("Frame Control: " "%04x\n", *((A_UINT16 *) &(pmacHdr->frameControl)));
    uiPrintf("DUR %04x ", WLAN_GET_DURATION(pmacHdr->durationNav));
    uiPrintf("ADDR: "); printMacAddress(pmacHdr->destAddr);
    uiPrintf(" "); printMacAddress(pmacHdr->srcAddr);
    uiPrintf(" "); printMacAddress(pmacHdr->bssId);
    uiPrintf(" SEQ/FRAG:" "%04x/%01x\n", WLAN_GET_SEQNUM(pmacHdr->seqControl),
             WLAN_GET_FRAGNUM(pmacHdr->seqControl));

    subtype = (A_UINT16) pmacHdr->frameControl.fSubtype;
    switch (subtype) {
    case SUBT_ASSOC_REQ: {
        WLAN_FRAME_ASSOC_REQ *p2 = (WLAN_FRAME_ASSOC_REQ *) pmacHdr;
        uiPrintf("CAP INFO: " "%04x", *((A_UINT16 *) &(p2->capInfo)));
        uiPrintf(LISTEN_INT "%04x\n", p2->listenInterval);
        p1 = (A_UINT8 *) &(((WLAN_FRAME_ASSOC_REQ *)pmacHdr)->ssid);
        break;
    }

    case SUBT_ASSOC_RESP:
    case SUBT_REASSOC_RESP: {
        WLAN_FRAME_ASSOC_RESP *p2 = (WLAN_FRAME_ASSOC_RESP *) pmacHdr;
        uiPrintf("CAP INFO: " "%04x", *((A_UINT16 *) &(p2->capInfo)));
        uiPrintf(STATUS_CODE "%04x  ", p2->statusCode);
        uiPrintf(ASSOC_ID "%04x\n", p2->assocId & 0x3fff);
        p1 = (A_UINT8 *) &(((WLAN_FRAME_ASSOC_RESP *)pmacHdr)->rateSet);
        break;
    }

    case SUBT_REASSOC_REQ: {
        WLAN_FRAME_REASSOC_REQ *p2 = (WLAN_FRAME_REASSOC_REQ *) pmacHdr;
        uiPrintf("CAP INFO: " "%04x", *((A_UINT16 *) &(p2->capInfo)));
        uiPrintf(LISTEN_INT "%04x", p2->listenInterval);
        uiPrintf(CURAP_ADDR); printMacAddress(p2->curAPAddr); uiPrintf("\n");
        p1 = (A_UINT8 *) &(((WLAN_FRAME_REASSOC_REQ *)pmacHdr)->ssid);
        break;
    }

    case SUBT_PROBE_REQ:
        p1 = (A_UINT8 *) &(((WLAN_FRAME_PROBE_REQ *)pmacHdr)->ssid);
        break;

    case SUBT_PROBE_RESP:
    case SUBT_BEACON: {
        WLAN_FRAME_PROBE_RESP *p2 = (WLAN_FRAME_PROBE_RESP *) pmacHdr;
        A_UINT32        val;
        A_DRIVER_BCOPY(&(p2->timestamp.high), &val, 4);
        uiPrintf(TIMESTAMP_STR "%08x", val);
        A_DRIVER_BCOPY(&(p2->timestamp.low), &val, 4);
        uiPrintf("%08x  ", val);
        uiPrintf(BEACON_INTRV "%04x\n", p2->beaconInterval);
        uiPrintf("CAP INFO: " "%04x\n", *((A_UINT16 *) &(p2->capInfo)));
        p1 = (A_UINT8 *) &(((WLAN_FRAME_PROBE_RESP *)pmacHdr)->buffer);
        break;
    }

    case SUBT_AUTH: {
        if (pmacHdr->frameControl.wep) {
            /* IV field in encrypted frame, use different frame */
            WLAN_FRAME_AUTH_ENCRYPT *p2 =
                (WLAN_FRAME_AUTH_ENCRYPT *)pmacHdr;
            uiPrintf(ALGO_NO "%04x ", p2->algNo);
            uiPrintf(TRANS_SEQ "%04x ", p2->transSeqNo);
            uiPrintf(STATUS_CODE "%04x\n", p2->statusCode);
            p1 = (A_UINT8 *) &(p2->challenge);

        } else {
            /* regulat AUTH frame */
            WLAN_FRAME_AUTH_CHLG    *p2 = (WLAN_FRAME_AUTH_CHLG *)pmacHdr;
            uiPrintf(ALGO_NO "%04x ", p2->algNo);
            uiPrintf(TRANS_SEQ "%04x ", p2->transSeqNo);
            uiPrintf(STATUS_CODE "%04x\n", p2->statusCode);
            p1 = (A_UINT8 *) &(p2->challenge);

        }
        break;
    }

    case SUBT_DEAUTH:
    case SUBT_DISASSOC: {
        WLAN_FRAME_DEAUTH       *p2 = (WLAN_FRAME_DEAUTH *)pmacHdr;
        uiPrintf(REASON_CODE "%04x\n", p2->reasonCode);
        break;
    }

    default:
        return;
    }

    /* Now, p1 is the beginning of the variable size fields */
    /* len is the offset into the frame */
    len = (int) (p1 - (A_UINT8 *)pmacHdr);

    /* A valid element should have at least 3 bytes - elementID,
     * length and one value byte. If not, we stop. Last 4 bytes of frame are FCS.
     */
    while (len < (frameSize - 3)) {
        displayInfoElementShort((INFO_ELEMENT *) p1);
        /* Increment len by no. of bytes in this element.
         * Element Id and Length byte are added to the value
         * of length field. Even if length field is 0,
         * this will increment len by 2. Don't use INFO_ELEMENT_SIZE
         * macro, it will return 0 if length is 0.
         */
        elen = ((INFO_ELEMENT *)p1)->length +
               sizeof(A_UINT8) + sizeof(A_UINT8);
        len += elen;
        p1  += elen;
    }
}

/**************************************************************************
 * displayCtrlFrame - Display a complete controlframe
 *
 * Displays a complete control frame.
 *
 * RETURNS: N/A
 */
void
displayCtrlFrame(WLAN_MGT_MAC_HEADER *pmacHdr)
{
    printFrameControl(&pmacHdr->frameControl);
    uiPrintf(DUR_ID "%08x   ", WLAN_GET_DURATION(pmacHdr->durationNav));

    switch (pmacHdr->frameControl.fSubtype) {
    case SUBT_PSPOLL: {
        WLAN_FRAME_PSPOLL *p2 = (WLAN_FRAME_PSPOLL *)pmacHdr;
        uiPrintf(BSSID_ADDR); printMacAddress(p2->bssId); uiPrintf("  ");
        uiPrintf(TRANS_ADDR  ); printMacAddress(p2->transAddr); uiPrintf("\n");
        break;
    }

    case SUBT_RTS: {
        WLAN_FRAME_RTS *p2 = (WLAN_FRAME_RTS *)pmacHdr;
        uiPrintf(REC_ADDR); printMacAddress(p2->recAddr); uiPrintf("  ");
        uiPrintf(TRANS_ADDR  ); printMacAddress(p2->transAddr); uiPrintf("\n");
        break;
    }

    case SUBT_CTS:
    case SUBT_ACK: {
        WLAN_FRAME_CTS *p2 = (WLAN_FRAME_CTS *)pmacHdr;
        uiPrintf(REC_ADDR  ); printMacAddress(p2->recAddr); uiPrintf("\n");
        break;
    }

    case SUBT_CFEND:
    case SUBT_CFENDACK: {
        WLAN_FRAME_CFEND *p2 = (WLAN_FRAME_CFEND *)pmacHdr;
        uiPrintf(REC_ADDR  ); printMacAddress(p2->recAddr); uiPrintf("\n");
        uiPrintf(BSSID_ADDR); printMacAddress(p2->bssId); uiPrintf("  ");
        break;
    }
    }
}

/**************************************************************************
 * displayStates - Display a SIB Station state and transitional state
 *
 * RETURNS: N/A
 */
void
displayStates(A_UINT32 sState, A_UINT32 tState)
{
    uiPrintf("State:");
    if (sState & STATE_AUTH)        uiPrintf(" AUTH");
    if (sState & STATE_ASSOC)       uiPrintf(" ASSOC");
    if (sState & STATE_JOINED)      uiPrintf(" JOINED");
    if (sState & STATE_AP_UP)       uiPrintf(" AP-UP");
    if (sState & STATE_AP_DOWN)     uiPrintf(" AP-DOWN");
    if (sState == 0)                uiPrintf(" None");

    uiPrintf(", transState: 0x%08x\n", tState);
}

/**************************************************************************
 * displaySib - Display a SIB Station Information Block
 *
 * Command exported to user to important fields of SIB , by supplying
 * the pointer.
 *
 * RETURNS: N/A
 */
void
displaySib(SIB_ENTRY *pSib)
{
    uiPrintf("SIB_ENTRY at %p, pNext %p\n",
             (void *)pSib, (void *)pSib->pNext);
    uiPrintf(STA_ADDR); printMacAddress(pSib->macAddr); uiPrintf(" ");
    uiPrintf("Service %4s ",
             (pSib->serviceType == WLAN_STA_SERVICE) ? "STA" : "AP");
    displayStates(pSib->staState, pSib->transState);
}
