/*
 * $Id: //depot/sw/branches/1.3_USB_LINUX_port/src/USB/wlan/host/common/fragde.c#1 $
 *
 * Copyright  2000-2003 Atheros Communications, Inc.,  All Rights Reserved.
 */

/*
DESCRIPTION

This module implements the fragmentation/de-fragmentation functions.

Please refer to the Access Point Software Desgin Doc for details

Fragmentation
=============

An output frame is presented as a descriptor chain. For each descriptor
the physical buffer length could be different. The fragmentation will
have to first examine the length of frame, if it is not longer than
the dot11FragmentionThreshold for the STA then no fragmentation is
needed. Otherwise we walk through the descriptors until the partial
length of the frame reaches dot11FragmentionThreshold, we need to
allocate a new descriptor chain for the next dot11FragmentionThreshold
length and so on until the last descriptor chain is shorter than
the dot11FragmentionThreshold. We need to add a 802.11 header for each
new allocated descriptor chain which is the duplicate of the original
header and with each header including the original one but except the
last one has More Frag bit set.

The result would a descriptor chain which chains all fragments together

Defragmentation
===============

An incoming frame could be just a fragment of a complete frame by
checking the More Frag bit in its 802.11 header in the first descriptor
of the descriptor chain. A partial frame descriptor chain ( only limited
number < 3 is allowed ) is reserved for the complete frame. First we need
to examine the frame length and More Frag bit of the incoming descriptor
chain.

if ((frame length < dot11FragmentionThreshold) & More Frag bit clear)
    no defragmentation needed
    this is the only fragment of the frame
    pass up, Done

if ((frame length < dot11FragmentionThreshold) & More Frag bit set)
    defragmentation needed
    this is the last fragment of the frame

if ((frame length = dot11FragmentionThreshold) & More Frag bit set)
    defragmentation needed
    this is one of the fragment of the frame

for the latter two situation

we need allocate a 802.11 header which is the duplicate of the header
of the incoming fragment with More Frag bit cleared and link the rest
descriptors of all fragments together and get ride of the 802.11 headers.
Also need to update the frame length.
*/

/* includes */

#include "wlandrv.h"
#include "wlanext.h"
#include "wlanframe.h"
#include "wlanSend.h"
#include "wlanfragde.h"
#include "ui.h"


#ifdef DEBUG
int fragVerbose = 0;
int membufDebug = 0;
int fragErrs    = 0;
#endif

LOCAL ATHEROS_DESC * tailDescFind(ATHEROS_DESC *pDesc);
LOCAL A_UINT32       wlanDeFragArFree(WLAN_DEV_INFO *);
LOCAL void           wlanDeFragArAge(WLAN_DEV_INFO *, A_BOOL);
LOCAL A_UINT32       wlanDeFragArSearch(WLAN_DEV_INFO *, WLAN_MACADDR *,
                                        A_UINT16 seqNum, A_UINT16 fragNum);

/*******************************************************************************
 * wlanFrag - fragmentation function
 *
 * AP: No fragmentation for broadcast/multicast frames including beacons
 *     No fragmentation for control frames cause they are short enough.
 * STA: Infrastructure mode all frames are unicast frames since they are all
 *      sent to AP.
 *      Ad-hoc mode the broadcast/multicast including beacons are not
 *      fragmentable.
 *
 * The function is called during TX service
 *
 * This function returns a TX descriptor chain as follows:
 *
 * {header + fraghdr1 + fragdata1 + fraghdr2 + fragdata2 + ... fraghdrN + fragdataN}
 *
 *
 * RETURNS: ERROR if unsuccessful
 *           OK if successful
 */
ATHEROS_DESC *
wlanFrag(WLAN_DEV_INFO *pdevInfo, ATHEROS_DESC *pDesc) /* fragment pDesc */
{
    WLAN_DATA_MAC_HEADER3 *pOrigWlanHdr;  /* Original WLAN Header */
    ATHEROS_DESC *pFragDescReturn;        /* Descriptor that is Returned */
    ATHEROS_DESC *pTmpDesc;               /* Temporary Descriptor */

    ATHEROS_DESC *pFragHdrDesc;           /* Fragment Header Descriptor */
    WLAN_DATA_MAC_HEADER3 *pFragWlanHdr;  /* Fragment WLAN Header */
    ATHEROS_DESC *pFragDataDesc;          /* Fragment Data Descriptor */     

    A_UINT16     pduSize;                 /* Calculated max payload */
    A_UINT16     origFrameLen;            /* Original Frame Length */   
    A_UINT16     hdrLen;                  /* Header Length */

    A_UINT16     residLength;             /* Residual Length Left to Frag */
    A_UINT16     uiPosition;              /* Byte index */
    A_UINT16     fragNum = 0;             /* Current Frag Number */
    A_UINT16     numFrags;                /* Number of Fragments */
    A_UINT16     fragThreshold;            

    /* decide if we need to frag the frame */

    origFrameLen  = (A_UINT16)pDesc->frameLength;
    fragThreshold = (A_UINT16)pdevInfo->staConfig.userFragThreshold;

    if (pDesc->ffFlag) {
        return pDesc;
    }

    /* Skip fragmentation if it's too short */
    if (origFrameLen <= fragThreshold) {
        return pDesc;
    }

    /* Skip or for AP/ad hoc multicast */
    if ((pdevInfo->localSta->serviceType == WLAN_AP_SERVICE) &&
        isGrp(&pDesc->pBufferVirtPtr.header3->address1)) {
        return pDesc;
    }

    /* Start of Fragmentation */

    /* Calculate PDU size which is the (fragThreshold - Header Length) */
    hdrLen = headerLengthGet(pDesc) + ivLengthGet(pDesc);
    pduSize = fragThreshold - hdrLen;

    /* Given PDU size, calculate number of Frags needed */
    numFrags = ((origFrameLen - hdrLen)/pduSize +
                (((origFrameLen - hdrLen)%pduSize == 0) ? 0 : 1));

    /* Prepare first Descriptor, set WLAN more flag */
    pOrigWlanHdr = pDesc->pBufferVirtPtr.header3;
    pOrigWlanHdr->frameControl.moreFrag = 1;

    pFragDescReturn = pDesc;

    pTmpDesc              = pDesc;
    pTmpDesc->frameLength = pduSize + hdrLen;

    /* Length left to fragment */
    residLength = (A_UINT16) pTmpDesc->bufferLength - hdrLen;

#ifdef DEBUG
    if (membufDebug)
        uiPrintf("\nFrag starts...FrameLen: %d hdrLen: %d pduSize: %d numFrags: %d\n", 
                 origFrameLen, hdrLen, pduSize, numFrags);
#endif

    while (fragNum < numFrags) {

        while (residLength >= pduSize) {
            /* calculate new buffer length */
            residLength            -= pduSize;
            uiPosition              = (A_UINT16)pTmpDesc->bufferLength - residLength;
            pTmpDesc->bufferLength  = uiPosition;

            /* complete a fragment if buffer length is equal to fragThreshold */
            /* no more descripor for this fragment */
            pTmpDesc->more = 0;

            if ((residLength == 0) && (pTmpDesc->pNextVirtPtr == NULL)) {
                break;
            }

#ifdef DEBUG
            if (membufDebug)
                uiPrintf("residLength: %d, pduSize: %d\n", residLength, pduSize);
#endif

            /* start a new fragment by creating a new header */
            if (CREATE_HEADER_BUFF_AND_DESC(pdevInfo, hdrLen, &pFragHdrDesc) != A_OK) {
                /* clean up before leave */
                freeBuffandDescChain(pdevInfo, pFragDescReturn);
                LOG_DROP_FRAME(pdevInfo, tx);
#ifdef DEBUG
                if (fragVerbose)
                    uiPrintf("\nFragment header buffer allocation failed.\n");
                ++fragErrs;
#endif
                return NULL;
            }

            /* Copy WLAN Frame Header */
            A_BCOPY(pOrigWlanHdr, pFragHdrDesc->pBufferVirtPtr.ptr, hdrLen);
            pFragWlanHdr = pFragHdrDesc->pBufferVirtPtr.header3;

            /* set the header and descriptor */
            WLAN_SET_FRAGNUM(pFragWlanHdr->seqControl, ++fragNum);
            if (fragNum != (A_UINT16)(numFrags - 1)) {
                pFragWlanHdr->frameControl.moreFrag = 1;
                pFragHdrDesc->frameLength           = pduSize + hdrLen;
            } else {
                pFragWlanHdr->frameControl.moreFrag = 0;
                pFragHdrDesc->frameLength           = origFrameLen - pduSize*fragNum;
            }

            /* Copy TX Desc Header */
            pFragHdrDesc->more = 1;
            A_DESC_COPY(pFragDescReturn, pFragHdrDesc);
            ATH_OSPKTREF_DUP(pFragDescReturn, pFragHdrDesc);

#ifdef DEBUG
            if (membufDebug)
                uiPrintf("bufRefUpdate -- residLength: %d, pduSize: %d\n", residLength, pduSize);
#endif

            if (residLength == 0) {
                /* link the descriptors */
                pFragDataDesc              = pTmpDesc->pNextVirtPtr;
                pTmpDesc->pNextVirtPtr     = pFragHdrDesc;
                pFragHdrDesc->pNextVirtPtr = pFragDataDesc;
                pTmpDesc                   = pFragHdrDesc;
            } else {
                /* allocate the descriptor for remaining data portion */
                if ((pFragDataDesc = memAllocateDescriptor(pdevInfo)) == NULL) {
                    LOG_DROP_FRAME(pdevInfo, tx);
                    /* clean up before leave */
                    freeBuffandDescChain(pdevInfo, pFragDescReturn);
                    freeBuffandDescChain(pdevInfo, pFragHdrDesc);
#ifdef DEBUG
                    if (fragVerbose)
                        uiPrintf("\ndescriptor allocation error!\n");
                    ++fragErrs;
#endif
                    return NULL;
                }
                A_DESC_COPY(pTmpDesc, pFragDataDesc);
                ATH_OSPKTREF_DUP(pTmpDesc, pFragDataDesc);
                ATH_OSBUFREF_DUP(pTmpDesc, pFragDataDesc);

                /* set descriptor field */
                pFragDataDesc->pBufferVirtPtr.byte = pTmpDesc->pBufferVirtPtr.byte + uiPosition;
                pFragDataDesc->bufferPhysPtr       = pTmpDesc->bufferPhysPtr + uiPosition;
                pFragDataDesc->more                = 1;
                pFragDataDesc->bufferLength        = residLength;

                /* keep the original buffer characteristics here */
                pFragDataDesc->pOrigBufferVirtPtr  = pTmpDesc->pOrigBufferVirtPtr;

                /* link descriptors */
                pFragDataDesc->pNextVirtPtr = pTmpDesc->pNextVirtPtr;
                pTmpDesc->pNextVirtPtr      = pFragHdrDesc;
                pFragHdrDesc->pNextVirtPtr  = pFragDataDesc;

                pTmpDesc = pFragDataDesc;
            }
        }

        /* continue a fragment */
        if (pTmpDesc->pNextVirtPtr == NULL) {
            pTmpDesc->more = 0;
            break;
        } else {
            pTmpDesc->more = 1;
            pTmpDesc = pTmpDesc->pNextVirtPtr;
            residLength += (A_UINT16) pTmpDesc->bufferLength;
        }

    }

#ifdef DEBUG
    if (membufDebug) {
        pTmpDesc = pFragDescReturn;
        uiPrintf("Frag Chain:\n");
        while (pTmpDesc) {
            uiPrintf("\tdesc: 0x%x buflen: %d framelen: %d FirstDesc: %x  Last Desc: %x Buffer: %x More: %d\n", 
                     pTmpDesc, pTmpDesc->bufferLength, pTmpDesc->frameLength,
                     pTmpDesc->pTxFirstDesc, pTmpDesc->pTxLastDesc,
                     pTmpDesc->pBufferVirtPtr.byte,
                     pTmpDesc->more);
            pTmpDesc = pTmpDesc->pNextVirtPtr;
        }
    }
#endif
    return pFragDescReturn;
}


/*******************************************************************************
* wlanDeFrag - De-fragmentation function
*
* AP: de-fragmentation for broadcast/multicast frames from WM
*     de-fragmentation for data and management frames
* STA: no de-fragmentation for broadcast/multicast frames in any mode
*
* the function is called in the RX service
*
* RETURNS:    NULL if the fragment is consumed by the pool
*             pointer of assemblied complete frame structure.
*/
ATHEROS_DESC *
wlanDeFrag(WLAN_DEV_INFO *pdevInfo, ATHEROS_DESC *pFragDesc)
{
    PARTIAL_DESC *deFragDescArray = pdevInfo->defragLanes;
    ATHEROS_DESC *pLastDesc, *pReassembled = NULL, *ptDesc, *pFirstDesc;
    A_UINT32     bufIndex;  /* index for pre-allocated defrag structures */
    A_UINT32     moreFrag;
    A_UINT32     totalLen;
    A_UINT32     len;

    /* wind up our aging clock */
    wlanDeFragArAge(pdevInfo, FALSE);

    /* check the current fragment frame whether fragmented or not */
    if (((moreFrag = moreFragGet(pFragDesc)) == 0) &&
        (fragNumGet(pFragDesc) == 0)) {
        /* not fragmented, return  */
        // uiPrintf("\n non fraged frame\n");
        return pFragDesc;
    }

    /* initial MPDU of fragmentated MSDU. */
    if (fragNumGet(pFragDesc) == 0) {
        /* first descriptor and find free partial frame to begin Msdu rx */
        if ((bufIndex = wlanDeFragArFree(pdevInfo)) == A_ERROR) {

            /* give us one more shot at it */
            wlanDeFragArAge(pdevInfo, TRUE);

            /* check again. if not,  reaches max limit partial frame numbers */
            if ((bufIndex = wlanDeFragArFree(pdevInfo)) == A_ERROR) {
                freeBuffandDescChain(pdevInfo, pFragDesc);
#ifdef DEBUG
                fragErrs++;
                if (fragVerbose) {
                    uiPrintf("\nwlanDeFrag -- out of max partial frag desc!\n");
                }
#endif
                return NULL;
            }
        }

        deFragDescArray[bufIndex].inUse     = TRUE;
        deFragDescArray[bufIndex].rta       = &pFragDesc->pBufferVirtPtr.header3->address2;
        deFragDescArray[bufIndex].rsn       = seqNumGet(pFragDesc);
        deFragDescArray[bufIndex].rCur      = fragNumGet(pFragDesc);
        deFragDescArray[bufIndex].timeStamp = A_MS_TICKGET();
        deFragDescArray[bufIndex].pDesc     = pFragDesc;
        deFragDescArray[bufIndex].hdrLen    = headerLengthGet(pFragDesc) +
                                              ivLengthGet(pFragDesc);
        deFragDescArray[bufIndex].pduLen    = lengthGet(pFragDesc) -
                                              deFragDescArray[bufIndex].hdrLen;

        pFragDesc->pBufferVirtPtr.header->frameControl.moreFrag = 0;

        return NULL;
    } else {
        /* intermediate or final Mpdu of the fragmented Msdu */

        /* search for the right buffer */
        bufIndex = wlanDeFragArSearch(pdevInfo,
                                      &pFragDesc->pBufferVirtPtr.header3->address2,
                                      seqNumGet(pFragDesc),
                                      fragNumGet(pFragDesc));

        if (bufIndex == A_ERROR) {
            /* free the input descriptor and buffer */
            // uiPrintf("\ninvalid fragment!\n");
            freeBuffandDescChain(pdevInfo, pFragDesc);
#ifdef DEBUG
            fragErrs++;
            if (fragVerbose) {
                uiPrintf("\nwlanDeFrag -- can't find partial frag desc (invalid fragment)!\n");
            }
#endif
            return NULL;
        }

         /* make sure it fits into WLAN pdu space and frags arrive in order */
         deFragDescArray[bufIndex].pduLen += lengthGet(pFragDesc) -
                                             deFragDescArray[bufIndex].hdrLen;

        if (deFragDescArray[bufIndex].pduLen > MAX_WLAN_BODY_SIZE ||
            deFragDescArray[bufIndex].rCur+1 != fragNumGet(pFragDesc))
        {
            /* frame is too long, throw away all fragments */

            /* free the input descriptor and buffer */
            freeBuffandDescChain(pdevInfo, pFragDesc);

            /* free the defrag partial frame buffer back to pool*/
            freeBuffandDescChain(pdevInfo, deFragDescArray[bufIndex].pDesc);
            deFragDescArray[bufIndex].inUse = FALSE;
#ifdef DEBUG
            fragErrs++;
            if (fragVerbose) {
                uiPrintf("\nwlanDeFrag -- drop bad fragment!\n");
            }
#endif
            return NULL;
        }

        /* frame passes all checks */

        deFragDescArray[bufIndex].rCur++;
        pLastDesc = tailDescFind(deFragDescArray[bufIndex].pDesc);

        /* skip duplicate headers */
        if (deFragDescArray[bufIndex].hdrLen == pFragDesc->status.rx.dataLength) {
            /* first descriptor only contains the header */
            pLastDesc->pNextVirtPtr = pFragDesc->pNextVirtPtr;

            /* free the input descriptor and buffer */
            freeBuffandDesc(pdevInfo, pFragDesc);
        } else {
            /* first descriptor contains the header + data */
            pLastDesc->pNextVirtPtr = pFragDesc;

            /* adjust buffer pointer */
            pFragDesc->bufferPhysPtr       += deFragDescArray[bufIndex].hdrLen;
            pFragDesc->pBufferVirtPtr.byte += deFragDescArray[bufIndex].hdrLen;

            pFragDesc->status.rx.dataLength -= deFragDescArray[bufIndex].hdrLen;
        }

        /* got em all */
        if (moreFrag == 0) {

#define COALESCE_DEFRAGS
#ifdef COALESCE_DEFRAGS
            /* Coalesce buffers 2 thru N into the first buffer. */

            pFirstDesc = deFragDescArray[bufIndex].pDesc;
            len        = pFirstDesc->status.rx.dataLength;

            for (totalLen = len, ptDesc = pFirstDesc->pNextVirtPtr; ptDesc;
                 ptDesc = ptDesc->pNextVirtPtr)
            {
                len = ptDesc->status.rx.dataLength;
                A_BCOPY(ptDesc->pBufferVirtPtr.byte,
                        &pFirstDesc->pBufferVirtPtr.byte[totalLen],
                        len);
                totalLen += len;
                ASSERT(totalLen <= AR_BUF_SIZE);
            }

            /* free 2..N */
            freeBuffandDescChain(pdevInfo, pFirstDesc->pNextVirtPtr);
            ASSERT(totalLen == (deFragDescArray[bufIndex].pduLen +
		            deFragDescArray[bufIndex].hdrLen));

            /* adjust len of our only desc */
            pFirstDesc->status.rx.dataLength = (A_UINT16)totalLen;

            /* XXX what if the desc are the last ones on q, and self-linked? */
            pFirstDesc->pNextVirtPtr = NULL;
#endif

            /* free the partial frame back to the pool */
            pReassembled = deFragDescArray[bufIndex].pDesc;
            deFragDescArray[bufIndex].inUse = FALSE;

            /* return the completed assembled frame */
            return pReassembled;
        }

        return NULL;
    }
}

/*******************************************************************************
* wlanArFree - free entities of De-fragmentation Mpdu partial arrays
*
* the function is called in the de-fragmentation process
*
* RETURNS:    A_ERROR if unsuccessful, no entries free
*            the index of a free entry if successful
*/
A_UINT32 wlanDeFragArFree(WLAN_DEV_INFO *pdevInfo)
{
    PARTIAL_DESC *deFragDescArray = pdevInfo->defragLanes;
    A_UINT32     arrayIndex;

    for (arrayIndex = 0; arrayIndex < pdevInfo->numDefragLanes; arrayIndex++) {
        if (deFragDescArray[arrayIndex].inUse == FALSE) {
            return arrayIndex;
        }
    }

    return A_ERROR;
}

/*******************************************************************************
* wlanArSearch - search entities of De-fragmentation Mpdu partial arrays to
*                find the entity with matching sequence number, fragment number
*                destination address.
*
* the function is called in the de-fragmentation process
*
* RETURNS:   -1 if unsuccessful, no match found
*            the index of the first match entry if successful
*/
A_UINT32 wlanDeFragArSearch(WLAN_DEV_INFO *pdevInfo, WLAN_MACADDR *addr,
                            A_UINT16 seqNum, A_UINT16 fragNum)
{
    PARTIAL_DESC *deFragDescArray = pdevInfo->defragLanes;
    A_UINT32     arrayIndex;

    for (arrayIndex = 0; arrayIndex < pdevInfo->numDefragLanes; arrayIndex++) {
        if ((deFragDescArray[arrayIndex].inUse == TRUE) &&
            (deFragDescArray[arrayIndex].rsn == (A_UINT16)seqNum) &&
            (deFragDescArray[arrayIndex].rCur == (fragNum - 1)) &&
            (A_MACADDR_COMP(deFragDescArray[arrayIndex].rta, addr) == 0))
        {
            return arrayIndex;
        }
    }

    return A_ERROR;
}

/*******************************************************************************
* wlanDeFragArAge - free entities of De-fragmentation Mpdu partial arrays which
*                   age > maxReceiveLifeTime.
*                   If we can't find any entries that exceeds maxReceiveLifeTime,
*                   then we'll use the entry with the oldest timestamp.
*/
void
wlanDeFragArAge(WLAN_DEV_INFO *pdevInfo, A_BOOL purgeOldest)
{
    PARTIAL_DESC *pDefragTbl = (PARTIAL_DESC *)&pdevInfo->defragLanes;
    PARTIAL_DESC *pOldest = NULL;
    A_UINT32     arrayIndex, now;
    A_UINT32     oldestTs = 0;      /* Oldest timestamp */
    

    now = A_MS_TICKGET();
    for (arrayIndex = 0; arrayIndex < pdevInfo->numDefragLanes; arrayIndex++, pDefragTbl++) {
        if (pDefragTbl->inUse == TRUE) {
            if (oldestTs == 0 || oldestTs > pDefragTbl->timeStamp) {
                oldestTs = pDefragTbl->timeStamp;
                pOldest = pDefragTbl;
            }
            if ((now - pDefragTbl->timeStamp) >= pdevInfo->staConfig.maxReceiveLifeTime)
            {
                pDefragTbl->inUse = FALSE;
                freeBuffandDescChain(pdevInfo, pDefragTbl->pDesc);
                pDefragTbl->pDesc = NULL;
                return;
            }
        }
    }
    if (purgeOldest == TRUE) {
        if (pOldest) {
            pOldest->inUse = FALSE;
            freeBuffandDescChain(pdevInfo, pOldest->pDesc);
            pOldest->pDesc = NULL;
        }
    }
}

/*******************************************************************************
* tailDescFind - search for the tail of the descriptor chain.
*
* This function is called in the de-fragmentation process.
*
* RETURNS: the pointer of the tail descriptor
*/
ATHEROS_DESC *
tailDescFind(ATHEROS_DESC *pDesc)
{
    /* traverse list of input descriptors to get tail descriptor */
    while (pDesc->pNextVirtPtr != NULL) {
        pDesc = pDesc->pNextVirtPtr;
    }

    return pDesc;
}

#if defined(VXWORKS) && defined(DEBUG)
void
fragShow(int unit)
{
    WLAN_DEV_INFO *pDev            = gDrvInfo.pDev[unit];
    PARTIAL_DESC  *deFragDescArray = pDev->defragLanes;
    A_UINT32      arrayIndex;

    uiPrintf("FragErrs: %d\n", fragErrs);
    for (arrayIndex = 0; arrayIndex < pDev->numDefragLanes; arrayIndex++) {
        uiPrintf("deFragDescArray[%d] inUse:%d, seqNum:%d, fragNum:%d, hdrLen:%d, pduLen:%d\n",
                 arrayIndex, deFragDescArray[arrayIndex].inUse,
                 deFragDescArray[arrayIndex].rsn,
                 deFragDescArray[arrayIndex].rCur,
                 deFragDescArray[arrayIndex].hdrLen,
                 deFragDescArray[arrayIndex].pduLen);
    }
}
#endif
