/*
 *
 * Copyright  2000-2003 Atheros Communications, Inc.,  All Rights Reserved.
 *
 * General support routines for 802.11 frames
 */

#include "wlantype.h"
#include "wlanproto.h"
#include "wlandrv.h"
#include "wlanext.h"
#include "wlansmeext.h"
#include "ui.h"
#include "display.h"
#include "eap.h"

/* mgtElementGet - Get a pointer to an element in the frame.
 *
 * This routine walks through the frame looking for the
 * information element (802.11 section 7.3.2) in the frame
 * and returns a pointer to the desired element.
 *
 * PARAMETERS:
 *  pmacHdr     Pointer to the 802.11 frame
 *  franeSize   Total size of the frame, including FCS.
 *  elementId   Element ID to search for, e.g. ELE_TIM
 *
 * RETURNS: pointer to INFO_ELEMENT data structure.
 *
 * Returns NULL if it is not a management frame or matching
 * elementId is not found.
 */
INFO_ELEMENT *
mgtElementGet(WLAN_MGT_MAC_HEADER *pmacHdr, A_UINT16 frameSize,
              A_UINT8 elementId)
{
    A_UINT16     subtype;
    A_UINT8      *p1;
    A_UINT16     len, elen;
    INFO_ELEMENT *ie;

    if (pmacHdr->frameControl.fType != FRAME_MGT) {
        return NULL;
    }

    subtype = (A_UINT16) pmacHdr->frameControl.fSubtype;

    switch (subtype) {
    case SUBT_ASSOC_REQ:
        p1 = (A_UINT8 *) &(((WLAN_FRAME_ASSOC_REQ *)pmacHdr)->ssid);
        break;

    case SUBT_ASSOC_RESP:
    case SUBT_REASSOC_RESP:
        p1 = (A_UINT8 *) &(((WLAN_FRAME_ASSOC_RESP *)pmacHdr)->rateSet);
        break;

    case SUBT_REASSOC_REQ:
        p1 = (A_UINT8 *) &(((WLAN_FRAME_REASSOC_REQ *)pmacHdr)->ssid);
        break;

    case SUBT_AUTH:
        if (((WLAN_FRAME_AUTH *)pmacHdr)->transSeqNo == 2) {
            p1 = (A_UINT8 *) &(((WLAN_FRAME_AUTH_CHLG *)pmacHdr)->challenge);
        } else if (((WLAN_FRAME_AUTH *)pmacHdr)->transSeqNo == 3) {
            p1 = (A_UINT8 *) &(((WLAN_FRAME_AUTH_ENCRYPT *)pmacHdr)->challenge);
        } else {
            p1 = NULL;
        }
        break;

    case SUBT_PROBE_REQ:
        p1 = (A_UINT8 *) &(((WLAN_FRAME_PROBE_REQ *)pmacHdr)->ssid);
        break;

    case SUBT_PROBE_RESP:
        p1 = (A_UINT8 *) &(((WLAN_FRAME_PROBE_RESP *)pmacHdr)->buffer);
        break;

    case SUBT_BEACON:
        p1 = (A_UINT8 *) &(((WLAN_FRAME_BEACON *)pmacHdr)->buffer);
        break;

    default:
        p1 = NULL;
    }

    if (p1 == NULL) {
        return NULL;
    }

    /*
     * Now, p1 is the beginning of the variable size fields.
     * len is the offset into the frame.
     */
    len = (A_UINT16)(p1 - (A_UINT8 *)pmacHdr);
    ie = (INFO_ELEMENT *)p1;

    /*
     * A valid element should have at least 2 bytes - elementID,
     * length. value byte may not be there if length is 0 (null ssid)
     */
    while (ie && (len + (ie->length + 2) <= frameSize)) {
        if (ie->elementID == elementId) {
            return ie;
        } else {
            elen = ie->length + 2;
            len += elen;
            ie = (INFO_ELEMENT *)((A_UINT8 *)ie + elen);
        }
    }

    return NULL;
}

void
mgtGetInfoElements(INFO_ELEMENT_SET *pIeSet, INFO_ELEMENT *pIe,
                   const A_UINT16 ieLen)
{
    A_UINT16 len = 0;
    A_UINT16 elen;

    while (pIe && (len + (pIe->length + 2) <= ieLen)) {
        switch (pIe->elementID) {
        case ELE_VENDOR_PRIVATE: {
            VENDOR_IE *pVendor = (VENDOR_IE *)pIe;

            if (!A_BCOMP(ouiMicrosoft, pVendor->oui, sizeof(pVendor->oui))) {
                MS_VENDOR_IE *pMsVendor = (MS_VENDOR_IE *)pIe;

                switch (pMsVendor->ouiType) {
                case WPA_OUI_TYPE:
                    switch (((WPA_IE *)pIe)->version) {
                    case WPAIE_VERSION:
                        pIeSet->pWpaIe = pIe;
                        break;
                    
                    default:
                        /* Other version to be added here */
                        break;
                    }
                    break;
#ifdef WME
                case WMEv1_OUI_TYPE:
                    if (VALID_WMEv1_PARAM_IE((WMEv1_PARAM_IE *)pIe) || 
                        VALID_WMEv1_SHORT_IE((WMEv1_PARAM_IE *)pIe)) {
                        pIeSet->pWmeV1Ie = (WMEv1_PARAM_IE *)pIe;
                    } 
                    break;
#endif
                default:
                    /* Other microsoft Vendor IE cases can be 
                     * added here or be added in a table in pIeSet
                     * which  can be added later 
                     */
                    break;
                }
            } else if (!A_BCOMP(ouiAtheros, pVendor->oui, sizeof(pVendor->oui))) {
                ATH_VENDOR_IE *pAthVendor = (ATH_VENDOR_IE *)pIe;

                switch (pAthVendor->ouiType) {
                case ATH_OUI_TYPE_CAP:
                    if (((ATH_ADVCAP_IE *)pIe)->version == ATH_OUI_VER_CAP) {
                        pIeSet->pAthCap = ((ATH_ADVCAP_IE *)pIe);
                    }
                    break;
                case ATH_OUI_TYPE_WME:
                    if (VALID_ATH_WME_PARAM_IE((ATH_WME_PARAM_IE *)pIe)) {
                        pIeSet->pAthWmeIe = (ATH_WME_PARAM_IE *)pIe;
                    }
                    break;
                default :
                    /* Other Atheros Types can be added here */
                    break;
                }
            }
            break;
        }

        default:
            pIeSet->pInfoElement[eidMap[pIe->elementID]] = pIe;
            break;
        }

        elen  = pIe->length + 2;
        len  += elen;
        pIe   = (INFO_ELEMENT *)((A_UINT8 *)pIe + elen);
    }
}

/* dk header file compatability */
#ifndef WEP_IV_FIELD_SIZE
#define WEP_IV_FIELD_SIZE WEP_IV_FIELD
#endif

/*
 *  pWpaIe may be unaligned!
 */
void
wpaIeEndianChangeEle(WPA_IE *pWpaIe, int direction)
{
    A_UINT16 authCount, cipherCount;
    A_UINT32 *pAuth;

    if (pWpaIe != NULL) {
        A_UINT16    tmp16 = 0;
        A_UINT32    tmp32;
        A_UINT32    i;

        /* swap unicast cipher counter first for rx(0) */
        if (!direction) {
            /* swap the unicast selector count */
            tmp16 = A_swab16(pWpaIe->uCipherCount);
            pWpaIe->uCipherCount = tmp16;

            /* Swap the auth selector count too */
            A_BCOPY(&pWpaIe->uCiphers[pWpaIe->uCipherCount],
                    &tmp16, sizeof(tmp16));
            tmp16 = A_swab16(tmp16);
            A_BCOPY(&tmp16, &pWpaIe->uCiphers[pWpaIe->uCipherCount],
                    sizeof(tmp16));
        }

        cipherCount = pWpaIe->uCipherCount;

        /* validate the element first, set length = 0 if invalid */
        if (cipherCount > 4) {
            pWpaIe->length = 0;
            return;
        }

        /* the OUI is never swapped */

        /* swap WPA spec revision */
        tmp16 = A_swab16(pWpaIe->version);
        pWpaIe->version = tmp16;

        /* swap multicast cipher */
        tmp32 = A_swab32(pWpaIe->mCipher[0]);
        pWpaIe->mCipher[0] = tmp32;

        /* swap unicast ciphers */
        for (i = 0; i < cipherCount; i++) {
            tmp32 = A_swab32(pWpaIe->uCiphers[i]);
            pWpaIe->uCiphers[i] = tmp32;
        }

        /* swap auth selectors */
        A_BCOPY(&pWpaIe->uCiphers[i], &authCount, sizeof(authCount));
        if (authCount != 1) {
            uiPrintf("bad auth count %x\n", authCount);
        }

        pAuth = (A_UINT32 *)((A_UINT8 *)&pWpaIe->uCiphers[cipherCount] +
                             sizeof(A_UINT16));

        for (i = 0; i < authCount; i++) {
            A_BCOPY(pAuth, &tmp32, sizeof(tmp32));
            tmp32 = A_swab32(tmp32);
            A_BCOPY(&tmp32, pAuth, sizeof(tmp32));
            pAuth++;
        }

        /* swap unicast cipher and auth counter afterwards for tx(1) */
        if (direction) {
            A_BCOPY(&pWpaIe->uCipherCount, &tmp16, sizeof(tmp16));
            tmp16 = A_swab16(tmp16);
            A_BCOPY(&tmp16, &pWpaIe->uCipherCount, sizeof(tmp16));

            tmp16 = A_swab16(authCount);
            A_BCOPY(&tmp16, &pWpaIe->uCiphers[cipherCount], sizeof(tmp16));
        }
    }
}

void
athWmeIeGetInfo(WLAN_DEV_INFO *pDev, OP_BSS *pOpBss, ATH_WME_PARAM_IE *pAthWmeIe)
{
    WMEv1_AC_RECORD *pAc;
    int len;
    if (!(pDev->staConfig.abolt & ABOLT_WME_ELE)) {
        return;
    }

    if ((!pAthWmeIe) || (!VALID_ATH_WME_PARAM_IE(pAthWmeIe))) {
        return;
    }

    pAc = &pAthWmeIe->info.ac[0];
    len = pAthWmeIe->length + FIELD_OFFSET(ATH_WME_PARAM_IE, oui[0]);
    while (len >= FIELD_OFFSET(ATH_WME_PARAM_IE, info.ac[1])) {
        pOpBss->phyChAPs.ac[pAc->aci] = *pAc;
#ifdef BIG_ENDIAN
        pOpBss->phyChAPs.ac[pAc->aci].txOpLimit = A_swab16(pOpBss->phyChAPs.ac[pAc->aci].txOpLimit);
#endif
        len -= sizeof(WMEv1_AC_RECORD);
        ++pAc;
    }
}

#ifdef WME
void
wmeV1IeGetInfo(WLAN_DEV_INFO *pDev, OP_BSS *pOpBss, WMEv1_PARAM_IE *pWmeV1Ie)
{
    WMEv1_AC_RECORD *pAc;
    int len;
    if  (!pDev->staConfig.WmeEnabled) {
        return;
    }

    if ((!pWmeV1Ie) || (!VALID_WMEv1_PARAM_IE(pWmeV1Ie))) {
        return;
    }

    pAc = &pWmeV1Ie->info.ac[0];
    len = pWmeV1Ie->length + FIELD_OFFSET(WMEv1_PARAM_IE, oui[0]);
    while (len >= FIELD_OFFSET(WMEv1_PARAM_IE, info.ac[1])) {
        pOpBss->phyChAPs.ac[pAc->aci] = *pAc;
#ifdef BIG_ENDIAN
        pOpBss->phyChAPs.ac[pAc->aci].txOpLimit = A_swab16(pOpBss->phyChAPs.ac[pAc->aci].txOpLimit);
#endif
        len -= sizeof(WMEv1_AC_RECORD);
        ++pAc;
    }
}
INFO_ELEMENT *
wmeV1IeGenerate(WLAN_DEV_INFO *pDev, OP_BSS *pOpBss, INFO_ELEMENT *pElement)
{
    WMEv1_PARAM_IE *pWmeV1Ie = (WMEv1_PARAM_IE *)pElement;
    int size = ACI_MAX;

    if (!pDev->staConfig.WmeEnabled) {
        return pElement;
    }

    pWmeV1Ie->elementID  = ELE_VENDOR_PRIVATE;
    pWmeV1Ie->length     = (A_UINT8)FIELD_OFFSET(WMEv1_PARAM_IE, info.ac[size]) -
                               (A_UINT8)FIELD_OFFSET(WMEv1_PARAM_IE, oui);
    A_BCOPY(ouiMicrosoft, pWmeV1Ie->oui, sizeof(pWmeV1Ie->oui));
    pWmeV1Ie->ouiType    = WMEv1_OUI_TYPE;
    pWmeV1Ie->ouiSubType = WMEv1_PARAM_OUI_SUBTYPE;
    pWmeV1Ie->version    = WMEv1_PARAM_OUI_VERSION;

    A_BCOPY(&pOpBss->phyChAPs, &pWmeV1Ie->info, FIELD_OFFSET(WMEv1_PARAM, ac[size]));

#ifdef BIG_ENDIAN
    pWmeV1Ie->info.ac[ACI_BE].txOpLimit = A_swab16(pWmeV1Ie->info.ac[ACI_BE].txOpLimit);
    pWmeV1Ie->info.ac[ACI_BK].txOpLimit = A_swab16(pWmeV1Ie->info.ac[ACI_BK].txOpLimit);
    pWmeV1Ie->info.ac[ACI_Vi].txOpLimit = A_swab16(pWmeV1Ie->info.ac[ACI_Vi].txOpLimit);
    pWmeV1Ie->info.ac[ACI_Vo].txOpLimit = A_swab16(pWmeV1Ie->info.ac[ACI_Vo].txOpLimit);
#endif

    return ((INFO_ELEMENT *)(&pWmeV1Ie->info.ac[size]));
}

INFO_ELEMENT *
wmeV1ShortIeGenerate(WLAN_DEV_INFO *pDev, INFO_ELEMENT *pElement)
{
    WMEv1_PARAM_IE *pWmeV1Ie = (WMEv1_PARAM_IE *)pElement;

    if (!pDev->staConfig.WmeEnabled) {
        return pElement;
    }

    pWmeV1Ie->elementID  = ELE_VENDOR_PRIVATE;
    pWmeV1Ie->length     = (A_UINT8)FIELD_OFFSET(WMEv1_PARAM_IE, info._pad) -
                               (A_UINT8)FIELD_OFFSET(WMEv1_PARAM_IE, oui);
    A_BCOPY(ouiMicrosoft, pWmeV1Ie->oui, sizeof(pWmeV1Ie->oui));
    pWmeV1Ie->ouiType    = WMEv1_OUI_TYPE;
    pWmeV1Ie->ouiSubType = WMEv1_SHORT_OUI_SUBTYPE;
    pWmeV1Ie->version    = WMEv1_PARAM_OUI_VERSION;
    pWmeV1Ie->info.info  = 0;

    return ((INFO_ELEMENT *)(&pWmeV1Ie->info._pad));
}

#endif

INFO_ELEMENT *
athWmeIeGenerate(WLAN_DEV_INFO *pDev, OP_BSS *pOpBss, INFO_ELEMENT *pElement)
{
    ATH_WME_PARAM_IE *pAthWmeIe = (ATH_WME_PARAM_IE *)pElement;

    if (!(pDev->staConfig.abolt & ABOLT_WME_ELE)) {
        return pElement;
    }

    /* use only the first AC for now */
    pAthWmeIe->elementID  = ELE_VENDOR_PRIVATE;
    pAthWmeIe->length     = (A_UINT8)FIELD_OFFSET(ATH_WME_PARAM_IE, info.ac[ACI_BK]) -
                               (A_UINT8)FIELD_OFFSET(ATH_WME_PARAM_IE, oui);
    A_BCOPY(ouiAtheros, pAthWmeIe->oui, sizeof(pAthWmeIe->oui));
    pAthWmeIe->ouiType    = ATH_OUI_TYPE_WME;
    pAthWmeIe->ouiSubType = ATH_OUI_SUBTYPE_WME_PARAM;
    pAthWmeIe->version    = ATH_OUI_VERSION_WME_PARAM;
    A_BCOPY(&pOpBss->phyChAPs, &pAthWmeIe->info, FIELD_OFFSET(WMEv1_PARAM, ac[ACI_BK]));

#ifdef BIG_ENDIAN
    pAthWmeIe->info.ac[ACI_BE].txOpLimit = A_swab16(pAthWmeIe->info.ac[ACI_BE].txOpLimit);
#endif

    return ((INFO_ELEMENT *)(&pAthWmeIe->info.ac[ACI_BK]));
}

#ifdef BIG_ENDIAN
/*
 * cryptoEndianChange: currently the IV is constructed and
 * inspected in host order and has to be swapped before tx
 * and/or after rx
 * direction: 1 == tx, 0 == rx
 */
void
cryptoEndianChange(WLAN_FRAME_HEADER *pFrame, int direction)
{
    FRAME_CONTROL *pFc = &pFrame->frameControl;
#ifdef OLDBAD
    WLAN_FRAME_3_DATA_ENCRYPT_EXTIV *p3ef;      /* 3 addr encrypted frame */
    WLAN_FRAME_DATA_ENCRYPT_EXTIV_PAD *p4ef;    /* 4 addr encrypted frame */
#endif
    A_UINT32 *pIV, *pextIV;

    ASSERT(pFc->wep);
    if (pFc->fType != FRAME_DATA) {
        return;
    }

    pIV = (A_UINT32 *)(GETIVPTR(pFrame));
    pextIV = (A_UINT32 *)(GETEXTIVPTR(pFrame));

#ifdef OLDbAD
    if (pFc->ToDS && pFc->FromDS) {
        p4ef = (WLAN_FRAME_DATA_ENCRYPT_EXTIV_PAD *)pFrame;
        if (!direction) {
            p4ef->iv = A_swab32(p4ef->iv);
            if (p4ef->iv & EXT_IV_BIT) {
                p4ef->extIv = A_swab32(p4ef->extIv);
            }
        } else {
            if (p4ef->iv & EXT_IV_BIT) {
                p4ef->extIv = A_swab32(p4ef->extIv);
            }
            p4ef->iv = A_swab32(p4ef->iv);
        }
    } else {
        p3ef = (WLAN_FRAME_3_DATA_ENCRYPT_EXTIV *)pFrame;
        if (!direction) {
            p3ef->iv = A_swab32(p3ef->iv);
            if (p3ef->iv & EXT_IV_BIT) {
                p3ef->extIv = A_swab32(p3ef->extIv);
            }
        } else {
            if (p3ef->iv & EXT_IV_BIT) {
                p3ef->extIv = A_swab32(p3ef->extIv);
            }
            p3ef->iv = A_swab32(p3ef->iv);
        }
    }
#endif
    if (!direction) {
        *pIV = A_swab32(*pIV);
        if (*pIV & EXT_IV_BIT) {
            *pextIV = A_swab32(*pextIV);
        }
    } else {
        if (*pIV & EXT_IV_BIT) {
            *pextIV = A_swab32(*pextIV);
        }
        *pIV = A_swab32(*pIV);
    }

    return;
}

static void
athIeEndianChange(ATH_ADVCAP_IE *pAthCap, int direction)
{
    A_UINT16 tmp16 = 0;
    if (pAthCap != NULL) {
        tmp16 = A_swab16(pAthCap->version);
        pAthCap->version = tmp16;
        tmp16 = A_swab16(pAthCap->defKeyIndex);
        pAthCap->defKeyIndex = tmp16;
    }
}

static void
mgtIEEndianChange(INFO_ELEMENT *ie,const A_UINT16 ieLen, int direction)
{
    A_UINT16 len = 0;
    A_UINT16 elen;

    while (ie && (len + (ie->length + 2) <= ieLen)) {

        switch(ie->elementID) {
        case ELE_VENDOR_PRIVATE:
            if (!A_BCOMP(ouiMicrosoft,((VENDOR_IE *)ie)->oui, 3)) {
                
                switch (((MS_VENDOR_IE *)ie)->ouiType) {
                case WPA_OUI_TYPE:
                    wpaIeEndianChangeEle((WPA_IE *)ie, direction);
                    break;

                default:
                    /* Other microsoft Vendor IE cases can be 
                     * added here */
                     break;
                }
            } else if (!A_BCOMP(ouiAtheros, ((VENDOR_IE *)ie)->oui, 3)) {
                
                switch (((ATH_VENDOR_IE *)ie)->ouiType) {
                case ATH_OUI_TYPE_CAP:
                    athIeEndianChange((ATH_ADVCAP_IE *)ie, direction);
                    break;
                case ATH_OUI_TYPE_WME:
                    /* no endian change required here */
                    break;

                default :
                    /* Other Atheros Types can be added here */
                    break;
                }
            } break;

        default:
            /* Other Elements ignored here */
            break;
        }
        elen = ie->length + 2;
        len += elen;
        ie = (INFO_ELEMENT *)((A_UINT8 *)ie + elen);
    }
}

/*
 * mgtFrameEndianChange: currently the mgmt frame elements are
 * constructed and inspected in host order and have to be swapped
 * before tx and/or after rx
 * direction: 1 == tx, 0 == rx
 */
void
mgtFrameEndianChange(WLAN_MGT_MAC_HEADER *pFrame, A_UINT16 len, int direction)
{
    FRAME_CONTROL *pFc           = &pFrame->frameControl;
    A_UINT8       *pInfoElements = NULL;
    A_UINT16       infoElemLen   = 0;
    A_UINT16       tmp16;

    ASSERT(pFc->fType == FRAME_MGT);

    switch (pFc->fSubtype) {
    case SUBT_ASSOC_REQ:
    case SUBT_REASSOC_REQ: {
        tmp16 = A_swab16(*((A_UINT16 *)&((WLAN_FRAME_ASSOC_REQ *)pFrame)->capInfo));
        ((WLAN_FRAME_ASSOC_REQ *)pFrame)->capInfo = *(CAP_INFO *)&tmp16;
        ((WLAN_FRAME_ASSOC_REQ *)pFrame)->listenInterval =
            A_swab16(((WLAN_FRAME_ASSOC_REQ *)pFrame)->listenInterval);

        if (pFc->fSubtype == SUBT_ASSOC_REQ) {
            pInfoElements = (A_UINT8 *) &(((WLAN_FRAME_ASSOC_REQ *)pFrame)->ssid);
            infoElemLen = (A_UINT16)(len - (pInfoElements - 
                          (A_UINT8 *)&((WLAN_FRAME_ASSOC_REQ *)pFrame)->macHeader));
        }
        else if (pFc->fSubtype == SUBT_REASSOC_REQ) {
            pInfoElements = (A_UINT8 *) &(((WLAN_FRAME_REASSOC_REQ *)pFrame)->ssid);
            infoElemLen = (A_UINT16)(len - (pInfoElements - 
                          (A_UINT8 *)&(((WLAN_FRAME_REASSOC_REQ  *)pFrame)->macHeader)));
        }
        mgtIEEndianChange((INFO_ELEMENT *)pInfoElements,infoElemLen,direction);
        break;
    }

    case SUBT_ASSOC_RESP:
    case SUBT_REASSOC_RESP: {
        tmp16 = A_swab16(*((A_UINT16 *)&((WLAN_FRAME_ASSOC_RESP *)pFrame)->capInfo));
        ((WLAN_FRAME_ASSOC_RESP *)pFrame)->capInfo = *(CAP_INFO *)&tmp16;
        ((WLAN_FRAME_ASSOC_RESP *)pFrame)->statusCode =
            A_swab16(((WLAN_FRAME_ASSOC_RESP *)pFrame)->statusCode);
        ((WLAN_FRAME_ASSOC_RESP *)pFrame)->assocId =
            A_swab16(((WLAN_FRAME_ASSOC_RESP *)pFrame)->assocId);

        pInfoElements = (A_UINT8 *) &(((WLAN_FRAME_ASSOC_RESP *)pFrame)->rateSet);
        infoElemLen = (A_UINT16)(len - (pInfoElements - 
                                (A_UINT8 *)&(((WLAN_FRAME_ASSOC_RESP *)pFrame)->macHeader)));
        mgtIEEndianChange((INFO_ELEMENT *)pInfoElements,infoElemLen,direction);
        break;
    }

    case SUBT_PROBE_RESP:
    case SUBT_BEACON: {
        /* swab 32 bit fields independently, don't need to swap them */
        ((WLAN_FRAME_PROBE_RESP *)pFrame)->timestamp.high =
            A_swab32(((WLAN_FRAME_PROBE_RESP *)pFrame)->timestamp.high);
        ((WLAN_FRAME_PROBE_RESP *)pFrame)->timestamp.low =
            A_swab32(((WLAN_FRAME_PROBE_RESP *)pFrame)->timestamp.low);
        ((WLAN_FRAME_PROBE_RESP *)pFrame)->beaconInterval =
            A_swab16(((WLAN_FRAME_PROBE_RESP *)pFrame)->beaconInterval);
        tmp16 = A_swab16(*((A_UINT16 *)&((WLAN_FRAME_PROBE_RESP *)pFrame)->capInfo));
        ((WLAN_FRAME_PROBE_RESP *)pFrame)->capInfo = *(CAP_INFO *)&tmp16;

        pInfoElements = (A_UINT8 *) &(((WLAN_FRAME_BEACON *)pFrame)->buffer);
        infoElemLen = (A_UINT16)(len - (pInfoElements - 
                      (A_UINT8 *)&((WLAN_FRAME_BEACON *)pFrame)->macHeader));

        mgtIEEndianChange((INFO_ELEMENT *)pInfoElements,infoElemLen,direction);
        break;
    }

    case SUBT_AUTH:
        if (pFc->wep) {
            ((WLAN_FRAME_AUTH_ENCRYPT *)pFrame)->algNo =
                A_swab16(((WLAN_FRAME_AUTH_ENCRYPT *)pFrame)->algNo);
            ((WLAN_FRAME_AUTH_ENCRYPT *)pFrame)->transSeqNo =
                A_swab16(((WLAN_FRAME_AUTH_ENCRYPT *)pFrame)->transSeqNo);
            ((WLAN_FRAME_AUTH_ENCRYPT *)pFrame)->statusCode =
                A_swab16(((WLAN_FRAME_AUTH_ENCRYPT *)pFrame)->statusCode);
            ((WLAN_FRAME_AUTH_ENCRYPT *)pFrame)->iv =
                A_swab32(((WLAN_FRAME_AUTH_ENCRYPT *)pFrame)->iv);
        } else {
            ((WLAN_FRAME_AUTH *)pFrame)->algNo =
                A_swab16(((WLAN_FRAME_AUTH *)pFrame)->algNo);
            ((WLAN_FRAME_AUTH *)pFrame)->transSeqNo =
                A_swab16(((WLAN_FRAME_AUTH *)pFrame)->transSeqNo);
            ((WLAN_FRAME_AUTH *)pFrame)->statusCode =
                A_swab16(((WLAN_FRAME_AUTH *)pFrame)->statusCode);
        }
        break;

    case SUBT_DEAUTH:
    case SUBT_DISASSOC:
        /* reasonCode has same offset in both frame types */
        ((WLAN_FRAME_DEAUTH *)pFrame)->reasonCode =
            A_swab16(((WLAN_FRAME_DEAUTH *)pFrame)->reasonCode);

        break;

        /*
         * rest of management subtypes don't have any fields that
         * need to be byte swapped
         */
    }
}
#endif

/*******************************************************************************
* lengthGet - calculate the length of the descriptor chain.
*
* This function is called in the de-fragmentation process
*
* RETURNS: length
*/
A_UINT16
lengthGet(ATHEROS_DESC *pDesc)
{
    A_UINT32 len = 0;

    /* traverse list of input descriptors to get length */
    while (pDesc != NULL) {
        len  += pDesc->status.rx.dataLength;
        pDesc = pDesc->pNextVirtPtr;
    }

    return (A_UINT16)len;
}

/*
* headerLengthGet - calculate the header length of the descriptor chain.
*
* Assumes the frame is in host order, not network order.  The frame is
* interpreted to be in host order, not network order.
*
* RETURNS: frame header length, including any hardware padding.
*/
A_UINT16
headerLengthGet(ATHEROS_DESC *pDesc)
{
    WLAN_DATA_MAC_HEADER3 *pWlanHdr;
    A_UINT16               hdrLen = 0;
    FRAME_CONTROL         *frameControl;

    pWlanHdr     = pDesc->pBufferVirtPtr.header3;
    frameControl = &pWlanHdr->frameControl;

    switch (frameControl->fType) {
    case FRAME_MGT:
        hdrLen = MAC_HDR_MGT_SIZE;
        break;

    case FRAME_CTRL:
        if (frameControl->fSubtype == SUBT_CTS ||
            frameControl->fSubtype == SUBT_ACK)
        {
            hdrLen = MAC_HDR_CTRL1_SIZE;
        } else {
            hdrLen = MAC_HDR_CTRL2_SIZE;
        }
        break;

    case FRAME_DATA:

        if ((frameControl->FromDS == 1) && (frameControl->ToDS == 1)) {
            hdrLen = WLAN_HDR4_SIZE;
        } else {
            hdrLen = WLAN_HDR_SIZE;
        }
        if (frameControl->fSubtype == SUBT_QOS) {
            hdrLen += sizeof (QOS_CNTL);
        }

        break;
    default:
        uiPrintf("Bad frame type\n");
        break;
    }

    return A_ROUNDUP(hdrLen, sizeof(A_UINT32));
}

A_UINT32
icvLengthGet(WLAN_PRIV_RECORD *pPriv)
{
    if (pPriv == NULL) {
        return 0;
    }

    if (pPriv->keyType == PRIV_KEY_TYPE_AES_CCM ||
        pPriv->keyType == PRIV_KEY_TYPE_AES_OCB)
    {
        return AES_ICV_FIELD_SIZE;
    }

    return WEP_ICV_FIELD_SIZE;
}

/*
 * ivLengthGet - calculate the length of the IV.  Return 0 if not present.
 * calculate the iv length based on key type
 */
A_UINT16
ivLengthGet(ATHEROS_DESC *pDesc)
{
    WLAN_DATA_MAC_HEADER3 *pWlanHdr     = pDesc->pBufferVirtPtr.header3;
    A_UINT16               ivLen        = 0;
    WLAN_PRIV_RECORD       *pPriv       = pDesc->pKey;

    if (pPriv && pWlanHdr->frameControl.wep) {
        ivLen = WEP_IV_FIELD_SIZE;

        if (pPriv->keyType == PRIV_KEY_TYPE_AES_CCM ||
            pPriv->keyType == PRIV_KEY_TYPE_TKIP ||
            pPriv->keyType == PRIV_KEY_TYPE_TKIP_SW)
        {
            ivLen += EXT_IV_FIELD_SIZE;
        }
    }
    return ivLen;
}

/*
 *  Assumption:  the frame is still in 802.11 encapsulation, not DIX.
 */
A_BOOL
isEapolFrame(WLAN_DEV_INFO *pdevInfo, A_UINT8 *llc)
{
    if (A_BCOMP(llc, (char *)&eapolSnap, sizeof(eapolSnap)) == 0) {
        return TRUE;
    }

    return FALSE;
}


/*
 *  pLLChdr is a ptr to the start of the LLC header
 */
A_BOOL
isL2UF(A_UINT8 *pLLChdr)
{
    static const A_UINT8 l2uf[] = { 0x00, 0x00, 0xf5, 0x81, 0x80, 0x00 };

    if (A_BCOMP(pLLChdr, l2uf, sizeof(l2uf)) == 0) {
        return TRUE;
    }
    return FALSE;
}


#if defined(DEBUG)

int doIpDecode = 0;
#define ALL_IPDECODE  -1
#define TIME_IPDECODE  0x80000000
#define DHCP_IPDECODE  0x01
#define IP_IPDECODE    0x02
#define EAPOL_IPDECODE 0x04

static A_UINT8 ipSnapHdr[] = {
    0xAA, 0xAA, 0x03, 0x00, 0x00, 0x00, 0x08, 0x00
};

static A_BOOL
isIpFrag(struct ip *ip)
{
    if ((ip->ip_off & IP_MF) || (ip->ip_off & IP_OFFMASK) != 0) {
        return TRUE;
    }
    return FALSE;
}

static A_BOOL
isIpOptionLess(struct ip *ip)
{
    return (ip->ip_hl == 5);
}


/*
 *  Print out IP and higher protocol information.  Assume buffer
 *  is in network order, and all protocol headers are pulled up 
 *  into first buffer.
 */
void
ipDecode(WLAN_DEV_INFO *pDevInfo, ATHEROS_DESC *pDesc, A_BOOL isRx)
{
    A_UINT8        *snapHdr;
    A_INT32        dot11HdrLen = headerLengthGet(pDesc) + ivLengthGet(pDesc);
    A_INT32        bufferLen   = isRx ? pDesc->status.rx.dataLength :
                                 pDesc->bufferLength;
    struct ip      *ipHdr;
    struct udphdr  *udpHdr;
    struct dhcphdr *dhcpHdr;
    A_UINT16       DPort, SPort;
    A_BOOL         fragment = FALSE;
    A_BOOL         options = FALSE;
    A_UINT8        protocol;
    A_UINT32       myIpDecode = 0;
    WLAN_FRAME_HEADER *pWlanHdr;
    EAPOL_HEADER   *eapolHdr;

    if (!doIpDecode) {
        return;
    }

    pWlanHdr = pDesc->pBufferVirtPtr.header;

    /* LLC is always co-resident with dot11 header */
    snapHdr = (A_UINT8 *)(pDesc->pBufferVirtPtr.byte + dot11HdrLen);

    if (bufferLen < (A_INT32)(dot11HdrLen + sizeof(ipSnapHdr))) {
        return;
    }

    if (isEapolFrame(pDevInfo, snapHdr) == TRUE) {
        myIpDecode = EAPOL_IPDECODE;
        if (isRx) {
            eapolHdr = (EAPOL_HEADER *)(snapHdr + sizeof(eapolSnap));
        } else {
            /* our encaps policy means we have to start at next desc */
            if (pDesc->bufferLength < dot11HdrLen +
                sizeof(eapolSnap) + sizeof(*eapolHdr))
            {
                pDesc = pDesc->pNextVirtPtr;
                eapolHdr = (EAPOL_HEADER *)pDesc->pBufferVirtPtr.ptr;
            } else {
                eapolHdr = (EAPOL_HEADER *)(snapHdr + sizeof(eapolSnap));
            }
        }
    }

    if (A_BCOMP(snapHdr, ipSnapHdr, sizeof(ipSnapHdr)) == 0) {
        myIpDecode = IP_IPDECODE;
        if (isRx) {
            ipHdr = (struct ip *)(snapHdr + sizeof(ipSnapHdr));
        } else {
            /* our encaps policy means we have to start at next desc */
            if (pDesc->bufferLength < dot11HdrLen +
                sizeof(ipSnapHdr) + sizeof(*ipHdr))
            {
                pDesc = pDesc->pNextVirtPtr;
                ipHdr = (struct ip *)pDesc->pBufferVirtPtr.ptr;
            } else {
                ipHdr = (struct ip *)(snapHdr + sizeof(ipSnapHdr));
            }
        }

        if (isIpFrag(ipHdr)) {
            fragment = TRUE;
        } else if (! isIpOptionLess(ipHdr)) {
            options = TRUE;
        } else {
            protocol = ipHdr->ip_p;
            if (ipHdr->ip_p == IPPROTO_UDP || ipHdr->ip_p == IPPROTO_TCP) { 
                if (!isRx && pDesc->bufferLength <
                    (sizeof(*ipHdr) + sizeof(*udpHdr)))
                {
                    if (pDesc->pNextVirtPtr == NULL) {
                        return;
                    }
                    pDesc  = pDesc->pNextVirtPtr;
                    udpHdr = (struct udphdr *)pDesc->pBufferVirtPtr.ptr;
                } else {
                    udpHdr = (struct udphdr *)(ipHdr + 1);
                }

                /* We're only intereted in port numbers, so use UDP shell */
                DPort = be2cpu16(udpHdr->uh_dport);
                SPort = be2cpu16(udpHdr->uh_sport);
                if (ipHdr->ip_p == IPPROTO_UDP &&
                    (DPort == IPPORT_BOOTPS || DPort == IPPORT_BOOTPC))
                {
                    myIpDecode = DHCP_IPDECODE;
                }
           }
        }
    }

    if (doIpDecode == -1 || (doIpDecode & myIpDecode)) {
        if (isRx) {
            uiPrintf("RX: ");
        } else {
            uiPrintf("TX: ");
        }
        if (pDevInfo->staConfig.opMode == OP_MODE_STA) {
            printMacAddress(pWlanHdr->address2);
        } else {
            printMacAddress(pWlanHdr->address3);
        }
        uiPrintf("-> ");
        printMacAddress(pWlanHdr->address1);

        if (doIpDecode & myIpDecode & EAPOL_IPDECODE) {
            uiPrintf(": EAPOL ");
            switch (eapolHdr->eapolType) {
            case (EAPOL_TYPE_PACKET):
                uiPrintf("Packet : ");
                break;
            case (EAPOL_TYPE_START):
                uiPrintf("Start : ");
                break;
            case (EAPOL_TYPE_LOGOFF):
                uiPrintf("Logoff : ");
                break;
            case (EAPOL_TYPE_KEYMSG):
                uiPrintf("Keymsg : ");
                break;
            case (EAPOL_TYPE_ASF_ALERT):
                uiPrintf("ASF Alert : ");
                break;
            default:
                uiPrintf("Unknown Type %d : ", eapolHdr->eapolType);
                break;
            }
        }

        if (doIpDecode & (IP_IPDECODE | DHCP_IPDECODE)) {
            uiPrintf(": IP: ");
            if (fragment) {
                uiPrintf("fragment");
            } else if (options) {
                uiPrintf("options present");
            } else {
                if (ipHdr->ip_p == IPPROTO_UDP) {
                    uiPrintf("UDP(%d/%d): ", SPort, DPort);
                } else if (ipHdr->ip_p == IPPROTO_TCP) {
                    uiPrintf("TCP(%d/%d): ", SPort, DPort);
                }

                if (doIpDecode & myIpDecode & DHCP_IPDECODE) {
                    if (!isRx && pDesc->bufferLength <
                        (sizeof(*udpHdr) + sizeof(*dhcpHdr)))
                    {
                        if (pDesc->pNextVirtPtr == NULL) {
                            return;
                        }
                        pDesc   = pDesc->pNextVirtPtr;
                        dhcpHdr = (struct dhcphdr *)pDesc->pBufferVirtPtr.ptr;
                    } else {
                        dhcpHdr = (struct dhcphdr *)(udpHdr + 1);
                    }
                    uiPrintf("DHCP ");
                    if (isRx) {
                        /* Validate it's to me */
                        if (A_MACADDR_COMP(&pDevInfo->localSta->macAddr,
                                       (WLAN_MACADDR *)dhcpHdr->chaddr) != 0)
                        {
                            uiPrintf("addr mismatch.\n");
                            return;
                        }
                    }

                    switch (dhcpHdr->op) {
                    case BOOTREQUEST:
                        uiPrintf("REQUEST");
                        break;
                    case BOOTREPLY:
                        uiPrintf("REPLY");
                        break;
                    }
                }
            }
        }
        if (doIpDecode & TIME_IPDECODE) {
            uiPrintf(" (%d)", A_MS_TICKGET());
        }
        uiPrintf("\n");
    }
}
#endif /* DEBUG */
