/*
 *
 * Key table and key cache mgmt code.  HW support is in the HAL.
 *
 * Copyright © 2000-2003 Atheros Communications, Inc.,  All Rights Reserved.
 *
 *****
 *
 * The key support is split into 3 levels.  The key table, they
 * key cache, and the hal.  The first two are implemented in this file.
 *
 * Data structures:
 *      devInfo->keyCache[] -- shadow table of WLAN_PRIV_RECORDs
 *                              representing the key cache
 *      devInfo->keyTable[] -- global table of WLAN_PRIV_RECORDs
 *                              of all possible keys; some entries
 *                              may be shared by multiple SIBs
 *      devInfo->keyCacheSib[] -- reverse mapping from keyCache to SIB entry
 *
 * There are two types of key table entries, static and dynamic.
 *
 * Dynamic keys are meant for temporary use, for instance with key
 * mgmt protocols such as 802.1x.  Static keys are meant to be loaded
 * at initialization time.  The NDIS driver does update the static
 * keys via the XP OIDs, so the keys are not truly static.  The
 * AP does treat them as static.
 *
 * Shared keys 1-4 must be static keys.
 *
 * The key table is allocated at start-up as one array for both
 * key types with wlanKeyTableInit().  Once initialized the key
 * table is fixed in size.  The caller should account for the
 * size it will need during its lifetime.
 *
 * NDIS uses 5 static keys.  4 shared keys and 1 unique key.
 *
 * The AP allocates static keys for each configured key, and
 * one dynamic key for each potential station.
 *
 * Static keys are added with wlanKeyTableAdd() with their index
 * into the key table.  Dynamic keys are added with
 * wlanKeyTableAddDynamic().
 *
 * Key mgmt and backing store is not managed here, but by the
 * caller.
 *
 * ---
 *
 * The hardware key cache holds keys for the encryption hardware.  It
 * contains 47 bits of the IEEE mac address, a key type, key length
 * and up to 128b of key material.
 *
 * The transmit path *always* allocates a key cache entry per
 * destination SIB.  Software retry uses the dest mask bit to
 * filter frames following a non-ACKED frame and this is indexed
 * by the key cache entry in the descriptor.  The entry index
 * is stored in pSib->hwIndex.
 *
 * A common scenario is when there is some initial key setup by
 * some config mgmt layer.  This key may or may not be used for
 * encryption, and later key mgmt code might derive new keys.
 * The lifecycle of key indices for the STA goes like this:
 * AuthRequest: wlanKeyEnable: set pSib->pPriv to the sw key table entry.
 *                             no key cache slot is allocated
 * Tx path: setupEncrypt: allocate key cache slot, save this
 *                        index in pSib->hwIndex
 * AssocResp: wlanKeyUpdate: key type is updated as a result of the
 *                           cipher suite negotiation
 * DisassocReq: hw key index is NOT removed from pSib, but we
 *              use wlanKeyInvalidateAll() to go thru the whole
 *              key cache and mark all users of a given sw key table
 *              index as not being PRIV_UNIQUEKEY users anymore, clear
 *              their backptrs to sw key table, and clear their cache
 *              entry with halResetKeyCacheEntry().
 *
 * The hardware key cache is a limited resource.  For the AP,
 * if more stations are associated than the availiable non-shared
 * key cache slots, then the software will try to find an
 * idle key cache entry to evict.  If the victim is using a
 * unique key, the rx side will have to fault in the key.
 * This will result in one dropped frame.
 *
 * The key cache code assumes it owns pSib->hwIndex and it
 * is guarded by keySem, even upon removal as wlanKeyCacheFree()
 * must be called when retiring a sib.
 *
 * ---
 *
 * To add a static key:
 *      wlanKeyTableAdd();
 *      wlanKeyCacheSet();
 *
 * To add a unique key:
 *      wlanKeyTableAdd();
 *
 * To associate a key with a SIB entry:
 *      wlanKeyEnable();                // link pSib and the key
 *      wlanKeyCacheFault();            // called per tx frame
 *
 * To use a dynamic key:
 *      wlanKeyTableAddDynamic();       // update keytable and enable
 *      wlanKeyCacheFault();            // called per tx frame
 *
 */

#include "wlantype.h"
#include "wlanproto.h"
#include "wlandrv.h"
#include "wlansme.h"
#include "wlansmeext.h"
#include "wlanext.h"
#include "display.h"
#include "ui.h"

#include "crypto/ocb.h"
#include "crypto/ccm.h"
#include "crypto/tkip.h"
#include "usbstub.h"

/* Prototypes */

static A_UINT16
wlanKeyCacheSetLocked(WLAN_DEV_INFO *pdevInfo, SIB_ENTRY *pSib,
                      WLAN_MACADDR *pmacAddr, WLAN_PRIV_RECORD *pKey);

static A_BOOL
isDynamicKey(WLAN_PRIV_RECORD *pKey);

#ifdef DEBUG
int keyDebugLevel = 0; // patchable
#else
const int keyDebugLevel = 0; // not patchable
#endif

/*
 * wlanKeyAuthTypeSet
 *
 * Sets the authAllow booleans based on the authentication type
 */
void
wlanKeyAuthTypeSet(WLAN_STA_CONFIG *pConfig, WLAN_AUTH_TYPE authType)
{
    ASSERT(pConfig);

    switch(authType) {
    case AUTH_TYPE_DEFAULT:
        uiPrintf("wlanKeyAuthTypeSet: DEFAULT\n");
        break;

    case AUTH_TYPE_OPEN:
        uiPrintf("wlanKeyAuthTypeSet: OPEN\n");
        pConfig->authAllowOpen   = TRUE;
        pConfig->authAllowShared = FALSE;
        pConfig->authOpenFirst   = TRUE;
        break;

    case AUTH_TYPE_SHARED:
        uiPrintf("wlanKeyAuthTypeSet: Shared\n");
        pConfig->authAllowOpen   = FALSE;
        pConfig->authAllowShared = TRUE;
        pConfig->authOpenFirst   = FALSE;
        break;

    case AUTH_TYPE_AUTO:
        uiPrintf("wlanKeyAuthTypeSet: Auto\n");
        pConfig->authAllowOpen   = TRUE;
        pConfig->authAllowShared = TRUE;
        pConfig->authOpenFirst   = FALSE;
        break;

    case AUTH_TYPE_WPA:
        uiPrintf("wlanKeyAuthTypeSet: WPA\n");
        pConfig->authAllowOpen   = TRUE;
        pConfig->authAllowShared = FALSE;
        pConfig->authOpenFirst   = TRUE;
        break;

    case AUTH_TYPE_WPAPSK:
        uiPrintf("wlanKeyAuthTypeSet: WPA PSK\n");
        pConfig->authAllowOpen   = TRUE;
        pConfig->authAllowShared = FALSE;
        pConfig->authOpenFirst   = TRUE;
        break;

    default:
        /* Do nothing */
        break;
    }
}

/*
 * wlanKeyCacheInit - Initialize the key cache hardware and software.
 *
 * Allocate keyTable with nstatic + ndynamic entries.
 * Allocate keyCache data structures to the size of the key cache.
 *
 * RETURNS: A_OK or A_* error code.
 */
A_STATUS
wlanKeyTableInit(WLAN_DEV_INFO *pdevInfo, A_UINT16 nstatic, A_UINT16 ndynamic)
{
    A_UINT16 nentries = nstatic + ndynamic;

    /* always allocate slots for the shared keys. */
    if (nentries == 0) {
        nentries = MAX_SHARED_KEYS;
    }

    A_SEM_INIT(pdevInfo->keySem, 0, CREATE_LOCKED);

    pdevInfo->keyCacheSize = pdevInfo->devCap.connectionIdMax+1;

    /*
     * Allocate the requested sized key table.  AP is bigger than
     * the station driver.
     */
    if (pdevInfo->keyTable == NULL) {
        pdevInfo->keyTable = (WLAN_PRIV_RECORD *)
            A_DRIVER_MALLOC(sizeof(WLAN_PRIV_RECORD) * nentries);
        pdevInfo->keyCacheSib = (SIB_ENTRY **)
            A_DRIVER_MALLOC(sizeof(SIB_ENTRY *) * pdevInfo->keyCacheSize);
        pdevInfo->keyCache = (WLAN_PRIV_RECORD **)
            A_DRIVER_MALLOC(sizeof(WLAN_PRIV_RECORD *) * pdevInfo->keyCacheSize);

        if (pdevInfo->keyTable == NULL ||
            pdevInfo->keyCacheSib == NULL ||
            pdevInfo->keyCache == NULL)
        {
            ASSERT(pdevInfo->keyTableSize == 0);
            wlanKeyTableFree(pdevInfo);
            return A_NO_MEMORY;
        }

        if (keyDebugLevel > 1) {
            uiPrintf("keytable nentries: %d, dynamic: %d",
                     nentries, nentries - nstatic);
            uiPrintf("keyCacheSize: %d\n", pdevInfo->keyCacheSize);
        }

        /*
         * Make sure that we set these values after allocating the table to
         * account for the case where allocation fails.  Don't want to call
         * wlanKeyTableRemoveDynamic() inside wlanKeyTableFree() if there's
         * nothing to remove.
         */
        pdevInfo->keyTableSize       = nentries;
        pdevInfo->keyTableStaticSize = nstatic;
    }

    ASSERT(pdevInfo->keyTableSize == nentries);
    ASSERT(pdevInfo->keyTableStaticSize == nstatic);

    A_MEM_ZERO(pdevInfo->keyTable, sizeof(WLAN_PRIV_RECORD) * nentries);

    /* Clear the software cache */
    A_MEM_ZERO(pdevInfo->keyCacheSib,
               sizeof(SIB_ENTRY *) * pdevInfo->keyCacheSize);
    A_MEM_ZERO(pdevInfo->keyCache,
               sizeof(WLAN_PRIV_RECORD *) * pdevInfo->keyCacheSize);

#if !NDIS_WDM
    /* Clear the hardware cache */
    halResetKeyCache(pdevInfo);
#endif

    return A_OK;
}

void
wlanReleaseKeyTableData(WLAN_DEV_INFO *pdevInfo)
{
    A_UINT32         i;
    WLAN_PRIV_RECORD *pKey;

    for (i = 0; i < pdevInfo->keyTableSize; i++) {
        pKey = &pdevInfo->keyTable[i];
        if (!isKeyTableSlotEmpty(pKey)) {
            wlanKeyTableRemoveDynamic(pdevInfo, pKey);
        }
    }
}

/*
 * wlanKeyTableFree - Free the key table and key cache software resources.
 *
 * Called by NDIS when unloading the driver.
 */
void
wlanKeyTableFree(WLAN_DEV_INFO *pdevInfo)
{
    if (pdevInfo->keyTable) {
        int i;

        for (i = 0; i < pdevInfo->keyTableSize; i++) {
            wlanKeyTableRemoveDynamic(pdevInfo, &pdevInfo->keyTable[i]);
        }

        A_DRIVER_FREE(pdevInfo->keyTable,
                      sizeof(WLAN_PRIV_RECORD) * pdevInfo->keyTableSize);
        pdevInfo->keyTable = NULL;
    }

    if (pdevInfo->keyCacheSib) {
        A_DRIVER_FREE(pdevInfo->keyCacheSib,
                      sizeof(SIB_ENTRY *) * pdevInfo->keyCacheSize);
        pdevInfo->keyCacheSib = NULL;
    }

    if (pdevInfo->keyCache) {
        A_DRIVER_FREE(pdevInfo->keyCache,
                      sizeof(WLAN_PRIV_RECORD *) * pdevInfo->keyCacheSize);
        pdevInfo->keyCache = NULL;
    }

    A_SEM_DELETE(pdevInfo->keySem);

    pdevInfo->keyTableSize       = 0;
    pdevInfo->keyCacheSize       = 0;
    pdevInfo->keyTableStaticSize = 0;
}

/*
 * caller should hold the sib entry lock
 */
A_BOOL
isDynamicKey(WLAN_PRIV_RECORD *pKey)
{
    if (pKey == NULL || ((pKey->keyFlags & PRIV_DYNAMIC) == 0)) {
        return FALSE;
    }

    return TRUE;
}

/*
 * Check/Free key table entries.
 */
A_BOOL
isKeyTableSlotEmpty(WLAN_PRIV_RECORD *pKey)
{
    return ((pKey->keyFlags & PRIV_KEY_LOCKED) == 0);
}

/* Called by sibCseStateClean */
void
wlanKeyPrivateDataDestroy(WLAN_PRIV_RECORD *pKey)
{
    if (pKey->initValue != NULL) {
        /*
         * free the init value memory space
         */
        switch (pKey->keyType) {
        case PRIV_KEY_TYPE_AES_CCM:
            ccm_close(pKey);
            break;

        case PRIV_KEY_TYPE_WEP:
            if (pKey->keyFlags & PRIV_CKIP_MIC) {
                ckip_close(pKey);
            }
            break;

        case PRIV_KEY_TYPE_CKIP:
            ckip_close(pKey);
            break;

        case PRIV_KEY_TYPE_TKIP:
        case PRIV_KEY_TYPE_TKIP_SW:
            tkip_close(pKey);
            break;

        default:
            break;
        }
        pKey->initValue = NULL;
    }
}

void
wlanKeyCacheDecompMaskClear(WLAN_DEV_INFO *pDevInfo, WLAN_PRIV_RECORD *pKey)
{
    WLAN_PRIV_RECORD *key;
    A_UINT16         i;
	A_UINT32 connAttribValue;

    /* Remove key cache entries */
    for (i = 0; i < pDevInfo->keyCacheSize; i++) {
        key = pDevInfo->keyCache[i];
        if (key == pKey) {
            wlanInstallBssKey(pDevInfo, i, 0, pKey);
            connAttribValue = FALSE;
            wdcUpdateConnectionAttribute(pDevInfo->targetHandle,
                                        i,
                                        CONN_COMPRESSION,
                                        sizeof(connAttribValue),
                                        (TARGET_CONFIG_VAL)&connAttribValue);
	    }
    }
}

void
wlanKeyTableSlotFree(WLAN_PRIV_RECORD *pKey)
{
    A_UINT32 flags;

    if (pKey) {
        wlanKeyPrivateDataDestroy(pKey);

        ASSERT(pKey->initValue == NULL);
        flags = pKey->keyFlags;         // retain flags

        A_MEM_ZERO(pKey, sizeof(WLAN_PRIV_RECORD));
        /*
         * We retain the flags because it contains lock bit and
         * the dynamic bit
         */
        pKey->keyFlags = flags;
    }
}

/*
 * Set a key table entry. Station side only.
 */
A_STATUS
wlanKeyTableSet(WLAN_DEV_INFO *pDevInfo, SIB_ENTRY *pSib,
                A_BOOL connectionKey, A_UINT32 cipher,
                A_UINT16 keyIndex,
                A_UINT32 keyLength, A_UINT8 *keyValue)
{
    WLAN_PRIV_RECORD    *pKey;
    SIB_ENTRY           *lSib = pDevInfo->localSta;

    /* Sanity check */
    if (keyLength == 0) {
        return A_ERROR;
    }

    if (keyIndex >= pDevInfo->keyTableSize) {
        return A_ERROR;
    }

    if ((connectionKey) && (!pSib)) {
        return A_ERROR;
    }

    /* Reset the key table entry */
    pKey = &pDevInfo->keyTable[keyIndex];

    wlanKeyPrivateDataDestroy(pKey);

    A_MEM_ZERO(pKey->keyVal, sizeof(pKey->keyVal));
    A_MEM_ZERO(pKey->micRxKeyVal, sizeof(pKey->micRxKeyVal));
    A_MEM_ZERO(pKey->micTxKeyVal, sizeof(pKey->micTxKeyVal));

    pKey->keyLength = (A_UINT32)keyLength;
    pKey->keyIV     = 0;
    pKey->extKeyIV  = 0;
    A_BCOPY(keyValue, pKey->keyVal, keyLength/8);

    pKey->keyFlags = (connectionKey ?
                      PRIV_UNIQUEKEY :
                      (keyIndex & PRIV_FRAMEKEY));
    pKey->keyFlags |= PRIV_KEY_LOCKED;

    /* Set key type and related and relative info */
    if (IS_CIPHER_AES_CCM(cipher)) {
        pKey->keyType = PRIV_KEY_TYPE_AES_CCM;
        keyPrivateInit(pKey);

    } else if (IS_CIPHER_TKIP(cipher)) {
        pKey->keyType = PRIV_KEY_TYPE_TKIP;
        ASSERT(pKey->keyLength == 256);

        if (connectionKey) {
            /*
             *  the key->keyLength covers all 3 keys for TKIP
             */
            A_BCOPY(&pKey->keyVal[16], pKey->micRxKeyVal,
                    TKIP_MIC_FIELD_SIZE);
            A_BCOPY(&pKey->keyVal[24], pKey->micTxKeyVal,
                    TKIP_MIC_FIELD_SIZE);
            A_MEM_ZERO(&pKey->keyVal[16], 2*TKIP_MIC_FIELD_SIZE);
            pKey->keyLength = MAX_KEY_LEN_BITS;

        } else {
            /*
             * but there are only 2 keys for group keys
             */
            A_BCOPY(&pKey->keyVal[16], pKey->micRxKeyVal,
                    TKIP_MIC_FIELD_SIZE);
            A_MEM_ZERO(&pKey->micTxKeyVal, TKIP_MIC_FIELD_SIZE);
            A_MEM_ZERO(&pKey->keyVal[16], 2*TKIP_MIC_FIELD_SIZE);
            pKey->keyLength = MAX_KEY_LEN_BITS;
        }

        keyPrivateInit(pKey);

    } else if ((pSib) && (lSib->ckipEnabled || lSib->ckipUseMic)) {
        ASSERT(pKey->keyLength <= 128);

        if (lSib->ckipEnabled) {
            pKey->keyType = PRIV_KEY_TYPE_CKIP;
        } else {
            pKey->keyType = PRIV_KEY_TYPE_WEP;
        }
        if (lSib->ckipUseMic) {
            pKey->keyFlags |= PRIV_CKIP_MIC;
        }
        keyPrivateInit(pKey);

    } else {
        pKey->keyType = PRIV_KEY_TYPE_WEP;
    }

    pKey->keyRSC = (A_INT64) -1;

    return A_OK;
}

void
wlanInstallConnectionKey(WLAN_DEV_INFO *pDevInfo, SIB_ENTRY *pSib,
                         WLAN_MACADDR *pBssId, WLAN_PRIV_RECORD *pKey)
{
    CONNECTION_KEY_RECORD connRecord;

    /* Prepare the data to be sent to the target */
    /* XXX Copy first. Fix endianness after */
    A_MACADDR_COPY(pBssId, &connRecord.bssId);
    A_BCOPY(pKey, &connRecord.connKey, sizeof(*pKey));

    /* Endianness business ... */
    connRecord.connKey.keyIV = A_swab32(pKey->keyIV);
    connRecord.connKey.extKeyIV = A_swab32(pKey->extKeyIV);
    connRecord.connKey.keyFlags = A_swab32(pKey->keyFlags);
    connRecord.connKey.keyLength = A_swab32(pKey->keyLength);
    connRecord.connKey.keyType = A_swab32(pKey->keyType);

    /* Send the msg to the target */
    wdcUpdateConnectionAttribute(pDevInfo->targetHandle,
                                 pSib->wdcConnId,
                                 CONN_KEY_UPDATE,
                                 sizeof(connRecord),
                                 &connRecord);
}

void
wlanDeleteConnectionKey(WLAN_DEV_INFO *pDevInfo, SIB_ENTRY *pSib,
                        WLAN_PRIV_RECORD *pKey)
{
    CONNECTION_KEY_RECORD connRecord;
    A_UINT32 flags;
    A_UINT32 keyType;

    A_MEM_ZERO(&connRecord, sizeof(connRecord));

    /* Indicate a destrucion and the key type, because of TKIP */
    flags = PRIV_KEY_DESTROY;

    /* No key. Clear connection. Still need to release the key cache entry */
    if (pKey) {
        keyType = pKey->keyType;
    } else {
        keyType = PRIV_KEY_TYPE_NULL;
    }

    connRecord.connKey.keyFlags = A_swab32(flags);
    connRecord.connKey.keyType = (A_swab32(keyType));

    /* Send the msg to the target */
    wdcUpdateConnectionAttribute(pDevInfo->targetHandle,
                                 pSib->wdcConnId,
                                 CONN_KEY_UPDATE,
                                 sizeof(connRecord),
                                 &connRecord);
}

void
wlanInstallBssKey(WLAN_DEV_INFO *pDevInfo, A_UINT32 bssKeyIdx,
                  A_UINT32 isDefaultBssKey, WLAN_PRIV_RECORD *pKey)
{
    BSS_KEY_RECORD bssRecord;

    ASSERT(pKey);

    /* Prepare the data to be sent to the target */
    bssRecord.bssKeyIdx = A_swab32(bssKeyIdx);
    bssRecord.isDefaultBssKey = A_swab32(isDefaultBssKey);
    A_BCOPY(pKey, &bssRecord.bssKey, sizeof(*pKey));

    /* Endianness business ... */
    bssRecord.bssKey.keyIV = A_swab32(pKey->keyIV);
    bssRecord.bssKey.extKeyIV = A_swab32(pKey->extKeyIV);
    bssRecord.bssKey.keyFlags = A_swab32(pKey->keyFlags);
    bssRecord.bssKey.keyLength = A_swab32(pKey->keyLength);
    bssRecord.bssKey.keyType = A_swab32(pKey->keyType);

    /* Send the msg to the target */
    wdcUpdateBssAttribute(pDevInfo->targetHandle,
                          0,
                          BSS_KEY_UPDATE,
                          sizeof(bssRecord),
                          &bssRecord);
}

void
wlanDeleteBssKey(WLAN_DEV_INFO *pDevInfo, A_UINT32 bssKeyIdx,
                 WLAN_PRIV_RECORD *pKey)
{
    BSS_KEY_RECORD bssRecord;
    A_UINT32 flags;

    A_MEM_ZERO(&bssRecord, sizeof(bssRecord));

    bssRecord.bssKeyIdx = A_swab32(bssKeyIdx);
    /* Indicate a destrucion and the key type, because of TKIP */
    flags = PRIV_KEY_DESTROY;
    bssRecord.bssKey.keyFlags = A_swab32(flags);
    bssRecord.bssKey.keyType = A_swab32(pKey->keyType);

    /* Send the msg to the target */
    wdcUpdateBssAttribute(pDevInfo->targetHandle,
                          0,
                          BSS_KEY_UPDATE,
                          sizeof(bssRecord),
                          &bssRecord);
}

/*
 * wlanKeyTableAdd - Add/update the key table entry at keyIndex.
 *
 * The key is only stored in the software key table, and is not stored
 * in the key cache with this routine.
 *
 * The key can be either static or dynamic.  NDIS will remove and
 * then add a "static" key as the OS will control traffic during this
 * update.
 *
 * If the keyIndex is KEYINDEX_DYNAMIC, a new key dynamic key table
 * entry will be allocated.
 *
 * If the keyIndex to be updated contains a key, all references to
 * this key are flushed from the key cache.
 *
 * keyLength is measured in bits.
 *
 */
WLAN_PRIV_RECORD *
wlanKeyTableAdd(WLAN_DEV_INFO *pdevInfo, A_UINT16 keyIndex, A_UINT32 keyType,
                A_UINT32 keyLength, A_UINT8 *keyValue)
{
    WLAN_PRIV_RECORD *pKey;
    A_UINT16         i;
    SIB_ENTRY        *pSib;

    if (keyLength == 0) {
        return NULL;
    }

    A_SEM_LOCK(pdevInfo->keySem, WAIT_FOREVER);

    /*
     *  Want a brand new dynamic entry.  Go fish.
     */
    if (keyIndex == KEYINDEX_DYNAMIC) {
        for (i = pdevInfo->keyTableStaticSize;
             i < pdevInfo->keyTableSize; i++)
        {
            if (isKeyTableSlotEmpty(pdevInfo->keyTable + i)) {
                keyIndex = i;
                break;
            } else {
                ASSERT(pdevInfo->keyTable[i].keyFlags & PRIV_DYNAMIC);
            }
        }
    }

    /* Invalid parameter, or KEYINDEX_DYNAMIC lookup search failed */
    if (keyIndex >= pdevInfo->keyTableSize) {
        A_SEM_UNLOCK(pdevInfo->keySem);
        return FALSE;
    }

    pKey = &pdevInfo->keyTable[keyIndex];

    wlanKeyPrivateDataDestroy(pKey);

    A_MEM_ZERO(pKey->keyVal, sizeof(pKey->keyVal));
    A_MEM_ZERO(pKey->micRxKeyVal, sizeof(pKey->micRxKeyVal));
    A_MEM_ZERO(pKey->micTxKeyVal, sizeof(pKey->micTxKeyVal));

    pKey->keyType   = (A_UINT16)keyType;
    pKey->keyLength = (A_UINT16)keyLength;
    pKey->keyIV     = 0;
    pKey->extKeyIV  = 0;
    A_BCOPY(keyValue, pKey->keyVal, keyLength/8);

    keyPrivateInit(pKey);

    if (pdevInfo->localSta->serviceType == WLAN_STA_SERVICE) {
        /* Init this to -1 so we can compare against frames strictly greater */
        pKey->keyRSC    = (A_INT64) -1;
    } else {
        /* AP requires keyRsc to 0 since it will advertize this to the STA */
        pKey->keyRSC    = (A_INT64) 0;
    }

    /*
     * For shared keys, we need to update the bits to put in
     * the wlan header IV field.  These need to be set as
     * shared keys, even when mapped as a unique key for
     * use with transmit blocking and software retry.
     * XXX this is wrong.  The caller of wlanKeyTableAdd() should
     * know what the tx/rx policy is, not here!
     */
    pKey->keyFlags = (keyIndex < MAX_SHARED_KEYS) ?
                     (keyIndex & PRIV_FRAMEKEY) : PRIV_UNIQUEKEY;
    pKey->keyFlags |= PRIV_KEY_LOCKED;
    if (keyIndex >= pdevInfo->keyTableStaticSize) {
        pKey->keyFlags |= PRIV_DYNAMIC;
    }

    /* Scour cache to update stale entries that were using this keyIndex */
    for (i = MAX_SHARED_KEYS; i < pdevInfo->keyCacheSize; i++) {
        if (pdevInfo->keyCache[i] == pKey) {
            pSib = pdevInfo->keyCacheSib[i];
            ASSERT(pSib->hwIndex == i || !(pKey->keyFlags & PRIV_UNIQUEKEY));
            ASSERT(pSib->pPriv == pKey || !(pKey->keyFlags & PRIV_UNIQUEKEY));
#ifdef DEBUG
            if (keyDebugLevel > 0) {
                uiPrintf("Key rewrite to hwidx %d\n", pSib->hwIndex);
            }
#endif
            wlanKeyCacheSetLocked(
                pdevInfo,
                pdevInfo->keyCacheSib[i],
                (pKey->keyFlags & PRIV_UNIQUEKEY) ? &pSib->macAddr : NULL,
                pKey);
        }
    }

    A_SEM_UNLOCK(pdevInfo->keySem);

    return pKey;
}

/*
 * wlanKeyTableAddDynamic - wrapper to wlanKeyTableAdd and wlanKeyEnable
 * for dynamic keys.
 *
 * Code that uses dynamic keys for 802.1x does not worry about the
 * index into the key table that is used, it only knows the pointer
 * to the key that was allocated to it.  For this reason, and because
 * we will not want to add w/o using a dynamic key, we combine the
 * the Add and Enable functions.
 *
 * If pKey == NULL or is a static key, we allocate a new dynamic key,
 * otherwise we update the key to the new value.
 *
 * Caller must not hold the sib entry lock.
 */
A_BOOL
wlanKeyTableAddDynamic(WLAN_DEV_INFO *pdevInfo, SIB_ENTRY *pSib,
                       A_UINT32 keyType, A_UINT32 keyLength, A_UINT8 *keyValue)
{
    WLAN_PRIV_RECORD *pKey;
    A_UINT16         keyIndex;

    A_SIB_ENTRY_LOCK(pSib);
    pKey = pSib->pPriv;

    if (isDynamicKey(pKey)) {
        /*
         * If dynamic, we assume that there is only one
         * user of this key and we are updating it.
         *
         * No attempt is made at sync with the pending tx
         * queue traffic other than we update this key
         * in place to avoid complications with software
         * retry.
         */
        keyIndex = pKey - pdevInfo->keyTable;
    } else {
        keyIndex = KEYINDEX_DYNAMIC;
    }

    pKey = wlanKeyTableAdd(pdevInfo, keyIndex, keyType, keyLength, keyValue);
    if (pKey == NULL) {
        A_SIB_ENTRY_UNLOCK(pSib);
        return FALSE;
    }

    /*
     * Associate the key and update the contents (wlanKeyEnable step)
     * and update the key cache HW if we have a key cache index.
     *
     * wlanKeyEnable() now removes dynamic keys so we cannot use it
     * directly.
     */
    A_SEM_LOCK(pdevInfo->keySem, WAIT_FOREVER);
    pSib->pPriv = pKey;
    if (pSib->hwIndex != HWINDEX_INVALID) {
        wlanKeyCacheSetLocked(pdevInfo, pSib,
                              &pSib->macAddr, pSib->pPriv);
    }
    A_SEM_UNLOCK(pdevInfo->keySem);

    A_SIB_ENTRY_UNLOCK(pSib);

    return TRUE;
}

/*
 * wlanKeyTableRemoveDynamic - remove an entry from the key table.
 *
 * Can be used for either static or dynamic keys.
 *
 * Also removes all instances of this key from the hardware key cache.
 *
 * The caller must ensure that this key is not busy by any
 * SIB entry.  It must also ensure that any SIB entry which
 * used this key will not become active with a null pPriv.
 *
 * Caller is assumed to hold A_SIB_TAB_LOCK(pSibTable) if it is
 * needed.  XXX in the future we have to layer this better.
 */
void
wlanKeyTableRemoveDynamic(WLAN_DEV_INFO *pdevInfo, WLAN_PRIV_RECORD *pKey)
{
    SIB_TABLE *pSibTable = pdevInfo->pSibTable;
    A_UINT32  i;

    if (pKey == NULL) {
        ASSERT(pKey);
        return;
    }

    /* always lock sib table first and then keySem */
    A_SEM_LOCK(pdevInfo->keySem, WAIT_FOREVER);

    /* Clean the key cache first */
    wlanKeyCacheKeyFree(pdevInfo, pKey);

    /* Invalidate all references to this key */
    /*
     *  XXX Should eliminate this search; better to
     *  keep a chain of sibs on the pKey.  Or maybe
     *  add another linkage element in the sib entry
     *  for this purpose.
     */
    for (i = 0; i < pSibTable->sibTableSize; i++) {
        if (pSibTable->sta[i].pPriv == pKey) {
            pSibTable->sta[i].pPriv   = NULL;
            pSibTable->sta[i].hwIndex = HWINDEX_INVALID;
        }
    }

    wlanKeyTableSlotFree(pKey);
    pKey->keyFlags = 0;         /* Clearing PRIV_KEY_LOCKED */

    A_SEM_UNLOCK(pdevInfo->keySem);
}

/*
 * Remove a static key by index.  Used by NDIS.
 */
void
wlanKeyTableRemove(WLAN_DEV_INFO *pdevInfo, A_UINT16 keyIndex)
{
    wlanKeyTableRemoveDynamic(pdevInfo, &pdevInfo->keyTable[keyIndex]);
}

/*
 * Remove dynamic key table and key cache entries for this SIB.
 */
void
wlanKeyFree(WLAN_DEV_INFO *pdevInfo, SIB_ENTRY *pSib)
{
    if (pSib) {
        if (isDynamicKey(pSib->pPriv)) {
            wlanKeyTableRemoveDynamic(pdevInfo, pSib->pPriv);
        } else {
            wlanKeyCacheFree(pdevInfo, pSib);
        }
    }
}

A_UINT16
wlanKeyCacheSetLocked(WLAN_DEV_INFO *pdevInfo,
                      SIB_ENTRY *pSib,
                      WLAN_MACADDR *pmacAddr,
                      WLAN_PRIV_RECORD *pKey)
{
    A_UINT16 hwidx;
    A_UINT32 keyType;
    A_BOOL   doObfuscate = pdevInfo->keyObfuscated;
    A_BOOL   hwEncrypt = TRUE;
    SIB_ENTRY **keyCacheSib = pdevInfo->keyCacheSib;
    A_UINT32  connAttribValue;

    if (pKey) {
        keyType = pKey->keyType;
    } else {
        keyType = PRIV_KEY_TYPE_NULL;
    }

    /* Host WDC - hwidx is now the connection ID  (wdcConnId) */
    hwidx = pSib->wdcConnId;
    keyCacheSib[hwidx] = pSib;


    if (keyType == PRIV_KEY_TYPE_TKIP && hwEncrypt == FALSE) {
        ASSERT(hwidx >= MAX_SHARED_KEYS);
        pKey->keyType = PRIV_KEY_TYPE_TKIP_SW;
        pSib->swEncryptionEnabled = TRUE;
    }

#ifdef DEBUG
    if (keyDebugLevel > 0) {
        if (pmacAddr) {
            printMacAddress(*pmacAddr);
            uiPrintf(" ");
        }
        if (pKey == NULL) {
            uiPrintf("Plumbing key (no Pkey)");
        } else {
            uiPrintf("Plumbing key from swindex %d, flags = 0x%02X ",
                     pKey - &pdevInfo->keyTable[0],
                     pKey->keyFlags);
        }
        uiPrintf("into hwindex %d, type=0x%x pSib=0x%x\n",
                 hwidx,
                 keyType,
                 (A_UINT32)pSib);
        if (keyDebugLevel > 1 && pKey) {
            int i;
            uiPrintf("\tkeyValue = ");
            for (i=0; i < (pKey->keyLength/8); i++) {
                uiPrintf("%02x", pKey->keyVal[i]);
            }
            uiPrintf("\n");
        }
    }
#endif

    pdevInfo->keyCache[hwidx] = pKey;
    halSetKeyCacheEntry(pdevInfo, hwidx, pmacAddr, pKey, doObfuscate);
    if (pSib && pSib->athAdvCapElement.info.useCompression && pmacAddr) {
        connAttribValue = TRUE;
        wdcUpdateConnectionAttribute(pdevInfo->targetHandle,
                                     pSib->wdcConnId,
                                     CONN_COMPRESSION,
                                     0,
                                     (TARGET_CONFIG_VAL)&connAttribValue);
    }

    return hwidx;
}

/*
 * wlanKeyCacheSet - set a key in the key cache.
 *
 * If pKey is NULL, we do what it takes to allocate a keyless
 * slot for transmit blocking.
 *
 * If pMacAddr is NULL we will use the key info in pSib->pKey, but will
 * not fill in the MAC address.  This is used for unique tx slots for
 * shared keys to avoid using the wrong key for rx with shared key 0.
 *
 * Used directly to set shared keys.  Called by wlanKeyCacheFault() for
 * unique keys and unique instances of shared keys.
 *
 * RETURNS: hardware slot entry.
 */
A_UINT16
wlanKeyCacheSet(WLAN_DEV_INFO *pdevInfo, SIB_ENTRY *pSib,
                WLAN_MACADDR *pmacAddr, WLAN_PRIV_RECORD *pKey)
{
    A_UINT16 hwidx;

    if (pSib != NULL) {
        A_SIB_ENTRY_LOCK(pSib);
    }

    A_SEM_LOCK(pdevInfo->keySem, WAIT_FOREVER);

    hwidx = wlanKeyCacheSetLocked(pdevInfo, pSib, pmacAddr, pKey);

    A_SEM_UNLOCK(pdevInfo->keySem);
    if (pSib != NULL) {
        A_SIB_ENTRY_UNLOCK(pSib);
    }

    return hwidx;
}

/*
 * Free the key cache resources used by pSib.
 *
 * Called when recycling a sib entry.
 */
void
wlanKeyCacheFree(WLAN_DEV_INFO *pdevInfo, SIB_ENTRY *pSib)
{
    A_UINT16 hwIndex;

    ASSERT(pSib != NULL);

    hwIndex = pSib->hwIndex;
    if (hwIndex == HWINDEX_INVALID) {
        return;
    }

    A_SEM_LOCK(pdevInfo->keySem, WAIT_FOREVER);

    ASSERT(pSib == pdevInfo->keyCacheSib[hwIndex]);
    ASSERT(pSib->pPriv == pdevInfo->keyCache[hwIndex]);  /* NULL is ok */

    halKeyCacheFree(pdevInfo, hwIndex);

    pdevInfo->keyCache[hwIndex] = NULL;
    pSib->hwIndex               = HWINDEX_INVALID;

    A_SEM_UNLOCK(pdevInfo->keySem);
}

/*
 * Remove all instances of pDelKey from the key cache.
 *
 * Caller should hold keySem.
 * This effectively reverses wlanKeyCacheSet().
 */
void
wlanKeyCacheKeyFree(WLAN_DEV_INFO *pdevInfo, WLAN_PRIV_RECORD *pDelKey)
{
    WLAN_PRIV_RECORD *pKey;
    SIB_ENTRY        *pSib;
    A_UINT16         i;

    if (pDelKey == NULL) {
        return;
    }

    /* Remove key cache entries */
    for (i = 0; i < pdevInfo->keyCacheSize; i++) {
        pSib = pdevInfo->keyCacheSib[i];
        pKey = pdevInfo->keyCache[i];

        /* our key + no sib or our sib */
        if (pSib && pKey == pDelKey) {
            halKeyCacheFree(pdevInfo, i);
            pdevInfo->keyCache[i] = NULL;
            pSib->pPriv           = NULL;
            pSib->hwIndex         = HWINDEX_INVALID;
        }
    }
}

/*
 * Frees up all Key Cache entries
 */
void
wlanKeyCacheFreeAll(WLAN_DEV_INFO *pdevInfo)
{
    SIB_ENTRY        *pSib;
    A_UINT16         i;

    /* Remove key cache entries */
    for (i = 0; i < pdevInfo->keyCacheSize; i++) {
        pSib = pdevInfo->keyCacheSib[i];
        if (pSib != NULL) {
            halKeyCacheFree(pdevInfo, i);
            pSib->hwIndex = HWINDEX_INVALID;
            pdevInfo->keyCache[i] = NULL;
        }
    }
}

/*
 * Invalidate connection key, clean up its cache entry.
 */
void
wlanKeyInvalidateAll(WLAN_DEV_INFO *pDev, WLAN_PRIV_RECORD *pInvalKey)
{
    A_UINT16 i;
    SIB_ENTRY *pSib = NULL;
    for (i = 0; i < pDev->pSibTable->sibTableSize; i++) {
        pSib = &pDev->pSibTable->sta[i];
        if (pSib && pInvalKey && pSib->pPriv == pInvalKey) {
            wlanDeleteConnectionKey(pDev, pSib, pInvalKey);
            pSib->pPriv = NULL;
        }
    }
}

/*
 * wlanKeyEnable - enable key at keyTable[hwIndex] for this SIB by
 * setting pSib->pPriv to this keyTable entry.
 *
 * Handle cases where a key slot is already allocated:
 *      - SW retry is enabled, and mgmt frames trigged slot allocation.
 *      - Sta has associated multiple times w/o de-authentication.
 *      - re-key.
 *
 * NOTE: We always set-up a unique key slot for tx, even when using
 *       shared keys.  SW retry requires this to access the tx filter
 *       hardware.
 */
void
wlanKeyEnable(WLAN_DEV_INFO *pdevInfo, SIB_ENTRY *pSib, A_UINT16 keyIndex)
{
    A_UINT16 hwidx;

    A_SEM_LOCK(pdevInfo->keySem, WAIT_FOREVER);

    /*
     * Remove previous dynamic key table entry while keeping
     * our key cache slot that we've worked so hard for.
     */
    if (isDynamicKey(pSib->pPriv)) {
        wlanKeyTableSlotFree(pSib->pPriv);
    }

    pSib->pPriv = &pdevInfo->keyTable[keyIndex];

    /* Always update with the new key info for dynamic keys */
    if (pSib->hwIndex != HWINDEX_INVALID) {
        hwidx = wlanKeyCacheSetLocked(pdevInfo, pSib,
                                      (pSib->pPriv->keyFlags & PRIV_UNIQUEKEY) ?
                                      &pSib->macAddr : NULL, pSib->pPriv);
        ASSERT(hwidx == pSib->hwIndex);
    }

    A_SEM_UNLOCK(pdevInfo->keySem);
}

/*
 * wlanKeyCacheFault - ensure key is cached by the hardware.
 *
 * Ensure we do not load a MAC address in a unique instance of a shared key.
 *
 * RETURNS: key cache index, and status indicating if a fault occured.  Both
 * the key index and the fault status has to be checked to know if the fault
 * was completed w/o error.
 */
A_UINT16
wlanKeyCacheFault(WLAN_DEV_INFO *pdevInfo, SIB_ENTRY *pSib, A_BOOL *faulted)
{
    WLAN_MACADDR *mac;

    *faulted = FALSE;
    if (pSib->hwIndex != HWINDEX_INVALID) {
        return pSib->hwIndex;
    }

    if ((pSib->pPriv == NULL) || (pSib->pPriv->keyFlags & PRIV_UNIQUEKEY)) {
        mac = &pSib->macAddr;
    } else {
        mac = NULL;
    }

    pSib->hwIndex = wlanKeyCacheSet(pdevInfo, pSib, mac, pSib->pPriv);
    if (pSib->hwIndex != HWINDEX_INVALID) {
        *faulted = TRUE;
    }

    return pSib->hwIndex;
}
void
keyMask(A_UINT8 *pInKeyVal, A_UINT8 *pOutKeyVal, A_UINT8 keyMask, A_INT32 len)
{
    A_INT8 j;

    for (j = 0; j < len; j++) {
        pOutKeyVal[j] = pInKeyVal[j] ^ keyMask;
    }
}

/*
 *  The job of this routine is simple:  map the encryption type
 *  of the cipher into the key type.  Call the crypto suite's
 *  init function to instantiate any private data.  Finally, call
 *  wlanKeyCacheSet() to push the key into the hw.
 */
A_STATUS
wlanKeyUpdateSingle(WLAN_DEV_INFO *pdevInfo, SIB_ENTRY *pSib, A_UINT32 context,
                    A_UINT16 keyIndex, A_UINT32 cipher)
{
    SIB_ENTRY           *lSib = pdevInfo->localSta;
    WLAN_PRIV_RECORD    *key = &pdevInfo->keyTable[keyIndex];
    A_UINT8             mask = pdevInfo->keyObfuscated ? KEY_XOR : 0;

    if (isKeyTableSlotEmpty(key)) {
        return A_ERROR;
    }

    /* XXX until we get key cleanup, well, cleaned up */
    key->keyFlags &= ~PRIV_CKIP_MIC;

    if (IS_CIPHER_AES_CCM(cipher)) {
        key->keyType = PRIV_KEY_TYPE_AES_CCM;

        ASSERT(key->keyLength == MAX_KEY_LEN_BITS);

        keyPrivateInit(key);

    } else if (lSib->ckipEnabled || lSib->ckipUseMic) {
        ASSERT(key->keyLength <= 128);

        if (lSib->ckipEnabled) {
            key->keyType = PRIV_KEY_TYPE_CKIP;
        } else {
            key->keyType = PRIV_KEY_TYPE_WEP;
        }
        if (lSib->ckipUseMic) {
            key->keyFlags |= PRIV_CKIP_MIC;
        }
        keyPrivateInit(key);

    } else if (IS_CIPHER_TKIP(cipher)) {
        key->keyType = PRIV_KEY_TYPE_TKIP;
        ASSERT(key->keyLength == 256);

        if (key->keyFlags & PRIV_UNIQUEKEY) {
            /*
             *  the key->keyLength covers all 3 keys for TKIP
             */
            if (lSib->serviceType == WLAN_STA_SERVICE) {
                A_BCOPY(&key->keyVal[16], key->micRxKeyVal,
                        TKIP_MIC_FIELD_SIZE);
                A_BCOPY(&key->keyVal[24], key->micTxKeyVal,
                        TKIP_MIC_FIELD_SIZE);
            } else {
                A_BCOPY(&key->keyVal[16], key->micTxKeyVal,
                        TKIP_MIC_FIELD_SIZE);
                A_BCOPY(&key->keyVal[24], key->micRxKeyVal,
                        TKIP_MIC_FIELD_SIZE);
            }
            A_MEM_ZERO(&key->keyVal[16], 2*TKIP_MIC_FIELD_SIZE);
            key->keyLength = MAX_KEY_LEN_BITS;

        } else {
            /*
             * but there are only 2 keys for group keys
             */
            if (lSib->serviceType == WLAN_STA_SERVICE) {
                A_BCOPY(&key->keyVal[16], key->micRxKeyVal,
                        TKIP_MIC_FIELD_SIZE);
                A_MEM_ZERO(&key->micTxKeyVal, TKIP_MIC_FIELD_SIZE);
            } else {
                A_BCOPY(&key->keyVal[16], key->micTxKeyVal,
                        TKIP_MIC_FIELD_SIZE);
                A_MEM_ZERO(&key->micRxKeyVal, TKIP_MIC_FIELD_SIZE);
            }
            A_MEM_ZERO(&key->keyVal[16], 2*TKIP_MIC_FIELD_SIZE);
            key->keyLength = MAX_KEY_LEN_BITS;
        }

        keyPrivateInit(key);

    } else {
        key->keyType = PRIV_KEY_TYPE_WEP;
    }

    /* update key type display, effect on station only now */
    if (keyIndex == pdevInfo->staConfig.defaultKey) {
        pdevInfo->keyType = key->keyType;
    }

    /* update key value and key type in hw key cache */
    wlanKeyCacheSet(pdevInfo, pSib,
                    (key->keyFlags & PRIV_UNIQUEKEY) ?
                    &pSib->macAddr : NULL, key);

    return A_OK;
}

/*
 * wlanKeyUpdate - set key type based on cipher negotiation.
 * only set up key type for unique keys. Also initialize the MIC for
 * AES cipher.
 *
 * If context is DOT1X then we should set the len to 128 for AES, 104 for WEP.
 *
 * RETURNS: status
 */
A_STATUS
wlanKeyUpdate(WLAN_DEV_INFO *pdevInfo, SIB_ENTRY *pSib, A_UINT32 context)
{
    WLAN_PRIV_RECORD *key = NULL;
    A_UINT32         mCipher = 0;
    A_UINT32         uCipher = 0;
    A_UINT16         index;
    A_UINT8          i;

    ASSERT(pSib != NULL);

    if (pdevInfo->localSta->serviceType == WLAN_STA_SERVICE) {
        /*
         * Install the shared keys first.  Note that the STA
         * may have its defaultKey (ie uniqueKey) in this
         * area, so we treat it with care.
         */

        /*
         * When multicast cipher is NULL, that means clear
         * for bc/mc traffic.
         * To support shared key encryption, need to set shared key
         * type to the unicast cipher type.
         */
        if (VALID_CIPHER_OUI(pSib->mCipher)) {
            mCipher = (IS_CIPHER_NULL(pSib->mCipher)) ?
                      pSib->uCipher : pSib->mCipher;
        }

        for (i = 0; i < MAX_SHARED_KEYS; i++) {
            key = &pdevInfo->keyTable[i];

            if (!isKeyTableSlotEmpty(key)) {
                wlanKeyUpdateSingle(pdevInfo, pSib, context, i, mCipher);
            }
        }

        /* set default/unique key type for infrastructure BSS */
        key = &pdevInfo->keyTable[pdevInfo->staConfig.defaultKey];

        /* no unique keys for Ad-hoc */
        if (pdevInfo->bssDescr->bsstype == INDEPENDENT_BSS) {
            return A_OK;
        }

        if (isKeyTableSlotEmpty(key)) {
            /* May not have default key yet in WPA */
            return A_OK;
        }

        /*
         * shared key encryption, i.e. zero config or ACU shared key only case
         */

        if (VALID_CIPHER_OUI(pSib->uCipher) &&
            VALID_CIPHER_OUI(pSib->mCipher))
        {
            if (context != CONTEXT_DOT1X &&
                pdevInfo->staConfig.defaultKey < MAX_SHARED_KEYS)
            {
                uCipher = IS_CIPHER_NULL(pSib->mCipher) ?
                          pSib->uCipher : pSib->mCipher;
            } else {
                /* for .1x or ACU unique key cases */
                uCipher = pSib->uCipher;
            }
        }
    } else {
        ASSERT(pdevInfo->localSta->serviceType == WLAN_AP_SERVICE);
        if (pdevInfo->staConfig.dot1xMode == KEYSRC_SERVER &&
            context != CONTEXT_DOT1X)
        {
            return A_OK;
        } else {
            key = pSib->pPriv;

            /*
             * enforce the AES policy for dot1x FLASH mode and
             * AUTO cipher mode,
             * that unique key is required
             */
            if ((pdevInfo->staConfig.dot1xMode == KEYSRC_LOCAL) &&
                key && !(key->keyFlags & PRIV_UNIQUEKEY))
            {
                pSib->uCipher = pdevInfo->bssDescr->wpaIe.mCipher[0];
            }
        }
        if (VALID_CIPHER_OUI(pSib->uCipher) &&
            VALID_CIPHER_OUI(pSib->mCipher))
        {
            /*
             *  should base on multicast cipher for shadow
             *  key when mCipher is not clear
             */
            uCipher = ((key && key->keyFlags & PRIV_UNIQUEKEY) ||
                       IS_CIPHER_NULL(pSib->mCipher)) ?
                           pSib->uCipher : pSib->mCipher;
        }
    }

    /* return OK and expect 802.1x to send us a key later */
    if (key == NULL) {
        return A_OK;
    }

    index = key - &pdevInfo->keyTable[0];
#ifdef DEBUG
    if (pSib->hwIndex == HWINDEX_INVALID) {
        uiPrintf("keyUpdate:  Invalid hwindex\n");
    }
#endif

    wlanKeyUpdateSingle(pdevInfo, pSib, context, index, uCipher);

    /* enable sw encryption if needed */
    pSib->swEncryptionEnabled =
        !cipherSupported(pdevInfo, key->keyType);
    pSib->swMicEnabled =
        !micSupported(pdevInfo, key->keyType);

    return A_OK;
}

/*
 *  Find info (keySlot, len, value) used for multicast traffic for this
 *  SIB.  Used by 802.1x txKey() code.
 */
A_UINT16
wlanKeyMcGet(WLAN_DEV_INFO *pdevInfo, SIB_ENTRY *pSib,
             A_UINT8 *buf, A_INT32 *len)
{
    A_UINT16         keyIndex;
    int              mcCipher;
    WLAN_PRIV_RECORD *pTable  = pdevInfo->keyTable;
    WLAN_STA_CONFIG  *pConfig = &pdevInfo->staConfig;

    if (!VALID_MCSE_ELEMENT(&pSib->mcseSet)) {
        /* Non-CSN STA, use WEP */
        keyIndex = pConfig->defaultKey;
        *len     = pTable[keyIndex].keyLength;
        A_BCOPY(pTable[keyIndex].keyVal, buf, *len / 8);
    } else {
        mcCipher = pSib->mCipher;

        switch (mcCipher) {
        case ATH_CSE_AES_CCM:
            keyIndex = pConfig->defaultKey;
            *len     = pTable[keyIndex].keyLength;
            A_BCOPY(pTable[keyIndex].keyVal, buf, *len / 8);
            break;

        case ATH_CSE_BASIC_WEP:
        default:
            keyIndex = pConfig->defaultKey;
            *len     = pTable[keyIndex].keyLength;
            A_BCOPY(pTable[keyIndex].keyVal, buf, *len / 8);
            break;
        }
    }

    if (*len == 0) {
        /* This happens when the sib's MC key is invalid */
        keyIndex = KEYINDEX_INVALID;
    }

    return keyIndex;
}

/*
 * Initialize cipher suites into bssDescr and localSta.  Note bssDescr
 * will get overwritten by probe responses.  The real code should
 * build the values in localSta, ignoring bssDescr.
 */
A_STATUS
initCipher(WLAN_DEV_INFO *pDevInfo)
{
#ifndef BUILD_AP
    WLAN_STA_CONFIG *pConfig      = &pDevInfo->staConfig;
    SIB_ENTRY       *lSib         = pDevInfo->localSta;
    WPA_IE          *pWpaIe       = &pDevInfo->bssDescr->wpaIe;
    UCSE_SET         bssUcseSet;
    AUTHSE_SET       bssAuthSet;
    MCSE_SET         bssMcseSet;
    UINT16           bsswpaCap;

    /* always default the encryption type to WEP */
    pDevInfo->keyType = PRIV_KEY_TYPE_WEP;

    lSib->wpaCapabilities = 0;
    lSib->authSel         = 0;
    lSib->uCipher         = 0;
    lSib->mCipher         = 0;

    /*
     *  The WPA IEs we read from the beacon go into bssDescr if we're
     *  scanning, and into apSib if we're joining.  Can't touch dis.
     *  We will return these in BSSID_LIST_EX OID.
     *
     *  The STA current cipher capabilities is kept in
     *  staConfig->encryptionAlg.  We will now look for the
     *  most secure union of the AP advertised ciphers with
     *  the STA's capabilities.  The union is found in the
     *  lSib->workingSet and lSib->wpaIE.
     */
    if (pDevInfo->staConfig.wpaEnabled) {
        A_UINT32 staAuth    = 0;
        A_UINT16 caps       = 0;
        int      i;

        if (!VALID_WPA_ELEMENT(&pDevInfo->bssDescr->wpaIe) ||
            wpaIeToWpaElements(&pDevInfo->bssDescr->wpaIe, &bssUcseSet,
                               &bssMcseSet, &bssAuthSet, &bsswpaCap) != A_OK)
        {
            return A_ERROR;
        }

        if (!pConfig->encryptionAlg) {
            return A_ERROR;
        }

        /*
         * Look for a unicast cipher
         * If we are in AUTO cipher selection mode we will
         * choose a cipher in the following preferred order:
         * AES
         * TKIP
         * CKIP+MIC
         * CKIP
         */
        for (i = 0; i < (bssUcseSet.uCipherCount); i++) {
            if (pConfig->encryptionAlg == ENCRYPTION_AUTO) {
                switch (bssUcseSet.uCiphers[i]) {
                case (WPA_CSE_CKIP):
                    if (!lSib->uCipher) {
                        lSib->uCipher = WPA_CSE_CKIP;
                    }
                    break;
                case (WPA_CSE_CKIP_MMH):
                    if (lSib->uCipher != WPA_CSE_AES_CCMP ||
                        lSib->uCipher != WPA_CSE_TKIP)
                    {
                        lSib->uCipher = WPA_CSE_CKIP_MMH;
                    }
                    break;
                case (WPA_CSE_TKIP):
                    if (lSib->uCipher != WPA_CSE_AES_CCMP) {
                        lSib->uCipher = WPA_CSE_TKIP;
                    }
                    break;
                case (WPA_CSE_AES_CCMP):
                    lSib->uCipher = WPA_CSE_AES_CCMP;
                    break;
                default:
                    break;
                }
            } else {
                switch (bssUcseSet.uCiphers[i]) {
                case (WPA_CSE_CKIP):
                case (WPA_CSE_CKIP_MMH):
                    if (pConfig->encryptionAlg == ENCRYPTION_CKIP) {
                        lSib->uCipher = bssUcseSet.uCiphers[i];
                    }
                    break;
                case (WPA_CSE_TKIP):
                    if (pConfig->encryptionAlg == ENCRYPTION_TKIP) {
                        lSib->uCipher = WPA_CSE_TKIP;
                    }
                    break;
                case (WPA_CSE_AES_CCMP):
                    if (pConfig->encryptionAlg == ENCRYPTION_AES_CCM) {
                        lSib->uCipher = WPA_CSE_AES_CCMP;
                    }
                    break;
                default:
                    break;
                }
            }
        }

        if (lSib->uCipher) {
           uiPrintf("Using unicast cipher 0x%x\n", lSib->uCipher);
           if (IS_CIPHER_CKIP(lSib->uCipher)) {
               lSib->ckipEnabled = TRUE;
           }
           if (IS_CIPHER_CKIP_MMH(lSib->uCipher)) {
               lSib->ckipUseMic = TRUE;
           }
        } else {
           uiPrintf("Must have a matching cipher (available = %d, configured = %d)\n",
                   bssUcseSet.uCiphers[i], pConfig->encryptionAlg);
           return A_ERROR;
        }

        /*
         * For the multicast/broadcast cipher, the AP gives us a choice of one
         * so let's try to find it in the list of ciphers.
         */
        switch (bssMcseSet.mCipher[0]) {
        case WPA_CSE_AES_CCMP:
            if (pConfig->encryptionAlg == ENCRYPTION_AUTO ||
                pConfig->encryptionAlg == ENCRYPTION_AES_CCM)
            {
                lSib->mCipher = bssMcseSet.mCipher[0];
            }
            break;

        case WPA_CSE_TKIP:
            if (pConfig->encryptionAlg == ENCRYPTION_AUTO ||
                pConfig->encryptionAlg == ENCRYPTION_AES_CCM ||
                pConfig->encryptionAlg == ENCRYPTION_TKIP)
            {
                lSib->mCipher = bssMcseSet.mCipher[0];
            }
            break;

        case WPA_CSE_CKIP:
        case WPA_CSE_CKIP_MMH:
            if (pConfig->encryptionAlg == ENCRYPTION_AUTO ||
                pConfig->encryptionAlg == ENCRYPTION_CKIP)
            {
                lSib->mCipher = bssMcseSet.mCipher[0];
            }
            break;

        case WPA_CSE_WEP40:
        case WPA_CSE_WEP104:
            if (pConfig->encryptionAlg == ENCRYPTION_AUTO ||
                pConfig->encryptionAlg == ENCRYPTION_AES_CCM ||
                pConfig->encryptionAlg == ENCRYPTION_TKIP)
            {
                lSib->mCipher = bssMcseSet.mCipher[0];
            }
            break;

        case WPA_CSE_NULL:
            uiPrintf("Do not support group key-only APs\n");
            break;

        default:
            break;
        }

        ASSERT(lSib->mCipher != 0);

        if (lSib->mCipher) {
            uiPrintf("Using multicast cipher 0x%x\n", lSib->mCipher);
        }

        /*
         * Verify that BSS and STA agree on Authentication Mode
         */
        switch (pDevInfo->staConfig.authType) {
        case AUTH_TYPE_WPA:
            staAuth = WPA_AUTHSE_8021X_UNSPEC;
            break;

        case AUTH_TYPE_WPAPSK:
            staAuth = WPA_AUTHSE_8021X_PSK;
            break;

        default:
            break;
        }

        for (i = 0; i < (bssAuthSet.authSelectorCount); i++) {
/* XXX Divy. CCKM is not part of ccx 1.0, currenntly required for Predator */
#if 0
#ifdef WLAN_CONFIG_CCX
            /*
             * CCKM is an optimization to EAP/TLS
             * so it requires a special check
             */
            if ((bssAuthSet.authSelectors[i] == WPA_AUTHSE_CCKM) &&
                (staAuth == WPA_AUTHSE_8021X_UNSPEC))
            {
                lSib->cckmEnabled = TRUE;
                lSib->authSel = bssAuthSet.authSelectors[i];
                uiPrintf("Using auth mode 0x%x\n", lSib->authSel);
                break;
            }
#endif /* WLAN_CONFIG_CCX */
#endif
            if (staAuth == bssAuthSet.authSelectors[i]) {
                lSib->authSel = bssAuthSet.authSelectors[i];
                uiPrintf("Using auth mode 0x%x\n", lSib->authSel);
                break;
            }
        }

        if (lSib->authSel == 0) {
            return A_ERROR;
        }

        lSib->wpaCapabilities = caps;

    } else { /* not WPA */

        if (pConfig->privacyInvoked) {
            /* WEP enabled auth: shared key auth tried first */
            pConfig->authAllowOpen   = TRUE;
            pConfig->authAllowShared = TRUE;
            pConfig->authOpenFirst   = FALSE;
            if (pDevInfo->bssDescr->ckipEnabled == TRUE) {
                lSib->ckipEnabled = TRUE;
            }
            if (pDevInfo->bssDescr->ckipUseMic == TRUE) {
                lSib->ckipUseMic = TRUE;
            }
        } else {
            /* WEP disabled auth: open auth enabled and tried first */
            pConfig->authAllowOpen   = TRUE;
            pConfig->authAllowShared = FALSE;
            pConfig->authOpenFirst   = TRUE;
        }
   }

    /*
     * See if the driver is restricted to use either Open or Shared
     * authentication only, via the authTypeUseOnly registry entru
     */
    if (pDevInfo->defaultStaConfig.authTypeUseOnly == AUTH_TYPE_DEFAULT) {
        wlanKeyAuthTypeSet(pConfig, pDevInfo->staConfig.authType);
    } else {
        wlanKeyAuthTypeSet(pConfig, pDevInfo->defaultStaConfig.authTypeUseOnly);
    }
#endif // #ifndef BUILD_AP
    return A_OK;
}

#if defined(VXWORKS) && defined(DEBUG)
/*
 * sibKeyCacheSteal - Clear key cache for sib idx.  Used for debugging
 * the key cache code only.
 */
void
sibKeyCacheSteal(int unit, int idx)
{
    WLAN_DEV_INFO *pDevInfo;
    SIB_TABLE     *pSibTable;
    SIB_ENTRY     *pSib;

    pDevInfo  = gDrvInfo.pDev[unit];
    pSibTable = pDevInfo->pSibTable;
    pSib      = &pSibTable->sta[idx];

    A_SIB_TAB_LOCK(pSibTable);                  /* always take first */
    A_SEM_LOCK(pDevInfo->keySem, WAIT_FOREVER);

    wlanKeyCacheKeyFree(pDevInfo, pSib->pPriv);

    A_SEM_UNLOCK(pDevInfo->keySem);
    A_SIB_TAB_UNLOCK(pSibTable);
}

void
keyValShow(char *p, WLAN_PRIV_RECORD *pKey)
{
    int         i, len = pKey->keyLength;
    A_UINT8     *val = pKey->keyVal;

    uiPrintf("pKey=0x%x ", (A_UINT32)pKey);
    uiPrintf("%s %3d bit %sKIP ", p, len,
             pKey->keyType == PRIV_KEY_TYPE_CKIP ? "C" : "T");
    for (i = 0; i < len / 8; i++) {
        uiPrintf("%02x", val[i]);
    }
    uiPrintf("\n");
}

/*
 * keyShow - dump info about the key table and key cache.  Used for
 * debugging only.
 */
int
keyShow(int unit)
{
    WLAN_DEV_INFO    *pdevInfo = gDrvInfo.pDev[unit];
    WLAN_PRIV_RECORD *pKey;
    int              idx, len, i;
    A_UINT8          *val = NULL;
    char             *pStr;

    A_SEM_LOCK(pdevInfo->keySem, WAIT_FOREVER);

    uiPrintf("Key Table Size = %d\n", pdevInfo->keyTableSize);
    uiPrintf("Key Cache Size = %d\n", pdevInfo->keyCacheSize);

    for (idx = 0; idx < pdevInfo->keyTableSize; idx++) {
        pKey = &pdevInfo->keyTable[idx];
        if (!isKeyTableSlotEmpty(pKey)) {
            uiPrintf("keyTable[%d] ", idx);

            len = pKey->keyLength;
            val = pKey->keyVal;

            switch (pKey->keyType) {
            case PRIV_KEY_TYPE_AES_CCM:
                pStr = "CCM";
                break;

            case PRIV_KEY_TYPE_WEP:
                pStr = "WEP";

                break;

            case PRIV_KEY_TYPE_CKIP:
                pStr = "CKIP";
                break;

            case PRIV_KEY_TYPE_TKIP_SW:
                pStr = "TKIP_SW";
                break;

            case PRIV_KEY_TYPE_TKIP:
                pStr = "TKIP";
                break;

            case PRIV_KEY_TYPE_NULL: // Pass-through
                pStr = "CLR";
                break;

            default:
                len = 0;
                pStr = "Unknown key type";
                break;
            }

            uiPrintf("%3d bit %s", len, pStr);
            uiPrintf(" Keyflags = <");
            if (pKey->keyFlags & PRIV_DYNAMIC) {
                uiPrintf("dynamic");
            }
            if (pKey->keyFlags & PRIV_UNIQUEKEY) {
                uiPrintf(",unique");
            }
            if (pKey->keyFlags & PRIV_CKIP_KP) {
                uiPrintf(",ckip_kp");
            }
            if (pKey->keyFlags & PRIV_CKIP_MIC) {
                uiPrintf(",ckip_mic");
            }
            if (pKey->keyFlags & PRIV_TKIP_SWMIC) {
                uiPrintf(",tkip_swmic");
            }
            if (pKey->keyFlags & PRIV_KEY_LOCKED) {
                uiPrintf(",locked");
            }
            uiPrintf("> framekey=%d ", pKey->keyFlags & PRIV_FRAMEKEY);

            for (i = 0; i < len / 8; i++) {
                uiPrintf("%02x", val[i]);
            }
            uiPrintf("\n");
        }
    }

    uiPrintf ("\n");
    for (idx = 0; idx < pdevInfo->keyCacheSize; idx++) {
        if (pdevInfo->keyCache[idx]) {
            pKey = pdevInfo->keyCache[idx];
            uiPrintf("keyCache[%d] -> keyTable[%d]", idx,
                     pdevInfo->keyCache[idx]-pdevInfo->keyTable);
            if (pdevInfo->keyCacheSib[idx]) {
                uiPrintf(" sib: %p  ", (void *)pdevInfo->keyCacheSib[idx]);
                printMacAddress(pdevInfo->keyCacheSib[idx]->macAddr);
            }
            uiPrintf("\n");
            if (pKey) {
                if (pKey->keyType == PRIV_KEY_TYPE_CKIP ||
                    pKey->keyType == PRIV_KEY_TYPE_TKIP)
                {
                    keyValShow("encrypt", pdevInfo->keyCache[idx+32]);
                    keyValShow("demic", pdevInfo->keyCache[idx+64]);
                    keyValShow("enmic", pdevInfo->keyCache[idx+64+32]);
                }
            }
        }
    }
    A_SEM_UNLOCK(pdevInfo->keySem);

    return 0;
}
#endif // defined(VXWORKS) && defined(DEBUG)

