/*
 *
 * Copyright (c) 2000-2003 Atheros Communications, Inc., All Rights Reserved
 *
 * Station Managment Entity
 */

#include "wlandrv.h"
#include "wlanDev.h"
#include "wlanext.h"
#include "wlansme.h"
#include "wlansmeext.h"
#include "wlanSend.h"
#include "wlanPhy.h"
#include "wlanbeacon.h"
#include "wlanchannel.h"
#include "display.h"
#include "ui.h"
#include "crypto/ocb.h"
#include "crypto/ccm.h"
#include "ccx.h"
#include "stacserv.h"
#include "usbstub.h"

#ifdef DEBUG
int mlmeDebugLevel;

static A_CHAR *
subtypeToString(A_UINT32 subtype)
{
    /* The following extern should be moved to a header file. */
    extern char *mgtTypeStr[];

    return mgtTypeStr[subtype];
}

#else /* DEBUG */

#ifdef VXWORKS
#define subtypeToString(vargs...)
#else
#define subtypeToString
#endif

#endif /* DEBUG */

/*
 * LOCAL PROTOTYPES
 */

static void
wlanMlmeJoinConfirm(WLAN_DEV_INFO *pDev, MLME_RESULT result);
static void
wlanMlmePowermgtConfirm(WLAN_DEV_INFO *pDev, MLME_RESULT result, A_BOOL powerStatus);
static A_STATUS
mlmeProcessJoinFrame(WLAN_DEV_INFO *pDev, ATHEROS_DESC *pDesc,
                     INFO_ELEMENT_SET *pIeSet);
static A_BOOL
mlmeBeaconTimCheck(TIM_ELEMENT *pTim, A_UINT16 assocId, A_BOOL *pbrmc);
static MLME_RESULT
wlanResetConfirm(WLAN_DEV_INFO *pDev);
static MLME_RESULT
wlanMlmeAuthConfirm(WLAN_DEV_INFO *pDev, WLAN_MACADDR *pDestAddr,
                    A_UINT16 authType, MLME_RESULT resultCode);
static MLME_RESULT
wlanMlmeAuthIndication(WLAN_DEV_INFO *pDev, WLAN_MACADDR *pDestAddr, A_UINT16 authType);
static MLME_RESULT
wlanMlmeDeauthConfirm(WLAN_DEV_INFO *pDev, WLAN_MACADDR *pDestAddr, MLME_RESULT resultCode);
static MLME_RESULT
wlanMlmeDeauthIndication(WLAN_DEV_INFO *pDev, WLAN_MACADDR *pDestAddr, MLME_RESULT resultCode);
static MLME_RESULT
wlanMlmeAssocConfirm(WLAN_DEV_INFO *pDev, MLME_RESULT resultCode);
static MLME_RESULT
wlanMlmeAssocIndication(WLAN_DEV_INFO *pDev, WLAN_MACADDR *pDestAddr);
static MLME_RESULT
wlanMlmeReassocConfirm(WLAN_DEV_INFO *pDev, MLME_RESULT resultCode);
static MLME_RESULT
wlanMlmeDisassocIndication(WLAN_DEV_INFO *pDev, WLAN_MACADDR *pDestAddr, MLME_RESULT reasonCode);
static MLME_RESULT
wlanMlmeDisassocConfirm(WLAN_DEV_INFO *pDev, WLAN_MACADDR *pmacAddr, MLME_RESULT resultCode);
static void
wlanMlmeScanConfirm(WLAN_DEV_INFO *pDev, MLME_RESULT result, A_BOOL interrupted);
static MLME_RESULT
wlanMlmeReassocIndication(WLAN_DEV_INFO *pDev, WLAN_MACADDR *pDestAddr);
static void
sibCseStateClean(WLAN_DEV_INFO *pDev, SIB_ENTRY *pSib);
void
sibDecompStateSet(WLAN_DEV_INFO *pDev, SIB_ENTRY *pSib, A_BOOL decomp);
static void
sibPowerStateClean(WLAN_DEV_INFO *pDev, SIB_ENTRY *pSib);
static A_UINT16
mlmeCountryCodeElementLength(WLAN_DEV_INFO *pDev, COUNTRY_INFO_LIST *pCountryInfo);
static void
mlmeBeaconReceive(WLAN_DEV_INFO *pDev, ATHEROS_DESC *pDesc);
static void
mlmeProbeRespReceive(WLAN_DEV_INFO *pDev, ATHEROS_DESC *pDesc);
MLME_RESULT
mlmeDisassocReceive(WLAN_DEV_INFO *pDev, OP_BSS *pOpBss,
                    WLAN_FRAME_DISASSOC *disassocFrame, A_UINT16 frameLen);
MLME_RESULT
mlmeAssocReassocRequest(WLAN_DEV_INFO *pDev, WLAN_MACADDR *pDestAddr,
                        A_UINT32 failureTimeout, CAP_INFO *capabilityInfo,
                        A_UINT16 listenInterval, A_BOOL reassoc);
A_UINT16
mlmeAssocReassocApReceive(WLAN_DEV_INFO *pDev, OP_BSS *pOpBss,
                          void *pFrame, A_UINT16 frameLen, A_BOOL reassoc);
A_UINT16
mlmeAssocApReceive(WLAN_DEV_INFO *pDev, OP_BSS *pOpBss,
                   WLAN_FRAME_ASSOC_REQ *assocFrame, A_UINT16 frameLen);
A_UINT16
mlmeReassocApReceive(WLAN_DEV_INFO *pDev, OP_BSS *pOpBss,
                     WLAN_FRAME_REASSOC_REQ *pFrame, A_UINT16 frameLen);
A_UINT16
mlmeAssocReassocStaReceive(WLAN_DEV_INFO *pDev,void *pResFrame,
                           A_UINT16 frameLen, A_BOOL reassoc);
MLME_RESULT
mlmeDeauthReceive(WLAN_DEV_INFO *pDev, OP_BSS *pOpBss,
                  WLAN_FRAME_DEAUTH *deauthFrame, A_UINT16 frameLen);
A_UINT16
mlmeAuthAp1Receive(WLAN_DEV_INFO *pDev, OP_BSS *pOpBss,
                   WLAN_FRAME_AUTH *authFrame, A_UINT16 len);
A_UINT16
mlmeAuthSta2Receive(WLAN_DEV_INFO *pDev, WLAN_FRAME_AUTH_CHLG *authFrame, A_UINT16 len);
A_UINT16
mlmeAuthAp3Receive(WLAN_DEV_INFO *pDev, OP_BSS *pOpBss,
                   WLAN_FRAME_AUTH_ENCRYPT *authFrame,
                   A_UINT16 len, A_BOOL decryptError);
A_UINT16
mlmeAuthSta4Receive(WLAN_DEV_INFO *pDev, WLAN_FRAME_AUTH *authFrame, A_UINT16 len);
A_UINT16
mlmeAssocStaReceive(WLAN_DEV_INFO *pDev, WLAN_FRAME_ASSOC_RESP *assocFrame, A_UINT16 len);
A_UINT16
mlmeReassocStaReceive(WLAN_DEV_INFO *pDev, WLAN_FRAME_REASSOC_RESP *reassocFrame, A_UINT16 len);
static void
mlmeProbeRequestReceive(WLAN_DEV_INFO *pDev, OP_BSS *pOpBss,
                        WLAN_FRAME_PROBE_REQ *probReqFrame, A_UINT16 len);
static void
xSectionRateSet(WLAN_RATE_SET *pSrc1, WLAN_RATE_SET *pSrc2, WLAN_RATE_SET *pDest);
static void
staWlanModeSet(WLAN_DEV_INFO *pDev, SIB_ENTRY *pSib);
static void
wlanElementToFullRateSet(RATE_SET *pRateSet, EXT_RATE_SET *pExtRateSet,
                         WLAN_RATE_SET *pShadowRateSet);
static MLME_RESULT
wlanResetConfirm(WLAN_DEV_INFO *pDev);
static void
mgtFrameHdrInit(WLAN_MGT_MAC_HEADER *pFrame, A_UINT8 subtype, WLAN_MACADDR *destAddr);
static void
chlgTextFill(WLAN_DEV_INFO *pDev, CHLG_TEXT *pChlg, A_UINT8 len);
static void
clearBasicRateFlag(WLAN_RATE_SET *pWlanRateSet);
static void
processNonErpElement(WLAN_DEV_INFO *pDevInfo,NONERP_ELEMENT *pNonErpElement);

/*
 * FUNCTIONS
 */


/*
 * mlmePoll
 *
 * Main polling function for SME/MLME.  Called periodically from an OS timer.
 */
void
mlmePoll(WLAN_DEV_INFO *pDev)
{
    A_UINT32 oldop;

    mlmePrintf("mlmePoll: ");

    ASSERT(pDev);

    /* Clear any scanning. Don't let any more BSSes creep in. */
    pDev->localSta->transState = TSTATE_QUIET;

    // invalidate the operation code before we switch based on its old value
    // because mlmeScanFunction() may call it again and set a new
    // value for operation.
    oldop = pDev->smeTimer.operation;
    pDev->smeTimer.operation = INVALID_TIMEOUT;

    switch (oldop) {
    case INVALID_TIMEOUT:
        mlmePrintf("No operation pending\n");
        break;
    case MLME_AUTH_TIMEOUT: {
        A_UINT16 authAlgNo = (A_UINT16)pDev->localSta->authenAlgo;

        mlmePrintf("MLME_AUTH_TIMEOUT\n");
        wlanMlmeAuthConfirm(pDev, NULL, authAlgNo, MLME_TIMEOUT);
        break;
    }

    case MLME_ASSOC_TIMEOUT:
        mlmePrintf("MLME_ASSOC_TIMEOUT\n");
        wlanMlmeAssocConfirm(pDev, MLME_TIMEOUT);
        break;

    case MLME_REASSOC_TIMEOUT:
        mlmePrintf("MLME_REASSOC_TIMEOUT\n");
        wlanMlmeReassocConfirm(pDev, MLME_TIMEOUT);
        break;

    case MLME_JOIN_TIMEOUT:
        mlmePrintf("MLME_JOIN_TIMEOUT\n");
        pDev->localSta->staState &= ~(STATE_JOINED | STATE_AUTH | STATE_ASSOC);
        wlanMlmeJoinConfirm(pDev,  MLME_TIMEOUT);
        break;

    case MLME_SCAN_TIMEOUT:
        mlmePrintf("MLME_SCAN_TIMEOUT\n");
        if (pDev->flushInProgress == TRUE) {
            A_TIMEOUT(&pDev->smeTimer, pDev->smeTimer.timeVal, (A_UINT32)pDev, MLME_SCAN_TIMEOUT);
        } else {
            mlmeScanFunction(pDev, SF_ADVANCE_NOW);
        }
        break;

    case MLME_PWRMGT_TIMEOUT:
        mlmePrintf("MLME_PWRMGT_TIMEOUT\n");
        wlanMlmePowermgtConfirm(pDev,  MLME_TIMEOUT, TRUE);
        break;
    default:
        ASSERT(0);
    }
}

/*
 * mgtFrameHdrInit
 *
 * Utility routine to initialize a management frame header
 */
void
mgtFrameHdrInit(WLAN_MGT_MAC_HEADER *pFrame, A_UINT8 subtype,
                WLAN_MACADDR *destAddr)
{
    /*
     * Fill in Frame Control fields. All management frames have FromDS
     * and ToDS set to 0. Most management frames are not encrypted.
     */
    A_MEM_ZERO((char *)pFrame, sizeof(*pFrame));

    pFrame->frameControl.fType    = FRAME_MGT;
    pFrame->frameControl.fSubtype = subtype;
    pFrame->frameControl.protoVer = PROTOCOL_VER_0;

    A_MACADDR_COPY(destAddr, &pFrame->destAddr);
}

#ifdef BUILD_AP
static A_STATUS
wlanMgmtFrameTransmit(WLAN_DEV_INFO *pDev, ATHEROS_DESC *pDesc)
{
    QUEUE_DETAILS *pTxQueue = &pDesc->pOpBss->txQueue;
    A_STATUS       status;

#ifdef WME
    if (pDev->staConfig.WmeEnabled) {
        pTxQueue = &pDesc->pOpBss->wmequeues[pDesc->txQandMode & txQMask];
    }
#endif
    if (IS_CHAN_G(pDev->staConfig.pChannel->channelFlags) &&
        pDesc->pDestSibEntry &&
        (pDesc->pDestSibEntry->wlanMode == STA_MODE_G) &&
        (pDev->staConfig.abolt & ABOLT_BURST))
    {
        /* Since we change the Q let us update the qmode as well */
        pTxQueue = &pDesc->pOpBss->burstQueue;
    }

    pDesc->pTargetQueueHandle = pTxQueue->pTargetQueueHandle;

    /*
     * lock out the tx completion code, and prepare and send
     * frame to hw
     */
    mgtFrameEndianChange(pDesc->pBufferVirtPtr.mgtHeader,
            pDesc->bufferLength, CPU2LE);
    A_SEM_LOCK(pTxQueue->qSem, WAIT_FOREVER);
    status = wlanFrameTransmit(pDev, pTxQueue, pDesc);
    A_SEM_UNLOCK(pTxQueue->qSem);

    return status;
}
#endif // BUILD_AP

/*
 * wlanMlmeInit
 *
 * Initializes the SME/MLME information structure
 *
 * NOTE: The MLME layer expects to own the data it passes.  We must make
 *       copies of it here.
 */
MLME_RESULT
wlanMlmeInit(WLAN_DEV_INFO *pDev, SME_INFO_STRUCT *pSmeInfo)
{
    if (pSmeInfo == NULL ||
        (pSmeInfo->serviceType != WLAN_STA_SERVICE &&
         pSmeInfo->serviceType != WLAN_AP_SERVICE))
    {
        mlmePrintf("wlanMlmeInit: ERROR: invalid service type\n");
        return MLME_INVALID_PARAM;
    }

    pDev->pSmeInfo = A_DRIVER_MALLOC(sizeof(SME_INFO_STRUCT));
    if (pDev->pSmeInfo == NULL) {
        return MLME_DRIVER_ERROR;
    }

    *pDev->pSmeInfo = *pSmeInfo; // copy

    A_SEM_INIT(pDev->baseBss.aidSem, 0, CREATE_UNLOCKED);
    if (isXrAp(pDev)) {
        A_SEM_INIT(pDev->xrBss.aidSem, 0, CREATE_UNLOCKED);
    }

    return MLME_SUCCESS;
}

void
wlanMlmeFree(WLAN_DEV_INFO *pDev)
{
    if (pDev->pSmeInfo != NULL) {
        A_DRIVER_FREE(pDev->pSmeInfo, sizeof(SME_INFO_STRUCT));
        pDev->pSmeInfo = NULL;
    }
    A_SEM_DELETE(pDev->baseBss.aidSem);
    if (isXrAp(pDev)) {
        A_SEM_DELETE(pDev->xrBss.aidSem);
    }
}

/*******************************************************************
 * initializeSme
 * initializes connection to SME by setting up callback pointers and timer.
 */
void
initializeSme(WLAN_DEV_INFO *pDevInfo, A_UINT32 service)
{
    MLME_RESULT result;

    SME_INFO_STRUCT *pSmeInfo = (service == WLAN_STA_SERVICE) ?
                                &staSmeInfo : &apSmeInfo;
    pSmeInfo->serviceType = (A_UINT16)service;
    result                = wlanMlmeInit(pDevInfo, pSmeInfo);

    if (service == WLAN_STA_SERVICE) {
        A_INIT_TIMER(pDevInfo, &pDevInfo->smeTimer, mlmePollHandler, FALSE);
    }
}


/***************************/
/* Authentication routines */
/***************************/

/*
 * MLME Authentication primitives perform authentication with the
 * AP. There is no authentication in Ad-hoc.
 *
 * PARAMETERS: Parameters for all of the authentication routines are described
 *            here together.
 *
 * pDestAddr - MAC address of the other STA to be auth'ed with. It is
 *            the AP's STA address.
 * authType - OPEN_SYSTEM or SHARED_KEY.
 * authFailureTimeout - TU. Abort the operation if not completed within this
 *            time. Observed for the overall duration of frame exchange.
 *
 */

/* wlanMlmeAuthRequest - start authentication process
 *
 * Starts the authentication sequence with the peer STA. Returns
 * after initiating the operation. It sends out first frame.
 *
 * RETURNS: MLME_SUCCESS, MLME_INVALID_PARAM, MLME_NOT_SUPPORETD
 */

MLME_RESULT
wlanMlmeAuthRequest(WLAN_DEV_INFO *pDev,
                    WLAN_MACADDR  *pDestAddr,
                    A_UINT16      algNo,
                    A_UINT32      authFailureTimeout)
{
    SIB_ENTRY       *pSib;
    SIB_ENTRY       *lSib;
    A_UINT16        newFrameLen;
    ATHEROS_DESC    *pDesc;
    WLAN_FRAME_AUTH *pFrame;

    mlmePrintf("wlanMlmeAuthReq:  entered.\n");

    /* Validate parameters */
    if (!pDestAddr) {
        mlmePrintf("wlanMlmeAuthReq: ERROR: Invalid destination address\n");
        return MLME_INVALID_PARAM;
    }

    /* Search sib table for pDestAddr, it should be an AP */
    if ((pSib = sibEntryFind(pDev, pDestAddr, NULL)) == NULL) {
        return MLME_INVALID_PARAM;
    }

    pSib->authenAlgo = algNo;

    mlmePrintf("wlanMlmeAuthRequest: algNo = %d\n", algNo);

    if (!pSib->capInfo.ess || pSib->capInfo.ibss ||
        pSib->serviceType != WLAN_AP_SERVICE)
    {
        // Destination address is not an infrastructure access point.
        return MLME_INVALID_PARAM;
    }

    if (algNo == AUTH_SHARED_KEY) {
        /* shared key can take longer (esp with debug on) */
        authFailureTimeout *= 2;
    }

    /* store algno for use when operation times out */
    lSib = pDev->localSta;
    lSib->authenAlgo = algNo;

    /* Lock sta */
    A_SIB_ENTRY_LOCK(lSib);

    /* check current state */
    if (lSib->transState != TSTATE_QUIET) {
        /* we are doing something. */
        A_SIB_ENTRY_UNLOCK(lSib);
        return MLME_REFUSED;
    }
    /* if already authenticated, allow preauthentication to other AP's. */
    if (!(lSib->staState & STATE_JOINED) ||
        ((lSib->staState & STATE_AUTH) &&
         (A_MACADDR_COMP(&pDev->baseBss.bssId, pDestAddr) == 0)))
    {
        /* Trying to authenticate again with the same AP. */
        /* @@@@ We should actually send a message to connection services
         * force deauthentication now.
         */
        A_SIB_ENTRY_UNLOCK(lSib);
        return MLME_REFUSED;
    }

    /* modify sta state */

    /* Compute frame length. No challenge text */
    newFrameLen = sizeof(WLAN_FRAME_AUTH);

    if ((createBuffandDesc(pDev, newFrameLen, &pDesc)) != 0) {
        A_SIB_ENTRY_UNLOCK(lSib);
        /* Failed to allocate buffer for frame and descriptor */
        return MLME_DRIVER_ERROR;
    }
    pFrame = pDesc->pBufferVirtPtr.auth;
    mgtFrameHdrInit(&pFrame->macHeader, SUBT_AUTH, pDestAddr);
    A_MACADDR_COPY(&pDev->baseBss.bssId, &pFrame->macHeader.bssId);
    A_MACADDR_COPY(&pDev->localSta->macAddr, &pFrame->macHeader.srcAddr);

    pFrame->algNo      = algNo;    /* Open sys or shared key */
    pFrame->statusCode = 0;        /* reserved in frame 1 */
    pFrame->transSeqNo = 1;        /* frame 1 */

    if (mlmeDebugLevel > 10) {
        displayMgtFrame((WLAN_MGT_MAC_HEADER *)pFrame, newFrameLen);
    }

    /* send first frame */
    lSib->transState     = TSTATE_AUTH_FRAME1_SENT;
    pDesc->pDestSibEntry = pSib;
    pDesc->pOpBss        = &pDev->baseBss;
    SETQHANDLE(pDesc, TXQ_ID_FOR_AC1);
    pDesc->bIndicateXmitStatus = TRUE;
    if (powerFrameTransmit(pDev, pDesc) == A_OK) {
        if (lSib->smePendDesc == NULL) {
            lSib->smePendDesc = (ATHEROS_DESC *)0xffffffff;
        }
        /* set timeout */
        A_TIMEOUT(&pDev->smeTimer, authFailureTimeout,
                  (A_UINT32) pDev, MLME_AUTH_TIMEOUT);
        A_SIB_ENTRY_UNLOCK(lSib);
        return MLME_SUCCESS;
    } else {
        lSib->transState = TSTATE_QUIET;
        A_SIB_ENTRY_UNLOCK(lSib);
        return MLME_DRIVER_ERROR;
    }
}

A_UINT16
mlmeAuthSta2Receive(WLAN_DEV_INFO        *pdevInfo,
                    WLAN_FRAME_AUTH_CHLG *authFrame,   /* Incoming frame pointer */
                    A_UINT16             len)          /* Total frame length */
{
    WLAN_FRAME_AUTH_ENCRYPT *pFrame;    /* return frame encrypted */
    SIB_ENTRY    *lSib;                 /* Local SIB */
    SIB_ENTRY    *pSib = NULL;          /* Peer (AP) SIB */
    ATHEROS_DESC *pDesc;
    A_UINT16     newFrameLen;

    /* validate parameters */
    lSib = pdevInfo->localSta;

    if (lSib->staState & STATE_AUTH ||
        lSib->transState != TSTATE_AUTH_FRAME1_SENT) {
        /* Wrong state */
        return MLME_REFUSED;
    }
    /* Ensure that it comes from desired AP (preauthenticate not supported yet) */
    if ((A_MACADDR_COMP(&(pdevInfo->baseBss.bssId), &(authFrame->macHeader.bssId))) != 0) {
        /* Got this frame from wrong AP !! */
        return MLME_REFUSED;
    }

    switch (pdevInfo->localSta->authenAlgo) {
    case AUTH_OPEN_SYSTEM:
    case AUTH_LEAP:
        /* This is the last frame in open system or leap */

        if (authFrame->statusCode == WLAN_SUCCESS) {
            /* Change the state in the SIB of the AP */
            if ((pSib = sibEntryFind(pdevInfo, &authFrame->macHeader.srcAddr, NULL)) == NULL) {
                /* Program error, should never happen */
                ASSERT(0);
                return MLME_INVALID_PARAM;
            }
            pSib->staState |= STATE_AUTH;

            /* Now update local station's SIB */
            lSib->staState |= STATE_AUTH;
            lSib->transState = TSTATE_QUIET;
        } else {
            lSib->transState = TSTATE_QUIET;
        }

        A_UNTIMEOUT(&(pdevInfo->smeTimer));
        /* Need to call callback routine here. */
        wlanMlmeAuthConfirm(pdevInfo, &(authFrame->macHeader.srcAddr),
                            (A_UINT16) pdevInfo->localSta->authenAlgo,
                            (MLME_RESULT) authFrame->statusCode);
        return MLME_SUCCESS;
        break;

    case AUTH_SHARED_KEY:
        pSib = sibEntryFind(pdevInfo, &authFrame->macHeader.srcAddr, NULL);
        if (pSib == NULL) {
            return MLME_INVALID_PARAM;
        }
        break;

    default:
        lSib->transState = TSTATE_QUIET;
        A_UNTIMEOUT(&(pdevInfo->smeTimer));
        wlanMlmeAuthConfirm(pdevInfo, &(authFrame->macHeader.srcAddr),
                            (A_UINT16) pdevInfo->localSta->authenAlgo,
                            MLME_NOT_SUPPORTED);
        return MLME_NOT_SUPPORTED;
        break;
    }

    /* Our request may fail if oversubscribed, mac denied, etc. */
    if (authFrame->statusCode != WLAN_SUCCESS) {
        lSib->transState = TSTATE_QUIET;
        A_UNTIMEOUT(&(pdevInfo->smeTimer));
        wlanMlmeAuthConfirm(pdevInfo, &(authFrame->macHeader.srcAddr),
                            (A_UINT16) pdevInfo->localSta->authenAlgo,
                            (MLME_RESULT)authFrame->statusCode);

        return MLME_SUCCESS;
    }

    /* Build the response frame:
     * In shared key, need to create another frame with same text
     * and send it back with encryption.  Include challenge text
     * and iv in the frame length.
     */
    newFrameLen = sizeof(WLAN_FRAME_AUTH) +
        WEP_ICV_FIELD_SIZE +
        INFO_ELEMENT_SIZE(authFrame->challenge);

    if ((createBuffandDesc(pdevInfo, newFrameLen, &pDesc)) != 0) {
        /* Failed to allocate buffer for frame and descriptor */
        return MLME_DRIVER_ERROR;
    }
    pFrame = pDesc->pBufferVirtPtr.authChalIv;
    mgtFrameHdrInit(&pFrame->macHeader, SUBT_AUTH, &pdevInfo->baseBss.bssId);
    A_MACADDR_COPY(&pdevInfo->baseBss.bssId, &pFrame->macHeader.bssId);
    A_MACADDR_COPY(&pdevInfo->localSta->macAddr, &pFrame->macHeader.srcAddr);
    pFrame->macHeader.frameControl.wep = 1;     /* encrypted */
    ASSERT(authFrame->algNo == AUTH_SHARED_KEY);
    pFrame->algNo      = authFrame->algNo;      /* Open sys or shared key */
    pFrame->statusCode = 0;                     /* reserved in frame 3 */
    pFrame->transSeqNo = 3;                     /* frame 1 */

    /* Copy the challenge text into the frame */
    mlmeElementCopy((INFO_ELEMENT *)&authFrame->challenge,
                    (INFO_ELEMENT *)&pFrame->challenge);

    if (mlmeDebugLevel > 10) {
        displayMgtFrame((WLAN_MGT_MAC_HEADER *)pFrame, newFrameLen);
    }

    /* send response frame */
    lSib->transState = TSTATE_AUTH_FRAME3_SENT;
    pDesc->pDestSibEntry = pSib;
    ASSERT(pSib->pPriv);
    pDesc->pOpBss        = &pdevInfo->baseBss;
    SETQHANDLE(pDesc, TXQ_ID_FOR_AC1);
    pDesc->bIndicateXmitStatus = TRUE;
    if (powerFrameTransmit(pdevInfo, pDesc) == A_OK) {
        if (lSib->smePendDesc == NULL) {
            lSib->smePendDesc = (ATHEROS_DESC *)0xffffffff;
        }
        return MLME_SUCCESS;
    } else {
        lSib->transState = TSTATE_QUIET;
        return MLME_DRIVER_ERROR;
    }
}

A_UINT16
mlmeAuthSta4Receive(WLAN_DEV_INFO   *pdevInfo,
                    WLAN_FRAME_AUTH *authFrame, /* Incoming frame pointer */
                    A_UINT16        len)        /* Total frame length */
{
    SIB_ENTRY    *lSib = pdevInfo->localSta;
    SIB_ENTRY    *pSib;

    /* validate parameters */
    if (lSib->staState & STATE_AUTH ||
        lSib->transState != TSTATE_AUTH_FRAME3_SENT) {
        /* Wrong state */
        return MLME_REFUSED;
    }

    /* Ensure that it comes from desired AP (preauthenticate not supported yet) */
    if ((A_MACADDR_COMP(&(pdevInfo->baseBss.bssId), &(authFrame->macHeader.srcAddr))) != 0) {
        /* Got this frame from wrong AP !! */
        return MLME_REFUSED;
    }

    lSib->transState = TSTATE_QUIET;
    /* This is the last frame in shared key */
    if (authFrame->statusCode == WLAN_SUCCESS) {
        /* Change the state in the SIB of the AP */
        if ((pSib = sibEntryFind(pdevInfo, &authFrame->macHeader.srcAddr, NULL)) == NULL) {
            /* Program error, should never happen */
            return MLME_INVALID_PARAM;
        }
        pSib->staState |= STATE_AUTH;
        lSib->staState |= STATE_AUTH;
    }
    // else wlanMlmeAuthConfirm() is responsible for any error reporting.

    A_UNTIMEOUT(&(pdevInfo->smeTimer));

    wlanMlmeAuthConfirm(pdevInfo, &(authFrame->macHeader.srcAddr),
            (A_UINT16) pdevInfo->localSta->authenAlgo,
            (MLME_RESULT) authFrame->statusCode);

    return MLME_SUCCESS;
}

/*
 * wlanMlmeAuthConfirm - confirm the result of AuthRequest operation.
 *
 * Calls the authCallback routine to deliver the result of authentication
 * operation.
 *
 * RETURNS: MLME_SUCCESS, MLME_INVALID_PARAM, MLME_NOT_SUPPORTED,
 *            MLME_TOOMANY_REQ, MLME_TIMEOUT, MLME_REFUSED
 */

MLME_RESULT
wlanMlmeAuthConfirm(WLAN_DEV_INFO *pdevInfo,
                    WLAN_MACADDR  *pDestAddr,
                    A_UINT16      authType,
                    MLME_RESULT   resultCode)
{
    if (pdevInfo->pSmeInfo->authCallback) {
        (*pdevInfo->pSmeInfo->authCallback)(pdevInfo, pDestAddr,
                                            authType, resultCode);
    }
    return MLME_SUCCESS;
}

/*
 * wlanMlmeAuthIndication - indicate the result of AuthRequest operation in AP
 *
 * Calls the authCallback routine to deliver the result of authentication
 * operation.
 */

MLME_RESULT
wlanMlmeAuthIndication(WLAN_DEV_INFO *pdevInfo,
                       WLAN_MACADDR  *pDestAddr,
                       A_UINT16      authType)
{
    if (pdevInfo->pSmeInfo->authIndicationCallback) {
        (*pdevInfo->pSmeInfo->authIndicationCallback)(pdevInfo, pDestAddr,
                                                      authType);
    }
    return MLME_SUCCESS;
}

/*
 * mlmeAuthAp1Receive - Process the received Auth frame #1 on AP
 *
 * This routine handles the incoming associate requests on AP. It checks the
 * state, assigns a new association id and sends back the response frame.
 * This routine is not part of MLME API.
 */

A_UINT16
mlmeAuthAp1Receive(WLAN_DEV_INFO   *pdevInfo,
                   OP_BSS          *pOpBss,
                   WLAN_FRAME_AUTH *authFrame,
                   A_UINT16        len)
{
    SIB_ENTRY    *pSib;
    A_UINT16     statusCode;
    A_UINT16     newFrameLen;
    ATHEROS_DESC *pDesc;

    union {
        WLAN_FRAME_AUTH_CHLG *shared;   /* return frame w/text */
        WLAN_FRAME_AUTH      *open;     /* return frame regular */
    } pFrame;

    /* Check if I am up */
    if (!(pdevInfo->localSta->staState & STATE_AP_UP)) {
        return MLME_REFUSED;
    }

    pSib = sibEntryFind(pdevInfo, &authFrame->macHeader.srcAddr, pOpBss);
    if (!pSib) {
        /* The sib should have been created while the frame
         * was received.  We should be disapatched before
         * sib can be aged and recycled.
         */
        return MLME_REFUSED;
    }

#ifdef BUILD_AP
    if (pdevInfo->localSta->transState & TSTATE_SCANNING) {
        /* Scanning in progress so reject this request */
        return MLME_TOOMANY_REQ;
    }
#endif

    statusCode = wlanMlmeDeauthRequest(pdevInfo, pOpBss,
                                       &authFrame->macHeader.srcAddr,
                                       REASON_UNSPECIFIED, FALSE);

    mlmePrintf("mlmeAuthAp1Receive: got %d from wlanMlmeDeauthRequest\n",
               statusCode);

    /* Verify state */
    statusCode = WLAN_SUCCESS;

    /*
     * We don't need to check transState because even if it is non-zero like
     * TSTATE_FRAME2_SENT, receiving frame 1 will reset it. This can happen
     * if remote sta didn't get an ack for previous frame 1 that it sent out.
     */
    pSib->authenAlgo = authFrame->algNo;

    if ((authFrame->algNo == AUTH_SHARED_KEY) &&
        (pdevInfo->staConfig.authAllowShared == TRUE))
    {
        /* If !privacyInvoked, we return WLAN_UNSUPPORTED_ALG below */
        if (pSib->pChlg == NULL) {
            pSib->pChlg = (CHLG_TEXT *)A_DRIVER_MALLOC(sizeof(CHLG_TEXT));
            if (pSib->pChlg == NULL) {
                return MLME_DRIVER_ERROR;
            }
        }

        chlgTextFill(pdevInfo, pSib->pChlg, AUTH_SHARED_KEY_CHLG_SIZE);

        /* Compute frame length. Include challenge text */
        newFrameLen = sizeof(WLAN_FRAME_AUTH) + INFO_ELEMENT_SIZE(*pSib->pChlg);

        /* Build the response frame */
        if ((createBuffandDesc(pdevInfo, newFrameLen, &pDesc)) != 0) {
            /* Failed to allocate buffer for frame and descriptor */
            A_DRIVER_FREE(pSib->pChlg, sizeof(CHLG_TEXT));
            pSib->pChlg = NULL;
            return MLME_DRIVER_ERROR;
        }

        pFrame.shared = pDesc->pBufferVirtPtr.authChal;
        mgtFrameHdrInit(&pFrame.shared->macHeader, SUBT_AUTH, &pSib->macAddr);
        A_MACADDR_COPY(&pOpBss->bssId, &pFrame.shared->macHeader.bssId);
        A_MACADDR_COPY(&pOpBss->bssId, &pFrame.shared->macHeader.srcAddr);
        /* Copy the challenge text into the frame */
        mlmeElementCopy((INFO_ELEMENT *)pSib->pChlg,
                        (INFO_ELEMENT *)&pFrame.shared->challenge);
    } else if ((authFrame->algNo == AUTH_OPEN_SYSTEM) &&
               (pdevInfo->staConfig.authAllowOpen == TRUE))
    {
        /* Simple open system auth.  No challenge text.
         * We may need to query some local database or table to validate
         * that this MAC address is allowed to enter the BSS network.
         */
        newFrameLen = sizeof(WLAN_FRAME_AUTH);

        /* Build the response frame */
        if ((createBuffandDesc(pdevInfo, newFrameLen, &pDesc)) != 0) {
            /* Failed to allocate buffer for frame and descriptor */
            return MLME_DRIVER_ERROR;
        }

        pFrame.open = pDesc->pBufferVirtPtr.auth;
        mgtFrameHdrInit(&pFrame.open->macHeader, SUBT_AUTH, &pSib->macAddr);
        A_MACADDR_COPY(&pOpBss->bssId, &pFrame.open->macHeader.bssId);
        A_MACADDR_COPY(&pOpBss->bssId, &pFrame.open->macHeader.srcAddr);
    } else {
        /*
         * Either shared or open auth can be configured to
         * be unsupported.
         */
        statusCode = WLAN_UNSUPPORTED_ALG;

#ifdef DEBUG
        switch (authFrame->algNo) {
        case AUTH_SHARED_KEY:
        case AUTH_OPEN_SYSTEM:
            break;

        case AUTH_LEAP:
            mlmePrintMacAddress(pSib->macAddr);
            mlmePrintf(": Unsupported 802.11 authentication service: LEAP\n");
            break;

        default:
            mlmePrintMacAddress(pSib->macAddr);
            mlmePrintf(":  Unsupported 802.11 authentication service: 0x%x\n",
                       authFrame->algNo);
            break;
        }
#endif

        /* Build an error response frame */
        newFrameLen = sizeof(WLAN_FRAME_AUTH);

        if ((createBuffandDesc(pdevInfo, newFrameLen, &pDesc)) != 0) {
            /* Failed to allocate buffer for frame and descriptor */
            return MLME_DRIVER_ERROR;
        }

        pFrame.open = pDesc->pBufferVirtPtr.auth;
        mgtFrameHdrInit(&pFrame.open->macHeader, SUBT_AUTH, &pSib->macAddr);
        A_MACADDR_COPY(&pOpBss->bssId, &pFrame.open->macHeader.bssId);
        A_MACADDR_COPY(&pOpBss->bssId, &pFrame.open->macHeader.srcAddr);
    }

    pFrame.open->algNo      = authFrame->algNo; // return same algno
    pFrame.open->statusCode = statusCode;
    pFrame.open->transSeqNo = 2;

    mlmePrintf("wlan%d mlmeAuthAp1Receive algNo = %d\n",
               pdevInfo->devno, authFrame->algNo);
    mlmePrintMacAddress(pSib->macAddr);
    mlmePrintf("  ");
    if (mlmeDebugLevel > 10) {
        displayStates(pSib->staState, pSib->transState);
        displayMgtFrameShort(&pFrame.open->macHeader, newFrameLen);
    }

    /* send response frame */
    pSib->transState     = TSTATE_AUTH_FRAME2_SENT;
    pDesc->pDestSibEntry = pSib;
    pDesc->pOpBss        = pOpBss;
    SETQHANDLE(pDesc, TXQ_ID_FOR_AC1);
    pDesc->bIndicateXmitStatus = TRUE;
    if (powerFrameTransmit(pdevInfo, pDesc) == A_OK) {
        pSib->stats.Authentications++;
        if (pSib->smePendDesc == NULL) {
            pSib->smePendDesc = (ATHEROS_DESC *)0xffffffff;
        }
        return MLME_SUCCESS;
    }

    pSib->transState = TSTATE_QUIET;

    return MLME_DRIVER_ERROR;
}

A_UINT16
mlmeAuthAp3Receive(WLAN_DEV_INFO           *pdevInfo,
                   OP_BSS                  *pOpBss,
                   WLAN_FRAME_AUTH_ENCRYPT *authFrame,
                   A_UINT16                len,
                   A_BOOL                  decryptError)
{
    SIB_ENTRY       *pSib;
    A_UINT16        statusCode;
    A_UINT16        newFrameLen;
    ATHEROS_DESC    *pDesc;
    WLAN_FRAME_AUTH *pFrame;

    /* Check if I am up */
    if (!(pdevInfo->localSta->staState & STATE_AP_UP)) {
        return MLME_REFUSED;
    }

    if ((pSib = sibEntryFind(pdevInfo, &authFrame->macHeader.srcAddr, pOpBss)) == NULL) {
        mlmePrintf("mlmeAuthAp3Receive: ERROR: failed to find sta in table\n");
        return MLME_REFUSED;
    }

    /*
     * STA is already authenticated is not an error.
     * This can happen if the STA didn't get the frame #4 that
     * AP sent out last time and it retransmitted frame #3.
     *
     * Pick through things that could be wrong to return the
     * right errno.
     */
    if ((pSib->transState != TSTATE_AUTH_FRAME2_SENT) &&
        (pSib->transState != TSTATE_AUTH_FRAME4_SENT)) {
        statusCode = WLAN_OUT_OF_SEQUENCE;
    } else if (decryptError) {
        /* check before looking at the frame as it may be corrupt */
        statusCode = WLAN_CHLG_FAILURE;
    } else if (authFrame->algNo != AUTH_SHARED_KEY) {
        statusCode = WLAN_UNSUPPORTED_ALG;
    } else if (pSib->pChlg == NULL) {
        statusCode = WLAN_OUT_OF_SCOPE;
    } else if (authFrame->transSeqNo != 3) {
        statusCode = WLAN_OUT_OF_SEQUENCE;
    } else if ((authFrame->challenge.elementID != ELE_CHLG_TEXT) ||
               (authFrame->challenge.length != pSib->pChlg->length) ||
               (memcmp(&authFrame->challenge.chlgText, pSib->pChlg->chlgText,
                       pSib->pChlg->length) != 0))
    {
        statusCode = WLAN_CHLG_FAILURE;
    } else {
        statusCode = WLAN_SUCCESS;
    }
    if (statusCode != WLAN_SUCCESS) {
        mlmePrintf("mlmeAuthAp3Receive: shared key auth failure %d for ",
                   statusCode);
        mlmePrintMacAddress(pSib->macAddr);
        mlmePrintf("\n");
    }
    if (pSib->pChlg != NULL) {
        A_DRIVER_FREE(pSib->pChlg, sizeof(CHLG_TEXT));
        pSib->pChlg = NULL;
    }

    /* Build the response frame - simple frame with no challenge. */
    newFrameLen = sizeof(WLAN_FRAME_AUTH);
    if ((createBuffandDesc(pdevInfo, newFrameLen, &pDesc)) != 0) {
        /* Failed to allocate buffer for frame and descriptor */
        return MLME_DRIVER_ERROR;
    }

    pFrame = pDesc->pBufferVirtPtr.auth;
    mgtFrameHdrInit(&pFrame->macHeader, SUBT_AUTH, &pSib->macAddr);
    A_MACADDR_COPY(&pOpBss->bssId, &pFrame->macHeader.bssId);
    A_MACADDR_COPY(&pOpBss->bssId, &pFrame->macHeader.srcAddr);
    pFrame->algNo      = authFrame->algNo;
    pFrame->statusCode = statusCode;
    pFrame->transSeqNo = 4;

    mlmePrintf("\nmlmeAuthAp3Receive\n");
    if (mlmeDebugLevel > 10) {
        displayMgtFrame(&pFrame->macHeader, newFrameLen);
    }

    /* send response frame */
    pSib->transState     = TSTATE_AUTH_FRAME4_SENT;
    pDesc->pDestSibEntry = pSib;
    pDesc->pOpBss        = pOpBss;
    SETQHANDLE(pDesc, TXQ_ID_FOR_AC1);
    pDesc->bIndicateXmitStatus = TRUE;
    if (powerFrameTransmit(pdevInfo, pDesc) == A_OK) {
        if (pSib->smePendDesc == NULL) {
            pSib->smePendDesc = (ATHEROS_DESC *) 0xffffffff;
        }

        return MLME_SUCCESS;
    } else {
        pSib->transState = TSTATE_QUIET;

        return MLME_DRIVER_ERROR;
    }

    /*
     * According to the spec, sta is not considered authenticated until
     * AP transmits frame #4 successfully with an ACK from sta. It is done
     * by mlmeMgtTxCompletion().
     */
}

/*
 * chlgTextFill - Generate challenge text for WEP authentication
 *
 * Fills the Challenge Text element with random byte values.
 * Spec section 8.1.2.2 dictates that it must use WEP PRNG.
 */
void
chlgTextFill(WLAN_DEV_INFO *pdevInfo, CHLG_TEXT *pChlg, A_UINT8 len)
{
    int i;

    pChlg->elementID = ELE_CHLG_TEXT;
    pChlg->length    = len;

    for (i =0; i < len && i < ELE_CHLG_TEXT_SIZE; i++) {
        pChlg->chlgText[i] = (A_UINT8)halGetRandomSeed(pdevInfo);
    }
}

/*
 * wlanMlmeDeauthRequest - terminates current authentication.
 *
 * It is a noticification. There is no notion of success/failure.
 * It is called by SME on either STA or AP.
 * Spec 5.4.3.2 says that it must deauth and disassociate if an
 * association exists. We need to worry about operations in progress.
 * I think that any auth or assoc in progress must be cancelled if
 * they are eventually completed. Not implemented yet.
 *
 * RETURNS: MLME_SUCCESS or MLME_INVALID_PARAM
 */

MLME_RESULT
wlanMlmeDeauthRequest(WLAN_DEV_INFO *pdevInfo,
                      OP_BSS        *pOpBss,
                      WLAN_MACADDR  *pDestAddr,
                      A_UINT16       reasonCode,
                      A_BOOL         sendFrame)
{
    SIB_ENTRY         *pSib;
    A_UINT16          frameLen;
    ATHEROS_DESC      *pDesc;
    WLAN_FRAME_DEAUTH *pFrame;
    SIB_ENTRY         *lSib = pdevInfo->localSta;
    A_UINT32          oldState;

    /*
     * This can be done on either AP or STA. Don't assume it's
     * the local STA.
     */
    if (lSib->transState & (TSTATE_JOINING | TSTATE_SCANNING)) {
        /* Scan or join in progress */
        return MLME_TOOMANY_REQ;
    }

    /* It's ok to not find the destination SIB. */
    pSib = sibEntryFind(pdevInfo, pDestAddr, pOpBss);

    mlmePrintf("wlanMlmeDeauthRequest -- ");
    if (pSib) {
        oldState = pSib->staState;
        mlmePrintMacAddress(pSib->macAddr);
        mlmePrintf(" in wlan%d\n", pdevInfo->devno);
    } else {
        oldState = lSib->staState;
        mlmePrintMacAddress(*pDestAddr);
        mlmePrintf(" No SIB found in wlan%d\n", pdevInfo->devno);
    }

    if ((lSib->staState & STATE_ASSOC) ||
        (pSib && (pSib->staState & STATE_ASSOC)))
    {
        /*
         * The receiving STA is associated too. Cancel that, but
         * don't send a Disassociate frame, as the
         * Deauthenticate frame that we send below should suffice.
         */
         wlanMlmeDisassocRequest(pdevInfo, pOpBss, pDestAddr,
                                 REASON_ASSOC_LEAVING, FALSE);
    }

    /*
     * Since this is a notification, we must reflect the state change
     * immediately so that data transmit/receive will terminate
     * immediately.
     */
    lSib->staState  &= ~(STATE_AUTH|STATE_ASSOC);
    lSib->transState = TSTATE_QUIET;

    if (pSib != NULL) {
        pSib->staState  &= ~(STATE_AUTH|STATE_ASSOC);
        pSib->transState = TSTATE_QUIET;
        pSib->stats.DeAuthentications++;
    }

    pdevInfo->localSta->stats.DeAuthentications++;

    if (sendFrame) {
        frameLen = sizeof(WLAN_FRAME_DEAUTH);

        if ((createBuffandDesc(pdevInfo, frameLen, &pDesc)) != 0) {
            /* Failed to allocate buffer for frame and descriptor */
            return MLME_DRIVER_ERROR;
        }

        pFrame = pDesc->pBufferVirtPtr.deAuth;
        mgtFrameHdrInit(&pFrame->macHeader, SUBT_DEAUTH, pDestAddr);
        A_MACADDR_COPY(&pOpBss->bssId, &pFrame->macHeader.bssId);
        A_MACADDR_COPY(
            (lSib->serviceType == WLAN_AP_SERVICE) ? &pOpBss->bssId :
            &lSib->macAddr,
            &pFrame->macHeader.srcAddr);

        pFrame->reasonCode = reasonCode;

        if (mlmeDebugLevel > 10) {
            displayMgtFrame(&pFrame->macHeader, frameLen);
        }

        pDesc->pDestSibEntry = pSib;
        pDesc->pOpBss        = pOpBss;
        SETQHANDLE(pDesc, TXQ_ID_FOR_AC1);

        /*
         * don't ask for Tx Completion
         * notification. The SIB may be reused by then.
         */
        pDesc->bIndicateXmitStatus = FALSE;
        powerFrameTransmit(pdevInfo, pDesc); // return status doesn't matter
    }
    if (oldState & STATE_AUTH) {
        wlanMlmeDeauthConfirm(pdevInfo, pDestAddr, reasonCode);
    }
    return MLME_SUCCESS;

}

/*
 * wlanMlmeDeauthConfirm - confirm a deauthentication operation.
 *
 * It will be called on transmission of deauth frame.
 *
 * RETURNS: MLME_SUCCESS, MLME_INVALID_PARAM, MLME_TOOMANY_REQ
 *
 */
MLME_RESULT
wlanMlmeDeauthConfirm(WLAN_DEV_INFO *pdevInfo,
                      WLAN_MACADDR  *pDestAddr,
                      MLME_RESULT   resultCode)
{
    /* If callback routine exists, call it */
    if (pdevInfo->pSmeInfo->deauthCallback) {
        (*pdevInfo->pSmeInfo->deauthCallback)(pdevInfo, pDestAddr, resultCode);
    }
    return MLME_SUCCESS;
}

MLME_RESULT
wlanMlmeDeauthIndication(WLAN_DEV_INFO *pdevInfo,
                         WLAN_MACADDR  *pDestAddr,
                         MLME_RESULT   resultCode)
{
    /* If callback routine exists, call it */
    if (pdevInfo->pSmeInfo->deauthIndicationCallback) {
        (*pdevInfo->pSmeInfo->deauthIndicationCallback)(pdevInfo, pDestAddr, resultCode);
    }
    return MLME_SUCCESS;
}

MLME_RESULT
mlmeDeauthReceive(WLAN_DEV_INFO     *pdevInfo,
                  OP_BSS            *pOpBss,
                  WLAN_FRAME_DEAUTH *deauthFrame,
                  A_UINT16          frameLen)
{
    SIB_ENTRY    *pSib;
    SIB_ENTRY    *lSib = pdevInfo->localSta;

#ifndef BUILD_AP
    /* Ensure that it comes from desired AP; accept deauth from authenticated bssid only */
    if ((A_MACADDR_COMP(&pOpBss->bssId, &deauthFrame->macHeader.srcAddr)) != 0) {
        /* Got this frame from wrong AP !! */
        return MLME_REFUSED;
    }
#endif

    mlmePrintf("mlmeDeuuthReceive:\n");
    mlmePrintMacAddress(deauthFrame->macHeader.srcAddr);

    pSib = sibEntryFind(pdevInfo, &deauthFrame->macHeader.srcAddr, pOpBss);
    if (pSib == NULL) {
        mlmePrintf("mlmeDeauthReceive: Warning: "
                   "Deauth frame from unknown source\n");
    }

    if (lSib->serviceType == WLAN_STA_SERVICE) {
        if (lSib->staState & STATE_ASSOC) {
            // The receiving STA is associated too. Cancel that.
            wlanMlmeDisassocRequest(pdevInfo, pOpBss, &deauthFrame->macHeader.srcAddr,
                                    REASON_AUTH_LEAVING, FALSE);
        }

        lSib->staState &= ~(STATE_ASSOC|STATE_AUTH);
        lSib->transState = TSTATE_QUIET;
    }

    if (pSib) {
        if (pSib->staState & STATE_ASSOC && pSib->assocId != 0) {
            /* JK - inform the target to reset swretry states */
            sibPowerStateClean(pdevInfo, pSib);
            ASSERT(pSib->pOpBss == pOpBss);
            mlmeAssocIdRelease(pdevInfo, pOpBss, pSib->assocId);  /* Currently assigned ID */
            pSib->staState  &= ~(STATE_ASSOC);
            pSib->assocId = 0;
            sibCseStateClean(pdevInfo, pSib);
            sibDecompStateSet(pdevInfo, pSib, FALSE);
            wlanMlmeDisassocIndication(pdevInfo, &deauthFrame->macHeader.srcAddr,
                                       deauthFrame->reasonCode);
        }

        /* STA is not authenticated. Error. ignore. */
        pSib->staState  &= ~(STATE_ASSOC|STATE_AUTH);
        pSib->transState = TSTATE_QUIET;
        pSib->stats.DeAuthentications++;
        pSib->stats.DeAuthReasonCode = deauthFrame->reasonCode;
    }

    wlanMlmeDeauthIndication(pdevInfo, &deauthFrame->macHeader.srcAddr,
                             (MLME_RESULT)deauthFrame->reasonCode);

    return MLME_SUCCESS;
}

/************************/
/* Association routines */
/************************/

/*
 * wlanMlmeAssocRequest - Request to associate with a AP
 *
 * This routine initiates the association operation with a AP.
 * It will fail if not authenticated with the requested AP or already
 * associated with some AP.
 *
 * RETURNS: MLME_SUCCESS, MLME_INVALID_PARAM, MLME_REFUSED,
 *            MLME_NOT_SUPPORTED
 */
MLME_RESULT
wlanMlmeAssocRequest(WLAN_DEV_INFO *pdevInfo,
                     WLAN_MACADDR  *pDestAddr,
                     A_UINT32      failureTimeout,  /* in TUs */
                     CAP_INFO      *capabilityInfo,
                     A_UINT16      listenInterval)
{
    return mlmeAssocReassocRequest(pdevInfo, pDestAddr, failureTimeout,
                                   capabilityInfo, listenInterval, FALSE);
}

/*
 * wlanMlmeReassocRequest - Request to reassociate with another AP
 *
 * This routine initiates the reassociation operation with a AP.
 * It will fail if not authenticated with the requested AP or already
 * not associated with some AP.
 *
 * RETURNS: MLME_SUCCESS, MLME_INVALID_PARAM, MLME_REFUSED,
 *          MLME_NOT_SUPPORTED
 */
MLME_RESULT
wlanMlmeReassocRequest(WLAN_DEV_INFO *pdevInfo,
                       WLAN_MACADDR  *pDestAddr,
                       A_UINT32      failureTimeout,
                       CAP_INFO      *capabilityInfo,
                       A_UINT16      listenInterval)
{
    ccxCckmMicCalc(pdevInfo, pDestAddr);
    return mlmeAssocReassocRequest(pdevInfo, pDestAddr, failureTimeout,
                                   capabilityInfo, listenInterval, TRUE);
}

/*
 * mlmeAssocReassocRequest
 *
 * This routine is a local common routine which is called
 * by both wlanMlmeAssocRequest and wlanMlmeReassocRequest
 *
 * RETURNS: MLME_SUCCESS, MLME_INVALID_PARAM, MLME_REFUSED,
 *          MLME_NOT_SUPPORTED
 */
MLME_RESULT
mlmeAssocReassocRequest(WLAN_DEV_INFO *pdevInfo,
                        WLAN_MACADDR  *pDestAddr,
                        A_UINT32      failureTimeout,
                        CAP_INFO      *capabilityInfo,
                        A_UINT16      listenInterval,
                        A_BOOL        isReassoc)
{
    WLAN_FRAME_ASSOC_REQ    *pAssocFrame   = NULL;
    WLAN_FRAME_REASSOC_REQ  *pReassocFrame = NULL;
    WLAN_MGT_MAC_HEADER     *pMacHeader;
    CAP_INFO                *pCapInfo;
    SIB_ENTRY               *pSib;
    SIB_ENTRY               *lSib;
    A_UINT16                frameLen;
    ATHEROS_DESC            *pDesc;
    INFO_ELEMENT            *pTempIe;
    WLAN_RATE_SET           *pOperationalRateSet;
    RATE_SET                rateSet;
    EXT_RATE_SET            extRateSet;
    WLAN_RATE_SET           tmpRateSet;
    UCSE_SET                ucseSet;
    MCSE_SET                mcseSet;
    AUTHSE_SET              authSet;

    /* validate parameters */
    if (!pDestAddr) {
        return MLME_INVALID_PARAM;
    }

    pSib = sibEntryFind(pdevInfo, pDestAddr, NULL);
    if (!pSib) {
        return MLME_INVALID_PARAM;
    }

    lSib = pdevInfo->localSta;

    /* check current state */
    if (!(lSib->staState & STATE_AUTH) ||
        (lSib->transState == TSTATE_ASSOC_REQ_SENT) ||
        (lSib->transState == TSTATE_REASSOC_REQ_SENT))
    {
        /* Not authenticated yet, or in the middle of assoc. Refuse assoc */
        return MLME_REFUSED;
    }

    ASSERT((pdevInfo->staConfig.wpaEnabled == FALSE) ||
           (VALID_CIPHER_OUI(lSib->uCipher) &&
            VALID_CIPHER_OUI(lSib->mCipher) &&
            VALID_AUTH_OUI(lSib->authSel)));

    /* This should have been validated by cservSelectBSS() */
    ASSERT(pdevInfo->staConfig.privacyInvoked == TRUE ||
           pSib->capInfo.privacy == FALSE);

    if (isReassoc) {
        mlmePrintf("re-assoc request...\n");
    } else {
        mlmePrintf("assoc request...\n");
    }

    /* Convert WLAN_RATE_SET (internal representation) to RATE_SET/EXT_RATE_SET (802.11) */
    pOperationalRateSet = &pSib->workingRateSet;

    /* clear the basic rate set flag */
    wlanRateSetCopy(pOperationalRateSet, &tmpRateSet);
    clearBasicRateFlag(&tmpRateSet);
    pOperationalRateSet = &tmpRateSet;

    wlanFullToRateSet(pOperationalRateSet, &rateSet,
                      pdevInfo->staConfig.gdraft5);
    wlanFullToExtRateSet(pOperationalRateSet, &extRateSet,
                         pdevInfo->staConfig.gdraft5);

    /*
     * Need to compare STA rate set with bss basic rates
     * before even starting assoc process. Also, match
     * AcpInfo fields too.
     */

    /*
     * Compare the advanced capabitilies and choose  the ones
     * which the AP to which we are going to join  supports
     */
    pdevInfo->turboPrimeInfo.turboPrimeAllowed = FALSE;
    pdevInfo->turboPrimeInfo.friendlyTurboAllowed = FALSE;
    if (pdevInfo->staConfig.abolt & ABOLT_ATH_ADVCAP) {

        /* Request only the capabilties supported by AP */
        lSib->athAdvCapElement.info.useTurboPrime =
            (pdevInfo->staConfig.abolt & ABOLT_TURBO_PRIME &&
            (IS_CHAN_G(pdevInfo->staConfig.pChannel->channelFlags) ||
             IS_CHAN_A(pdevInfo->staConfig.pChannel->channelFlags))) ?
            pdevInfo->bssDescr->athAdvCapElement.info.useTurboPrime : 0;
        lSib->athAdvCapElement.info.useFriendlyTurbo =
            (lSib->athAdvCapElement.info.useTurboPrime &&
             pdevInfo->staConfig.abolt & ABOLT_FRIENDLY_TURBO) ?
             pdevInfo->bssDescr->athAdvCapElement.info.useFriendlyTurbo : 0;
        lSib->athAdvCapElement.info.useCompression =
            (pdevInfo->staConfig.compressionSupport) ?
            pdevInfo->bssDescr->athAdvCapElement.info.useCompression : 0;
        lSib->athAdvCapElement.info.useFastFrame =
            (pdevInfo->staConfig.abolt & ABOLT_FAST_FRAME) ?
            pdevInfo->bssDescr->athAdvCapElement.info.useFastFrame : 0;
        lSib->athAdvCapElement.defKeyIndex =
            (pdevInfo->bssDescr->athAdvCapElement.defKeyIndex != HWINDEX_INVALID) ?
            pdevInfo->staConfig.defaultKey : HWINDEX_INVALID;
        if (!(*((A_UINT8 *)&lSib->athAdvCapElement.info))) {
            lSib->athAdvCapElement.length = 0;
        } else {
            lSib->athAdvCapElement.length = sizeof(ATH_ADVCAP_IE) -
                                             (A_UINT32)&(((ATH_ADVCAP_IE *)0)->oui);
        }
    }

    /* Compute length of the desired frame */
    frameLen = sizeof(WLAN_MGT_MAC_HEADER) +
               sizeof(CAP_INFO) +
               sizeof(A_UINT16) +      // listen interval
               (2 * sizeof(A_UINT8)) + pdevInfo->bssDescr->ssid.length +
               INFO_ELEMENT_SIZE(rateSet) + INFO_ELEMENT_SIZE(extRateSet);

    /* reassoc request has an additional field curApAddr in the frame */
    if (isReassoc) {
        frameLen += sizeof(pReassocFrame->curAPAddr);
    }

    if (pdevInfo->staConfig.wpaEnabled) {
        /*
         * allocate the length for the WPA element using sparse structure
         * framelen will be adjusted at the end
         */
        frameLen += sizeof(WPA_IE);
        if (isReassoc && lSib->cckmEnabled) {
            frameLen += ccxGetCckmIeSize();
        }
    }

    frameLen += (VALID_ATH_ADVCAP_ELEMENT(&lSib->athAdvCapElement) ?
                   INFO_ELEMENT_SIZE(lSib->athAdvCapElement) : 0);

#ifdef WME
    if (pSib->qosdataEnabled) {
        frameLen += wmeV1ShortIeLength(pdevInfo);
    }
#endif


    if (pdevInfo->bssDescr->isCiscoAp) {
        frameLen += sizeof(AIRONET_IE);
        frameLen += sizeof(CCX_AP_ADDR);
    }

    if (createBuffandDesc(pdevInfo, frameLen, &pDesc) != 0) {
        /* Failed to allocate buffer for frame and descriptor */
        return MLME_DRIVER_ERROR;
    }

    A_MEM_ZERO(pDesc->pBufferVirtPtr.byte, frameLen);

    if (isReassoc) {
        pReassocFrame = pDesc->pBufferVirtPtr.reAssoc;
        pMacHeader    = &pReassocFrame->macHeader;
        pCapInfo      = &pReassocFrame->capInfo;
    } else {
        pAssocFrame   = pDesc->pBufferVirtPtr.assoc;
        pMacHeader    = &pAssocFrame->macHeader;
        pCapInfo      = &pAssocFrame->capInfo;
    }

    mgtFrameHdrInit(pMacHeader, isReassoc ?
                    SUBT_REASSOC_REQ : SUBT_ASSOC_REQ, pDestAddr);
    A_MACADDR_COPY(&pdevInfo->baseBss.bssId, &pMacHeader->bssId);
    A_MACADDR_COPY(&pdevInfo->localSta->macAddr, &pMacHeader->srcAddr);

    /* ANSI C required. Copying 16 bit structure */
    *pCapInfo = pdevInfo->staConfig.capInfo;

    if (
#define WAR_9374
#ifndef WAR_9374
        ccxMixedPrivacyPolicyCheck(pdevInfo) == A_OK &&
#endif
        !pdevInfo->staConfig.wpaEnabled)
    {
        pCapInfo->privacy = pdevInfo->staConfig.privacyInvoked;
    }

    // use short preamble if both of us support it, otherwise use long.
    pCapInfo->shortPreamble =
        pdevInfo->bssDescr->capabilityInfo.shortPreamble &&
        pdevInfo->staConfig.shortPreamble;

    /* Request Short Slot Time if G mode and we are configured to do so. */
    pCapInfo->shortSlotTime =
        IS_CHAN_G_OR_108G(pdevInfo->staConfig.pChannel->channelFlags) ?
        pdevInfo->staConfig.shortSlotTime : 0;

    if (isReassoc) {
        pReassocFrame->listenInterval = listenInterval;

        /* copy the ap address which is stored in lastApAddr */
        A_MACADDR_COPY(&pdevInfo->lastApAddr, &pReassocFrame->curAPAddr);

        /* Copy ssid and bump the pointer to next element */
        pTempIe = (INFO_ELEMENT *)&pReassocFrame->ssid;
    } else {
         pAssocFrame->listenInterval = listenInterval;

        /* Copy ssid and bump the pointer to next element */
        pTempIe = (INFO_ELEMENT *)&pAssocFrame->ssid;
    }

    pTempIe = mlmeElementCopy((INFO_ELEMENT *)&pdevInfo->bssDescr->ssid,
                              (INFO_ELEMENT *)pTempIe);


    pTempIe = mlmeElementCopy((INFO_ELEMENT *)&rateSet,
                              (INFO_ELEMENT *)pTempIe);

    if (VALID_ERS_ELEMENT(&extRateSet)) {
        pTempIe = mlmeElementCopy((INFO_ELEMENT *)&extRateSet,
                                  (INFO_ELEMENT *)pTempIe);
    }
    if (pdevInfo->staConfig.wpaEnabled) {
        A_UINT32 isize;

        ucseSet.uCipherCount      = 1;
        ucseSet.uCiphers[0]       = lSib->uCipher;
        mcseSet.mCipher[0]        = lSib->mCipher;
        authSet.authSelectorCount = 1;
        authSet.authSelectors[0]  = lSib->authSel;

        wpaElementsToWpaIe(&ucseSet, &mcseSet, &authSet,
                           lSib->wpaCapabilities,(WPA_IE *)pTempIe);

        isize = pTempIe->length + sizeof(pTempIe->elementID) +
        sizeof(pTempIe->length);
        pTempIe = (INFO_ELEMENT *)((A_UINT8 *)pTempIe + isize);

        if (isReassoc && lSib->cckmEnabled) {
            int bytesAdded;

            bytesAdded = ccxCckmAddIe(lSib, (A_UINT8 *)pTempIe);
            pTempIe = (INFO_ELEMENT *)((A_UINT8 *)pTempIe + bytesAdded);
        }
    }

    if (VALID_ATH_ADVCAP_ELEMENT(&lSib->athAdvCapElement)) {
        pTempIe = mlmeElementCopy((INFO_ELEMENT *)&lSib->athAdvCapElement,
                                  (INFO_ELEMENT *)pTempIe);
    }

#ifdef WME
    if (pSib->qosdataEnabled) {
        pTempIe = wmeV1ShortIeGenerate(pdevInfo, pTempIe);
    }
#endif

    /* Negotiate CKIP */
    pTempIe = ccxCkipStaRequest(pdevInfo, pdevInfo->bssDescr, (AIRONET_IE *)pTempIe);

    /* Insert our IP addr, if any */
    pTempIe = ccxStaIpAddrInsert(pdevInfo, (CCX_AP_ADDR *)pTempIe, 0);

    /* Done filling in frame, trim to proper len */
    if ((A_UINT8 *)pTempIe < (pDesc->pBufferVirtPtr.byte + frameLen)) {
        A_MEM_ZERO(pTempIe, frameLen - ((A_UINT8 *)pTempIe -
                                        pDesc->pBufferVirtPtr.byte));
        frameLen = (A_UINT8 *)pTempIe - pDesc->pBufferVirtPtr.byte;
    }

    if (pdevInfo->staConfig.wpaEnabled) {
        WLAN_PRIV_RECORD *pKey;

        if (frameLen > sizeof(pdevInfo->assocInfo.reqBuf)) {
            return MLME_DRIVER_ERROR;
        }

        /* Save this frame for upper layers to query */
        A_MEM_ZERO(pdevInfo->assocInfo.reqBuf,
                   sizeof(pdevInfo->assocInfo.reqBuf));
        A_BCOPY(pDesc->pBufferVirtPtr.byte, pdevInfo->assocInfo.reqBuf,
                frameLen);
        pdevInfo->assocInfo.reqLen = frameLen;
        pdevInfo->assocInfo.isReassoc = isReassoc;

        /* XXX should use pPriv */
        pKey = &pdevInfo->keyTable[pdevInfo->staConfig.defaultKey];
        wlanKeyTableSlotFree(pKey);
    }

    pDesc->frameLength  = frameLen;
    pDesc->bufferLength = frameLen;

    if (mlmeDebugLevel > 10) {
        displayMgtFrame(pMacHeader, frameLen);
    }

    /* send frame */
    if (isReassoc) {
        lSib->transState = TSTATE_REASSOC_REQ_SENT;
    } else {
        lSib->transState = TSTATE_ASSOC_REQ_SENT;
    }
    pDesc->pDestSibEntry = pSib;
    pDesc->pOpBss        = &pdevInfo->baseBss;
    SETQHANDLE(pDesc, TXQ_ID_FOR_AC1);
    pDesc->bIndicateXmitStatus = TRUE;
    if (powerFrameTransmit(pdevInfo, pDesc) == A_OK) {
        if (lSib->smePendDesc == NULL) {
            lSib->smePendDesc = (ATHEROS_DESC *)0xffffffff;
        }
        A_TIMEOUT(&pdevInfo->smeTimer, failureTimeout, (A_UINT32)pdevInfo,
                  isReassoc ? MLME_REASSOC_TIMEOUT : MLME_ASSOC_TIMEOUT);

        return MLME_SUCCESS;
    }

    /* else */
    lSib->transState = TSTATE_QUIET;
    return MLME_DRIVER_ERROR;
}

static void
mlmeAssocIdInit(WLAN_DEV_INFO *pdevInfo, OP_BSS *pOpBss)
{
    /* clear all AID bits; AID 0 is illegal - mark it allocated */
    A_MEM_ZERO(pOpBss->assocArray, AID_ARRAY_SZ);
    pOpBss->assocArray[0]        = 0x1;
    pOpBss->numAssociatedClients = 0;
    pOpBss->numSleepingClients   = 0;
}

/*
 * mlmeAssocIdNew - Search and assign association ID
 *
 * Called after receiving associate request. It will search thru
 * the bit array and assign a new one. Search is always from to low
 * to high order bits so that we assign the lowest available ID.
 * This reduces the possible size of TIM element in beacon.
 * Need to use locking mechanism to protect assocArray.
 */
A_UINT16
mlmeAssocIdNew(WLAN_DEV_INFO *pdevInfo, OP_BSS *pOpBss)
{
    A_UINT8       *assocArray = pOpBss->assocArray;
    A_UINT8       bitIndex;
    A_UINT8       byteIndex;
    WIRELESS_MODE mode;

    A_SEM_LOCK(pOpBss->aidSem, WAIT_FOREVER);

    /* Assign the next available Association ID */
    for (byteIndex = 0; byteIndex < AID_ARRAY_SZ; byteIndex++) {
        if (assocArray[byteIndex] == 0xff) {
            continue;
        }

        for (bitIndex = 0; bitIndex < 8; bitIndex ++) {
            if (assocArray[byteIndex] & (0x1 << bitIndex)) {
                continue;
            }

            /* Found a place */
            assocArray[byteIndex] |= (0x1 << bitIndex);
            ++pOpBss->numAssociatedClients;
            A_SEM_UNLOCK(pOpBss->aidSem);

            if (pOpBss->numAssociatedClients == 2) {
                mode = wlanFindModeFromRateTable(pdevInfo, pOpBss);
                wlanUpdatePhyChAPs(pdevInfo, pOpBss, mode);
                pdevInfo->baseBss.phyChAPs.info++;
                wlanUpdateTxQueues(pdevInfo);
            }

            /* Bits 14 and 15 must be 1 according to spec */
            return (A_UINT16)((byteIndex * 8 + bitIndex) | 0xc000);
        }

        ASSERT(0);
    }

    A_SEM_UNLOCK(pOpBss->aidSem);

    return 0;
}

/*
 * mlmeAssocIdRelease - Make assocId available again
 *
 * Called by disassociate primitive. It will clear the corresponding
 * bit and make the assocId available for others.
 * Need to use locking mechanism to protect assocArray.
 */
void
mlmeAssocIdRelease(WLAN_DEV_INFO *pdevInfo, OP_BSS *pOpBss, A_UINT16 assocId)
{
    A_UINT8       *assocArray = pOpBss->assocArray;
    A_INT8        bitIndex;
    int           byteIndex;
    WIRELESS_MODE mode;

    assocId &= 0x3fff;
    if (assocId > ASSOCID_MAX || assocId == 0) {
        mlmePrintf("mlmeAssocIdRelease: ERROR: Illegal AssocId %04x\n", assocId);
        return;
    }

    A_SEM_LOCK(pOpBss->aidSem, WAIT_FOREVER);

    byteIndex = assocId >> 3;     /* 8 bits per byte */
    bitIndex  = assocId & 0x7;    /* Lowest 3 bits, modulo 8 */

    assocArray[byteIndex] = assocArray[byteIndex] & (~(0x1 << (assocId & 0x7)));
    --pOpBss->numAssociatedClients;
    if (pOpBss->numAssociatedClients == 1) {
        mode = wlanFindModeFromRateTable(pdevInfo, pOpBss);
        wlanUpdatePhyChAPs(pdevInfo, pOpBss, mode);
        pdevInfo->baseBss.phyChAPs.info++;
        wlanUpdateTxQueues(pdevInfo);
    }

    A_SEM_UNLOCK(pOpBss->aidSem);
}

/*
 * mlmeAssocReceive - Process the received Assoc Request frame on AP.
 *
 * This routine handles the incoming associate requests on AP. It checks the
 * state, assigns a new association id and sends back the response frame.
 * This routine is not part of MLME API.
 */
A_UINT16
mlmeAssocApReceive(WLAN_DEV_INFO        *pdevInfo,
                   OP_BSS               *pOpBss,
                   WLAN_FRAME_ASSOC_REQ *assocFrame,
                   A_UINT16             frameLen)
{
    return mlmeAssocReassocApReceive(pdevInfo, pOpBss, assocFrame,
                                     frameLen, FALSE);
}

/*
 * mlmeReassocReceive - Process the received Reassoc Request frame on AP.
 *
 * This routine handles the incoming reassociate requests on AP. It checks the
 * state, assigns a new association id and sends back the response frame.
 * This routine is not part of MLME API.
 */
A_UINT16
mlmeReassocApReceive(WLAN_DEV_INFO          *pdevInfo,
                     OP_BSS                 *pOpBss,
                     WLAN_FRAME_REASSOC_REQ *pFrame,
                     A_UINT16               frameLen)
{
    return mlmeAssocReassocApReceive(pdevInfo, pOpBss, pFrame, frameLen, TRUE);
}

/*
 * wlanMlmeAssocReassocRequest
 *
 * This routine is a local common routine which is called
 * by both wlanMlmeAssocApReceive and wlanMlmeReassocApReceive
 *
 */
A_UINT16
mlmeAssocReassocApReceive(WLAN_DEV_INFO *pdevInfo,
                          OP_BSS        *pOpBss,
                          void          *pReqFrame,
                          A_UINT16      frameLen,
                          A_BOOL        isReassoc)
{
    WLAN_FRAME_ASSOC_RESP   *pFrame;
    SIB_ENTRY               *pSib;
    SIB_ENTRY               *lSib;
    A_UINT16                statusCode;
    A_UINT16                newFrameLen;
    ATHEROS_DESC            *pDesc;
    RATE_SET                *pRateSet;
    EXT_RATE_SET            *pExtRateSet;
    RATE_SET                rateSet;
    EXT_RATE_SET            extRateSet;
    WLAN_RATE_SET           wlanStaRateSet, wlan11bRateSet;
    CONN_RATE_SET           connRateSet;
    int                     lock = 0;
    SSID                    *pSsid;
    WLAN_MGT_MAC_HEADER     *pMacHeader;
    CAP_INFO                capInfo;
    INFO_ELEMENT            *pTempIe      = NULL;
    A_BOOL                  flagProtect   = FALSE;
    A_BOOL                  flagShort     = TRUE;
    A_BOOL                  flagPreamble  = TRUE;
    WLAN_FRAME_ASSOC_REQ    *assocFrame   = NULL;
    WLAN_FRAME_REASSOC_REQ  *reassocFrame = NULL;
    UCSE_SET                bssUcseSet;
    MCSE_SET                bssMcseSet;
    AUTHSE_SET              bssAuthSet;
    UCSE_SET                ucseSet;
    MCSE_SET                mcseSet;
    AUTHSE_SET              authSet;
    WPA_IE                  *pWpaIe = NULL;
    int                     i;
    INFO_ELEMENT_SET        ieSet;
    A_UINT8                 *pInfoElements = NULL;
    A_UINT16                infoElemLen = 0;

    if (!(pdevInfo->localSta->staState & STATE_AP_UP)) {
        return MLME_REFUSED;
    }

    lSib = pdevInfo->localSta;

    if (isReassoc) {
        reassocFrame = (WLAN_FRAME_REASSOC_REQ *)pReqFrame;
        pMacHeader   = &reassocFrame->macHeader;
        pSsid        = &reassocFrame->ssid;
        capInfo      = reassocFrame->capInfo;
     } else {
        assocFrame = (WLAN_FRAME_ASSOC_REQ *)pReqFrame;
        pMacHeader = &assocFrame->macHeader;
        pSsid      = &assocFrame->ssid;
        capInfo    = assocFrame->capInfo;
    }

    /* Validate SSID in the request frame. */
    if (pdevInfo->bssDescr->ssid.length != pSsid->length ||
        A_BCOMP(pSsid, &pdevInfo->bssDescr->ssid, pSsid->length) != 0)
    {
        return MLME_INVALID_PARAM;
    }

    pSib = sibEntryFind(pdevInfo, &pMacHeader->srcAddr, pOpBss);
    if (!pSib) {
        /* 802.11 section 5.5(b) says that AP should send a deauth frame to sender */
        wlanMlmeDeauthRequest(pdevInfo, pOpBss, &pMacHeader->srcAddr,
                              REASON_CLASS2, TRUE);
        mlmePrintf("mlme[Re]AssocApReceive -- can't find SIB\n");
        pdevInfo->localSta->stats.DeAuthentications++;
        return MLME_INVALID_PARAM;
    }

    /* Verify state */

#ifdef BUILD_AP
    /*
     * There's a race condition between us doing tx completion on the
     * auth response which went out and getting the subsequent assoc
     * request from the station; the tx completion on the auth response
     * is needed to get us in the right state to process the assoc
     * request! so the following hack
     */
    if ((!(pSib->staState & STATE_AUTH))
        && ((pSib->transState == TSTATE_AUTH_FRAME2_SENT) ||
            (pSib->transState == TSTATE_AUTH_FRAME4_SENT)))
    {
        processTxQueue(pdevInfo, &pSib->pOpBss->txQueue);
    }
#endif

    if (!(pSib->staState & STATE_AUTH)) {
        /*
         * STA is not authenticated. Error. Should send a deauth frame.
         * 802.11 section 5.5(b) says that AP should send a deauth frame to sender
         */
        wlanMlmeDeauthRequest(pdevInfo, pOpBss, &pMacHeader->srcAddr,
                              REASON_CLASS2, TRUE);
        mlmePrintf("mlme[Re]AssocApReceive -- Not Authenticated\n");

        pdevInfo->localSta->stats.DeAuthentications++;
        return MLME_REFUSED;
    }

    if (pSib->staState & STATE_ASSOC) {
        /* STA is already associated. Error. Should send a disassoc frame */
        /* 802.11 section 5.5(b) says that AP should send a disassoc frame to sender */
        wlanMlmeDisassocRequest(pdevInfo, pOpBss, &pMacHeader->srcAddr,
                                REASON_UNSPECIFIED, TRUE);
        mlmePrintf("mlme[Re]AssocApReceive -- Already Associated\n");

        pdevInfo->localSta->stats.DisAssociations++;
        return MLME_REFUSED;
    }

    if (pdevInfo->localSta->transState & TSTATE_SCANNING) {
        mlmePrintf("mlme[Re]AssocApReceive -- Not Authenticated\n");
        return MLME_REFUSED;
    }

    /* set the default value for statusCode */
    statusCode = WLAN_SUCCESS;

    if (pdevInfo->trafficBlockTimestamp != 0) {
        if ((A_MS_TICKGET() - pdevInfo->trafficBlockTimestamp) <
            MIC_COUNTERMEASURE_TIMEOUT)
        {
            mlmePrintf("Assoc attempted by ");
            mlmePrintMacAddress(pSib->macAddr);
            mlmePrintf(" refused\n");
            /* do NOT update the timestamp here! */
            statusCode = WLAN_OUT_OF_SCOPE;
        } else {
            /* Gate should be opened now */
            mlmePrintf("Clearing trafficBlockTS\n");
            pdevInfo->trafficBlockTimestamp = 0;
        }
    }

    /* Need to compare various bits in assoc frame capInfo field */
    if (capInfo.cfPollable && !capInfo.cfPollReq) {
        /*
         * Spec section 7.3.1.4, table 16. We want to avoid the
         * case where STA is CF pollable and requesting to be
         * placed on the CF polling list.
         */
        statusCode = WLAN_UNSUPPORTED_CAP;
        mlmePrintf("mlme[Re]AssocApReceive -- Unsupported capInfo found!\n");
    }

    pSib->capInfo.shortPreamble = capInfo.shortPreamble;

    A_MEM_ZERO(&ieSet,sizeof(ieSet));
    if (isReassoc) {
        pInfoElements = (A_UINT8 *) &(reassocFrame->ssid);
        infoElemLen = (A_UINT16)(frameLen - (pInfoElements -
                                 (A_UINT8 *)&reassocFrame->macHeader));
    }
    else {
        pInfoElements = (A_UINT8 *) &(assocFrame->ssid);
        infoElemLen = (A_UINT16)(frameLen - (pInfoElements -
                                 (A_UINT8 *)&assocFrame->macHeader));
    }
    mgtGetInfoElements(&ieSet,(INFO_ELEMENT *)pInfoElements,infoElemLen);

    /* STA rate set compared with BSS basic rate set */
    pRateSet    = (RATE_SET *)
        ieSet.pInfoElement[eidMap[ELE_SUPPORTED_RATES]];

    pExtRateSet = (EXT_RATE_SET *)
        ieSet.pInfoElement[eidMap[ELE_EXT_SUPPORTED_RATES]];

    wlanElementToFullRateSet(pRateSet, pExtRateSet, &wlanStaRateSet);

    /*
     * Check current rate table basic rate to determine association eligibility.
     * If 11g mode and 11g draft 5.0 compatibility, also check
     * the 11b rate table basic rate.
     * See bug 6249
     */
#if NDIS_WDM
    /*
     * (LW) Attention !!!
     * This is temp hack, should revisit later
     */
    wlanRateTbl2RateSet(pdevInfo->hwRateTable[WIRELESS_MODE_11b],
                        &wlan11bRateSet);
#else
    wlanRateTbl2RateSet(HACK_TO_PARDEV(pdevInfo)->hwRateTable[WIRELESS_MODE_11b],
                        &wlan11bRateSet);
#endif
    if (basicRateSubsetCheck(&pOpBss->rateSet, &wlanStaRateSet) == FALSE &&
        (!IS_CHAN_G(pdevInfo->staConfig.pChannel->channelFlags) ||
         !pdevInfo->staConfig.gdraft5 ||
         !basicRateSubsetCheck(&wlan11bRateSet, &wlanStaRateSet)))
    {
        if (mlmeDebugLevel) {
            displayMgtFrame(pMacHeader, frameLen);
        }
        mlmePrintf("mlme[Re]AssocApReceive -- Unsupported rates found!\n");
        statusCode = WLAN_UNSUPPORTED_RATE;
    }

    /* Update the working rate set to be used for communicating
     * to this sta. It's an intersection of the two sets.
     */
    xSectionRateSet(&pOpBss->rateSet, &wlanStaRateSet, &pSib->workingRateSet);

    /* update the sta wireless mode */
    staWlanModeSet(pdevInfo, pSib);

    /* Update target connection */
    wlanRateSet2WdcRateSet(&pSib->workingRateSet, &connRateSet);
    wdcUpdateConnectionAttribute(pdevInfo->targetHandle,
                                 pSib->wdcConnId,
                                 CONN_WORKING_RATE_SET,
                                 sizeof(connRateSet),
                                 &connRateSet);

    wdcUpdateConnectionAttribute(pdevInfo->targetHandle,
                                 pSib->wdcConnId,
                                 CONN_WLAN_MODE,
                                 sizeof(pSib->wlanMode),
                                 (TARGET_CONFIG_VAL)&pSib->wlanMode);

    /*
     * Get the STA's cipher suite. Make the element invalid first.
     */
    sibCseStateClean(pdevInfo, pSib);

    if (pdevInfo->staConfig.wpaEnabled) {
        A_UINT16 wpaCap = 0;
        A_UINT16 bssWpaCap = 0;

        pWpaIe = (WPA_IE *)ieSet.pWpaIe;
        if ((pWpaIe != NULL) && VALID_WPA_ELEMENT(pWpaIe)) {
            if (wpaIeToWpaElements(pWpaIe, &ucseSet, &mcseSet,
                                   &authSet, &wpaCap) != A_OK)
            {
                statusCode = WLAN_OUT_OF_SCOPE;
            } else if (wpaIeToWpaElements(
                           &pdevInfo->bssDescr->wpaIe, &bssUcseSet,
                           &bssMcseSet,&bssAuthSet, &bssWpaCap) != A_OK)
            {
                statusCode = WLAN_OUT_OF_SCOPE;
            }

            if (ucseSet.uCipherCount == 1) {
                for (i = 0; i < bssUcseSet.uCipherCount; i++) {
                    if (ucseSet.uCiphers[0] == bssUcseSet.uCiphers[i]) {
                        mlmePrintf("Assoc: unicast cipher = 0x%x\n",
                                   bssUcseSet.uCiphers[i]);
                        pSib->uCipher = bssUcseSet.uCiphers[i];
                        break;
                    }
                }
                if (i >= bssUcseSet.uCipherCount) {
                    mlmePrintf("Assoc: unicast cipher not found\n");
                    statusCode = WLAN_UNSUPPORTED_ALG;
                }
            } else {
                mlmePrintf("Assoc: Not exactly one unicast cipher\n");
                statusCode = WLAN_UNSUPPORTED_ALG;
            }
            if (statusCode == WLAN_SUCCESS) {
                if (mcseSet.mCipher[0] == bssMcseSet.mCipher[0]) {
                    mlmePrintf("Assoc: multicast cipher = 0x%x\n",
                                   bssMcseSet.mCipher[0]);
                    pSib->mCipher = bssMcseSet.mCipher[0];
                } else {
                    mlmePrintf("Assoc: multicast cipher not found\n");
                    statusCode = WLAN_UNSUPPORTED_ALG;
                }
            }

            if ((statusCode == WLAN_SUCCESS) &&
                                authSet.authSelectorCount == 1) {
                for (i = 0; i < (bssAuthSet.authSelectorCount ); i++) {
                    if (authSet.authSelectors[0] ==
                        bssAuthSet.authSelectors[i]) {
                        mlmePrintf("Assoc: Auth Mode=0x%x\n",
                                   bssAuthSet.authSelectors[i]);
                        pSib->authSel = bssAuthSet.authSelectors[i];
                        break;
                    }
                }
                if (i >= bssAuthSet.authSelectorCount) {
                    mlmePrintf("Assoc: Authentication Mode not found\n");
                    statusCode = WLAN_UNSUPPORTED_ALG;
                }
            } else if (authSet.authSelectorCount != 1) {
                mlmePrintf("Assoc: Not exactly one Authentication Mode\n");
                statusCode = WLAN_UNSUPPORTED_ALG;
            }
        } else {
            mlmePrintf("Assoc: Missing or bad WPA IE from ");
            mlmePrintMacAddress(pMacHeader->srcAddr);
            mlmePrintf("\n");
            statusCode = WLAN_UNKNOWN_FAILURE;
        }
        if (statusCode == WLAN_SUCCESS) {
            /* Save WPA IE for upper layers to query */
            if (frameLen > sizeof(pdevInfo->assocInfo.reqBuf)) {
                return MLME_REFUSED;
            }

            A_MEM_ZERO(pdevInfo->assocInfo.reqBuf,
                       sizeof(pdevInfo->assocInfo.reqBuf));
            A_BCOPY(pMacHeader, pdevInfo->assocInfo.reqBuf, frameLen);
            pdevInfo->assocInfo.reqLen = frameLen;

        } else {
            /* Reset the working Cipher */
            pSib->uCipher = 0;
            pSib->mCipher = 0;
            pSib->authSel = 0;

        }
    }

    /* Update the privacy, preamble bits at the same time */
    pSib->capInfo = capInfo;

    /* Check Advanced Capability Element intersection */
    pSib->athAdvCapElement.length = 0;
    if ((ieSet.pAthCap != NULL) && VALID_ATH_ADVCAP_ELEMENT(ieSet.pAthCap)) {
        if ((ieSet.pAthCap->info.useTurboPrime &&
                !pdevInfo->bssDescr->athAdvCapElement.info.useTurboPrime) ||
            (ieSet.pAthCap->info.useFriendlyTurbo &&
                !pdevInfo->bssDescr->athAdvCapElement.info.useFriendlyTurbo) ||
            (ieSet.pAthCap->info.useCompression &&
                !pdevInfo->bssDescr->athAdvCapElement.info.useCompression) ||
            (ieSet.pAthCap->info.useFastFrame &&
                !pdevInfo->bssDescr->athAdvCapElement.info.useFastFrame))
        {
            statusCode = WLAN_UNSUPPORTED_CAP;
            *((A_UINT8 *)&pSib->athAdvCapElement.info) = 0;
        } else {
            mlmeElementCopy((INFO_ELEMENT *)ieSet.pAthCap,
                            (INFO_ELEMENT *)&pSib->athAdvCapElement);
        }
    } else {
        *((A_UINT8 *)&pSib->athAdvCapElement.info) = 0;
    }

    /* TURBO_PRIME */
    if (!pSib->athAdvCapElement.info.useTurboPrime) {
        pdevInfo->turboPrimeInfo.legacyActivity = A_MS_TICKGET();
        pdevInfo->turboPrimeInfo.legacyAssoc    = TRUE;
        pdevInfo->turboPrimeInfo.legacyPresent  = TRUE;
        pdevInfo->turboPrimeInfo.nonFriendlyStaAssoc = TRUE;
        pdevInfo->turboPrimeInfo.nonFriendlyStaPresent  = TRUE;
    }

    if (!pSib->athAdvCapElement.info.useFriendlyTurbo) {
        pdevInfo->turboPrimeInfo.nonFriendlyStaAssoc = TRUE;
        pdevInfo->turboPrimeInfo.nonFriendlyStaPresent  = TRUE;
    }

#ifdef WME
    if (ieSet.pWmeV1Ie != NULL) {
        pSib->qosdataEnabled = pdevInfo->staConfig.WmeEnabled ? 1 : 0;
        mlmePrintf ("WME Info Element received from Association Request, set qosdataEnabled=%d\n",pSib->qosdataEnabled);
    } else {
        pSib->qosdataEnabled = 0;
    }
#endif

    /*
     * according to the 802.11 spec, no information is present in
     * a station's capInfo.privacy bit! some sneaky implementations
     * attach sematics to that bit to request encryption from mixed
     * mode APs (APs which support both encrypted and unencrypted
     * traffic at the same time and consequently do not set the
     * "required encryption" bit in capInfo elements generated by
     * them - our AP does not support such a mixed mode, so we
     * shouldn't be looking into that bit for that reason either -
     * but use it to flag out sneaky clients nevertheless
     */
    if (statusCode == WLAN_SUCCESS &&
        !pdevInfo->staConfig.privacyInvoked && capInfo.privacy)
    {
        mlmePrintf("mlme[Re]AssocApReceive: "
                   "sneaky station requesting privacy! AP privacy off\n");
        statusCode = WLAN_UNSUPPORTED_CAP;
    }

    pSib->privacyInvoked = pdevInfo->staConfig.privacyInvoked;

    /* Added for 11g */
    /* A long slot time sta has associated, turn off short slot time */
    if (!pSib->capInfo.shortSlotTime) {
        flagShort = FALSE;
        pdevInfo->protectInfo.updateLong = TRUE;
    }

    if (!pSib->capInfo.shortPreamble) {
        flagPreamble = FALSE;
        pdevInfo->protectInfo.updatePreamble = TRUE;
    }

    if (IS_CHAN_G(pdevInfo->staConfig.pChannel->channelFlags) &&
        pSib->wlanMode == STA_MODE_B)
    {
        /* update protection and slot time status for 11g */
        if (pdevInfo->protectInfo.option &
            (PROT_OPTION_1 | PROT_OPTION_2 | PROT_OPTION_4))
        {
            flagProtect = TRUE;
        }

        pdevInfo->protectInfo.updatePresent = TRUE;
    }

    /* update 11g bss info */
#ifdef BUILD_AP
    lock = intLock();
    if (flagProtect) {
        pdevInfo->bssDescr->nonErpElement.info.present = 1;
    }
    if (!flagShort){
        pdevInfo->bssDescr->capabilityInfo.shortSlotTime = 0;
    }
    if (!flagPreamble){
        pdevInfo->bssDescr->nonErpElement.info.preamble = 1;
    }
    intUnlock(lock);
#endif  /* BUILD_AP */

    /* Convert WLAN_RATE_SET (internal representation) to RATE_SET/EXT_RATE_SET (802.11) */
    wlanFullToRateSet(&pSib->workingRateSet, &rateSet, pdevInfo->staConfig.gdraft5);
    wlanFullToExtRateSet(&pSib->workingRateSet, &extRateSet, pdevInfo->staConfig.gdraft5);

    /* Compute length of the desired frame */
    newFrameLen = sizeof(WLAN_MGT_MAC_HEADER) +
                  sizeof(CAP_INFO) +
                  sizeof(A_UINT16) + /* Association ID */
                  sizeof(pFrame->statusCode) +
                  INFO_ELEMENT_SIZE(rateSet) +
                  INFO_ELEMENT_SIZE(extRateSet) +
#ifdef WME
                  athWmeIeLength(pdevInfo) +
                  (pSib->qosdataEnabled ? wmeV1IeLength(pdevInfo) : 0);
#else
                  athWmeIeLength(pdevInfo);
#endif

    /* AP should not send any WPA info back in assocResp */

    if ((ieSet.pAthCap != NULL) && VALID_ATH_ADVCAP_ELEMENT(ieSet.pAthCap)) {
        /* Send an Adv Response only if a station asked for it */
        newFrameLen +=  (VALID_ATH_ADVCAP_ELEMENT(&pSib->athAdvCapElement)) ?
                        INFO_ELEMENT_SIZE(pSib->athAdvCapElement) :
                        (VALID_ATH_ADVCAP_ELEMENT(&pdevInfo->bssDescr->athAdvCapElement) ?
                        INFO_ELEMENT_SIZE(pdevInfo->bssDescr->athAdvCapElement) : 0);
    }

    /* Allocate assoc response frame */
    if ((createBuffandDesc(pdevInfo, newFrameLen, &pDesc)) != 0) {
        /* Failed to allocate buffer for frame and descriptor */
        return MLME_DRIVER_ERROR;
    }

    /*
     * Both assoc response and reassoc response has the same frame format.
     * Using the assoc structures even for reassoc frames.
     */

    pFrame = pDesc->pBufferVirtPtr.assocResp;

    /* prepare assoc response frame */
    mgtFrameHdrInit(&pFrame->macHeader,
                    isReassoc ? SUBT_REASSOC_RESP : SUBT_ASSOC_RESP, &pSib->macAddr);

    A_MACADDR_COPY(&pOpBss->bssId, &pFrame->macHeader.bssId);
    A_MACADDR_COPY(&pOpBss->bssId, &pFrame->macHeader.srcAddr);

    /*
     * according to Spec section 7.3.1.4:
     * APs controls Privacy, CF-Pollable and CF-Poll Request subfields of
     * Association Response, and Reassociation Response management frames
     */
    pFrame->capInfo = lSib->capInfo;

    if (IS_CHAN_G_OR_108G(pdevInfo->staConfig.pChannel->channelFlags)) {
        /* 11g: if BSS is actually using short slot time, return that capability */
        pFrame->capInfo.shortSlotTime =
            (pdevInfo->bssDescr->capabilityInfo.shortSlotTime ?
                                pSib->capInfo.shortSlotTime : 0);
    }

    if (pSib->assocId != 0) {
        /* Release the association ID */
        ASSERT(pSib->pOpBss == pOpBss);
        mlmeAssocIdRelease(pdevInfo, pOpBss, pSib->assocId);
        pSib->assocId = 0;
    }

    if (statusCode == WLAN_SUCCESS) {
        if ((pFrame->assocId = mlmeAssocIdNew(pdevInfo, pOpBss)) == 0) {
            /* Failed to get a new assoc id. Table must be full */
            statusCode = WLAN_TOOMANY_ASSOC;
        } else {
            pSib->stats.Associations++;
        }
    } else {
        pFrame->assocId = 0;
    }

    if (isReassoc) {
        pSib->listenInterval = reassocFrame->listenInterval;
        pSib->stats.Reassociations++;
    } else {
        pSib->listenInterval = assocFrame->listenInterval;
        pSib->stats.Associations++;
    }

    pFrame->statusCode = statusCode;

    pTempIe = mlmeElementCopy((INFO_ELEMENT *)&rateSet,
                              (INFO_ELEMENT *)&pFrame->rateSet);
    if (VALID_ERS_ELEMENT(&extRateSet)) {
        pTempIe = mlmeElementCopy((INFO_ELEMENT *)&extRateSet,
                                  (INFO_ELEMENT *)pTempIe);
    }

    if ((ieSet.pAthCap != NULL) && VALID_ATH_ADVCAP_ELEMENT(ieSet.pAthCap)) {
        if (VALID_ATH_ADVCAP_ELEMENT(&pSib->athAdvCapElement)) {
            pTempIe = mlmeElementCopy((INFO_ELEMENT *)&pSib->athAdvCapElement,
                                      (INFO_ELEMENT *)pTempIe);
        } else if (VALID_ATH_ADVCAP_ELEMENT(&pdevInfo->bssDescr->athAdvCapElement)){
            pTempIe = mlmeElementCopy((INFO_ELEMENT *)&pdevInfo->bssDescr->athAdvCapElement,
                                      (INFO_ELEMENT *)pTempIe);
        }
    }

    pTempIe = athWmeIeGenerate (pdevInfo, pOpBss, pTempIe);
#ifdef WME
    if (pSib->qosdataEnabled) {
        mlmePrintf ("Generating WME Info Element in Assoc resp...\n");
        pTempIe = wmeV1IeGenerate(pdevInfo, pOpBss, pTempIe);
    }
#endif

    mlmePrintf("\nwlan%d mlme[Re]AssocApReceive response\n", pdevInfo->devno);
    if (mlmeDebugLevel > 10) {
        displayMgtFrame(&pFrame->macHeader, newFrameLen);
    }

    /* send frame */
    if (isReassoc) {
        pSib->transState = TSTATE_REASSOC_RESP_SENT;
    } else {
        pSib->transState = TSTATE_ASSOC_RESP_SENT;
    }
    pDesc->pDestSibEntry = pSib;
    pDesc->pOpBss        = pOpBss;
    SETQHANDLE(pDesc, TXQ_ID_FOR_AC1);
    pDesc->bIndicateXmitStatus = TRUE;
    if (powerFrameTransmit(pdevInfo, pDesc) == A_OK) {
        if (pSib->smePendDesc == NULL) {
            pSib->smePendDesc = (ATHEROS_DESC *)0xffffffff;
        }

        return MLME_SUCCESS;
    }

    pSib->transState = TSTATE_QUIET;

    return MLME_DRIVER_ERROR;
}

A_UINT16
mlmeAssocStaReceive(WLAN_DEV_INFO *pDev, WLAN_FRAME_ASSOC_RESP *pFrame,
                    A_UINT16 frameLen)
{
    return mlmeAssocReassocStaReceive(pDev, pFrame, frameLen, FALSE);
}

A_UINT16
mlmeReassocStaReceive(WLAN_DEV_INFO *pDev, WLAN_FRAME_REASSOC_RESP *pFrame,
                      A_UINT16 frameLen)
{
    return mlmeAssocReassocStaReceive(pDev, pFrame, frameLen, TRUE);
}

A_UINT16
mlmeAssocReassocStaReceive(WLAN_DEV_INFO *pdevInfo, void *pResFrame,
                           A_UINT16 frameLen, A_BOOL isReassoc)
{
    SIB_ENTRY             *lSib;
    SIB_ENTRY             *pSib;
    AIRONET_IE            *aie = NULL;
    RATE_SET              *pRateSet = NULL;
    EXT_RATE_SET          *pExtRateSet = NULL;
    WLAN_RATE_SET         wlanApRateSet;
    WLAN_FRAME_ASSOC_RESP *pFrame;
    INFO_ELEMENT_SET      ieSet;
    A_UINT8               *pInfoElements = NULL;
    A_UINT16              infoElemLen = 0;
    CCX_AP_ADDR           *apAddr;
    CONN_RATE_SET         connRateSet;
    A_UINT32              bssAttribValue;

    lSib = pdevInfo->localSta;

    /*
     * Both assoc response and reassoc response has the same frame format.
     * Using the assoc structures even for reassoc frames.
     */
    pFrame = (WLAN_FRAME_ASSOC_RESP *)pResFrame;

    /*
     * Ensure that it comes from desired AP (preauthenticate
     * not supported yet)
     */
    if ((A_MACADDR_COMP(&pdevInfo->baseBss.bssId,
                        &pFrame->macHeader.srcAddr)) != 0)
    {
        /* Got this frame from wrong AP !! */
        return MLME_REFUSED;
    }

    pSib = sibEntryFind(pdevInfo, &pFrame->macHeader.srcAddr, NULL);
    if (!pSib) {
        /* Program error, should never happen */
        //ASSERT(0);
        return MLME_INVALID_PARAM;
    }

    if (!(lSib->staState & STATE_AUTH)) {
        /*
         * Our authentication got cancelled while assoc request was
         * in progress, then this frame arrived. We should treat is as fail.
         * This can happen when last assoc attempt takes too long, just when the
         * cows start coming home, this frame rushes in.
         * Change statusCode so that it will escape out of rest of the logic.
         */
        pFrame->statusCode = WLAN_UNKNOWN_FAILURE;

        /* TODO:
         *
         * We should actually send disassoc request frame to AP and explicitly
         * cancel association, but it may mess up connection services logic.
         * For right now we will let AP figure it out automatically
         * by out of state requests.
         */
    }

    if (ccxMixedPrivacyPolicyCheck(pdevInfo) == A_OK) {
        if (pFrame->capInfo.privacy != pdevInfo->staConfig.privacyInvoked) {
            mlmePrintf("Attempt to assoc to mixed cell failed\n");
            pFrame->statusCode = WLAN_UNSUPPORTED_CAP;
        }
    }

    /* check the status and record the results */
    if (pFrame->statusCode == WLAN_SUCCESS) {
        if (pdevInfo->staConfig.wpaEnabled) {
            if (frameLen > sizeof(pdevInfo->assocInfo.respBuf)) {
                return MLME_DRIVER_ERROR;
            }

            /* save this frame */
            A_MEM_ZERO(pdevInfo->assocInfo.respBuf,
                       sizeof(pdevInfo->assocInfo.respBuf));
            A_BCOPY(pFrame, pdevInfo->assocInfo.respBuf, frameLen);
            pdevInfo->assocInfo.respLen = frameLen;
        }

        /*
         * Update the working rate set to be used for communicating
         * to this AP. It's an intersection of the rates supported
         * by this device and the rates supported by the AP.
         */
        A_MEM_ZERO(&ieSet, sizeof(ieSet));
        pInfoElements = (A_UINT8 *)&pFrame->rateSet;
        infoElemLen = (A_UINT16)(frameLen - (pInfoElements -
                                (A_UINT8 *)&pFrame->macHeader));
        mgtGetInfoElements(&ieSet, (INFO_ELEMENT *)pInfoElements, infoElemLen);

        /* STA rate set compared with BSS basic rate set */
        pRateSet    = (RATE_SET *)
            ieSet.pInfoElement[eidMap[ELE_SUPPORTED_RATES]];
        pExtRateSet = (EXT_RATE_SET *)
            ieSet.pInfoElement[eidMap[ELE_EXT_SUPPORTED_RATES]];
        wlanElementToFullRateSet(pRateSet, pExtRateSet, &wlanApRateSet);

        xSectionRateSet(&wlanApRateSet, &pdevInfo->baseBss.rateSet,
                        &pSib->workingRateSet);
        wlanRateSetCopy(&pSib->workingRateSet, &lSib->workingRateSet);
        lSib->wdcConnId = pSib->wdcConnId;

        /* Update target connection */
        wlanRateSet2WdcRateSet(&pSib->workingRateSet, &connRateSet);
        wdcUpdateConnectionAttribute(pdevInfo->targetHandle,
                                     pSib->wdcConnId,
                                     CONN_WORKING_RATE_SET,
                                     sizeof(connRateSet),
                                     &connRateSet);

        pSib->athAdvCapElement.length = 0;
        A_MEM_ZERO(&pSib->athAdvCapElement.info, sizeof(pSib->athAdvCapElement.info));
        mlmeElementCopy((INFO_ELEMENT *)ieSet.pAthCap,
                        (INFO_ELEMENT *)&pSib->athAdvCapElement);

        if (VALID_ATH_ADVCAP_ELEMENT(&pSib->athAdvCapElement)) {
            pdevInfo->turboPrimeInfo.turboPrimeAllowed =
                                pSib->athAdvCapElement.info.useTurboPrime;
            pdevInfo->turboPrimeInfo.friendlyTurboAllowed =
                                pSib->athAdvCapElement.info.useFriendlyTurbo;
            pSib->athAdvCapElement.defKeyIndex =
                                pdevInfo->bssDescr->athAdvCapElement.defKeyIndex;
        }

        /* The assoc resp overrides any history */
        aie = (AIRONET_IE *)ieSet.pInfoElement[eidMap[ELE_AIRONET]];
        ccxAIEAssocRespProcess(pdevInfo, aie);

        apAddr = (CCX_AP_ADDR *)ieSet.pInfoElement[eidMap[ELE_CCX_AP_ADDR]];
        ccxApAddrAssocRespProcess(pdevInfo, apAddr);

        pdevInfo->bssDescr->athWmeIe.length = 0;
        if ((pdevInfo->staConfig.abolt & ABOLT_WME_ELE) && ieSet.pAthWmeIe) {
            mlmeElementCopy((INFO_ELEMENT *)ieSet.pAthWmeIe,
                    (INFO_ELEMENT *)&pdevInfo->bssDescr->athWmeIe);
            athWmeIeGetInfo(pdevInfo, &pdevInfo->baseBss, ieSet.pAthWmeIe);
            wlanUpdateTxQueues(pdevInfo);
        }

#ifdef WME
        pdevInfo->bssDescr->wmeV1Ie.length = 0;

        if (pdevInfo->staConfig.WmeEnabled && ieSet.pWmeV1Ie) {
            mlmePrintf ("Copying WME ie in mlmeAssocReassocStaReceive()....\n");
            pdevInfo->localSta->qosdataEnabled = 1;
            pSib->qosdataEnabled = 1;
            pdevInfo->usingQos = pdevInfo->localSta->qosdataEnabled;
            mlmeElementCopy((INFO_ELEMENT *)ieSet.pWmeV1Ie,
                    (INFO_ELEMENT *)&pdevInfo->bssDescr->wmeV1Ie);
            wmeV1IeGetInfo(pdevInfo, &pdevInfo->baseBss, ieSet.pWmeV1Ie);

            wlanUpdateTxQueues(pdevInfo);
        } else {
            pdevInfo->localSta->qosdataEnabled = 0;
            pSib->qosdataEnabled = 0;
            pdevInfo->usingQos = pdevInfo->localSta->qosdataEnabled;
        }
#endif

        /*
         * Cisco CKIP negociated.
         * Static WEP keys are treated here.
         * Dynamic keys will be treated when we have them.
         */

        if (pSib->pPriv &&
            ((lSib->ckipEnabled) || (lSib->ckipUseMic))) {
            WLAN_PRIV_RECORD *pKey = pSib->pPriv;
            WLAN_MACADDR nullMacAddr;
            A_UINT32 defIdx;

            /* Update connection key */
            pKey = pSib->pPriv;
            ASSERT(pKey);
            if (lSib->ckipEnabled) {
                pKey->keyType = PRIV_KEY_TYPE_CKIP;
            }

            if (lSib->ckipUseMic) {
                pKey->keyFlags |= PRIV_CKIP_MIC;
            } else {
                pKey->keyFlags &= ~PRIV_CKIP_MIC;
            }

            keyPrivateInit(pKey);
            A_MEM_ZERO(&nullMacAddr, sizeof(nullMacAddr));
            wlanInstallConnectionKey(pdevInfo,
                                     pSib,
                                     &nullMacAddr,
                                     pKey);

            setupSwEncryption(pdevInfo);

            /* Update the BSS key */
            defIdx = pdevInfo->staConfig.defaultKey;

            /* Asymetric WEP key not supported in this case */
            ASSERT(pKey == &pdevInfo->keyTable[defIdx]);

            keyPrivateInit(pKey);
            wlanInstallBssKey(pdevInfo,
                              defIdx,
                              1,            /* Default BSS key */ 
                              pKey);
        }

        /*
         *  TODO: Verify that shortpreamble bit is valid in
         *  AP's response according to spec.
         */
        lSib->capInfo.shortPreamble = pFrame->capInfo.shortPreamble;

        pdevInfo->useShortSlotTime  = pdevInfo->staConfig.shortSlotTime ?
                                      pFrame->capInfo.shortSlotTime : FALSE;

        bssAttribValue = (A_UINT32)pdevInfo->useShortSlotTime;
        wdcUpdateBssAttribute(pdevInfo->targetHandle, 0,
                              SHORT_SLOT_TIME_11g,
                              sizeof(bssAttribValue),
                              (TARGET_CONFIG_VAL)&bssAttribValue);

        lSib->assocId               = pFrame->assocId;

        /* Based on the Assoc ID, tell the hardware, which TIM bit to look for */
        wdcStaAssocBss(pdevInfo->targetHandle, 0, lSib->assocId);

        /* set up decompression Mask */
        sibDecompStateSet(pdevInfo, pSib, TRUE);

        /* Does capInfo represent BSS cap info? Should we keep a copy of it? */
        lSib->staState |= STATE_ASSOC;
        pSib->staState |= STATE_ASSOC;

        /* Copy the bssId, this will be used in reassociation */
        A_MACADDR_COPY(&pdevInfo->baseBss.bssId, &pdevInfo->lastApAddr);

        mlmePrintf("mlme[Re]AssocStaReceive: Successful Associate, assocId = %x\n", lSib->assocId);
#ifndef BUILD_AP
        osIndicateResourcesAvailable(pdevInfo);
#endif
    } else {
        mlmePrintf("mlme[Re]AssocStaReceive: ERROR: Associate failed, ");
        mlmePrintf("code = %x\n", pFrame->statusCode);
        if (pFrame->statusCode == REASON_NOT_AUTH) {
            wlanMlmeDeauthRequest(pdevInfo, &pdevInfo->baseBss,
                                  &pdevInfo->baseBss.bssId,
                                  REASON_UNSPECIFIED, TRUE);
        }
    }

    lSib->transState = TSTATE_QUIET;

    A_UNTIMEOUT(&pdevInfo->smeTimer);

    if (isReassoc) {
        return wlanMlmeReassocConfirm(pdevInfo, (MLME_RESULT)pFrame->statusCode);
    } else {
        return wlanMlmeAssocConfirm(pdevInfo, (MLME_RESULT)pFrame->statusCode);
    }
}


/*
 * wlanMlmeAssocConfirm - Confirm the result of assoc. request
 *
 * This routine calls back assocCallback() routine to deliver the result
 * of assoc request.
 *
 * RETURNS: MLME_SUCCESS, MLME_INVALID_PARAM, MLME_REFUSED,
 *          MLME_TIMEOUT, MLME_NOT_SUPPORTED
 */
MLME_RESULT
wlanMlmeAssocConfirm(WLAN_DEV_INFO *pdevInfo, MLME_RESULT resultCode)
{
    mlmePrintf("wlanMlmeAssocConfirm: Associate Confirm\n");

#ifndef BUILD_AP
    /*
     * We're done with scanning, but if we were interrupted mid-scan, the next
     * background scan that rolls around will not work properly.
     */
    if (pdevInfo->pscanInfo) {
        pdevInfo->pscanInfo->newScan = TRUE;
    }
#endif /* BUILD_AP */

    /* If callback routine exists, call it */
    if (pdevInfo->pSmeInfo->assocCallback) {
        (*pdevInfo->pSmeInfo->assocCallback)(pdevInfo, resultCode);
    }

    return resultCode == WLAN_SUCCESS ? MLME_SUCCESS : MLME_NOT_SUPPORTED;
}

/*
 * wlanMlmeReassocConfirm - Confirm the result of reassoc. request
 *
 * this routine calls back reassocCallback() routine to deliver the result
 * of assoc request.
 *
 * RETURNS: MLME_SUCCESS, MLME_INVALID_PARAM, MLME_REFUSED,
 *          MLME_TIMEOUT, MLME_NOT_SUPPORTED
 */
MLME_RESULT
wlanMlmeReassocConfirm(WLAN_DEV_INFO *pdevInfo, MLME_RESULT resultCode)
{
    mlmePrintf("wlanMlmeReassocConfirm: Successful Reassociate\n");

#ifndef BUILD_AP
    /*
     * We're done with scanning, but if we were interrupted mid-scan, the next
     * background scan that rolls around will not work properly.
     */
    pdevInfo->pscanInfo->newScan = TRUE;
#endif /* BUILD_AP */

    /* If callback routine exists, call it */
    if (pdevInfo->pSmeInfo->reassocCallback) {
        (*pdevInfo->pSmeInfo->reassocCallback)(pdevInfo, resultCode);
    }

    return MLME_SUCCESS;
}

/*
 * wlanMlmeAssocIndication - Inform upper layer of associtaion.
 *
 * Currently unimplemented
 */
MLME_RESULT
wlanMlmeAssocIndication(WLAN_DEV_INFO *pdevInfo, WLAN_MACADDR *pDestAddr)
{
    /* If callback routine exists, call it */
    if (pdevInfo->pSmeInfo->assocIndicationCallback) {
        (*pdevInfo->pSmeInfo->assocIndicationCallback)(pdevInfo, pDestAddr);
    }
    return MLME_SUCCESS;
}

/*
 * wlanMlmeReassocIndication - Inform upper layer of reassocitaion
 *
 */
MLME_RESULT
wlanMlmeReassocIndication(WLAN_DEV_INFO *pdevInfo,
                          WLAN_MACADDR  *pDestAddr)
{

    /* If callback routine exists, call it */
    if (pdevInfo->pSmeInfo->reassocIndicationCallback) {
        (*pdevInfo->pSmeInfo->reassocIndicationCallback)(pdevInfo, pDestAddr);
    }
    return MLME_SUCCESS;
}


MLME_RESULT
wlanMlmeDisassocRequest(WLAN_DEV_INFO *pdevInfo,
                        OP_BSS        *pOpBss,
                        WLAN_MACADDR  *pDestAddr,
                        A_UINT16      reasonCode,
                        A_BOOL        sendFrame)
{
    WLAN_FRAME_DISASSOC *pFrame;
    SIB_ENTRY           *pSib = NULL;
    SIB_ENTRY           *lSib = pdevInfo->localSta;
    A_UINT16            frameLen;
    ATHEROS_DESC        *pDesc;
    MLME_RESULT         retVal;
    A_UINT32            oldState = 0;

    /*
     * This can be done on either AP or STA. Don't assume it's
     * the local STA.
     */

    if (A_MACADDR_COMP(&broadcastMacAddr, pDestAddr) == 0) {
        pSib = NULL;
    } else {
        /* It's ok to not find the destination SIB. */
        pSib = sibEntryFind(pdevInfo, pDestAddr, pOpBss);
    }

    mlmePrintf("wlanMlmeDisassocRequest -- ");
    if (pSib) {
        mlmePrintMacAddress(pSib->macAddr);
        mlmePrintf(" in wlan%d\n", pdevInfo->devno);
    } else {
        mlmePrintMacAddress(*pDestAddr);
        mlmePrintf(" No SIB found in wlan%d\n", pdevInfo->devno);
    }

    /*
     * Since this is a notification, we must reflect the state change
     * immediately so that data transmit/receive will terminate
     * immediately.
     */
    if (lSib->serviceType == WLAN_STA_SERVICE) {

        if (pSib && pSib->pPriv) {
            sibDecompStateSet(pdevInfo, pSib, FALSE);
            if (pdevInfo->staConfig.wpaEnabled) {
                int defaultIndex = pdevInfo->staConfig.defaultKey;

                /* Delete key records */
                wlanDeleteConnectionKey(pdevInfo, pSib, pSib->pPriv);
                wlanDeleteBssKey(pdevInfo,
                                 defaultIndex,
                                 &pdevInfo->keyTable[defaultIndex]);

                wlanKeyTableSlotFree(pSib->pPriv);
            } else {
                /*
                 *  Demote the old AP to shared, so we will
                 *  not install a macAddr into the cache slot
                 *  (which is most likely < 4) as a result of
                 *  wlanKeyUpdate()
                 */
                if ((pSib->pPriv - &pdevInfo->keyTable[0]) < MAX_SHARED_KEYS) {
                    pSib->pPriv->keyFlags &= ~PRIV_UNIQUEKEY;
                }
            }
        }

        oldState = lSib->staState;

        /* Never refuse the disassociate request */
        lSib->staState &= ~STATE_ASSOC;
        lSib->transState = TSTATE_QUIET;

        /*
         * STA driver may call this function even when it knows that it's
         * not associated. This can happen when STA reboots but AP doesn't
         * know. AP still thinks STA is associated. So, we send this frame
         * without checking if STA is associated or not.
         */
        if (lSib->assocId != 0) {
            /* Physically disassociate */
            lSib->assocId = 0;
            wdcDetachBss(pdevInfo->targetHandle, 0);
            sibCseStateClean(pdevInfo, lSib);
            if (pSib) {
                sibCseStateClean(pdevInfo, pSib);
            }
        }
    } else if (lSib->serviceType == WLAN_AP_SERVICE) {
        /*
         * AP software may call this even when it knows that STA is not associated
         * in its association table. This can happen if AP reboots or has already
         * disassoced the STA while STA still thinks that it is associated.
         * Another case is when the L2 update frame is received on the DSM.
         */

        /* If actual association exists on AP, release it. */
        if (pSib) {
            oldState = pSib->staState;

            /* JK - inform the target to reset the swretry queue */
            pSib->staState &= ~STATE_ASSOC;
            if (pSib->assocId != 0) {
                sibPowerStateClean(pdevInfo, pSib);
                /* Currently assigned ID */
                ASSERT(pSib->pOpBss == pOpBss);
                mlmeAssocIdRelease(pdevInfo, pOpBss, pSib->assocId);
                pSib->assocId = 0;
            }
            sibCseStateClean(pdevInfo, pSib);
            sibDecompStateSet(pdevInfo, pSib, FALSE);
            pSib->stats.DisAssociations++;
            pSib->stats.DisAssocReasonCode = reasonCode;
            pdevInfo->localSta->stats.DisAssociations++;
        }
    }

    if (sendFrame) {
        frameLen = sizeof(WLAN_FRAME_DISASSOC);

        if ((createBuffandDesc(pdevInfo, frameLen, &pDesc)) != 0) {
            /* Failed to allocate buffer for frame and descriptor */
            return MLME_DRIVER_ERROR;
        }

        pFrame = pDesc->pBufferVirtPtr.disAssoc;
        mgtFrameHdrInit(&pFrame->macHeader, SUBT_DISASSOC, pDestAddr);
        A_MACADDR_COPY(&pOpBss->bssId, &pFrame->macHeader.bssId);
        A_MACADDR_COPY(
            (lSib->serviceType == WLAN_AP_SERVICE) ? &pOpBss->bssId :
            &lSib->macAddr,
            &pFrame->macHeader.srcAddr);

        pFrame->reasonCode   = reasonCode;  /* reason for disassoc */

        if (mlmeDebugLevel > 10) {
            displayMgtFrame(&pFrame->macHeader, frameLen);
        }

        /* return status doesn't matter */
        pDesc->pDestSibEntry = pSib;
        pDesc->pOpBss        = pOpBss;
        SETQHANDLE(pDesc, TXQ_ID_FOR_AC1);
        pDesc->bIndicateXmitStatus = TRUE;
        powerFrameTransmit(pdevInfo, pDesc);
        retVal = MLME_SUCCESS;

    } else {
        retVal = pSib ? MLME_SUCCESS : MLME_INVALID_PARAM;
    }

    if (oldState & STATE_ASSOC) {
        wlanMlmeDisassocConfirm(pdevInfo, pDestAddr, reasonCode);
    }

    return retVal;
}


/*
 * wlanMlmeDisassocConfirm - Confirm the disassoc operation.
 *
 * This routine calls back disassocCallback() to confirm the operation.
 *
 * RETURNS: MLME_SUCCESS, MLME_INVALID_PARAM, MLME_REFUSED,
 *          MLME_TIMEOUT, MLME_NOT_SUPPORTED
 */
MLME_RESULT
wlanMlmeDisassocConfirm(WLAN_DEV_INFO *pdevInfo,
                        WLAN_MACADDR  *pmacAddr,
                        MLME_RESULT   resultCode)
{
    /* If callback routine exists, call it */
    if (pdevInfo->pSmeInfo->disassocCallback) {
        (*pdevInfo->pSmeInfo->disassocCallback)(pdevInfo, pmacAddr, resultCode);
    }
    return MLME_SUCCESS;
}

/*
 * wlanMlmeDisassocIndication - Inform upper layer of disassociation
 *
 * Currently unimplemented
 */
MLME_RESULT
wlanMlmeDisassocIndication(WLAN_DEV_INFO *pdevInfo,
                           WLAN_MACADDR  *pDestAddr,
                           MLME_RESULT   resultCode)
{
    if (pdevInfo->pSmeInfo->disassocIndicationCallback) {
        (*pdevInfo->pSmeInfo->disassocIndicationCallback)(pdevInfo,
                                                          pDestAddr, resultCode);
    }
    return MLME_SUCCESS;
}

MLME_RESULT
mlmeDisassocReceive(WLAN_DEV_INFO       *pdevInfo,
                    OP_BSS              *pOpBss,
                    WLAN_FRAME_DISASSOC *disassocFrame,
                    A_UINT16            frameLen)
{
    SIB_ENTRY    *pSib;
    SIB_ENTRY    *lSib = pdevInfo->localSta;

#ifndef BUILD_AP
    /* Ensure that it comes from desired AP; accept disassoc from our bssid only */
    if ((A_MACADDR_COMP(&(pOpBss->bssId),
                        &(disassocFrame->macHeader.srcAddr))) != 0)
    {
        /* Got this frame from wrong AP !! */
        return MLME_REFUSED;
    }
#endif

    pSib = sibEntryFind(pdevInfo, &disassocFrame->macHeader.srcAddr, pOpBss);
    /*
     * pSib can be NULL if AP rebooted and doesn't know anything about
     * the sending STA
     */

    if (lSib->serviceType == WLAN_STA_SERVICE) {

        lSib->staState  &= ~STATE_ASSOC;
        lSib->transState = TSTATE_QUIET;

        /*
         * STA driver may call this function even when it knows that it's
         * not associated. This can happen when STA reboots but AP doesn't
         * know. AP still thinks STA is associated. So, we send this frame
         * without checking if STA is associated or not. It could also be
         * a broadcast disassoc.
         */
        if (lSib->assocId != 0) {
            /* Physically disassociate */
            lSib->assocId = 0;
            wdcDetachBss(pdevInfo->targetHandle, 0);
        }
        sibCseStateClean(pdevInfo, lSib);
        sibDecompStateSet(pdevInfo, pSib, FALSE);
    }

    if (lSib->serviceType == WLAN_AP_SERVICE && pSib != NULL &&
        pSib->assocId != 0) {
        /* JK - inform the target to reset the swretry queue */
        sibPowerStateClean(pdevInfo, pSib);
        /* Release the association ID for pSib. */
        ASSERT(pSib->pOpBss == pOpBss);
        mlmeAssocIdRelease(pdevInfo, pOpBss, pSib->assocId);
        pSib->staState  &= ~STATE_ASSOC;
        pSib->assocId = 0;
        sibCseStateClean(pdevInfo, pSib);
        sibDecompStateSet(pdevInfo, pSib, FALSE);
    }

    if (pSib != NULL) {
        pSib->staState  &= ~STATE_ASSOC;
        pSib->transState = TSTATE_QUIET;
        pSib->stats.DisAssociations++;
        pSib->stats.DisAssocReasonCode = disassocFrame->reasonCode;
    }

    wlanMlmeDisassocIndication(pdevInfo, &disassocFrame->macHeader.srcAddr,
                               disassocFrame->reasonCode);

    return MLME_SUCCESS;
}

/*************************/
/* MLME support routines */
/*************************/

/*
 * basicRateSubsetCheck - Check if bss basic rates are part of sta rates
 *
 * Spec section 7.3.2.2
 * STA can be associated only if the all bss basic rates are
 * supported by the sta. The basic rates in the WLAN_RATE_SET element
 * are represented by msb being 1. This routine takes basic rates
 * (with msb = 1) in psub and compares its lower 7 bits with lower 7
 * bits of each rate value in the rate element psuper.
 */
A_BOOL
basicRateSubsetCheck(WLAN_RATE_SET *psub,    /* expected subset */
                     WLAN_RATE_SET *psuper)  /* set expected to be superset */
{
    int    i, j;
    A_BOOL match;

    if (psub == NULL || psuper == NULL) {
        mlmePrintf("basicRateSubsetCheck: Error: NULL basic rate element\n");
        return FALSE;
    }

    for (i = 0; i < psub->length; i++) {
        if ((psub->rates[i] & 0x80) == 0) {
            /* MSB is not 1. It is not a basic rate, skip it */
            continue;
        }

        match = FALSE;

        for (j = 0; j < psuper->length; j++) {
            if ((psub->rates[i] & 0x7f) == (psuper->rates[j] & 0x7f)) {
                /* lower 7 bits match. */
                match = TRUE;
                break;
            }
        }

        if (!match) {
            return match;
        }
    }

    /* It didn't fail any match. All basic rates in psub matched. */
    return TRUE;
}

/*
 * xSectionRateSet - Builds an intersection of two rate sets
 * into the third one. Used by AP and STA for building a working
 * rate set. The first parameter should contain the rateset with
 * the basic rate flag.
 */
void
xSectionRateSet(WLAN_RATE_SET *pSrc1,        /* set 1 */
                WLAN_RATE_SET *pSrc2,        /* set 2 */
                WLAN_RATE_SET *pDest)        /* cross section set */
{
    int     i, j;
    A_UINT8 k;

    k = 0;
    for (i = 0; i < pSrc1->length; i++) {
        for (j = 0; j < pSrc2->length; j++) {
            if ((pSrc1->rates[i] & 0x7f) == (pSrc2->rates[j] & 0x7f)) {

                /* lower 7 bits match. Add it to dest */
                /* The basis rate flag is set based on the pSrc1 */
                pDest->rates[k] = pSrc1->rates[i];

                k++;
                break;
            }
        }
    }
    pDest->length    = k;
}

/*
 clearBasicRateFlag - clear the basic rate flag in the rate set
 */

static void
clearBasicRateFlag(WLAN_RATE_SET *pWlanRateSet)
{
    int     i;

    for (i = 0; i < pWlanRateSet->length; i++) {
        pWlanRateSet->rates[i] = pWlanRateSet->rates[i] & 0x7f;
    }
}


/*
 * phyRateSubSetCheck - returns TRUE if there exists at least one rate of the
 * requested check phy type, else returns FALSE.
 *
 */
A_BOOL
phyRateSubSetCheck(WLAN_DEV_INFO *pDev, WIRELESS_MODE mode,
                   WLAN_RATE_SET *pRateSet, WLAN_PHY checkType)
{
#if NDIS_WDM
    /*
     * (LW) Attention !!!
     * This is temp hack, should revisit later
     */
    RATE_TABLE  *pRateTable = pDev->hwRateTable[mode];
#else
    RATE_TABLE  *pRateTable = HACK_TO_PARDEV(pDev)->hwRateTable[mode];
#endif
    WLAN_RATE_SET phyRateSet, destRateSet;
    A_UINT8     i, j = 0;

    if (pRateTable == NULL || pRateSet == NULL) {
        return FALSE;
    }

    /* construct a rate set of the requested check phy type */
    for (i = 0; i < pRateTable->rateCount; i++) {
        if (pRateTable->info[i].phy != checkType) {
            /* It is not requested phy type rate, skip it */
            continue;
        }

        phyRateSet.rates[j++] = pRateTable->info[i].dot11Rate;
    }

    phyRateSet.length = j;

    xSectionRateSet(pRateSet, &phyRateSet, &destRateSet);

    if (destRateSet.length !=0 ) {
        return TRUE;
    }

    return FALSE;
}

/*
 * staWlanModeSet - set STA's wireless mode
 */
void
staWlanModeSet(WLAN_DEV_INFO *pdevInfo, SIB_ENTRY *pSib)
{
    WLAN_CFLAGS cf = pdevInfo->staConfig.pChannel->channelFlags;

    if (IS_CHAN_5GHZ(cf)) {
        pSib->wlanMode = STA_MODE_A;             /* A client */
    } else {
        pSib->wlanMode = STA_MODE_B;             /* assume B client */

        /* Special handling for G mode */
        if (IS_CHAN_G(cf) &&
            phyRateSubSetCheck(pdevInfo, WIRELESS_MODE_11g,
                               &pSib->workingRateSet, WLAN_PHY_OFDM) == TRUE)
        {
            pSib->wlanMode = STA_MODE_G;     /* G client */
        }
        if (IS_CHAN_108G(cf) &&
            phyRateSubSetCheck(pdevInfo, WIRELESS_MODE_108g,
                               &pSib->workingRateSet, WLAN_PHY_TURBO) == TRUE)
        {
            pSib->wlanMode = STA_MODE_108G;   /* 11g turbo client */
        }
    }
}


/*
 * wlanElementToFullRateSet - build a combined rate set from each
 * Information Element rate set.
 */
void
wlanElementToFullRateSet(RATE_SET *pRateSet,
                         EXT_RATE_SET *pExtRateSet,
                         WLAN_RATE_SET *pShadowRateSet)
{
    int i = 0, j = 0;

    ASSERT(pShadowRateSet);

    if (pRateSet != NULL) {
        for (i = 0, j = 0; i < pRateSet->length; i++, j++) {
            pShadowRateSet->rates[j] = pRateSet->rates[i];
        }
    }

    if (pExtRateSet != NULL) {
        for (i = 0; i < pExtRateSet->length; i++, j++) {
            if (j == MAX_WLAN_RATE_SIZE) {
                break;
            }
            pShadowRateSet->rates[j] = pExtRateSet->rates[i];
        }
    }

    pShadowRateSet->length = (A_UINT8)j;
}

/*
 * wlanFullToRateSet - build the Supported Rate Set
 * element from the combined rate set.
 */
void
wlanFullToRateSet(WLAN_RATE_SET *pShadowRateSet, RATE_SET *pRateSet,
                  A_BOOL gdraft5)
{
    int i, n;

    n = ELE_RATE_SIZE_11B + (gdraft5 ? 4 : 0);
    pRateSet->elementID = ELE_SUPPORTED_RATES;
    pRateSet->length    = (pShadowRateSet->length > n) ? n :
        pShadowRateSet->length;
    for (i = 0; i < pRateSet->length; i++) {
        pRateSet->rates[i] = pShadowRateSet->rates[i];
    }
}

/*
 * wlanFullToExtRateSet - build the Extended Supported Rate Set
 * element from the combined rate set.  Assumes the Supported Rate Set
 * is fully populated before overflowing into the Extended Supported Rate Set.
 */
void
wlanFullToExtRateSet(WLAN_RATE_SET *pShadowRateSet, EXT_RATE_SET *pExtRateSet,
                     A_BOOL gdraft5)
{
    int i;

    if (gdraft5) {
        pExtRateSet->length = 0;
        return;
    }

    pExtRateSet->elementID = ELE_EXT_SUPPORTED_RATES;
    if (pShadowRateSet->length > ELE_RATE_SIZE_11B) {
        pExtRateSet->length = pShadowRateSet->length - ELE_RATE_SIZE_11B;
        for (i = 0; i < pExtRateSet->length; i++) {
            pExtRateSet->rates[i] = pShadowRateSet->rates[i + ELE_RATE_SIZE_11B];
        }
    } else {
        pExtRateSet->length = 0;
    }
}

void
wlanRateSetCopy(WLAN_RATE_SET *pSrc, WLAN_RATE_SET *pDest)
{
    int isize;

    ASSERT(pSrc != NULL);
    ASSERT(pDest != NULL);

    isize = pSrc->length + sizeof(pSrc->length);

    ASSERT(isize < sizeof(WLAN_RATE_SET));
    A_DRIVER_BCOPY(pSrc, pDest, isize);
}

void
wpaElementsToWpaIe(UCSE_SET *pUcseSet, MCSE_SET *pMcseSet, AUTHSE_SET *pAuthSet,
                   A_UINT16 wpaCap, WPA_IE *pWpaIe)
{
    A_UINT8 length, *pData;
    int     i;

    pWpaIe->elementID = ELE_VENDOR_PRIVATE;
    /* length calc is deferred */

    /* Insert OUI in endian-agnostic way */
    A_BCOPY(ouiMicrosoft, pWpaIe->oui, sizeof(pWpaIe->oui));

    pWpaIe->ouiType = WPA_OUI_TYPE;
    pWpaIe->version = WPAIE_VERSION;

    pData = (A_UINT8 *)&pWpaIe->version;

    if (VALID_MCSE_ELEMENT(pMcseSet)) {
        pWpaIe->mCipher[0] = pMcseSet->mCipher[0];
        pData = (A_UINT8 *)&pWpaIe->mCipher[0];

        if (VALID_UCSE_ELEMENT(pUcseSet)) {
            pWpaIe->uCipherCount = pUcseSet->uCipherCount;
            for (i = 0; i < pUcseSet->uCipherCount; i++) {
                pWpaIe->uCiphers[i] = pUcseSet->uCiphers[i];
            }

            pData = (A_UINT8 *)&pWpaIe->uCiphers[i];

            if (VALID_AUTHSE_ELEMENT(pAuthSet)) {
                A_UINT16 len = pAuthSet->authSelectorCount;

                A_BCOPY((char *)&len, pData, sizeof(A_UINT16));
                pData += sizeof(A_UINT16);

                for (i = 0; i < pAuthSet->authSelectorCount; i++) {
                    A_UINT32 sel = pAuthSet->authSelectors[i];
                    A_BCOPY(&sel, pData, sizeof(A_UINT32));
                    pData += sizeof(A_UINT32);
                }

                /*
                 * WPA capabilities
                 * Don't include the capabilities if they are zero.
                 */
                if (wpaCap) {
                    A_BCOPY((char *)&wpaCap, pData, sizeof(wpaCap));
                    pData += sizeof(A_UINT16);
                }
            } else {
                /* insert default auth element */
            }
        } else {
            /* insert default ucse element */
        }
    } else {
        /* insert default mcse element */
    }
    length = pData - (A_UINT8 *)pWpaIe;
    pWpaIe->length = length - 2;
}

/*
 *  Take a WPA IE and decompose it into Atheros-private CSN elements
 */

A_STATUS
wpaIeToWpaElements(WPA_IE *pWpaIe, UCSE_SET *pUcseSet, MCSE_SET *pMcseSet,
                   AUTHSE_SET *pAuthSet, A_UINT16 *pWpaCap)
{
    int      i = 0;
    A_UINT8  *pData = NULL;
    A_UINT32 uCipherLen = 0, AKMLen = 0;
    const A_UINT32 mCipherOffsetEnd = 10;
    const A_UINT32 uCipherOffsetEnd = 12;
    const A_UINT32 AKMOffsetEnd     = 14;
    const A_UINT32 capOffsetEnd     = 16;

    /* make invalid first, even if WPA IE not valie */
    A_MEM_ZERO(pUcseSet, sizeof(UCSE_SET));
    A_MEM_ZERO(pMcseSet, sizeof(MCSE_SET));
    A_MEM_ZERO(pAuthSet, sizeof(AUTHSE_SET));
    *pWpaCap = 0;

    if (!VALID_WPA_ELEMENT(pWpaIe)) {
        return A_ERROR;
    }

    if (pWpaIe->length >= mCipherOffsetEnd) {
        /* set up multicast cipher */
        pMcseSet->mCipher[0] = pWpaIe->mCipher[0];
    }

    if (pWpaIe->length >= uCipherOffsetEnd) {
        ASSERT(pWpaIe->uCipherCount > 0);
        /* set up unicast ciphers */
        pUcseSet->uCipherCount = pWpaIe->uCipherCount;
        uCipherLen = pUcseSet->uCipherCount * sizeof(A_UINT32);

        if (uCipherLen > ELE_UCSE_SIZE ||
            pWpaIe->length < (uCipherOffsetEnd + uCipherLen))
        {
            A_MEM_ZERO(pUcseSet, sizeof(UCSE_SET));
            A_MEM_ZERO(pMcseSet, sizeof(MCSE_SET));
            A_MEM_ZERO(pAuthSet, sizeof(AUTHSE_SET));
            *pWpaCap = 0;

            return A_ERROR;
        }

        for (i = 0; i < pWpaIe->uCipherCount; i++) {
            pUcseSet->uCiphers[i] = pWpaIe->uCiphers[i];
        }
    }

    if (pWpaIe->length >= (AKMOffsetEnd + uCipherLen)) {
        pData      = (A_UINT8 *)&pWpaIe->uCiphers[i];
        A_BCOPY(pData, &pAuthSet->authSelectorCount, sizeof(A_UINT16));
        pData     += sizeof(A_UINT16);
        AKMLen     = pAuthSet->authSelectorCount * sizeof(A_UINT32);

        if (AKMLen > ELE_AUTHSE_SIZE ||
            pWpaIe->length < (uCipherOffsetEnd + uCipherLen + AKMLen))
        {
            A_MEM_ZERO(pUcseSet, sizeof(UCSE_SET));
            A_MEM_ZERO(pMcseSet, sizeof(MCSE_SET));
            A_MEM_ZERO(pAuthSet, sizeof(AUTHSE_SET));
            *pWpaCap = 0;

            return A_ERROR;
        }

        /* set up the authentication elements */
        for (i = 0; i < pAuthSet->authSelectorCount; i++) {
            A_BCOPY(pData, &pAuthSet->authSelectors[i],
                    sizeof(A_UINT32));
            pData += sizeof(A_UINT32);
        }
    }

    if (pWpaIe->length >= (capOffsetEnd + uCipherLen + AKMLen)) {
        /* Now store the WPA IE capabilities info */
        A_BCOPY(pData,(char *)pWpaCap, sizeof(A_UINT16));
    }

    return A_OK;
}

A_BOOL
mlmeBeaconTimCheck(TIM_ELEMENT *pTim, A_UINT16 assocId, A_BOOL *pbrmc)
{
    A_UINT8 mask;
    A_UINT8 index, indexn1, indexn2;

    assocId &= 0x3fff; // Clear top 2 bits

    ASSERT(pTim);
    ASSERT(assocId >= ASSOCID_MIN && assocId <= ASSOCID_MAX);

    if (pbrmc != NULL) {
        *pbrmc = FALSE;
    }

    index = assocId / 8;
    mask  = 1 << (assocId & 7);

    /* See if our assocId is outside the bitmap range */
    indexn1 = pTim->bitmapControl & 0xfe; /* Clear lsb */
    indexn2 = pTim->length + indexn1 - 4;

    if (index < indexn1 || index > indexn2) {
        /* Index out of range */
        return FALSE;
    }

    index -= indexn1;

    if (pbrmc && (pTim->bitmapControl & 0x01)) {
        *pbrmc = TRUE;
    }

    return (A_BOOL)(pTim->bitmap[index] & mask);
}

INFO_ELEMENT *
mlmeElementCopy(INFO_ELEMENT *pSrc, INFO_ELEMENT *pDest)
{
    int isize;

    if (pSrc == NULL || pDest == NULL) {
        return NULL;
    }

    isize = pSrc->length + sizeof(pSrc->elementID) + sizeof(pSrc->length);

    A_DRIVER_BCOPY(pSrc, pDest, isize);
    return (INFO_ELEMENT *)((A_UINT8 *)pDest + isize);
}

LOCAL void
smePendDescProcess (WLAN_DEV_INFO *pdevInfo, SIB_ENTRY *pSib)
{
    ATHEROS_DESC *pTmp;
    ATHEROS_DESC *pNext;
    SIB_ENTRY    *opSib;

    opSib = (pdevInfo->localSta->serviceType == WLAN_STA_SERVICE) ?
            pdevInfo->localSta : pSib;

    if (!opSib) {
        return;
    }

    if (opSib->smePendDesc != NULL &&
        opSib->smePendDesc != (ATHEROS_DESC *)0xffffffff)
    {
        /* We have some mgt frames pending for this SIB. process them. */
        pTmp = opSib->smePendDesc;
        opSib->smePendDesc = NULL;
        while (pTmp != NULL) {
            mlmePrintf("smePendDescProcess: Cleaning up pending rx %p\n", (void *)pTmp);
            pNext = pTmp->pNextVirtPtr;

            /*
             * mlmeMgtRxCompletion expects frame in little endian
             * format - this is definitely non-optimal but can be
             * optimized going forward by reorganizing the code a
             * bit
             */
            mgtFrameEndianChange(pTmp->pBufferVirtPtr.mgtHeader,
                    pTmp->status.rx.dataLength, CPU2LE);
            mlmeMgtRxCompletion(pdevInfo, pTmp);

            pTmp = pNext;
        }
    }

    opSib->smePendDesc = NULL; // if it was 0xffffffff, change to 0
}

/*
 * mlmeMgtTxCompletion - Process completion of mgt transmit descriptor
 *
 * This routine updates the station state in SIB based on success or
 * failure of mgt frame that was scheduled earlier. Certain MLME primitives
 * are not called completed until that frame is transmitted and ack received.
 *
 * INPUT PARAMETERS
 *    pDesc - Pointer to descriptor with valid tx status field.
 */
void
mlmeMgtTxCompletion(WLAN_DEV_INFO *pdevInfo, ATHEROS_DESC *pDesc)
{
    WLAN_MGT_MAC_HEADER *pFrame;
    SIB_ENTRY           *pSib;
    A_BOOL               result;
    CONN_RATE_SET        connRateSet;

    pFrame = pDesc->pBufferVirtPtr.mgtHeader;
    mgtFrameEndianChange(pFrame, pDesc->bufferLength, LE2CPU);

    /* sw tx status isn't cleaned up always - so make sure its done also */
    result = (pDesc->pTxLastDesc->status.tx.status == TRANSMIT_OK);

    /*
     * dereference the sib off the desc and make sure the
     * sib is still valid - our sme/sib state mc is known
     * to have races; the validity check and the assert
     * are there because the earlier sibLookup in the code
     * is being replaced by the dereference off the desc
     */
    pSib = pDesc->pDestSibEntry;
    ASSERT(!pSib ||
           ( (pSib->used & SIB_VALID) &&
             (A_MACADDR_COMP(&pFrame->destAddr, &pSib->macAddr) == 0)) );

    mlmeL2Printf("mlmeMgtTxCompletion result: %s. %s, dst ",
                 result ? "ok" : "error",
                 subtypeToString(pFrame->frameControl.fSubtype));

    mlmeL2PrintMacAddress(pFrame->destAddr);
    mlmeL2Printf(" in wlan%d\n", pdevInfo->devno);
    switch (pFrame->frameControl.fSubtype) {
    case SUBT_AUTH: {
        WLAN_FRAME_AUTH *p1 = (WLAN_FRAME_AUTH *)pFrame;
        A_UINT16        algNo;
        A_UINT16        transSeqNo;
        A_UINT16        statusCode;

        if (pFrame->frameControl.wep) {
            algNo      = ((WLAN_FRAME_AUTH_ENCRYPT *)pFrame)->algNo;
            transSeqNo = ((WLAN_FRAME_AUTH_ENCRYPT *)pFrame)->transSeqNo;
            statusCode = ((WLAN_FRAME_AUTH_ENCRYPT *)pFrame)->statusCode;
        } else {
            algNo      = ((WLAN_FRAME_AUTH *)pFrame)->algNo;
            transSeqNo = ((WLAN_FRAME_AUTH *)pFrame)->transSeqNo;
            statusCode = ((WLAN_FRAME_AUTH *)pFrame)->statusCode;
        }

        if (!result && (transSeqNo == 1 || transSeqNo == 3)) {
            /* We are a STA. Can't send out frames. Abort */
            pdevInfo->localSta->transState = TSTATE_QUIET;
            A_UNTIMEOUT(&pdevInfo->smeTimer);
            wlanMlmeAuthConfirm(pdevInfo, &pFrame->destAddr,
                                (A_UINT16) pdevInfo->localSta->authenAlgo,
                                MLME_TIMEOUT);
            pdevInfo->localSta->stats.txAuthenticateFailStatus = statusCode;
            A_MACADDR_COPY(&pFrame->destAddr,
                           &pdevInfo->localSta->stats.txAuthenticateFailSta);
        } else if ((transSeqNo == 2 && algNo == AUTH_OPEN_SYSTEM) ||
                   (transSeqNo == 4 && algNo == AUTH_SHARED_KEY))
        {
            /* We are an AP */
            if (pSib->transState != TSTATE_AUTH_FRAME2_SENT &&
                pSib->transState != TSTATE_AUTH_FRAME4_SENT)
            {
                /* Station was deauthed meanwhile */
                result = FALSE;
            }
            if (!result || (statusCode != WLAN_SUCCESS)) {
                pSib->transState = TSTATE_QUIET;
                pSib->staState  &= ~(STATE_AUTH);

                pdevInfo->localSta->stats.txAuthenticateFailStatus = statusCode;
                A_MACADDR_COPY(&pFrame->destAddr,
                               &pdevInfo->localSta->stats.txAuthenticateFailSta);
            } else {
                /* Final successful frame has been transmitted ok. */
                pSib->staState  |= STATE_AUTH;
                pSib->transState = TSTATE_QUIET;
                pSib->authenAlgo = p1->algNo;
                pdevInfo->localSta->authenAlgo = p1->algNo;
                wlanMlmeAuthIndication(pdevInfo, &pFrame->destAddr, algNo);
            }
        }

        // do nothing for successful tx of frame 2 in AUTH_SHARED_KEY
        break;
    }

    case SUBT_ASSOC_REQ:
        if (!result) {
            /* failed to generate first frame on sta.  Abort */
            A_UNTIMEOUT(&pdevInfo->smeTimer);
            pdevInfo->localSta->transState = TSTATE_QUIET;
            wlanMlmeAssocConfirm(pdevInfo, MLME_TIMEOUT);
        }
        break;

    case SUBT_REASSOC_REQ:
        if (!result) {
            /* failed to generate first frame on sta. Abort */
            A_UNTIMEOUT(&pdevInfo->smeTimer);
            pdevInfo->localSta->transState = TSTATE_QUIET;
            wlanMlmeReassocConfirm(pdevInfo, MLME_TIMEOUT);
        }
        break;

    case SUBT_ASSOC_RESP: {
        WLAN_FRAME_ASSOC_RESP *pResp = (WLAN_FRAME_ASSOC_RESP *)pFrame;

        /* Make sure sta hasn't lost authentication meanwhile */
        if ((pSib->transState != TSTATE_ASSOC_RESP_SENT &&
             pSib->transState != TSTATE_REASSOC_RESP_SENT) ||
            !(pSib->staState & STATE_AUTH)) {
            result = FALSE;
        }
        pSib->transState = TSTATE_QUIET;
        if (result && pResp->statusCode == WLAN_SUCCESS) {
            /* Successful transmission of response on AP */
            pSib->staState |= STATE_ASSOC;

            /* recapture the assoc Id in case of the race condition */
            pSib->assocId = pResp->assocId;
            ASSERT(pSib->assocId);

            /* Update target connection */
            wlanRateSet2WdcRateSet(&pSib->workingRateSet, &connRateSet);
            wdcUpdateConnectionAttribute(pdevInfo->targetHandle,
                                         pSib->wdcConnId,
                                         CONN_WORKING_RATE_SET,
                                         sizeof(connRateSet),
                                         &connRateSet);
            wlanMlmeAssocIndication(pdevInfo, &pFrame->destAddr);
        } else {
            /* Call disassocrequest for cleanup */
            wlanMlmeDisassocRequest(pdevInfo, pDesc->pOpBss, &pSib->macAddr,
                REASON_AUTH_EXPIRED, FALSE);
        }
        break;
    }

    case SUBT_REASSOC_RESP: {
        WLAN_FRAME_REASSOC_RESP *pResp = (WLAN_FRAME_REASSOC_RESP *)pFrame;

        pSib->transState = TSTATE_QUIET;
        if (result && pResp->statusCode == WLAN_SUCCESS) {
            /* Successful transmission of response on AP */
            pSib->staState |= STATE_ASSOC;

            /* recapture the assoc Id in case of the race condition */
            pSib->assocId = pResp->assocId;
            ASSERT(pSib->assocId);
            /* Update target connection */
            wlanRateSet2WdcRateSet(&pSib->workingRateSet, &connRateSet);
            wdcUpdateConnectionAttribute(pdevInfo->targetHandle,
                                         pSib->wdcConnId,
                                         CONN_WORKING_RATE_SET,
                                         sizeof(connRateSet),
                                         &connRateSet);
            wlanMlmeReassocIndication(pdevInfo, &pFrame->destAddr);
        } else {
            pSib->staState      &= ~STATE_ASSOC;
            pSib->listenInterval = 0;
            /* Release the AID since we don't need it. */
            if (pSib->assocId) {
                sibPowerStateClean(pdevInfo, pSib);
                ASSERT(pSib->pOpBss == pDesc->pOpBss);
                mlmeAssocIdRelease(pdevInfo, pSib->pOpBss, pSib->assocId);
            }
            sibCseStateClean(pdevInfo, pSib);
            pSib->assocId = 0;
        }
        break;
    }

    case SUBT_PROBE_REQ:
        if (pSib) {
            pSib->stats.TxProbeRequests++;
        }
        break;

    case SUBT_PROBE_RESP:
    case SUBT_DISASSOC:
    case SUBT_DEAUTH:
    default:
        /* Ignore the success or failure of transmit. */
        break;
    }

    smePendDescProcess(pdevInfo, pSib);

    return;
}

/*
 * mlmeMgtRxCompletion - Routine to process incoming mgt frame
 *
 * It calls proper mlme routine based on incoming frame type and state of the station
 */
void
mlmeMgtRxCompletion(WLAN_DEV_INFO *pdevInfo, ATHEROS_DESC *pDesc)
{
    WLAN_MGT_MAC_HEADER *pMacHdr = pDesc->pBufferVirtPtr.mgtHeader;
    WLAN_MACADDR        *pDest;
    A_UINT16             ret;
    A_UINT16             frameLen;

    pDest = pdevInfo->localSta->serviceType == WLAN_AP_SERVICE ?
            &pDesc->pOpBss->bssId : &pdevInfo->localSta->macAddr;

    if (A_MACADDR_COMP(pDest, &pMacHdr->destAddr) &&
        A_MACADDR_COMP(&broadcastMacAddr, &pMacHdr->destAddr))
    {
        // mgt frame not directed to us or broadcast
        freeBuffandDesc(pdevInfo, pDesc);
        return;
    }

    /* The "wep" bit can only be on for auth frames. */
    if (pMacHdr->frameControl.wep &&
        (pMacHdr->frameControl.fSubtype != SUBT_AUTH))
    {
        freeBuffandDesc(pdevInfo, pDesc);
        return;
    }

    frameLen = pDesc->status.rx.dataLength;
    mgtFrameEndianChange(pMacHdr, frameLen, LE2CPU);

    mlmeL2Printf("mlmeMgtRxCompletion: %s, ",
                 subtypeToString(pMacHdr->frameControl.fSubtype));
    mlmeL2PrintMacAddress(pMacHdr->srcAddr);
    mlmeL2Printf(" in wlan%d\n", pdevInfo->devno);
    if (mlmeDebugLevel > 10) {
        displayMgtFrame(pDesc->pBufferVirtPtr.mgtHeader, frameLen);
    }

    switch (pMacHdr->frameControl.fSubtype) {
    /*
     * In this switch, we just take care of non auth/assoc frames, beacons
     * first because they are most frequent.
     */

    case SUBT_BEACON:
        mlmeBeaconReceive(pdevInfo, pDesc);
        freeBuffandDesc(pdevInfo, pDesc);
        return;

    case SUBT_PROBE_REQ:
        mlmeProbeRequestReceive(pdevInfo, pDesc->pOpBss,
                                (WLAN_FRAME_PROBE_REQ *)pMacHdr, frameLen);
        freeBuffandDesc(pdevInfo, pDesc);
        return;

    case SUBT_PROBE_RESP:
        if (pDesc->pSrcSibEntry) {
            pDesc->pSrcSibEntry->stats.RxProbeResponses++;
        }
        mlmeProbeRespReceive(pdevInfo, pDesc);
        freeBuffandDesc(pdevInfo, pDesc);
        return;

    default:
        break;
    }

    /*
     * Now see if we can process this RX descriptor.
     *
     * if smePendDesc is NULL, there is no tx descr pending completion
     * if smePendDesc is 0xffffffff, some tx descr is pending completion, but no rx yet.
     * if smePendDesc is a valid pointer, we already have received frames and they are waiting
     * for tx frame to finish.
     */

    if (pdevInfo->localSta->smePendDesc != NULL) {
        pDesc->pNextVirtPtr = NULL;
        if (pdevInfo->localSta->smePendDesc == (ATHEROS_DESC *) 0xffffffff) {
            /*
             * Some TX pending, but no RX pending as of yet. pDesc becomes
             * the first descriptor in the list.
             */
            pdevInfo->localSta->smePendDesc = pDesc;
            mlmePrintf("mgtRxCompletion: first pending rx descriptor %p\n", (void *)pDesc);
        } else {
            ATHEROS_DESC *pTmp = pdevInfo->localSta->smePendDesc;

            mlmePrintf("mgtRxCompletion: additional pending rx descriptor %p\n", (void *)pDesc);
            while (pTmp->pNextVirtPtr != NULL) {
                pTmp = pTmp->pNextVirtPtr;
            }
            pTmp->pNextVirtPtr = pDesc;
        }

        /* Return from here. Don't delete pDesc. */
        return;
    }

    mlmePrintf("mlmeMgtRxCompletion: %s, ",
                 subtypeToString(pMacHdr->frameControl.fSubtype));
    mlmePrintMacAddress(pMacHdr->srcAddr);
    mlmePrintf(" in wlan%d\n", pdevInfo->devno);
    if (mlmeDebugLevel > 1) {
        displayMgtFrame(pDesc->pBufferVirtPtr.mgtHeader, frameLen);
    }

    switch (pMacHdr->frameControl.fSubtype) {
    case SUBT_ASSOC_REQ:
        pdevInfo->localSta->stats.Associations++;
        ret = mlmeAssocApReceive(pdevInfo, pDesc->pOpBss,
                                 (WLAN_FRAME_ASSOC_REQ *)pMacHdr, frameLen);
        break;

    case SUBT_ASSOC_RESP:
        ret = mlmeAssocStaReceive(pdevInfo, (WLAN_FRAME_ASSOC_RESP *)pMacHdr,
                                  frameLen);
        break;

    case SUBT_REASSOC_REQ:
        pdevInfo->localSta->stats.Reassociations++;
        ret = mlmeReassocApReceive(pdevInfo, pDesc->pOpBss,
                                   (WLAN_FRAME_REASSOC_REQ *)pMacHdr, frameLen);
        break;

    case SUBT_REASSOC_RESP:
        ret = mlmeReassocStaReceive(pdevInfo,
                                    (WLAN_FRAME_REASSOC_RESP *)pMacHdr, frameLen);
        break;

    case SUBT_DISASSOC:
        ret = (A_UINT16)
              mlmeDisassocReceive(pdevInfo, pDesc->pOpBss,
                                  (WLAN_FRAME_DISASSOC *)pMacHdr, frameLen);
        break;

    case SUBT_AUTH: {
        WLAN_FRAME_AUTH *pFrame = (WLAN_FRAME_AUTH *)pMacHdr;

        if (pFrame->macHeader.frameControl.wep) {
            /*
             * WEP is set, must be frame #3 for the AP.  Call will
             * handle other sequence numbers with the wep bit on.
             */
            ret = mlmeAuthAp3Receive(pdevInfo, pDesc->pOpBss,
                                     (WLAN_FRAME_AUTH_ENCRYPT *)pMacHdr,
                                     frameLen,
                                     (A_BOOL)pDesc->status.rx.decryptError);
            break;
        }

        switch (pFrame->transSeqNo) {
        case 1:
            /* frame 1 received on AP */
            pdevInfo->localSta->stats.Authentications++;
            ret = mlmeAuthAp1Receive(pdevInfo, pDesc->pOpBss,
                                     (WLAN_FRAME_AUTH *)pMacHdr, frameLen);
            break;
        case 2:
            /* frame 2 received on STA */
            ret = mlmeAuthSta2Receive(pdevInfo, (WLAN_FRAME_AUTH_CHLG *)pMacHdr,
                                      frameLen);
            break;
        case 4:
            /* frame 4 received on STA */
            ret = mlmeAuthSta4Receive(pdevInfo, (WLAN_FRAME_AUTH *)pMacHdr,
                                      frameLen);
            break;
        default:
            mlmePrintf("mlmeMgtRxCompletion: !! Bad auth transaction seq num: %d\n",
                       pFrame->transSeqNo);
            break;
        }

        break;
    }

    case SUBT_DEAUTH:
        ret = (A_UINT16)
              mlmeDeauthReceive(pdevInfo, pDesc->pOpBss,
                                (WLAN_FRAME_DEAUTH *)pMacHdr, frameLen);
        break;

    default:
        mlmePrintf("mlmeMgtRxCompletion: unexpected subtype %d\n",
                   pMacHdr->frameControl.fSubtype);
        break;
    }

    freeBuffandDesc(pdevInfo, pDesc);
}

/* Only used by STA, AP uses chipReset. TODO - commonize. */

/******************************************************************
 * drvHwReset - Reset the hardware and restore WLAN operation
 *
 * DESCRIPTION: Resets the hardware and restores WLAN operation without
 * affecting SME state, power management state, beacon operation etc.
 *
 * On Entry: Caller must stop tx/rx and disable interrupts
 * On Exit:  interrupts are disabled.
 */
A_STATUS
drvHwReset(WLAN_DEV_INFO *pdevInfo, CHAN_VALUES *pChval, A_BOOL keepRCContent)
{
#ifndef BUILD_AP
    A_STATUS status;

    if (pdevInfo->bssDescr->bsstype == INDEPENDENT_BSS &&
        IS_CHAN_2GHZ(pChval->channelFlags))
    {
        switch (cservAdHocBand2Wmode(pdevInfo->staConfig.AdhocBand)) {
            case MODE_SELECT_11B:
            case MODE_SELECT_11G:
            case MODE_SELECT_108G:
                pdevInfo->bssDescr->opMode = wlanCFlagsToWirelessMode(pdevInfo, CHANNEL_B);
                pChval->channelFlags &= ~CHANNEL_ALL;
                pChval->channelFlags |= CHANNEL_B;
                break;
        }
    }

    /* Reset the hardware. Indicate that it is a channel change */
    status = wlanDevReset(pdevInfo, WLAN_STA_SERVICE, pChval, TRUE, keepRCContent);
    if (status != A_OK) {
        return status;
    }

    pdevInfo->globISRReg = 0;

    /* Restore LEDs */
    wdcSetLedState(pdevInfo->targetHandle,pdevInfo->pOSHandle->connectionStatus);

    /* Restore HW encrypt engine setting */
    setupSwEncryption(pdevInfo);

#endif /* #ifndef BUILD_AP */
    /*
     * Restore power save. Must do this before PS-Polls can
     * be re-enabled in wdcStaAssocBss()
     */
    powerHwResetNotify(pdevInfo);

    /* Restore SME related hardware state */
    if (pdevInfo->bssDescr->bsstype == INFRASTRUCTURE_BSS) {
        BSSDESCR *pBss = pdevInfo->bssDescr;

        if (pdevInfo->localSta->staState & STATE_ASSOC) {
            /*
             * Set AID and re-enable PS-Polls; also make an attempt to
             * resync beacon timers - on Venice really need a new beacon
             * to be able to do that.. nevertheless make somewhat of an
             * an attempt here; really needed to enable the bmiss interrupt
             * lest the AP has gone away!
             */
            wdcStaAssocBss(pdevInfo->targetHandle, 0, pdevInfo->localSta->assocId);
        }
    } else {
        /* 11g Ad-Hoc always uses long slot time */
        if (IS_CHAN_G(pdevInfo->staConfig.pChannel->channelFlags)) {
            pdevInfo->staConfig.shortSlotTime = FALSE;
            pdevInfo->defaultStaConfig.shortSlotTime = FALSE;
            pdevInfo->bssDescr->capabilityInfo.shortSlotTime = 0;
            pdevInfo->localSta->capInfo.shortSlotTime = 0;
            pdevInfo->useShortSlotTime = FALSE;
        }
        /* Restore beacon generation (if currently enabled) */
        if (IS_CONN_OR_JOINED(pdevInfo) &&
            (pdevInfo->staConfig.phwChannel == pdevInfo->staConfig.pChannel) &&
            pdevInfo->baseBss.pBeaconInfo && pdevInfo->baseBss.pBeaconInfo->beaconEnabled)
        {
            wdcStartBss(pdevInfo->targetHandle, 0, NULL);
        }
    }

    return A_OK;
}



/******************************************************************
 * drvChangeChannel - Change A2/D2 to a new channel
 *
 * DESCRIPTION: The routine does whatever is necessary to shut down
 * tx, rx, drain queues, change channel in the device, it should
 * reset the chip if required to change channel. Later, it restarts
 * the receive queue with the receive filter value of rxFilter.
 * This routine is primarily used during scan to switch thru a list
 * of channels and listening to the beacons.
 */
A_STATUS
drvChangeChannel(WLAN_DEV_INFO *pDev, CHAN_VALUES *pChval, A_BOOL force, A_BOOL keepRCContent)
{
    A_STATUS status = A_OK;

    ASSERT(pDev && pChval);

    mlmePrintf("drvChangeChannel: Switching from channel %d to %d\n",
               pDev->staConfig.phwChannel->channel, pChval->channel);

#ifdef BUILD_AP

    chipReset(pDev, pChval, keepRCContent);

#else /* BUILD_AP */

    if (pDev->NicResetInProgress ||
       (pDev->staConfig.phwChannel == pChval && !force))
    {
        /*
         * Don't channel change if a NIC reset is in progress. This avoids
         * recursion when StopReceive is called (SME may call DrvChangeChannel again)
         * Also don't bother to change if we're already where we want to be.
         */
        mlmePrintf("drvChangeChannel: NicResetInProgress or already on channel\n");
        return A_OK;
    }

    /* Reject NDIS send around change channel */
#if NDIS_WDM
    /*
     * (LW) Attention !!!
     * This is temp hack, should revisit later
     */
    pDev->invalidTxChannel = TRUE;
#else
    HACK_TO_PARDEV(pDev)->invalidTxChannel = TRUE;
#endif

#if 0
    /* LW
     * no need for the following code because target will handle stop recieve
     */
    halDisableInterrupts(HACK_TO_PARDEV(pDev), HAL_INT_GLOBAL);

    /* xxxxPGSxxxx - TODO stop receive and transmit are blocking operations and need to be removed
     * StopTransmit is macroed out already.
     */
    StopSwReceive(pDev);
    StopTransmit(HACK_TO_PARDEV(pDev), DO_CLEAR_TX_Q, DO_COMPLETE, DO_WAIT);
    StopReceive(pDev);
#endif

	/* 
	 * We need to do fast channel change during turbo switch 
	 * No FLUSH during turbo switch
	 */
   	if (!keepRCContent) {
        /*
         * We will flush twice around channel change,
         * increase number twice in here to prevent we change channel again
         * before we finish previous channel change.
         */
        NdisInterlockedIncrement(&pDev->numFlushCall);
        NdisInterlockedIncrement(&pDev->numFlushCall);

        wdcFlush(pDev->targetHandle,pDev->disableEnableCounter);
    }

    /* Reset the chip, changing the channel as well */
    status = drvHwReset(pDev, pChval, keepRCContent);
    ASSERT(status == A_OK);

   	if (!keepRCContent) {
        wdcFlush(pDev->targetHandle,pDev->disableEnableCounter);
        pDev->flushInProgress = TRUE;
    }

    pDev->rcvChangeInProgress = TRUE;

#if 0
    /* LW
     * no need for the following code because target will handle start recieve
     */

    halEnableInterrupts(HACK_TO_PARDEV(pDev), HAL_INT_GLOBAL);
    StartReceiveUnit(pDev, (RX_UCAST  | RX_MCAST | RX_BCAST |
                            RX_BEACON | RX_PROBE_REQ));
    RestartTransmit(HACK_TO_PARDEV(pDev));
#endif

    /* Allow NDIS send if now on configured channel */
    if (pDev->staConfig.phwChannel == pDev->staConfig.pChannel) {
#if NDIS_WDM
        /*
        * (LW) Attention !!!
        * This is temp hack, should revisit later
        */
        pDev->invalidTxChannel = FALSE;
#else
        HACK_TO_PARDEV(pDev)->invalidTxChannel = FALSE;
#endif
        osIndicateResourcesAvailable(pDev);
        ATH_RELEASE_SPINLOCK(pDev->lock);
        drainSendWaitQueue(pDev);
        ATH_ACQUIRE_SPINLOCK(pDev->lock);
    }

#endif /* BUILD_AP */

    return status;
}

A_BOOL
mlmeBsssetIsFull(BSSDESCR_SET *pSet)
{
    ASSERT(pSet->count < BSSDESCR_SET_SIZE);
    return (pSet->count < BSSDESCR_SET_SIZE) ? FALSE : TRUE;
}

void
mlmeBsssetAdd(WLAN_DEV_INFO *pDevInfo, BSSDESCR *pSrc, BSSDESCR_SET *pDest)
{
    BSSDESCR   *pOrig;
    A_CHAR      apState;
    WLAN_CFLAGS cflags;
    A_BOOL      isSrc11g = FALSE;

    if (!pSrc->pChannel) {
        return;
    }

    cflags = pSrc->pChannel->channelFlags;

    /*
     * Sometimes legacy AP's will support only a subset of the rates, reducing
     * the station's capabilities to just those of the legacy BSS
     */
    pSrc->opMode = wlanCFlagsToWirelessMode(pDevInfo, cflags);

    mlmePrintf("mlmeBsssetAdd: ");
    mlmePrintSsid(&pSrc->ssid);
    mlmePrintf(" chan %d, ORIG opMode %d, rate %d ofdm %d\n",
        pSrc->pChannel->channel,
        wlanCFlagsToWirelessMode(pDevInfo, cflags),
        pSrc->supportedRateSet.length,
        phyRateSubSetCheck(pDevInfo, WIRELESS_MODE_11g,
                           &pSrc->supportedRateSet, WLAN_PHY_OFDM));

    if (IS_CHAN_NON_108G_2GHZ(cflags) &&
        pDevInfo->staConfig.NetBand & MODE_SELECT_11G &&
        !IS_CHANNEL_14(pSrc->pChannel->channel) &&
        phyRateSubSetCheck(pDevInfo, WIRELESS_MODE_11g,
                           &pSrc->supportedRateSet, WLAN_PHY_OFDM))
    {
        isSrc11g = TRUE;
    }

    if (pSrc->opMode == WIRELESS_MODE_11b && isSrc11g) {
        /*
         * 11g STA changed to 11b now trying to join an 11g AP,
         * change opMode back to 11g.
         * Note: We come here because wlanMlmeJoinRequest has
         * modified the channel to CCK. Time to revert it back to OFDM.
         * Reminder: wlanMlmeJoinRequest will change the channel type back to OFDM.
         */
        pSrc->opMode = wlanCFlagsToWirelessMode(pDevInfo,(cflags &(~CHANNEL_CCK)) | CHANNEL_OFDM);
    }

    if (pSrc->opMode == WIRELESS_MODE_11g && !isSrc11g) {
        /*
         * 11g STA finds an 11b AP, change opMode to 11b.
         * Reminder: wlanMlmeJoinRequest will change the channel type to CCK.
         */
        pSrc->opMode = wlanCFlagsToWirelessMode(pDevInfo,(cflags &(~CHANNEL_OFDM)) | CHANNEL_CCK);
    }

    mlmePrintf("opMode %d\n", pSrc->opMode);

    if ((pSrc->opMode == WIRELESS_MODE_XR) &&
        (!phyRateSubSetCheck(pDevInfo, WIRELESS_MODE_XR,
                             &pSrc->supportedRateSet, WLAN_PHY_XR)))
    {
        pSrc->opMode = wlanCFlagsToWirelessMode(pDevInfo, cflags & (~CHANNEL_XR));
#if defined(XR_HACKERY) && (XR_DEFAULT_RATE >= 6000)
        if (pSrc->beaconInterval > 200) {
            pSrc->opMode = WIRELESS_MODE_XR;
        }
#endif
    }


#ifndef BUILD_AP
    if (pDevInfo->bssDescr->bsstype == INDEPENDENT_BSS &&
        A_MACADDR_COMP(&pDevInfo->bssDescr->bssId, &pSrc->bssId) == 0)
    {
        pSrc->opMode = wlanCFlagsToWirelessMode(pDevInfo,
                           wlanMode2Cflags(cservAdHocBand2Wmode(
                                               pDevInfo->staConfig.AdhocBand)));
        mlmePrintf("Adhoc opMode %d\n", pSrc->opMode);
    }
#endif /* BUILD_AP */

    mlmePrintf("New opMode %d\n", pSrc->opMode);

    pOrig = mlmeBsssetFind(pDest, &pSrc->bssId, cflags);

    if (pOrig != NULL) {
        SSID_ARRAY *pSsids = &pDevInfo->pscanInfo->desSsids;
        SSID       tmpSsid;
        WPA_IE     tmpWpaIe;
        A_BOOL     sameSsid, replaceAll;
        A_BOOL     isSrcSsidDesired = FALSE, isOrigSsidDesired = FALSE;
        int        i;
        A_BOOL tmpPrivacy = FALSE;

        /*
         * Some APs hide either their main SSID or multiple SSIDs that they
         * support through one BSSID. They do this by not broadcasting the
         * SSID in Beacons, and only respond to Probe Requests that include
         * one of the SSIDs that they support. See bugs 8312 and 8313.
         *
         * If we already have a real SSID for this BSSID, and that SSID
         * matches one of our configured SSIDs, then we don't store this
         * new SSID as it would overwrite our valid (but probably hidden)
         * SSID.
         *
         * We also save off other information that appears only in the
         * Probe Request and not in the Beacon.
         */
        apState = pOrig->apState;

        for (i = 0; i < A_ATOMIC_READ(&pSsids->count); i++) {
            if (isSsidEqual(&pSrc->ssid, &pSsids->elements[i])) {
                isSrcSsidDesired = TRUE;
            }
            if (isSsidEqual(&pOrig->ssid, &pSsids->elements[i])) {
                isOrigSsidDesired = TRUE;
            }
        }

        sameSsid = isSsidEqual(&pSrc->ssid, &pOrig->ssid);

        replaceAll = sameSsid || isSrcSsidDesired || !isOrigSsidDesired;

        if (!replaceAll) {
            /* Save 'em off */
            tmpSsid  = pOrig->ssid;
            tmpWpaIe = pOrig->wpaIe;
            tmpPrivacy = (A_BOOL) pOrig->capabilityInfo.privacy;
        }

        *pOrig = *pSrc;

        if (!replaceAll) {
            /* Restore 'em */
            pOrig->ssid  = tmpSsid;
            pOrig->wpaIe = tmpWpaIe;
            pOrig->capabilityInfo.privacy = tmpPrivacy;

            /* Don't forget that we've previously seen a Probe Request. */
            pOrig->probeResp = TRUE;
        }

        /* Always save and restore apState, too. */
        pOrig->apState = apState;

        return;
    }

    if (pDest->count < BSSDESCR_SET_SIZE) {
        pDest->bssDescrArray[pDest->count++] = *pSrc;
    }
}

void
mlmeBsssetAge(BSSDESCR_SET *pSrc, A_UINT32 agingPeriod, BSSID *pBssidToKeep)
{
    A_UINT16 i   = 0;
    A_UINT32 now = A_MS_TICKGET();

    ASSERT(pSrc);

    if (pSrc->count <= 0) {
        return;
    }

    /* Convert the agingPeriod from seconds to ms */
    agingPeriod *= 1000;

    while (i < pSrc->count) {
        BSSDESCR *pBss = &pSrc->bssDescrArray[i];

        /*
         * Update the scanTime of the BssidToKeep to current time
         * so that it won't get removed from our list.
         * Ignore if pBssid is NULL.
         */
        if (pBssidToKeep && A_MACADDR_COMP(pBssidToKeep, &pBss->bssId) == 0) {
            pBss->scanTime = now;
        }

        if (now - pBss->scanTime > agingPeriod) {
            /*
             * Too old, discard this BSSDESCR.
             * Move the last entry here, order is not important.
             * Don't move the last entry onto itself.
             */
            pSrc->count--;

            if (i < pSrc->count) {
                *pBss = pSrc->bssDescrArray[pSrc->count];
            }

        } else {
            i++;
        }
    }
}

BSSDESCR *
mlmeBsssetFind(BSSDESCR_SET *pSet, WLAN_MACADDR *pAddr, A_UINT16 channelFlags)
{
    int i;

    if (pSet == NULL || pAddr == NULL) {
        return NULL;
    }

    for (i = 0; i < pSet->count; i++) {
        if (A_MACADDR_COMP(pAddr, &pSet->bssDescrArray[i].bssId) == 0 &&
            (pSet->bssDescrArray[i].pChannel->channelFlags &
             CHANNEL_ALL) == (channelFlags & CHANNEL_ALL)) {
            /* Found the matching BSSID. */

            return &pSet->bssDescrArray[i];
        }
    }

    return NULL;
}

void
mlmeBsssetDel(BSSDESCR_SET *pSet, WLAN_MACADDR *pAddr)
{
    int i = 0;

    if (pSet == NULL || pAddr == NULL) {
        return;
    }

    while (i < pSet->count) {
        if (A_MACADDR_COMP(pAddr, &pSet->bssDescrArray[i].bssId) == 0) {
            /*
             * Found a matching BSSID. Move the last entry here,
             * order is not important.
             */
            pSet->count--;

            if (i < pSet->count) {
                /* No need to move last entry. */
                pSet->bssDescrArray[i] = pSet->bssDescrArray[pSet->count];
            }

        } else {
            i++;
        }
    }
}

/*
 * isNullSsid
 *
 * A null ssid is either 0 bytes in length or non-zero length but all bytes
 * are 0. Strange but true. It's a common method used by vendors like Cisco.
 */
A_BOOL
isNullSsid(SSID *pSsid)
{
    A_UINT8 i;

    if (pSsid == NULL) {
        return TRUE;
    }

    if (pSsid->length == 1 && pSsid->ssid[0] == ' ') {
        /* Proxim's method of hiding ssid is one blank character. */
        return TRUE;
    }

    i = pSsid->length;

    while (i > 0) {
        if (pSsid->ssid[--i] != 0) {
            return FALSE;
        }
    }

    return TRUE;
}

void
ssidArrayAdd(SSID_ARRAY *pArray, A_CHAR *buf, A_UINT8 length)
{
    int  i = 0;
    SSID tmpSsid;

    ASSERT(pArray);
    ASSERT(buf);

    /* Array is full */
    if (A_ATOMIC_READ(&pArray->count) >= SSID_ARRAY_SIZE) {
        return;
    }

    if (length == 3 && A_TOLOWER(buf[0]) == 'a' &&
        A_TOLOWER(buf[1]) == 'n' && A_TOLOWER(buf[2]) == 'y')
    {
        /* "any" is converted to zero length ssid */
        length = 0;
    }

    /* Don't add zero-length element if there's already something in array */
    if (A_ATOMIC_READ(&pArray->count) > 0 && length == 0) {
        return;
    }

    tmpSsid.elementID = ELE_SSID;
    tmpSsid.length    = length;
    A_BCOPY(buf, &tmpSsid.ssid, length);

    for (i = 0; i < A_ATOMIC_READ(&pArray->count); i++) {
        if (isSsidEqual(&pArray->elements[i], &tmpSsid)) {
            /* Duplicate entry */
            return;
        }
    }

    pArray->elements[i] = tmpSsid;
#ifdef BUILD_AP
    pArray->count++;
    pArray->numValid++;

#else
    NdisInterlockedIncrement(&pArray->count);
    NdisInterlockedIncrement(&pArray->numValid);
#endif

    ASSERT(A_ATOMIC_READ(&pArray->count) >= A_ATOMIC_READ(&pArray->numValid));

    mlmePrintf("ssidArrayAdd: Ssid[%d] = \"", i);
    mlmePrintSsid(&tmpSsid);
    mlmePrintf("\"\n");
}

void
ssidArrayInit(SSID_ARRAY *pArray)
{
    ASSERT(pArray);

//    pArray->count = pArray->numValid = 0;
    A_ATOMIC_SET(&pArray->count, 0);
    A_ATOMIC_SET(&pArray->numValid, 0);
}

A_BOOL
isSsidEqual(SSID *s1, SSID *s2)
{
    if (s1 == NULL || s2 == NULL) {
        return FALSE;
    }

    return !A_BCOMP(s1, s2, s1->length+2);
}

A_INT32
ssidArrayFind(SSID_ARRAY *pArray, SSID *pSsid)
{
    A_INT32 i;

    if (pArray == NULL || pSsid == NULL || A_ATOMIC_READ(&pArray->count) <= 0) {
        return -1;
    }

    ASSERT(A_ATOMIC_READ(&pArray->count) <= SSID_ARRAY_SIZE);

    for (i = 0; i < A_ATOMIC_READ(&pArray->count); i++) {
        if (isSsidEqual(pSsid, &pArray->elements[i])) {
            return i;
        }
    }
    return -1;
}

static void
wlanStaJoinBss(WLAN_DEV_INFO *pDev)
{
    BSSDESCR            *pBss = pDev->bssDescr;
    A_UINT32            bmissTime, sleepDuration;
    WDC_BSS_ATTRIBUTES  wdcBssConfig;
    A_BOOL              turboPrime;
    WIRELESS_MODE       mode;

    A_MEM_ZERO(&wdcBssConfig, sizeof(wdcBssConfig));

    ASSERT(pBss->DTIMPeriod);
    if (!pBss->DTIMPeriod) {
        pBss->DTIMPeriod = 1;
    }

    wdcBssConfig.bssType            = pBss->bsstype;
    wdcBssConfig.beaconInterval     = pBss->beaconInterval;
    wdcBssConfig.dtimInterval       = pBss->DTIMPeriod * pBss->beaconInterval;
    wdcBssConfig.cfpInterval        = wdcBssConfig.dtimInterval * pBss->cfParamSet.cfpPeriod;
    wdcBssConfig.cfpMaxDuration     = pBss->cfParamSet.cfpMaxDuration;
    wdcBssConfig.atimWindow         = pDev->staConfig.atimWindow;
    wdcBssConfig.defaultRateIndex   = pDev->baseBss.defaultRateIndex;

    turboPrime = VALID_ATH_ADVCAP_ELEMENT(&pBss->athAdvCapElement) &&
                    pBss->athAdvCapElement.info.useTurboPrime &&
                    (pDev->staConfig.abolt & ABOLT_TURBO_PRIME);
    mode = wlanCFlagsToWirelessMode(pDev, pDev->staConfig.pChannel->channelFlags);
    switch (mode) {
    case WIRELESS_MODE_11a:
        wdcBssConfig.wlanMode = turboPrime ? WLAN_MODE_11a_TURBO_PRIME : WLAN_MODE_11a;
        break;
    case WIRELESS_MODE_TURBO:
        wdcBssConfig.wlanMode = WLAN_MODE_11a_TURBO;
        break;
    case WIRELESS_MODE_11b:
        wdcBssConfig.wlanMode = WLAN_MODE_11b;
        break;
    case WIRELESS_MODE_11g:
        wdcBssConfig.wlanMode = turboPrime ? WLAN_MODE_11g_TURBO_PRIME : WLAN_MODE_11g;
        break;
    case WIRELESS_MODE_108g:
        wdcBssConfig.wlanMode = WLAN_MODE_11g_TURBO;
        break;
    case WIRELESS_MODE_XR:
        ASSERT(0);
        break;
    default:
        ASSERT(0);
        break;
    }

    if ((wdcBssConfig.wlanMode == WLAN_MODE_11g) && (wdcBssConfig.bssType == INDEPENDENT_BSS)) {
        pDev->staConfig.shortSlotTime = FALSE;
        pDev->defaultStaConfig.shortSlotTime = FALSE;
        pDev->bssDescr->capabilityInfo.shortSlotTime = 0;
        pDev->localSta->capInfo.shortSlotTime = 0;
        pDev->useShortSlotTime = FALSE;
    }
    wdcBssConfig.shortSlotTime11g = pDev->useShortSlotTime;

    bmissTime = MS_TO_TU(pDev->staConfig.noBeaconTimeout);
    bmissTime = A_ROUNDUP(bmissTime, wdcBssConfig.beaconInterval);
    wdcBssConfig.bmissThreshold = A_MAX(bmissTime / wdcBssConfig.beaconInterval, 10);
    wdcBssConfig.bmissThreshold = (wdcBssConfig.bmissThreshold + 1) / 2;

    sleepDuration = (pDev->staConfig.sleepMode == POWERMGT_SLEEP) ?
                    MS_TO_TU(pDev->staConfig.sleepTimePwrSave) :
                    MS_TO_TU(pDev->staConfig.sleepTimePerf);
    pDev->powerMgmt.sleepDuration = sleepDuration;
    wdcBssConfig.sleepDuration    = sleepDuration;

    /* Adjust our transmit power if the BSS is using TPC */
    wdcBssConfig.tpcPowerLimit = MAX_TX_POWER;
    if (IS_ELEMENT_USED(&pBss->tpcIe)) {
        wdcBssConfig.tpcPowerLimit = pBss->tpcIe.pwrLimit;
    }

    wdcStaJoinBss(pDev->targetHandle, 0, &pBss->bssId, &wdcBssConfig);

    return;
}

/*
 * mlmeBeaconReceive - Parse and process received beacon frame
 *
 * Spec section 11.1.2.3
 * Beacons will be typically reach the MLME software only when
 * enabled for scan or join operations.
 */
void
mlmeBeaconReceive(WLAN_DEV_INFO *pdevInfo, ATHEROS_DESC *pDesc)
{
    SIB_ENTRY           *apSib;
    IBSS_PARAM_SET      *pIbssParams = NULL;
    BSSDESCR            *pBss = pdevInfo->bssDescr;
    ATH_ADVCAP_IE       *pAthAdvCap;
    WLAN_RATE_SET       beaconRateSet;
    WLAN_RATE_SET       wlanBeaconRateSet;
    WLAN_FRAME_BEACON   *bFrame;
    INFO_ELEMENT_SET    ieSet;
    NONERP_ELEMENT      *pNonErpElement = NULL;
    RATE_SET            *pRateSet       = NULL;
    EXT_RATE_SET        *pExtRateSet    = NULL;
    SSID                *pSsid          = NULL;
    TIM_ELEMENT         *pTim           = NULL;
    DS_PARAM_SET        *pDsParams      = NULL;
    TPC_IE              *pTpc           = NULL;
    const A_UINT16      len             = (A_UINT16)pDesc->status.rx.dataLength;
    A_UINT8             *pInfoElements  = NULL;
    A_UINT16            infoElemLen     = 0;
    INFO_ELEMENT        *pWpaIe         = NULL;
    AIRONET_IE          *aie            = NULL;
    A_UINT32            oldVal;
    A_UINT32            bssAttribValue;

    bFrame = pDesc->pBufferVirtPtr.beacon;

    mlmeL2Printf("mlmeBeaconReceive: Beacon from ");
    mlmeL2PrintMacAddress(bFrame->macHeader.bssId);
    mlmeL2Printf(" on channel %d, transState: %x\n",
                 pdevInfo->staConfig.phwChannel->channel,
                 pdevInfo->localSta->transState);

    /* Find and store the beacon elements */
    A_MEM_ZERO(&ieSet, sizeof(ieSet));
    pInfoElements = (A_UINT8 *)&bFrame->buffer;
    infoElemLen   = (A_UINT16)(len - (pInfoElements - (A_UINT8 *)bFrame));
    mgtGetInfoElements(&ieSet, (INFO_ELEMENT *)pInfoElements, infoElemLen);

    /* Basic sanity checks */
    pSsid     = (SSID *)ieSet.pInfoElement[eidMap[ELE_SSID]];
    pTim      = (TIM_ELEMENT *)ieSet.pInfoElement[eidMap[ELE_TIM]];
    pDsParams = (DS_PARAM_SET *)ieSet.pInfoElement[eidMap[ELE_DS_PARAM_SET]];

    if (pSsid == NULL || (bFrame->capInfo.ess && pTim == NULL)) {
        /* Improperly formatted beacon. Invalid SSID or TIM. */
        mlmePrintf("Invalid SSID or TIM in beacon, dropped\n");
        return;
    }

    if (!wlanDoesDsElementMatch(pdevInfo, pDsParams)) {
        /*
         * We picked up this beacon on a wrong channel.
         * Can happen in 11b. Beacons can be
         * heard anywhere in the 25 MHz vicinity. Ignore it.
         */
        return;
    }

    /*
     * If we already have an entry for the sta of this AP, mark
     * its state as AP up and running. If we don't have an entry,
     * don't bother.
     */
    apSib = sibEntryFind(pdevInfo, &bFrame->macHeader.srcAddr, NULL);
    if (apSib != NULL) {
        /* May not be our AP, but update state & timestamp */
        apSib->staState      |= STATE_AP_UP;
        apSib->lastBeaconTime = bFrame->timestamp;
    }

    /* If there's a TPC element, make sure it's valid */
    pTpc = (TPC_IE *)ieSet.pInfoElement[eidMap[ELE_TPC]];
    if (pTpc && !VALID_TPC_ELEMENT(pTpc)) {
        pTpc = NULL;
    }

    if (pdevInfo->localSta->staState & STATE_JOINED) {
        A_BOOL isIbss = (pBss->bsstype == INDEPENDENT_BSS);
        A_BOOL isBcMcTim;
        A_BOOL isDirectedTim;

        /* Is this beacon from our BSS ? */
        if ((A_MACADDR_COMP(&pdevInfo->baseBss.bssId,
                            &bFrame->macHeader.bssId)) == 0)
        {
            A_UINT16 aid = pdevInfo->localSta->assocId;

            pdevInfo->localSta->lastBeaconTime = bFrame->timestamp;

            /* process TPC */
            if (pTpc) {
                if (!IS_ELEMENT_USED(&pBss->tpcIe) ||
                    pBss->tpcIe.pwrLimit != pTpc->pwrLimit)
                {
                    /*
                     * Either this is the first time we've seen the IE, or
                     * the power limit contained therein has just changed.
                     *
                     * Would be costly to update the BSSDESCR in the scan list
                     * here. Someday we will not have a separate copy and it
                     * will just update automatically.
                     */
                    pBss->tpcIe = *pTpc;
                    bssAttribValue = (A_UINT32)pTpc->pwrLimit;
                    wdcUpdateBssAttribute(pdevInfo->targetHandle, 0,
                                          TPC_POWER_LIMIT,
                                          sizeof(bssAttribValue),
                                          (TARGET_CONFIG_VAL)&bssAttribValue);
                }
            } else if (IS_ELEMENT_USED(&pBss->tpcIe)) {
                /*
                 * The IE has gone away.  Assume that we can go back to full
                 * strength. Clear out our IE info in our BSSDESCR.
                 */
                A_MEM_ZERO(&pBss->tpcIe, sizeof(pBss->tpcIe));
                bssAttribValue = MAX_TX_POWER;
                wdcUpdateBssAttribute(pdevInfo->targetHandle, 0,
                                      TPC_POWER_LIMIT,
                                      sizeof(bssAttribValue),
                                      (TARGET_CONFIG_VAL)&bssAttribValue);
            }

            /*
             * If associated to infrastructure, perform TIM/PSPoll processing.
             * Check TIM whether we're sleeping or not. AP may think we are.
             * Sending null data packet will cure it.
             * process the nonERP IE to update the nonERP protection schems
             */
            if (pTim && pBss->bsstype == INFRASTRUCTURE_BSS  &&
                (pdevInfo->localSta->staState & STATE_ASSOC) &&
                bFrame->capInfo.ess)
            {
                if (pdevInfo->powerMgmt.needResyncToBeacon) {
                    /*
                     * We may have associated without having received a single
                     * Beacon - probe responses don't have the dtim period in
                     * them.  In this scenario, we need to set the DTIM period
                     * and TIM offset, they are only available in Beacon frames.
                     */
                    if (pBss->DTIMPeriod != pTim->dtimPeriod) {
                        pBss->DTIMPeriod = pTim->dtimPeriod;
                        bssAttribValue = (A_UINT32)pTim->dtimPeriod * pBss->beaconInterval;
                        wdcUpdateBssAttribute(pdevInfo->targetHandle, 0,
                                              DTIM_INTERVAL,
                                              sizeof(bssAttribValue),
                                              (TARGET_CONFIG_VAL)&bssAttribValue);
                    }
                    wdcStaAssocBss(pdevInfo->targetHandle, 0, pdevInfo->localSta->assocId);
                    pdevInfo->powerMgmt.lastRememberedBeacon = TSF_TO_TU(bFrame->timestamp);
                    pdevInfo->powerMgmt.needResyncToBeacon   = FALSE;
                    powerSet(pdevInfo, WAKE_UP, REDUCE_PWR_FROM_1, FALSE, 0);
                }

                /* process TIM */
                isDirectedTim = mlmeBeaconTimCheck(pTim, aid, &isBcMcTim);
                powerTimIndication(pdevInfo, isDirectedTim, isBcMcTim);

                /* We can now notify AP regarding PM activity */
                pdevInfo->localSta->receivedBeaconFromHomeAP = TRUE;

                /* For 11g mode */
                /* process NONERP IE */
                pNonErpElement = (NONERP_ELEMENT *)
                            ieSet.pInfoElement[eidMap[ELE_NONERP]];

                processNonErpElement(pdevInfo, pNonErpElement);

                pBss->wpaIe.length = 0;

                pWpaIe = ieSet.pWpaIe;
                if (pWpaIe != NULL && VALID_WPA_ELEMENT((WPA_IE *)pWpaIe)) {
                    mlmeElementCopy(pWpaIe, (INFO_ELEMENT *)&pBss->wpaIe);
                }

                /* Process short slot time cap */
                oldVal = (A_UINT32)pdevInfo->useShortSlotTime;
                pdevInfo->useShortSlotTime = pdevInfo->staConfig.shortSlotTime ?
                                             bFrame->capInfo.shortSlotTime : FALSE;
                if (oldVal != (A_UINT32)pdevInfo->useShortSlotTime) {
                    bssAttribValue = (A_UINT32)pdevInfo->useShortSlotTime;
                    wdcUpdateBssAttribute(pdevInfo->targetHandle, 0,
                                          SHORT_SLOT_TIME_11g,
                                          sizeof(bssAttribValue),
                                          (TARGET_CONFIG_VAL)&bssAttribValue);
                }

                /* Process athAdvCapElement */
                pAthAdvCap = (ATH_ADVCAP_IE *)ieSet.pAthCap;
                if ((pAthAdvCap != NULL) &&
                    VALID_ATH_ADVCAP_ELEMENT(pAthAdvCap))
                {
                    /* TURBO_PRIME */
                    A_UINT8 currBoost = pdevInfo->bssDescr->athAdvCapElement.info.boost;
                    mlmeElementCopy((INFO_ELEMENT *)pAthAdvCap,
                                    (INFO_ELEMENT *)&pdevInfo->bssDescr->athAdvCapElement);

                    if (pdevInfo->staConfig.abolt & ABOLT_TURBO_PRIME &&
                        pdevInfo->turboPrimeInfo.turboPrimeAllowed &&
                        currBoost != pAthAdvCap->info.boost)
                    {
                        /* Going back and forth between Turbo and Non Turbo modes. */
                        turboPrimeSwitch(pdevInfo, pdevInfo->bssDescr->athAdvCapElement.info.boost);
                    }
                }

                if ((pdevInfo->staConfig.abolt & ABOLT_WME_ELE) && ieSet.pAthWmeIe) {
                    if (ieSet.pAthWmeIe->info.info != pBss->athWmeIe.info.info) {
                        mlmeElementCopy((INFO_ELEMENT *)ieSet.pAthWmeIe, (INFO_ELEMENT *)&pBss->athWmeIe);
                        athWmeIeGetInfo(pdevInfo, &pdevInfo->baseBss, ieSet.pAthWmeIe);
                        wlanUpdateTxQueues(pdevInfo);
                    }
                }

#ifdef WME
                if (pdevInfo->staConfig.WmeEnabled && ieSet.pWmeV1Ie) {
                    if (ieSet.pWmeV1Ie->info.info != pBss->wmeV1Ie.info.info) {
                        mlmeElementCopy((INFO_ELEMENT *)ieSet.pWmeV1Ie, (INFO_ELEMENT *)&pBss->wmeV1Ie);
                        wmeV1IeGetInfo(pdevInfo, &pdevInfo->baseBss, ieSet.pWmeV1Ie);
                        /* for now, always update EDCA parms; should look at info bit */
                        wlanUpdateTxQueues(pdevInfo);
                    }
                    apSib->qosdataEnabled = 1;
                } else {
                    apSib->qosdataEnabled = 0;
                }
#endif
            }

            if (isIbss && bFrame->capInfo.ibss) {
                if (pdevInfo->powerMgmt.needResyncToBeacon) {
                    pdevInfo->powerMgmt.needResyncToBeacon = FALSE;
                    wdcStartBss(pdevInfo->targetHandle, 0, NULL);
                }
                cservAdHocTrafficReceived(pdevInfo);
            }
        } else if (isIbss && bFrame->capInfo.ibss     &&
                   pSsid->length == pBss->ssid.length &&
                   A_BCOMP(&pSsid->ssid, &pBss->ssid.ssid, pSsid->length) == 0)
        {
            /*
             * It's ad-hoc network, ssid matches but bssid doesn't.
             * 802.11 protocol requires that if two different bssid's for the same ad-hoc
             * ssid are found in the network, they should merge. One with an
             * earlier timestamp must prevail.
             */

            if (TSF_COMP(pDesc->status.rx.timestamp, bFrame->timestamp) < 0) {
                /* ours is a higher timestamp, we started earlier. */
                return;
            }

            /* Now we need to change our BSSID. */
            A_MACADDR_COPY(&bFrame->macHeader.bssId, &pdevInfo->baseBss.bssId);
            A_MACADDR_COPY(&bFrame->macHeader.bssId, &pBss->bssId);

            /*
             * Adopt the ATIM window of the network we are joining.
             * This needs to be set before starting beacons
             */
            pIbssParams = (IBSS_PARAM_SET *)
                ieSet.pInfoElement[eidMap[ELE_IBSS_PARAM_SET]];
            if (pIbssParams) {
                pdevInfo->staConfig.atimWindow = pIbssParams->atimWindow;
            }

            /*
             * Adopt the same basic rate set of the network we are joining
             * but not the extended rate set
             */
            pRateSet = (RATE_SET *)
                    ieSet.pInfoElement[eidMap[ELE_SUPPORTED_RATES]];
            wlanElementToFullRateSet(pRateSet, NULL, &beaconRateSet);
            xSectionRateSet(&beaconRateSet, &pdevInfo->baseBss.rateSet,
                            &wlanBeaconRateSet);
            wlanRateSetCopy(&wlanBeaconRateSet, &pdevInfo->baseBss.rateSet);

            pdevInfo->localSta->lastBeaconTime = bFrame->timestamp;

            /*
             * The following is hacky! The elegant method would
             * have the state m/c attempt to join the new BSS and
             * start beacon generation once joined. We'll start
             * beacon generation now to get the beacon updated and
             * use needResyncToBeacon to get the tsfs chugging
             * correctly
             */
            wlanStaJoinBss(pdevInfo);
            wlanBeaconInit(pdevInfo);
            pdevInfo->powerMgmt.needResyncToBeacon = TRUE;
        }
    } /* if STATE_JOINED */

    if (pdevInfo->localSta->transState & TSTATE_JOINING) {
        A_STATUS status =
            mlmeProcessJoinFrame(pdevInfo, pDesc, &ieSet);

        if (status != A_OK) {
            return;
        }
    }

    if (pdevInfo->localSta->transState & TSTATE_SCANNING) {
        SCAN_INFO    *pInfo        = pdevInfo->pscanInfo;
        A_BOOL        wasProbeSent = FALSE;
        BSSDESCR     *pNewBss;
        INFO_ELEMENT *p1;

        /* We are in the middle of scanning process */
        /* See if we have already received beacon from this AP */
        if (pdevInfo->pscanInfo == NULL) {
            /* something wrong. We are scanning but set is not allocated */
            mlmePrintf("mlmeBeaconReceive: ERROR: scanning but no set allocated\n");
            return;
        }

        if (pdevInfo->pscanInfo->bssSet.count == BSSDESCR_SET_SIZE) {
            /* We maxed out on scanned beacons. Punt. */
            return;
        }

        /*
         * Assume that space for entire set is already allocated
         * by whoever initiated the scan.
         */
        mlmePrintf("New beacon found. SSID=\"");
        mlmePrintSsid(pSsid);
        mlmePrintf("\"\n");

        if ((pNewBss = (BSSDESCR *)A_DRIVER_MALLOC(sizeof(*pNewBss))) == NULL) {
            return;
        }

        A_MEM_ZERO((char *)pNewBss, sizeof(*pNewBss));

        A_MACADDR_COPY(&bFrame->macHeader.bssId, &pNewBss->bssId);

        pTim = (TIM_ELEMENT *)ieSet.pInfoElement[eidMap[ELE_TIM]];

        pNewBss->DTIMPeriod     = pTim ? pTim->dtimPeriod : DEFAULT_DTIM_PERIOD;
        pNewBss->beaconInterval = bFrame->beaconInterval;
        pNewBss->bsstype        = (bFrame->capInfo.ibss == 1) ?
                                  INDEPENDENT_BSS : INFRASTRUCTURE_BSS;
        pNewBss->pChannel       = pdevInfo->staConfig.phwChannel;
        pNewBss->rssi           = pDesc->status.rx.rssi;
        pNewBss->timestamp      = bFrame->timestamp;
        pNewBss->capabilityInfo = bFrame->capInfo;

        mlmeElementCopy((INFO_ELEMENT *)pSsid, (INFO_ELEMENT *)&pNewBss->ssid);
        mlmeElementCopy((INFO_ELEMENT *)pDsParams, (INFO_ELEMENT *)&pNewBss->dsParamSet);

        /*
         * Need to make a correction. Copy only the basic rate
         * values (with msb = 1) into the BSS description
         */
        pRateSet    = (RATE_SET *)
            ieSet.pInfoElement[eidMap[ELE_SUPPORTED_RATES]];
        pExtRateSet = (EXT_RATE_SET *)
            ieSet.pInfoElement[eidMap[ELE_EXT_SUPPORTED_RATES]];
        /* save a copy for NDIS OIDs */
        mlmeElementCopy((INFO_ELEMENT *) pRateSet, (INFO_ELEMENT *)&pNewBss->rateSetIe);
        /* save a copy for NDIS OIDs */
        mlmeElementCopy((INFO_ELEMENT *) pExtRateSet, (INFO_ELEMENT *)&pNewBss->extRateSetIe);
        wlanElementToFullRateSet(pRateSet, pExtRateSet, &wlanBeaconRateSet);
        wlanRateSetCopy(&wlanBeaconRateSet, &pNewBss->supportedRateSet);

        /* Process TPC element */
        if (pTpc) {
            pNewBss->tpcIe = *pTpc;
        }

        /* get nonERP element */
        pNewBss->nonErpElement.length = 0;
        mlmeElementCopy((INFO_ELEMENT *)
                        ieSet.pInfoElement[eidMap[ELE_NONERP]],
                        (INFO_ELEMENT *)&pNewBss->nonErpElement);

        /* make the element invalid first */
        /* need remove this when the mlmeElementCopy changed */
        pWpaIe = ieSet.pWpaIe;
        if (pWpaIe != NULL && VALID_WPA_ELEMENT((WPA_IE *)pWpaIe)) {
            mlmeElementCopy(pWpaIe, (INFO_ELEMENT *)&pNewBss->wpaIe);
        } else {
            A_MEM_ZERO(&pNewBss->wpaIe, sizeof(pNewBss->wpaIe));
        }

        pNewBss->athAdvCapElement.length = 0;
        if (ieSet.pAthCap != NULL && VALID_ATH_ADVCAP_ELEMENT(ieSet.pAthCap)) {
            mlmeElementCopy((INFO_ELEMENT *)ieSet.pAthCap,
                        (INFO_ELEMENT *)&pNewBss->athAdvCapElement);
        }

        aie = (AIRONET_IE *)ieSet.pInfoElement[eidMap[ELE_AIRONET]];
        if (aie != NULL) {
            ccxAironetIeProcess(pdevInfo, pNewBss, aie);
        }

        mlmeElementCopy((INFO_ELEMENT *)
                        ieSet.pInfoElement[eidMap[ELE_CF_PARAM_SET]],
                        (INFO_ELEMENT *)&pNewBss->cfParamSet);

        mlmeElementCopy((INFO_ELEMENT *)
                        ieSet.pInfoElement[eidMap[ELE_IBSS_PARAM_SET]],
                        (INFO_ELEMENT *)&pNewBss->ibssParamSet);

        if (pNewBss->bsstype == INFRASTRUCTURE_BSS) {
            p1 = ieSet.pInfoElement[eidMap[ELE_COUNTRY_INFO]];

            if (p1 && (p1->length + sizeof(p1->elementID) +
                       sizeof(p1->length)) <= sizeof(pNewBss->ccList))
            {
                mlmeElementCopy(p1, (INFO_ELEMENT *)&pNewBss->ccList);
            }
        }

        pNewBss->pChannel = pdevInfo->staConfig.phwChannel;
        pNewBss->scanTime = A_MS_TICKGET(); // save the time of reception

        /*
         * Take note of the fact that we've seen a BSS here. BSSes can have
         * one or more SSIDs that are not broadcast.  Please see bugs 8312 and
         * 8313 for more details.
         */
        pdevInfo->staConfig.phwChannel->bssSeenHere = TRUE;

        if (pInfo->scantype == PASSIVE_SCAN ||
            IS_CHAN_PASSIVE(pdevInfo->staConfig.phwChannel->channelFlags))
        {
            /*
             * Some SSIDs are not broadcast in the Beacon.  To find these SSIDs
             * we need to send a Probe Request containing the SSID we're looking
             * for.
             * Channel/domain is set for passive scan, but since we saw a Beacon
             * on this channel, it should be okay to send Probe Request(s).
             */
            SSID_ARRAY *pSsids = &pdevInfo->pscanInfo->desSsids;
            int        i;

            for (i = 0; i < A_ATOMIC_READ(&pSsids->count); i++) {
                if (pSsids->elements[i].length > 0) {
                    wasProbeSent = TRUE;
                    wlanMlmeProbeRequest(pdevInfo, &bFrame->macHeader.srcAddr,
                                         &pSsids->elements[i]);
                }
            }
        }

        if (wasProbeSent && !pInfo->wasTimeAdded) {
            /*
             * Make sure that we stick around long enough to get the Probe
             * Responses back.
             */
            pInfo->scanStartTime += pInfo->minActiveTime;
            pInfo->wasTimeAdded   = TRUE;
        }

        mlmePrintf("mlmeBeaconReceive: New beacon found -- update bss set, channel %d, flags %x, bssSet.count: %d\n",
                   pNewBss->pChannel->channel,
                   pNewBss->pChannel->channelFlags,
                   pdevInfo->pscanInfo->bssSet.count);

        /* add or replace the existing entry */
        mlmeBsssetAdd(pdevInfo, pNewBss, &pdevInfo->pscanInfo->bssSet);
        A_DRIVER_FREE(pNewBss, sizeof(*pNewBss));

        if (pdevInfo->localSta->serviceType == WLAN_STA_SERVICE) {
            mlmeScanFunction(pdevInfo, SF_ADVANCE_NOW);
        }
    } /* TSTATE_SCANNING */
#ifdef BUILD_AP
    else {
        /* For 11g mode */
        if (IS_CHAN_G_OR_108G(pdevInfo->staConfig.pChannel->channelFlags)) {

            pRateSet = (RATE_SET *)
                ieSet.pInfoElement[eidMap[ELE_SUPPORTED_RATES]];
            pExtRateSet = (EXT_RATE_SET *)
                ieSet.pInfoElement[eidMap[ELE_EXT_SUPPORTED_RATES]];
            wlanElementToFullRateSet(pRateSet, pExtRateSet, &wlanBeaconRateSet);
            pNonErpElement = (NONERP_ELEMENT *)
                ieSet.pInfoElement[eidMap[ELE_NONERP]];

            /*
             * Check nonErp element if present, then check legacy 11b beacons
             * and 11g beacons with broken nonErp IE.
             */
            if ((pNonErpElement != NULL && pNonErpElement->info.present) ||
                (pNonErpElement == NULL &&
                 phyRateSubSetCheck(pdevInfo, WIRELESS_MODE_11g,
                                    &wlanBeaconRateSet, WLAN_PHY_CCK)))
            {
                pdevInfo->protectInfo.otherPresent = TRUE;
                pdevInfo->protectInfo.timePres = A_MS_TICKGET();
            }
        }
    }
#endif  /* BUILD_AP */
}

/*
 * mlmeProcessJoinFrame
 *
 * Processes either a Beacon or a Probe Response frame that
 * we received while joining a BSS, taking all of the pertinent
 * information from the frame and storing it all in its proper place.
 */
A_STATUS
mlmeProcessJoinFrame(WLAN_DEV_INFO *pDev, ATHEROS_DESC *pDesc,
                     INFO_ELEMENT_SET *pIeSet)
{
    WLAN_RATE_SET       combinedRateSet;
    WLAN_RATE_SET       beaconRateSet, wlanBeaconRateSet;
    RATE_SET            *pRateSet;
    EXT_RATE_SET        *pExtRateSet;
    WLAN_FRAME_BEACON   *pFrame;
    WPA_IE              *pWpaIe;
    ATHADVCAP_INFO      *pAthAdvCap = NULL;
    SIB_ENTRY           *pApSib;
    BSSDESCR            *pBss    = pDev->bssDescr;
    SIB_ENTRY           *pSib    = pDev->localSta;
    WLAN_STA_CONFIG     *pConfig = &pDev->staConfig;
    A_BOOL              bNewConnection = FALSE;
    CONN_ATTRIBUTES     connAttribs;
    A_UINT32            connAttribValue;
    A_UINT32            bssAttribValue;
    CONN_RATE_SET       connRateSet;

    ASSERT(pSib->transState & TSTATE_JOINING);
    ASSERT(pBss);

    /* Probe Response and Beacon structures are identical */
    pFrame = pDesc->pBufferVirtPtr.beacon;

    /* Is this frame from the BSS we want to join? */
    if (A_MACADDR_COMP(&pBss->bssId, &pFrame->macHeader.bssId) != 0) {
        return A_ERROR;
    }

    if ((pBss->bsstype == INFRASTRUCTURE_BSS && !pFrame->capInfo.ess) ||
        (pBss->bsstype == INDEPENDENT_BSS && !pFrame->capInfo.ibss))
    {
        /*
         * BSSID matches, but type of network doesn't. This can happen when
         * the same BSSID is used by an infrastructure BSS on one channel
         * and an IBSS on another.
         */
        mlmePrintf("mlmeProcessJoinFrame: Network type mismatch\n");
        return A_ERROR;
    }

    /* Complete the join process. */
    pSib->staState  |= STATE_JOINED;
    pSib->transState = TSTATE_QUIET;

    /* Copy timestamp and capability info */
    pSib->lastBeaconTime = pFrame->timestamp;
    pSib->capInfo.ess    = pFrame->capInfo.ess;
    pSib->capInfo.ibss   = pFrame->capInfo.ibss;

    /* Combine the the normal and extended rate information. */
    pRateSet    = (RATE_SET *)
        pIeSet->pInfoElement[eidMap[ELE_SUPPORTED_RATES]];
    pExtRateSet = (EXT_RATE_SET *)
        pIeSet->pInfoElement[eidMap[ELE_EXT_SUPPORTED_RATES]];
    wlanElementToFullRateSet(pRateSet, pExtRateSet, &combinedRateSet);

    /* Set the baseBss BSSID */
    A_MACADDR_COPY(&pFrame->macHeader.bssId, &pDev->baseBss.bssId);

    /*
     * Previously there was code which allowed the STA to be authenticated
     * and/or assiciated but not joined.  Ensure that this is no longer
     * allowed.
     */
    ASSERT(!(pSib->staState & (STATE_AUTH | STATE_ASSOC)));

    if (pBss->bsstype == INDEPENDENT_BSS) {
        IBSS_PARAM_SET *pIbssParams;

        /*
         * Set up privacy, if any.
         * TODO: This code should be centralized in security.c or
         *       somesuch.
         */
        if (pSib->privacyInvoked) {
            A_BOOL faulted;

            if (initCipher(pDev) != A_OK) {
                pSib->staState  &= ~STATE_JOINED;
                pSib->transState = TSTATE_QUIET;

                return A_ERROR;
            }
            setupSwEncryption(pDev);
        }

        /*
         * Adopt the ATIM window of the network we are joining.
         * This needs to be set before starting beacons
         * wdcStaJoinBss should have happened by now!
         * Make sure that the rest of the parameters are applied too!
         */
        pIbssParams = (IBSS_PARAM_SET *)
            pIeSet->pInfoElement[eidMap[ELE_IBSS_PARAM_SET]];
        if (pIbssParams && (pConfig->atimWindow != pIbssParams->atimWindow)) {
            pConfig->atimWindow = pIbssParams->atimWindow;
            bssAttribValue = (A_UINT32)pConfig->atimWindow;
            wdcUpdateBssAttribute(pDev->targetHandle, 0,
                                  ATIM_WINDOW,
                                  sizeof(bssAttribValue),
                                  (TARGET_CONFIG_VAL)&bssAttribValue);
        }
        ASSERT((!IS_CHAN_G(pDev->staConfig.pChannel->channelFlags)) || (!pDev->useShortSlotTime));

        /* Adopt the same basic rate set of the network we are joining */
        wlanElementToFullRateSet(pRateSet, NULL, &beaconRateSet);
        xSectionRateSet(&beaconRateSet, &pDev->baseBss.rateSet,
                        &wlanBeaconRateSet);
        wlanRateSetCopy(&wlanBeaconRateSet, &pDev->baseBss.rateSet);

        /* Start beacons and change our LED state. */
        wlanBeaconInit(pDev);
        wdcSetLedState(pDev->targetHandle,TRUE);
    } else {
        if (!wlanIsActiveScanAllowed(pDev)) {
            /*
             * Above routine will return FALSE in ETSI domain
             * Bring down STA shutoff time to 2 seconds. Closer to regulatory
             * value. If AP reboots because of radar, STA will give up the
             * connection in 2 seconds instead of the normal timeout.
             */
            pConfig->noBeaconTimeout = 2000;
        } else {
            pConfig->noBeaconTimeout = pDev->defaultStaConfig.noBeaconTimeout;
        }

        /*
         * We may not have received a beacon yet. Rather than test to see if
         * this is a beacon or not, we can simply require that we wait for
         * the next beacon to initialize things like the TIM offset, DTIM
         * period, etc.
         */
        if (!pDev->powerMgmt.needResyncToBeacon) {
            powerSetResyncToBeacon(pDev);
        }
    }

    /* Update our local bssid in the virt device */
    A_MACADDR_COPY(&pFrame->macHeader.bssId, &pDev->baseBss.bssId);

    /* We need to create an entry for the STA of this AP */
    pApSib = sibEntryFind(pDev, &pFrame->macHeader.srcAddr, NULL);
    if (pApSib == NULL) {
        bNewConnection = TRUE;
        pApSib = sibEntryAlloc(pDev, &pFrame->macHeader.srcAddr,
                               &pDev->baseBss);

        /* Add the AP/BSS SIB to the SIB hash table. */
        if (sibEntryAdd(pApSib) != WLAN_OK) {
            uiPrintf("mlmeBeaconReceive: ERROR: failed to add pApSib to hash"
                     " table\n");
            pSib->staState  &= ~STATE_JOINED;
            pSib->transState = TSTATE_QUIET;

            return A_ERROR;
        }
    }
    else {
        /* Clean up stale keys from the target */
        /* XXX Divy. Should abstract the test on connection validity */
        ASSERT(pApSib->wdcConnId < (A_INT32)(pDev->pSibTable->sibTableSize));
        if (pApSib->pPriv) {
            wlanDeleteConnectionKey(pDev, pApSib, pApSib->pPriv);
        }
    }

    pApSib->staState    = STATE_AP_UP;
    pApSib->transState  = TSTATE_QUIET;
    pApSib->serviceType = WLAN_AP_SERVICE;
    pApSib->capInfo     = pFrame->capInfo;

    /*
     * Update the working rate set to be used for communicating
     * to this AP. It's an intersection of the rates supported
     * by this device and the rates supported by the AP.  This is a
     * provisional setting as it may change during association.
     */
    xSectionRateSet(&combinedRateSet, &pDev->baseBss.rateSet,
                    &pApSib->workingRateSet);

    /* Set the wireless mode of the AP/BSS. */
    staWlanModeSet(pDev, pApSib);

    pWpaIe = (WPA_IE *)pIeSet->pWpaIe;
    if (pWpaIe) {
        if (VALID_WPA_ELEMENT(pWpaIe)) {
            pApSib->wpaEnabled      = TRUE;
            pApSib->uCipher         = pSib->uCipher;
            pApSib->mCipher         = pSib->mCipher;
            pApSib->authSel         = pSib->authSel;
            pApSib->wpaCapabilities = pSib->wpaCapabilities;
        } else {
            pApSib->wpaEnabled      = FALSE;
            pApSib->uCipher         = 0;
            pApSib->mCipher         = 0;
            pApSib->authSel         = 0;
            pApSib->wpaCapabilities = 0;
        }
    } else {
        pApSib->wpaEnabled      = FALSE;
        pApSib->mCipher         = 0;
        pApSib->authSel         = 0;
        pApSib->wpaCapabilities = 0;
    }

    if ((pIeSet->pAthCap != NULL) &&
         VALID_ATH_ADVCAP_ELEMENT(pIeSet->pAthCap))
    {
        pAthAdvCap = &pIeSet->pAthCap->info;
    }

    /* Create/Update target side connection */
    if (bNewConnection) {
        /* New connection => Create target side connection */
        A_MEM_ZERO(&connAttribs, sizeof(connAttribs));
        connAttribs.keytype             = PRIV_KEY_TYPE_NULL;

        connAttribs.compression         =
            (pAthAdvCap && pAthAdvCap->useCompression &&
             (pDev->staConfig.compressionSupport));

        connAttribs.longPreambleOnly    = 0;                /* TODO */
        connAttribs.currentTxRateAnt    = 0;                /* TODO */
        connAttribs.serviceType         = pApSib->serviceType;
        wlanRateSet2WdcRateSet(&pApSib->workingRateSet, &connAttribs.workingRateSet);
        connAttribs.wlanMode            = pApSib->wlanMode;

        wdcCreateConnection(pDev->targetHandle,
                            &pApSib->macAddr,
                            pApSib->wdcConnId,
                            0,
                            &connAttribs);
        NdisInterlockedIncrement(&pDev->numFlushCall);
        wdcFlush(pDev->targetHandle,pDev->disableEnableCounter);

    } else {

        /* Update existing connection */

        /* Service Type */
        connAttribValue = pApSib->serviceType;
        wdcUpdateConnectionAttribute(pDev->targetHandle,
                                     pApSib->wdcConnId,
                                     CONN_SERVICE_TYPE,
                                     sizeof(connAttribValue),
                                     (TARGET_CONFIG_VAL)&connAttribValue);


        /* WLAN mode */
        connAttribValue = pApSib->wlanMode;
        wdcUpdateConnectionAttribute(pDev->targetHandle,
                                     pApSib->wdcConnId,
                                     CONN_WLAN_MODE,
                                     sizeof(connAttribValue),
                                     (TARGET_CONFIG_VAL)&connAttribValue);


        /* Working Rate Set */
        wlanRateSet2WdcRateSet(&pApSib->workingRateSet, &connRateSet);
        wdcUpdateConnectionAttribute(pDev->targetHandle,
                                     pApSib->wdcConnId,
                                     CONN_WORKING_RATE_SET,
                                     sizeof(connRateSet),
                                     &connRateSet);

    }

    /*
     * Reserve a key slot for the connection.
     * In WEP case, plumb the actual key.
     */

    if (pDev->staConfig.privacyInvoked && !pDev->staConfig.wpaEnabled) {
        WLAN_PRIV_RECORD *pKey = NULL;
        int i;
        /* Install the BSS keys */
        for (i = 0; i < MAX_SHARED_KEYS; i++) {
            A_UINT32 isDefaultBssKey = 0;

            pKey = &pDev->keyTable[i];
            if (pKey->keyLength) {
                /* Reset the key type to pure WEP */
                pKey->keyType = PRIV_KEY_TYPE_WEP;
                pKey->keyFlags &= ~PRIV_CKIP_MIC;

                isDefaultBssKey = (pDev->staConfig.defaultKey == i);
                wlanInstallBssKey(pDev, i, isDefaultBssKey, pKey);
            }
        }
        uiPrintf("mlmeProcessJoinFrame: installing default key\n");
        pApSib->pPriv = &pDev->keyTable[pDev->staConfig.defaultKey];
        if (pApSib->pPriv != NULL) {
            WLAN_MACADDR nullMacAddr;
            /*
             * Do not set a MAC address for the transmit key.
             * The AP might decide to use another transmit key.
             */
            A_MEM_ZERO(&nullMacAddr, sizeof(nullMacAddr));
            wlanInstallConnectionKey(pDev,
                                     pApSib,
                                     &nullMacAddr,
                                     pApSib->pPriv);
        }
    } else {
        WLAN_PRIV_RECORD key;

        uiPrintf("mlmeProcessJoinFrame: installing dummy default key\n");
        A_MEM_ZERO(&key, sizeof(key));
        key.keyType = PRIV_KEY_TYPE_NULL;

        wlanInstallConnectionKey(pDev,
                                 pApSib,
                                 &pApSib->macAddr,
                                 &key);
    }

    /* Cancel the outstanding SME join timeout. */
    A_UNTIMEOUT(&pDev->smeTimer);

    mlmePrintf("Successfully joined, calling joinCallback\n");
    wlanMlmeJoinConfirm(pDev, MLME_SUCCESS);

    return A_OK;
}


/*
 * wlanMlmeJoinRequest - Join a BSS
 *
 * Spec. 10.3.3.1, 11.1.2.4
 * JOIN primitive logically establishes a binding to the AP.
 * It doesn't need to send any packet to the AP. It will turn on
 * the beacon delivery to driver and latch on to the next beacon.
 * MAC time should be synchronized with that beacon.
 * It is not clear if it needs to send a probe request or not.
 *
 * RETURNS: MLME_SUCCESS, MLME_INVALID_PARAM
 */

MLME_RESULT
wlanMlmeJoinRequest(WLAN_DEV_INFO *pDev, BSSDESCR *pBss, A_UINT32 timeout)
{
    A_BOOL          forceChanChange = FALSE;

    ASSERT(pBss);

    if (pBss == NULL) {
        return MLME_INVALID_PARAM;
    } else if (!pDev->staConfig.privacyInvoked &&
               pBss->capabilityInfo.privacy)
    {
        /* AP is asking for encryption, but we don't have it enabled. */
        mlmePrintSsid(&pBss->ssid);
        mlmePrintf(" wants encryption, and we don't have it, join failed\n");
        return MLME_INVALID_PARAM;
    }

    /* Figure out short/long preamble setting before trying to join. */
    pDev->localSta->capInfo.shortPreamble = pDev->staConfig.shortPreamble &&
                                            pBss->capabilityInfo.shortPreamble;

    mlmePrintf("wlanMlmeJoinRequest: chan=%d, SSID=\"",
               pBss->pChannel->channel);
    mlmePrintSsid(&pBss->ssid);
    mlmePrintf("\" @ time = %d ms\n", A_MS_TICKGET());

    /*
     * An 11g STA that wants to associate to an 11b AP will configure itself
     * as an 11b STA so cwmin and other params are appropriate for an 11b BSS.
     */
    if (IS_CHAN_NON_108G_2GHZ(pBss->pChannel->channelFlags) &&
        pDev->staConfig.NetBand & MODE_SELECT_11G)
    {
        /*
         * Force drvChangeChannel to change, even if pBss->pChannel and
         * pDev->staConfig.phwChannel point to same place.
         */
        forceChanChange = TRUE;

        if ((pBss->opMode == WIRELESS_MODE_11b) &&
            IS_CHAN_OFDM(pBss->pChannel->channelFlags))
        {
            /* 11g STA trying to join an 11b AP, change to 11b channel. */
            pBss->pChannel->channelFlags &= ~CHANNEL_OFDM;
            pBss->pChannel->channelFlags |= CHANNEL_CCK;
            mlmePrintf("wlanMlmeJoinRequest: force g sta -> b ap\n");
        } else if (pBss->opMode == WIRELESS_MODE_11g &&
                   IS_CHAN_CCK(pBss->pChannel->channelFlags))
        {
            /* 11g STA changed to 11b now trying to join an 11g AP, change back to 11g channel. */
            pBss->pChannel->channelFlags &= ~CHANNEL_CCK;
            pBss->pChannel->channelFlags |= CHANNEL_OFDM;
            mlmePrintf("wlanMlmeJoinRequest: force g sta -> b ap -> g ap\n");
        }
    }

    /* Change to the BSS's channel */
    if (drvChangeChannel(pDev, pBss->pChannel, forceChanChange, FALSE) != A_OK) {
        /* Noisy channel */
        mlmePrintf("wlanMlmeJoinRequest: drvChangeChannel() failed. Busy channel?\n");
        return MLME_INVALID_PARAM;
    }

    /* Record the fact that we're now on our "assigned" channel. */
    pDev->staConfig.pChannel = pDev->staConfig.phwChannel;
#if NDIS_WDM
    /*
     * (LW) Attention !!!
     * This is temp hack, should revisit later
     */
    pDev->invalidTxChannel   = FALSE;
#else
    HACK_TO_PARDEV(pDev)->invalidTxChannel   = FALSE;
#endif

    /*
     * Since this BSS descr can come from any outside context, we need to make
     * our own copy and store it.
     */
    if (pDev->bssDescr == NULL) {
        pDev->bssDescr = (BSSDESCR *)A_DRIVER_MALLOC(sizeof(*pBss));
        if (pDev->bssDescr == NULL) {
            mlmePrintf("wlanMlmeJoinRequest: Could not alloc mem for bssDescr!\n");
            return MLME_DRIVER_ERROR;
        }
    }

    *pDev->bssDescr = *pBss;

    /* Initialize encryption */
    if (initCipher(pDev) != A_OK) {
        mlmePrintf("wlanMlmeJoinRequest: initCipher() failed!\n");
        return MLME_DRIVER_ERROR;
    }

    if (!VALID_CF_ELEMENT(&pBss->cfParamSet)) {
        pBss->cfParamSet.cfpPeriod      = 0;
        pBss->cfParamSet.cfpMaxDuration = 0;
    }

    wlanStaJoinBss(pDev);

    ASSERT(!(pDev->localSta->staState & (STATE_AUTH | STATE_ASSOC)));
    pDev->localSta->transState = TSTATE_JOINING;

    /*
     * If we're allowed, send a Probe Request to the BSS that we're joining.
     * If not, we'll wait for a beacon, instead.
     */
    if (!IS_CHAN_PASSIVE(pDev->staConfig.pChannel->channelFlags)) {
        mlmePrintf("wlanMlmeJoinRequest: sending Probe Request\n");
        if (wlanMlmeProbeRequest(pDev, &pBss->bssId, &pBss->ssid) !=
            MLME_SUCCESS)
        {
            /* Something wrong with probe transmission. */
            mlmePrintf("wlanMlmeJoinRequest: ERROR: Probe Request failed!\n");
        }
    }

    /* Set a timer to cancel after joinFailureTimeout */
    A_TIMEOUT(&pDev->smeTimer, timeout, (A_UINT32)pDev, MLME_JOIN_TIMEOUT);

    return MLME_SUCCESS;
}

void
wlanMlmeJoinConfirm(WLAN_DEV_INFO *pdevInfo, MLME_RESULT result)
{
    /* If callback routine exists, call it */
    if (pdevInfo->pSmeInfo->joinCallback) {
        (*pdevInfo->pSmeInfo->joinCallback)(pdevInfo, result);
    }
}

/*
 * mlmeScanFunction
 */
void
mlmeScanFunction(WLAN_DEV_INFO *pDev, SF_ENUM mode)
{
    A_STATUS    status        = A_OK;
    A_BOOL      scanPassively = TRUE;
    A_BOOL      finished      = FALSE;
    A_BOOL      doAdvance     = TRUE;
    CHAN_VALUES *pCurChan     = NULL;
    SCAN_INFO   *pInfo        = pDev->pscanInfo;
    A_INT32     i;

    ASSERT(pInfo);

    mlmePrintf("%d.%03d | mlmeScanFunction: ", A_MS_TICKGET() / 1000,
               A_MS_TICKGET() % 1000);

    if (pInfo->newScan) {
        mlmePrintf("New scan, ");

        pInfo->newScan = FALSE;

        /* Don't skip past the first channel */
        doAdvance = FALSE;
    } else if (mode == SF_DONT_ADVANCE_NOW) {
        /*
         * We are restarting the scan, having previously stopped the scan
         * mid-stream.  Ensure that we completely scan the channel that was
         * previously interrupted, even if it means scanning it again.
         */
        doAdvance = FALSE;
    } else {
        A_INT32 timeSpent = A_MS_TICKGET() - pInfo->scanStartTime;
        A_INT32 timeLeft  = pInfo->minTime - timeSpent;

        if (timeLeft > 10) { // 10 ms resolution is best we can get from A_TIMEOUT()
            /*
             * We haven't spent enough time on the current channel, yet.
             * Reset the timer to wait for the time remaining in order
             * to spend at least minChannelTime on this channel.
             */
            mlmePrintf("need to wait longer on this channel\n");
            A_TIMEOUT(&pDev->smeTimer, (A_UINT32)timeLeft, (A_UINT32)pDev,
                      MLME_SCAN_TIMEOUT);
            return;
        }

        /* Cancel any previous scan timers. */
        A_UNTIMEOUT(&pDev->smeTimer);

    }

    /* Change channel until there is no interference detected. */
    while (!finished) {
        ASSERT(pInfo->curChannelIndex < pDev->staConfig.pClist->listSize);

        /* Advance our index to the next channel. */
        if (doAdvance) {
            pInfo->curChannelIndex++;
        } else {
            doAdvance = TRUE;
        }

        mlmePrintf("next chan %d/%d", pInfo->curChannelIndex + 1,
                   pInfo->pScanList->listSize);

        if (pInfo->curChannelIndex >= pInfo->pScanList->listSize) {
            /* We finished last channel on the list. */
            finished = TRUE;
            break;
        }

        pCurChan = pInfo->pScanList->pChanArray[pInfo->curChannelIndex];

        mlmePrintf(" %d%s %s  \n", pCurChan->channel,(IS_CHAN_5GHZ(pCurChan->channelFlags)? "A" :
                               (IS_CHAN_G(pCurChan->channelFlags)?"G":"B")),
                               IS_CHAN_PASSIVE(pCurChan->channelFlags)?"P":"A") ;

        status = drvChangeChannel(pDev, pCurChan, FALSE, FALSE);

        if (!(pCurChan->channelFlags & CHANNEL_CW_INT)) {
            break;
        }

        mlmePrintf("mlmeScanFunction: Interference on %d, ", pCurChan->channel);
    }

    if (finished) {
        mlmePrintf(", finished\n");

        /*
         * Confirm that the scan is complete, and call any callback routines
         * provided by the caller
         */
        wlanMlmeScanConfirm(pDev, MLME_SUCCESS, FALSE);
        return;
    }

    /*
     * Change state to scanning only after changing the channel, otherwise
     * some beacons from old channel may get recorded as on new channel
     */
    pDev->localSta->transState |= TSTATE_SCANNING;
    pInfo->scanStartTime        = A_MS_TICKGET();
    pInfo->wasTimeAdded         = FALSE;

    /* Now we get into the scanning process */

    scanPassively = pInfo->scantype == PASSIVE_SCAN ||
                    IS_CHAN_PASSIVE(pCurChan->channelFlags);

    mlmePrintf("mlmeScanFunction -- %s scan channel %d (%d MHz)\n",
               scanPassively ? "passive" : "active",
               wlanConvertGHztoCh(pCurChan->channel, pCurChan->channelFlags),
               pCurChan->channel);

    if (!scanPassively) {
        SSID blankSsid;

        blankSsid.elementID = ELE_SSID;
        blankSsid.length    = 0;

        /* Send out a directed Probe Request if we have valid SSIDs. */
        for (i = 0; i < A_ATOMIC_READ(&pInfo->desSsids.count); i++) {
            if (pInfo->desSsids.elements[i].length > 0) {
                wlanMlmeProbeRequest(pDev, &broadcastMacAddr,
                                     &pInfo->desSsids.elements[i]);
            }
        }

        /* Send out a broadcast Probe Request. */
        if (wlanMlmeProbeRequest(pDev, &broadcastMacAddr, &blankSsid) !=
            MLME_SUCCESS)
        {
            /* Something wrong with probe transmission. Skip */
            mlmePrintf("mlmeScanFunction: ERROR: failed probe request on channel %x\n",
                       pCurChan->channel);
        }

        pInfo->maxTime = pInfo->maxActiveTime;
        pInfo->minTime = pInfo->minActiveTime;
    } else {
        pInfo->maxTime = pInfo->maxPassiveTime;
        pInfo->minTime = pInfo->minPassiveTime;
    }

    A_TIMEOUT(&pDev->smeTimer, pInfo->maxTime, (A_UINT32)pDev, MLME_SCAN_TIMEOUT);
}

/*
 * wlanMlmeScanRequest - scan for potential BSSs.
 * MLME SCAN primitive
 *
 * Spec section 10.3.2.1
 *
 * PARAMETERS:
 *  bsstype        - INFRASTRUCURE_BSS or INDEPENDENT_BSS or both. It will accept
 *                     beacons from either or both.
 *  bssid          - A specific BSSID to scan for or broadcast MAC address for
 *                   accepting all of them.
 *  ssidStr        - Name of Service set. 32 bytes max. NULL value means accept any
 *                     SSID.
 *  scanType       - ACTIVE_SCAN or PASSIVE_SCAN
 *  channelList    - List of channels to scan.
 *  minChannelTime - Listen to each channel for at least this long.
 *  maxChannelTime - TU. Must switch to next channel after this time.
 *
 * RETURNS: MLME_SUCCESS or MLME_INVALID_PARAM
 */
MLME_RESULT
wlanMlmeScanRequest(WLAN_DEV_INFO     *pdevInfo,
                    BSSTYPE           bsstype,
                    BSSID             *pBssid,
                    SSID_ARRAY        *pSsids,
                    SCANTYPE          scantype,
                    WLAN_SCAN_LIST    *pScanList,
                    A_UINT32          minActiveTime,
                    A_UINT32          maxActiveTime,
                    A_UINT32          minPassiveTime,
                    A_UINT32          maxPassiveTime,
                    A_BOOL            isTempScan)
{
    SCAN_INFO *pInfo = pdevInfo->pscanInfo;

    ASSERT(pScanList);

    mlmePrintf("wlanMlmeScanRequest: Entered\n");

    if (pdevInfo->localSta->transState & TSTATE_SCANNING) {
        /* We're already scanning.  Reject the request. */
        return MLME_TOOMANY_REQ;
    }

    /*
     * Only allow up to one temporary scan, and if this is the first scan,
     * then it's not temporary, is it.
     */
    if (isTempScan) {
        if (!pInfo || pInfo->pLastInfo) {
            /*
             * By clearing isTempScan, we will allow the previous scan in
             * progress to continue.
             */
            isTempScan = FALSE;
        }
    }

    if (!pInfo || (isTempScan && !pInfo->newScan)) {
        pInfo = (SCAN_INFO *)A_DRIVER_MALLOC(sizeof(*pInfo));
        if (!pInfo) {
            mlmePrintf("wlanMlmeScanRequest: Failed to allocate memory for "
                       "scan info\n");
            return MLME_REFUSED;
        }

        /* Save off the previous information if this is a temporary scan */
        pInfo->pLastInfo    = isTempScan ? pdevInfo->pscanInfo : NULL;
        pInfo->newScan      = TRUE;

        pdevInfo->pscanInfo = pInfo;
    } else {
        WLAN_SCAN_LIST **ppList;

        /*
         * Since a scan list was allocated before calling this function
         * we need to either free the previous scan list if this is a new
         * scan, or free the one that was passed in because we want to
         * ignore it if we're in the middle of our current scan.
         */
        if (pInfo->newScan) {
            ppList = &pInfo->pScanList;
        } else {
            ppList = &pScanList;
        }

        wlanFreeScanList(*ppList);
        ppList = NULL;
    }

    if (pInfo->newScan) {
        if (pScanList == NULL) {
            mlmePrintf("wlanMlmeScanRequest: Scan List cannot be null for a new scan \n");
            return MLME_REFUSED;
        }
        pInfo->bssSet.count     = 0;
        pInfo->curChannelIndex  = 0;
        pInfo->bsstype          = bsstype;
        pInfo->scantype         = scantype;
        pInfo->pScanList        = pScanList;
        pInfo->bssId            = *pBssid;
        pInfo->maxActiveTime    = maxActiveTime;
        pInfo->minActiveTime    = minActiveTime;
        pInfo->maxPassiveTime   = maxPassiveTime;
        pInfo->minPassiveTime   = minPassiveTime;
        pInfo->wasTimeAdded     = FALSE;

        ASSERT(pScanList->listSize <= pdevInfo->staConfig.pClist->listSize);

        A_MEM_ZERO((char *)&pInfo->bssSet.bssDescrArray,
                   sizeof(pInfo->bssSet.bssDescrArray));
    }

    if (pSsids != NULL) {
        pInfo->desSsids = *pSsids;
    } else {
        mlmePrintf("wlanMlmeScanRequest: Allowing association to ANY SSID.\n");
        ssidArrayInit(&pInfo->desSsids);
        ssidArrayAdd(&pInfo->desSsids, "", 0);

    }

    /* Kick off the scan, making sure we scan the next channel in the list. */
    mlmeScanFunction(pdevInfo, SF_DONT_ADVANCE_NOW);

    return MLME_SUCCESS;
}

/*
 * wlanMlmeScanConfirm
 *
 * It's called when a scan completes or is interrupted.
 *
 */
void
wlanMlmeScanConfirm(WLAN_DEV_INFO *pDev, MLME_RESULT result,
                    A_BOOL interrupted)
{
    SCAN_INFO *pInfo = pDev->pscanInfo;
    A_BOOL    done   = FALSE;

    mlmePrintf("> mlmeScanConfirm\n");

    pDev->localSta->transState = TSTATE_QUIET;

    if (!pInfo) {
        mlmePrintf("< mlmeScanConfirm: SCAN_INFO struct is NULL\n");
        return;
    }

    mlmePrintf("mlmeScanConfirm: Returning to channel %d\n",
               pDev->staConfig.pChannel->channel);

    /* Switch back to our original channel. */
    drvChangeChannel(pDev, pDev->staConfig.pChannel, FALSE, FALSE);

    if (!interrupted) {
        done = wlanMlmeScanDone(pDev, FALSE);

        /*
         * The above call may have reverted to the previous scan info struct
         * so we have to re-grab our pointer here. Assert that any temporary
         * scan info structs are gone by this point.
         */
        pInfo = pDev->pscanInfo;
        ASSERT(pInfo->pLastInfo == NULL);
    }

    /* If callback routine exists, call it. */
    if (pDev->pSmeInfo->scanCallback) {
        (*pDev->pSmeInfo->scanCallback)(pDev, &pInfo->bssSet, result,
                                        interrupted, done);
    }

    mlmePrintf("< mlmeScanConfirm\n");
}

/*
 * wlanMlmeScanDone
 *
 * It's called when a scan completes, or to cancel a scan in progress.
 * Returns TRUE if a non-temporary scan completes
 *
 */
A_BOOL
wlanMlmeScanDone(WLAN_DEV_INFO *pDev, A_BOOL cancelAll)
{
    SCAN_INFO *pInfo   = pDev->pscanInfo;
    A_BOOL    complete = FALSE;

    mlmePrintf("> wlanMlmeScanDone: cancelAll = %s\n",
               cancelAll ? "TRUE" : "FALSE");

    /*
     * Background scans don't occur before foreground scans today, but some
     * day they may.
     */
    if (!pInfo) {
        mlmePrintf("< wlanMlmeScanDone: CSERV_INFO struct is NULL\n");
        return FALSE;
    }

    /*
     * If this is the end of a "temporary" scan, then restore the previous
     * scan information so that the interrupted scan can continue where
     * it left off.
     */
    if (pInfo->pLastInfo) {
        SCAN_INFO *pLastInfo = pInfo->pLastInfo;
        int       i;

        /*
         * We need to copy over the BSS information into the previous info
         * struct before freeing this info struct.  Otherwise we'll lose that
         * information as it hasn't yet been relayed back to the upper layer.
         */
        mlmePrintf("wlanMlmeScanDone: Copying %d BSSes from temp to main bss set list.\n",
                 pInfo->bssSet.count);
        for (i = 0; i < pInfo->bssSet.count; i++) {
            mlmeBsssetAdd(pDev, &pInfo->bssSet.bssDescrArray[i],
                          &pLastInfo->bssSet);
        }

        A_DRIVER_FREE(pInfo, sizeof(*pInfo));
        pInfo = pDev->pscanInfo = pLastInfo;
    } else {
        /* The next scan will start off anew */
        pInfo->newScan = TRUE;
        complete       = TRUE;

    }

    /*
     * The cancelAll flag is used once we successfully connect so that our
     * first background scan will always start anew.
     */
    if (cancelAll) {
        pInfo->newScan = TRUE;
    }

    mlmePrintf("< wlanMlmeScanDone: complete = %s\n",
               complete ? "TRUE" : "FALSE");

    return complete;
}

/*
 * wlanMlmeProbeRequest
 *
 * This routine sends out a probe packet. Not required
 * by MLME, but used internally by algorithms.
 * This routine can be called for active scan even before we have a valid
 * ssid or bssid in the SIB of local station. Hence, those parameters are passed in.
 */
MLME_RESULT
wlanMlmeProbeRequest(WLAN_DEV_INFO *pdevInfo,
                     WLAN_MACADDR  *bssid,
                     SSID          *ssid)
{
    WLAN_FRAME_PROBE_REQ *pFrame;
    SIB_ENTRY            *pSib = NULL;
    A_UINT16             frameLen;
    ATHEROS_DESC         *pDesc;
    INFO_ELEMENT         *pTempIe;
    RATE_SET             rateSet;
    EXT_RATE_SET         extRateSet;

    mlmeL2Printf("wlanMlmeProbeRequest -- channel %d\n",
               pdevInfo->staConfig.phwChannel->channel);

    if (!isGrp(bssid)) {
        pSib = sibEntryFind(pdevInfo, bssid, NULL);
    }

    /* Convert WLAN_RATE_SET (internal representation) to RATE_SET/EXT_RATE_SET (802.11) */
    wlanFullToRateSet(&pdevInfo->baseBss.rateSet, &rateSet,
                      pdevInfo->staConfig.gdraft5);
    wlanFullToExtRateSet(&pdevInfo->baseBss.rateSet, &extRateSet,
                         pdevInfo->staConfig.gdraft5);

    /* Compute frame length. No challenge text */
    /* Don't use INFO_ELEMENT_SIZE macro for size of ssid, it can be null */

    frameLen = sizeof(WLAN_MGT_MAC_HEADER) +
               (ssid->length + sizeof(ssid->elementID) + sizeof(ssid->length)) +
               INFO_ELEMENT_SIZE(rateSet) +
               INFO_ELEMENT_SIZE(extRateSet);

    if ((createBuffandDesc(pdevInfo, frameLen, &pDesc)) != 0) {
        /* Failed to allocate buffer for frame and descriptor */
        return MLME_DRIVER_ERROR;
    }

    pFrame = pDesc->pBufferVirtPtr.probe;
    mgtFrameHdrInit(&pFrame->macHeader, SUBT_PROBE_REQ, bssid);
    A_MACADDR_COPY(bssid, &pFrame->macHeader.bssId);
    A_MACADDR_COPY(&pdevInfo->localSta->macAddr, &pFrame->macHeader.srcAddr);

    pTempIe = mlmeElementCopy((INFO_ELEMENT *)ssid,
                              (INFO_ELEMENT *)&pFrame->ssid);
    pTempIe = mlmeElementCopy((INFO_ELEMENT *)&rateSet,
                              (INFO_ELEMENT *)pTempIe);

    if (VALID_ERS_ELEMENT(&extRateSet)) {
        pTempIe = mlmeElementCopy((INFO_ELEMENT *)&extRateSet,
                                   (INFO_ELEMENT *)pTempIe);
    }

#ifdef DEBUG_11B_PROBE_REQUESTS
    /*
     * The following is useful when debugging scanning/probing in 802.11b
     * as there is no channel number in the Probe Request but there is in
     * the Probe Response, and frames can be heard across neighboring
     * channels.
     */
    frameLen += 3;
    pTempIe->elementID = ELE_VENDOR_PRIVATE;
    pTempIe->length    = 1;
    pTempIe->value[0]  = (A_CHAR)wlanConvertGHztoCh(pdevInfo->staConfig.phwChannel->channel,
                                            pdevInfo->staConfig.phwChannel->channelFlags);
#endif

    if (mlmeDebugLevel > 10) {
        displayMgtFrame(&pFrame->macHeader, frameLen);
    }

    pDesc->pDestSibEntry = pSib;
    pDesc->pOpBss        = &pdevInfo->baseBss;
    SETQHANDLE(pDesc, TXQ_ID_FOR_AC1);

    pDesc->bIndicateXmitStatus = TRUE;
    if (powerFrameTransmit(pdevInfo, pDesc) == A_OK) {
        return MLME_SUCCESS;
    }

    return MLME_DRIVER_ERROR;
}


/*
 * wlanMlmeProbeRequestReceive
 *
 * This routine receives a probe request on AP. Not required
 * by MLME, but used internally by algorithms.
 */
void
mlmeProbeRequestReceive(WLAN_DEV_INFO        *pdevInfo,
                        OP_BSS               *pOpBss,
                        WLAN_FRAME_PROBE_REQ *probeReqFrame,
                        A_UINT16             frameLen)
{
    WLAN_FRAME_PROBE_RESP *pFrame;
    A_UINT16              newFrameLen;
    ATHEROS_DESC          *pDesc;
    SSID                  *pSsid = NULL;
    INFO_ELEMENT          *pTempIe;
    COUNTRY_INFO_LIST     countryInfo;
    RATE_SET              *pRateSet;
    EXT_RATE_SET          *pExtRateSet;
    A_BOOL                isGMode = FALSE;
    RATE_SET              thisRateSet;
    EXT_RATE_SET          thisExtRateSet;
    WLAN_RATE_SET         wlanStaRateSet;
    A_UINT8               *pInfoElements = NULL;
    A_UINT16              infoElemLen = 0;
    INFO_ELEMENT_SET      ieSet;

    /* Check if our AP function is on */
    if ((!(pdevInfo->localSta->staState & STATE_AP_UP)) &&
        (pdevInfo->bssDescr->bsstype == INFRASTRUCTURE_BSS)) {
        return;
    }
    /* Check if we are joined in adhoc */
    if ((!(pdevInfo->localSta->staState & STATE_JOINED)) &&
        (pdevInfo->bssDescr->bsstype == INDEPENDENT_BSS)) {
        return;
    }

    /* Don't respond to mlmeProbeRequest when off channel */
#ifndef BUILD_AP
#if NDIS_WDM
    /*
     * (LW) Attention !!!
     * This is temp hack, should revisit later
     */
    if (pdevInfo->invalidTxChannel) {
        return;
    }
#else
    if (HACK_TO_PARDEV(pdevInfo)->invalidTxChannel) {
        return;
    }
#endif
#endif

    /* Find and store the Probe Request elements */
    A_MEM_ZERO(&ieSet, sizeof(ieSet));
    pInfoElements = (A_UINT8 *)&probeReqFrame->ssid;
    infoElemLen = (A_UINT16)(frameLen - (pInfoElements -
                            (A_UINT8 *)&probeReqFrame->macHeader));
    mgtGetInfoElements(&ieSet,(INFO_ELEMENT *)pInfoElements,infoElemLen);
    pSsid     = (SSID *)ieSet.pInfoElement[eidMap[ELE_SSID]];


    /* don't response to the STA with NULL SSID when SSID suppression enabled */
    if ( pdevInfo->staConfig.ssidSuppressEnabled &&
         (pSsid == NULL || pSsid->length == 0) )
    {
        return;
    }

    if (pSsid != NULL &&
        pSsid->length > 0 &&
        (pdevInfo->bssDescr->ssid.length != pSsid->length ||
         A_BCOMP(&pSsid->ssid, &pdevInfo->bssDescr->ssid.ssid,
                 pSsid->length) != 0))
    {
        /* Requested SSID is not null but it doesn't match ours */
        return;
    }

    /* Avoid sending probe response to broadcast address, causes TI sta failure in WiFi */
    /* 802.11 spec section 11.1.3.2.1 wants probe responses to be unicast */

    if (A_MACADDR_COMP(&broadcastMacAddr, &probeReqFrame->macHeader.srcAddr) == 0) {
        return;
    }

    isGMode     = IS_CHAN_G_OR_108G(pdevInfo->staConfig.pChannel->channelFlags);
    pRateSet    = (RATE_SET *)
        ieSet.pInfoElement[eidMap[ELE_SUPPORTED_RATES]];
    pExtRateSet = (EXT_RATE_SET *)
        ieSet.pInfoElement[eidMap[ELE_EXT_SUPPORTED_RATES]];

    wlanElementToFullRateSet(pRateSet, pExtRateSet, &wlanStaRateSet);

    wlanFullToRateSet((WLAN_RATE_SET *)&pOpBss->rateSet, &thisRateSet,
                      pdevInfo->staConfig.gdraft5);

    thisExtRateSet.length = 0;  /* invalidate before any use */
    if (IS_CHAN_G(pdevInfo->staConfig.pChannel->channelFlags)) {
        wlanFullToExtRateSet((WLAN_RATE_SET *)&pOpBss->rateSet,
                             &thisExtRateSet, pdevInfo->staConfig.gdraft5);
        /*
         * Some vendors' 11b clients cannot support extended supported rate set.
         * Also, if 11g draft 5.0 compatibility, truncate supported rate set.
         */
        if (!phyRateSubSetCheck(pdevInfo, WIRELESS_MODE_11g, &wlanStaRateSet,
                                WLAN_PHY_OFDM)) {
            thisExtRateSet.length = 0;
            if (pdevInfo->staConfig.gdraft5 &&
                thisRateSet.length > ELE_RATE_SIZE_11B)
            {
                thisRateSet.length = ELE_RATE_SIZE_11B;
            }
        }
    }

    /* Compute our response length */
    newFrameLen = sizeof(WLAN_MGT_MAC_HEADER) + sizeof(WLAN_TIMESTAMP) +
                  sizeof(pFrame->beaconInterval) + sizeof(CAP_INFO) +
                  INFO_ELEMENT_SIZE(pdevInfo->bssDescr->ssid) +
                  INFO_ELEMENT_SIZE(thisRateSet) +
                  INFO_ELEMENT_SIZE(pdevInfo->bssDescr->dsParamSet) +
                  (VALID_CF_ELEMENT(&pdevInfo->bssDescr->cfParamSet) ?
                   INFO_ELEMENT_SIZE(pdevInfo->bssDescr->cfParamSet) : 0) +
                  ((pdevInfo->bssDescr->bsstype == INDEPENDENT_BSS) ?
                   INFO_ELEMENT_SIZE(pdevInfo->bssDescr->ibssParamSet) : 0);

    mlmeCountryCodeElementFill(pdevInfo, &countryInfo);
    newFrameLen += mlmeCountryCodeElementLength(pdevInfo, &countryInfo);

    newFrameLen += isGMode ? sizeof(NONERP_ELEMENT) : 0;

    newFrameLen += isGMode ? INFO_ELEMENT_SIZE(thisExtRateSet) : 0;

    if (pdevInfo->staConfig.wpaEnabled) {
        newFrameLen += (VALID_WPA_ELEMENT(&pdevInfo->bssDescr->wpaIe) ?
                       INFO_ELEMENT_SIZE(pdevInfo->bssDescr->wpaIe) : 0);
    }

    newFrameLen += (VALID_ATH_ADVCAP_ELEMENT(&pdevInfo->bssDescr->athAdvCapElement) ?
                   INFO_ELEMENT_SIZE(pdevInfo->bssDescr->athAdvCapElement) : 0);

    newFrameLen += athWmeIeLength(pdevInfo);
#ifdef WME
    newFrameLen += wmeV1IeLength(pdevInfo);
#endif

    if ((createBuffandDesc(pdevInfo, newFrameLen, &pDesc)) != 0) {
        /* Failed to allocate buffer for frame and descriptor */
        return;
    }
    pFrame = pDesc->pBufferVirtPtr.probeResp;

    mgtFrameHdrInit(&pFrame->macHeader, SUBT_PROBE_RESP,
                    &probeReqFrame->macHeader.srcAddr);
    A_MACADDR_COPY(&pOpBss->bssId, &pFrame->macHeader.bssId);
    A_MACADDR_COPY(
        (pdevInfo->localSta->serviceType == WLAN_AP_SERVICE) ?
        &pOpBss->bssId : &pdevInfo->localSta->macAddr,
        &pFrame->macHeader.srcAddr);

    pFrame->beaconInterval = pdevInfo->bssDescr->beaconInterval;
    pFrame->capInfo        = pdevInfo->bssDescr->capabilityInfo;

    pTempIe = (INFO_ELEMENT *)&pFrame->buffer;
    pTempIe = mlmeElementCopy((INFO_ELEMENT *)
                              &pdevInfo->bssDescr->ssid, pTempIe);
    pTempIe = mlmeElementCopy((INFO_ELEMENT *)&thisRateSet, pTempIe);
    pTempIe = mlmeElementCopy((INFO_ELEMENT *)
                              &pdevInfo->bssDescr->dsParamSet, pTempIe);

    if (VALID_CF_ELEMENT(&pdevInfo->bssDescr->cfParamSet)) {
        pTempIe = mlmeElementCopy((INFO_ELEMENT *)
                                  &pdevInfo->bssDescr->cfParamSet, pTempIe);
    }
    if (pdevInfo->bssDescr->bsstype == INDEPENDENT_BSS) {
        pTempIe = mlmeElementCopy((INFO_ELEMENT *)
                                  &pdevInfo->bssDescr->ibssParamSet, pTempIe);
    }

    /* Put the country code element after the TIM */
    if (pdevInfo->staConfig.multiDomainCapEnabled) {
        pTempIe = mlmeElementCopy((INFO_ELEMENT *)&countryInfo, pTempIe);
    }

    /*
     *  Some WPA STAs can't yet grok the NONERP element
     *  appearing before the WPA element, so insert the latter
     *  first.
     */
    if (VALID_WPA_ELEMENT(&pdevInfo->bssDescr->wpaIe)) {
        pTempIe = mlmeElementCopy((INFO_ELEMENT *)
                                  &pdevInfo->bssDescr->wpaIe, pTempIe);
    }

    /* put NONERP element after country list */
    if (isGMode && pdevInfo->bssDescr->nonErpElement.length) {
        pTempIe = mlmeElementCopy((INFO_ELEMENT *)
                &pdevInfo->bssDescr->nonErpElement, pTempIe);
    }

    if (isGMode && VALID_ERS_ELEMENT(&thisExtRateSet)) {
        pTempIe = mlmeElementCopy((INFO_ELEMENT *)&thisExtRateSet, pTempIe);
    }

    if (pdevInfo->bssDescr->athAdvCapElement.length) {
        pTempIe = mlmeElementCopy((INFO_ELEMENT *)&pdevInfo->bssDescr->athAdvCapElement, pTempIe);
    }

    pTempIe = athWmeIeGenerate(pdevInfo, pOpBss, pTempIe);

#ifdef WME
    pTempIe = wmeV1IeGenerate (pdevInfo, pOpBss, pTempIe);
#endif

    if (mlmeDebugLevel > 10) {
        displayMgtFrame(&pFrame->macHeader, newFrameLen);
    }

    pDesc->pDestSibEntry = NULL;
    pDesc->pOpBss        = pOpBss;
    SETQHANDLE(pDesc, TXQ_ID_FOR_AC1);
    pDesc->bIndicateXmitStatus = TRUE;
    powerFrameTransmit(pdevInfo, pDesc);
}

void
mlmeProbeRespReceive(WLAN_DEV_INFO *pdevInfo, ATHEROS_DESC *pDesc)
{
    WLAN_FRAME_PROBE_RESP *bFrame;
    BSSDESCR              *pBss;
    A_UINT16              frameLen;
    DS_PARAM_SET          *pDsParams;
    INFO_ELEMENT_SET      ieSet;
    BSSTYPE               bssType;
    WLAN_RATE_SET         wlanApRateSet;
    SCAN_INFO             *pScanInfo      = pdevInfo->pscanInfo;
    SSID                  *pSsid          = NULL;
    RATE_SET              *pRateSet       = NULL;
    EXT_RATE_SET          *pExtRateSet    = NULL;
    A_UINT8               *pInfoElements  = NULL;
    A_UINT16              infoElemLen     = 0;
    INFO_ELEMENT          *pWpaIe         = NULL;
    NONERP_ELEMENT        *pNonErpElement = NULL;
    ATH_ADVCAP_IE         *pAthCap        = NULL;
    AIRONET_IE            *aie            = NULL;
    TIM_ELEMENT           *pTim           = NULL;
    A_UINT32              oldVal;
    A_UINT32              bssAttribValue;
#ifdef WME
    SIB_ENTRY             *pSib           = NULL;
#endif

    bFrame    = pDesc->pBufferVirtPtr.probeResp;
    frameLen  = pDesc->status.rx.dataLength;

    /* Find and store the beacon elements */
    A_MEM_ZERO(&ieSet,sizeof(ieSet));
    pInfoElements = (A_UINT8 *)&(bFrame->buffer);
    infoElemLen   = (A_UINT16)(frameLen - (pInfoElements -
                               (A_UINT8 *)&bFrame->macHeader));
    mgtGetInfoElements(&ieSet,(INFO_ELEMENT *)pInfoElements,infoElemLen);

    if (pdevInfo->localSta->transState & TSTATE_JOINING) {
        mlmeProcessJoinFrame(pdevInfo, pDesc, &ieSet);
        return;
    } else

    if (!(pdevInfo->localSta->transState & TSTATE_SCANNING)) {
        /* not joining or scanning --> ignore this frame */
        return;
    }

    ASSERT(pScanInfo != NULL);

    /* Basic sanity checks */
    pSsid     = (SSID *)ieSet.pInfoElement[eidMap[ELE_SSID]];
    pDsParams = (DS_PARAM_SET *)
                ieSet.pInfoElement[eidMap[ELE_DS_PARAM_SET]];
    bssType   = (bFrame->capInfo.ibss == 1) ?
                INDEPENDENT_BSS : INFRASTRUCTURE_BSS;

    mlmeL2Printf("mlmeProbeRespReceive -- ");
    mlmeL2PrintMacAddress(bFrame->macHeader.bssId);
    mlmeL2Printf(" channel %d\n", pdevInfo->staConfig.phwChannel->channel);

    if (!wlanDoesDsElementMatch(pdevInfo, pDsParams) ||
        pScanInfo->bssSet.count == BSSDESCR_SET_SIZE ||
        (pScanInfo->bsstype != ANY_BSS &&
         bssType != pScanInfo->bsstype)              ||
        pSsid == NULL)
    {
        /* We either:
         *
         * 1) Picked up this beacon on a wrong channel. Can happen in 11b.
         *    Beacons can be heard anywhere in the 25 MHz vicinity. Ignore it.
         * 2) Ran out of space to put the BSSDESCR.
         * 3) The BSS type doesn't match what we're looking for.
         * 4) Found an invalid SSID in the frame.
         */
        return;
    }

    pBss = (BSSDESCR *)A_DRIVER_MALLOC(sizeof(*pBss));
    if (!pBss) {
        return;
    }

    A_MEM_ZERO(pBss, sizeof(*pBss));

    pTim = (TIM_ELEMENT *)ieSet.pInfoElement[eidMap[ELE_TIM]];

    pBss->DTIMPeriod     = pTim ? pTim->dtimPeriod : DEFAULT_DTIM_PERIOD;
    pBss->beaconInterval = bFrame->beaconInterval;
    pBss->bsstype        = bssType;
    pBss->pChannel       = pdevInfo->staConfig.phwChannel;
    pBss->rssi           = pDesc->status.rx.rssi;
    pBss->timestamp      = bFrame->timestamp;
    pBss->capabilityInfo = bFrame->capInfo;
    pBss->bssId          = bFrame->macHeader.bssId;
    pBss->probeResp      = TRUE;

    mlmeElementCopy((INFO_ELEMENT *)pSsid, (INFO_ELEMENT *)&pBss->ssid);
    mlmeElementCopy((INFO_ELEMENT *)pDsParams, (INFO_ELEMENT *)&pBss->dsParamSet);

    pRateSet    = (RATE_SET *)
        ieSet.pInfoElement[eidMap[ELE_SUPPORTED_RATES]];
    pExtRateSet = (EXT_RATE_SET *)
        ieSet.pInfoElement[eidMap[ELE_EXT_SUPPORTED_RATES]];

    /* save copies for NDIS OIDs */
    mlmeElementCopy((INFO_ELEMENT *)pRateSet, (INFO_ELEMENT *)&pBss->rateSetIe);
    mlmeElementCopy((INFO_ELEMENT *)pExtRateSet, (INFO_ELEMENT *)&pBss->extRateSetIe);

    wlanElementToFullRateSet(pRateSet, pExtRateSet, &wlanApRateSet);
    wlanRateSetCopy(&wlanApRateSet, &pBss->supportedRateSet);

    mlmeElementCopy((INFO_ELEMENT *)
                    ieSet.pInfoElement[eidMap[ELE_CF_PARAM_SET]],
                    (INFO_ELEMENT *)&pBss->cfParamSet);
    mlmeElementCopy((INFO_ELEMENT *)
                    ieSet.pInfoElement[eidMap[ELE_IBSS_PARAM_SET]],
                    (INFO_ELEMENT *)&pBss->ibssParamSet);

    if (pBss->bsstype == INFRASTRUCTURE_BSS) {
        COUNTRY_INFO_LIST *pList;
        TPC_IE            *pTpc;

        /* TPC element is only available in infrastructure mode. */
        pTpc = (TPC_IE *)ieSet.pInfoElement[eidMap[ELE_TPC]];
        if (pTpc && VALID_TPC_ELEMENT(pTpc)) {
            pBss->tpcIe = *pTpc;
        }

        pList = (COUNTRY_INFO_LIST *)
            ieSet.pInfoElement[eidMap[ELE_COUNTRY_INFO]];
        if (pList &&
           (pList->length + sizeof(pList->elementID) + sizeof(pList->length)) <=
            sizeof(*pList))
        {
            pBss->ccList = *pList;
        }
    }

    /*
     *  Process nonErpElement and ShortslotTime. sent by our AP
     *  during background scan.
     */
    pNonErpElement = (NONERP_ELEMENT *)ieSet.pInfoElement[eidMap[ELE_NONERP]];

    if ((pdevInfo->localSta->staState & STATE_JOINED) &&
        (A_MACADDR_COMP(&pdevInfo->baseBss.bssId, &bFrame->macHeader.bssId) == 0))
    {
        processNonErpElement(pdevInfo, pNonErpElement);

        oldVal = (A_UINT32)pdevInfo->useShortSlotTime;
        pdevInfo->useShortSlotTime = pdevInfo->staConfig.shortSlotTime ?
                                     bFrame->capInfo.shortSlotTime : FALSE;

        if (oldVal != (A_UINT32)pdevInfo->useShortSlotTime) {
            bssAttribValue = (A_UINT32)pdevInfo->useShortSlotTime;
            wdcUpdateBssAttribute(pdevInfo->targetHandle, 0,
                                  SHORT_SLOT_TIME_11g,
                                  sizeof(bssAttribValue),
                                  (TARGET_CONFIG_VAL)&bssAttribValue);
        }
    }

    if (pNonErpElement) {
        mlmeElementCopy((INFO_ELEMENT *)pNonErpElement,(INFO_ELEMENT *)&pBss->nonErpElement);
    } else {
        pBss->nonErpElement.length = 0;
    }

    /*
     * Make the element invalid first.
     * TODO: need remove this when the mlmeElementCopy changed
     */
    pBss->wpaIe.length = 0;

    pWpaIe = ieSet.pWpaIe;
    if (pWpaIe != NULL && VALID_WPA_ELEMENT((WPA_IE *)pWpaIe)) {
        mlmeElementCopy(pWpaIe, (INFO_ELEMENT *)&pBss->wpaIe);
    }
    pBss->athAdvCapElement.length = 0;
    pAthCap = ieSet.pAthCap;
    if (pAthCap != NULL && VALID_ATH_ADVCAP_ELEMENT(pAthCap)) {
        mlmeElementCopy((INFO_ELEMENT *)pAthCap,
                        (INFO_ELEMENT *)&pBss->athAdvCapElement);
    }

#ifdef WME
    /*
     * If probe Response doesn't have WME IE, then no need to request
     * for it.
     */
    pSib = sibEntryFind(pdevInfo, &bFrame->macHeader.srcAddr, NULL);
    pSib->qosdataEnabled = (ieSet.pWmeV1Ie != NULL);
#endif

    aie = (AIRONET_IE *)ieSet.pInfoElement[eidMap[ELE_AIRONET]];
    if (aie != NULL) {
        ccxAironetIeProcess(pdevInfo, pBss, aie);
    }

    pBss->pChannel = pdevInfo->staConfig.phwChannel;
    pBss->scanTime = A_MS_TICKGET();

    /*
     * Take note of the fact that we've seen a BSS here. BSSes can have
     * one or more SSIDs that are not broadcast.  Please see bugs 8312 and
     * 8313 for more details.
     */
    pdevInfo->staConfig.phwChannel->bssSeenHere = TRUE;

    mlmePrintf("mlmeProbeRespReceive: update bss set, channel %d, flags %x, bssSet.count: %d\n",
               pBss->pChannel->channel,
               pBss->pChannel->channelFlags,
               pdevInfo->pscanInfo->bssSet.count);

    /* Add or replace the existing entry */
    mlmeBsssetAdd(pdevInfo, pBss, &pdevInfo->pscanInfo->bssSet);
    A_DRIVER_FREE(pBss, sizeof(*pBss));

    if (pdevInfo->localSta->serviceType == WLAN_STA_SERVICE) {
        mlmeScanFunction(pdevInfo, SF_ADVANCE_NOW);
    }
}

/* wlanMlmeStartRequest
 *
 * Tells the MAC entity to operate as a BSS
 * It sets various BSS related parameters into the MAC, enables beacon
 * generation, DTIM generation, enable power save buffering, etc.
 *
 * This will be implemented as a synchronous operation.
 *
 * RETURNS:    MLME_SUCCESS, MLME_INVALID_PARAM, MLME_ALREADY_IN_BSS
 */
MLME_RESULT
wlanMlmeStartRequest(WLAN_DEV_INFO *pdevInfo, BSSDESCR *pbssDescr)
{
    A_BOOL      faulted;
    CHAN_VALUES *pChan;
    SSID_ARRAY  *pSsids = &pdevInfo->staConfig.cfgSsids;

    if (!pdevInfo->staConfig.privacyInvoked &&
        pbssDescr->capabilityInfo.privacy)
    {
        /*
         * IBSS is trying to use WEP, but we don't have it enabled.
         * cannot start this SSID.
         */
        return MLME_INVALID_PARAM;
    }

    /* ideally, we shouldn't get anything which says ANY bss */
    if (pbssDescr->bsstype == ANY_BSS) {
        pbssDescr->bsstype = INDEPENDENT_BSS;
        pbssDescr->capabilityInfo.ess  = 0;
        pbssDescr->capabilityInfo.ibss = 1;
    }

    /* Copy BSS description into pdevInfo of the STA */
    if (pdevInfo->bssDescr == NULL) {
        pdevInfo->bssDescr = (BSSDESCR *)A_DRIVER_MALLOC(sizeof(BSSDESCR));
        if (pdevInfo->bssDescr == NULL) {
            return MLME_DRIVER_ERROR;
        }
    }
    *pdevInfo->bssDescr = *pbssDescr;

    A_MACADDR_COPY(&pbssDescr->bssId, &pdevInfo->baseBss.bssId);
    mlmeAssocIdInit(pdevInfo, &pdevInfo->baseBss);
    if (isXrAp(pdevInfo)) {
        A_MACADDR_COPY(&pbssDescr->bssId, &pdevInfo->xrBss.bssId);
        setXrBssIdMask(&pdevInfo->xrBss.bssId);
        mlmeAssocIdInit(pdevInfo, &pdevInfo->xrBss);
    }

    /* There's only one desired ssid. */
    pSsids->elements[0] = pbssDescr->ssid;
    A_ATOMIC_SET(&pSsids->count, 1);

    pChan = pdevInfo->bssDescr->pChannel;

    pdevInfo->bssDescr->dsParamSet.elementID = ELE_DS_PARAM_SET;
    pdevInfo->bssDescr->dsParamSet.length    = 1;
    pdevInfo->bssDescr->dsParamSet.currentChannel =
        (A_UINT8) wlanConvertGHztoCh(pChan->channel, pChan->channelFlags);

    /* Enter the new channel into HW config */
    pdevInfo->staConfig.pChannel = pChan;

#ifndef BUILD_AP
    if (pbssDescr->bsstype == INFRASTRUCTURE_BSS) {
        pdevInfo->localSta->staState |= STATE_AP_UP;
    }

    /* Move to the desired channel */
    if (drvChangeChannel(pdevInfo, pChan, pbssDescr->bsstype ==
                         INFRASTRUCTURE_BSS ? FALSE : TRUE, FALSE) != A_OK)
    {
        return MLME_INVALID_PARAM;
    }

    pdevInfo->powerMgmt.needResyncToBeacon = FALSE;
#endif

    wlanStaJoinBss(pdevInfo);

    if (pbssDescr->bsstype == INDEPENDENT_BSS) {
        pdevInfo->localSta->staState  |= STATE_JOINED;
        pdevInfo->localSta->transState = TSTATE_QUIET;

        pdevInfo->localSta->capInfo.ess           = pbssDescr->capabilityInfo.ess;
        pdevInfo->localSta->capInfo.ibss          = pbssDescr->capabilityInfo.ibss;
        pdevInfo->localSta->capInfo.shortPreamble = pbssDescr->capabilityInfo.shortPreamble;
        pdevInfo->localSta->capInfo.privacy       = pbssDescr->capabilityInfo.privacy;

        /* Set up privacy, if any */
        if (pdevInfo->staConfig.privacyInvoked) {
            WLAN_PRIV_RECORD *pKey = NULL;
            int i;

            /* Install the BSS keys */
            for (i = 0; i < MAX_SHARED_KEYS; i++) {
                A_UINT32 isDefaultBssKey = 0;

                pKey = &pdevInfo->keyTable[i];
                if (pKey->keyLength) {
                    /* Reset the key type to pure WEP */
                    pKey->keyType = PRIV_KEY_TYPE_WEP;
                    pKey->keyFlags &= ~PRIV_CKIP_MIC;

                    isDefaultBssKey = (pdevInfo->staConfig.defaultKey == i);
                    wlanInstallBssKey(pdevInfo, i, isDefaultBssKey, pKey);
                }
            }
            if (initCipher(pdevInfo) != A_OK) {
                pdevInfo->localSta->staState  &= ~STATE_JOINED;
                pdevInfo->localSta->transState = TSTATE_QUIET;
                return MLME_REFUSED;
            }
            setupSwEncryption(pdevInfo);
        }

        wlanBeaconInit(pdevInfo);
    }

    /* drvChangeChannel set correct rate set. Copy it into bssdescr */
    wlanRateSetCopy(&pdevInfo->baseBss.rateSet, &pbssDescr->supportedRateSet);
    /* Copy it to rateSetIe for NDIS OIDs */
    pbssDescr->rateSetIe.elementID = ELE_SUPPORTED_RATES;
    pbssDescr->rateSetIe.length    = pbssDescr->supportedRateSet.length;
    A_DRIVER_BCOPY(&pbssDescr->supportedRateSet.rates, &pbssDescr->rateSetIe.rates,
                    pbssDescr->rateSetIe.length);

#ifndef BUILD_AP
    /* No adhoc in 11g, hence no extended rate set */
    pbssDescr->extRateSetIe.length = 0;
#endif

    /* xxxxPGSxxxx - this needs moved into the target bss code */
#if 0
    halSetBasicRate(pdevInfo, &pbssDescr->supportedRateSet);
#endif

    /*
     * For infrastructure mode the AP beacon generation is
     * turned on in the station init code itself! It could ideally
     * be done here, but am not sure if mlmeStart gets called for
     * that case at all!
     */
    wlanMlmeStartConfirm(pdevInfo, MLME_SUCCESS);

    return MLME_SUCCESS;
}

/* wlanMlmeStartConfirm - Confirm the start of BSS
 *
 * Since the request is implemented as a synchronous operation,
 * this is an empty routine.
 */
MLME_RESULT
wlanMlmeStartConfirm(WLAN_DEV_INFO *pdevInfo, MLME_RESULT resultCode)
{
    /* If callback routine exists, call it */
    if (pdevInfo->pSmeInfo->startCallback) {
        (*pdevInfo->pSmeInfo->startCallback)(pdevInfo, resultCode);
    }

    return resultCode;
}

void
mlmePowerCallback(WLAN_DEV_INFO *pdevInfo, A_STATUS resultCode,
                  A_BOOL powerStatus)
{
    /* Got a callback from power mgt code. Cancel the timer. */
    A_UNTIMEOUT(&(pdevInfo->smeTimer));
    wlanMlmePowermgtConfirm(pdevInfo, resultCode, powerStatus);
}

/*
 * wlanMlmePwrmgtRequest - Change PS mode of STA
 *
 * MLME Power management primitive changes the curent power saving mode.
 * This call will change the setting and return. MLME module will
 * send a null frame to AP to indicate the state transition or use an
 * existing scheduled transmission. When going into PS mode, the packets
 * already scheduled to transmit will be transmitted before switching the
 * mode. If the transmission attempts by MAC hardware fail, then software will
 * suspend those packets instead of going into PS mode. When going into
 * active mode, if wakeup is TRUE, MAC is immediately
 * forced into active mode instead of waiting for beacons.
 *
 * RETURNS: MLME_SUCCESS, MLME_NOT_SUPPORTED or MLME_INVALID_PARAM
 */
MLME_RESULT
wlanMlmePwrmgtRequest(WLAN_DEV_INFO *pdevInfo,
                      A_UINT16      powermgtMode, /* PS_MODE_ACTIVE,PS_MODE_POWER_SAVE */
                      A_BOOL        wakeup)       /* TRUE for immediate active mode */
{
    A_STATUS    pwStatus = A_ERROR;

    if (pdevInfo->localSta->transState != TSTATE_QUIET) {
        // if tstate is not 0, mlme operation is in progress, cannot use
        // sme timer, cannot change power mode
        return MLME_REFUSED;
    }

#ifndef BUILD_AP
    // sanity check
    switch (powermgtMode) {
    case PS_MODE_ACTIVE:
        // CALL NDIS driver power save routine to wake up
        pwStatus = powerSet(pdevInfo,
                            WAKE_UP, wakeup ? SET_PWR_TO_1 : REDUCE_PWR_FROM_1,
                            TRUE, mlmePowerCallback);
        break;

    case PS_MODE_POWER_SAVE:
        // CALL NDIS driver power save routine to go to sleep
        // according to 802.11 spec, wakeup flag in POWER_SAVE mode has no
        // meaning. We will use that to signify FAKE_SLEEP mode. In a way,
        // we are telling driver to do sleep semantics but wakeup immediately.
        if (wakeup) {
            // special case to take us into/out of FAKE_SLEEP mode
            pwStatus = powerSet(pdevInfo, FAKE_SLEEP, ENABLE, pdevInfo->localSta->receivedBeaconFromHomeAP,
                                mlmePowerCallback);
        } else {
            pwStatus = powerSet(pdevInfo, FAKE_SLEEP, DISABLE, pdevInfo->localSta->receivedBeaconFromHomeAP,
                                mlmePowerCallback);
            // Presumably now back on home channel; don't notify AP next time
            // until we hear from it.
            pdevInfo->localSta->receivedBeaconFromHomeAP = FALSE;
        }
        break;

    default:
        ASSERT(0);
    }
#endif
    if (pwStatus == A_PENDING) {
        // power mgt frame is in the queue, hence it returned A_PENDING, set a timer
        // hardcoded value of 50 ms
        // mark the state as busy, callback will clear it.
        pdevInfo->localSta->transState = TSTATE_PWRMGT_PENDING;
        A_TIMEOUT(&pdevInfo->smeTimer, 50, (A_UINT32) pdevInfo,
                  MLME_PWRMGT_TIMEOUT);
        return MLME_OP_PENDING;
    }

    return (pwStatus == A_OK )? MLME_SUCCESS : MLME_NOT_SUPPORTED;
}

/* wlanMlmePowermgtConfirm - Confirm the powermgt operation's result
 *
 * This routine is used to call pwrmgtCallback() to inform the operation
 * result. One possible reason for failure is that MAC fails to transmit
 * successfully a packet with PwrMgt bit set.
 *
 * RETURNS: MLME_SUCCESS, MLME_INVALID_PARAM
 */

void
wlanMlmePowermgtConfirm(WLAN_DEV_INFO *pdevInfo,
                        MLME_RESULT   result,
                        A_BOOL        powerStatus)
{
    pdevInfo->localSta->transState = TSTATE_QUIET;
    /* If callback routine exists, call it */
    if (pdevInfo->pSmeInfo->pwrMgtCallback) {
        (*pdevInfo->pSmeInfo->pwrMgtCallback)(pdevInfo, result, powerStatus);
    }
    return;
}


void
mlmeOperationCancel(WLAN_DEV_INFO *pDev)
{
    mlmePrintf("> mlmeOperationCancel\n");

    A_UNTIMEOUT(&pDev->smeTimer);
    wlanMlmeScanConfirm(pDev, MLME_SUCCESS, TRUE);

    mlmePrintf("< mlmeOperationCancel\n");

}

/* wlanResetRequest - Reset the MAC entity
 *
 * This call resets the MAC hardware and software structures that
 * are used to handle that specific MAC. There will be other routines
 * and commands to reset the hardware, driver, etc. but not done as
 * part of MLME operation.
 *
 * PARAMETERS:
 *   deauth - Whether or not to send a Deauthentication frame to the AP
 *
 * RETURNS: MLME_SUCCESS, WLAN_UNKNOWN_FAILURE
 */

#ifdef NOTUSED
/*
 * wmeNotificationSetupRequest
 *
 * This routine sends out a WME Notification Setup Request frame.
 */
MLME_RESULT
wmeNotificationSetupRequest(WLAN_DEV_INFO *pdevInfo,
                     WLAN_MACADDR  *bssid,
                     SSID          *ssid)
{
    WLAN_FRAME_PROBE_REQ *pFrame;
    SIB_ENTRY            *pSib = NULL;
    A_UINT16             frameLen;
    ATHEROS_DESC         *pDesc;
    INFO_ELEMENT         *pTempIe;
    RATE_SET             rateSet;
    EXT_RATE_SET         extRateSet;

    mlmeL2Printf("wmeNotificationSetupRequest \n");

    if (!isGrp(bssid)) {
        pSib = sibEntryFind(pdevInfo, bssid, NULL);
    }

     /* Compute frame length. */
    /* Don't use INFO_ELEMENT_SIZE macro for size of ssid, it can be null */

    frameLen = sizeof(WLAN_MGT_MAC_HEADER) +
               (ssid->length + sizeof(ssid->elementID) + sizeof(ssid->length)) +
               INFO_ELEMENT_SIZE(rateSet) +
               INFO_ELEMENT_SIZE(extRateSet);

    if ((createBuffandDesc(pdevInfo, frameLen, &pDesc)) != 0) {
        /* Failed to allocate buffer for frame and descriptor */
        return MLME_DRIVER_ERROR;
    }

    pFrame = pDesc->pBufferVirtPtr.probe;
    mgtFrameHdrInit(&pFrame->macHeader, SUBT_ACTION, bssid);
    A_MACADDR_COPY(bssid, &pFrame->macHeader.bssId);
    A_MACADDR_COPY(&pdevInfo->localSta->macAddr, &pFrame->macHeader.srcAddr);

    pTempIe = mlmeElementCopy((INFO_ELEMENT *)ssid,
                              (INFO_ELEMENT *)&pFrame->ssid);
    pTempIe = mlmeElementCopy((INFO_ELEMENT *)&rateSet,
                              (INFO_ELEMENT *)pTempIe);

    if (VALID_ERS_ELEMENT(&extRateSet)) {
        pTempIe = mlmeElementCopy((INFO_ELEMENT *)&extRateSet,
                                   (INFO_ELEMENT *)pTempIe);
    }

#ifdef DEBUG_11B_PROBE_REQUESTS
    /*
     * The following is useful when debugging scanning/probing in 802.11b
     * as there is no channel number in the Probe Request but there is in
     * the Probe Response, and frames can be heard across neighboring
     * channels.
     */
    frameLen += 3;
    pTempIe->elementID = ELE_VENDOR_PRIVATE;
    pTempIe->length    = 1;
    pTempIe->value[0]  = (A_CHAR)wlanConvertGHztoCh(pdevInfo->staConfig.phwChannel->channel,
                                            pdevInfo->staConfig.phwChannel->channelFlags);
#endif

    if (mlmeDebugLevel > 10) {
        displayMgtFrame(&pFrame->macHeader, frameLen);
    }

    pDesc->pDestSibEntry = pSib;
    pDesc->pOpBss        = &pdevInfo->baseBss;
    SETQHANDLE(pDesc, TXQ_ID_FOR_AC1);
    pDesc->bIndicateXmitStatus = TRUE;
    if (powerFrameTransmit(pdevInfo, pDesc) == A_OK) {
        return MLME_SUCCESS;
    }

    return MLME_DRIVER_ERROR;
}
#endif

void
wlanResetRequest(WLAN_DEV_INFO *pDev, SEND_DEAUTH_ENUM deauth)
{
    A_BOOL sendDeauth = (deauth == SEND_DEAUTH);

    /*
     * Do a systematic MAC shutdown.
     * Currently implemented only on station.
     * First, tell the AP that we are disconnecting.
     */
    if (pDev->localSta->staState & STATE_AUTH) {
        wlanMlmeDeauthRequest(pDev, &pDev->baseBss, &pDev->baseBss.bssId,
                              REASON_AUTH_LEAVING, sendDeauth);
    }

    pDev->localSta->staState              &= ~STATE_CONNECTED;
    pDev->localSta->transState             = 0;
    pDev->turboPrimeInfo.turboPrimeAllowed = FALSE;
    pDev->turboPrimeInfo.friendlyTurboAllowed = FALSE;

#ifdef BUILD_AP
    chipReset(pDev, pDev->staConfig.pChannel, FALSE);
#else /* BUILD_AP */
    ResetOnError(pDev, DONT_CLEAR_TX_Q, RESET_DONT_INC_CTR, DO_WAIT);
#endif
    /*
     * Make sure we don't get any more of these interrupts until we're
     * associated again.
     */
    halDisableInterrupts(HACK_TO_PARDEV(pDev), HAL_INT_BMISS);

    wlanResetConfirm(pDev);
}

/* wlanResetConfirm - Confirm the reset of MAC
 * This routine calls reseCallback() to deliver the Success/fail
 * result of reset operation.
 */
MLME_RESULT
wlanResetConfirm(WLAN_DEV_INFO *pdevInfo)
{
    /* If callback routine exists, call it */
    if (pdevInfo->pSmeInfo->resetCallback) {
        (*pdevInfo->pSmeInfo->resetCallback)(pdevInfo);
    }
    return MLME_SUCCESS;
}

/* set sw/hw decompression mask */
void
sibDecompStateSet(WLAN_DEV_INFO *pdevInfo, SIB_ENTRY *pSib, A_BOOL decomp)
{
    A_UINT32 connAttribValue;

    ASSERT(pSib);
    ASSERT(!pSib->privacyInvoked || pSib->pPriv);

    if (VALID_ATH_ADVCAP_ELEMENT(&pSib->athAdvCapElement) &&
        pSib->athAdvCapElement.info.useCompression)
    {
        /* for clear and disconnect cases */
        if (!decomp || !pdevInfo->staConfig.privacyInvoked /* ??? */) {
            pSib->athAdvCapElement.info.useCompression = decomp;
        }
    } else {
        /* no compression */
        decomp = FALSE;
    }

    /* Set compression in the shared WEP keys case */
    if ((pdevInfo->staConfig.privacyInvoked) && (decomp)) {
        if(!pdevInfo->staConfig.wpaEnabled)  {
            WLAN_PRIV_RECORD *pKey;
            A_UINT32 keyIndex = pSib->athAdvCapElement.defKeyIndex;

            /*
             * Allow compression if the AP advertized a proper index,
             * i.e. if it is configured for static WEP and compression.
             */
            if (keyIndex < MAX_SHARED_KEYS) {
                pKey = &pdevInfo->keyTable[keyIndex];

                pKey->keyType = PRIV_KEY_TYPE_WEP_COMP;
                wlanInstallConnectionKey(pdevInfo,
                                     pSib,
                                     &pSib->macAddr,
                                     pKey);
            } else {
                decomp = FALSE;
            }
        } else {
            decomp = FALSE;
        }
    }

    connAttribValue = decomp;
    wdcUpdateConnectionAttribute(pdevInfo->targetHandle,
                                 pSib->wdcConnId,
                                 CONN_COMPRESSION,
                                 sizeof(connAttribValue),
                                 (TARGET_CONFIG_VAL)&connAttribValue);
}

/* clean up all cipher negotiation related stuff */
void
sibCseStateClean(WLAN_DEV_INFO *pDevInfo, SIB_ENTRY *pSib)
{
    WLAN_PRIV_RECORD *pKey = pSib->pPriv;

    pSib->uCipher = 0;
    pSib->mCipher = 0;
    pSib->authSel = 0;

    /* clear sw encryption enable flag */
    pSib->swEncryptionEnabled = FALSE;
    pSib->swMicEnabled        = FALSE;
    pSib->ckipEnabled         = FALSE;
    pSib->ckipUseMic          = FALSE;
    pSib->cckmEnabled         = FALSE;

    /* Reset key types to WEP if necessary */
    if (pKey) {
        if (pKey->keyType == PRIV_KEY_TYPE_CKIP) {
            pKey->keyType = PRIV_KEY_TYPE_WEP;
        }
        pKey->keyFlags &= ~PRIV_CKIP_MIC;
    }
}

/* clean up all power saving related stuff */
void
sibPowerStateClean(WLAN_DEV_INFO *pdevInfo, SIB_ENTRY *pSib)
{
#ifdef BUILD_AP
    /* currently, this is used only on AP */
    if (pSib) {
        /* update the power save state */
        if (pSib->psMode == PS_MODE_POWER_SAVE) {
            pSib->psMode = 0;
            pSib->pOpBss->numSleepingClients--;
        }

        /* flush the power save queue */
        psStateReset(pdevInfo, pSib, FALSE);

        /* update the TIM if needed */
        if (!pSib->numTxPending) {
            wlanTimUpdate(pdevInfo, pSib->assocId, FALSE);
        }
    }
#endif
}

A_UINT16
mlmeCountryCodeElementLength(WLAN_DEV_INFO *pDevInfo,
                             COUNTRY_INFO_LIST*pCountryInfo)
{
    if (pDevInfo->localSta->serviceType == WLAN_AP_SERVICE &&
        pDevInfo->staConfig.multiDomainCapEnabled)
    {
        ASSERT(pCountryInfo);
        return (pCountryInfo->length + sizeof(pCountryInfo->elementID) +
                sizeof(pCountryInfo->length));
    }

    return 0;
}

void
mlmeCountryCodeElementFill(WLAN_DEV_INFO *pDevInfo,
                           COUNTRY_INFO_LIST *pCountryInfo)
{
    if (pDevInfo->localSta->serviceType == WLAN_AP_SERVICE &&
        pDevInfo->staConfig.multiDomainCapEnabled)
    {
        wlanGetCountryInfoList(pDevInfo, pCountryInfo);
        pCountryInfo->elementID = ELE_COUNTRY_INFO;
    }
}

/*  Process the nonErpElement Information Element */

static void
processNonErpElement(WLAN_DEV_INFO *pDevInfo,NONERP_ELEMENT *pNonErpElement)
{

    A_STATUS status;
    A_UINT32 cfgVal;

    if (pNonErpElement != NULL) {
        mlmeElementCopy((INFO_ELEMENT *)pNonErpElement,
        (INFO_ELEMENT *)&pDevInfo->bssDescr->nonErpElement);

        if ((pDevInfo->staConfig.modeCTS == 2) &&
            (pDevInfo->protectOn != pNonErpElement->info.protect))
        {
            pDevInfo->protectOn = pNonErpElement->info.protect;
            cfgVal = (A_UINT32)pDevInfo->protectOn;
            wdcTargetSetConfiguration(pDevInfo->targetHandle, CFG_GMODE_PROTECTION,
                                      0, (TARGET_CONFIG_VAL)(&cfgVal), NULL);
        }

        if (pDevInfo->nonErpPreamble != pNonErpElement->info.preamble) {

            pDevInfo->nonErpPreamble = pNonErpElement->info.preamble;
            cfgVal = (A_UINT32)pDevInfo->nonErpPreamble;
            wdcTargetSetConfiguration(pDevInfo->targetHandle, CFG_GMODE_NON_ERP_PREAMBLE,
                                      0, (TARGET_CONFIG_VAL)(&cfgVal), NULL);
        }
    }

    return;
}

/*******************************************************************************
* turboPrimeSwitch
*
* This routine is called when the STA wants to switch dynamic turbo mode.
*
* Dynamic turbo mode switch routine.
* en == 1: switch from base to turbo
* en == 0: switch from turbo to base
* With dynamic turbo, channel center does not change.
*/
void
turboPrimeSwitch(WLAN_DEV_INFO *pdevInfo, A_UINT8 en)
{
    A_UINT32 bssAttribValue;
    if (en) {
        pdevInfo->bssDescr->pChannel->channelFlags |= CHANNEL_TURBO;
        uiPrintf("Switch to turbo mode\n");
    } else {
        pdevInfo->bssDescr->pChannel->channelFlags &= ~CHANNEL_TURBO;
        uiPrintf("Switch back to base mode\n");
    }

    pdevInfo->staConfig.pChannel->channelFlags   =
                pdevInfo->bssDescr->pChannel->channelFlags;
    pdevInfo->staConfig.phwChannel->channelFlags =
                pdevInfo->bssDescr->pChannel->channelFlags;

#ifndef BUILD_AP
    cservUpdateLastActivityTime(pdevInfo);
    pdevInfo->turboPrimeInfo.timer = A_MS_TICKGET();
#endif

    drvChangeChannel(pdevInfo, pdevInfo->bssDescr->pChannel, TRUE, TRUE);
    mlmePrintf("Boost change to %x %d\n",
                     pdevInfo->bssDescr->pChannel->channelFlags,
                     pdevInfo->bssDescr->athAdvCapElement.info.boost);

    /*
     * Update target default rate index, it is used for turbo prime
     */
    bssAttribValue = (A_UINT32)pdevInfo->baseBss.defaultRateIndex;
    wdcUpdateBssAttribute(pdevInfo->targetHandle,
                          0,
                          DEFAULT_RATE_INDEX,
                          sizeof(bssAttribValue),
                          (TARGET_CONFIG_VAL)&bssAttribValue);

    pdevInfo->turboPrimeInfo.currentBoostState = en;
}

/*******************************************************************************
* updtSupFeatureInfo
*
* This routine is called to update the Supported SuperG Feature Info
*
* mode is used only on AP.
*
*/
void updtSupFeatureInfo(WLAN_DEV_INFO *pdevInfo, WIRELESS_MODE mode)
{
    if (pdevInfo->staConfig.abolt) {
        pdevInfo->staConfig.abolt =
            pdevInfo->staConfig.abolt &
            ((!pdevInfo->devCap.compressSupport) ?
            ~ABOLT_COMPRESSION : pdevInfo->staConfig.abolt)  &
            ((!pdevInfo->devCap.burstSupport) ?
            ~ABOLT_BURST : pdevInfo->staConfig.abolt)  &
            ((!pdevInfo->devCap.fastFramesSupport) ?
             ~ABOLT_FAST_FRAME : pdevInfo->staConfig.abolt)  &
            ((!pdevInfo->devCap.turboGSupport) ?
             ~ABOLT_TURBO_G : pdevInfo->staConfig.abolt)  &
            ((!pdevInfo->devCap.turboPrimeSupport) ?
             ~(ABOLT_TURBO_PRIME | ABOLT_FRIENDLY_TURBO) : pdevInfo->staConfig.abolt)  &
            ((!pdevInfo->devCap.chapTuningSupport) ?
             ~ABOLT_WME_ELE : pdevInfo->staConfig.abolt);

#if defined (BUILD_AP)
        if (mode != WIRELESS_MODE_11g && mode != WIRELESS_MODE_11a) {
            pdevInfo->staConfig.abolt &= ~(ABOLT_TURBO_PRIME | ABOLT_FRIENDLY_TURBO);
        }
#endif

    }
}

#if defined (DEBUG) && defined (BUILD_AP)
void
dumpoverlapbss(WLAN_DEV_INFO *pdevInfo)
{
    uiPrintf("\nUnit  %d", pdevInfo->devno);
    uiPrintf("\nOther nonErp Present:  %s",
             pdevInfo->protectInfo.otherPresent ? "YES" : " NO");
    uiPrintf("\nOther Pres timestamp:  %d %d",
             pdevInfo->protectInfo.timePres, A_MS_TICKGET());
    uiPrintf("\nThis AP protection:    %s",
             pdevInfo->protectOn ? "YES" : " NO");
    uiPrintf("\nThis AP long slot:     %s",
             pdevInfo->useShortSlotTime ? " NO" : "YES");
    uiPrintf("\n");
}

#endif  /* DEBUG && BUILD_AP */

/*******************************************************************************
* WLAN to WDC conversion routines
*/

void
wlanRateSet2WdcRateSet(WLAN_RATE_SET *pRateSet, CONN_RATE_SET *pConnRateSet )
{
    /* WDC rate set structure and WLAN rate set structure are identical */
    wlanRateSetCopy(pRateSet, (WLAN_RATE_SET *) pConnRateSet);
}
