/*
 * $Id: //depot/sw/branches/1.3_USB_LINUX_port/src/USB/wlan/host/common/statab.c#2 $
 *
 * This file contains the data structures and routines to manage
 * the shared data items in STA table
 *
 * Copyright (c) 2000-2003 Atheros Communications, Inc., All Rights Reserved
 */

#include "wlandrv.h"
#include "wlanext.h"
#include "wlansme.h"
#include "wlansmeext.h"
#include "wlanchannel.h"
#include "wdcApi.h"
#include "wlanCapDe.h"
#include "display.h"
#include "ui.h"

#ifdef BUILD_AP
#include "dot1x.h"
#ifdef INCLUDE_RCW
#include "rc.h"
#include "ws_global.h"

SCB *dot1xSessionGet(WLAN_DEV_INFO *, SIB_ENTRY *);
int dot1xReAuthPeriodGet(void);
extern char *dot1xStateStr[];
void displayDot1xStats(SCB *s);

SIB_ENTRY *pStaListTbl[MAX_STA_NUM];
int       staListNum;

#endif /* INCLUDE_RCW */
#endif /* BUILD_AP */


/* Forward references */
LOCAL A_STATUS  sibEntryAddLocked(WLAN_DEV_INFO *pDev, SIB_ENTRY *);
#ifndef UPSD
LOCAL INLINE SIB_ENTRY *
#else
SIB_ENTRY *
#endif
sibEntryFindLocked(WLAN_DEV_INFO *pDev, WLAN_MACADDR *pAddr, OP_BSS *pOpBss);
LOCAL void      sibEntryDelLocked(WLAN_DEV_INFO *pDev, SIB_TABLE *pTable, SIB_ENTRY *pSib);
LOCAL SIB_ENTRY *sibVictimFindLocked(WLAN_DEV_INFO *pDev, SIB_ENTRY *localSib, BSSDESCR *apDescr);
LOCAL void sibKeyShow(SIB_ENTRY *, int);
LOCAL char *sibCipherType(A_UINT32 cipher);

/* Static global vars */
LOCAL char txt_11a[]   = "5GHz 54Mbps (802.11a)";
LOCAL char txt_turbo[] = "5GHz 108Mbps (802.11a Turbo)";
LOCAL char txt_11b[]   = "2.4GHz 11Mbps (802.11b)";
/* LOCAL char txt_11bg[]  = "2.4GHz 11Mbps and 54Mbps (802.11b and 802.11g)"; */
LOCAL char txt_11g[]   = "2.4GHz 54Mbps (802.11g)";
LOCAL char txt_108g[]  = "2.4GHz 108Mbps (802.11g Turbo)";

#ifdef  DEBUG
int agingDebug = 0;
#endif

/*
DESCRIPTION

This module implements the management functions for AP's Station
Information Database.

The SIB is related to the AP host key table and MAC key Cache.

At the first step only one semaphore is utilized to implement the mutual
exclusion.

Functions:

*   find the entry which has the expected macAddr
*   add an entry
*   delete an entry
*   get an used entry
*   get a free entry
*   modify an entry
*
*/

typedef struct rate_to_str_map {
    char   str[4];
    int    rate;
} RATE_TO_STR_MAP;

static const RATE_TO_STR_MAP dataRateStrTbl[] = {
    {  "1",    1000},
    {  "2",    2000},
    {  "5.5",  5500},
    { "11",   11000},
    {  "6",    6000},
    {  "9",    9000},
    { "12",   12000},
    { "18",   18000},
    { "24",   24000},
    { "36",   36000},
    { "48",   48000},
    { "54",   54000},
    { "72",   72000},
    { "96",   96000},
    {"108",  108000},
    {"?",         0}

};


/***************************************************
* sibTableInit - Initialize the SIB table at system initialization stage
*
* This routine initilizes the SIB Table during the system initialization
* stage
*
* RETURNS: always A_OK.
*/

A_STATUS
sibTableInit(SIB_TABLE *pMySibTable, A_UINT32 sibTableSize)
{
    A_UINT16  i;
    SIB_TABLE *pSibTable;

    ASSERT(pMySibTable);
    ASSERT(sibTableSize <= MAX_STA_NUM);

    pSibTable = pMySibTable;
    pSibTable->sibTableSize = sibTableSize;

    /* Initialize the table semaphores */
    A_SEM_INIT(pMySibTable->sibTableSem, 0, CREATE_LOCKED);

    /*
     * As we go through the loop, make a linked list of
     * all the entries into pFreeList. Searching for a free
     * entry becomes easy with this linked list.
     */
    pMySibTable->pFreeList = NULL;

    for (i = 0; i < pSibTable->sibTableSize; i++) {
        A_MEM_ZERO(&pMySibTable->sta[i], sizeof(struct sibEntry));
        A_SEM_INIT(pMySibTable->sta[i].sibLock, 0, CREATE_LOCKED);

        pMySibTable->sta[i].wdcConnId = i;
        pMySibTable->sta[i].used  = SIB_FREE; /* unused entry */
        pMySibTable->sta[i].pNext = pMySibTable->pFreeList;
        pMySibTable->pFreeList    = &pMySibTable->sta[i];

#ifdef BUILD_AP
        A_SEM_INIT(pMySibTable->sta[i].txPSQueue.qSem, 0, CREATE_UNLOCKED);
#endif
    }

    pMySibTable->currentEntryNum = 0;

    for (i = 0; i < MAX_STA_HASH_NUM; i++) {
        pMySibTable->pHashList[i] =  NULL;
    }

    return A_OK;
}

/***********************************************************************
* sibTableExit - Free the resouce we allocate on initialization stage
*
* This routine uninitilizes the SIB Table during the system quit
* stage
*
* RETURNS: always A_OK.
*/

A_STATUS
sibTableExit(SIB_TABLE *pMySibTable)
{
    A_UINT16  i;
    SIB_TABLE *pSibTable;

    pSibTable = pMySibTable;

    /* Delete the table semaphores */
    A_SEM_DELETE(pMySibTable->sibTableSem);

    for (i = 0; i < pSibTable->sibTableSize; i++) {
        /* Delete the sib semaphores */
        A_SEM_DELETE(pMySibTable->sta[i].sibLock);
    }

    return A_OK;
}

/***********************************************************************
* sibFreeTargetConn - Free the target side connections
*
*
* RETURNS: always A_OK.
*/

A_STATUS
sibFreeTargetConn(WLAN_DEV_INFO *pDevInfo, SIB_TABLE *pSibTable)
{
    A_UINT16  i;
    SIB_ENTRY *pSib;

    /* Cleanup target side connections */
    for (i = 0; i < pSibTable->sibTableSize; i++) {

        pSib = &pSibTable->sta[i];

        /* Ignore the local SIB */
        if (pSib != pDevInfo->localSta && (pSib->used & SIB_VALID)) {
            /*
             * Flush pending tx requests
             * before deleting the target connection
             */
            NdisInterlockedIncrement(&pDevInfo->numFlushCall);
            wdcFlush(pDevInfo->targetHandle,pDevInfo->disableEnableCounter);

            /* Remove connection keys */
            wlanDeleteConnectionKey(pDevInfo, pSib, pSib->pPriv);

            /* Delete target connection */
            wdcDeleteConnection(pDevInfo->targetHandle, pSib->wdcConnId);

            sibEntryDel(pDevInfo, pSib);
        }
    }

    return A_OK;
}


A_INT32
wlanMACHash(A_UINT32 devno, WLAN_MACADDR *pmacAddr)
{
    return (A_INT32) (pmacAddr->octets[3] ^ pmacAddr->octets[4] ^
                      pmacAddr->octets[5] ^ devno) & (MAX_STA_HASH_NUM - 1);
}

static INLINE A_INT32
inlinewlanMACHash(A_UINT32 devno, WLAN_MACADDR *pmacAddr)
{
    return (A_INT32) (pmacAddr->octets[3] ^ pmacAddr->octets[4] ^
                      pmacAddr->octets[5] ^ devno) & (MAX_STA_HASH_NUM - 1);
}

/*******************************************************************************
*
* sibEntryFind - find the entry which has matched macAddress
*
* This routine searches the entire SIB database to find the match entry which
* has the same MAC address.  We depend on the wlanMACHash returning different
* values for each device.
*
* RETURNS:  the pointer of the entry if successful
*           NULL if unsuccessful
*/
#ifndef UPSD
LOCAL INLINE SIB_ENTRY *
#else
SIB_ENTRY *
#endif
sibEntryFindLocked(WLAN_DEV_INFO *pDevInfo, WLAN_MACADDR *macAddr, OP_BSS *pOpBss)
{
    SIB_ENTRY *sibEntry;
    A_UINT32  hash;

    hash = inlinewlanMACHash(pDevInfo->devno, macAddr);
    sibEntry = pDevInfo->pSibTable->pHashList[hash];
    while (sibEntry) {
        /* parsing through all the entries in linked list */
        if ((sibEntry->used & SIB_VALID) &&
            (A_MACADDR_COMP(&sibEntry->macAddr, macAddr) == 0)
            && ((pOpBss == NULL) || (sibEntry->pOpBss == pOpBss)))
        {
            sibEntry->staLastActivityTime = A_MS_TICKGET();
            return sibEntry;
        }
        sibEntry = sibEntry->pNext;
    }

    return NULL;
}

SIB_ENTRY *
sibEntryFind(WLAN_DEV_INFO *pDevInfo, WLAN_MACADDR *macAddr, OP_BSS *pOpBss)
{
    SIB_TABLE *pSibTable = pDevInfo->pSibTable;
    SIB_ENTRY *sibEntry;

    A_SIB_TAB_LOCK(pSibTable);
    sibEntry = sibEntryFindLocked(pDevInfo, macAddr, pOpBss);
    A_SIB_TAB_UNLOCK(pSibTable);

    return sibEntry;
}

/*******************************************************************************
* sibEntryDel - delete the #entryIndex entry which is used
*
* This routine delete the entry with index # of entryIndex
*
* RETURNS:  OK if successful
*           ERROR if unsuccessful
*/
A_STATUS
sibEntryDel(WLAN_DEV_INFO *pDevInfo, SIB_ENTRY *pSib)
{
    SIB_TABLE *pSibTable = pDevInfo->pSibTable;

    A_SIB_TAB_LOCK(pSibTable);
    A_SIB_ENTRY_LOCK(pSib);

    /* check the status of the entry */
    if (!(pSib->used & (SIB_USED|SIB_VALID)) ||
        (pSib->used & SIB_BUSY))
    {
        A_SIB_ENTRY_UNLOCK(pSib);
        A_SIB_TAB_UNLOCK(pSibTable);
        return A_ERROR;
    }

    sibEntryDelLocked(pDevInfo, pSibTable, pSib);

    A_SIB_ENTRY_UNLOCK(pSib);
    A_SIB_TAB_UNLOCK(pSibTable);

    return A_OK;
}

/*
 *  Note, there are some members of the SIB entry that cannot
 *  be set to zero...the SIB entry lock in particular.
 */
LOCAL void
sibEntryDelLocked(WLAN_DEV_INFO *pDevInfo, SIB_TABLE *pSibTable, SIB_ENTRY *pSib)
{
    SIB_ENTRY *pEntry;
    A_UINT32  hindex;
#ifdef WME
    UINT i;
#endif

#ifdef DEBUG
    /* Only need a little information usually. */
    uiPrintf("wlan%d delete STA: ", pDevInfo->devno);
    printMacAddress(pSib->macAddr);
    uiPrintf(" (%d)\n", A_MS_TICKGET());
#endif
    if (pSib->assocId != 0) {
        /* Release the association ID */
        mlmeAssocIdRelease(pDevInfo, pSib->pOpBss, pSib->assocId);
        pSib->assocId = 0;
    }
    pSib->used = SIB_FREE; // after this, sibEntryFind() won't find this entry

    /* change the status of the entry */
    pSib->staState = STATE_QUIET;

    if (pSib->pChlg) {
        /* This could be left over */
        A_DRIVER_FREE(pSib->pChlg, sizeof(CHLG_TEXT));
        pSib->pChlg = NULL;
    }

    /* change the current number of the SIBTable entries */
    pSibTable->currentEntryNum -= 1;

    /* remove it from the linked list */
    hindex = wlanMACHash(pDevInfo->devno, &pSib->macAddr);
    pEntry = pSibTable->pHashList[hindex];
    if (pEntry == pSib) {
        pSibTable->pHashList[hindex] = pSib->pNext;
    } else {
        while (pEntry->pNext != NULL && pEntry->pNext != pSib) {
            pEntry = pEntry->pNext;
        }
        /* We must find a match for pSib, if not, something
         * is seriously wrong. It will crash.
         */
        pEntry->pNext = pSib->pNext;
        /* Place it back at the beginning of the free list */
    }

    A_MACADDR_COPY(&nullMacAddr, &pSib->macAddr);

    pSib->staLifeStartTime = pSib->staLastActivityTime = 0;
    A_MEM_ZERO(&pSib->stats, sizeof(pSib->stats));

    pSib->psMode                    = 0;
#ifdef WME
    for (i = 0; i < NWME; i++) {
	    pSib->lastRxSeqControl[i].data = 0;
	    pSib->lastLastRxSeqControl[i].data = 0;
    }
#else
    pSib->lastRxSeqControl.data     = 0;
    pSib->lastLastRxSeqControl.data = 0;
#endif
    pSib->privacyInvoked            = FALSE;
    pSib->pNext                     = pSibTable->pFreeList;
    pSibTable->pFreeList            = pSib;
}

/*******************************************************************************
* sibEntryAlloc - Allocate a SIB entry from sibTable.
*
* This routine searches the SIB table for a free entry, marks
* it used and returns it to the caller. It is not marked valid or
* added to the hash table.
*
* RETURNS:  the pointer of the entry if successful
*           NULL if unsuccessful
*/

SIB_ENTRY *
sibEntryAlloc(WLAN_DEV_INFO *pDevInfo, WLAN_MACADDR *macAddr, OP_BSS *pOpBss)
{
    SIB_TABLE *pSibTable = pDevInfo->pSibTable;
    SIB_ENTRY *sibEntry, *victim;
    A_UINT32  wdcConnId;
#ifdef WME
    UINT i;
#endif

    A_SIB_TAB_LOCK(pSibTable);

    sibEntry = sibEntryFindLocked(pDevInfo, macAddr, pOpBss);
    if (sibEntry) {
        A_SIB_TAB_UNLOCK(pSibTable);
        return sibEntry;
    }

    /*
     *  If SIB table is full, go looking for a stale STA
     */
    if (pSibTable->currentEntryNum >= pSibTable->sibTableSize) {
        if ((victim = sibVictimFindLocked(pDevInfo, pDevInfo->localSta, pDevInfo->bssDescr)) != NULL) {
            A_SIB_TAB_UNLOCK(pSibTable);

            wdcConnId = victim->wdcConnId;

            /* Delete host connection
             * - delete host side first, so that the host doesn't continue to use the
             *   connection while the target is in the process of deleting the connection.
             */

            /* Delete the connection key while we still have it */
            ASSERT(victim->wdcConnId < (A_INT32)(pDevInfo->pSibTable->sibTableSize));
            wlanDeleteConnectionKey(pDevInfo, victim, victim->pPriv);
            victim->pPriv = NULL;

            if (sibEntryDel(pDevInfo, victim) == A_ERROR) {
                uiPrintf("sibEntryAlloc:  Could not remove inactive "
                         "SIB to make room for new SIB.\n");
                return NULL;
            }

            /* Flush pending tx requests
             * before deleting the target connection
             */
            NdisInterlockedIncrement(&pDevInfo->numFlushCall);
            wdcFlush(pDevInfo->targetHandle,pDevInfo->disableEnableCounter);

            /* Delete target connection */
            wdcDeleteConnection(pDevInfo->targetHandle, wdcConnId);

            A_SIB_TAB_LOCK(pSibTable);
        } else {
            A_SIB_TAB_UNLOCK(pSibTable);
            uiPrintf("Access denied ");
            printMacAddress(*macAddr);
            uiPrintf(" out of resources (SIB)\n");
            return NULL;
        }
    }

    sibEntry = pSibTable->pFreeList;
    ASSERT(sibEntry != NULL);

    if (sibEntry->used != SIB_FREE) {
        A_SIB_TAB_UNLOCK(pSibTable);
        uiPrintf("sibEntryAlloc: ERROR: Program bug! Entry not marked free\n");
        return NULL;
    }
    /* Move free list pointer to the next entry */
    pSibTable->pFreeList = sibEntry->pNext;

    /* Initialize various fields to defaults */
    A_MACADDR_COPY(macAddr, &sibEntry->macAddr);
    sibEntry->pNext = NULL;
    sibEntry->used  = SIB_USED;

    /* Initial timestamps */
    sibEntry->staLifeStartTime = A_MS_TICKGET();

    sibEntry->pOpBss                = pOpBss;
    sibEntry->hwIndex               = HWINDEX_INVALID;
    sibEntry->pPriv                 = NULL;
    sibEntry->pBridgeApSibEntry     = NULL;
    sibEntry->pChlg                 = NULL;

#ifdef WME
    for (i = 0; i < NWME; i++) {
        sibEntry->lastRxSeqControl[i].data = -1; /* impossible value */
    }
#else
    sibEntry->lastRxSeqControl.data = -1; /* impossible value */
#endif
    sibEntry->pTxFFDesc             = NULL;
    sibEntry->pRxFFDesc             = NULL;
    /*
     *  The previous user of this TXMASK entry may have left it
     *  set, so be sure our first frame clears it.
     */
    sibEntry->antTx             = 0;
    A_MEM_ZERO((char *)&sibEntry->stats, sizeof(sibEntry->stats));

    TRACEBUF_INIT(Tx, sibEntry);
    TRACEBUF_INIT(Rx, sibEntry);

    A_SIB_TAB_UNLOCK(pSibTable);

    return sibEntry;
}

/**************************************************************************
* sibEntryAdd - add a entry to the SIBTable
*
* This routine check the specific entry and set it to the valid status
*
* The caller must pass an entry already marked as "used".
*
* RETURNS: A_OK or A_WIN_DRIVER_ERROR.
*/
A_STATUS
sibEntryAdd(SIB_ENTRY *sibEntry)
{
    A_STATUS      status;
    WLAN_DEV_INFO *pDevInfo;
    SIB_TABLE     *pSibTable;

    pDevInfo  = sib2DevInfo(sibEntry);
    pSibTable = pDevInfo->pSibTable;
    A_SIB_TAB_LOCK(pSibTable);
    if (sibEntryFindLocked(pDevInfo, &sibEntry->macAddr, sibEntry->pOpBss)) {
        A_SIB_TAB_UNLOCK(pSibTable);
        return A_OK;
    }
    status = sibEntryAddLocked(pDevInfo, sibEntry);
    A_SIB_TAB_UNLOCK(pSibTable);
    return status;
}

LOCAL A_STATUS
sibEntryAddLocked(WLAN_DEV_INFO *pDevInfo, SIB_ENTRY *sibEntry)
{
    A_UINT32  hindex;
    SIB_TABLE *pSibTable = pDevInfo->pSibTable;

    if (sibEntry->used != SIB_USED) {
        /* must be wrong if reach here */
        uiPrintf("sibEntryAdd: ERROR: sibEntry in wrong state!\n");
        return A_WIN_DRIVER_ERROR;
    }

    /* It is in the correct state. */

    /*
     * Add it to the linked list
     * Add it to the beginning of the list. Since sibEntryAdd
     * will be soon followed by transmit/receive, some thread
     * will soon start searching for it. Better to keep it
     * at the beginning.
     */
    hindex = wlanMACHash(pDevInfo->devno, &sibEntry->macAddr);
    sibEntry->used = SIB_USED | SIB_VALID;

    A_SIB_ENTRY_LOCK(sibEntry);
    sibEntry->staLastActivityTime = A_MS_TICKGET();
    sibEntry->pNext               = pSibTable->pHashList[hindex];
    A_SIB_ENTRY_UNLOCK(sibEntry);

    pSibTable->pHashList[hindex] = sibEntry;
    pSibTable->currentEntryNum++ ;

#ifdef DEBUG
    /* Only need a little information usually. */
    uiPrintf("wlan%d added STA: ", pDevInfo->devno);
    printMacAddress(sibEntry->macAddr);
    uiPrintf(" (%d)\n", A_MS_TICKGET());
#endif

    return A_OK;
}

/*******************************************************************************
* sibEntryAgeCheck - Check if a sib entry is stale, and should be removed.
*
* Assumes caller holds the per-SIB lock.
*
* RETURNS:  FALSE if this sib is still valid. TRUE if the sib can be
*           removed with sibEntryAge().
*/
A_BOOL
sibEntryAgeCheck(SIB_ENTRY *pSib, A_UINT32 aging_period)
{
    QUEUE_DETAILS *pQueue;
    A_UINT32      lasttime;
    A_BOOL        idle;

    if (aging_period == 0) {
        return FALSE;
    }

    lasttime = pSib->staLastActivityTime;
    pQueue   = &pSib->txPSQueue;
    idle     = TRUE;
    if (pQueue->pDescQueueHead ||
        A_ATOMIC_READ(&pQueue->qFrameCount) ||
        (pSib->used & SIB_BUSY))
    {
        idle = FALSE;
    }

    if ((A_MS_TICKGET() - lasttime) >= aging_period) {
#if defined(DEBUG) && defined(BUILD_AP)
        if (agingDebug) {
            char buf[80];

            cliUptimeGet(buf);
            printf("%s -- ", buf);
            printMacAddress(pSib->macAddr);
            printf(" %s\n",
                   !idle ? "busy" : pSib->assocId & 0x3fff ? "disassociated" : "aged out");
        }
#endif
        return idle;
    }
    return FALSE;                       /* still has time left */
}

/*
 * sibEntryAge - handle an elderly SIB in 2 easy steps.
 *   1a) clean up any association state.  This does not send a frame.
 *   1b) send a deauth frame.  For power save stations this will likely
 *       be dropped, but currently is sent anyway.  The station will
 *       happily think it's associated.
 *   2)  remove the sib entry if disassociated.
 *
 * Should not hold the sib lock when calling.
 *
 * XXX Should handle the deauth frame better.  Either not send for
 *     PS stations to save air, or send it before tearing down
 *     the SIB to let it go on the PS queue and tear down the SIB
 *     after the tx is done.  In some ways it's better for the STA
 *     to not know the AP has aged it to avoid thrashing.
 */
void
sibEntryAge(WLAN_DEV_INFO *pDevInfo, SIB_ENTRY *pSib)
{

    if (pSib->staState & STATE_AUTH) {
#ifdef DEBUG
        /* Only need a little information usually. */
        uiPrintf("wlan%d aged STA: ",pDevInfo->devno);
        printMacAddress(pSib->macAddr);
        uiPrintf(" (%d)\n", A_MS_TICKGET());
#endif
        wlanMlmeDeauthRequest(pDevInfo, pSib->pOpBss, &pSib->macAddr, REASON_AUTH_EXPIRED, TRUE);
    } else {
        sibEntryDel(pDevInfo, pSib);
    }
}

void
psStateReset(WLAN_DEV_INFO *pDevInfo, SIB_ENTRY *pSib, COMPLETION_ENUM complete)
{
    /* release all the PS queue resource assuming that PS queues are
     * in the host side
     */
    freeBuffandDescChain(pDevInfo, pSib->txPSQueue.pDescQueueHead);

    return;
}


void
sibTableStateReset(WLAN_DEV_INFO *pDevInfo, COMPLETION_ENUM complete)
{
    SIB_TABLE *pSibTable = pDevInfo->pSibTable;
    SIB_ENTRY *pSib;
    A_UINT32  i;

    A_SIB_TAB_LOCK(pSibTable);
    for (i = 0; i < pSibTable->sibTableSize; i++) {
        pSib = &pSibTable->sta[i];
        if (((pSib->used & SIB_VALID) == 0) ||
            (pSib == pDevInfo->localSta))
        {
            continue;
        }
        /* JK - should delete the target connection too */
        /* free PS states */
        psStateReset(pDevInfo, pSib, complete);
        pSib->psMode = 0;
    }
    A_SIB_TAB_UNLOCK(pSibTable);
    return;
}

char *sibStateStrGet(SIB_ENTRY *pSib)
{
    char *p = "none";

    if (pSib->staState & STATE_AUTH)
        p = "authenticated";
    if (pSib->staState & STATE_ASSOC)
        p = "associated";
    if (pSib->staState & STATE_JOINED)
        p = "joined";
    if (pSib->staState & STATE_AP_UP)
        p = "up";
    if (pSib->staState & STATE_AP_DOWN)
        p = "down";

    return p;
}


/*
 *  We've run out of free slots in the SIB table.
 *  Return the LRU item so we can kick it out.
 *
 *  Do not victimize ourselves, or the AP BSSDESCR.
 *
 *  Assumption: A_SIB_TAB_LOCK(pSibTable) has been called.
 */
LOCAL SIB_ENTRY *
sibVictimFindLocked(WLAN_DEV_INFO *pDevInfo, SIB_ENTRY *localSib, BSSDESCR *apDescr)
{
    SIB_ENTRY          *pSib, *victim = NULL;
    A_UINT32           idx;
    const WLAN_MACADDR dummyAddr = {{0x00, 0x03, 0x7f, 0x00, 0x00, 0x00}};

    if (localSib == NULL) {
        return NULL;
    }

    for (idx = 0; idx < pDevInfo->pSibTable->sibTableSize; ++idx) {
        pSib = &pDevInfo->pSibTable->sta[idx];

        if (!(pSib->used & SIB_VALID) || (pSib->used & SIB_BUSY)) {
            continue;
        }

        /* Don't torch ourself */
        if (pSib == localSib) {
            continue;
        }

        /* Don't torch our partner AP if there is one */
        if (localSib->serviceType == WLAN_STA_SERVICE &&
            pSib->serviceType == WLAN_AP_SERVICE &&
            apDescr != NULL &&
            A_MACADDR_COMP(&pSib->macAddr, &apDescr->bssId) == 0)
        {
            continue;
        }

        if (localSib->serviceType == WLAN_STA_SERVICE &&
            A_MACADDR_COMP(&pSib->macAddr, &dummyAddr) == 0)
        {
            continue;
        }

        /* if we already have a better victim */
        if (victim &&
            pSib->staLastActivityTime >= victim->staLastActivityTime)
        {
            continue;
        }

        if ((localSib->serviceType == WLAN_STA_SERVICE)) {

            /*
             * Predator STA:
             *
             * The current code base does not maintain a SIB
             * reference count. Instead, an aging mechanism is used to ensure
             * the SIB is no longer is in use.
             *
             * Predator has a limited number of connections (SIBs),
             * so the aging mechanism cannot be relied upon to find
             * a SIB that is not in use.
             *
             * Adding SIB reference counting is considered too
             * difficult at this point for several reasons
             *
             * 1. Time constrained.  We do not want to have to
             *    rewrite/debug the upper WLAN code.
             * 2. A thorough understanding of each modules
             *    use of the SIB is required
             * 3. The data path acquires the SIB pointer
             *    multiple times. There is not a one-to-one
             *    correspondence between SIB reference and
             *    SIB de-reference.
             * 4. Reference count debugging would be a nightmare
             *
             * Instead, Predator STA will use the following SIB replacement
             * strategy:
             *
             * Infra:
             *      Replace the least recently used SIB.
             *
             *      For the STA only one connection is active,
             *      so deleting the oldest connection is safe.
             *
             *
             * Adhoc:
             *      SIB aging
             *
             *      Adhoc connections are created when a
             *      frame is received from a STA.
             *      An adhoc SIB will be replaced if the connection
             *      has been inactive for a period of
             *      time (aging period). If all connections are
             *      active, then new connections will be rejected.
             *      Note: no aging is applied to ESS SIBs. These
             *      can be immediately reused.
             */
            if ( apDescr != NULL && (apDescr->bsstype == INDEPENDENT_BSS) &&
                pSib->capInfo.ibss)
            {
                /* assert a limit (60s) to prevent thrashing and ensure
                 * no ref counts are out to this one
                 */
                if (!sibEntryAgeCheck(pSib, 60 * 1000)) {
                    continue;
                }
            }

        } else {
            /* assert a limit (60s) to prevent thrashing and ensure
             * no ref counts are out to this one
             */
            if (!sibEntryAgeCheck(pSib, 60 * 1000)) {
                continue;
            }
        }

        victim = pSib;
    }

    return victim;
}

int
macaddrcmp(SIB_ENTRY **pp1, SIB_ENTRY **pp2)
{
    SIB_ENTRY *p1;
    SIB_ENTRY *p2;

    p1 = pp1[0];
    p2 = pp2[0];
    return A_BCOMP(p1->macAddr.octets, p2->macAddr.octets, sizeof(WLAN_MACADDR));
}

/*******************************************************************************
 * roundUpRate
 *
 * Given a rate (in kbps), returns the rate (in kbps) that corresponds the the
 * next highest rate.  This is useful for displaying transmit rate from a LPFed
 * value, so that the user is not confused by the LPFed value.
 */
static A_UINT32
roundUpRate(WLAN_DEV_INFO *pDev, A_UINT32 rate, A_BOOL validChk)
{
    A_UINT8           i;
    A_UINT32          thisRate = 0;
    const RATE_TABLE  *pRates;
    WIRELESS_MODE     mode;

    mode = wlanCFlagsToWirelessMode(pDev, pDev->staConfig.pChannel->channelFlags);
    pRates = pDev->hwRateTable[mode];

    if (pDev->staConfig.rateCtrlEnable == FALSE) {
        return rate;
    }

    /*
     * Since the rates in the Rate Table may not be monotonically increasing,
     * the entire table must be traversed to find the closest matching rate.
     */
    if (rate > pRates->info[pRates->rateCount-1].rateKbps) {
        /* This case should never happen but needed for defensive programming */
        thisRate = pRates->info[pRates->rateCount-1].rateKbps;
    } else {
        for (i = 0; i < pRates->rateCount; i++) {
            if (!validChk || pRates->info[i].valid) {
                if (i == 0 && rate <= pRates->info[i].rateKbps) {
                    thisRate = pRates->info[i].rateKbps;
                    break;
                }
                if (pRates->info[i].rateKbps >= rate) {
                    /* Must account for possible non-monotinicity in  rate table. */
                    if (thisRate <= 0 || pRates->info[i].rateKbps < thisRate) {
                        thisRate = pRates->info[i].rateKbps;
                    }
                }
            }
        }
    }
    return thisRate;
}

A_UINT32
staTxRateKbpsGet(WLAN_DEV_INFO *pDevInfo, SIB_ENTRY *pSib)
{
    A_UINT32 rate;

    rate = roundUpRate(pDevInfo, A_RATE_OUT(pSib->stats.txRateKb), TRUE);

    if (IS_CHAN_TURBO(pDevInfo->staConfig.pChannel->channelFlags)) {
        rate *= 2;
    }

    return rate;
}

A_UINT32
staTxRateGet(WLAN_DEV_INFO *pDevInfo, SIB_ENTRY *pSib)
{
    A_UINT32 rate;

    rate  = staTxRateKbpsGet(pDevInfo, pSib);
    rate /= 1000;

    return rate;
}

/*
 * Fix the rate, or use best rate if rateKbps is 0.
 */
void
staTxRateSet(WLAN_DEV_INFO *pDevInfo, A_UINT16 rateKbps)
{
    if (!rateKbps) {
        pDevInfo->staConfig.rateCtrlEnable = TRUE;
        return;
    }

    wlanUpdateDefaultRate(pDevInfo, rateKbps);

    pDevInfo->staConfig.rateCtrlEnable = FALSE;
}

A_UINT32
staRxRateGet(WLAN_DEV_INFO *pDevInfo, SIB_ENTRY *pSib)
{
    A_UINT32 rate;

    rate = roundUpRate(pDevInfo, pSib->stats.RcvRate, FALSE);
    /* Multiply first before dividing to take care of 5.5Mbps */
    if (IS_CHAN_TURBO(pDevInfo->staConfig.pChannel->channelFlags)) {
        rate *= 2;
    }
    rate /= 1000;
    return rate;
}

A_UINT8
staRateValueToIndex(A_UINT32 txRateKbps, SIB_ENTRY *pSib)
{
    const RATE_TABLE *pRateTable;
    A_UINT8      rate  = 0;
    A_UINT8      index = 0;

    if (NULL == pSib) {
        return index;
    }

    pRateTable  = pSib->pOpBss->pRateTable;

    while (rate < pRateTable->rateCount) {
        if (pRateTable->info[rate].valid &&
            txRateKbps == pRateTable->info[rate].rateKbps)
        {
            index = rate;
            break;
        }
        rate++;
    }

    return index;
}

A_UINT8
staGetSigQuality(A_UINT8 txRate, SIB_ENTRY *pSib)
{
    const RATE_TABLE *pRateTable;
    A_UINT32     rateKbps;
    A_UINT32     maxRateKbps;
    A_UINT8      qual = 0;

    if (NULL == pSib) {
        return qual;
    }

    pRateTable  = pSib->pOpBss->pRateTable;
    maxRateKbps = pRateTable->info[pRateTable->rateCount - 1].rateKbps;
    rateKbps    = pRateTable->info[txRate].rateKbps;

    qual = (A_UINT8)((rateKbps * 100) / maxRateKbps);
    return qual;
}

#ifdef BUILD_AP
#ifdef INCLUDE_RCW

LOCAL int
macAddr2Sib(char *addr, SIB_ENTRY *pSibCopy, int devno)
{
    int           cnt, idx;
    SIB_ENTRY     *pSib;
    SIB_TABLE     *pSibTable;
    char          macaddr[6];
    WLAN_DEV_INFO *pDevInfo;


    strtoWlanMAC(addr, (WLAN_MACADDR *)macaddr);

    A_MEM_ZERO(pSibCopy, sizeof(struct sibEntry));

    pDevInfo  = gDrvInfo.pDev[devno];
    pSibTable = pDevInfo->pSibTable;
    A_SIB_TAB_LOCK(pSibTable);
    cnt = pSibTable->currentEntryNum;
    if (cnt) {
        for (idx = pSibTable->sibTableSize - 1; idx >= 0; idx--) {
            pSib = &pSibTable->sta[idx];
            if (bcmp((char *)pSib->macAddr.octets, macaddr, sizeof(macaddr)) == 0) {
                A_BCOPY((char *)pSib, (char *)pSibCopy, sizeof(SIB_ENTRY));
                if (pSib->used & SIB_VALID &&
                    (pSib->assocId & 0x3fff || pSib->serviceType == WLAN_AP_SERVICE))
                {
                    A_SIB_TAB_UNLOCK(pSibTable);
                    return OK;
                }
                break;
            }
        }
    }
    A_SIB_TAB_UNLOCK(pSibTable);

    return ERROR;
}

/*
 * Initial station table for display
 */
int
staListTblInit(int devno)
{
    int       idx, j;
    int       cnt;
    SIB_ENTRY *pSib;
    SIB_TABLE *pSibTable;

    pSibTable = gDrvInfo.pDev[devno]->pSibTable;
    if (pSibTable == NULL) {
        return staListNum = 0;
    }

    j = 0;
    A_SIB_TAB_LOCK(pSibTable);

    cnt = pSibTable->currentEntryNum;
    if (cnt) {
        for (idx = pSibTable->sibTableSize - 1; idx >= 0; idx--) {
            pSib = &pSibTable->sta[idx];
            if (pSib->used & SIB_VALID &&
                (pSib->assocId & 0x3fff || pSib->serviceType == WLAN_AP_SERVICE))
            {
                pStaListTbl[j++] = pSib;
                cnt--;
                if (cnt == 0) {
                    break;
                }
            }
        }
    }

    A_SIB_TAB_UNLOCK(pSibTable);

    if (j > 1) {
        qsort(&pStaListTbl[1], j-1, sizeof(SIB_ENTRY *), (void *)&macaddrcmp);
    }

    return staListNum = j;
}


int
staMacAddrGet(int index, char *p)
{
    char      macaddr[20];
    SIB_ENTRY *pSib;

    pSib = pStaListTbl[index];
    sprintf(macaddr, "%02X:%02X:%02X:%02X:%02X:%02X",
            pSib->macAddr.octets[0], pSib->macAddr.octets[1],
            pSib->macAddr.octets[2], pSib->macAddr.octets[3],
            pSib->macAddr.octets[4], pSib->macAddr.octets[5]);
    strcpy(p, macaddr);

    return index;
}

int
staStatusGet(int index, char *p)
{
    SIB_ENTRY *pSib;

    pSib = pStaListTbl[index];
    strcpy(p, sibStateStrGet(pSib));

    return index;
}


int
staIdGet(int index, char *p)
{
    SIB_ENTRY *pSib;

    pSib = pStaListTbl[index];
    if (pSib->serviceType == WLAN_STA_SERVICE) {
        sprintf(p, "STA %d", pSib->assocId & 0x3fff);
    } else {
        strcpy(p, "AP");
    }

    return index;
}

static const char *
staGetRateStr(WLAN_DEV_INFO *pDevInfo, A_UINT8 rateIdx, char **pModeStr)
{
    int              i;
    int              rateShift;
    int              thisRate;
    const RATE_TABLE *pRates;
    WIRELESS_MODE    mode;

    mode      = wlanCFlagsToWirelessMode(pDevInfo, pDevInfo->staConfig.pChannel->channelFlags);
    pRates    = pDevInfo->hwRateTable[mode];
    rateShift = 1;

    switch (mode) {
    case WIRELESS_MODE_TURBO:
        rateShift = 2;
        *pModeStr = txt_turbo;
        break;
    case WIRELESS_MODE_11a:
        *pModeStr = txt_11a;
        break;
    case WIRELESS_MODE_11g:
        rateShift = !pDevInfo->turboPrimeInfo.currentBoostState ? 1 : 2;
        *pModeStr = txt_11g;
        break;
    case WIRELESS_MODE_108g:
        rateShift = 2;
        *pModeStr = txt_108g;
        break;
    case WIRELESS_MODE_11b:
        *pModeStr = txt_11b;
        break;
    default:
        *pModeStr = "unknown";
        break;
    }

    thisRate = rateShift * pRates->info[rateIdx].rateKbps;
    for (i = 0; i < sizeof(dataRateStrTbl) / sizeof(RATE_TO_STR_MAP) - 1; i++) {
        if (thisRate == dataRateStrTbl[i].rate) {
            break;
        }
    }

    return dataRateStrTbl[i].str;
}

void
staInfoGet(char *pMacAddr, rcw_SibTbl *pInfo, int devno)
{
    WLAN_DEV_INFO *pDevInfo;
    SIB_ENTRY     sibCopy;
    SIB_ENTRY     *pSib;
    char          *pCipher, *pModeStr;
    const char    *pRateStr;
    SCB           *s;

    pSib = (SIB_ENTRY *)&sibCopy;
    if (macAddr2Sib(pMacAddr, pSib, devno) != OK) {            /* XXX mwlan */
        return;
    }

    pDevInfo = gDrvInfo.pDev[devno];
    pRateStr = staGetRateStr(pDevInfo, pSib->stats.RcvRate, &pModeStr);
    sprintf(pInfo->sib_Id, "%d", pSib->assocId & 0x3fff);
    sprintf(pInfo->sib_MacAddr, "%02X:%02X:%02X:%02X:%02X:%02X",
            pSib->macAddr.octets[0], pSib->macAddr.octets[1],
            pSib->macAddr.octets[2], pSib->macAddr.octets[3],
            pSib->macAddr.octets[4], pSib->macAddr.octets[5]);
    strcpy(pInfo->sib_State, sibStateStrGet(pSib));
    strcpy(pInfo->sib_Identity, "");
    if ((s = dot1xSessionGet(pDevInfo, pSib))) {
        if (strlen(s->radiusInfo.identity) <= sizeof(pInfo->sib_Identity) - 1) {
            strcpy(pInfo->sib_Identity, s->radiusInfo.identity);
        } else {
            strncpy(pInfo->sib_Identity, s->radiusInfo.identity, sizeof(pInfo->sib_Identity));
            pInfo->sib_Identity[sizeof(pInfo->sib_Identity)-1] = 0;
        }
    }
    strcpy(pInfo->sib_PwrSave, pSib->psMode ? "on" : "off");
    strcpy(pInfo->sib_NonErpType, "");
    if (pSib->serviceType == WLAN_STA_SERVICE) {
        strcpy(pInfo->sib_WlanMode, pSib->wlanMode == STA_MODE_A ? txt_11a :
               (pSib->wlanMode == STA_MODE_B ? txt_11b :
                (pSib->wlanMode == STA_MODE_G ? txt_11g : txt_108g)));
        if (pSib->wlanMode == STA_MODE_G) {
            sprintf(pInfo->sib_NonErpType, "%s %s",
                    pDevInfo->bssDescr->nonErpElement.info.present ? "Non-ERP" : "",
                    pDevInfo->bssDescr->nonErpElement.info.protect ? "protection" : "");
        }
    } else {
        strcpy(pInfo->sib_WlanMode, pModeStr);
    }
    strcpy(pInfo->sib_Wep, pSib->privacyInvoked ? "yes" : "no");
    strcpy(pInfo->sib_CipherType, "None");
    strcpy(pInfo->sib_UcastCipher, "None");
    strcpy(pInfo->sib_McastCipher, "None");

    if (pSib->privacyInvoked) {
        if (pSib->serviceType == WLAN_STA_SERVICE) {
            if (VALID_CIPHER_OUI(pSib->uCipher)) {
                strcpy(pInfo->sib_UcastCipher,
                       sibCipherType(pSib->uCipher));
            }
            if (VALID_CIPHER_OUI(pSib->mCipher)) {
                strcpy(pInfo->sib_McastCipher,
                       sibCipherType(pSib->mCipher));
            }
        }
        if (VALID_CIPHER_OUI(pSib->uCipher)) {
            strcpy (pInfo->sib_CipherType, "");
            pCipher = sibCipherType(pSib->uCipher);
            strcat(pInfo->sib_CipherType, " ");
            strcat(pInfo->sib_CipherType, pCipher);
            pCipher = "";
        }
    }

    strcpy(pInfo->sib_AuthType, authTypeGet(pSib->authenAlgo));

    // Since the user application doesnt maintain a seperate reassociation count
    // add it to the association count
    pInfo->sib_AssocCnt    = pSib->stats.Associations;
    pInfo->sib_ReassocCnt  = pSib->stats.Reassociations;
    pInfo->sib_DisassocCnt = pSib->stats.DisAssociations;
    pInfo->sib_AuthCnt     = pSib->stats.Authentications;
    pInfo->sib_DeauthCnt   = pSib->stats.DeAuthentications;
    pInfo->rx_Cnt          = pSib->stats.GoodReceives;
    pInfo->rx_Data         = pSib->stats.RxDataFrames;
    pInfo->rx_Mcast        = pSib->stats.MulticastReceives;
    pInfo->rx_Mgmt         = pSib->stats.RxMgmtFrames;
    pInfo->rx_Ctrl         = pSib->stats.RxCtrlFrames;
    pInfo->rx_Errs         = pSib->stats.ReceiveErrors;
    pInfo->rx_AckErr       = pSib->stats.AckRcvFailures;
    pInfo->rx_CrcErr       = pSib->stats.RcvCrcErrors;
    pInfo->rx_DecryptErr   = pSib->stats.RcvDecryptCrcErrors;
    pInfo->rx_DiscardErr   = pSib->stats.RxDiscardFrames;
    pInfo->rx_DmaErr       = pSib->stats.RcvDmaOverrunErrors;
    pInfo->rx_DupFrameErr  = pSib->stats.MultipleRxDuplicates;
    pInfo->rx_FcsErr       = pSib->stats.FcsFailCnt;
    pInfo->rx_PhyErr       = pSib->stats.RcvPhyErrors;
    pInfo->rx_RtsErr       = pSib->stats.RtsFailCnt;
    pInfo->rx_WepXErr      = pSib->stats.RcvWEPExcludedCount;
    pInfo->rx_Ssl          = pSib->stats.rssi;

    strcpy(pInfo->rx_Rate, pRateStr);

    pInfo->tx_Cnt          = pSib->stats.GoodTransmits;
    pInfo->tx_Data         = pSib->stats.TxDataFrames;
    pInfo->tx_Mcast        = pSib->stats.TxMulticastFrames +
                             pSib->stats.TxBroadcastFrames;
    pInfo->tx_Mgmt         = pSib->stats.TxMgmtFrames;
    pInfo->tx_Ctrl         = pSib->stats.TxCtrlFrames;
    pInfo->tx_Errs         = pSib->stats.TransmitErrors;
    pInfo->tx_DiscardErr   = pSib->stats.TxFramesDropped;
    pInfo->tx_DmaErr       = pSib->stats.TxDmaUnderrun;
    pInfo->tx_XretryErr    = pSib->stats.TxExcessiveRetries;
    pInfo->tx_Ssl          = A_RSSI_OUT(pSib->stats.ackRssi);

    if (pDevInfo->staConfig.rateCtrlEnable) {
        int rate = staTxRateGet(pDevInfo, pSib);
        sprintf(pInfo->tx_sfRate, "%d", rate);
        sprintf(pInfo->tx_lfRate, "%d", rate);
    } else {
        strcpy(pInfo->tx_sfRate, staGetRateStr(pDevInfo, pDevInfo->baseBss.defaultRateIndex, &pModeStr));
        strcpy(pInfo->tx_lfRate, staGetRateStr(pDevInfo, pDevInfo->baseBss.defaultRateIndex, &pModeStr));
    }
}


LOCAL SIB_ENTRY *
findMacAddrSib(WLAN_DEV_INFO *pDevInfo, char *addr)
{
    int         cnt, idx;
    SIB_ENTRY   *pSib;          /* a entry from SIB Table */
    SIB_TABLE   *pSibTable;
    char        macaddr[6];

    strtoWlanMAC(addr, (WLAN_MACADDR *)macaddr);

    pSibTable = pDevInfo->pSibTable;
    A_SIB_TAB_LOCK(pSibTable);
    cnt = pSibTable->currentEntryNum;
    if (cnt) {
        for (idx = pSibTable->sibTableSize - 1; idx >= 0; idx--) {
            pSib = &pSibTable->sta[idx];
            if (bcmp(pSib->macAddr.octets, macaddr, sizeof(macaddr)) == 0) {
                if (pSib->used & SIB_VALID) {
                    A_SIB_TAB_UNLOCK(pSibTable);
                    return pSib;
                }
                break;
            }
        }
    }

    A_SIB_TAB_UNLOCK(pSibTable);

    return NULL;
}


void
staDot1xInfoGet(char *pMacAddr, rcw_Dot1x *pInfo)
{
    SCB           *s;
    SIB_ENTRY     *pSib;
    int           i;
    WLAN_DEV_INFO *pDevInfo;


    bzero((char *)pInfo, sizeof(rcw_Dot1x));
    pInfo->reAuthPeriod = dot1xReAuthPeriodGet();

    for (i = 0; i < WLAN_MAX_DEV; i++) {
        pDevInfo = gDrvInfo.pDev[i];
        if (pDevInfo) {
            pSib = findMacAddrSib(pDevInfo, pMacAddr);
            if (pSib) {
                s = dot1xSessionGet(pDevInfo, pSib);
                if (s) {
                    strcpy(pInfo->identity, s->radiusInfo.identity);
                    strcpy(pInfo->authState, dot1xStateStr[s->stateAuth.state]);
                    strcpy(pInfo->asState, dot1xStateStr[s->stateAS.state]);
                    pInfo->EntersConnecting =
                        s->AuthCounter.authEntersConnecting;
                    pInfo->EapLogoffsWhileConnecting =
                        s->AuthCounter.authEapLogoffsWhileConnecting;
                    pInfo->EntersAuthenticating =
                        s->AuthCounter.authEntersAuthenticating;
                    pInfo->SuccessesWhileAuthenticating =
                        s->AuthCounter.authAuthSuccessesWhileAuthenticating;
                    pInfo->TimeoutsWhileAuthenticating =
                        s->AuthCounter.authAuthTimeoutsWhileAuthenticating;
                    pInfo->FailWhileAuthenticating =
                        s->AuthCounter.authAuthFailWhileAuthenticating;
                    pInfo->FailWhileAuthenticating =
                        s->AuthCounter.authAuthFailWhileAuthenticating;
                    pInfo->ReauthsWhileAuthenticating =
                        s->AuthCounter.authAuthReauthsWhileAuthenticating;
                    pInfo->EapStartsWhileAuthenticating =
                        s->AuthCounter.authAuthEapStartsWhileAuthenticating;
                    pInfo->EapLogoffWhileAuthenticating =
                        s->AuthCounter.authAuthEapLogoffWhileAuthenticating;
                    pInfo->ReauthsWhileAuthenticated =
                        s->AuthCounter.authAuthReauthsWhileAuthenticated;
                    pInfo->EapStartsWhileAuthenticated =
                        s->AuthCounter.authAuthEapStartsWhileAuthenticated;
                    pInfo->EapLogoffWhileAuthenticated =
                        s->AuthCounter.authAuthEapLogoffWhileAuthenticated;
                    pInfo->Responses =
                        s->AuthServerCounter.backendResponses;
                    pInfo->AccessChallenges =
                        s->AuthServerCounter.backendAccessChallenges;
                    pInfo->OtherRequestsToSupplicant =
                        s->AuthServerCounter.backendOtherRequestsToSupplicant;
                    pInfo->NonNakResponsesFromSupplicant =
                        s->AuthServerCounter.backendNonNakResponsesFromSupplicant;
                    pInfo->AuthSuccesses =
                        s->AuthServerCounter.backendAuthSuccesses;
                    pInfo->AuthFails =
                        s->AuthServerCounter.backendAuthFails;
                }
                break;
            }
        }
    }
}

#endif  /* INCLUDE_RCW */

/*
 * wlanIdCmp
 *
 * Sort the SIBs by their wlan ID
 * - used by qsort in sibSmeShow
 */
static int
wlanIdCmp(SIB_ENTRY **pp1, SIB_ENTRY **pp2)
{
    SIB_ENTRY     *p1;
    SIB_ENTRY     *p2;
    WLAN_DEV_INFO *pDev1;
    WLAN_DEV_INFO *pDev2;

    p1    = pp1[0];
    pDev1 = sib2DevInfo(p1);
    p2    = pp2[0];
    pDev2 = sib2DevInfo(p2);

    if (pDev1->devno > pDev2->devno) {
        return +1;
    } else if (pDev1->devno == pDev2->devno) {
        return 0;
    }

    return -1;
}

/*
 *  Display AP association table
 */
void
sibSmeShow(int unit)
{
    int           cnt, idx, j;
    SIB_ENTRY     *pSib;
    SIB_ENTRY     *pTbl[MAX_STA_NUM];
    SIB_TABLE     *pSibTable;
    WLAN_DEV_INFO *pDevInfo;
    char          *pStr;

    pDevInfo  = gDrvInfo.pDev[unit];
    pSibTable = pDevInfo->pSibTable;
    j = 0;

    A_SIB_TAB_LOCK(pSibTable);

    cnt = pSibTable->currentEntryNum;
    if (cnt) {
        for (idx = pSibTable->sibTableSize - 1; idx >= 0; idx--) {
            pSib = &pSibTable->sta[idx];
            if (pSib->used & SIB_VALID &&
                (pSib->serviceType == WLAN_AP_SERVICE ||
                 pSib->staState & STATE_ASSOC))
            {
                pTbl[j++] = pSib;
                cnt--;
                if (cnt == 0) {
                    break;
                }
            }
        }
    }

    A_SIB_TAB_UNLOCK(pSibTable);

    if (j > 1) {
        qsort(&pTbl[0], j, sizeof(SIB_ENTRY *), (void *)&wlanIdCmp);
    }

    uiPrintf(" STA     DEV\tMAC Address\t\tState\t\tKeytype\n");
    for (idx = 0; idx < j; idx++) {
        pSib = pTbl[idx];

        if (pSib->serviceType == WLAN_AP_SERVICE) {
            uiPrintf("%4s     ","AP");
        } else {
            uiPrintf("%4d%s", pSib->assocId & 0x3fff, isXrBssSib(pDevInfo, pSib) ? "(XR) " : "     ");
        }

        uiPrintf("wlan%d\t%02X:%02X:%02X:%02X:%02X:%02X\t%s",
                 pDevInfo->devno,
                 pSib->macAddr.octets[0], pSib->macAddr.octets[1],
                 pSib->macAddr.octets[2], pSib->macAddr.octets[3],
                 pSib->macAddr.octets[4], pSib->macAddr.octets[5],
                 sibStateStrGet(pSib));
        if (pSib->pPriv) {
            switch (pSib->pPriv->keyType) {
            case PRIV_KEY_TYPE_AES_CCM: pStr = "AES"; break;
            case PRIV_KEY_TYPE_WEP:     pStr = "WEP"; break;
            case PRIV_KEY_TYPE_CKIP:    pStr = "CKIP"; break;
            case PRIV_KEY_TYPE_TKIP:    pStr = "TKIP"; break;
            case PRIV_KEY_TYPE_TKIP_SW: pStr = "TKIP(sw)"; break;
            case PRIV_KEY_TYPE_NULL: pStr = "NULL"; break;
            default:                    pStr = "unknown"; break;
            }
            uiPrintf("\t%s", pStr);
        }
         uiPrintf("\n");
    }
    uiPrintf("\n");
}

void
sibStationShow(int unit, int staId)
{
    int           idx;
    int           cnt;
    SIB_ENTRY     sibCopy;
    SIB_ENTRY     *pSib;
    const char    *pRateStr;
    char          *pModeStr;
    WLAN_DEV_INFO *pDevInfo;
    SIB_TABLE     *pSibTable;
    SCB           *pSCB;

    pDevInfo = gDrvInfo.pDev[unit];
    if (pDevInfo) {
        pSibTable = pDevInfo->pSibTable;
        cnt = pSibTable->currentEntryNum;
        if (cnt) {
            for (idx = pSibTable->sibTableSize - 1 ; idx >= 0; idx--) {
                pSib = &pSibTable->sta[idx];
                A_SIB_TAB_LOCK(pSibTable);
                A_BCOPY((char *)pSib, (char *)&sibCopy, sizeof(SIB_ENTRY));
                A_SIB_TAB_UNLOCK(pSibTable);
                if (pSib->used & SIB_VALID && pSib->assocId & 0x3fff) {
                    if (pSib->serviceType == WLAN_STA_SERVICE &&
                        (staId == 0 || staId == (pSib->assocId & 0x3fff)))
                    {
                        uiPrintf("*************************************************\n");
                        printMacAddress(pSib->macAddr);
                        uiPrintf(", State: %s, AID: %d%s\n",
                                 sibStateStrGet(pSib),
                                 pSib->assocId & 0x3fff, isXrBssSib(pDevInfo, pSib) ? "(XR)" : "");
                        uiPrintf("Authentication Type: %s\n", authTypeGet(pSib->authenAlgo));
                        uiPrintf("WLAN Mode: %s", pSib->wlanMode == STA_MODE_A ? txt_11a :
                                 (pSib->wlanMode == STA_MODE_B ? txt_11b :
                                  (pSib->wlanMode == STA_MODE_G ? txt_11g : txt_108g)));
                        if (pSib->wlanMode == STA_MODE_G) {
                            uiPrintf(",  nonERP present: %s, nonERP protected: %s\n",
                                     pDevInfo->bssDescr->nonErpElement.info.present ? "YES" : "NO",
                                     pDevInfo->bssDescr->nonErpElement.info.protect ? "YES" : "NO");
                        } else {
                            uiPrintf("\n");
                        }

                        uiPrintf("Connection ID is %d\n", pSib->wdcConnId);

#if defined(DEBUG)
                        sibKeyShow(pSib, TRUE);
#else
                        sibKeyShow(pSib, FALSE);
#endif
                        uiPrintf("Compression: %s",
                                  pSib->athAdvCapElement.info.useCompression ? "ON" : "OFF");
                        uiPrintf("Power Save Mode: %s\n", pSib->psMode ? "ON" : "OFF");
                        if (pSib->stats.RcvDecryptCrcErrors)
                            uiPrintf("Encryption Decryption Error: %d - key may be misconfigured.\n",
                                     pSib->stats.RcvDecryptCrcErrors);
                        pRateStr = staGetRateStr(pDevInfo, pSib->stats.RcvRate, &pModeStr);
                        uiPrintf("Rx Data Rate: %s, RxSignalStrength: %d, AckSignalStrength: %d\n",
                                 pRateStr,
                                 pSib->stats.rssi,
                                 A_RSSI_OUT(pSib->stats.ackRssi));

                        uiPrintf("              MSDU       Data    Mcast     Mgmt     Ctrl    Errors\n");
                        uiPrintf("Rx\t%10u %10d %8u %8u %8u %9u\n",
                                 pSib->stats.GoodReceives,
                                 pSib->stats.RxDataFrames,
                                 pSib->stats.MulticastReceives,
                                 pSib->stats.RxMgmtFrames,
                                 pSib->stats.RxCtrlFrames,
                                 pSib->stats.ReceiveErrors);
                        uiPrintf("Tx\t%10u %10u %8u %8u %8u %9u\n",
                                 pSib->stats.GoodTransmits,
                                 pSib->stats.TxDataFrames,
                                 pSib->stats.TxMulticastFrames + pSib->stats.TxBroadcastFrames,
                                 pSib->stats.TxMgmtFrames,
                                 pSib->stats.TxCtrlFrames,
                                 pSib->stats.TransmitErrors);
                        uiPrintf("\n");
                        /*
                         * Print dot1x stats
                         */
                        pSCB = dot1xSessionGet(pDevInfo, pSib);
                        if (pSCB) {
                            displayDot1xStats(pSCB);
                        }
                        uiPrintf("\n");
                    }
                }
            }
        }
    }
}

void
sibKeyShow(SIB_ENTRY *pSib, int verbose)
{
    WLAN_PRIV_RECORD *pKey;
    char        *pStr;
    int         i;

    uiPrintf("Encryption: %s",
             pSib->privacyInvoked ? "ON" : "OFF");
    if (pSib->serviceType == WLAN_STA_SERVICE) {
        if (pSib->privacyInvoked) {
            pKey = pSib->pPriv;
            if (pKey != NULL) {
                uiPrintf(" Keyflags = <");
                if (pKey->keyFlags & PRIV_DYNAMIC) {
                    uiPrintf("dynamic");
                }
                if (pKey->keyFlags & PRIV_UNIQUEKEY) {
                    uiPrintf(",unique");
                }
                if (pKey->keyFlags & PRIV_CKIP_KP) {
                    uiPrintf(",ckip_kp");
                }
                if (pKey->keyFlags & PRIV_CKIP_MIC) {
                    uiPrintf(",ckip_mic");
                }
                if (pKey->keyFlags & PRIV_TKIP_SWMIC) {
                    uiPrintf(",tkip_swmic");
                }
                if (pKey->keyFlags & PRIV_KEY_LOCKED) {
                    uiPrintf(",locked");
                }
                uiPrintf("> framekey=%d ", pKey->keyFlags & PRIV_FRAMEKEY);

                uiPrintf("pPriv =0x%x", (A_UINT32)pKey);
                switch (pKey->keyType) {
                    case PRIV_KEY_TYPE_AES_CCM: pStr = "AES"; break;
                    case PRIV_KEY_TYPE_WEP:     pStr = "WEP"; break;
                    case PRIV_KEY_TYPE_CKIP:    pStr = "CKIP"; break;
                    case PRIV_KEY_TYPE_TKIP:
                    case PRIV_KEY_TYPE_TKIP_SW: pStr = "TKIP"; break;
                    default:                    pStr = "unknown"; break;
                }
                uiPrintf(" %s", pStr);
                if (verbose) {
                    uiPrintf(" %dbit key: ", pKey->keyLength);
                    for (i = 0; i < (pKey->keyLength/8); i++) {
                        uiPrintf("%02x", pKey->keyVal[i]);
                    }
                }
                if (pSib->hwIndex == HWINDEX_INVALID) {
                    uiPrintf(" hwIndex none\n");
                } else {
                    uiPrintf(" hwIndex %d\n", pSib->hwIndex);
                }
                if (verbose) {
                    if (pKey->akeyLength) {
                        uiPrintf("%dbit AES key: ", pKey->keyLength);
                        for (i = 0; i < (pKey->keyLength/8); i++) {
                            uiPrintf("%02x", pKey->keyVal[i]);
                        }
                        uiPrintf("\n");
                    }
                    uiPrintf("RX MIC key: ");
                    for (i = 0; i < 8; i++) {
                        uiPrintf("%02x", pKey->micRxKeyVal[i]);
                    }
                    uiPrintf("\nTX MIC key: ");
                    for (i = 0; i < 8; i++) {
                        uiPrintf("%02x", pKey->micTxKeyVal[i]);
                    }
                    uiPrintf("\n");
                }
            }
        }
    }
    sibCipherShow(pSib);
    uiPrintf("\n");
#ifdef DEBUG
    uiPrintf("Key cache misses: %d\n", pSib->stats.RcvKeyCacheMisses);
#endif
}

LOCAL char *
sibCipherType(A_UINT32 cipher)
{
    char *pStr = "UNKNOWN";
    if (IS_CIPHER_NULL(cipher)) {
        pStr = "NULL";
    }
    if (IS_CIPHER_WEP(cipher)) {
        pStr = "WEP";
    }
    if (IS_CIPHER_AES_CCM(cipher)) {
        pStr = "AES";
    }
    if (IS_CIPHER_TKIP(cipher)) {
        pStr = "TKIP";
    }
    return pStr;
}

void
sibCipherShow(SIB_ENTRY *pSib)
{
    char *pStr;

    uiPrintf(" Ciphers:");
    if (pSib->serviceType == WLAN_STA_SERVICE) {
        pStr = "NONE";
        if (VALID_CIPHER_OUI(pSib->uCipher)) {
            pStr = sibCipherType(pSib->uCipher);
        }
        uiPrintf(" %s(unicast),", pStr);
        pStr = "NONE";
        if (VALID_CIPHER_OUI(pSib->mCipher)) {
            pStr = sibCipherType(pSib->mCipher);
        }
        uiPrintf(" %s(multicast), ", pStr);
        pStr = "NONE";
        if (VALID_CIPHER_OUI(pSib->uCipher)) {
            uiPrintf("supports");
            pStr = sibCipherType(pSib->uCipher);
        }
        uiPrintf(" %s", pStr);
    } else {
        pStr = "NONE";
        if (VALID_CIPHER_OUI(pSib->uCipher)) {
            pStr = sibCipherType(pSib->uCipher);
        }
        uiPrintf(" %s", pStr);
    }
}

#if defined(DEBUG)

WLAN_MACADDR  *pStaMacAddr;

SIB_ENTRY *
addsta(int cnt, int unit, int serialize, int verbose)
{
    WLAN_DEV_INFO *pDevInfo;
    WLAN_MACADDR  macAddr;
    SIB_ENTRY     *pSib = NULL;
    int           ii    = 0;
    int           seed  = 1;

    pDevInfo = gDrvInfo.pDev[unit];
    srand(halGetRandomSeed(pDevInfo));

    if (cnt == 0) {
        cnt = 1;
    }

    while (cnt--) {
        macAddr.words[0] = serialize ? 0x0011 : rand() & 0xfeff;
        macAddr.words[1] = serialize ? 0x2233 : rand();
        macAddr.words[2] = serialize ? seed++ : rand();

        pSib = sibEntryAlloc(pDevInfo, &macAddr, &pDevInfo->baseBss);
        if (!pSib) {
            printf("Can't allocate SIB\n");
            return pSib;
        }
        pStaMacAddr     = &pSib->macAddr;
        pSib->assocId   = mlmeAssocIdNew(pDevInfo, &pDevInfo->baseBss) & 0x3fff;
        pSib->staState |= (STATE_AUTH | STATE_ASSOC);
        if (sibEntryAdd(pSib) == OK) {
            if (verbose)
            printf("sib[%d]: %p, %d added\n", ++ii, (void *)pSib, pSib->assocId & 0x3fff);
        } else {
            printf("Can't add SIB[%d]: %d added\n", ++ii, pSib->assocId & 0x3fff);
        }
    }
    return pSib;
}

#ifdef DEBUG

void
lockKeyCache()
{
    WLAN_DEV_INFO *pdevInfo;
    SIB_ENTRY *pSib;
    A_UINT16 i;
    int unit = 0;

    pdevInfo = gDrvInfo.pDev[unit];
    pSib = addsta(1, unit, 1, 1);
    pSib->numTxPending = 5;

    printf("Lockthem: unit %d keyCachesize %d\n", unit,
           pdevInfo->keyCacheSize);

    for (i = 6; i < (pdevInfo->keyCacheSize >> 2); i++)  {
        pdevInfo->keyCacheSib[i] = pSib;
        pdevInfo->keyCacheSib[i+32] = pSib;
        pdevInfo->keyCacheSib[i+64] = pSib;
        pdevInfo->keyCacheSib[i+32+64] = pSib;
    }
    pdevInfo->keyCacheSib[0] = pSib;
    pdevInfo->keyCacheSib[0+32] = pSib;
    pdevInfo->keyCacheSib[0+64] = pSib;
    pdevInfo->keyCacheSib[0+32+64] = pSib;

    pdevInfo->keyCacheSib[3] = pSib;
    pdevInfo->keyCacheSib[3+32] = pSib;
    pdevInfo->keyCacheSib[3+64] = pSib;
    pdevInfo->keyCacheSib[3+32+64] = pSib;
}
#endif /* DEBUG */

void
sibShow(int staId)
{
    WLAN_DEV_INFO *pDevInfo;
    int           i, j, idx;
    int           cnt;
    SIB_ENTRY     sibCopy;
    SIB_ENTRY     *pSib;
    const char    *pRateStr;
    char          *pModeStr;
    SIB_TABLE     *pSibTable;

    for (j = 0; j < WLAN_MAX_DEV; j++) {
        pDevInfo = gDrvInfo.pDev[j];
        if (pDevInfo) {
            pSibTable = pDevInfo->pSibTable;
            cnt = pSibTable->currentEntryNum;
            if (cnt) {
                uiPrintf("wlan%d\n", pDevInfo->devno);
                for (idx = pSibTable->sibTableSize - 1; idx >= 0; idx--) {
                    pSib = &pSibTable->sta[idx];
                    A_SIB_TAB_LOCK(pSibTable);
                    A_BCOPY((char *)pSib, (char *)&sibCopy, sizeof(SIB_ENTRY));
                    A_SIB_TAB_UNLOCK(pSibTable);
                    pSib = (SIB_ENTRY *)&sibCopy;
                    if ((pSib->used & SIB_VALID) == 0) {
                        continue;
                    }
                    if ((staId != -1) && (staId != 0) &&
                        (staId != (pSib->assocId & 0x3fff)))
                    {
                        continue;
                    }

                    pRateStr = staGetRateStr(pDevInfo, pSib->stats.RcvRate, &pModeStr);

                    if (pSib->serviceType == WLAN_STA_SERVICE) {
                        uiPrintf("AID %d%s, ", pSib->assocId & 0x3fff, isXrBssSib(pDevInfo, pSib) ? "(XR)" : "");
                    }

                    printMacAddress(pSib->macAddr);
                    uiPrintf("%4s, idx: %d, pOpBss: %p\n",
                             (pSib->serviceType == WLAN_STA_SERVICE) ? "STA" : "AP",
                             idx, (void *)pSib->pOpBss);
                    displayStates(pSib->staState, pSib->transState);
                    uiPrintf("Current Ant: %d, Rx Ant: %d\n",
                             pSib->antTx + 1, pSib->stats.RxAntenna);
                    uiPrintf("Ant 1: %d, Ant 2: %d, Switch: %d, Delta: %d\n",
                             pSib->stats.AntCnt[0], pSib->stats.AntCnt[1],
                             pSib->stats.AntSwCnt,
                             pSib->stats.AntCnt[0] - pSib->stats.AntCnt[1]);
                    if (pSib->serviceType != WLAN_STA_SERVICE) {
                        uiPrintf("Channel %d\n", pSib->Channel);
                        uiPrintf("Mode: %s\n", pModeStr);
                        uiPrintf("Encryption: %s\n", pDevInfo->staConfig.privacyInvoked ? "ON" : "OFF");
                        sibCipherShow(pSib);
                        uiPrintf("\n");
                    } else {
                        int rate = staTxRateGet(pDevInfo, pSib);
                        uiPrintf("Start: %d, Last active: %d\n",
                                 pSib->staLifeStartTime/1000,
                                 pSib->staLastActivityTime/1000);
                        uiPrintf("sfRate: %d, lfRate: %d\n", rate, rate);
                        uiPrintf("Authentication Type:\t %s\n", authTypeGet(pSib->authenAlgo));
                        sibKeyShow(pSib, TRUE);

                        uiPrintf("WLAN Mode: %s", pSib->wlanMode == STA_MODE_A ? txt_11a :
                                 (pSib->wlanMode == STA_MODE_B ? txt_11b : txt_11g));
                        if (pSib->wlanMode == STA_MODE_G) {
                            uiPrintf(",  nonERP present: %s, nonERP protected: %s\n",
                                     pDevInfo->bssDescr->nonErpElement.info.present ? "YES" : "NO",
                                     pDevInfo->bssDescr->nonErpElement.info.protect ? "YES" : "NO");
                        } else {
                            uiPrintf("\n");
                        }

                        uiPrintf("Connection ID is %d\n", pSib->wdcConnId);

                        uiPrintf("Compression: %s",
                                  pSib->athAdvCapElement.info.useCompression ? "ON" : "OFF");
                    }

                    if (pSib->stats.Authentications || pSib->stats.DeAuthentications) {
                        uiPrintf("Authentications: \t%5d  Deauthentications: %5d",
                                 pSib->stats.Authentications,
                                 pSib->stats.DeAuthentications);
                        if (pSib->stats.DeAuthReasonCode) {
                            uiPrintf("  Reason Code: %d\n",
                                     pSib->stats.DeAuthReasonCode);
                        } else {
                            uiPrintf("\n");
                        }
                    }
                    if (pSib->stats.Associations || pSib->stats.DisAssociations ||
                        pSib->stats.Reassociations)
                    {
                        uiPrintf("Associations:    \t%5d  Reassociations:    %5d  Disassociations:   %5d",
                                 pSib->stats.Associations,
                                 pSib->stats.Reassociations,
                                 pSib->stats.DisAssociations);
                        if (pSib->stats.DisAssocReasonCode) {
                            uiPrintf("  Reason Code: %d\n",
                                     pSib->stats.DisAssocReasonCode);
                        } else {
                            uiPrintf("\n");
                        }
                    }
                    uiPrintf("Power Save Mode: %s\n", pSib->psMode ? "ON" : "OFF");
                    if (pSib->stats.rssi) {
                        uiPrintf("RcvRate:\t\t %10s\nAckSignalStrength:\t %10d\t RcvSignalStrength:\t %10d\n",
                                 pRateStr,
                                 A_RSSI_OUT(pSib->stats.ackRssi),
                                 pSib->stats.rssi);
                    }
                    uiPrintf("Tx MSDUs:\t\t %10d\t Rx MSDUs:\t\t %10d\n",
                             pSib->stats.GoodTransmits,
                             pSib->stats.GoodReceives);
                    if (pSib->stats.TxMulticastFrames  + pSib->stats.TxBroadcastFrames || pSib->stats.MulticastReceives)
                        uiPrintf("TxMulticasts:\t\t %10d\t MulticastReceives:\t %10d\n",
                                 pSib->stats.TxMulticastFrames + pSib->stats.TxBroadcastFrames,
                                 pSib->stats.MulticastReceives);
                    if (pSib->stats.TxCtrlFrames ||  pSib->stats.RxCtrlFrames)
                        uiPrintf("TxCtrlFrames:\t\t %10d\t RxCtrlFrames:\t\t %10d\n",
                                 pSib->stats.TxCtrlFrames, pSib->stats.RxCtrlFrames);
                    if (pSib->stats.TxDataFrames ||  pSib->stats.RxDataFrames)
                        uiPrintf("TxDataFrames:\t\t %10d\t RxDataFrames:\t\t %10d\n",
                                 pSib->stats.TxDataFrames, pSib->stats.RxDataFrames);
                    if (pSib->stats.TxMgmtFrames || pSib->stats.RxMgmtFrames)
                        uiPrintf("TxMgmtFrames:\t\t %10d\t RxMgmtFrames:\t\t %10d\n",
                                 pSib->stats.TxMgmtFrames, pSib->stats.RxMgmtFrames);
                    if (pSib->stats.TransmitErrors || pSib->stats.ReceiveErrors)
                        uiPrintf("TxErrors:\t\t %10d\t RxErrors:\t\t %10d\n",
                                 pSib->stats.TransmitErrors,
                                 pSib->stats.ReceiveErrors);
                    if (pSib->stats.RcvPhyErrors)
                        uiPrintf("RxPhyErrors:\t\t %10d\n",
                                 pSib->stats.RcvPhyErrors);
                    if (pSib->stats.TxExcessiveRetries || pSib->stats.RcvCrcErrors)
                        uiPrintf("TxExcessiveRetries:\t %10d\t RxCrcErrors:\t\t %10d\n",
                                 pSib->stats.TxExcessiveRetries,
                                 pSib->stats.RcvCrcErrors);
                    if (pSib->stats.RcvDecryptCrcErrors || pSib->stats.RcvKeyCacheMisses)
                        uiPrintf("\t\t\t\t\t RcvDecryptCrcErrors:\t %10d\n"
                                 "\t\t\t\t\t RxKeyCacheMisses:\t %10d\n",
                                 pSib->stats.RcvDecryptCrcErrors,
                                 pSib->stats.RcvKeyCacheMisses);
                    if (pSib->numTxPending || pSib->numFilteredPending)
                        uiPrintf("Pending transmit frames: %10d\t Pending Filtered frames: %9d\n",
                                 pSib->numTxPending, pSib->numFilteredPending);
                    if (pSib->stats.swRetryTotalCnt || pSib->stats.swRetryMaxRetriesExceeded)
                        uiPrintf("SW Retries:\t\t %10d\t Exceeded SW Retry limits: %8d\n",
                                 pSib->stats.swRetryTotalCnt,
                                 pSib->stats.swRetryMaxRetriesExceeded);
                    if (pSib->stats.TxFiltered || pSib->stats.MultipleRxDuplicates)
                        uiPrintf("TxFiltered:\t\t %10d\t RxDuplicateFrames:\t %10d\n",
                                 pSib->stats.TxFiltered,
                                 pSib->stats.MultipleRxDuplicates);
                    if (pSib->stats.TxFramesDropped || pSib->stats.RxDiscardFrames)
                        uiPrintf("TxFramesDropped:\t %10d\t RxDiscardFrames:\t %10d\n",
                                 pSib->stats.TxFramesDropped,
                                 pSib->stats.RxDiscardFrames);
                    if (pSib->stats.RtsFailCnt || pSib->stats.FcsFailCnt)
                        uiPrintf("RtsFailCnt:\t\t %10d\t FcsFailCnt:\t\t %10d\n",
                                 pSib->stats.RtsFailCnt,
                                 pSib->stats.FcsFailCnt);
                    if (pSib->stats.TotalRetries || pSib->stats.AckRcvFailures)
                        uiPrintf("TxTotalRetries:\t\t %10d\t AckRcvFailures:\t %10d\n",
                                 pSib->stats.TotalRetries,
                                 pSib->stats.AckRcvFailures);
                    if (pSib->stats.RcvWEPExcludedCount)
                        uiPrintf("RcvWEPExcludedCount:\t %10d\n", pSib->stats.RcvWEPExcludedCount);

                    for (i = 1; i < 16; i++) {
                        if (pSib->stats.RetryBins[i] || pSib->stats.shortFrameRetryBins[i]) {
                            if (i == 1)
                                uiPrintf("%2d   TxShortFrameRetry:\t %10d\t %2d   TxLongFrameRetry:\t %10d\n",
                                         i, pSib->stats.shortFrameRetryBins[i],
                                         i, pSib->stats.RetryBins[i]);
                            else
                                uiPrintf("%2d TxShortFrameRetries:\t %10d\t %2d TxLongFrameRetries:\t %10d\n",
                                         i, pSib->stats.shortFrameRetryBins[i],
                                         i, pSib->stats.RetryBins[i]);
                        }
                    }
                    /* power save related statistics */
                    if (pSib->serviceType == WLAN_STA_SERVICE) {
                        if (pSib->stats.psRequestFrames || pSib->stats.psSavedFrames)
                            uiPrintf("psRequestFrames:\t %10d\t psSavedFrames:\t\t %10d\n",
                                     pSib->stats.psRequestFrames,
                                     pSib->stats.psSavedFrames);
                        if (pSib->stats.psAgedFrames)
                            uiPrintf("psAgedFrames:\t\t %10d\n", pSib->stats.psAgedFrames);
                        if (pSib->stats.compSuccessCnt || pSib->stats.RcvDecompCrcErrors)
                            uiPrintf("compSuccessFrames:\t %10d\t RxDecompCrcErrors:\t %10d\n",
                                     pSib->stats.compSuccessCnt, pSib->stats.RcvDecompCrcErrors);
                    } else {
                        if (pSib->stats.psMulticastFrames)
                            uiPrintf("psMulticastFrames:\t %10d\n", pSib->stats.psMulticastFrames);
                        if (pSib->stats.compSuccessCnt || pSib->stats.RcvDecompCrcErrors)
                            uiPrintf("compSuccessFrames:\t %10d\t RxDecompCrcErrors:\t %10d\n",
                                     pSib->stats.compSuccessCnt, pSib->stats.RcvDecompCrcErrors);
                        if (pSib->stats.CompCPC0Cnt)
                            uiPrintf("  compCPC0Counter:\t %10d\n", pSib->stats.CompCPC0Cnt);
                        if (pSib->stats.CompCPC1Cnt)
                            uiPrintf("  compCPC1Counter:\t %10d\n", pSib->stats.CompCPC1Cnt);
                        if (pSib->stats.CompCPC2Cnt)
                            uiPrintf("  compCPC2Counter:\t %10d\n", pSib->stats.CompCPC2Cnt);
                        if (pSib->stats.CompCPC3Cnt)
                            uiPrintf("  compCPC3Counter:\t %10d\n", pSib->stats.CompCPC3Cnt);
                    }
                    uiPrintf("\n");
                    cnt--;
                    if (cnt == 0)
                        break;
                }
            }
        }
    }
}

void
bssShow(int unit)
{
    WLAN_DEV_INFO *pDev = gDrvInfo.pDev[unit];
    int           i, len;

    uiPrintf("SSID: %s\n",
             pDev->bssDescr->ssid.ssid);
    uiPrintf("BSS type: %d\n",
             pDev->bssDescr->bsstype);
    uiPrintf("Channel: %d\n",
             pDev->bssDescr->pChannel ?
             pDev->bssDescr->pChannel->channel : 0);
    uiPrintf("Beacon Interval: %d\n",
             pDev->bssDescr->beaconInterval);
    uiPrintf("DTIM Interval: %d\n",
             pDev->bssDescr->DTIMPeriod);
    len = pDev->baseBss.rateSet.length;
    for (i = 0; i < len; i++) {
        uiPrintf("rate[%d]: %d\n", i, pDev->baseBss.rateSet.rates[i]);
    }
}

void
sibAgingShow(void)
{
    SIB_ENTRY *pSib;
    int           i, idx;
    int           cnt;
    SIB_ENTRY     sibCopy;
    QUEUE_DETAILS *pQueue;
    A_UINT32      lasttime;
    SIB_TABLE     *pSibTable;
    WLAN_DEV_INFO *pDevInfo;

    for (i = 0; i < WLAN_MAX_DEV; i++) {
        pDevInfo = gDrvInfo.pDev[i];
        if (pDevInfo) {
            pSibTable = gDrvInfo.pDev[i]->pSibTable;
            cnt       = pSibTable->currentEntryNum;
            if (cnt) {
                for (idx = pSibTable->sibTableSize - 1; idx >= 0; idx--) {
                    pSib = &pSibTable->sta[idx];
                    A_SIB_TAB_LOCK(pSibTable);
                    A_BCOPY((char *)pSib, (char *)&sibCopy, sizeof(SIB_ENTRY));
                    A_SIB_TAB_UNLOCK(pSibTable);
                    pSib = (SIB_ENTRY *)&sibCopy;
                    if (!(pSib->used & SIB_VALID))
                        continue;

                    printMacAddress(pSib->macAddr);
                    printf (" AID: %d%s\n", pSib->assocId & 0x3fff, isXrBssSib(pDevInfo, pSib) ? "(XR)" : "");
                    lasttime = pSib->staLastActivityTime;
                    pQueue   = &pSib->txPSQueue;
                    printf ("psQHead: %p, psQFrameCount: %d\nnumTxPending: %d, numFilteredPending: %d, swRetryQFrames: %d\n",
                            (void *)pQueue->pDescQueueHead,
                            pQueue->qFrameCount,
                            pSib->numTxPending,
                            pSib->numFilteredPending,
                            pSib->swRetryQueue.qFrameCount);
                    printf("time: %d\n", A_MS_TICKGET() - lasttime);
                    cnt--;
                    if (cnt == 0)
                        break;
                }
            }
        }
    }
}


void
sizeShow()
{
    uiPrintf("sizeof(ATHEROS_DESC): %d\n", sizeof(ATHEROS_DESC));
    uiPrintf("sizeof(SIB_ENTRY): %d\n", sizeof(SIB_ENTRY));
    uiPrintf("sizeof(SIB_TABLE): %d\n", sizeof(SIB_TABLE));
    uiPrintf("sizeof(WLAN_DEV_INFO): %d\n", sizeof(WLAN_DEV_INFO));
}

#endif /* DEBUG */
#endif /* BUILD_AP */


WLAN_DEV_INFO *
sib2DevInfo(SIB_ENTRY *pSib)
{
    int           i;
    SIB_TABLE     *pSibTable;
    WLAN_DEV_INFO *pDev;

    for (i = 0; i < WLAN_MAX_DEV; i++) {
        pDev = gDrvInfo.pDev[i];
        if (pDev) {
            pSibTable = pDev->pSibTable;
            if (pSib >= &pSibTable->sta[0] &&
                pSib <= &pSibTable->sta[pSibTable->sibTableSize - 1])
            {
                return pDev;
            }
        }
    }

    ASSERT(0);
    return NULL;
}

