/*
 * $Id: //depot/sw/branches/1.3_USB_LINUX_port/src/USB/wlan/host/common/wlanPhy.c#2 $
 *
 * Copyright (c) 2000-2003 Atheros Communications, Inc., All Rights Reserved
 *
 */

#include "wlandrv.h"
#include "wlanext.h"
#include "wlansmeext.h"
#include "wlanPhy.h"
#include "wlanchannel.h"
#include "wlanReceive.h"
#include "halApiLinux.h"
#include "ui.h"

static A_UINT16
ofdmComputeTxTime(const RATE_TABLE *pRateTable, A_UINT32 frameLen, A_UINT16 rateIndex, A_BOOL flag)
{
    A_UINT32 kbps    = pRateTable->info[rateIndex].rateKbps;
    A_UINT32 bitsPerSymbol, numBits, numSymbols, txTime;

#define OFDM_SIFS_TIME        16
#define OFDM_PREAMBLE_TIME    20
#define OFDM_PLCP_BITS        22
#define OFDM_SYMBOL_TIME       4

    bitsPerSymbol = (kbps * OFDM_SYMBOL_TIME) / 1000;

    numBits       = OFDM_PLCP_BITS + (frameLen << 3);
    numSymbols    = (numBits + bitsPerSymbol - 1) / bitsPerSymbol;
    txTime        = OFDM_SIFS_TIME + OFDM_PREAMBLE_TIME + (numSymbols * OFDM_SYMBOL_TIME);

    return (A_UINT16)txTime;
}


static A_UINT16
turboComputeTxTime(const RATE_TABLE *pRateTable, A_UINT32 frameLen, A_UINT16 rateIndex, A_BOOL flag)
{
    A_UINT32 kbps    = pRateTable->info[rateIndex].rateKbps;
    A_UINT32 bitsPerSymbol, numBits, numSymbols, txTime;

#define TURBO_SIFS_TIME         8
#define TURBO_PREAMBLE_TIME    14
#define TURBO_PLCP_BITS        22
#define TURBO_SYMBOL_TIME       4

    /* we still save OFDM rates in kbps - so double them */
    bitsPerSymbol = ((kbps << 1) * TURBO_SYMBOL_TIME) / 1000;

    numBits       = TURBO_PLCP_BITS + (frameLen << 3);
    numSymbols    = (numBits + bitsPerSymbol - 1) / bitsPerSymbol;
    txTime        = TURBO_SIFS_TIME + TURBO_PREAMBLE_TIME + (numSymbols * TURBO_SYMBOL_TIME);

    return (A_UINT16)txTime;
}


static A_UINT16
cckComputeTxTime(const RATE_TABLE *pRateTable, A_UINT32 frameLen, A_UINT16 rateIndex, A_BOOL shortPreamble)
{
    A_UINT32 kbps    = pRateTable->info[rateIndex].rateKbps;
    A_UINT32 numBits = frameLen << 3;
    A_UINT32 phyTime, txTime;

#define CCK_SIFS_TIME        10
#define CCK_PREAMBLE_BITS   144
#define CCK_PLCP_BITS        48

    phyTime = CCK_PREAMBLE_BITS + CCK_PLCP_BITS;
    if (shortPreamble && pRateTable->info[rateIndex].shortPreamble) {
        phyTime >>= 1;
    }

    txTime = CCK_SIFS_TIME + phyTime + ((numBits * 1000)/kbps);

    return (A_UINT16)txTime;
}


static A_UINT16
xrComputeTxTime(const RATE_TABLE *pRateTable, A_UINT32 frameLen, A_UINT16 rateIndex, A_BOOL flag)
{
    A_UINT32 kbps    = pRateTable->info[rateIndex].rateKbps;
    A_UINT32 bitsPerSymbol, numBits, numSymbols, txTime;

#define XR_SIFS_TIME            16
#define XR_PREAMBLE_TIME(_kpbs) (((_kpbs) < 1000) ? 173 : 76)
#define XR_PLCP_BITS            22
#define XR_SYMBOL_TIME           4

    bitsPerSymbol = (kbps * XR_SYMBOL_TIME) / 1000;

    numBits       = XR_PLCP_BITS + (frameLen << 3);
    numSymbols    = (numBits + bitsPerSymbol - 1) / bitsPerSymbol;
    txTime        = XR_SIFS_TIME + XR_PREAMBLE_TIME(kbps) + (numSymbols * XR_SYMBOL_TIME);

    return (A_UINT16)txTime;
}


PHY_TX_TIME_FUNC
phyComputeTxTime[WLAN_PHY_MAX] = {
    ofdmComputeTxTime,
    turboComputeTxTime,
    cckComputeTxTime,
    xrComputeTxTime
};


/**************************************************************************
 * Rate Table(s) and Modes:
 *
 **************************************************************************/


A_STATUS
wlanSetCckRateMode(WLAN_DEV_INFO *pDev, CCK_RATE_MODE mode)
{
#if NDIS_WDM
    /*     
     * (LW) Attention !!!
     * This is temp hack, should revisit later
     */
    RATE_TABLE *pRateTable = pDev->hwRateTable[WIRELESS_MODE_11b];
#else
    RATE_TABLE *pRateTable = HACK_TO_PARDEV(pDev)->hwRateTable[WIRELESS_MODE_11b];
#endif
    int i;

    switch (mode) {
    case CCK_RATE_MODE_LONG_RANGE:
        /* User selected only 1 and 2 Mb, long range setting */
        pRateTable->rateCount = 2;
        break;

    case CCK_RATE_MODE_BASIC_RATES:
        /* 5.5 and 11 Mb/s are not basic rates */
        for (i = 2; i < pRateTable->rateCount; ++i) {
            pRateTable->info[i].dot11Rate &= 0x7F;
        }
        break;

    default:
        ASSERT(0);
    }

    return A_OK;
}

A_STATUS
wlanSet11gBasicRates(WLAN_DEV_INFO *pDev, A_UINT16 mode)
{
    int i, j;
#if NDIS_WDM
    /*     
     * (LW) Attention !!!
     * This is temp hack, should revisit later
     */
    RATE_TABLE *pRateTable = pDev->hwRateTable[WIRELESS_MODE_11g];
#else
    RATE_TABLE *pRateTable = HACK_TO_PARDEV(pDev)->hwRateTable[WIRELESS_MODE_11g];
#endif

    static const BASIC_RATE_TABLE basicRates11g[] = {
        { 2, {2, 4}},                     /*  BASIC11G_DOT11  1,2 */
        { 4, {2, 4, 11, 22}},             /*  BASIC11G_DOT11B 1,2,5.5,11 */
        { 7, {2, 4, 11, 22, 12, 24, 48}}, /*  BASIC11G_DOT11G 1,2,5.5,11,6,12,24 */
        { 3, {12, 24, 48}},               /*  BASIC11G_OFDM   6,12,24 */
    };

    ASSERT(mode <= BASIC11G_MAX);

    for (i = 0; i < pRateTable->rateCount; i++) {
        /* Assume not a basic rate */
        pRateTable->info[i].dot11Rate &= 0x7f;   

        for (j = 0; j < basicRates11g[mode].rateCount; j++) {
            if (pRateTable->info[i].dot11Rate == basicRates11g[mode].basicRates[j]) {
                /* Basic rate */
                pRateTable->info[i].dot11Rate |= 0x80;
            }
        }
    }
    return A_OK;
}

WIRELESS_MODE wlanFindModeFromRateTable(WLAN_DEV_INFO *pDev, OP_BSS *pOpBss)
{
    WIRELESS_MODE mode;

    /* hacky */
    for (mode = WIRELESS_MODE_11a; mode < WIRELESS_MODE_MAX; ++mode) {
#if NDIS_WDM
        /*     
        * (LW) Attention !!!
        * This is temp hack, should revisit later
        */
        if (pOpBss->pRateTable == pDev->hwRateTable[mode]) {
#else
        if (pOpBss->pRateTable == HACK_TO_PARDEV(pDev)->hwRateTable[mode]) {
#endif
            break;
        }
    }
    ASSERT(mode < WIRELESS_MODE_MAX);

    return mode;
}

#ifdef WME
/*
 * Initialize the STA with the default WME parameters.
 */
void
wlanInitStaWmeParams(WLAN_DEV_INFO *pDev)
{
    typedef struct { 
        A_UINT8 aifs; 
        A_UINT8 logCwMin;
        A_UINT8 logCwMax; 
        A_UINT16 txOpLimit;
        A_UINT8 acm;
    } PHY_PARAM_TYPE;
    PHY_PARAM_TYPE *pPhyParam;

    static PHY_PARAM_TYPE phyParamForACI_BE[WIRELESS_MODE_MAX] = {
        /* WIRELESS_MODE_11a   */ {          3,                4,               6,                  0,              0 },
        /* WIRELESS_MODE_TURBO */ {          3,                4,               5,                  0,              0 },
        /* WIRELESS_MODE_11b   */ {          3,                5,               7,                  0,              0 },
        /* WIRELESS_MODE_11g   */ {          3,                4,               6,                  0,              0 },
        /* WIRELESS_MODE_108g  */ {          3,                4,               5,                  0,              0 },
        /* WIRELESS_MODE_XR    */ {    XR_AIFS,     XR_LOG_CWMIN,     XR_LOG_CWMAX,                 0,              0 }};
    static PHY_PARAM_TYPE phyParamForACI_BK[WIRELESS_MODE_MAX] = {
        /* WIRELESS_MODE_11a   */ {          7,                4,               10,                 0,              0 },
        /* WIRELESS_MODE_TURBO */ {          7,                3,               10,                 0,              0 },
        /* WIRELESS_MODE_11b   */ {          7,                5,               10,                 0,              0 },
        /* WIRELESS_MODE_11g   */ {          7,                4,               10,                 0,              0 },
        /* WIRELESS_MODE_108g  */ {          7,                3,               10,                 0,              0 },
        /* WIRELESS_MODE_XR    */ {    XR_AIFS,     XR_LOG_CWMIN,     XR_LOG_CWMAX,                 0,              0 }};
    static PHY_PARAM_TYPE phyParamForACI_Vi[WIRELESS_MODE_MAX] = {
        /* WIRELESS_MODE_11a   */ {          1,                3,               4,                  3,              0 },
        /* WIRELESS_MODE_TURBO */ {          1,                2,               3,                  3,              0 },
        /* WIRELESS_MODE_11b   */ {          1,                4,               5,                  6,              0 },
        /* WIRELESS_MODE_11g   */ {          1,                3,               4,                  3,              0 },
        /* WIRELESS_MODE_108g  */ {          1,                2,               3,                  3,              0 },
        /* WIRELESS_MODE_XR    */ {    XR_AIFS,     XR_LOG_CWMIN,     XR_LOG_CWMAX,                 0,              0 }};
    static PHY_PARAM_TYPE phyParamForACI_Vo[WIRELESS_MODE_MAX] = {
        /* WIRELESS_MODE_11a   */ {          1,                2,               3,                  1,              0 },
        /* WIRELESS_MODE_TURBO */ {          1,                1,               2,                  1,              0 },
        /* WIRELESS_MODE_11b   */ {          1,                3,               4,                  3,              0 },
        /* WIRELESS_MODE_11g   */ {          1,                2,               3,                  1,              0 },
        /* WIRELESS_MODE_108g  */ {          1,                1,               2,                  1,              0 },
        /* WIRELESS_MODE_XR    */ {    XR_AIFS,     XR_LOG_CWMIN,     XR_LOG_CWMAX,                 0,              0 }};
 
    int qnum;
    WIRELESS_MODE mode;

    for (mode = WIRELESS_MODE_11a; mode < WIRELESS_MODE_MAX; mode++) {
        for (qnum = 0; qnum < NWME; qnum++) {
            switch (QNUM_TO_AC(qnum)) {
            case ACI_BK:
                pPhyParam = &(phyParamForACI_BK[mode]);
                break;
            case ACI_Vi:
                pPhyParam = &phyParamForACI_Vi[mode];
                break;
            case ACI_Vo:
                pPhyParam = &phyParamForACI_Vo[mode];
                break;
            case ACI_BE:
            default:
                pPhyParam = &phyParamForACI_BE[mode];
                break;
            }
            pDev->staConfig.wmeParamValues[mode].wmeParamsPerQ[qnum].acm = pPhyParam->acm;
            pDev->staConfig.wmeParamValues[mode].wmeParamsPerQ[qnum].aifs = pPhyParam->aifs;
            pDev->staConfig.wmeParamValues[mode].wmeParamsPerQ[qnum].logCwMin = pPhyParam->logCwMin;
            pDev->staConfig.wmeParamValues[mode].wmeParamsPerQ[qnum].logCwMax = pPhyParam->logCwMax;
            pDev->staConfig.wmeParamValues[mode].wmeParamsPerQ[qnum].txOpLimit = pPhyParam->txOpLimit;
        }

    }
}

#endif

void
wlanUpdatePhyChAPs(WLAN_DEV_INFO *pDev, OP_BSS *pOpBss, WIRELESS_MODE mode)
{
    static const struct           { A_UINT8 aifs; A_UINT8 logCwMin; A_UINT8 logCwMax; }
    phyParam[WIRELESS_MODE_MAX] = {
          /* WIRELESS_MODE_11a   */ {          2,                4,               10  },
          /* WIRELESS_MODE_TURBO */ {          1,                3,               10  },
          /* WIRELESS_MODE_11b   */ {          2,                5,               10  },
          /* WIRELESS_MODE_11g   */ {          2,                4,               10  },
          /* WIRELESS_MODE_108g  */ {          1,                3,               10  },
          /* WIRELESS_MODE_XR    */ {    XR_AIFS,     XR_LOG_CWMIN,     XR_LOG_CWMAX  }};

#ifdef WME
    if (pDev->staConfig.WmeEnabled) {
        /* set up the channel access parameters for the physical device */
        pOpBss->phyChAPs.ac[ACI_BE].aci       = ACI_BE;
        pOpBss->phyChAPs.ac[ACI_BE].acm       = pDev->staConfig.wmeParamValues[mode].wmeParamsPerQ[AC_TO_QNUM(ACI_BE)].acm;
        pOpBss->phyChAPs.ac[ACI_BE].aifs      = pDev->staConfig.wmeParamValues[mode].wmeParamsPerQ[AC_TO_QNUM(ACI_BE)].aifs;
        pOpBss->phyChAPs.ac[ACI_BE].logCwMin  = pDev->staConfig.wmeParamValues[mode].wmeParamsPerQ[AC_TO_QNUM(ACI_BE)].logCwMin;
        pOpBss->phyChAPs.ac[ACI_BE].logCwMax  = pDev->staConfig.wmeParamValues[mode].wmeParamsPerQ[AC_TO_QNUM(ACI_BE)].logCwMax;
        pOpBss->phyChAPs.ac[ACI_BE].txOpLimit = pDev->staConfig.wmeParamValues[mode].wmeParamsPerQ[AC_TO_QNUM(ACI_BE)].txOpLimit;
        pOpBss->phyChAPs.ac[ACI_BK].aci       = ACI_BK;
        pOpBss->phyChAPs.ac[ACI_BK].acm       = pDev->staConfig.wmeParamValues[mode].wmeParamsPerQ[AC_TO_QNUM(ACI_BK)].acm;
        pOpBss->phyChAPs.ac[ACI_BK].aifs      = pDev->staConfig.wmeParamValues[mode].wmeParamsPerQ[AC_TO_QNUM(ACI_BK)].aifs;
        pOpBss->phyChAPs.ac[ACI_BK].logCwMin  = pDev->staConfig.wmeParamValues[mode].wmeParamsPerQ[AC_TO_QNUM(ACI_BK)].logCwMin;
        pOpBss->phyChAPs.ac[ACI_BK].logCwMax  = pDev->staConfig.wmeParamValues[mode].wmeParamsPerQ[AC_TO_QNUM(ACI_BK)].logCwMax;
        pOpBss->phyChAPs.ac[ACI_BK].txOpLimit = pDev->staConfig.wmeParamValues[mode].wmeParamsPerQ[AC_TO_QNUM(ACI_BK)].txOpLimit;
        pOpBss->phyChAPs.ac[ACI_Vi].aci       = ACI_Vi;
        pOpBss->phyChAPs.ac[ACI_Vi].acm       = pDev->staConfig.wmeParamValues[mode].wmeParamsPerQ[AC_TO_QNUM(ACI_Vi)].acm;
        pOpBss->phyChAPs.ac[ACI_Vi].aifs      = pDev->staConfig.wmeParamValues[mode].wmeParamsPerQ[AC_TO_QNUM(ACI_Vi)].aifs;
        pOpBss->phyChAPs.ac[ACI_Vi].logCwMin  = pDev->staConfig.wmeParamValues[mode].wmeParamsPerQ[AC_TO_QNUM(ACI_Vi)].logCwMin;
        pOpBss->phyChAPs.ac[ACI_Vi].logCwMax  = pDev->staConfig.wmeParamValues[mode].wmeParamsPerQ[AC_TO_QNUM(ACI_Vi)].logCwMax;
        pOpBss->phyChAPs.ac[ACI_Vi].txOpLimit = pDev->staConfig.wmeParamValues[mode].wmeParamsPerQ[AC_TO_QNUM(ACI_Vi)].txOpLimit;
        pOpBss->phyChAPs.ac[ACI_Vo].aci       = ACI_Vo;
        pOpBss->phyChAPs.ac[ACI_Vo].acm       = pDev->staConfig.wmeParamValues[mode].wmeParamsPerQ[AC_TO_QNUM(ACI_Vo)].acm;
        pOpBss->phyChAPs.ac[ACI_Vo].aifs      = pDev->staConfig.wmeParamValues[mode].wmeParamsPerQ[AC_TO_QNUM(ACI_Vo)].aifs;
        pOpBss->phyChAPs.ac[ACI_Vo].logCwMin  = pDev->staConfig.wmeParamValues[mode].wmeParamsPerQ[AC_TO_QNUM(ACI_Vo)].logCwMin;
        pOpBss->phyChAPs.ac[ACI_Vo].logCwMax  = pDev->staConfig.wmeParamValues[mode].wmeParamsPerQ[AC_TO_QNUM(ACI_Vo)].logCwMax;
        pOpBss->phyChAPs.ac[ACI_Vo].txOpLimit = pDev->staConfig.wmeParamValues[mode].wmeParamsPerQ[AC_TO_QNUM(ACI_Vo)].txOpLimit;
    } else {
        /* set up the channel access parameters for the physical device */
        pOpBss->phyChAPs.ac[ACI_BE].aci       = ACI_BE;
        pOpBss->phyChAPs.ac[ACI_BE].aifs      = phyParam[mode].aifs;
        pOpBss->phyChAPs.ac[ACI_BE].logCwMin  = phyParam[mode].logCwMin;
        pOpBss->phyChAPs.ac[ACI_BE].logCwMax  = phyParam[mode].logCwMax;

        /* assume TxOP = 0 for all ACs for now */
        pOpBss->phyChAPs.ac[ACI_BE].acm       = 0;
        pOpBss->phyChAPs.ac[ACI_BE].txOpLimit = 0;
        pOpBss->phyChAPs.info = 0;
        pOpBss->phyChAPs.ac[ACI_BK].aci       = ACI_BK;
        pOpBss->phyChAPs.ac[ACI_BK].acm       = 0;
        pOpBss->phyChAPs.ac[ACI_BK].aifs      = 6;
        pOpBss->phyChAPs.ac[ACI_BK].logCwMin  = 6;
        pOpBss->phyChAPs.ac[ACI_BK].logCwMax  = 10;
        pOpBss->phyChAPs.ac[ACI_BK].txOpLimit = 0;
        pOpBss->phyChAPs.ac[ACI_Vi].aci       = ACI_Vi;
        pOpBss->phyChAPs.ac[ACI_Vi].acm       = 0;
        pOpBss->phyChAPs.ac[ACI_Vi].aifs      = 3;
        pOpBss->phyChAPs.ac[ACI_Vi].logCwMin  = 4;
        pOpBss->phyChAPs.ac[ACI_Vi].logCwMax  = 6;
        pOpBss->phyChAPs.ac[ACI_Vi].txOpLimit = 0;
        pOpBss->phyChAPs.ac[ACI_Vo].aci       = ACI_Vo;
        pOpBss->phyChAPs.ac[ACI_Vo].acm       = 0;
        pOpBss->phyChAPs.ac[ACI_Vo].aifs      = 1;
        pOpBss->phyChAPs.ac[ACI_Vo].logCwMin  = 2;
        pOpBss->phyChAPs.ac[ACI_Vo].logCwMax  = 4;
        pOpBss->phyChAPs.ac[ACI_Vo].txOpLimit = 0;
    }

    if (pDev->staConfig.abolt & ABOLT_BURST) {
        pOpBss->phyChAPs.ac[ACI_BE].txOpLimit = US_TO_TXOP(TU_TO_US(pDev->staConfig.burstTime));
    }

#ifdef BUILD_AP
    if (pDev->staConfig.abolt & ABOLT_WME_ELE) {
        static const A_UINT8
        logCwMin[WIRELESS_MODE_MAX] = {
              /* WIRELESS_MODE_11a   */   3,
              /* WIRELESS_MODE_TURBO */   3,
              /* WIRELESS_MODE_11b   */   4,
              /* WIRELESS_MODE_11g   */   3,
              /* WIRELESS_MODE_108g  */   3,
              /* WIRELESS_MODE_XR    */   1 };

        if (pOpBss->numAssociatedClients < 2) {
            pOpBss->phyChAPs.ac[ACI_BE].logCwMin = logCwMin[mode];
            pOpBss->phyChAPs.info = 1;
        } else {
            pOpBss->phyChAPs.info = 0;
        }
    }
#endif
#else
    /* set up the channel access parameters for the physical device */
    pOpBss->phyChAPs.ac[ACI_BE].aci       = ACI_BE;
    pOpBss->phyChAPs.ac[ACI_BE].aifs      = phyParam[mode].aifs;
    pOpBss->phyChAPs.ac[ACI_BE].logCwMin  = phyParam[mode].logCwMin;
    pOpBss->phyChAPs.ac[ACI_BE].logCwMax  = phyParam[mode].logCwMax;

    if (pDev->staConfig.abolt & ABOLT_BURST) {
        pOpBss->phyChAPs.ac[ACI_BE].txOpLimit = US_TO_TXOP(TU_TO_US(pDev->staConfig.burstTime));
    }

#ifdef BUILD_AP
    if (pDev->staConfig.abolt & ABOLT_WME_ELE) {
        static const A_UINT8
        logCwMin[WIRELESS_MODE_MAX] = {
              /* WIRELESS_MODE_11a   */   3,
              /* WIRELESS_MODE_TURBO */   3,
              /* WIRELESS_MODE_11b   */   4,
              /* WIRELESS_MODE_11g   */   3,
              /* WIRELESS_MODE_108g  */   3,
              /* WIRELESS_MODE_XR    */   1 };

        if (pOpBss->numAssociatedClients < 2) {
            pOpBss->phyChAPs.ac[ACI_BE].logCwMin = logCwMin[mode];
            pOpBss->phyChAPs.info = 1;
        } else {
            pOpBss->phyChAPs.info = 0;
        }
    }
#endif
#endif
}

A_STATUS
wlanSetupPhyInfo(WLAN_DEV_INFO *pDev, OP_BSS* pOpBss, WIRELESS_MODE mode, A_UINT16 defaultRate)
{
#if NDIS_WDM
    /*     
     * (LW) Attention !!!
     * This is temp hack, should revisit later
     */
    RATE_TABLE    *pRateTable  = pDev->hwRateTable[mode];
#else
    RATE_TABLE    *pRateTable  = HACK_TO_PARDEV(pDev)->hwRateTable[mode];
#endif
    WLAN_RATE_SET *pRateSet    = &pOpBss->rateSet;
    A_UINT8       i;

#if 0
    /* WAR 8226 */
    if (pOpBss->pRateTable == pRateTable) {
        /* already initialized */
        return A_OK;
    }
#endif

    /* inherit the provided hw rate table */
    pOpBss->pRateTable = pRateTable;

    wlanUpdatePhyChAPs(pDev, pOpBss, mode);

    /* finish setting up the rate table for reverse lookup */
    for (i = 0; i < RATE_TABLE_SIZE; ++i) {
        pRateTable->rateCodeToIndex[i] = -1;
    }

    /* turboPrime - instead of looping from beginning to end of array, loop in opposite direction */
    for (i = pRateTable->rateCount-1; i < 0xff; i--) {
        A_UINT8 rateCode = pRateTable->info[i].rateCode;

        pRateTable->rateCodeToIndex[rateCode] = i;
        pRateTable->rateCodeToIndex[rateCode | pRateTable->info[i].shortPreamble] = i;
    }

    /* construct the rate set from the rate table */
    wlanRateTbl2RateSet(pRateTable, pRateSet);

    /* 
     * Force basic rates to appear first in the rate set.
     *
     * This is used to provide better backwards compatibility with
     * pre-draft 6.1 STAs/APs that do not support the
     * extended supported rate IE.
     *
     * e.g. 
     * supported rate IE = 1,2,5.5,11,6,12,24 (all basic rates, 8 elements max)
     *
     * Would allow pre-draft 6.1 11g clients to communicate up to 24 Mbps
     * and also associate if in 11g only mode (basic rates= 1-11, 6,12,24)
     *
     */
    if ((mode == WIRELESS_MODE_11g) && !pDev->staConfig.gdraft5) {
        static const WLAN_RATE_SET prefRates = { ELE_RATE_SIZE_11B, 
                                {2, 4, 11, 22, 12, 24, 48, 72}}; 
        /* Sort by perferred rates  1, 2, 5.5, 11, 6, 12, 24, 36 */
        wlanSortByPrefRateSet(pRateSet, &prefRates);
    }

    /* initialize the default rate index based on the default rate */
    for (i = 0; i < pRateTable->rateCount; i++) {
        if (pRateTable->info[i].rateKbps == defaultRate) {
            pOpBss->defaultRateIndex = i;
            return A_OK;
        }
    }

    /* default rate not found */
    return A_ERROR;
}


WIRELESS_MODE
wlanCFlagsToWirelessMode(WLAN_DEV_INFO *pdevInfo, WLAN_CFLAGS cflags)
{
    if (IS_CHAN_CCK(cflags)) {
        return WIRELESS_MODE_11b;
    }
    if (IS_CHAN_G(cflags)) {
        return WIRELESS_MODE_11g;
    }
    if (IS_CHAN_108G(cflags)) {
        return WIRELESS_MODE_108g;
    }
    if (IS_CHAN_TURBO(cflags)) {
        return WIRELESS_MODE_TURBO;
    }
    if (IS_CHAN_XR(cflags)) {
        return WIRELESS_MODE_XR;
    }
    return WIRELESS_MODE_11a;
}

/* TURBO_PRIME - extra param keepRCContent*/
void
wlanUpdateWirelessMode(WLAN_DEV_INFO *pdevInfo, CHAN_VALUES *pChval, A_BOOL keepRCContent)
{
    WIRELESS_MODE mode;
    WIRELESS_MODE tblMode;
    A_UINT8       defaultRateIndex;
    A_UINT16      defaultRateKbps;
    A_STATUS      status;
    CONN_RATE_SET connRateSet;
    SIB_ENTRY     *lSib;

    mode = wlanCFlagsToWirelessMode(pdevInfo, pChval->channelFlags);

    defaultRateIndex = pdevInfo->staConfig.rateCtrlEnable ? LOWEST_RATE_INDEX
                       : pdevInfo->staConfig.defaultRateIndex[mode];

    /*
     * TURBO_PRIME
     * Dynamic turbo uses base combo table when switching modes.
     */
    tblMode = mode;
    if (keepRCContent && ((mode == WIRELESS_MODE_108g) || (mode == WIRELESS_MODE_TURBO))) {
        if (mode == WIRELESS_MODE_108g) {
            tblMode = WIRELESS_MODE_11g;
        } else {
            tblMode = WIRELESS_MODE_11a;
        }
#if NDIS_WDM
        /*     
        * (LW) Attention !!!
        * This is temp hack, should revisit later
        */
        defaultRateIndex += pdevInfo->staConfig.rateCtrlEnable ?
                                 pdevInfo->hwRateTable[tblMode]->rateCount -
                                 pdevInfo->hwRateTable[tblMode]->numTurboRates
                                 : 0;
#else
        defaultRateIndex += pdevInfo->staConfig.rateCtrlEnable ?
                                 HACK_TO_PARDEV(pdevInfo)->hwRateTable[tblMode]->rateCount -
                                 HACK_TO_PARDEV(pdevInfo)->hwRateTable[tblMode]->numTurboRates
                                 : 0;
#endif
    }

#ifdef XR_HACKERY
    if (mode == WIRELESS_MODE_XR) {
        defaultRateKbps = XR_DEFAULT_RATE;
    }
#endif

    /*
     * Inherit the provided hw rate table, unless using TURBO_PRIME, 
     * as indicated by keepRCContent.  As this means we preserve RC
     * information across dynamic turbo mode changes, also need
     * to preserve the corresponding rate table.
     */
    if (keepRCContent) {
        wlanUpdatePhyChAPs(pdevInfo, &pdevInfo->baseBss, mode);
        (&pdevInfo->baseBss)->defaultRateIndex = defaultRateIndex;
    } else {
#if NDIS_WDM
        /*     
        * (LW) Attention !!!
        * This is temp hack, should revisit later
        */
        defaultRateKbps  = pdevInfo->hwRateTable[tblMode]->info[defaultRateIndex].rateKbps;
#else
        defaultRateKbps  = HACK_TO_PARDEV(pdevInfo)->hwRateTable[tblMode]->info[defaultRateIndex].rateKbps;
#endif
        status = wlanSetupPhyInfo(pdevInfo, &pdevInfo->baseBss, mode, defaultRateKbps);
        ASSERT(status == A_OK);
    }

    /* TURBO_PRIME - extra check to keep rate control structure intact */
    lSib = pdevInfo->localSta;
    if (lSib && keepRCContent) {
        wlanRateSet2WdcRateSet(&lSib->workingRateSet, &connRateSet);
        wdcUpdateConnectionAttribute(pdevInfo->targetHandle,
                                     lSib->wdcConnId,
                                     CONN_WORKING_RATE_SET_KEEPRC,
                                     sizeof(connRateSet),
                                     &connRateSet);
    }
}

void
wlanUpdateDefaultRate(WLAN_DEV_INFO *pdevInfo, A_UINT16 rateKbps)
{
    OP_BSS        *pOpBss = &pdevInfo->baseBss;
    A_UINT8       i;
    RATE_TABLE    *pRateTable;
    WIRELESS_MODE mode;

    /* Set the default rate index based on the desired default rate */
    for (mode = 0; mode < WIRELESS_MODE_MAX; mode++) {
#if NDIS_WDM
        /*     
        * (LW) Attention !!!
        * This is temp hack, should revisit later
        */
        pRateTable = pdevInfo->hwRateTable[mode];
#else
        pRateTable = HACK_TO_PARDEV(pdevInfo)->hwRateTable[mode];
#endif
        if (!pRateTable) {
            continue;
        }
        for (i = 0; i < pRateTable->rateCount; i++) {
            if (pRateTable->info[i].rateKbps == rateKbps) {

                pdevInfo->staConfig.defaultRateIndex[mode] = i;

                if (pRateTable == pOpBss->pRateTable) {
                    pOpBss->defaultRateIndex = i;
                }
            }
        }
    }
}

void
wlanRateTbl2RateSet(RATE_TABLE *pRateTable, WLAN_RATE_SET *pRateSet)
{
    int i;

    /* construct the rate set from the rate table */
    pRateSet->length = pRateTable->initialRateMax + 1;
    if (pRateSet->length > MAX_WLAN_RATE_SIZE) {
        pRateSet->length = MAX_WLAN_RATE_SIZE;
    }
    for (i = 0; i < pRateSet->length; i++) {
        pRateSet->rates[i] = pRateTable->info[i].dot11Rate;
    }

}

/* 
 * Sort the WLAN_RATE_SET by preferred rates first
 *
 * Order of preferred rates and non-preferred rates is maintained.
 *
 * eg. pRateSet = 1,2,5.5,11,6,9,12,18,24,36,48,54
 *     pPrefRateSet = 1,2,5.5,11,6,12,24,36
 *
 *     Output   = 1,2,5.5,11,6,12,24,36,9,18,48,54
 */
void
wlanSortByPrefRateSet(WLAN_RATE_SET *pRateSet, const WLAN_RATE_SET *pPrefRateSet)
{
    A_UINT8 i, j, prefIndex, otherIndex;
    WLAN_RATE_SET prefOutput;
    WLAN_RATE_SET otherOutput;

    ASSERT(pRateSet->length <= MAX_WLAN_RATE_SIZE);

    if (pRateSet->length > MAX_WLAN_RATE_SIZE) {
        pRateSet->length = MAX_WLAN_RATE_SIZE;
    }

    /* Separate preferred and other (non-preferred) rates */
    prefIndex  = 0;
    otherIndex = 0;
    for (i = 0; i < pRateSet->length; i++) {
        for (j = 0; j < pPrefRateSet->length; j++ ) {
            if ((pRateSet->rates[i] & 0x7f) == pPrefRateSet->rates[j]) {
                prefOutput.rates[prefIndex] = pRateSet->rates[i];
                prefIndex++;
                break;
            }
        }
        if (j == pPrefRateSet->length) {
            /* Non-preferred rate. Place at end of array */
            otherOutput.rates[otherIndex] = pRateSet->rates[i];
            otherIndex++;
        }
    }

    prefOutput.length   = prefIndex;
    otherOutput.length  = otherIndex;

    /* Copy preferred rates */
    wlanRateSetCopy(&prefOutput, pRateSet);

    /* Append non-preferred rates */
    for (i = 0; i < otherOutput.length; i++) {
        pRateSet->rates[pRateSet->length + i] = otherOutput.rates[i];
    }
    pRateSet->length = prefOutput.length + otherOutput.length;
}


#if defined(DEBUG) || defined(_DEBUG)
int aniDebug = 0;
#endif

#define ANI_STATS_RESET(_pDev, _pChan, _level) {                        \
    PHY_ERR_STATS_RESET(_pDev, OFDM_FALSE_DETECT, aniTriggerAction,     \
            ANI_OFDM_TRIGGER_COUNT_##_level, ANI_TRIGGER_THRESHOLD);         \
    PHY_ERR_STATS_RESET(_pDev, CCK_FALSE_DETECT, aniTriggerAction,      \
            ANI_CCK_TRIGGER_COUNT_##_level, ANI_TRIGGER_THRESHOLD);          \
}

static void
aniTriggerAction(WLAN_DEV_INFO *pdevInfo, PHY_ERR_TYPE phyErr)
{
    CHAN_VALUES *pChan = pdevInfo->staConfig.phwChannel;
    WIRELESS_MODE mode;

    ASSERT(pChan);
    ASSERT((phyErr == OFDM_FALSE_DETECT) || (phyErr == CCK_FALSE_DETECT));

    if ((!pChan->noiseImmunityLevel) && (!pChan->spurImmunityBias)) {
        halAniControl(pdevInfo, SET_NOISE_IMMUNITY_LEVEL, pChan->noiseImmunityLevel = 4);
        aniLog0(pChan, "Noise Immunity Level to Full\n");
        ANI_STATS_RESET(pdevInfo, pChan, LOW);
        return;
    }

    if ((pChan->noiseImmunityLevel < 4) && (!pChan->spurImmunityBias)) {
        halAniControl(pdevInfo, SET_NOISE_IMMUNITY_LEVEL, ++pChan->noiseImmunityLevel);
        aniLog1(pChan, "Noise Immunity Level to %d\n", pChan->noiseImmunityLevel);
        ANI_STATS_RESET(pdevInfo, pChan, HIGH);
        return;
    }

    mode = wlanCFlagsToWirelessMode(pdevInfo, pChan->channelFlags);

    if (((mode == WIRELESS_MODE_11a) || (mode == WIRELESS_MODE_TURBO)) && pChan->spurImmunityLevel) {
        if (pChan->spurImmunityLevel < 7) {
            halAniControl(pdevInfo, SET_SPUR_IMMUNITY_LEVEL, ++pChan->spurImmunityLevel);
            aniLog1(pChan, "Spur Immunity Level increased to %d\n", pChan->spurImmunityLevel);
            ANI_STATS_RESET(pdevInfo, pChan, LOW);
            return;
        }
        /* otherwise let it fall through to turn off ofdm weak signal detection */
    }

    if (((mode == WIRELESS_MODE_11a) || (mode == WIRELESS_MODE_11g)) && (phyErr != CCK_FALSE_DETECT) && (!pChan->ofdmWeakSigDetectionOff)) {
        halAniControl(pdevInfo, SET_OFDM_WEAK_SIGNAL_DETECTION, !(pChan->ofdmWeakSigDetectionOff = TRUE));
        aniLog0(pChan, "OFDM Weak Signal Detection to OFF\n");
        ANI_STATS_RESET(pdevInfo, pChan, LOW);
        return;
    }

    if (pChan->spurImmunityBias) {
        pChan->spurImmunityBias = FALSE;
        aniTriggerAction(pdevInfo, phyErr);
        return;
    }

    if ((mode == WIRELESS_MODE_11b) || (mode == WIRELESS_MODE_11g)) {
        if (!pChan->cckWeakSigThresholdHigh) {
            ASSERT(pChan->firstepLevel < 2);
            halAniControl(pdevInfo, SET_CCK_WEAK_SIGNAL_THR, pChan->cckWeakSigThresholdHigh = TRUE);
            aniLog0(pChan, "CCK Weak Signal Threshold to High\n");
        }
        if (pChan->firstepLevel < 2) {
            halAniControl(pdevInfo, SET_FIRSTEP_LEVEL, ++pChan->firstepLevel);
            aniLog1(pChan, "Firstep Level increased to %d\n", pChan->firstepLevel);
            ANI_STATS_RESET(pdevInfo, pChan, LOW);
            return;
        }
    }

    /*
     * at this point we can't really do anything - except turn off
     * collecting phy error stats to reduce CPU load
     */
    aniLog0(pChan, "ANI at max and still see errors - disabling stats to help CPU\n");
    pChan->phyErrStatsDisabled = TRUE;
    wlanRxFilter(pdevInfo, RX_PHY_ERR, RX_FILTER_CLEAR);
    ANI_STATS_RESET(pdevInfo, pChan, HIGH);
}

void
aniPollFunction(WLAN_DEV_INFO *pdevInfo)
{
    CHAN_VALUES    *pChan = pdevInfo->staConfig.phwChannel;
    PHY_ERR_RECORD *pErr;
    WIRELESS_MODE   mode;

    ASSERT(pChan);

    /*
     * see if we want to proceed - we proceed if the duration
     * corresponding to the last triggerCount errors is large
     * enough
     */
    pErr = PHY_ERR_RECORD_LOOKUP(pdevInfo, OFDM_FALSE_DETECT);
    if (PHY_ERR_STATS_TIME(pErr) <= 5 * OPT_USEC_TO_MSEC(pErr->triggerThreshold)) {
        return;
    }
    pErr = PHY_ERR_RECORD_LOOKUP(pdevInfo, CCK_FALSE_DETECT);
    if (PHY_ERR_STATS_TIME(pErr) <= 5 * OPT_USEC_TO_MSEC(pErr->triggerThreshold)) {
        return;
    }

    if (pChan->phyErrStatsDisabled) {
        aniLog0(pChan, "Maybe noise is gone - turning on stats again\n");
        pChan->phyErrStatsDisabled = FALSE;
        wlanRxFilter(pdevInfo, RX_PHY_ERR, RX_FILTER_SET);
        ANI_STATS_RESET(pdevInfo, pChan, LOW);
        return;
    }

    mode = wlanCFlagsToWirelessMode(pdevInfo, pChan->channelFlags);

    if (((mode == WIRELESS_MODE_11a) || (mode == WIRELESS_MODE_TURBO)) &&
        pChan->ofdmWeakSigDetectionOff && (!pChan->spurImmunityLevel))
    {
        halAniControl(pdevInfo, SET_OFDM_WEAK_SIGNAL_DETECTION, !(pChan->ofdmWeakSigDetectionOff = FALSE));
        halAniControl(pdevInfo, SET_SPUR_IMMUNITY_LEVEL, ++pChan->spurImmunityLevel);
        aniLog0(pChan, "OFDM Weak Signal Detection ON again and Increase Spur Immunity\n");
        ANI_STATS_RESET(pdevInfo, pChan, LOW);
        return;
    }

    if (pChan->noiseImmunityLevel) {
        ASSERT(!pChan->spurImmunityBias);
        halAniControl(pdevInfo, SET_NOISE_IMMUNITY_LEVEL, --pChan->noiseImmunityLevel);
        aniLog1(pChan, "Noise Immunity Level reduced to %d\n", pChan->noiseImmunityLevel);
        ANI_STATS_RESET(pdevInfo, pChan, LOW);
        return;
    }

    if (pChan->firstepLevel) {
        halAniControl(pdevInfo, SET_FIRSTEP_LEVEL, --pChan->firstepLevel);
        aniLog1(pChan, "Firstep Level reduced to %d\n", pChan->firstepLevel);
        ANI_STATS_RESET(pdevInfo, pChan, LOW);
        return;
    }

    if (pChan->cckWeakSigThresholdHigh) {
        halAniControl(pdevInfo, SET_CCK_WEAK_SIGNAL_THR, pChan->cckWeakSigThresholdHigh = FALSE);
        aniLog0(pChan, "CCK Weak Signal Threshold to Low\n");
        ANI_STATS_RESET(pdevInfo, pChan, LOW);
        return;
    }

    if (pChan->spurImmunityLevel) {
        pChan->spurImmunityBias = TRUE;
        halAniControl(pdevInfo, SET_SPUR_IMMUNITY_LEVEL, --pChan->spurImmunityLevel);
        aniLog1(pChan, "Spur Immunity Level reduced to %d\n", pChan->spurImmunityLevel);
        ANI_STATS_RESET(pdevInfo, pChan, LOW);
        return;
    }

    if (pChan->ofdmWeakSigDetectionOff) {
        pChan->spurImmunityBias = TRUE;
        halAniControl(pdevInfo, SET_OFDM_WEAK_SIGNAL_DETECTION, !(pChan->ofdmWeakSigDetectionOff = FALSE));
        aniLog0(pChan, "OFDM Weak Signal Detection to ON\n");
        ANI_STATS_RESET(pdevInfo, pChan, LOW);
        return;
    }
}

void
aniReset(WLAN_DEV_INFO *pdevInfo)
{
    CHAN_VALUES *pChan = pdevInfo->staConfig.phwChannel;

    ASSERT(pChan);

    /* restore the ANI paramters and reset the stats on every hardware reset */

    if (pChan->noiseImmunityLevel) {
        halAniControl(pdevInfo, SET_NOISE_IMMUNITY_LEVEL, pChan->noiseImmunityLevel);
    }
    if (pChan->spurImmunityLevel) {
        halAniControl(pdevInfo, SET_SPUR_IMMUNITY_LEVEL, pChan->spurImmunityLevel);
    }
    if (pChan->ofdmWeakSigDetectionOff) {
        halAniControl(pdevInfo, SET_OFDM_WEAK_SIGNAL_DETECTION, !pChan->ofdmWeakSigDetectionOff);
    }
    if (pChan->cckWeakSigThresholdHigh) {
        halAniControl(pdevInfo, SET_CCK_WEAK_SIGNAL_THR, pChan->cckWeakSigThresholdHigh);
    }
    if (pChan->firstepLevel) {
        halAniControl(pdevInfo, SET_FIRSTEP_LEVEL, pChan->firstepLevel);
    }

    if (pChan->phyErrStatsDisabled) {
        wlanRxFilter(pdevInfo, RX_PHY_ERR, RX_FILTER_CLEAR);
        return;
    }

    ANI_STATS_RESET(pdevInfo, pChan, HIGH);
    wlanRxFilter(pdevInfo, RX_PHY_ERR, RX_FILTER_SET);
}

#ifdef DEBUG
void
aniReport(A_UINT unit)
{
    WLAN_DEV_INFO *pdevInfo;
    CHAN_VALUES   *pChan;

    if (unit >= gDrvInfo.nWorkingDev) {
        return;
    }
    pdevInfo = gDrvInfo.pDev[unit];
    pChan = pdevInfo->staConfig.phwChannel;

    aniLog1(pChan, "Noise Immunity Level %d\n", pChan->noiseImmunityLevel);
    aniLog1(pChan, "Spur Immunity Level %d\n", pChan->spurImmunityLevel);
    aniLog1(pChan, "OFDM Weak Signal Detection %s\n", pChan->ofdmWeakSigDetectionOff ? "OFF" : "ON");
    aniLog1(pChan, "CCK Weak Signal Threshold %s\n", pChan->cckWeakSigThresholdHigh ? "HIGH" : "LOW");
    aniLog1(pChan, "Firstep Level %d\n", pChan->firstepLevel);
    aniLog1(pChan, "Phy Error stats collection is %s\n", pChan->phyErrStatsDisabled ? "DISABLED" : "ENABLED");
    if (!pChan->phyErrStatsDisabled) {
        aniLog1(pChan, "  OFDM trigger count is %d/sec\n",
                PHY_ERR_RECORD_LOOKUP(pdevInfo, OFDM_FALSE_DETECT)->triggerCount);
        aniLog1(pChan, "  CCK trigger count is %d/sec\n",
                PHY_ERR_RECORD_LOOKUP(pdevInfo, CCK_FALSE_DETECT)->triggerCount);
    }
}
#endif
