/*
 * $Id: //depot/sw/branches/1.3_USB_LINUX_port/src/USB/wlan/host/common/wlanUtil.c#1 $
 *
 * Copyright  2000-2003 Atheros Communications, Inc.,  All Rights Reserved.
 *
 * General support routines for the wlan layer
 */

#include "wlantype.h"
#include "wlanext.h"

/*
 * XXX: TODO: Get rid of nasty extern
 */
static A_BOOL
getHexString(A_CHAR **srcString, A_UINT32 *hexVal);



LOCAL A_UINT8 getSetBitPos(A_UINT32 n);

/***********************************************************
 * asciiToMacaddr - Converts ascii string to ethernet mac address
 * DESCRIPTION:
 * This routine parses an ASCII string that looks like a standard
 * ethernet mac address like nnnnnnnnnnnn or nn:nn:nn:nn:nn:nn or
 * '.' or '-' instead of ':'. It is flexible about single digit
 * values too. It places resulting 6 bytes address in pmac.
 *
 * RETURNS: FALSE if conversion failed, TRUE if it succeeds
 */
A_BOOL
asciiToMacaddr(A_UCHAR      *pstr,  // Input ascii string, may not be null terminated
               A_UINT16     alen,   // length of input ascii string
               WLAN_MACADDR *pmac)  // pointer to resulting mac address
{
    int     macIndex   = 0;
    int     asciiIndex = 0;
    int     count      = 0;
    A_UINT8 tval       = 0;

    while (asciiIndex < alen && macIndex < 6) {
        if (pstr[asciiIndex] >= '0' && pstr[asciiIndex] <= '9') {
            tval = (tval << 4) | (pstr[asciiIndex] - '0');
            count++;
        } else if (pstr[asciiIndex] >= 'a' && pstr[asciiIndex] <= 'f') {
            tval = (tval << 4) | (pstr[asciiIndex] - 'a' + 10);
            count++;
        } else if (pstr[asciiIndex] >= 'A' && pstr[asciiIndex] <= 'F') {
            tval = (tval << 4) | (pstr[asciiIndex] - 'A' + 10);
            count++;
        } else if (pstr[asciiIndex] == ':' ||
                   pstr[asciiIndex] == '-' ||
                   pstr[asciiIndex] == '.' ||
                   asciiIndex == (alen-1)) {
            count = 2;
        } else {
            /* bad character */
            break;
        }
        if (count == 2) {
            pmac->octets[macIndex++] = tval;
            count = 0;
            tval = 0;
        }
        asciiIndex++;
    }

    if (macIndex != 6) {
        return FALSE;
    }

    return TRUE;
}

/***********************************************************
 * asciiToKey - Converts ascii string to WEP key
 * DESCRIPTION:
 * This routine parses an ASCII string of hex digits
 * and converts them into a byte array of corresponding values.
 * It also updates key length field.
 *
 * RETURNS: FALSE if conversion failed, TRUE if it succeeds
 *
 */
A_BOOL
asciiToKey(A_UCHAR          *pstr,              // Input ascii string, may not be null terminated
           A_UINT16         alen,               // length of input string
           WLAN_PRIV_RECORD *pKey,              // WEP key record pointer
           A_INT32          keylen,             // length of desired hardware key in bytes
           A_INT32          encryptedKeyLen)    // length of encrypted key stored in memory
{
    int     keyIndex   = 0;
    int     asciiIndex = 0;
    int     count      = 0;
    A_UINT8 tval       = 0;

    while (asciiIndex < alen && keyIndex < encryptedKeyLen) {
        if (pstr[asciiIndex] >= '0' && pstr[asciiIndex] <= '9') {
            tval = (tval << 4) | (pstr[asciiIndex] - '0');
            count++;
        } else if (pstr[asciiIndex] >= 'a' && pstr[asciiIndex] <= 'f') {
            tval = (tval << 4) | (pstr[asciiIndex] - 'a' + 10);
            count++;
        } else if (pstr[asciiIndex] >= 'A' && pstr[asciiIndex] <= 'F') {
            tval = (tval << 4) | (pstr[asciiIndex] - 'A' + 10);
            count++;
        } else {
            /* bad character */
            pKey->keyLength = 0;        // make it an invalid key
            return FALSE;
        }
        if (count == 2) {
            pKey->keyVal[keyIndex++] = tval;
            count = 0;
            tval = 0;
        }
        asciiIndex++;
    }

    if (keyIndex != encryptedKeyLen) {
        pKey->keyLength = 0;            // make it an invalid key
        return FALSE;
    }

    pKey->keyLength = keylen;
    return TRUE;
}

#ifdef BUILD_AP
A_BOOL
asciiToPassphraseKey(A_UINT8 *pstr, A_UINT8 *pPpKey, int encryptedKeyLen)
{
    int     keyIndex   = 0;
    int     asciiIndex = 0;
    int     count      = 0;
    A_UINT8 tval       = 0;

    int     alen       = strlen(pstr); // Length os ascii string

    /* Convert the ascii value read from file to Hex value and store in cfg struct */
    while (asciiIndex < alen && keyIndex < encryptedKeyLen) {
        if (pstr[asciiIndex] >= '0' && pstr[asciiIndex] <= '9') {
            tval = (tval << 4) | (pstr[asciiIndex] - '0');
            count++;
        } else if (pstr[asciiIndex] >= 'a' && pstr[asciiIndex] <= 'f') {
            tval = (tval << 4) | (pstr[asciiIndex] - 'a' + 10);
            count++;
        } else if (pstr[asciiIndex] >= 'A' && pstr[asciiIndex] <= 'F') {
            tval = (tval << 4) | (pstr[asciiIndex] - 'A' + 10);
            count++;
        } else {
            /* bad character */
            return A_EINVAL;
        }
        /* 2 ascii chars make one hex char */
        if (count == 2) {
            /* store hex string in apcfg struct */
            pPpKey[keyIndex++] = tval;
            count = 0;
            tval = 0;
        }
        asciiIndex++;
    }

    return A_OK;
}
#endif

A_BOOL
asciiToPasswd(A_UCHAR *hexString, A_UINT16 hexStrLen,
              A_UINT8 *pOut, A_UINT16 outLen)
{
    int i, j=0;
    A_UINT8 tval = 0;

    for (i = 0; i < hexStrLen; i++) {
        if (hexString[i] >= '0' && hexString[i] <= '9') {
            tval = (tval << 4) | (hexString[i] - '0');
        } else if (hexString[i] >= 'a' && hexString[i] <= 'f') {
            tval = (tval << 4) | (hexString[i] - 'a' + 10);
        } else if (hexString[i] >= 'A' && hexString[i] <= 'F') {
            tval = (tval << 4) | (hexString[i] - 'A' + 10);
        } else {
            /* bad character */
            return FALSE;
        }

        if ((i % 2) == 1) {
            pOut[j++] = tval;
            tval = 0;
        }
    }

    return TRUE;
}

A_BOOL
asciiToMacAddr(A_UCHAR *hexStr, WLAN_MACADDR *pMacAddr)
{
    return asciiHexToChar(hexStr, 12, pMacAddr->octets);
}

A_BOOL
asciiHexToChar(A_UCHAR *hexStr, A_UINT16 hexStrLen, A_UINT8 *pOut)
{
    int     i, j = 0;
    A_UINT8 tval = 0;

    for (i = 0; i < hexStrLen; i++) {
        if (hexStr[i] >= '0' && hexStr[i] <= '9') {
            tval = (tval << 4) | (hexStr[i] - '0');
        } else if (hexStr[i] >= 'a' && hexStr[i] <= 'f') {
            tval = (tval << 4) | (hexStr[i] - 'a' + 10);
        } else if (hexStr[i] >= 'A' && hexStr[i] <= 'F') {
            tval = (tval << 4) | (hexStr[i] - 'A' + 10);
        } else {
            /* bad character */
            return FALSE;
        }

        if ((i % 2) == 1) {
            pOut[j++] = tval;
            tval = 0;
        }
    }

    return TRUE;
}

/**********************************************************************
 * getHexString
 *
 * Returns a pointer to the character after the first consecutive
 * occurence of 0x
 */
static A_BOOL
getHexString(A_CHAR **srcString, A_UINT32 *hexVal)
{
    A_CHAR *pChar, *pNext;
    A_BOOL status;

    pChar = *srcString;

    while (pChar[0] != 0) {
        if ((pChar[0] == '0') && ((pChar[1] == 'x') || (pChar[1] == 'X'))) {
            status = convertHexToU32(&(pChar[2]), hexVal, &pNext);
            if (status == FALSE) {
                return FALSE;
            } else {
                *srcString = pNext;
                return TRUE;
            }
        }
        pChar++;
    }
    return FALSE;
}

/**********************************************************************
 * convertHexToU32
 *
 * Converts up to 8 character values from hex to int starting
 * at hexStr. Returns hexStrEnd to first character beyond the hex value
 */
A_BOOL
convertHexToU32(A_CHAR *hexStr, A_UINT32 *hexVal, A_CHAR **hexStrEnd)
{
    A_UINT32 pos;      /* position in string */
    A_UINT32 num = 0;  /* return number */

    /* Shift values starting from the high nibbles as we read the string */
    for (pos = 0; pos < 8; pos++) {
        A_CHAR ch = hexStr[pos];

        if (hexStr[pos] >= '0'  && hexStr[pos] <= '9' ) {
            num |= ((ch - '0') & 0x0f) << (7 - pos) * 4;
        } else if (ch >='a' && ch <= 'f') {
            num |= ((ch - 'a' + 10) & 0x0f) << (7 - pos) * 4;
        } else if (ch >='A' && ch <= 'F') {
            num |= ((ch - 'A' + 10) & 0x0f) << (7 - pos) * 4;
        } else {
            break;
        }
    }

    if (pos == 0) {
        /* Nothing converted */
        return FALSE;
    }

    /* Shift the high nibbles into the correct position */
    *hexVal    = num >> ((7 - (pos - 1)) * 4);
    *hexStrEnd = &(hexStr[pos]);

    return TRUE;
}

/**********************************************************************
 * parseRegLine
 *
 * Parses the current line for an offset value hex pair.
 * Returns TRUE if offset/value are valid
 *
 * REQUIRES: Input line to be a string (\0 terminated)
 */
A_BOOL
parseRegLine(A_CHAR *pChar, A_UINT32 *offset, A_UINT *value)
{
    A_CHAR *pBuf = pChar;
    A_BOOL status = FALSE;

    /* Ignore comment lines starting with "#" */
    if (*pBuf != '#') {
        /* Find and convert register offset */
        status = getHexString(&pBuf, offset);
        if (status) {
            /* Find and convert register value */
            status = getHexString(&pBuf, value);
        }
    }
    return status;
}
/**********************************************************************
 * bitMapAllocEntry
 *
 * Allocates an available bit from a 16 bit bitMap.
 *
 * INPUT : (1)Pointer to a 16 bit Bit Map.
 *         (2)Location where the virtual port number is returned
 * 1 indicates bit allocated, 0 free.
 *
 * Returns TRUE if alloc succeeded.
 *
 */
A_BOOL
bitMapAllocEntry(A_UINT16 *bitMap, A_UINT8 *bitPos)
{
    A_UINT16 bitMask,n;

    n = *bitMap;

    if (! ~n)
        return FALSE;

    bitMask =  ~n & (~n ^ (~n - 1));
    *bitPos = getSetBitPos(bitMask);
    *bitMap |= bitMask;

    return TRUE;
}
/**********************************************************************
 * bitMapFreeEntry
 *
 * Frees a  bit in the 16 bit bitMap.
 *
 * INPUT : (1)Pointer to a 16 bit Bit Map.
 *         (2)bit position to be freed.
 *
 *
 * Returns TRUE if free succeeded.
 *
 */
A_BOOL
bitMapFreeEntry(A_UINT16 *bitMap, A_UINT8 bitPos)
{
    A_UINT16 bitMask;
    A_BOOL rc = FALSE;

    if (bitPos > 15)
        return FALSE;

    bitMask = 1 << bitPos;

    if (*bitMap & bitMask) {
        *bitMap &= ~bitMask;
        rc = TRUE;
    }
    return rc;
}
/***********************************************************
 * getSetBitPos - Gets the Position of the bit that is
 * set in a 32 bit number. At least 1 bit should be set.
 */

A_UINT8
getSetBitPos(A_UINT32 n)
{

    /*
     * 1,2,4 and 8 are the powers of 2 among the numbers from 0 to 15.
     */

    A_INT8  pwrNblTbl[] = {-1,0,1,-1,2,-1,-1,-1,3,-1,-1,-1,-1,-1,-1,-1 };
    A_UINT8 bitPosBase  = 0,i;
    A_UINT8 numNbl      = ((sizeof(A_UINT32) * NUM_BITS_BYTE)/NUM_BITS_NBL);

    for ( i = 0; i < numNbl; i++) {
        if (n & NBL_MASK) {
            break;
        }
        else {
            bitPosBase += NUM_BITS_NBL;
            n >>= NUM_BITS_NBL;
        }
    }
    return bitPosBase + pwrNblTbl[n & NBL_MASK];
}
/***********************************************************
 * wlanWdsMacHash - Finds the Hash Table Index for tables
 * keyed off the MAC Address.
 *
 */

A_INT32
wlanWdsMacHash(A_UINT32 devno, WLAN_MACADDR *pMacAddr, A_UINT32 maxEntries)
{
    return (A_INT32) (pMacAddr->octets[3] ^ pMacAddr->octets[4] ^
                      pMacAddr->octets[5] ^ devno) & (maxEntries - 1);
}

