/*
 * $Id: //depot/sw/branches/1.3_USB_LINUX_port/src/USB/wlan/host/common/wlanbeacon.c#1 $
 *
 * Copyright (c) 2000-2003 Atheros Communications, Inc., All Rights Reserved
 *
 * Implements beacon and CAB related functionality for both AP and STA.
 */

#include "wlanbeacon.h"
#include "wlanframe.h"
#include "wlanproto.h"
#include "wlanext.h"
#include "wlansmeext.h"
#include "wlansme.h"
#include "wlanchannel.h"
#include "ui.h"
#include "display.h"
#include "usbstub.h"

#ifdef BUILD_AP
#include "apcfg.h"
#endif

#define BEACON_MISS_THRESHOLD      10
#define NONERP_BSS_AGING_PERIOD    (10 * 1000)  /* 10 seconds */

/*
 * Allow some extra TURBO_PRIME channels for debug purposes prior
 * to the 3.0 release.
 */
#define TURBO_DEBUG_CHANNELS 1

/* TURBO_PRIME */
void updateAthAdvCapElement(WLAN_DEV_INFO *pDev);

void *
beaconAthOuiTypeGet(A_UCHAR *p1, A_UINT16 size, A_UINT8 ouiType)
{
    A_UINT16 len, elen;

    /*
     * Now, p1 is the beginning of the variable size fields.
     * len is the offset into the frame from beginning of the variable size fields.
     */
    len = 0;

    /*
     * A valid element should have at least 2 bytes - elementID,
     * length. value byte may not be there if length is 0 (null ssid)
     */
    while (len <= size) {
        elen = ((INFO_ELEMENT *)p1)->length + 2;

        if (((INFO_ELEMENT *)p1)->elementID == ELE_VENDOR_PRIVATE) {
            if (!A_BCOMP(ouiAtheros, ((VENDOR_IE *)p1)->oui, 3)) {
                if (((ATH_VENDOR_IE *)p1)->ouiType == ouiType) {
                    return ((void *)p1);
                }
            }
        } 
        len += elen;
        p1  += elen;
    }

    return NULL;
}

#ifdef WME
void *
beaconMsOuiTypeGet(A_UCHAR *p1, A_UINT16 size, A_UINT8 ouiType)
{
    A_UINT16 len, elen;

    /*
     * Now, p1 is the beginning of the variable size fields.
     * len is the offset into the frame from beginning of the variable size fields.
     */
    len = 0;

    /*
     * A valid element should have at least 2 bytes - elementID,
     * length. value byte may not be there if length is 0 (null ssid)
     */
    while (len <= size) {
        elen = ((INFO_ELEMENT *)p1)->length + 2;

        if (((INFO_ELEMENT *)p1)->elementID == ELE_VENDOR_PRIVATE) {
            if (!A_BCOMP(ouiMicrosoft, ((VENDOR_IE *)p1)->oui, 3)) {
                if (((MS_VENDOR_IE *)p1)->ouiType == ouiType) {
                    return ((void *)p1);
                }
            }
        } 
        len += elen;
        p1  += elen;
    }

    return NULL;
}
#endif

#ifdef BUILD_AP
/******************************************************************************
* buildTim
*
* This routine builds the TIM element of an infrastructure beacon.
*
* Returns: length of the TIM element in bytes
*/
LOCAL A_UINT16
buildTim(BEACON_INFO *pBeaconInfo, A_BOOL cab)
{
    A_UINT8     index, indexN1, indexN2;
    TIM_ELEMENT *pTim;
    A_UCHAR     *pBitmap;

    ASSERT(pBeaconInfo && pBeaconInfo->pTim);

    pBitmap = pBeaconInfo->timBitmap;
    pTim    = pBeaconInfo->pTim;

    indexN1 = indexN2 = 0;
    if (pBeaconInfo->timBitsSet) {
        /*
         * Find N1, which is equal to the largest even number such that
         * bits numbered 1 through (N1 * 8) - 1 in the bitmap are all 0.
         */
        for (index = 0; index < ELE_BITMAP_SIZE; index ++) {
            if (pBitmap[index] != 0) {
                indexN1 = index & 0xfe;
                break;
            }
        }

        /*
         * Find N2, which is equal to the smallest number such that
         * bits numbered (N2 + 1) * 8 through 2007 in the bitmap are all 0.
         */
        for (index = ELE_BITMAP_SIZE - 1; index >= indexN1; index--) {
            if (pBitmap[index] != 0) {
                indexN2 = index;
                break;
            }
        }
    }

    ASSERT(indexN1 <= indexN2);

    /* Copy our partial virtual bitmap into the beacon's TIM element. */
    A_BCOPY(pBitmap + indexN1, pTim->bitmap, indexN2 - indexN1 + 1);

    /* Update the Bitmap Control field. */
    pTim->bitmapControl = indexN1 | (cab ? 1 : 0);

    /* Update the length field. */
    pTim->length = (indexN2 - indexN1) + 4;

    /* Decrement the DTIM Count modulo the DTIM Period. */
    if (pTim->dtimCount == 0) {
        pTim->dtimCount = pTim->dtimPeriod - 1;
    } else {
        pTim->dtimCount--;
    }

    return ELEM_ID_LEN + pTim->length;
}

/* The following are used for 11g mode */

/******************************************************************************
* isOverlapPresent
*
* This routine checks for the NONERP present element of overlapping BSS
* infrastructure beacons.
*
* Returns: 1 if any other nonERP BSS present
*          0 if no nonERP BSS present
*/
LOCAL A_BOOL
isOverlapPresent(WLAN_DEV_INFO *pDev)
{

    if (pDev->protectInfo.otherPresent &&
        ((A_MS_TICKGET() - pDev->protectInfo.timePres) >= NONERP_BSS_AGING_PERIOD))
    {
        pDev->protectInfo.otherPresent = FALSE;
    }

    return pDev->protectInfo.otherPresent;
}

/******************************************************************************
* updateNonErpState
*
* This routine updates the NONERP protect element of
* infrastructure beacons.
*/
LOCAL void
updateNonErpState(WLAN_DEV_INFO *pDev)
{
    BSSDESCR *pBss = pDev->bssDescr;
    if (pDev->staConfig.modeCTS == PROT_MODE_AUTO) {

        if (pDev->protectInfo.option & (PROT_OPTION_2 | PROT_OPTION_4)) {
            pBss->nonErpElement.info.protect = pDev->protectInfo.updateProtect ||
                                               isOverlapPresent(pDev);
        } else {
            pBss->nonErpElement.info.protect = pBss->nonErpElement.info.present ||
                                               isOverlapPresent(pDev);
        }

    } else {
        pBss->nonErpElement.info.protect = pDev->staConfig.modeCTS;
    }
}

#ifdef DEBUG
int dbgProtect = 0;
#endif

/* TURBO_PRIME */
#define TURBO_PRIME_CFP 20

/******************************************************************************
* apBeaconGen
*
* This routine generates the beacon frame for the given BSS based upon
* the snapshot of the current TIM element.  This routine is called only
* by the AP, hence the name.
*
* Assumes: procedure can be called multiple times
*
* Returns: A_OK, A_EBUSY
*/
LOCAL A_STATUS
apBeaconGen(WLAN_DEV_INFO *pDev, OP_BSS *pOpBss, QUEUE_DETAILS *pCabQueue)
{
    BEACON_INFO      *pBeaconInfo  = pOpBss->pBeaconInfo;
    QUEUE_DETAILS    *pBeaconQueue = &pOpBss->beaconQueue;
    ATHEROS_DESC     *pBeaconDesc  = pBeaconQueue->pDescQueueHead;
    QUEUE_DETAILS    *pPsQueue     = &pOpBss->mcPSQueue;
    A_BOOL            cabTraffic   = FALSE;
    ATH_ADVCAP_IE    *pAthAdvCap   = NULL;
    ATH_WME_PARAM_IE *pAthWmeIe    = NULL;
#ifdef WME
    WMEv1_PARAM_IE   *pWmeV1Ie    = NULL;
#endif

    ATHEROS_DESC     *pTail, *pTmp;
    A_UINT32          pfc;
    A_BOOL            fakeDtim     = FALSE;


    /* See if the beacon queue has stopped */
    pfc = halNumTxPending(HACK_TO_PARDEV(pDev), pBeaconQueue->halTxQueueNum);
    if (pfc) {
        pBeaconInfo->missedBeaconCount++;
#ifdef DEBUG
        if (pBeaconInfo->missedBeaconCount < BEACON_MISS_THRESHOLD) {
            isrPrintf("wlan%d%s: isBeaconDone: TXE still high on beacon QCU - %d in a row (PFC = %d)%s\n",
                      pDev->devno, (pOpBss == &pDev->xrBss) ? "(XR)" : "",
                      pBeaconInfo->missedBeaconCount, pfc,
                      halGetTxDescDone(HACK_TO_PARDEV(pDev), (AR_DESC *)pBeaconQueue->pDescQueueHead) ?
                        "; desc marked DONE! hw bug?" : ".");
        } else if (pBeaconInfo->missedBeaconCount == BEACON_MISS_THRESHOLD) {
            halDmaDebugDump(HACK_TO_PARDEV(pDev));
            isrPrintf("wlan%d%s: isBeaconDone: TXE on beacon QCU stuck.\n",
                    pDev->devno, (pOpBss == &pDev->xrBss) ? "(XR)" : "");
        }
#endif
        return A_EBUSY;
    }
    if (pBeaconInfo->missedBeaconCount) {
#ifdef DEBUG
        halDmaDebugDump(HACK_TO_PARDEV(pDev));
        isrPrintf("wlan%d%s: isBeaconDone: Resuming beacon transmission after %d missed beacons!\n",
                  pDev->devno, (pOpBss == &pDev->xrBss) ? "(XR)" : "",
                  pBeaconInfo->missedBeaconCount);
#endif
        pBeaconInfo->missedBeaconCount = 0;
    }

    /*
     * TURBO_PRIME
     * When switching turbo modes dynamically, set a fake dtim to increase
     * the chances of the station waking up to hear the beacon with the
     * change notification.  When switching, any CAB frames are held until
     * the next DTIM.
     */
    if (pDev->turboPrimeInfo.updateFlag) {
        fakeDtim = TRUE;
        pDev->bssDescr->cfParamSet.cfpMaxDuration = A_swab16(TURBO_PRIME_CFP);
    } else {
        pDev->bssDescr->cfParamSet.cfpMaxDuration = 0;
    }

    if (!pCabQueue) {
        pTmp = descQPopHead(pBeaconQueue);
        ASSERT(pTmp == pBeaconDesc);
#if defined(PCI_INTERFACE)
	// xxxJDCxxx
        arProcessTxQueueLocked(HACK_TO_PARDEV(pDev), (AR_QUEUE_INFO *)pBeaconQueue);
#endif
        if ((pTmp = descQGetHead(pBeaconQueue)) && pTmp->staleFlag) {
            freeBuffandDesc(pDev, descQPopHead(pBeaconQueue));
        }
        /*
         * set stale flag on beacon desc lest code below
         * mistakes it for CAB
         */
        pBeaconDesc->staleFlag = TRUE;
        descQPushHead(pBeaconQueue, pBeaconDesc);
        pCabQueue = pBeaconQueue;
    }
    pTail = pCabQueue->pDescQueueTail;

    /* will there be any leftover cab to be drained? */
    if (pTail && (!pTail->staleFlag) && !fakeDtim) {
        A_DESC_CACHE_INVAL(pTail);
        if (!halGetTxDescDone(HACK_TO_PARDEV(pDev), (AR_DESC *)pTail)) {
            cabTraffic = TRUE;
        }
    }

    /* If this is a DTIM, send any buffered BC/MC frames. */
    if (pBeaconInfo->pTim->dtimCount == 0 && pPsQueue->pDescQueueHead && !fakeDtim) {
        /*
         * Set the more bit on the tail frame of the current CAB queue; no
         * harm done if the frame has already gone out
         */
        if (pTail && (!pTail->staleFlag)) {
            FRAME_CONTROL *pFc = &pTail->pTxFirstDesc->pBufferVirtPtr.header->frameControl;

            pFc->moreData = 1;
            A_DATA_CACHE_FLUSH(pFc, sizeof(*pFc));
        }

#if defined(PCI_INTERFACE)
	// xxxJDCxxx
        arQueueTxDescriptor(HACK_TO_PARDEV(pDev), (AR_QUEUE_INFO *)pCabQueue,
                          (AR_DESC *)pPsQueue->pDescQueueHead, (AR_DESC *)pPsQueue->pDescQueueTail);
#endif

        /* Flush the PS queue. */
        pPsQueue->pDescQueueHead = NULL;
        pPsQueue->pDescQueueTail = NULL;
        pPsQueue->qFrameCount    = 0;

        cabTraffic = TRUE;
    }

    /* Rebuild the TIM element. */
    pBeaconInfo->TIMLen = buildTim(pBeaconInfo, fakeDtim || cabTraffic);

    /* TURBO_PRIME */
    /* Update the CFP */
    if (pDev->bssDescr->cfParamSet.cfpPeriod > 0)       // If CFP enabled
    {
        A_UINT16    beaDurRem;  /* temp byteswap copy */
        beaDurRem = A_swab16(pDev->bssDescr->cfParamSet.cfpDurRemain);

        /* This will be the first beacon to go out of the CFP */
        if ((pBeaconInfo->pTim->dtimCount == 0) &&
            (pDev->bssDescr->cfParamSet.cfpCount == 0))
        {
            pDev->bssDescr->cfParamSet.cfpDurRemain = 
                                        pDev->bssDescr->cfParamSet.cfpMaxDuration;
        }
        else if (beaDurRem > pDev->bssDescr->beaconInterval) {
            /* These are beacons to go out while still in CFP */
                beaDurRem -= pDev->bssDescr->beaconInterval;
                pDev->bssDescr->cfParamSet.cfpDurRemain = A_swab16(beaDurRem);
        }
        else
        {
            /* These are beacons to go out after CFP has expired */
            pDev->bssDescr->cfParamSet.cfpDurRemain = 0;
        }
    }
    /* Get copy of current CFParam set */
    mlmeElementCopy((INFO_ELEMENT *)&pDev->bssDescr->cfParamSet,
                    (INFO_ELEMENT *)pDev->baseBss.pBeaconInfo->pCfPtr);

    if (pBeaconInfo->pTim->dtimCount == pBeaconInfo->pTim->dtimPeriod - 1) {
        if (pDev->bssDescr->cfParamSet.cfpPeriod > 1) {
            if (pDev->bssDescr->cfParamSet.cfpCount == 0) {
                pDev->bssDescr->cfParamSet.cfpCount = 
                        pDev->bssDescr->cfParamSet.cfpPeriod - 1;
            } else {
                pDev->bssDescr->cfParamSet.cfpCount--;
            }
        }
    }

#if NDIS_WDM
    /*     
     * (LW) Attention !!!
     * This is temp hack, should revisit later
     */
    if (pOpBss->pRateTable == pDev->hwRateTable[WIRELESS_MODE_11g]) {
#else
    if (pOpBss->pRateTable == HACK_TO_PARDEV(pDev)->hwRateTable[WIRELESS_MODE_11g]) {
#endif
        BSSDESCR *pBss = pDev->bssDescr;
        NONERP_ELEMENT *pNonErpElement;
        A_UINT16 tmp;

        pNonErpElement = (NONERP_ELEMENT *)((A_UCHAR *)pBeaconInfo->postTIMBuf + 
                         (pDev->staConfig.multiDomainCapEnabled ? 
                         INFO_ELEMENT_SIZE(pDev->bssDescr->ccList) : 0));

        /* 11G mode: update the non ERP element if it exists */
        updateNonErpState(pDev);
#if defined(PCI_INTERFACE)
        HACK_TO_PARDEV(pDev)->protectOn = pBss->nonErpElement.info.protect;
        HACK_TO_PARDEV(pDev)->nonErpPreamble = pBss->nonErpElement.info.preamble;
#endif

        /* update the beacon field */
        pNonErpElement->info = pBss->nonErpElement.info;

        /* 11G mode: update short slot time */
        if (pDev->useShortSlotTime != MAKE_BOOL(pBss->capabilityInfo.shortSlotTime)) {
            pDev->useShortSlotTime = MAKE_BOOL(pBss->capabilityInfo.shortSlotTime);
            wdcUpdateBssAttribute(pDev->targetHandle, 0,
                SHORT_SLOT_TIME_11g, (A_UINT32)&pDev->useShortSlotTime);
        }

#ifdef DEBUG
        if ((dbgProtect & 0x01) &&
            (pDev->protectInfo.timer == pDev->protectInfo.period))
        {
            isrPrintf(" %d %d %d %d \n",
                      pBss->nonErpElement.info.present ? 1 : 0,
                      pBss->nonErpElement.info.protect ? 1 : 0,
                      pBss->nonErpElement.info.preamble ? 1 : 0,
                      pBss->capabilityInfo.shortSlotTime ? 0 : 1);
        }
#endif

        /* update the beacon field */
        tmp = le2cpu16(*((A_UINT16 *)(&pBeaconDesc->pBufferVirtPtr.beacon->capInfo)));
        ((CAP_INFO *)&tmp)->shortSlotTime = pDev->useShortSlotTime ?
                                            TRUE : FALSE;
        *((A_UINT16 *)(&pBeaconDesc->pBufferVirtPtr.beacon->capInfo)) = cpu2le16(tmp);
    }

    /* Append country code element, nonErp element, WPA element after the TIM. */
    A_BCOPY(pBeaconInfo->postTIMBuf,
            (A_CHAR *)pBeaconInfo->pTim + pBeaconInfo->TIMLen,
            pBeaconInfo->postTIMLen);

    /*
     * Recompute the beacon frame length, update the beacon descriptor,
     * and queue the Beacon and CAB frames to HW.
     */
    pBeaconInfo->frameLen = pBeaconInfo->nonTIMLen + pBeaconInfo->TIMLen;

    /* update the Adv Cap element */
    pAthAdvCap = beaconAthOuiTypeGet(&pBeaconDesc->pBufferVirtPtr.beacon->buffer[0],
                                     pBeaconInfo->frameLen, ATH_OUI_TYPE_CAP);
    if (pAthAdvCap != NULL) {
        pAthAdvCap->info = pDev->bssDescr->athAdvCapElement.info;
    }

    pAthWmeIe = beaconAthOuiTypeGet(&pBeaconDesc->pBufferVirtPtr.beacon->buffer[0],
                                       pBeaconInfo->frameLen, ATH_OUI_TYPE_WME);
    if (pAthWmeIe && (pAthWmeIe->info.info != pOpBss->phyChAPs.info)) {
        athWmeIeGenerate(pDev, pOpBss, (INFO_ELEMENT *)pAthWmeIe);
    }

#ifdef WME
    pWmeV1Ie = beaconMsOuiTypeGet(&pBeaconDesc->pBufferVirtPtr.beacon->buffer[0],
                                       pBeaconInfo->frameLen, WMEv1_OUI_TYPE);
    if (pWmeV1Ie && (pWmeV1Ie->info.info != pOpBss->phyChAPs.info)) {
        wmeV1IeGenerate(pDev, pOpBss, (INFO_ELEMENT *)pWmeV1Ie);
    }
#endif

    /*
     * AP Beacon Antenna Diversity:
     * Toggle the tx antenna every 4 beacons
     */
    halSetupBeaconDesc(pDev, pBeaconDesc, pBeaconInfo->txBeaconCount & 4 ? 2 : 1, TRUE);

    A_DATA_CACHE_FLUSH(pBeaconDesc->pBufferVirtPtr, pBeaconInfo->frameLen);
    A_DESC_CACHE_FLUSH(pBeaconDesc);
    A_PIPEFLUSH();

    /*
     * Enable the cab queue before the beacon queue to avoid the remote
     * race condition where the beacon queue is ready, fires, and triggers
     * the CAB queue, but the CAB queue is not yet ready
     */
    halSetTxDP(HACK_TO_PARDEV(pDev), pBeaconQueue->halTxQueueNum, pBeaconDesc->thisPhysPtr);
    if (cabTraffic) {
        halStartTxDma(HACK_TO_PARDEV(pDev), pCabQueue->halTxQueueNum);
    }

    halStartTxDma(HACK_TO_PARDEV(pDev), pBeaconQueue->halTxQueueNum);

    pBeaconInfo->txBeaconCount++;

    return A_OK;
}
#endif


/******************************************************************************
* wlanApBeaconGen
*
* This routine generates the beacon frame based upon the snapshot of the
* current TIM element.  This routine is called only by the AP, hence the name.
*
* Assumes: procedure can be called multiple times
*
* Returns: A_OK, A_EBUSY
*/
A_STATUS
wlanApBeaconGen(WLAN_DEV_INFO *pDev)
{
#ifdef BUILD_AP
    int      beaconCount = pDev->baseBss.pBeaconInfo->txBeaconCount;
    A_STATUS status;

    /* TURBO_PRIME */
    if (pDev->staConfig.abolt & ABOLT_TURBO_PRIME) {
        updateAthAdvCapElement(pDev);
    }
    /* 11g mode */
    if (IS_CHAN_G(pDev->staConfig.pChannel->channelFlags) &&
        pDev->protectInfo.option & (PROT_OPTION_2 | PROT_OPTION_3 | PROT_OPTION_4))
    {
        BSSDESCR *pBss = pDev->bssDescr;
        A_UINT32 deltaCountCCK;

        deltaCountCCK = pDev->protectInfo.countCckFrames - pDev->protectInfo.countLast;

        /*
         * When number of cck data frames exceeds up threshold,
         * turn on protection.
         */
        if (deltaCountCCK > pDev->protectInfo.upThresh) {
            pDev->protectInfo.updateState = UPDATE_ON;
            if (pDev->protectInfo.option & PROT_OPTION_3) {
                pBss->nonErpElement.info.present = 1;
            }
            pDev->protectInfo.updateProtect = TRUE;
        }

        /* Sample period timer */
        if (pDev->protectInfo.timer) {
            pDev->protectInfo.timer--;
        } else {
            pDev->protectInfo.timer = pDev->protectInfo.period;

            /*
             * When number of cck data frames below down threshold,
             * turn off protection.
             */
            if (pDev->protectInfo.updateState == UPDATE_ON &&
                deltaCountCCK < pDev->protectInfo.dnThresh)
            {
                pDev->protectInfo.updateState = UPDATE_OFF;
                if (pDev->protectInfo.option & PROT_OPTION_3) {
                    pBss->nonErpElement.info.present = 0;
                }
                pDev->protectInfo.updateProtect = FALSE;
            }
#ifdef DEBUG
            if (dbgProtect & 0x01) {
                isrPrintf("+%d %d %d, %d %d ",
                    deltaCountCCK,
                    pDev->protectInfo.updateState,
                    pDev->protectInfo.otherPresent,
                    pDev->protectInfo.updatePresent,
                    pDev->protectInfo.updateProtect);
            }
#endif
            /* restart counting for next sample period. */
            pDev->protectInfo.countLast = pDev->protectInfo.countCckFrames;
        }
    }

    status = apBeaconGen(pDev, &pDev->baseBss, &pDev->baseBss.cabQueue);
    if (status != A_OK) {
        return status;
    }

    if (isXrAp(pDev) && ((beaconCount & (XR_BEACON_FACTOR - 1)) == 0)) {
        status = apBeaconGen(pDev, &pDev->xrBss, NULL);
    }
#endif  /* BUILD_AP */

    return A_OK;
}


/******************************************************************************
* beaconFree
*
* This routine frees all memory allocated in wlanBeaconInit().
*/
LOCAL void
beaconFree(WLAN_DEV_INFO* pDev, OP_BSS *pOpBss)
{
#ifdef BUILD_AP
    if (A_SEM_VALID(pOpBss->mcPSQueue.qSem)) {
        resetTxQueue(pDev, pOpBss->mcPSQueue.targetQueueHandle, DONT_COMPLETE);
        A_SEM_DELETE(pOpBss->mcPSQueue.qSem);
    }

    /*
     * the following should free up the cab queue for
     * normal BSS and the grpPollQueue for the XR BSS
     */
#if (grpPollQueue != cabQueue)
#error code needs work
#endif
    if (A_SEM_VALID(pOpBss->altQueue.qSem)) {
        resetTxQueue(pDev, pOpBss->altQueue.targetQueueHandle, DONT_COMPLETE);
        A_SEM_DELETE(pOpBss->altQueue.qSem);
        halReleaseTxQueue(HACK_TO_PARDEV(pDev), pOpBss->altQueue.halTxQueueNum);
    }
#endif

    if (A_SEM_VALID(pOpBss->beaconQueue.qSem)) {
        freeBuffandDescChain(pDev, pOpBss->beaconQueue.pDescQueueHead);
        pOpBss->beaconQueue.pDescQueueHead = pOpBss->beaconQueue.pDescQueueTail = NULL;
        A_ATOMIC_SET(&pOpBss->beaconQueue.qFrameCount,0);
        A_SEM_DELETE(pOpBss->beaconQueue.qSem);
    }

    if (pOpBss->pBeaconInfo) {
        A_SEM_DELETE(pOpBss->pBeaconInfo->timSemaphoreId);
        A_DRIVER_FREE(pOpBss->pBeaconInfo, sizeof(BEACON_INFO));
        pOpBss->pBeaconInfo = NULL;
    }
}


/******************************************************************************
* beaconInit
*
* This routine some initialization common to setting up beacons
* for various BSS's - it allocates the global beacon descriptor,
* and the powersave queue for multicasts/broadcasts on the AP.
* Returns: success or failure
*/
LOCAL A_STATUS
beaconInit(OP_BSS *pOpBss)
{
    BEACON_INFO *pBeaconInfo;

    /* Allocate and zero-out the beacon information structure. */
    ASSERT(pOpBss->pBeaconInfo == NULL);
    pBeaconInfo = (BEACON_INFO *)A_DRIVER_MALLOC(sizeof(BEACON_INFO));
    if (pBeaconInfo == NULL) {
        uiPrintf("beaconInit: Error allocating beacon struct!\n");
        return A_ERROR;
    }
    A_MEM_ZERO(pBeaconInfo, sizeof(BEACON_INFO));
    pOpBss->pBeaconInfo = pBeaconInfo;

    /* Create and init the semaphore for TIM */
    A_SEM_INIT(pBeaconInfo->timSemaphoreId, 0, CREATE_UNLOCKED);
    if (!A_SEM_VALID(pBeaconInfo->timSemaphoreId)) {
        uiPrintf("beaconInit: cannot create TIM semaphore!\n");
        return A_ERROR;
    }

#ifdef BUILD_AP
    /*
     * setup queue for buffered mc/bc traffic - should be
     * OK to allocate it and leave it unused for station
     * side (adhoc mode)
     */
    if (descQInit(&pOpBss->mcPSQueue) != A_OK) {
        uiPrintf("beaconInit: cannot initialize PS queue for mc/bc traffic!\n");
        return A_ERROR;
    }
#endif

    return A_OK;
}


/******************************************************************************
* setupBeacon
*
* This routine sets up the beacon frame for the given BSS.
* It returns the descriptor corresponding to the beacon.
* buffers for beacon and TIM element
* (if AP) or IBSS element (if client).
* It returns the descriptor corresponding to the beacon.
*/
LOCAL ATHEROS_DESC *
setupBeacon(WLAN_DEV_INFO *pDev, OP_BSS *pOpBss, A_UINT16 beaconInterval)
{
    A_BOOL             isAp        = (pDev->localSta->serviceType == WLAN_AP_SERVICE);
    BSSDESCR          *pBss        = pDev->bssDescr;
    BEACON_INFO       *pBeaconInfo = pOpBss->pBeaconInfo;
    ATHEROS_DESC      *pBeaconDesc;
    WLAN_FRAME_BEACON *pBeacon;
    INFO_ELEMENT      *pElement;
    TIM_ELEMENT       *pTim;

    /*
     * Allocate the beacon buffer and descriptor - don't
     * worry about calculating the exact beacon length
     * for allocating the buffer - just get a large enough
     * buffer; exact length varies given that the TIM will
     * vary depending on the state of the system
     */
    if (createBuffandDesc(pDev, AR_BUF_SIZE, &pBeaconDesc) != A_OK) {
        uiPrintf("setupBeacon: cannot allocate beacon!\n");
        return NULL;
    }
    pBeacon = pBeaconDesc->pBufferVirtPtr.beacon;
    A_MEM_ZERO(pBeacon, AR_BUF_SIZE);

    /*
     * Start filling up the beacon frame with the mac hdr
     * followed by the frame body; the timestamp value is
     * supplied by HW when transmitted
     */
    pBeacon->macHeader.frameControl.fType    = FRAME_MGT;
    pBeacon->macHeader.frameControl.fSubtype = SUBT_BEACON;
    pBeacon->macHeader.frameControl.protoVer = PROTOCOL_VER_0;
    A_MACADDR_COPY(&broadcastMacAddr, &pBeacon->macHeader.destAddr);
    A_MACADDR_COPY(&pOpBss->bssId, &pBeacon->macHeader.bssId);
    A_MACADDR_COPY(isAp ? &pOpBss->bssId : &pDev->localSta->macAddr, &pBeacon->macHeader.srcAddr);
    pBeacon->beaconInterval = beaconInterval;
    pBeacon->capInfo        = pBss->capabilityInfo;

    pElement = (INFO_ELEMENT *)&pBeacon->buffer[0];

    /* Fill in the SSID taking care of suppress SSID support */
    pElement = mlmeElementCopy((INFO_ELEMENT *)&pBss->ssid, pElement);
    if (isAp && pDev->staConfig.ssidSuppressEnabled) {
        A_MEM_ZERO(((A_UCHAR *)pElement - pBss->ssid.length), pBss->ssid.length);
    }

    /* Fill in the Supported Rates. */
    wlanFullToRateSet(&pOpBss->rateSet, (RATE_SET *)pElement, pDev->staConfig.gdraft5);
    pElement = (INFO_ELEMENT *)((A_UCHAR *)pElement + INFO_ELEMENT_SIZE(*pElement));

    /*
     * Build the DS parameter set and copy it into the beacon.
     * We'll do this for both 11b and 11a. Cisco uses it for 11a too.
     */
    pBss->dsParamSet.elementID      = ELE_DS_PARAM_SET;
    pBss->dsParamSet.length         = 1;
    pBss->dsParamSet.currentChannel =
        (A_UINT8) wlanConvertGHztoCh(pBss->pChannel->channel, pBss->pChannel->channelFlags);
    pElement = mlmeElementCopy((INFO_ELEMENT *)&pBss->dsParamSet, pElement);

    if (VALID_CF_ELEMENT(&pBss->cfParamSet)) {
        pBss->cfParamSet.cfpCount = pBss->cfParamSet.cfpPeriod - 1;
        pBeaconInfo->pCfPtr = (A_UCHAR *)pElement;   /* cache it for future reference */
        pElement = mlmeElementCopy((INFO_ELEMENT *)&pBss->cfParamSet, pElement);
    }

    if (pBss->bsstype == INDEPENDENT_BSS) {
        pElement = mlmeElementCopy((INFO_ELEMENT *)&pBss->ibssParamSet, pElement);
    }

    if (isAp) {
        /*
         * Initialize the TIM element in the beacon.
         * Since the first beacon sent out will not be at TSF = 0,
         * dtimCount should be DTIMPeriod - 1
         */
        pTim                = (TIM_ELEMENT *)pElement;
        pTim->elementID     = ELE_TIM;
        pTim->length        = 4;
        pTim->dtimPeriod    = pBss->DTIMPeriod;
        pTim->dtimCount     = pTim->dtimPeriod - 1;
        pTim->bitmapControl = 0;
        pTim->bitmap[0]     = 0;

        pBeaconInfo->pTim = pTim;   /* cache it for future reference */
        pElement = (INFO_ELEMENT *)((A_UCHAR *)pElement + INFO_ELEMENT_SIZE(*pElement));

        /* Add country code element if enabled */
        if (pDev->staConfig.multiDomainCapEnabled) {
            mlmeCountryCodeElementFill(pDev, (COUNTRY_INFO_LIST *)pElement);
            pElement = (INFO_ELEMENT *)((A_UCHAR *)pElement + INFO_ELEMENT_SIZE(*pElement));
        }

#if NDIS_WDM
        /*     
        * (LW) Attention !!!
        * This is temp hack, should revisit later
        */
        if (pOpBss->pRateTable == pDev->hwRateTable[WIRELESS_MODE_11g]) {
            pElement = mlmeElementCopy((INFO_ELEMENT *)&pBss->nonErpElement, pElement);
        }
#else
        if (pOpBss->pRateTable == HACK_TO_PARDEV(pDev)->hwRateTable[WIRELESS_MODE_11g]) {
            pElement = mlmeElementCopy((INFO_ELEMENT *)&pBss->nonErpElement, pElement);
        }
#endif
        wlanFullToExtRateSet(&pOpBss->rateSet, (EXT_RATE_SET *)pElement, pDev->staConfig.gdraft5);
        pElement = (INFO_ELEMENT *)((A_UCHAR *)pElement + INFO_ELEMENT_SIZE(*pElement));

        if (pDev->staConfig.wpaEnabled && VALID_WPA_ELEMENT(&pBss->wpaIe)) {
            pElement = mlmeElementCopy((INFO_ELEMENT *)&pBss->wpaIe, pElement);
        }

        if (pBss->athAdvCapElement.length != 0) {
            pElement = mlmeElementCopy((INFO_ELEMENT *)&pBss->athAdvCapElement, pElement);
        }

        pElement = athWmeIeGenerate(pDev, pOpBss, pElement);

#ifdef WME 
        pElement = wmeV1IeGenerate(pDev, pOpBss, pElement);
#endif

        pBeaconInfo->TIMLen     = INFO_ELEMENT_SIZE(*pBeaconInfo->pTim);
        pBeaconInfo->postTIMLen = (A_UCHAR *)pElement - ((A_UCHAR *)pBeaconInfo->pTim + pBeaconInfo->TIMLen);
    }

    pBeaconInfo->frameLen  = (A_UCHAR *)pElement - (A_UCHAR *)pBeacon;
    pBeaconInfo->nonTIMLen = pBeaconInfo->frameLen - pBeaconInfo->TIMLen;
    ASSERT(pBeaconInfo->frameLen <= AR_BUF_SIZE);

    pBeaconDesc->bufferLength = pBeaconInfo->frameLen;
    pBeaconDesc->frameLength  = pBeaconInfo->frameLen;

    pBeaconDesc->pOpBss       = pOpBss;
    pBeaconDesc->pTxFirstDesc = pBeaconDesc;
    pBeaconDesc->pTxLastDesc  = pBeaconDesc;

    /* endian swap for all the elements */
    mgtFrameEndianChange((WLAN_MGT_MAC_HEADER *)pBeacon, pBeaconInfo->frameLen, CPU2LE);

    /* cache data after TIM to copy over in wlanBeaconGen() */
    ASSERT(pBeaconInfo->postTIMLen <= POST_TIM_LEN);
    A_BCOPY(((A_CHAR *)pBeaconInfo->pTim + pBeaconInfo->TIMLen),
             pBeaconInfo->postTIMBuf,
             pBeaconInfo->postTIMLen);

    return pBeaconDesc;
}


/******************************************************************************
* xrBeaconInit
*
* Initialization of xr beacon and xr group poll queue
* The XR beacon queue needs to be initialized at some
* offset from the normal beacon - so that its CBR timer
* fires half-way through the normal BSS's beacon period
*
* RETURNS: OK if succesful
*          ERROR if unsuccessful
*/
#ifdef BUILD_AP
LOCAL void
xrBeaconInitISR(WLAN_DEV_INFO *pDev)
{
    QUEUE_DETAILS     *pQueue = &pDev->xrBss.beaconQueue;
    HAL_TX_QUEUE_INFO  queueParam;

    if (!pQueue->pDescQueueHead) {
        return;
    }

    /*
     * always use base BSS's parameters in determining AP's access
     * to the XR channel
     */
    A_QUEUE_INFO_INIT(&queueParam, &pDev->baseBss.phyChAPs.ac[ACI_BE]);
    queueParam.mode             = TXQ_MODE_NORMAL;
    queueParam.priority         = TXQ_ID_FOR_XR_BEACON;
    queueParam.cbrPeriod        = TU_TO_US(pDev->bssDescr->beaconInterval * XR_BEACON_FACTOR);
    queueParam.cbrOverflowLimit = 1;
    queueParam.readyTime        = TU_TO_US(pDev->bssDescr->beaconInterval * (XR_BEACON_FACTOR - 1));
    pQueue->halTxQueueNum       = halSetupTxQueue(HACK_TO_PARDEV(pDev), &queueParam);
    ASSERT(queueParam.readyTime);

#ifdef DEBUG
    if (pDev->baseBss.pBeaconInfo->txBeaconCount) {
        isrPrintf("wlan%d::xrBeaconInitISR: too late? base BSS beacon count = %d\n",
                  pDev->devno, pDev->baseBss.pBeaconInfo->txBeaconCount);
    }
#endif
}
#endif
LOCAL A_STATUS
xrBeaconInit(WLAN_DEV_INFO *pDev)
{
#ifdef BUILD_AP
    QUEUE_DETAILS         *pQueue;
    HAL_TX_QUEUE_INFO      queueParam;
    WLAN_DATA_MAC_HEADER3 *pGrpPollFrame;
    WLAN_FRAME_CFEND      *pCFEndFrame;
    ATHEROS_DESC          *pDesc;

    if (!isXrAp(pDev)) {
        return A_OK;
    }

    if (beaconInit(&pDev->xrBss) != A_OK) {
        uiPrintf("xrBeaconInit: beaconInit failed\n");
        return A_ERROR;
    }

    pQueue = &pDev->xrBss.beaconQueue;
    ASSERT(pQueue->pDescQueueHead == NULL);
    if (descQInit(pQueue) != A_OK) {
        uiPrintf("xrBeaconInit: cannot initialize beaconQueue\n");
        return A_ERROR;
    }
    pDesc = setupBeacon(pDev, &pDev->xrBss, pDev->bssDescr->beaconInterval * XR_BEACON_FACTOR);
    if (!pDesc) {
        uiPrintf("xrBeaconInit: failed to create beacon frame!\n");
        return A_ERROR;
    }
    descQPushTail(pQueue, pDesc, pDesc);
    if (A_DELAYED_EXEC(pDev, pDev->bssDescr->beaconInterval/2, xrBeaconInitISR) != A_OK) {
        uiPrintf("xrBeaconInit: attempt to delayed init failed!\n");
        return A_ERROR;
    }

    pQueue = &pDev->xrBss.grpPollQueue;
    ASSERT(pQueue->pDescQueueHead == NULL);
    if (descQInit(pQueue) != A_OK) {
        uiPrintf("xrBeaconInit: cannot initialize grpPollQueue\n");
        return A_ERROR;
    }
    /*
     * Again, use base BSS's parameters in determining AP's
     * access for the XR channel; disabling post-backoff causes
     * the CF-End to go out sooner following a failed group poll
     */
    A_QUEUE_INFO_INIT(&queueParam, &pDev->baseBss.phyChAPs.ac[ACI_BE]);
    queueParam.mode             = TXQ_MODE_NORMAL;
    queueParam.priority         = TXQ_ID_FOR_XR_GRPPOLL;
    queueParam.logCwMin        += XR_DNLINK_QUEUE_CWMIN_SHIFT;
    queueParam.cbrPeriod        = TU_TO_US(5);
    queueParam.cbrOverflowLimit = 1;
    queueParam.burstTime        = TU_TO_US(5);
    queueParam.qFlags           = TXQ_FLAG_BACKOFF_DISABLE;
    pQueue->halTxQueueNum       = halSetupTxQueue(HACK_TO_PARDEV(pDev), &queueParam);

    if (createBuffandDesc(pDev, sizeof(WLAN_DATA_MAC_HEADER3), &pDesc) != A_OK) {
        uiPrintf("xrBeaconInit: not enough memory for Group Poll frame!\n");
        return A_ERROR;
    }
    descQPushTail(pQueue, pDesc, pDesc);

    pDesc->pTxFirstDesc = pDesc;
    pDesc->pTxLastDesc  = pDesc;

    pDesc->hw.txControl.frameLength  = sizeof(WLAN_DATA_MAC_HEADER3) + FCS_FIELD_SIZE;
    pDesc->hw.txControl.bufferLength = sizeof(WLAN_DATA_MAC_HEADER3);
    pDesc->hw.txControl.more         = 0;

    pGrpPollFrame = pDesc->pBufferVirtPtr.header3;
    pGrpPollFrame->frameControl.fType    = FRAME_DATA;
    pGrpPollFrame->frameControl.fSubtype = SUBT_NODATA_CFPOLL;
    pGrpPollFrame->frameControl.FromDS   = 1;
    pGrpPollFrame->frameControl.ToDS     = 0;
    pGrpPollFrame->frameControl.protoVer = PROTOCOL_VER_0;
    pGrpPollFrame->frameControl.pwrMgt   = 0;
    pGrpPollFrame->frameControl.wep      = 0;
    pGrpPollFrame->frameControl.moreData = 0;
    pGrpPollFrame->frameControl.moreFrag = 0;
    WLAN_SET_DURATION_CFP(pGrpPollFrame->durationNav);
    A_MACADDR_COPY(&broadcastMacAddr, &pGrpPollFrame->address1);
    A_MACADDR_COPY(&pDev->xrBss.bssId, &pGrpPollFrame->address2);
    A_MACADDR_COPY(&pDev->xrBss.bssId, &pGrpPollFrame->address3);

    if (createBuffandDesc(pDev, sizeof(WLAN_DATA_MAC_HEADER3), &pDesc) != A_OK) {
        uiPrintf("xrBeaconInit: not enough memory for CF-End frame!\n");
        return A_ERROR;
    }
    descQPushTail(pQueue, pDesc, pDesc);

    pDesc->pTxFirstDesc = pDesc;
    pDesc->pTxLastDesc  = pDesc;
    pDesc->hw.txControl.frameLength  = sizeof(WLAN_FRAME_CFEND) + FCS_FIELD_SIZE;
    pDesc->hw.txControl.bufferLength = sizeof(WLAN_FRAME_CFEND);
    pDesc->hw.txControl.more         = 0;

    pCFEndFrame = pDesc->pBufferVirtPtr.cfEnd;
    pCFEndFrame->frameControl.fType     = FRAME_CTRL;
    pCFEndFrame->frameControl.fSubtype  = SUBT_CFEND;
    pCFEndFrame->frameControl.FromDS    = 0;
    pCFEndFrame->frameControl.ToDS      = 0;
    pCFEndFrame->frameControl.protoVer  = PROTOCOL_VER_0;
    pCFEndFrame->frameControl.pwrMgt    = 0;
    pCFEndFrame->frameControl.wep       = 0;
    pCFEndFrame->frameControl.moreData  = 0;
    pCFEndFrame->frameControl.moreFrag  = 0;
    WLAN_SET_DURATION_NAV(pCFEndFrame->durationNav, 0);
    A_MACADDR_COPY(&broadcastMacAddr, &pCFEndFrame->recAddr);
    A_MACADDR_COPY(&pDev->xrBss.bssId, &pCFEndFrame->bssId);

    halSetupGrpPollChain(pDev, pQueue->pDescQueueHead, pQueue->pDescQueueTail, HWINDEX_INVALID);
    halSetTxDP(HACK_TO_PARDEV(pDev), pQueue->halTxQueueNum, pQueue->pDescQueueHead->thisPhysPtr);
    halStartTxDma(HACK_TO_PARDEV(pDev), pQueue->halTxQueueNum);
#endif

    return A_OK;
}

/******************************************************************************
* wlanBeaconInit
*
* Setup the CAB and beacon queues; for adhoc beacons the
* cwMinShift is 1 as adhoc beacons use twice the cwMin;
* beacon queue should be the higher priority, and CAB
* should be one lower
*
* RETURNS: OK if succesful
*          ERROR if unsuccessful
*/
A_STATUS
wlanBeaconInit(WLAN_DEV_INFO *pDev)
{
    A_BOOL            isAp = (pDev->localSta->serviceType == WLAN_AP_SERVICE);
    BSSDESCR         *pBss = pDev->bssDescr;
    QUEUE_DETAILS    *pQueue;
    ATHEROS_DESC     *pBeaconDesc;

    /*
     * This is a temporary hack since we currently do not have the ability
     * to disable/enable beacons outside of wlanBeaconRelease() and
     * wlanBeaconInit(). (see bug 4095)
     */
    beaconFree(pDev, &pDev->baseBss);

    /* TURBO_PRIME */
    /*
     * Whenever beacons are initialized, check if turbo prime is allowed
     * as we may have changed channel prior to initializing this beacon.
     */
    pDev->turboPrimeInfo.turboPrimeAllowed =
             (pDev->staConfig.abolt & ABOLT_TURBO_PRIME &&
#ifdef BUILD_AP
                 (apCfgFreqSpecGet(pDev->devno) == MODE_SELECT_11G ||
                  apCfgFreqSpecGet(pDev->devno) == MODE_SELECT_11A) &&
#endif
    /*   6 */    (pDev->staConfig.pChannel->channel == 2437 ||
#ifdef TURBO_DEBUG_CHANNELS
    /* -19 */     pDev->staConfig.pChannel->channel == 2312 ||
    /*  -8 */     pDev->staConfig.pChannel->channel == 2367 ||
    /*  15 */     pDev->staConfig.pChannel->channel == 2512 ||
    /*  18 */     pDev->staConfig.pChannel->channel == 2572 ||
#endif
    /*  40 */     pDev->staConfig.pChannel->channel == 5200 ||
    /*  48 */     pDev->staConfig.pChannel->channel == 5240 ||
    /*  56 */     pDev->staConfig.pChannel->channel == 5280 ||
    /* 153 */     pDev->staConfig.pChannel->channel == 5765 ||
    /* 161 */     pDev->staConfig.pChannel->channel == 5805)) ? TRUE : FALSE;

    pDev->bssDescr->athAdvCapElement.info.useTurboPrime = pDev->turboPrimeInfo.turboPrimeAllowed;

    if (beaconInit(&pDev->baseBss) != A_OK) {
        uiPrintf("wlanBeaconInit: beaconInit failed\n");
        beaconFree(pDev, &pDev->baseBss);
        return A_ERROR;
    }

#ifdef BUILD_AP
    if (isAp) {
        HAL_TX_QUEUE_INFO queueParam;
        pQueue = &pDev->baseBss.cabQueue;
        if (descQInit(pQueue) != A_OK) {
            uiPrintf("wlanBeaconInit: cannot initialize cab queue!\n");
            beaconFree(pDev, &pDev->baseBss);
            return A_ERROR;
        }
        A_QUEUE_INFO_INIT(&queueParam, &pDev->baseBss.phyChAPs.ac[ACI_BE]);
        queueParam.mode     = TXQ_MODE_CAB;
        queueParam.priority = TXQ_ID_FOR_CAB;
        if (pDev->staConfig.abolt & ABOLT_WME_ELE) {
            /*
             * if the AP is configured for channel access params
             * tuning, burst out the beacon and CAB traffic
             */
            queueParam.aifs     = 1;
            queueParam.logCwMin = 0;
            queueParam.logCwMax = 0;
        }
        queueParam.readyTime  = TU_TO_US(pBss->beaconInterval) - 
                                (SW_BEACON_RESPONSE_TIME - DMA_BEACON_RESPONSE_TIME) -
                                ADDITIONAL_SWBA_BACKOFF;
        queueParam.qFlags     = TXQ_FLAG_TXINT_ENABLE;
        pQueue->halTxQueueNum = halSetupTxQueue(HACK_TO_PARDEV(pDev), &queueParam);
    }
#endif

    pQueue = &pDev->baseBss.beaconQueue;
    if (descQInit(pQueue) != A_OK) {
        uiPrintf("wlanBeaconInit: cannot initialize beacon queue!\n");
        beaconFree(pDev, &pDev->baseBss);
        return A_ERROR;
    }

    pBeaconDesc = setupBeacon(pDev, &pDev->baseBss, pBss->beaconInterval);
    if (!pBeaconDesc) {
        uiPrintf("wlanBeaconInit: failed to create beacon frame!\n");
        beaconFree(pDev, &pDev->baseBss);
        return A_ERROR;
    }
    /* can't use descQPushTail lest it screw up physPtr for adhoc beacon */
    pQueue->pDescQueueHead = pQueue->pDescQueueTail = pBeaconDesc;

    A_TX_DESC_CHAIN_TO_WDC(pBeaconDesc);
    wdcStartBss(pDev->targetHandle, 0, &pBeaconDesc->bufDesc);

    /* powersave info */
    pDev->baseBss.pBeaconInfo->calBeaconCount
        = (pDev->staConfig.calibrationTime * 1000) / pBss->beaconInterval;

    pDev->baseBss.pBeaconInfo->beaconEnabled = TRUE;
    pDev->baseBss.pBeaconInfo->bssStartCount++;

    if (xrBeaconInit(pDev) != A_OK) {
        uiPrintf("wlanBeaconInit: could not initialize XR beacons\n");
        wlanBeaconRelease(pDev);
        return A_ERROR;
    }

    return A_OK;
}


/******************************************************************************
* wlanBeaconRelease
*
* This routine resets the beacon on a AP chip reset
*/
void
wlanBeaconRelease(WLAN_DEV_INFO *pDev)
{
    beaconFree(pDev, &pDev->baseBss);
    beaconFree(pDev, &pDev->xrBss);
}


/******************************************************************************
* wlanTimUpdate
*
* This routine updates the virtual bitmap of the TIM element
* to indicate buffered frames at the AP.
*
* Params:  set  TRUE   if you want to set the bit in the bitmap
*               FALSE  if you want to clear it
*
*/
void
wlanTimUpdate(WLAN_DEV_INFO *pDev, A_UINT16 aid, A_BOOL set)
{
    A_UINT8      byteIndex, bitIndex;
    BEACON_INFO  *pBeaconInfo;
    A_UCHAR      *pBitmap;
    
    if (pDev->localSta->serviceType == WLAN_STA_SERVICE) 
        return;

    ASSERT(pDev && pDev->baseBss.pBeaconInfo);

    pBeaconInfo = pDev->baseBss.pBeaconInfo;
    pBitmap     = pBeaconInfo->timBitmap;

    aid &= 0x3fff;
    ASSERT((aid >= ASSOCID_MIN) && (aid <= ASSOCID_MAX));

    A_SEM_LOCK(pBeaconInfo->timSemaphoreId, WAIT_FOREVER);

    /* calculate the index for byte and bit */
    byteIndex = aid / 8;
    bitIndex  = 1 << (aid & 0x7);

    if ((pBitmap[byteIndex] & bitIndex) == (set ? bitIndex : 0)) {
        /* Bit is already in the right state. */
        A_SEM_UNLOCK(pBeaconInfo->timSemaphoreId);
        return;
    }

    if (set) {
        pBitmap[byteIndex] |= bitIndex;
        pBeaconInfo->timBitsSet++;
    } else {
        pBitmap[byteIndex] &= ~bitIndex;
        pBeaconInfo->timBitsSet--;
    }

    A_SEM_UNLOCK(pBeaconInfo->timSemaphoreId);
}


#if defined(DEBUG) && defined(BUILD_AP)
void
beaconShow(int unit)
{
    WLAN_DEV_INFO       *pDev         = gDrvInfo.pDev[unit];
    BEACON_INFO         *pBeaconInfo  = pDev->baseBss.pBeaconInfo;
    QUEUE_DETAILS       *pBeaconQueue = &pDev->baseBss.beaconQueue;
    WLAN_MGT_MAC_HEADER *pBuf;

    if (pBeaconInfo == NULL) {
        uiPrintf("pBeaconInfo is NULL!\n");
        return;
    }

    if (pDev->bssDescr == NULL) {
        uiPrintf("bssDescr is NULL!\n");
        return;
    }

    uiPrintf("Beacon Status: %s\n",
             pBeaconInfo->beaconEnabled ? "enabled" : "disabled");
    uiPrintf("Beacon Interval: %d\n", pDev->bssDescr->beaconInterval);
    uiPrintf("DTIM Interval: %d\n", pDev->bssDescr->DTIMPeriod);
    uiPrintf("DTIM Count: %d\n", pBeaconInfo->pTim->dtimCount);
    uiPrintf("Missed Beacon Count: %d\n", pBeaconInfo->missedBeaconCount);
    uiPrintf("Transmitted Beacon Count: %d\n", pBeaconInfo->txBeaconCount);
    uiPrintf("BSS Start Count: %d\n", pBeaconInfo->bssStartCount);
    uiPrintf("Beacon Watchdog delay tick Count: %d\n", pBeaconInfo->wdgDelayTicks);
    uiPrintf("TIM len: %d + non-TIM len: %d = frame len: %d\n\n",
             pBeaconInfo->TIMLen, pBeaconInfo->nonTIMLen, pBeaconInfo->frameLen);
    uiPrintf("bptr: %p\n", (void *)pBeaconQueue->pDescQueueHead->pBufferVirtPtr.byte);

    pBuf = A_DRIVER_MALLOC(pBeaconInfo->frameLen);
    if (!pBuf) {
        return;
    }
    A_BCOPY(pBeaconQueue->pDescQueueHead->pBufferVirtPtr.byte, pBuf,
            pBeaconInfo->frameLen);
    mgtFrameEndianChange(pBuf, pBeaconInfo->frameLen, LE2CPU);
    displayMgtFrame(pBuf, pBeaconInfo->frameLen);
    A_DRIVER_FREE(pBuf, pBeaconInfo->frameLen);
}

#endif /* DEBUG */

#ifdef BUILD_AP
/* TURBO_PRIME */
void updateAthAdvCapElement(WLAN_DEV_INFO *pDev)
{
    A_UINT8  curBoost;
    BSSDESCR *pBss     = pDev->bssDescr;
    A_UINT32 delta     = pDev->turboPrimeInfo.primeByteCount -
                         pDev->turboPrimeInfo.primeByteCountPrev;
    A_BOOL   isDtim    = pDev->baseBss.pBeaconInfo->pTim->dtimCount == 1 ||
                         pDev->bssDescr->DTIMPeriod == 1;
    A_BOOL   isCapable = !pDev->turboPrimeInfo.legacyPresent ||
                         pDev->turboPrimeInfo.misc & 1;
#ifdef DEBUG
    int oldState = pDev->turboPrimeInfo.state;
#endif

    curBoost = pBss->athAdvCapElement.info.boost;
    pDev->turboPrimeInfo.rateRecn |= pDev->turboPrimeInfo.misc & 2;

    if (pDev->turboPrimeInfo.timer) {
        pDev->turboPrimeInfo.timer--;
    }

    switch (pDev->turboPrimeInfo.state) {
    case PRIME_BASE:
        if (delta > pDev->turboPrimeInfo.upThresh &&
            pDev->turboPrimeInfo.rateRecn &&
            !pDev->turboPrimeInfo.timer &&
            isDtim &&
            isCapable &&
            pDev->turboPrimeInfo.turboPrimeAllowed)
        {
            pDev->turboPrimeInfo.state = PRIME_BOOST;
            pBss->athAdvCapElement.info.boost = 1;
            pDev->turboPrimeInfo.timer = pDev->turboPrimeInfo.periodIn;
        }
        break;

    case PRIME_BOOST:
        if ((delta < pDev->turboPrimeInfo.dnThresh ||
            !pDev->turboPrimeInfo.rateRecn) &&
            pDev->turboPrimeInfo.timer &&
            isCapable &&
            pDev->turboPrimeInfo.turboPrimeAllowed)
        {
            pDev->turboPrimeInfo.state = PRIME_HOLD;
            pDev->turboPrimeInfo.saveTimer = pDev->turboPrimeInfo.timer;
            pDev->turboPrimeInfo.timer = pDev->turboPrimeInfo.periodHold;
        } else if (isDtim &&
            (!pDev->turboPrimeInfo.timer ||
             !isCapable ||
             !pDev->turboPrimeInfo.turboPrimeAllowed))
        {
            pDev->turboPrimeInfo.state = PRIME_BASE;
            pBss->athAdvCapElement.info.boost = 0;
            pDev->turboPrimeInfo.timer = pDev->turboPrimeInfo.periodOut;
        }
        break;

    case PRIME_HOLD:
        if (pDev->turboPrimeInfo.saveTimer) {
            pDev->turboPrimeInfo.saveTimer--;
        }
        if (delta > pDev->turboPrimeInfo.upThresh &&
            pDev->turboPrimeInfo.rateRecn &&
            pDev->turboPrimeInfo.saveTimer &&
            isCapable &&
            pDev->turboPrimeInfo.turboPrimeAllowed)
        {
            pDev->turboPrimeInfo.state = PRIME_BOOST;
            pBss->athAdvCapElement.info.boost = 1;
            pDev->turboPrimeInfo.timer = pDev->turboPrimeInfo.saveTimer;
        }
        else if (isDtim &&
            (!pDev->turboPrimeInfo.timer ||
             !isCapable ||
             !pDev->turboPrimeInfo.turboPrimeAllowed))
        {
            pDev->turboPrimeInfo.state = PRIME_BASE;
            pBss->athAdvCapElement.info.boost = 0;
            pDev->turboPrimeInfo.timer = pDev->turboPrimeInfo.periodOut;
        }
         break;

    default:
        ASSERT(0);
        break;
    }

#ifdef DEBUG
    if (dbgProtect & 0x02 && oldState != pDev->turboPrimeInfo.state) {
        isrPrintf("+%d %d %d, %4d %4d\n",
            oldState,
            pDev->turboPrimeInfo.state,
            pBss->athAdvCapElement.info.boost,
            pDev->turboPrimeInfo.timer,
            pDev->turboPrimeInfo.saveTimer);
    }
    if (dbgProtect & 0x04 && oldState != pDev->turboPrimeInfo.state) {
        isrPrintf(" %6d %d %d %d %d\n",
            delta,
            isDtim,
            isCapable,
            pDev->turboPrimeInfo.rateRecn,
            pDev->turboPrimeInfo.turboPrimeAllowed);
    }
#endif

    /* restart counting for next sample period. */
    pDev->turboPrimeInfo.primeByteCountPrev = pDev->turboPrimeInfo.primeByteCount;

    pDev->turboPrimeInfo.updateFlag =
                    (curBoost != pBss->athAdvCapElement.info.boost) ? 1 : 0;

}

/************************************************************************/
A_STATUS
wlanBeaconRestore(WLAN_DEV_INFO *pDev)
{
    HAL_TX_QUEUE_INFO queueParam;
    QUEUE_DETAILS    *pQueue;
    A_UINT32          queueMask;

    ASSERT(pDev->localSta->serviceType == WLAN_AP_SERVICE);

    pQueue = &pDev->baseBss.cabQueue;
    A_QUEUE_INFO_INIT(&queueParam, &pDev->baseBss.phyChAPs.ac[ACI_BE]);
    queueParam.mode     = TXQ_MODE_CAB;
    queueParam.priority = TXQ_ID_FOR_CAB;
    if (pDev->staConfig.abolt & ABOLT_WME_ELE) {
        /*
         * if the AP is configured for channel access params
         * tuning, burst out the beacon and CAB traffic
         */
        queueParam.aifs     = 1;
        queueParam.logCwMin = 0;
        queueParam.logCwMax = 0;
    }
    queueParam.readyTime  = TU_TO_US(pDev->bssDescr->beaconInterval) - 
                            (SW_BEACON_RESPONSE_TIME - DMA_BEACON_RESPONSE_TIME) -
                            ADDITIONAL_SWBA_BACKOFF;
    queueParam.qFlags     = TXQ_FLAG_TXINT_ENABLE;

    queueMask = (1 << queueParam.priority);
    pDev->pHalInfo->txQueueAllocMask &= ~queueMask;

    halSetupTxQueue(pDev, &queueParam);

    pQueue = &pDev->baseBss.beaconQueue;
    A_QUEUE_INFO_INIT(&queueParam, &pDev->baseBss.phyChAPs.ac[ACI_BE]);
    queueParam.mode     = TXQ_MODE_BEACON;
    queueParam.priority = TXQ_ID_FOR_BEACON;
    if (pDev->staConfig.abolt & ABOLT_WME_ELE) {
        /*
         * if the AP is configured for channel access params
         * tuning, burst out the beacon and CAB traffic
         */
        queueParam.aifs     = 1;
        queueParam.logCwMin = 0;
        queueParam.logCwMax = 0;
    }
    if (pDev->staConfig.abolt & ABOLT_TURBO_PRIME) {
        queueParam.qFlags = TXQ_FLAG_TXDESCINT_ENABLE;
    }

    queueMask = (1 << queueParam.priority);
    pDev->pHalInfo->txQueueAllocMask &= ~queueMask;

    halSetupTxQueue(pDev, &queueParam);

    return A_OK;
}

#endif
