/*
 *
 * Copyright (c) 2000-2003 Atheros Communications, Inc., All Rights Reserved
 *
 * This module contains the regulatory domain table and accessor functions
 * for the information in the table.
 * The channel list creation is also contained in this module.
 *
 * "The country table and respective Regulatory Domain channel and power settings
 * are based on available knowledge as of software release. The underlying global
 * regulatory and spectrum rules change on a regular basis, therefore, no warranty
 * is given that the channel and power information herein is complete, accurate or
 * up to date.  Developers are responsible for regulatory compliance of end-products
 * developed using the enclosed data per all applicable national requirements.
 * Furthermore, data in this table does not guarantee that spectrum is available and
 * that regulatory approval is possible in every case. Knowldegable regulatory
 * compliance or government contacts should be consulted by the manufacturer to ensure
 * that the most current and accurate settings are used in each end-product.
 * This table was designed so that developers are able to update the country table
 * mappings as well as the Regulatory Domain definitions in order to incorporate the
 * most current channel and power settings in the end-product."
 */

#include "wlandrv.h"
#include "wlanext.h"
#include "wlanchannel.h"
#include "wlansmeext.h"
#include "wlanbeacon.h"
#include "stacserv.h"
#include "radarext.h"
#include "display.h"
#include "ui.h"

#ifdef BUILD_AP
#include "wrn/bridge/bridge.h"
#include "ar5hwcEnd.h"
#include "apcfg.h"
#else
#include "stacserv.h"
#endif

int debugScan = 0;
A_BOOL mkkChannelsModified = FALSE;

/* Remove to enforce EEPROM restriction of country codes */
#define ALLOW_DEBUG_COUNTRY

#define FCC_OUTDOOR_FIRST_FREQ 5725

/*
 * Country/Region Codes from MS WINNLS.H
 * Numbering from ISO 3166
 */
enum CountryCode {
    CTRY_ALBANIA              = 8,       // Albania
    CTRY_ALGERIA              = 12,      // Algeria
    CTRY_ARGENTINA            = 32,      // Argentina
    CTRY_ARMENIA              = 51,      // Armenia
    CTRY_AUSTRALIA            = 36,      // Australia
    CTRY_AUSTRIA              = 40,      // Austria
    CTRY_AZERBAIJAN           = 31,      // Azerbaijan
    CTRY_BAHRAIN              = 48,      // Bahrain
    CTRY_BELARUS              = 112,     // Belarus
    CTRY_BELGIUM              = 56,      // Belgium
    CTRY_BELIZE               = 84,      // Belize
    CTRY_BOLIVIA              = 68,      // Bolivia
    CTRY_BRAZIL               = 76,      // Brazil
    CTRY_BRUNEI_DARUSSALAM    = 96,      // Brunei Darussalam
    CTRY_BULGARIA             = 100,     // Bulgaria
    CTRY_CANADA               = 124,     // Canada
    CTRY_CHILE                = 152,     // Chile
    CTRY_CHINA                = 156,     // People's Republic of China
    CTRY_COLOMBIA             = 170,     // Colombia
    CTRY_COSTA_RICA           = 188,     // Costa Rica
    CTRY_CROATIA              = 191,     // Croatia
    CTRY_CYPRUS               = 196,
    CTRY_CZECH                = 203,     // Czech Republic
    CTRY_DENMARK              = 208,     // Denmark
    CTRY_DOMINICAN_REPUBLIC   = 214,     // Dominican Republic
    CTRY_ECUADOR              = 218,     // Ecuador
    CTRY_EGYPT                = 818,     // Egypt
    CTRY_EL_SALVADOR          = 222,     // El Salvador
    CTRY_ESTONIA              = 233,     // Estonia
    CTRY_FAEROE_ISLANDS       = 234,     // Faeroe Islands
    CTRY_FINLAND              = 246,     // Finland
    CTRY_FRANCE               = 250,     // France
    CTRY_GEORGIA              = 268,     // Georgia
    CTRY_GERMANY              = 276,     // Germany
    CTRY_GREECE               = 300,     // Greece
    CTRY_GUATEMALA            = 320,     // Guatemala
    CTRY_HONDURAS             = 340,     // Honduras
    CTRY_HONG_KONG            = 344,     // Hong Kong S.A.R., P.R.C.
    CTRY_HUNGARY              = 348,     // Hungary
    CTRY_ICELAND              = 352,     // Iceland
    CTRY_INDIA                = 356,     // India
    CTRY_INDONESIA            = 360,     // Indonesia
    CTRY_IRAN                 = 364,     // Iran
    CTRY_IRAQ                 = 368,     // Iraq
    CTRY_IRELAND              = 372,     // Ireland
    CTRY_ISRAEL               = 376,     // Israel
    CTRY_ITALY                = 380,     // Italy
    CTRY_JAMAICA              = 388,     // Jamaica
    CTRY_JAPAN                = 392,     // Japan
    CTRY_JAPAN1               = 393,     // Japan (JP1)
    CTRY_JAPAN2               = 394,     // Japan (JP0)
    CTRY_JAPAN3               = 395,     // Japan (JP1-1)
    CTRY_JAPAN4               = 396,     // Japan (JE1)
    CTRY_JAPAN5               = 397,     // Japan (JE2)   
    CTRY_JAPAN6               = 399,     // Japan (JP6)   
    CTRY_JORDAN               = 400,     // Jordan
    CTRY_KAZAKHSTAN           = 398,     // Kazakhstan
    CTRY_KENYA                = 404,     // Kenya
    CTRY_KOREA_NORTH          = 408,     // North Korea
    CTRY_KOREA_ROC            = 410,     // South Korea
    CTRY_KOREA_ROC2           = 411,     // South Korea
    CTRY_KUWAIT               = 414,     // Kuwait
    CTRY_LATVIA               = 428,     // Latvia
    CTRY_LEBANON              = 422,     // Lebanon
    CTRY_LIBYA                = 434,     // Libya
    CTRY_LIECHTENSTEIN        = 438,     // Liechtenstein
    CTRY_LITHUANIA            = 440,     // Lithuania
    CTRY_LUXEMBOURG           = 442,     // Luxembourg
    CTRY_MACAU                = 446,     // Macau
    CTRY_MACEDONIA            = 807,     // the Former Yugoslav Republic of Macedonia
    CTRY_MALAYSIA             = 458,     // Malaysia
    CTRY_MEXICO               = 484,     // Mexico
    CTRY_MONACO               = 492,     // Principality of Monaco
    CTRY_MOROCCO              = 504,     // Morocco
    CTRY_NETHERLANDS          = 528,     // Netherlands
    CTRY_NEW_ZEALAND          = 554,     // New Zealand
    CTRY_NICARAGUA            = 558,     // Nicaragua
    CTRY_NORWAY               = 578,     // Norway
    CTRY_OMAN                 = 512,     // Oman
    CTRY_PAKISTAN             = 586,     // Islamic Republic of Pakistan
    CTRY_PANAMA               = 591,     // Panama
    CTRY_PARAGUAY             = 600,     // Paraguay
    CTRY_PERU                 = 604,     // Peru
    CTRY_PHILIPPINES          = 608,     // Republic of the Philippines
    CTRY_POLAND               = 616,     // Poland
    CTRY_PORTUGAL             = 620,     // Portugal
    CTRY_PUERTO_RICO          = 630,     // Puerto Rico
    CTRY_QATAR                = 634,     // Qatar
    CTRY_ROMANIA              = 642,     // Romania
    CTRY_RUSSIA               = 643,     // Russia
    CTRY_SAUDI_ARABIA         = 682,     // Saudi Arabia
    CTRY_SINGAPORE            = 702,     // Singapore
    CTRY_SLOVAKIA             = 703,     // Slovak Republic
    CTRY_SLOVENIA             = 705,     // Slovenia
    CTRY_SOUTH_AFRICA         = 710,     // South Africa
    CTRY_SPAIN                = 724,     // Spain
    CTRY_SWEDEN               = 752,     // Sweden
    CTRY_SWITZERLAND          = 756,     // Switzerland
    CTRY_SYRIA                = 760,     // Syria
    CTRY_TAIWAN               = 158,     // Taiwan
    CTRY_THAILAND             = 764,     // Thailand
    CTRY_TRINIDAD_Y_TOBAGO    = 780,     // Trinidad y Tobago
    CTRY_TUNISIA              = 788,     // Tunisia
    CTRY_TURKEY               = 792,     // Turkey
    CTRY_UAE                  = 784,     // U.A.E.
    CTRY_UKRAINE              = 804,     // Ukraine
    CTRY_UNITED_KINGDOM       = 826,     // United Kingdom
    CTRY_UNITED_STATES        = 840,     // United States
    CTRY_URUGUAY              = 858,     // Uruguay
    CTRY_UZBEKISTAN           = 860,     // Uzbekistan
    CTRY_VENEZUELA            = 862,     // Venezuela
    CTRY_VIET_NAM             = 704,     // Viet Nam
    CTRY_YEMEN                = 887,     // Yemen
    CTRY_ZIMBABWE             = 716      // Zimbabwe
};

/* Enumerated Regulatory Domain Information */
enum EnumRd {
    /*
     * The following regulatory domain definitions are
     * found in the EEPROM. Each regulatory domain
     * can operate in either a 5GHz or 2.4GHz wireless mode or
     * both 5GHz and 2.4GHz wireless modes.
     * In general, the value holds no special
     * meaning and is used to decode into either specific
     * 2.4GHz or 5GHz wireless mode for that particular
     * regulatory domain.
     *
     */
    NO_ENUMRD     = 0x00,
    NULL1_WORLD   = 0x03, /* For 11b-only countries (no 11a allowed) */
    NULL1_ETSIB   = 0x07, /* Israel */
    NULL1_ETSIC   = 0x08,
    FCC1_FCCA     = 0x10, // USA
    FCC1_WORLD    = 0x11, // Hong Kong

    FCC2_FCCA     = 0x20, // Canada
    FCC2_WORLD    = 0x21, // Australia & HK
    FCC2_ETSIC    = 0x22,
    FCC3_FCCA     = 0x31, // USA & Canada w/5470 band, 11h, DFS enabled   

    ETSI1_WORLD   = 0x37,
    ETSI3_ETSIA   = 0x32, // France (optional)
    ETSI2_WORLD   = 0x35, // Hungary & others
    ETSI3_WORLD   = 0x36, // France & others
    ETSI4_WORLD   = 0x30,
    ETSI4_ETSIC   = 0x38,
    ETSI5_WORLD   = 0x39,
    ETSI6_WORLD   = 0x34, // Bulgaria
    ETSI_RESERVED = 0x33, // Reserved (Do not used)

    MKK1_MKKA     = 0x40, // Japan (JP1)
    MKK1_MKKB     = 0x41, // Japan (JP0)
    APL4_WORLD    = 0x42, // Singapore
    MKK2_MKKA     = 0x43, // Japan with 4.9G channels
    APL_RESERVED  = 0x44, // Reserved (Do not used) 
    APL2_WORLD    = 0x45, // Korea
    APL2_APLC     = 0x46,
    APL3_WORLD    = 0x47,
    MKK1_FCCA     = 0x48, // Japan (JP1-1)
    APL2_APLD     = 0x49, // Korea with 2.3G channels
    MKK1_MKKA1    = 0x4A, // Japan (JE1)
    MKK1_MKKA2    = 0x4B, // Japan (JE  2)
    MKK1_MKKC     = 0x4C, // Japan (MKK1_MKKA,except Ch14)

    APL1_WORLD    = 0x52, // Latin America
    APL1_FCCA     = 0x53,
    APL1_APLA     = 0x54,    
    APL1_ETSIC    = 0x55,
    APL2_ETSIC    = 0x56, // Venezuala
    APL5_WORLD    = 0x58, // Chile

    /*
     * World mode SKUs
     */
    WOR0_WORLD    = 0x60,       /* World0 (WO0 SKU) */
    WOR1_WORLD    = 0x61,       /* World1 (WO1 SKU) */
    WOR2_WORLD    = 0x62,       /* World2 (WO2 SKU) */
    WOR3_WORLD    = 0x63,       /* World3 (WO3 SKU) */
    WOR4_WORLD    = 0x64,       /* World4 (WO4 SKU) */
    WOR5_ETSIC    = 0x65,       /* World5 (WO5 SKU) */    

    WOR01_WORLD   = 0x66,       /* World0-1 (WW0-1 SKU) */
    WOR02_WORLD   = 0x67,       /* World0-2 (WW0-2 SKU) */
    EU1_WORLD     = 0x68,       /* Same as World0-2 (WW0-2 SKU), except active scan ch1-13. No ch14 */

    /*  "Regulatory domains ending in a number (e.g. APL1,
     *   MKK1, ETSI4,etc) apply to 5GHz channel and power information.
     *   Reg. domains ending in a letter (e.g. APLA, FCCA, etc.)
     *   apply to 2.4GHz channel and power information."
     */

    /*
     * The following wireless modes are either 2.4 or 5GHz
     */
    APL1          = 0x0150,     /* LAT & Asia */
    APL2          = 0x0250,     /* LAT & Asia */
    APL3          = 0x0350,     /* Taiwan */
    APL4          = 0x0450,     /* Singapore */
    APL5          = 0x0550,     /* Chile */

    ETSI1         = 0x0130,     /* Europe & others */
    ETSI2         = 0x0230,     /* Europe & others */
    ETSI3         = 0x0330,     /* Europe & others */
    ETSI4         = 0x0430,     /* Europe & others */
    ETSI5         = 0x0530,     /* Europe & others */
    ETSI6         = 0x0630,     /* Europe & others */
    ETSIA         = 0x0A30,     /* France */
    ETSIB         = 0x0B30,     /* Israel */
    ETSIC         = 0x0C30,     /* Latin America */

    FCC1          = 0x0110,     /* US & others */
    FCC2          = 0x0120,     /* Canada, Australia & New Zealand */
    FCC3          = 0x0130,     /* US w/new middle band & DFS */    
    FCCA          = 0x0A10,

    APLD          = 0x0D50,     /* South Korea */

    MKK1          = 0x0140,     /* Japan */
    MKK2          = 0x0240,     /* Japan Extended */
    MKKA          = 0x0A40,     /* Japan */
    
    NULL1         = 0x0198,
    WORLD         = 0x0199,
    DEBUG_REG_DMN = 0x01ff
};

#define DEF_REGDMN              FCC1_FCCA
#define COUNTRY_ERD_FLAG        0x8000
#define WORLDWIDE_ROAMING_FLAG  0x4000
#define SUPER_DOMAIN_MASK       0x0fff
#define COUNTRY_CODE_MASK       0x03ff
#define NUM_OF_COUNTRIES        (sizeof(allCountries)/sizeof(COUNTRY_CODE_TO_ENUM_RD))
#define NUM_OF_SUPER_DOMAINS    (sizeof(superDomainTbl)/sizeof(SUPER_DOMAIN))
#define NUM_OF_TURBO_DOMAINS    (sizeof(regDmnWm5Turbo)/sizeof(REG_DMN_WM_FREQ_TABLE))
#define NUM_OF_11A_DOMAINS      (sizeof(regDmnWm5) / sizeof(REG_DMN_WM_FREQ_TABLE))
#define NUM_OF_11B_DOMAINS      (sizeof(regDmnWm2) / sizeof(REG_DMN_WM_FREQ_TABLE))
#define NUM_OF_108G_DOMAINS     (sizeof(regDmnWm2Turbo)/sizeof(REG_DMN_WM_FREQ_TABLE))
#define NUM_OF_LEGACY_NAMES     (sizeof(allLegacyNames)/sizeof(REG_DMN_LEGACY_NAMES))
#define NUM_OF_REG_DOMAINS      (sizeof(allEnumRds)/sizeof(ENUM_RD_TO_WIRELESS_MODE_RD))
#define CF_INTERFERENCE         (CHANNEL_CW_INT | CHANNEL_RADAR_INT)
#define CHANNEL_14              (2484)  /* 802.11g operation is not permitted on channel 14 */
#define IS_11G_CH14(ch,cf)      (((ch) == CHANNEL_14) && ((cf) == CHANNEL_G))

typedef struct RegDmnLegacyNames {
    A_UCHAR     regDmnName[8];
    REG_DOMAIN  regDmn;
    CTRY_CODE   countryCode;
} REG_DMN_LEGACY_NAMES;

static const REG_DMN_LEGACY_NAMES allLegacyNames[] = {
    {"NONE", NO_ENUMRD,   CTRY_DEBUG          },
    {"FCC",  FCC1_FCCA,   CTRY_UNITED_STATES  },
    {"MKK",  MKK1_MKKA,   CTRY_JAPAN          },
    {"ETSI", ETSI4_WORLD, CTRY_UNITED_KINGDOM }
};

#define YES TRUE
#define NO  FALSE

typedef struct CountryCodeToEnumRd {
    CTRY_CODE       countryCode;
    REG_DOMAIN      regDmnEnum;
    const A_UCHAR   isoName[3];
    const A_UCHAR   name[32];
    A_BOOL          allow11g;    
    A_BOOL          allow11aTurbo;
    A_BOOL          allow11gTurbo;
} COUNTRY_CODE_TO_ENUM_RD;

/* Index into table to avoid DEBUG and NO COUNTRY SET entries */
#define CTRY_ONLY_INDEX 2
/*
 * Country Code Table to Enumerated RD
 */
static const COUNTRY_CODE_TO_ENUM_RD allCountries[] = {
    {CTRY_DEBUG,       NO_ENUMRD,     "DB", "DEBUG",      YES, YES, YES    },
    {CTRY_DEFAULT,     DEF_REGDMN,    "NA", "NO_COUNTRY_SET", YES, YES, YES },
    {CTRY_ALBANIA,     NULL1_WORLD,   "AL", "ALBANIA",    YES, NO, YES     },
    {CTRY_ALGERIA,     NULL1_WORLD,   "DZ", "ALGERIA",    YES, NO, YES     },
    {CTRY_ARGENTINA,   APL3_WORLD,    "AR", "ARGENTINA",   NO, NO, NO     },
    {CTRY_ARMENIA,     ETSI4_WORLD,   "AM", "ARMENIA",    YES, NO, YES     },
    {CTRY_AUSTRALIA,   FCC2_WORLD,    "AU", "AUSTRALIA",  YES, YES, YES    },
    {CTRY_AUSTRIA,     ETSI5_WORLD,   "AT", "AUSTRIA",    YES, NO, YES     },
    {CTRY_AZERBAIJAN,  ETSI4_WORLD,   "AZ", "AZERBAIJAN", YES, YES, YES    },
    {CTRY_BAHRAIN,     NULL1_WORLD,   "BH", "BAHRAIN",    YES, NO, YES     },
    {CTRY_BELARUS,     NULL1_WORLD,   "BY", "BELARUS",    YES, NO, YES     },
    {CTRY_BELGIUM,     ETSI2_WORLD,   "BE", "BELGIUM",    YES, NO, YES     },
    {CTRY_BELIZE,      APL1_ETSIC,    "BZ", "BELIZE",     YES, YES, YES   },
    {CTRY_BOLIVIA,     APL1_ETSIC,    "BO", "BOLVIA",     YES, YES, YES    },
    {CTRY_BRAZIL,      NULL1_ETSIC,   "BR", "BRAZIL",      NO, NO, NO     },
    {CTRY_BRUNEI_DARUSSALAM,APL1_WORLD,"BN", "BRUNEI DARUSSALAM", YES, YES, YES },
    {CTRY_BULGARIA,    ETSI6_WORLD,   "BG", "BULGARIA",   YES, NO, YES    },
    {CTRY_CANADA,      FCC2_FCCA,     "CA", "CANADA",     YES, YES, YES   },
    {CTRY_CHILE,       APL5_WORLD,    "CL", "CHILE",      NO,  YES, NO    },
    {CTRY_CHINA,       APL1_WORLD,    "CN", "CHINA",      YES, YES, YES    },
    {CTRY_COLOMBIA,    FCC1_FCCA,     "CO", "COLOMBIA",   YES, NO, YES     },
    {CTRY_COSTA_RICA,  NULL1_WORLD,   "CR", "COSTA RICA", YES, NO, YES     },
    {CTRY_CROATIA,     ETSI3_WORLD,   "HR", "CROATIA",    YES, NO, YES     },
    {CTRY_CYPRUS,      ETSI1_WORLD,   "CY", "CYPRUS",     YES, YES, YES   },
    {CTRY_CZECH,       ETSI3_WORLD,   "CZ", "CZECH REPUBLIC", NO, NO, NO  },
    {CTRY_DENMARK,     ETSI1_WORLD,   "DK", "DENMARK",    YES, NO, YES     },
    {CTRY_DOMINICAN_REPUBLIC,FCC1_FCCA,"DO", "DOMINICAN REPUBLIC", YES, YES, YES },
    {CTRY_ECUADOR,     NULL1_WORLD,   "EC", "ECUADOR",     NO, NO,NO     },
    {CTRY_EGYPT,       NULL1_WORLD,   "EG", "EGYPT",      YES, NO, YES    },
    {CTRY_EL_SALVADOR, NULL1_WORLD,   "SV", "EL SALVADOR",YES, NO, YES     },    
    {CTRY_ESTONIA,     ETSI1_WORLD,   "EE", "ESTONIA",    YES, NO, YES     },
    {CTRY_FINLAND,     ETSI1_WORLD,   "FI", "FINLAND",    YES, NO, YES     },
    {CTRY_FRANCE,      ETSI3_WORLD,   "FR", "FRANCE",     YES, NO, YES     },
    {CTRY_GEORGIA,     ETSI4_WORLD,   "GE", "GEORGIA",    YES, YES, YES    },
    {CTRY_GERMANY,     ETSI1_WORLD,   "DE", "GERMANY",    YES, NO, YES     },
    {CTRY_GREECE,      NULL1_WORLD,   "GR", "GREECE",     YES, NO, YES     },
    {CTRY_GUATEMALA,   FCC1_FCCA,     "GT", "GUATEMALA",  YES, YES, YES    },
    {CTRY_HONDURAS,    NULL1_WORLD,   "HN", "HONDURAS",   YES, NO, YES     },
    {CTRY_HONG_KONG,   FCC2_WORLD,    "HK", "HONG KONG",  YES, YES, YES    },
    {CTRY_HUNGARY,     ETSI2_WORLD,   "HU", "HUNGARY",    YES, NO, YES     },
    {CTRY_ICELAND,     ETSI1_WORLD,   "IS", "ICELAND",    YES, NO, YES     },
    {CTRY_INDIA,       NULL1_WORLD,   "IN", "INDIA",      YES, NO, YES     },
    {CTRY_INDONESIA,   NULL1_WORLD,   "ID", "INDONESIA",  YES, NO, YES     },
    {CTRY_IRAN,        APL1_WORLD,    "IR", "IRAN",       YES, YES, YES    },
    {CTRY_IRELAND,     ETSI1_WORLD,   "IE", "IRELAND",    YES, NO, YES     },
    {CTRY_ISRAEL,      NULL1_ETSIB,   "IL", "ISRAEL",     YES, NO, YES     },
    {CTRY_ITALY,       ETSI1_WORLD,   "IT", "ITALY",      YES, NO, YES     },
    {CTRY_JAPAN,       MKK1_MKKA,     "JP", "JAPAN",      YES, NO, NO     },
    {CTRY_JAPAN1,      MKK1_MKKB,     "J1", "JAPAN1",     YES, NO, NO     },
    {CTRY_JAPAN2,      MKK1_FCCA,     "J2", "JAPAN2",     YES, NO, NO     },    
    {CTRY_JAPAN3,      MKK2_MKKA,     "J3", "JAPAN3",     YES, NO, NO     },
    {CTRY_JAPAN4,      MKK1_MKKA1,    "J4", "JAPAN4",     YES, NO, NO     },
    {CTRY_JAPAN5,      MKK1_MKKA2,    "J5", "JAPAN5",     YES, NO, NO     },    
    {CTRY_JORDAN,      NULL1_WORLD,   "JO", "JORDAN",     YES, NO, YES     },
    {CTRY_KAZAKHSTAN,  NULL1_WORLD,   "KZ", "KAZAKHSTAN", YES, NO, YES    },
    {CTRY_KOREA_NORTH, APL2_WORLD,    "KP", "NORTH KOREA",YES, YES, YES    },
    {CTRY_KOREA_ROC,   APL2_WORLD,    "KR", "KOREA REPUBLIC", YES, YES, YES },
    {CTRY_KOREA_ROC2,  APL2_APLD,     "K2", "KOREA REPUBLIC2", YES, YES, YES },
    {CTRY_KUWAIT,      NULL1_WORLD,   "KW", "KUWAIT",     YES, NO, YES     },
    {CTRY_LATVIA,      NULL1_WORLD,   "LV", "LATVIA",     YES, NO, YES     },
    {CTRY_LEBANON,     NULL1_WORLD,   "LB", "LEBANON",    YES, NO, YES     },
    {CTRY_LIECHTENSTEIN,ETSI2_WORLD,  "LI", "LIECHTENSTEIN", YES, NO, YES  },
    {CTRY_LITHUANIA,   ETSI1_WORLD,   "LT", "LITHUANIA",  YES, NO, YES     },
    {CTRY_LUXEMBOURG,  ETSI1_WORLD,   "LU", "LUXEMBOURG", YES, NO, YES     },
    {CTRY_MACAU,       FCC2_WORLD,    "MO", "MACAU",      YES, YES, YES    },
    {CTRY_MACEDONIA,   NULL1_WORLD,   "MK", "MACEDONIA",  YES, NO, YES     },
    {CTRY_MALAYSIA,    NULL1_WORLD,   "MY", "MALAYSIA",    NO, NO, NO     },
    {CTRY_MEXICO,      FCC1_FCCA,     "MX", "MEXICO",     YES, YES, YES    },
    {CTRY_MONACO,      ETSI4_WORLD,   "MC", "MONACO",     YES, YES, YES    },
    {CTRY_MOROCCO,     NULL1_WORLD,   "MA", "MOROCCO",    YES, NO, YES     },
    {CTRY_NETHERLANDS, ETSI1_WORLD,   "NL", "NETHERLANDS",YES, NO, YES     },
    {CTRY_NEW_ZEALAND, FCC2_ETSIC,    "NZ", "NEW ZEALAND",YES, NO, YES     },
    {CTRY_NORWAY,      ETSI1_WORLD,   "NO", "NORWAY",     YES, NO, YES     },
    {CTRY_OMAN,        NULL1_WORLD,   "OM", "OMAN",       YES, NO, YES     },
    {CTRY_PAKISTAN,    NULL1_WORLD,   "PK", "PAKISTAN",   YES, NO, YES     },
    {CTRY_PANAMA,      FCC1_FCCA,     "PA", "PANAMA",     YES, YES, YES    },
    {CTRY_PERU,        NULL1_WORLD,   "PE", "PERU",       YES, NO, YES     },
    {CTRY_PHILIPPINES, FCC1_WORLD,    "PH", "PHILIPPINES",YES, YES, YES    },
    {CTRY_POLAND,      ETSI1_WORLD,   "PL", "POLAND",     YES, NO, YES     },
    {CTRY_PORTUGAL,    ETSI1_WORLD,   "PT", "PORTUGAL",   YES, NO, YES     },
    {CTRY_PUERTO_RICO, FCC1_FCCA,     "PR", "PUERTO RICO",YES, YES, YES    },
    {CTRY_QATAR,       NULL1_WORLD,   "QA", "QATAR",      YES, NO, YES     },
    {CTRY_ROMANIA,     NULL1_WORLD,   "RO", "ROMANIA",    YES, NO, YES     },
    {CTRY_RUSSIA,      NULL1_WORLD,   "RU", "RUSSIA",     YES, NO, YES     },
    {CTRY_SAUDI_ARABIA,NULL1_WORLD,   "SA", "SAUDI ARABIA", YES, NO, YES   },
    {CTRY_SINGAPORE,   APL4_WORLD,    "SG", "SINGAPORE",  YES, YES, YES    },
    {CTRY_SLOVAKIA,    ETSI3_WORLD,   "SK", "SLOVAK REPUBLIC", YES, NO, YES },
    {CTRY_SLOVENIA,    ETSI1_WORLD,   "SI", "SLOVENIA",   YES, NO, YES     },
    {CTRY_SOUTH_AFRICA,ETSI1_WORLD,   "ZA", "SOUTH AFRICA", YES, YES, YES  },
    {CTRY_SPAIN,       ETSI2_WORLD,   "ES", "SPAIN",      YES, NO, YES     },
    {CTRY_SWEDEN,      ETSI1_WORLD,   "SE", "SWEDEN",     YES, NO, YES     },
    {CTRY_SWITZERLAND, ETSI2_WORLD,   "CH", "SWITZERLAND",YES, NO, YES     },
    {CTRY_SYRIA,       NULL1_WORLD,   "SY", "SYRIA",      YES, NO, YES    },
    {CTRY_TAIWAN,      APL3_WORLD,    "TW", "TAIWAN",     YES, YES, YES    },
    {CTRY_THAILAND,    APL2_WORLD,    "TH", "THAILAND",   YES, YES, YES    },
    {CTRY_TRINIDAD_Y_TOBAGO,ETSI4_WORLD,"TT", "TRINIDAD & TOBAGO", YES, NO, YES    },
    {CTRY_TUNISIA,     ETSI3_WORLD,   "TN", "TUNISIA",    YES, NO, YES     },
    {CTRY_TURKEY,      ETSI3_WORLD,   "TR", "TURKEY",     YES, NO, YES     },
    {CTRY_UKRAINE,     NULL1_WORLD,   "UA", "UKRAINE",    YES, NO, YES     },
    {CTRY_UAE,         NULL1_WORLD,   "AE", "UNITED ARAB EMIRATES", YES, NO, YES },
    {CTRY_UNITED_KINGDOM, ETSI1_WORLD, "GB", "UNITED KINGDOM", YES, NO, YES },
    {CTRY_UNITED_STATES, FCC1_FCCA,   "US", "UNITED STATES", YES, YES, YES },
    {CTRY_URUGUAY,     APL2_WORLD,    "UY", "URUGUAY",    YES, NO, YES     },
    {CTRY_UZBEKISTAN,  NULL1_WORLD,   "UZ", "UZBEKISTAN", YES, NO, YES     },    
    {CTRY_VENEZUELA,   APL2_ETSIC,    "VE", "VENEZUELA",   NO, NO, NO     },
    {CTRY_VIET_NAM,    NULL1_WORLD,   "VN", "VIET NAM",   YES, NO, YES     },
    {CTRY_YEMEN,       NULL1_WORLD,   "YE", "YEMEN",      YES, NO, YES    },
    {CTRY_ZIMBABWE,    NULL1_WORLD,   "ZW", "ZIMBABWE",   YES, NO, YES     }    
};

typedef struct EnumRdToWirelessModeRd {
    A_UINT16  regDmnEnum;
    A_UINT16  regDmnWirelessMode5;
    A_UINT16  regDmnWirelessMode2;
} ENUM_RD_TO_WIRELESS_MODE_RD;

static const ENUM_RD_TO_WIRELESS_MODE_RD allEnumRds[] = {
    {NO_ENUMRD,     DEBUG_REG_DMN, DEBUG_REG_DMN},
    {NULL1_WORLD,   NULL1,   WORLD   },
    {NULL1_ETSIB,   NULL1,   ETSIB   },
    {NULL1_ETSIC,   NULL1,   ETSIC   },

    {FCC2_FCCA,     FCC2,    FCCA    },
    {FCC2_WORLD,    FCC2,    WORLD   },
    {FCC2_ETSIC,    FCC2,    ETSIC   },

    {ETSI1_WORLD,   ETSI1,   WORLD   },
    {ETSI2_WORLD,   ETSI2,   WORLD   },
    {ETSI3_WORLD,   ETSI3,   WORLD   },
    {ETSI4_WORLD,   ETSI4,   WORLD   },
    {ETSI5_WORLD,   ETSI5,   WORLD   },
    {ETSI6_WORLD,   ETSI6,   WORLD   },

    {ETSI3_ETSIA,   ETSI3,   WORLD   },

    {FCC1_WORLD,    FCC1,    WORLD   },
    {FCC1_FCCA,     FCC1,    FCCA    },

    {APL1_WORLD,    APL1,    WORLD   },
    {APL2_WORLD,    APL2,    WORLD   },
    {APL3_WORLD,    APL3,    WORLD   },
    {APL4_WORLD,    APL4,    WORLD   },
    {APL5_WORLD,    APL5,    WORLD   },    

    {APL1_ETSIC,    APL1,    ETSIC   },
    {APL2_ETSIC,    APL2,    ETSIC   },
    {APL2_APLD,     APL2,    APLD    },

    {MKK1_MKKA,     MKK1,    MKKA    },
    {MKK1_MKKB,     MKK1,    MKKA    },
    {MKK2_MKKA,     MKK2,    MKKA    },
    {MKK1_FCCA,     MKK1,    FCCA    },    
    {MKK1_MKKA1,    MKK1,    MKKA    },
    {MKK1_MKKA2,    MKK1,    MKKA    },

    /*
     * The area below is reserved for super domains
     */
    {WOR0_WORLD,    WOR0_WORLD,   WOR0_WORLD    },
    {WOR1_WORLD,    WOR1_WORLD,   WOR1_WORLD    },
    {WOR2_WORLD,    WOR2_WORLD,   WOR2_WORLD    },
    {WOR3_WORLD,    WOR3_WORLD,   WOR3_WORLD    },
    {WOR4_WORLD,    WOR4_WORLD,   WOR4_WORLD    },
    {WOR5_ETSIC,    WOR5_ETSIC,   WOR5_ETSIC    },
    {WOR01_WORLD,   WOR01_WORLD,  WOR01_WORLD   },
    {WOR02_WORLD,   WOR02_WORLD,  WOR02_WORLD   },
    {EU1_WORLD,     EU1_WORLD,    EU1_WORLD     }
};

#define TURBO_CHANNEL_SEPARATION    (40)
#define OFDM5_CHANNEL_SEPARATION    (20)
#define LOW2GHZ_CHANNEL_SEPARATION  (5)
#define HIGH2GHZ_CHANNEL_SEPARATION (20)
#define IS24DEBUG(ch)               (((ch) >= 2512) && ((ch) <= 2732))

#define MAX_CHANNEL_GROUPS 6

/* 5 GHz table settings */
typedef struct RegDmnWmChannelGroup {
    A_UINT16    lowChannel;   // Low Channel center in MHz
    A_UINT16    highChannel;  // High Channel center in MHz
    A_UINT8     powerDfs;     // Maximum power (dBm) for this channel range when using DFS
    A_UINT8     antennaMax;   // Maximum allowed antenna gain
} REG_DMN_WM_CHANNEL_GROUP;

typedef struct RegDmnWmFreqTable {
    A_UINT16                 regDmnWirelessMode;
    A_UINT16                 entries;
    A_UINT8                  conformanceTestLimit;
    REG_DMN_WM_CHANNEL_GROUP chanGroup[MAX_CHANNEL_GROUPS];
} REG_DMN_WM_FREQ_TABLE;

/*
 * Regulatory Domain Tables
 *
 * (Information is based on "Country Table 8-29-02.xls")
 * Tables have entries ordered with AP/Adhoc planting preference in mind.
 *  -Indoor channels are ordered before outdoor
 *  -Higher power level indoor channels are first
 *
 */
static REG_DMN_WM_FREQ_TABLE regDmnWm5[] = {
    {DEBUG_REG_DMN, 2, FCC, {
    {5120, 5700,  5,  6},
    {5745, 5825,  5,  6},
    {   0,    0,  0,  0},
    {   0,    0,  0,  0}}},

    {APL1, 1, FCC, {
    {5745, 5825, 30, 0},
    {   0,    0,  0, 0},
    {   0,    0,  0, 0},
    {   0,    0,  0, 0}}},

    {APL2, 1, FCC, {
    {5745, 5805, 23, 0},
    {   0,    0,  0, 0},
    {   0,    0,  0, 0},
    {   0,    0,  0, 0}}},

    {APL3, 2, FCC, {
    {5280, 5320, 17, 6},
    {5745, 5805, 30, 6},
    {   0,    0,  0, 0},
    {   0,    0,  0, 0}}},

    {APL4, 2, FCC, {
    {5180, 5240, 20, 0},
    {5745, 5825, 20, 0},
    {   0,    0,  0, 0},
    {   0,    0,  0, 0}}},

    {APL5, 2, FCC, {
    {5745, 5825, 17, 0},
    {   0,    0,  0, 0},
    {   0,    0,  0, 0},
    {   0,    0,  0, 0}}},

    /*
     * The ETSI1 domain requires TPC
     * As the TPC specification are unfinished, 3dBi are
     * removed from each of the ETSI1 power selections
     */
    {ETSI1, 2, ETSI, {
    {5180, 5320, 20, 0},  // Should be 23 with TPC
    {5500, 5700, 27, 0},  // Should be 30 with TPC
    {   0,    0,  0, 0},
    {   0,    0,  0, 0}}},

    {ETSI2, 1, ETSI, {
    {5180, 5240, 18, 0},
    {   0,    0,  0, 0},
    {   0,    0,  0, 0},
    {   0,    0,  0, 0}}},

    {ETSI3, 1, ETSI, {
    {5180, 5320, 20, 0},
    {   0,    0,  0, 0},
    {   0,    0,  0, 0},
    {   0,    0,  0, 0}}},

    {ETSI4, 1, ETSI, {
    {5180, 5320, 18, 0},
    {   0,    0,  0, 0},
    {   0,    0,  0, 0},
    {   0,    0,  0, 0}}},

    {ETSI5, 1, ETSI, {
    {5180, 5240, 15, 0},
    {   0,    0,  0, 0},
    {   0,    0,  0, 0},
    {   0,    0,  0, 0}}},

    {ETSI6, 2, ETSI, {
    {5180, 5280, 23, 0},
    {5500, 5700, 30, 0},
    {   0,    0,  0, 0},
    {   0,    0,  0, 0}}},

    /*
     * Values artificially decreased to meet FCC testing
     * procedures
     */
    {FCC1, 3, FCC, {
    {5260, 5320, 23, 6},
    {5180, 5240, 17, 6},
    {5745, 5825, 30, 6},
    {   0,    0,  0, 0}}},

    {FCC2, 2, FCC, {
    {5180, 5320, 23, 6},
    {5745, 5825, 30, 6},
    {   0,    0,  0, 0},
    {   0,    0,  0, 0}}},

    {MKK1, 1, MKK, {
    {5170, 5230, 23, 0},
    {   0,    0,  0, 0},
    {   0,    0,  0, 0},
    {   0,    0,  0, 0}}},

    {MKK2, 3, MKK, {
    {4920, 4980, 23, 0},
    {5040, 5080, 23, 0},
    {5170, 5230, 23, 0},
    {   0,    0,  0, 0}}}
};

static const REG_DMN_WM_FREQ_TABLE regDmnWm5Turbo[] = {
    {DEBUG_REG_DMN, 2, FCC | CTL_TURBO, {
    {5130, 5670,  5, 6},
    {5150, 5650,  5, 6},
    {   0,    0,  0, 0},
    {   0,    0,  0, 0}}},

    {FCC1, 3, FCC | CTL_TURBO, {
    {5250, 5290, 23, 6},
    {5210, 5210, 17, 6},
    {5760, 5800, 30, 6},
    {   0,    0,  0, 0}}}
};

/* Special scan tables for turbo prime interference checks */ 
#define NUM_PRIME_A_CHANNELS 5  /* number additional to static turbo channels */
#define NUM_PRIME_A_DOMAINS (sizeof(regDmnWm5TurboPrime)/sizeof(REG_DMN_WM_FREQ_TABLE))
static const REG_DMN_WM_FREQ_TABLE regDmnWm5TurboPrime[] = {
    {FCC1, 3, FCC | CTL_TURBO, {
    {5240, 5280, 23, 6},
    {5200, 5200, 17, 6},
    {5765, 5805, 30, 6},
    {   0,    0,  0, 0}}}
};

/* This number is currently 0 since 'g' base ch same as turbo. */
#define NUM_PRIME_G_CHANNELS 0  /* number additional to static turbo channels */
#define NUM_PRIME_G_DOMAINS (sizeof(regDmnWm2TurboPrime)/sizeof(REG_DMN_WM_FREQ_TABLE))
static const REG_DMN_WM_FREQ_TABLE regDmnWm2TurboPrime[] = {
    {FCCA, 1, FCC | CTL_11B, {
    {   0,    0,  0, 0},
    {   0,    0,  0, 0},
    {   0,    0,  0, 0},
    {   0,    0,  0, 0}}}
};

static const REG_DMN_WM_FREQ_TABLE regDmnWm2Turbo[] = {
    {DEBUG_REG_DMN, 3, NO_CTL, {
    {2312, 2372,  5, 6},        /* Channel -19 to -7 */
    {2437, 2437,  5, 6},        /* Channel 6 only */
    {2512, 2732,  5, 6},        /* Channel 15 and higher  */
    {0,       0,  0, 0}}},      /* */

    {FCCA, 1, FCC | CTL_11B, {
    {2437, 2437, 20, 6},
    {   0,    0,  0, 0},
    {   0,    0,  0, 0},
    {   0,    0,  0, 0}}},

    {ETSIA, 1, ETSI | CTL_11B, {
    {2437, 2437, 20, 6},
    {   0,    0,  0, 0},
    {   0,    0,  0, 0},
    {   0,    0,  0, 0}}},

    {ETSIB, 1, ETSI | CTL_11B, {
    {2437, 2437, 20, 6},
    {   0,    0,  0, 0},
    {   0,    0,  0, 0},
    {   0,    0,  0, 0}}},

    {ETSIC, 1, ETSI | CTL_11B, {
    {2437, 2437, 20, 6}, 
    {   0,    0,  0, 0},
    {   0,    0,  0, 0},
    {   0,    0,  0, 0}}},

    {MKKA, 1, MKK | CTL_11B, {
    {2437, 2437, 20, 6}, 
    {   0,    0,  0, 0},
    {   0,    0,  0, 0},
    {   0,    0,  0, 0}}},

    {WORLD, 1, ETSI | CTL_11B, {
    {2437, 2437, 20, 6}, 
    {   0,    0,  0, 0},
    {   0,    0,  0, 0},
    {   0,    0,  0, 0}}}
    
};

static const REG_DMN_WM_FREQ_TABLE regDmnWm2[] = {
    {DEBUG_REG_DMN, 4, NO_CTL, {
    {2312, 2372,  5, 6},      /* Channel -19 to -7 */
    {2412, 2472,  5, 6},
    {2484, 2484,  5, 6},
    {2512, 2732,  5, 6}}},

    {APLD, 2, NO_CTL, {
    {2412, 2472, 20, 0},        /* Channel 1 - 13 */
    {2312, 2372, 20, 0},        /* Negative Channels */
    {   0,    0,  0, 0},
    {   0,    0,  0, 0}}},

    {ETSIA, 1, NO_CTL, {
    {2457, 2472, 20, 0},        /* Channel 10 - 13 */
    {   0,    0,  0, 0},
    {   0,    0,  0, 0},
    {   0,    0,  0, 0}}},

    {ETSIB, 1, ETSI | CTL_11B, {
    {2432, 2442, 20, 0},        /* Channel 5 - 7 */
    {   0,    0,  0, 0},
    {   0,    0,  0, 0},
    {   0,    0,  0, 0}}},

    {ETSIC, 1, ETSI | CTL_11B, {
    {2412, 2472, 30, 0},        /* Channel 1 - 13 */
    {   0,    0,  0, 0},
    {   0,    0,  0, 0},
    {   0,    0,  0, 0}}},

    {FCCA, 1, FCC | CTL_11B, {
    {2412, 2462, 27, 6},        /* Channel 1 - 11 */
    {   0,    0,  0, 0},
    {   0,    0,  0, 0},
    {   0,    0,  0, 0}}},

    {MKKA, 2, MKK | CTL_11B, {
    {2412, 2472, 20, 0},        /* Channel 1 - 13 */
    {2484, 2484, 20, 0},        /* Channel 14 */
    {   0,    0,  0, 0},
    {   0,    0,  0, 0}}},

    {WORLD, 1, ETSI | CTL_11B, {
    {2412, 2472, 20, 0},        /* Channel 1 - 13 */
    {   0,    0,  0, 0},
    {   0,    0,  0, 0},
    {   0,    0,  0, 0}}},
};

static const REG_DMN_WM_FREQ_TABLE regDmnNone[] = {
    {0, 0, NO_CTL, {
    {   0,    0,  0, 0},
    {   0,    0,  0, 0},
    {   0,    0,  0, 0},
    {   0,    0,  0, 0}}},
};

/*
 * 802.11a Common Mode Power Levels
 */
typedef struct _Common_Mode_Power {
    A_UINT16    lchan;  /* Low channel */
    A_UINT16    hchan;  /* Highest channel */
    A_UINT8     pwrLvl; /* power level in dBm */
} CM_PWR;

#define CM_PWR_TBL_LEN (sizeof(cmPwrTbl)/sizeof(CM_PWR))

static CM_PWR cmPwrTbl[] =
{
    { 4900, 5000, 17 },
    { 5000, 5100, 17 },
    { 5150, 5250, 15 }, /* ETSI & MKK */
    { 5250, 5350, 18 }, /* ETSI */
    { 5470, 5725, 20 }, /* ETSI */
    { 5725, 5825, 20 }, /* Singapore */
    { 5825, 5850, 23 }  /* Korea */
};
    

typedef struct _adhoc_disallow_RD {
    A_UINT16    rd;     /* 2.4 or 5GHz regulatory domain */
    WLAN_CFLAGS cflags; /* 2.4 or 5GHz channel flag */
    A_UINT16    mode;   /* 2.4 or 5GHz wireless mode flag(s) */
} ADHOC_DISALLOW_RD;

#define ADHOC_DISALLOW_RD_TBL_LEN (sizeof(adHocDisallowRdTbl)/sizeof(ADHOC_DISALLOW_RD))

/*
 * The following table defines the wireless mode that DOES
 * NOT supported adHoc network in its regulatory domain
 *
 * If adhoc network is not supported in 802.11a, then in
 * the same token, it won't supported the turbo mode also.
 */
static ADHOC_DISALLOW_RD adHocDisallowRdTbl[] =
{
    { WOR1_WORLD, CHANNEL_A, MODE_SELECT_11A | MODE_SELECT_TURBO },
    { WOR2_WORLD, CHANNEL_T, MODE_SELECT_11A                     },
    { WOR4_WORLD, CHANNEL_A, MODE_SELECT_11A | MODE_SELECT_TURBO },
    { WOR5_ETSIC, CHANNEL_A, MODE_SELECT_11A | MODE_SELECT_TURBO },    
    { ETSI1_WORLD,CHANNEL_A, MODE_SELECT_11A | MODE_SELECT_TURBO },
    { ETSI2_WORLD,CHANNEL_A, MODE_SELECT_11A | MODE_SELECT_TURBO },
    { ETSI3_WORLD,CHANNEL_A, MODE_SELECT_11A | MODE_SELECT_TURBO },
    { ETSI3_ETSIA,CHANNEL_A, MODE_SELECT_11A | MODE_SELECT_TURBO },
    { ETSI4_WORLD,CHANNEL_A, MODE_SELECT_11A | MODE_SELECT_TURBO },
    { ETSI5_WORLD,CHANNEL_A, MODE_SELECT_11A | MODE_SELECT_TURBO },
    { ETSI6_WORLD,CHANNEL_A, MODE_SELECT_11A | MODE_SELECT_TURBO },
    { ETSI1,      CHANNEL_A, MODE_SELECT_11A | MODE_SELECT_TURBO },
    { ETSI2,      CHANNEL_A, MODE_SELECT_11A | MODE_SELECT_TURBO },
    { ETSI3,      CHANNEL_A, MODE_SELECT_11A | MODE_SELECT_TURBO },
    { ETSI4,      CHANNEL_A, MODE_SELECT_11A | MODE_SELECT_TURBO },
    { ETSI5,      CHANNEL_A, MODE_SELECT_11A | MODE_SELECT_TURBO },
    { ETSI6,      CHANNEL_A, MODE_SELECT_11A | MODE_SELECT_TURBO },
    { ETSIA,      CHANNEL_A, MODE_SELECT_11A | MODE_SELECT_TURBO },
    { ETSIB,      CHANNEL_A, MODE_SELECT_11A | MODE_SELECT_TURBO },
    { ETSIC,      CHANNEL_A, MODE_SELECT_11A | MODE_SELECT_TURBO },
    { MKK1_MKKA,  CHANNEL_T,                   MODE_SELECT_TURBO },
    { MKK2_MKKA,  CHANNEL_A, MODE_SELECT_11A | MODE_SELECT_TURBO },
    { MKK1_MKKB,  CHANNEL_A, MODE_SELECT_11A | MODE_SELECT_TURBO },
    { MKK1_FCCA,  CHANNEL_T,                   MODE_SELECT_TURBO },
    { MKK1_MKKA1, CHANNEL_T,                   MODE_SELECT_TURBO },
    { MKK1_MKKA2, CHANNEL_T,                   MODE_SELECT_TURBO },
    { MKK1,       CHANNEL_T,                   MODE_SELECT_TURBO },
    { MKK2,       CHANNEL_T,                   MODE_SELECT_TURBO },    
    { MKKA,       CHANNEL_T,                   MODE_SELECT_TURBO }
};

typedef struct FCCPassiveScanChannelList {
    A_UINT16            mode;
    A_UINT16            numClist;
    WLAN_CHANNEL        *pClTbl[4];
} FCC_PSCAN_CLIST;

typedef struct _passive_scan_RD {
    A_UINT16    rd;             /* 2.4/5GHz regulatory domain */
    A_UINT16    eepromRD;       /* EEPROM regulatory domain */
    WLAN_CFLAGS cflags;
    FCC_PSCAN_CLIST *pList;
} PASSIVE_SCAN_RD;

/* FCC passive scan channel list: 36, 40, 44, 48, 52, 56, 60, 64 */
static WLAN_CHANNEL fccPassive1[] =
{ 5180, 5200, 5220, 5240, 5260, 5280, 5300, 5320, 0 };
/* FCC passive scan channel list: 100,104,108,112,116,120,124,128,132,136,140 */static WLAN_CHANNEL fccPassive2[] =
{ 5500, 5520, 5540, 5560, 5580, 5600, 5620, 5640, 5660, 5680, 5700, 0 };
/* FCC passive scan list for turbo channels: 48,50,56,58,108,116,124,132 */
static WLAN_CHANNEL fccPassiveTurbo[] =
{ 5240, 5250, 5280, 5290, 5540, 5580, 5620, 5660, 0 };

static FCC_PSCAN_CLIST fccPassiveScanTbl =
{ CHANNEL_A, 2,
    { (WLAN_CHANNEL *)&fccPassive1, (WLAN_CHANNEL *)&fccPassive2, (void *)0, (void *)0 } };

static FCC_PSCAN_CLIST fccPassiveScanTurboTbl =
{ CHANNEL_T, 1,
    { (WLAN_CHANNEL *)&fccPassiveTurbo, (void *)0, (void *)0, (void *)0 } };

/* MKK Passive scan list: All channels except 36, 40, 44, 48, no turbo */
static WLAN_CHANNEL mkkPassive[] =
{ 5170, 5190, 5210, 5230, 5260, 5280, 5300, 5320, 5500, 5520, 5540, 5560, 5580, 5600, 5620,
  5640, 5660, 5680, 5700, 0 };
static FCC_PSCAN_CLIST mkkPassiveScanTbl =
{ CHANNEL_A, 1,
    { (WLAN_CHANNEL *)&mkkPassive, (void *)0, (void *)0, (void *)0 } };

#define PS_RD_TBL_LEN (sizeof(passiveScanRdTbl)/sizeof(PASSIVE_SCAN_RD))

/*
 * The following table defines the wireless mode that requires
 * passive scanning in its regulatory domain. 
 *
 * For ETSI and FCC3 RDs, the eepromRD is NOT used to determine if a channel scan-type is 
 * passive.
 */
static PASSIVE_SCAN_RD passiveScanRdTbl[] =
{
    { FCC3,  0,         CHANNEL_A, &fccPassiveScanTbl },
    { FCC3,  0,         CHANNEL_T, &fccPassiveScanTurboTbl },
    { ETSI1, 0,         CHANNEL_A, NULL },
    { ETSI2, 0,         CHANNEL_A, NULL },
    { ETSI3, 0,         CHANNEL_A, NULL },
    { ETSI4, 0,         CHANNEL_A, NULL },
    { ETSI5, 0,         CHANNEL_A, NULL },
    { ETSI6, 0,         CHANNEL_A, NULL },
    { ETSIA, 0,         CHANNEL_A, NULL },
    { ETSIB, 0,         CHANNEL_A, NULL },
    { ETSIC, 0,         CHANNEL_A, NULL },
    { EU1_WORLD, EU1_WORLD, CHANNEL_A, NULL },
    { MKK2,  0 , CHANNEL_A, &mkkPassiveScanTbl },
    { MKKA,  MKK2_MKKA, CHANNEL_B, NULL },
    { MKKA,  MKK2_MKKA, CHANNEL_G, NULL },
    { MKK1,  0 , CHANNEL_A, &mkkPassiveScanTbl },
    { MKKA,  MKK1_MKKB, CHANNEL_B, NULL },
    { MKKA,  MKK1_MKKB, CHANNEL_G, NULL }
};

typedef struct _dfs_RD {
    A_UINT16    rd;
} DFS_RD;

#define DFS_RD_TBL_LEN (sizeof(dfsRdTbl)/sizeof(DFS_RD))

/*
 * The following table defines the wireless mode that enables
 * DFS in its regulatory domain
 */
static DFS_RD dfsRdTbl[] = {
    { ETSI1 },
    { ETSI2 },
    { ETSI3 },
    { ETSI4 },
    { ETSI5 },
    { ETSI6 },
    { ETSIA },
    { ETSIB },
    { ETSIC }
};


typedef struct x11b_only_RD {
    A_UINT16    rd;
} X11B_ONLY_RD;

#define X11B_ONLY_RD_TBL_LEN (sizeof(X11bOnlyRdTbl)/sizeof(X11B_ONLY_RD))

/*
 * The following table defines the 802.11b only regulatory
 * domain
 */
static X11B_ONLY_RD X11bOnlyRdTbl[] =
{
    { NULL1_WORLD },
    { NULL1_ETSIB },
    { NULL1_ETSIC },
    { NULL1 }
};

typedef struct _one_country_RD {
    A_UINT16    rd;
    CTRY_CODE   country;
} SINGLE_COUNTRY_RD;

#define SC_RD_TBL_LEN (sizeof(singleCountryRdTbl)/sizeof(SINGLE_COUNTRY_RD))
/*
 * The following table defines a single country
 * regulatory domain
 */
static SINGLE_COUNTRY_RD singleCountryRdTbl[] = {
    { MKK1_MKKA, CTRY_JAPAN  },
    { MKK1_MKKB, CTRY_JAPAN1 },
    { MKK1_FCCA, CTRY_JAPAN2 },    
    { MKK2_MKKA, CTRY_JAPAN3 },
    { MKK1_MKKA1, CTRY_JAPAN4 },
    { MKK1_MKKA2, CTRY_JAPAN5 }
};

#define NFC_RD_TBL_LEN (sizeof(nfcRdTbl)/sizeof(REG_DOMAIN))
/*
 * The following table defines regulatory domain that requires
 * noise floor calibration
 */
static REG_DOMAIN nfcRdTbl[] = {
    MKK1_MKKA,
    MKK2_MKKA,
    MKK1_MKKB,
    MKK1_FCCA,
    MKK1_MKKA1,
    MKK1_MKKA2    
};

/*
 * WorldWide Roaming Channels
 * The channels are organzied in order of the passive scan.
 */

typedef WLAN_CHANNEL ROAMING_CHANNEL;

typedef struct WorldWideRoamingChannelList {
    A_UINT16            mode;
    ROAMING_CHANNEL     *pList;
} WWR_CLIST;


typedef WLAN_CHANNEL    ADHOC_CLIST;

typedef struct _super_domain {
    A_UINT16            domain;
    START_ADHOC_OPTION  adHocMode;
    ADHOC_CLIST         *adHocClist;  /* Ad Hoc default channel list */
    A_UINT16            rclTblSize;
    WWR_CLIST           *RclTbl[10];
} SUPER_DOMAIN;

static ROAMING_CHANNEL rcl1[] =         /* 8 FCC channel: 52, 56, 60, 64, 36, 40, 44, 48 */
{ 5260, 5280, 5300, 5320, 5180, 5200, 5220, 5240, 0 };
static ROAMING_CHANNEL rcl2[] =         /* 4 MKK channels: 34, 38, 42, 46 */
{ 5170, 5190, 5210, 5230, 0 };
static ROAMING_CHANNEL rcl2Alt[] =     /* 4 new MKK channels: 36, 40, 44, 48 */
{ 5180, 5200, 5220, 5240, 0 };
static ROAMING_CHANNEL rcl3[] =         /* 2.4Ghz ch: 1,6,11,7,13 */
{ 2412, 2437, 2462, 2442, 2472, 0 };
static ROAMING_CHANNEL rcl3x[] =        /* 2.4Ghz ch: 1,6,11,7 */
{ 2412, 2437, 2462, 2442, 0 };
static ROAMING_CHANNEL rcl3a[] =        /* 2.4Ghz ch: 1,6,11,7 */
{ 2412, 2437, 2462, 2442, 0 };
static ROAMING_CHANNEL rcl3ax[] =       /* 2.4Ghz ch: 1,6,11,7 */
{ 2412, 2437, 2462, 2442, 0 };
static ROAMING_CHANNEL rcl4[] =         /* 5 FCC channel: 149, 153, 161, 165 */
{ 5745, 5765, 5785, 5805, 5825, 0 };
static ROAMING_CHANNEL rcl5[] =         /* 3 turbo channels */
{ 5210, 5250, 5290, 0 };
static ROAMING_CHANNEL rcl6[] =         /* 2 turbo channels */
{ 5760, 5800, 0 };
static ROAMING_CHANNEL rcl7[] =         /* 11 ETSI channel: 100,104,108,112,116,120,124,128,132,136,140 */
{ 5500, 5520, 5540, 5560, 5580, 5600, 5620, 5640, 5660, 5680, 5700, 0 };
static ROAMING_CHANNEL rcl8[] =         /* 2.4Ghz ch: 2,3,4,5,8,9,10,12 */
{ 2417, 2422, 2427, 2432, 2447, 2452, 2457, 2467, 0 };
static ROAMING_CHANNEL rcl8x[] =        /* 2.4Ghz ch: 2,3,4,5,8,9,10 */
{ 2417, 2422, 2427, 2432, 2447, 2452, 2457, 0 };
static ROAMING_CHANNEL rcl8a[] =        /* 2.4Ghz ch: 2,3,4,5,8,9,10 */
{ 2417, 2422, 2427, 2432, 2447, 2452, 2457, 0 };
static ROAMING_CHANNEL rcl8ax[] =       /* 2.4Ghz ch: 2,3,4,5,8,9,10 */
{ 2417, 2422, 2427, 2432, 2447, 2452, 2457, 0 };
static ROAMING_CHANNEL rcl9[] =         /* 2.4Ghz ch: 14 */
{ 2484, 0 };
static ROAMING_CHANNEL rcl10[] =        /* Added Korean channels 2312-2372 */
{ 2312, 2317, 2322, 2327, 2332, 2337, 2342, 2347, 2352, 2357, 2362, 2367, 2372, 0 };
static ROAMING_CHANNEL rcl11[] =        /* Added Japan channels in 4.9/5.0 spectrum */
{ 5040, 5060, 5080, 4920, 4940, 4960, 4980, 0 };
static ROAMING_CHANNEL rcl12[] =        /* 2.4Ghz Turbo channel 6 */
{ 2437, 0 };

static ROAMING_CHANNEL ecmChannelTbl[] =  /* 2.4Ghz channel 12 - 14 */
{ 2472, 2467, 2484, 0 };

typedef struct ecmChannelScan {
    WLAN_CHANNEL        chan;
    A_BOOL              scanMode;
} ECM_CHAN_SCAN;

#define A_SCAN                  FALSE
#define P_SCAN                  TRUE
#define ECM_CHAN_TBL_LEN        3
#define ECM_SKU_TBL_LEN         (sizeof(ecmSkuTbl)/sizeof(ECM_SKU_TBL))

typedef struct ECM_SKU {
    A_UINT16            sku;
    ECM_CHAN_SCAN       tbl[ECM_CHAN_TBL_LEN];
} ECM_SKU_TBL;

static ECM_SKU_TBL ecmSkuTbl[6] = {
    { WOR0_WORLD,  { {2472, P_SCAN}, {2467, P_SCAN}, {2484, P_SCAN} } },
    { WOR02_WORLD, { {2472, P_SCAN}, {2467, P_SCAN}, {2484, P_SCAN} } },
    { WOR1_WORLD,  { {2472, P_SCAN}, {2467, P_SCAN}, {2484, P_SCAN} } },
    { WOR5_ETSIC,  { {2472, P_SCAN}, {2467, P_SCAN}, {2484, P_SCAN} } },
    { MKK1_MKKA1, { {2472, A_SCAN}, {2467, A_SCAN}, {2484, P_SCAN} } },
    { MKK1_MKKA2, { {2472, P_SCAN}, {2467, P_SCAN}, {2484, P_SCAN} } }
};
    
static ADHOC_CLIST adhocWorldClist[] =  /* Channel: 36, 40, 44, 48 */
{ 5180, 5200, 5220, 5240, 0 };
static ADHOC_CLIST adhocMkkClist[] =    /* Channel: 34, 38, 42, 46 */
{ 5170, 5190, 5210, 5230, 0 };
static ADHOC_CLIST adhocMkkClistAlt[] =    /* Channel: 36, 40, 44, 48 */
{ 5180, 5200, 5220, 5240, 0 };
static ADHOC_CLIST adhoc11bClist[] =    /* Channel: 10, 11 */
{ 2457, 2462, 0 };

static WWR_CLIST wwrcl1     = { MODE_SELECT_11A,   rcl1  };
static WWR_CLIST wwrcl2     = { MODE_SELECT_11A,   rcl2  };
static WWR_CLIST wwrcl3     = { MODE_SELECT_11B,   rcl3  };
static WWR_CLIST wwrcl3x    = { MODE_SELECT_11B,   rcl3x  };
static WWR_CLIST wwrcl3a    = { MODE_SELECT_11B,   rcl3a };
static WWR_CLIST wwrcl3ax   = { MODE_SELECT_11B,   rcl3ax };
static WWR_CLIST wwrcl4     = { MODE_SELECT_11A,   rcl4  };
static WWR_CLIST wwrcl5     = { MODE_SELECT_TURBO, rcl5  };
static WWR_CLIST wwrcl6     = { MODE_SELECT_TURBO, rcl6  };
static WWR_CLIST wwrcl7     = { MODE_SELECT_11A,   rcl7  };
static WWR_CLIST wwrcl8     = { MODE_SELECT_11B,   rcl8  };
static WWR_CLIST wwrcl8x    = { MODE_SELECT_11B,   rcl8x  };
static WWR_CLIST wwrcl8a    = { MODE_SELECT_11B,   rcl8a };
static WWR_CLIST wwrcl8ax   = { MODE_SELECT_11B,   rcl8ax };
static WWR_CLIST wwrcl9     = { MODE_SELECT_11B,   rcl9  };
static WWR_CLIST wwrcl10    = { MODE_SELECT_11B,   rcl10 };
static WWR_CLIST wwrcl11    = { MODE_SELECT_11A,   rcl11 };
static WWR_CLIST wwrcl12    = { MODE_SELECT_108G,  rcl12 };

/*
 * Scanning order of Extended Channel Mode channels,
 * Extended Channel Mode channels are channels 12 - 14.
 */
static const WWR_CLIST * socTbl [] = {
    &wwrcl3,  &wwrcl1,  &wwrcl2, &wwrcl8,  &wwrcl9, &wwrcl4, &wwrcl5, &wwrcl6,
    &wwrcl7, &wwrcl10, &wwrcl11, &wwrcl12, NULL
};

/*
 *  Extended Channel Mode World SKU Table
 */
static const SUPER_DOMAIN superDomainTbl [] = {
    { WOR0_WORLD,  START_ADHOC_PER_11D, NULL,            10,
      { &wwrcl1,   &wwrcl2, &wwrcl3,  &wwrcl4, &wwrcl5, &wwrcl6,  &wwrcl7, &wwrcl8,  &wwrcl9, &wwrcl12 } },
    { WOR01_WORLD, START_ADHOC_PER_11D, NULL,           9,
      { &wwrcl1,   &wwrcl2, &wwrcl3x, &wwrcl4, &wwrcl5, &wwrcl6,  &wwrcl7, &wwrcl8x, &wwrcl12 } },
    { WOR02_WORLD, START_ADHOC_PER_11D, NULL,           9,
      { &wwrcl1,   &wwrcl2, &wwrcl3,  &wwrcl4, &wwrcl5, &wwrcl6,  &wwrcl7, &wwrcl8,  &wwrcl12 } },
    { EU1_WORLD,   START_ADHOC_PER_11D, NULL,           9,
      { &wwrcl1,   &wwrcl2, &wwrcl3,  &wwrcl4, &wwrcl5, &wwrcl6,  &wwrcl7, &wwrcl8,  &wwrcl12 } },
    { WOR1_WORLD,  START_ADHOC_NO_11A,  NULL,            8,
      { &wwrcl1,   &wwrcl2, &wwrcl3,  &wwrcl4, &wwrcl7, &wwrcl8,  &wwrcl9, &wwrcl12, NULL } },
    { WOR2_WORLD,  START_ADHOC_IN_11A,  adhocWorldClist, 10,
      { &wwrcl1,   &wwrcl2, &wwrcl3,  &wwrcl4, &wwrcl5, &wwrcl6,  &wwrcl7, &wwrcl8,  &wwrcl9, &wwrcl12 } },
    { WOR3_WORLD,  START_ADHOC_PER_11D, NULL,            8,
      { &wwrcl1,   &wwrcl2, &wwrcl3,  &wwrcl4, &wwrcl5, &wwrcl6,  &wwrcl8, &wwrcl12, NULL } },
    { WOR4_WORLD,  START_ADHOC_NO_11A, NULL,             7,
      { &wwrcl1,   &wwrcl3a, &wwrcl4, &wwrcl5, &wwrcl6, &wwrcl8a, &wwrcl12, NULL,    NULL } },    
    { WOR5_ETSIC,  START_ADHOC_NO_11A, NULL,             5,
      { &wwrcl1,   &wwrcl3,  &wwrcl4, &wwrcl8, &wwrcl12, NULL } }
};

/*
 * Scanning order of non-Extended Channel Mode channels
 * Same as Extended Channel Mode channels but exclude channels 12 - 14.
 */
static const WWR_CLIST * NonEcmSocTbl [] = {
    &wwrcl3x,  &wwrcl1,  &wwrcl2, &wwrcl8x, &wwrcl4, &wwrcl5, &wwrcl6, &wwrcl7,
    &wwrcl10, &wwrcl11, &wwrcl12, NULL
};

/*
 *  Non-Extended Channel Mode World SKU Table
 */
static const SUPER_DOMAIN NonEcmSuperDomainTbl [] = {
    { WOR0_WORLD, START_ADHOC_PER_11D, NULL,            9,
      { &wwrcl1,  &wwrcl2, &wwrcl3x, &wwrcl4, &wwrcl5, &wwrcl6,   &wwrcl7,  &wwrcl8x, &wwrcl12 } },
    { WOR01_WORLD, START_ADHOC_PER_11D, NULL,           9,
      { &wwrcl1,  &wwrcl2, &wwrcl3x, &wwrcl4, &wwrcl5, &wwrcl6,   &wwrcl7,  &wwrcl8x, &wwrcl12 } },
    { WOR02_WORLD, START_ADHOC_PER_11D, NULL,           9,
      { &wwrcl1,  &wwrcl2, &wwrcl3,  &wwrcl4, &wwrcl5, &wwrcl6,   &wwrcl7,  &wwrcl8x, &wwrcl12 } },
    { EU1_WORLD,  START_ADHOC_PER_11D, NULL,           9,
      { &wwrcl1,  &wwrcl2, &wwrcl3,  &wwrcl4, &wwrcl5, &wwrcl6,   &wwrcl7,  &wwrcl8x, &wwrcl12 } },
    { WOR1_WORLD, START_ADHOC_NO_11A,  NULL,            7,
      { &wwrcl1,  &wwrcl2, &wwrcl3x, &wwrcl4, &wwrcl7, &wwrcl8x,  &wwrcl12, NULL,     NULL } },
    { WOR2_WORLD, START_ADHOC_IN_11A,  adhocWorldClist, 9,
      { &wwrcl1,  &wwrcl2, &wwrcl3x, &wwrcl4, &wwrcl5, &wwrcl6,   &wwrcl7,  &wwrcl8x, &wwrcl12 } },
    { WOR3_WORLD, START_ADHOC_PER_11D, NULL,            8,
      { &wwrcl1,  &wwrcl2, &wwrcl3x, &wwrcl4, &wwrcl5, &wwrcl6,   &wwrcl8x, &wwrcl12, NULL } },
    { WOR4_WORLD, START_ADHOC_NO_11A, NULL,             7,
      { &wwrcl1, &wwrcl3ax, &wwrcl4, &wwrcl5, &wwrcl6, &wwrcl8ax, &wwrcl12, NULL,     NULL } },    
    { WOR5_ETSIC, START_ADHOC_NO_11A, NULL,             6,
      { &wwrcl1, &wwrcl3ax, &wwrcl4, &wwrcl8ax, &wwrcl12, NULL } },    
};

typedef struct _adhoc_chan_RD {
    A_UINT16            rd;             /* EEPROM regulatory domain */
    WLAN_CHANNEL        *pClist;
} ADHOC_CHAN_RD;

#define ADHOC_CHAN_RD_TBL_LEN (sizeof(adhocChanRdTbl)/sizeof(ADHOC_CHAN_RD))

/*
 * The following table defines the adhoc channel list of
 * the regulatory domain
 */
static ADHOC_CHAN_RD adhocChanRdTbl[] = {
    { FCC1_FCCA, adhocWorldClist },
    { MKK1_MKKA, adhocMkkClist   },
    { MKK1_FCCA, adhocMkkClist   },
    { MKK1_MKKA1,adhocMkkClist   },
    { MKK1_MKKA2,adhocMkkClist   }    
};


/*
 * Implicit Regulatory Domain Channel Lists and Table
 */

typedef struct ImplicitRegDmn {
    REG_DOMAIN  regDmn;
    CTRY_CODE   countryCode;
    WLAN_CFLAGS channelFlags;
    WLAN_CHANNEL *cList;
} IMPLICIT_RD_CHANNEL;

static ROAMING_CHANNEL irdList1[] = {
    5170, 5190, 5210, 5230, 0                           /* 4 MKK channels */
};

static ROAMING_CHANNEL irdList2[] = {
    5210, 5250, 5290, 5760, 5800, 0                     /* 5 turbo channels */
};

static ROAMING_CHANNEL irdList3[] = {
    5500, 5520, 5540, 5560, 5580, 5600, 5620, 5640,     /* 5.47 - 5.72 ETSI channels */
    5660, 5680, 5700, 0                                 /*      "     "      "       */
};

#define NUM_IMPLICIT_RD         3

static const IMPLICIT_RD_CHANNEL irdChannelTbl [NUM_IMPLICIT_RD] = {
    { MKK1,  CTRY_JAPAN,          CHANNEL_A, irdList1 },
    { FCC1,  CTRY_UNITED_STATES,  CHANNEL_T, irdList2 },
    /*
     * Change from United Kingdom to Netherlands to match with ETSI1
     * A ETSI1 country will be needed if Netherlands changes its
     * regulatory domain
     */
    { ETSI1, CTRY_NETHERLANDS,     CHANNEL_A, irdList3 }
};

/***************************************************************************
 *      Cisco's non-ISO view of the 802.11d country code element
 *
      ID  Len   Country         Base Num  Pwr
                                 Chn Chn
    0x07, 0x06, 0x4E, 0x41, 0x00,  1, 11, 200   ; Americas 'NA\0'
    0x07, 0x06, 0x45, 0x55, 0x00,  1, 13,  50   ; Europe 'EU\0'
    0x07, 0x06, 0x4a, 0x41, 0x50, 14,  1,  90   ; Japan 'JAP'
    0x07, 0x06, 0x53, 0x50, 0x41, 10,  2,  50   ; Spain 'SPA'
    0x07, 0x06, 0x46, 0x52, 0x41, 10,  4,  50   ; France 'FRA'
    0x07, 0x06, 0x42, 0x45, 0x4c, 12,  2,  50   ; Belgium 'BEL'
    0x07, 0x06, 0x49, 0x53, 0x52,  3,  7,  50   ; Israel 'ISR'
    0x07, 0x06, 0x43, 0x41, 0x4e,  1,  8,  50   ; Canada 'CAN' (Outdoor)
    0x07, 0x06, 0x41, 0x55, 0x53,  1, 11, 200   ; Australia 'AUS'
    0x07, 0x06, 0x4a, 0x41, 0x57,  1, 14,  50   ; Japan wideband 'JPW'
    0x07, 0x06, 0x43, 0x4E, 0x20,  1, 11,   5   ; China 'CN '
 *
 ***************************************************************************/
typedef struct cisco_country_code {
    CTRY_CODE   countryCode;
    REG_DOMAIN  regDmn;
    COUNTRY_INFO_LIST   info;
} CISCO_COUNTRY_INFO_LIST;


static CISCO_COUNTRY_INFO_LIST ciscoCcTbl[] = {
    { CTRY_UNITED_STATES, FCC1_FCCA,
      { 0x07, 0x06, { 0x4E, 0x41, 0x00 },         /* Americas 'NA\0' */
        { {  1, 11, 200 }, {0, 0, 0}, {0, 0, 0}, {0, 0, 0} }, 0 } },
    { CTRY_UNITED_KINGDOM, ETSI1_WORLD,
      { 0x07, 0x06, { 0x45, 0x55, 0x00 },         /* Europe 'EU\0' */
        { {  1, 13,  50 }, {0, 0, 0}, {0, 0, 0}, {0, 0, 0} }, 0 } },
    { CTRY_JAPAN, MKK1_MKKA,
      { 0x07, 0x06, { 0x4a, 0x41, 0x50, },        /*  Japan 'JAP' */
        { { 14,  1,  90 }, {0, 0, 0}, {0, 0, 0}, {0, 0, 0} }, 0 } },
    { CTRY_SPAIN, ETSI1_WORLD,
      { 0x07, 0x06, { 0x53, 0x50, 0x41 },         /* Spain 'SPA' */
        { { 10,  2,  50 }, {0, 0, 0}, {0, 0, 0}, {0, 0, 0} }, 0 } },
    { CTRY_FRANCE, ETSI3_WORLD,
      { 0x07, 0x06, { 0x46, 0x52, 0x41 },       /* France 'FRA' */
        { { 10,  4,  50 }, {0, 0, 0}, {0, 0, 0}, {0, 0, 0} }, 0 } },
    { CTRY_BELGIUM, ETSI2_WORLD,
      { 0x07, 0x06, { 0x42, 0x45, 0x4c },         /* Belgium 'BEL' */
        { { 12,  2,  50 }, {0, 0, 0}, {0, 0, 0}, {0, 0, 0} }, 0 } },
    { CTRY_ISRAEL, NULL1_ETSIB,
      { 0x07, 0x06, { 0x49, 0x53, 0x52 },         /* Israel 'ISR' */
        { {  3,  7,  50 }, {0, 0, 0}, {0, 0, 0}, {0, 0, 0} }, 0 } },
    { CTRY_CANADA, FCC2_FCCA,
      { 0x07, 0x06, { 0x43, 0x41, 0x4e },         /* Canada 'CAN' (Outdoor) */
        { {  1,  8,  50 }, {0, 0, 0}, {0, 0, 0}, {0, 0, 0} }, 0 } },
    { CTRY_AUSTRALIA, FCC2_WORLD,
      { 0x07, 0x06, { 0x41, 0x55, 0x53 },         /* Australia 'AUS' */
        { {  1, 11, 200 }, {0, 0, 0}, {0, 0, 0}, {0, 0, 0} }, 0 } },
    { CTRY_JAPAN, MKK1_MKKA,
      { 0x07, 0x06, { 0x4a, 0x41, 0x57 },         /* Japan wideband 'JPW' */
        { {  1, 14,  50 }, {0, 0, 0}, {0, 0, 0}, {0, 0, 0} }, 0 } },
    { CTRY_CHINA, APL1_WORLD,
      { 0x07, 0x06, { 0x43, 0x4E, 0x20 },         /* China 'CN ' */
        { { 1, 11,   5  }, {0, 0, 0}, {0, 0, 0}, {0, 0, 0} }, 0 } }
};

#ifdef INCLUDE_RCW
extern void (*siteSurveyCallback)(BSSDESCR *pBss);
#endif /* INCLUDE_RCW */

#if defined(DEBUG) && !defined(BUILD_AP)
A_UINT16 softRD = 0;
#endif

#ifdef DEBUG
int wlanScanDebugLevel;
#endif /* DEBUG */

/* Forward declarations */
static A_UINT16
wlanEepromCcRdGet(WLAN_DEV_INFO *pDevInfo);

static A_UINT16
wlanSupDomGet(WLAN_DEV_INFO *pDevInfo, A_UINT16 *tblIdx);

static SUPER_DOMAIN *
wlanSupDomTblGet(WLAN_DEV_INFO *pDevInfo);

static A_UINT16
wlanGetNumChannelsForDomain(WLAN_DEV_INFO *pDevInfo, CTRY_CODE countryCode,
                            A_UINT16 modeSelect, A_UINT16 enableOutdoor);

static A_STATUS
wlanAllocScanList(WLAN_SCAN_LIST **pSList, A_UINT16 entries);

static A_STATUS
wlanAllocChannelList(WLAN_CHANNEL_LIST **pCList, A_UINT16 entries);

static A_UINT16
wlanGetWmRD(WLAN_DEV_INFO *pDevInfo, CTRY_CODE countryCode, WLAN_CFLAGS channelFlags);

static A_BOOL
wlanIsWorldWideRoaming(WLAN_DEV_INFO *pDevInfo);

static CTRY_CODE
wlanXlatNonIsoCcode(A_UCHAR *countryCodeName);

static WLAN_CHANNEL *
wlanGetAdHocClist(REG_DOMAIN rd);

static A_BOOL
wlanIsPassiveScanChannel(WLAN_DEV_INFO *, REG_DOMAIN rd, CHAN_VALUES *);

static A_BOOL
wlanIsDfsRD(WLAN_DEV_INFO *, REG_DOMAIN rd);

static A_BOOL
wlanIsCountry11g(CTRY_CODE cc);

static A_BOOL
wlanIsEcmChanPassive(WLAN_DEV_INFO *pDevInfo, A_UINT16 channel);

static A_BOOL
wlanVerifyChan4Ecm(WLAN_DEV_INFO *, A_UINT16 channel, WLAN_CFLAGS cf);

static CTRY_CODE
wlanGetSingletonCountryRD(REG_DOMAIN rd);

void
printmodes(A_UINT16 mode);

void
clistShow(WLAN_DEV_INFO *pDevInfo);

void
slistShow(WLAN_DEV_INFO *pDevInfo);

#define TOTAL_CLIST_MODES   5
typedef struct {
    A_UINT16 modeSelect;
    WLAN_CFLAGS channelFlags;
} CLIST_MODES;

const static CLIST_MODES modeTable[TOTAL_CLIST_MODES] = {
    {MODE_SELECT_TURBO, CHANNEL_T},
    {MODE_SELECT_11A,   CHANNEL_A},
    {MODE_SELECT_11B,   CHANNEL_B},
    {MODE_SELECT_11G,   CHANNEL_G},
    {MODE_SELECT_108G,  CHANNEL_108G}
};

A_UINT16
wlanHalRD(WLAN_DEV_INFO *pDevInfo)
{
#if defined(DEBUG) && !defined(BUILD_AP)
    if (softRD) {
        return softRD;
    }
#endif

    return (A_UINT16)pDevInfo->devCap.regDomain;
}

/***********************************************************
 * wlanEepromCcRdInit
 *
 * Initialize the soft EEPROM flag
 */
void
wlanEepromCcRdInit(WLAN_DEV_INFO *pDevInfo)
{
    pDevInfo->staConfig.sku.eeprom.Inited = FALSE;
}


/***********************************************************
 * wlanEepromCcRdGet
 *
 * Read the software copy of the EEPROM
 */
static A_UINT16
wlanEepromCcRdGet(WLAN_DEV_INFO *pDevInfo)
{
    /* Read from hardware first */
    if (pDevInfo->staConfig.sku.eeprom.Inited == FALSE) {
        pDevInfo->staConfig.sku.eeprom.Inited = TRUE;
        pDevInfo->staConfig.sku.eeprom.RdCc = wlanHalRD(pDevInfo);
    }
    pDevInfo->staConfig.sku.eeprom.RdCc &= ~WORLDWIDE_ROAMING_FLAG;
    return pDevInfo->staConfig.sku.eeprom.RdCc;
}

/***********************************************************
 * wlanSupDomGet
 *
 * Scan super domain table for a match with EEPROM
 */
A_UINT16
wlanSupDomGet(WLAN_DEV_INFO *pDevInfo, A_UINT16 *tblIdx)
{
    A_UINT16 eepRdCc, ij, rd, tblSize;

    eepRdCc = wlanEepromCcRdGet(pDevInfo);
    if (eepRdCc & COUNTRY_ERD_FLAG) {
        return 0;
    }

    rd = eepRdCc & SUPER_DOMAIN_MASK;
    if (rd) {
        tblSize = NUM_OF_SUPER_DOMAINS;
        for (ij = 0; ij < tblSize; ij++) {
            if (superDomainTbl[ij].domain == rd) {
                *tblIdx = ij;
#ifdef  DEBUG
                uiPrintf("wlanSupDomGet: rd %x, eepRdCc %x\n", rd, eepRdCc);
#endif
                return rd;
            }
        }
        rd = 0;
    }
#ifdef  BUILD_AP
    *tblIdx = 0;
    rd = superDomainTbl[0].domain;
#endif
#ifdef DEBUG
    uiPrintf("wlanSupDomGet: rd %x, eepRdCc %x\n", rd, eepRdCc);
#endif
    return rd;
}

SUPER_DOMAIN *
wlanSupDomTblGet(WLAN_DEV_INFO *pDevInfo)
{
    A_UINT16      eepRdCc, ij, rd, tblSize;
    SUPER_DOMAIN  *pTbl = NULL;

    eepRdCc = wlanHalRD(pDevInfo);
    if (eepRdCc & COUNTRY_ERD_FLAG) {
        return pTbl;
    }

    rd = eepRdCc & SUPER_DOMAIN_MASK;
    if (rd) {
        tblSize = NUM_OF_SUPER_DOMAINS;
        for (ij = 0; ij < tblSize; ij++) {
            if (!pDevInfo->staConfig.extendedChanMode) {
                if (NonEcmSuperDomainTbl[ij].domain == rd) {
                    return (SUPER_DOMAIN *)&NonEcmSuperDomainTbl[ij];
                }
            } else {
                if (superDomainTbl[ij].domain == rd) {
                    return (SUPER_DOMAIN *)&superDomainTbl[ij];                    
                }
            }
        }
    }
    return pTbl;
}

void
wlanEepromRdCcWriteRd(WLAN_DEV_INFO *pDevInfo, REG_DOMAIN rd)
{
    pDevInfo->staConfig.sku.eeprom.RdCc = rd & SUPER_DOMAIN_MASK;
    uiPrintf("wlanEepromRdCcWriteRd: rd: 0x%x, 0x%x\n",
             rd, pDevInfo->staConfig.sku.eeprom.RdCc);
}

void
wlanEepromRdCcWriteCc(WLAN_DEV_INFO *pDevInfo, CTRY_CODE cc)
{
    pDevInfo->staConfig.sku.eeprom.RdCc = COUNTRY_ERD_FLAG |
                                          (cc & COUNTRY_CODE_MASK);
    uiPrintf("wlanEepromRdCcWriteCc: cc: %d, 0x%x\n",
             cc, pDevInfo->staConfig.sku.eeprom.RdCc);
}

static const REG_DMN_WM_FREQ_TABLE *
wlanGetRdmfTable(A_UINT16 channelFlags)
{
    if (channelFlags & CHANNEL_5GHZ) {
        return (channelFlags & CHANNEL_TURBO) ? regDmnWm5Turbo : regDmnWm5;
    }
    if (channelFlags & CHANNEL_2GHZ) {
        return (channelFlags & CHANNEL_TURBO) ? regDmnWm2Turbo : regDmnWm2;
    }

    // either 2 or 5 GHz flags must be set
    ASSERT(0);
    return NULL;
}

static int
wlanGetRdmfTblLen(WLAN_CFLAGS channelFlags)
{
    if (IS_CHAN_5GHZ(channelFlags)) {
        return (channelFlags & CHANNEL_TURBO) ? NUM_OF_TURBO_DOMAINS : NUM_OF_11A_DOMAINS;
    }
    if (IS_CHAN_2GHZ(channelFlags)) {
        return (channelFlags & CHANNEL_TURBO) ? NUM_OF_108G_DOMAINS : NUM_OF_11B_DOMAINS;
    }

    // either 2 or 5 GHz flags must be set
    ASSERT(0);
    return 0;
}

static A_UINT16
wlanGetChannelSpread(A_UINT16 channelFlags)
{
    if (IS_CHAN_2GHZ(channelFlags)) {
        // TODO: currently we return same value for CCK or OFDM in 2 GHz
        return IS_CHAN_TURBO(channelFlags) ? TURBO_CHANNEL_SEPARATION : LOW2GHZ_CHANNEL_SEPARATION;
    } else if (IS_CHAN_5GHZ(channelFlags)) {
        if (IS_CHAN_TURBO(channelFlags)) {
            return TURBO_CHANNEL_SEPARATION;
        } else {
            return OFDM5_CHANNEL_SEPARATION;
        }
    }

    // either 2 or 5 GHz flags must be set
    ASSERT(0);
    return 20; // Returning 0 may throw it into a loop ?!?
}

static const REG_DMN_WM_FREQ_TABLE *
wlanGetCcRdTbl(WLAN_DEV_INFO *pDevInfo, CTRY_CODE ccode, WLAN_CFLAGS cflags)
{
    const REG_DMN_WM_FREQ_TABLE *pCcTbl;
    int      i, numRd;
    A_UINT16 wmode;


    pCcTbl = wlanGetRdmfTable(cflags);
    numRd  = wlanGetRdmfTblLen(cflags);
    wmode  = wlanGetWmRD(pDevInfo, ccode, cflags);

    /* Search through Regulatory Domain (Enum) table for a match */
    for (i = 0; i < numRd; i++) {
        if (pCcTbl[i].regDmnWirelessMode == wmode) {
            return &pCcTbl[i];
        }
    }
    return regDmnNone;
}

static const REG_DMN_WM_FREQ_TABLE *
wlanGetRdTbl(WLAN_DEV_INFO *pDevInfo, CTRY_CODE ccode, WLAN_CFLAGS cflags)
{
    return wlanGetCcRdTbl(pDevInfo, ccode, cflags);
}

static const A_UINT16
wlanGetRdMode(WLAN_DEV_INFO *pDevInfo, CTRY_CODE ccode, WLAN_CFLAGS cflags)
{
    return wlanGetRdTbl(pDevInfo, ccode, cflags)->regDmnWirelessMode;
}

/***********************************************************
 * wlanGetCountryIndex
 *
 * Match the given countryCode in the static country Table
 */
static A_UINT16
wlanGetCtryIdx(CTRY_CODE countryCode)
{
    A_UINT16 index;

    for (index = 0; index < NUM_OF_COUNTRIES; index++) {
        if (countryCode == allCountries[index].countryCode) {
            return index;
        }
    }
    uiPrintf("wlanGetCtryIdx: Failed to find index in country code table %d\n", countryCode);
//    ASSERT(0);
    return 0;
}

/***********************************************************
 * wlanGetRDTblIdx
 *
 * Get the Wireless Mode Regulatory Domain based on the country code
 * and wireless mode
 */
static A_INT16
wlanGetRDTblIdx(REG_DOMAIN regDmnEnum)
{
    A_UINT16 index;

    for (index = 0; index < NUM_OF_REG_DOMAINS; index++) {
        if (regDmnEnum == allEnumRds[index].regDmnEnum) {
            return index;
        }
    }
//    uiPrintf("wlanGetRDTblIdx: Failed to find index in regDmnEnum table %d\n", regDmnEnum);
    return 0;
}

static A_UINT16
wlanGetRDIdxByCtry(WLAN_DEV_INFO *pDevInfo, CTRY_CODE countryCode)
{
    A_UINT16 countryIndex, eepRdCc;
    A_INT16  regDmnEnumIndex;

    /*
     * Use country code to look up regulatory domain for known country code.
     * Otherwise, use default country code and regulatory domain if
     * - if EEPROM is blank.
     * - if EEPROM's regulatory domain is unknown.
     */
    if (countryCode == CTRY_DEFAULT) {
        regDmnEnumIndex = wlanGetRDTblIdx(DEF_REGDMN);
        eepRdCc = wlanEepromCcRdGet(pDevInfo); /* No country set - USE EEPROM value */
        if (eepRdCc) {
            A_INT16 rd = wlanGetRDTblIdx(eepRdCc);

            if (rd == -1) {
                uiPrintf("wlanGetWmRD: Can't determine country, use default %d\n",
                         regDmnEnumIndex);
            } else {
                regDmnEnumIndex = rd;
            }
        }
    } else {
        countryIndex    = wlanGetCtryIdx(countryCode);
        regDmnEnumIndex = wlanGetRDTblIdx(allCountries[countryIndex].regDmnEnum);
    }

    return regDmnEnumIndex;
}

/***********************************************************
 * wlanGetWmRD
 *
 * Get the Wireless Mode Regulatory Domain based on the country code
 * and wireless mode
 */
static A_UINT16
wlanGetWmRD(WLAN_DEV_INFO *pDevInfo, CTRY_CODE countryCode, WLAN_CFLAGS channelFlags)
{
    A_UINT16 countryIndex, eepRdCc;
    A_INT16  regDmnEnumIndex;

    /*
     * Use country code to look up regulatory domain for known country code.
     * Otherwise, use default country code and regulatory domain if
     * - if EEPROM is blank.
     * - if EEPROM's regulatory domain is unknown.
     */
    if (countryCode == CTRY_DEFAULT) {
        regDmnEnumIndex = wlanGetRDTblIdx(DEF_REGDMN);
        eepRdCc = wlanEepromCcRdGet(pDevInfo); /* No country set - USE EEPROM value */
        if (eepRdCc) {
            A_INT16 rd = wlanGetRDTblIdx(eepRdCc);

            if (rd == -1) {
                uiPrintf("wlanGetWmRD: Can't determine country, use default %d\n",
                         regDmnEnumIndex);
            } else {
                regDmnEnumIndex = rd;
            }
        }
    } else {
        countryIndex    = wlanGetCtryIdx(countryCode);
        regDmnEnumIndex = wlanGetRDTblIdx(allCountries[countryIndex].regDmnEnum);
    }
    if (IS_CHAN_2GHZ(channelFlags)) {
        return allEnumRds[regDmnEnumIndex].regDmnWirelessMode2;
    } else {
        return allEnumRds[regDmnEnumIndex].regDmnWirelessMode5;
    }
}

A_UINT16
wlanGetChannelFlags (A_UINT16 modeSelect, WLAN_CHANNEL channel)
{
    const REG_DMN_WM_FREQ_TABLE *pCcTable;
    int       i, j, k, regDmnEnumEntries;
    A_UINT16  searchChannel, channelSpread, cflags;


    for (k = 0; k < TOTAL_CLIST_MODES; k++) {
        if (!(modeTable[k].modeSelect & modeSelect)) {
            /*
             * Corresponding bit in channelFlags is not set.
             * We don't want add this channel type.
             */
            continue;
        }

        cflags        = modeTable[k].channelFlags;
        pCcTable      = wlanGetRdmfTable(cflags);
        channelSpread = wlanGetChannelSpread(cflags);
        
        /* Search through Regulatory Domain (Enum) table for a match */
        regDmnEnumEntries = wlanGetRdmfTblLen(cflags);
        for (i = 0; i < regDmnEnumEntries; i++) {
            for (j = 0; j < pCcTable[i].entries; j++) {
                /* Find this channel's entry */
                for (searchChannel = pCcTable[i].chanGroup[j].lowChannel;
                     searchChannel <= pCcTable[i].chanGroup[j].highChannel;
                     searchChannel += (IS24DEBUG(searchChannel) ?
                                       HIGH2GHZ_CHANNEL_SEPARATION : channelSpread))
                {
                    if (searchChannel == channel) {
                        return cflags;
                    }
                }
            }
        }
    }
    return 0;
}

/***********************************************************
 * wlanMKKModifyChannels
 *
 * Modify the channel numbers for Japan if the product follows
 * new certification. The algorithm is,
 * If the EEPROM version is >=5.3 or the use channels bit is
 * set in software, then change the regulatory domain definition
 * of MKK1 and MKK2 to contain the new channels (remove old ones)
 * If world mode, modify super domain table, and let 
 * wlanCCChannelInit() figure out the correct channels from
 * the regulatory domain.
 * Modify Adhoc channel list for MKK.
 */
void
wlanModifyMKKChannels (WLAN_DEV_INFO *pDevInfo) {

    A_UINT32 validMKKBands = 0;
    A_UINT32 i, j, index = 0;
    A_UINT16 validMKKBandBits[] = {REG_KK_USE_EVEN_U1, 
                                   REG_KK_USE_U2, 
                                   REG_KK_USE_MIDBAND};
    A_UINT16 validMKKBandBitsSize = sizeof(validMKKBandBits)/sizeof(A_UINT16);

    static const REG_DMN_WM_CHANNEL_GROUP newMKKChannels[] = {
            {   5180,    5240,  20, 0},
            {   5260,    5320,  20, 0},
            {   5500,    5700,  20, 0}};
    
    if (mkkChannelsModified) {
        return;
    }
    mkkChannelsModified = TRUE;
    
    validMKKBands = wlanGetRegCapabilities(pDevInfo);

    if (!validMKKBands || validMKKBands == REGCAPBITS_VALUE_SET) {
        return;
    }

    /* For world wide SKUs modify the super domain table to contain
     * the even channels for Japan. We will modify the channel list
     * for MKK1/MKK2 based on EEPROM bits and wlanInitChannelList()
     * will build channel list with even channels if required, but
     * it needs to match channels in super domain table.
     * Change the adhoc list also to point to even channels.
     */
    if (validMKKBands & REG_KK_USE_EVEN_U1) {
        A_BCOPY(adhocMkkClistAlt, adhocMkkClist, sizeof(adhocMkkClistAlt)); 
        if (wlanIsWwrSKU(pDevInfo)) {
            A_BCOPY(rcl2Alt, rcl2, sizeof(rcl2Alt)); 
        }
    }

    for (i = 0; i < NUM_OF_11A_DOMAINS; i++) {

        if (regDmnWm5[i].regDmnWirelessMode != MKK1 &&
                regDmnWm5[i].regDmnWirelessMode != MKK2) {
            continue;
        }

        /* UNI1 band is currently last band in the channel list
         * definition, change it in future if that is not true.
         */
        if (regDmnWm5[i].regDmnWirelessMode == MKK1) {
            index = 0;
        } else if (regDmnWm5[i].regDmnWirelessMode == MKK2) {
            index = 2;
        }

        /* If UNII1 bit is not used then clear it out */
        if ((!(validMKKBands & REG_KK_USE_EVEN_U1)) && (!(validMKKBands & REG_KK_USE_ODD_U1))) {
            regDmnWm5[i].entries--;
        }

        /* If UNII1 odd is enabled then mover forward, channel list is built
         * with UNI-1 odd by default */
        if (validMKKBands & REG_KK_USE_ODD_U1) {
            index++;
        }

        for (j = 0; j < validMKKBandBitsSize; j++) {

            if (validMKKBands & validMKKBandBits[j]) {
                regDmnWm5[i].chanGroup[index].lowChannel 
                    = newMKKChannels[j].lowChannel;
                regDmnWm5[i].chanGroup[index].highChannel 
                    = newMKKChannels[j].highChannel;
                regDmnWm5[i].chanGroup[index].powerDfs 
                    = newMKKChannels[j].powerDfs;
                regDmnWm5[i].chanGroup[index].antennaMax 
                    = newMKKChannels[j].antennaMax;
                regDmnWm5[i].entries++;
                if ((validMKKBandBits[j] == REG_KK_USE_EVEN_U1) && (!(validMKKBands & REG_KK_USE_ODD_U1))) {
                    /* We will only replace the UNII1 band, when ODD is disabled */
                    regDmnWm5[i].entries--;
                }
                index++;
            }
        }
    }
}

A_BOOL
wlanIsHwSupportMode (WLAN_DEV_INFO *pDevInfo, A_UINT16 modeSelect)
{
    /* Get modes support by the hardware */
    A_UINT32 wMode;

    wMode = pDevInfo->devCap.wirelessModes;

    return (modeSelect & wMode) ? TRUE : FALSE;
}

WLAN_CFLAGS
wlanMode2Cflags(int mode)
{
    switch (mode) {
        case MODE_SELECT_11A:   return CHANNEL_A;
        case MODE_SELECT_11B:   return CHANNEL_B;
        case MODE_SELECT_11G:   return CHANNEL_G;
        case MODE_SELECT_TURBO: return CHANNEL_T;
        case MODE_SELECT_108G:  return CHANNEL_108G;
    }
    uiPrintf("wlanMode2Cflags: unknown mode %x\n", mode);
    return 0;
}

A_UINT32
wlanCflags2Mode(WLAN_CFLAGS cFlags)
{
    switch ((cFlags & CHANNEL_ALL) & ~CHANNEL_XR) {
        case CHANNEL_A:    
            return MODE_SELECT_11A;
        case CHANNEL_B:
            return MODE_SELECT_11B;
        case CHANNEL_G:
            return MODE_SELECT_11G;
        case CHANNEL_T:
            return MODE_SELECT_TURBO;
        case CHANNEL_108G:
            return MODE_SELECT_108G;
    }
    uiPrintf("wlanCflags2Mode: unknown cFlags %x\n", cFlags);
    return 0;
}

/*
 * wlanInitRoamingChannelList
 *
 * Build a dynamic roaming channel list
 */
A_STATUS
wlanInitRoamingChannelList(WLAN_DEV_INFO *pDevInfo, WLAN_CHANNEL_LIST **pCList,
                           A_UINT16 wireless_selection)
{
    A_STATUS     status = A_OK;
    A_UINT16     clistSize;
    A_UINT16     cflags, channel;
    A_UINT16     totalChannels = 0;
    A_UINT16     sd, tblIdx, tmode;
    int          ij, k, tblSize;
    WWR_CLIST   *pRc;
    A_UINT16     modeSelect;

    modeSelect = wireless_selection;
    
    pDevInfo->staConfig.sku.adHocMode = 0;
    sd = wlanSupDomGet(pDevInfo, &tblIdx);
    if (!sd) {
        return A_OK;
    }

    pDevInfo->staConfig.sku.adHocMode = superDomainTbl[tblIdx].adHocMode;

    /*
     * Use the super domain to determine the number of roaming channels
     */

    clistSize = 0;
    if (!pDevInfo->staConfig.extendedChanMode) {
        tblSize = NonEcmSuperDomainTbl[tblIdx].rclTblSize;
    } else {
        tblSize = superDomainTbl[tblIdx].rclTblSize;
    }

    for (ij = 0; ij < tblSize; ij++) {
        if (!pDevInfo->staConfig.extendedChanMode) {
            pRc = NonEcmSuperDomainTbl[tblIdx].RclTbl[ij];
        } else {
            pRc = superDomainTbl[tblIdx].RclTbl[ij];
        }
        
        tmode = pRc->mode;

        for (k = 0; pRc->pList[k]; k++) {
            /*
             * Is mode supported by hardware?
             * Is mode valid for regulatory mode?
             * If 2.4GHz channel, then test all 2.4GHz channel flags
             */
            if (wlanIsHwSupportMode(pDevInfo, tmode) &&
                ((modeSelect & tmode) ||
                 (IS_CHAN_2GHZ(wlanMode2Cflags(tmode)) &&
                  (modeSelect & MODE_SELECT_2GHZ))))
            {
                clistSize++;
            }
        }
    }
    ASSERT(clistSize);

    /*
     * Allocate space for the channel list
     */
    status = wlanAllocChannelList(pCList, clistSize);
    if (status != A_OK || *pCList == NULL) {
        return status;
    }

    uiPrintf("wlanInitRoamingChannelList -- size: %d\n", clistSize);

    /*
     * Initialize the channel list from the roaming list
     * add channel and channel flags
     */
    if (!pDevInfo->staConfig.extendedChanMode) {
        tblSize = NonEcmSuperDomainTbl[tblIdx].rclTblSize;
    } else {
        tblSize = superDomainTbl[tblIdx].rclTblSize;
    }
    for (ij = 0; ij < tblSize; ij++) {
        if (!pDevInfo->staConfig.extendedChanMode) {
            pRc = NonEcmSuperDomainTbl[tblIdx].RclTbl[ij];
        }
        else {
            pRc = superDomainTbl[tblIdx].RclTbl[ij];
        }
        tmode = pRc->mode;
        printmodes(modeSelect);
        uiPrintf("[%d]:", ij);
        printmodes(tmode);        

        for (k = 0; pRc->pList[k]; k++) {
            if (wlanIsHwSupportMode(pDevInfo, tmode) &&
                ((modeSelect & tmode) ||
                 (IS_CHAN_2GHZ(wlanMode2Cflags(tmode)) &&
                  (modeSelect & MODE_SELECT_2GHZ))))
            {
                channel = pRc->pList[k];
                cflags  = wlanGetChannelFlags(tmode, channel);
                uiPrintf("channel[%d] %d, mode: %x", k, channel, modeSelect);
                wlanPrintFlags(cflags);
                ASSERT(cflags);
                /*
                 * Fix up the 2.4Ghz channel flag because the table
                 * is set up for 11b only.
                 */
                if (IS_CHAN_NON_108G_2GHZ(cflags)) {
                    cflags &= ~CHANNEL_ALL;
                    cflags |= (modeSelect & MODE_SELECT_11G) ? CHANNEL_G : CHANNEL_B;
                }
                if (cflags) {
                    /* Add passive scan flags */
                    cflags &= ~CHANNEL_PASSIVE;
                    if (tmode & MODE_SELECT_11B || tmode & MODE_SELECT_11G) {
                        if (wlanIs2GHzEcmChan(channel)) {
                            if (channel == CHANNEL_14) {
                                cflags &= ~CHANNEL_G;
                                cflags |= CHANNEL_B;
                            }
                            if (wlanIsEcmChanPassive(pDevInfo, channel) == TRUE) {
                                cflags |= CHANNEL_PASSIVE;
                            }
                        }
                    } else {
                        cflags |= CHANNEL_PASSIVE;
                    }
                    (*pCList)->chanArray[totalChannels].channelFlags = cflags;
                    (*pCList)->chanArray[totalChannels].channel = channel;
                    totalChannels++;
                    uiPrintf("[%d] %d", totalChannels, channel);
                    wlanPrintFlags(cflags);
                }
            }
        }
    }
    ASSERT(clistSize == totalChannels);
    (*pCList)->listSize = totalChannels;
    uiPrintf("wlanInitRoamingChannelList: %d channels\n", totalChannels);

    if (totalChannels == 0) {
#ifdef DEBUG
        uiPrintf("wlanInitRoamingChannelList: No WWR channel found\n");
#endif
        return A_EINVAL;
    }
    return status;
}

/*******************************************************************************
 * wlanInitChannelList
 *
 * This routine takes a regulatory domain (enumerated) and builds the channel list for base
 * and adds turbo mode channels if selected
 *
 * Inputs:
 *      - pointer to WLAN_DEV_INFO
 *      - pointer to the channel list
 *      - target country code
 *      - one ore more wireless modes (11a, turbo, 11b, 11g)
 *      - enable outdoor channel flag
 *      - don't build global scan list flag
 *
 * Country code is used to identify the regulatory domain.
 * Both regulatory domain and wireless mode(s) are used to build
 * the channel list by scanning the regulatory domain table and
 * pick up the channels with matching wireless modes.
 *
 * Returns: A_OK, A_EINVAL
 *
 * TODO:
 * add super domain support
 */
A_STATUS
wlanInitChannelList(WLAN_DEV_INFO *pDevInfo, WLAN_CHANNEL_LIST **pCList,
                    CTRY_CODE countryCode,  A_UINT16 wireless_selection,
                    A_UINT16 enableOutdoor, A_BOOL noGlobal)
{
    const REG_DMN_WM_FREQ_TABLE *pCcTable, *pCcPrimeTable = NULL;
    A_STATUS     status = A_OK;
    WLAN_CHANNEL lowChannel, highChannel;
    A_UINT16     totalChannels = 0;
    A_UINT16     maxRdChannel = 7000;
    A_UINT16     addChannel, channelSpread;
    int          i, j, k;
    A_UINT16     clistSize;
    A_UINT16     wmode;
    WLAN_CFLAGS  tcflags;
    A_UINT16     modeSelect;
    A_BOOL       includeExtraChannels = FALSE;
    A_UINT32     wlanMode;
    

    enableOutdoor = TRUE;       /* Always TRUE, no 1st gen hw allowed */
    modeSelect = wireless_selection & MODE_SELECT_ALL;

    /*
     * If 11b & 11g, then clear 11b to prevent the same channel
     * for both 11b & 11g
     */
    if ((modeSelect & (MODE_SELECT_11G | MODE_SELECT_11B)) ==
        (MODE_SELECT_11G | MODE_SELECT_11B))
    {
        modeSelect &= ~MODE_SELECT_11B;
    }

#if defined(DEBUG) && !defined(BUILD_AP)
    uiPrintf("wlanInitChannelList -- country: %s, mode: %x",
             wlanGetCountryCodeName(countryCode), modeSelect);
    printmodes(modeSelect);
    uiPrintf("\n");
#endif

    if ((modeSelect & MODE_SELECT_TURBO) &&
        (wlanIsCountryAllow11aTurbo(pDevInfo, countryCode) != TRUE))
    {
        modeSelect &= ~MODE_SELECT_TURBO;
    }

    if ((modeSelect & MODE_SELECT_11G) &&
        (wlanIsCountry11g(countryCode) != TRUE))
    {
        modeSelect &= ~MODE_SELECT_11G;
        modeSelect |= MODE_SELECT_11B;        
    }

    if ((modeSelect & MODE_SELECT_108G) &&
        (wlanIsCountry11g(countryCode) != TRUE))
    {
        modeSelect &= ~MODE_SELECT_108G;
        modeSelect |= MODE_SELECT_11B;        
    }

    if (wlanIsWorldWideRoaming(pDevInfo) == TRUE && noGlobal == FALSE) {
        return wlanInitRoamingChannelList(pDevInfo, pCList, modeSelect);
    }

    clistSize = wlanGetNumChannelsForDomain(pDevInfo, countryCode, modeSelect,
                                            enableOutdoor);

    /*
     * Only use idx 0 since this is a special table.
     * After mode usage is re-architected, the get table accessor
     * routines will be modified to select turbo prime table as
     * well as normal tables.
     */
    if (wireless_selection & MODE_SELECT_TPA &&
        modeSelect & MODE_SELECT_TURBO &&
        NUM_PRIME_A_CHANNELS > 0)
    {
        includeExtraChannels = TRUE;
        clistSize += NUM_PRIME_A_CHANNELS;
        pCcPrimeTable = &regDmnWm5TurboPrime[0];
        ASSERT(pDevInfo->staConfig.opMode == OP_MODE_AP);
    }

    if (wireless_selection & MODE_SELECT_TPG &&
        modeSelect & MODE_SELECT_108G &&
        NUM_PRIME_G_CHANNELS > 0)
    {
        includeExtraChannels = TRUE;
        clistSize += NUM_PRIME_G_CHANNELS;
        pCcPrimeTable = &regDmnWm2TurboPrime[0];
        ASSERT(pDevInfo->staConfig.opMode == OP_MODE_AP);
    }

    if (clistSize == 0) {
        return A_EINVAL;
    }

    status = wlanAllocChannelList(pCList, clistSize);
    if (status != A_OK || *pCList == NULL) {
        return status;
    }

    if ((!enableOutdoor) && (countryCode == CTRY_UNITED_STATES)) {
        maxRdChannel = FCC_OUTDOOR_FIRST_FREQ;
    }

    for (k = 0; k < TOTAL_CLIST_MODES; k++) {
        tcflags = modeTable[k].channelFlags;
        if (!(modeTable[k].modeSelect & modeSelect)) {
            /*
             * Corresponding bit in channelFlags is not set.
             * We don't want add that type of channel.
             */
            continue;
        }

        /* Skip channel modes not supported by this hardware */
        wlanMode = pDevInfo->devCap.wirelessModes;
        if (!(wlanMode & modeTable[k].modeSelect)) {
            continue;
        }

        lowChannel = IS_CHAN_5GHZ(tcflags) ?
                     (WLAN_CHANNEL)pDevInfo->devCap.low5GhzChan :
                     (WLAN_CHANNEL)pDevInfo->devCap.low2GhzChan;

        highChannel = IS_CHAN_5GHZ(tcflags) ?
                      (WLAN_CHANNEL)pDevInfo->devCap.high5GhzChan :
                      (WLAN_CHANNEL)pDevInfo->devCap.high2GhzChan;

        pCcTable      = wlanGetRdTbl(pDevInfo, countryCode, tcflags);
        channelSpread = wlanGetChannelSpread(tcflags);
        wmode         = wlanGetRdMode(pDevInfo, countryCode, tcflags);

#if defined(DEBUG) && !defined(BUILD_AP)
        uiPrintf("wmode: %x, tcflags %4x ", wmode, tcflags);
        wlanPrintFlags(tcflags);
#endif

        /*
         * Build channel list.
         * If scanning turbo prime interference, add extra channels.
         */
        for (i = 0; i < (includeExtraChannels ? 2 : 1); i++) {
            for (j = 0; j < pCcTable->entries; j++) {
                /* Add entries to the channel list */
                for (addChannel = pCcTable->chanGroup[j].lowChannel;
                     addChannel <= pCcTable->chanGroup[j].highChannel;
                    addChannel += (IS24DEBUG(addChannel) ? HIGH2GHZ_CHANNEL_SEPARATION : channelSpread))
                {
                    if (wlanVerifyChan4Ecm(pDevInfo, addChannel, modeTable[k].channelFlags) &&
                        (addChannel >= lowChannel) && (addChannel <= highChannel) &&
                        (addChannel <= maxRdChannel))
                    {
                        CHAN_VALUES cv;
                        tcflags = modeTable[k].channelFlags;
                        (*pCList)->chanArray[totalChannels].channel = addChannel;
                        if (addChannel == CHANNEL_14) {
                            tcflags &= ~CHANNEL_G;
                            tcflags |= CHANNEL_B;
                        }

                        cv.channel = addChannel;
                        cv.channelFlags = tcflags;
                        tcflags &= ~CHANNEL_PASSIVE;
                        if (wlanIsPassiveScanChannel(pDevInfo, wmode, &cv) == TRUE) {
                            tcflags |= CHANNEL_PASSIVE;
                        } else {
                            if (modeTable[k].modeSelect == MODE_SELECT_11B ||
                                modeTable[k].modeSelect == MODE_SELECT_11G) {
                                if (wlanIs2GHzEcmChan(addChannel)) {
                                    if (wlanIsEcmChanPassive(pDevInfo, addChannel) == TRUE) {
                                        tcflags |= CHANNEL_PASSIVE;
                                    }
                                }
                            }
                        }
                        (*pCList)->chanArray[totalChannels].channelFlags = tcflags;
                        totalChannels++;
                        ASSERT(totalChannels <= clistSize);
                    }
                }
            }
            pCcTable = pCcPrimeTable;
        }
    }

    (*pCList)->listSize = totalChannels;
#if defined(DEBUG) && !defined(BUILD_AP)
    uiPrintf("wlanInitChannelList -- %d channels\n", totalChannels);
#endif
    if (totalChannels == 0) {
#ifdef DEBUG
        uiPrintf("wlanInitChannelList: Country Code: %s has no entry\n",
            wlanGetCountryCodeName(countryCode));
#endif
        wlanFreeChannelList(*pCList);
        return A_EINVAL;
    }
    return status;
}

/***********************************************************
 * wlanGetNumChannelsForDomain
 *
 * Count the number of channels in this domain table adding in
 * all possible modes.
 *
 * TODO:
 * if super domain, then return the number of super domain
 * channels
 */
A_UINT16
wlanGetNumChannelsForDomain(WLAN_DEV_INFO *pDevInfo, CTRY_CODE countryCode,
                            A_UINT16 modeSelect, A_UINT16 enableOutdoor)
{
    const REG_DMN_WM_FREQ_TABLE *pCcTable;
    WLAN_CHANNEL   lowChannel, highChannel;
    A_UINT16       totalChannels = 0;
    A_UINT16       searchChannel, channelSpread;
    A_UINT16       maxRdChannel = 7000;
    int            j, k;
    A_UINT32       wMode;

    if ((!enableOutdoor) && (countryCode == CTRY_UNITED_STATES)) {
        maxRdChannel = FCC_OUTDOOR_FIRST_FREQ;
    }

    for (k = 0; k < TOTAL_CLIST_MODES; k++) {
        if (!(modeTable[k].modeSelect & modeSelect)) {
            // Corresponding bit in channelFlags is not set,
            // we don't want add that type of channels.
            continue;
        }

        wMode = pDevInfo->devCap.wirelessModes;
        /* Skip wireless modes not supported by this hardware */
        if (!(wMode & modeTable[k].modeSelect)) {
            continue;
        }

        lowChannel = IS_CHAN_5GHZ(modeTable[k].channelFlags) ?
                     (WLAN_CHANNEL)pDevInfo->devCap.low5GhzChan :
                     (WLAN_CHANNEL)pDevInfo->devCap.low2GhzChan;

        highChannel = IS_CHAN_5GHZ(modeTable[k].channelFlags) ?
                      (WLAN_CHANNEL)pDevInfo->devCap.high5GhzChan :
                      (WLAN_CHANNEL)pDevInfo->devCap.high2GhzChan;

        pCcTable      = wlanGetRdTbl(pDevInfo, countryCode, modeTable[k].channelFlags);
        channelSpread = wlanGetChannelSpread(modeTable[k].channelFlags);

        for (j = 0; j < pCcTable->entries; j++) {
            for (searchChannel  = pCcTable->chanGroup[j].lowChannel;
                 searchChannel <= pCcTable->chanGroup[j].highChannel;
                 searchChannel += (IS24DEBUG(searchChannel) ? HIGH2GHZ_CHANNEL_SEPARATION : channelSpread))
            {
                if (wlanVerifyChan4Ecm(pDevInfo, searchChannel, modeTable[k].channelFlags) &&
                    (searchChannel >= lowChannel) && (searchChannel <= highChannel) &&
                    (searchChannel <= maxRdChannel))
                {
                    totalChannels++;
                }
            }
        }
    }

    if (totalChannels == 0) {
#ifdef DEBUG
        uiPrintf("wlanGetNumChannelsForDomain: No matching channels were found\n");
#endif
        return A_EINVAL;
    }

    return totalChannels;
}

/***********************************************************
 * asciiToClist - Converts ascii string to channel list
 *
 * This routine parses an ASCII string of decimal digits and ';' and '-'
 * and converts them into a list of numbers.  Ranges are incremented by 4
 * (IEEE 5Mhz channels) to bump by 20Mhz.
 *
 * RETURNS: FALSE if conversion failed, TRUE if it succeeds
 */
#define CHANNEL_LIST_SIZE 128

A_STATUS
asciiToClist(WLAN_DEV_INFO *pDevInfo, A_UCHAR *pStr, A_UINT16 alen, WLAN_CHANNEL_LIST **pList)
{
    A_UINT16        lindex = 0;
    A_UINT16        asciiIndex, addChannel;
    A_UINT16        fnum = 0, cnum = 0;
    WLAN_CHANNEL    lowChannel, highChannel;
    A_UINT16        channelSpread = OFDM5_CHANNEL_SEPARATION;
    A_UINT16        channelFlags = CHANNEL_A;
    CHAN_VALUES     tempChannels[CHANNEL_LIST_SIZE];
    A_UINT16        i;
    A_STATUS        status = A_OK;
    A_UINT32        wMode;

    ASSERT(pList);

    if ((pStr == NULL) || (alen <= 0)) {
        return A_EINVAL;
    }

    for (asciiIndex = 0; asciiIndex < alen && lindex < CHANNEL_LIST_SIZE; asciiIndex++) {
        switch (pStr[asciiIndex]) {
        case '0':
        case '1':
        case '2':
        case '3':
        case '4':
        case '5':
        case '6':
        case '7':
        case '8':
        case '9':
            cnum = cnum * 10 + (pStr[asciiIndex] - '0');
            break;
        case '-':
            fnum = cnum;
            cnum = 0;
            break;
        case 'g':
        case 'G':
            channelFlags = CHANNEL_G;
            channelSpread = LOW2GHZ_CHANNEL_SEPARATION;
            break;
        case 't':
        case 'T':
            if (channelFlags == CHANNEL_G) {
                channelFlags = CHANNEL_108G;
                channelSpread = TURBO_CHANNEL_SEPARATION;
            } else {
                channelFlags = CHANNEL_T;
                channelSpread = TURBO_CHANNEL_SEPARATION;
            }
            break;
        case 'b':
        case 'B':
            channelFlags = CHANNEL_B;
            channelSpread = LOW2GHZ_CHANNEL_SEPARATION;
            break;
        case 'a':
        case 'A':
            channelFlags = CHANNEL_A;
            channelSpread = OFDM5_CHANNEL_SEPARATION;
            break;
        case 'x':
        case 'X':
            channelFlags |= CHANNEL_XR;
            break;
        case ',':
            if (fnum == 0) {
                fnum = cnum;
            }

            wMode = pDevInfo->devCap.wirelessModes;
            /* Channel mode must be supported in hardware to be in the channel list */
            if (wMode &
                wlanCflags2Mode(channelFlags))
            {
                lowChannel = IS_CHAN_5GHZ(channelFlags) ?
                             (WLAN_CHANNEL)pDevInfo->devCap.low5GhzChan :
                             (WLAN_CHANNEL)pDevInfo->devCap.low2GhzChan;

                highChannel = IS_CHAN_5GHZ(channelFlags) ?
                              (WLAN_CHANNEL)pDevInfo->devCap.high5GhzChan :
                              (WLAN_CHANNEL)pDevInfo->devCap.high2GhzChan;

                
                /* Convert channels up for scripted testing */
                if ((fnum > 0) && (fnum < 24)) {
                    fnum = wlanConvertChtoGHz(fnum, CHANNEL_2GHZ);
                } else if ((fnum > 0) && (fnum < 256)) {
                    fnum = wlanConvertChtoGHz(fnum, CHANNEL_5GHZ);;
                }
                if ((cnum > 0) && (cnum < 24)) {
                    cnum = wlanConvertChtoGHz(cnum, CHANNEL_2GHZ);
                } else if ((cnum > 0) && (cnum < 256)) {
                    cnum = wlanConvertChtoGHz(cnum, CHANNEL_5GHZ);;
                }
                if (fnum < lowChannel || cnum > highChannel) {
                    // one of the nuber doesn't fit into the valid range
                    fnum = cnum = 0;
                    break;
                }
                for (addChannel = fnum; addChannel <= cnum && fnum != 0 && lindex < CHANNEL_LIST_SIZE;
                     addChannel += (IS24DEBUG(addChannel) ? HIGH2GHZ_CHANNEL_SEPARATION : channelSpread))
                {
                    if (addChannel >= lowChannel && addChannel <= highChannel &&
                        lindex < CHANNEL_LIST_SIZE)
                    {
                        tempChannels[lindex].channel      = addChannel;
                        tempChannels[lindex].channelFlags = channelFlags;
                        lindex++;
                    }
                }
            }

            fnum = cnum = 0;
            break;
        default:
            return A_ERROR;
        }
    }

    if (asciiIndex == alen) {
        if (fnum == 0) {
            fnum = cnum;
        }

        wMode = pDevInfo->devCap.wirelessModes;
        /* Channel mode must be supported in hardware to be in the channel list */
        if (wMode & wlanCflags2Mode(channelFlags)) {
            lowChannel = IS_CHAN_5GHZ(channelFlags) ?
                         (WLAN_CHANNEL)pDevInfo->devCap.low5GhzChan :
                         (WLAN_CHANNEL)pDevInfo->devCap.low2GhzChan;

            highChannel = IS_CHAN_5GHZ(channelFlags) ?
                          (WLAN_CHANNEL)pDevInfo->devCap.high5GhzChan :
                          (WLAN_CHANNEL)pDevInfo->devCap.high2GhzChan;

            /* Convert channels up for scripted testing */
            if ((fnum > 0) && (fnum < 24)) {
                fnum = wlanConvertChtoGHz(fnum, CHANNEL_2GHZ);
            } else if ((fnum > 0) && (fnum < 256)) {
                fnum = wlanConvertChtoGHz(fnum, CHANNEL_5GHZ);;
            }

            if ((cnum > 0) && (cnum < 24)) {
                cnum = wlanConvertChtoGHz(cnum, CHANNEL_2GHZ);
            } else if ((cnum > 0) && (cnum < 256)) {
                cnum = wlanConvertChtoGHz(cnum, CHANNEL_5GHZ);;
            }

            if (fnum >= lowChannel && cnum <= highChannel) {
                for (addChannel = fnum; addChannel <= cnum && fnum != 0 && lindex < CHANNEL_LIST_SIZE;
                    addChannel += (IS24DEBUG(addChannel) ? HIGH2GHZ_CHANNEL_SEPARATION : channelSpread))
                {
                    if (addChannel >= lowChannel && addChannel <= highChannel &&
                        lindex < CHANNEL_LIST_SIZE)
                    {
                        tempChannels[lindex].channel = addChannel;
                        tempChannels[lindex].channelFlags = channelFlags;
                        lindex++;
                    }
                }
            }
        }
    }

    /* If we got here with no channels set - don't use clist */
    if (lindex == 0) {
        return A_ERROR;
    }

    status = wlanAllocChannelList(pList, lindex);
    if (status != A_OK) {
        return status;
    }

    for (i = 0; i < lindex; i++) {
        (*pList)->chanArray[i].channel      = tempChannels[i].channel;
        (*pList)->chanArray[i].channelFlags = tempChannels[i].channelFlags;
        uiPrintf("%2d. New channel freq %d, flags %x\n", i, (*pList)->chanArray[i].channel,
                 (*pList)->chanArray[i].channelFlags);
    }
    (*pList)->listSize = lindex;

    return status;
}

/***********************************************************
 * wlanAllocChannelList
 *
 * Allocates the number of entries into the provided channel list pointer
 */
A_STATUS
wlanAllocChannelList(WLAN_CHANNEL_LIST **pCList, A_UINT16 entries)
{
    int size = sizeof(WLAN_CHANNEL_LIST) + (entries * sizeof(CHAN_VALUES));

    *pCList = (WLAN_CHANNEL_LIST *)A_DRIVER_MALLOC(size);
    if (!(*pCList)) {
        uiPrintf("wlanAllocChannelList: Could not allocate memory for channel list of size %d entries\n",
                 entries);
        return A_NO_MEMORY;
    }
    A_MEM_ZERO(*pCList, size);
    (*pCList)->chanArray = (CHAN_VALUES *)(((A_UINT8 *)*pCList) + sizeof(WLAN_CHANNEL_LIST));

    return A_OK;
}

/***********************************************************
 * wlanFreeChannelList
 *
 * Frees all entries in this channel list
 */
A_STATUS
wlanFreeChannelList(WLAN_CHANNEL_LIST *pCList)
{
    A_UINT16 entries;

    if (pCList) {
        entries = pCList->listSize;
        if (entries) {
            A_DRIVER_FREE(pCList, sizeof(WLAN_CHANNEL_LIST) + (entries * sizeof(CHAN_VALUES)));
        }
        pCList = NULL;
    }

    return A_OK;
}

void
wlanPrintFlags (WLAN_CFLAGS flags)
{
    if (IS_CHAN_TURBO(flags))      uiPrintf (" turbo");
    if (IS_CHAN_XR(flags))         uiPrintf (" XR");
    if (IS_CHAN_CCK(flags))        uiPrintf (" cck");
    if (IS_CHAN_OFDM(flags))       uiPrintf (" ofdm");
    if (IS_CHAN_2GHZ(flags))       uiPrintf (" 2.4");
    if (IS_CHAN_5GHZ(flags))       uiPrintf (" 5");
    if (IS_CHAN_PASSIVE(flags))    uiPrintf (" passive");
    if (flags & CHANNEL_CW_INT)     uiPrintf (" CW Int");
    if (flags & CHANNEL_RADAR_INT)  uiPrintf (" Radar Int");
    if (flags & CHANNEL_BUSY)       uiPrintf (" busy");
    if (flags & CHANNEL_DONT_SCAN)  uiPrintf (" wont scan");
    uiPrintf("\n");
}

void
printmodes (A_UINT16 mode)
{
    if (mode & MODE_SELECT_11A)     uiPrintf (" 11a");
    if (mode & MODE_SELECT_TURBO)   uiPrintf (" turbo");
    if (mode & MODE_SELECT_11B)     uiPrintf (" 11b");
    if (mode & MODE_SELECT_11G)     uiPrintf (" 11g");
    if (mode & MODE_SELECT_108G)    uiPrintf (" 108g");
    uiPrintf("\n");
}

void
clistShow(WLAN_DEV_INFO *pDevInfo)
{
    WLAN_CHANNEL_LIST *pCList;
    int               cnt = 0, i;

    if (!pDevInfo) {
        return;
    }

    pCList = pDevInfo->staConfig.pClist;
    if (!pCList) {
        return;
    }

#ifndef BUILD_AP
    if (pDevInfo->pscanInfo &&
        pDevInfo->staConfig.phwChannel && pDevInfo->staConfig.pChannel)
    {
        uiPrintf("clist: scan idx: %d, hw: %d, staCfg: %d\n",
                 pDevInfo->pscanInfo->curChannelIndex,
                 pDevInfo->staConfig.phwChannel->channel,
                 pDevInfo->staConfig.pChannel->channel);
    }
#endif
    if (pDevInfo->staConfig.pChannel)
        uiPrintf("channel list: size: %d, channel: %d\n",
                 pCList->listSize, pDevInfo->staConfig.pChannel->channel);
    if (pDevInfo->staConfig.phwChannel)
        uiPrintf("Hw channel: %d\n",
                 pDevInfo->staConfig.phwChannel->channel);
    for (i = 0; i < pCList->listSize; i++) {
        uiPrintf("%2d: %4d (%3d), ", i+1,
                 pCList->chanArray[i].channel,
                 wlanConvertGHztoCh(pCList->chanArray[i].channel,
                                    pCList->chanArray[i].channelFlags));
        wlanPrintFlags(pCList->chanArray[i].channelFlags);

        cnt += pCList->chanArray[i].channelFlags & CHANNEL_DONT_SCAN ? 0 : 1;
    }

    uiPrintf ("%d scan channel, wont scan: %d\n", cnt, pCList->listSize - cnt);
}

void
slistShow(WLAN_DEV_INFO *pDevInfo)
{
    WLAN_SCAN_LIST *pSlist;
    int            i;

    if (!pDevInfo || !pDevInfo->pscanInfo) {
        return;
    }

    pSlist = pDevInfo->pscanInfo->pScanList;
    if (!pSlist) {
        return;
    }

    uiPrintf("scan list: size: %d, scan index: %d\n",
             pSlist->listSize, pDevInfo->pscanInfo->curChannelIndex);
    for (i = 0; i < pSlist->listSize; i++) {
        uiPrintf("%2d: %4d, %4x ", i+1,
                 pSlist->pChanArray[i]->channel,
                 pSlist->pChanArray[i]->channelFlags);
        wlanPrintFlags(pSlist->pChanArray[i]->channelFlags);
    }
}

/***********************************************************
 * wlanFirstChannel
 *
 * Returns the first valid channel on the channel list
 *
 */
CHAN_VALUES *
wlanFirstChannel(WLAN_DEV_INFO *pDevInfo, WLAN_CHANNEL_LIST *pClist)
{
    int i;

    for (i = 0; i < pClist->listSize; i++) {
        if (!(pClist->chanArray[i].channelFlags & CHANNEL_DONT_SCAN)) {
            return &pClist->chanArray[i];
        }
    }

    for (i = 0; i < pClist->listSize; i++) {
        if (!(pClist->chanArray[i].channelFlags & CHANNEL_DONT_SCAN)) {
            return &pClist->chanArray[i];
        }
    }
    return &pClist->chanArray[0];
}

/***********************************************************
 * wlanGetBksChannel
 *
 * Returns the valid channel on the channel list
 *
 */
CHAN_VALUES *
wlanGetBksChannel(WLAN_CHANNEL_LIST *pClist, int idx)
{
    int i;

    for (i = idx; i < pClist->listSize; i++) {
        if (!(pClist->chanArray[i].channelFlags & CHANNEL_DONT_SCAN)) {
            return &pClist->chanArray[i];
        }
    }
    return NULL;
}

/***********************************************************
 * wlanClistCmp
 *
 * Compare the two channel lists
 * Return FALSE if the lists are identical.
 * Return TRUE if the lists are different.
 *
 */
A_UINT32
wlanClistCmp(WLAN_CHANNEL_LIST *p1, WLAN_CHANNEL_LIST *p2)
{
    int i;

    if (p1->listSize != p2->listSize) {
        uiPrintf("wlanClistCmp -- size: %d, %d\n", p1->listSize, p2->listSize);
        return 1;
    }

    for (i = 0; i < p1->listSize; i++) {
        if (p1->chanArray[i].channel != p2->chanArray[i].channel) {
            return 1;
        }
    }
    return 0;
}

/***********************************************************
 * wlanIsChannelMatch
 *
 * Because both MKK & Turbo use the same channel 5210.
 * We also need to test the channel flags.
 * But it's not enough to test for matching channel flags because the
 * channel flags in the channel list might be altered to accommodate
 * 11b <=> 11g mode transition.
 */
A_BOOL wlanIsChannelMatch(CHAN_VALUES *pCv1, CHAN_VALUES *pCv2)
{
    WLAN_CFLAGS cf1, cf2;

    
    if (pCv1->channel != pCv2->channel) {
        return FALSE;
    }

    cf1 = pCv1->channelFlags & CHANNEL_ALL;
    cf2 = pCv2->channelFlags & CHANNEL_ALL;
    
    if (cf1 == cf2) {
        return TRUE;
    }

    if (IS_CHAN_5GHZ(cf1) && IS_CHAN_5GHZ(cf2)) {
        return (cf1 == cf2) ? TRUE : FALSE;
    }
            
    if (IS_CHAN_2GHZ(cf1) && IS_CHAN_2GHZ(cf2)) {
        if (IS_CHAN_108G(cf1) || IS_CHAN_108G(cf2)) {
            return (IS_CHAN_108G(cf1) && IS_CHAN_108G(cf2)) ? TRUE : FALSE;
        }
        return TRUE;
    }
    return FALSE;
}


/***********************************************************
 * wlanClistUpdate
 *
 * Use the new channel list to remove channels that does
 * not belonged to the new country code from the channel
 * list in staConfig by marking its channel flag as
 * "CHANNEL_DONT_SCAN".
 *
 */
void
wlanClistUpdate (WLAN_CHANNEL_LIST *pClist, WLAN_CHANNEL_LIST *pNewList)
{
    int          i, j;
    WLAN_CHANNEL chan;
    WLAN_CFLAGS  cflags;


    for (i = 0; i < pClist->listSize; i++) {
        CHAN_VALUES *pCv1 = &pClist->chanArray[i];
        pClist->chanArray[i].channelFlags |= CHANNEL_DONT_SCAN;
        cflags = pClist->chanArray[i].channelFlags & CHANNEL_ALL;
        chan   = pClist->chanArray[i].channel;

        for (j = 0; j < pNewList->listSize; j++) {
            CHAN_VALUES *pCv2 = &pNewList->chanArray[j];
            if (wlanIsChannelMatch(pCv1, pCv2)) {
                pClist->chanArray[i].channelFlags &= ~CHANNEL_DONT_SCAN;
                break;
            }
        }
    }

    /* Sanity check */
    for (i = j = 0; i < pClist->listSize; i++) {
        j += pClist->chanArray[i].channelFlags & CHANNEL_DONT_SCAN ? 0 : 1;
    }
    ASSERT(j);
}

/***********************************************************
 * wlanInitCcChannelList
 *
 * Use the new country code to update the current channel list
 * by setting the "DONT_SCAN" channel flags bit to channels that
 * does not belonged to the new country code
 *
 */
WLAN_INIT_CC_CLIST
wlanInitCcChannelList(WLAN_DEV_INFO *pDevInfo)
{
    WLAN_CHANNEL_LIST *pCList;
    WLAN_CHANNEL_LIST *pNewClist;
    A_STATUS          status = WLAN_INIT_CC_CLIST_NEW;
    int               ccSize;

    ASSERT(pDevInfo);

    uiPrintf("From the old channel list\n");
    clistShow(pDevInfo);

    /*
     * Update country code in the software EEPROM
     * and create a new channel list
     */
    uiPrintf("wlanInitCcChannelList: new cc: %d\n", pDevInfo->staConfig.countryCode);
    wlanEepromRdCcWriteCc(pDevInfo, pDevInfo->staConfig.countryCode);

    if (wlanInitChannelList(pDevInfo, &pNewClist,
                            pDevInfo->staConfig.countryCode,
                            pDevInfo->staConfig.NetBand,
                            TRUE, TRUE) != A_OK)
    {
        /* Can't create channel list */
        uiPrintf("wlanInitCcChannelList: Can't create channel list\n");
        return WLAN_INIT_CC_CLIST_ERR;
    }

    /*
     * The following changes is to take care of the corner case where
     * the new country code's channel list does not match the default
     * channel list of the global SKU. For example, a new channel list
     * is created in response to the country code CTRY_JAPAN in the
     * 802.11d from a 802.11b channel. But if the SKU is WOR4_WORLD,
     * then the new channel list will contain channels that are invalid
     * ito the WOR4_WORLD SKU.
     */
    if (wlanIsWwrSKU(pDevInfo)) {
        SUPER_DOMAIN *pTbl = wlanSupDomTblGet(pDevInfo);

        if (pTbl) {
            WLAN_CHANNEL_LIST *pXClist;
            A_UINT16          clistSize;
            int               ii, ij, k, tblSize;
            CHAN_VALUES       *pCv;

            uiPrintf("New global channel list size: %d\n", pNewClist->listSize);
            clistSize = 0;
            for (ii = 0; ii < pNewClist->listSize; ii++) {
                int csize = clistSize;

                pCv     = &pNewClist->chanArray[ii];
                tblSize = pTbl->rclTblSize;
                for (ij = 0; ij < tblSize; ij++) {
                    WWR_CLIST *pRc;
                    CHAN_VALUES cv1;
                    
                    pRc = pTbl->RclTbl[ij];
                    cv1.channelFlags = wlanMode2Cflags(pRc->mode);
                    for (k = 0; pRc->pList[k]; k++) {
                        cv1.channel = pRc->pList[k];
                        if (wlanIsChannelMatch(&cv1, pCv)) {
                            uiPrintf("add channel[%d] %d", clistSize, pCv->channel);
                            wlanPrintFlags(pCv->channelFlags);
                            clistSize++;
                            break;                                
                        }
                    }
                }

                if (clistSize == csize) {
                    uiPrintf("del channel[%d] %d", ii, pNewClist->chanArray[ii].channel);
                    wlanPrintFlags(pNewClist->chanArray[ii].channelFlags);
                    pNewClist->chanArray[ii].channel = 0;
                }
            }

            if (clistSize == 0) {
                uiPrintf("wlanInitCcChannelList: No change in channel list\n");
                return WLAN_INIT_CC_CLIST_NO_CHG;
            }

            status = wlanAllocChannelList(&pXClist, clistSize);
            if (status != A_OK) {
                wlanFreeChannelList(pNewClist);
                /* Can't create channel list */
                uiPrintf("wlanInitCcChannelList: Can't create channel list\n");
                return WLAN_INIT_CC_CLIST_ERR;
            }

            pXClist->listSize = 0;
            for (ii = 0; ii < pNewClist->listSize; ii++) {
                if (pNewClist->chanArray[ii].channel) {
                    pXClist->chanArray[pXClist->listSize++] = pNewClist->chanArray[ii];
                }
            }
            wlanFreeChannelList(pNewClist);
            pNewClist = pXClist;
        }
    }

    if (wlanClistCmp(pDevInfo->staConfig.pClist, pNewClist) == 0) {
        uiPrintf("wlanInitCcChannelList: No change in channel list\n");
        return WLAN_INIT_CC_CLIST_NO_CHG;
    }

    /*
     * Update the channel list by marking those channels that does not
     * belonged to the new country code as invalid for scanning.
     */
    pCList = pDevInfo->staConfig.pClist;
    ccSize = pCList->listSize;
    uiPrintf("wlanInitCcChannelList -- list size: %d (old), %d (new)\n", ccSize, pNewClist->listSize);
    ASSERT(ccSize);
    ASSERT(pNewClist->listSize);

    wlanClistUpdate(pCList, pNewClist);
    wlanFreeChannelList(pNewClist);

    uiPrintf("Updated channel list\n");
    clistShow(pDevInfo);

    return status;
}

/***********************************************************
 * wlanBkSlistSort
 *
 * Rearrange the channel list for background scanning
 * and return the first sorted channel
 */
CHAN_VALUES *
wlanBkSlistSort(WLAN_DEV_INFO *pDevInfo)
{
    WLAN_SCAN_LIST *pScanList = NULL;
    int            i, j, k, m, limit, numBss;
    CSERV_INFO     *pCsInfo;
    BSSDESCR       **pTbl, *pTmp;
    CHAN_VALUES    *pCv;

    ASSERT(pDevInfo);

    pCv     = wlanFirstChannel(pDevInfo, pDevInfo->staConfig.pClist);
    pCsInfo = pDevInfo->pCsInfo;

    uiPrintf("wlanBkSlistSort: %p\n", (void *)pCsInfo);

    if (!pCsInfo || !pCsInfo->scanInfo.bssSet.count) {
        return pCv;
    }

    if (pDevInfo->staConfig.userClist) {
        uiPrintf("use clist\n");
        return pCv;
    }

    if (pDevInfo->staConfig.countryCode == CTRY_DEBUG) {
        return pCv;
    }

    numBss = pCsInfo->scanInfo.bssSet.count;
    if ((pTbl = (BSSDESCR **)A_DRIVER_MALLOC(sizeof(BSSDESCR *) * numBss)) == NULL) {
        return pCv;
    }

    uiPrintf("numBss %d\n", numBss);

    for (i = 0; i < numBss; i++) {
        pTbl[i] = &pCsInfo->scanInfo.bssSet.bssDescrArray[i];
        uiPrintf("BSS[%2d] ", i);
        printMacAddress(pTbl[i]->bssId);
        uiPrintf(" ");
        printSsid(&pTbl[i]->ssid);
        uiPrintf(" rssi: %d, %d", pTbl[i]->rssi, pTbl[i]->pChannel->channel);
        uiPrintf(", ranking: %d", pTbl[i]->rank);
        wlanPrintFlags(pTbl[i]->pChannel->channelFlags);
        ASSERT(pTbl[i]->pChannel->channel);
    }

    /*
     * Sort the BSS in ascending ranking order and put its
     * channels in the new channel list before putting it
     * back to channel list in pDevInfo.
     */
    pTmp = numBss == 1 ? pTbl[0] : NULL;
    for (limit = numBss - 1; limit; limit--) {
        for (i = 0; i < limit; i++) {
            if (pTbl[i]->rank > pTbl[i+1]->rank) {
                pTmp      = pTbl[i];
                pTbl[i]   = pTbl[i+1];
                pTbl[i+1] = pTmp;
            }
        }
    }

    if (pTmp) {
        WLAN_CHANNEL schan;
        WLAN_CFLAGS  cflags;
        int          match;
        WWR_CLIST    **pSoc;
        
#ifdef DEBUG
        for (i = 0; i < numBss; i++) {
            uiPrintf("channel %d, ranking: %d\n", pTbl[i]->pChannel->channel, pTbl[i]->rank);
        }
#endif

        if (wlanInitScanList(pDevInfo, pDevInfo->staConfig.pClist, &pScanList, FALSE) != A_OK) {
            uiPrintf("wlanScanMode: wlanInitScanList fails \n");
            return pCv;
        }
        
        /*
         * Store the sorted BSS channel list to the new scan list
         */
        pScanList->listSize = 0;
        for (i = numBss - 1; i >= 0; i--) {
            CHAN_VALUES *pCv1 = pTbl[i]->pChannel;
            schan  = pTbl[i]->pChannel->channel;
            cflags = pTbl[i]->pChannel->channelFlags & CHANNEL_ALL;
            match  = 0;

            /* Skip all the "don't scan" channels */
            if (pTbl[i]->pChannel->channelFlags & CHANNEL_DONT_SCAN) {
                continue;
            }
                
            for (k = 0; k < pScanList->listSize; k++) {
                CHAN_VALUES *pCv2 = pScanList->pChanArray[k];
                if (wlanIsChannelMatch(pCv1, pCv2)) {
                    match++;
                    break;
                }
            }

            if (schan && !match) {
                pScanList->pChanArray[k] = pTbl[i]->pChannel;
                pScanList->listSize++;
#ifdef DEBUG
                uiPrintf("[%d]: %d, %4x ", k, schan, pScanList->pChanArray[k]->channelFlags);
                wlanPrintFlags(pScanList->pChanArray[k]->channelFlags);
#endif
            }
        }

        /*
         * Store the rest of the channel list to the new scan list
         */
        pSoc = pDevInfo->staConfig.extendedChanMode == 0 ? (WWR_CLIST **)NonEcmSocTbl : (WWR_CLIST **)socTbl;
        for (i = 0; pSoc[i] ; i++) {
            for (j = 0; pSoc[i]->pList[j]; j++) {
                cflags = wlanMode2Cflags(pSoc[i]->mode);
                schan  = pSoc[i]->pList[j];

                for (k = 0; k < pDevInfo->staConfig.pClist->listSize; k++) {
                    CHAN_VALUES *pCv1 = &pDevInfo->staConfig.pClist->chanArray[k];
                    if (schan == pDevInfo->staConfig.pClist->chanArray[k].channel &&
                        !(pDevInfo->staConfig.pClist->chanArray[k].channelFlags & CHANNEL_DONT_SCAN))
                    {
                        match = 0;
                        for (m = 0; m < pScanList->listSize; m++) {
                            CHAN_VALUES *pCv2 = pScanList->pChanArray[m];
                            if (wlanIsChannelMatch(pCv1, pCv2)) {
                                match++;
                                break;
                            }
                        }
                        if (!match) {
                            pScanList->pChanArray[m] = &pDevInfo->staConfig.pClist->chanArray[k];
                            pScanList->listSize++;
#ifdef  DEBUG
                            uiPrintf("[%d]: %d, %4x ", m, schan, pScanList->pChanArray[m]->channelFlags);
                            wlanPrintFlags(pScanList->pChanArray[m]->channelFlags);
#endif
                        }
                        break;
                    }
                }
            }
        }

        if (pCsInfo->bkScanList) {
            wlanFreeScanList(pCsInfo->bkScanList);
        }

        pCsInfo->bkScanList = pScanList;
        pCv = pScanList->pChanArray[0];
    }

    A_DRIVER_FREE(pTbl, sizeof(BSSDESCR *) * numBss);

    return pCv;
}

/***********************************************************
 * wlanInitScanList
 *
 * Create a scanList which points into the selected portion of
 * the given channelList
 */
A_STATUS
wlanInitScanList(WLAN_DEV_INFO *pDevInfo, WLAN_CHANNEL_LIST *pCListFrom,
                 WLAN_SCAN_LIST **pSListTo, A_BOOL hiddenOnly)
{
    WLAN_CHANNEL  schan;
    WLAN_CFLAGS   cflags;
    A_STATUS      status = A_OK;
    A_UINT16      i, ia, j, k;
    WWR_CLIST     **pSoc;

    
    ASSERT(pDevInfo);
    ASSERT(*pSListTo == NULL);

#ifdef  DEBUG
    if (wlanScanDebugLevel) {
        uiPrintf("wlanInitScanList: listSize: %d\n", pCListFrom->listSize);
    }
#endif
    /*
     * Even though it uses the number of channel entries to
     * call wlanAllocScanList to allocate the scan list, the scan
     * list might end up being a subset of the channel list
     * because of adhoc mode or country code restriction
     */
    status = wlanAllocScanList(pSListTo, pCListFrom->listSize);
    if (status != A_OK) {
        return status;
    }

    /*
     * Copy everything verbatim for user defined clist or
     * DEBUG country code
     */
    if (pDevInfo->staConfig.userClist ||
        pDevInfo->staConfig.countryCode == CTRY_DEBUG)
    {
        for (i = 0; i < pCListFrom->listSize; i++) {
            (*pSListTo)->pChanArray[i] = &pCListFrom->chanArray[i];
        }
        (*pSListTo)->listSize = pCListFrom->listSize;
        return status;
    }

    /* Arrange the scanning order of channels in the scan list */
    ia = 0;
    pSoc = pDevInfo->staConfig.extendedChanMode == 0 ? (WWR_CLIST **)NonEcmSocTbl : (WWR_CLIST **)socTbl;
    for (i = 0; pSoc[i] ; i++) {
        for (j = 0; pSoc[i]->pList[j]; j++) {
            CHAN_VALUES cv1;
            cflags = wlanMode2Cflags(pSoc[i]->mode);
            schan  = pSoc[i]->pList[j];
            cv1.channel      = schan;
            cv1.channelFlags = cflags;            

            for (k = 0; k < pCListFrom->listSize; k++) {
                CHAN_VALUES *pChan = &pCListFrom->chanArray[k];

                if (schan == pChan->channel &&
                    !(pChan->channelFlags & CHANNEL_DONT_SCAN))
                {
                    if (hiddenOnly && !pChan->bssSeenHere) {
                        break;
                    }
                    if (wlanIsChannelMatch(&cv1, pChan)) {
                        (*pSListTo)->pChanArray[ia++] = pChan;
#ifdef  DEBUG
                        if (wlanScanDebugLevel) {
                            uiPrintf("[%2d]: %d (%3d)", ia, schan, wlanConvertGHztoCh(schan, pChan->channelFlags));
                            wlanPrintFlags(pChan->channelFlags);
                        }
#endif
                        break;
                    }
                }
            }
        }
    }

#ifdef  DEBUG
    if (wlanScanDebugLevel) {
        uiPrintf("wlanInitScanList: new scan count: %d, listSize: %d\n", ia, pCListFrom->listSize);
    }
#endif

    (*pSListTo)->listSize = ia;

    return status;
}

/***********************************************************
 * wlanInitSingleScanList
 *
 * Create a scanList which points into the selected portion of
 * the given channelList
 */
A_STATUS
wlanInitSingleScanList(CHAN_VALUES *pChannelFrom, WLAN_SCAN_LIST **pSListTo)
{
    A_STATUS status = A_OK;

    ASSERT(*pSListTo == NULL);

    status = wlanAllocScanList(pSListTo, 1);
    if (status != A_OK) {
        return status;
    }

    uiPrintf("InitSingleScan -- %d, %x ", pChannelFrom->channel,
             pChannelFrom->channelFlags);
    wlanPrintFlags(pChannelFrom->channelFlags);
    ASSERT(!(pChannelFrom->channelFlags & CHANNEL_DONT_SCAN));
    (*pSListTo)->pChanArray[0] = pChannelFrom;
    (*pSListTo)->listSize      = 1;

    return status;
}

/***********************************************************
 * wlanAllocScanList
 *
 * Allocates the number of ptr entries into the scan list
 */
A_STATUS
wlanAllocScanList(WLAN_SCAN_LIST **pSList, A_UINT16 entries)
{
    int size;

    /* Allocate the Channel List */
    size    = sizeof(WLAN_SCAN_LIST) + (entries * sizeof(CHAN_VALUES *));
    *pSList = (WLAN_SCAN_LIST *)A_DRIVER_MALLOC(size);

    if (!(*pSList)) {
        uiPrintf("wlanAllocScanList: Could not allocate memory for scan list of size %d entries\n",
                 entries);
        return A_NO_MEMORY;
    }

    A_MEM_ZERO(*pSList, size);
    (*pSList)->pChanArray = (CHAN_VALUES **)(((A_UINT8 *)*pSList) + sizeof(WLAN_SCAN_LIST));

    return A_OK;
}

/***********************************************************
 * wlanFreeScanList
 *
 * Frees the scan list
 */
A_STATUS
wlanFreeScanList(WLAN_SCAN_LIST *pSList)
{
    A_UINT16 entries;

    if (pSList) {
        entries = pSList->listSize;
        A_DRIVER_FREE(pSList, sizeof(WLAN_SCAN_LIST) + (entries * sizeof(CHAN_VALUES *)));
        pSList = NULL;
    }

    return A_OK;
}

/***********************************************************
 * wlanGetChannelPtr
 *
 * Find this channel's entry in the channel list
 * Returns NULL if no channel ptr is found in the list
 */
CHAN_VALUES *
wlanGetChannelPtr(WLAN_CHANNEL_LIST *pCList, WLAN_CHANNEL channel, A_UINT16 modeSelect)
{
    int i, j;

    ASSERT(pCList);

    /* Match all selected modes to find the channel flags */
    for (i = 0; i < TOTAL_CLIST_MODES; i++) {
        if (modeSelect & modeTable[i].modeSelect) {
            for (j = 0; j < pCList->listSize; j++) {
                /* If the same flags exist then we've hit the right channel */
                if ( (pCList->chanArray[j].channelFlags & CHANNEL_ALL) ==
                     modeTable[i].channelFlags )
                {
                    if (pCList->chanArray[j].channel == channel) {
                        return &pCList->chanArray[j];
                    }
                }
            }
        }
    }

    return NULL;
}

/***********************************************************
 * wlanIsValidChannel
 *
 * Search the regulatory table to see if this channel exists
 * for this device
 */
A_BOOL
wlanIsValidChannel(WLAN_DEV_INFO *pDevInfo, CTRY_CODE countryCode, A_UINT16 modeSelect, WLAN_CHANNEL channel)
{
    int mode;

    /* Match all selected modes to find the channel flags */
    for (mode = 0; mode < TOTAL_CLIST_MODES; mode++) {
        if (modeSelect & modeTable[mode].modeSelect) {
            const REG_DMN_WM_FREQ_TABLE *pCcTable;
            WLAN_CHANNEL    cSpread, schan, loChan, hiChan;
            int             j;
            WLAN_CFLAGS     cflags;

            cflags   = modeTable[mode].channelFlags;
            pCcTable = wlanGetCcRdTbl(pDevInfo, countryCode, cflags);
            cSpread  = wlanGetChannelSpread(cflags);

            loChan = IS_CHAN_5GHZ(cflags) ?
                     (WLAN_CHANNEL)pDevInfo->devCap.low5GhzChan :
                     (WLAN_CHANNEL)pDevInfo->devCap.low2GhzChan;

            hiChan = IS_CHAN_5GHZ(cflags) ?
                     (WLAN_CHANNEL)pDevInfo->devCap.high5GhzChan :
                     (WLAN_CHANNEL)pDevInfo->devCap.high2GhzChan;

            for (j = 0; j < pCcTable->entries; j++) {
                /* Find this channel's entry */
                for (schan = pCcTable->chanGroup[j].lowChannel;
                     schan <= pCcTable->chanGroup[j].highChannel;
                     schan += (IS24DEBUG(schan) ? HIGH2GHZ_CHANNEL_SEPARATION : cSpread))
                {
                    if (wlanVerifyChan4Ecm(pDevInfo, schan, cflags) == TRUE &&
                        schan >= loChan && schan <= hiChan && schan == channel) {
                        return TRUE;
                    }
                }
            }
            break;
        }
    }
    return FALSE;
}


/***********************************************************
 * wlanRdTblPtr
 *
 * If super domain is configured in the EEPROM but the
 * the country code remains undefined  and we can't found
 * any 11d beacon or any info that can tell us what the
 * regulatory domain is.
 * Then use the channel to track down a regulatory domain
 * table.
 *
 * Called by:
 * - wlanGetAntennaReduction
 * - wlanGetCtl
 * - wlanGetChannelPower
 *
 */
static const REG_DMN_WM_FREQ_TABLE *
wlanRdTblPtr(WLAN_DEV_INFO *pDevInfo, CHAN_VALUES *pChannel)
{
    const REG_DMN_WM_FREQ_TABLE *pCcTbl;
    WLAN_CFLAGS cflags, saveflags;
    int         i;

    cflags = pChannel->channelFlags;

    /*
     * For dynamic turbo, use base channel for rd table lookup if no country.
     * This is because table lookup is is based on channel flags rather than
     * driver mode.  This depends on the base channel being same as dynamic turbo.
     * This will need to change when the channel flag / wireless mode / driver state
     * is re-architected.
     */
    saveflags = cflags;
    if (pDevInfo->turboPrimeInfo.turboPrimeAllowed &&
        IS_CHAN_TURBO(cflags))
    {
        cflags &= ~CHANNEL_TURBO;
    }

    if (pDevInfo->staConfig.countryCode == CTRY_DEFAULT) {
        REG_DOMAIN  rd;

        rd = wlanHalRD(pDevInfo);
        rd = (rd & COUNTRY_ERD_FLAG) ? 0 : rd & SUPER_DOMAIN_MASK;
        if (rd) {
            for (i = 0; i < NUM_OF_SUPER_DOMAINS; i++) {
                if (superDomainTbl[i].domain == rd) {
                    int ij, j, numRd;

                    pCcTbl = wlanGetRdmfTable(cflags);
                    numRd  = wlanGetRdmfTblLen(cflags);
                    for (ij = 0; ij < numRd; ij++) {
                        for (j = 0; j < pCcTbl[ij].entries; j++) {
                            if ((pChannel->channel >= pCcTbl[ij].chanGroup[j].lowChannel) &&
                                (pChannel->channel <= pCcTbl[ij].chanGroup[j].highChannel))
                            {
                                return &pCcTbl[ij];
                            }
                        }
                    }
                }
            }
        }
    }

    /*
     * For dynamic turbo, use base channel for rd table lookup.  This is
     * because table lookup is is based on channel flags rather than driver mode.
     * In the case lookup is being done based on country, need additional check
     * to ensure the rd is valid for turbo first, then use base to get correct channel
     * info.  This is related to the fact that dynamic turbo uses the same channel
     * center frequencies as base mode.
     * This will need to change when the channel flag / wireless mode / driver state
     * is re-architected.
     */
    if (pDevInfo->turboPrimeInfo.turboPrimeAllowed &&
        IS_CHAN_TURBO(saveflags) &&
        regDmnNone == wlanGetRdTbl(pDevInfo, pDevInfo->staConfig.countryCode, saveflags))
    {
        return regDmnNone;
    }

    return wlanGetRdTbl(pDevInfo, pDevInfo->staConfig.countryCode, cflags);
}


/***********************************************************
 * wlanGetAntennaReduction
 *
 * Find the max allowed antenna gain and apply any regulatory
 * domain specific changes.
 *
 * NOTE: a negative reduction is possible in RD's that only
 * measure radiated power (e.g., ETSI) which would increase
 * that actual conducted output power (though never beyond
 * the calibrated target power).
 */
A_INT8
wlanGetAntennaReduction(WLAN_DEV_INFO *pDevInfo, CHAN_VALUES *pChannel, A_INT8 twiceAntennaGain)
{
    const REG_DMN_WM_FREQ_TABLE *pCcTable;
    A_INT8  antennaMax;
    int     j;

    pCcTable = wlanRdTblPtr(pDevInfo, pChannel);
    for (j = 0; j < pCcTable->entries; j++) {
        /* Find this channel's entry */
        if ((pChannel->channel >= pCcTable->chanGroup[j].lowChannel) &&
            (pChannel->channel <= pCcTable->chanGroup[j].highChannel))
        {
            antennaMax = twiceAntennaGain - (pCcTable->chanGroup[j].antennaMax * 2);
            return (antennaMax < 0) ? 0 : antennaMax;
        }
    }

    /* Failed to find the correct index - may be a debug channel */
    return 0;
 }

/***********************************************************
 * wlanGetCtl
 *
 * Search the regulatory table and return the test group
 *
 * Called by ar5211Reset when changing channel
 *
 * TODO: CTL for 11B CommonMode when regulatory domain is unknown
 *
 */
A_UINT8
wlanGetCtl(WLAN_DEV_INFO *pDevInfo, CHAN_VALUES *pChannel)
{
    const REG_DMN_WM_FREQ_TABLE *pCcTbl;
    WLAN_CFLAGS cflags = pChannel->channelFlags;

    /* Special CTL to signify WWR SKU without a known country */
    if ((pDevInfo->staConfig.countryCode == CTRY_DEFAULT) && wlanIsWwrSKU(pDevInfo)) {
        if (IS_CHAN_B(cflags)) {
            return SD_NO_CTL | CTL_11B;
        } else if (IS_CHAN_G_OR_108G(cflags)) {
            return SD_NO_CTL | CTL_11G;
        } else if (IS_CHAN_T(cflags)) {
            return SD_NO_CTL | CTL_TURBO;
        } else {
            return SD_NO_CTL | CTL_11A;
        }
    }

    pCcTbl = wlanRdTblPtr(pDevInfo, pChannel);
    if (IS_CHAN_2GHZ(cflags) && IS_CHAN_OFDM(cflags)) {
        if ((pCcTbl->conformanceTestLimit & 0xf) == CTL_11B) {
            return (pCcTbl->conformanceTestLimit & ~0xf) | CTL_11G;
        }
    }
    return pCcTbl->conformanceTestLimit;
}

/***********************************************************
 * wlanGetChannelPower
 *
 * Search the regulatory table and return the channel's power
 * (in dBm)
 *
 * For 802.11d, use the 802.11d max tx power setting if the
 * channel matches and its value is less than the one in table,
 * For common mode, use the tx power setting for that frequency
 * Otherwise, use the regulatory domain's table setting
 *
 */
A_UINT8
wlanGetChannelPower(WLAN_DEV_INFO *pDevInfo, CHAN_VALUES *pChannel)
{
    const REG_DMN_WM_FREQ_TABLE *pCcTable;
    int       j;
    A_UINT16  cflags, wchan;


    wchan    = pChannel->channel;
    cflags   = pChannel->channelFlags;
    pCcTable = wlanRdTblPtr(pDevInfo, pChannel);

    if (pDevInfo->staConfig.sku.v.b.commonMode) {
        if (cflags & CHANNEL_2GHZ) {
            return 18;
        }
        for (j = 0; j < CM_PWR_TBL_LEN; j++) {
            if (wchan >= cmPwrTbl[j].lchan && wchan < cmPwrTbl[j].hchan) {
                return cmPwrTbl[j].pwrLvl;
            }
        }
    }

    for (j = 0; j < pCcTable->entries; j++) {
        /* Find this channel's entry */
        if (wchan >= pCcTable->chanGroup[j].lowChannel &&
            wchan <= pCcTable->chanGroup[j].highChannel)
        {
            if (pDevInfo->staConfig.sku.v.b.cc11d) {
                COUNTRY_INFO_LIST *pClist;
                A_UINT16          chan;
                int               k, len, numChan;
                A_UINT8           maxTxPwr;

                pClist = &pDevInfo->staConfig.sku.cc11dInfo;
                len    = pClist->length - 3;

                for (k = 0; len >= 3; len -= 3, k++) {
                    chan     = pClist->subBand[k].firstChannel;
                    maxTxPwr = pClist->subBand[k].maxTxPwr;

                    for (numChan = 0; numChan < pClist->subBand[k].numChannel; numChan++) {
                        if (maxTxPwr && maxTxPwr <= pCcTable->chanGroup[j].powerDfs &&
                            wchan == wlanConvertChtoGHz(chan, cflags))
                        {
                            return maxTxPwr;
                        }
                    }
                }
            }
            return pCcTable->chanGroup[j].powerDfs;
        }
    }

    /* Failed to find the correct index - may be a debug channel */
    uiPrintf("GetChannelPower -- %d entries, %d, cflags %x, cc: %d\n",
             pCcTable->entries, wchan, cflags, pDevInfo->staConfig.countryCode);

    return 5;
}


/***********************************************************
 * wlanPrintChannelList
 *
 * Print out the channel list based only on the given reg
 * domain and wireless mode(s)
 */
void
wlanPrintChannelList(WLAN_DEV_INFO *pDevInfo, CTRY_CODE countryCode,
                     A_UINT16 wirelessMode, A_UINT16 enableOutdoor)
{
    const REG_DMN_WM_FREQ_TABLE *pCcTable;
    int             j, k, newPrint = 0;
    A_UINT16        cSpread, schan;
    WLAN_CHANNEL    loChan = 0, hiChan = 0;
    A_UINT16        maxRdChannel = 7000;
    WLAN_CFLAGS     cflags;
    A_UINT32        wMode;

    if (!enableOutdoor && countryCode == CTRY_UNITED_STATES) {
        maxRdChannel = FCC_OUTDOOR_FIRST_FREQ;
    }

    for (k = 0; k < TOTAL_CLIST_MODES; k++) {
        if (!(modeTable[k].modeSelect & wirelessMode)) {
            /*
             * Corresponding bit in channelFlags is not set.
             * We don't want add that type of channel.
             */
            continue;
        }

        cflags   = modeTable[k].channelFlags;
        pCcTable = wlanGetRdTbl(pDevInfo, countryCode, cflags);
        cSpread  = wlanGetChannelSpread(cflags);

        wMode = pDevInfo->devCap.wirelessModes;
        /* Skip channel modes not supported by this hardware */
        if (!(wMode & modeTable[k].modeSelect)) {
            continue;
        }

        loChan = IS_CHAN_5GHZ(cflags) ?
                 (WLAN_CHANNEL)pDevInfo->devCap.low5GhzChan :
                 (WLAN_CHANNEL)pDevInfo->devCap.low2GhzChan;

        hiChan = IS_CHAN_5GHZ(cflags) ?
                 (WLAN_CHANNEL)pDevInfo->devCap.high5GhzChan :
                 (WLAN_CHANNEL)pDevInfo->devCap.high2GhzChan;


        uiPrintf("\t");
        for (j = 0; j < pCcTable->entries; j++) {
            /* Find this channel's entry */
            for (schan = pCcTable->chanGroup[j].lowChannel;
                 schan <= pCcTable->chanGroup[j].highChannel;
                 schan += (IS24DEBUG(schan) ? HIGH2GHZ_CHANNEL_SEPARATION : cSpread))
            {
                if (wlanVerifyChan4Ecm(pDevInfo, schan, modeTable[k].channelFlags) == TRUE &&
                    schan >= loChan && schan <= hiChan && schan <= maxRdChannel) {
                    uiPrintf("%04d%s", schan, (++newPrint % 8) ? " " : "\n\t");
                }
            }
        }
        uiPrintf("\n");
    }
}

/***********************************************************
 * wlanGetCountryCodeTableSize
 *
 * Return the static size of the entire country code table
 */
A_UINT16
wlanGetCountryCodeTableSize(WLAN_DEV_INFO *pDevInfo)
{
    A_UINT16 eepRdCc, totalCountries = 0, i;

    eepRdCc = wlanEepromCcRdGet(pDevInfo);

    if (eepRdCc & COUNTRY_ERD_FLAG) {
        /* EEP setting is a country - config shall match */
        totalCountries = 1;
    } else if (eepRdCc == DEBUG_REG_DMN || eepRdCc == NO_ENUMRD) {
        /* Set to Debug or AllowAnyCountry - allow any setting - 2 to remove Debug */
        totalCountries = NUM_OF_COUNTRIES - 1;
    } else {
        for (i = 1; i < NUM_OF_COUNTRIES; i++) {
            if (allCountries[i].regDmnEnum == eepRdCc) {
                totalCountries++;
            }
        }
    }

    ASSERT(totalCountries);
    return totalCountries;
}

/***********************************************************
 * wlanGetIndexedCountryCodeName
 *
 * Return a ptr to the country name at the given index
 */
const A_UCHAR *
wlanGetIndexedCountryCodeName(WLAN_DEV_INFO *pDevInfo, int index)
{
    A_UINT16       eepRdCc, localIndex = 0, i;
    const A_UCHAR  *name = NULL;

    eepRdCc = wlanEepromCcRdGet(pDevInfo);

    if (eepRdCc & COUNTRY_ERD_FLAG) {
        /* EEP setting is a country - config shall match */
        ASSERT(index == 0);
        name = allCountries[wlanGetCtryIdx((A_UINT16)(eepRdCc & ~COUNTRY_ERD_FLAG))].name;
    } else if (eepRdCc == DEBUG_REG_DMN || eepRdCc == NO_ENUMRD) {
        /* Set to Debug or AllowAnyCountry - allow any setting - 2 to remove Debug */
        ASSERT(index < NUM_OF_COUNTRIES - 1);
        name = allCountries[index + 1].name;
    } else {
        for (i = 1; i < NUM_OF_COUNTRIES; i++) {
            if (allCountries[i].regDmnEnum == eepRdCc) {
                if (index == localIndex) {
                    name = allCountries[i].name;
                    break;
                }
                localIndex++;
            }
        }
    }

    ASSERT(name);
    return name;
}

/***********************************************************
 * wlanGetIndexedCountryCodeIsoName
 *
 * Return the static size of the entire country code table
 */
const A_UCHAR *
wlanGetIndexedCountryCodeIsoName(WLAN_DEV_INFO *pDevInfo, int index)
{
    A_UINT16       eepRdCc, localIndex = 0, i;
    const A_UCHAR  *isoName = NULL;

    eepRdCc = wlanEepromCcRdGet(pDevInfo);

    if (eepRdCc & COUNTRY_ERD_FLAG) {
        /* EEP setting is a country - config shall match */
        ASSERT(index == 0);
        isoName = allCountries[wlanGetCtryIdx((A_UINT16)(eepRdCc & ~COUNTRY_ERD_FLAG))].isoName;
    } else if (eepRdCc == DEBUG_REG_DMN || eepRdCc == NO_ENUMRD) {
        /* Set to Debug or AllowAnyCountry - allow any setting - 2 to remove Debug */
        ASSERT(index < NUM_OF_COUNTRIES - 1);
        isoName = allCountries[index + 1].isoName;
    } else {
        for (i = 1; i < NUM_OF_COUNTRIES; i++) {
            if (allCountries[i].regDmnEnum == eepRdCc) {
                if (index == localIndex) {
                    isoName = allCountries[i].isoName;
                    break;
                }
                localIndex++;
            }
        }
    }

    ASSERT(isoName);
    return isoName;
}

/***********************************************************
 * wlanGetCountryCodeIndex
 *
 * Find the table index matching the set country code
 */
A_UINT16
wlanGetCountryCodeIndex(WLAN_DEV_INFO *pDevInfo, CTRY_CODE countryCode)
{
    A_UINT16 eepRdCc, localIndex = 0, i;

    /* Set country display to NA if debug is present */
    if (countryCode == CTRY_DEBUG) {
        return 0;
    }

    eepRdCc = wlanEepromCcRdGet(pDevInfo);

    if (eepRdCc & COUNTRY_ERD_FLAG) {
        /* EEP setting is a country - config shall match */
        return 0;
    } else if ((eepRdCc == DEBUG_REG_DMN) || (eepRdCc == NO_ENUMRD)) {
        /* Set to Debug or AllowAnyCountry - allow any setting -  remove Debug */
        return wlanGetCtryIdx(countryCode) - 1;
    } else {
        for (i = 1; i < NUM_OF_COUNTRIES; i++) {
            if (allCountries[i].regDmnEnum == eepRdCc) {
                if (countryCode == allCountries[i].countryCode) {
                    return localIndex;
                }
                localIndex++;
            }
        }
    }

    uiPrintf("wlanGetCountryCodeIndex: No matching country code index %d found!\n", countryCode);
    return 0;
}

/***********************************************************
 * wlanGetCountryCodeName
 *
 * Return a pointer to the ISO name of the country
 */
const A_UCHAR *
wlanGetCountryCodeName(CTRY_CODE countryCode)
{
    int i;

    for (i = 0; i < NUM_OF_COUNTRIES; i++) {
        if (countryCode == allCountries[i].countryCode) {
            return allCountries[i].isoName;
        }
    }

    uiPrintf("wlanGetCountryCodeName: Could not find matching name for country code\n");

    return allCountries[0].isoName;
}

/***********************************************************
 * wlanGetCountryCodeByName
 *
 * Return the country code that matches the given string
 * includeLocal allows matching of the internal data types such
 * as the DB (debug) country and NA (no country present) types.
 */
CTRY_CODE
wlanGetCountryCodeByName(A_UCHAR *countryCodeName, A_BOOL includeLocal)
{
    int i = 0;

    if (!includeLocal) {
        i = CTRY_ONLY_INDEX;
    }

    for (; i < NUM_OF_COUNTRIES; i++) {
        if (A_STRNCMP(countryCodeName, allCountries[i].isoName, 2) == 0) {
            return allCountries[i].countryCode;
        }
    }

    return wlanXlatNonIsoCcode(countryCodeName);
}

/***********************************************************
 * wlanIsCountryCodeValid
 *
 * Returns TRUE if cfg'ed country is allowed by the EEPROM setting
 */
A_BOOL
wlanIsCountryCodeValid(WLAN_DEV_INFO *pDevInfo, CTRY_CODE countryCode)
{
    A_UINT16 eepRdCc, i;

    /* Default setting requires no checks */
    if (countryCode == CTRY_DEFAULT) {
        return TRUE;
    }
#ifdef ALLOW_DEBUG_COUNTRY
    if (countryCode == CTRY_DEBUG) {
        return TRUE;
    }
#endif

    eepRdCc = wlanEepromCcRdGet(pDevInfo);

    if (eepRdCc & COUNTRY_ERD_FLAG) {
        /* EEP setting is a country - config shall match */
        return (countryCode == (eepRdCc & ~COUNTRY_ERD_FLAG)) ? TRUE : FALSE;
    } else if ((eepRdCc == DEBUG_REG_DMN) || (eepRdCc == NO_ENUMRD)) {
        /* Set to Debug or AllowAnyCountry mode - allow any setting */
        return TRUE;
    } else {
        for (i = 0; i < NUM_OF_COUNTRIES; i++) {
            if ((countryCode == allCountries[i].countryCode) &&
                (allCountries[i].regDmnEnum == eepRdCc))
            {
                return TRUE;
            }
        }
    }

    return FALSE;
}

/***********************************************************
 * wlanPrintCountryCodeList
 *
 * Prints the list of available countries for this EEPROM setting
 */
void
wlanPrintCountryCodeList(WLAN_DEV_INFO *pDevInfo)
{
    int      i;
    A_UINT16 eepRdCc, countryCode;

#ifdef ALLOW_DEBUG_COUNTRY
    uiPrintf("DB - DEBUG\n");
#endif

    eepRdCc = wlanEepromCcRdGet(pDevInfo);
    if ((eepRdCc == DEBUG_REG_DMN) || (eepRdCc == NO_ENUMRD)) {
        /* Print all countries */
        for (i = 0; i < NUM_OF_COUNTRIES; i++) {
            uiPrintf("%s - %s\n", allCountries[i].isoName, allCountries[i].name);
        }
    } else if (eepRdCc & COUNTRY_ERD_FLAG) {
        /* Print the country that matches the hardware EEPROM setting */
        countryCode = eepRdCc & ~COUNTRY_ERD_FLAG;
        for (i = 0; i < NUM_OF_COUNTRIES; i++) {
            if (countryCode == allCountries[i].countryCode) {
                uiPrintf("%s - %s\n", allCountries[i].isoName, allCountries[i].name);
                break;
            }
        }
        if (i == NUM_OF_COUNTRIES) {
            uiPrintf("Hardware setting does not match a known value\n");
        }
    } else {
        /* Print only countries that fall in the set reg domain */
        for (i = 0; i < NUM_OF_COUNTRIES; i++) {
            if (eepRdCc == allCountries[i].regDmnEnum) {
                uiPrintf("%s - %s\n", allCountries[i].isoName, allCountries[i].name);
            }
        }
    }
}

/***********************************************************
 * wlanGetDefaultCountry
 *
 * Returns country from the EEPROM or CTRY_DEFAULT
 */
A_UINT16
wlanGetDefaultCountry(WLAN_DEV_INFO *pDevInfo)
{
    A_UINT16 eepRdCc, i;
    A_UINT16 ccode;

    eepRdCc = wlanEepromCcRdGet(pDevInfo);
    if (eepRdCc & COUNTRY_ERD_FLAG) {
        ccode = eepRdCc & ~COUNTRY_ERD_FLAG;
        for (i = 0; i < NUM_OF_COUNTRIES; i++) {
            if (ccode == allCountries[i].countryCode) {
                return ccode;
            }
        }
    }

    ccode = wlanGetSingletonCountryRD(eepRdCc);

    return (ccode ? ccode : CTRY_DEFAULT);
}


/***********************************************************
 * wlanIs11aTurboAllowed
 *
 * Checks it is ok to use turbo mode
 *
 * If global SKU, then verify whether turbo is allowed or not
 * Otherwise, get regulatrory domain or country code.
 * If country code, then extract the country's regulatory domain
 * that country.
 * Get the 5GHz of the regulatory domain and check if the
 * regulatory domain is in the turbo regulatory domain table
 * or not.
 */
A_BOOL
wlanIs11aTurboAllowed(WLAN_DEV_INFO *pDevInfo)
{
    const REG_DMN_WM_FREQ_TABLE *pTbl;
    A_UINT16  ccode, eepRdCc, rd;
    int       i, j, tblLen;


    if (wlanIsWwrSKU(pDevInfo)) {
        rd = wlanHalRD(pDevInfo);

        for (i = 0; i < NUM_OF_SUPER_DOMAINS; i++) {
            if (superDomainTbl[i].domain == rd) {
                for (j = 0; j < superDomainTbl[i].rclTblSize; j++) {
                    if (superDomainTbl[i].RclTbl[j]) {
                        if (superDomainTbl[i].RclTbl[j]->mode & MODE_SELECT_TURBO) {
                            return TRUE;
                        }
                    }
                }
            }
        }
        return FALSE;
    }

    eepRdCc = wlanEepromCcRdGet(pDevInfo);
    if (eepRdCc & COUNTRY_ERD_FLAG) {
        ccode = eepRdCc & ~COUNTRY_ERD_FLAG;
        for (i = 0; i < NUM_OF_COUNTRIES; i++) {
            if (ccode == allCountries[i].countryCode) {
                if (allCountries[i].allow11aTurbo == YES) {
                    return TRUE;
                }
            }
        }
        return FALSE;
    }

    rd = allEnumRds[wlanGetRDTblIdx(eepRdCc)].regDmnWirelessMode5;

    pTbl   = wlanGetRdmfTable(CHANNEL_T);
    tblLen = wlanGetRdmfTblLen(CHANNEL_T);

    for (i = 0; i < tblLen; i++) {
        if (pTbl[i].regDmnWirelessMode == rd) {
            return TRUE;
        }
    }

    return FALSE;
}

/***********************************************************
 * wlanIsCountryAllow11aTurbo
 *
 * Checks this country allows turbo mode in 5Ghz
 *
 */
A_BOOL
wlanIsCountryAllow11aTurbo(WLAN_DEV_INFO *pDevInfo, CTRY_CODE ccode)
{
    int      i;

    if (ccode == CTRY_DEFAULT) {
        return wlanIs11aTurboAllowed(pDevInfo);
    }
    
    for (i = 0; i < NUM_OF_COUNTRIES; i++) {
        if (ccode == allCountries[i].countryCode) {
            return (allCountries[i].allow11aTurbo == YES) ? TRUE : FALSE;
        }
    }

    uiPrintf("wlanIsCountryAllow11aTurbo -- wlan%d: Invalid country code: %d\n", pDevInfo->devno, ccode);
    ASSERT(0);

    return FALSE;
}

/***********************************************************
 * wlanIsCountryAllow11gTurbo
 *
 * Checks this country allows turbo mode in 2.4Ghz
 *
 */
A_BOOL
wlanIsCountryAllow11gTurbo(WLAN_DEV_INFO *pDevInfo, CTRY_CODE ccode)
{
    int      i;
    
    for (i = 0; i < NUM_OF_COUNTRIES; i++) {
        if (ccode == allCountries[i].countryCode) {
            return (allCountries[i].allow11gTurbo == YES) ? TRUE : FALSE;
        }
    }

    return FALSE;
}

/***********************************************************
 * wlanGetCountryInfoList
 *
 * Search the eeprom and return the country string +
 * all available subband channel group info, such as first
 * channel number in the group, number of channels in the
 * group, and the max transmit power.
 */
void
wlanGetCountryInfoList(WLAN_DEV_INFO *pDevInfo, COUNTRY_INFO_LIST *pCountryInfo)
{
    const REG_DMN_WM_FREQ_TABLE *pCcTable;

    int          i, regDmnEnumEntries, j = 0;
    A_UINT16     channelSpread;
    WLAN_CHANNEL lowChannel = 0, highChannel = 0;
    A_UINT16     regDmnWirelessMode;
    A_UINT16     countryCode, channelFlags;

    countryCode = pDevInfo->staConfig.countryCode;
    ASSERT(countryCode != CTRY_DEFAULT);
    ASSERT(countryCode != CTRY_DEBUG);

    for (i = 0; i < NUM_OF_COUNTRIES; i++) {
        if (countryCode == allCountries[i].countryCode) {
            break;
        }
    }
    ASSERT(i < NUM_OF_COUNTRIES);

    /* Zero out country Info to clear out values - most importantly any pad */
    A_MEM_ZERO(pCountryInfo, sizeof(COUNTRY_INFO_LIST));

    pCountryInfo->countryString[0] = allCountries[i].isoName[0];
    pCountryInfo->countryString[1] = allCountries[i].isoName[1];
    /*
     * TODO: outdoor/indoor: the question man has asked forever
     */
    pCountryInfo->countryString[2] = ' ';

    channelFlags = pDevInfo->staConfig.pChannel->channelFlags;

    pCcTable           = wlanGetRdmfTable(channelFlags);
    regDmnEnumEntries  = wlanGetRdmfTblLen(channelFlags);
    channelSpread      = wlanGetChannelSpread(channelFlags);
    regDmnWirelessMode = wlanGetWmRD(pDevInfo, countryCode, channelFlags);

    lowChannel = IS_CHAN_5GHZ(channelFlags) ?
                 (WLAN_CHANNEL)pDevInfo->devCap.low5GhzChan :
                 (WLAN_CHANNEL)pDevInfo->devCap.low2GhzChan;

    highChannel = IS_CHAN_5GHZ(channelFlags) ?
                  (WLAN_CHANNEL)pDevInfo->devCap.high5GhzChan :
                  (WLAN_CHANNEL)pDevInfo->devCap.high2GhzChan;


    /* Search through Regulatory Domain (Enum) table for a match */
    for (i = 0; i < regDmnEnumEntries; i++) {
        if (pCcTable[i].regDmnWirelessMode == regDmnWirelessMode) {
            for (j = 0; j < pCcTable[i].entries; j++) {
                if (pCcTable[i].chanGroup[j].lowChannel >= lowChannel &&
                    pCcTable[i].chanGroup[j].highChannel <= highChannel)
                    /*
                     * TODO: Account for no outdoor channels?
                     */
                {
                    pCountryInfo->subBand[j].firstChannel =
                        (A_UINT8) wlanConvertGHztoCh(pCcTable[i].chanGroup[j].lowChannel, channelFlags);
                    pCountryInfo->subBand[j].numChannel =
                        ((pCcTable[i].chanGroup[j].highChannel - pCcTable[i].chanGroup[j].lowChannel) /
                        channelSpread) + 1;
                    pCountryInfo->subBand[j].maxTxPwr = pCcTable[i].chanGroup[j].powerDfs;
                }
            }
            break;
        }
    }
    ASSERT(i < regDmnEnumEntries);
    pCountryInfo->length = sizeof(pCountryInfo->countryString) +
                           sizeof(A_UINT8) * 3 * j;

    /* Round up size to be even number of bytes for 802.11d */
    pCountryInfo->length = (pCountryInfo->length + 1) & ~0x1;

#if 0
    uiPrintf("--Country Code of Beacon--\n");
    uiPrintf("RAW WORDS\n");
    for (i = 0; i < sizeof(COUNTRY_INFO_LIST) / sizeof(A_UINT32); i++) {
        uiPrintf("\t0x%08X\n", ((A_UINT32 *)(pCountryInfo))[i]);
    }
    uiPrintf("Fields\n");
    uiPrintf("\tLength: %d\n", pCountryInfo->length);
    uiPrintf("\tCountry: %c%c%c\n", pCountryInfo->countryString[0],
             pCountryInfo->countryString[1],
             pCountryInfo->countryString[2]);
    for (i = 0; i < j; i++) {
        uiPrintf("\tFirst Channel %3d\tNum Channels %3d\tChannel Power %3d\n",
        pCountryInfo->subBand[i].firstChannel,
        pCountryInfo->subBand[i].numChannel,
        pCountryInfo->subBand[i].maxTxPwr);
    }
#endif

    return;
}

/***********************************************************
 * wlanGetCountryCodeByRegDmnName
 *
 * Search the regulatory table by name and return the country code
 */
A_UINT16
wlanGetCountryCodeByRegDmnName(A_UCHAR *regName)
{
    int i;

    /* Search through Regulatory Domain (Enum) table for a match */
    for (i = 0; i < NUM_OF_LEGACY_NAMES; i++) {
        if (A_STRCMP(allLegacyNames[i].regDmnName, regName) == 0) {
            return allLegacyNames[i].countryCode;
        }
    }
    return CTRY_INVALID;
}

/***********************************************************
 * wlanGetRegDmnByName
 *
 * Search the regulatory table by name and return the domain
 */
A_UINT16
wlanGetRegDmnByName(A_UCHAR *regName)
{
    int i;

    /* Search through Regulatory Domain (Enum) table for a match */
    for (i = 0; i < NUM_OF_LEGACY_NAMES; i++) {
        if (A_STRCMP(allLegacyNames[i].regDmnName, regName) == 0) {
            return allLegacyNames[i].regDmn;
        }
    }

    /* Previous call to get country code should have stopped this call */
    ASSERT(0);
    return FCC;
}

/***********************************************************
 * wlanPrintRegDmnList
 *
 * Fill the given string with all reg domain names space separated
 */
void
wlanPrintRegDmnList(void)
{
    A_UINT16 i;

    for (i = 0; i < NUM_OF_LEGACY_NAMES; i++) {
        uiPrintf("%s ", allLegacyNames[i].regDmnName);
    }

    uiPrintf("\n");
}

/***********************************************************
 * wlanConvertGHztoCh
 *
 * Convert 2.4 & 5 GHz frequency to IEEE channel number
 * 11B range uses 1 thru 14 with 14 being a special case
 *
 * Return value is a signed 16-bit to represent negative channel
 * values for 4.9 and positive values up to 6 GHz (over 128).
 */
A_INT16
wlanConvertGHztoCh(A_UINT16 freq, WLAN_CFLAGS channelFlags)
{
    A_INT16 channel;

    if (IS_CHAN_2GHZ(channelFlags)) {
        // 2 GHz band
        if ( freq == 2484 ) {
            channel = 14;
        } else if ( freq < 2484 ) {
            channel =  (freq - 2407) / 5;
        } else {
            // additional 2 GHz channels (15-26)
            channel = (( freq - 2512 ) / 20) + 15;
        }
    } else {
        // 5 GHz band, channel 0 = 5 GHz, increments of 5 MHz
        channel = ( freq - 5000 ) / 5;
    }
    return channel;
}


/***********************************************************
 * wlanConvertChtoGHz
 *
 * Convert 2 & 5 GHz channel number to frequency
 */
A_UINT16 wlanConvertChtoGHz(A_INT16 ch, A_UINT16 channelFlags)
{
    A_UINT16 freq;

    if (IS_CHAN_2GHZ(channelFlags)) {
        if ( ch == 14 ) {
            freq = 2484;
        } else if ( ch < 14 ) {
            freq = 2407 + ch * 5;
        } else {
            // additional 2 GHz channels (15-26)
            freq = 2512 + ((ch - 15) * 20);
        }
    } else {
        if((IS_BASE_4GHZ(channelFlags)) || (ch>= 180)) {
            // 4 GHz band
            freq = 4000 + ch * 5;
        } else{
            // 5 GHz band
            freq = 5000 + ch * 5;
        }
    }

    return freq;
}

/***********************************************************
 * wlanEnableDFS
 *
 * Check the EEPROM set regulatory domain and enable DFS if
 * required for operation.
 *
 * RETURNS: TRUE if DFS is enabled
 */
A_BOOL
wlanEnableDFS(WLAN_DEV_INFO *pDevInfo)
{
    A_UINT16    rd;
    WLAN_CFLAGS cflags;

    cflags = pDevInfo->staConfig.phwChannel->channelFlags;
    rd     = wlanGetWmRD(pDevInfo, pDevInfo->staConfig.countryCode, cflags);
    if (wlanIsDfsRD(pDevInfo, rd) == FALSE) {
        return FALSE;
    }

#ifdef DEBUG
    uiPrintf("wlanEnableDFS: Enabling radar detection features\n");
#endif
#ifdef BUILD_AP
    etsiFeaturesEnable(pDevInfo);
#endif

    return TRUE;
}

/***********************************************************
 * wlanIsActiveScanAllowed
 *
 * Return the scantype allowed for the given ERD
 */
A_BOOL
wlanIsActiveScanAllowed(WLAN_DEV_INFO *pDevInfo)
{
    REG_DOMAIN  rd;
    WLAN_CFLAGS cflags;

    ASSERT(pDevInfo->staConfig.pChannel);

    cflags = pDevInfo->staConfig.pChannel->channelFlags & CHANNEL_ALL;
    rd     = wlanGetWmRD(pDevInfo, pDevInfo->staConfig.countryCode, cflags);

    return wlanIsPassiveScanChannel(pDevInfo, rd, pDevInfo->staConfig.pChannel) == TRUE ? FALSE : TRUE;
}

/***********************************************************
 * wlanIsAdHocAllowed
 *
 * Return TRUE if adhoc is allowed else FALSE
 */
A_BOOL
wlanIsAdHocAllowed(WLAN_DEV_INFO *pDevInfo, WLAN_CFLAGS cflags)
{
    REG_DOMAIN rd;
    int i;

    /* Get the country found RD for this mode - see if it's restricted in the disallow table */
    rd = wlanGetWmRD(pDevInfo, pDevInfo->staConfig.countryCode, cflags);
    for (i = 0; i < ADHOC_DISALLOW_RD_TBL_LEN; i++) {
        if ((adHocDisallowRdTbl[i].rd == rd) && (cflags == adHocDisallowRdTbl[i].cflags)) {
            return FALSE;
        }
    }

    /* Check if the base RD supports Adhoc */
    if (wlanUnsupportedAdHocBand(pDevInfo) == cflags) {
        return FALSE;
    }

    switch (cflags) {
        case CHANNEL_A:
        case CHANNEL_T:
            return wlanIs11bOnlyRD(rd) ? FALSE : TRUE;
    }

    return TRUE;
}

/***********************************************************
 * wlanIsTelecSupportRequired
 *
 * Check if Telec (japanese regulatory domain) support is 
 * enforced.
 */
A_BOOL
wlanIsTelecSupportRequired(WLAN_DEV_INFO *pDevInfo)
{
    A_UINT16 i, rd, rdIdx;

    rdIdx = wlanGetRDIdxByCtry(pDevInfo, pDevInfo->staConfig.countryCode);
    rd = allEnumRds[rdIdx].regDmnEnum;

    for (i = 0; i < NFC_RD_TBL_LEN; i++) {
        if (nfcRdTbl[i] == rd) {
            return TRUE;
        }
    }

    return FALSE;
}

/***********************************************************
 * wlanIsNfCheckRequired
 *
 * Return the scantype allowed for the given ERD
 */
A_BOOL
wlanIsNfCheckRequired(WLAN_DEV_INFO *pDevInfo)
{
    A_UINT16 i, rd;

    ASSERT(pDevInfo->staConfig.phwChannel);
    rd = wlanGetWmRD(pDevInfo, pDevInfo->staConfig.countryCode,
                     pDevInfo->staConfig.phwChannel->channelFlags);

    for (i = 0; i < NFC_RD_TBL_LEN; i++) {
        if (nfcRdTbl[i] == rd) {
            return TRUE;
        }
    }
    return FALSE;
}

/***********************************************************
 * wlanIsCountry11g
 *
 * Return TRUE if 11g is allowed else FALSE
 */
A_BOOL
wlanIsCountry11g(CTRY_CODE ccode)
{
    int      i;


    for (i = 0; i < NUM_OF_COUNTRIES; i++) {
        if (ccode == allCountries[i].countryCode) {
            return (allCountries[i].allow11g == YES) ? TRUE : FALSE;
        }
    }

    uiPrintf("wlanIsCountry11g -- Invalid country code: %d\n", ccode);
    ASSERT(0);

    return FALSE;
}

/***********************************************************
 * wlanVerifyChan4Ecm
 *
 * Return TRUE if it is an ECM channel
 */
A_BOOL
wlanVerifyChan4Ecm(WLAN_DEV_INFO *pDevInfo, A_UINT16 channel, WLAN_CFLAGS cf)
{
    if (wlanIs2GHzEcmChan(channel) == TRUE) {
#ifdef BUILD_AP
        if (cf == CHANNEL_G && channel == CHANNEL_14) {
            return FALSE;
        }
#endif
        return pDevInfo->staConfig.extendedChanMode ? TRUE : FALSE;
    }
    return TRUE;
}


/***********************************************************
 * wlanIs2GHzEcmChan
 *
 * Return TRUE if channel 12, 13 or 14
 */
A_BOOL
wlanIs2GHzEcmChan(A_UINT16 channel)
{
    int      i;

    for (i = 0; i < ecmChannelTbl[i]; i++) {
        if (channel == ecmChannelTbl[i]) {
            return TRUE;
        }
    }
    return FALSE;
}

A_BOOL
wlanIsEcmChanPassive(WLAN_DEV_INFO *pDevInfo, A_UINT16 channel)
{
    int      i, j;


    for (i = 0; i < ECM_SKU_TBL_LEN; i++) {
        if (wlanHalRD(pDevInfo) == ecmSkuTbl[i].sku) {
            for (j = 0; j < ECM_CHAN_TBL_LEN; j++) {
                if (channel == ecmSkuTbl[i].tbl[j].chan) {
                    return ecmSkuTbl[i].tbl[j].scanMode;
                }
            }
        }
    }
    return FALSE;
}

/***********************************************************
 * wlanGetWirelessMode
 *
 * Return the radio's frequency specifications that the hardware supports
 * for infrastructure mode.  This is dependent on country code.
 */
A_UINT32
wlanGetWirelessMode(WLAN_DEV_INFO *pDevInfo, CTRY_CODE countryCode)
{
    A_UINT32 mode;

    /* Return minimal config if the device is not attached to get config items up */
    if (pDevInfo == NULL) {
        return MODE_SELECT_11A;
    }

    /* Get modes support by the hardware */
    mode = pDevInfo->devCap.wirelessModes;

    if (mode & MODE_SELECT_TURBO) {
        /* Some countries don't allow turbo */
        if (wlanIsCountryAllow11aTurbo(pDevInfo, countryCode) != TRUE) {
            mode &= ~MODE_SELECT_TURBO;
        }
    }
    if (mode & MODE_SELECT_108G) {
        /* Some countries don't allow turbo G */
        if (wlanIsCountryAllow11gTurbo(pDevInfo, countryCode) != TRUE) {
            mode &= ~MODE_SELECT_108G;
        }
    }
    if (mode & MODE_SELECT_11A) {
        /* Some regulatory domains don't allow 11a */
        if (wlanIs11aAllowed(pDevInfo, countryCode) != TRUE) {
            mode &= ~MODE_SELECT_11A;
        }
    }
    if (mode & MODE_SELECT_11G) {
        /* Some countries don't allow 11g */
        if (wlanIsCountry11g(countryCode) != TRUE) {
            mode &= ~MODE_SELECT_11G;
        }
    }

    // uiPrintf("wlanGetWirelessMode -- cc: %d, mode %x\n", countryCode, mode);

    return mode;
}

/***********************************************************
 * wlanGetAdhocWirelessMode
 *
 * Return the radio's frequency specifications that the hardware supports
 * for adhoc mode.  This is dependent on country code.
 */
A_UINT32
wlanGetAdhocWirelessMode(WLAN_DEV_INFO *pDevInfo, CTRY_CODE countryCode)
{
    A_UINT32 mode;

    /* Get modes supported in infrastructure */
    mode = wlanGetWirelessMode(pDevInfo, countryCode);

    /* Some regulatory domains don't allow adhoc mode at certain frequencies */
    mode &= ~wlanUnsupportedAdHocMode(pDevInfo);

#if 0
    uiPrintf("wlanGetAdhocWirelessMode %x ", mode);
    printmodes((A_UINT16)mode);
    uiPrintf("\n");
#endif

    return mode;
}


/*******************************************************************************
 * AP Scan and related functions
 *******************************************************************************/

#ifdef BUILD_AP

/* Scan time in milliSeconds */
enum {
    AP_ACTIVE_SCAN_MSTIME  = 300,
    AP_PASSIVE_SCAN_MSTIME = 4000
};

/*******************************************************************************
 * bssDescCmp
 *
 * Sort the BSS descriptor elements by their channel
 * - used by qsort in bssDescShow
 */
static int
bssDescCmp(BSSDESCR *p1, BSSDESCR *p2)
{
    if (p1->pChannel->channel > p2->pChannel->channel) {
        return +1;
    } else if (p1->pChannel->channel == p2->pChannel->channel) {
        return 0;
    }
    return -1;
}

/*******************************************************************************
 * bssDescShow
 *
 * Sort and Print the BSS Description set
 */
static void
bssDescShow(BSSDESCR_SET *pBssSet)
{
    BSSDESCR  *pbssDescr;
    int       index, ap = 0, adhoc = 0;

    /* Sort the BSS descriptor elements by their channel */
    qsort(&(pBssSet->bssDescrArray[0]), pBssSet->count,
          sizeof(BSSDESCR), (void *)&bssDescCmp);

    /* Print the BSS descriptors and their info */
    uiPrintf("BSS Index  BSS Type  Channel       RSSI  BSSID              SSID\n");

    for (index = 0; index < pBssSet->count; index++) {
        pbssDescr = &pBssSet->bssDescrArray[index];
        if (pbssDescr->bsstype == INFRASTRUCTURE_BSS) {
            ap++;
        } else {
            adhoc++;
        }
        uiPrintf("[%2d],     %7s   %d.%03d (%3d)   %3d   ", index,
                 pbssDescr->bsstype == INFRASTRUCTURE_BSS ? "AP BSS" : "Ad-Hoc",
                 pbssDescr->pChannel->channel / 1000, pbssDescr->pChannel->channel % 1000,
                 wlanConvertGHztoCh(pbssDescr->pChannel->channel, pbssDescr->pChannel->channelFlags),
                 pbssDescr->rssi);
        printMacAddress(pbssDescr->bssId);
        uiPrintf("  ");
        printSsid(&pbssDescr->ssid);
        uiPrintf("\n");
        if (pbssDescr->ccList.elementID == ELE_COUNTRY_INFO) {
            int  i, len = pbssDescr->ccList.length - 3;
            char *pStr = (char *)&pbssDescr->ccList.countryString;

            uiPrintf("%c%c%c ", *pStr++, *pStr++, *pStr);

            for (i = 0; len >= 3; len -= 3, i++) {
                uiPrintf(" [%3d (%d) %2d %2d]",
                         pbssDescr->ccList.subBand[i].firstChannel,
                         wlanConvertChtoGHz(pbssDescr->ccList.subBand[i].firstChannel,
                                            pbssDescr->pChannel->channelFlags),
                         pbssDescr->ccList.subBand[i].numChannel,
                         pbssDescr->ccList.subBand[i].maxTxPwr);
            }

            uiPrintf("\n");
        }
    }

    uiPrintf("AP: %d, Ad-Hoc: %d. Total BSS: %d\n", ap,
             adhoc, pBssSet->count);
}


/*******************************************************************************
 * bssDesc2Web
 *
 * Sort and stuff the BSS Description set
 */
static void
bssDesc2Web(BSSDESCR_SET *pBssSet)
{
    BSSDESCR  *pbssDescr;
    int index;
    

    /* Sort the BSS descriptor elements by their channel */
    qsort(&(pBssSet->bssDescrArray[0]), pBssSet->count,
          sizeof(BSSDESCR), (void *)&bssDescCmp);

#ifdef INCLUDE_RCW
    for (index = 0; index < pBssSet->count; index++) {
        pbssDescr = &pBssSet->bssDescrArray[index];
        if (pbssDescr->bsstype == INFRASTRUCTURE_BSS) {
            if (siteSurveyCallback) {
                (*siteSurveyCallback) (pbssDescr);
            }
        }
    }
#endif /* INCLUDE_RCW */

}

/*******************************************************************************
 * wlanScanMode
 *
 * Initialize the scan followed by scanning the scanInfo channels
 * using mlmeScanFunction
 */
static A_STATUS
wlanScanMode(WLAN_DEV_INFO *pDevInfo, APSCAN_OP scanOp, SCANTYPE scanType, CHAN_VALUES *pRch, int verbose)
{
    SCAN_INFO      *pScanInfo;
    int            resultCode;
    A_INT32        maxChanTicks, maxChanTime;
    WLAN_SCAN_LIST *pScanList = NULL;
    A_UCHAR        *pName = "Unknown";


    pRch = NULL;

    /* maxChanTime in milliseconds */
    maxChanTime  = (scanType == ACTIVE_SCAN) ? AP_ACTIVE_SCAN_MSTIME : AP_PASSIVE_SCAN_MSTIME;
    maxChanTicks = (maxChanTime * sysClkRateGet()) / 1000;

    /* TODO: commonize these strings for web and other printing */
    switch (pDevInfo->staConfig.pClist->chanArray[0].channelFlags & CHANNEL_ALL) {
    case CHANNEL_A:
        pName = "5GHz 54Mbps (802.11a)";
        break;
    case CHANNEL_T:
        pName = "5GHz 108Mbps (Turbo)";
        break;
    case CHANNEL_B:
        pName = "2.4GHz 11Mbps (802.11b)";
        break;
    case CHANNEL_G:
        pName = "2.4GHz 54Mbps";
        break;
    case CHANNEL_108G:
        pName = "2.4GHz 108Mbps (Turbo)";
        break;
    }

    /* collect all parameters required for mlmeScanRequest */
    if (wlanInitScanList(pDevInfo, pDevInfo->staConfig.pClist, &pScanList, FALSE) != A_OK) {
        uiPrintf("wlanScanMode: wlanInitScanList fails \n");
        return A_EINVAL;
    }

    if (verbose) {
        uiPrintf("\n%s scanning %s channels for %d seconds...\n",
                 scanType == PASSIVE_SCAN ? "Passive" : "Active", pName,
                 (pScanList->listSize * maxChanTime + 999) / 1000);
    }
    halRxFilter(pDevInfo, HAL_RX_BEACON, HAL_RX_FILTER_SET);
    resultCode = wlanMlmeScanRequest(pDevInfo, ANY_BSS,
                                     &pDevInfo->baseBss.bssId, NULL, scanType,
                                     pScanList,
                                     maxChanTime, maxChanTime,
                                     maxChanTime, maxChanTime, FALSE);
    if (resultCode) {
        uiPrintf("Scan Request result %x\n", resultCode);
        halRxFilter(pDevInfo, HAL_RX_BEACON, HAL_RX_FILTER_CLEAR);
        return A_EINVAL;
    }

    pScanInfo = pDevInfo->pscanInfo;
    ASSERT(pScanInfo);

    if (pDevInfo->staConfig.radarDetectEnable) {
        initRadarDetect(pDevInfo);
    }

    /* Scan from current channel index through listSize */
    while (pScanInfo->curChannelIndex < pScanList->listSize) {
        if (isApShutdown() == TRUE) {
            exit(0);
        }

        taskDelay(maxChanTicks);

        /* turn off any more frames coming in */
        halRxFilter(pDevInfo, 0, HAL_RX_FILTER_TEMP);

        /* give up CPU for a fraction of a second. allow received beacons to get processed. */
        taskDelay(5);

        /* re-enable the rx filter */
        halRxFilter(pDevInfo, 0, HAL_RX_FILTER_RESTORE);

        mlmeScanFunction(pDevInfo, SF_ADVANCE_NOW);

        /*
         * mlmeScanFunction() might increment the curChannelIndex (and not actually scan), so
         * we check to make sure we haven't advanced past the list.  One of these days the AP
         * will use the same interface to MLME that the STA uses.
         */
        if (pScanInfo->curChannelIndex < pScanList->listSize &&
            pScanList->pChanArray[pScanInfo->curChannelIndex]->channelFlags & CF_INTERFERENCE)
        {
            if (verbose) {
                uiPrintf("Channel Interference (radar or carrier wave) detected on channel %d MHz\n",
                         pScanList->pChanArray[pScanInfo->curChannelIndex]->channel);
            }
        }
    }

    if (scanOp == AP_SCAN_SELECT_CHAN && pDevInfo->staConfig.radarDetectEnable) {
        WLAN_SCAN_LIST    *pRadarScanList = NULL;
        WLAN_CHANNEL_LIST *pMainCList;
        BSSDESCR_SET      *pMainBssSet;


        pMainCList = pDevInfo->staConfig.pClist;

        /*
         * Save the main wireless mode scan list
         * Allocate a new channel list of one for extended scanning
         */
        if ((pMainBssSet = (BSSDESCR_SET *)A_DRIVER_MALLOC(sizeof(BSSDESCR_SET))) == NULL) {
                uiPrintf("wlanScanMode failed to allocate memory\n");
                halRxFilter(pDevInfo, HAL_RX_BEACON, HAL_RX_FILTER_CLEAR);
                return A_EINVAL;
        }
        
        A_BCOPY((char *)&pDevInfo->pscanInfo->bssSet, (char *)pMainBssSet, sizeof(BSSDESCR_SET));

        while (1) {
            WLAN_CHANNEL nwChan;
            WLAN_CFLAGS  cflags;

            pRch = wlanFindChannel(pDevInfo, pMainBssSet, pMainCList, NULL, 0);
            if (!pRch) {
                break;
            }

            nwChan = pRch->channel;
            A_MEM_ZERO(&pDevInfo->pscanInfo->bssSet, sizeof(BSSDESCR_SET));

            maxChanTime  = radarScanTime * 1000;
            maxChanTicks = radarScanTime * sysClkRateGet();

            pRadarScanList = NULL;
            if (wlanInitSingleScanList(pRch, &pRadarScanList) != A_OK) {
                uiPrintf("wlanScanMode: wlanInitSingleScanList fails \n");
                A_DRIVER_FREE(pMainBssSet, sizeof(BSSDESCR_SET));
                halRxFilter(pDevInfo, HAL_RX_BEACON, HAL_RX_FILTER_CLEAR);
                return A_EINVAL;
            }
            if (verbose) {
                uiPrintf("Select channel %d for %d seconds of interference detection...\n",
                         nwChan, maxChanTime/1000);
            }
            resultCode = wlanMlmeScanRequest(pDevInfo, ANY_BSS,
                                             &pDevInfo->baseBss.bssId,
                                             NULL, scanType, pRadarScanList,
                                             maxChanTime, maxChanTime,
                                             maxChanTime, maxChanTime, FALSE);
            if (resultCode) {
                if (verbose) {
                    uiPrintf("Scan Request result %x\n", resultCode);
                }
                A_DRIVER_FREE(pMainBssSet, sizeof(BSSDESCR_SET));
                halRxFilter(pDevInfo, HAL_RX_BEACON, HAL_RX_FILTER_CLEAR);
                return A_EINVAL;
            }

            if (isApShutdown() == TRUE) {
                exit(0);
            }

            taskDelay(maxChanTicks + sysClkRateGet());

            /* turn off any more frames coming in */
            halRxFilter(pDevInfo, 0, HAL_RX_FILTER_TEMP);

            /* give up CPU for a fraction of a second. allow received beacons to get processed. */
            taskDelay(5);

            cflags = pRadarScanList->pChanArray[0]->channelFlags;
            if (cflags & CF_INTERFERENCE) {
                uiPrintf("Channel Interference (radar or carrier wave) detected on channel %d MHz\n",
                         pRadarScanList->pChanArray[0]->channel);
            }

            /* re-enable the rx filter */
            halRxFilter(pDevInfo, 0, HAL_RX_FILTER_RESTORE);

            /* Scan curChannelIndex */
            mlmeScanFunction(pDevInfo, SF_ADVANCE_NOW);

            if (cflags & CF_INTERFERENCE) {
                int i, ifcnt;

                for (i = 0; i < pMainBssSet->count; i++) {
                    if (pMainBssSet->bssDescrArray[i].pChannel->channel == nwChan) {
                        A_BCOPY((char *)&pScanInfo->bssSet.bssDescrArray,
                                (char *)&pMainBssSet->bssDescrArray[i], sizeof(BSSDESCR));
                        break;
                    }
                }

                for (i = ifcnt = 0; i < pMainCList->listSize; i++) {
                    if (pMainCList->chanArray[i].channel == nwChan) {
                        pMainCList->chanArray[i].channelFlags |= cflags & CF_INTERFERENCE;
                        uiPrintf("Channel Interference on channel %d MHz\n",
                                 pRadarScanList->pChanArray[0]->channel);
                        ifcnt++;
                        break;
                    }

                    if (pMainCList->chanArray[i].channelFlags & CF_INTERFERENCE) {
                        ifcnt++;
                    }
                }

                uiPrintf("interference count: %d, total: %d\n", ifcnt, pMainCList->listSize);

                if (ifcnt == pMainCList->listSize) {
                    uiPrintf("Interferences on all %d channels\n", pMainCList->listSize);
                    break;
                }
            } else {
                break;
            }
        }

        pDevInfo->pscanInfo->bssSet = *pMainBssSet;

        A_DRIVER_FREE(pMainBssSet, sizeof(*pMainBssSet));
    }

    if (pDevInfo->localSta->serviceType == WLAN_AP_SERVICE) {
        halRxFilter(pDevInfo, HAL_RX_BEACON, HAL_RX_FILTER_CLEAR);
    }

    return A_OK;
}

/*******************************************************************************
 * apScan
 *
 *  Scan all available channels in search for AP
 *  If scanType is passive scan, then listen for beacon
 *  If scanType is active scan, then send probe request and listen for reply
 */
A_STATUS
apScan(WLAN_DEV_INFO *pDevInfo, APSCAN_OP scanOp, SCANTYPE scanType)
{
    SCAN_INFO         *pScanInfo;
    A_STATUS          status;
    CHAN_VALUES       *pNewChannel = NULL;
    A_UINT16          intMode = 0;
    WLAN_CHANNEL_LIST *pMainCList = NULL, *pIntCList = NULL;
    BSSDESCR_SET      *pMainBssSet;
    BSSDESCR_SET      *pIntBssSet = NULL;
    A_BOOL            worldWideScan = FALSE;
    int               unit = pDevInfo->devno;
    int               verbose = AP_SCAN_SITE_SURVEY ? 0 : 1;


    /* delete the sme timer if any */
    A_DELETE_TIMER(&pDevInfo->smeTimer);

    if (scanOp != AP_SCAN_SELECT_CHAN) {
        bridgePortRemove("ar", pDevInfo->devno);

        /* let traffic drain */
        taskDelay(sysClkRateGet());

        /* Cancel the Beacon watchdog and Beacons*/
        beaconWdgCancel(pDevInfo);
        halDisableInterrupts(pDevInfo->targetHandle, HAL_INT_SWBA);
    }

    if (scanOp == AP_SCAN_WORLDWIDE) {
        A_UINT16 mode = MODE_SELECT_ALL;

        worldWideScan = TRUE;
        scanOp        = AP_SCAN_ALL_BSS;
        pMainCList    = pDevInfo->staConfig.pClist;

        pDevInfo->staConfig.pClist = NULL;

        if (wlanInitRoamingChannelList(pDevInfo, &pDevInfo->staConfig.pClist, mode) == A_OK) {
            pIntCList = pDevInfo->staConfig.pClist;
        }
    }

#ifdef DEBUG
    if (verbose) {
        clistShow(pDevInfo);
    }
#endif

    status = wlanScanMode(pDevInfo, scanOp == AP_SCAN_SITE_SURVEY ? AP_SCAN_ALL_BSS : scanOp,
                          scanType, pNewChannel, verbose);
    if (status != A_OK) {
        uiPrintf("wlanScanMode returns ERROR %d\n", status);
        ASSERT(0);
    }

    /* Determine interference mode */
    switch(pDevInfo->staConfig.pClist->chanArray[0].channelFlags & CHANNEL_ALL) {
    case CHANNEL_A:
        intMode = pDevInfo->devCap.wirelessModes & MODE_SELECT_TURBO;
        if (intMode) {          /* Supported by regulatory domain */
            intMode = wlanGetWirelessMode(pDevInfo, pDevInfo->staConfig.countryCode) & MODE_SELECT_TURBO ? MODE_SELECT_TURBO : 0;
            if (intMode && apCfgAboltGet(pDevInfo->devno) & ABOLT_TURBO_PRIME) {
                intMode |= MODE_SELECT_TPA;
            }
        }
        break;
    case CHANNEL_T:
        intMode = MODE_SELECT_11A;
        break;
    case CHANNEL_B:
    case CHANNEL_G:
        intMode = pDevInfo->devCap.wirelessModes & MODE_SELECT_108G;
        if (intMode) {          /* Supported by regulatory domain */
            intMode = wlanGetWirelessMode(pDevInfo, pDevInfo->staConfig.countryCode) & MODE_SELECT_108G ? MODE_SELECT_108G : 0;
            if (intMode && apCfgAboltGet(pDevInfo->devno) & ABOLT_TURBO_PRIME) {
                intMode |= MODE_SELECT_TPG;
            }
        }
        break;
    case CHANNEL_108G:        
        intMode = 0;
        break;
    default:
        ASSERT(0);
        break;
    }

    if (worldWideScan == TRUE) {
        pDevInfo->staConfig.pClist = pMainCList;
    }

    pMainCList = pDevInfo->staConfig.pClist;

    /* Save the main wireless mode scan list */
    if ((pMainBssSet = (BSSDESCR_SET *)A_DRIVER_MALLOC(sizeof(*pMainBssSet))) == NULL) {
        uiPrintf("apScan failed to allocate memory \n");
        halRxFilter(pDevInfo, HAL_RX_BEACON, HAL_RX_FILTER_CLEAR);
        return A_EINVAL;
    }
    *pMainBssSet = pDevInfo->pscanInfo->bssSet;
    A_MEM_ZERO(&pDevInfo->pscanInfo->bssSet, sizeof(pDevInfo->pscanInfo->bssSet));

    if ((scanOp != AP_SCAN_SITE_SURVEY) && (worldWideScan == FALSE) && intMode) {
        /* Scan any interfering wireless mode - if possible
         *
         * Some list "acrobatics" are required here as the scan tools expect
         * to use the staConfig channel list and the scanInfo BSS list.  Hence
         * we "pointer save" the channel list and have to copy the BSS list from
         * the first scan.
         */
        pMainCList = pDevInfo->staConfig.pClist;
        pDevInfo->staConfig.pClist = NULL;

        if (wlanInitChannelList(pDevInfo, &(pDevInfo->staConfig.pClist), pDevInfo->staConfig.countryCode,
                                intMode, apCfgOutdoorChanGet(unit), TRUE) != A_OK)
        {
#ifdef DEBUG
            uiPrintf("Auto Channel Scan: Interfering wireless mode cannot be scanned\n");
#endif
        } else {
            /* Scan interfering mode */

            /* Includes turbo prime channels if AP is configured for Dynamic Turbo.
             * wlanInitChannelList() will insert those channels if required.
             * Assumes number of turbo prime channels is same as what
             * wlanGetNumChannelsForDomain() returns since only FCC1 supports it.
             * The current architecture does not support the cleaner use of tables
             * unless the mode usage is redefined.
             */
            status = wlanScanMode(pDevInfo, scanOp, scanType, pNewChannel, verbose);

            if (status != A_OK) {
                uiPrintf("wlanScanMode returns ERROR %d on interferer's mode\n", status);
                ASSERT(0);
            }

            pIntCList  = pDevInfo->staConfig.pClist;
            pIntBssSet = &pDevInfo->pscanInfo->bssSet;
        }
        pDevInfo->staConfig.pClist = pMainCList;
    }

    /* enable unicasts to get PROBE_RESP in active scan */
    halRxFilter(pDevInfo, HAL_RX_UCAST, HAL_RX_FILTER_SET);

    if (scanOp != AP_SCAN_SELECT_CHAN) {
        if (pDevInfo->localSta->serviceType == WLAN_AP_SERVICE) {
            /* Re-enable beacons */
            wlanBeaconRelease(pDevInfo);
            wlanBeaconInit(pDevInfo);
            beaconWdgStart(pDevInfo);
        }

        bridgePortAdd("ar", pDevInfo->devno);
    }

    /* pScanInfo is not malloc'ed until after the wlanScanMode is run */
    pScanInfo = pDevInfo->pscanInfo;
    ASSERT(pScanInfo);
    pDevInfo->pscanInfo->bssSet = *pMainBssSet;
    pDevInfo->pCsInfo->scanInfo.bssSet = *pMainBssSet;

    if (scanOp == AP_SCAN_ALL_BSS) {
        /*
         * Finish with the last channel on the list.
         * Clear the state. Don't let any more bss's creep in
         */
        if (pMainBssSet->count > 0) {
            uiPrintf("=> BSS'es from the selected wireless mode <=\n");
            bssDescShow(pMainBssSet);
        }
        if (pIntBssSet && (pIntBssSet->count > 0)) {
            uiPrintf("=> BSS'es from the interfering wireless mode <=\n");
            bssDescShow(pIntBssSet);
        }
        if (pIntCList) {
            wlanFreeChannelList(pIntCList);
        }
        A_DRIVER_FREE(pMainBssSet, sizeof(*pMainBssSet));
        return A_OK;
    }

    if (scanOp == AP_SCAN_SITE_SURVEY) {
        /*
         * Finish with the last channel on the list.
         * Clear the state. Don't let any more bss's creep in
         */
        if (pMainBssSet->count) {
            bssDesc2Web(pMainBssSet);
        }
        A_DRIVER_FREE(pMainBssSet, sizeof(*pMainBssSet));
        return A_OK;
    }

    if (pNewChannel == NULL) {
        pNewChannel = wlanFindChannel(pDevInfo, pMainBssSet, pMainCList, pIntBssSet, 0);
    }
    
    if (scanOp == AP_SCAN_CLEAN_CHAN) {
        if (pNewChannel) {
            uiPrintf("Auto Channel Scan would select channel center frequency %d MHz (IEEE %d)\n",
                     pNewChannel->channel, wlanConvertGHztoCh(pNewChannel->channel,
                     pNewChannel->channelFlags));
        } else {
            uiPrintf("Auto Channel Scan did not detect a good channel\n");
        }

        if (pIntCList) {
            wlanFreeChannelList(pIntCList);
        }

        A_DRIVER_FREE(pMainBssSet, sizeof(*pMainBssSet));
        return A_OK;
    }

    if (pNewChannel) {
        uiPrintf("Auto Channel Scan selected %d MHz, channel %d\n",
                 pNewChannel->channel,
                 wlanConvertGHztoCh(pNewChannel->channel, pNewChannel->channelFlags));

        if (pNewChannel != pDevInfo->staConfig.pChannel) {
            pDevInfo->staConfig.pChannel = pNewChannel;
            pDevInfo->bssDescr->pChannel = pNewChannel;
            apCfgRadioChannelSet(pDevInfo->devno, pNewChannel->channel);
        }
    }

    if (pIntCList) {
        wlanFreeChannelList(pIntCList);
    }

    /* Retune to the configured channel */
    chipReset(pDevInfo, pDevInfo->staConfig.pChannel, FALSE);

    if (pDevInfo->staConfig.radarDetectEnable) {
        initRadarDetect(pDevInfo);
    }

    A_DRIVER_FREE(pMainBssSet, sizeof(*pMainBssSet));

    return A_OK;
}
#endif // BUILD_AP

/*
 * The defines below increase or decrease the cost accounting that selects
 * a channel for initial placement
 */
enum ChanCostEnum {
    CHAN_COST_BASE_NEXT_TO_TURBO = 10,
    CHAN_COST_NEW_GEN            = -20,
    CHAN_COST_PRIME              = 30,
    CHAN_COST_5_ADJACENT         = 18,
    CHAN_COST_MIRROR             = 50,
    CHAN_COST_UNUSABLE           = 512,
    ADDITIONAL_INT_COST          = 3
} CHAN_COST_ENUM;

#define A_MAX_CAST(x, y)        ((A_UINT16)A_MAX((x), (y)))
#define PRINTCOST(x, y, z)      {                                         \
    int a;                                                                \
    for (a = 0; a < (z); a++) {                                           \
       uiPrintf(" -- Channel %d Cost %d\n", (y)[a].channel, (x)[a].cost); \
    }                                                                     \
}

typedef struct ChannelListCost {
    A_INT16 cost;
    A_INT16 maxCost;
    A_INT16 fitCost;
    A_INT16 numBadGuys;
} CHANNEL_LIST_COST;

/*******************************************************************************
 * wlanAddToCost
 *
 * Adds the given cost into the Cost List and updates the max and number of
 * interferers.
 */
static void
wlanAddToCost(CHANNEL_LIST_COST *pCListCost, A_UINT16 index, A_INT16 cost)
{
    if (cost > 0 ) {
       pCListCost[index].numBadGuys++;
    }
    if (cost > pCListCost[index].maxCost) {
        /* New signal is bigger - set it as the main interfering cost */
        pCListCost[index].maxCost = cost;
    }
}

#define WLAN_IS_OVERLAP(channel1, spread1, channel2, spread2)                   \
    (                                                                           \
    ((((channel2) - ((spread2) / 2) + 1) > ((channel1) - ((spread1) / 2))) &&   \
     (((channel2) - ((spread2) / 2) + 1) < ((channel1) + ((spread1) / 2)))) ||  \
    ((((channel2) + ((spread2) / 2) - 1) > ((channel1) - ((spread1) / 2))) &&   \
     (((channel2) + ((spread2) / 2) - 1) < ((channel1) + ((spread1) / 2))))     \
    )

/*******************************************************************************
 * wlanFindChannel
 *
 * Given the bss/channel list, the interferer bss/channel list, the configured
 * channel - find an acceptable runtime channel.
 *
 * A channelCost list is created.  The given channel statistics
 * will increase (less desirable) and decrease (more desirable)
 */
CHAN_VALUES *
wlanFindChannel(WLAN_DEV_INFO *pDevInfo, BSSDESCR_SET *pBssSet,
                WLAN_CHANNEL_LIST *pCList, BSSDESCR_SET *pIntBssSet,
                WLAN_CFLAGS cflags)
{
    CHANNEL_LIST_COST *pCListCost;
    CHAN_VALUES       *pNewChannel = NULL;
    A_BOOL            chan5Ghz;
    A_UINT16          i, j;
    A_UINT16          channelSpread, intSpread, bssChannel, bssRssi;
    int               winningLowChannel, numLowestCost;
    A_INT16           lowestCost;
    static A_INT16    CHAN_COST_CCK[5] = {0, 2, 3, 8, 25};

    ASSERT(pCList->listSize);

    /* Allocate cost-function list holder to match channel list size */
    pCListCost = (CHANNEL_LIST_COST *)
                 A_DRIVER_MALLOC(sizeof(*pCListCost) * pCList->listSize);
    if (!pCListCost) {
        uiPrintf("wlanFindChannel: Unable to allocate a cost list\n");
        return NULL;
    }
    A_MEM_ZERO(pCListCost, sizeof(*pCListCost) * pCList->listSize);

    /*
     * Initialize Cost
     * Subtract cost for the relative power levels - give higher power channels preference
     * Only power levels up to 23 dBm are counted
     *
     * Add cost for carrier wave or radar detection present to remove the channel from the list
     *
     * If AP, for dynamic turbo channels, decrease cost to favor their selection.
     */
    for (i = 0; i < pCList->listSize; i++) {
        chan5Ghz = IS_CHAN_5GHZ(pCList->chanArray[i].channelFlags);

        if (chan5Ghz && (pCList->chanArray[i].channel < 5120 ||
            pCList->chanArray[i].channel > 5430))
        {
            /* Increase cost of using non 1st gen channels */
            pCListCost[i].cost = CHAN_COST_NEW_GEN;
        } else {
            /* Apply a 1.5 multiplier to power levels */
            pCListCost[i].cost = ((wlanGetChannelPower(pDevInfo, &(pCList->chanArray[i])) >> 1) * -3);

#if defined(DEBUG) && !defined(BUILD_AP)
            uiPrintf("channel %d, power: %d\n", pCList->chanArray[i].channel,
                     wlanGetChannelPower(pDevInfo, &(pCList->chanArray[i])));
#endif
        }

        if (wlanIsTurboPrimeChannel(pDevInfo, &(pCList->chanArray[i]))) {
            pCListCost[i].cost -= CHAN_COST_PRIME;
            #ifdef DEBUG
            if (debugScan) {
                uiPrintf("\nAdjust prime channel %d %d\n", pCList->chanArray[i].channel, pCListCost[i].cost);
            }
            #endif
        }

        if (pCList->chanArray[i].channelFlags & (CF_INTERFERENCE | CHANNEL_DONT_SCAN)) {
            pCListCost[i].cost = CHAN_COST_UNUSABLE;
        }
    }

#if defined(DEBUG) && !defined(BUILD_AP)
    PRINTCOST(pCListCost, pCList->chanArray, pCList->listSize)
#endif

    ASSERT(pBssSet);
    /* Add cost for detected BSS'es */
    for (j = 0; j < pBssSet->count; j++) {
        bssChannel = pBssSet->bssDescrArray[j].pChannel->channel;
        bssRssi    = (A_UINT16)pBssSet->bssDescrArray[j].rssi;

        for (i = 0; i < pCList->listSize; i++) {
            channelSpread = wlanGetChannelSpread(pCList->chanArray[i].channelFlags);

            if (pCList->chanArray[i].channel == bssChannel) {
                /* Add this BSS'es cost to its channel */
                wlanAddToCost(pCListCost, i, bssRssi);
            }

            chan5Ghz = IS_CHAN_5GHZ(pCList->chanArray[i].channelFlags);
            if (chan5Ghz) {
                if (bssChannel == (pCList->chanArray[i].channel + channelSpread) ||
                    bssChannel == (pCList->chanArray[i].channel - channelSpread))
                {
                    /* Add this BSS'es cost to any adjacent channels */
                    wlanAddToCost(pCListCost, i, A_MAX_CAST(bssRssi - CHAN_COST_5_ADJACENT, 0));
                } else if (bssChannel == (pCList->chanArray[i].channel + 80)  ||
                           bssChannel == (pCList->chanArray[i].channel - 80)  ||
                           bssChannel == (pCList->chanArray[i].channel + 160) ||
                           bssChannel == (pCList->chanArray[i].channel - 160))
                {
                    /* Add this BSS'es Cost to any possible mirrors */
                    wlanAddToCost(pCListCost, i, A_MAX_CAST(bssRssi - CHAN_COST_MIRROR, 0));
                }
            } else {
                /* 2.4 GHz channels are spaced every 5 MHz and interfere up to 20 MHz away */
                int mhzAway;

                for (mhzAway = 5; mhzAway <= 20; mhzAway += 5) {
                    if (bssChannel == (pCList->chanArray[i].channel + mhzAway) ||
                        bssChannel == (pCList->chanArray[i].channel - mhzAway))
                    {
                        /* Add this BSS'es cost to any adjacent channels */
                        wlanAddToCost(pCListCost, i,
                                      A_MAX_CAST(bssRssi - CHAN_COST_CCK[mhzAway / 5], 0));
                    }
                }
            }
        }
    }

    /* Add Cost of channels and BSS'es from an interfering wireless mode */
    if (pIntBssSet != NULL) {
        for (j = 0; j < pIntBssSet->count; j++) {
            intSpread  = wlanGetChannelSpread(pIntBssSet->bssDescrArray[j].pChannel->channelFlags);
            bssChannel = pIntBssSet->bssDescrArray[j].pChannel->channel;
            bssRssi    = (A_UINT16)pIntBssSet->bssDescrArray[j].rssi;

            for (i = 0; i < pCList->listSize; i++) {
                chan5Ghz = IS_CHAN_5GHZ(pCList->chanArray[i].channelFlags);

                if (chan5Ghz) {
                    channelSpread = wlanGetChannelSpread(pCList->chanArray[i].channelFlags);

                    if (WLAN_IS_OVERLAP(bssChannel, intSpread, pCList->chanArray[i].channel, channelSpread)) {
                        /* Add this BSS'es cost to any overlapping channel */
                        wlanAddToCost(pCListCost, i, bssRssi);
                    }

                    if (WLAN_IS_OVERLAP(bssChannel, intSpread, pCList->chanArray[i].channel + channelSpread, channelSpread) ||
                        WLAN_IS_OVERLAP(bssChannel, intSpread, pCList->chanArray[i].channel - channelSpread, channelSpread))
                    {
                        /* Add this BSS'es cost to any adjacent channels */
                        wlanAddToCost(pCListCost, i, A_MAX_CAST(bssRssi - CHAN_COST_5_ADJACENT, 0));
                    }

                    /* Add this BSS'es Cost to any possible mirrors */
                    if (WLAN_IS_OVERLAP(bssChannel, intSpread, pCList->chanArray[i].channel + 80, channelSpread)  ||
                        WLAN_IS_OVERLAP(bssChannel, intSpread, pCList->chanArray[i].channel - 80, channelSpread)  ||
                        WLAN_IS_OVERLAP(bssChannel, intSpread, pCList->chanArray[i].channel + 160, channelSpread) ||
                        WLAN_IS_OVERLAP(bssChannel, intSpread, pCList->chanArray[i].channel - 160, channelSpread))
                    {
                        wlanAddToCost(pCListCost, i, A_MAX_CAST(bssRssi - CHAN_COST_MIRROR, 0));
                    }
                } else {
                    /* 2.4 GHz channels are spaced every 5 MHz and interfere up to 20 MHz away */
                    /* Both 11b and 11g use the same spacing so no overlap check is required */
                    int mhzAway;

                    for (mhzAway = 5; mhzAway <= 20; mhzAway += 5) {
                        if (bssChannel == (pCList->chanArray[i].channel + mhzAway) ||
                            bssChannel == (pCList->chanArray[i].channel - mhzAway))
                        {
                            /* Add this BSS'es cost to any adjacent channels */
                            wlanAddToCost(pCListCost, i,
                                          A_MAX_CAST(bssRssi - CHAN_COST_CCK[mhzAway / 5], 0));
                        }
                    }
                }
            }
        }
    }

    for (i = 0; i < pCList->listSize; i++) {
        /* Add the worst Cost plus 3 per additional interferer to the cost value */
        if (pCListCost[i].numBadGuys) {
            pCListCost[i].cost += (pCListCost[i].maxCost + ((pCListCost[i].numBadGuys - 1) * ADDITIONAL_INT_COST));
        }
    }

#if defined(DEBUG) && !defined(BUILD_AP)
    PRINTCOST(pCListCost, pCList->chanArray, pCList->listSize)
#endif

#ifndef BUILD_AP
    /*
     * Get rid of unwanted adhoc mode by marking the channel
     * as unusable if the channel type don't match.
     * (Use only channel 10 & 11 for 802.11b)
     */
    if (pDevInfo->staConfig.bssType == INDEPENDENT_BSS) {
        wlanPrintFlags(cflags);

        for (i = 0; i < pCList->listSize; i++) {
            WLAN_CFLAGS  ccf;
            WLAN_CHANNEL channel;

            ccf     = pCList->chanArray[i].channelFlags & CHANNEL_ALL;
            channel = pCList->chanArray[i].channel;
            
            if (cflags == CHANNEL_108G) {
                pCListCost[i].cost = (ccf == cflags) ? 0 : CHAN_COST_UNUSABLE;
                continue;
            } 
            
            if (ccf != cflags) {
                if (ccf == CHANNEL_B) {
                    if (channel == 2457 || channel == 2462) {
                        continue;
                    }
                    pCListCost[i].cost = cflags == CHANNEL_B ? 0 : CHAN_COST_UNUSABLE;
                } else {
                    pCListCost[i].cost = CHAN_COST_UNUSABLE;
                }
                uiPrintf(" -- Channel %d Cost %d\n", channel, pCListCost[i].cost);
            } else {
                if (ccf == CHANNEL_B) {
                    if (channel != 2457 && channel != 2462) {
                        pCListCost[i].cost = cflags == CHANNEL_B ? 0 : CHAN_COST_UNUSABLE;
                    }
                    uiPrintf(" -- Channel %d Cost %d\n", channel, pCListCost[i].cost);
                }
            }
        }
    }
#endif

    lowestCost    = CHAN_COST_UNUSABLE - 1;
    numLowestCost = 0;

    for (i = 0; i < pCList->listSize; i++) {
        chan5Ghz = IS_CHAN_5GHZ(pCList->chanArray[i].channelFlags);
        if (chan5Ghz || IS_CHAN_108G(cflags)) {
            /* 5GHz channels "fit" into a single channel */
            pCListCost[i].fitCost = pCListCost[i].cost;
        } else {
            int m;

            /*
             * Find lowest cost "fit" for 2.4 channels which must "fit"
             * across multiple frequencies
             */
            for (m = i - 3; m <= i + 3; m++) {
                if (m < 0) {
                    pCListCost[i].fitCost += pCListCost[0].cost - CHAN_COST_CCK[A_ABS(i - m)];
                } else if (m >= pCList->listSize) {
                    pCListCost[i].fitCost += pCListCost[pCList->listSize - 1].cost - CHAN_COST_CCK[A_ABS(i - m)];
                } else {
                    pCListCost[i].fitCost += pCListCost[m].cost - CHAN_COST_CCK[A_ABS(i - m)];
                }
            }
        }
#if defined(DEBUG) && !defined(BUILD_AP)
        uiPrintf("channel %d, cost: %d, fitCost: %d, lowestCost: %d\n",
                 pCList->chanArray[i].channel, pCListCost[i].cost, pCListCost[i].fitCost, lowestCost);
#endif
        /* Find lowest cost and count how many are at this cost */
        if (pCListCost[i].fitCost == lowestCost) {
            numLowestCost++;
        } else if (pCListCost[i].fitCost < lowestCost) {
            lowestCost    = pCListCost[i].fitCost;
            numLowestCost = 1;
        }
    }

#ifdef DEBUG
    if (debugScan) {
        uiPrintf("lowestCost %d, numLowestCost %d\n", lowestCost, numLowestCost);
        uiPrintf("wlanFindChannel Cost Histogram: higher values indicate more interference\n");
        /* Display cost histogram */
        for (i = 0; i < pCList->listSize; i++) {
            uiPrintf("%4d %3d%s", pCList->chanArray[i].channel, pCListCost[i].fitCost,
                     (i + 1) % 7 ? "  " : "\n");
        }
        uiPrintf("\n\n");
    }
#endif
    if (numLowestCost == 0) {
        uiPrintf("wlanFindChannel: WARNING - All channels are unusable\n");
#if defined(DEBUG) && !defined(BUILD_AP)
        PRINTCOST(pCListCost, pCList->chanArray, pCList->listSize)
#endif
        A_DRIVER_FREE(pCListCost, sizeof(CHANNEL_LIST_COST) * pCList->listSize);

        return NULL;
    }

    /*
     * Randomize the set of low cost channels
     * for the case of multiple AP's booting at once
     */
    winningLowChannel = A_RAND() % numLowestCost;

    /* Return the winner */
    for (i = 0; i < pCList->listSize; i++) {
        if (pCListCost[i].fitCost == lowestCost) {
            if (winningLowChannel == 0) {
                pNewChannel = &(pCList->chanArray[i]);
                break;
            }
            winningLowChannel--;
        }
    }

    ASSERT(i != pCList->listSize);

#ifdef DEBUG
    /* Normalize */
    lowestCost = CHAN_COST_UNUSABLE - 1;
    for (i = 0; i < pCList->listSize; i++) {
        if (pCListCost[i].fitCost < lowestCost) {
            lowestCost = pCListCost[i].fitCost;
        }
    }

    for (i = 0; i < pCList->listSize; i++) {
        pCListCost[i].fitCost -= lowestCost;
    }

    uiPrintf("wlanFindChannel Cost Histogram: higher values indicate more interference\n");

    /* Display cost histogram */
    for (i = 0; i < pCList->listSize; i++) {
        uiPrintf("%4d %3d%s", pCList->chanArray[i].channel, pCListCost[i].fitCost,
                 (i + 1) % 7 ? "  " : "\n");
    }

    uiPrintf("\n\n");
#endif

    A_DRIVER_FREE(pCListCost, sizeof(*pCListCost) * pCList->listSize);

    return pNewChannel;
}

/*******************************************************************************
 * wlanFindChannelMatch
 *
 * Given the channel list, channel flags, and frequency,
 * find a channel in the list that matches flags and frequency.
 */
CHAN_VALUES *
wlanFindChannelMatch(WLAN_DEV_INFO *pDevInfo, WLAN_CHANNEL_LIST *pCList,
                     WLAN_CFLAGS cflags, A_UINT16 cfreq)
{
    A_UINT16    i;
    CHAN_VALUES cv1;
    

    ASSERT(pCList->listSize);
    cv1.channel      = cfreq;
    cv1.channelFlags = cflags;    

    for (i = 0; i < pCList->listSize; i++) {
        CHAN_VALUES *pCv2 = &pCList->chanArray[i];
        if (wlanIsChannelMatch(&cv1, pCv2)) {
            return &pCList->chanArray[i];
        }
    }

    return NULL;
}

/***********************************************************
 * wlanIsWorldWideRoaming
 *
 * Returns TRUE if worlwide roaming is allowed by the EEPROM setting
 */
static A_BOOL
wlanIsWorldWideRoaming(WLAN_DEV_INFO *pDevInfo)
{
    if (wlanEepromCcRdGet(pDevInfo) & WORLDWIDE_ROAMING_FLAG) {
        return TRUE;
    }

#ifndef BUILD_AP
    {
        A_UINT16 idx;

        if (wlanSupDomGet(pDevInfo, &idx)) {
            return TRUE;
        }
    }
#endif

    return FALSE;
}

A_BOOL
wlanIsImplicitRegDmn(CHAN_VALUES *pChannel, CTRY_CODE *pCcode)
{
    WLAN_CHANNEL chan;
    WLAN_CFLAGS  cflags;
    A_UINT16     i;

    cflags = pChannel->channelFlags;

    uiPrintf("wlanIsImplicitRegDmn: %d, %x ", pChannel->channel,  cflags);
    wlanPrintFlags(cflags);

    for (i = 0; i < NUM_IMPLICIT_RD; i++) {
        for (chan = 0; irdChannelTbl[i].cList[chan]; chan++) {
            CHAN_VALUES cv2;

            cv2.channel      = irdChannelTbl[i].cList[chan];
            cv2.channelFlags = irdChannelTbl[i].channelFlags;
            if (wlanIsChannelMatch(pChannel, &cv2)) {
                *pCcode = irdChannelTbl[i].countryCode;
                uiPrintf("wlanIsImplicitRegDmn: %d, %d\n", pChannel->channel, *pCcode);
                return TRUE;
            }
        }
    }
    return FALSE;
}

A_UCHAR *
wlanGetCountryName(WLAN_DEV_INFO *pDevInfo, CTRY_CODE countryCode)
{
    int i;

    for (i = 0; i < NUM_OF_COUNTRIES; i++) {
        if (countryCode == allCountries[i].countryCode) {
            return (A_UCHAR *)allCountries[i].name;
        }
    }

    return "unknown";
}

A_BOOL
wlan11bMode(WLAN_DEV_INFO *pDevInfo)
{
    return (wlanGetWirelessMode(pDevInfo, pDevInfo->staConfig.countryCode) & MODE_SELECT_11B) ? TRUE : FALSE;
}

A_BOOL
wlan11gMode(WLAN_DEV_INFO *pDevInfo)
{
    return (wlanGetWirelessMode(pDevInfo, pDevInfo->staConfig.countryCode) & MODE_SELECT_11G) ? TRUE : FALSE;
}

A_BOOL
wlanIsWwrSKU(WLAN_DEV_INFO *pDevInfo)
{
    REG_DOMAIN  rd;
    int         i;

    rd = wlanHalRD(pDevInfo);
    rd = (rd & COUNTRY_ERD_FLAG) ? 0 : rd & SUPER_DOMAIN_MASK;

    for (i = 0; i < NUM_OF_SUPER_DOMAINS; i++) {
        if (superDomainTbl[i].domain == rd) {
            return TRUE;
        }
    }
    return FALSE;
}

ATH_COUNTRY
wlanGetSKU(WLAN_DEV_INFO *pDevInfo)
{
    if (wlanGetDefaultCountry(pDevInfo) != CTRY_DEFAULT) {
        /* Single country */
        return ATH_COUNTRY_ONE;
    } else {
        if (wlanIsWwrSKU(pDevInfo)) {
            return ATH_COUNTRY_WORLD;
        } else {
            return ATH_COUNTRY_ALL;
        }
    }
}


/*
 * wlanSelectAdHocChannel
 *
 * Assuming that cservSelectAdHoc has already called wlanFindChannel
 * to choose a channel to start the adhoc network.
 * This function is called to determine whether to use the channel
 * choosen by wlanFindChannel or use other channel else instead.
 * The channel selection is as follows:
 * - If user selects 11b ad hoc, then use channl 10 or 11.
 * - For regulatory domain FCC1_FCCA (0x10) or MKK1_MKKA (0x40), choose
 *   from either 36, 40, 44, 48 for FCC, or from 34, 38, 42, 48 for MKK.
 * - For super domain,
 *   - WOR0_WORLD & WOR3_WORLD, channel selection is driven by the 11d mechanism.
 *   - APL5_WORLD & APL5_APLB, choose either 149, 153, 157 or 161.
 *   - WOR2_WORLD, choose either 36, 40, 44 or 48.
 *   - WOR1_WORLD, choose either 2.4GHz channel 10 or 11 if 802.11b
 *     adhoc is selected.
 * - Otherwise, use channel selected by cservSelectAdHoc.
 *
 * Return:
 *      A_OK - Channel selected in pDevInfo->bssDescr->pChannel.
 *      A_ERROR - No channel selected.
 */
A_STATUS
wlanSelectAdHocChannel(WLAN_DEV_INFO *pDevInfo, BSSDESCR_SET *pBssSet, WLAN_CFLAGS cflags)
{
    A_STATUS      status = A_ERROR;
    CHAN_VALUES   *pCv, *pChannel = NULL;
    WLAN_CHANNEL  achan = 0, bchan = 0, adchan;
    int           cnt, i, j, start11b;
    REG_DOMAIN    rd;
    WLAN_CHANNEL  *pAdHocClist = NULL;
    WLAN_STA_CONFIG *pConfig = &pDevInfo->staConfig;

    uiPrintf("wlanSelectAdHocChannel: bss channel %d, ",
             pDevInfo->bssDescr->pChannel ? pDevInfo->bssDescr->pChannel->channel : 0);
    wlanPrintFlags(cflags);

    if (pDevInfo->staConfig.userClist) {
        uiPrintf("wlanSelectAdHocChannel: use clist\n");
        return A_OK;
    }

    pCv = (CHAN_VALUES *)&pDevInfo->staConfig.sku.adHocChannel;
    if (pDevInfo->bssDescr->pChannel)
        *pCv = *pDevInfo->bssDescr->pChannel;

    start11b = 0;
    pDevInfo->staConfig.sku.adHocMode = START_ADHOC_IN_11A;

    if (cflags == CHANNEL_T) {
        uiPrintf("wlanSelectAdHocChannel -- Turbo\n");
        return A_OK;
    }

    if (cflags == CHANNEL_108G) {
        uiPrintf("wlanSelectAdHocChannel -- 108g\n");
        return A_OK;
    }

    if (cflags == CHANNEL_B || cflags == CHANNEL_G) {
        if (wlanIsHwSupportMode(pDevInfo, MODE_SELECT_11B | MODE_SELECT_11G) == FALSE) {
            uiPrintf("wlanSelectAdHocChannel -- No 11b/g hardware support\n");
            return A_ERROR;
        }
        uiPrintf("wlanSelectAdHocChannel: mode = 11%s\n", cflags == CHANNEL_B ? "b": "g");
        start11b = 1;
        pDevInfo->staConfig.sku.adHocMode = START_ADHOC_IN_11B;
    } else {
        /*
         * Read the actual RD setting in EEPROM
         * If super domain is defined, then setup the adhoc channel
         * list and adhoc mode.
         * If country code is defined, then setup the adhoc channel
         * list for FCC and MKK.
         */
        uiPrintf("wlanSelectAdHocChannel: mode = 11a\n");
        rd = wlanHalRD(pDevInfo);
        rd = (rd & COUNTRY_ERD_FLAG) ? 0 : rd & SUPER_DOMAIN_MASK;

        uiPrintf("wlanSelectAdHocChannel: rd = 0x%x\n", rd);
        if (rd) {
            for (i = 0; i < NUM_OF_SUPER_DOMAINS; i++) {
                if (superDomainTbl[i].domain == rd) {
                    pAdHocClist = superDomainTbl[i].adHocClist;
                    pDevInfo->staConfig.sku.adHocMode = superDomainTbl[i].adHocMode;
                    if (pDevInfo->staConfig.sku.adHocMode == START_ADHOC_NO_11A) {
                        pDevInfo->bssDescr->pChannel = NULL;
                        return A_ERROR;
                    }
                    if (pDevInfo->staConfig.sku.adHocMode == START_ADHOC_PER_11D) {
                        uiPrintf("wlanSelectAdHocChannel: startAdHoc 11d\n");
                        return A_OK;
                    }
                    break;
                }
            }
        }

        uiPrintf("wlanSelectAdHocChannel: rd: 0x%x, ccode: %d, adhoc: %p, status: %d\n",
                 rd, pDevInfo->staConfig.countryCode, (void *)pAdHocClist, status);

        if (pDevInfo->staConfig.sku.adHocMode != START_ADHOC_NO_11A) {
            if (!pAdHocClist) {
                rd = wlanHalRD(pDevInfo);
                pAdHocClist = wlanGetAdHocClist(rd);
                if (!pAdHocClist) {
                    if (pCv) {
                        uiPrintf("wlanSelectAdHocChannel: ccode: %d, channel: %d\n",
                                 pDevInfo->staConfig.countryCode, pCv->channel);
                    }
                    return pCv ? A_OK : A_ERROR;
                }

                uiPrintf("wlanSelectAdHocChannel: ccode: %d, rd: %x\n",
                         pDevInfo->staConfig.countryCode, rd);
            }
        }

        if (pAdHocClist) {
            /*
             * Use the preset channel list to scan the BSS
             * and pick the channel that is not in BSS
             */

            adchan = pAdHocClist[0];
            cnt    = 0;

            for (i = 0; pAdHocClist[i]; i++) {
                cnt++;
            }

            /*
             * Check if we've already chosen the channel
             * in the preset channel list
             */
            for (i = 0; pAdHocClist[i]; i++) {
                if (pDevInfo->bssDescr->pChannel &&
                    pDevInfo->bssDescr->pChannel->channel == pAdHocClist[i])
                {
                    break;
                }
            }

            if (i >= cnt) {
                for (i = 0; pAdHocClist[i]; i++) {
                    achan = pAdHocClist[i];
                    for (j = 0; j < pBssSet->count; j++) {
                        bchan = pBssSet->bssDescrArray[j].pChannel->channel;
                        if (achan == bchan) {
                            uiPrintf("achan %d, bchan %d\n", achan, bchan);
                            adchan = pAdHocClist[(i+1) % cnt];
                        }
                    }
                }

                pCv->channel      = adchan;
                pCv->channelFlags = CHANNEL_A;
            }

            status = A_OK;
            uiPrintf("wlanSelectAdHocChannel: pick adchan = %d, use preset 11a channel %d\n",
                     adchan, pCv->channel);
        }
    }

    if (status != A_OK || start11b) {
        if (wlan11bMode(pDevInfo) || wlan11gMode(pDevInfo)) {
            {
                /* Search for available 11b/11g channel */
                adchan = adhoc11bClist[0];
                cnt    = 0;

                for (i = 0; adhoc11bClist[i]; i++) {
                    cnt++;
                }

                /*
                 * Check if we've already chosen the channel
                 * in the reserved channel list
                 */
                for (i = 0; adhoc11bClist[i]; i++) {
                    if (pDevInfo->bssDescr->pChannel &&
                        pDevInfo->bssDescr->pChannel->channel == adhoc11bClist[i])
                    {
                        break;
                    }
                }

                if (i >= cnt) {
                    for (i = 0; adhoc11bClist[i]; i++) {
                        achan = adhoc11bClist[i];
                        for (j = 0; j < pBssSet->count; j++) {
                            bchan = pBssSet->bssDescrArray[j].pChannel->channel;
                            if (achan == bchan) {
                                uiPrintf("wlanSelectAdHocChannel: achan %d, bchan %d\n", achan, bchan);
                                adchan = adhoc11bClist[(i+1) % cnt];
                            }
                        }
                    }

                    pCv->channel      = adchan;
                    pCv->channelFlags = start11b ? cflags : CHANNEL_B;
                }

                status = A_OK;
                uiPrintf("wlanSelectAdHocChannel: use reserved 11b/g channel %d\n", pCv->channel);
            }
        }
    }

    uiPrintf("wlanSelectAdHocChannel: status: %d, channel %d, flags: %x ",
             status, pCv->channel, pCv->channelFlags);
    wlanPrintFlags(pCv->channelFlags);

    if (status == A_OK) {
        pChannel = wlanFindChannelMatch(pDevInfo, pConfig->pClist,
                                    pCv->channelFlags, pCv->channel);
    }

    if (pChannel) {
        pDevInfo->bssDescr->pChannel = pChannel;
    }

    return status;
}

/***********************************************************
 * wlanXlatNonIsoCcode
 *
 * Convert non-ISO country code string to ISO country code
 */
static CTRY_CODE
wlanXlatNonIsoCcode(A_UCHAR *pCc)
{
    int i;

    for (i = 0; i < sizeof(ciscoCcTbl)/sizeof(CISCO_COUNTRY_INFO_LIST); i++) {
        if (A_STRNCMP(ciscoCcTbl[i].info.countryString, pCc, 3) == 0) {
            return ciscoCcTbl[i].countryCode;
        }
    }

    uiPrintf("wlanXlatNonIsoCcode: No country code for %s!\n", pCc);

    return CTRY_INVALID;
}

/***********************************************************
 * wlanClistClnup
 *
 * Clear passive scan flag from channel list if non-ETSI
 */
void
wlanClistClnup(WLAN_DEV_INFO *pDevInfo)
{
    WLAN_CHANNEL_LIST *pCList;
    WLAN_CFLAGS       cflags;
    REG_DOMAIN        rd;
    int               i;


    ASSERT(pDevInfo);
#if DEBUG
    uiPrintf("wlanClistClnup: cc: %d\n", pDevInfo->staConfig.countryCode);
#endif

    /*
     * For world SKU, if ignore11dBeacon is enabled or no 11d beacon is
     * found, then don't purge the passive scan flag
     */
    if (wlanIsWwrSKU(pDevInfo) == TRUE) {
        if (pDevInfo->staConfig.ignore11dBeacon ||
            pDevInfo->staConfig.sku.v.b.cc11d == FALSE) {
            return;
        }
    }

    /* Purge the passive scan flag for non-ETSI */
    cflags = pDevInfo->staConfig.phwChannel->channelFlags;
    rd     = wlanGetWmRD(pDevInfo, pDevInfo->staConfig.countryCode, cflags);

    if (wlanIsPassiveScanChannel(pDevInfo, rd, pDevInfo->staConfig.phwChannel) != TRUE) {
        pCList = pDevInfo->staConfig.pClist;
        for (i = 0; i < pCList->listSize; i++) {
            cflags = pCList->chanArray[i].channelFlags;
            if (wlanIs2GHzEcmChan(pCList->chanArray[i].channel) == FALSE) {
                pCList->chanArray[i].channelFlags = cflags & ~CHANNEL_PASSIVE;
            } else {
                if (wlanIsEcmChanPassive(pDevInfo, pCList->chanArray[i].channel) == FALSE) {
                    pCList->chanArray[i].channelFlags = cflags & ~CHANNEL_PASSIVE;
                }
            }
        }
    }
#if DEBUG
    if (wlanScanDebugLevel) {
        clistShow(pDevInfo);
    }
#endif
}

/***********************************************************
 * wlanClistNewScanCallback
 *
 * Modify the channel list given that we're about to start a
 * new scan.
 *
 * Clears the bssSeenHere flag for each channel in the list.
 */
void
wlanClistNewScanCallback(WLAN_DEV_INFO *pDev)
{
    WLAN_CHANNEL_LIST *pClist = pDev->staConfig.pClist;
    int               i;

    ASSERT(pClist);

    for (i = 0; i< pClist->listSize; i++) {
        pClist->chanArray[i].bssSeenHere = FALSE;
    }
}

/***********************************************************
 * wlanDoesDsElementMatch
 *
 * Returns TRUE if the DS element passed in described the
 * channel to which we are currently tuned or the pointer
 * is NULL, FALSE otherwise.
 */
A_BOOL
wlanDoesDsElementMatch(WLAN_DEV_INFO *pDev, DS_PARAM_SET *pDsParams)
{
    A_UCHAR ourChan;

    ASSERT(pDev && pDev->staConfig.phwChannel);

    ourChan = (A_UINT8)wlanConvertGHztoCh(pDev->staConfig.phwChannel->channel,
                                 pDev->staConfig.phwChannel->channelFlags);

    if (pDsParams == NULL || pDsParams->currentChannel == ourChan) {
        return TRUE;
    } else {
        return FALSE;
    }
}

/***********************************************************
 * wlanUnsupportedAdHocBand
 *
 * Return the unsupported adhoc frequency specification (band)
 * of the regulatory domain
 */
WLAN_CFLAGS
wlanUnsupportedAdHocBand(WLAN_DEV_INFO *pDevInfo)
{
    REG_DOMAIN  rd;
    int         i;
    WLAN_CFLAGS cflags = 0;
    A_UINT16    ccode;


    rd = wlanHalRD(pDevInfo);
    if (rd & COUNTRY_ERD_FLAG) {
        ccode = rd & ~COUNTRY_ERD_FLAG;
        rd    = allCountries[wlanGetCtryIdx(ccode)].regDmnEnum;
    }

    for (i = 0; i < ADHOC_DISALLOW_RD_TBL_LEN; i++) {
        if (adHocDisallowRdTbl[i].rd == rd) {
            cflags = adHocDisallowRdTbl[i].cflags;
            break;
        }
    }
    return cflags;
}

/***********************************************************
 * wlanUnsupportedAdHocMode
 *
 * Return the unsupported adhoc wireless mode of the
 * regulatory domain
 */
A_UINT16
wlanUnsupportedAdHocMode(WLAN_DEV_INFO *pDevInfo)
{
    REG_DOMAIN  rd;
    int         i;
    A_UINT16    ccode, mode = 0;

    rd = wlanHalRD(pDevInfo);
    if (rd == 0) {
        if (pDevInfo->staConfig.countryCode) {
            rd = allCountries[wlanGetCtryIdx(pDevInfo->staConfig.countryCode)].regDmnEnum;
        }
    }
    

    if (rd & COUNTRY_ERD_FLAG) {
        ccode = rd & ~COUNTRY_ERD_FLAG;
        rd    = allCountries[wlanGetCtryIdx(ccode)].regDmnEnum;
    }

    for (i = 0; i < ADHOC_DISALLOW_RD_TBL_LEN; i++) {
        if (adHocDisallowRdTbl[i].rd == rd) {
            mode = adHocDisallowRdTbl[i].mode;
            break;
        }
    }
    return mode;
}

/***********************************************************
 * wlanGetAdHocClist
 *
 * Return the adhoc channel list of the regulatory domain
 */
static WLAN_CHANNEL *
wlanGetAdHocClist(REG_DOMAIN rd)
{
    int i;

    for (i = 0; i < ADHOC_CHAN_RD_TBL_LEN; i++) {
        if (adhocChanRdTbl[i].rd == rd) {
            return adhocChanRdTbl[i].pClist;
        }
    }

    return 0;
}

A_BOOL
wlanIsAdhocRD(REG_DOMAIN rd, WLAN_CHANNEL wchan, WLAN_CFLAGS cflags)
{
    const REG_DMN_WM_FREQ_TABLE *pCcTbl;
    WLAN_CFLAGS  xcflags = 0;
    int          i, j, numRd;

    for (i = 0; i < ADHOC_DISALLOW_RD_TBL_LEN; i++) {
        if (adHocDisallowRdTbl[i].rd == rd) {
            xcflags = adHocDisallowRdTbl[i].cflags;
            break;
        }
    }

    if (!xcflags) {
        return TRUE;
    }

    if (xcflags == cflags) {
        return FALSE;
    }

    pCcTbl = wlanGetRdmfTable(cflags);
    numRd  = wlanGetRdmfTblLen(cflags);
    for (i = 0; i < numRd; i++) {
        for (j = 0; j < pCcTbl[i].entries; j++) {
            if ((wchan >= pCcTbl[i].chanGroup[j].lowChannel) &&
                (wchan <= pCcTbl[i].chanGroup[j].highChannel))
            {
                return FALSE;
            }
        }
    }
    return TRUE;
}

/***********************************************************
 * wlanIsPassiveScanChannel
 *
 * Return the passive channel scan frequency specification
 * of the regulatory domain
 */
static A_BOOL
wlanIsPassiveScanChannel(WLAN_DEV_INFO *pDevInfo, REG_DOMAIN rd, CHAN_VALUES *pCv)
{
    A_BOOL found = FALSE;
    int i;
    WLAN_CFLAGS cflags = pCv->channelFlags & CHANNEL_ALL;

    for (i = 0; i < PS_RD_TBL_LEN; i++) {
        if (passiveScanRdTbl[i].rd == rd) {
            if (passiveScanRdTbl[i].eepromRD == wlanHalRD(pDevInfo) &&
                passiveScanRdTbl[i].cflags == cflags)
            {
                found = TRUE;
                break;
            }

            if (passiveScanRdTbl[i].eepromRD == 0 &&
                passiveScanRdTbl[i].cflags == cflags)
            {
                found = TRUE;
                break;
            }
        }
    }
    if (found && (passiveScanRdTbl[i].pList != NULL)) {
        FCC_PSCAN_CLIST *pList = passiveScanRdTbl[i].pList;
        if (pList) {
            int j;
            found = FALSE;
            for (j = 0; j < pList->numClist; j++) {
                WLAN_CHANNEL *pClist = pList->pClTbl[j];
                while (*pClist) {
                    if (*pClist == pCv->channel) {
                        found = TRUE;
                        break;
                    }
                    pClist++;
                }
            }
        }
    }
    return found;
}

/***********************************************************
 * wlanIsDfsRD
 *
 * Return TRUE if the regulatory domain in question requires DFS
 */
A_BOOL
wlanIsDfsRD(WLAN_DEV_INFO *pDevInfo, REG_DOMAIN rd)
{
    int i;
    A_UINT32 regCaps = wlanGetRegCapabilities(pDevInfo);

    for (i = 0; i < DFS_RD_TBL_LEN; i++) {
        if (dfsRdTbl[i].rd == rd) {
            return TRUE;
        }
    }

    /* 
     * Depending upon the capabilities setting thru hardware (EEPROM) or SW (regCapMask).
     * We will need to check for MKK 5 GHz SKUs and legacy FCC SKUs (FCC 0x10)
     */
    if ((rd == MKK1) || (rd == MKK2)) {
        if ((regCaps & REG_KK_USE_U2) || (regCaps & REG_KK_USE_MIDBAND)) {
            return TRUE;
        }
    }

    if ((rd == FCC1) || (rd == FCC2)) {
        if (regCaps & REG_FCC_USE_MIDBAND) {
            return TRUE;
        }
    }

    return FALSE;
}

/***********************************************************
 * wlanIs11bCountry
 *
 * Return TRUE if the country allows only 11b
 */
A_BOOL
wlanIs11bCountry(WLAN_DEV_INFO *pDevInfo, CTRY_CODE ccode)
{
    REG_DOMAIN  rd;
    A_BOOL      status;

    rd     = wlanGetWmRD(pDevInfo, ccode, CHANNEL_B);
    status = wlanIs11bOnlyRD(rd);

    uiPrintf("is11b -- rd %x, cc: %d, status %d\n", rd, ccode, status);

    return status;
}

/***********************************************************
 * wlanIs11gCountry
 *
 * Return TRUE if the country allows only 11g
 */
A_BOOL
wlanIs11gCountry(WLAN_DEV_INFO *pDevInfo, CTRY_CODE ccode)
{
    REG_DOMAIN  rd;
    A_BOOL      status;

    rd     = wlanGetWmRD(pDevInfo, ccode, CHANNEL_B);
    status = wlanIs11gOnlyRD(rd);

    uiPrintf("is11g -- rd %x, cc: %d, status %d\n", rd, ccode, status);

    return status;
}

/***********************************************************
 * wlanIs11aAllowed
 *
 * Return TRUE if the regulatory domain allows 11a
 */
A_BOOL
wlanIs11aAllowed(WLAN_DEV_INFO *pDevInfo, CTRY_CODE ccode)
{
    REG_DOMAIN rd;

    rd = wlanGetWmRD(pDevInfo, ccode, CHANNEL_A);

    return wlanIs11bOnlyRD(rd) ? FALSE : TRUE;
}

/***********************************************************
 * wlanIs11bOnlyRD
 *
 * Return TRUE if the regulatory domain is 11b only
 */
A_BOOL
wlanIs11bOnlyRD(REG_DOMAIN rd)
{
    int i;

    for (i = 0; i < X11B_ONLY_RD_TBL_LEN; i++) {
        if (X11bOnlyRdTbl[i].rd == rd) {
            return TRUE;
        }
    }
    return FALSE;
}

/***********************************************************
 * wlanIs11gOnlyRD
 *
 * Return TRUE if the regulatory domain is 11g only
 */
A_BOOL
wlanIs11gOnlyRD(REG_DOMAIN rd)
{
    return wlanIs11bOnlyRD(rd);;
}

/***********************************************************
 * wlanGetSingletonCountryRD
 *
 * Return the country code of the regulatory domain
 * that is used by only one country
 */
static CTRY_CODE
wlanGetSingletonCountryRD(REG_DOMAIN rd)
{
    int i;

    for (i = 0; i < SC_RD_TBL_LEN; i++) {
        if (singleCountryRdTbl[i].rd == rd) {
            return singleCountryRdTbl[i].country;
        }
    }
    return 0;
}

/* 
 * This function returns a mask of the regulatory capabilities and the precedence
 * is assigned from EEPROM or software.
 */
A_UINT32
wlanGetRegCapabilities(WLAN_DEV_INFO *pDevInfo)
{
    A_UINT32            regCapBits;

    /* If EEPROM value is set (bit 1 is set) then return the value, 
     * otherwise return any software value set
     */
    regCapBits = pDevInfo->devCap.regCapBits;

    if ((!(regCapBits & REGCAPBITS_VALUE_SET)) && (!(regCapBits & REG_KK_USE_NEW_11A)))  {
        /* 
         * This device has legacy EEPROM and has not been updated to indicate it is certified
         * to operate under the new 11a Japan laws.  In this case we must continue to operate
         * on the funky odd channels, so bail out and return 0
         */
        return 0;
    }

    /* Clear out midband support, not approved by regulatory yet! */
    regCapBits &= ~(REG_FCC_USE_MIDBAND | REG_KK_USE_MIDBAND);

    /*
     * For legacy devices the UNI-1 Even support must be set based on if the "Japan New 11a"
     * flag is set.
     */
    if (!(regCapBits & REGCAPBITS_VALUE_SET)) {
        /*
         * For legacy devices the UNI-1 Even support must be set based on if the "Japan New 11a"
         * flag is set.  The HAL will only returns capabilities for "Japan New 11a" and "Odd 
         * UNI-1 channel support"
         */
        regCapBits |= (regCapBits & REG_KK_USE_NEW_11A) ? REG_KK_USE_EVEN_U1 : 0;
    }
    
    /* For EERPOM v5.3 or greater can not be certified under new rules to operate in ODD only
     * we will prevent this configuration by disabling odd support when no even support
     * is enabled
     */
    if (regCapBits & REGCAPBITS_VALUE_SET) {
        if (!(regCapBits & REG_KK_USE_EVEN_U1)) {
            regCapBits &= ~REG_KK_USE_ODD_U1;
        }
    }

    return regCapBits;

}

A_BOOL
wlanIs5211Channel14(SIB_ENTRY *pSib)
{
    WLAN_DEV_INFO *pDevInfo = sib2DevInfo(pSib);    

#if !NDIS_WDM    
    /* Current implementation won't check for country code MKK */
    if (pDevInfo->staConfig.pChannel->channel == CHANNEL_14 &&
        pDevInfo->devCap.chanSpreadSupport == FALSE)
    {
        uiPrintf("channel 14 -- no hw support\n");
        return TRUE;
    }
#endif

    return FALSE;
}

A_BOOL
wlanVerifyEepromCcRd(WLAN_DEV_INFO *pDevInfo)
{
    A_UINT16 eepRdCc = wlanHalRD(pDevInfo);    
    A_UINT16 ccode, index;

    if (eepRdCc & COUNTRY_ERD_FLAG) {
        ccode = eepRdCc & ~COUNTRY_ERD_FLAG;
        for (index = 0; index < NUM_OF_COUNTRIES; index++) {
            if (ccode == allCountries[index].countryCode) {
                return TRUE;
            }
        }
    } else {
        for (index = 0; index < NUM_OF_REG_DOMAINS; index++) {
            if (eepRdCc == allEnumRds[index].regDmnEnum) {
                return TRUE;
            }
        }
    }
    uiPrintf("wlanVerifyEepromCcRd -- invalid value %#x\n", eepRdCc);
    return FALSE;
}

#ifdef BUILD_AP
void
wlanSiteSurveySetup(WLAN_DEV_INFO *pDevInfo)
{
    
    apScan(pDevInfo, AP_SCAN_SITE_SURVEY,
           wlanIsActiveScanAllowed(pDevInfo) ? ACTIVE_SCAN : PASSIVE_SCAN);

    /* Restore overlapping bss protection if enabled to do so */
    if (apCfgFreqSpecGet(pDevInfo->devno) == MODE_SELECT_11G &&
        apCfg11gOverlapBssGet(pDevInfo->devno))
    {
        halRxFilter(pDevInfo, HAL_RX_BEACON, HAL_RX_FILTER_SET);
    }
}
#endif

/*
 * Allow some extra TURBO_PRIME channels for debug purposes prior
 * to the 3.0 release.
 */
#define TURBO_DEBUG_CHANNELS 1

/***********************************************************
 * wlanIsTurboPrimeChannel
 *
 * Check if channel is Turbo Prime capable.
 *
 */
A_BOOL
wlanIsTurboPrimeChannel(WLAN_DEV_INFO *pDevInfo, CHAN_VALUES *pChannel)
{

    if (pDevInfo->staConfig.abolt & ABOLT_TURBO_PRIME &&
        pDevInfo->staConfig.opMode == OP_MODE_AP      &&
        (pChannel->channel == 2437  ||   /*   6 */ 
#ifdef TURBO_DEBUG_CHANNELS
         pChannel->channel == 2312  ||   /* -19 */
         pChannel->channel == 2367  ||   /*  -8 */
         pChannel->channel == 2512  ||   /*  15 */
         pChannel->channel == 2572  ||   /*  18 */
#endif
         pChannel->channel == 5200  ||   /*  40 */
         pChannel->channel == 5240  ||   /*  48 */
         pChannel->channel == 5280  ||   /*  56 */
         pChannel->channel == 5765  ||   /* 153 */
         pChannel->channel == 5805) &&   /* 161 */
        (IS_CHAN_G(pChannel->channelFlags)  ||
         IS_CHAN_A(pChannel->channelFlags)))
    {
        return TRUE;
    } else {
        return FALSE;
    }
}
/***********************************************************
 * wlanIsMKKSku
 *
 * The parameter is what is retrieved from halGetCapabilities[HAL_GET_REG_DMN].
 */
A_BOOL
wlanIsMKKSku(A_UINT16 regDmn)
{
    A_BOOL  retValue = FALSE;

    switch (regDmn) {
    case MKK1_MKKA:
    case MKK1_MKKB:
    case MKK2_MKKA:
    case MKK1_FCCA:
    case MKK1_MKKA1:
    case MKK1_MKKA2:
    case MKK1_MKKC:
    case (CTRY_JAPAN | COUNTRY_ERD_FLAG):
    case (CTRY_JAPAN1 | COUNTRY_ERD_FLAG):
    case (CTRY_JAPAN2 | COUNTRY_ERD_FLAG):
    case (CTRY_JAPAN3 | COUNTRY_ERD_FLAG):
    case (CTRY_JAPAN4 | COUNTRY_ERD_FLAG):
    case (CTRY_JAPAN5 | COUNTRY_ERD_FLAG):
    case (CTRY_JAPAN6 | COUNTRY_ERD_FLAG):
        retValue = TRUE;
    }
    return retValue;
}

/**************************************************************
 * wlanApplyMKKIrreversibleUpdate
 *
 */
A_STATUS
wlanApplyMKKIrreversibleUpdate(WLAN_DEV_INFO *pDevInfo, A_UINT32 enOddU1, A_UINT32 undoUpdate)
{
    A_STATUS          status;
	A_UINT16          regDmn;

    DEV_CAPABILITIES *pCap = &(pDevInfo->devCap);

    regDmn = wlanHalRD(pDevInfo);

    if (!wlanIsMKKSku(regDmn)) {
        return A_OK; /* The device is not a MKK SKU or is already not required */
    }

    status = wdcApplyMKKIrreversibleUpdate(pDevInfo->targetHandle, enOddU1, undoUpdate);

    /* 
     * Change the HAL regulatory capabilities to reflect certified under MKK new 11a rules
     * and Odd UNI-1 KK support is based on config data, first default it to enabled.
     */
    pCap->regCapBits |= (REG_KK_USE_NEW_11A | REG_KK_USE_ODD_U1);
    if (!enOddU1) {
        pCap->regCapBits &= ~REG_KK_USE_ODD_U1;
    }

    return status;
}
A_BOOL
wlanVerifyUserRd(REG_DOMAIN regDomain)
{
	int index;

	for( index=0; index < NUM_OF_REG_DOMAINS; index++ ) {
		if (regDomain == allEnumRds[index].regDmnEnum ) {
			return TRUE;
		}
	}

	return FALSE;
}

