/*
 * Copyright  2000-2003 Atheros Communications, Inc.,  All Rights Reserved.
 */
#ifndef _CCX_DEFS_H_
#define _CCX_DEFS_H_

/* For use with a Beacon Measurement Request Specification */
typedef enum measScanMode {
    PASSIVE,
    ACTIVE,
    BEACON_TABLE
} MEAS_SCAN_MODE;

typedef enum radioMeasType {
    MEAS_UNUSED = 0,
    MEAS_CHANNEL_LOAD,
    MEAS_NOISE_HISTOGRAM,
    MEAS_BEACON,
    MEAS_FRAME,
    MEAS_HIDDEN_NODE
} CCX_MEASUREMENT_TYPE;

/*
 * CCX "ether types", i.e., frame types recognized within the Aironet
 * SNAP domain.
 */
typedef enum ccxEtherType {
    CCX_ETHTYPE_NULL        = 0x0000,
    CCX_ETHTYPE_CKIP        = 0x0002
} CCX_ETHER_TYPE;

/* DDP Message types */
typedef enum ddpMessageType {
    DDP_MSG_REPORT          = 0x30,
    DDP_MSG_RADIO_MEAS      = 0x32,
    DDP_MSG_ROGUEAP         = 0x40
} DDP_MSG_TYPE;

/* DDP Function types */
typedef enum ddpFunctionType {
    DDP_FUNC_NULL           = 0x00,
    DDP_FUNC_MEAS_REQUEST   = 0x01,
    DDP_FUNC_MEAS_REPORT    = 0x81,
    DDP_FUNC_ROGUEAP        = 0x8E
} DDP_FUNC_TYPE;

/* Element ID's for CCX */
typedef enum ccxElementId {
    ID_MEAS_REQUEST         = 38,
    ID_MEAS_REPORT          = 39,
    ID_AP_TX_POWER          = 150,
    ID_RADIO_MGMT_CAP       = 221
} CCX_ELEMENT_ID;

/* DDP Report tag codes */
typedef enum ddpReportCode {
    DDP_RPT_TAG_NEARBY_AP   = 0x9B
} DDP_REP_CODE;

/* Sadly, these do not map to the official Cisco DDP codes */
typedef enum {
    MH_CCX_RAP_TIMER_EXPIRE = 1,
    MH_CCX_RAP_AP2STA_CHALLENGE_FAIL,
    MH_CCX_RAP_STA2AP_CHALLENGE_FAIL,
    MH_CCX_RAP_AUTH_SUCCESS,
    MH_CCX_RAP_NEW_CREDENTIALS,
} MH_CCX_RAP_FAIL_CODE;

/* These map to the official settings for rogue AP. */
typedef enum {
    DDP_RAP_AUTH_SUCCESS          = 0, // hunky-dory
    DDP_RAP_UNSUPPORTED_ALG       = 1, // AP did not use LEAP alg for dot11 auth
    DDP_RAP_AUTH_TIMER_EXPIRE     = 2, // overall LEAP timer expriation
    DDP_RAP_CHALLENGE_FROMAP_FAIL = 3, // Never got EAP-Success
    DDP_RAP_CHALLENGE_TOAP_FAIL   = 4, // Got EAP-Success, but bad resp from AP
} DDP_RAP_AUTH_STATUS_CODE;

typedef struct measQEntry_s {
    TAILQ_ENTRY(measQEntry_s) link;
    A_UINT16         token;      // Unique among request elements in frame
    A_UINT8          mode;       // Bit field
    A_UINT8          type;       // See CCX_MEASUREMENT_TYPE: one of SPEC_XXX

    A_UINT8          channelNo;
    A_UINT8          spare;
#define scanMode spare
    A_UINT16         duration;
    A_TIMER          durationTimer;
} MEAS_Q_ENTRY;
typedef TAILQ_HEAD(measQEntryHead_s, measQEntry_s) MEAS_Q_HEAD;

typedef struct measSchedQueue_s {
    TAILQ_ENTRY(measSchedQueue_s) link;    // For queue manipulation
    MEAS_Q_HEAD           measQueueHead;   // ditto
    WLAN_DEV_INFO         *pDev;           // "Dispatch" function needs this
    A_TIMER               startTimer;      // Timer for dispatch function
    A_BOOL                done;            // Can this request chain be deleted
    A_UINT32              numItems;        // num of requests
    A_UINT8               activationDelay; // # of TBTT's before start
    A_UINT8               measOffset;      // start time in TU's after
    A_UINT16              dialogToken;     // Like a transaction id
} MEAS_SCHED_Q;
typedef TAILQ_HEAD(schedQueueHead_s, measSchedQueue_s) MEAS_SCHED_Q_HEAD;


/*
 * Radio Measurement Frame bodies
 */

/*
 * CCA asserted time as a percentage of measurement interval
 * AFAIK, Identical to TGh draft 3.3.4
 */
typedef struct reportChannelLoad {
    A_UINT8        ccaBusyFraction;
} __ATTRIB_PACK SPEC_CHANNEL_LOAD;

/*
 * RPI (received power indication) binned over dB range
 * AFAIK, Identical to TGh draft 3.3.4
 */
typedef struct reportNoiseHistogram {
    A_UINT8        rpiDensity[8];
} __ATTRIB_PACK SPEC_NOISE_HISTOGRAM;

/*
 * Beacons and probe responses received on channel of interest
 * Not in TGh
 */
typedef struct reportBeaconInfo {
    A_UINT8        phyType;
    A_UINT8        rssi;
    WLAN_MACADDR   bssid;
    A_UINT32       parentTsf;
    A_UINT32       targetTsf;
    A_UINT16       beaconInterval;
    A_UINT16       capabilityInfo;
    A_UINT8        recvdElements[0];
} __ATTRIB_PACK SPEC_BEACON_INFO;

/*
 * Number of frames not in my BSS over measurement interval
 * Not in TGh
 */
typedef struct reportFrameInfo {
    WLAN_MACADDR   txAddr;
    WLAN_MACADDR   bssid;
    A_UINT8        rssi;
    A_UINT8        cntFrames;
    A_UINT32       quadruplet[1]; // pg. 42
} __ATTRIB_PACK SPEC_FRAME_INFO;


/*
 * CCX Information Elements
 */

typedef struct mgmtElement {
    A_UINT16        tag;        // Id or tag
    A_UINT16        length;     // Length of ele not including these 4 bytes
} __ATTRIB_PACK MGMT_ELEMENT;

#define AP_LIST_ELE_SIZE        20  // doesn't include SSID chars
typedef struct apListElement {
    MGMT_ELEMENT     ele;
    A_UINT8          oui[4];
    WLAN_MACADDR     bssid;
    A_UINT16         channel;
    A_UINT16         ssidLength;
    A_UINT8          ssid[0];
    /*
     * There's actually a timeSinceLastDisassoc that sits here but its location
     * is dependent on the size of the SSID.  So we have a separate typedef to
     * use for this.
     */
} __ATTRIB_PACK AP_LIST_ELEMENT;

typedef struct RogueApElement {
    A_UINT16        failureCode;
    WLAN_MACADDR    macAddr;
    A_UCHAR         rogueName[16];
} __ATTRIB_PACK ROGUE_AP_ELEMENT;

typedef struct radioMeasElement {
    MGMT_ELEMENT     ele;
    A_UINT16         token;      // Unique among request elements in frame
    A_UINT8          mode;       // Bit field
    A_UINT8          type;       // See CCX_MEASUREMENT_TYPE: one of SPEC_XXX

    /* Specification of the measurement request/report */
    A_UINT8          channelNo;
    A_UINT8          spare;
#define scanMode spare
    A_UINT16         measDuration;

    /* The specific measurement spec follows here */

} __ATTRIB_PACK RADIO_MEAS_ELEMENT;

/*
 * CCX Frame Headers
 */

/* DDP Header */
typedef struct DdpHeader {
    A_UINT16        ddpId     : 4,
                    ddpLength : 12;
    A_UINT8         msgType;
    A_UINT8         funcCode;
    WLAN_MACADDR    destAddr;
    WLAN_MACADDR    srcAddr;
} __ATTRIB_PACK DDP_HEADER;

/* Measurement Request Header */
typedef struct measRequestHeader {
    A_UINT16           dialogToken;     // Like a transaction id
    A_UINT8            activationDelay; // # of TBTT's before start
    A_UINT8            measOffset;      // start time in TU's after
                                        // activationDelay expires
} __ATTRIB_PACK MEAS_REQUEST_HEADER;

/* Measurement Report Header */
typedef struct measReportHeader {
    A_UINT16           dialogToken;     // Matches measurement req or 0
    A_UINT8            measMode;        // Bit field
    A_UINT8            measType;        // See CCX_MEASUREMENT_TYPE
} __ATTRIB_PACK MEAS_REPORT_HEADER;

typedef struct measHeader {
    union {
        MEAS_REQUEST_HEADER  request;
        MEAS_REPORT_HEADER   report;
    } h;
    RADIO_MEAS_ELEMENT element[1];      // array of elements
} __ATTRIB_PACK CCX_MEAS_HEADER;

/*
 * This struct hangs off of the device info struct to keep track of all things
 * CCX-related.
 */
typedef struct CcxInfo {
    /* Used for Fast Roaming Assist */
    A_UINT16        lastChannel;
    WLAN_MACADDR    lastBssid;
    SSID            lastSsid;
    A_UINT32        disassocTime;       // in ms
    A_BOOL          sentLastApFrame;

    /* Rogue AP stuff */
#define ROGUE_LIST_NELEMENTS 8
    ROGUE_AP_ELEMENT rogueList[ROGUE_LIST_NELEMENTS];

    /* Preferred AP stuff */
    A_UINT32        numPreferredAps;
    WLAN_MACADDR    prefBssid[MAX_PREFERRED_APS];

    /* Radio measurements stuff */
    MEAS_SCHED_Q_HEAD schedQueueHead;
    A_SEM_TYPE        schedQueueLock;   // not used now
    A_UINT8           startTime;        // # of TBTT's 'til next measurement
    A_UINT8           startOffset;      // # of TU's 'til next measurement

    /* ccx statistics */
    struct ccxStats {
        A_UINT32    cntRxMeasRequestFrames;
        A_UINT32    cntTxMeasRequestFrames;
        A_UINT32    cntRxMeasReportFrames;
        A_UINT32    cntTxMeasReportFrames;
        A_UINT32    cntMeasRequestsCancelled;
        A_UINT32    cntMeasRequestsRefused;
    } stats;
} CCX_INFO;
    
/*
 * CCKM Definitions
 */
#define ELE_CCKM_REQ_SIZE       24      /* CCKM Request element size */
#define CCKM_OUI_TYPE         0x00

/*
 * CCKM Association Request Element
 * A Station desiring CCKM fast reassociation will include this IE in the
 * Association Request. See CCX2.X document for more info.
 * timestamp is the current TSF timer value (least sig byte first)
 * reqnumber is the reassociation req number (least sig byte first)
 * mic = HMAC-MD5(KRK, STA-ID | BSSID | RSNIE | Timestamp | RN)
 */
typedef struct cckmReqIE {
    A_UINT8             cr_elementID;
    A_UINT8             cr_length;
    A_UINT8             cr_oui[3];
    A_UINT8             cr_ouiType;
    WLAN_TIMESTAMP      cr_timestamp;      /* 64 bit timestamp */
    A_UINT32            cr_reasnumber;
    A_UINT8             cr_mic[8];
} __ATTRIB_PACK CCKM_REQ_IE;

#define VALID_CCKM_REQ_ELEMENT(_cckmreqep)                     \
        (   ((_cckmreqep)->cr_elementID == ELE_CCKM)       &&  \
            ((_cckmreqep)->cr_length == ELE_CCKM_REQ_SIZE) &&  \
            ((_cckmreqep)->cr_oui[0] == 0x00)              &&  \
            ((_cckmreqep)->cr_oui[1] == 0x40)              &&  \
            ((_cckmreqep)->cr_oui[2] == 0x96)              &&  \
            ((_cckmreqep)->cr_ouiType == 0x00) )

/*
 * CCKM Association Response Element
 * An AP supporting CCKM fast reassociation will include this IE in the
 * Association Response to a request with a CCKM Element. See CCX 2.X doc for
 * more information.
 * reqnumber is the reassociation req number (least sig byte first)
 * mic = HMAC-MD5(802.1x-mic-key, STA-ID | RSNIE | RN | KeyID-unicast 
 *      | KeyID-multicast | RSC | Multicast-key Length | EGTK)
 */
typedef struct cckmResIE {
    A_UINT8             cr_elementID;
    A_UINT8             cr_length;
    A_UINT8             cr_oui[3];
    A_UINT8             cr_ouiType;
    A_UINT32            cr_reqnumber;
    A_UINT8             cr_uKeyId;              // unicast
    A_UINT8             cr_mKeyId;              // multicast
    A_UINT8             cr_rsc[8];              // replay sequence counter
    A_UINT16            cr_mKeyLength;          // multicast key length
    A_UINT8             cr_mic[8];
    A_UINT8             cr_egtk[1];             // encrypted group key
} __ATTRIB_PACK CCKM_RESP_IE;

#define VALID_CCKM_RESP_ELEMENT(_cckmresep)                        \
        (   ((_cckmresep)->cr_elementID == ELE_CCKM)           &&  \
            ((_cckmresep)->cr_length >= sizeof (CCKM_RESP_IE)) &&  \
            ((_cckmresep)->cr_oui[0] == 0x00)                  &&  \
            ((_cckmresep)->cr_oui[1] == 0x40)                  &&  \
            ((_cckmresep)->cr_oui[2] == 0x96)                  &&  \
            ((_cckmresep)->cr_ouiType == 0x00) )

typedef struct athSuplAuthStatus {
    A_UINT32              length; // total length, incl length field
#define ATH_SUPL_AUTH_STATUS_API_VERSION            1
    A_UINT32              version; // version of this OID
    MH_CCX_RAP_FAIL_CODE  code;
    WLAN_MACADDR          Bssid;  // BSSID of failing AP
} MH_SUPP_AUTH_STATUS, *PMH_SUPP_AUTH_STATUS;


/* MACROS */

/*
 * If extended authentication (i.e., above 802.11) is used, then we need to wait
 * to send DDP frames until we have keys so that the frames won't be dropped
 * either by our STA or by the AP.
 */
#define ABLE_TO_PASS_TRAFFIC(x)                                         \
            (!(x)->staConfig.privacyInvoked                 ||          \
             ((x)->staConfig.defaultKey != KEYINDEX_INVALID &&          \
              (x)->keyTable[(x)->staConfig.defaultKey].keyLength != 0))

#endif /* #ifndef _CCX_DEFS_H_ */
