/*
 * Copyright (c) 2000-2002 Atheros Communications, Inc., All Rights Reserved
 *
 * Definitions for some performance debugging histograms
 *  - compiled in for debug only
 * Should work on most platforms
 * Defined as macros to be extensible in capability
 * See ar5hwcEnd.c for examples of usage
 *
 * $Id: //depot/sw/branches/1.3_USB_LINUX_port/src/USB/wlan/host/include/histograms.h#1 $
 */

#ifndef _HISTOGRAMS_H_
#define _HISTOGRAMS_H_

#ifdef AR531X 
#define HISTOGRAM_TIMER_TICK sysTickGet
#else
#define HISTOGRAM_TIMER_TICK A_MS_TICKGET
#endif

#ifndef NDEBUG

/*
 * use HISTOGRAM_DEFINE to declare and define histgram bins
 * giving them a name; ndev is number of devices used to get the
 * named bin per device; range provides the value range (0-range]
 * which needs binning; logBinSz is the log of the bin size
 */
#define HISTOGRAM_DEFINE(_name, _ndev, _range, _logBinSz)                   \
    LOCAL A_UINT32 _name##HistBin_[_ndev][(_range) >> (_logBinSz)];         \
    LOCAL A_UINT32 _name##HistMax_[_ndev];                                  \
    LOCAL A_UINT32 _name##HistLast_[_ndev];                                 \
    LOCAL A_UINT32 _name##HistRange_ = (_range);                            \
    LOCAL A_UINT32 _name##HistLogBinSz_ = (_logBinSz);                      \
          A_UINT32 _name##HistCnt_

/* bin a given value into the named bins for device number dev */
#define _CAPTURE_MAX(_max, _val)                                            \
    ((_max) = ((_max) > (_val)) ? (_max) : (_val))
#define HISTOGRAM_BIN_VALUE(_name, _dev, _val)                      do {    \
    (++(_name##HistBin_[_dev][                                              \
        ((((_val) >= (_name##HistRange_))                                   \
            ? ((_name##HistRange_) - 1)                                     \
            : (_val))                                                       \
        >> _name##HistLogBinSz_)]));                                        \
    _CAPTURE_MAX(_name##HistMax_[_dev], _val);                              \
    _name##HistLast_[_dev] = (_val);                                        \
} while (0)

/*
 * Counter bins: API is start...[count]+...stop
 */
#define HISTOGRAM_START_COUNTER(_name)                              do {    \
    extern A_UINT32 _name##HistCnt_;                                        \
    _name##HistCnt_ = 0;                                                    \
} while (0)
#define HISTOGRAM_COUNT(_name)                                      do {    \
    extern A_UINT32 _name##HistCnt_;                                        \
    (++(_name##HistCnt_));                                                  \
} while (0)
#define HISTOGRAM_STOP_COUNTER(_name, _dev)                                 \
    HISTOGRAM_BIN_VALUE(_name, _dev, _name##HistCnt_)

/*
 * Timer bins: API is start...stop
 * assume that the platform intending to use this has defined a function
 * corresponding to HISTOGRAM_TIMER_TICK - return the ticks in appropriate
 * units
 */
#define HISTOGRAM_START_TIMER(_name)                                do {    \
    extern A_UINT32 _name##HistCnt_;                                        \
    _name##HistCnt_ = HISTOGRAM_TIMER_TICK();                               \
} while (0)
#define HISTOGRAM_STOP_TIMER(_name, _dev)                           do {    \
    extern A_UINT32 _name##HistCnt_;                                        \
    _name##HistCnt_ = (HISTOGRAM_TIMER_TICK() - _name##HistCnt_);           \
    HISTOGRAM_BIN_VALUE(_name, _dev, _name##HistCnt_);                      \
} while (0)

/* dump histograms for given device */
#define HISTOGRAM_DUMP(_s, _name, _dev)                             do {    \
    A_UINT32 n;                                                             \
                                                                            \
    uiPrintf("Histograms for %s on dev %d\n", _s, _dev);                    \
    for (n = 0; n < (_name##HistRange_ >> _name##HistLogBinSz_); ++n) {     \
        if (_name##HistBin_[_dev][n]) {                                     \
            uiPrintf(" %04d-%04d: %d\n",                                    \
                n << _name##HistLogBinSz_,                                  \
                ((n + 1) << _name##HistLogBinSz_) - 1,                      \
                _name##HistBin_[_dev][n]);                                  \
            _name##HistBin_[_dev][n] = 0;                                   \
        }                                                                   \
    }                                                                       \
    uiPrintf(" Max Value: %d\n", _name##HistMax_[_dev]);                    \
    uiPrintf("Last Value: %d\n", _name##HistLast_[_dev]);                   \
    _name##HistMax_[_dev] = 0;                                              \
} while (0)

/*
 * Profiling help: works for functions! uses gcc built-ins to put
 * wrappers around functions and operate timer histograms on them
 */
#ifdef __GNUC__
#define HISTOGRAM_PROFILE(_type, _func, _args)                              \
HISTOGRAM_DEFINE(_func##_timer, 1, 256, 2);                                 \
void _func##Prof() { HISTOGRAM_DUMP(#_func" Profile", _func##_timer, 0); }  \
_type _func _args {                                                         \
    extern void _##_func();                                                 \
    void *result;                                                           \
    HISTOGRAM_START_TIMER(_func##_timer);                                   \
    result = __builtin_apply(_##_func, __builtin_apply_args(), 32);         \
    HISTOGRAM_STOP_TIMER(_func##_timer, 0);                                 \
    __builtin_return(result);                                               \
}                                                                           \
_type _##_func _args
#else
#define HISTOGRAM_PROFILE(_type, _func, _args) _type _func _args
#endif

#else
#define HISTOGRAM_DEFINE(_name, _ndev, _range, _logBinSz)
#define HISTOGRAM_BIN_VALUE(_name, _dev, _val)
#define HISTOGRAM_START_COUNTER(_name)
#define HISTOGRAM_COUNT(_name)
#define HISTOGRAM_STOP_COUNTER(_name, _dev)
#define HISTOGRAM_START_TIMER(_name)
#define HISTOGRAM_STOP_TIMER(_name, _dev)
#define HISTOGRAM_DUMP(_s, _name, _dev)
#define HISTOGRAM_PROFILE(_type, _func, _args) _type _func _args
#endif

#endif
