/*
 * Copyright  2000-2002 Atheros Communications, Inc.,  All Rights Reserved.
 *
 * This file contains the function prototypes and data structure declarations
 * specific to encapsulation/de-capsulation functions.  It is the same
 * for both the station and access point.
 *
 * $Id: //depot/sw/branches/1.3_USB_LINUX_port/src/USB/wlan/host/include/wlanCapDe.h#1 $
 */

#ifndef _WLANCAPDE_H_
#define _WLANCAPDE_H_

#ifdef __cplusplus
extern "C" {
#endif

struct wlanDevInfo;
struct atherosDesc;
struct queueDetails;

/* Encapsulation header for RFC1042 and Ethenet_tunnel */

#define RFC1042OUI        {{0xAA, 0xAA, 0x03, 0x00, 0x00, 0x00}}
#define ETHERNET_TUNNEL   {{0xAA, 0xAA, 0x03, 0x00, 0x00, 0xF8}}
#define AIRONET_SNAP      {{0xAA, 0xAA, 0x03, 0x00, 0x40, 0x96}}
#define ATHEROS_CAP_CONST {{0xAA, 0xAA, 0x03, 0x00, 0x03, 0x7F}}


/* Selective Translation protocol definition */

#define APPLE_TALK_ARP          0x80f3
#define NOVELL_IPX              0x8137
#define IPV4                    0x0800
#define CKIP_TYPE_FIELD         0x0002  /* used inside Aironet SNAP domain */

#define IS_ETHERTYPE(_typeOrLen)    ((_typeOrLen) >= 0x0600)

/********************************************************************
 * VLAN Support
 */

#define ETHERTYPE_802_1Q    0x8100

typedef struct VlanHdr {
#ifdef BIG_ENDIAN
    A_UINT8     userPri:3,
                cfi:1,
                vidHi:4;
#else
    A_UINT8     vidHi:4,
                cfi:1,
                userPri:3;
#endif
    A_UINT8     vidLo;
    A_UINT16    typeOrLen;
} __ATTRIB_PACK VLAN_HDR;

typedef A_UINT16 VLAN_ID;
#define TCI_GET_VID(_hdr)                                                   \
    ((((VLAN_ID)((_hdr)->vidHi)) << 8) | ((VLAN_ID)((_hdr)->vidLo)))
#define TCI_SET_VID(_hdr, _vid) {                                           \
    (_hdr)->vidHi = (A_UINT8)(((_vid) >> 8) & 0x0f);                        \
    (_hdr)->vidLo = (A_UINT8)((_vid) & 0xff);                               \
}


/********************************************************************
 * Framework for Atheros Layer 2 Protocols.
 *   Real EtherType requested from IEEE
 *   Minimal generic 32 bit header - affords 26 bits of immediate hdr
 *   for the specific protocol; the specific protocol itself may add
 *   more hdr bytes
 * First Atheros Layer 2 Protocol - 802.3 Tunnel over 802.11
 *   Used for supporting Fast Frames
 *   proto       is ATHL2P_802_3_TUNNEL
 *   frameType   is FRAME_TYPE_FAST_FRAME
 *   optHdrLen32 is additional header size in 32 bit words to maintain
 *                  backwards compatability on future addition of any
 *                  header options for the 802_3_TUNNEL protocol
 *   seqNum      is expected to increment by one each time to help
 *                  aggregation of 802.3 payload which may be split
 *                  across multiple fast frames (in order delivery is
 *                  required and assumed for aggregation)
 *   offset      is the remaining bytes of any partial 802.3 frame
 *                  sent in the last fast frame payload; it is used
 *                  to discard partially received headerless 802.3 frames
 */
#define ETHERTYPE_ATHL2P        0x88BD  /* assigned by IEEE */
typedef struct {
#ifdef BIG_ENDIAN
    A_UINT32 proto :  6,
             param : 26;
#else
    A_UINT32 param : 26,
             proto :  6;
#endif
} __ATTRIB_PACK ATHL2P_HDR;

#define ATHL2P_802_3_TUNNEL     0x0000
#define FRAME_TYPE_FAST_FRAME   0x0000
typedef struct {
#ifdef BIG_ENDIAN
    A_UINT32 proto       :  6,
             frameType   :  2,
             optHdrLen32 :  2,
             seqNum      : 11,
             offset      : 11;
#else
    A_UINT32 offset      : 11,
             seqNum      : 11,
             optHdrLen32 :  2,
             frameType   :  2,
             proto       :  6;
#endif
} __ATTRIB_PACK ATHL2P_802_3_TUNNEL_HDR;

#define CPU2BE_ATHL2P(_pHdr) do {                                       \
    (*((A_UINT32 *)(_pHdr))) = cpu2be32(*((A_UINT32 *)(_pHdr)));    \
} while (0)

#define BE2CPU_ATHL2P(_pHdr) do {                                       \
    (*((A_UINT32 *)(_pHdr))) = be2cpu32(*((A_UINT32 *)(_pHdr)));    \
} while (0)


/*************************************************************************
 * wlanDataFrameTransmit - entry point for transmit of a single data
 *   frame; called by the AP or STA code when the incoming buffers
 *   from the OS have been converted to ATHEROS_DESC representation.
 *   The sib corresponding to the destination should have been looked
 *   up and validated for a connection/association
 *   The frame still looks like a 802.3/Ethernet frame
 * Is a part of the encapsulation API since its still dealing with
 *   802.3 type of frames; the new generation stuff will directly deal
 *   with native 802.11 frames and would therefore directly call the
 *   core API in wlanSend.c
 *
 * RETURNS: error status
 *   Discards the frame on error
 */
A_STATUS
wlanDataFrameTransmit(struct wlanDevInfo *pDev, struct atherosDesc *pDesc);

A_STATUS
txFFDrain(struct wlanDevInfo *pDev, struct queueDetails *pQueue);

void
txFfFlush(WLAN_DEV_INFO *pDev);

void
ffInit(WLAN_DEV_INFO *pDev);

A_STATUS
rxNullHandler(WLAN_DEV_INFO *pDev, ATHEROS_DESC *pDesc,
                       A_UINT16 etherType);

/*******************************************************************************
 * Buffer management utilties - helpful for encap/decap code
 * NOTE: Currently only inlined in capde.c
 */
INLINE A_UINT32
bufferHeadroom(ATHEROS_DESC *pDesc);

INLINE void *
bufferPushHdr(ATHEROS_DESC *pDesc, A_UINT32 len);

INLINE void *
bufferPopHdr(ATHEROS_DESC *pDesc, A_UINT32 len);

INLINE void *
bufferGetHdr(ATHEROS_DESC *pDesc, A_UINT32 len);

/*************************************************************************
 * wlanDataFrameReceive - entry point for receive of a single data
 *   frame; decaps it and passes it up the stack
 */

A_STATUS
wlanDataFrameReceive(struct wlanDevInfo *pDev, struct atherosDesc *pDesc);

#define rxFFDrain(_pDev, _pSib) do {                        \
    if ((_pSib) && (_pSib)->pRxFFDesc) {                    \
        (_pSib)->stats.RxDiscardFrames++;                   \
        freeBuffandDescChain(_pDev, (_pSib)->pRxFFDesc);    \
        (_pSib)->pRxFFDesc = NULL;                          \
    }                                                       \
} while (0)


#ifdef VXWORKS
#include "wlanapDs.h"
#define athFrameReceive wlanDsFromWM
#else
#define athFrameReceive staFrameReceive
#endif

typedef struct _snapHandler {
    CAP_CONST   capConst;
    A_STATUS    (*pRxFunc)(WLAN_DEV_INFO *, ATHEROS_DESC *, UINT16 etherType);
} SNAP_HANDLER;

#define AIRONET_HANDLER { AIRONET_SNAP, ccxProcessRxFrame }

#ifdef __cplusplus
}
#endif

#endif /* _WLANCAPDE_H_ */

