/*
 * Copyright © 2000-2002 Atheros Communications, Inc.,  All Rights Reserved.
 *
 * This file contains the function prototypes and data structure declarations
 * used by channel list creation and regulatory domain functions.
 *
 * $Id: //depot/sw/branches/1.3_USB_LINUX_port/src/USB/wlan/host/include/wlanchannel.h#5 $
 */

#ifndef _WLAN_CHANNEL_H_
#define _WLAN_CHANNEL_H_

#ifdef __cplusplus
extern "C" {
#endif

/* Extra information about channels detected during hardware operation. */
#define CHANNEL_CW_INT    0x0001   // CW interference detected on the channel
#define CHANNEL_RADAR_INT 0x0002   // Radar interference detected on the channel
#define CHANNEL_BUSY      0x0004   // Busy, occupied directly or overlap from BSS on adjoining channel
#define CHANNEL_DONT_SCAN 0x0008   // Don't scan this channel - set if channel is invalid in this SKU
#define CHANNEL_PASSIVE   0x0010   // Only passive scan allowed in the channel

#define CHANNEL_TURBO     0x0100   // Turbo Channel
#define CHANNEL_CCK       0x0200   // CCK channel
#define CHANNEL_OFDM      0x0400   // OFDM channel
#define CHANNEL_XR        0x0800   // XR channel

#define CHANNEL_2GHZ      0x1000   // 2 GHz spectrum channel.
#define CHANNEL_5GHZ      0x2000   // 5 GHz spectrum channel
#define CHANNEL_4GHZ      0x4000   // 4 GHz Base.

#define CHANNEL_ALL       (CHANNEL_OFDM | CHANNEL_CCK | CHANNEL_TURBO | CHANNEL_XR | CHANNEL_5GHZ | CHANNEL_2GHZ)

#define CHANNEL_A         (CHANNEL_5GHZ | CHANNEL_OFDM)
#define CHANNEL_B         (CHANNEL_2GHZ | CHANNEL_CCK)
#define CHANNEL_G         (CHANNEL_2GHZ | CHANNEL_OFDM)     /* xxx: set the CCK flag also */
#define CHANNEL_PUREG     (CHANNEL_2GHZ | CHANNEL_OFDM)
#define CHANNEL_T         (CHANNEL_5GHZ | CHANNEL_OFDM | CHANNEL_TURBO)
#define CHANNEL_108G      (CHANNEL_2GHZ | CHANNEL_OFDM | CHANNEL_TURBO)
#define CHANNEL_XR_A      (CHANNEL_A     | CHANNEL_XR)
#define CHANNEL_XR_G      (CHANNEL_PUREG | CHANNEL_XR)
#define CHANNEL_XR_T      (CHANNEL_T     | CHANNEL_XR)

#define IS_CHAN_TURBO(x)  MAKE_BOOL(((x) & CHANNEL_TURBO))
#define IS_CHAN_CCK(x)    MAKE_BOOL(((x) & CHANNEL_CCK))
#define IS_CHAN_OFDM(x)   MAKE_BOOL(((x) & CHANNEL_OFDM))
#define IS_CHAN_XR(x)     MAKE_BOOL(((x) & CHANNEL_XR))
#define IS_CHAN_2GHZ(x)   MAKE_BOOL(((x) & CHANNEL_2GHZ))
#define IS_BASE_4GHZ(x)   MAKE_BOOL(((x) & CHANNEL_4GHZ)) 
#define IS_CHAN_5GHZ(x)   MAKE_BOOL(((x) & CHANNEL_5GHZ))
#define IS_CHAN_PASSIVE(x) MAKE_BOOL(((x) & CHANNEL_PASSIVE))

#define IS_CHAN_A(x)      (((x) & CHANNEL_ALL) == CHANNEL_A)
#define IS_CHAN_T(x)      (((x) & CHANNEL_ALL) == CHANNEL_T)
#define IS_CHAN_B(x)      (((x) & CHANNEL_ALL) == CHANNEL_B)
#define IS_CHAN_G(x)      (((x) & CHANNEL_ALL) == CHANNEL_G)
#define IS_CHAN_108G(x)   (((x) & CHANNEL_ALL) == CHANNEL_108G)
#define IS_CHAN_XR_A(x)   (((x) & CHANNEL_ALL) == CHANNEL_XR_A)
#define IS_CHAN_XR_G(x)   (((x) & CHANNEL_ALL) == CHANNEL_XR_G)
#define IS_CHAN_XR_T(x)   (((x) & CHANNEL_ALL) == CHANNEL_XR_T)
#define IS_CHAN_NON_108G_2GHZ(x) (IS_CHAN_2GHZ(x) && !IS_CHAN_108G(x))
#define IS_CHAN_G_OR_108G(x) (IS_CHAN_G(x) || IS_CHAN_108G(x))
#define IS_CHANNEL_14(x)  ((x) == 2484)

#define CHAN_TO_CHAR(x)   (IS_CHAN_A(x)    ? "A" :      \
                           IS_CHAN_B(x)    ? "B" :      \
                           IS_CHAN_G(x)    ? "G" :      \
                           IS_CHAN_T(x)    ? "T" :      \
                           IS_CHAN_108G(x) ? "t" : "?")
      
#define MODE_SELECT_11A   0x01
#define MODE_SELECT_TURBO 0x02
#define MODE_SELECT_11B   0x04
#define MODE_SELECT_11G   0x08
#define MODE_SELECT_108G  0x10
/* extra mode bits for dynamic turbo prime */
#define MODE_SELECT_TPA   0x20
#define MODE_SELECT_TPG   0x40

#define MODE_SELECT_2GHZ  (MODE_SELECT_11B | MODE_SELECT_11G | MODE_SELECT_108G)
#define MODE_SELECT_ALL   (MODE_SELECT_11A | MODE_SELECT_TURBO | MODE_SELECT_2GHZ)

/*
 * This bit is not exactly a mode, but it is set by NCP in the same field.
 * May move it elsewhere.
 */
#define MODE_SELECT_EXT_RANGE   0x100   // 11B using 1, 2, 5.5 and 11.

#define CTRY_DEFAULT            0       // Default country code
#define CTRY_DEBUG              0x1ff   // Debug country code
#define CTRY_INVALID            0x2ff   // Invalid country code
#define CTRY_WORLDWIDE_ROAM     0x3ff   // Worldwide roam country code
#define CHAN_11A_DEFAULT        5260
#define CHAN_11A_WWR_AD_HOC     5260    // Default worldwide roam ad-hoc 802.11a ???
#define CHAN_TURBO_WWR_AD_HOC   5210    // Default worldwide roam ad-hoc turbo ???
#define CHAN_11B_DEFAULT        2462    // Default 802.11b channel 11
#define CHAN_11G_DEFAULT        2462    // Default 802.11g channel 11
#define CHAN_108G_DEFAULT       2437    // Default 802.11g turbo channel 6
#define CHAN_TPA_DEFAULT        5200    // Default turbo prime A channel 40
#define CHAN_11B_WWR_AD_HOC     2457    // Default worldwide roam ad-hoc 802.11b channel 10

#define MAX_REG_DOM_NAME_SIZE   10      // Max string size of reg dom name inc NULL
/* Scan Types Allowed */
typedef enum {
    AP_SCAN_ALL_BSS,
    AP_SCAN_CLEAN_CHAN,
    AP_SCAN_SELECT_CHAN,
    AP_SCAN_WORLDWIDE,
    AP_SCAN_SITE_SURVEY    
} APSCAN_OP;

typedef enum {
    WLAN_INIT_CC_CLIST_NEW,
    WLAN_INIT_CC_CLIST_NO_CHG,
    WLAN_INIT_CC_CLIST_ERR
} WLAN_INIT_CC_CLIST;


/* NIC Country Capability */
typedef enum {
    ATH_COUNTRY_ALL,     // multi country
    ATH_COUNTRY_ONE,     // single country
    ATH_COUNTRY_WORLD    // World wide NIC
} ATH_COUNTRY;

/* Software Regulatory Capabilities definitions */
typedef enum {
    REGCAPBITS_VALUE_SET = 0x0001,     // Use software override to determine reg caps
    REG_FCC_USE_MIDBAND = 0x0002,       // FCC SKU uses midband channels
    REG_KK_USE_EVEN_U1 = 0x0004,     // Japan SKU uses even channels in UNII1 band
    REG_KK_USE_U2      = 0x0008,     // Japan SKU uses UNII2 band
    REG_KK_USE_MIDBAND = 0x0010,      // Japan SKU uses midband channels
    REG_KK_USE_ODD_U1 = 0x0020,      // Japan SKU uses odd UNI1 channels
    REG_KK_USE_NEW_11A = 0x0040,
} SW_REG_CAPABILITIES;

#define EN_KK_IRREVERSIBLE_UPDATE  0x00000004
#define EE_KK_ODD_U1_LEGACY        0x00000002
#define EN_KK_UNDO_UPDATE          0x00000001


typedef enum ConformanceTestLimits {
    FCC        = 0x10,
    MKK        = 0x40,
    ETSI       = 0x30,
    SD_NO_CTL  = 0xE0,
    NO_CTL     = 0xFF,
    CTL_MODE_M = 3,
    CTL_11A    = 0,
    CTL_11B    = 1,
    CTL_11G    = 2,
    CTL_TURBO  = 3
} ATH_CTLS;

/* Update Software EEPROM Functions */

void
wlanEepromCcRdInit(WLAN_DEV_INFO *pDevInfo);

void
wlanEepromRdCcWriteRd(WLAN_DEV_INFO *pDevInfo, A_UINT16 regDmn);

void
wlanEepromRdCcWriteCc(WLAN_DEV_INFO *pDevInfo, CTRY_CODE countryCode);

/* Channel List Initialization Functions */

void
wlanModifyMKKChannels(WLAN_DEV_INFO *);

extern A_BOOL mkkChannelsModified;

A_STATUS
wlanInitChannelList(WLAN_DEV_INFO *pDevInfo, WLAN_CHANNEL_LIST **pCList, A_UINT16 regDmnEnum,
                    A_UINT16 modeSelect, A_UINT16 enableOutdoor, A_BOOL ignoreGlobal);

A_STATUS
wlanInitSingleChannelList(WLAN_CHANNEL_LIST **pCList, A_UINT16 channel, A_UINT16 channelFlags);

A_STATUS
asciiToClist(WLAN_DEV_INFO *pDevInfo, A_UCHAR *pStr, A_UINT16 alen, WLAN_CHANNEL_LIST **pList);

/* Channel List Manipulation Functions */

A_STATUS
wlanFreeChannelList(WLAN_CHANNEL_LIST *pCList);

WLAN_INIT_CC_CLIST
wlanInitCcChannelList(WLAN_DEV_INFO *pDevInfo);

A_STATUS
wlanInitScanList(WLAN_DEV_INFO *pDevInfo, WLAN_CHANNEL_LIST *pCListFrom,
                 WLAN_SCAN_LIST **pSListTo, A_BOOL hiddenOnly);

A_STATUS
wlanInitSingleScanList(CHAN_VALUES *pChannelFrom, WLAN_SCAN_LIST **pSListTo);

A_STATUS
wlanFreeScanList(WLAN_SCAN_LIST *pSList);

CHAN_VALUES *
wlanGetChannelPtr(WLAN_CHANNEL_LIST *pCList, A_UINT16 channel, A_UINT16 modeSelect);

/* Channel List Accessor Functions */

A_BOOL
wlanIsValidChannel(WLAN_DEV_INFO *pDevInfo, CTRY_CODE countryCode, A_UINT16 modeSelect, A_UINT16 channel);

A_UINT8
wlanGetChannelPower(WLAN_DEV_INFO *pDevInfo, CHAN_VALUES *pChannel);

void
wlanPrintChannelList(WLAN_DEV_INFO *pDevInfo, CTRY_CODE countryCode,
                     A_UINT16 wirelessMode, A_UINT16 enableOutdoor);

/* Country Code Manipulation Functions */

A_BOOL
wlanIsCountryCodeValid(WLAN_DEV_INFO *pDevInfo, CTRY_CODE countryCode);

A_UINT16
wlanGetCountryCodeTableSize(WLAN_DEV_INFO *pDevInfo);

const A_UCHAR *
wlanGetIndexedCountryCodeName(WLAN_DEV_INFO *pDevInfo, int index);

const A_UCHAR *
wlanGetIndexedCountryCodeIsoName(WLAN_DEV_INFO *pDevInfo, int index);

A_UINT16
wlanGetCountryCodeIndex(WLAN_DEV_INFO *pDevInfo, CTRY_CODE countryCode);

const A_UCHAR *
wlanGetCountryCodeName(CTRY_CODE countryCode);

A_UINT16
wlanGetCountryCodeByName(A_UCHAR *countryCodeName, A_BOOL includeLocal);

A_UINT16
wlanGetDefaultCountry(WLAN_DEV_INFO *pDevInfo);

A_BOOL
wlanIsCountryAllow11aTurbo(WLAN_DEV_INFO *pDevInfo, CTRY_CODE);

A_BOOL
wlanIsCountryAllow11gTurbo(WLAN_DEV_INFO *pDevInfo, CTRY_CODE);

void
wlanPrintCountryCodeList(WLAN_DEV_INFO *pDevInfo);

A_INT8
wlanGetAntennaReduction(WLAN_DEV_INFO *pDevInfo, CHAN_VALUES *pChannel, A_INT8 twiceAntennaGain);

A_UINT8
wlanGetCtl(WLAN_DEV_INFO *pDevInfo, CHAN_VALUES *pChannel);

void
wlanGetCountryInfoList(WLAN_DEV_INFO *pDevInfo, COUNTRY_INFO_LIST *pCountryInfo);

/* Regulatory Domain Table Manipulation Functions */

A_UINT16
wlanGetCountryCodeByRegDmnName(A_UCHAR *regName);

A_UINT16
wlanGetRegDmnByName(A_UCHAR *regName);

void
wlanPrintRegDmnList(void);

/* Channel Conversion Functions */

A_INT16
wlanConvertGHztoCh(A_UINT16 freq, A_UINT16 channelFlags);

A_UINT16
wlanConvertChtoGHz(A_INT16 ch, A_UINT16 channelFlags);

WLAN_CFLAGS
wlanMode2Cflags(int mode);

A_UINT32
wlanCflags2Mode(WLAN_CFLAGS cFlags);
    
WIRELESS_MODE
wlanCFlagsToWirelessMode(WLAN_DEV_INFO *pdevInfo, WLAN_CFLAGS cflags);

/* Regulatory Domain checking Functions */

A_BOOL
wlanEnableDFS(WLAN_DEV_INFO *pDevInfo);

A_UINT32
wlanGetRegCapabilities(WLAN_DEV_INFO *pDevInfo);

A_BOOL
wlanIsDfsRD(WLAN_DEV_INFO *pDevInfo, REG_DOMAIN rd);

A_BOOL
wlanIsDfsChannel(WLAN_DEV_INFO *pDevInfo, CHAN_VALUES *pChannel);

A_BOOL
wlanIsActiveScanAllowed(WLAN_DEV_INFO *pDevInfo);

A_BOOL
wlanIsAdHocAllowed(WLAN_DEV_INFO *, WLAN_CFLAGS);

A_BOOL
wlanIsTelecSupportRequired(WLAN_DEV_INFO *pDevInfo);

A_BOOL
wlanIsNfCheckRequired(WLAN_DEV_INFO *pDevInfo);

A_BOOL
wlanIsImplicitRegDmn(CHAN_VALUES *pChannel, CTRY_CODE *countryCode);

A_UCHAR *
wlanGetCountryName(WLAN_DEV_INFO *pDevInfo, CTRY_CODE countryCode);

A_BOOL
wlanIsWwrSKU(WLAN_DEV_INFO *pDevInfo);

ATH_COUNTRY
wlanGetSKU(WLAN_DEV_INFO *pDevInfo);

CHAN_VALUES *
wlanBkSlistSort(WLAN_DEV_INFO *pDevInfo);

CHAN_VALUES *
wlanFirstChannel(WLAN_DEV_INFO *, WLAN_CHANNEL_LIST *);

CHAN_VALUES *
wlanGetBksChannel(WLAN_CHANNEL_LIST *, int index);

A_STATUS
wlanSelectAdHocChannel(WLAN_DEV_INFO *pDevInfo, BSSDESCR_SET *pBssSet, WLAN_CFLAGS cflags);

void
wlanClistClnup(WLAN_DEV_INFO *pDevInfo);

void
wlanClistNewScanCallback(WLAN_DEV_INFO *pDevInfo);

A_UINT16
wlanHalRD(WLAN_DEV_INFO *pDevInfo);

WLAN_CFLAGS
wlanUnsupportedAdHocBand(WLAN_DEV_INFO *pDevInfo);

A_UINT16
wlanUnsupportedAdHocMode(WLAN_DEV_INFO *pDevInfo);

A_BOOL
wlanIs11bCountry(WLAN_DEV_INFO *, CTRY_CODE);

A_BOOL
wlanIs11gCountry(WLAN_DEV_INFO *, CTRY_CODE);

A_BOOL
wlanIs11aAllowed(WLAN_DEV_INFO *pDevInfo, CTRY_CODE ccode);

A_BOOL
wlanIsTurboPrimeChannel(WLAN_DEV_INFO *pDevInfo, CHAN_VALUES *pChannel);

A_BOOL
wlanIs11bOnlyRD(REG_DOMAIN rd);

A_BOOL
wlanIs11gOnlyRD(REG_DOMAIN rd);

A_BOOL
wlanIs5211Channel14(SIB_ENTRY *pSib);

A_BOOL
wlanIs2GHzEcmChan(A_UINT16 channel);

A_BOOL
wlanVerifyEepromCcRd(WLAN_DEV_INFO *pDevInfo);

/* Radio Frequency Specification Manipulation Functions */

A_UINT32
wlanGetWirelessMode(WLAN_DEV_INFO *pDevInfo, CTRY_CODE countryCode);

A_UINT32
wlanGetAdhocWirelessMode(WLAN_DEV_INFO *pDevInfo, CTRY_CODE countryCode);


/* Channel Scan Functions */

A_STATUS
apScan(WLAN_DEV_INFO *pDevInfo, APSCAN_OP scanOp, SCANTYPE scan_type);

CHAN_VALUES *
wlanFindChannel(WLAN_DEV_INFO *pDevInfo, BSSDESCR_SET *pBssSet, WLAN_CHANNEL_LIST *pCList,
                BSSDESCR_SET *pIntBssSet, WLAN_CFLAGS cflags);

CHAN_VALUES *
wlanFindChannelMatch(WLAN_DEV_INFO *pDevInfo, WLAN_CHANNEL_LIST *pCList,
                     WLAN_CFLAGS cflags, A_UINT16 cfreq);

/* Utility functions */

A_BOOL
wlanDoesDsElementMatch(WLAN_DEV_INFO *pDev, DS_PARAM_SET *pDsParams);

void
clistShow(WLAN_DEV_INFO *pDevInfo);

void
wlanPrintFlags(WLAN_CFLAGS flags);

A_STATUS
wlanApplyMKKIrreversibleUpdate(WLAN_DEV_INFO *pDevInfo, A_UINT32 enOddU1, A_UINT32 undoUpdate);

A_BOOL
wlanIsMKKSku(A_UINT16 regDmn);

A_BOOL
wlanVerifyUserRd(REG_DOMAIN regDomain);


#ifdef __cplusplus
}
#endif

#endif /* _WLAN_CHANNEL_H_ */
