/*
 * Copyright (c) 2000-2002 Atheros Communications, Inc., All Rights Reserved
 *
 * Definitions for core driver
 * This is a common header file for all platforms and operating systems.
 */

#ifndef _WLANDRV_H_
#define _WLANDRV_H_

#include "wlanconfig.h"
#include "wlantype.h"
#include "wlanproto.h"
#include "wdcApi.h"
#include "wlanPhy.h"
#include "wlansme.h"
#include "queue.h"
#include "pci.h"
#include "wlanfragde.h"
#include "trace.h"
#include "shrcfg.h"

#ifdef __cplusplus
extern "C" {
#endif

/*
 * max num of devices supported by the driver; the AP can now
 * support 1 or 2 devices, but the station driver can only
 * support 1.
 */
#ifdef BUILD_AP
#define WLAN_MAX_DEV            2
#else
#define WLAN_MAX_DEV            1
#endif

#define MAX_ETH_CLIENTS         8
#define MIN_ETH_CLIENTS         1
#define ATHEROS_VENDOR_ID       0x168C  /* vendor ID for our device */

#define MAX_KEY_LEN_BITS        128
#define MAX_KEY_LEN_BYTES       (MAX_KEY_LEN_BITS / 8)
#define MAX_SHARED_KEYS         4
#define UNIQ_KEY_INDEX          4
#define HWINDEX_INVALID         0x7fff          /* hwIndex not allocated */
#define KEYINDEX_DYNAMIC        0x8000          /* dynamically allocate keyIndex */
#define KEYINDEX_INVALID        0x8001          /* invalid keyTable entry */
#define KEYINDEX_NONE           0x8002          /* implies no crypto in vlan support */

#define KEY_XOR                 0xaa
#define KRK_LENGTH_BYTES        16

/*
 * WLAN_PRIV_RECORD key types
 */
#define PRIV_KEY_TYPE_WEP       0x0
#define PRIV_KEY_TYPE_WEP_COMP  0x1
#define PRIV_KEY_TYPE_AES_OCB   0x2            // ar5211-compatible aes-ocb
#define PRIV_KEY_TYPE_AES_CCM   0x3
#define PRIV_KEY_TYPE_NULL      0x7
#define PRIV_KEY_TYPE_TKIP      0x10
#define PRIV_KEY_TYPE_TKIP_SW   0x11
#define PRIV_KEY_TYPE_CKIP      0x20           // ckip

extern A_BOOL cipherSupported(struct wlanDevInfo *pdev, A_UINT32 keyType);
extern A_BOOL micSupported(struct wlanDevInfo *pdev, A_UINT32 keyType);

typedef struct WlanPrivStruct {
    A_UINT32    keyIV;             /* IV generation control */
    A_UINT32    extKeyIV;          /* extended IV field for TKIP & CCM */
    A_UINT32    keyFlags;
    A_INT32     keyLength;         /* Length of key in bits */
    A_UINT32    keyType;           /* type of key, WEP, TKIP or AES */
    void       *initValue;        /* private data for encryption algorithms */
    A_UINT8     keyVal[32];                     /* Array for key */
    A_UINT8     micTxKeyVal[8];                 /* TX MIC key */
    A_UINT8     micRxKeyVal[8];                 /* RX MIC key */
    A_INT64     keyRSC;                         /* key Rx Seq Counter */
} WLAN_PRIV_RECORD __ATTRIB_ALIGN(8);

typedef struct WlanConnKeyStruct {
    WLAN_MACADDR        bssId;
    WLAN_PRIV_RECORD    connKey;
} CONNECTION_KEY_RECORD __ATTRIB_ALIGN(8);

typedef struct WlanBssKeyStruct {
    A_UINT32            bssKeyIdx;
    A_UINT32            isDefaultBssKey;
    WLAN_PRIV_RECORD    bssKey;
} BSS_KEY_RECORD __ATTRIB_ALIGN(8);

typedef struct aesNonce {
    A_UINT32            keyIV;
    WLAN_MACADDR        srcAddr;
    WLAN_MACADDR        dstAddr;
} AES_NONCE;

typedef enum {
    ENCRYPTION_DISABLED,
    ENCRYPTION_WEP,
    ENCRYPTION_AES_OCB,
    ENCRYPTION_AES_CCM,
    ENCRYPTION_CKIP, // No MMH used
    ENCRYPTION_TKIP, // implies Michael
    ENCRYPTION_AUTO,
} WLAN_ENCRYPTION_ALG;

typedef enum {
    MIC_DISABLED = 0, // includes WEP
    MIC_AES,
    MIC_CKIP_MMH,
    MIC_TKIP_MICHAEL,
} WLAN_MIC_ALG;

typedef enum {
    AUTH_TYPE_DEFAULT,
    AUTH_TYPE_OPEN,
    AUTH_TYPE_SHARED,
    AUTH_TYPE_AUTO,
    AUTH_TYPE_WPA,
    AUTH_TYPE_WPAPSK
} WLAN_AUTH_TYPE;

typedef enum {
    OP_MODE_AP,
    OP_MODE_STA,
    OP_MODE_PPT,
    OP_MODE_MPT,
    OP_MODE_REPEATER
} WLAN_OP_MODE;

/* WLAN_PRIV_RECORD keyFlags bit mappings */
#define PRIV_FRAMEKEY           0x03            /* Two bits of key in on air frame */
#define PRIV_UNIQUEKEY          0x04            /* Unique key */
#define PRIV_DYNAMIC            0x08            /* Key is generated by exchange protocol */
#define PRIV_CKIP_KP            0x10            /* CKIP KP flag */
#define PRIV_CKIP_MIC           0x20            /* CKIP MIC flag */
#define PRIV_TKIP_SWMIC         0x40            /* TKIP uses SW Michael */
#define PRIV_KEY_DESTROY        0x0400          /* Key to be destroyed */
#define PRIV_KEY_LOCKED         0x8000          /* Key is being used */

/* ATHEROS_DESC_RX_STATUS decryptError bit mappings */
#define DECRYPTERROR_CRC           0x01  /* Indicates there was a CRC Error */
#define DECRYPTERROR_MIC           0x02  /* Indicates there was a MICError */
#define MIC_COUNTERMEASURE_TIMEOUT 60000 /* 60 seconds for countermeasure */

/* Encryption struct for key backing store */
typedef struct {
    unsigned long P[16 + 2];
    unsigned long S[4][256];
} BLOWFISH_CTX;

struct queueDetails;

/*
 * Software copies of HW status.  Currently must be 2 DWORDS in length.
 */
typedef struct atherosDescTxStatus {
    enum {
        NOT_DONE,
        TRANSMIT_OK,
        EXCESSIVE_RETRIES,
        FILTERED
    }           status;
    A_UINT16    timestamp;          /* HW assigned timestamp */
    A_UINT16    seqNum;             /* HW assigned sequence number */
    A_UINT16    retryCount;         /* short + long retries */
} ATHEROS_DESC_TX_STATUS;

typedef struct atherosDescRxStatus {
    A_UINT16        dataLength;     /* RX frame length */
    WLAN_TIMESTAMP  timestamp;      /* HW assigned receive time stamp */
    A_UINT8         decryptError;   /* payload was not decrypted ok */
    A_UINT8         phyError;       /* phy error code from HW */
    A_UINT8         rate;           /* HW receive rate */
    A_UINT8         hwIndex;        /* key cache index */
    A_INT8          rssi;           /* signed RSSI */
    A_UINT8         antenna;        /* antenna */
    A_UINT8         keyCacheMiss;   /* key cache miss */
} ATHEROS_DESC_RX_STATUS;


/* Software descriptor structure */
typedef struct atherosDesc {
    struct queueDetails *freeQueue;         /* flag for which queue the descriptor, when free, belongs to */
    struct atherosDesc  *pNextVirtPtr;      /* virtual pointer to next descriptor */

#if 1   /* xxxxPGSxxxx - remove this. Also ATHEROS_DESC do not need to be allocated from shared memory anymore */
    A_UINT32            thisPhysPtr;        /* phys address of this descriptor */
#endif

    WLAN_FRAME_UNION    pBufferVirtPtr;     /* virtual pointer to buffer */
    void                *pOrigBufferVirtPtr;/* Original buffer virtual ptr, in case bufferVirtPtr changes */
    A_UINT32            bufferPhysPtr;      /* phys address of this buffer */
    A_UINT32            pOrigbufferPhysPtr; /* Original buffer physical ptr, in case bufferPhysPtr changes */
    A_UINT32            bufferLength;       /* buffer length */
    A_UINT32            frameLength;        /* frame length - valid for first descriptor only */

#ifdef VXWORKS
    CL_BLK_ID           pOSDescPtr;         /* OS specific */
    A_UINT32            _pad[3];
#else
    PNDIS_PACKET        pOSHandle;          /* ptr to NDIS pkt descriptor */
    PNDIS_BUFFER        OSbuffer;           /* ptr to NDIS buffer descriptor */
    A_UINT32            OSspecificInfo;     /* map register info for NDIS */
    A_UINT32            _pad[1];
#define ffRefHack       _pad[0]
#endif

    struct sibEntry     *pSrcSibEntry;      /* source STA SIB entry */
    struct sibEntry     *pDestSibEntry;     /* destination STA SIB entry */
    struct atherosDesc  *pTxLastDesc;       /* pointer to the last descriptor of the frame in a transmit chain */
    struct atherosDesc  *pTxFirstDesc;      /* pointer to the first descriptor of the frame in a transmit chain */
    struct OpBss        *pOpBss;            /* Operation Bss the frame belongs to! XR AP runs two BSSs, bridges etc. */
    WLAN_PRIV_RECORD    *pKey;              /* pointer to the key table entry for the frame */

    A_UINT32            hwIndex;
    A_UINT32            staleFlag:1,        /* already processed holding descriptor */
                        qosMode:4,          /* modes for qos frames */
                        swRetryFlag:1,      /* set if this is a retransmit frame */
                        ffFlag:1,           /* frame being tx is fast frame */
                        ffSeqNum:11,        /* ffSeqNum of recd and buffered fast frame */
                        is4AddrFrm:1,       /* Is the Frame a 4 Address Frame */
                        isEthFrm:1,         /* Is the Ethernet Frame */
                        bIndicateXmitStatus:1,/* indicate the transmit status or not */
                        more:1,             /* Set if packet continues in next descriptor */
                        reserved:11;        /* reserved field */
    A_UINT32            timeStamp;          /* time stamp for PS aging in MS */

    /* tx/rx status (after hardware processing) */
    union {
        ATHEROS_DESC_RX_STATUS rx;
        ATHEROS_DESC_TX_STATUS tx;
    } status;

    /* Should be cache aligned */
    WDC_BUFFER_DESC         bufDesc;        /* WDC buffer descriptor */
    WDC_RX_INFO         *wdcRxInfo;         /* Describes associated RX Buffer */

    A_UINT32            targetTxQId;        /* Target TX queue ID */
} ATHEROS_DESC;

#define A_DESC_INIT(_pDesc)     do {                                        \
    (_pDesc)->hwIndex         = HWINDEX_INVALID;                            \
    (_pDesc)->staleFlag       = 0;                                          \
    (_pDesc)->ffFlag          = 0;                                          \
    (_pDesc)->ffSeqNum        = 0;                                          \
    (_pDesc)->is4AddrFrm      = 0;                                          \
    (_pDesc)->isEthFrm        = 0;                                          \
    (_pDesc)->bIndicateXmitStatus = 0;                                      \
    (_pDesc)->more            = 0;                                          \
    (_pDesc)->reserved        = 0;                                          \
    (_pDesc)->pTxFirstDesc    = NULL;                                       \
    (_pDesc)->pTxLastDesc     = NULL;                                       \
    (_pDesc)->pDestSibEntry   = NULL;                                       \
} while (0)

#define A_DESC_COPY(_in, _out)          do {                                \
    (_out)->pSrcSibEntry    = (_in)->pSrcSibEntry;                          \
    (_out)->pDestSibEntry   = (_in)->pDestSibEntry;                         \
    (_out)->pOpBss          = (_in)->pOpBss;                                \
    (_out)->pKey            = (_in)->pKey;                                  \
    (_out)->hwIndex         = (_in)->hwIndex;                               \
    (_out)->staleFlag       = (_in)->staleFlag;                             \
    (_out)->ffFlag          = (_in)->ffFlag;                                \
    (_out)->is4AddrFrm      = (_in)->is4AddrFrm;                            \
    (_out)->isEthFrm        = (_in)->isEthFrm;                              \
    (_out)->qosMode         = (_in)->qosMode;                               \
    (_out)->timeStamp       = (_in)->timeStamp;                             \
    (_out)->targetTxQId     = (_in)->targetTxQId;                       \
} while (0)

#define A_TX_DESC_CHAIN_FORMAT(_pDesc)  do {                                \
    ATHEROS_DESC *pTail = _pDesc;                                           \
    while (pTail->more) {                                      \
        pTail->pTxFirstDesc = _pDesc;                                       \
        pTail = pTail->pNextVirtPtr;                                        \
        ASSERT(pTail);                                                      \
    }                                                                       \
    pTail->pTxFirstDesc = _pDesc;                                           \
    (_pDesc)->pTxLastDesc = pTail;                                          \
} while (0)

/* xxxxPGSxxx - A_RX_DESC_CHAIN_TO_TX could possibly be removed (see wlanDataFrameReceive) */
#define A_RX_DESC_CHAIN_TO_TX(_pDesc)   do {                                \
    ATHEROS_DESC *pTail = _pDesc;                                           \
    (_pDesc)->frameLength = 0;                                              \
    while (pTail) {                                                         \
        (_pDesc)->frameLength    += pTail->status.rx.dataLength;            \
        pTail->bufferLength      = pTail->status.rx.dataLength;             \
        pTail->more              = pTail->pNextVirtPtr ? 1 : 0;             \
        pTail = pTail->pNextVirtPtr;                                        \
    }                                                                       \
} while (0)

/* Convert ATHEROS_DESC to WDC_BUFFER_DESC (WDC) */
#define A_TX_DESC_CHAIN_TO_WDC(_pDesc)   do {                               \
    ATHEROS_DESC *pTail = (_pDesc);                                         \
    while (pTail) {                                                         \
        pTail->bufDesc.pBuffer          = pTail->pBufferVirtPtr.byte;       \
        pTail->bufDesc.bufferPhysAddr   = pTail->bufferPhysPtr;             \
        pTail->bufDesc.bufferLength     = pTail->bufferLength;              \
        pTail->bufDesc.frameLength      = pTail->frameLength;               \
        pTail->bufDesc.more             = pTail->more;                      \
        if (pTail->pNextVirtPtr && pTail->more) {                           \
            pTail->bufDesc.pNext = &pTail->pNextVirtPtr->bufDesc;           \
        } else {                                                            \
            pTail->bufDesc.pNext = NULL;                                    \
        }                                                                   \
        if (pTail->more)                                                    \
            pTail = pTail->pNextVirtPtr;                                    \
        else                                                                \
            pTail = NULL;                                                   \
    }                                                                       \
} while (0)

typedef struct _PACKET_QUEUE_ENTRY {
    struct _PACKET_QUEUE_ENTRY *next;
} PACKET_QUEUE_ENTRY, *PPACKET_QUEUE_ENTRY;

typedef struct _PACKET_QUEUE_HEADER {
    PPACKET_QUEUE_ENTRY head;
    PPACKET_QUEUE_ENTRY tail;
} PACKET_QUEUE_HEADER, *PPACKET_QUEUE_HEADER;

#define InitializeQueueHeader(QueueHeader)    do {     \
    (QueueHeader)->head = (QueueHeader)->tail = NULL;  \
} while (0)

#define IsQueueEmpty(QueueHeader) ((QueueHeader)->head == NULL)

#define RemoveHeadQueue(QueueHeader)                  \
    (QueueHeader)->head;                              \
    do {                                              \
        PPACKET_QUEUE_ENTRY pNext;                    \
        ASSERT((QueueHeader)->head);                  \
        pNext = (QueueHeader)->head->next;            \
        (QueueHeader)->head = pNext;                  \
        if (pNext == NULL) {                          \
            (QueueHeader)->tail = NULL;               \
        }                                             \
    } while (0)

#define InsertHeadQueue(QueueHeader, QueueEntry)  do {             \
    ((PPACKET_QUEUE_ENTRY)QueueEntry)->next = (QueueHeader)->head; \
    (QueueHeader)->head = (PPACKET_QUEUE_ENTRY)(QueueEntry);       \
    if ((QueueHeader)->tail == NULL) {                             \
        (QueueHeader)->tail = (PPACKET_QUEUE_ENTRY)(QueueEntry);   \
    }                                                              \
} while (0)

#define InsertTailQueue(QueueHeader, QueueEntry)   do {                 \
    ((PPACKET_QUEUE_ENTRY)QueueEntry)->next = NULL;                     \
    if ((QueueHeader)->tail) {                                          \
        (QueueHeader)->tail->next = (PPACKET_QUEUE_ENTRY)(QueueEntry);  \
    } else {                                                            \
        (QueueHeader)->head = (PPACKET_QUEUE_ENTRY)(QueueEntry);        \
    }                                                                   \
    (QueueHeader)->tail = (PPACKET_QUEUE_ENTRY)(QueueEntry);            \
} while (0)

#ifdef DEBUG
extern int lockDebugLevel;
#define lockPrintf            if (lockDebugLevel) uiPrintf
#else /* DEBUG */
#define lockDebugLevel          0
#define lockPrintf(...)
#endif

typedef struct {
    A_ATOMIC        num;
    A_UINT32        label;
    ULONG           flags;
#ifndef BUILD_AP
    KSPIN_LOCK  spinLock;
#endif
} A_SPIN_LOCK, *PA_SPIN_LOCK;

#ifdef BUILD_AP
#define ATH_ALLOCATE_SPINLOCK(a)
#define ATH_ACQUIRE_SPINLOCK(a)
#define ATH_RELEASE_SPINLOCK(a)
#define ATH_FREE_SPINLOCK(a)
#else
#define ATH_ALLOCATE_SPINLOCK(a,seq)     do {                  \
    a.label = seq;                                             \
    A_ATOMIC_SET(&(a.num), 0);                                    \
    NdisAllocateSpinLock(&(a.spinLock));                       \
} while (0)

#define ATH_ACQUIRE_SPINLOCK(a)     do {                            \
    NdisInterlockedIncrement(&(a.num));                             \
    lockPrintf("==>%s() line%d: Acquire lock%d, num = %d\n",        \
                __FUNCTION__,__LINE__,a.label,A_ATOMIC_READ(&(a.num)));\
    NdisAcquireSpinLock(&(a.spinLock));                             \
} while (0)

#define ATH_RELEASE_SPINLOCK(a)     do {                            \
    NdisInterlockedDecrement(&(a.num));                             \
    lockPrintf("<==%s() line%d: Release lock%d, num = %d\n",        \
                __FUNCTION__,__LINE__,a.label,A_ATOMIC_READ(&(a.num)));\
    ASSERT(A_ATOMIC_READ(&(a.num)) >= 0);                              \
    NdisReleaseSpinLock(&(a.spinLock));                             \
} while (0)

#define ATH_ACQUIRE_SPINLOCK_IRQ(a)     do {                            \
    NdisInterlockedIncrement(&(a.num));                             \
    lockPrintf("==>%s() line%d: Acquire irqlock%d, num = %d\n",        \
                __FUNCTION__,__LINE__,a.label,A_ATOMIC_READ(&(a.num)));\
    spin_lock_irqsave(&(a.spinLock), a.flags);                             \
} while (0)

#define ATH_RELEASE_SPINLOCK_IRQ(a)     do {                            \
    NdisInterlockedDecrement(&(a.num));                             \
    lockPrintf("<==%s() line%d: Release irqlock%d, num = %d\n",        \
                __FUNCTION__,__LINE__,a.label,A_ATOMIC_READ(&(a.num)));\
    ASSERT(A_ATOMIC_READ(&(a.num)) >= 0);                              \
    spin_unlock_irqrestore(&(a.spinLock), a.flags);                             \
} while (0)


#define ATH_FREE_SPINLOCK(a)     do {                  \
    NdisFreeSpinLock(&(a.spinLock));                   \
} while (0)
#endif

/*
 * definitions for TxQandMode
 */
#define txQMask         0xf                /* limited to 16 hw queues */
#define TX_ISQOSDATA    0x8000             /* encapsulate as QOS_DATA */
#define TX_IS_UPSD      0x4000             /* uses UPSD queues */
#define TX_IS_OFDM      0x2000             /* sent as OFDM */

/* Queue data structure */
typedef struct queueDetails {
    A_SEM_TYPE           qSem;               /* Semaphore to manage this queue */
    ATHEROS_DESC        *pDescQueueHead;     /* virt pointer to queue head */
    ATHEROS_DESC        *pDescQueueTail;     /* virtual pointer to queue tail */
    TXQ_ATTRIBUTES       descQueueAttribs;   /* attributes of the queue */
    A_UINT32             targetQueueId;      /* target queue ID */
    A_ATOMIC             qFrameCount;        /* frame count in this queue */
    int                  halTxQueueNum;
    A_BOOL               isTailSelfLinked;   /* should the tail descriptor be linked to itself? */
    A_BOOL               queuingBurst;       /* flag to indicate operations on a burst Q */
    A_INT32              qBurstCount;        /* counter for the pending burst seq */
    A_INT32              pendBFrameCount;    /* count of frames in pending Burst sequence */
    A_UINT32             burstCTSDur;        /* CTS duration for the being formed burst seq */
    ATHEROS_DESC        *pBurstHeadDesc;     /* frame desc  which is last one in a burst seq */
    ATHEROS_DESC        *pBurstTailDesc;     /* frame desc  which is first one in a burst seq */
    struct queueDetails *pStagingQueue;
} QUEUE_DETAILS;

#define A_QUEUE_INFO_INIT(_pQ, _pAc)    do {    \
    A_MEM_ZERO(_pQ, sizeof(TXQ_ATTRIBUTES));     \
    (_pQ)->aifs     = (_pAc)->aifs;             \
    (_pQ)->logCwMin = (_pAc)->logCwMin;         \
    (_pQ)->logCwMax = (_pAc)->logCwMax;         \
} while (0)

/*
 * Possible state bits for destState in the SIB.
 * The bits reflect current successful state of the station. e.g. STATE_AUTH
 * Authenticated bit won't be set until the frame exchange is completed with
 * successful status.
 */
#define STATE_QUIET         0x0     /* Initial empty state */
#define STATE_AUTH          0x1     /* Station is authenticated */
#define STATE_ASSOC         0x2     /* Station is associated */
#define STATE_JOINED        0x10    /* Station has joined an AP */
#define STATE_AP_DOWN       0x0100  /* This is an AP, down, or not started */
#define STATE_AP_UP         0x0400  /* This is an AP, up and running */
#define STATE_CONNECTED     (STATE_JOINED | STATE_AUTH | STATE_ASSOC)

#define IS_STATE_CONNECTED(x) (((x)->localSta) && (((x)->localSta->staState & STATE_CONNECTED) == STATE_CONNECTED))
/* bssDescr is a dynamically generated object, we need to make sure that it
 * is available before checking for connection status */
#define IS_CONN_OR_JOINED(x)  (((x)->bssDescr) &&                           \
        (((x)->bssDescr->bsstype == INFRASTRUCTURE_BSS &&                   \
            IS_STATE_CONNECTED(x)) ||                                       \
        ((x)->bssDescr->bsstype == INDEPENDENT_BSS &&                       \
            ((x)->localSta->staState & STATE_JOINED))))

/*
 * Possible state bits for transState in SIB_ENTRY
 * It is the current state of transactions going on for the station.
 * It will change when a request frame is sent or a response is received.
 * It will change when almost any communication is made with the station.
 * After going through a few states in transState, an operation like
 * Authentication is completed, then we change the corresponding state bit
 * in destState and clear the transState.
 */

#define TSTATE_QUIET                0        /* No transactions in progress */
#define TSTATE_AUTH_FRAME1_SENT     0x1      /* Frame 1 has been sent (on STA) */
#define TSTATE_AUTH_FRAME1_RCVD     0x2      /* Frame 1 is received (on AP) */
#define TSTATE_AUTH_FRAME2_SENT     0x4      /* Frame 2 has been sent (on AP) */
#define TSTATE_AUTH_FRAME2_RCVD     0x8      /* Frame 2 is received (on STA) */
#define TSTATE_AUTH_FRAME3_SENT     0x10     /* Frame 3 has been sent (on STA) */
#define TSTATE_AUTH_FRAME3_RCVD     0x20     /* Frame 3 is received (on AP) */
#define TSTATE_AUTH_FRAME4_SENT     0x40     /* Frame 4 has been sent (on AP) */
#define TSTATE_AUTH_FRAME4_RCVD     0x80     /* Frame 4 is received (on STA) */
#define TSTATE_AUTH_IN_PROGRESS     0xff     /* Any of above bits is set */
#define TSTATE_ASSOC_REQ_SENT       0x100    /* Assoc request is sent (on STA) */
#define TSTATE_ASSOC_REQ_RCVD       0x200    /* Assoc request is received (on AP) */
#define TSTATE_ASSOC_RESP_SENT      0x400    /* Assoc response is sent (on AP) */
#define TSTATE_ASSOC_RESP_RCVD      0x800    /* Assoc response received (on STA) */

#define TSTATE_SCANNING             0x1000   /* Scan in progress */
#define TSTATE_JOINING              0x2000   /* Join in progress */
#define TSTATE_DEAUTH_FRAME_SENT    0x4000   /* deauth in progress */
#define TSTATE_DISASSOC_FRAME_SENT  0x8000   /* Disassoc in progress */

#define TSTATE_REASSOC_REQ_SENT     0x10000  /* Reassoc request is sent (on STA) */
#define TSTATE_REASSOC_REQ_RCVD     0x20000  /* Reassoc request is received (on AP) */
#define TSTATE_REASSOC_RESP_SENT    0x40000  /* Reassoc response is sent (on AP) */
#define TSTATE_REASSOC_RESP_RCVD    0x80000  /* Reassoc response received (on STA) */
#define TSTATE_PWRMGT_PENDING       0x100000 /* Transmission of power mode packet pending */

/*
 * Possible value for wlanMode in SIB_ENTRY
 */
#define STA_MODE_A                  0        /* STA operates in 11a mode */
#define STA_MODE_G                  1        /* STA operates in 11g mode */
#define STA_MODE_B                  2        /* STA operates in 11b mode */
#define STA_MODE_108G               3        /* STA operates in 11g turbo mode */

/*
 * Possible values for protectionType for 11g nonErp protection
 */
#define PROT_TYPE_CTSONLY           0        /* 11g nonErp protection with CTS only */
#define PROT_TYPE_RTSCTS            1        /* 11g nonErp protection with RTS-CTS */

/*
 * Possible values for protectionMode for 11g nonErp protection
 */
#define PROT_MODE_NONE              0        /* 11g nonErp protection never  */
#define PROT_MODE_ALWAYS            1        /* 11g nonErp protection always */
#define PROT_MODE_AUTO              2        /* 11g nonErp protection automatically based on nonERP assoc */

/* Number of (OEM-defined) functions using GPIO pins currently defined */
#define NUM_GPIO_FUNCS 2

#define MAX_MULTICAST_ADDRESSES     32

#define NO_DOMAIN_NAME              "NONE"   // String Name of debug regulatory domain

typedef A_UINT16                    WLAN_CHANNEL;          /* Channel number */
typedef A_UINT16                    WLAN_CFLAGS;           /* Channel flags */

typedef struct channelValues {
    WLAN_CHANNEL channel;
    WLAN_CFLAGS  channelFlags;
    A_UINT8      noiseImmunityLevel;
    A_UINT8      spurImmunityLevel;
    A_UINT8      firstepLevel;
    A_BOOL       ofdmWeakSigDetectionOff;
    A_BOOL       cckWeakSigThresholdHigh;
    A_BOOL       spurImmunityBias;
    A_BOOL       phyErrStatsDisabled;
    A_BOOL       bssSeenHere;
    A_UINT8      gainI;
    A_BOOL       iqCalValid;
    A_INT8       iCoff;
    A_INT8       qCoff;
} CHAN_VALUES;

/* List of channels to be scanned. channel is to be used as a variable size array. */
typedef struct wlanChannelList {
    A_UINT16    listSize;
    CHAN_VALUES *chanArray;
} WLAN_CHANNEL_LIST;

typedef struct wlanScanList {
    A_UINT16    listSize;
    CHAN_VALUES **pChanArray;
} WLAN_SCAN_LIST;

typedef enum  {
    START_ADHOC_NO_11A,         /* Don't use 802.11a channel */
    START_ADHOC_PER_11D,        /* 802.11a + 802.11d ad-hoc (dynamic start channel) */
    START_ADHOC_IN_11A,         /* 802.11a ad-hoc (default start channel) */
    START_ADHOC_IN_11B          /* 802.11b ad-hoc */
} START_ADHOC_OPTION;


#ifdef WME
#define NWME          8            // should be NUM_TX_QUEUES, but isn't available here
                                   // also, current code treats the CAB queue specially; thus,
                                   // it should not be included in loops over the WME set of queues.
typedef struct {
    WMEv1_AC_RECORD wmeParamsPerQ[NWME];
} WMEv1_PARAM_PER_MODE;
#endif
/*
 * Default config information for STA. Config utilities
 * will fill in this structure. It will persist throuth
 * a reset, SIB entry will not.
 */
typedef struct StaConfig {
    struct wlanChannelList *pClist;         // The channel list created to use for selectable channels

    WLAN_MACADDR    macPermAddr;            // EEPROM's MAC Address - only used by driver attach and OID request
    WLAN_MACADDR    macAddr;                // MAC Address of device
    BSSTYPE         bssType;                // Type of BSS - Infrastructure, Independent, or ANY
    CAP_INFO        capInfo;                // Capibilities info
    SSID_ARRAY      cfgSsids;               // Array of configured Ssids.
    A_UINT8         defaultRateIndex[WIRELESS_MODE_MAX];
                                            // The following are used in .11g mode
    A_UINT8         modeCTS;                // G mode CTS protection for the station,
                                            // 0 = always off, 1 = always on, 2 = automatic
    A_UINT8         rateCTS;                // optional fixed rate for CTS protection and MC
    A_BOOL          shortSlotTime;          // Allow short slot time in g mode
    A_UINT8         protectionMode;         // G mode CTS protection for the station,
                                            // 0 = always off, 1 = always on, 2 = automatic
    A_UINT8         protectionRate;         // optional fixed rate for CTS protection and MC
    A_BOOL          protectionType;         // Use CTS-ONLY (0) or RTS-CTS (1)
    A_UINT8         gBeaconRate;            // optional fixed rate for 11g beacons
    A_BOOL          gdraft5;                // 11g Draft 5.0 compatability

    A_BOOL          privacyOptionImplemented;   // privacy option implemented
    A_BOOL          extendedChanMode;       // Extended Channel Mode flag
    A_BOOL          ignore11dBeacon;        // Ignore 11d beacon flag
    CHAN_VALUES     *pChannel;              // home channel, where it's AP is located
    CHAN_VALUES     *phwChannel;            // channel currently set in hardware, may be different
                                            // than Channel during scan
    A_UINT16        adhocChannelFreq;       // adhoc channel for starting IBSS
    WLAN_CFLAGS     adhocChannelFlags;      // adhoc channel type for starting IBSS
    A_UINT16        defaultFragThreshold;   // Default fragmentation threshold
    A_UINT16        userFragThreshold;      // User-selected fragmentation threshold
    A_UINT16        cfpDuration;            // CFP duration for NDIS based AP, if supported.
    A_UINT16        defaultDot11RTSThreshold;
    A_UINT16        userRTSThreshold;
    A_UINT32        maxTransmitMsduLifetime;//the elapsed time in TU, after the initial transmission of an MSDU, after which further
                                            // attempts to transmit the MSDU shall be terminated
    A_UINT32        maxReceiveLifeTime;     // the elapsed time in TU, after the initial reception of a fragmented MMPDU or MSDU,
                                            // after which further attempts to reassemble the MMPDU or
                                            // MSDU shall be terminated

    A_UINT32        NumberOfMcAddresses;    // Used for multicast address changes during resets
    WLAN_MACADDR    McTable[MAX_MULTICAST_ADDRESSES];

    A_UINT32        McastFilter0;           // Current selectively-set multicast filter bits
    A_UINT32        McastFilter1;

    A_UINT32        PacketFilter;           // The current Global Packet Filter
    A_UINT32        encryptionAlg;          // Encryption algorithm (AES/WEP)
    WLAN_AUTH_TYPE  authType;               // Authentication Type (Open/Shared/Auto/WPA)
    WLAN_AUTH_TYPE  authTypeUseOnly;        // Authentication Type only allowed (Open/Shared/Default)
    A_BOOL          privacyInvoked;         // Privacy enable status
    A_BOOL          enableACL;              // Enable MAC ACL checks
    A_BOOL          excludeUnlisted;        // Disallow unlisted MACs
    A_BOOL          authAllowOpen;          // Open authentication is ok
    A_BOOL          authAllowShared;        // Shared authentication is ok
    A_BOOL          authOpenFirst;          // STA first authentication request
    A_BOOL          rateCtrlEnable;         // Enable transmit rate control features
    A_BOOL          radioEnable;            // Radio is enabled. Can be turned off by rfSilent feature
    A_BOOL          filter8021x;            // 802.1X filtering. Don't encrypt .1x frames. Default 0.
    A_BOOL          multiDomainCapEnabled;  // MultiDomainCapability flag
    A_UINT16        countryCode;            // Country Code (MS numbering)
    A_UINT16        defaultKey;             // Default encryption key [0-4]
    WLAN_PRIV_RECORD keys[MAX_SHARED_KEYS+1];   // array to hold configured keys
    A_UINT16        sleepMode;              // Awake, aggressive sleep or agressive performance
    A_UINT16        mbiTimeout;             // Missed beacon timeout, AP assumed to be lost after this
    A_UINT16        sleepTimePwrSave;       // station wakes after this many mS in max power save mode
    A_UINT16        sleepTimePerf;          // station wakes after this many mS in max performance mode
    A_BOOL          sleepLogEnable;         // Log percentage of time chip is sleeping
    A_UINT16        rollingAvgPeriod;       // time over which to take a sleep rolling average, s
    A_UINT16        sleepSampleInterval;    // interval for checking sleep state, ms
    A_BOOL          overrideACstatus;       // ignore PnP AC event notification if TRUE
    A_UINT16        listenInterval;         // station wakes after this TU period to listen for beacons
    A_UINT16        beaconInterval;         // beacon interval, applies to AP or ad-hoc only.
    A_UINT16        atimWindow;             // atim window, applies to ad-hoc only.
    A_UINT32        inactivityTimePwrSave;  // in max PS mode, how long (in mS) w/o Tx/Rx before going back to sleep
    A_UINT32        inactivityTimePerf;     // in max perf mode, how long (in mS) w/o Tx/Rx before going back to sleep
    A_UINT16        calibrationTime;        // how often to do calibrations, in seconds (0=off)
    A_UINT32        noBeaconTimeout;        // time in ms to wait while not receiving beacons before bailing
    A_UINT32        swRetryMaxRetries;      // max number of retries to be attempted by sw
                                            // for any given frame
    A_UINT32        hwTxRetries;            // num of times hw retries the frame
    A_INT32         psQueueMaxDepth;        // Max depth of queue used while processing power saved frames
    A_UINT32        txTrigThresh;           // How many good tx's before decreasing tx trigger
    A_BOOL          swretryEnabled;         // feature enable flag
    A_UINT32        burstTime;              // max burst time of transmit queue in ms
    A_UINT16        tpScale;                // Scaling factor to be applied to max transmit power
    A_INT16         tpcHalfDbm5;            // Transmit Power Control for 5 GHz in half dBm
    A_INT16         tpcHalfDbm2;            // Transmit Power Control for 2 GHz in half dBm
    A_UINT16        RegDomain;              // Regulatory Domain
    A_UINT16        busConfig;              // PCI target retry and TRDY timeout
    A_UINT16        cacheLineSize;          // Saved from PCI config (set by BIOS)
                                            // Combination of phyType and phyFreq determines a, b or g.
    A_UINT16        NetBand;                // User's choice of a, b, turbo, g, etc. from registry
    A_UINT16        AdhocBand;              // User's choice of a, b, turbo, g, etc. to start adhoc
    A_BOOL          bInvalidNetBand;        // NetBand or AdhocBand setting is invalid.
    A_BOOL          radarDetectEnable;      // radar detection feature
    A_BOOL          friendlyDetectEnable;   // friendly turbo detection feature (radar)
    A_BOOL          shortPreamble;          // Allow short and long preamble in 11b mode
    A_UINT32        ssidSuppressEnabled;    // suppress ssid in beacons/probe responese
    A_UINT32        dot1xMode;              // Mode of 802.1x operation for AP

    A_BOOL          leapEnabled;            // Cisco LEAP support for STA
    A_BOOL          mdcLeapEnabled;         // MDC view of LEAP config
#ifdef LEAP_IN_DRIVER
    A_BOOL          leapInDriver;           // Use Leap supplicant in driver
    A_CHAR          leapUserName[64];       // LEAP user name from registry
    A_UINT32        leapUserNameLen;        // len of above
    A_CHAR          leapUserPasswd[64];     // LEAP password from registry
    A_UINT32        leapUserPasswdLen;      // len of above
#endif
    A_UINT32        leapTimeout;            // timeout (ms) for auth to complete

    struct {
        union {
            A_UINT16 w;
            struct {
                A_BOOL      bldRclist;      // Build World Wide Roaming Channel List
                A_BOOL      cc11d;          // 802.11d found
                A_BOOL      cc11a;          // 802.11a's 802.11d
                A_BOOL      cc11b;          // 802.11b's 802.11d
                A_BOOL      commonMode;     // Use common mode
                A_BOOL      startAdhoc;     // Start ad-hoc
            } b;
        } v;
        START_ADHOC_OPTION  adHocMode;      // AdHoc start channel option
        struct {                            // Memory copy of what's in EEPROM
            A_BOOL          Inited;         // EEPROM read
            A_UINT16        RdCc;           // Regulatory Domain/Country Code
                                            // Subject to change by roaming code
        } eeprom;
        CTRY_CODE           ccByOid;        // Country code OID change
        COUNTRY_INFO_LIST   cc11dInfo;      // 802.11d Country Code Element
        CHAN_VALUES         adHocChannel;   // AdHoc channel
    } sku;
    A_BOOL          wpaEnabled;
    A_BOOL          wdsEnabled;
    A_BOOL          userClist;              // Use user-defined channel list

    /* Connection Services configuration */
    A_INT32         roamRssiA;              // RSSI level below which we'll try to roam
    A_INT32         roamRssiB;
    A_INT32         roamRssiBOnly;
    A_INT32         roamRateA;
    A_INT32         roamRateB;
    A_INT32         roamRateBOnly;

    A_UINT32        bssAgingPeriod;         // How many seconds before old BSSes
                                            // in the scan list are aged out
    A_BOOL          clearListOnScan;        // Should we clear the BSS list first before
                                            // background scanning?
    A_UINT32        scanType;               // active/passive scan, important for ETSI
    A_BOOL          bkScanEnable;           // Enable background scanning features
    A_BOOL          compressionSupport;     // HW, SW, and cfg allow compression
    A_BOOL          fastFramesSupport;      // HW, SW, and cfg allow fast frames
    A_UINT32        scanTimePreSleep;       // Time to continue scanning before sleeping (in s)
    A_UINT32        sleepTimePostScan;      // Time to sleep before starting to scan again (in s)
    A_UCHAR         gpioPinFuncs[NUM_GPIO_FUNCS]; // GPIO pin for each associated function
    ANTENNA_CONTROL diversityControl;       // Enable/disable antenna diversity

    A_UINT32        quietDuration;          // duration of quiet period
    A_UINT32        quietOffset;            // how far into the beacon is the quiet period
    A_BOOL          quietAckCtsAllow;       // do we ack/cts during quiet period

    A_UINT16        overRideTxPower;        // Override of transmit power
    A_UINT32        abolt;                  // Enable various special features
    A_UINT32        opMode;

    A_BOOL          reAssocEnable;          // Enable reassociaton
    A_UINT16        enable32KHzClock;       // auto or manual on/off
    A_BOOL          removeNoGSubId;         // Allows 11G operation even if subVendor type is restricted
#ifdef WME
    WMEv1_PARAM_PER_MODE wmeParamValues[WIRELESS_MODE_MAX]; // default values for WME related parameters (phyChAPs)
#endif
    A_BOOL          WmeEnabled;             // enabled WME tx queues
    A_BOOL          GprsCBRPeriod;          // enables gprs tx queue and sets CBR period
    A_BOOL          UpsdEnabled;            // enables sw-poll queue for voip
    A_UINT8         burstSeqThreshold;      // Max num of frames in a burst
    A_BOOL          translateNullSsid;      // translate null ssid string to a string of nulls.

#define MAX_PREFERRED_APS 4
    WLAN_MACADDR    prefBssids[MAX_PREFERRED_APS]; // for CCX preferred AP support
    A_BOOL          mixedPrivacyAllow;      // allow assoc to clear BSS, even if we have keys (used by STA only)
    A_UCHAR         clientName[17];

    A_UINT8         ccxRadioMeasEnable;
    A_UINT8         ccxRmMaxOffChanTime;
    A_UINT8         ccxRmOffChanTimeWhenBusy;

    A_UINT32        descsToKeep;

    A_UINT32        prefSuperAG;             // User preference for Super A/G features
    A_UINT32        regCapBits;   /* Regulatory capabilities mask */
} WLAN_STA_CONFIG;

/*
 * Update the SuperAG Capabilities information for infrastructure mode.
 * The prefSuperAG affects the configurable SUPERAG aspects of the abolt. 
 * The non-configurable SuperAG aspects should be unchanged
 */
#define UPDATE_SUPERAG_CAP(_pDev, _prefSuperAG) do {                                        \
        ((_pDev)->staConfig.abolt = ((_pDev)->defaultStaConfig.abolt & (_prefSuperAG) & ABOLT_CFG_SUPER_FEATURES) | \
                                    ((_pDev)->defaultStaConfig.abolt & ~ABOLT_CFG_SUPER_FEATURES));  \
} while (0)

typedef struct WlanStats {

    /* Receive related stats */

    A_LONGSTATS             GoodReceives;
    A_LONGSTATS             RxUnicastFrames;
    A_LONGSTATS             RxBroadcastFrames;
    A_LONGSTATS             MulticastReceives;
    A_UINT32                RxDiscardFrames;
    A_LONGSTATS             GoodReceiveBytes;

    A_UINT32                RxMgmtFrames;
    A_UINT32                RxCtrlFrames;
    A_UINT32                RxDataFrames;
    A_LONGSTATS             RxBeacons;
    A_UINT32                RxProbeResponses;

    A_LONGSTATS             ReceiveErrors;
    A_UINT32                RcvCrcErrors;
    A_UINT32                FcsFailCnt;
    A_UINT32                RcvKeyCacheMisses;
    A_UINT32                RcvDmaOverrunErrors;
    A_UINT32                RcvPhyErrors;
    A_LONGSTATS             RcvDupSeqErrors;
    A_UINT32                MultipleRxDuplicates;
    A_LONGSTATS             RcvDecryptCrcErrors;
    A_LONGSTATS             RcvDecompCrcErrors;
    A_LONGSTATS             RcvDecipherErrors;
    A_LONGSTATS             RcvDemicErrors;
    A_LONGSTATS             RcvDeMMHErrors;
    A_UINT32                RcvWEPExcludedCount;


    /* Transmit related stats */

    A_LONGSTATS             GoodTransmits;
    A_LONGSTATS             TxUnicastFrames;
    A_LONGSTATS             TxBroadcastFrames;
    A_LONGSTATS             TxMulticastFrames;
    A_LONGSTATS             TxFramesDropped;
    A_LONGSTATS             GoodTransmitBytes;
    A_UINT32                TxProbeRequests;

    A_UINT32                TxFrames[4];
#define TxDataFrames        TxFrames[FRAME_DATA]
#define TxMgmtFrames        TxFrames[FRAME_MGT]
#define TxCtrlFrames        TxFrames[FRAME_CTRL]
#define TxInvalidFrames     TxFrames[FRAME_RESERVED]

    A_UINT32                TransmitErrors;
    A_UINT32                TxExcessiveRetries;
    A_UINT32                TxDmaUnderrun;
    A_UINT32                AckRcvFailures;
    A_LONGSTATS             RtsSuccessCnt;
    A_UINT32                RtsFailCnt;

    A_UINT32                compSuccessCnt;
    A_UINT32                CompCPC0Cnt;
    A_UINT32                CompCPC1Cnt;
    A_UINT32                CompCPC2Cnt;
    A_UINT32                CompCPC3Cnt;

    A_UINT32                TxFiltered;
    A_LONGSTATS             TotalRetries;
    A_UINT32                RetryBins[16];              // long frame bin
    A_UINT32                shortFrameRetryBins[16];
    A_UINT32                swRetryTotalCnt;            // sw retries for this connection
    A_UINT32                swRetryMaxRetriesExceeded;

    A_UINT32                psMulticastFrames;
    A_UINT32                psRequestFrames;
    A_UINT32                psAgedFrames;
    A_UINT32                psSavedFrames;


    /* Rates/Antenna/RSSI etc. */

    A_INT32                 ackRssi;
    A_UINT32                RcvRate;
    A_INT32                 rssi;
    A_UINT32                txRateKb;                   // Averaged rate used for transmit
    A_UINT32                RxAntenna;
    A_UINT32                AntCnt[2];
    A_UINT32                AntSwCnt;
    A_UINT32                DataRateMismatches; // Mismatch of basic rates

    /* MLME related stats */

    A_UINT32                Associations;
    A_UINT32                Reassociations;
    A_UINT32                DisAssociations;
    A_UINT32                DisAssocReasonCode;
    A_UINT32                Authentications;
    A_UINT32                DeAuthentications;
    A_UINT32                DeAuthReasonCode;
    A_UINT32                AuthReqRefused;
    A_UINT32                AssocReqRefused;
    A_UINT32                AssocTimeouts;
    A_UINT32                AuthTimeouts;

    /* Misc stats - particular to the OS */

    A_UINT32                HungCount;                  // How many times tx got hung
    A_UINT32                hwResetCount;               // How many times we reset hardware
    A_UINT32                osResetCount;               // How many times OS reset us
    A_UINT32                NdisSendRequest;            // Number of times NDIS requested us to send a frame
    A_UINT32                NdisSendReRequest;          // Number of times NDIS had to hold a frame for us
    A_UINT32                PrefAPMismatches; // Had to settle for non-preferred AP


    /* MIBs */

    WLAN_MACADDR            txDisAssociationSta;
    A_UINT32                txDisAssocReasonCode;
    WLAN_MACADDR            txDeAuthenticationSta;
    A_UINT32                txDeAuthReasonCode;
    WLAN_MACADDR            txAuthenticateFailSta;
    A_UINT32                txAuthenticateFailStatus;
} WLAN_STATS;

/* Define the filename for pulling debug register values */
#define ATH_REG_ADD_FILENAME "regAdd.txt"
#define MAX_REG_ADD_COUNT    50

enum {
    LEAP_CHALLENGE_LEN = 8,
    LEAP_CHALLRESP_LEN = 24,
};

/* Defines for basic rate modes for 11g */
#define BASIC11G_DOT11      0   /* 1,2 */
#define BASIC11G_DOT11B     1   /* 1,2,5.5,11 */
#define BASIC11G_DOT11G     2   /* 1,2,5.5,11,6,12,24 */
#define BASIC11G_OFDM       3   /* 6,12,24 */
#define BASIC11G_MIN        0
#define BASIC11G_MAX        3

struct wlanDevInfo;             /* Forward reference */


enum {
    TIMER_STOPPED,
    TIMER_FUNC_QUEUED,
    TIMER_FUNC_RUNNING,
    TIMER_FUNC_CANCELLED,
};

typedef void (*PFN_TIMER_CALLBACK)(void *pContext);

/* Generic timer structure to be used by driver code */
typedef struct athTimerStruct {
    struct wlanDevInfo  *pdevInfo;      /* standard devinfo pointer */
    A_UINT32        param;              /* parameter */
    PFN_TIMER_CALLBACK  pFunc;          /* Function to be called */
    A_UINT32        operation;          /* operation that we are timing on */
    A_UINT32        timeVal;            /* timer value in milliseconds */
    void            *osTimerObj;        /* OS specific timer object */
    A_BOOL          repeat;             /* true if periodic timer */
    A_BOOL          state;              /* true if timer cancelled */
    A_ATOMIC        timerCount;         /* number of timer event outstanding */
} A_TIMER;

/*
 * sibEntry is a data structure pivotal to operation of all modules.
 * It maintains the state, callback pointers, timeouts, etc.
 * We need to define and add more information fields to this structure
 * for storing information about rates, fragmentation, power management,
 * etc. Hence it is expected that this structure will grow significantly
 * as the development progresses.
 *
 * First field is the link pointer. Rest of the fields should be
 * optimized for packing the fields in the structure. First pointers,
 * 32 bit values, then 16 bit values then variable length and byte
 * fields.
 */

typedef struct  sibEntry {
    /*
     * sib management related stuff
     */
    struct sibEntry     *pNext;         /* pointer to next in the list */
    A_SEM_TYPE          sibLock;        /* Lock for individual SIB */
    A_UINT16            used;           /* flag to reflect the status of use as below */
#define SIB_FREE        0               /* SIB entry is free */
#define SIB_USED        1               /* SIB entry is taken - set by sibEntryAlloc() */
#define SIB_VALID       2               /* Valid data in SIB entry - set by sibEntryAdd() */
#define SIB_BUSY        4               /* Someone is using this SIB, don't delete */

    /*
     * sib identification
     */
    WLAN_MACADDR        macAddr;        /* MAC address of station */
    struct OpBss        *pOpBss;        /* operational BSS this sib belongs to.. relevant on AP */

    WLAN_SERVICE        serviceType;    /* STA / AP */
    CAP_INFO            capInfo;        /* Capability of station */
    A_UINT16            staState;       /* Current state of station */
    A_UINT32            transState;     /* Current state of transaction */
    A_UINT16            assocId;        /* association ID */
    A_UINT16            listenInterval; /* listen interval in units of beacon interval */
    A_UINT16            Channel;
    A_UINT16            authenAlgo;     /* authentication algorithm used by this STA */
    A_UINT16            wlanMode;       /* wireless LAN mode, A, B, or G */
    WLAN_PRIV_RECORD    *pPriv;         /* WEP key record */
    CHLG_TEXT           *pChlg;         /* ptr to challenge text holding place during authentication */
    A_BOOL              wpaEnabled;     /* WPA enabled on this connection */
    A_BOOL              pairwiseKeysInstalled;  // For WPA
/*
 *  The staConfig will have all values supported by STA.  The lSib will have
 *  the union of all values supported by the driver and those currently enabled by the
 *  supplicant.  The pSib will have the values sent in assnReq.
 */
    A_UINT32            uCipher;        /* Selected unicast cipher by this station */
    A_UINT32            mCipher;        /* Selected multicast cipher by this station */
    A_UINT32            authSel;        /* Selected Auth Selector by this station */
    A_UINT16            wpaCapabilities; /* Read from beacon */
    ATH_ADVCAP_IE       athAdvCapElement;/* Adv Cap element */

    A_BOOL              privacyInvoked;         /* privacy enable status */
    A_BOOL              swEncryptionEnabled;    /* sw encryption enable status */
    A_BOOL              swMicEnabled;           /* sw MIC enable status */

/*
 *  Strategy:  For TKIP, we need 3 key values (1 encrypt/decrypt, 1 tx MIC,
 *  1 rx MIC).  The MIC keys must be installed at (i+64) in the cache.
 *  We will manage the TKIP keys by assigning usage ranges.  Encryption keys go
 *  into indices 0..31, decryption keys go into slots 32..63, enmic keys go into
 *  64..95, and demic keys go into 96..127.
 */
    A_INT16             hwIndex;                /* tx encryption key index */

    A_UINT8             antTx;                  /* current transmit antenna */

    A_UINT32            staLifeStartTime;       /* the time of the first frame from this STA (ms) */
    A_UINT32            staLastActivityTime;    /* the time of the last frame from this STA (ms) */
    WLAN_TIMESTAMP      lastBeaconTime;         /* timestamp of last beacon */
#ifdef WME
    SEQ_CONTROL         lastRxSeqControl[NWME];       /* Last frame's sequence number */
    SEQ_CONTROL         lastLastRxSeqControl[NWME];   /* Memory of the sequence no. before last. */
#else

    SEQ_CONTROL         lastRxSeqControl;       /* Last frame's sequence number */
    SEQ_CONTROL         lastLastRxSeqControl;   /* Memory of the sequence no. before last. */
#endif
    A_BOOL              receivedBeaconFromHomeAP; /* modulates when to use fake sleep during bkscan */
    WLAN_STATS          stats;                  /* Structure of statistics counters */

    WLAN_RATE_SET       workingRateSet;         /* Working rates to be used for current tx*/
    A_UINT16            lastRateKbps;           /* Rate selected for the last pkt, in kbps */

    ATHEROS_DESC        *smePendDesc;           /* SME frames pending on tx completion */

    struct sibEntry     *pBridgeApSibEntry;     /* ptr to bridge AP sib if this sta is visible through a bridge */
    A_INT16             wdcConnId;              /* WDC Connection ID */

    A_UINT16            psMode;                 /* power saveing mode */
    QUEUE_DETAILS       txPSQueue;              /* queue to hold tx frames under power save mode */
    QUEUE_DETAILS       upsdQueue;              /* buffered frames for upsd voip downlink */
    QUEUE_DETAILS       upsdNullQueue;          /* place to hold qos-null downlink frame */
    ATHEROS_DESC        *pTxFFDesc;
    ATHEROS_DESC        *pRxFFDesc;

    A_BOOL              qosdataEnabled;         /* send QoS DATA frames */
    A_BOOL              ckipEnabled;
    A_BOOL              ckipUseMic;
#if !defined(BUILD_AP) && defined(LEAP_IN_DRIVER)
    struct LeapContext {
        A_UINT32    leapState;
        A_UINT8     leapSessionKey[16];         /* full-length derived key used to decrypt EAPOL-Key msgs */
    /* These next members are used to generate the session key */
        A_UINT8     challengeToRadius[LEAP_CHALLENGE_LEN];
        A_UINT8     challengeToRadiusResponse[LEAP_CHALLRESP_LEN];
        A_UINT8     challengeToClient[LEAP_CHALLENGE_LEN];
        A_UINT8     challengeToClientResponse[LEAP_CHALLRESP_LEN];

        A_UINT64    replayCtr;                  /* Copy of last replay ctr */
    } leapCtx;
#endif /* !BUILD_AP && LEAP_IN_DRIVER */
    A_BOOL              cckmEnabled;    /* CCKM enabled for this connection */
    A_BOOL              cckmKrkValid;  /* True if cckmKRK is valid */
    A_UINT8             cckmKrk[KRK_LENGTH_BYTES];
    A_UINT32            cckmRn;         /* Reassociation Request Number */
    A_UINT8             cckmMic[8];
    TRACEBUF_DEFINE(Tx)
    TRACEBUF_DEFINE(Rx)
} SIB_ENTRY;

/* SIB Table */
#ifdef BUILD_AP
#define MAX_STA_NUM 64          /* "big" # for APs */
#else
#define MAX_STA_NUM 61          /* key cache size + localsta */
#endif

/* SIB Hash Table. Pointers to SIB entries */
#define MAX_STA_HASH_NUM    256 /* Max. possible hash values */

typedef struct sibTable {
    SIB_ENTRY   sta[MAX_STA_NUM];
    SIB_ENTRY   *pHashList[MAX_STA_HASH_NUM];
    SIB_ENTRY   *pFreeList;
    A_UINT32    sibTableSize;
    A_SEM_TYPE  sibTableSem;
    A_UINT32    currentEntryNum;
} SIB_TABLE;


/*
 * MAC ACL data structures.  Created by WEP configuration.  Managed
 * with the same hash table scheme as the SIB_ENTRYs.
 */
typedef struct macACL_entry {
    /* Points to first macACL_entry for this hash value */
    struct macACL_entry *pNext;
    WLAN_MACADDR        mac;
    A_UINT16            flags;
#define MACACL_NONE             0x0000
#define MACACL_PRIVACYINVOKED   0x0001
#define MACACL_DENY             0x0002
#define MACACL_DEFAULT          0x0004
#define MACACL_KEY              0x0008
    A_UINT16            key;
} MACACL_ENTRY;

typedef struct macACLHashTable {
    MACACL_ENTRY    *pFirst[MAX_STA_HASH_NUM];
} MACACL_HASH_TABLE;

extern MACACL_HASH_TABLE macACLHashTable;

typedef void * OS_DEVICE_ID;

/* Settings for 802.1x operation */
enum {
    KEYSRC_LOCAL = 0,
    KEYSRC_SERVER,
    KEYSRC_MIXED,
};

/*
 * Following information needs to be stored on the STA during
 * the process of scanning. It has the requested parameters
 * and received beacon/probe resp information too.
 */

typedef struct scanInfoStruct {
    struct BSSDescriptionSet bssSet;
    struct scanInfoStruct    *pLastInfo; /* used for "temporary" scans */
    BSSID           bssId;
    SSID_ARRAY      desSsids;
    BSSTYPE         bsstype;
    A_BOOL          newScan;
    SCANTYPE        scantype;       /* active or passive */
    WLAN_SCAN_LIST  *pScanList;     /* array of ptrs to the list of channels to scan */
    A_UINT16        curChannelIndex;/* List index of current channel being scanned */
    A_UINT32        scanStartTime;  /* Timestamp of when scan started on this channel */
    A_UINT32        scanEndTime;    /* Timestamp of when scan finished, used for aging */
    A_UINT32        minActiveTime;
    A_UINT32        maxActiveTime;
    A_UINT32        minPassiveTime;
    A_UINT32        maxPassiveTime;
    A_UINT32        minTime;
    A_UINT32        maxTime;
    A_BOOL          wasTimeAdded;
} SCAN_INFO;

/* Local and Overlap Bss 11g Protection Info structure */
typedef enum  {
    UPDATE_OFF = 0,     /* Protection off */
    UPDATE_HYST = 1,    /* Protection off hysteresis */
    UPDATE_ON = 2       /* Protection on */
} PROTECTION_UPDATE_STATE;

typedef enum  {
    PROT_OPTION_1 = 0x01,   /* Level 1 */
    PROT_OPTION_2 = 0x02,   /* Level 2 */
    PROT_OPTION_3 = 0x04,   /* Level 3 */
    PROT_OPTION_4 = 0x08,   /* Level 4 */
    PROT_OPTION_5 = 0x10    /* Level 5 */
} PROTECTION_OPTION;

typedef struct apProtectInfo {
    A_BOOL          otherPresent;   /* nonErp present on another BSS */
    A_INT32         timePres;       /* the time (ms) of the last otherPresent seen */
    A_BOOL          updatePresent;  /* update local present flag */
    A_BOOL          updateProtect;  /* update local protect flag */
    A_BOOL          updateLong;     /* update local long flag */
    A_BOOL          updatePreamble; /* update barker preamble mode flag */
    A_UINT32        countCckFrames; /* count of frames from 11b stas */
    A_UINT32        countLast;      /* last count of frames from 11b stas */
    int             timer;          /* number of beacon intervals */
    int             updateState;    /* used for protection control */
    int             upThresh;       /* threshold of cck frames per period to turn on protection */
    int             dnThresh;       /* threshold of cck frames per period to turn off protection */
    int             period;         /* sample period for protection control, in beacon intervals */
    int             option;         /* bitmap to determine protection control policy for tuning */
} AP_PROTECT_INFO;

/*
 * WLAN_DRIVER_INFO structure holds global driver information.
 */
typedef struct wlanDrvInfo {
    A_UINT32               nWorkingDev;         /* Number of "working" devices */
    A_UINT32               nMaxDev;             /* Highest valid pDev index */
    struct wlanDevInfo     *pDev[WLAN_MAX_DEV]; /* Array of devinfo pointers */
} WLAN_DRV_INFO;

struct HwFuncs;                     /* Fwd Decl see hal.h */
struct eepMap;                      /* Fwd Decl see ar52XX.h */
struct cservInfoStruct;             /* Fwd Decl see stacserv.h */
struct CcxInfo;                     /* Fwd Decl see ccx.h */

typedef struct rfSilentInfo {
    A_UINT32        gpioSelect;     /* GPIO pin to be used for RF silent */
    A_UINT32        polarity;       /* Polarity of data from pin which disables RF */
    A_BOOL          eepEnabled;     /* Reflects EEPROM bit for rf silent capability/switch */
    A_BOOL          swRadioDisable; /* Radio disable requested by software app */
    A_BOOL          hwRadioDisable; /* Radio disable requested by hardware switch */
    A_BOOL          radioEnabled;   /* Actual state of radio */
} RF_SILENT_INFO;

typedef struct pwrMgmtInfo {
    A_UINT32        powerState;             // Current overall power state: D0 - D3
    A_UINT32        powerSleepSubstate;     // Current sleep state: awake/asleep/pending change
    A_BOOL          sleepTimerEnable;       // Allow timer to run; when TRUE begin sleeping
    ATHEROS_DESC   *powerChangeTxDesc;      // descriptor associated with pending power change frame
    void           (*PowerRequestCallback)(void *, int, A_BOOL);// routine to call (typically MLME) when power state changes
    A_UINT32        hibernate;              // Current hibernate request state
    A_UINT32        radio;                  // Current rf kill request state
    A_UINT32        onAC;                   // Current 'on AC' request state
    A_UINT32        forceSleep;             // Current 'deep sleep' request state
    A_UINT32        fakeSleep;              // Current 'fake sleep' request state
    A_BOOL          apPmState;              // TRUE if AP believes STA is sleeping
    A_UINT32        internalAwake_1;        // Current internal awake (scan, etc.) request state
    A_UINT32        internalAwakeRefCnt_1;  // Allow return to sleep iff == 0
    A_UINT32        internalAwake_2;        // Current internal awake (scan, etc.) request state
    A_UINT32        internalAwakeRefCnt_2;  // Allow return to sleep iff == 0
    A_UINT32        internalAwake_3;        // Current internal awake (scan, etc.) request state
    A_UINT32        internalAwakeRefCnt_3;  // Allow return to sleep iff == 0
    A_UINT32        sleepDuration;          // max time we let the hw go to sleep
    A_UINT32        lastRememberedBeacon;   // used by sleepHangCheck
    A_BOOL          needResyncToBeacon;
    A_BOOL          dtim;                   // TRUE if we got one and still getting CAB traffic
} PWR_MGMT_INFO;

typedef struct gpioFuncInfo {
    A_UINT32        enabled;                // function is enabled
    A_UINT32        pin;                    // pin used by this function
    A_BOOL          state;                  // current state of GPIO pin
    A_UINT32        counter1;               // general purpose counter
    A_UINT32        timerParm1;             // general purpose variable for timer routine
} GPIO_FUNC_INFO;

// Enums for various GPIO functions
typedef enum  {
    POLL_EVENT,
    RECEIVE_EVENT,
    TRANSMIT_EVENT
} GPIO_FUNC1;

typedef struct RogueList {
    WLAN_MACADDR    macAddr;
    A_UINT16        failureCode;
} rogueListType;

typedef enum  {
    PRIME_BASE  = 0x00,
    PRIME_BOOST = 0x01,
    PRIME_HOLD  = 0x02
} TURBO_PRIME_STATE;

#define FRIENDLY_TURBO_AR_MAX_NUMBER_OF_PEAKS            3
#define FRIENDLY_TURBO_AR_MAX_ACK_RADAR_DUR              511

typedef struct ackRegionStruct {
    A_UINT16 peakPtrList[FRIENDLY_TURBO_AR_MAX_NUMBER_OF_PEAKS];
    A_UINT32 regionSum;
} ACK_REGION;

typedef struct friendlyTurboStruct {
    A_UINT32        prevWidth;
    A_UINT32        phyErrCount[FRIENDLY_TURBO_AR_MAX_ACK_RADAR_DUR];
    ACK_REGION      ackRegion;
    A_BOOL          turboAllow;
} FRIENDLY_TURBO;

/* TURBO_PRIME */
typedef struct turbo_prime_info {
    A_BOOL          turboPrimeAllowed;  /* this ap is configured for turbo prime */
                                        /* this sta has negotiated for turbo prime */
    A_BOOL          friendlyTurboAllowed;  /* this ap is configured for friendly turbo */
                                        /* this sta has negotiated for friendly turbo */
    A_BOOL          nonFriendlyStaPresent; /* flag for legacy (STA with no friendly Turbo support (3.0)) present */
    A_BOOL          nonFriendlyStaAssoc;   /* flag for legacy (STA with no friendly Turbo support (3.0)) assoc */
    A_BOOL          updateFlag;         /* used for turbo switch control */
    A_BOOL          modeChangeActive;   /* mode change is occurring */
    A_BOOL          legacyPresent;      /* flag for legacy (non turbo prime) present */
    A_BOOL          legacyAssoc;        /* flag for legacy assoc */
    A_UINT32        legacyActivity;     /* timestamp of last legacy activity */
    A_UINT32        primeByteCount;     /* count of bytes from turbo prime stas  */
    A_UINT32        primeByteCountPrev; /* previous count of bytes from turbo prime stas  */
    int             state;              /* turbo prime state machine */
    int             timer;              /* turbo prime timer */
    int             saveTimer;          /* temporary holding time */
    int             rateRecn;           /* rate control recommendation */
    A_UINT8         currentBoostState;  /* current value */
                                        /* tunable parameters */
    int             upThresh;           /* on threshold, adjusted to beacon period */
    int             dnThresh;           /* off threshold, adjusted to beacon period */
    int             periodIn;           /* time in turbo, adjusted to # of beacons */
    int             periodHold;         /* hysteresis time, adjusted to # of beacons */
    int             periodOut;          /* time out of turbo, adjusted to # of beacons */
    int             misc;               /* misc */
    A_UINT32        packetThreshold;    /* threshold on count of pattern matches */
    A_UINT32        parThreshold;       /* threshold on Peak-to-average ratio of pattern matches */
    FRIENDLY_TURBO  friendlyTurbo;      /* Hold previous tick for friendly turbo check */
} TURBO_PRIME_INFO;

/* CFP Parameters */
typedef struct cfp_info_s {
    int             flags;
    int             cfPeriod;           /*  8 */
    int             cfpMaxDuration;     /* 16 */
    int             cfEndDelay;         /* 16 */
} CFP_INFO;


/*
 * Operational BSS:
 * The AP may have multiple BSSs supported via a single WLAN device - for
 * repeater/bridging, or for running an XR BSS separate from the basic BSS
 * for the device; on the station side it captures the operational parameters
 * corresponding to the BSS - may change as the BSS changes (e.g. station
 * hops from one BSS to the other).
 * Currently, this structure is rather flat - as a convenience; over time
 * it is expected to gain the richness and depth of WLAN_DRV_INFO as more
 * and more fields move into it from there
 */
typedef struct OpBss {
    BSSID             bssId;
    const RATE_TABLE  *pRateTable;              // current global rate table
    WLAN_RATE_SET     rateSet;                  // rate set corresponding to the current rateTable
    A_UINT8           defaultRateIndex;         // default rate to use for non-sib or non-ratectrl frames
    WMEv1_PARAM       phyChAPs;                 // channel access paramters for the PHY

    A_UINT8           telecSupportReqd;         // telec regulatory requirement support enable
    A_SEM_TYPE        aidSem;                   // protect assocArray
#define AID_ARRAY_SZ  ((ASSOCID_MAX + 7) / 8)
    A_UINT8           assocArray[AID_ARRAY_SZ]; // Bit array for Association ID's
    A_UINT32          numAssociatedClients;     // number of associated clients
    A_UINT32          numSleepingClients;       // number of associated clients in Power Save

    struct BeaconInfo *pBeaconInfo;
    QUEUE_DETAILS     beaconQueue;              // queue detail for beacon queue
    QUEUE_DETAILS     mcPSQueue;                // buffer mc/bc frames before putting them on CAB queue

    QUEUE_DETAILS     txQueue;                  // queue details for transmit queue
    QUEUE_DETAILS     burstQueue;               // queue details for Burst queue
    QUEUE_DETAILS     altQueue;                 // Alternate queue; used as:
#define cabQueue      altQueue                  //      cabQueue for normal BSS on AP
#define psPollQueue   altQueue                  //      psPollQueue for normal station
#define grpPollQueue  altQueue                  //      groupPollQueue for XR BSS on AP

#ifdef WME
    QUEUE_DETAILS     wmequeues[NWME];          // WME output queues
#endif
} OP_BSS;

/* NOTE: Changes to AC<->TXQ mapping must propagate to AC_TO_QNUM and QNUM_TO_AC macros */
#define TXQ_ID_FOR_DATA         0       /* default for AP or STA */
#define TXQ_ID_FOR_PSPOLL       1       /* STA only */
#define TXQ_ID_FOR_XR_DATA      1       /* AP only */
#define TXQ_ID_FOR_XR_GRPPOLL   2       /* AP only */
#define TXQ_ID_FOR_XR_BEACON    3       /* AP only */
#define TXQ_ID_FOR_AC0          0       /* WME AC0 (AC_BK), AP or STA */
#define TXQ_ID_FOR_AC1          3       /* WME AC1 (AC_BE), AP or STA */
#define TXQ_ID_FOR_AC2          4       /* WME AC2 (AC_VI), AP or STA */
#define TXQ_ID_FOR_AC3          5       /* WME AC3 (AC_VO), AP or STA */
#define TXQ_ID_FOR_VIDEO        4       /* AP or STA */
#define TXQ_ID_FOR_UPSD         6       /* AP only */
#define TXQ_ID_FOR_GBURST       7
#define TXQ_ID_FOR_GPRS         7       /* AP only */
#define TXQ_ID_FOR_HCF          7       /* AP only */
#define TXQ_ID_FOR_CAB          8
#define TXQ_ID_FOR_BEACON       9
#define QNUM_TO_AC(_q) ( \
    ((_q) == 0) ? ACI_BK : \
    ((_q) == 4) ? ACI_Vi : \
    ((_q) == 5) ? ACI_Vo : \
    ((_q) == 3) ? ACI_BE :  \
    ACI_BE)
#define AC_TO_QNUM(_ac) ( \
    ((_ac) == ACI_BK) ? TXQ_ID_FOR_AC0 : \
    ((_ac) == ACI_Vi) ? TXQ_ID_FOR_AC2 : \
    ((_ac) == ACI_Vo) ? TXQ_ID_FOR_AC3 : \
    ((_ac) == ACI_BE) ? TXQ_ID_FOR_AC1 : \
    TXQ_ID_FOR_AC1)

/*
 * Declarations for Virtual Port Support
 */


/*
 * Types of Virtual Ports
 */
typedef enum {
    VPORT_WDS_TYPE = 0,
    VPORT_XR_TYPE,
    MAX_VPORT_TYPE,
} VPORT_TYPE;

/*
 * A Maximum of 16 WDS virtual ports are supported per physical port
 *
 */
#define MAX_VPORT_WDS_NUM   16

/*
 * Capabilities are determined by the Target.
 */
typedef struct {
    A_UINT32    targetVersion;
    A_UINT32    targetRevision;
    A_UINT32    macVersion;
    A_UINT32    macRevision;
    A_UINT32    phyRevision;
    A_UINT32    analog5GhzRevision;
    A_UINT32    analog2GhzRevision;

    A_UINT32    regDomain;
    A_UINT32    countryCode;
    A_UINT32    regCapBits;

    A_UINT32    wirelessModes;
    A_UINT32    connectionIdMax;
    A_UINT32    txQueueIdMax;
    A_UINT32    bssIdMax;

    A_UINT32    chanSpreadSupport;
    A_UINT32    compressSupport;
    A_UINT32    burstSupport;
    A_UINT32    fastFramesSupport;
    A_UINT32    chapTuningSupport;
    A_UINT32    turboGSupport;
    A_UINT32    turboPrimeSupport;
    A_UINT32    deviceType;

    A_UINT32    low2GhzChan;
    A_UINT32    high2GhzChan;
    A_UINT32    low5GhzChan;
    A_UINT32    high5GhzChan;

    A_BOOL      supportCipherWEP;
    A_BOOL      supportCipherAES_OCB;
    A_BOOL      supportCipherAES_CCM;
    A_BOOL      supportCipherTKIP;
    A_BOOL      supportCipherCKIP;

    A_BOOL      supportMicAES_CCM;
    A_BOOL      supportMicTKIP;
    A_BOOL      supportMicCKIP;
    A_UINT32    twiceAntennaGain5G;
    A_UINT32    twiceAntennaGain2G;

//  ...
} DEV_CAPABILITIES;

extern void cryptCapInit(struct wlanDevInfo *pdev);

/*
 * WLAN_DEV_INFO structure will hold all kinds of device related information.
 * It will hold OS specific information about the device and a device number.
 * Most of the code doesn't need to know what's inside that structure.
 * The fields are very likely to change.
 */
typedef struct wlanDevInfo {
    OS_DEV_INFO            *pOSHandle;
    TARGET_HANDLE          targetHandle;

    DEV_CAPABILITIES       devCap;            // combine with above xxxxLMWxxxx

    WLAN_STA_CONFIG        defaultStaConfig;  // default values for station configuration info
    WLAN_STA_CONFIG        staConfig;         // current values for station configuration info
    PCI_CARD_INFO          pciInfo;           // Struct to hold PCI related information

    REGISTER_VAL           *pInitRegs;

    OP_BSS                  baseBss;
    OP_BSS                  xrBss;

    struct CcxInfo         *pCcxInfo;        // storage for CCX-specific stuff

    struct sibTable        *pSibTable;       // back ptr to SIB table assoc with this device
    A_UINT32               sibTableSize;     // size of SIB table
    struct pwrMgmtInfo     powerMgmt;        // Control struct for device power management
    struct rfSilentInfo    rfSilent;         // Control struct for RF silent operation
    struct gpioFuncInfo    gpioFunc[NUM_GPIO_FUNCS]; // Control struct for each GPIO pin
    struct cservInfoStruct *pCsInfo;         // connection services data structure
    struct smeInfoStruct   *pSmeInfo;        // SME storage.
    struct radarInfoStruct *pRadarInfo;      // Dynamic radar information

    A_UINT32        devno;              // Device number
    SIB_ENTRY       *localSta;          // Pointer to station structure for the STA of this device
    BSSDESCR        *bssDescr;          // BSS Description of current AP
    SCAN_INFO       *pscanInfo;         // scan related info
    A_UINT32        baseAddress;        // virtual base address of hardware (mapped)
    A_UINT32        physAddress;        // physical base address of hardware
    A_BOOL          mapped;             // memory mapping has been done?
    A_UINT32        mapLength;          // length of mapped region

#ifdef BUILD_AP
    char            *pDmaBuf;           // ptr to device memory
    int             dmaBufSize;         // size of user specified memory
    char            *pMclMem;           // ptr to MBLK/CL_BLR memory
    A_UINT32        flags;
    int             ivec;               // Device interrupt vector
    int             ilevel;             // Device interrupt level
    QUEUE_DETAILS   psBufferQueue;
    A_UINT16        psQueueDepth;
    A_UINT16        rxQueueDepth;
    A_UINT16        maxLoanedCount;
    WDOG_ID         beaconWatchDogId;   // beacon watchdog ID
    CL_BLK_ID       pDeCryptOSDesc;     // pointer to os descriptor of decrypt buf
#else
    void            *pDescMemory;       // pointer to start of descriptor memory block
    void            *pSmallbufMemory;   // pointer to start of smallbuf memory block
    void            *pLargebufMemory;   // pointer to start of largebuf memory block
    A_UINT32        descPhysAddr;       // physical address of descriptor memory block
    A_UINT32        smallbufPhysAddr;   // Physical address of smallbuf memory block

    QUEUE_DETAILS   smallBufDescQueue;  // queue for empty descr/small buff pairs
    QUEUE_DETAILS   largeBufDescQueue;  // queue for empty descr/large buff pairs
#endif

#if defined(DEBUG) && defined(PCI_INTERFACE)
    A_UINT16        *pEarDebug;         // pointer to Ear Debug when read from file
    int             earDebugLength;     // Length of ear debug information in bytes
#endif
    A_UINT16        numEthClients;      // number of ethernet client devices
                                        // for wireless client mode
    A_UINT16        numDefragLanes;
    PARTIAL_DESC    defragLanes[DEFRAG_ENTRIES_MAX];

    QUEUE_DETAILS   emptyDescQueue;     // queue of free Atheros descriptors
    QUEUE_DETAILS   txHeaderQueue;      // queue of free Atheros descriptors
    A_ATOMIC        freeDescCount;      // Transmit descriptor count
    A_UINT16        txHeaderDescCount;  // Receive descriptor count

    A_UINT32        txFramesDropped;
    A_UINT32        rxFramesDropped;
    A_UINT32        psFramesDropped;

    QUEUE_DETAILS   txStagingQueue;
    QUEUE_DETAILS   burstStagingQueue;
    QUEUE_DETAILS   rxQueue;            // host rx queue
    QUEUE_DETAILS   txPendingQueue;

    /* used for 11g mode */
    A_UINT8         protectOn;          // use to gate G mode nonERP protection
    A_UINT8         protectRateIdx;     // rate to send G mode nonERP protection frames
    A_BOOL          useShortSlotTime;   // flag to use short slot time
    AP_PROTECT_INFO protectInfo;        // local and overlapping bss protection info
    A_BOOL          nonErpPreamble;     // flag to use long preamble in this BSS

    TURBO_PRIME_INFO turboPrimeInfo;    // turbo prime info
    CFP_INFO         cfpInfo;           // cfp info

    /* Device status related fields */
    A_BOOL          NicResetInProgress;
    A_BOOL          NdisResetInProgress;
    A_BOOL          NdisHaltInProgress;
    A_BOOL          wdsNegInProgress;
    A_BOOL          NdisDisassociated;
    A_UINT32        hwResetCount;
    A_UINT32        txUrn;
    A_UINT32        txUrnBurst;
    A_UINT32        txUrnBurstMs;
    A_UINT32        MaskReg;            // value written to interrupt mask register
    A_UINT32        isrValue;           // Interrupt safe copy of globIsrReg
    A_UINT32        globISRReg;         // current ISR value
    A_UINT32        NdisFilterReg;      // value of filter requested by NDIS
    A_UINT32        rxFilter;           // host copy of rxFilter setting
    WLAN_MACADDR    mcTable[MAX_MULTICAST_ADDRESSES]; // list of multicast addresses to receive on
    A_UINT          mcTableCount;       // number in above list
    A_UINT16        macVersion;         // MAC silicon version (Crete, Oahu etc).
    A_UINT16        macRev;             // MAC silicon revision
    A_UINT16        phyRev;             // PHY baseband silicon revision
    A_UINT16        analog5GhzRev;      // analog silicon revision
    A_UINT16        analog2GhzRev;      // analog silicon revision
    A_UINT16        channelCap;         // Channel capabilities, CCK, 2.4 GHz, etc.
    A_UINT32        rxOrnCnt;           // Count of RX overruns
    A_UINT32        rxEOLcount;         // Count of RX EOLs

    A_ATOMIC        totalRxIndicated;   // how many we've sent up the stack
                                        // cannot unload driver unless count is 0
    A_UINT32        targetRxPending;    // rx buffers owned by target
    A_BOOL          targetStopRec;
    A_TIMER         staPollTimer;
    A_TIMER         smeTimer;           // timer object for auth, assoc, join sme functions
    A_TIMER         leapTimer;
    A_SEM_TYPE      keySem;             // protect key
    A_UINT16        keyTableSize;       // total entries in keyTable
    A_UINT16        keyTableStaticSize; // static entries in keyTable
    A_UINT32        keyCacheSize;       // size of hardware key cache
    WLAN_PRIV_RECORD *keyTable;         // encryption key list
    WLAN_PRIV_RECORD **keyCache;        // encryption keys in HW
    SIB_ENTRY        **keyCacheSib;     // SIB owner of keys in HW
    A_UINT32        keyType;            // type of key, WEP, WEP2 or AES, negotiated in most recent assoc.
    A_UINT16        dot1xUcKeyIndex;    // unicast key index for dot1x
    A_BOOL          keyObfuscated;      // obfuscate the key values

    BLOWFISH_CTX    *Ctx;               // Control struct for WEP key encryption
    void            *virtEncryptBuf;    // pointer to start of encrypt buf
    A_UINT32        physEncryptBuf;     // Physical address of encrypt buf
    void            *virtDecryptBuf;    // pointer to start of decrypt
    A_UINT32        physDecryptBuf;     // Physical address of decrypt buf
    A_BOOL          transmitBusy;       // Some frames still in tx queue
    A_INT32         noActivityInterval; // power down if not activity for this time
    A_INT32         noActivityCur;      // current no activity time
    A_UINT32        tx6PowerInHalfDbm;  // The current power output level
    A_UINT32        maxTxPowerAvail;    // The maximum ouput power in half dBm
    A_UINT32        noiseFloor;         // The current noise floor
    A_BOOL          dropAllRxFrames;    // allows us to stop receive more easily in s/w
    A_UINT32        phyActivityFlag;
    A_UINT32        phyHwLastFcsFailCount;
    A_LONGSTATS     phyHwLastBeaconCount;

    A_UINT16        sleepBinMax;        // max number of values allowed in a sleep bin
    A_UINT16        sleepBinIndex;      // current sleep bin in use

    A_BOOL          useFastDiversity;
    A_UINT8         cachedDefAnt;       // can be 1 or 2
    A_UINT16        countOtherRxAnt;    // count consecutive rx on other antenna

    PHY_ERR_RECORD  phyErrRecord[PHY_ERR_MAX];
    WLAN_MACADDR    lastApAddr;         // used in reassociation request

    // (LW)
    //Temp hack, should revisit later
    RATE_TABLE      *hwRateTable[WLAN_MODE_MAX];
    A_BOOL          noApActivity;
    A_BOOL          invalidTxChannel;
    A_BOOL          rcvChangeInProgress;
    A_BOOL          flushInProgress;
    A_UINT8         progressWaitQueue;
    A_ATOMIC        numFlushCall;

    PACKET_QUEUE_HEADER     sendWaitQueue;      /* use to queue NDIS packet */
    A_SPIN_LOCK             sendLock;           /* use to protect sendWaitQueue */
    A_SPIN_LOCK             lock;               /* global lock to protect SME, timer */
    A_SPIN_LOCK             mapLock;            /* use to protect mapregister */
    A_UINT32                nWaitSend;
    A_UINT32                busStatus;
    NDIS_EVENT              noPendingPMEvent;
    A_BOOL                  bWorkItemInProgress;

#define ASSOC_INFO_BUF_SZ 512
    struct assocInfo {
        /* used by TGi supps/authenticators */
        A_BOOL          isReassoc;
        A_UINT8         reqBuf[ASSOC_INFO_BUF_SZ];
        A_UINT16        reqLen;
        A_UINT8         respBuf[ASSOC_INFO_BUF_SZ];
        A_UINT16        respLen;
    } assocInfo;
    A_UINT32        trafficBlockTimestamp; // in ms
    A_UINT32        lastMicError;          // in ms
    A_BOOL          micErrorM2Sent;
#ifdef BUILD_AP
    void        *vportTbl[MAX_VPORT_TYPE];
    A_UINT16    vportWdsBitMap;
#endif
    int             netJobRegRXID;      /* NetJob Registry ID for RX */
    int             netJobRegTXID;      /* NetJob Registry ID for TX */
#ifdef WME
    /* for MLME debugging purposes, should be taken out later */
    A_BOOL          usingQos;
#endif
    A_UINT32        ffThresh;           /* Fast Frame TX Threshold */
    A_CHAR          serialNumber[16];  
    A_UINT32        disableEnableCounter;
} WLAN_DEV_INFO;

#define DEV_INFO_FROM_CONTEXT(_pContext) (WLAN_DEV_INFO *)(_pContext)

typedef struct _ATH_WORKER_ITEM_CONTEXT{
    WLAN_DEV_INFO       *pDevInfo;
    PNDIS_WORK_ITEM      pWorkItem;
    A_BOOL               bSuspend;
} ATH_WORKER_ITEM_CONTEXT, *PATH_WORKER_ITEM_CONTEXT;
/*
 * Macro definitions
 */

/* Low pass filtering macros */

#define RSSI_LPF_LEN       10          // Low pass filter length for averaging RSSI values
#define RSSI_EP_MULTIPLIER 1 << 7      // Powers of two mean compiler optimizes out * and /
#define A_RSSI_OUT(x)      (A_EP_RND((x), RSSI_EP_MULTIPLIER))
#define A_RSSI_IN(x)       (A_EP_MUL((x), RSSI_EP_MULTIPLIER))
#define A_RSSI_LPF(x, y)   (A_LPF((x), A_RSSI_IN((y)), RSSI_LPF_LEN))

#define RATE_LPF_LEN       10          // Low pass filter length for averaging rates
#define RATE_EP_MULTIPLIER 1 << 7      // Powers of two mean compiler optimizes out * and /
#define A_RATE_OUT(x)      (A_EP_RND((x), RATE_EP_MULTIPLIER))
#define A_RATE_IN(x)       (A_EP_MUL((x), RATE_EP_MULTIPLIER))
#define A_RATE_LPF(x, y)   (A_LPF((x), A_RATE_IN((y)), RATE_LPF_LEN))

/* Misc. defines */

#define RFGAIN_POLL_INTERVAL     600   // Periodic check for RF gain, 60 seconds.
#define MAX_11B_FRAME_TIME       20000 // 2346 bytes * 8 us, rounded up, in us
#define MIN_RSSI_FOR_HANG_CHECK  5
#define HANG_CHECK_INTERVAL      200   // in us
#define HANG_CHECK_SAMPLES       (MAX_11B_FRAME_TIME / HANG_CHECK_INTERVAL)

/*
 * Sleep data
 */
typedef struct SleepBin {
    A_INT16         SleepCount;         // number times chip detected sleeping
    A_INT16         TotalCount;         // number times chip polled
} SLEEP_BIN;

typedef enum {
    MICERR_KEYMSG = 1,
    MICERR_RXFRAME,
} MIC_ERROR_TYPE;

/* External declarations for shared global variables from wlanglobal.c */

extern WLAN_DRV_INFO        gDrvInfo;
extern WLAN_MACADDR         broadcastMacAddr;
extern WLAN_MACADDR         nullMacAddr;
extern const EAPOL_SNAP_HDR eapolSnap;

extern const A_UINT8        ouiMicrosoft[];
extern const A_UINT8        ouiAtheros[];
extern const A_UINT8        ouiAironet[];

extern const A_UINT8        eidMap[];

extern const A_UINT8        ouiAtheros[];
extern const A_UINT8        elementIDMapping[];

A_BOOL micFailNow(void);


/*
 * Macros to access the phyport handle fields given
 * the virtual port handle
 */

#define VPORT_BITMAP(v) (((WLAN_DEV_INFO *)(v)->vport_cmn.phyPort)->vportBitMap)
#define VPORT_TBL(v,t)  (((WLAN_DEV_INFO *)(v)->vport_cmn.phyPort)->vportTbl[(t)])
#define VPORT_BASEBSS(v) (&((WLAN_DEV_INFO *)(v)->vport_cmn.phyPort)->baseBss)

/*
 * Macros to access the Vport Information stored in
 * the physical port handle.
 */

#define GET_PHY_INFO(r)   (gDrvInfo.pDev[r])
#define GET_VPORT_TBL(p,t) ((p)->vportTbl[t])
#define GET_VPORT_WDS_BMAP(p) (&(p)->vportWdsBitMap)
/*
 * Receive Function Handler for Virtual Ports
 */

typedef void (*VPORT_RX_FN)(void *pVportHandle, ATHEROS_DESC *pDesc);

/*
 * Virtual Port Statistics
 */
typedef struct Vport_Cntr {
    A_UINT32 frmRxd;
    A_UINT32 uniFrmRxd;
    A_UINT32 multiFrmRxd;
    A_UINT32 frmTxd;
    A_UINT32 uniFrmTxd;
    A_UINT32 multiFrmTxd;
} VPORT_CNTR;

/*
 * Registration parameters, when a virtual port is bound to a physical port
 */

typedef struct Vport_reginfo {
    VPORT_TYPE vport_type;         /* Discriminator for Vritual Ports */
    void       *pVportHandle;      /* Virtual Port cookie */
    VPORT_RX_FN vportRxFn;         /* Receive Handler for the frames rxd on a virtual port */
    union {
        WLAN_MACADDR peerMacAddr;  /* Discriminator for WDS Virtual Ports */
        /* Add other discriminitaor fields here */
    } s;
} VPORT_REGINFO;

/*
 * Virtual Port Related Tables
 */

/*
 * WDS Vport Table
 */

typedef struct VportWdsEntry {
        struct VportWdsEntry *next;
        WLAN_MACADDR wdsPeerAddr;
        void *pVportHandle;
} VPORTWDS_ENTRY;

typedef struct vportWdsTbl {
        VPORT_RX_FN     vportRxFn;
        VPORTWDS_ENTRY *pWdsHashList[MAX_VPORT_WDS_NUM];  /* Accessed using the MAC Address */
        void           *pVportHandleTbl[MAX_VPORT_WDS_NUM]; /* Indexed by the Virtual Port Number */
}VPORTWDS_TBL;



/*
 * API to bind a virtual port to a physical port
 */

extern WLAN_DEV_INFO *phyVportReg(A_UINT32 devNo, VPORT_REGINFO *pRegInfo);
extern A_BOOL vportWdsEntryFind(WLAN_DEV_INFO *pDevInfo, WLAN_MACADDR *txAddr,
                                VPORT_RX_FN *vportRxFn,
                                void **pVportHandle);
extern A_BOOL phyVportDeReg(A_UINT32 devno, VPORT_REGINFO *pRegInfo);
extern A_BOOL vportWdsEntryDel(VPORTWDS_TBL *pVportWdsTbl, A_UINT32 devno,
                               WLAN_MACADDR *wdsPeerAddr);

// Now include the OS specific header files for our drivers
#include "wlanos.h"

#ifdef MDK_BUILD
#include "commoninit_fns.h"
#endif

#define IS_QUEUE_BIT_SET(bmask, qnum)    (bmask & (1 << qnum))

#ifdef __cplusplus
}
#endif

#endif /* _WLANDRV_H_ */

