/*************************************************************************
 * Copyright Â© 2000-2003 Atheros Communications, Inc., All Rights Reserved
 *
 * Atheros and the Atheros logo and design are trademarks of Atheros
 * Communications, Inc.
 *
 * Sample Code from Microsoft Windows 2000 Driver Development Kit is
 * used under license from Microsoft Corporation and was developed for
 * Microsoft by Intel Corp., Hillsboro, Oregon: Copyright (c) 1994-1997
 * by Intel Corporation.
 *
 * $Id: //depot/sw/branches/1.3_USB_LINUX_port/src/USB/wlan/host/sta/ar5210.c#12 $
 * Main routines of the NDIS driver
 */

#include "crypto/ocb.h"
#include "crypto/wep.h"
#include "wlandrv.h"
#include "wlanDev.h"
#include "wlanPhy.h"
#include "wlanext.h"
#include "wlansmeext.h"
#include "wlanDev.h"
#include "wlanbeacon.h"
#include "wlanchannel.h"
#include "wlanSend.h"
#include "wlanReceive.h"
#ifdef Linux
#include "linuxdrv.h"
#include "linuxext.h"
#include "net/iw_handler.h"
#endif
#include "stacserv.h"
#include "athusbapi.h"
#include "ccx.h"
#include "intercept.h"
#include "usbstub.h"
#include "wlanCapDe.h"

/* Forward declarations */

static A_STATUS
registryDecryptInit(WLAN_DEV_INFO *pDevInfo, UCHAR *pKey, int keyLen);

static void
leapPasswdDecrypt(WLAN_DEV_INFO *pDevInfo);

static A_BOOL
parseMapFile(A_UCHAR *FilePtr, A_UINT32 FileLength, REGISTER_VAL *addRegs);

static NDIS_STATUS
initializeRegAdd(WLAN_DEV_INFO *pDevInfo, NDIS_HANDLE MiniportAdapterHandle);

#if defined(DEBUG) && defined(PCI_INTERFACE)
static NDIS_STATUS
initializeEarDebug(WLAN_DEV_INFO *pDevInfo, NDIS_HANDLE MiniportAdapterHandle);
#endif

static void
freeWlanResources(WLAN_DEV_INFO *pDevInfo, A_BOOL delete);

void
freeResources(IN WLAN_DEV_INFO *pDevInfo);

static NDIS_STATUS
initializeWlan(WLAN_DEV_INFO *pDevInfo);

static NDIS_STATUS
athReset(PBOOLEAN AddressingReset, NDIS_HANDLE MiniportAdapterContext);

#ifdef Linux
NDIS_STATUS
athInitialize(PDEVICE_OBJECT pDeviceObject);
#endif

static VOID
athGetReturnedPackets(NDIS_HANDLE context, PNDIS_PACKET pPacket);

static NDIS_STATUS
initializeHal(WLAN_DEV_INFO *pDevInfo);

static VOID
athShutdownHandler(NDIS_HANDLE context);

VOID
athHalt(NDIS_HANDLE context);

static BOOLEAN
athCheckForHang(NDIS_HANDLE context);

static void
freeStaticNdisResources(WLAN_DEV_INFO *pDevInfo);

#ifdef NDIS51_MINIPORT
static VOID
athPnPEventNotifyHandler(NDIS_HANDLE context, NDIS_DEVICE_PNP_EVENT PnPEvent,
                         PVOID InformationBuffer, ULONG InformationBufferLength);
#endif // NDIS51_MINIPORT

#ifdef NDIS40_MINIPORT
static void
cardRemovalNT4(WLAN_DEV_INFO *pDevInfo);

static NDIS_STATUS
cardReinsertionNT4(WLAN_DEV_INFO *pDevInfo);
#endif  // NDIS40_MINIPORT

static A_STATUS
athGetDevCap(WLAN_DEV_INFO * pDevInfo);

static A_STATUS
athGetDevStatus(WLAN_DEV_INFO * pDevInfo);

A_STATUS
athSetDevConfig(WLAN_DEV_INFO * pDevInfo);

static void
athDataMsgRecvInd(TARGET_HANDLE targetHandle,
		  WDC_EVENT     event,
		  void          *param,
		  A_UINT32      infoSize,
		  A_UCHAR       *pInfo);


static void
athSendCompleteInd(TARGET_HANDLE targetHandle,
		   WDC_EVENT     event,
		   void          *param,
		   A_UINT32      infoSize,
		   A_UCHAR       *pInfo);

static void
athStatusNotify(TARGET_HANDLE targetHandle,
		   WDC_EVENT     event,
		   void          *param,
		   A_UINT32      infoSize,
		   A_UCHAR       *pInfo);

static void
athRcvChangeNotify(TARGET_HANDLE targetHandle,
		   WDC_EVENT     event,
		   void          *param,
		   A_UINT32      infoSize,
		   A_UCHAR       *pInfo);

static void
athFlushCompNotify(TARGET_HANDLE targetHandle,
		   WDC_EVENT     event,
		   void          *param,
		   A_UINT32      infoSize,
		   A_UCHAR       *pInfo);

static void
athBmissNotify(TARGET_HANDLE targetHandle,
		   WDC_EVENT     event,
		   void          *param,
		   A_UINT32      infoSize,
		   A_UCHAR       *pInfo);

static void
athWlanStatsNotify(TARGET_HANDLE targetHandle,
		       WDC_EVENT     event,
		       void          *param,
		       A_UINT32      infoSize,
		       A_UCHAR       *pInfo);

static NDIS_STATUS
athSetPacketFilter(IN WLAN_DEV_INFO *pDevInfo, IN ULONG newPacketFilter);

void athusb_set_multicast (struct net_device *pNetDevice);

/*
 * This is registered as an event handler for WDCEVT_DEVICE_AVAILABLE.
 */
void athDeviceFound(TARGET_HANDLE targetHandle,
                    WDC_EVENT     event,
                    void          *param,
                    A_UINT32      infoSize,
                    A_UCHAR       *pInfo)
{
    WLAN_DEV_INFO  *pDevInfo = (WLAN_DEV_INFO *)param;

    ASSERT(event == WDCEVT_DEVICE_AVAILABLE);
    ASSERT(infoSize == 0);

    pDevInfo->targetHandle = targetHandle;
}

/*
 * Get a fixed-sized capability from the Target.
 * Note that this macro RETURNS from the caller
 * on failure!
 * Attention: you should make sure pdata is resident
 * memory, not temporary memory from stack.
 */
#define GET_CAP_FROM_TARGET(pDev, capability, pdata)    \
do {                                                    \
    A_UINT32 capData;                                   \
                                                        \
    wdcTargetGetCapability(pDev->targetHandle,          \
                           capability,                  \
                           NULL,                        \
                           (TARGET_CAP_DATA *)&capData, \
                           &status);                    \
    if (status == A_OK) {                               \
        *pdata = capData;                               \
    } else {                                            \
        return status;                                  \
    }                                                   \
} while(0)

/*
 * Retrieve all Capability information from target side
 */
static A_STATUS
athGetDevCap(WLAN_DEV_INFO * pDevInfo)
{
    A_UINT32    capSize;
    A_STATUS    status;
    A_UINT32    capData;

    DEV_CAPABILITIES *pCap = &(pDevInfo->devCap);

    GET_CAP_FROM_TARGET(pDevInfo,
                        CAP_TARGET_VERSION,
                        &pCap->targetVersion);

    GET_CAP_FROM_TARGET(pDevInfo,
                        CAP_TARGET_REVISION,
                        &pCap->targetRevision);

    GET_CAP_FROM_TARGET(pDevInfo,
                        CAP_MAC_VERSION,
                        &pCap->macVersion);

    GET_CAP_FROM_TARGET(pDevInfo,
                        CAP_MAC_REVISION,
                        &pCap->macRevision);

    GET_CAP_FROM_TARGET(pDevInfo,
                        CAP_PHY_REVISION,
                        &pCap->phyRevision);

    GET_CAP_FROM_TARGET(pDevInfo,
                        CAP_ANALOG_5GHz_REVISION,
                        &pCap->analog5GhzRevision);

    GET_CAP_FROM_TARGET(pDevInfo,
                        CAP_ANALOG_2GHz_REVISION,
                        &pCap->analog2GhzRevision);

    GET_CAP_FROM_TARGET(pDevInfo,
                        CAP_REG_DOMAIN,
                        &pCap->regDomain);

    GET_CAP_FROM_TARGET(pDevInfo,
                        CAP_COUNTRY_CODE,
                        &pCap->countryCode);

    GET_CAP_FROM_TARGET(pDevInfo,
                        CAP_REG_CAP_BITS,
                        &pCap->regCapBits);

    GET_CAP_FROM_TARGET(pDevInfo,
                        CAP_WIRELESS_MODES,
                        &pCap->wirelessModes);

    if ((pDevInfo->staConfig.abolt & ABOLT_TURBO_G) != ABOLT_TURBO_G) {
        pCap->wirelessModes &= ~MODE_SELECT_108G;
    }

    GET_CAP_FROM_TARGET(pDevInfo,
                        CAP_CONNECTION_ID_MAX,
                        &pCap->connectionIdMax);

    GET_CAP_FROM_TARGET(pDevInfo,
                        CAP_TX_QUEUE_ID_MAX,
                        &pCap->txQueueIdMax);

    GET_CAP_FROM_TARGET(pDevInfo,
                        CAP_CHAN_SPREAD_SUPPORT,
                        &pCap->chanSpreadSupport);

    GET_CAP_FROM_TARGET(pDevInfo,
                        CAP_COMPRESS_SUPPORT,
                        &pCap->compressSupport);

    pCap->compressSupport = pCap->compressSupport && (pCap->wirelessModes & (MODE_SELECT_11A | MODE_SELECT_11G));

    /* There currently is no WME_ELE abolt capability msg from the
       target.  This should just work -- we should not need to
       query this information.   For now, this causes RX failures.
       Disable it here. XXX */

    pDevInfo->staConfig.abolt            &= ~ABOLT_WME_ELE;
    pDevInfo->defaultStaConfig.abolt     &= ~ABOLT_WME_ELE;

    if (!pCap->compressSupport) {
        pDevInfo->staConfig.abolt        &= ~ABOLT_COMPRESSION;
        pDevInfo->defaultStaConfig.abolt &= ~ABOLT_COMPRESSION;
    }

    GET_CAP_FROM_TARGET(pDevInfo,
                        CAP_BURST_SUPPORT,
                        &pCap->burstSupport);

    if (!pCap->burstSupport) {
        pDevInfo->staConfig.abolt        &= ~ABOLT_BURST;
        pDevInfo->defaultStaConfig.abolt &= ~ABOLT_BURST;
    }

    GET_CAP_FROM_TARGET(pDevInfo,
                        CAP_FAST_FRAMES_SUPPORT,
                        &pCap->fastFramesSupport);

    pCap->fastFramesSupport = pCap->fastFramesSupport && (pCap->wirelessModes & (MODE_SELECT_11A | MODE_SELECT_11G));

    if (!pCap->fastFramesSupport) {
        pDevInfo->staConfig.abolt        &= ~ABOLT_FAST_FRAME;
        pDevInfo->defaultStaConfig.abolt &= ~ABOLT_FAST_FRAME;
    }

    GET_CAP_FROM_TARGET(pDevInfo,
                        CAP_CHAP_TUNING_SUPPORT,
                        &pCap->chapTuningSupport);

    GET_CAP_FROM_TARGET(pDevInfo,
                        CAP_TURBOG_SUPPORT,
                        &pCap->turboGSupport);

    GET_CAP_FROM_TARGET(pDevInfo,
                        CAP_TURBO_PRIME_SUPPORT,
                        &pCap->turboPrimeSupport);

    GET_CAP_FROM_TARGET(pDevInfo,
                        CAP_DEVICE_TYPE,
                        &pCap->deviceType);

    GET_CAP_FROM_TARGET(pDevInfo,
                        CAP_LOW_2GHZ_CHAN,
                        &pCap->low2GhzChan);

    GET_CAP_FROM_TARGET(pDevInfo,
                        CAP_HIGH_2GHZ_CHAN,
                        &pCap->high2GhzChan);

    GET_CAP_FROM_TARGET(pDevInfo,
                        CAP_LOW_5GHZ_CHAN,
                        &pCap->low5GhzChan);

    GET_CAP_FROM_TARGET(pDevInfo,
                        CAP_HIGH_5GHZ_CHAN,
                        &pCap->high5GhzChan);

    GET_CAP_FROM_TARGET(pDevInfo,
                        CAP_TWICE_ANTENNAGAIN_5G,
                        &pCap->analog5GhzRevision);

    GET_CAP_FROM_TARGET(pDevInfo,
                        CAP_TWICE_ANTENNAGAIN_2G,
                        &pCap->analog2GhzRevision);

    cryptCapInit(pDevInfo);

    return A_OK;
}

/*
 * Retrieve all Status information from target side
 */
static A_STATUS
athGetDevStatus(WLAN_DEV_INFO * pDevInfo)
{
    A_UINT32            statusSz;
    A_STATUS            status;
    WLAN_MACADDR        *macAddr;
    A_CHAR              *pSerialNumber;

    /*
     * Get MAC_ADDRESS from target EEPROM
     */
    wdcTargetGetStatus(pDevInfo->targetHandle,
                       ST_MAC_ADDR,
                       &statusSz,
                       (TARGET_STATUS_DATA *)&macAddr,
                       &status);

    if (status == A_OK) {
        A_MACADDR_COPY(macAddr,
                       &pDevInfo->staConfig.macPermAddr);
        wdcStatusDataFree(macAddr, statusSz);
    } else {
        return status;
    }

    /*
     * Get Serial Number from target
     */
    wdcTargetGetStatus(pDevInfo->targetHandle,
                       ST_SERIAL_NUMBER,
                       &statusSz,
                       (TARGET_STATUS_DATA *)&pSerialNumber,
                       &status);

    if (status == A_OK) {
        A_BCOPY(pSerialNumber, pDevInfo->serialNumber,
                sizeof(pDevInfo->serialNumber));
        wdcStatusDataFree(pSerialNumber, statusSz);
    } else {
        return status;
    }

    return A_OK;
}

/*
 * Retrieve all Capability information from target side
 */
A_STATUS
athSetDevConfig(WLAN_DEV_INFO * pDevInfo)
{
    A_STATUS        status;
    A_UINT32        cfgVal;


    if (A_MACADDR_COMP(&pDevInfo->staConfig.macAddr, &nullMacAddr) != 0) {
        wdcTargetSetConfiguration(pDevInfo->targetHandle,
                                  CFG_MAC_ADDR,
                                  WLAN_MAC_ADDR_SIZE,
                                  (TARGET_CONFIG_VAL *)&pDevInfo->staConfig.macAddr,
                                  &status);
    } else if (A_MACADDR_COMP(&pDevInfo->staConfig.macPermAddr, &nullMacAddr) != 0) {
        wdcTargetSetConfiguration(pDevInfo->targetHandle,
                                  CFG_MAC_ADDR,
                                  WLAN_MAC_ADDR_SIZE,
                                  (TARGET_CONFIG_VAL *)&pDevInfo->staConfig.macPermAddr,
                                  &status);
    }

    cfgVal = (A_UINT32)pDevInfo->staConfig.RegDomain;
    if (cfgVal != 0xffff) {
        /*
         * If the Target doesn't support Regulatory Domain Capability Override,
         * this will fail, and we'll continue to use the Regulatory Domain
         * specified by the Target.
         */
        wdcTargetSetConfiguration(pDevInfo->targetHandle,
                                  CFG_REG_DOMAIN,
                                  4,
                                  (TARGET_CONFIG_VAL)&cfgVal,
                                  &status);
        pDevInfo->devCap.regDomain = (A_UINT16)cfgVal;
    }

    cfgVal = (A_UINT32)pDevInfo->staConfig.rateCtrlEnable;
    wdcTargetSetConfiguration(pDevInfo->targetHandle,
                              CFG_RATE_CONTROL_ENABLE,
                              4,
                              (TARGET_CONFIG_VAL)&cfgVal,
                              &status);

    cfgVal = (A_UINT32)pDevInfo->staConfig.abolt;
    wdcTargetSetConfiguration(pDevInfo->targetHandle,
                              CFG_ABOLT,
                              4,
                              (TARGET_CONFIG_VAL)&cfgVal,
                              &status);

    cfgVal = (A_UINT32)pDevInfo->staConfig.calibrationTime;
    wdcTargetSetConfiguration(pDevInfo->targetHandle,
                              CFG_CALIBRATION_INTERVAL,
                              4,
                              (TARGET_CONFIG_VAL)&cfgVal,
                              &status);

    if (pDevInfo->pInitRegs) {
        wdcTargetSetConfiguration(pDevInfo->targetHandle,
                                  CFG_INIT_REGS,
                                  MAX_REG_ADD_COUNT * 8,
                                  (TARGET_CONFIG_VAL) pDevInfo->pInitRegs,
                                  &status);
    }

    cfgVal = (A_UINT32)pDevInfo->staConfig.diversityControl;
    wdcTargetSetConfiguration(pDevInfo->targetHandle,
                              CFG_DIVERSITY_CTL,
                              4,
                              (TARGET_CONFIG_VAL)&cfgVal,
                              &status);


    return status;
}

#if defined(DEBUG)
/* Print all Target status information. */
/* TBD: Handle endianness */
static void
athShowTargetStatus(WLAN_DEV_INFO *pDevInfo)
{
    A_STATUS           status;
    A_UINT32           allStatusSize;
    TARGET_STATUS_DATA allStatusData;

    uiPrintf("athShowTargetStatus:\n");

    wdcTargetGetStatus(pDevInfo->targetHandle,
                       ST_ALL,
                       &allStatusSize,
                       &allStatusData,
                       &status);
    if (status != A_OK) {
        uiPrintf("wdcTargetGetStatus failed (%d)\n", status);
        return;
    }

    {
        A_UINT32            statusSize;
        TARGET_STATUS_DATA  statusItem;
        A_STATUS            s;

        s=wdcGetStatusItem(allStatusData, ST_SERVICE_TYPE,
                             &statusSize, &statusItem);
        uiPrintf("Service Type: %d\n",
            (s == A_OK) ? (A_UINT32)statusItem : -s);

        s=wdcGetStatusItem(allStatusData, ST_WLAN_MODE,
                             &statusSize, &statusItem);
        uiPrintf("WLAN Mode: %d\n",
            (s == A_OK) ? (A_UINT32)statusItem : -s);

        s=wdcGetStatusItem(allStatusData, ST_FREQ,
                             &statusSize, &statusItem);
        uiPrintf("Frequency: %d\n",
            (s == A_OK) ? (A_UINT32)statusItem : -s);

        s=wdcGetStatusItem(allStatusData, ST_BAND,
                             &statusSize, &statusItem);
        uiPrintf("Band: %d\n",
            (s == A_OK) ? (A_UINT32)statusItem : -s);

        s=wdcGetStatusItem(allStatusData, ST_LAST_RSSI,
                             &statusSize, &statusItem);
        uiPrintf("Last RSSI: %d\n",
            (s == A_OK) ? (A_UINT32)statusItem : -s);

        s=wdcGetStatusItem(allStatusData, ST_PS_FRAMES_DROPPED,
                             &statusSize, &statusItem);
        uiPrintf("PS Frames Dropped: %d\n",
            (s == A_OK) ? (A_UINT32)statusItem : -s);

        s=wdcGetStatusItem(allStatusData, ST_CACHED_DEF_ANT,
                             &statusSize, &statusItem);
        uiPrintf("Cached Default Antenna: %d\n",
            (s == A_OK) ? (A_UINT32)statusItem : -s);

        s=wdcGetStatusItem(allStatusData, ST_COUNT_OTHER_RX_ANT,
                             &statusSize, &statusItem);
        uiPrintf("Count Other RX Antenna: %d\n",
            (s == A_OK) ? (A_UINT32)statusItem : -s);

        s=wdcGetStatusItem(allStatusData, ST_USE_FAST_DIVERSITY,
                             &statusSize, &statusItem);
        uiPrintf("Use Fast Diversity: %d\n",
            (s == A_OK) ? (A_UINT32)statusItem : -s);

        s=wdcGetStatusItem(allStatusData, ST_MAC_ADDR,
                             &statusSize, &statusItem);
        if (s == A_OK) {
            WLAN_MACADDR *ma = (WLAN_MACADDR *)statusItem;
            uiPrintf("MAC Address: 0x%x 0x%x 0x%x 0x%x 0x%x 0x%x\n",
                      ma->octets[0],
                      ma->octets[1],
                      ma->octets[2],
                      ma->octets[3],
                      ma->octets[4],
                      ma->octets[5]);
        } else {
            uiPrintf("MAC Address: Unknown\n");
        }

        s=wdcGetStatusItem(allStatusData, ST_SERIAL_NUMBER,
                             &statusSize, &statusItem);
        if (s == A_OK) {
            uiPrintf("Serial Number: %s\n", (char *)statusItem);
        } else {
            uiPrintf("Serial Number: Unknown\n");
        }

    }

    wdcStatusDataFree(allStatusData, allStatusSize);
}
#else /* !DEBUG */
#define athShowTargetStatus(pDevInfo)
#endif /* DEBUG */

//-----------------------------------------------------------------------------
// Procedure:    athInitialize
//
// Description: This routine is called once per each supported adapter card in
//              the system. This routine is responsible for initializing each
//              adapter. This includes parsing all of the necessary parameters
//              from the registry, allocating and initializing shared memory
//              structures, configuring the Atheros chip, registering the
//              interrupt, and starting the receive unit.
//
// Arguments:
//        OpenErrorStatus (mini) - Returns more info about any failure
//        SelectedMediumIndex (mini) - Returns the index in MediumArray of the
//                                    medium that the miniport is using
//        MediumArraySize (mini) - An array of medium types that the driver
//                                supports
//        MiniportAdapterHandle (mini) - pointer to the adapter object data area.
//
//        WrapperConfigurationContext (both) - A value that we will pass to
//                                            NdisOpenConfiguration.
//
//
// Returns:
//        NDIS_STATUS_SUCCESS - If the adapter was initialized successfully.
//        <not NDIS_STATUS_SUCCESS> - If for some reason the adapter didn't
//                                    initialize
//-----------------------------------------------------------------------------
#ifdef Linux
ENTRY_FN(NDIS_STATUS, athInitialize,
         (PDEVICE_OBJECT pDeviceObject),
         (pDeviceObject), ICEPT_INITIALIZE)
{
    NDIS_HANDLE MiniportAdapterHandle = (NDIS_HANDLE)pDeviceObject->pStubHandle;
#endif
    ULONG                i;
    NDIS_STATUS          Status;
    NDIS_HANDLE          ConfigHandle;
    WLAN_DEV_INFO        *pDevInfo = NULL;
    OS_DEV_INFO          *pOsInfo  = NULL;
    CSERV_INFO           *pCsInfo  = NULL;
    CHAR                 countryName[4] = "NA";
    DRV_BUF_DESC         *pBufDesc, *tpBufDesc;
    A_UINT32             physAddr;
    ULONG                ndisLibVersion;
    WLAN_STA_CONFIG      *pConfig;
    WLAN_STA_CONFIG      *pDefConfig;
    A_STATUS             athStatus;

    uiPrintf("> %s\n", __FUNCTION__);


    /* Reject second instance of the card */
    if (gDrvInfo.nWorkingDev >= gDrvInfo.nMaxDev) {
        /* Multiple cards is an unsupported configuration for now */
        uiPrintf("< %s: Multiple devices are not supported yet.\n", __FUNCTION__);
        NdisWriteErrorLogEntry(MiniportAdapterHandle,
                               NDIS_ERROR_CODE_UNSUPPORTED_CONFIGURATION, 4,
                               MiniportAdapterHandle, 1, NULL, NULL);
        return NDIS_STATUS_FAILURE;
    }

    // Allocate the Adapter Object, exit if error occurs
    pDevInfo = (WLAN_DEV_INFO *)A_DRIVER_MALLOC(sizeof(*pDevInfo));

    if (pDevInfo == NULL) {
        uiPrintf("< %s: WLAN_DEV_INFO allocate failed!\n", __FUNCTION__);
        NdisWriteErrorLogEntry(MiniportAdapterHandle,
                               NDIS_ERROR_CODE_OUT_OF_RESOURCES, 4,
                               MiniportAdapterHandle, 5, NULL, NULL);
        return NDIS_STATUS_RESOURCES;
    }

    /* Store in array of devinfo pointers. */
    gDrvInfo.pDev[gDrvInfo.nWorkingDev++] = pDevInfo;

    // Zero out the adapter object space
    A_MEM_ZERO(pDevInfo, sizeof(*pDevInfo));

    pOsInfo = (OS_DEV_INFO *)A_DRIVER_MALLOC(sizeof(*pOsInfo));

    if (pOsInfo == NULL) {
        uiPrintf("< %s: OS_DEV_INFO allocate failed!\n", __FUNCTION__);
        NdisWriteErrorLogEntry(MiniportAdapterHandle,
                               NDIS_ERROR_CODE_OUT_OF_RESOURCES, 4,
                               MiniportAdapterHandle, 10, NULL, NULL);
        freeResources(pDevInfo);
        return NDIS_STATUS_RESOURCES;
    }

    pDevInfo->pOSHandle = pOsInfo;
    A_MEM_ZERO(pOsInfo, sizeof(*pOsInfo));

    // Store off our MiniportAdapterHandle
    pOsInfo->NicAdapterHandle = MiniportAdapterHandle;

    uiPrintf("%s: pDevInfo = %08x\n", __FUNCTION__, pDevInfo);


    // Parse all of our configuration parameters.
    Status = ParseRegistryParameters(pDevInfo, ConfigHandle,
                                     &pDevInfo->defaultStaConfig);

    // If a required configuration parameter was not present, then error out
    if (Status != NDIS_STATUS_SUCCESS) {
        uiPrintf("< %s: ParseRegistryParameters() failed!\n", __FUNCTION__);
        NdisCloseConfiguration(ConfigHandle);
        athLogError(pDevInfo, MiniportAdapterHandle, Status, 20);
        freeResources(pDevInfo);
        return NDIS_STATUS_NOT_ACCEPTED;
    }

    // We read out all of our config/init info, so close
    // the gateway to the registry
    NdisCloseConfiguration(ConfigHandle);

    // Initialize user Super A/G preference to abolt.
    // It may get changed with each profile.
    pDevInfo->defaultStaConfig.prefSuperAG = pDevInfo->defaultStaConfig.abolt;

    // Now that we have read in all the parameters into defaultStaConfig, copy
    // them to staConfig where they can be changed dynamically.
    pDevInfo->staConfig = pDevInfo->defaultStaConfig;

    if (pDevInfo->staConfig.bssType == INDEPENDENT_BSS) {
        /*
         * No SuperG Features in Adhoc Mode
         */
        pDevInfo->staConfig.abolt = 0;
    }

    // call NdisMSetAttributesEx in order to let NDIS know
    // what kind of driver and features we support
#ifdef Linux
    pOsInfo->NicAdapterHandle->priv = (void *)pDevInfo;
#endif

    // allocate & Initialize the required Nic Shared Memory areas
    Status = SetupSharedAdapterMemory(pDevInfo);

    // Check the status returned from SetupSharedAdapterMemory
    if (Status != NDIS_STATUS_SUCCESS) {
        // Since we couldn't allocate enough shared memory, free any resources
        // that we previously allocated and error out.
        uiPrintf("< %s: SetupSharedAdapterMemory() failed! (0x%x)\n", __FUNCTION__, Status);
        athLogError(pDevInfo, MiniportAdapterHandle, Status, 35);
        freeResources(pDevInfo);
        return NDIS_STATUS_RESOURCES;
    }

    Status = initializeRegAdd(pDevInfo, MiniportAdapterHandle);
    if (Status != NDIS_STATUS_SUCCESS) {
        uiPrintf("< %s: initializeRegAdd() failed! (0x%x)\n", __FUNCTION__, Status);
        athLogError(pDevInfo, MiniportAdapterHandle, Status, 36);
        freeResources(pDevInfo);
        return NDIS_STATUS_RESOURCES;
    }

#if defined(DEBUG) && defined(PCI_INTERFACE)
    Status = initializeEarDebug(pDevInfo, MiniportAdapterHandle);
    if (Status != NDIS_STATUS_SUCCESS) {
        uiPrintf("< %s: initializeEarDebug() failed! (0x%x)\n", __FUNCTION__, Status);
        athLogError(pDevInfo, MiniportAdapterHandle, Status, 38);
        freeResources(pDevInfo);
        return NDIS_STATUS_RESOURCES;
    }
#endif

    ffInit(pDevInfo);

    InitializeQueueHeader(&pDevInfo->sendWaitQueue);
    pDevInfo->progressWaitQueue = 0;
    ATH_ALLOCATE_SPINLOCK(pDevInfo->sendLock,1);
    ATH_ALLOCATE_SPINLOCK(pDevInfo->lock,2);
    ATH_ALLOCATE_SPINLOCK(pDevInfo->mapLock,3);
    // Attach the target driver (bind to correct HAL functions, get EEPROM info etc).
{
    A_STATUS            status;
    A_UINT32            cfgVal;

    status = wdcInit();
    if (status != A_OK) {
        uiPrintf("< %s:%d wdcInit() failed! (0x%x)\n", __FUNCTION__, __LINE__, status);
        athLogError(pDevInfo, MiniportAdapterHandle,
                    NDIS_ERROR_CODE_OUT_OF_RESOURCES, 40);
        freeResources(pDevInfo);
        return NDIS_STATUS_FAILURE;
    }

    wdcRegisterEventHandler(NULL, WDCEVT_DEVICE_AVAILABLE,
                            athDeviceFound, (void *)pDevInfo, &status);
    if (status != A_OK) {
        uiPrintf("< %s:%d wdcRegisterEventHandler() failed! (0x%x)\n", __FUNCTION__, __LINE__, status);
        athLogError(pDevInfo, MiniportAdapterHandle,
                    NDIS_ERROR_CODE_OUT_OF_RESOURCES, 40);
        freeResources(pDevInfo);
        return NDIS_STATUS_FAILURE;
    }

#if Linux
    wdcDeviceDiscovered((DEVICE_DATA)pDeviceObject, &status);
#endif

    if (status != A_OK) {
        uiPrintf("< %s: wdcDeviceDiscovered() failed! (0x%x)\n", __FUNCTION__, status);
        athLogError(pDevInfo, MiniportAdapterHandle,
                    NDIS_ERROR_CODE_OUT_OF_RESOURCES, 40);
        freeResources(pDevInfo);
        return NDIS_STATUS_FAILURE;
    }

    ASSERT(pDevInfo->targetHandle); /* Set in athDeviceFound */

    wdcRegisterEventHandler(pDevInfo->targetHandle, WDCEVT_DATA_INDICATION,
                            athDataMsgRecvInd, (void *)pDevInfo, &status);
    if (status != A_OK) {
        uiPrintf("< %s:%d wdcRegisterEventHandler() failed! (0x%x)\n", __FUNCTION__, __LINE__, status);
        athLogError(pDevInfo, MiniportAdapterHandle,
                    NDIS_ERROR_CODE_OUT_OF_RESOURCES, 40);
        freeResources(pDevInfo);
        return NDIS_STATUS_FAILURE;
    }

    wdcRegisterEventHandler(pDevInfo->targetHandle, WDCEVT_SEND_COMPLETE,
                            athSendCompleteInd, (void *)pDevInfo, &status);
    if (status != A_OK) {
        uiPrintf("< %s:%d wdcRegisterEventHandler() failed! (0x%x)\n", __FUNCTION__, __LINE__, status);
        athLogError(pDevInfo, MiniportAdapterHandle,
                    NDIS_ERROR_CODE_OUT_OF_RESOURCES, 40);
        freeResources(pDevInfo);
        return NDIS_STATUS_FAILURE;
    }

    wdcRegisterEventHandler(pDevInfo->targetHandle, WDCEVT_STATUS_NOTIFY,
                            athStatusNotify, (void *)pDevInfo, &status);
    if (status != A_OK) {
        uiPrintf("< %s:%d wdcRegisterEventHandler() failed! (0x%x)\n", __FUNCTION__, __LINE__, status);
        athLogError(pDevInfo, MiniportAdapterHandle,
                    NDIS_ERROR_CODE_OUT_OF_RESOURCES, 40);
        freeResources(pDevInfo);
        return NDIS_STATUS_FAILURE;
    }

    wdcRegisterEventHandler(pDevInfo->targetHandle, WDCEVT_RX_IN_SYNC,
                            athRcvChangeNotify, (void *)pDevInfo, &status);
    if (status != A_OK) {
        uiPrintf("< %s:%d wdcRegisterEventHandler() failed! (0x%x)\n", __FUNCTION__, __LINE__, status);
        athLogError(pDevInfo, MiniportAdapterHandle,
                    NDIS_ERROR_CODE_OUT_OF_RESOURCES, 40);
        freeResources(pDevInfo);
        return NDIS_STATUS_FAILURE;
    }

    wdcRegisterEventHandler(pDevInfo->targetHandle, WDCEVT_FLUSH_COMPLETE,
                            athFlushCompNotify, (void *)pDevInfo, &status);
    if (status != A_OK) {
        uiPrintf("< %s:%d wdcRegisterEventHandler() failed! (0x%x)\n", __FUNCTION__, __LINE__, status);
        athLogError(pDevInfo, MiniportAdapterHandle,
                    NDIS_ERROR_CODE_OUT_OF_RESOURCES, 40);
        freeResources(pDevInfo);
        return NDIS_STATUS_FAILURE;
    }

    wdcRegisterEventHandler(pDevInfo->targetHandle, WDCEVT_BEACON_MISS,
                            athBmissNotify, (void *)pDevInfo, &status);
    if (status != A_OK) {
        uiPrintf("< %s:%d wdcRegisterEventHandler() failed! (0x%x)\n", __FUNCTION__, __LINE__, status);
        athLogError(pDevInfo, MiniportAdapterHandle,
                    NDIS_ERROR_CODE_OUT_OF_RESOURCES, 40);
        freeResources(pDevInfo);
        return NDIS_STATUS_FAILURE;
    }

    wdcRegisterEventHandler(pDevInfo->targetHandle, WDCEVT_WLAN_STATS_UPDATE,
                            athWlanStatsNotify, (void *)pDevInfo, &status);
    if (status != A_OK) {
        uiPrintf("< %s:%d wdcRegisterEventHandler() failed! (0x%x)\n", __FUNCTION__, __LINE__, status);
        athLogError(pDevInfo, MiniportAdapterHandle,
                    NDIS_ERROR_CODE_OUT_OF_RESOURCES, 40);
        freeResources(pDevInfo);
        return NDIS_STATUS_FAILURE;
    }

    status = athGetDevCap(pDevInfo);
    if (status != A_OK) {
        uiPrintf("< %s: athGetDevCap() failed! (0x%x)\n", __FUNCTION__, status);
        athLogError(pDevInfo, MiniportAdapterHandle,
                    NDIS_ERROR_CODE_OUT_OF_RESOURCES, 40);
        freeResources(pDevInfo);
        return NDIS_STATUS_FAILURE;
    }
    /* Check for irreversible upgrade for MKK SKUs */
    if (pDevInfo->staConfig.regCapBits & EN_KK_IRREVERSIBLE_UPDATE) {
        wlanApplyMKKIrreversibleUpdate(pDevInfo, 
                                      (pDevInfo->staConfig.regCapBits & EE_KK_ODD_U1_LEGACY), 
                                      pDevInfo->staConfig.regCapBits & EN_KK_UNDO_UPDATE);
        athGetDevCap(pDevInfo);
    }


    status = athGetDevStatus(pDevInfo);
    if (status != A_OK) {
        uiPrintf("< %s: athGetDevStatus() failed! (0x%x)\n", __FUNCTION__, status);
        athLogError(pDevInfo, MiniportAdapterHandle,
                    NDIS_ERROR_CODE_OUT_OF_RESOURCES, 40);
        freeResources(pDevInfo);
        return NDIS_STATUS_FAILURE;
    }

    wdcBind(pDevInfo, pDevInfo->targetHandle, 0, &status);
    if (status != A_OK) {
        uiPrintf("< %s: wdcBind() failed! (0x%x)\n", __FUNCTION__, status);
        athLogError(pDevInfo, MiniportAdapterHandle,
                    NDIS_ERROR_CODE_OUT_OF_RESOURCES, 40);
        freeResources(pDevInfo);
        return NDIS_STATUS_RESOURCES;
    }

    status = athSetDevConfig(pDevInfo);
    if (status != A_OK) {
        uiPrintf("< %s: athSetDevConfig() failed! (0x%x)\n", __FUNCTION__, status);
        athLogError(pDevInfo, MiniportAdapterHandle,
                    NDIS_ERROR_CODE_OUT_OF_RESOURCES, 40);
        freeResources(pDevInfo);
        return NDIS_STATUS_FAILURE;
    }

    wdcTargetStart(pDevInfo->targetHandle, &status);
    if (status != A_OK) {
        uiPrintf("< %s: wdcTargetStart() failed! (0x%x)\n", __FUNCTION__, status);
        athLogError(pDevInfo, MiniportAdapterHandle,
                    NDIS_ERROR_CODE_OUT_OF_RESOURCES, 40);
        freeResources(pDevInfo);
        return NDIS_STATUS_FAILURE;
    }
}

#if NDIS_WDM
    /*
     * (LW) Attention !!!
     * This is temp hack, should revisit later
     */
    pDevInfo->invalidTxChannel = FALSE;
#else
    HACK_TO_PARDEV(pDevInfo)->invalidTxChannel = FALSE;
#endif

    pDevInfo->rcvChangeInProgress = FALSE;

    NdisInitializeEvent(&pDevInfo->noPendingPMEvent);
    pDevInfo->bWorkItemInProgress = FALSE;
    NdisSetEvent(&pDevInfo->noPendingPMEvent);

    /*
     * Process the country code after halAttach to verify goodness
     * Read EEPROM for country code setting if possible (EEPROM
     * should have either country code or regulatory domain setting)
     * If country code is unknown or regulatory domain is configured,
     * then get country code from the registry
     */
    pDevInfo->defaultStaConfig.countryCode = wlanGetDefaultCountry(pDevInfo);
    if (pDevInfo->defaultStaConfig.countryCode == CTRY_DEFAULT) {
        A_UINT16 countryCode;

        countryCode = wlanGetCountryCodeByName(pOsInfo->countryName.Buffer, TRUE);
        if (countryCode != CTRY_INVALID &&
            wlanIsCountryCodeValid(pDevInfo, countryCode))
        {
            pDevInfo->defaultStaConfig.countryCode = countryCode;
        }
    }
    pDevInfo->staConfig.countryCode = pDevInfo->defaultStaConfig.countryCode;

    /*
     * Setup 11g configuration.
     * TODO: Make protectionRateIdx dynamically adjust
     */
    pDevInfo->useShortSlotTime = FALSE;
    pDevInfo->nonErpPreamble   = FALSE;
    
    pDevInfo->protectOn        = (pDevInfo->staConfig.modeCTS == 1) ? 1 : 0;
    pDevInfo->protectRateIdx   = (pDevInfo->staConfig.rateCTS > 0) ?
                                 pDevInfo->staConfig.rateCTS - 1 : 0;
    
    pDevInfo->staConfig.gBeaconRate = LOWEST_RATE_INDEX;    // ad hoc beacons
    athAttachRateTables(pDevInfo);

    /* Initialize configured power control limiter to maximum 31.5 dBm */
    pDevInfo->defaultStaConfig.tpcHalfDbm2 = 60;
    pDevInfo->defaultStaConfig.tpcHalfDbm5 = 60;
    pDevInfo->staConfig.tpcHalfDbm2        = 60;
    pDevInfo->staConfig.tpcHalfDbm5        = 60;

    pDevInfo->staConfig.opMode = OP_MODE_STA;

    if ((athStatus = descQInit(&pDevInfo->txPendingQueue)) != A_OK) {
        return NDIS_STATUS_FAILURE;
    }

	/* Under Linux Device is started only during netdevice open */
#ifdef Linux
    pConfig              = &pDevInfo->staConfig;
    memcpy(pOsInfo->NicAdapterHandle->dev_addr, &pConfig->macPermAddr.octets[0], WLAN_MAC_ADDR_SIZE);
#else
    // Initialization HAL (halReset) and channel list (based on regulatory domain/clist).
    Status = initializeHal(pDevInfo);
    if (Status != NDIS_STATUS_SUCCESS) {
        uiPrintf("< %s: initializeHal() failed! (0x%x)\n", __FUNCTION__, Status);
        freeResources(pDevInfo);
        return Status;
    }
#endif


    // Setup and initialize the receive structures
    Status = SetupReceiveQueues(pDevInfo);
    if (Status != NDIS_STATUS_SUCCESS) {
        uiPrintf("< %s: SetupReceiveQueues() failed! (0x%x)\n", __FUNCTION__, Status);
        athLogError(pDevInfo, MiniportAdapterHandle, Status, 55);
        freeResources(pDevInfo);
        return NDIS_STATUS_RESOURCES;
    }


    // Pre-set power mgmt states
    pDevInfo->powerMgmt.hibernate  = DISABLED;
    pDevInfo->powerMgmt.radio      = ENABLED;
    pDevInfo->powerMgmt.onAC       = DISABLED;
    pDevInfo->powerMgmt.forceSleep = DISABLE;
    pDevInfo->powerMgmt.fakeSleep  = DISABLE;

    // Initialize CCX functionality
    athStatus = ccxInit(pDevInfo);
    if (athStatus != A_OK) {
        uiPrintf("< %s: ccxInit() failed! (0x%x)\n", __FUNCTION__, athStatus);
        freeResources(pDevInfo);
        return NDIS_STATUS_FAILURE;
    }

#if !NDIS_WDM
    // initialize GPIO functions
    initGPIO(pDevInfo);
#endif

    Status = initNdisTimers(pDevInfo);
    ASSERT(Status == NDIS_STATUS_SUCCESS);
    SYSTEMSTATE_SET(pDevInfo, ATHUSB_SYSTEM_INIT);

#ifndef Linux
    /* initializeWlan becomes the netdev-> open and will be called by the 
     * network subsystem */
    // Initialization WLAN layer.
    Status = initializeWlan(pDevInfo);
    if (Status != NDIS_STATUS_SUCCESS) {
        uiPrintf("< %s: initializeWlan() failed! (0x%x)\n", __FUNCTION__, Status);
        freeResources(pDevInfo);
        return Status;
    }
#endif

    // Store the current uptime, in seconds
    NdisGetSystemUpTime(&i);
    pOsInfo->startTime = i / 1000;

    uiPrintf("< %s\n", __FUNCTION__);
    /*
     * Indicate that we're done with initialization so the first connection
     * status call can be made, which has to be done at DISPATCH level.
     *
     * Also need to keep track of the very first link speed query which is the
     * only time pre-WinXP OS's set their TCP window size.
     */
    pOsInfo->postInit          = TRUE;
    pOsInfo->postInitLinkSpeed = TRUE;


    return NDIS_STATUS_SUCCESS;
}

//-----------------------------------------------------------------------------
// initializeHal
//
// DESCRIPTION:
//    Initializes the HAL.
//      - Create channel list from regulatory domain or clist
//      - halReset
//
//    NOTE: This function can be called outside of the MiniportInitialize context.
//          Do not allocate any NDIS resources in this function.
//
// RETURNS:
//    NDIS_STATUS
//-----------------------------------------------------------------------------
NDIS_STATUS
initializeHal(WLAN_DEV_INFO *pDevInfo)
{
    A_STATUS        Status                = A_OK;
    OS_DEV_INFO     *pOsInfo              = pDevInfo->pOSHandle;
    NDIS_HANDLE     MiniportAdapterHandle = pOsInfo->NicAdapterHandle;
    WLAN_STA_CONFIG *pConfig              = &pDevInfo->staConfig;
    WLAN_STA_CONFIG *pDefConfig           = &pDevInfo->defaultStaConfig;
    A_STATUS        clistUsed;
    A_UINT32        wMode;

    uiPrintf("> %s\n", __FUNCTION__);

    /*
     * Set a flag to block all outside calls from the OS. We will clear this
     * flag when we have allocated resources, etc. We can't unload the driver
     * if this call fails during a reconfiguration, and we don't want the OS
     * to call into our driver if we don't have resources allocated, etc. for
     * fear of dereferencing NULL pointers and the like (see bug 6502).
     */
    pOsInfo->openForBusiness = FALSE;

    wMode = pDevInfo->devCap.wirelessModes;
    if (!wMode) {
        // No wireless mode defined in EEPROM
        uiPrintf("< %s: no wireless mode defined in EEPROM!\n", __FUNCTION__);
        athLogError(pDevInfo, MiniportAdapterHandle,
                    NDIS_ERROR_CODE_HARDWARE_FAILURE, 45);
        return NDIS_STATUS_FAILURE;
    }

    /*
     * Clear the soft EEPROM flag so we can reread what's in the hardware
     * Process the country code after halAttach to verify goodness
     * Read EEPROM for country code setting if possible (EEPROM
     * should have either country code or regulatory domain setting)
     * If country code is unknown or regulatory domain is configured,
     * then get country code from the registry
     */

    if (wlanVerifyEepromCcRd(pDevInfo) == FALSE) {
        // No wireless mode defined in EEPROM
        uiPrintf("< %s: invalid country code/regulatory domain in EEPROM!\n", __FUNCTION__);
        athLogError(pDevInfo, MiniportAdapterHandle,
                    NDIS_ERROR_CODE_HARDWARE_FAILURE, 45);
        return NDIS_STATUS_FAILURE;
    }

    wlanEepromCcRdInit(pDevInfo);
    if (pConfig->sku.ccByOid == 0) {
        pDefConfig->countryCode = wlanGetDefaultCountry(pDevInfo);
    } else {
        pDefConfig->countryCode = pConfig->sku.ccByOid;
    }

    if (wlanIsWwrSKU(pDevInfo)) {
        pConfig->sku.v.b.commonMode = TRUE;   /* Use common mode */
    }

    uiPrintf("%s: defaultStaConfig.countryCode = %d\n", __FUNCTION__, pDefConfig->countryCode);

    if (pDefConfig->countryCode == CTRY_DEFAULT) {
        A_UINT16 countryCode;

        countryCode = wlanGetCountryCodeByName(pOsInfo->countryName.Buffer, TRUE);
        if (countryCode != CTRY_INVALID &&
            wlanIsCountryCodeValid(pDevInfo, countryCode))
        {
            pDefConfig->countryCode = countryCode;
            uiPrintf("%s: new country code = %d\n",__FUNCTION__, 
                     pDefConfig->countryCode);
        }
    }

    pConfig->countryCode = pDefConfig->countryCode;

    /*
     * Override the default INF setting of
     * adhoc netBand (11B) if the card does not support
     * 11B/G e.g. a CB21
     */
    if ((pConfig->AdhocBand == ATH_ADHOCBAND_BONLY ||
         pConfig->AdhocBand == ATH_ADHOCBAND_GONLY ||
         pConfig->AdhocBand == ATH_ADHOCBAND_108GONLY) &&
        !(wMode & MODE_SELECT_11B))
    {
        pConfig->AdhocBand = ATH_ADHOCBAND_AONLY;
    }

    /* Modify the regulatory domain channel list for MKK to contain
     * channels conforming to new government regulations.
     */
    wlanModifyMKKChannels(pDevInfo);

    /*
     * If a channel list (clist) is specified in the registry use that,
     * otherwise use the regulatory domain.
     */
    clistUsed = asciiToClist(pDevInfo, pOsInfo->clist.Buffer,
                             pOsInfo->clist.Length,
                             &pConfig->pClist);

    pConfig->userClist = (clistUsed == A_OK) ? TRUE : FALSE;

    if (clistUsed != A_OK) {
        A_UINT32  adhocWMode;

        /* Initially assume netBand is valid */
        pConfig->bInvalidNetBand = FALSE;

        /*
         * Verify hardware support for NetBand
         * If hardware does not supported 11g, then clear 11g and
         * use 11b if it is supported by the hardware.
         */
        if (pConfig->NetBand & MODE_SELECT_11G &&
            !(wMode & MODE_SELECT_11G))
        {
            pConfig->NetBand &= ~MODE_SELECT_11G;
            if (wMode & MODE_SELECT_11B) {
                pConfig->NetBand |= MODE_SELECT_11B;
            }
        }

        /*
         * If hardware does not support 11g turbo, then
         * clear 11g turbo and use 11g if it is available.
         * Otherwise, use 11b if it is available.
         */
        if (pConfig->NetBand & MODE_SELECT_108G &&
            !(wMode & MODE_SELECT_108G))
        {
            pConfig->NetBand &= ~MODE_SELECT_108G;

            if (wMode & MODE_SELECT_11G) {
                pConfig->NetBand |= MODE_SELECT_11G;
            } else if (wMode & MODE_SELECT_11B) {
                pConfig->NetBand |= MODE_SELECT_11B;
            }
        }

        /*
         * Validate NetBand
         *
         * A card swap can make the configured NetBand invalid, if the new
         * card has a different reg domain/country code.
         * If netBand is zero, then set it what the hardware supports.
         */
        wMode = wlanGetWirelessMode(pDevInfo, pConfig->countryCode);
        pConfig->NetBand &= wMode;

        if (pConfig->NetBand == 0) {
            pConfig->NetBand = (A_UINT16)wMode;
        }

        if (pConfig->bssType == INDEPENDENT_BSS) {
            /*
             * No SuperG Features in Adhoc Mode
             */
            pConfig->abolt = 0;

            /*
             * Validate AhocBand
             *
             * If card is 11a only, then make sure the adHoc is correct
             * because the default adhoc value could be set to 11b/g only.
             * If card doesn't support adhoc and don't support 802.11b/g,
             * force adhocBand = NONE.
             * otherwise adhocBand...
             * 1. must be supported by the hardware/current reg domain
             * 2. match one of the user configured wireless modes
             */
            adhocWMode = cservAdHocBand2Wmode(pConfig->AdhocBand);

            if ( (wMode == MODE_SELECT_11A ||
                  wMode == (MODE_SELECT_11A | MODE_SELECT_TURBO)) &&
                 (adhocWMode == MODE_SELECT_11B || adhocWMode == MODE_SELECT_11G ||
                  adhocWMode == MODE_SELECT_108G) )
            {
                /* Force to 11a only */
                adhocWMode = MODE_SELECT_11A;
            }

            if ( (adhocWMode & pConfig->NetBand &
                  wlanGetAdhocWirelessMode(pDevInfo,
                                           pConfig->countryCode)) == 0 )
            {
                /* Card does not support this netband */
                uiPrintf("%s: hardware does not support ad hoc mode!\n"
                         "-> adhowWMode = %x, NetBand = %x, getAdhoc = %x\n",
                         __FUNCTION__, adhocWMode, pConfig->NetBand,
                         wlanGetAdhocWirelessMode(pDevInfo, pConfig->countryCode));
                pConfig->AdhocBand = ATH_ADHOCBAND_NONE;
            }

            if (pConfig->AdhocBand == ATH_ADHOCBAND_BONLY ||
                pConfig->AdhocBand == ATH_ADHOCBAND_GONLY)
            {
                pConfig->NetBand &= ~MODE_SELECT_11G;
                pConfig->NetBand |=  MODE_SELECT_11B;
            }
        } else {
            /*
             * Restore the SuperG Feature Setting for the Infrastructure Mode.
             */
            UPDATE_SUPERAG_CAP(pDevInfo, pConfig->prefSuperAG);
#if !NDIS_WDM
            updtSupFeatureInfo(pDevInfo,0);
#endif
        }

        /*
         * Build channel list (Release any old channel list)
         */
        if (pConfig->pClist) {
            uiPrintf("%s: freeing pClist (%p, size = %d)\n", __FUNCTION__,
                     pConfig->pClist, pConfig->pClist->listSize);
            wlanFreeChannelList(pConfig->pClist);
            pConfig->pClist = NULL;
        }

        if (wlanInitChannelList(pDevInfo, &pConfig->pClist,
                                pConfig->countryCode,
                                pConfig->NetBand, TRUE, FALSE) != A_OK)
        {
            // Adapter failed to create a channel list
            uiPrintf("< %s: wlanInitChannelList() failed!\n", __FUNCTION__);
            athLogError(pDevInfo, MiniportAdapterHandle,
                        NDIS_ERROR_CODE_HARDWARE_FAILURE, 45);
            return NDIS_STATUS_FAILURE;
        }
    }

    /* Check for invalid adhoc configuration */
    if (pConfig->AdhocBand == ATH_ADHOCBAND_NONE &&
        pConfig->bssType == INDEPENDENT_BSS)
    {
        /* Mark as invalid, radio will be disabled */
        pConfig->bInvalidNetBand = TRUE;
    }

    pDefConfig->pChannel = pConfig->pChannel = wlanFirstChannel(pDevInfo, pConfig->pClist);
    pDefConfig->phwChannel = pConfig->phwChannel = pConfig->pChannel;

    if (pConfig->scanType == ANY_SCAN) {
        // default was not changed by registry or any other setting.
        // Let's change it if passive scan is required.
        pConfig->scanType = wlanIsActiveScanAllowed(pDevInfo) ?
                            ACTIVE_SCAN : PASSIVE_SCAN;
        uiPrintf("%s: scan type is %s\n", __FUNCTION__, 
                 pConfig->scanType == ACTIVE_SCAN ? "active" : "passive");
    }

#ifdef XR_HACKERY
    /*
     * XR HACKERY - Currently the station becomes XR by being given
     * an XR channel in the clist; following that:
     * - it does passive scan only (for now)
     * - it changes its mac addr as the AP keys off the sta's macaddr
     *   to classify it as an XR station (for now)
     */
    if (IS_CHAN_XR(pConfig->pChannel->channelFlags)) {
        uiPrintf("%s: XXXXXXXXRRRRRRRR HACK ALERT\n", __FUNCTION__);
        pConfig->scanType = PASSIVE_SCAN;
        pConfig->macPermAddr.st.half = XR_HACKERY_MACADDR;
    }
#endif

    /*
     * Set swRetry state before calling HAL. The HAL uses the swretry state
     * to set the number of transmit retries.
     */
    pConfig->swretryEnabled = (pConfig->swRetryMaxRetries != 0);

    /* If the registry macAddr is NULL, use the EEPROM macPermAddr */
    if (A_MACADDR_COMP(&pConfig->macAddr, &nullMacAddr) == 0) {
        A_MACADDR_COPY(&pConfig->macPermAddr,
                       &pConfig->macAddr);
        A_MACADDR_COPY(&pConfig->macPermAddr,
                       &pDefConfig->macAddr);
    }

#ifdef Linux
    memcpy(MiniportAdapterHandle->dev_addr, &pConfig->macAddr.octets[0], WLAN_MAC_ADDR_SIZE);
#endif
    // Reset the hardware. Interrupts will be disabled after reset.
    Status = wlanDevInit(pDevInfo, WLAN_STA_SERVICE);
    if (Status != NDIS_STATUS_SUCCESS) {
        // Since the adapter failed the self-test, free any resources that
        // we previously allocated and error out.
        uiPrintf("< %s: wlanDevInit() failed! (0x%x)\n", __FUNCTION__, Status);
        athLogError(pDevInfo, MiniportAdapterHandle,
                    NDIS_ERROR_CODE_HARDWARE_FAILURE, 45);
        return NDIS_STATUS_FAILURE;
    }

    uiPrintf("< %s\n", __FUNCTION__);

    return Status;
}

//-----------------------------------------------------------------------------
// stationInitialize
//
// Station environment initialize
//
//-----------------------------------------------------------------------------
static A_STATUS
stationInitialize(WLAN_DEV_INFO *pdevInfo)
{
    SIB_ENTRY       *staSib;
    WLAN_STA_CONFIG *pConfig;
    BSSDESCR        *pBss;
    int             i;

    pConfig = &pdevInfo->staConfig;

    /*
     * Some fields in staConfig structure are already initialized,
     * e.g. macAddr, bssId, etc.  Rest of them are set to default
     * values here.
     */

    /* Initialize capabilities info fields to default values */
    pConfig->capInfo.cfPollable = 0;
    pConfig->capInfo.cfPollReq  = 0;

    // get short preamble setting from registry, default is 1.
    pConfig->capInfo.shortPreamble = pConfig->shortPreamble;
    if (pConfig->bssType == INDEPENDENT_BSS) {
        pConfig->capInfo.ess  = 0;
        pConfig->capInfo.ibss = 1;
    } else {
        pConfig->capInfo.ess  = 1;
        pConfig->capInfo.ibss = 0;
    }

    pConfig->capInfo.privacy = 0;

    /* Setup basic rates for starting an adhoc network. */
    if (pdevInfo->staConfig.NetBand & MODE_SELECT_EXT_RANGE) {
        wlanSetCckRateMode(pdevInfo, CCK_RATE_MODE_LONG_RANGE);
    }

#ifndef NDIS_DRIVER
    // Set fragmentation threshold. Should be done from user set defaults
#ifdef FRAGS_SUPPORTED
    pConfig->defaultFragThreshold = dot11FRAGMENTATION_THRESHOLD;
#else /* FRAGS_SUPPORTED */
    // set it to a high value so that fragmentation never occurs
    pConfig->defaultFragThreshold = MAX_WLAN_FRAME_SIZE;
#endif /* FRAGS_SUPPORTED */

#else /* NDIS_DRIVER */
    pConfig->defaultFragThreshold = dot11FRAGMENTATION_THRESHOLD;
#endif /* NDIS_DRIVER */

    pdevInfo->staConfig.maxReceiveLifeTime = dot11MaxReceiveLifeTime;

    staSib = sibEntryAlloc(pdevInfo, &pConfig->macAddr, &pdevInfo->baseBss);
    ASSERT(staSib);

    pdevInfo->localSta  = staSib;
    staSib->serviceType = WLAN_STA_SERVICE;
    staSib->staState    = 0;
    staSib->transState  = TSTATE_QUIET;
    staSib->capInfo     = pConfig->capInfo;

    A_MACADDR_COPY(&pConfig->macAddr, &staSib->macAddr);

#if !NDIS_WDM
    /* Update the abolt values based on the Supported capabilities */
    updtSupFeatureInfo(pdevInfo,0);
#endif

    pConfig->compressionSupport =
        ((pdevInfo->staConfig.abolt & ABOLT_COMPRESSION) &&
         (pdevInfo->devCap.compressSupport));

    pConfig->fastFramesSupport =
        ((pdevInfo->staConfig.abolt & ABOLT_FAST_FRAME) &&
         (pdevInfo->devCap.fastFramesSupport));

    /* Init local STA Adv Element */
    if (pdevInfo->staConfig.abolt & ABOLT_ATH_ADVCAP) {

        staSib->athAdvCapElement.elementID  = ELE_VENDOR_PRIVATE;
        staSib->athAdvCapElement.length     = sizeof(ATH_ADVCAP_IE) -
                                              (A_UINT32)&(((ATH_ADVCAP_IE *)0)->oui);
        /* Insert OUI  */
        A_BCOPY(ouiAtheros, staSib->athAdvCapElement.oui, sizeof(staSib->athAdvCapElement.oui));

        staSib->athAdvCapElement.ouiType = ATH_OUI_TYPE_CAP;
        staSib->athAdvCapElement.version = ATH_OUI_VER_CAP;

        staSib->athAdvCapElement.info.useTurboPrime =
            (pdevInfo->staConfig.abolt & ABOLT_TURBO_PRIME) ? 1 : 0;
        
        staSib->athAdvCapElement.info.useFriendlyTurbo = 
            (pdevInfo->staConfig.abolt & ABOLT_FRIENDLY_TURBO) ? 1 : 0;

        staSib->athAdvCapElement.info.useCompression =
            pConfig->compressionSupport;

        staSib->athAdvCapElement.info.useFastFrame =
            pConfig->fastFramesSupport;
    }

    /* Add STA to SIB hash table */
    if (sibEntryAdd(staSib) != WLAN_OK) {
        uiPrintf("stationInitialize: ERROR: failed to add to hash table\n");
        return A_ERROR;
    }

    /* Set rate fields according to baseband CCK or OFDM */
    wlanUpdateWirelessMode(pdevInfo, pdevInfo->staConfig.pChannel, FALSE);

    /* Build our BSS descriptor that hangs off the DEV_INFO struct */

    /* Basic fields, addresses and SSID */
    if ((pBss = (BSSDESCR *)A_DRIVER_MALLOC(sizeof(*pBss))) == NULL) {
        uiPrintf("stationInitialize: ERROR: failed to allocate BSSDESCR!\n");
        return A_ERROR;
    }
    A_MEM_ZERO(pBss, sizeof(*pBss));

    /* First configured SSID temporarily becomes our SSID. */
    pBss->ssid = pConfig->cfgSsids.elements[0];

    /*
     * cfPollable and cfPollReq fields are currently set to 0
     * because our AP is not a point coordinator.
     */

    pBss->pChannel       = pConfig->pChannel;
    pBss->capabilityInfo = pConfig->capInfo;
    pBss->bsstype        = pConfig->bssType;
    pBss->beaconInterval = pConfig->beaconInterval;
    pBss->DTIMPeriod     = DEFAULT_DTIM_PERIOD;

    pBss->ibssParamSet.elementID  = ELE_IBSS_PARAM_SET;
    pBss->ibssParamSet.length     = ELE_IBSS_PARAM_SIZE;
    pBss->ibssParamSet.atimWindow = pConfig->atimWindow;

    pdevInfo->bssDescr = pBss;

    staSib->staState   = STATE_QUIET;
    staSib->transState = TSTATE_QUIET;

    return A_OK;
}

//-----------------------------------------------------------------------------
// initializeWlan
//
// DESCRIPTION:
//    WLAN initialization code common to athInitialize() and cardReinsertionNT4().
//
//  athInitialize() is our miniport entry point that handles adapter initialization.
//  cardReinsertionNT4() is called when a PC card is re-inserted on Windows NT4
//  running CardWizard. cardReinsertionNT4() needs to do initialization similar
//  to  athInitialize() except for the NDIS resource allocation (the driver
//  is not unloaded when the PC card is removed).
//
//  NOTE: do not allocate any static NDIS resources in this function
//
// RETURNS:
//    nothing
//
//-----------------------------------------------------------------------------
NDIS_STATUS
initializeWlan(WLAN_DEV_INFO *pDevInfo)
{
    A_STATUS         Status                = A_OK;
    OS_DEV_INFO      *pOsInfo              = pDevInfo->pOSHandle;
    NDIS_HANDLE      MiniportAdapterHandle = pOsInfo->NicAdapterHandle;
    WLAN_STA_CONFIG  *pConfig              = &pDevInfo->staConfig;
    WLAN_PRIV_RECORD *pKey, *pDefStaKey;
    int              i;

    ASSERT(pOsInfo);

    uiPrintf("> %s\n", __FUNCTION__);

    // Allocate connection services data structure
    pDevInfo->pCsInfo = (CSERV_INFO *)A_DRIVER_MALLOC(sizeof(CSERV_INFO));

    if (pDevInfo->pCsInfo == NULL) {
        uiPrintf("< %s: CSERV_INFO allocation failed!\n", __FUNCTION__);
        athLogError(pDevInfo, MiniportAdapterHandle,
                    NDIS_ERROR_CODE_OUT_OF_RESOURCES, 10);
        return NDIS_STATUS_RESOURCES;
    }

    A_MEM_ZERO(pDevInfo->pCsInfo, sizeof(CSERV_INFO));

    // Allocate SIB Table
    pDevInfo->pSibTable = (SIB_TABLE *)A_DRIVER_MALLOC(sizeof(SIB_TABLE));
    if (pDevInfo->pSibTable == NULL) {
        uiPrintf("< %s: SIB_TABLE allocation failed!\n", __FUNCTION__);
        athLogError(pDevInfo, MiniportAdapterHandle,
                    NDIS_ERROR_CODE_OUT_OF_RESOURCES, 10);
        return NDIS_STATUS_RESOURCES;
    }

    A_MEM_ZERO(pDevInfo->pSibTable, sizeof(SIB_TABLE));

    if (pOsInfo->cardCfgId.Length > 0) {
        /*
         *  Make sure we have a hw key seed;
         *  it's needed for decrypting the encryption keys from the
         *  registry
         */
        Status = registryDecryptInit(pDevInfo,
                                     pOsInfo->cardCfgId.Buffer,
                                     pOsInfo->cardCfgId.Length);
        if (Status != NDIS_STATUS_SUCCESS) {
            uiPrintf("< %s: registryDecrypt() failed!\n", __FUNCTION__);
            athLogError(pDevInfo, MiniportAdapterHandle, Status, 22);
            return NDIS_STATUS_RESOURCES;
        }
    }

#ifdef WLAN_CONFIG_LEAP
    /* Now get the LEAP password, if present */
    if (pConfig->leapEnabled) {
        /* paranoia...ACU should do this */
        pConfig->privacyInvoked = TRUE;

#ifndef Linux
        if (pConfig->leapInDriver == TRUE) {
            /* Decrypt the leapUserPasswd */
            leapPasswdDecrypt(pDevInfo);
        }
#endif
    }
#endif  /* WLAN_CONFIG_LEAP */

    /* Just in case we ever need CKIP */
    ckip_internal_init();

    /* Initialize CCX stuff that can change between profiles below */
    ccxInitPreferredAps(pDevInfo, MAX_PREFERRED_APS, pConfig->prefBssids);

    /*
     *  privacyInvoked might be set by zeroconfig, so don't assume
     *  it came from static keys.
     */
    if (pConfig->privacyInvoked) {
        /* Our default policy for non-dynamic keys */
        pConfig->authAllowOpen   = TRUE;
        pConfig->authAllowShared = TRUE;
        pConfig->authOpenFirst   = FALSE;

        /* Decrypt WEP keys */
        if (pOsInfo->cardCfgId.Length > 0) {
            processWEPKeys(pDevInfo);
        }

    } else {
        // WEP disabled auth: open auth enabled and tried first
        pConfig->authAllowOpen   = TRUE;
        pConfig->authAllowShared = FALSE;
        pConfig->authOpenFirst   = TRUE;
    }

    /* Added to Set the registry values for the Authentication Type */
    wlanKeyAuthTypeSet(pConfig, pDevInfo->defaultStaConfig.authType);

    if (pConfig->sleepLogEnable) {
        A_UINT32 size = pConfig->rollingAvgPeriod * sizeof(SLEEP_BIN);

        // Try to get space to store sleep data, then get a timer to track sleep performance
        pOsInfo->pSleepBins = A_DRIVER_MALLOC(size);
        if (pOsInfo->pSleepBins == NULL) {
            uiPrintf("< %s: SLEEP_BINs allocation failed!\n", __FUNCTION__);
            athLogError(pDevInfo, MiniportAdapterHandle, NDIS_ERROR_CODE_OUT_OF_RESOURCES, 57);
            return NDIS_STATUS_RESOURCES;
        }

        A_MEM_ZERO(pOsInfo->pSleepBins, size);

        // assume one second per bin
        ASSERT(pConfig->sleepSampleInterval > 0);
        pDevInfo->sleepBinMax   = 1000 / pConfig->sleepSampleInterval;
        pDevInfo->sleepBinIndex = 0;
    }

    // At this point, we will initialize the station entity. Although
    // station management is logically one level above hardware driver, we
    // don't yet have an external OID/ioctl mechanism to initialize station.
    // This initialization routine is our only chance.
    if (sibTableInit(pDevInfo->pSibTable, pDevInfo->devCap.connectionIdMax) != A_OK) {
        uiPrintf("< %s: sibTableInit() failed!\n", __FUNCTION__);
        athLogError(pDevInfo, MiniportAdapterHandle,
                    NDIS_ERROR_CODE_OUT_OF_RESOURCES, 56);
        return NDIS_STATUS_RESOURCES;
    }

    pDevInfo->devno = 0; // Device instance number, currently we support only one

    stationInitialize(pDevInfo);
    initializeSme(pDevInfo, WLAN_STA_SERVICE);

    pDevInfo->numDefragLanes = DEFRAG_ENTRIES_DEFAULT;

    // set up encryption keys, encryption types, etc.
    if ((Status = securityInitialization(pDevInfo)) != A_OK) {
        uiPrintf("< %s: securityInitialization() failed!\n", __FUNCTION__);
        athLogError(pDevInfo, MiniportAdapterHandle, Status, 61);
        return Status;
    }

    pDevInfo->pOSHandle->connectionStatus = FALSE;
    pDevInfo->pOSHandle->resourceAvail    = TRUE;
    pDevInfo->pOSHandle->wzcActive        = TRUE;

#if NDIS_WDM
    pDevInfo->transmitBusy = FALSE;
#else
    // Reset transmit hang check
    HACK_TO_PARDEV(pDevInfo)->TransmitBusy = FALSE;
#endif

    // start up the power management
    if ((Status = powerInitialization(pDevInfo)) != A_OK) {
        uiPrintf("< %s: powerInitialization() failed!\n", __FUNCTION__);
        athLogError(pDevInfo, MiniportAdapterHandle, Status, 63);
        return NDIS_STATUS_FAILURE;
    }

    if ((Status = powerSet(pDevInfo, WAKE_UP, SET_PWR_TO_2, FALSE, 0)) != A_OK) {
        uiPrintf("< %s: powerSet() failed!\n", __FUNCTION__);
        athLogError(pDevInfo, MiniportAdapterHandle, Status, 64);
        return NDIS_STATUS_FAILURE;
    }

    /*
     * LW
     * No we can set queue info properly, since queue parameter change
     * only take effect during channel change, so use wlanDevReset to
     * set right queue info, it look ugly, but lower risk in this stage.
     */

    wlanDevReset(pDevInfo, WLAN_STA_SERVICE, pDevInfo->staConfig.pChannel, TRUE, FALSE);

    // Start the receive unit
    StartReceiveUnit(pDevInfo, (RX_UCAST  | RX_MCAST | RX_BCAST |
                                RX_BEACON | RX_PROBE_REQ));

#if !NDIS_WDM
    // Enable board interrupts -- we can now receive frames
    halEnableInterrupts(HACK_TO_PARDEV(pDevInfo), HAL_INT_GLOBAL);
#endif

    // initialize various high level station services
    cservInit(pDevInfo);

    startNdisTimers(pDevInfo);

    //
    // Disable radio if the hardware switch or software setting is set
    // to disabled, or both.  Initially assume enabled.
    //
    pDevInfo->rfSilent.radioEnabled = TRUE;
#if !NDIS_WDM
    gpioFunc0(pDevInfo, TRUE);
    gpioFunc1(pDevInfo, TRUE);
    if (pDevInfo->rfSilent.eepEnabled &&
       (pDevInfo->rfSilent.polarity ==
        halGpioGet(pDevInfo, pDevInfo->rfSilent.gpioSelect)))
    {
        athDrvRadioDisable(pDevInfo, &pDevInfo->rfSilent.hwRadioDisable);
    }
#endif

    if (!pConfig->radioEnable || pConfig->bInvalidNetBand) {
        athDrvRadioDisable(pDevInfo, &pDevInfo->rfSilent.swRadioDisable);
    }

    // May go to sleep at this point
    powerSet(pDevInfo, WAKE_UP, REDUCE_PWR_FROM_2, FALSE, 0);

    // Allow calls to our driver entry points
    pOsInfo->openForBusiness = TRUE;

#ifdef Linux
    athusb_set_multicast (MiniportAdapterHandle);
    netif_start_queue (MiniportAdapterHandle);
#endif

    uiPrintf("< %s\n", __FUNCTION__);

    return NDIS_STATUS_SUCCESS;
}

//*********************************************************************
// parseMapFile
//
// Parses a memory region containing paired strings of hex digits.  The second value is placed into
// the array element's second field if the first value matches an array element's first field
//
// Assumes format of "0x01234567" for each, and that there's at least one white space char at the end
// of the file that we can mark/use as such.
//
// This function is NDIS specific as it assumes the entire file buffer
// is passed in at a single time.
//
// Parameters:
//  Pointer to memory region
//  Length in bytes of memory region
//  Register structure in which to put data
// Returns:
//  Sets number of valid register pairs found, or number of default pairs, whichever is least, into Adapter
//
static A_BOOL
parseMapFile(A_UCHAR *filePtr, A_UINT32 fileLength, REGISTER_VAL *addRegs)
{
    A_UINT32     offset, value, i;
    A_UINT32     regCount = 0;
    A_CHAR       *pScan, *pBuf;
    A_BOOL       status;
    A_CHAR       buf[80];

    pScan = filePtr;

    while (*pScan != '\0') {
        /* Register file full check */
        if (regCount > MAX_REG_ADD_COUNT - 1) {
            break;
        }

        /* Copy next line into buffer and terminate */
        pBuf = buf;
        i = 0;
        while ((*pScan != '\0') && (*pScan != '\r') &&
            (*pScan != '\n') && (i < 79))
        {
            pBuf[i] = *pScan++;
            i++;
        }
        while ((*pScan == '\r') || (*pScan == '\n')) {
            /* Move pScan to the next line in this fileBuffer */
            pScan++;
        }
        pBuf[i] = '\0';

        /* Parse the newly string terminated buffer */
        if (parseRegLine(buf, &offset, &value) == TRUE) {
            addRegs[regCount].Offset = offset;
            addRegs[regCount].Value  = value;
            regCount++;
        }
    }

    return regCount ? TRUE : FALSE;
}

//-----------------------------------------------------------------------------
// initializeRegAdd
//
// DESCRIPTION:
//    If an additive register file exists, a set of register entries is allocated
//    and filled from the register file.
//
// RETURNS:
//    status of allocation success
//
//-----------------------------------------------------------------------------
static NDIS_STATUS
initializeRegAdd(WLAN_DEV_INFO *pDevInfo, NDIS_HANDLE MiniportAdapterHandle)
{
    return NDIS_STATUS_SUCCESS;
}

#if defined(DEBUG) && defined(PCI_INTERFACE)
//-----------------------------------------------------------------------------
// initializeEarDebug
//
// DESCRIPTION:
//    Allows pushing a binary file to the EAR parser to test EAR functionality
//    without rewriting EEPROM.  An ear.bin will override any EAR in EEPROM.
//
// RETURNS:
//    status of allocation success
//
//-----------------------------------------------------------------------------
static NDIS_STATUS
initializeEarDebug(WLAN_DEV_INFO *pDevInfo, NDIS_HANDLE MiniportAdapterHandle)
{
    NDIS_STATUS             Status = NDIS_STATUS_SUCCESS;
    NDIS_PHYSICAL_ADDRESS   earBinPhysAddr;
    NDIS_HANDLE             earBinHandle;
    A_UINT                  earBinLength = 0;
    A_UCHAR                 *MappedFile;
    NDIS_STRING             NdisEarBinName = NDIS_STRING_CONST("ear.bin");

    earBinPhysAddr.LowPart = earBinPhysAddr.HighPart = -1;

    NdisOpenFile(&Status, &earBinHandle, &earBinLength, &NdisEarBinName,
                 earBinPhysAddr);

    if (Status == NDIS_STATUS_SUCCESS) {
        pDevInfo->pEarDebug = (A_UINT16 *) A_DRIVER_MALLOC(earBinLength);
        if (pDevInfo->pEarDebug == NULL) {
            uiPrintf("Ear Debug Array allocate memory failed \n");
            NdisWriteErrorLogEntry(MiniportAdapterHandle,
                                   NDIS_ERROR_CODE_OUT_OF_RESOURCES, 4,
                                   MiniportAdapterHandle, 10, NULL, NULL);
            NdisCloseFile(earBinHandle);
            return NDIS_STATUS_RESOURCES;
        }
        A_MEM_ZERO(pDevInfo->pEarDebug, earBinLength);

        NdisMapFile(&Status, &MappedFile, earBinHandle);
        if (Status == NDIS_STATUS_SUCCESS) {
            A_DRIVER_BCOPY(MappedFile, (A_UCHAR *)pDevInfo->pEarDebug, earBinLength);
            pDevInfo->earDebugLength = earBinLength;
        }
        NdisCloseFile(earBinHandle);
    }

    return NDIS_STATUS_SUCCESS;
}
#endif // DEBUG && PCI_INTERFACE



//-----------------------------------------------------------------------------
// Procedure:   athHalt
//
// Description: Removes an adapter instance that was previously initialized.
//              To "halt" or "remove" an adapter, we disable its interrupt,
//              sbort its receive unit (otherwise it would continue to DMA in
//              data), and release all of the resources (memory, i/o space,
//              etc.) that the adapter instance was using.
//              This routine is only called when the adapter is "stopped"
//              or unloaded with a "net stop ar5210b". To see what is called
//              at machine shutdown see athShutdownHandler.
//-----------------------------------------------------------------------------
ENTRY_FN_VOID(athHalt, (NDIS_HANDLE context), (context), ICEPT_HALT)
{
    WLAN_DEV_INFO *pDevInfo = DEVINFO_FROM_CONTEXT(context);
    OS_DEV_INFO   *pOsInfo  = pDevInfo->pOSHandle;

    ASSERT(pDevInfo && pOsInfo);

    uiPrintf("> %s\n", __FUNCTION__);

    pDevInfo->NdisHaltInProgress = TRUE;

    SYSTEMSTATE_SET(pDevInfo, ATHUSB_SYSTEM_HALT);

    athShutdownDriver(pDevInfo, SHUTDOWN_EVENT);

    /*
     * Delete this instance from the array of multiple instances.
     * WinME doesn't unload driver, it may call athHalt() followed by athInitialize()
     * again to reenable the same device.
     */
    gDrvInfo.nWorkingDev -= (gDrvInfo.nWorkingDev > 0) ? 1 : 0;

    uiPrintf("< %s: bye bye!\n", __FUNCTION__);
}


//-----------------------------------------------------------------------------
// athResetComplete
//
// PARAMETERS: NDIS_HANDLE MiniportAdapterContext
//
// DESCRIPTION: This function is called by a timer indicating our
//                reset is done (by way of .5 seconds expiring)
//
// RETURNS: nothing, but sets NdisMResetComplete, enables ints
//            and starts the receive unit
//-----------------------------------------------------------------------------
ENTRY_FN_VOID(athResetComplete,(void *pContext), (pContext), ICEPT_RESETDONE)
{
    WLAN_DEV_INFO *pDevInfo = DEV_INFO_FROM_CONTEXT(pContext);
    OS_DEV_INFO   *pOsInfo  = pDevInfo->pOSHandle;

    ASSERT(pDevInfo);

    uiPrintf("> %s\n", __FUNCTION__);

    // Probably a rare case, but cover our bases anyway...
    if (!pOsInfo->openForBusiness) {
        NdisMResetComplete(pOsInfo->NicAdapterHandle, NDIS_STATUS_SUCCESS, FALSE);

        ATH_ACQUIRE_SPINLOCK_IRQ(pDevInfo->lock);
        osIndicateResourcesAvailable(pDevInfo);
        osIndicateConnectStatus(pDevInfo, FALSE);
        uiPrintf("< %s: ZOMBIE MODE -- can't touch HW! "
                 "-- reset completed\n", __FUNCTION__);
        ATH_RELEASE_SPINLOCK_IRQ(pDevInfo->lock);
        return;
    }

    if (pDevInfo->busStatus != ATHUSB_BUS_STATE_NORMAL) {
        // USB bus still hang
        if (pOsInfo->resetCheckCounter) {
            pOsInfo->resetCheckCounter--;
            // set timer again to call us back later
            A_TIMEOUT(&pOsInfo->NicAsyncResetTimer, CONNECT_CHECK_TIME, (A_UINT32)pDevInfo, 0);
            uiPrintf("< %s: waiting for bus reset\n", __FUNCTION__);
            return;
        }
    }

    // if we were connected prior to the reset, and we're not either re-joined
    // or re-associated yet, wait awhile longer
    if (pOsInfo->resetConnectionStatus) {
        if ((((pDevInfo->bssDescr->bsstype == INDEPENDENT_BSS) &&
             !(pDevInfo->localSta->staState & STATE_JOINED))) ||
             ((pDevInfo->bssDescr->bsstype == INFRASTRUCTURE_BSS) &&
             ((pDevInfo->localSta->staState & (STATE_AUTH | STATE_ASSOC)) !=
               (STATE_AUTH | STATE_ASSOC))) )
        {
            if (pOsInfo->resetCheckCounter) {
                pOsInfo->resetCheckCounter--;
                // set timer again to call us back later
                A_TIMEOUT(&pOsInfo->NicAsyncResetTimer, CONNECT_CHECK_TIME, (A_UINT32)pDevInfo, 0);
                uiPrintf("< %s: waiting for reconnection\n", __FUNCTION__);
                return;
            }
        }
    }

    NdisMResetComplete(pOsInfo->NicAdapterHandle, NDIS_STATUS_SUCCESS, TRUE);

    ATH_ACQUIRE_SPINLOCK_IRQ(pDevInfo->lock);

    pDevInfo->NdisResetInProgress = FALSE;
    pDevInfo->localSta->stats.osResetCount++;
    powerSet(pDevInfo, WAKE_UP, REDUCE_PWR_FROM_3, FALSE, 0);

    osIndicateResourcesAvailable(pDevInfo);

    // if we lost connection during the reset and couldn't regain it within CONNECTION_RESTORE_TIME, tell the OS...
    // it's gone
    if (pOsInfo->resetCheckCounter == 0) {
        if (pDevInfo->busStatus != ATHUSB_BUS_STATE_NORMAL &&
            pOsInfo->resetConnectionStatus) {
            cservStopBss(pDevInfo, DISCONNECT_NOW, REMOVE_BSS, DONT_SEND_DEAUTH);
        } else {
            osIndicateConnectStatus(pDevInfo, FALSE);
        }
    }

    ATH_RELEASE_SPINLOCK_IRQ(pDevInfo->lock);
    uiPrintf("< %s: reset completed\n", __FUNCTION__);
}

#ifdef NDIS51_MINIPORT // Extensions for NDIS 5.1

//-----------------------------------------------------------------------------
// athPnPEventNotifyHandler
//
// PARAMETERS: NDIS_HANDLE MiniportAdapterContext
//
// DESCRIPTION: This function handles AC/battery and surprise removal notifications
//
// RETURNS: nothing
//-----------------------------------------------------------------------------
ENTRY_FN_VOID(athPnPEventNotifyHandler,
              (NDIS_HANDLE           MiniportAdapterContext,
               NDIS_DEVICE_PNP_EVENT PnPEvent,
               PVOID                 InformationBuffer,
               ULONG                 InformationBufferLength),
              (MiniportAdapterContext, PnPEvent,
               InformationBuffer, InformationBufferLength), ICEPT_PNP)
{
    WLAN_DEV_INFO       *pDev             = DEVINFO_FROM_CONTEXT(MiniportAdapterContext);
    PNDIS_POWER_PROFILE pNdisPowerProfile = (PNDIS_POWER_PROFILE)InformationBuffer;

    uiPrintf("> %s\n", __FUNCTION__);

    if (!pDev->pOSHandle->openForBusiness) {
        uiPrintf("< %s: ZOMBIE MODE -- can't touch HW\n", __FUNCTION__);
        return;
    }

    if (PnPEvent == NdisDevicePnPEventPowerProfileChanged) {
        if (pDev->staConfig.overrideACstatus) {
            uiPrintf("< %s: ignoring (overrideACstatus == TRUE)\n", __FUNCTION__);
            return;
        }
        if (*pNdisPowerProfile == NdisPowerProfileAcOnLine) {
            if (wlanDevPresent(pDev)) {
                ATH_ACQUIRE_SPINLOCK(pDev->lock);
                powerSet(pDev, AC_ONLINE, ENABLE, FALSE, 0);
                ATH_RELEASE_SPINLOCK(pDev->lock);
            } else {
                uiPrintf("Ignore change since device isn't ready\n");
            }
            uiPrintf("< %s: now plugged in\n", __FUNCTION__);
        } else {
            if (wlanDevPresent(pDev)) {
                ATH_ACQUIRE_SPINLOCK(pDev->lock);
                powerSet(pDev, AC_ONLINE, DISABLE, TRUE, 0);
                ATH_RELEASE_SPINLOCK(pDev->lock);
            } else {
                uiPrintf("Ignore change since device isn't ready\n");
            }
            uiPrintf("< %s: now unplugged\n", __FUNCTION__);
        }
    } else if (PnPEvent == NdisDevicePnPEventSurpriseRemoved) {
        uiPrintf("Device surprise remove\n");

        /*
         * After return from this function, we can't send anything to bus driver
         */
        pDev->busStatus = ATHUSB_BUS_STATE_SURPRISE_REMOVED;
        wdcSurpriseRemoved(pDev->targetHandle);
        if (pDev->bWorkItemInProgress == TRUE) {
            /*
             * If resume work item already fire, let it finished
             */
            NdisWaitEvent(&pDev->noPendingPMEvent, 0);
        }
        uiPrintf("< %s: SurpriseRemoved\n", __FUNCTION__);
    }
}

#endif // NDIS51_MINIPORT


//-----------------------------------------------------------------------------
// athSleepTrack
//
// PARAMETERS: NDIS_HANDLE MiniportAdapterContext
//
// DESCRIPTION: This function is called by a timer to record the chip sleep status
//              for measuring the percentage of time actually spent asleep.
//
// RETURNS: nothing
//
//-----------------------------------------------------------------------------
ENTRY_FN_VOID(athSleepTrack,(void *pContext), (pContext), ICEPT_SLEEPTRACK)
{
    WLAN_DEV_INFO *pDevInfo    = DEV_INFO_FROM_CONTEXT(pContext);
    SLEEP_BIN     *pSleepBins;
    
    ASSERT(pDevInfo);

    if (!pDevInfo->pOSHandle->openForBusiness) {
        return;
    }

    pSleepBins = pDevInfo->pOSHandle->pSleepBins;
    ASSERT(pSleepBins);

    if (halGetPowerStatus(HACK_TO_PARDEV(pDevInfo))) {
        pSleepBins[pDevInfo->sleepBinIndex].SleepCount++;
    }

    // Put sleep counts in a per second bin.  If bin is full (1 second has elapsed), go
    // to next bin, wrapping if needed.  Empty it before using it.
    if (pSleepBins[pDevInfo->sleepBinIndex].TotalCount++ >= pDevInfo->sleepBinMax) {
        pDevInfo->sleepBinIndex++;
        if (pDevInfo->sleepBinIndex >= pDevInfo->staConfig.rollingAvgPeriod) {
            pDevInfo->sleepBinIndex = 0;
        }

        if (pSleepBins[pDevInfo->sleepBinIndex].TotalCount != 0) {
            pSleepBins[pDevInfo->sleepBinIndex].TotalCount = 0;
            pSleepBins[pDevInfo->sleepBinIndex].SleepCount = 0;
        }
    }

}

//-----------------------------------------------------------------------------
// athPnPEventComplete
//
// PARAMETERS: NDIS_HANDLE MiniportAdapterContext
//
// DESCRIPTION: This function is called by a timer when entering standby or
//              hibernate.  If any other timer functions had fired before they
//              were cancelled at that time, this will be the last one run, so
//              now we can indicate to NDIS that the request has been completed.
//
// RETURNS: nothing
//
//-----------------------------------------------------------------------------
ENTRY_FN_VOID(athPnPEventComplete,(void *pContext), (pContext), ICEPT_HIBERNATE)
{
    WLAN_DEV_INFO *pDevInfo = DEV_INFO_FROM_CONTEXT(pContext);
    
    ASSERT(pDevInfo);

    NdisMSetInformationComplete(pDevInfo->pOSHandle->NicAdapterHandle, NDIS_STATUS_SUCCESS);
    NdisSetEvent(&pDevInfo->noPendingPMEvent);
}

//-----------------------------------------------------------------------------
// Procedure:   ResetOnError
//
// Description: Instructs the Miniport to issue a hardware reset to the
//              network adapter. The driver also resets its software state.
//              This function also optionally drains the transmit queues.
//
// Returns:     A_OK or A_ERROR
//-----------------------------------------------------------------------------
A_STATUS
ResetOnError(WLAN_DEV_INFO *pDevInfo, CLEAR_TX_Q_ENUM clearTxQueue,
             RESET_CTR_ENUM ctrEnum, DRAIN_ENUM waitForDrain)
{
    CSERV_INFO      *pCsInfo  = pDevInfo->pCsInfo;
    WLAN_STA_CONFIG *pCfg     = &pDevInfo->staConfig;
    A_BOOL          skipStats = ctrEnum == RESET_DONT_INC_CTR;

    ASSERT(pDevInfo->pCsInfo);

    uiPrintf("> %s: clearTxQueue = %d, waitForDrain = %d\n", __FUNCTION__,
             clearTxQueue, waitForDrain);

    if (pDevInfo->NicResetInProgress == TRUE) {
        uiPrintf("< %s: reset in progress\n", __FUNCTION__);
        return A_OK;
    }

    pDevInfo->NicResetInProgress = TRUE;

    if (!skipStats) {
        pDevInfo->localSta->stats.hwResetCount++;
        pDevInfo->hwResetCount++;
    }

    cservPollReset(pDevInfo); // reset poll counters so that they won't interfere

    cservOperationCancel(pDevInfo);

    // Must be awake to access hardware
    powerSet(pDevInfo, WAKE_UP, SET_PWR_TO_3, FALSE, 0);

#if 0
    /* LW
     * target will handle receive and transmit stop
     */

    // Disable interrupts while we re-init the transmit structures
    halDisableInterrupts(HACK_TO_PARDEV(pDevInfo), HAL_INT_GLOBAL);

    StopSwReceive(pDevInfo);
    StopTransmit(pDevInfo, clearTxQueue, DO_COMPLETE, waitForDrain);
    StopReceive(pDevInfo);
#endif
    // Free our beacon resources here.  If we're currently in an IBSS, we'll rejoin
    // after our reset.
    wlanBeaconRelease(pDevInfo);

    /*
     * LW
     * This is temp solutio to let target happy
     */
    //if (waitForDrain)
    {
        NdisInterlockedIncrement(&pDevInfo->numFlushCall);
        wdcFlush(pDevInfo->targetHandle,pDevInfo->disableEnableCounter);
    }

    if (wlanDevReset(pDevInfo, WLAN_STA_SERVICE, pCfg->pChannel, FALSE, pDevInfo->turboPrimeInfo.turboPrimeAllowed) != A_OK) {
        pDevInfo->NicResetInProgress = FALSE;
        powerSet(pDevInfo, WAKE_UP, REDUCE_PWR_FROM_3, FALSE, 0);
        uiPrintf("< %s: wlanDevReset() failed!\n", __FUNCTION__);
        return A_ERROR;
    }

    NdisInterlockedIncrement(&pDevInfo->numFlushCall);
    wdcFlush(pDevInfo->targetHandle,pDevInfo->disableEnableCounter);

    powerHwResetNotify(pDevInfo);

    pDevInfo->globISRReg = 0; // clear the stored value of interrupt bits

    /* Allow NDIS Send if now on configured channel */
#if NDIS_WDM
    /*
     * (LW) Attention !!!
     * This is temp hack, should revisit later
     */
    if (pDevInfo->staConfig.phwChannel == pDevInfo->staConfig.pChannel) {
        pDevInfo->invalidTxChannel = FALSE;
    }
#else
    if (pDevInfo->staConfig.phwChannel == pDevInfo->staConfig.pChannel) {
        HACK_TO_PARDEV(pDevInfo)->invalidTxChannel = FALSE;
    }
#endif

    // restore h/w encrypt engine setting
    setupSwEncryption(pDevInfo);

    if (athDrvRadioStatus(pDevInfo, FALSE)) {
#if 0
        /* LW
         * target will handle target start over again
         */

        /* Radio was enabled prior to reset */
        /* Enable intr before restarting tx and rx */
        halEnableInterrupts(HACK_TO_PARDEV(pDevInfo), HAL_INT_GLOBAL);

        RestartTransmit(pDevInfo);
        StartReceiveUnit(pDevInfo, 0);
#endif
        pDevInfo->NicResetInProgress = FALSE;

        // clear the resources condition caused by power save or bkgnd scanning
        osIndicateResourcesAvailable(pDevInfo);

        // Resume power setting at previous level
        powerSet(pDevInfo, WAKE_UP, REDUCE_PWR_FROM_3, FALSE, 0);
    } else {
        /* LW
         * need design for radio disable case
         */
        halPhyDisable(HACK_TO_PARDEV(pDevInfo));

        // Re-enable interrupts to catch switch change on the GPIO
        halEnableInterrupts(HACK_TO_PARDEV(pDevInfo), HAL_INT_GLOBAL);

        // Restore power state, then attempt to silence the radio and enter
        // full sleep, ok if unsuccessful
        powerSet(pDevInfo, WAKE_UP, REDUCE_PWR_FROM_3, FALSE, 0);
        powerSet(pDevInfo, RADIO_ONLINE, DISABLE, FALSE, 0);
    }

    uiPrintf("< %s\n", __FUNCTION__);

    return A_OK;
}

//-----------------------------------------------------------------------------
// Procedure:   athShutdownDriver
//
// Description: This routine handles the shutdown of the driver.
//
// Arguments:
//
// pDevInfo - Device context
// event    - passes the reason for shutting down the driver.
//
// Returns:
//-----------------------------------------------------------------------------
void
athShutdownDriver(IN WLAN_DEV_INFO *pDevInfo, IN SHUTDOWN_TYPE event)
{
    A_STATUS      status;

    uiPrintf("> %s: ", __FUNCTION__);
#ifdef Linux
        netif_stop_queue(pDevInfo->pOSHandle->NicAdapterHandle);
#endif

    switch (event) {
    case SHUTDOWN_EVENT:
        uiPrintf("shutdown event\n");

        if (pDevInfo->bWorkItemInProgress == TRUE) {
            NdisWaitEvent(&pDevInfo->noPendingPMEvent, 0);
        }

        pDevInfo->NicResetInProgress = TRUE;
        stopNdisTimers(pDevInfo, FALSE);

        /* Set the device to Sleep now */
        //wdcTargetSetPowerMode(pDevInfo->targetHandle, TARGET_DEVICE_PWRDN, &status);

        /* 
         * We need notify AP we will go away, otherwise AP may
         * still send traffic to us and we may take long time
         * to associate back to AP if we come back before AP age us out
         */
        if (wlanDevPresent(pDevInfo) && (IS_STATE_CONNECTED(pDevInfo))) {
            ATH_ACQUIRE_SPINLOCK_IRQ(pDevInfo->lock);
            cservStopBss(pDevInfo, DISCONNECT_NOW, REMOVE_BSS, SEND_DEAUTH);
            ATH_RELEASE_SPINLOCK_IRQ(pDevInfo->lock);
        
            /*
             * Sleep 50 millisecond to let deauthentication frame go out
             * before we go away
             */
            NdisMSleep(50000);
        }
       

        break;

    case CONFIG_EVENT:
        uiPrintf("config event\n");

        /*
         * TODO: Still needs some work, need to review resetOnError
         * and when in reset already case.
         */
        stopNdisTimers(pDevInfo, FALSE);
        ATH_ACQUIRE_SPINLOCK(pDevInfo->lock);
        cservStopBss(pDevInfo, DISCONNECT_NOW, REMOVE_BSS, SEND_DEAUTH);
        ATH_RELEASE_SPINLOCK(pDevInfo->lock);
        break;

    case HIBERNATE_EVENT:
        uiPrintf("hibernate event\n");
        pDevInfo->NicResetInProgress = TRUE;
        stopNdisTimers(pDevInfo, FALSE);
        break;

    default:
        uiPrintf("unknown shutdown event!!\n");
        ASSERT(0);
    }

    /* During system shutdown we wouldnt have any other bottom halves running
     * The system hangs here with the error messages.  Anyway we are shutting
     * down.  No harm in losing a couple of packets
     */

    switch (event) {
    case CONFIG_EVENT:
        /* Free target side connections.
         * Note: this cannot be done in freeWlanResources because
         * freeWlanResources can be called after the WDC interface
         * is torn down (i.e. on MiniportHalt).
         */
        ATH_ACQUIRE_SPINLOCK_IRQ(pDevInfo->sendLock);
        ATH_ACQUIRE_SPINLOCK(pDevInfo->lock);
        sibFreeTargetConn(pDevInfo, pDevInfo->pSibTable);

        /* Free WLAN resources. Also cancels all NDIS timers. */        
        freeWlanResources(pDevInfo, FALSE);
        ATH_RELEASE_SPINLOCK(pDevInfo->lock);
        ATH_RELEASE_SPINLOCK_IRQ(pDevInfo->sendLock);
        break;

    case SHUTDOWN_EVENT:
        freeResources(pDevInfo);
        break;

    case HIBERNATE_EVENT:
        break;
    }

    uiPrintf("< %s: bye bye!\n", __FUNCTION__);
}

//-----------------------------------------------------------------------------
// Procedure:   athRestartDriver
//
// Description: Restart driver (Initialize HAL and WLAN layer)
//
//              Used in conjunction with athShutdownDriver for
//              dynamic configuration.
//
// Arguments:
//
// Returns:
//-----------------------------------------------------------------------------
NDIS_STATUS
athRestartDriver(IN WLAN_DEV_INFO *pDevInfo)
{
    NDIS_STATUS Status;

    uiPrintf("> %s\n", __FUNCTION__);

    // Re-initialize the HAL
    Status = initializeHal(pDevInfo);
    if (Status != NDIS_STATUS_SUCCESS) {
        uiPrintf("< %s: initializeHal() failed!\n", __FUNCTION__);
        return Status;
    }

    // Re-initialize the rest of the driver
    Status = initializeWlan(pDevInfo);
    if (Status == NDIS_STATUS_SUCCESS) {
        return Status;
    }

    uiPrintf("< %s: initializeWlan() failed!\n", __FUNCTION__);

    // Free WLAN resources. This makes sure that the timers are cancelled.
    freeWlanResources(pDevInfo, TRUE);

    uiPrintf("< %s\n", __FUNCTION__);

    return Status;
}

NDIS_STATUS
initNdisTimers(WLAN_DEV_INFO *pDevInfo)
{
    OS_DEV_INFO     *pOsInfo = pDevInfo->pOSHandle;
    WLAN_STA_CONFIG *pConfig = &pDevInfo->staConfig;

    uiPrintf("> %s\n", __FUNCTION__);

    SYSTEMSTATE_SET(pDevInfo, ATHUSB_SYSTEM_TIMER_INIT);

    // Init one shot timers for reset, hibernate, leap and sme
    // Init periodic timers for calibration, sleep logging and STA poll
    A_INIT_TIMER(pDevInfo, &pOsInfo->NicAsyncResetTimer, athResetComplete, FALSE);
    A_INIT_TIMER(pDevInfo, &pOsInfo->NicPnPEventTimer, athPnPEventComplete, FALSE);
#if defined(WLAN_CONFIG_LEAP) && defined(LEAP_IN_DRIVER)
    A_INIT_TIMER(pDevInfo, &pDevInfo->leapTimer, leapTimerSvc, FALSE);
#endif /* WLAN_CONFIG_LEAP */
    if (pConfig->sleepLogEnable) {
        A_INIT_TIMER(pDevInfo, &pOsInfo->NicSleepLogTimer, athSleepTrack, TRUE);
    }
    A_INIT_TIMER(pDevInfo, &pDevInfo->smeTimer, mlmePollHandler, FALSE);
    A_INIT_TIMER(pDevInfo, &pDevInfo->staPollTimer, staPollHandler, TRUE);    

    uiPrintf("< %s\n", __FUNCTION__);

    return NDIS_STATUS_SUCCESS;
}

void
stopNdisTimers(WLAN_DEV_INFO *pDevInfo, A_BOOL delete)
{
    OS_DEV_INFO     *pOsInfo = pDevInfo->pOSHandle;
    WLAN_STA_CONFIG *pConfig = &pDevInfo->staConfig;
    A_UINT32         count = 0;

    ASSERT(pOsInfo);

    /*
     * Note, can not wait for all timers to
     * be dequeued here since there are
     * functions calls to here that run
     * at higher than IRQL_PASSIVE_LEVEL.
     */
    SYSTEMSTATE_SET(pDevInfo, ATHUSB_SYSTEM_TIMER_STOP);

    uiPrintf("> %sdelete = %d\n", __FUNCTION__, delete);

    // NOTE WELL: A_DELETE_TIMER() will protect against us passing in NULL
    //            pointers
    if (delete) {

        SYSTEMSTATE_CLEAR(pDevInfo, ATHUSB_SYSTEM_TIMER_INIT);

        // Delete all timers
#if defined(WLAN_CONFIG_LEAP) && defined(LEAP_IN_DRIVER)
        A_DELETE_TIMER(&pDevInfo->leapTimer);
#endif
        A_DELETE_TIMER(&pDevInfo->smeTimer);
        A_DELETE_TIMER(&pDevInfo->staPollTimer);

        A_DELETE_TIMER(&pOsInfo->NicAsyncResetTimer);
        A_DELETE_TIMER(&pOsInfo->NicPnPEventTimer);

        if (pConfig->sleepLogEnable) {
            A_DELETE_TIMER(&pOsInfo->NicSleepLogTimer);
        }
    } else {
        // Cancel all timers
#if defined(WLAN_CONFIG_LEAP) && defined(LEAP_IN_DRIVER)
        A_UNTIMEOUT(&pDevInfo->leapTimer);
#endif
        A_UNTIMEOUT(&pDevInfo->smeTimer);
        A_UNTIMEOUT(&pDevInfo->staPollTimer);

        A_UNTIMEOUT(&pOsInfo->NicAsyncResetTimer);
        A_UNTIMEOUT(&pOsInfo->NicPnPEventTimer);

        if (pConfig->sleepLogEnable) {
            A_UNTIMEOUT(&pOsInfo->NicSleepLogTimer);
        }

        /*
         * Only enabled the workaround for HT for XP
         * TODO May need to add to both W2K and .NET
         */
#ifdef NDIS51_MINIPORT
        if ((pDevInfo->pOSHandle->ndisDriverMajorVersion == 5) &&
            (pDevInfo->pOSHandle->ndisDriverMinorVersion == 1))
        {
            while ((TimerCount(pDevInfo)) && (count++ < 6)) {
                NdisMSleep(20000);
            }
        }
#endif
    }

    uiPrintf("< %s\n", __FUNCTION__);
}

// start periodic timers
void
startNdisTimers(WLAN_DEV_INFO *pDevInfo)
{
    OS_DEV_INFO     *pOsInfo = pDevInfo->pOSHandle;
    WLAN_STA_CONFIG *pConfig = &pDevInfo->staConfig;

    ASSERT(pOsInfo);

    uiPrintf("> %s\n", __FUNCTION__);

    if (pConfig->sleepLogEnable) {
        // set a timer for sleep tracking
        A_TIMEOUT(&pOsInfo->NicSleepLogTimer, pConfig->sleepSampleInterval,
                  (A_UINT32)pDevInfo, 0);
    }

    A_TIMEOUT(&pDevInfo->staPollTimer, CSERV_TIME_STAPOLL, (A_UINT32) pDevInfo, 0);

    uiPrintf("< %s\n", __FUNCTION__);
}

//-----------------------------------------------------------------------------
// Procedure:   freeResources
//
// Description: Deregisters handlers, interrupts, detach from HAL, and frees all memory.
//
//
// Returns:
//-----------------------------------------------------------------------------
void
freeResources(WLAN_DEV_INFO *pDevInfo)
{
    OS_DEV_INFO             *pOsInfo = pDevInfo->pOSHandle;

    uiPrintf("> %s\n", __FUNCTION__);


    // Free any WDC allocated resources.
    //
    // NOTE: Do not call any WDC functions after calling wdcUnbind().
    //
    if (pDevInfo->targetHandle) {
        wdcTargetStop(pDevInfo->targetHandle, NULL);
        wdcUnbind(pDevInfo, pDevInfo->targetHandle);
        pDevInfo->targetHandle = NULL; /* sanity */
    }

    // Free Wlan/HAL related resources. Also will shutdown NDIS timers.
    freeWlanResources(pDevInfo, TRUE);

    // Free static NDIS related resources.
    freeStaticNdisResources(pDevInfo);

    // Free CCX stuff
    ccxFree(pDevInfo);

    // Free additional info structures we created in MiniportInitialize
    if (pDevInfo->pInitRegs) {
        A_DRIVER_FREE(pDevInfo->pInitRegs, sizeof(REGISTER_VAL) * MAX_REG_ADD_COUNT);
        pDevInfo->pInitRegs = NULL;
    }

    if (pOsInfo) {
        A_DRIVER_FREE(pOsInfo, sizeof(OS_DEV_INFO));
        pOsInfo = NULL;
    }

    ATH_FREE_SPINLOCK(pDevInfo->lock);
    ATH_FREE_SPINLOCK(pDevInfo->sendLock);
    ATH_FREE_SPINLOCK(pDevInfo->mapLock);

    A_DRIVER_FREE(pDevInfo, sizeof(WLAN_DEV_INFO));
    pDevInfo = NULL;

#ifdef DEBUG_PKT_LOG
    logEventDelete();
#endif

    uiPrintf("< %s\n", __FUNCTION__);

}


//-----------------------------------------------------------------------------
// Procedure:    freeStaticNdisResources
//
// Description:  Free static NDIS resources.
//               Note that all of these resources can only be allocated at MiniportInitialization i.e.
//               they can not be re-allocated dynamically after MiniportInitialization.
//
//
// Returns:
//-----------------------------------------------------------------------------
void
freeStaticNdisResources(WLAN_DEV_INFO *pDevInfo)
{
    OS_DEV_INFO  *pOsInfo = pDevInfo->pOSHandle;
    ATHEROS_DESC *pDesc;
    DRV_BUF_DESC *pBufDesc, *tpBufDesc;

    ASSERT(pOsInfo);

    uiPrintf("> %s\n", __FUNCTION__);

    // Release NDIS' packet descriptors
    A_SEM_LOCK(pDevInfo->rxQueue.qSem, WAIT_FOREVER);
    while (pDesc = descQPopHead(&pDevInfo->rxQueue)) {
        if (pDesc->pOSHandle) {
            NdisFreePacket(pDesc->pOSHandle);
            pDesc->pOSHandle = NULL;
        }
    }
    A_SEM_UNLOCK(pDevInfo->rxQueue.qSem);
    A_SEM_DELETE((pDevInfo->rxQueue).qSem);

    // Release NDIS' pools
    if (pOsInfo->rxPacketPool) {
        NdisFreePacketPool(pOsInfo->rxPacketPool);
        pOsInfo->rxPacketPool = NULL;
    }

    if (pOsInfo->rxBufferPool) {
        NdisFreeBufferPool(pOsInfo->rxBufferPool);
        pOsInfo->rxBufferPool = NULL;
    }

    A_SEM_LOCK(pDevInfo->txPendingQueue.qSem, WAIT_FOREVER);
    while (pDesc = descQPopHead(&pDevInfo->txPendingQueue)) {
        freeBuffandDescChain(pDevInfo, pDesc);
    }
    A_SEM_UNLOCK(pDevInfo->txPendingQueue.qSem);

    A_SEM_DELETE((pDevInfo->txPendingQueue).qSem);

    // Free small buffers
    if (pDevInfo->pSmallbufMemory) {
        athNdisFreeMem(pDevInfo, (NUM_SMALL_BUFF*SMALL_BUFF_SIZE),
                       pDevInfo->smallbufPhysAddr, pDevInfo->pSmallbufMemory);
        pDevInfo->pSmallbufMemory = NULL;
    }

    // delete smallbuf queue semaphore
    A_SEM_DELETE((pDevInfo->smallBufDescQueue).qSem);

    // Free large buffers
    pBufDesc = pDevInfo->pLargebufMemory;
    while (pBufDesc) {
        tpBufDesc = pBufDesc->nextBufDesc;
        A_CORRUPTION_BUG_CHECK(pBufDesc);
        athNdisFreeMem(pDevInfo, LARGE_BUFF_SIZE, pBufDesc->thisPhysAddr, pBufDesc);
        pBufDesc = tpBufDesc;
    }

    // delete largetbuf queue semaphore
    A_SEM_DELETE((pDevInfo->largeBufDescQueue).qSem);

    pDevInfo->pLargebufMemory = NULL;
    pDevInfo->virtEncryptBuf  = NULL;
    pDevInfo->virtDecryptBuf  = NULL;

    // Free descriptors
    if (pDevInfo->pDescMemory) {
        athNdisFreeMem(pDevInfo, (NUM_DESCS * DESC_SIZE), pDevInfo->descPhysAddr,
                       pDevInfo->pDescMemory);
        pDevInfo->pDescMemory = NULL;
    }

    // delete descbuf queue semaphore
    A_SEM_DELETE((pDevInfo->emptyDescQueue).qSem);

    // Let the map registers go
    if (pOsInfo->mapRegsAvail) {
        NdisMFreeMapRegisters(pOsInfo->NicAdapterHandle);
        pDevInfo->pOSHandle->mapRegsAvail = 0;
    }

    // Unmap the NIC
    if (pDevInfo->mapped) {
        NdisMUnmapIoSpace(pOsInfo->NicAdapterHandle, (PVOID) pDevInfo->baseAddress,
                          pDevInfo->mapLength);
        pDevInfo->mapped = FALSE;
    }

    if (pOsInfo->pSleepBins) {
        A_DRIVER_FREE(pOsInfo->pSleepBins,
                      pDevInfo->staConfig.rollingAvgPeriod * sizeof(struct SleepBin));
    }
}

//-----------------------------------------------------------------------------
// Procedure:   freeWlanResources
//
// Description: WLAN free resource code shared between
//              freeResources() and cardReinsertionNT4().
//
//              cardReinsertionNT4 needs to free data structures
//              but not NDIS resources.  Also, any data structure allocated once in
//              athInitialize should not be freed here (e.g. OS_DEV_INFO, WLAN_DEV_INFO).
//-----------------------------------------------------------------------------
void
freeWlanResources(WLAN_DEV_INFO *pDevInfo, A_BOOL delete)
{
    ASSERT(pDevInfo);

    uiPrintf("> %s: delete = %d\n", __FUNCTION__, delete);

    /* Block all callers into our driver until we're back up again */
    pDevInfo->pOSHandle->openForBusiness = FALSE;

    /* Cancel pending timers */
    stopNdisTimers(pDevInfo, delete);

#if NDIS_WDM
    wlanDevRelease(pDevInfo);
#else
    // This should absorb the rest over time
    wlanDevStop(pDevInfo);
    wlanDevRelease(pDevInfo);

    // free beacon memory and return beacon desc/buf back to pool
    // before the pool is freed.
    wlanBeaconRelease(pDevInfo);

    // Free PS Poll resources
    powerPSPollFree(pDevInfo);
#endif
    // Free Channel list
    wlanFreeChannelList(pDevInfo->staConfig.pClist);
    pDevInfo->staConfig.pClist     = NULL;
    pDevInfo->staConfig.phwChannel = NULL;
    pDevInfo->staConfig.pChannel   = NULL;

    if (pDevInfo->bssDescr) {
        A_DRIVER_FREE(pDevInfo->bssDescr, sizeof(BSSDESCR));
        pDevInfo->bssDescr = NULL;
    }

    wlanKeyTableFree(pDevInfo);

    if (pDevInfo->pscanInfo) {
        SCAN_INFO *pInfo = pDevInfo->pscanInfo;

        /* Clean up all possibly nested information structs */
        while (pInfo) {
            SCAN_INFO *pTemp = pInfo->pLastInfo;

            if (pInfo->pScanList) {
                wlanFreeScanList(pInfo->pScanList);
            }

            A_DRIVER_FREE(pInfo, sizeof(*pInfo));
            pInfo = pTemp;
        }
        pDevInfo->pscanInfo = NULL;
    }

    if (pDevInfo->pCsInfo) {
        if (pDevInfo->pCsInfo->bkScanList) {
            wlanFreeScanList(pDevInfo->pCsInfo->bkScanList);
        }
        A_DRIVER_FREE(pDevInfo->pCsInfo, sizeof(CSERV_INFO));
        pDevInfo->pCsInfo = NULL;
    }

    if (pDevInfo->Ctx) {
        A_DRIVER_FREE(pDevInfo->Ctx, (sizeof(BLOWFISH_CTX)));
        pDevInfo->Ctx = NULL;
    }

    if (pDevInfo->pSibTable) {
        sibTableExit(pDevInfo->pSibTable);
        A_DRIVER_FREE(pDevInfo->pSibTable, sizeof(SIB_TABLE));
        pDevInfo->pSibTable = NULL;
        pDevInfo->localSta  = NULL;
    }

    wlanMlmeFree(pDevInfo);

    uiPrintf("< %s\n", __FUNCTION__);
}

//-----------------------------------------------------------------------------
// Procedure:   athDrvRadioEnable
//
// Description: Enables radio operation, if neither of the users of this function
//              (h/w switch and s/w application(s)) have requested disable at
//              this time.
//
// Parameters:  Address of callers radio disable flag, either of hw or sw
//
// Returns:     A_OK if radio is enabled.  Any other value indicates radio enable
//              failed and radio is not functional.
//-----------------------------------------------------------------------------
A_STATUS
athDrvRadioEnable(WLAN_DEV_INFO *pDevInfo, A_BOOL *pxwRadioDisable)
{
    A_STATUS status = A_OK;

    uiPrintf("> %s: *pxwRadioDisable = %d\n", __FUNCTION__, *pxwRadioDisable);

    if (*pxwRadioDisable == FALSE) {
        uiPrintf("< %s: nothing to do\n", __FUNCTION__);
        return A_OK;
    }

    *pxwRadioDisable = FALSE;

    // Two users (s/w & h/w) of this function - only allow enable when both have so requested.
    // Also do not allow radio enable when the NetBand is invalid.
    if (pDevInfo->rfSilent.radioEnabled ||
        pDevInfo->rfSilent.swRadioDisable ||
        pDevInfo->rfSilent.hwRadioDisable ||
        pDevInfo->staConfig.bInvalidNetBand)
    {
        uiPrintf("< %s: already enabled or myriad of other reasons to leave\n", __FUNCTION__);
        return status;
    }

    NdisInterlockedIncrement(&pDevInfo->disableEnableCounter);
    pDevInfo->flushInProgress   =   FALSE;
    A_ATOMIC_SET (&pDevInfo->numFlushCall, 0);

    wdcTargetEnable(pDevInfo->targetHandle,&status);

    ATH_ACQUIRE_SPINLOCK_IRQ(pDevInfo->lock);

    pDevInfo->NicResetInProgress    = FALSE;
    pDevInfo->rfSilent.radioEnabled = TRUE;
    status = ResetOnError(pDevInfo, DONT_CLEAR_TX_Q, RESET_DONT_INC_CTR, DO_WAIT);
    if (status != A_OK) {
        uiPrintf("< %s: ResetOnError() failed!\n", __FUNCTION__);
        ATH_RELEASE_SPINLOCK_IRQ(pDevInfo->lock);
        return status;
    }

    // Allow power to the chips
    status = powerSet(pDevInfo, RADIO_ONLINE, ENABLE, FALSE, 0);
    if (status != A_OK) {
        uiPrintf("< %s: powerSet() failed!\n", __FUNCTION__);
        ATH_RELEASE_SPINLOCK_IRQ(pDevInfo->lock);
        return status;
    }

    status = powerInitialization(pDevInfo);
    ASSERT(status == A_OK);

    // Turn on LED
    gpioFunc0(pDevInfo, TRUE);
    gpioFunc1(pDevInfo, TRUE);

    uiPrintf("< %s\n", __FUNCTION__);
    ATH_RELEASE_SPINLOCK_IRQ(pDevInfo->lock);
    return status;
}

//-----------------------------------------------------------------------------
// Procedure:   athDrvRadioDisable
//
// Description: Disables radio operation, if not already done.
//
// Parameters:  Address of callers radio disable flag, either one of hw or sw
//
// Returns:     A_OK if radio is disabled.  Any other value indicates radio disable
//              failed and its' functionality is indeterminate.
//-----------------------------------------------------------------------------
A_STATUS
athDrvRadioDisable(WLAN_DEV_INFO *pDevInfo, A_BOOL *pxwRadioDisable)
{
    A_STATUS status = A_OK;

    uiPrintf("> %s: *pxwRadioDisable = %d\n", __FUNCTION__, *pxwRadioDisable);

    if (*pxwRadioDisable == TRUE) {
        uiPrintf("< %s: nothing to do\n", __FUNCTION__);
        return A_OK;
    }

    *pxwRadioDisable = TRUE;

    if (!pDevInfo->rfSilent.radioEnabled) {
        uiPrintf("< %s: already disabled\n", __FUNCTION__);
        return A_OK;
    }

    ATH_ACQUIRE_SPINLOCK_IRQ(pDevInfo->lock);

    pDevInfo->NicResetInProgress    = TRUE;
    pDevInfo->rfSilent.radioEnabled = FALSE;

    pDevInfo->localSta->staState    = STATE_QUIET;
    pDevInfo->localSta->transState  = TSTATE_QUIET;

    osIndicateConnectStatus(pDevInfo, FALSE); // notify OS upper layer

    // Do not set cservOp = 0 because cserv operations are not cleaned up properly
    cservPollReset(pDevInfo); // reset poll counters so that they won't interfere

    // Don't just cancel timer, cancel the operation.
    // A_UNTIMEOUT(&pDevInfo->smeTimer);
    cservOperationCancel(pDevInfo);

    // Bring chips out of sleep temporarily to clean up uncompleted rx/tx
    if (powerSet(pDevInfo, WAKE_UP, SET_PWR_TO_2, FALSE, 0) != A_OK) {
        uiPrintf("< %s: powerSet() failed!\n", __FUNCTION__);
        pDevInfo->rfSilent.radioEnabled = TRUE;
        pDevInfo->NicResetInProgress    = FALSE;
        ATH_RELEASE_SPINLOCK_IRQ(pDevInfo->lock);
        return A_ERROR;
    }

    ATH_RELEASE_SPINLOCK_IRQ(pDevInfo->lock);

#if 0
    // Disable interrupts while we re-init the transmit structures
    halDisableInterrupts(HACK_TO_PARDEV(pDevInfo), HAL_INT_GLOBAL);

    // turn off receive filters. When pcu comes out of reset, filter should be all 0
    wlanRxFilter(pDevInfo, 0, RX_FILTER_INIT);

    // Flush TX queue.
    StopTransmit(pDevInfo, DO_CLEAR_TX_Q, DO_COMPLETE, DONT_WAIT);

    // Flush RX queue. NOTE: do not complete the receives (to NDIS, or SME etc)
    // StopSwReceive will prevent completion of RX packets.
    StopSwReceive(pDevInfo);
    StopReceive(pDevInfo);

    // Put hardware into a passive state
    halReset(HACK_TO_PARDEV(pDevInfo), WLAN_STA_SERVICE, NULL, FALSE);
    halPhyDisable(HACK_TO_PARDEV(pDevInfo));

    // Re-enable interrupts to catch switch change on the GPIO
    halEnableInterrupts(HACK_TO_PARDEV(pDevInfo), HAL_INT_GLOBAL);

    // Turn off LED
    gpioFunc0(pDevInfo, FALSE);
    gpioFunc1(pDevInfo, FALSE);
#endif

    NdisInterlockedIncrement(&pDevInfo->numFlushCall);
    wdcFlush(pDevInfo->targetHandle,pDevInfo->disableEnableCounter);

    wdcTargetDisable(pDevInfo->targetHandle,&status);

    ATH_ACQUIRE_SPINLOCK_IRQ(pDevInfo->lock);
    // Attempt to change state to complete sleep, ok if unsuccessful
    powerSet(pDevInfo, WAKE_UP, REDUCE_PWR_FROM_2, FALSE, 0);
    powerSet(pDevInfo, RADIO_ONLINE, DISABLE, FALSE, 0);

    uiPrintf("< %s\n", __FUNCTION__);

    ATH_RELEASE_SPINLOCK_IRQ(pDevInfo->lock);

    return A_OK;
}

//-----------------------------------------------------------------------------
// Procedure:   athDrvRadioStatus
//
// Description: Return availability of radio.
//              Optionally, if disabled, wake up the chip for long enough
//              to get a switch change interrupt.  Re-sleep the chip; the
//              DPC will run later to dis/enable the radio as necessary.
//
// Parameters:  checkSwitch: TRUE if above option is desired
//
// Returns:     TRUE if radio is enabled
//-----------------------------------------------------------------------------
A_BOOL
athDrvRadioStatus(WLAN_DEV_INFO *pDevInfo, A_BOOL checkSwitch)
{
    if (pDevInfo->rfSilent.radioEnabled) {
        return TRUE;
    }

    if (checkSwitch) {
        halSetPowerMode(HACK_TO_PARDEV(pDevInfo), AWAKE, TRUE);
        udelay(100);
        halSetPowerMode(HACK_TO_PARDEV(pDevInfo), FULL_SLEEP, TRUE);
    }

    return FALSE;
}


//-----------------------------------------------------------------------------
// Procedure:   securityInitialization
//
// Description: Allocate memory to support this feature.  Figure out cipher to use.
//              Program keys into s/w table and into h/w.
//
// Input:
//
// Returns:     RESOURCES or NDIS_ERROR_CODE_UNSUPPORTED_CONFIGURATION error
// TODO: Modify the logic to go through the routine even if privacyInvoked is false but
// keys are present.
//-----------------------------------------------------------------------------

A_STATUS
securityInitialization(WLAN_DEV_INFO *pDevInfo)
{
    A_STATUS         Status = A_OK;
    WLAN_PRIV_RECORD *pKey;
    A_UINT16         i;
    keystruct        *aeskeystruct;
    SIB_TABLE        *pSibTable = pDevInfo->pSibTable;

    uiPrintf("> %s\n", __FUNCTION__);

    pDevInfo->keyObfuscated = TRUE;

    /*
     * Only need a small key table currently for NDIS --
     * The 4 shared keys and one unique key.  The unique
     * key is loaded later per BSSID as needed.
     */
    Status = wlanKeyTableInit(pDevInfo, MAX_SHARED_KEYS + 1, 0);
    if (Status != A_OK) {
        uiPrintf("< %s: wlanKeyTableInit() failed!\n", __FUNCTION__);
        return NDIS_STATUS_RESOURCES;
    }

    /*
     *  Invalidate all SIB keycache mappings;
     *  these are preserved across hibernation
     */
    for (i = 0; i < MAX_STA_NUM; i++) {
        pSibTable->sta[i].hwIndex = HWINDEX_INVALID;
    }

    /*
     *  Program the shared keys.  Note that some installations
     *  may want to use 802.11 shared key authentication PLUS LEAP.
     *  This means they will need shared keys installed just to
     *  complete the shared key auth, then install dynamic keys
     *  after completing LEAP authentication.
     */
    for (i = 0; i < MAX_SHARED_KEYS; i++) {
        if (pDevInfo->staConfig.keys[i].keyLength != 0) {
            WLAN_PRIV_RECORD *pTempKey;
            A_UINT32 cipher, status;

            pTempKey = &pDevInfo->staConfig.keys[i];
            cipher = (pTempKey->keyLength == (104 / 13) ? WPA_CSE_WEP104
                                                        : WPA_CSE_WEP40);
            status = wlanKeyTableSet(pDevInfo,
                                     NULL,  /* No Sib */
                                     0,     /* Not connection key */
                                     cipher,
                                     i,
                                     pTempKey->keyLength * 8,
                                     pTempKey->keyVal);

            if (status != A_OK) {
                uiPrintf("securityInitialization : failed to install BSS key %i\n",
                         i);
            }
        }
    }
/*
 * XXX Divy. Consider unicast WEP key disabled for now.
 * Wait for Apps version to be decided (3.0 or 3.3)
 */
#if 0
    /*
     * Add unique key if it exists and is selected.  Will be sent to
     * the hardware cache when we know the BSSID to send to.
     */
    if ( (pDevInfo->staConfig.defaultKey == UNIQ_KEY_INDEX) &&
         (pDevInfo->staConfig.keys[UNIQ_KEY_INDEX].keyLength != 0) )
    {
        WLAN_PRIV_RECORD *pTempKey;

        pTempKey = &pDevInfo->staConfig.keys[UNIQ_KEY_INDEX];

        pKey = wlanKeyTableAdd(pDevInfo, UNIQ_KEY_INDEX,
                               pTempKey->keyType,
                               pTempKey->keyLength * 8,
                               pTempKey->keyVal);
        ASSERT(pKey);
        if (pKey) {
            wlanKeyCacheSet(pDevInfo, UNIQ_KEY_INDEX, NULL, NULL, pKey);
        }
    }
#endif

    A_MEM_ZERO(&pDevInfo->bssDescr->wpaIe, sizeof(pDevInfo->bssDescr->wpaIe));

    initCipher(pDevInfo);
    Crc32Init();

#if defined(WLAN_CONFIG_LEAP) && defined(LEAP_IN_DRIVER)
    if (pDevInfo->staConfig.leapEnabled &&
        pDevInfo->staConfig.leapInDriver &&
        leapInit(pDevInfo) != A_OK)
    {
        uiPrintf("< %s: leapInit() failed!\n", __FUNCTION__);
        return NDIS_STATUS_RESOURCES;
    }
#endif  /* WLAN_CONFIG_LEAP && LEAP_IN_DRIVER */

    uiPrintf("< %s\n", __FUNCTION__);

    return A_OK;
}

static A_STATUS
registryDecryptInit(WLAN_DEV_INFO *pDevInfo, UCHAR *pKey, int keyLen)
{
    uiPrintf("> %s: pKey = 0x%08x, keyLen = %d\n", __FUNCTION__, pKey, keyLen);

    if (pDevInfo->Ctx == NULL) {
        pDevInfo->Ctx = (BLOWFISH_CTX *)A_DRIVER_MALLOC(sizeof(BLOWFISH_CTX));
        if (pDevInfo->Ctx == NULL) {
            uiPrintf("< %s: BLOWFISH_CTX allocation failed!\n", __FUNCTION__);
            athLogError(pDevInfo, pDevInfo->pOSHandle->NicAdapterHandle,
                        NDIS_ERROR_CODE_OUT_OF_RESOURCES, 23);
            return NDIS_STATUS_RESOURCES;
        }
    }

    Blowfish_Init(pDevInfo->Ctx, pKey, keyLen);

    uiPrintf("< %s\n", __FUNCTION__);

    return NDIS_STATUS_SUCCESS;
}

//-----------------------------------------------------------------------------
// Procedure:    processWEPKeys
//
// Description: Get a crypt seed from Registry, use it to init the decryptor
//              For each key, decrypt, XOR and place back into config table
//
// Returns:     RESOURCES error if unable to allocate some memory, else A_OK
//-----------------------------------------------------------------------------
void
processWEPKeys(WLAN_DEV_INFO *pDevInfo)
{
    A_UINT16        i;
    A_UINT32        L, R;

    // replace encrypted key in config structs with decrypted/XORed copy
    for (i = 0; i < MAX_SHARED_KEYS + 1; i++) {
        if (pDevInfo->staConfig.keys[i].keyLength != 0) {
            L = *(A_UINT32 *)&pDevInfo->staConfig.keys[i].keyVal[0];
            R = *(A_UINT32 *)&pDevInfo->staConfig.keys[i].keyVal[4];
            Blowfish_Decrypt(pDevInfo->Ctx, &L, &R);

            *(A_UINT32 *)&pDevInfo->staConfig.keys[i].keyVal[0]        = L;
            *(A_UINT32 *)&pDevInfo->staConfig.keys[i].keyVal[4]        = R;
            *(A_UINT32 *)&pDevInfo->defaultStaConfig.keys[i].keyVal[0] = L;
            *(A_UINT32 *)&pDevInfo->defaultStaConfig.keys[i].keyVal[4] = R;

            if (pDevInfo->staConfig.keys[i].keyLength != 5) {
                L = *(A_UINT32 *)&pDevInfo->staConfig.keys[i].keyVal[8];
                R = *(A_UINT32 *)&pDevInfo->staConfig.keys[i].keyVal[12];
                Blowfish_Decrypt(pDevInfo->Ctx, &L, &R);

                *(A_UINT32 *)&pDevInfo->staConfig.keys[i].keyVal[8]         = L;
                *(A_UINT32 *)&pDevInfo->staConfig.keys[i].keyVal[12]        = R;
                *(A_UINT32 *)&pDevInfo->defaultStaConfig.keys[i].keyVal[8]  = L;
                *(A_UINT32 *)&pDevInfo->defaultStaConfig.keys[i].keyVal[12] = R;
            }
        }
    }
}

#if defined(WLAN_CONFIG_LEAP) && defined(LEAP_IN_DRIVER)
/*
 *  Decrypt LEAP password, blowfish style.  Blowfish requires the passwd
 *  string has been zero-padded to a 8 byte boundary.  We learn the
 *  real length from the leapUserPasswdLength from the registry.
 */
static void
leapPasswdDecrypt(WLAN_DEV_INFO *pDevInfo)
{
    int      i;
    int      len;
    A_UINT32 L, R;

    /* Roundup to include the padding used in encryption */
    len  = A_ROUNDUP(pDevInfo->staConfig.leapUserPasswdLen, 8);

    for (i = 0; i < len; i += 8) {
        L = *(A_UINT32 *)&pDevInfo->staConfig.leapUserPasswd[i];
        R = *(A_UINT32 *)&pDevInfo->staConfig.leapUserPasswd[i+4];
        Blowfish_Decrypt(pDevInfo->Ctx, &L, &R);
        *(A_UINT32 *)&pDevInfo->staConfig.leapUserPasswd[i]   = L;
        *(A_UINT32 *)&pDevInfo->staConfig.leapUserPasswd[i+4] = R;
        *(A_UINT32 *)&pDevInfo->defaultStaConfig.leapUserPasswd[i]   = L;
        *(A_UINT32 *)&pDevInfo->defaultStaConfig.leapUserPasswd[i+4] = R;
    }
}
#endif  /* WLAN_CONFIG_LEAP */

#ifdef NDIS40_MINIPORT

//-----------------------------------------------------------------------------
// Procedure:    cardRemovalNT4
//
// Description:  Supports CardWizard removal notification on Windows NT 4.
//
//                 Similar to athHalt except that the card is already
//                 powered down (i.e. we can't talk to the hardware).
//
// Returns:
//-----------------------------------------------------------------------------
void
cardRemovalNT4(WLAN_DEV_INFO *pDevInfo)
{
    // Prevent send processing
    pDevInfo->NicResetInProgress = TRUE;

    // We lost our connection
    osIndicateConnectStatus(pDevInfo, FALSE);

    // Cancel pending timers, otherwise they may get fired off
    // after we are gone.
    stopNdisTimers(pDevInfo, TRUE);

    // Flush transmit & receive queues
    StopSwReceive(pDevInfo);
    StopTransmit(pDevInfo, DO_CLEAR_TX_Q, DO_COMPLETE, DONT_WAIT);
    StopReceive(pDevInfo);

    // Don't free any driver data structure because the
    // driver is still loaded and entry points can be called
    // (by NDIS or via DeviceIoctl etc).
}

//-----------------------------------------------------------------------------
// Procedure:    cardReinsertionNT4
//
// Description:  Supports CardWizard re-insertion notification on Windows NT 4.
//
//               Because the card was powered down and then power on,
//               we must reset both the software and hardware state.
//               Similar to athInitialize except we will not allocate
//               DMA buffers and hardware resources (e.g. interrupt etc.)
//
//               The driver will remain loaded even if an error occurs
//               during re-insertion.  Therefore, no resources will
//               be freed if an error occurs.
//
// Returns:      NDIS_STATUS_SUCCESS if successful.
//-----------------------------------------------------------------------------
NDIS_STATUS
cardReinsertionNT4(WLAN_DEV_INFO *pDevInfo)
{
    NDIS_STATUS Status;

    /* Stop NDIS timers and free all WLAN resources */
    freeWlanResources(pDevInfo, TRUE);

    Status = athRestartDriver(pDevInfo);

    return Status;
}

#endif // NDIS40_MINIPORT

/* phyHangCheck - try to detect if PHY is hung
 *
 * We can guess that PHY is hung when:
 *   no rx or tx activity in the last epoch and
 *   no adhoc beacons transmitted (in adhoc mode only) and
 *   no infrastructure beacons received and
 *   hw reported FCS errors haven't changed and
 *   PHY receive RSSI is stuck at one value for more than a
 *     maximum frame length
 *
 * If we think that the HW is hung, we issue a reset.
 */
void
phyHangCheck(WLAN_DEV_INFO *pDevInfo)
{
#if TBD /* xxxxLMWxxxx */
/* Do we really need to do this at all?  Looks like old code for
old version of 11b hardware that would hang.   IF we need something
to check for hangs, part of it belongs on the target side; we might
want a new WDCEVT to indicate hung hardware. */
    A_UINT32    lastRssi;
    int         i       = 0;
    WLAN_STATS  *pStats = &pDevInfo->localSta->stats;

    if (!IS_CHAN_CCK(pDevInfo->staConfig.phwChannel->channelFlags) ||
        !powerEnabled(pDevInfo))
    {
        /* HW only hangs in 11b. Can't touch HW while chip is asleep. */
        return;
    }

    /* Read FCS & beacon counts into stats data structure. */
    halMibControl(pDevInfo, UPDATE_SW_ALL);

    if (HACK_TO_PARDEV(pDevInfo)->phyActivityFlag) {
        /* Some send or receive activity since the last time we were called. */
        goto done;
    }

    /* Check if beacons are being received/sent. */
    if (!(pDevInfo->localSta->staState & STATE_JOINED)     &&
         pDevInfo->localSta->transState != TSTATE_SCANNING &&
         pDevInfo->localSta->transState != TSTATE_JOINING)
    {
        goto done;
    }

    if (pDevInfo->bssDescr->bsstype == INDEPENDENT_BSS &&
        pDevInfo->baseBss.pBeaconInfo != NULL &&
        pDevInfo->baseBss.beaconQueue.pDescQueueHead)
    {
        ATHEROS_DESC *pBeaconDesc = pDevInfo->baseBss.beaconQueue.pDescQueueHead;

        if (halGetTxDescDone(HACK_TO_PARDEV(pDevInfo), (AR_DESC *)pBeaconDesc)) {
            /*
             * We sent some beacons in ad hoc mode. Calling
             * halSetupBeaconDesc is a round-about but safe
             * way of clearing up our status bits for the
             * next time around. The call is race-free w.r.t.
             * the hw accessing the beacon desc because in
             * this case it just updates the control words
             * with the same values as before.
             */
            halSetupBeaconDesc(pDevInfo, pBeaconDesc, 0, FALSE);
            goto done;
        }
    }

    if (pDevInfo->phyHwLastBeaconCount != pStats->RxBeacons) {
        /* Received a beacon since last time we checked. */
        goto done;
    }

    /* See if the PHY's RSSI has moved since the last time we checked. */
    lastRssi                = HACK_TO_PARDEV(pDevInfo)->phyHwLastRssi;
    HACK_TO_PARDEV(pDevInfo)->phyHwLastRssi = halGetCurRssi(HACK_TO_PARDEV(pDevInfo));

    if (pDevInfo->phyHwLastFcsFailCount != pStats->FcsFailCnt ||
        HACK_TO_PARDEV(pDevInfo)->phyHwLastRssi <= MIN_RSSI_FOR_HANG_CHECK)
    {
        goto done;
    }

    while (HACK_TO_PARDEV(pDevInfo)->phyHwLastRssi == lastRssi &&
           i < HANG_CHECK_SAMPLES)
    {
        udelay(HANG_CHECK_INTERVAL);
        HACK_TO_PARDEV(pDevInfo)->phyHwLastRssi = halGetCurRssi(HACK_TO_PARDEV(pDevInfo));
        i++;
    }

    if (i < HANG_CHECK_SAMPLES) {
        goto done;
    }

    /*
     * If we've made it this far, then we believe that the HW is hung.
     * Take a big stick and hit it hard. reset will shake out of the jam.
     */
    ResetOnError(pDevInfo, DONT_CLEAR_TX_Q, RESET_DONT_INC_CTR, DONT_WAIT);

    /*
     * Mark this chanel as passive scan so that we won't send probe requests
     * anymore, reducing the probability of receiver hang.
     */
    pDevInfo->staConfig.phwChannel->channelFlags |= CHANNEL_PASSIVE;

done:
    /* Reset flags and update counters before going into next round. */
    pDevInfo->phyHwLastFcsFailCount = pStats->FcsFailCnt;
    pDevInfo->phyHwLastBeaconCount  = pStats->RxBeacons;
    HACK_TO_PARDEV(pDevInfo)->phyHwLastRssi         = halGetCurRssi(HACK_TO_PARDEV(pDevInfo));
    HACK_TO_PARDEV(pDevInfo)->phyActivityFlag       = 0;
#endif
}

static void
athDataMsgRecvInd(TARGET_HANDLE targetHandle,
		  WDC_EVENT     event,
		  void          *param,
		  A_UINT32      infoSize,
		  A_UCHAR       *pInfo)
{
    WLAN_DEV_INFO *pDevInfo  = (WLAN_DEV_INFO *) param;
    WDC_RX_INFO   *wdcRxInfo = (WDC_RX_INFO   *) pInfo;
    ATHEROS_DESC  *pDesc;

    ASSERT(event == WDCEVT_DATA_INDICATION);
    ASSERT(infoSize == sizeof(WDC_RX_INFO));

    A_SEM_LOCK(pDevInfo->rxQueue.qSem, WAIT_FOREVER);
    pDesc = descQPopHead(&pDevInfo->rxQueue);
    A_SEM_UNLOCK(pDevInfo->rxQueue.qSem);

    if (pDesc) {
        pDesc->pBufferVirtPtr.ptr = wdcRxInfo->rxData;
//      pDesc->bufferPhysPtr      = pDesc->pBufferVirtPtr.ptr;
//      pDesc->pNextVirtPtr       = NULL;
//      pDesc->freeQueue          = &pDev->rxQueue;
//      pDesc->ffRefHack          = 1;
        pDesc->wdcRxInfo          = wdcRxInfo;

        ATH_ACQUIRE_SPINLOCK_IRQ(pDevInfo->lock);
        hostRxCallback(pDevInfo, pDesc, pDevInfo, wdcRxInfo->receiveInfo);
        ATH_RELEASE_SPINLOCK_IRQ(pDevInfo->lock);
    } else {
        wdcRxDone(pDevInfo->targetHandle, wdcRxInfo);
    }

    return;
}


static void
athSendCompleteInd(TARGET_HANDLE targetHandle,
		   WDC_EVENT     event,
		   void          *param,
		   A_UINT32      infoSize,
		   A_UCHAR       *pInfo)
{
    WLAN_DEV_INFO      *pDevInfo   = (WLAN_DEV_INFO *) param;
    TRANSMIT_INFO      *pTxInfo    = (TRANSMIT_INFO *) pInfo;
    ATHEROS_DESC       *pDesc;

    ASSERT(event    == WDCEVT_SEND_COMPLETE);
    ASSERT(infoSize == sizeof(*pTxInfo));

    pDesc = (ATHEROS_DESC *) pTxInfo->hostFrameHandle;

    ATH_ACQUIRE_SPINLOCK_IRQ(pDevInfo->sendLock);
    txHostCallback(pDevInfo, pDesc, pTxInfo);
    ATH_RELEASE_SPINLOCK_IRQ(pDevInfo->sendLock);

    return;
}

static void
athStatusNotify(TARGET_HANDLE targetHandle,
		   WDC_EVENT     event,
		   void          *param,
		   A_UINT32      infoSize,
		   A_UCHAR       *pInfo)
{
    WLAN_DEV_INFO      *pDevInfo   = (WLAN_DEV_INFO *) param;
    A_UINT32           status    = *((A_UINT32 *)(pInfo));

    ASSERT(event    == WDCEVT_STATUS_NOTIFY);
    ASSERT(infoSize == sizeof(A_UINT32));

    if (status == ATHUSB_BUS_STATE_FATAL) {
        abortSendWaitQueue(pDevInfo);
    }

    if ((pDevInfo->busStatus == ATHUSB_BUS_STATE_ERROR) && 
        ((status == ATHUSB_BUS_STATE_NORMAL) || status == ATHUSB_BUS_STATE_FATAL))
    {
        if (pDevInfo->localSta) {
            pDevInfo->localSta->stats.hwResetCount++;
        }

        pDevInfo->hwResetCount++;
    }

    pDevInfo->busStatus = status;

    return;
}

static void
athRcvChangeNotify(TARGET_HANDLE targetHandle,
		          WDC_EVENT     event,
		          void          *param,
		          A_UINT32      infoSize,
		          A_UCHAR       *pInfo)
{
    WLAN_DEV_INFO   *pDevInfo   = (WLAN_DEV_INFO *) param;

    ASSERT(event == WDCEVT_RX_IN_SYNC);

    pDevInfo->rcvChangeInProgress = FALSE;

    return;
}

static void
athFlushCompNotify(TARGET_HANDLE targetHandle,
		          WDC_EVENT     event,
		          void          *param,
		          A_UINT32      infoSize,
		          A_UCHAR       *pInfo)
{
    WLAN_DEV_INFO      *pDevInfo   = (WLAN_DEV_INFO *) param;

    ASSERT(event == WDCEVT_FLUSH_COMPLETE);

    if(*((A_UINT32*)pInfo) != pDevInfo->disableEnableCounter)
        return;
    ASSERT(A_ATOMIC_READ(&pDevInfo->numFlushCall));

    NdisInterlockedDecrement(&pDevInfo->numFlushCall);

    if (A_ATOMIC_READ(&pDevInfo->numFlushCall) == 0) {
        pDevInfo->flushInProgress = FALSE;
    }

    return;
}

static void
athBmissNotify(TARGET_HANDLE targetHandle,
		       WDC_EVENT     event,
		       void          *param,
		       A_UINT32      infoSize,
		       A_UCHAR       *pInfo)
{
    WLAN_DEV_INFO      *pDevInfo   = (WLAN_DEV_INFO *) param;
    A_LONGSTATS        RxBeacons = *((A_LONGSTATS *)pInfo);

    ASSERT(event == WDCEVT_BEACON_MISS);

    if ((pDevInfo->localSta->staState & STATE_CONNECTED) == STATE_CONNECTED ) {
        ATH_ACQUIRE_SPINLOCK_IRQ(pDevInfo->lock);
        
        if (RxBeacons) {
            pDevInfo->localSta->stats.RxBeacons = RxBeacons;
        }
        powerSet(pDevInfo, WAKE_UP, SET_PWR_TO_3, FALSE, 0);
        cservBmissNotify(pDevInfo);
        powerSet(pDevInfo, WAKE_UP, REDUCE_PWR_FROM_3, FALSE, 0);
        ATH_RELEASE_SPINLOCK_IRQ(pDevInfo->lock);
    }

    return;
}

static void
athWlanStatsNotify(TARGET_HANDLE targetHandle,
		       WDC_EVENT     event,
		       void          *param,
		       A_UINT32      infoSize,
		       A_UCHAR       *pInfo)
{
    WLAN_DEV_INFO      *pDevInfo   = (WLAN_DEV_INFO *) param;
    TARGET_STATS_DATA  devStats    = (TARGET_STATS_DATA) pInfo;
    A_UINT32           statsSize;
    A_UINT8            *pStatsData;
    WLAN_STATS         *pHostStats;

    ASSERT(event == WDCEVT_WLAN_STATS_UPDATE);
    ASSERT(devStats != NULL);

    if (!pDevInfo->pOSHandle->openForBusiness || devStats == NULL) {
        return;
    }

    pHostStats = &pDevInfo->localSta->stats;

    /* Receive related stats  */
    if (wdcGetStatsItem(devStats, STATS_RX_FCS_FAIL_CNT,
                        &statsSize, (void **)&pStatsData) == A_OK) 
    {
        ASSERT(statsSize == sizeof(A_UINT32));
        pHostStats->FcsFailCnt = *((A_UINT32 *)pStatsData);
    } 

    if (wdcGetStatsItem(devStats, STATS_RX_DMA_OVERRUN_ERRORS,
                        &statsSize, (void **)&pStatsData) == A_OK) 
    {
        ASSERT(statsSize == sizeof(A_UINT32));
        pHostStats->RcvDmaOverrunErrors = *((A_UINT32 *)pStatsData);
    } 

    if (wdcGetStatsItem(devStats, STATS_RX_BEACON_COUNT,
                        &statsSize, (void **)&pStatsData) == A_OK) 
    {
        ASSERT(statsSize == sizeof(A_LONGSTATS));
        pHostStats->RxBeacons = *((A_LONGSTATS *)pStatsData);
    } 


    /* Transmit related stats */

    if (wdcGetStatsItem(devStats, STATS_TX_GOOD_FRAMES,
                        &statsSize, (void **)&pStatsData) == A_OK) 
    {
        ASSERT(statsSize == sizeof(A_LONGSTATS));
        pHostStats->GoodTransmits = *((A_LONGSTATS *)pStatsData);
    } 

    if (wdcGetStatsItem(devStats, STATS_TX_UNICAST_FRAMES,
                        &statsSize, (void **)&pStatsData) == A_OK) 
    {
        ASSERT(statsSize == sizeof(A_LONGSTATS));
        pHostStats->TxUnicastFrames = *((A_LONGSTATS *)pStatsData);
    } 

    if (wdcGetStatsItem(devStats, STATS_TX_BROADCAST_FRAMES,
                        &statsSize, (void **)&pStatsData) == A_OK) 
    {
        ASSERT(statsSize == sizeof(A_LONGSTATS));
        pHostStats->TxBroadcastFrames = *((A_LONGSTATS *)pStatsData);
    } 

    if (wdcGetStatsItem(devStats, STATS_TX_MUTLICAST_FRAMES,
                        &statsSize, (void **)&pStatsData) == A_OK) 
    {
        ASSERT(statsSize == sizeof(A_LONGSTATS));
        pHostStats->TxMulticastFrames = *((A_LONGSTATS *)pStatsData);
    } 

    if (wdcGetStatsItem(devStats, STATS_TX_FRAMES_DROPPED,
                        &statsSize, (void **)&pStatsData) == A_OK) 
    {
        ASSERT(statsSize == sizeof(A_LONGSTATS));
        pHostStats->TxFramesDropped = *((A_LONGSTATS *)pStatsData);
    } 

    if (wdcGetStatsItem(devStats, STATS_TX_GOOD_BYTES,
                        &statsSize, (void **)&pStatsData) == A_OK) 
    {
        ASSERT(statsSize == sizeof(A_LONGSTATS));
        pHostStats->GoodTransmitBytes = *((A_LONGSTATS *)pStatsData);
    } 

    if (wdcGetStatsItem(devStats, STATS_TX_ERRORS,
                        &statsSize, (void **)&pStatsData) == A_OK) 
    {
        ASSERT(statsSize == sizeof(A_UINT32));
        pHostStats->TransmitErrors = *((A_UINT32 *)pStatsData);
    } 

    if (wdcGetStatsItem(devStats, STATS_TX_EXCESSIVE_RETRIES,
                        &statsSize, (void **)&pStatsData) == A_OK) 
    {
        ASSERT(statsSize == sizeof(A_UINT32));
        pHostStats->TxExcessiveRetries = *((A_UINT32 *)pStatsData);
    } 

    if (wdcGetStatsItem(devStats, STATS_TX_DMA_UNDERRUN,
                        &statsSize, (void **)&pStatsData) == A_OK) 
    {
        ASSERT(statsSize == sizeof(A_UINT32));
        pHostStats->TxDmaUnderrun = *((A_UINT32 *)pStatsData);
    } 

    if (wdcGetStatsItem(devStats, STATS_TX_ACK_RCV_FAILURES,
                        &statsSize, (void **)&pStatsData) == A_OK) 
    {
        ASSERT(statsSize == sizeof(A_UINT32));
        pHostStats->AckRcvFailures = *((A_UINT32 *)pStatsData);
    } 

    if (wdcGetStatsItem(devStats, STATS_TX_RTS_SUCCESS_CNT,
                        &statsSize, (void **)&pStatsData) == A_OK) 
    {
        ASSERT(statsSize == sizeof(A_LONGSTATS));
        pHostStats->RtsSuccessCnt = *((A_LONGSTATS *)pStatsData);
    } 

    if (wdcGetStatsItem(devStats, STATS_TX_RTS_FAIL_CNT,
                        &statsSize, (void **)&pStatsData) == A_OK) 
    {
        ASSERT(statsSize == sizeof(A_UINT32));
        pHostStats->RtsFailCnt = *((A_UINT32 *)pStatsData);
    } 

    if (wdcGetStatsItem(devStats, STATS_TX_COMP_SUCCESS_CNT,
                        &statsSize, (void **)&pStatsData) == A_OK) 
    {
        ASSERT(statsSize == sizeof(A_UINT32));
        pHostStats->compSuccessCnt = *((A_UINT32 *)pStatsData);
    } 

    if (wdcGetStatsItem(devStats, STATS_TX_COMP_CPC0_CNT,
                        &statsSize, (void **)&pStatsData) == A_OK) 
    {
        ASSERT(statsSize == sizeof(A_UINT32));
        pHostStats->CompCPC0Cnt = *((A_UINT32 *)pStatsData);
    } 

    if (wdcGetStatsItem(devStats, STATS_TX_COMP_CPC1_CNT,
                        &statsSize, (void **)&pStatsData) == A_OK) 
    {
        ASSERT(statsSize == sizeof(A_UINT32));
        pHostStats->CompCPC1Cnt = *((A_UINT32 *)pStatsData);
    } 

    if (wdcGetStatsItem(devStats, STATS_TX_COMP_CPC2_CNT,
                        &statsSize, (void **)&pStatsData) == A_OK) 
    {
        ASSERT(statsSize == sizeof(A_UINT32));
        pHostStats->CompCPC2Cnt = *((A_UINT32 *)pStatsData);
    } 

    if (wdcGetStatsItem(devStats, STATS_TX_COMP_CPC3_CNT,
                        &statsSize, (void **)&pStatsData) == A_OK) 
    {
        ASSERT(statsSize == sizeof(A_UINT32));
        pHostStats->CompCPC3Cnt = *((A_UINT32 *)pStatsData);
    } 

    if (wdcGetStatsItem(devStats, STATS_TX_FILTERED,
                        &statsSize, (void **)&pStatsData) == A_OK) 
    {
        ASSERT(statsSize == sizeof(A_UINT32));
        pHostStats->TxFiltered = *((A_UINT32 *)pStatsData);
    } 

    if (wdcGetStatsItem(devStats, STATS_TX_TOTAL_RETRIES,
                        &statsSize, (void **)&pStatsData) == A_OK) 
    {
        ASSERT(statsSize == sizeof(A_LONGSTATS));
        pHostStats->TotalRetries = *((A_LONGSTATS *)pStatsData);
    } 

    /* xxxxPGSxxxx- TODO - RetryBins[16], shortFrameRetryBins[16] */

    if (wdcGetStatsItem(devStats, STATS_TX_SW_RETRY_TOTAL_CNT,
                        &statsSize, (void **)&pStatsData) == A_OK) 
    {
        ASSERT(statsSize == sizeof(A_UINT32));
        pHostStats->swRetryTotalCnt = *((A_UINT32 *)pStatsData);
    } 
    
    if (wdcGetStatsItem(devStats, STATS_TX_SW_RETRY_MAX_RETRIES_EXCEEDED,
                        &statsSize, (void **)&pStatsData) == A_OK) 
    {
        ASSERT(statsSize == sizeof(A_UINT32));
        pHostStats->swRetryMaxRetriesExceeded = *((A_UINT32 *)pStatsData);
    } 

    /* Rates/Antenna/RSSI etc. */

    if (wdcGetStatsItem(devStats, STATS_ACK_RSSI,
                        &statsSize, (void **)&pStatsData) == A_OK) 
    {
        ASSERT(statsSize == sizeof(A_INT32));
        pHostStats->ackRssi = *((A_INT32 *)pStatsData);
    } 

    if (wdcGetStatsItem(devStats, STATS_TX_RATE_KB,
                        &statsSize, (void **)&pStatsData) == A_OK) 
    {
        ASSERT(statsSize == sizeof(A_UINT32));
        pHostStats->txRateKb = *((A_UINT32 *)pStatsData);
    } 

    /* xxxxPGSxxx - TODO    STATS_ANT_CNT AntCnt[2] */

    if (wdcGetStatsItem(devStats, STATS_ANT_SW_CNT,
                        &statsSize, (void **)&pStatsData) == A_OK) 
    {
        ASSERT(statsSize == sizeof(A_UINT32));
        pHostStats->AntSwCnt = *((A_UINT32 *)pStatsData);
    } 

    if (wdcGetStatsItem(devStats, STATS_CURR_TX_POWER,
                        &statsSize, (void **)&pStatsData) == A_OK) 
    {
        ASSERT(statsSize == sizeof(A_UINT32));
        pDevInfo->tx6PowerInHalfDbm = *((A_UINT32 *)pStatsData);
    } 

    if (wdcGetStatsItem(devStats, STATS_MAX_TX_POWER,
                        &statsSize, (void **)&pStatsData) == A_OK) 
    {
        ASSERT(statsSize == sizeof(A_UINT32));
        pDevInfo->maxTxPowerAvail = *((A_UINT32 *)pStatsData);
    } 

    if (wdcGetStatsItem(devStats, STATS_NOISE_FLOOR,
                        &statsSize, (void **)&pStatsData) == A_OK) 
    {
        ASSERT(statsSize == sizeof(A_UINT32));
        pDevInfo->noiseFloor = *((A_UINT32 *)pStatsData);
    }

    return;
}

#ifdef NDIS51_MINIPORT
LONG
TimerCount(WLAN_DEV_INFO *pDevInfo)
{
    OS_DEV_INFO          *pOsInfo;
    LONG                  currentCount;


    ASSERT(pDevInfo);

    pOsInfo = pDevInfo->pOSHandle;

    /*
     * Now need to walk thru all the timers
     * to find the total outstanding count
     */
    currentCount  = pOsInfo->NicAsyncResetTimer.timerCount;
    currentCount += pOsInfo->NicPnPEventTimer.timerCount;
#if defined(WLAN_CONFIG_LEAP) && defined(LEAP_IN_DRIVER)
    currentCount += pDevInfo->leapTimer.timerCount;
#endif
    currentCount += pOsInfo->NicSleepLogTimer.timerCount;
    currentCount += pDevInfo->smeTimer.timerCount;
    currentCount += pDevInfo->staPollTimer.timerCount;

    ASSERT(currentCount >= 0);

    return currentCount;
}
#endif
