#include "crypto/ocb.h"
#include "crypto/wep.h"
#include "wlandrv.h"
#include "wlanDev.h"
#include "wlanPhy.h"
#include "wlanext.h"
#include "wlansmeext.h"
#include "wlanDev.h"
#include "wlanbeacon.h"
#include "wlanchannel.h"
#include "wlanSend.h"
#include "wlanReceive.h"
#include "wlanvers.h"
#include "linuxdrv.h"
#include "linuxext.h"
#include "net/iw_handler.h"
#include "stacserv.h"
#include "athusbapi.h"
#include "ccx.h"
#include "intercept.h"
#include "usbstub.h"
#include "wlanCapDe.h"
#include "linux/config.h"
#include "linux/module.h"
#include "linux/kernel.h"

/* TODO : There is so much code under here, calls for a new file */

/* Dont know the max RSSI value yet.  It can reach upto 80 from what we see
 * But lets keep 60 as the max; anything above is 100% */
#define RSSI_MAX        60

/* Definitions Taken from apcfg.h */
#define MIN_FRAG_THRESHOLD      256
#define MAX_FRAG_THRESHOLD      2346

#define MIN_RTS_THRESHOLD      1
#define MAX_RTS_THRESHOLD      2346

static NDIS_STATUS
GetDeviceStatus (WLAN_DEV_INFO *pDevInfo);

static void
athDeviceReconfig (WLAN_DEV_INFO *pDevInfo);

static NDIS_STATUS
athSetPacketFilter(IN WLAN_DEV_INFO *pDevInfo, IN ULONG newPacketFilter);

NDIS_STATUS
athInitialize(PDEVICE_OBJECT pDeviceObject);

A_STATUS
athSetDevConfig(WLAN_DEV_INFO * pDevInfo);

void
freeResources(IN WLAN_DEV_INFO *pDevInfo);

VOID
athHalt(NDIS_HANDLE context);

static int athusb_siwpriv_authtype(WLAN_DEV_INFO *pDevInfo, A_UINT32 Value);

static struct iw_handler_def athusb_iw_handler_def;

static int DriverOpen (struct net_device *pNetDevice)
{
    return (athRestartDriver(DEVINFO_FROM_CONTEXT(pNetDevice)));
}

static int DriverStop (struct net_device *pNetDevice)
{
    WLAN_DEV_INFO *pDevInfo = DEVINFO_FROM_CONTEXT(pNetDevice);

    if (wlanDevPresent(pDevInfo)) {
        athShutdownDriver(DEVINFO_FROM_CONTEXT(pNetDevice),CONFIG_EVENT);
    }
    return (0);
}

static int isValidPacket (WLAN_DEV_INFO *pDevInfo, struct sk_buff *pSkb)
{
    struct ethhdr *hdr = (struct ethhdr *)(pSkb->data);
    unsigned short etherType;
    if (pDevInfo->staConfig.wpaEnabled) {
        WLAN_PRIV_RECORD *pKey = NULL;
        if (pDevInfo->staConfig.defaultKey != KEYINDEX_INVALID) {
            pKey = &pDevInfo->keyTable[pDevInfo->staConfig.defaultKey];
        }
        etherType = ntohs(hdr->h_proto);
        if ((pKey && pKey->keyLength) || (etherType == EAPOL_TYPEORLEN)) {
            return (1);
        } else {
            return (0);
        }
    }
    return (1);
}


static int DriverTransmit (struct sk_buff *pSkb, struct net_device *pNetDevice)
{
    WLAN_DEV_INFO *pDevInfo = DEVINFO_FROM_CONTEXT(pNetDevice);
    NDIS_STATUS Status;

    /* Check device and connection status before trying to send any packet */
    if (!wlanDevPresent(pDevInfo) || !netif_carrier_ok(pNetDevice) ||
            !isValidPacket(pDevInfo, pSkb)) {
        dev_kfree_skb(pSkb);
        return (0);
    }

    ATH_ACQUIRE_SPINLOCK_IRQ(pDevInfo->sendLock);
    Status = athSendSinglePacket(DEVINFO_FROM_CONTEXT(pNetDevice), pSkb);
    ATH_RELEASE_SPINLOCK_IRQ(pDevInfo->sendLock);

    return (0);// Always return Zero. SKB is freed both on suscess /error
}

struct net_device_stats *athusb_get_stats(struct net_device *pNetDevice)
{
    WLAN_DEV_INFO	*pDevInfo= DEVINFO_FROM_CONTEXT(pNetDevice);
    WLAN_STATS *wlanStats=&pDevInfo->localSta->stats;
    struct net_device_stats *pOSStats = &pDevInfo->pOSHandle->stats;
	
    memset (pOSStats, 0 , sizeof (struct net_device_stats));
    if (pDevInfo && pDevInfo->localSta) {
        wlanStats=&pDevInfo->localSta->stats;
    } else {
        return pOSStats;
    }
    
    pOSStats->rx_packets=wlanStats->GoodReceives;
    pOSStats->tx_packets= wlanStats->GoodTransmits;
    pOSStats->rx_bytes=wlanStats->GoodReceiveBytes;
    pOSStats->tx_bytes=wlanStats->GoodTransmitBytes;
    pOSStats->rx_errors=wlanStats->ReceiveErrors;
    pOSStats->tx_errors=wlanStats->TransmitErrors;
    pOSStats->rx_dropped=wlanStats->RxDiscardFrames;
    pOSStats->tx_dropped=wlanStats->TxFramesDropped;
    pOSStats->multicast=wlanStats->MulticastReceives;

    pOSStats->rx_over_errors=wlanStats->RcvDmaOverrunErrors;
    pOSStats->rx_crc_errors=wlanStats->RcvCrcErrors;

#if 0
        pOSStats->rx_length_errors=0;
        pOSStats->collisions=0;
        pOSStats->rx_frame_errors=0;
        pOSStats->rx_fifo_errors=0;
        pOSStats->rx_missed_errors=0;

        pOSStats->tx_aborted_errors=0;
        pOSStats->tx_carrier_errors=0;
        pOSStats->tx_fifo_errors=0;
        pOSStats->tx_heartbeat_errors=0;
        pOSStats->tx_window_errors=0;

        stats->rx_compressed=0;
        stats->tx_compressed=0;
#endif
        return pOSStats;
}

struct iw_statistics *athusb_get_wireless_stats(struct net_device *pNetDevice)
{
	WLAN_DEV_INFO *pDevInfo =DEVINFO_FROM_CONTEXT(pNetDevice);
	WLAN_STATS *wlanStats   = &pDevInfo->localSta->stats;	
	struct iw_statistics *pOSStats= &pDevInfo->pOSHandle->wi_stats;	
    int Rssi;

    memset (pOSStats, 0, sizeof (struct iw_statistics));
    if (pDevInfo && pDevInfo->localSta) {
        wlanStats = &pDevInfo->localSta->stats;	
    } else {
        return (pOSStats);
    }

    /* The status */	
    pOSStats->status = 0;
    
    Rssi = A_RSSI_OUT(wlanStats->rssi);
    if (Rssi > RSSI_MAX) {
        Rssi = RSSI_MAX;
    }
    pOSStats->qual.qual = (Rssi * 100) / RSSI_MAX;
    pOSStats->qual.noise = 0;                   /* we dont have */
    pOSStats->qual.level = RSSI_TO_NDIS(Rssi);  /* in dBm */
    pOSStats->qual.updated = 7;

    /* Packets discarded in the wireless adapter due to wireless
     * specific problems */
    pOSStats->discard.nwid = wlanStats->PrefAPMismatches;/* SSID Mismatch */
    pOSStats->discard.code = wlanStats->RcvKeyCacheMisses +
                             wlanStats->RcvDecryptCrcErrors +
                             wlanStats->RcvWEPExcludedCount;/* RxWepErr */
    pOSStats->discard.fragment = 0;
    pOSStats->discard.retries = wlanStats->TotalRetries;
    pOSStats->discard.misc = wlanStats->RcvDecompCrcErrors +
                             wlanStats->RcvDecipherErrors +
                             wlanStats->RcvDemicErrors +
                             wlanStats->RcvDeMMHErrors +
                             wlanStats->AckRcvFailures +
                             wlanStats->RtsFailCnt;
    pOSStats->miss.beacon = 0;

    return (pOSStats);
}

static int athusb_set_mac_address(struct net_device *pNetDevice, void *pContext)
{
    WLAN_DEV_INFO *pDevInfo = DEVINFO_FROM_CONTEXT(pNetDevice);
    struct sockaddr *pMacAddr = (struct sockaddr *)pContext;

    memcpy (&pDevInfo->staConfig.macAddr.octets[0], pMacAddr->sa_data,
            pNetDevice->addr_len);
    athDeviceReconfig (pDevInfo);
    return 0;
}

void athusb_set_multicast (struct net_device *pNetDevice)
{
    WLAN_DEV_INFO *pDevInfo = DEVINFO_FROM_CONTEXT(pNetDevice); 
    ULONG newPacketFilter;
    
    newPacketFilter = (NDIS_PACKET_TYPE_DIRECTED | NDIS_PACKET_TYPE_BROADCAST);

    if (!wlanDevPresent(pDevInfo)) {
        uiPrintf("Can't set multicast list because device isn't ready\n");
        return;
    }
    
    if ((pNetDevice->flags & IFF_PROMISC) ||
            (pNetDevice->flags & IFF_ALLMULTI) ||
            (pNetDevice->mc_count > MAX_MULTICAST_ADDRESSES))
    {
        newPacketFilter |= (NDIS_PACKET_TYPE_MULTICAST |
                NDIS_PACKET_TYPE_ALL_MULTICAST);
    } else {
        int i;
        struct dev_mc_list *pMcList = pNetDevice->mc_list;
        for (i = 0; i < pNetDevice->mc_count; i++) {
            memcpy(&pDevInfo->mcTable[i], pMcList->dmi_addr,
                    ETH_LENGTH_OF_ADDRESS); 
            pMcList = pMcList->next;
        }
        newPacketFilter |= NDIS_PACKET_TYPE_MULTICAST;
    }

    ATH_ACQUIRE_SPINLOCK_IRQ(pDevInfo->lock);
    // Wake up to access hardware
    powerSet(pDevInfo, WAKE_UP, SET_PWR_TO_3, FALSE, 0);
    athSetPacketFilter(pDevInfo, newPacketFilter);
    
    // Return to previous power setting
    powerSet(pDevInfo, WAKE_UP, REDUCE_PWR_FROM_3, FALSE, 0); 
    ATH_RELEASE_SPINLOCK_IRQ(pDevInfo->lock);
}

void *DriverProbe (PDEVICE_OBJECT pDeviceObject) {
    struct net_device *pNetDevice;

#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0))
    pNetDevice = kmalloc (sizeof(struct net_device), GFP_KERNEL);
    if (pNetDevice == NULL ) {
        return (NULL);
    }
    memset (pNetDevice, 0, sizeof (struct net_device));

    strcpy (pNetDevice->name, "ath%d");
    if (init_etherdev(pNetDevice, 0) == NULL) {
        goto ProbeExit;
    }
#else
    pNetDevice = alloc_netdev (0, "ath%d", ether_setup);
    if (pNetDevice == NULL ) {
        return (NULL);
    }
#endif

    SET_MODULE_OWNER(pNetDevice);
    pDeviceObject->pStubHandle = pNetDevice;

    pNetDevice->open = DriverOpen;
    pNetDevice->stop = DriverStop;
    pNetDevice->hard_start_xmit = DriverTransmit;
    pNetDevice->get_stats = athusb_get_stats;
    pNetDevice->get_wireless_stats = athusb_get_wireless_stats;
    pNetDevice->wireless_handlers = &athusb_iw_handler_def;
    pNetDevice->set_multicast_list = athusb_set_multicast;
    pNetDevice->set_mac_address = athusb_set_mac_address;

    /* A lot of allocation and one time initializations are done in
     * athInitialize.  Lets not use it as the netdev->open function */
    if (athInitialize(pDeviceObject) != NDIS_STATUS_SUCCESS) {
        goto ProbeExit;
    }

    if (register_netdev(pNetDevice)) {
        printk (KERN_ERR " %s: unabled to register device\n", pNetDevice->name);
        freeResources(DEVINFO_FROM_CONTEXT(pNetDevice));
        goto ProbeExit;
    }

    return (pNetDevice);

ProbeExit:
    gDrvInfo.nWorkingDev -= (gDrvInfo.nWorkingDev > 0) ? 1 : 0;
    A_DRIVER_FREE(pNetDevice, sizeof (struct net_device));
    return (NULL);
}

int DriverDisconnect (PDEVICE_OBJECT pDeviceObject)
{
    struct net_device *pNetDevice = pDeviceObject->pStubHandle;
    WLAN_DEV_INFO *pDevInfo = DEVINFO_FROM_CONTEXT(pNetDevice);

   // pDevInfo->busStatus = ATHUSB_BUS_STATE_SURPRISE_REMOVED;
   // wdcSurpriseRemoved(pDevInfo->targetHandle);

    athHalt(pNetDevice);

    if (pDeviceObject->pStubHandle) {
        unregister_netdev (pDeviceObject->pStubHandle);
#if (LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,0))
        free_netdev (pDeviceObject->pStubHandle);
#endif
    }

    return (0);
}

int athusb_giwname (struct net_device *pNetDevice,
        struct iw_request_info *pInfo,
        char *pName,
        char *pExtra)
{
    WLAN_DEV_INFO *pDevInfo     = DEVINFO_FROM_CONTEXT(pNetDevice);
    WLAN_STA_CONFIG *pStaConfig;
    WLAN_MODE mode;
    A_UINT16 NetBand;

    pStaConfig = &pDevInfo->staConfig;

    if (IS_CONN_OR_JOINED(pDevInfo)) {
        mode = wlanCFlagsToWirelessMode (pDevInfo,
                pDevInfo->staConfig.pChannel->channelFlags);
        NetBand = 1 << mode;
    } else {
        NetBand = pStaConfig->NetBand;
    }

    switch (NetBand) {
        case MODE_SELECT_11B :
            strncpy (pName, "IEEE 802.11b", IFNAMSIZ);
            break;
        case MODE_SELECT_11G :
            strncpy (pName, "IEEE 802.11g", IFNAMSIZ);
            break;
        case MODE_SELECT_11A :
            strncpy (pName, "IEEE 802.11a", IFNAMSIZ);
            break;
        case MODE_SELECT_TURBO :
            strncpy (pName, "IEEE 802.11a+Turbo", IFNAMSIZ);
            break;
        case MODE_SELECT_108G :
            strncpy (pName, "IEEE 802.11g+Turbo", IFNAMSIZ);
            break;
        default :
            strncpy (pName, "IEEE 802.11a/b/g/turbo", IFNAMSIZ);
            break;
    }
    return (0);
}

#define ONE_MHZ     (1 * 1000 * 1000)
int athusb_siwfreq(struct net_device *pNetDevice,
			 struct iw_request_info *pInfo,
			 struct iw_freq *pFreq,
			 char *pExtra)
{
	WLAN_DEV_INFO *pDevInfo = DEVINFO_FROM_CONTEXT(pNetDevice);
    A_UINT16 Frequency;
    A_UINT16 Channel;
    A_UINT16 AdhocBand;

	if(pFreq->e == 1) {
        Frequency = (pFreq->m * 10) / ONE_MHZ;
    } else if((pFreq->m >= 256) || (pFreq->e > 0)) {
		return (-EOPNOTSUPP);
    } else {
        /* We try to do Auto AdHoc Band selection here - right now do 
         * only for A and G */
        Channel = pFreq->m;
        if ((Channel > 0) && (Channel < 24)) {
            Frequency = wlanConvertChtoGHz(Channel, CHANNEL_2GHZ);
            AdhocBand = ATH_ADHOCBAND_GONLY;
        } else if ((Channel > 0) && (Channel < 256)) {
            Frequency = wlanConvertChtoGHz(Channel, CHANNEL_5GHZ);;
            AdhocBand = ATH_ADHOCBAND_AONLY;
        } else {
            return (-EOPNOTSUPP);
        }
	}
    
    /*
     * We dont have a desired channel feature for Infrastructure mode. Should
     * we complain about it here for non adhoc modes or just set adhoc 
     * frequency?
     */
    ATH_ACQUIRE_SPINLOCK_IRQ(pDevInfo->lock);
    pDevInfo->staConfig.adhocChannelFreq = Frequency;
//    pDevInfo->staConfig.AdhocBand = AdhocBand;
    ATH_RELEASE_SPINLOCK_IRQ (pDevInfo->lock);

    if (IS_CONN_OR_JOINED(pDevInfo) &&
            (pDevInfo->bssDescr->bsstype == INDEPENDENT_BSS))
    {
        athDeviceReconfig (pDevInfo);
    }
	return 0;
}


int athusb_giwfreq(struct net_device *pNetDevice,
			 struct iw_request_info *pInfo,
			 struct iw_freq *pFreq,
			 char *pExtra)
{
    WLAN_DEV_INFO *pDevInfo = DEVINFO_FROM_CONTEXT(pNetDevice);
    CHAN_VALUES *pChan = NULL;
    A_UINT16 Channel = 0;

    /* Get the Current Operating Channel */
    if (IS_CONN_OR_JOINED(pDevInfo)) {
        /* Available both in staConfig and bssDescr. Guess both are same */
        ASSERT(pDevInfo->bssDescr);
        pChan = pDevInfo->bssDescr->pChannel;
    }

    /* If not connected, get the default Ad-Hoc Frequency */
    if (pChan != NULL) {
        Channel = pChan->channel;
    } else {
        Channel = pDevInfo->staConfig.adhocChannelFreq;
    }


    /* We maintain Channel/Frequency info in MHZ - Convert it to e and m */
    /* m = freq / 10^e */
    if (Channel) {
        pFreq->e = 1;
        pFreq->m = Channel * (ONE_MHZ / 10);
    } else {
        return (-EOPNOTSUPP);
    }
	return 0;
}

int athusb_siwmode(struct net_device *pNetDevice,
			 struct iw_request_info *pInfo,
			 A_UINT32 *pMode,
			 char *pExtra)
{
    WLAN_DEV_INFO *pDevInfo = DEVINFO_FROM_CONTEXT(pNetDevice);
    BSSTYPE     oldType;
    A_STATUS    status;

    if (pDevInfo->busStatus == ATHUSB_BUS_STATE_SURPRISE_REMOVED) {
        uiPrintf("Can't set infra mode because device is gone\n");
        return NDIS_STATUS_NOT_ACCEPTED;
    }
    oldType = pDevInfo->staConfig.bssType;

    switch (*pMode) {
        case IW_MODE_INFRA:
            pDevInfo->staConfig.bssType = INFRASTRUCTURE_BSS;
            /*
             * Restore the SuperG Feature Setting for the Infrastructure Mode.
             */
            UPDATE_SUPERAG_CAP(pDevInfo, pDevInfo->staConfig.prefSuperAG); 
            break;
        case IW_MODE_ADHOC :
            pDevInfo->staConfig.bssType = INDEPENDENT_BSS;
            break;
        case IW_MODE_AUTO :
            pDevInfo->staConfig.bssType = ANY_BSS;
            break;
        default:
            return (-EOPNOTSUPP);
            break;
    }

    if (pDevInfo->staConfig.bssType == oldType) {
        /* We are already in the desired mode; just return */
        return 0;
    }

    if (!wlanDevPresent (pDevInfo)) {
        return 0;
    }
   
    // Make a note that the WZC service is now active.
    pDevInfo->pOSHandle->wzcActive = TRUE;
    
    /* Remove tx key, if WPA mode */
    if (pDevInfo->staConfig.wpaEnabled) { 
        status = powerSet(pDevInfo, WAKE_UP, SET_PWR_TO_3, FALSE, 0);
        if (status != A_OK) {
            return (NDIS_STATUS_NOT_ACCEPTED);
        } 
        /*
         * Clear the decomp mask before our key gets clobbered
         */ 
        wlanKeyCacheDecompMaskClear(pDevInfo, &pDevInfo->keyTable[
                pDevInfo->staConfig.defaultKey]);

        powerSet(pDevInfo, WAKE_UP, REDUCE_PWR_FROM_3, FALSE, 0);

        wlanKeyTableSlotFree(&pDevInfo->keyTable[
                pDevInfo->staConfig.defaultKey]);
        // XXX need a better, but safe invalid setting
        pDevInfo->staConfig.defaultKey = 0;
    }
    /*
     * In Infrastucture mode we lock the key cache using the
     * pSib.  In ad-hoc we can't so entries are never locked.
     * Because of this we need to reset the keycache on every
     * transition from ad-hoc to infrastructure and viceversa.
     */
    ATH_ACQUIRE_SPINLOCK_IRQ(pDevInfo->lock);

    status = powerSet(pDevInfo, WAKE_UP, SET_PWR_TO_3, FALSE, 0);
    if (status != A_OK) {
        return (NDIS_STATUS_NOT_ACCEPTED);
    }
    wlanReleaseKeyTableData(pDevInfo);
    powerSet(pDevInfo, WAKE_UP, REDUCE_PWR_FROM_3, FALSE, 0);
    ATH_RELEASE_SPINLOCK_IRQ(pDevInfo->lock);

    athDeviceReconfig (pDevInfo);

    return (0);
}

int athusb_giwmode(struct net_device *pNetDevice,
			 struct iw_request_info *pInfo,
			 A_UINT32 *pMode,
			 char *pExtra)
{
    WLAN_DEV_INFO *pDevInfo = DEVINFO_FROM_CONTEXT(pNetDevice);

    if (IS_CONN_OR_JOINED(pDevInfo)) {
        *pMode = (pDevInfo->bssDescr->bsstype == INDEPENDENT_BSS) ?
            (IW_MODE_ADHOC) : (IW_MODE_INFRA);
    } else {
        switch (pDevInfo->staConfig.bssType) {
        case INFRASTRUCTURE_BSS:
            *pMode = IW_MODE_INFRA;
            break;
        case INDEPENDENT_BSS :
            *pMode = IW_MODE_ADHOC;
            break;
        case ANY_BSS:
            *pMode = IW_MODE_AUTO;
            break;
        default:
            return (-EOPNOTSUPP);
            break;
        }
    }

    return (0);
}

static int athusb_giwrange(struct net_device *pNetDevice,
        struct iw_request_info *pInfo,
        struct iw_point *pData,
        char *pExtra)
{
	WLAN_DEV_INFO *pDevInfo = DEVINFO_FROM_CONTEXT(pNetDevice);
	struct iw_range *pRange = (struct iw_range *) pExtra;
    WLAN_CHANNEL_LIST *pCList;
    A_UINT16 length, listSize, i;

	pData->length = sizeof(struct iw_range);
	memset(pRange, 0, sizeof(struct iw_range));

	pRange->min_nwid = 0x0000;
	pRange->max_nwid = 0x0000;

	pRange->we_version_source = 15; /* we support WE15 and above */
	pRange->we_version_compiled = WIRELESS_EXT;

    pCList = pDevInfo->staConfig.pClist;
    if (pCList) {
        listSize = (pCList->listSize < IW_MAX_FREQUENCIES) ? pCList->listSize :
            IW_MAX_FREQUENCIES;
        pRange->num_channels = listSize;
        pRange->num_frequency = listSize;
        for(i = 0; i < listSize; i++) {
            pRange->freq[i].i = wlanConvertGHztoCh(pCList->chanArray[i].channel,
                    pCList->chanArray[i].channelFlags);
            pRange->freq[i].e = 1;
            pRange->freq[i].m = pCList->chanArray[i].channel * (ONE_MHZ / 10);
        }
    }

	pRange->max_qual.qual = 100;
	pRange->max_qual.level = 0;
	pRange->max_qual.noise = 0;

	pRange->sensitivity = 65535;

    /* FIXME : If max bitrates is set to 8, Wireless Tools 25 reports an error,
     * works OK with WT 26 */
    length = pDevInfo->baseBss.rateSet.length;
    length = (length < IW_MAX_BITRATES) ? length : IW_MAX_BITRATES;

	for(i = 0 ; i < length ; i++) {
		int bitrate =  pDevInfo->baseBss.rateSet.rates[i] & 0x7f;
        pRange->bitrate[i] = (bitrate * 1000000) / 2;
	}
	pRange->num_bitrates = length;


	pRange->min_rts = MIN_RTS_THRESHOLD;
	pRange->max_rts = MAX_RTS_THRESHOLD;

	pRange->min_frag = MIN_FRAG_THRESHOLD;
	pRange->max_frag = MAX_FRAG_THRESHOLD;

    pRange->max_encoding_tokens = MAX_SHARED_KEYS;
    pRange->num_encoding_sizes = 3;
    pRange->encoding_size[0] = 5;
    pRange->encoding_size[1] = 13;
    pRange->encoding_size[2] = 16;

	/* Transmit Power Settings - Hard code them. There is no reasonable to 
     * get this information from the card */
	pRange->txpower[0] = 10;    /* 10mW */
	pRange->txpower[1] = 14;    /* 20mW */
	pRange->txpower[2] = 15;    /* 30mW */
	pRange->txpower[3] = 17;    /* 50mW */
	pRange->txpower[4] = 18;    /* 63mW */
	pRange->txpower[5] = 20;    /* 100mW */
	pRange->num_txpower = 6;
	pRange->txpower_capa = IW_TXPOW_DBM;

#if 0
	range->min_pmp = 0;
	range->max_pmp = 5000000;	/* 5 secs */
	range->min_pmt = 0;
	range->max_pmt = 65535 * 1024;	/* ??? */
	range->pmp_flags = IW_POWER_PERIOD;
	range->pmt_flags = IW_POWER_TIMEOUT;
	range->pm_capa = IW_POWER_PERIOD | IW_POWER_TIMEOUT | IW_POWER_ALL_R;

	range->retry_capa = IW_RETRY_LIMIT | IW_RETRY_LIFETIME;
	range->retry_flags = IW_RETRY_LIMIT;
	range->r_time_flags = IW_RETRY_LIFETIME;
	range->min_retry = 1;
	range->max_retry = 65535;
	range->min_r_time = 1024;
	range->max_r_time = 65535 * 1024;
#endif

	return 0;
}


int athusb_siwap (struct net_device *pNetDevice,
        struct iw_request_info *pInfo,
        struct sockaddr *pSockAddr,
        char *pExtra)
{
    WLAN_DEV_INFO *pDevInfo = DEVINFO_FROM_CONTEXT(pNetDevice);
    WLAN_STA_CONFIG *pStaConfig = &pDevInfo->staConfig;

    /* No harm in setting the desired BSS for all modes */
    memcpy (pStaConfig->prefBssids[0].octets, &pSockAddr->sa_data,
            ETH_LENGTH_OF_ADDRESS);

    if(pDevInfo->staConfig.bssType == INDEPENDENT_BSS) {
        /* We are currently not in INFRA mode; just return */
        return (0);
    }

    // if we are on the same SSID and associated, return good status
    if (pDevInfo->pOSHandle->connectionStatus) {
        if (A_MACADDR_COMP((WLAN_MACADDR *)&pSockAddr->sa_data,
                &pDevInfo->baseBss.bssId) != 0)
        {
            athDeviceReconfig(pDevInfo);
        }
    }
    return (0);
}

static int athusb_listscan(WLAN_DEV_INFO *pDevInfo, int siwscan)
{
    OS_DEV_INFO *pOsInfo = pDevInfo->pOSHandle;

    // Need to clear any cached SSIDs from the OS.
    ssidArrayInit(&pOsInfo->ssids); 
    
    if (!athDrvRadioStatus(pDevInfo, FALSE) || !wlanDevPresent(pDevInfo)
            || !(pDevInfo->pCsInfo)) {
        /* We are in RFsilent operation */
        uiPrintf("Can't do the scan because device isn't ready\n");
        if (pDevInfo->busStatus == ATHUSB_BUS_STATE_SURPRISE_REMOVED) {
            return (-EINVAL);
        } else {
            return (0);
        }
    }
    
    if (siwscan) {
        if (pOsInfo->scan_timestamp) {
            return 0;
        }
        pOsInfo->scan_timestamp = jiffies;
    }

    if (IS_CONN_OR_JOINED(pDevInfo)) {
        /*
         * make sure only one cservXXX function call at one time
         */
        ATH_ACQUIRE_SPINLOCK_IRQ(pDevInfo->lock);
        cservBkScanAll(pDevInfo, SCAN_NOW);
        ATH_RELEASE_SPINLOCK_IRQ(pDevInfo->lock);
    } else {
        /*
         * make sure only one cservXXX function call at one time
         */
        ATH_ACQUIRE_SPINLOCK_IRQ(pDevInfo->lock);
        /* not connected, quickly step thru all channels */
        cservScan(pDevInfo, pDevInfo->staConfig.scanType, 0, FALSE);
        ATH_RELEASE_SPINLOCK_IRQ(pDevInfo->lock);
    }
    return (0);

}

static int athusb_siwscan(struct net_device *pNetDevice,
        struct iw_request_info *pInfo,
        struct iw_param *pData,
        char *pExtra)
{
    WLAN_DEV_INFO *pDevInfo = DEVINFO_FROM_CONTEXT(pNetDevice);
    return (athusb_listscan(pDevInfo, 1));
}

static int athusb_giwscan(struct net_device *pNetDevice,
        struct iw_request_info *pInfo,
        struct iw_point *pData,
        char *pExtra)
{
    WLAN_DEV_INFO *pDevInfo = DEVINFO_FROM_CONTEXT(pNetDevice);
    CSERV_INFO *pCsInfo = pDevInfo->pCsInfo;
    OS_DEV_INFO *pOsInfo = pDevInfo->pOSHandle;
    BSSDESCR_SET *pBssSet = NULL;
    struct iw_event IwEvent;
    char *pCurrentEv = pExtra;
    char *pEnd = pCurrentEv + IW_SCAN_MAX_DATA;
    char *pCurrentVal;
    int Rssi, index, i;
    unsigned char CustomBuf[256];

    /* Checking for scan_timestamp here doesnt work well for WPA supplicant
     * No harm in removing the check though
     * if (!pOsInfo->scan_timestamp || !pCsInfo) {
     */
    if (!pCsInfo) {
        return (-EOPNOTSUPP);
    }

    if (time_before(jiffies, pOsInfo->scan_timestamp + 3 * HZ)) {
        return (-EAGAIN);
    }
    pOsInfo->scan_timestamp = 0;
/*    if (pCsInfo->isScanRunDone == FALSE) {
        return (-EAGAIN);
    } */
    
    pBssSet = &pCsInfo->scanInfo.bssSet;
    if (!athDrvRadioStatus(pDevInfo, FALSE) || !wlanDevPresent(pDevInfo)) {
        /* Radio is off; null our list */
        pBssSet->count = 0;
    }

    for (index = 0; index < pBssSet->count; index++) {
        BSSDESCR *pOurs = &pBssSet->bssDescrArray[index];

        /* First entry mult be the AP MAC address */
        memset (&IwEvent, 0, sizeof (IwEvent));
        IwEvent.cmd = SIOCGIWAP;
        IwEvent.u.ap_addr.sa_family = ARPHRD_ETHER;
        memcpy(IwEvent.u.ap_addr.sa_data, &pOurs->bssId, ETH_LENGTH_OF_ADDRESS);
        IwEvent.len = IW_EV_ADDR_LEN;
        pCurrentEv = iwe_stream_add_event (pCurrentEv, pEnd, &IwEvent,
                IW_EV_ADDR_LEN);

        /* Add ESSID Info */
        memset(&IwEvent, 0, sizeof(IwEvent));
        IwEvent.cmd = SIOCGIWESSID;
        IwEvent.u.data.length = le2cpu16(pOurs->ssid.length);
        if (IwEvent.u.data.length > 32)
            IwEvent.u.data.length = 32;
        IwEvent.u.data.flags = 1;
        IwEvent.len = IW_EV_POINT_LEN + IwEvent.u.data.length;
        pCurrentEv = iwe_stream_add_point(pCurrentEv, pEnd, &IwEvent,
                pOurs->ssid.ssid);

        /* mode */
        memset(&IwEvent, 0, sizeof(IwEvent));
        IwEvent.cmd = SIOCGIWMODE;
        IwEvent.u.mode = (pOurs->bsstype == INFRASTRUCTURE_BSS) ? 
            IW_MODE_INFRA : IW_MODE_ADHOC;
        IwEvent.len = IW_EV_UINT_LEN;
        pCurrentEv = iwe_stream_add_event(pCurrentEv, pEnd, &IwEvent,
                IW_EV_UINT_LEN);

        /* Frequency */
        memset(&IwEvent, 0, sizeof(IwEvent));
        IwEvent.cmd = SIOCGIWFREQ;
        IwEvent.u.freq.m = pOurs->pChannel->channel*100000;
        IwEvent.u.freq.e = 1;
        IwEvent.len = IW_EV_FREQ_LEN;
        pCurrentEv = iwe_stream_add_event(pCurrentEv, pEnd, &IwEvent,
                IW_EV_FREQ_LEN);

        /* Signal Quality */
        memset(&IwEvent, 0, sizeof(IwEvent));
        IwEvent.cmd = IWEVQUAL;
        Rssi = A_RSSI_OUT(pOurs->rssi);
        if (Rssi > RSSI_MAX) {
            Rssi = RSSI_MAX;
        }
        IwEvent.u.qual.qual = (Rssi * 100) / RSSI_MAX;
        IwEvent.u.qual.noise = 0;
        IwEvent.u.qual.level = RSSI_TO_NDIS(Rssi);  /* in dBm */
        IwEvent.u.qual.updated = 7;

        IwEvent.len = IW_EV_QUAL_LEN;
        pCurrentEv = iwe_stream_add_event(pCurrentEv, pEnd, &IwEvent,
                IW_EV_QUAL_LEN);

        /* Encryption */
        memset(&IwEvent, 0, sizeof(IwEvent));
        IwEvent.cmd = SIOCGIWENCODE;
        if (pOurs->capabilityInfo.privacy)
            IwEvent.u.data.flags = IW_ENCODE_ENABLED | IW_ENCODE_NOKEY;
        else
            IwEvent.u.data.flags = IW_ENCODE_DISABLED;
        IwEvent.u.data.length = 0;
        IwEvent.len = IW_EV_POINT_LEN + IwEvent.u.data.length;
        pCurrentEv = iwe_stream_add_point(pCurrentEv, pEnd, &IwEvent, "");

        /* Data Rate */
        memset(&IwEvent, 0, sizeof(IwEvent));
        IwEvent.cmd = SIOCGIWRATE;
        pCurrentVal = pCurrentEv + IW_EV_LCP_LEN;
        for (i = 0; i < pOurs->supportedRateSet.length; i++) {
            if (pOurs->supportedRateSet.rates[i] == 0)
                break;
            IwEvent.u.bitrate.value =
                ((pOurs->supportedRateSet.rates[i] & 0x7f) * (1000000/2));
            pCurrentVal = iwe_stream_add_value (pCurrentEv, pCurrentVal,
                    pEnd, &IwEvent, IW_EV_PARAM_LEN);
        }
        /* Check if we added any event */
        if ((pCurrentVal - pCurrentEv) > IW_EV_LCP_LEN)
            pCurrentEv = pCurrentVal;

        /* WPA IE */
        if (VALID_WPA_ELEMENT(&pOurs->wpaIe)) {
            A_UINT8 *pBuf = CustomBuf;
            A_UINT8 *pWpaIe;
            memset(&IwEvent, 0, sizeof(IwEvent));
            IwEvent.cmd = IWEVCUSTOM;

            pWpaIe = (A_UINT8 *)&pOurs->wpaIe;
            pBuf += sprintf(pBuf, "wpa_ie=");
            for (i = 0; i < pOurs->wpaIe.length+2; i++) {
                pBuf += sprintf(pBuf, "%02x", pWpaIe[i]);
            }
            IwEvent.u.data.length = strlen(CustomBuf);
            pCurrentEv = iwe_stream_add_point (pCurrentEv, pEnd, &IwEvent, CustomBuf);
        }
    }

    pData->length = (pCurrentEv - pExtra);
    pData->flags = 0;

    return (0);
}

int athusb_giwap (struct net_device *pNetDevice,
        struct iw_request_info *pInfo,
        struct sockaddr *pSockAddr,
        char *pExtra)
{
    WLAN_DEV_INFO *pDevInfo = DEVINFO_FROM_CONTEXT(pNetDevice);

    if (IS_CONN_OR_JOINED(pDevInfo)) {
        memcpy (&pSockAddr->sa_data, pDevInfo->baseBss.bssId.octets,
                ETH_LENGTH_OF_ADDRESS);
    } else { 
        memset (&pSockAddr->sa_data, 0, ETH_LENGTH_OF_ADDRESS);
    }
    return (0);
}

int athusb_siwessid(struct net_device *pNetDevice,
			 struct iw_request_info *pInfo,
			 struct iw_point *pData,
             char *pSsid)
{
    WLAN_DEV_INFO *pDevInfo = DEVINFO_FROM_CONTEXT(pNetDevice);
    WLAN_STA_CONFIG *pStaConfig = NULL;
    SSID localSsid;

	if (pData->flags == 0) {		/* ANY */
        strcpy(localSsid.ssid,"");
        localSsid.length = 0;
	} else {
        if (pData->length > 32) {
            pData->length = 32;
        }
        if (pSsid[pData->length-1] == '\0') {
            pData->length--;
        }
        memcpy (localSsid.ssid, pSsid, pData->length);
        localSsid.length = pData->length;
    }

    if (IS_CONN_OR_JOINED(pDevInfo)) {
        if ((pDevInfo->bssDescr->ssid.length == localSsid.length) &&
            (memcmp (pDevInfo->bssDescr->ssid.ssid, localSsid.ssid,
                     localSsid.length) == 0))
        {
            return 0;
        }
    }

    pStaConfig = &pDevInfo->staConfig;
    ATH_ACQUIRE_SPINLOCK_IRQ(pDevInfo->lock);
    ssidArrayInit (&pStaConfig->cfgSsids);
    ssidArrayAdd (&pStaConfig->cfgSsids, localSsid.ssid, localSsid.length);
    ATH_RELEASE_SPINLOCK_IRQ (pDevInfo->lock);
    athDeviceReconfig (pDevInfo);
	return (0);
}

int athusb_giwessid(struct net_device *pNetDevice,
			 struct iw_request_info *pInfo,
			 struct iw_point *pData,
             char *pSsidName)
{
    WLAN_DEV_INFO *pDevInfo = DEVINFO_FROM_CONTEXT(pNetDevice);
    SSID *pSsid = NULL;

    if (IS_CONN_OR_JOINED(pDevInfo)) {
        pSsid = &pDevInfo->bssDescr->ssid;
    } else {
        pSsid = &pDevInfo->staConfig.cfgSsids.elements[0];
    }

    pData->flags = 1;
    pData->length = pSsid->length;
    memcpy (pSsidName, pSsid->ssid, pData->length);
    return (0);
}

int athusb_siwrate (struct net_device *pNetDevice,
        struct iw_request_info *info,
        struct iw_param *pData,
        char *pExtra)
{
    WLAN_DEV_INFO *pDevInfo = DEVINFO_FROM_CONTEXT(pNetDevice);
    A_UINT32 TxRate;

    if (pData->fixed == 0) {
        TxRate = 0;
    }
    else if ((pData->value >= 0) && pData->value < IW_MAX_BITRATES) {
		TxRate =  pDevInfo->baseBss.rateSet.rates[pData->value] & 0x7f;
        TxRate = (TxRate * 1000) / 2;
    } else {
        TxRate = pData->value/1000;
    }

    staTxRateSet(pDevInfo, TxRate);
    athDeviceReconfig (pDevInfo);

    return (0);
}


int athusb_giwrate (struct net_device *pNetDevice,
        struct iw_request_info *info,
        struct iw_param *pData,
        char *pExtra)
{
    WLAN_DEV_INFO *pDevInfo = DEVINFO_FROM_CONTEXT(pNetDevice);

    if (pDevInfo->localSta) {
        pData->value = staTxRateKbpsGet(pDevInfo, pDevInfo->localSta) * 1000;
    } else {
        pData->value = 0;
    }
    pData->fixed = (pDevInfo->staConfig.rateCtrlEnable == FALSE);
    pData->disabled = 0;

    return (0);
}

int athusb_siwrts (struct net_device *pNetDevice,
        struct iw_request_info *info,
        struct iw_param *pData,
        char *pExtra)
{
    WLAN_DEV_INFO *pDevInfo = DEVINFO_FROM_CONTEXT(pNetDevice);
    WLAN_STA_CONFIG *pStaConfig = &pDevInfo->staConfig;
    A_UINT32 newValue = pData->value;

    if (pData->disabled) {
        newValue = MAX_RTS_THRESHOLD;
    } else if ((newValue < MIN_RTS_THRESHOLD) ||
            (newValue > MAX_RTS_THRESHOLD))
    {
        return (-EINVAL);
    }
        
    if (newValue != pStaConfig->userRTSThreshold) {
        pStaConfig->userRTSThreshold = newValue;
        athDeviceReconfig (pDevInfo);
    }
    return (0);
}

int athusb_giwrts (struct net_device *pNetDevice,
        struct iw_request_info *info,
        struct iw_param *pData,
        char *pExtra)
{
    WLAN_DEV_INFO *pDevInfo = DEVINFO_FROM_CONTEXT(pNetDevice);
    WLAN_STA_CONFIG *pStaConfig = &pDevInfo->staConfig;
    
    pData->value = pStaConfig->userRTSThreshold;
    pData->disabled = (pData->value >= MAX_RTS_THRESHOLD);
    pData->fixed = 1;

    return (0);
}


int athusb_siwfrag (struct net_device *pNetDevice,
        struct iw_request_info *info,
        struct iw_param *pData,
        char *pExtra)
{
    WLAN_DEV_INFO *pDevInfo = DEVINFO_FROM_CONTEXT(pNetDevice);
    WLAN_STA_CONFIG *pStaConfig = &pDevInfo->staConfig;
    A_UINT32 newValue = pData->value;

    if (pData->disabled) {
        newValue = MAX_FRAG_THRESHOLD;
    } else if ((newValue < MIN_FRAG_THRESHOLD) ||
            (newValue > MAX_FRAG_THRESHOLD))
    {
        return (-EINVAL);
    }
        
    if (newValue != pStaConfig->userFragThreshold) {
        pStaConfig->userFragThreshold = newValue;
        athDeviceReconfig (pDevInfo);
    }
    return (0);
}

int athusb_giwfrag (struct net_device *pNetDevice,
        struct iw_request_info *info,
        struct iw_param *pData,
        char *pExtra)
{
    WLAN_DEV_INFO *pDevInfo = DEVINFO_FROM_CONTEXT(pNetDevice);
    WLAN_STA_CONFIG *pStaConfig = &pDevInfo->staConfig;
    
    pData->value = pStaConfig->userFragThreshold;
    pData->disabled = (pData->value >= MAX_FRAG_THRESHOLD);
    pData->fixed = 1;

    return (0);
}

int athusb_siwtxpow (struct net_device *pNetDevice,
        struct iw_request_info *info,
        struct iw_param *pData,
        char *pExtra)
{
    WLAN_DEV_INFO *pDevInfo = DEVINFO_FROM_CONTEXT(pNetDevice);
    WLAN_STA_CONFIG *pStaConfig = &pDevInfo->staConfig;
    A_UINT16 tpcHalfDbm;

    if (!wlanDevPresent(pDevInfo)) {
        return (-EOPNOTSUPP);
    }

    if (pData->disabled) {
        athDrvRadioDisable (pDevInfo, &pDevInfo->rfSilent.swRadioDisable);
        return (0);
    }

    if (pData->value >= 0 && pData->value <= 30) {
        tpcHalfDbm = pData->value * 2;
    } else {
        tpcHalfDbm = 60;
    }
    pStaConfig->tpcHalfDbm5 = pStaConfig->tpcHalfDbm2 = tpcHalfDbm;

    if (!athDrvRadioStatus (pDevInfo, FALSE) &&
            pDevInfo->pOSHandle->openForBusiness)
    {
        athDrvRadioEnable (pDevInfo, &pDevInfo->rfSilent.swRadioDisable);
    }


    athDeviceReconfig(pDevInfo);
    return (0);
}

int athusb_giwtxpow (struct net_device *pNetDevice,
        struct iw_request_info *info,
        struct iw_param *pData,
        char *pExtra)
{
    WLAN_DEV_INFO *pDevInfo = DEVINFO_FROM_CONTEXT(pNetDevice);
    WLAN_STA_CONFIG *pStaConfig = &pDevInfo->staConfig;

    pData->value = pStaConfig->tpcHalfDbm5 / 2;
    pData->fixed = 1;
    pData->disabled = athDrvRadioStatus(pDevInfo, FALSE);
    pData->flags = IW_TXPOW_DBM;
    return (0);
}

int athusb_siwretry (struct net_device *pNetDevice,
        struct iw_request_info *info,
        struct iw_param *pData,
        char *pExtra)
{
    WLAN_DEV_INFO *pDevInfo = DEVINFO_FROM_CONTEXT(pNetDevice);

    if (pData->disabled) {
        pDevInfo->staConfig.swRetryMaxRetries = 0;
        return (0);
    }
    if (pData->flags & IW_RETRY_LIFETIME ||
            pData->flags & IW_RETRY_MIN) {
        return (-EOPNOTSUPP);
    }
    if (pData->value < 0 || pData->value > 15) {
        return (-EINVAL);
    }
    pDevInfo->staConfig.swRetryMaxRetries = pData->value;

    athDeviceReconfig(pDevInfo);
    return (0);
}

int athusb_giwretry (struct net_device *pNetDevice,
        struct iw_request_info *info,
        struct iw_param *pData,
        char *pExtra)
{
    WLAN_DEV_INFO *pDevInfo = DEVINFO_FROM_CONTEXT(pNetDevice);
    pData->disabled = (pDevInfo->staConfig.swRetryMaxRetries == 0);
    pData->flags = IW_RETRY_LIMIT;
    pData->value = pDevInfo->staConfig.swRetryMaxRetries;
    return (0);
}

int athusb_siwpower (struct net_device *pNetDevice,
        struct iw_request_info *info,
        struct iw_param *pData,
        char *pExtra)
{
//    WLAN_DEV_INFO *pDevInfo = DEVINFO_FROM_CONTEXT(pNetDevice);
    return (-EOPNOTSUPP);
}

int athusb_giwpower (struct net_device *pNetDevice,
        struct iw_request_info *info,
        struct iw_param *pData,
        char *pExtra)
{
//    WLAN_DEV_INFO *pDevInfo = DEVINFO_FROM_CONTEXT(pNetDevice);

    pData->disabled = 1;
    return (0);
}

int athusb_siwencode (struct net_device *pNetDevice,
        struct iw_request_info *info,
        struct iw_point *pData,
        char *pKey)
{
    WLAN_DEV_INFO *pDevInfo = DEVINFO_FROM_CONTEXT(pNetDevice);
    WLAN_STA_CONFIG *pStaConfig = &pDevInfo->staConfig;
    int keyIndex = 0;
    A_BOOL privacyInvoked = pStaConfig->privacyInvoked;
    A_UINT16 authType = pStaConfig->authType;

    keyIndex = pData->flags & IW_ENCODE_INDEX;
    if (keyIndex <= 0) {
        /* Key index is not set - use default key */
        keyIndex = pStaConfig->defaultKey;
    } else {
        keyIndex--;
    }

    if (keyIndex >= MAX_SHARED_KEYS) {
        /* If the current or previous default tx key is out of range we really
         * cant do much */
        return (-EINVAL);
    }

    if ((pData->flags & IW_ENCODE_DISABLED) &&
            (pStaConfig->privacyInvoked == TRUE))
    {
        privacyInvoked = FALSE;
    }

    /* Check and change authentication modes */
    if (pData->flags & IW_ENCODE_OPEN) {
        authType = AUTH_TYPE_OPEN;
    } else if (pData->flags & IW_ENCODE_RESTRICTED) {
        authType = AUTH_TYPE_SHARED;
    }

    if ((pData->flags & IW_ENCODE_NOKEY) == 0) {
        if (pData->length > 0) {
            /* Just set the key, dont change default tx key */
            if (pData->length > 16) {
                return (-EOPNOTSUPP);
            }
            memcpy(pStaConfig->keys[keyIndex].keyVal, pKey, pData->length);
            pStaConfig->keys[keyIndex].keyLength = pData->length;
        }

        if ((pData->length == 0) && !(pData->flags & IW_ENCODE_INDEX) &&
                !(pData->flags & IW_ENCODE_DISABLED)) {
            /* This is how encryption is enabled ; Forget about the keyIndex
             * we got here; use the default */
            keyIndex = pStaConfig->defaultKey;
            if (!pStaConfig->keys[keyIndex].keyLength || 
                    (pStaConfig->keys[keyIndex].keyLength > 16))
            {
                return (-EINVAL);
            }
            privacyInvoked = TRUE;
        }
    } else {
        if (pData->length == 0 && (pData->flags & IW_ENCODE_INDEX)) {
            /* We are changing the default Tx Key.  If the keylength is invalid
             * then dont set it */
            if (!pStaConfig->keys[keyIndex].keyLength || 
                    (pStaConfig->keys[keyIndex].keyLength > 16))
            {
                return (-EINVAL);
            }
            pStaConfig->defaultKey = keyIndex;
        }
    }

    pStaConfig->authType = authType;
    if (pStaConfig->privacyInvoked || privacyInvoked) {
        pStaConfig->privacyInvoked = privacyInvoked;
        athDeviceReconfig(pDevInfo);
    }

    return (0);
}

static int athusb_setregulatory (WLAN_DEV_INFO *pDevInfo, unsigned char *countryIsoName)
{
    A_UINT16         countryCode;
    
    countryCode = wlanGetCountryCodeByName(countryIsoName, FALSE);
    
    if (countryCode == pDevInfo->staConfig.countryCode) {
        return 0;
    } 
    
    if ((countryCode == CTRY_INVALID) ||
            !wlanIsCountryCodeValid(pDevInfo, countryCode))
    {
        printk("OID_ATH_RD failed\n");
        return (NDIS_STATUS_FAILURE);
    }

    if (!wlanDevPresent(pDevInfo)) {
        return (NDIS_STATUS_NOT_ACCEPTED);
    } 

    /* Save country code in a safe place */
    pDevInfo->staConfig.sku.ccByOid = countryCode;

    if (GetDeviceStatus (pDevInfo) != NDIS_STATUS_SUCCESS) {
        pDevInfo->staConfig.countryCode = countryCode;
        return 0;
    }

    ATH_ACQUIRE_SPINLOCK(pDevInfo->lock);
    /* Must be awake to access hardware */
    powerSet(pDevInfo, WAKE_UP, SET_PWR_TO_3, FALSE, 0);

    ATH_RELEASE_SPINLOCK(pDevInfo->lock);

    /* Shutdown the driver so we can safely change the configuration */
    /* We awaken chip in here */
    athShutdownDriver(pDevInfo, CONFIG_EVENT);

    ATH_ACQUIRE_SPINLOCK(pDevInfo->lock);
    // Set new country code
    pDevInfo->staConfig.countryCode = countryCode;

    // Re-initialize the driver; power is left in default state
    athRestartDriver(pDevInfo);

    /* Restore power state */
    powerSet(pDevInfo, WAKE_UP, REDUCE_PWR_FROM_3, FALSE, 0);
    ATH_RELEASE_SPINLOCK(pDevInfo->lock);

    return (0);
}

int athusb_giwencode (struct net_device *pNetDevice,
        struct iw_request_info *info,
        struct iw_point *pData,
        char *pKey)
{
    WLAN_DEV_INFO *pDevInfo = DEVINFO_FROM_CONTEXT(pNetDevice);
    WLAN_STA_CONFIG *pStaConfig = &pDevInfo->staConfig;
    int keyIndex = (pData->flags & IW_ENCODE_INDEX) - 1;

    if ((keyIndex < 0) || (keyIndex >= MAX_SHARED_KEYS)) {
        keyIndex = pStaConfig->defaultKey;
    }
    
    if (pStaConfig->privacyInvoked /* && !pStaConfig->wpaEnabled */) {
        pData->length = pStaConfig->keys[keyIndex].keyLength;
        memcpy(pKey, pStaConfig->keys[keyIndex].keyVal, pData->length);
        pData->flags = keyIndex + 1;
        pData->flags |= IW_ENCODE_ENABLED;
    } else {
        pData->length = 0;
        pData->flags = IW_ENCODE_DISABLED;
    }

    if (pStaConfig->authType == AUTH_TYPE_SHARED) {
        pData->flags |= IW_ENCODE_RESTRICTED;
    } else {
        pData->flags |= IW_ENCODE_OPEN;
    }
    return (0);
}

int athusb_setvalue (struct net_device *pNetDevice,
        struct iw_request_info *pInfo,
        void *pData,
        char *pExtra)
{
    WLAN_DEV_INFO *pDevInfo = DEVINFO_FROM_CONTEXT(pNetDevice);
    int *ParamList = (int *)pExtra;
    int Parameter = ParamList[0];
    int Value = ParamList[1];
    switch (Parameter) {
        case ATHUSB_SUBIOCTL_NETBAND:
            Value = Value & MODE_SELECT_ALL;
            if (Value == pDevInfo->staConfig.NetBand) {
                return 0;
            }
            pDevInfo->staConfig.NetBand = Value;
            break;
        case ATHUSB_SUBIOCTL_ADHOCBAND:
            if (Value < ATH_ADHOCBAND_BONLY || Value > ATH_ADHOCBAND_108GONLY) {
                return (-EINVAL);
            }
            /* 11g mode is not yet supported for Ad-Hoc, use 11b */
            if (Value == ATH_ADHOCBAND_GONLY) {
                Value = ATH_ADHOCBAND_BONLY;
            }
            if (Value == pDevInfo->staConfig.AdhocBand) {
                return (0);
            }
            pDevInfo->staConfig.AdhocBand = Value;
            break;
        case ATHUSB_SUBIOCTL_SHORTPREAMBLE:
            if (Value < 0 || Value > 1) {
                return (-EINVAL);
            }
            if (Value == pDevInfo->staConfig.shortPreamble) {
                return (0);
            }
            pDevInfo->staConfig.shortPreamble = Value;
            break;
        case ATHUSB_SUBIOCTL_ABOLT:
            if (Value < 0 || Value > 0xffff) {
                return (-EINVAL);
            }
            if (Value == pDevInfo->staConfig.abolt) {
                return (0);
            }
            pDevInfo->staConfig.abolt = Value;
            break;
        case ATHUSB_SUBIOCTL_AUTHTYPE:
            athusb_siwpriv_authtype(pDevInfo, Value);
            break;
        case ATHUSB_SUBIOCTL_PRIVACY:
        {
            PRIVACY_FILTER Filter = Value;
            if (Filter == PrivFilter8021xWEP) {
                pDevInfo->staConfig.filter8021x = TRUE;
            } else {
                pDevInfo->staConfig.filter8021x = FALSE;
            }
            break;
        }
        case ATHUSB_SUBIOCTL_ENCRYPTION:
        {
            ENCRYPTION_ALGORITHM encAlg = Value;
            if (encAlg < EncryptionDisabled || encAlg >EncryptionAuto) {
                return (-EINVAL);
            }
            pDevInfo->staConfig.encryptionAlg = encAlg;
            if (pDevInfo->staConfig.encryptionAlg != ENCRYPTION_DISABLED) {
                pDevInfo->staConfig.privacyInvoked  = TRUE;
            } else {
                pDevInfo->staConfig.privacyInvoked  = FALSE;
            }
            break;
        }
        case ATHUSB_SUBIOCTL_WPA:
            if (Value == 0) {
                athusb_siwpriv_authtype(pDevInfo, AuthModeOpen);
            } else {
                return (0);
            }
            break;
        case ATHUSB_SUBIOCTL_CHANNEL:
        case ATHUSB_SUBIOCTL_RECV_DATARATE:
        case ATHUSB_SUBIOCTL_POWERSAVE_STATE:
        case ATHUSB_SUBIOCTL_SIGNAL_QUALITY:
        case ATHUSB_SUBIOCTL_NOISE_FLOOR:
        case ATHUSB_SUBIOCTL_DRIVER_VERSION:
        case ATHUSB_SUBIOCTL_TRANSMIT_POWER:
        case ATHUSB_SUBIOCTL_VENDOR_VERSION:
        case ATHUSB_SUBIOCTL_NETTYPE_INUSE:
        case ATHUSB_SUBIOCTL_NETWORK_TYPE:
        {
            return (-EINVAL);
        }
        case ATHUSB_SUBIOCTL_XMIT_DATARATE:
        {
            staTxRateSet(pDevInfo, (A_UINT16)Value);
            return (0);
        }
        case ATHUSB_SUBIOCTL_LIST_SCAN:
        {
            athusb_listscan(pDevInfo, 0);
            return (0);
        }
        case ATHUSB_SUBIOCTL_RADIO_ENABLE:
        {
            A_BOOL      enable = (Value) ? 1 : 0;
            A_STATUS    athStatus;

            if (!wlanDevPresent(pDevInfo)) {
                return NDIS_STATUS_NOT_ACCEPTED;
            }
            if (pDevInfo->localSta == NULL) {
                return (NDIS_STATUS_FAILURE);
            }

            pDevInfo->staConfig.radioEnable = enable;
            athStatus = enable ?  athDrvRadioEnable(pDevInfo, &pDevInfo->rfSilent.swRadioDisable) : athDrvRadioDisable(pDevInfo, &pDevInfo->rfSilent.swRadioDisable);

            if (athStatus != A_OK) {
            }

            return (0);
        }

        case ATHUSB_SUBIOCTL_ANTENNA_SWITCH:
        {
            ANTENNA_CONTROL         settings;
            CHAN_VALUES             *pChval    = pDevInfo->staConfig.phwChannel;

            settings = (A_UINT16)Value;
            if (settings >= ANTENNA_DUMMY_MAX) {
                return (NDIS_STATUS_NOT_ACCEPTED);
            } 
            
            wdcTargetSetConfiguration(pDevInfo->targetHandle,
                    CFG_DIVERSITY_CTL,
                    4,
                    (TARGET_CONFIG_VAL)settings,
                    NULL);

            return (0);
        }
	
	case ATHUSB_SUBIOCTL_REGULATORY:
	{
		if (!wlanVerifyUserRd(Value)) {
			return ( -EINVAL);
		}
		pDevInfo->staConfig.RegDomain = (A_UINT16) Value;
		pDevInfo->devCap.regDomain = (A_UINT16) Value;
		return (0);
	}
	
	case ATHUSB_SUBIOCTL_REGULATORY_DOMAIN:
        { 
           /* A_UCHAR          countryIsoName[3];
            countryIsoName[0] = (A_UINT16)Value >> 8;
            countryIsoName[1] = (A_UINT8)Value;
            countryIsoName[2] = '\0';*/
		A_UCHAR* countryIsoName;
		countryIsoName=wlanGetCountryCodeName((A_UINT16)Value);
            return (athusb_setregulatory(pDevInfo, countryIsoName));
        }
	case ATHUSB_SUBIOCTL_BKSCAN:
	{
		if(Value > 1){
			return (-EINVAL);
		}
		if(Value==pDevInfo->staConfig.bkScanEnable){
			return (0);
		}
		pDevInfo->staConfig.bkScanEnable=Value;
		break;
	}
        case ATHUSB_SUBIOCTL_CALIBRATION_TIME:
            if (Value < 0 || Value > 65535) {
                return (-EINVAL);
            }
            if (Value == pDevInfo->staConfig.calibrationTime) {
                return (0);
            }
            pDevInfo->staConfig.calibrationTime = Value;
            break;
        case ATHUSB_SUBIOCTL_TRIGGER_THRESHOLD:
            if (Value < 100 || Value > 32767) {
                return (-EINVAL);
            }
            if (Value == pDevInfo->staConfig.txTrigThresh) {
                return (0);
            }
            pDevInfo->staConfig.txTrigThresh = Value;
            break;
#ifdef WLAN_CONFIG_LEAP
	case ATHUSB_SUBIOCTL_LEAPENABLED:
	{
    		if (Value) {
			pDevInfo->staConfig.leapEnabled=TRUE;
		} else {
			pDevInfo->staConfig.leapEnabled=FALSE;
		}
		break;
	}
#endif
        case ATHUSB_SUBIOCTL_REG_UPDATE:
        {
            if (Value ==0 || Value ==1) {
                wlanApplyMKKIrreversibleUpdate(pDevInfo, Value, 0);
                return (0);
            }
            return (-EINVAL);
        }

	    case ATHUSB_SUBIOCTL_REGCAPBITS:
        {
            if (Value ==4 || Value ==6) {
                pDevInfo->staConfig.regCapBits = Value;
                wlanApplyMKKIrreversibleUpdate(pDevInfo, (pDevInfo->staConfig.regCapBits & EE_KK_ODD_U1_LEGACY), 
                                               pDevInfo->staConfig.regCapBits & EN_KK_UNDO_UPDATE);
                return (0);
            }
            return (-EINVAL);
        }
	
#ifdef DEBUG
        case ATHUSB_SUBIOCTL_POWERDEBUG:
        {
            extern int powerDebugLevel;
            if (Value < 0 || Value > 256) {
                return (-EINVAL);
            }
            powerDebugLevel = Value;
            return (0);
        }
        case ATHUSB_SUBIOCTL_RECEIVEDEBUG:
        {
            extern int rxDebugLevel;
            if (Value < 0 || Value > 256) {
                return (-EINVAL);
            }
            rxDebugLevel = Value;
            return (0);
        }
        case ATHUSB_SUBIOCTL_TRANSMITDEBUG:
        {
            extern int txDebugLevel;
            if (Value < 0 || Value > 256) {
                return (-EINVAL);
            }
            txDebugLevel = Value;
            return (0);
        }
        case ATHUSB_SUBIOCTL_MLMEDEBUG:
        {
            extern int mlmeDebugLevel;
            if (Value < 0 || Value > 256) {
                return (-EINVAL);
            }
            mlmeDebugLevel = Value;
            return (0);
        }
        case ATHUSB_SUBIOCTL_LOCKDEBUG:
        {
            extern int lockDebugLevel;
            if (Value < 0 || Value > 9) {
                return (-EINVAL);
            }
            lockDebugLevel = Value;
            return (0);
        }
        case ATHUSB_SUBIOCTL_CSERVDEBUG:
        {
            extern int cservDebugLevel;
            if (Value < 0 || Value > 9) {
                return (-EINVAL);
            }
            cservDebugLevel = Value;
            return (0);
        }
#endif
        default:
            break;
    }
    athDeviceReconfig(pDevInfo);
    return (0);
}

/*
 * Adjust station value for display if 11g mode.
 * If STA is connected with 11b rates, then show it as 11b channel
 * even if the channel flags are 11g.
 */
static A_UINT16
adjustStationFlagsForDisplay(WLAN_DEV_INFO *pDevInfo)
{
    /*
     * If sta is connected on a 11b/g channel, channel type will be
     * the connected AP's rate set
     */
    if (IS_CONN_OR_JOINED(pDevInfo)) {
        if (IS_CHAN_2GHZ(pDevInfo->staConfig.pChannel->channelFlags) &&
            !phyRateSubSetCheck(pDevInfo, WIRELESS_MODE_11g,
                                &pDevInfo->bssDescr->supportedRateSet,
                                 WLAN_PHY_OFDM))
        {
           /* No OFDM rates. Must be 11b */
           return (pDevInfo->staConfig.pChannel->channelFlags & ~(CHANNEL_TURBO | CHANNEL_OFDM));
        } else
        {
            return pDevInfo->staConfig.pChannel->channelFlags;
        }
    }
    return pDevInfo->staConfig.phwChannel->channelFlags;
}


int athusb_getvalue (struct net_device *pNetDevice,
        struct iw_request_info *pInfo,
        void *pData,
        char *pExtra)
{
    WLAN_DEV_INFO *pDevInfo = DEVINFO_FROM_CONTEXT(pNetDevice);
    int *ParamList = (int *)pExtra;
    int Parameter = ParamList[0];

    switch (Parameter) {
        case ATHUSB_SUBIOCTL_NETBAND:
            ParamList[0] = pDevInfo->staConfig.NetBand;
            break;
        case ATHUSB_SUBIOCTL_ADHOCBAND:
            ParamList[0] = pDevInfo->staConfig.AdhocBand;
            break;
        case ATHUSB_SUBIOCTL_SHORTPREAMBLE:
            ParamList[0] = pDevInfo->staConfig.shortPreamble;
            break;
        case ATHUSB_SUBIOCTL_ABOLT:
            ParamList[0] = pDevInfo->staConfig.abolt;
            break;
        case ATHUSB_SUBIOCTL_CALIBRATION_TIME:
            ParamList[0] = pDevInfo->staConfig.calibrationTime;
            break;
        case ATHUSB_SUBIOCTL_TRIGGER_THRESHOLD:
            ParamList[0] = pDevInfo->staConfig.txTrigThresh;
            break;
        case ATHUSB_SUBIOCTL_AUTHTYPE:
            ParamList[0] = pDevInfo->staConfig.authType-1;
            break;
        case ATHUSB_SUBIOCTL_ENCRYPTION:
            ParamList[0] = pDevInfo->staConfig.encryptionAlg;
            break;
        case ATHUSB_SUBIOCTL_WPA:
            ParamList[0] = pDevInfo->staConfig.wpaEnabled;
            break;
        case ATHUSB_SUBIOCTL_CHANNEL:
        {
            CHAN_VALUES *pChan;
            A_UINT16    configChannelFlag; 
            
            if (IS_CONN_OR_JOINED(pDevInfo)) {
                pChan = pDevInfo->staConfig.pChannel;
            } else {
                pChan = pDevInfo->staConfig.phwChannel;
            }

            if (pChan) {
                /* Adjust station value for display */
                configChannelFlag = adjustStationFlagsForDisplay(pDevInfo); 
                ParamList[0] = (configChannelFlag << 16) | pChan->channel;
            } else {
                ParamList[0] = 0;
            }
            break;
        }
        case ATHUSB_SUBIOCTL_XMIT_DATARATE:
        {
            if (pDevInfo->localSta) {
                ParamList[0] = staTxRateGet(pDevInfo, pDevInfo->localSta);
            } else {
                ParamList[0] = 0;
            }
            break;
        }
        case ATHUSB_SUBIOCTL_RECV_DATARATE:
        {
            if (pDevInfo->localSta) {
                ParamList[0] = staRxRateGet(pDevInfo, pDevInfo->localSta);
            } else {
                ParamList[0] = 0;
            }
            break;
        }
        case ATHUSB_SUBIOCTL_POWERSAVE_STATE:
        {
            switch (pDevInfo->powerMgmt.powerState) {
            case D0_STATE:
                ParamList[0] = pDevInfo->powerMgmt.powerSleepSubstate;
                break; 
            case D1_STATE:
                ParamList[0] = pDevInfo->powerMgmt.powerSleepSubstate;
                break; 
            case D2_STATE:
                ParamList[0] = STATE_SLEEP;
                break; 
            case D3_STATE:
                ParamList[0] = STATE_SLEEP;
                break;
            default:
                ParamList[0] = 5; // Unknown
            }
            break;
        }
        case ATHUSB_SUBIOCTL_SIGNAL_QUALITY:
        {
            SIB_ENTRY *pSib  = pDevInfo->localSta;
            A_UINT32  txRateVal;
            A_UINT8   txRate; 
            if (NULL == pSib) {
                ParamList[0] = 0;
                break;
            } 

            txRateVal = staTxRateKbpsGet(pDevInfo, pSib);
            txRate    = staRateValueToIndex(txRateVal, pSib);

            /* Returns the signal quality as a percentage. */
            ParamList[0] = staGetSigQuality(txRate, pSib);

            break;
        }
        case ATHUSB_SUBIOCTL_NOISE_FLOOR:
            ParamList[0] = pDevInfo->noiseFloor;
            break;
        case ATHUSB_SUBIOCTL_LIST_SCAN:
            return (-EINVAL);
        case ATHUSB_SUBIOCTL_DRIVER_VERSION:
            ParamList[0] = (0x05 << 8) | 0x01;
            break;
        case ATHUSB_SUBIOCTL_TRANSMIT_POWER:
            /* Returns the current half dBm txPower for 6mb Rate */
            ParamList[0] = pDevInfo->tx6PowerInHalfDbm;
            break;
        case ATHUSB_SUBIOCTL_RADIO_ENABLE:
            ParamList[0] = (A_BOOL) pDevInfo->rfSilent.swRadioDisable |
                (A_BOOL) pDevInfo->rfSilent.hwRadioDisable << 1;
            break;
        case ATHUSB_SUBIOCTL_VENDOR_VERSION:
            ParamList[0] = (ATH_SW_VER_MAJOR << 16) + ATH_SW_VER_MINOR;
            break;
        case ATHUSB_SUBIOCTL_NETWORK_TYPE:
            if (pDevInfo->bssDescr) {
                ParamList[0] = 
                    pDevInfo->bssDescr->bsstype == INFRASTRUCTURE_BSS ? 0 : 1;
            } else {
                ParamList[0] = 0;
            }
                return (0);
            break;
        case ATHUSB_SUBIOCTL_NETTYPE_INUSE:
        {
            PATH_802_11_NETWORK_TYPE pType =
                (PATH_802_11_NETWORK_TYPE)ParamList;

            if (IS_CONN_OR_JOINED(pDevInfo)) {
                if (IS_CHAN_B(pDevInfo->bssDescr->pChannel->channelFlags)) {
                    *pType = Ath802_11DS;
                } else {
                    if ((IS_CHAN_G(pDevInfo->bssDescr->pChannel->channelFlags)) ||
                            (IS_CHAN_108G(pDevInfo->bssDescr->pChannel->channelFlags)))
                    {
                        *pType = Ath802_11OFDM24;
                    } else {
                        *pType = Ath802_11OFDM5;
                    }
                }
            } else {
                if ((pDevInfo->staConfig.NetBand & MODE_SELECT_11B) == MODE_SELECT_11B) {
                    *pType = Ath802_11DS;
                } else {
                    if ((pDevInfo->staConfig.NetBand & MODE_SELECT_11G) == 
                            MODE_SELECT_11G)
                    {
                        *pType = Ath802_11OFDM24;
                    } else {
                        *pType = Ath802_11OFDM5;
                    }
                }
            }

            return (0);
        }
        case ATHUSB_SUBIOCTL_ANTENNA_SWITCH:
        {
            return (-EINVAL);
        }
	case ATHUSB_SUBIOCTL_REGULATORY:
	{
		ParamList[0]=pDevInfo->devCap.regDomain;
	}
        case ATHUSB_SUBIOCTL_REGULATORY_DOMAIN:
        {
            const A_UCHAR *pCcString;
	    A_UINT16 pCcValue;

            /* This now returns a two character code identifying the country */
            pCcString = wlanGetCountryCodeName(pDevInfo->staConfig.countryCode);
	    pCcValue=wlanGetCountryCodeByName(pCcString,FALSE);
	    ParamList[0]=pCcValue;
//            ParamList[0] = (pCcString[0] << 8) | pCcString[1];
//            printk ("Regulatory : %x\n", ParamList[0]);
            break;
        }
	case ATHUSB_SUBIOCTL_BKSCAN:
	{
		ParamList[0]=pDevInfo->staConfig.bkScanEnable;
		break;
	}
	case ATHUSB_SUBIOCTL_REGCAPBITS:
	{
		ParamList[0]=pDevInfo->staConfig.regCapBits;
		break;
    }
        case ATHUSB_SUBIOCTL_PRIVACY:
        {
            PRIVACY_FILTER *pFilter = (PRIVACY_FILTER *)pExtra;
            if (pDevInfo->staConfig.filter8021x) {
                *pFilter = PrivFilter8021xWEP;
            } else {
                *pFilter = PrivFilterAcceptAll;
            }
            break;
        }
#ifdef WLAN_CONFIG_LEAP
	case ATHUSB_SUBIOCTL_LEAPENABLED:
		ParamList[0]=pDevInfo->staConfig.leapEnabled;
		break;
#endif
		
#ifdef DEBUG
        case ATHUSB_SUBIOCTL_POWERDEBUG:
        {
            extern int powerDebugLevel;
            ParamList[0] = powerDebugLevel;
            break;
        }
        case ATHUSB_SUBIOCTL_RECEIVEDEBUG:
        {
            extern int rxDebugLevel;
            ParamList[0] = rxDebugLevel;
            break;
        }
        case ATHUSB_SUBIOCTL_TRANSMITDEBUG:
        {
            extern int txDebugLevel;
            ParamList[0] = txDebugLevel;
            break;
        }
        case ATHUSB_SUBIOCTL_MLMEDEBUG:
        {
            extern int mlmeDebugLevel;
            ParamList[0] = mlmeDebugLevel;
            break;
        }
        case ATHUSB_SUBIOCTL_LOCKDEBUG:
        {
            extern int lockDebugLevel;
            ParamList[0] = lockDebugLevel;
            break;
        }
        case ATHUSB_SUBIOCTL_CSERVDEBUG:
        {
            extern int cservDebugLevel;
            ParamList[0] = cservDebugLevel;
            break;
        }
#endif
        default:
            return (-EOPNOTSUPP);
            break;
    }

    return (0);
}

int athusb_setmlme (struct net_device *pNetDevice,
        struct iw_request_info *pInfo,
        void *pData,
        char *pExtra)
{
    WLAN_DEV_INFO *pDevInfo = DEVINFO_FROM_CONTEXT(pNetDevice);
    struct athusb_mlme *pMlme = (struct athusb_mlme *)pExtra;

    if (pMlme->mlmeRequest != MLME_REQUEST_DEAUTH &&
            pMlme->mlmeRequest != MLME_REQUEST_DISASSOC) {
        return (0);
    }
    if (GetDeviceStatus (pDevInfo) != NDIS_STATUS_SUCCESS) {
        /* Device is not yet ready - No need to send error message back */
        return (0);
    }
    if (IS_CONN_OR_JOINED(pDevInfo)) {
        /*
         * make sure only one cservXXX function call at one time
         */
        ATH_ACQUIRE_SPINLOCK_IRQ(pDevInfo->lock);

        /* Must be awake to access hardware */
        powerSet(pDevInfo, WAKE_UP, SET_PWR_TO_2, FALSE, 0);

        cservStopBss(pDevInfo, DISCONNECT_NOW, REMOVE_BSS, SEND_DEAUTH);
        // Return to previous power setting but turn off the radio
        // and record the reason why
        powerSet(pDevInfo, WAKE_UP, REDUCE_PWR_FROM_2, FALSE, 0);
        ATH_RELEASE_SPINLOCK_IRQ(pDevInfo->lock);
    }
    return 0;
}

int athusb_removekey(WLAN_DEV_INFO *pDevInfo, A_UINT32 KeyIndex)
{
    if (!wlanDevPresent(pDevInfo)) {
        uiPrintf("Can't remove key because device isn't ready\n");
        return (NDIS_STATUS_NOT_ACCEPTED);
    }
    
    /* XXX Only support First Four for now */
    if (KeyIndex > MAX_SHARED_KEYS) {
        return (NDIS_STATUS_INVALID_DATA);
    } 
    
    if (pDevInfo->staConfig.keys[KeyIndex].keyLength == 0 ||
            pDevInfo->keyTable == NULL) {
        return (NDIS_STATUS_SUCCESS);
    }

    ATH_ACQUIRE_SPINLOCK_IRQ(pDevInfo->lock);
    /* Must be awake to access hardware */
    powerSet(pDevInfo, WAKE_UP, SET_PWR_TO_3, FALSE, 0);

    /*
     *  We really have no other choice but to quiesce the txq
     *  so that the swretry state machine gets advanced to
     *  a dormant state.  This is ok since ripping the keys
     *  away is a traffic-dropping event anyway.
     */
    StopTransmit(pDevInfo, DO_CLEAR_TX_Q, DO_COMPLETE, DO_WAIT);

    /*
     *  XXX When we apply keymapping, we will have to do a mapping
     *  lookup here.  But for now, this works.
     */
    wlanKeyTableRemove(pDevInfo, (A_UINT16)KeyIndex);

    if (KeyIndex == pDevInfo->staConfig.defaultKey) {
        // we deleted the default key, disable wep
        pDevInfo->staConfig.defaultKey      = 0; // default from parse.c
        pDevInfo->staConfig.privacyInvoked  = FALSE;
    }
	memset(&pDevInfo->staConfig.keys[KeyIndex].keyVal, 0,
            pDevInfo->staConfig.keys[KeyIndex].keyLength);
    pDevInfo->staConfig.keys[KeyIndex].keyLength = 0;

    RestartTransmit(pDevInfo);
    // Return to previous power setting
    powerSet(pDevInfo, WAKE_UP, REDUCE_PWR_FROM_3, FALSE, 0); 
    ATH_RELEASE_SPINLOCK_IRQ(pDevInfo->lock);
    return (NDIS_STATUS_SUCCESS);
}


int athusb_setkey (struct net_device *pNetDevice,
        struct iw_request_info *pInfo,
        union iwreq_data *pData,
        char *pExtra)
{
    WLAN_DEV_INFO *pDevInfo = DEVINFO_FROM_CONTEXT(pNetDevice);
    struct athusb_wpakey *pWpaKey = (struct athusb_wpakey *)pExtra;
    PNDIS_802_11_KEY    pKeyDesc;
    int i;
    NDIS_STATUS Status = 0;


    if (pWpaKey->wpaAlgorithm == EncryptionDisabled || pWpaKey->KeyLength == 0){
        return athusb_removekey(pDevInfo, pWpaKey->KeyIndex);
    }

	if (pWpaKey->KeyLength > IW_ENCODING_TOKEN_MAX)
	{
		return (-EINVAL);
	}

	if (pWpaKey->SeqLength > IW_ENCODING_TOKEN_MAX)
	{
		return (-EINVAL);
	}

    pKeyDesc = A_DRIVER_MALLOC (sizeof(NDIS_802_11_KEY)+IW_ENCODING_TOKEN_MAX);
	memset(pKeyDesc, 0, sizeof(NDIS_802_11_KEY)+IW_ENCODING_TOKEN_MAX);

	memcpy(&pKeyDesc->KeyMaterial, pWpaKey->KeyData, pWpaKey->KeyLength);

	pKeyDesc->Length = sizeof(NDIS_802_11_KEY)+IW_ENCODING_TOKEN_MAX;
	pKeyDesc->KeyLength = pWpaKey->KeyLength;

	for (i = 0, pKeyDesc->KeyRSC = 0 ; i < pWpaKey->SeqLength ; i++)
		pKeyDesc->KeyRSC |= (pWpaKey->Sequence[i] << (i * 8));

	if (pWpaKey->KeyIndex == 0) /* pairwise key */
		pKeyDesc->KeyIndex = (1 << 31) | (1 << 30);
	else
		pKeyDesc->KeyIndex = pWpaKey->KeyIndex;
	
	if(pWpaKey->DefaultTx){
		pKeyDesc->KeyIndex |= (1 << 31);
	}

    memcpy(&pKeyDesc->BSSID, pWpaKey->Address, ETH_ALEN);

    /*
     * This is one of the OIDs we must awaken for
     * always, even if radio is disabled
     */
    ATH_ACQUIRE_SPINLOCK_IRQ(pDevInfo->lock);
    powerSet(pDevInfo, WAKE_UP, SET_PWR_TO_3, FALSE, 0); 
    if (athKeyPlumb(pDevInfo, pKeyDesc, ADD_KEY) != A_OK) {
        Status = (-1);
    }
    // Return to previous power setting
    powerSet(pDevInfo, WAKE_UP, REDUCE_PWR_FROM_3, FALSE, 0);
    ATH_RELEASE_SPINLOCK_IRQ(pDevInfo->lock);

	memcpy(&pDevInfo->staConfig.keys[pWpaKey->KeyIndex].keyVal,
            pWpaKey->KeyData, pWpaKey->KeyLength);
    pDevInfo->staConfig.keys[pWpaKey->KeyIndex].keyLength = pWpaKey->KeyLength;

    A_DRIVER_FREE(pKeyDesc, 0);
    return (Status);
}

int athusb_setcountry (struct net_device *pNetDevice,
        struct iw_request_info *pInfo,
        char *pCountry,
        char *pExtra)
{
    WLAN_DEV_INFO    *pDevInfo = DEVINFO_FROM_CONTEXT(pNetDevice);
    A_UCHAR          countryIsoName[3];

    countryIsoName[0] = pCountry[0];
    countryIsoName[1] = pCountry[1];
    countryIsoName[2] = '\0';

    return (athusb_setregulatory(pDevInfo, countryIsoName));
}

int athusb_getcountry (struct net_device *pNetDevice,
        struct iw_request_info *pInfo,
        struct iw_point *pData,
        char *pCountry)
{
    const A_UCHAR *pCcString; 
    WLAN_DEV_INFO    *pDevInfo = DEVINFO_FROM_CONTEXT(pNetDevice);
    
    /* This now returns a two character code identifying the country */
    pCcString    = wlanGetCountryCodeName(pDevInfo->staConfig.countryCode);
    pCountry[0] = pCcString[0];
    pCountry[1] = pCcString[1];
    pCountry[2] = '\0';
    pData->length = 3;


    return (0);
}

static int
countBitsSet(unsigned int x)
{
    int cnt = 0;

    while (x) {
        cnt++;
        x &= x - 1;
    }

    return cnt;
}

int athusb_getnetwork_list (struct net_device *pNetDevice,
        struct iw_request_info *pInfo,
        struct iw_point *pData,
        char *pExtra)
{
    WLAN_DEV_INFO    *pDevInfo = DEVINFO_FROM_CONTEXT(pNetDevice);
    A_UINT32 mode =  pDevInfo->staConfig.NetBand;

    ATH_802_11_NETWORK_TYPE_LIST *pList =
        (ATH_802_11_NETWORK_TYPE_LIST *)pExtra; 

    pList->NumberOfItems = 0;
    pList->NetworkType[pList->NumberOfItems++] = Ath802_11OFDM5;

    if (mode & MODE_SELECT_11A) {
        pList->NetworkType[pList->NumberOfItems++] = Ath802_11OFDM5;
    }

    if (mode & (MODE_SELECT_11G | MODE_SELECT_108G)) {
        pList->NetworkType[pList->NumberOfItems++] = Ath802_11OFDM24;
    }

    if (mode & MODE_SELECT_11B) {
        pList->NetworkType[pList->NumberOfItems++] = Ath802_11DS;
    }

    pData->length = sizeof(ATH_802_11_NETWORK_TYPE_LIST) +
        sizeof(ATH_802_11_NETWORK_TYPE) * countBitsSet(mode);

    return NDIS_STATUS_SUCCESS;
}

int athusb_getdevinfo (struct net_device *pNetDevice,
        struct iw_request_info *pInfo,
        struct iw_point *pData,
        char *pExtra)
{
    WLAN_DEV_INFO    *pDevInfo = DEVINFO_FROM_CONTEXT(pNetDevice);
    PAtheros5000_DeviceInfo pE  = (PAtheros5000_DeviceInfo)pExtra;

    /*
     * ACU need deviceID greater than 18 to support SuperG,
     * this is wrong strategy because USB don't have PCI ID.
     * For ACU, it should ask driver to report its capability,
     * we may run new driver on old hardware or old driver
     * on new hardware, device ID isn't sufficient information
     * to determine SuperG capability
     */
    pE->AthVendorID             = 0x0cf3;
    pE->AthDeviceID             = 19;
    pE->AthSubVendorID          = 0;
    pE->AthSubVendor_DeviceID   = 0;
    pE->AthMacRev               = ((USHORT)pDevInfo->devCap.macVersion << 4) | (USHORT)pDevInfo->devCap.macRevision;
    pE->AthPhyRev               = (USHORT)pDevInfo->devCap.phyRevision;
    pE->AthAnalog5GhzRev        = (USHORT)pDevInfo->devCap.analog5GhzRevision;
    pE->AthAnalog2GhzRev        = (USHORT)pDevInfo->devCap.analog2GhzRevision;
    pE->AthHwCapWirelessModes   = (USHORT)pDevInfo->devCap.wirelessModes;
    pE->AthInfraWirelessModes   = wlanGetWirelessMode(pDevInfo,
            pDevInfo->staConfig.countryCode);
    pE->AthAdhocWirelessModes   = wlanGetAdhocWirelessMode(pDevInfo,
            pDevInfo->staConfig.countryCode);
    pE->AthNicSKU               = wlanGetSKU(pDevInfo);

    pData->length = sizeof(Atheros5000_DeviceInfo);

    return NDIS_STATUS_SUCCESS;
}

int athusb_getathlist (struct net_device *pNetDevice,
        struct iw_request_info *pInfo,
        struct iw_point *pData,
        char *pExtra)
{
    WLAN_DEV_INFO    *pDevInfo = DEVINFO_FROM_CONTEXT(pNetDevice);
    Atheros5000_BssInfo *pTheirs;
    CSERV_INFO          *pCsInfo = pDevInfo->pCsInfo;
    BSSDESCR_SET        *pBssSet;
    A_INT32             i, count;
    
    if (pCsInfo == NULL) {
        pData->length = 0;
        return (0);
    }

    pBssSet  = &pCsInfo->scanInfo.bssSet;
    count    = pBssSet->count; 

    if (pData->length < sizeof(*pTheirs) * count) {
        return (-ENOMEM);
    } 
    
    if (!wlanDevPresent(pDevInfo)) {
        count = 0;
    } 
    
    for (i = 0; i < count; i++) {
        BSSDESCR    *pOurs   = (BSSDESCR *)&pBssSet->bssDescrArray[i];
        WLAN_CFLAGS channelFlags;

        pTheirs = (Atheros5000_BssInfo *)pExtra + i;

        pTheirs->AtimWindow       = pOurs->ibssParamSet.atimWindow;
        pTheirs->BeaconInterval   = pOurs->beaconInterval;

        /* Set the reported channelFlags correctly for an 11g/11b AP by
         * examining the channel flags in the BSS
         */
        channelFlags = pOurs->pChannel->channelFlags; 
        
        if (IS_CHAN_NON_108G_2GHZ(channelFlags)) {
            channelFlags = (pOurs->opMode == WIRELESS_MODE_11g) ?
                CHANNEL_G : CHANNEL_B;
        }

        pTheirs->Channel          = ((ULONG)channelFlags) << 16 |
            pOurs->pChannel->channel;
        pTheirs->NetworkType      = pOurs->bsstype == INFRASTRUCTURE_BSS ? 0 : 1;
        pTheirs->Privacy          = pOurs->capabilityInfo.privacy;
        pTheirs->Rssi             = pOurs->rssi; 
        
        A_MEM_ZERO(&pTheirs->Ssid, Atheros5000_BssInfo_Ssid_SIZE);
        A_BCOPY(&pOurs->ssid.ssid, &pTheirs->Ssid, pOurs->ssid.length);
        A_BCOPY(&pOurs->supportedRateSet, &pTheirs->SupportedRates, 8);
        A_BCOPY(&pOurs->bssId.octets, &pTheirs->Bssid, 6);
        pTheirs->version = ATH_BSS_INFO_VERSION_CURRENT;
        pTheirs->xrCapable = FALSE;
        pTheirs->xrCurrent = FALSE;
        if (VALID_ATH_ADVCAP_ELEMENT(&pOurs->athAdvCapElement)) {
            pTheirs->superCap = IS_SUPER_FEATURE_ON(&pOurs->athAdvCapElement.info);
        } else {
            pTheirs->superCap = 0;
        }
    }

    pData->length = count * sizeof(*pTheirs);
    return NDIS_STATUS_SUCCESS;
}

static const struct iw_priv_args athusb_privargs[] = {
    { ATHUSB_IOCTL_GET_ATH_LIST,
        0, IW_PRIV_TYPE_BYTE | 2047, "get_athlist" },
    { ATHUSB_IOCTL_SETMLME,
        IW_PRIV_TYPE_BYTE | IW_PRIV_SIZE_FIXED | sizeof(struct athusb_mlme),
        0, "setmlme" },
    { ATHUSB_IOCTL_GET_DEVICE_INFO,
        0, IW_PRIV_TYPE_BYTE | 1024, "get_devinfo" },
    { ATHUSB_IOCTL_SETKEY,
        IW_PRIV_TYPE_BYTE | IW_PRIV_SIZE_FIXED | sizeof(struct athusb_wpakey),
        0, "wpakey" },
    { ATHUSB_IOCTL_GET_NETSUPPORT,
        0, IW_PRIV_TYPE_BYTE | 1024, "get_netlist" },
    { ATHUSB_IOCTL_SET_COUNTRY,
        IW_PRIV_TYPE_CHAR | IW_PRIV_SIZE_FIXED | 2, 0, "country" },
    { ATHUSB_IOCTL_GET_COUNTRY,
        0, IW_PRIV_TYPE_CHAR |IW_PRIV_SIZE_FIXED | 2, "get_country" },
    { ATHUSB_IOCTL_SET_VALUE,
        IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 2, 0, "value" },
    { ATHUSB_IOCTL_GET_VALUE,
        IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1,
        IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1, "get_value" },
    { ATHUSB_IOCTL_SET_VALUE,
        IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1, 0, "" },
    { ATHUSB_IOCTL_GET_VALUE,
        0, IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1, "" },
    { ATHUSB_SUBIOCTL_NETBAND,
        IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1, 0, "netband" },
    { ATHUSB_SUBIOCTL_NETBAND,
        0, IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1, "get_netband" },
    { ATHUSB_SUBIOCTL_ADHOCBAND,
        IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1, 0, "adhocband" },
    { ATHUSB_SUBIOCTL_ADHOCBAND,
        0, IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1, "get_adhocband" },
    { ATHUSB_SUBIOCTL_SHORTPREAMBLE,
        IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1, 0, "spreamble" },
    { ATHUSB_SUBIOCTL_SHORTPREAMBLE,
        0, IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1, "get_spreamble" },
    { ATHUSB_SUBIOCTL_ABOLT,
        IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1, 0, "abolt" },
    { ATHUSB_SUBIOCTL_ABOLT,
        0, IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1, "get_abolt" },
    { ATHUSB_SUBIOCTL_CALIBRATION_TIME,
        IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1, 0, "calibtime" },
    { ATHUSB_SUBIOCTL_CALIBRATION_TIME,
        0, IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1, "get_calibtime" },
    { ATHUSB_SUBIOCTL_TRIGGER_THRESHOLD,
        IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1, 0, "triggeradj" },
    { ATHUSB_SUBIOCTL_TRIGGER_THRESHOLD,
        0, IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1, "get_triggeradj" },
    { ATHUSB_SUBIOCTL_AUTHTYPE,
        IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1, 0, "authtype" },
    { ATHUSB_SUBIOCTL_AUTHTYPE,
        0, IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1, "get_authtype" },
    { ATHUSB_SUBIOCTL_PRIVACY,
        IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1, 0, "privacy" },
    { ATHUSB_SUBIOCTL_PRIVACY,
        0, IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1, "get_privacy" },
    { ATHUSB_SUBIOCTL_ENCRYPTION,
        IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1, 0, "keyalg" },
    { ATHUSB_SUBIOCTL_ENCRYPTION,
        0, IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1, "get_keyalg" },
    { ATHUSB_SUBIOCTL_WPA,
        IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1, 0, "wpa" },
    { ATHUSB_SUBIOCTL_WPA,
        0, IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1, "get_wpa" },
    { ATHUSB_SUBIOCTL_CHANNEL,
        IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1, 0, "channel" },
    { ATHUSB_SUBIOCTL_CHANNEL,
        0, IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1, "get_channel" },
    { ATHUSB_SUBIOCTL_XMIT_DATARATE,
        IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1, 0, "txrate" },
    { ATHUSB_SUBIOCTL_XMIT_DATARATE,
        0, IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1, "get_txrate" },
    { ATHUSB_SUBIOCTL_RECV_DATARATE,
        IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1, 0, "rxrate" },
    { ATHUSB_SUBIOCTL_RECV_DATARATE,
        0, IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1, "get_rxrate" },
    { ATHUSB_SUBIOCTL_POWERSAVE_STATE,
        IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1, 0, "powerstate" },
    { ATHUSB_SUBIOCTL_POWERSAVE_STATE,
        0, IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1, "get_powerstate" },
    { ATHUSB_SUBIOCTL_SIGNAL_QUALITY,
        IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1, 0, "sigquality" },
    { ATHUSB_SUBIOCTL_SIGNAL_QUALITY,
        0, IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1, "get_sigquality" },
    { ATHUSB_SUBIOCTL_NOISE_FLOOR,
        IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1, 0, "noisefloor" },
    { ATHUSB_SUBIOCTL_NOISE_FLOOR,
        0, IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1, "get_noisefloor" },
    { ATHUSB_SUBIOCTL_LIST_SCAN,
        IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1, 0, "scan" },
    { ATHUSB_SUBIOCTL_LIST_SCAN,
        0, IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1, "get_scan" },
    { ATHUSB_SUBIOCTL_DRIVER_VERSION,
        IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1, 0, "version" },
    { ATHUSB_SUBIOCTL_DRIVER_VERSION,
        0, IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1, "get_version" },
    { ATHUSB_SUBIOCTL_TRANSMIT_POWER,
        IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1, 0, "txpower" },
    { ATHUSB_SUBIOCTL_TRANSMIT_POWER,
        0, IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1, "get_txpower" },
    { ATHUSB_SUBIOCTL_RADIO_ENABLE,
        IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1, 0, "radio" },
    { ATHUSB_SUBIOCTL_RADIO_ENABLE,
        0, IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1, "get_radio" },
    { ATHUSB_SUBIOCTL_VENDOR_VERSION,
        IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1, 0, "vendor" },
    { ATHUSB_SUBIOCTL_VENDOR_VERSION,
        0, IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1, "get_vendor" },
    { ATHUSB_SUBIOCTL_NETTYPE_INUSE,
        IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1, 0, "nettype_use" },
    { ATHUSB_SUBIOCTL_NETTYPE_INUSE,
        0, IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1, "get_nettype_use" },
    { ATHUSB_SUBIOCTL_NETWORK_TYPE,
        IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1, 0, "nettype" },
    { ATHUSB_SUBIOCTL_NETWORK_TYPE,
        0, IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1, "get_nettype" },
    { ATHUSB_SUBIOCTL_ANTENNA_SWITCH,
        IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1, 0, "antenna" },
    { ATHUSB_SUBIOCTL_ANTENNA_SWITCH,
        0, IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1, "get_antenna" },
    { ATHUSB_SUBIOCTL_REGULATORY,
        IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1, 0, "regulatory" },
    { ATHUSB_SUBIOCTL_REGULATORY,
        0, IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1, "get_regulatory" },
    { ATHUSB_SUBIOCTL_REGULATORY_DOMAIN,
        IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1, 0, "regdomain" },
    { ATHUSB_SUBIOCTL_REGULATORY_DOMAIN,
        0, IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1, "get_regdomain" },

#ifdef WLAN_CONFIG_LEAP
    { ATHUSB_SUBIOCTL_LEAPENABLED,
	IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1, 0, "leapenabled" },
    { ATHUSB_SUBIOCTL_LEAPENABLED,
	0, IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1, "get_leapenabled" },
#endif
    { ATHUSB_SUBIOCTL_BKSCAN,
        IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1, 0, "bkscan" },
    { ATHUSB_SUBIOCTL_BKSCAN,
        0, IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1, "get_bkscan" },
    { ATHUSB_SUBIOCTL_REG_UPDATE,
        IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1, 0, "eepromupdate" },
    { ATHUSB_SUBIOCTL_REGCAPBITS,
        IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1, 0, "regcapbits" },
    { ATHUSB_SUBIOCTL_REGCAPBITS,
        0, IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1, "get_regcapbits" },
#ifdef DEBUG
    { ATHUSB_SUBIOCTL_POWERDEBUG,
        IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1, 0, "powerdebug" },
    { ATHUSB_SUBIOCTL_POWERDEBUG,
        0, IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1, "get_powerdebug" },
    { ATHUSB_SUBIOCTL_RECEIVEDEBUG,
        IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1, 0, "rxdebug" },
    { ATHUSB_SUBIOCTL_RECEIVEDEBUG,
        0, IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1, "get_rxdebug" },
    { ATHUSB_SUBIOCTL_TRANSMITDEBUG,
        IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1, 0, "txdebug" },
    { ATHUSB_SUBIOCTL_TRANSMITDEBUG,
        0, IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1, "get_txdebug" },
    { ATHUSB_SUBIOCTL_MLMEDEBUG,
        IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1, 0, "mlmedebug" },
    { ATHUSB_SUBIOCTL_MLMEDEBUG,
        0, IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1, "get_mlmedebug" },
    { ATHUSB_SUBIOCTL_LOCKDEBUG,
        IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1, 0, "lockdebug" },
    { ATHUSB_SUBIOCTL_LOCKDEBUG,
        0, IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1, "get_lockdebug" },
    { ATHUSB_SUBIOCTL_CSERVDEBUG,
        IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1, 0, "cservdebug" },
    { ATHUSB_SUBIOCTL_CSERVDEBUG,
        0, IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1, "get_cservdebug" },
#endif
};

static const iw_handler athusb_handler[] = {
    (iw_handler) NULL,                /* SIOCSIWCOMMIT */
    (iw_handler) athusb_giwname,      /* SIOCGIWNAME */
    (iw_handler) NULL,                /* SIOCSIWNWID */
    (iw_handler) NULL,                /* SIOCGIWNWID */
    (iw_handler) athusb_siwfreq,      /* SIOCSIWFREQ */
    (iw_handler) athusb_giwfreq,      /* SIOCGIWFREQ */
    (iw_handler) athusb_siwmode,      /* SIOCSIWMODE */
    (iw_handler) athusb_giwmode,      /* SIOCGIWMODE */
    (iw_handler) NULL,                /* SIOCSIWSENS */
    (iw_handler) NULL,                /* SIOCGIWSENS */
    (iw_handler) NULL,                /* SIOCSIWRANGE */
    (iw_handler) athusb_giwrange,     /* SIOCGIWRANGE */
    (iw_handler) NULL,                /* SIOCSIWPRIV */
    (iw_handler) NULL,                /* SIOCGIWPRIV */
    (iw_handler) NULL,                /* SIOCSIWSTATS */
    (iw_handler) NULL,                /* SIOCGIWSTATS */
    (iw_handler) NULL,                /* SIOCSIWSPY */
    (iw_handler) NULL,                /* SIOCGIWSPY */
    (iw_handler) NULL,                /* ** HOLE ** */
    (iw_handler) NULL,                /* ** HOLE ** */
    (iw_handler) athusb_siwap,        /* SIOCSIWAP */
    (iw_handler) athusb_giwap,        /* SIOCGIWAP */
    (iw_handler) NULL,                /* -- hole -- */
    (iw_handler) NULL,                /* SIOCGIWAPLIST */
#ifdef SIOCGIWSCAN
    (iw_handler) athusb_siwscan,      /* SIOCSIWSCAN */
    (iw_handler) athusb_giwscan,      /* SIOCGIWSCAN */
#else
    (iw_handler) NULL,                /* SIOCSIWSCAN */
    (iw_handler) NULL,                /* SIOCGIWSCAN */
#endif /* SIOCGIWSCAN */
	(iw_handler) athusb_siwessid,     /* SIOCSIWESSID */
    (iw_handler) athusb_giwessid,     /* SIOCGIWESSID */
    (iw_handler) NULL,                /* SIOCSIWNICKN */
    (iw_handler) NULL,                /* SIOCGIWNICKN */
    (iw_handler) NULL,                /* ** HOLE ** */
    (iw_handler) NULL,                /* ** HOLE ** */
    (iw_handler) athusb_siwrate,      /* SIOCSIWRATE */
    (iw_handler) athusb_giwrate,      /* SIOCGIWRATE */
    (iw_handler) athusb_siwrts,       /* SIOCSIWRTS */
    (iw_handler) athusb_giwrts,       /* SIOCGIWRTS */
    (iw_handler) athusb_siwfrag,      /* SIOCSIWFRAG */
    (iw_handler) athusb_giwfrag,      /* SIOCGIWFRAG */
    (iw_handler) athusb_siwtxpow,     /* SIOCSIWTXPOW */
    (iw_handler) athusb_giwtxpow,     /* SIOCGIWTXPOW */
    (iw_handler) athusb_siwretry,     /* SIOCSIWRETRY */
    (iw_handler) athusb_giwretry,     /* SIOCGIWRETRY */
    (iw_handler) athusb_siwencode,    /* SIOCSIWENCODE */
    (iw_handler) athusb_giwencode,    /* SIOCGIWENCODE */
    (iw_handler) athusb_siwpower,     /* SIOCSIWPOWER */
    (iw_handler) athusb_giwpower,     /* SIOCGIWPOWER */
};

static const iw_handler athusb_privhandler[] = {
    (iw_handler) athusb_setvalue,           /* SIOCIWFIRSTPRIV + 0 */
    (iw_handler) athusb_getvalue,           /* SIOCIWFIRSTPRIV + 1 */
    (iw_handler) athusb_setmlme,            /* SIOCIWFIRSTPRIV + 2 */
    (iw_handler) athusb_getdevinfo,         /* SIOCIWFIRSTPRIV + 3 */
    (iw_handler) athusb_setkey,             /* SIOCIWFIRSTPRIV + 4 */
    (iw_handler) athusb_getathlist,         /* SIOCIWFIRSTPRIV + 5 */
    (iw_handler) NULL,                      /* SIOCIWFIRSTPRIV + 6 */
    (iw_handler) athusb_getnetwork_list,    /* SIOCIWFIRSTPRIV + 7 */ 
    (iw_handler) athusb_setcountry,         /* SIOCIWFIRSTPRIV + 8 */
    (iw_handler) athusb_getcountry,         /* SIOCIWFIRSTPRIV + 9 */
};

static struct iw_handler_def athusb_iw_handler_def = {
    num_standard:       sizeof(athusb_handler)/sizeof(iw_handler),
    num_private:        sizeof(athusb_privhandler)/sizeof(iw_handler),
    standard:           (iw_handler *)athusb_handler,
    private:            (iw_handler *)athusb_privhandler,
    private_args:       (struct iw_priv_args *) athusb_privargs,
    num_private_args:   sizeof(athusb_privargs) / sizeof(struct iw_priv_args),
};

static NDIS_STATUS GetDeviceStatus (WLAN_DEV_INFO *pDevInfo)
{
    if (!athDrvRadioStatus (pDevInfo, FALSE)) {
        return (NDIS_STATUS_NOT_ACCEPTED);
    }

    if (!wlanDevPresent(pDevInfo)) {
        return (NDIS_STATUS_NOT_ACCEPTED);
    }

    if (pDevInfo->pOSHandle->openForBusiness == FALSE) {
        return (NDIS_STATUS_NOT_ACCEPTED);
    }
    return (NDIS_STATUS_SUCCESS);
}

static void athDeviceReconfig (WLAN_DEV_INFO *pDevInfo)
{
    if (GetDeviceStatus (pDevInfo) == NDIS_STATUS_SUCCESS) {

        ATH_ACQUIRE_SPINLOCK_IRQ(pDevInfo->lock);
        // Must be awake to access hardware
        powerSet(pDevInfo, WAKE_UP, SET_PWR_TO_3, FALSE, 0);
        ATH_RELEASE_SPINLOCK_IRQ(pDevInfo->lock);

        athShutdownDriver(pDevInfo, CONFIG_EVENT);

        ATH_ACQUIRE_SPINLOCK_IRQ(pDevInfo->lock);
#if 0
        {
            WLAN_STA_CONFIG *pStaConfig = &pDevInfo->staConfig;
            int i;
        ssidArrayInit(&pStaConfig->cfgSsids);
        ssidArrayAdd(&pStaConfig->cfgSsids,"", 0);
        ssidArrayAdd(&pStaConfig->cfgSsids,"", 0);
        ssidArrayAdd(&pStaConfig->cfgSsids,"", 0);

        pStaConfig->privacyInvoked = 1;
        pStaConfig->defaultKey = 0;
        pStaConfig->leapEnabled = 0;
        pStaConfig->leapInDriver = FALSE;
        pStaConfig->wpaEnabled = FALSE;
        pStaConfig->authType = 1;
        pStaConfig->encryptionAlg = 6;
        pStaConfig->mixedPrivacyAllow = 0;

        for (i = 0; i < MAX_PREFERRED_APS; i++) {
            memset(&pStaConfig->prefBssids[i], 0, 6);
        }

        }
#endif
    
        /* Some of the parameters are passsed onto wdc in setdevconfig
         * For dynamic restarting we need to call this everytime */
        athSetDevConfig(pDevInfo);

        athRestartDriver(pDevInfo);
        powerSet(pDevInfo, WAKE_UP, REDUCE_PWR_FROM_3, FALSE, 0);
        ATH_RELEASE_SPINLOCK_IRQ(pDevInfo->lock);
    }
}

NDIS_STATUS
athSetPacketFilter(IN WLAN_DEV_INFO *pDevInfo, IN ULONG newPacketFilter)
{
    ULONG           newParameterField = 0;

    // Verify bits, if any bits are set that we don't support, leave
    if (newPacketFilter & ~(NDIS_PACKET_TYPE_DIRECTED |
                            NDIS_PACKET_TYPE_MULTICAST |
                            NDIS_PACKET_TYPE_BROADCAST |
                            NDIS_PACKET_TYPE_ALL_MULTICAST))
    {
        return NDIS_STATUS_NOT_SUPPORTED;
    }

    /*
     * if NDIS is setting the unicast filter, get ready to receive
     * beacons and probe requests also
     */
    if (newPacketFilter & NDIS_PACKET_TYPE_DIRECTED) {
        newParameterField |= RX_UCAST;
        newParameterField |= RX_BEACON;
        newParameterField |= RX_PROBE_REQ;
    }
    /*
     * Receive all multi/broadcasts to allow sleep after CAB
     */
    newParameterField |= (RX_BCAST | RX_MCAST);

    // Give it to the NIC and save it, as well as NDISs' version
    wlanRxFilter(pDevInfo, newParameterField, RX_FILTER_SET);
    uiPrintf("athSetPacketFilter: new h/w value = %04x\n", newParameterField);

    pDevInfo->NdisFilterReg = newPacketFilter;

    /*
     * Turn on all the mcast filter bits to receive all such frames
     */
    wdcInitRxMulticastFilter(pDevInfo->targetHandle, FALSE);

    return NDIS_STATUS_SUCCESS;
}

#ifdef WLAN_CONFIG_WPA
A_STATUS
leapInit(WLAN_DEV_INFO *pDevInfo) {
    return (0);
}

void leapTimerSvc (void *context) {
}

void
leapAssocCallback(WLAN_DEV_INFO *pDevInfo, SIB_ENTRY *pSib) {
}

void
leapReceive(WLAN_DEV_INFO *pDevInfo, ATHEROS_DESC *pDesc) {
    freeBuffandDesc(pDevInfo, pDesc);
}
#endif

/*
 * Plumb an encryption key into the hardware.  Follow the
 * cookbook encoded into the NDIS_802_11_KEY structure.
 */
A_STATUS
athKeySet(WLAN_DEV_INFO *pDevInfo)
{
    WLAN_PRIV_RECORD    *pKey;
    A_UINT16            keyIndex;
    A_STATUS            Status = A_OK;
    SIB_ENTRY           *pSib;
    A_STATUS            status;
    WLAN_PRIV_RECORD *pTempKey;
    WLAN_MACADDR nullMacAddr;
    WLAN_PRIV_RECORD *oldKey;

    memset (&nullMacAddr, 0, sizeof (nullMacAddr));
    pKey = &pDevInfo->keyTable[0];
    keyIndex = 0;

    /* Clear the key table entry */
    wlanKeyTableSlotFree(pKey);
    pKey->keyFlags = 0; 
    
    /* Clean up previously BSS key if not erased already */
    oldKey = &pDevInfo->keyTable[pDevInfo->staConfig.defaultKey];
    wlanDeleteBssKey(pDevInfo, pDevInfo->staConfig.defaultKey, oldKey);

    pDevInfo->staConfig.defaultKey = keyIndex;

    /* Install the key as connection if the transmitUsage is set */
    /* Look up the AP's SIB */
    pSib = sibEntryFind(pDevInfo, &pDevInfo->baseBss.bssId, NULL);
    ASSERT(pSib); 
    
    pTempKey = &pDevInfo->staConfig.keys[0];
    /* Set the key */
    status = wlanKeyTableSet(pDevInfo, pSib,
            1, 0, (A_UINT16)0,
            pTempKey->keyLength * 8,
            pTempKey->keyVal);
    if (status != A_OK) {
        return status;
    } 
    
    pKey->keyType = PRIV_KEY_TYPE_WEP;
    pKey->keyFlags |= PRIV_UNIQUEKEY;
    pKey->keyFlags &= ~PRIV_FRAMEKEY;
    pSib->pPriv = pKey; 
    
    wlanInstallConnectionKey(pDevInfo, pSib, &pDevInfo->bssDescr->bssId, pKey);

    /* Enable compression now that we have the pairwise keys */
    if (VALID_ATH_ADVCAP_ELEMENT(&pSib->athAdvCapElement) &&
            pSib->athAdvCapElement.info.useCompression) {
        A_UINT32 attribValue = TRUE;

        wdcUpdateConnectionAttribute(pDevInfo->targetHandle,
                pSib->wdcConnId,
                CONN_COMPRESSION,
                sizeof(attribValue),
                (TARGET_CONFIG_VAL)&attribValue);
    }
    /* Install BSS key */
    wlanInstallBssKey(pDevInfo, (A_UINT32)0, 1, pKey);
    ccxKeyPlumbedCallback(pDevInfo);

    return Status;
}

/*
 * Plumb an encryption key into the hardware.  Follow the
 * cookbook encoded into the NDIS_802_11_KEY structure.
 */
A_STATUS
athKeyPlumb(WLAN_DEV_INFO *pDevInfo, void *pBuf, KEY_STRUCT_TYPE type)
{
    WLAN_PRIV_RECORD    *pKey;
    A_UINT16            keyIndex;
    A_STATUS            Status = A_OK;
    SIB_ENTRY           *pSib;
    A_UINT16            i;
    A_UINT32            cipher = 0;
    PNDIS_802_11_KEY    pKeyDesc;
    PNDIS_802_11_WEP    pWepKeyDesc;
    A_STATUS            status;
    SIB_ENTRY            *lSib = pDevInfo->localSta;

    NDIS_KEY_PROPS      *nkp;
	
    A_BOOL              connectionKey = 0;
    A_BOOL              bssKey = 0;

    A_BOOL              contextWpa = 0;
    A_BOOL              context802_1x = 0;
    A_BOOL              contextStaticWEP = 0;


    /* ADD_WEP and ADD_KEY buffers share the same header */
    pKeyDesc = (PNDIS_802_11_KEY)pBuf;
    nkp = (NDIS_KEY_PROPS *)&pKeyDesc->KeyIndex;

    /* Determine key management context */
    if (pDevInfo->staConfig.wpaEnabled) {
        contextWpa = TRUE;
    } else {
        /* WEP keys. 802.1x or static ? */
        if (IS_STATE_CONNECTED(pDevInfo)) {
            context802_1x = TRUE;
        } else {
            contextStaticWEP = TRUE;
        }
    }

    /* Determine the key range */
    /* XXX Divy Test to be optimized after tests for 802.1X... */
    if (nkp->pairWise) {
        connectionKey = TRUE;
        bssKey = FALSE;
    } else {
        if ((context802_1x) && (nkp->usageTransmit)) {
            connectionKey = TRUE;
            bssKey = FALSE;
        } else {
            connectionKey = FALSE;
            bssKey = TRUE;
        }
    }

    if (contextWpa) {
        uiPrintf("athKeyPlumb: WPA context\n");
    } else if ((context802_1x) && (connectionKey)) {
        uiPrintf("athKeyPlumb: 802.1x connection key\n");
    } else if ((context802_1x) && (!connectionKey)) {
        uiPrintf("athKeyPlumb: 802.1x BSS key\n");
    } else if ((contextStaticWEP) && (connectionKey)) {
        uiPrintf("athKeyPlumb: static WEP connection key\n");
    } else if ((contextStaticWEP) && (!connectionKey)) {
        uiPrintf("athKeyPlumb: static WEP BSS key\n");
    } else {
        uiPrintf("athKeyPlumb: unknown context\n");
    }

    /*
     * WPA context.
     * Assume we're in a ESS. One connection only at a time.
     */
    if (contextWpa) {
        /* Sanity checks */
        /* We require ADD_KEY for WPA */
        if (type == ADD_WEP) {
            return A_ERROR;
        }


        /* WPA spec sez to ignore Transmit bit for group keys */
        if (!nkp->pairWise) {
            nkp->usageTransmit = FALSE;
        }

        if ((!nkp->usageTransmit) && (nkp->pairWise)) {
            return A_ERROR;
        }

        if ((nkp->usageTransmit) && (nkp->pairWise) && (nkp->index != 0)) {
            return A_ERROR;
        }

        if ((pKeyDesc->KeyLength > 32) || (pKeyDesc->KeyLength < 5) || (nkp->authenticatorKey)) {
            return A_ERROR;
        }

        if (nkp->pairWise) {
            /* Make sho we're tawkin bout same dude */
            if (A_MACADDR_COMP((WLAN_MACADDR *)&pKeyDesc->BSSID,
                               &broadcastMacAddr) == 0)
            {
                return A_ERROR;
            }
        }

        if (nkp->index >= MAX_SHARED_KEYS) {
            return A_ERROR;
        }

        if (!IS_STATE_CONNECTED(pDevInfo)) {
            return A_OK;
        }

        uiPrintf(" -- key index = 0x%08x, ", nkp->index);
        uiPrintf("Tx = %d, pairwise = %d, ",
                nkp->usageTransmit, nkp->pairWise);
        uiPrintf("initSC = %d len = %d\n",
                 nkp->initSC, pKeyDesc->KeyLength);
        uiPrintf("Key Value = ");
        for (i = 0; i < pKeyDesc->KeyLength; i++) {
             uiPrintf("%02X", pKeyDesc->KeyMaterial[i]);
        }
        uiPrintf("\n");

        pKey = &pDevInfo->keyTable[nkp->index];
        
        StopTransmit(pDevInfo, DO_CLEAR_TX_Q, DO_COMPLETE, DO_WAIT);

        /* Clear the key table entry */
        wlanKeyTableRemove(pDevInfo, (A_UINT16)nkp->index);

        RestartTransmit(pDevInfo);
        
        pKey->keyFlags = 0;

        if (nkp->usageTransmit) {
            if (pDevInfo->staConfig.defaultKey != nkp->index) {
                StopTransmit(pDevInfo, DO_CLEAR_TX_Q, DO_COMPLETE, DO_WAIT);
                /* XXX Divy. Who else could still be using this key */
                /* Invalidate all the users of this key */
                wlanKeyInvalidateAll(pDevInfo, &pDevInfo->keyTable[
                                        pDevInfo->staConfig.defaultKey]);
                /*
                 *  It's too dangerous to the swretry state machine to let
                 *  us change our defaultkey while frames are pending,
                 *  so dump em.
                 */

                RestartTransmit(pDevInfo);
            }
            pDevInfo->staConfig.defaultKey = (A_UINT16)nkp->index;
        }
        /* XXX Divy. Why not cleaning any previous BSS key at this point ? */

        /* Get the cipher */
        ASSERT(VALID_CIPHER_OUI(pDevInfo->localSta->mCipher));
        ASSERT(VALID_CIPHER_OUI(pDevInfo->localSta->uCipher));
        cipher = (connectionKey ?
                  pDevInfo->localSta->uCipher :
                  pDevInfo->localSta->mCipher);

        /* Look up the AP's SIB */
        pSib = sibEntryFind(pDevInfo, &pDevInfo->baseBss.bssId, NULL);
        ASSERT(pSib);

        /* Set the key */
        status = wlanKeyTableSet(pDevInfo,
                                 pSib,
                                 connectionKey,
                                 cipher,
                                 (A_UINT16)nkp->index,
                                 pKeyDesc->KeyLength * 8,
                                 pKeyDesc->KeyMaterial);

        if (status != A_OK) {
            return status;
        }

        if (nkp->initSC) {
            /* Subtract 1 so that we allow the value of keyRSC itself */
            pKey->keyRSC = pKeyDesc->KeyRSC - 1;
        } else {
            pKey->keyRSC = (A_INT64) -1;
        }

        /* Enable the connection key and install it */
        if (connectionKey) {
            /*
             * Flush any pending EAPOL-KEY message
             * before the key gets plumbed.
             */
            NdisInterlockedIncrement(&pDevInfo->numFlushCall);
            wdcFlush(pDevInfo->targetHandle, pDevInfo->disableEnableCounter);
          
            pSib->pPriv = pKey;
            wlanInstallConnectionKey(pDevInfo,
                                     pSib,
                                     &pDevInfo->bssDescr->bssId,
                                     pKey);
            /* Enable compression now that we have the pairwise keys */
            if (VALID_ATH_ADVCAP_ELEMENT(&pSib->athAdvCapElement) &&
                pSib->athAdvCapElement.info.useCompression) {
                A_UINT32 attribValue = TRUE;

                wdcUpdateConnectionAttribute(pDevInfo->targetHandle,
                                             pSib->wdcConnId,
                                             CONN_COMPRESSION,
                                             sizeof(attribValue),
                                             (TARGET_CONFIG_VAL)&attribValue);
            }
        }
        if (bssKey) {
            /* Install BSS key */
            wlanInstallBssKey(pDevInfo,
                              (A_UINT32)nkp->index,
                              1,                        /* Default bss key */
                              &pDevInfo->keyTable[nkp->index]);
        }

        if (connectionKey) {
            /* enable sw encryption if needed */
            pSib->swEncryptionEnabled =
                !cipherSupported(pDevInfo, pKey->keyType);
            pSib->swMicEnabled        =
                !micSupported(pDevInfo, pKey->keyType);
        }

    } else {
        /* WEP keys only from that point */
        pWepKeyDesc = (PNDIS_802_11_WEP)pBuf;

        /* If we used an ADD_KEY OID, cons it into a 802_11_WEP struct */
        if (type == ADD_KEY) {
            A_UINT8 tempKey[16];

            if (pKeyDesc->KeyLength > MAX_KEY_LEN_BYTES || (pKeyDesc->KeyLength < 5)) {
                return A_ERROR;
            }

            /* Have to copy twice since it could overlap */
            A_BCOPY(pKeyDesc->KeyMaterial, tempKey, pKeyDesc->KeyLength);
            A_BCOPY(tempKey, pWepKeyDesc->KeyMaterial, pKeyDesc->KeyLength);

            /* RSC and BSSID is unused for legacy */
        }

        nkp = (NDIS_KEY_PROPS *)&pWepKeyDesc->KeyIndex;

        uiPrintf(" -- key index = 0x%08x, ", nkp->index);
        uiPrintf("Tx = %d, pairwise = %d, ",
                nkp->usageTransmit, nkp->pairWise);
        uiPrintf("initSC = %d len = %d\n",
                 nkp->initSC, pWepKeyDesc->KeyLength);
        uiPrintf("Key Value = ");
        for (i = 0; i < pWepKeyDesc->KeyLength; i++) {
             uiPrintf("%02X", pWepKeyDesc->KeyMaterial[i]);
        }
        uiPrintf("\n");

        pKey = &pDevInfo->keyTable[nkp->index];
        keyIndex = (A_UINT16)nkp->index;
        
        StopTransmit(pDevInfo, DO_CLEAR_TX_Q, DO_COMPLETE, DO_WAIT);

        /* Clear the key table entry */
        wlanKeyTableRemove(pDevInfo, (A_UINT16)nkp->index);

        RestartTransmit(pDevInfo);

        pKey->keyFlags = 0;
      
        if (nkp->usageTransmit) {
            pDevInfo->staConfig.defaultKey = keyIndex;
        }
        
        pKey->keyFlags = keyIndex & PRIV_FRAMEKEY;

        /*
         * 802.1X context.
         * Assume we're in a ESS. One connection only at a time.
         */
        if (context802_1x) {
             /* Install the key as connection if the transmitUsage is set */
            /* Look up the AP's SIB */
            cipher = (pWepKeyDesc->KeyLength == (104 / 13) ? WPA_CSE_WEP104
                                                           : WPA_CSE_WEP40);
            pSib = sibEntryFind(pDevInfo, &pDevInfo->baseBss.bssId, NULL);
            ASSERT(pSib);

            /* Set the key */
            status = wlanKeyTableSet(pDevInfo,
                                     pSib,
                                     connectionKey,
                                     cipher,
                                     (A_UINT16)nkp->index,
                                     pWepKeyDesc->KeyLength * 8,
                                     pWepKeyDesc->KeyMaterial);
            if (status != A_OK) {
                return status;
            }

            if (connectionKey) {
                
                pSib->pPriv = pKey;
           
                
                /* Enable compression now that we have the pairwise keys */
                if (VALID_ATH_ADVCAP_ELEMENT(&pSib->athAdvCapElement) &&
                    pSib->athAdvCapElement.info.useCompression) {
                    A_UINT32 attribValue = TRUE;
                    A_UINT16 beaconInt = pDevInfo->bssDescr->beaconInterval;
                    A_UINT16 listenInt = (pDevInfo->staConfig.listenInterval
                                            + beaconInt - 1) / beaconInt;

                    wlanMlmeReassocRequest(pDevInfo,
                                           &pDevInfo->bssDescr->bssId,
                                           CSERV_TIME_AA_MAX,
                                           &lSib->capInfo,
                                           listenInt);

                }
                else {
                       wlanInstallConnectionKey(pDevInfo,
                                         pSib,
                                         &pDevInfo->bssDescr->bssId,
                                         pKey);

                }

            }
            if (bssKey) {
                /* Install BSS key */
                wlanInstallBssKey(pDevInfo,
                                  (A_UINT32)nkp->index,
                                  1,                        /* Default bss key */
                                  pKey);
            }
            return A_OK;

        } else {
            /*
             * static WEP context.
             * Assume no connection at this point.
             */
            if (!contextStaticWEP) {
                return A_ERROR;
            }
            cipher = (pWepKeyDesc->KeyLength == (104 / 13) ? WPA_CSE_WEP104
                                                           : WPA_CSE_WEP40);
            status = wlanKeyTableSet(pDevInfo,
                                     NULL,
                                     0,
                                     cipher,
                                     (A_UINT16)nkp->index,
                                     pWepKeyDesc->KeyLength * 8,
                                     pWepKeyDesc->KeyMaterial);
            if (status != A_OK) {
                return status;
            }
        }
    }
    ccxKeyPlumbedCallback(pDevInfo);

    return Status;
}

static int athusb_siwpriv_authtype(WLAN_DEV_INFO *pDevInfo, A_UINT32 Value)
{
    IEEE80211_AUTHENTICATION_MODE mode;
    SIB_ENTRY *lSib = pDevInfo->localSta;
    WLAN_AUTH_TYPE oldType;
    A_BOOL clearCse = FALSE; 
    
    mode = Value;
    oldType = pDevInfo->staConfig.authType;

    switch (mode) {
    case AuthModeOpen:
        // Authenticate in the open.
        pDevInfo->staConfig.wpaEnabled      = FALSE;
        pDevInfo->staConfig.authType        = AUTH_TYPE_OPEN;
        if (oldType == AUTH_TYPE_WPA || oldType == AUTH_TYPE_WPAPSK) {
            clearCse = TRUE;
        }
        break;

    case AuthModeShared:
        // Authenticate using the default key.
        // This may be a shared or a unique key.
        pDevInfo->staConfig.authType        = AUTH_TYPE_SHARED;
        pDevInfo->staConfig.wpaEnabled      = FALSE;
        if (oldType == AUTH_TYPE_WPA || oldType == AUTH_TYPE_WPAPSK) {
            clearCse = TRUE;
        }
        break;

    case AuthModeAutoSwitch:
        // Try to authenticate with shared key first,
        // then fall back to open system.
        pDevInfo->staConfig.authType        = AUTH_TYPE_AUTO;
        pDevInfo->staConfig.wpaEnabled      = FALSE;
        if (oldType == AUTH_TYPE_WPA || oldType == AUTH_TYPE_WPAPSK) {
            clearCse = TRUE;
        }
        break;

    case AuthModeWPA:
    case AuthModeWPAPSK:
        // WPA authentication. Only open systems allowed.
        pDevInfo->staConfig.wpaEnabled      = TRUE;
        pDevInfo->staConfig.privacyInvoked  = TRUE;
        /* Maybe set defaultKey = INVALID ? */
       
        if (mode == AuthModeWPA) {
            pDevInfo->staConfig.authType = AUTH_TYPE_WPA;
        } else if (mode == AuthModeWPAPSK) {
            pDevInfo->staConfig.authType = AUTH_TYPE_WPAPSK;
        } else {
            ASSERT(mode == AuthModeWPANone);
        }
        if (oldType == AUTH_TYPE_OPEN || oldType == AUTH_TYPE_SHARED ||
                oldType == AUTH_TYPE_AUTO)
        {
            clearCse = TRUE;
        }
        break;

    case AuthModeWPANone:
        /*
         * Specifies WPA version 1 security. Specifies the
         * use of preshared key without IEEE 802.1x in
         * IBSS mode. (taken from Microsoft guidelines)
         */
        uiPrintf("-- WPA in IBSS is unsupported\n");
        return NDIS_STATUS_INVALID_DATA;

    default:
        uiPrintf("-- Unknown mode!\n");
        return NDIS_STATUS_INVALID_DATA;
    }

    if (lSib && (clearCse == TRUE)) {
        /* Destroy any extant IE settings */
        lSib->uCipher = 0;
        lSib->mCipher = 0;
        lSib->authSel = 0;
        lSib->wpaCapabilities = 0;
    }

    if (pDevInfo->defaultStaConfig.authTypeUseOnly == AUTH_TYPE_DEFAULT) {
        wlanKeyAuthTypeSet(&(pDevInfo->staConfig),
                pDevInfo->staConfig.authType);
    } else {
        wlanKeyAuthTypeSet(&(pDevInfo->staConfig),
                pDevInfo->defaultStaConfig.authTypeUseOnly);
    }

    return (NDIS_STATUS_SUCCESS);
}


