/*
 * $Id: //depot/sw/branches/1.3_USB_LINUX_port/src/USB/wlan/host/sta/ccx.c#3 $
 *
 * Copyright (c) 2000-2003 Atheros Communications, Inc., All Rights Reserved
 *
 * Implementation of Cisco Certified eXtensions (CCX)
 */

#include "wlantype.h"
#include "wlanproto.h"
#include "wlanext.h"
#include "wlansmeext.h"
#include "wlanCapDe.h"
#include "stacserv.h"
#include "wlanchannel.h"
#include "ccxdefs.h"
#include "ccx.h"
#include "usbstub.h"

#ifdef WLAN_CONFIG_CCX

#define RM_POLICY_CHECK(pDev) ((pDev)->staConfig.ccxRadioMeasEnable) ? TRUE : \
                                                                       FALSE

/*
 * GLOBALS
 */

const        A_UINT8   aironetOui[] = { 0x00, 0x40, 0x96, 0x00 };
static const CAP_CONST aironetSnap  = AIRONET_SNAP;

/*
 * PROTOTYPES
 */

static void
ccxProcessDdpPacket(WLAN_DEV_INFO *pDev, ATHEROS_DESC *pDesc, A_UINT8 funcCode,
                    void *pContext);

static void
ccxProcessRadioMeas(WLAN_DEV_INFO *pDev, ATHEROS_DESC *pDesc, A_UINT8 funcCode);

static A_STATUS
ccxSendLastApFrame(WLAN_DEV_INFO *pDev);

static A_STATUS
ccxSendDdpFrame(WLAN_DEV_INFO *pDev, void *pData, A_UINT16 dataLen,
                A_UINT8 msgType, A_UINT8 funcCode, WLAN_MACADDR *pDestAddr,
                WLAN_MACADDR *pSrcAddr, A_UINT16 etherType);

static A_STATUS
ccxAddPreferredAp(WLAN_DEV_INFO *pDev, WLAN_MACADDR *pBssid);

static void
ccxRogueListDel(WLAN_DEV_INFO *pDev, WLAN_MACADDR *pBssid);

static void
ccxRmDispatch(MEAS_SCHED_Q *pHead);

static void
ccxRmPurgeRequests(CCX_INFO *pCcxInfo);

static INLINE A_STATUS
ccxScheduleMeasRequest(MEAS_SCHED_Q *pHead);

static void
ccxRmRemoveTransaction(MEAS_SCHED_Q *pQueue);

static void
ccxRmComplete(MEAS_Q_ENTRY *pEntry);

/*
 * IMPLEMENTATION
 */

/**************************************
 * Initializes anything CCX-related.  Called at driver load.
 */
A_STATUS
ccxInit(WLAN_DEV_INFO *pDev)
{
    CCX_INFO *pCcxInfo;

    ASSERT(pDev->pCcxInfo == NULL);

    pDev->pCcxInfo = A_DRIVER_MALLOC(sizeof(*pDev->pCcxInfo));
    pCcxInfo       = pDev->pCcxInfo;

    if (NULL == pCcxInfo) {
        return A_ERROR;
    }
    A_MEM_ZERO(pCcxInfo, sizeof(*pCcxInfo));

    TAILQ_INIT(&pCcxInfo->schedQueueHead);
    A_SEM_INIT(pCcxInfo->schedQueueLock, 0, CREATE_UNLOCKED);

    /* TODO: Copy Preferred AP information from the STA configuration */

    return A_OK;
}

/**************************************
 * Frees up CCX-related structs, etc.  Called on driver unload.
 */
void
ccxFree(WLAN_DEV_INFO *pDev)
{
    CCX_INFO     *pCcxInfo = pDev->pCcxInfo;

    if (!pCcxInfo) {
        return;
    }

    ccxRmPurgeRequests(pCcxInfo);

    A_DRIVER_FREE(pCcxInfo, sizeof(*pCcxInfo));
    pDev->pCcxInfo = NULL;
}


void
ccxRmPurgeRequests(CCX_INFO *pCcxInfo)
{
    MEAS_SCHED_Q *pQueue, *pNext;

    /* Clean up the queue of scheduled requests */
    pQueue = TAILQ_FIRST(&pCcxInfo->schedQueueHead);
    while (pQueue) {
        pNext = TAILQ_NEXT(pQueue, link);

        ccxRmRemoveTransaction(pQueue);

        pQueue = pNext;
    }
    A_SEM_DELETE(pCcxInfo->schedQueueLock);
}

/**************************************
 * Processes a received CCX DDP frame that the STA might be interested in.
 *
 * Returns: A_OK    if the frame was a DDP frame that we've processed without
 *                  error.
 *          A_ERROR if there was a problem parsing the frame.
 *
 * In any case, the frame will be consumed here.
 */
A_STATUS
ccxProcessRxFrame(WLAN_DEV_INFO *pDev, ATHEROS_DESC *pDesc, A_UINT16 etherType)
{
    DDP_HEADER  *pHdr;
    A_UINT32    frameLen;
    A_STATUS    status     = A_OK;

    frameLen = pDesc->frameLength;

    switch (etherType) {
    case CCX_ETHTYPE_NULL:
        pHdr = (DDP_HEADER *)pDesc->pBufferVirtPtr.byte;
        if (frameLen < sizeof(*pHdr) || be2cpu16(pHdr->ddpLength) > frameLen) {
            status = A_ERROR;
            goto finished;
        }

        /* Pop the DDP header */
        bufferPopHdr(pDesc, sizeof(*pHdr));

        switch (be2cpu16(pHdr->msgType)) {
        case DDP_MSG_REPORT:
            ccxProcessDdpPacket(pDev, pDesc, pHdr->funcCode, NULL);
            break;
        case DDP_MSG_RADIO_MEAS:
            if (TRUE == RM_POLICY_CHECK(pDev)) {
                ccxProcessRadioMeas(pDev, pDesc, pHdr->funcCode);
            }
            break;
        case DDP_MSG_ROGUEAP:
            ASSERT(pHdr->msgType != DDP_MSG_ROGUEAP);
            break;
        default:
            /* Unknown DDP message type */
            status = A_ERROR;
            break;
        }
        break;

    default:
        /* Unknown ether type */
        status = A_ERROR;
        break;
    }

finished:
    freeBuffandDesc(pDev, pDesc);
    return status;
}

static void
ccxProcessRadioMeas(WLAN_DEV_INFO *pDev, ATHEROS_DESC *pDesc, A_UINT8 funcCode)
{
    CCX_INFO            *pCcxInfo       = pDev->pCcxInfo;
    MEAS_REQUEST_HEADER *pRequestHdr;
    MEAS_REPORT_HEADER  *pReportHdr;
    A_INT16             frameLen;

    ASSERT(pDesc);

    frameLen = (A_UINT16) pDesc->frameLength;

    switch ((funcCode)) {
    case DDP_FUNC_MEAS_REQUEST: {
        A_STATUS        status;
        MEAS_SCHED_Q    *pQueue;
     
        if (frameLen < sizeof(*pRequestHdr)) {
            break;
        }

        pCcxInfo->stats.cntRxMeasRequestFrames++;
        pRequestHdr = (MEAS_REQUEST_HEADER *)
                      bufferPopHdr(pDesc, sizeof(*pRequestHdr));

        pQueue = (MEAS_SCHED_Q *) A_DRIVER_MALLOC(sizeof(*pQueue));
        if (NULL == pQueue) {
            break;
        }
        A_MEM_ZERO(pQueue, sizeof(*pQueue));
        TAILQ_INIT(&pQueue->measQueueHead);

        /* Setup parameters for these measurements */
        pQueue->pDev              = pDev;
        pQueue->dialogToken       = be2cpu16(pRequestHdr->dialogToken);
        pQueue->activationDelay   = pRequestHdr->activationDelay;
        pQueue->measOffset        = pRequestHdr->measOffset;

        /* Collect the requests */
        ccxProcessDdpPacket(pDev, pDesc, funcCode, pQueue);

        /* If no requests were added, no need to schedule anything */
        if (pQueue->numItems < 1) {
            A_DRIVER_FREE(pQueue, sizeof(*pQueue));
            break;
        }

        /* Schedule a timer if needed */
        status = ccxScheduleMeasRequest(pQueue);
        if (A_ERROR == status) {
            ccxRmRemoveTransaction(pQueue);
            break;
        }

        /* Queue this transaction */
        TAILQ_INSERT_TAIL(&pCcxInfo->schedQueueHead, pQueue, link);

        if (A_PENDING != status) {
            /* Not scheduled, need to process inline */
            ccxRmDispatch(pQueue);
        }

        break;
    }

    case DDP_FUNC_MEAS_REPORT:
        pCcxInfo->stats.cntRxMeasReportFrames++;
        pReportHdr = (MEAS_REPORT_HEADER *)
                     bufferPopHdr(pDesc, sizeof(*pReportHdr));
        /* What do we do with received measurement reports? */
        ccxProcessDdpPacket(pDev, pDesc, funcCode, NULL);
        break;

    default:
        break;
    }
}

#define CCX_SETUP_MEAS_REQUEST(pRequest, pEle)                      \
            (pRequest)->type      = (pEle)->type;                   \
            (pRequest)->mode      = (pEle)->mode;                   \
            (pRequest)->token     = be2cpu16((pEle)->token);        \
            (pRequest)->channelNo = (pEle)->channelNo;              \
            (pRequest)->scanMode  = (pEle)->scanMode;               \
            (pRequest)->duration  = be2cpu16((pEle)->measDuration);

/**************************************
 * Process a received DDP "report" frame
 */
void
ccxProcessDdpPacket(WLAN_DEV_INFO *pDev, ATHEROS_DESC *pDesc, A_UINT8 funcCode,
                    void *pContext)
{
    MGMT_ELEMENT  *pElement;
    A_INT16       frameLen;

    ASSERT(pDesc);

    pElement = (MGMT_ELEMENT *) pDesc->pBufferVirtPtr.byte;
    frameLen = (A_UINT16) pDesc->frameLength;

    /* Go through each element */
    while (frameLen > sizeof(*pElement)) {
        A_UINT16 eleLen = be2cpu16(pElement->length) + sizeof(*pElement);

        /* Check to be sure that the length given in the element is valid */
        if (eleLen > frameLen) {
            return;
        }

        /* Process the element based on its tag */
        switch (be2cpu16(pElement->tag)) {
        case DDP_RPT_TAG_NEARBY_AP: {
            AP_LIST_ELEMENT *pEle = (AP_LIST_ELEMENT *) pElement;
            SSID            ssid;

            /* Make sure the OUI field matches */
            if (A_BCOMP(&pEle->oui, &aironetOui, sizeof(aironetOui)) != 0) {
                break;
            }

            ssid.elementID = ELE_SSID;
            ssid.length    = (A_UINT8)be2cpu16(pEle->ssidLength);
            A_BCOPY(&pEle->ssid, &ssid.ssid, ssid.length);

            cservNearbyApReport(pDev, be2cpu16(pEle->channel),
                                &pEle->bssid, &ssid);
            break;
        }

        case ID_MEAS_REQUEST: {
            MEAS_SCHED_Q   *pQueue    = (MEAS_SCHED_Q *) pContext;
            MEAS_Q_ENTRY     *pRequest;
          
            if (frameLen < sizeof(RADIO_MEAS_ELEMENT)) {
                return;
            }

            pRequest = (MEAS_Q_ENTRY *) A_DRIVER_MALLOC(sizeof(*pRequest));
            if (NULL == pRequest) {
                return;
            }
            A_MEM_ZERO(pRequest, sizeof(*pRequest));

            CCX_SETUP_MEAS_REQUEST(pRequest, (RADIO_MEAS_ELEMENT *) pElement);

            TAILQ_INSERT_TAIL(&pQueue->measQueueHead, pRequest, link);
            pQueue->numItems++;

            break;
        }

        case ID_MEAS_REPORT: {
            RADIO_MEAS_ELEMENT *pEle = (RADIO_MEAS_ELEMENT *) pElement;
            /* Need to handle reports? */
            break;
        }

        default:
            /* Unknown tag */
            uiPrintf("ccxDdpProcessPacket: Unknown tag: %d", pElement->tag);
            break;
        }

        bufferPopHdr(pDesc, eleLen);
        pElement = (MGMT_ELEMENT *) pDesc->pBufferVirtPtr.byte;
        frameLen = (A_UINT16) pDesc->frameLength;
    }
}

/**************************************
 * Sends a "rogue AP" informational frame to the AP with which the
 * STA is currently associated.  Contains the rogue AP's BSSID as
 * well as a failure code.
 */
A_STATUS
ccxSendRogueApFrame(WLAN_DEV_INFO *pDev, WLAN_MACADDR *pAddr,
                    A_UINT16 failCode)
{
    ROGUE_AP_ELEMENT *pEle;
    A_STATUS         status;

    pEle = (ROGUE_AP_ELEMENT *)A_DRIVER_MALLOC(sizeof(*pEle));
    if (!pEle) {
        return A_ERROR;
    }
    
    /* Fill in the Rogue AP element */
    pEle->failureCode = cpu2be16(failCode);
    A_MACADDR_COPY(pAddr, &pEle->macAddr);
    A_MEM_ZERO(pEle->rogueName, sizeof(pEle->rogueName));

    status = ccxSendDdpFrame(pDev, pEle, sizeof(*pEle),
                             DDP_MSG_ROGUEAP, DDP_FUNC_ROGUEAP,
                             &pDev->bssDescr->bssId,
                             &pDev->localSta->macAddr, CCX_ETHTYPE_NULL);

    /* Previous call has copied the data it needs */
    A_DRIVER_FREE(pEle, sizeof(*pEle));

    return status;
}

/**************************************
 * Sends a frame to our newly-associated AP with information pertaining to
 * our last association.
 */
A_STATUS
ccxSendLastApFrame(WLAN_DEV_INFO *pDev)
{
    AP_LIST_ELEMENT *pEle;
    A_STATUS        status;
    A_UINT16        len, timeElapsed, *pSeconds;
    CCX_INFO        *pInfo  = pDev->pCcxInfo;
    SSID            *pSsid  = &pInfo->lastSsid;
    WLAN_MACADDR    *pBssid = &pInfo->lastBssid;

    if (!pSsid || !pSsid->length || pInfo->sentLastApFrame) {
        /*
         * We've never disassociated before or we've already sent this frame,
         * so no reason to send one now.
         */
        return A_ERROR;
    }
    
    ASSERT(pSsid->length <= ELE_SSID_SIZE);

    /* Figure out the time elapsed */
    timeElapsed = (A_UINT16)((A_MS_TICKGET() - pInfo->disassocTime) / 1000);

    /*
     * Figure out the length of the element since the SSID length is
     * variable for some strange reason. See comment near
     * AP_LIST_ELEMENT to figure out how this crazy thing works.
     */
    len = AP_LIST_ELE_SIZE + pSsid->length;

    pEle = (AP_LIST_ELEMENT *)A_DRIVER_MALLOC(len);
    if (!pEle) {
        return A_ERROR;
    }

    /* Fill in the Last AP element */
    pEle->ele.tag     = cpu2be16(DDP_RPT_TAG_NEARBY_AP);
    pEle->ele.length  = cpu2be16(len - sizeof(pEle->ele));
    A_DRIVER_BCOPY(&aironetOui, pEle->oui, sizeof(pEle->oui));
    A_MACADDR_COPY(pBssid, &pEle->bssid);
    pEle->channel     = cpu2be16(pInfo->lastChannel);
    pEle->ssidLength  = cpu2be16(pSsid->length);
    A_DRIVER_BCOPY(pSsid->ssid, pEle->ssid, pSsid->length);

    /*
     * Now place the last part of the "structure" that comes
     * after the variable-length SSID.
     */
    pSeconds = (A_UINT16 *)(pEle->ssid + pSsid->length);

    ASSERT((A_UINT8 *)pSeconds = (A_UINT8 *)pEle + len - sizeof(A_UINT16));
    *pSeconds = cpu2be16(timeElapsed);

    status = ccxSendDdpFrame(pDev, pEle, len,
                             DDP_MSG_REPORT, DDP_FUNC_NULL,
                             &pDev->bssDescr->bssId,
                             &pDev->localSta->macAddr, CCX_ETHTYPE_NULL);

    /* Previous call has copied the data it needs */
    A_DRIVER_FREE(pEle, len);

    if (status == A_OK) {
        pInfo->sentLastApFrame = TRUE;
    }
    
    return status;
}

/**************************************
 * Generic routine which takes an input buffer, DDP encapsulates the frame,
 * 802.3 encapsulates the frame, and then passes it on to be 802.11
 * encapsulated and sent to the currently-associated AP.
 */
A_STATUS
ccxSendDdpFrame(WLAN_DEV_INFO *pDev, void *pData, A_UINT16 dataLen,
                A_UINT8 msgType, A_UINT8 funcCode, WLAN_MACADDR *pDestAddr,
                WLAN_MACADDR *pSrcAddr, A_UINT16 etherType)
{
    ATHEROS_DESC     *pDesc;
    DDP_HEADER       *pDdpHdr;
    LLC_SNAP_HEADER  *pSnapHdr;
    LAN_FRAME_HEADER *pHdr;
    A_UINT16         totalLen, headroom, typeOrLen;
    A_STATUS         status;

    ASSERT(pData && dataLen >= 0);
    ASSERT(pDestAddr && pSrcAddr);

    /* Determine needed headroom for encapsulating the frame later. */
    headroom = sizeof(WLAN_DATA_MAC_HEADER3) + sizeof(DDP_HEADER) +
               MAX_IV_FIELD_SIZE;
    totalLen = headroom + dataLen + MAX_ICV_FIELD_SIZE;

    /* Get a buffer and descriptor pair */
    if (createBuffandDesc(pDev, totalLen, &pDesc) != A_OK) {
        return A_ERROR;
    }

    /* Provide sufficient room for headers. */
    bufferPopHdr(pDesc, headroom);

    /* Set the payload length */
    pDesc->bufferLength = dataLen;
    pDesc->frameLength  = dataLen;
    
    /* Fill in the payload data */
    A_DRIVER_BCOPY(pData, pDesc->pBufferVirtPtr.byte, dataLen);
    
    /* Push space for the DDP header */
    bufferPushHdr(pDesc, sizeof(*pDdpHdr));
    pDdpHdr = (DDP_HEADER *)pDesc->pBufferVirtPtr.byte;

    /* Fill in the DDP header */
    pDdpHdr->msgType   = msgType;
    pDdpHdr->funcCode  = funcCode;
    A_MACADDR_COPY(pDestAddr, &pDdpHdr->destAddr);
    A_MACADDR_COPY(pSrcAddr, &pDdpHdr->srcAddr);
    /* ddpLength includes DDP header plus the data */
    pDdpHdr->ddpLength = cpu2be16(pDesc->frameLength);

    /* Push space for the SNAP header */
    bufferPushHdr(pDesc, sizeof(LLC_SNAP_HEADER));
    pSnapHdr = pDesc->pBufferVirtPtr.llcSnapHeader;

    /* Fill in the SNAP header */
    A_BCOPY(&aironetSnap, pSnapHdr, sizeof(CAP_CONST));
    pSnapHdr->etherType = cpu2be16(etherType);

    /* Record the frame length before we add the 802.3 header. */
    typeOrLen = (A_UINT16)pDesc->frameLength;

    /* Push space for the 802.3 header. */
    bufferPushHdr(pDesc, sizeof(LAN_FRAME_HEADER));
    pHdr = pDesc->pBufferVirtPtr.lanHeader;

    /* Fill in the 802.3 header */
    A_MACADDR_COPY(pDestAddr, &pHdr->destAddr);
    A_MACADDR_COPY(pSrcAddr, &pHdr->srcAddr);
    pHdr->lanTypeOrLen = cpu2be16(typeOrLen);

    /*
     * HACK ALERT:
     * Station uses the basic virtual device only. This should be done in a
     * more layered fashion, but it's not.  Yet.
     */
    pDesc->pOpBss = &pDev->baseBss;

    status = wlanDataFrameTransmit(pDev, pDesc);

    return status;
}

/*
 * Use the OS timer and allow some slop.
 * If we end up needing better resolution than what the OS can give, change
 * this function to instrument a timer with our hardware.
 *
 * Returns:
 *      TRUE  - A timeout was scheduled.
 *      FALSE - No timeout scheduled.
 */
static INLINE A_STATUS
ccxScheduleMeasRequest(MEAS_SCHED_Q *pQueue)
{
    A_INT32 timeout = pQueue->activationDelay + pQueue->measOffset;

#define OS_TIMER_RES 20
    if (timeout > OS_TIMER_RES / 2) {
        if (A_OK != A_INIT_TIMER(pQueue->pDev, &pQueue->startTimer,
                                 ccxRmDispatch, FALSE))
        {
            return A_ERROR;
        }
        A_TIMEOUT(&pQueue->startTimer, timeout, (A_UINT32) pQueue, 0);

        return A_PENDING;
    }

    return A_OK;
}

/*
 * Timer function.
 */
void
ccxRmDispatch(MEAS_SCHED_Q *pQueue)
{
    MEAS_Q_ENTRY        *pEntry;
    WLAN_DEV_INFO       *pDev    = pQueue->pDev;
    WLAN_STA_CONFIG     *pConfig = &pDev->staConfig;

    pEntry = TAILQ_FIRST(&pQueue->measQueueHead);
    while (pEntry) {
        A_BOOL          processed = FALSE;
        MEAS_Q_ENTRY    *pNext    = TAILQ_NEXT(pEntry, link);

        /* Ignore certain off-channel requests */
        if (pEntry->channelNo != pConfig->pChannel->channel) {
            if (pEntry->duration > pConfig->ccxRmMaxOffChanTime) {
                processed = TRUE;
            }

            if (TRUE/* TODO: Need a busy check */ &&
                pEntry->duration > pConfig->ccxRmOffChanTimeWhenBusy)
            {
                processed = TRUE;
            }
        }

        if (!processed) {
            switch (pEntry->type) {
            case MEAS_CHANNEL_LOAD:
                uiPrintf("MEAS_CHANNEL_LOAD Request not handled.\n");
                // MAC_RCCNT
                halMibControl(pDev, CLEAR_ALL_COUNTERS);
                halMibControl(pDev, GET_CCA_PERCENTAGE);
                break;

            case MEAS_NOISE_HISTOGRAM:
                uiPrintf("MEAS_NOISE_HISTOGRAM Request not handled.\n");
                processed = TRUE;
                break;

            case MEAS_BEACON:
                uiPrintf("MEAS_BEACON Request not handled.\n");
                processed = TRUE;
                break;

            case MEAS_FRAME:
                uiPrintf("MEAS_FRAME Request not handled.\n");
                processed = TRUE;
                break;

            default:
                uiPrintf("ERROR: Unknown radio measurement: %d",
                         pEntry->type);
                processed = TRUE;
            }
        }

        if (processed) {
            TAILQ_REMOVE(&pQueue->measQueueHead, pEntry, link);
        }

        pEntry = pNext;
    }

    if (NULL == TAILQ_FIRST(&pQueue->measQueueHead)) {
        ccxRmRemoveTransaction(pQueue);
    }
}

/*
 * Timer function (not done).
 */
void
ccxRmComplete(MEAS_Q_ENTRY *pEntry)
{
    uiPrintf("==> %s(): ",__FUNCTION__);

    return;
}

void
ccxRmRemoveTransaction(MEAS_SCHED_Q *pQueue)
{
    MEAS_Q_ENTRY        *pEntry;
    CCX_INFO            *pCcxInfo = pQueue->pDev->pCcxInfo;

    /* TODO: semaphore (nothing in NDIS) */

    MEAS_Q_ENTRY   *pMeasurement  = TAILQ_FIRST(&pQueue->measQueueHead);

    A_DELETE_TIMER(&pQueue->startTimer);
    /* XXX: Cancel ongoing transaction? */

    /* Clean up the queue of measurements for this request */
    while (pMeasurement) {
        MEAS_Q_ENTRY *pNextMeas = TAILQ_NEXT(pMeasurement, link);
        TAILQ_REMOVE(&pQueue->measQueueHead, pMeasurement, link);
        A_DELETE_TIMER(&pMeasurement->durationTimer);
        A_DRIVER_FREE(pMeasurement, sizeof(*pMeasurement));
        pMeasurement = pNextMeas;
    }

    TAILQ_REMOVE(&pCcxInfo->schedQueueHead, pQueue, link);
    A_DRIVER_FREE(pQueue, sizeof(*pQueue));
}

/**************************************
 * Called on dissassociation from an AP
 */
void
ccxDisassocCallback(WLAN_DEV_INFO *pDev)
{
    CCX_INFO    *pInfo = pDev->pCcxInfo;
    CHAN_VALUES *pChan = pDev->staConfig.pChannel;
    A_UINT16    chanNum;

    ASSERT(pInfo);
    ASSERT(pChan);

    /* Convert the channel from MHz to IEEE */
    chanNum = wlanConvertGHztoCh(pChan->channel, pChan->channelFlags);

    pInfo->disassocTime = A_MS_TICKGET();
    pInfo->lastChannel  = chanNum;
    pInfo->lastSsid     = pDev->bssDescr->ssid;
    A_MACADDR_COPY(&pDev->bssDescr->bssId, &pInfo->lastBssid);

    /*
     * Clear our state flag so that we'll send an AP report frame
     * at the next association.
     */
    pInfo->sentLastApFrame = FALSE;
}

/**************************************
 * Called when a key is plumbed from upper-layer supplicants, etc.
 */
void
ccxKeyPlumbedCallback(WLAN_DEV_INFO *pDev)
{
    A_STATUS status;
    char     *string;

    /*
     * If we're using encryption, we have to wait until we have a valid key
     * so that our frame won't be dropped either by the STA or the AP on
     * the other end.
     */
    if (ABLE_TO_PASS_TRAFFIC(pDev) &&
        IS_STATE_CONNECTED(pDev))
    {
        status = ccxSendLastApFrame(pDev);
    } else {
        status = A_ERROR;
    }

    string = (status == A_OK) ? "Sent" : "Did not send";
    
    uiPrintf("ccxKeyPlumbedCallback: %s Previous AP DDP frame.\n", string);
}

/**************************************
 * Called on association to an AP
 */
void
ccxAssocCallback(WLAN_DEV_INFO *pDev)
{
    A_STATUS status;
    char     *string;

    if (ABLE_TO_PASS_TRAFFIC(pDev)) {
        status = ccxSendLastApFrame(pDev);
    } else {
        status = A_ERROR;
    }

    string = (status == A_OK) ? "Sent" : "Did not send";
    
    uiPrintf("ccxAssocCallback: %s Previous AP DDP frame.\n", string);
}

/**************************************
 * Called after LEAP authentication is finished.
 */
void
ccxLeapAuthCallback(WLAN_DEV_INFO *pDev, WLAN_MACADDR *peerAddr,
                    MLME_RESULT resultCode)
{
    ASSERT(pDev->staConfig.leapEnabled);

    if (pDev->staConfig.leapEnabled &&
        resultCode == WLAN_UNSUPPORTED_ALG)
    {
        ccxRogueListAdd(pDev, peerAddr, DDP_RAP_UNSUPPORTED_ALG);
    }
}

/**************************************
 * Takes an array of MAC addresses (BSSIDs) and makes them
 * the CCX preferred AP list.
 */
void
ccxInitPreferredAps(WLAN_DEV_INFO *pDev, A_UINT32 numBssids,
                    WLAN_MACADDR pBssids[])
{
    A_UINT32 i;
    A_STATUS addStatus;

    ASSERT(pDev->pCcxInfo);
    ASSERT(pBssids);

    pDev->pCcxInfo->numPreferredAps = 0;

    for (i = 0; i < numBssids; i++) {

        if (A_MACADDR_COMP(&pBssids[i], &nullMacAddr) != 0) {
            addStatus = ccxAddPreferredAp(pDev, &pBssids[i]);
            ASSERT(addStatus == A_OK);
        }
    }
}

/**************************************
 * Adds a BSSID to the preferred AP list.
 */
A_STATUS
ccxAddPreferredAp(WLAN_DEV_INFO *pDev, WLAN_MACADDR *pBssid)
{
    CCX_INFO *pInfo = pDev->pCcxInfo;
    A_UINT32 num;

    ASSERT(pInfo);

    num = pInfo->numPreferredAps;
    if (num >= MAX_PREFERRED_APS) {
        return A_ERROR;
    }

    A_MACADDR_COPY(pBssid, &pInfo->prefBssid[num]);

    pInfo->numPreferredAps++;

    return A_OK;
}

/**************************************
 * Called before attempting to roam.  Modifies the rssiAdder of a BSS
 * based on the preferred status of a BSS.
 */
#define PREFERRED_RSSI_ADDER 200
void
ccxProcessPreferredAps(WLAN_DEV_INFO *pDev, BSSDESCR_SET *pSet)
{
    CCX_INFO *pInfo;
    A_UINT32 i, j;

    ASSERT(pDev->pCcxInfo && pSet);
    pInfo = pDev->pCcxInfo;

    /* If no preferred BSSIDs are set, return */
    if (!pInfo->numPreferredAps) {
        return;
    }
    
    for (i = 0; i < pSet->count; i++) {
        BSSDESCR *pBss = &pSet->bssDescrArray[i];

        for (j = 0; j < pInfo->numPreferredAps; j++) {
            WLAN_MACADDR *pBssid = &pInfo->prefBssid[j];

            if (A_MACADDR_COMP(pBssid, &pBss->bssId) == 0) {
                pBss->rssiAdder = PREFERRED_RSSI_ADDER;
            }
        }
    }
}

/**************************************
 * Add an AP the the rogue AP list.
 */
A_STATUS
ccxRogueListAdd(WLAN_DEV_INFO *pDev, WLAN_MACADDR *pBssid,
                A_UINT16 failureCode)
{
    int              i;
    ROGUE_AP_ELEMENT *pEle;
    A_CHAR           apIndex = pDev->pCsInfo->scanInfo.bssSet.setIdx;

    /* MDS: should make this a fn call into connection services */
    pDev->pCsInfo->scanInfo.bssSet.bssDescrArray[apIndex].apState |= BAD_AP;

    ASSERT(pDev->pCcxInfo);

    for (i = 0; i < ROGUE_LIST_NELEMENTS; i++) {
        pEle = &pDev->pCcxInfo->rogueList[i];

        if (A_MACADDR_COMP(&pEle->macAddr, pBssid) == 0) {
            /* Already in the list */
            return A_OK;
        }

        if (A_MACADDR_COMP(&pEle->macAddr, &nullMacAddr) == 0) {
            A_MACADDR_COPY(pBssid, &pEle->macAddr);
            pEle->failureCode = failureCode;

            return A_OK;
        }
    }

    return A_NO_MEMORY;
}

/**************************************
 * Delete an entry from the Rogue AP list.
 */
void
ccxRogueListDel(WLAN_DEV_INFO *pDev, WLAN_MACADDR *pBssid)
{
    int              i;
    ROGUE_AP_ELEMENT *pEle;

    for (i = 0; i < ROGUE_LIST_NELEMENTS; i++) {
        pEle = &pDev->pCcxInfo->rogueList[i];
        if (A_MACADDR_COMP(&pEle->macAddr, pBssid) == 0) {
            A_MACADDR_COPY(&nullMacAddr, &pEle->macAddr);
            pEle->failureCode = 0;
            break;
        }
    }
}

/**************************************
 * Flush the rogue AP list, if any, out to the AP
 */
void
ccxRogueListFlush(WLAN_DEV_INFO *pDev, WLAN_MACADDR *pMac)
{
    int i;

    ASSERT(A_MACADDR_COMP(pMac, &pDev->bssDescr->bssId) == 0);

    /* make sure this guy is marked ok */
    ccxRogueListDel(pDev, pMac);

    /* send extant list to AP */
    for (i = 0; i < ROGUE_LIST_NELEMENTS; i++) {
        ROGUE_AP_ELEMENT *pEle = &pDev->pCcxInfo->rogueList[i];

        if (A_MACADDR_COMP(&pEle->macAddr, &nullMacAddr) != 0 &&
            A_MACADDR_COMP(&pEle->macAddr, &pDev->bssDescr->bssId) != 0)
        {
            ccxSendRogueApFrame(pDev, &pEle->macAddr, pEle->failureCode);
        }

        A_MEM_ZERO(pEle, sizeof(*pEle));
    }
}

/*
 * CCKM Routines
 */

/*
 * Calculates the MIC that will be sent on the
 * CCKM reassociation request.
 * MICmn = HMAC-MD5(KRK, STA-ID | BSSID | RSNIEmn | Timestamp | RN)
 */
void
ccxCckmMicCalc(WLAN_DEV_INFO *pdevInfo, WLAN_MACADDR  *pDestAddr)
{
    SIB_ENTRY   *lSib;
    UCSE_SET    ucseSet;
    MCSE_SET    mcseSet;
    AUTHSE_SET  authSet;
    A_UINT8     cckmMicBuf[64];
    A_UINT32    cckmMicBufLen = 0;

    lSib = pdevInfo->localSta;

    if (lSib->cckmEnabled == TRUE) {
        ASSERT(pdevInfo->staConfig.wpaEnabled == TRUE);
        if (lSib->cckmKrkValid == TRUE) {
            /*
             * If CCKM is enabled we need to calculate the hmac_md5
             * MIC that needs to be added to the CCKM Assoc Req IE
             * HMAC-MD5(KRK, STA-ID | BSSID | RSNie | Timestamp | RN)
             * where KRK = key refresh key used to calculate MIC
             * STA-ID = Station's MAC address
             * BSSID = AP's MAC
             * RSNie = WPA Information Element
             * Timestamp = Last TSF timer from beacon (least sig byte first)
             * RN = Reassociation Request Number (incremented each reassoc)
             */
            lSib->cckmRn++;
            A_MACADDR_COPY(&lSib->macAddr,
                           (WLAN_MACADDR *)&cckmMicBuf[cckmMicBufLen]);
            cckmMicBufLen += WLAN_MAC_ADDR_SIZE;
            A_MACADDR_COPY(pDestAddr,
                           (WLAN_MACADDR *)&cckmMicBuf[cckmMicBufLen]);
            cckmMicBufLen += WLAN_MAC_ADDR_SIZE;
            /*
             * Construct the WPA IE
             */
            ucseSet.uCipherCount      = 1;
            ucseSet.uCiphers[0]       = lSib->uCipher;
            mcseSet.mCipher[0]        = lSib->mCipher;
            authSet.authSelectorCount = 1;
            authSet.authSelectors[0]  = lSib->authSel;

            wpaElementsToWpaIe(&ucseSet, &mcseSet, &authSet,
                           lSib->wpaCapabilities,
                           (WPA_IE *)&cckmMicBuf[cckmMicBufLen]);

            cckmMicBufLen +=
                INFO_ELEMENT_SIZE(*(INFO_ELEMENT *)&cckmMicBuf[cckmMicBufLen]);

            A_BCOPY(&lSib->lastBeaconTime, &cckmMicBuf[cckmMicBufLen],
                    sizeof (WLAN_TIMESTAMP));
            cckmMicBufLen += sizeof (WLAN_TIMESTAMP);
            A_BCOPY(&lSib->cckmRn, &cckmMicBuf[cckmMicBufLen],
                    sizeof (lSib->cckmRn));
            cckmMicBufLen += sizeof (lSib->cckmRn);
            ASSERT(cckmMicBufLen < 64);
            hmac_md5(cckmMicBuf, cckmMicBufLen, lSib->cckmKrk,
                     KRK_LENGTH_BYTES, lSib->cckmMic);
        } else {
            /*
             * We don't have a valid KRK. Disable cckm so we will
             * do a regular reassociation (vs. a CCKM one)
             */
            lSib->cckmEnabled = FALSE;
        }
    }
}

/*
 * Adds a CCKM Reassociation Request Element to buf if needed.
 * Should be called by mlmeAssocReassocRequest only if we are
 * doing a reassociation.
 * Returns the numbers of bytes added to buf.  If no IE was
 * added it will return 0.
 */
int
ccxCckmAddIe(SIB_ENTRY *lSib, A_UINT8 *buf)
{
    CCKM_REQ_IE *pCckmIe;
    int bytesadded = 0;

    if (lSib->cckmEnabled) {
        pCckmIe = (CCKM_REQ_IE *)buf;

        pCckmIe->cr_elementID = ELE_CCKM;
        pCckmIe->cr_length = ELE_CCKM_REQ_SIZE;
        A_BCOPY(ouiAironet, pCckmIe->cr_oui, sizeof(pCckmIe->cr_oui));
        pCckmIe->cr_ouiType = CCKM_OUI_TYPE;
        A_BCOPY(&lSib->lastBeaconTime, &pCckmIe->cr_timestamp,
                sizeof (WLAN_TIMESTAMP));
        A_BCOPY(&lSib->cckmRn, &pCckmIe->cr_reasnumber,
                sizeof (lSib->cckmRn));
        A_BCOPY(&lSib->cckmMic, &pCckmIe->cr_mic,
                sizeof (lSib->cckmMic));
        mlmePrintf("requesting CCKM Reauth\n");
        bytesadded = sizeof(*pCckmIe);
    }

    return bytesadded;
}

A_UINT16
ccxGetCckmIeSize(void)
{
    return sizeof(CCKM_REQ_IE);
}

void
ccxCckmKrkInvalidate(WLAN_DEV_INFO *pDevInfo)
{
    SIB_ENTRY *lSib = pDevInfo->localSta;

    if (lSib->cckmEnabled == TRUE) {
        A_MEM_ZERO(lSib->cckmKrk,KRK_LENGTH_BYTES);
        lSib->cckmKrkValid = FALSE;
        lSib->cckmEnabled = FALSE;
    }
}


void
ccxMixedPrivacyOptionConfigure(WLAN_DEV_INFO *pDevInfo, A_BOOL mixedPrivacyOption)
{
    pDevInfo->staConfig.mixedPrivacyAllow = mixedPrivacyOption;
    return;
}

A_STATUS
ccxMixedPrivacyPolicyCheck(WLAN_DEV_INFO *pDevInfo)
{
    if (pDevInfo->staConfig.mixedPrivacyAllow) {
        return A_OK;
    } else {
        return A_ERROR;
    }
}

void
ccxAironetIeProcess(WLAN_DEV_INFO *pDevInfo, BSSDESCR *pBss, AIRONET_IE *pAie)
{
    ASSERT(pBss);
    
    /* If the IE exists, then it must be a Cisco AP. */
    pBss->isCiscoAp = TRUE;
    
    pBss->ckipEnabled = pBss->ckipUseMic = FALSE;

    if (pAie->flags & AIE_CKIP_KP) {
        pBss->ckipEnabled = TRUE;
    }
    if (pAie->flags & AIE_CKIP_MIC) {
        pBss->ckipUseMic = TRUE;
    }

    /* XXX put in CWmin/CWmax processing here */

    A_BCOPY(pAie->name, pBss->ccxApName, sizeof(pBss->ccxApName));
}

INFO_ELEMENT *
ccxStaIpAddrInsert(WLAN_DEV_INFO *pDevInfo, CCX_AP_ADDR *ccxApAddrIe,
                   A_UINT32 ipAddr)
{

    if (!pDevInfo->bssDescr->isCiscoAp) {
        /*
         * Not associating to an AP that understands this IE, so we
         * don't include it.
         */
        return (INFO_ELEMENT *)ccxApAddrIe;
    }
    
    ccxApAddrIe->elementID = ELE_CCX_AP_ADDR;
    ccxApAddrIe->length    = sizeof(*ccxApAddrIe) - 2;
    A_BCOPY(ouiAironet, ccxApAddrIe->oui, sizeof(ccxApAddrIe->oui));
    ccxApAddrIe->mbz       = 0;
    ccxApAddrIe->ipAddr    = cpu2be32(ipAddr);
    ccxApAddrIe->reserved  = 0;

    /* advance to next IE */
    return ((INFO_ELEMENT *)(ccxApAddrIe + 1));
}

INFO_ELEMENT *
ccxCkipStaRequest(WLAN_DEV_INFO *pDevInfo, BSSDESCR *pBss, AIRONET_IE *pAie)
{
    SIB_ENTRY *lSib = pDevInfo->localSta;

    ASSERT(pAie != NULL);

    if (!pDevInfo->bssDescr->isCiscoAp) {
        /*
         * Not associating to an AP that understands this IE, so we
         * don't include it.
         */
        return (INFO_ELEMENT *)pAie;
    }

    A_MEM_ZERO(pAie, sizeof(*pAie));

    pAie->elementID = ELE_AIRONET;
    pAie->length    = sizeof(*pAie) - 2;

    /* This is required for the AP to listen to us */
    pAie->devType = AIE_DEVICETYPE;

    if (pBss->ckipEnabled || pBss->ckipUseMic) {
        pAie->flags = AIE_CKIP_KP | AIE_CKIP_MIC;
    }

    A_BCOPY(pDevInfo->staConfig.clientName, pAie->name, sizeof(pAie->name));

    return ((INFO_ELEMENT *)(pAie + 1));
}

void
ccxAIEAssocRespProcess(WLAN_DEV_INFO *pDevInfo, AIRONET_IE *pAie)
{
    SIB_ENTRY *lSib = pDevInfo->localSta;

    if (pAie != NULL) {
        lSib->ckipEnabled = (pAie->flags & AIE_CKIP_KP) ? 1 : 0;
        lSib->ckipUseMic  = (pAie->flags & AIE_CKIP_MIC) ? 1 : 0;
    } else {
        lSib->ckipEnabled = 0;
        lSib->ckipUseMic  = 0;
    }
}

void
ccxApAddrAssocRespProcess(WLAN_DEV_INFO *pDevInfo, CCX_AP_ADDR *pApAddr)
{
    if (pApAddr != NULL) {
        A_BCOPY(&pApAddr->ipAddr, &pDevInfo->bssDescr->ccxApAddr,
                sizeof(pApAddr->ipAddr));
    } else {
        A_MEM_ZERO(&pDevInfo->bssDescr->ccxApAddr,
                   sizeof(pDevInfo->bssDescr->ccxApAddr));
    }
}

A_STATUS
ccxAuthStatusProcess(WLAN_DEV_INFO *pDev, void *pData, A_UINT32 dataLen,
                     A_UINT32 *pBytesNeeded)
{
    A_STATUS              status      = A_OK;
    PMH_SUPP_AUTH_STATUS pAuthStatus = (PMH_SUPP_AUTH_STATUS) pData;

    if (sizeof(*pAuthStatus) > dataLen) {
        *pBytesNeeded = sizeof(*pAuthStatus);
        return A_EMSGSIZE;
    }

    switch (pAuthStatus->code) {
    case MH_CCX_RAP_AUTH_SUCCESS:
        if (pDev->staConfig.leapEnabled) {
            ccxRogueListFlush(pDev, &pAuthStatus->Bssid);
        }
        break;
    case MH_CCX_RAP_TIMER_EXPIRE:
        if (pDev->staConfig.leapEnabled) {
            /* record this guy as a loser */
            ccxRogueListAdd(pDev, (WLAN_MACADDR *)&pAuthStatus->Bssid,
                            DDP_RAP_AUTH_TIMER_EXPIRE);
        }
        break;
    case MH_CCX_RAP_AP2STA_CHALLENGE_FAIL:
        if (pDev->staConfig.leapEnabled) {
            /* record this guy as a loser */
            ccxRogueListAdd(pDev, (WLAN_MACADDR *)&pAuthStatus->Bssid,
                            DDP_RAP_CHALLENGE_FROMAP_FAIL);
        }
        break;
    case MH_CCX_RAP_STA2AP_CHALLENGE_FAIL:
        if (pDev->staConfig.leapEnabled) {
            /* record this guy as a loser */
            ccxRogueListAdd(pDev, (WLAN_MACADDR *)&pAuthStatus->Bssid,
                            DDP_RAP_CHALLENGE_TOAP_FAIL);
        }
        break;
    default:
        break;
    }

    return status;
}


#endif /* WLAN_CONFIG_CCX */

