/*
 * $Id: //depot/sw/branches/1.3_USB_LINUX_port/src/USB/wlan/host/sta/hw.c#4 $
 * 
 * Copyright © 2000-2003 Atheros Communications, Inc., All Rights Reserved
 *
 * access hardware/memory buffer/descriptor management routines
 */

#ifdef Linux
#include "wlandrv.h"
#include "wlanext.h"
#include "linuxext.h"
#endif
#include "stacserv.h"
#include "intercept.h"
#include "wdcApi.h"
#include "usbstub.h"

/**************************************************************************
 * hwAllocateDescMemory - allocate memory for descriptor allocation
 *
 * This allocates some physically contiguous memory for descriptor
 * allocation. In the NT test environment, this malloc's an area
 * of memory that will be used for descriptors.
 *
 * RETURNS: status
 */
A_STATUS
hwAllocateDescMemory(WLAN_DEV_INFO *pDevInfo, A_UINT16 memSize)
{
    A_STATUS        status;
    int             i;
    ATHEROS_DESC    *pDesc;

    /* memSize is ignored */
    status = athNdisAllocMem(pDevInfo,
                             NUM_DESCS * DESC_SIZE,
                             &pDevInfo->descPhysAddr,
                             &pDevInfo->pDescMemory);
    if (pDevInfo->pDescMemory == NULL) {
        return A_NO_MEMORY;
    }

    /* Now go ahead and build the descriptor's queue */
    A_SEM_INIT(pDevInfo->emptyDescQueue.qSem, 0, CREATE_UNLOCKED);
    for (i = 0; i < NUM_DESCS; i++) {
        pDesc = &((ATHEROS_DESC *) pDevInfo->pDescMemory)[i];

        pDesc->thisPhysPtr  = pDevInfo->descPhysAddr + sizeof(*pDesc) * i;
        pDesc->pOSHandle    = NULL;
        pDesc->ffRefHack    = 0;

        pDesc->pNextVirtPtr = NULL;
        pDesc->freeQueue    = &pDevInfo->emptyDescQueue;

        descQPushTail(&pDevInfo->emptyDescQueue, pDesc, pDesc);
    }

    A_ATOMIC_SET(&pDevInfo->freeDescCount, NUM_DESCS);

    return A_OK;
}

/**************************************************************************
 * hwAllocateBufMemory - allocate memory for small, large and crypt
 *   buffers; memsize passed in is ignored
 *
 * RETURNS: N/A
 */
A_STATUS
hwAllocateBufMemory(WLAN_DEV_INFO *pDevInfo, A_UINT16 memSize)
{
    ATHEROS_DESC    *pDesc;
    DRV_BUF_DESC    *pBufDesc, *tpBufDesc;
    A_UINT32        physAddr;
    int             i;
    PNDIS_PACKET    pSkb;

    athNdisAllocMem(pDevInfo, NUM_SMALL_BUFF * SMALL_BUFF_SIZE,
                    &pDevInfo->smallbufPhysAddr,
                    &pDevInfo->pSmallbufMemory);
    if (pDevInfo->pSmallbufMemory == NULL) {
        return A_NO_MEMORY;
    }

    descQInit(&pDevInfo->smallBufDescQueue);
    for (i = 0; i < NUM_SMALL_BUFF; i++) {
        pDesc = descQPopHead(&pDevInfo->emptyDescQueue);
        if (pDesc == NULL) {
            A_ATOMIC_SET(&pDevInfo->freeDescCount,0);
            return A_NO_MEMORY;
        }

        pDesc->pBufferVirtPtr.ptr = (char *)pDevInfo->pSmallbufMemory + (i * SMALL_BUFF_SIZE);
        pDesc->bufferPhysPtr      = pDevInfo->smallbufPhysAddr + (i * SMALL_BUFF_SIZE);
        pDesc->pOrigBufferVirtPtr = pDesc->pBufferVirtPtr.ptr;
        pDesc->pOrigbufferPhysPtr = pDesc->bufferPhysPtr;

        /* set default free queue - will be changed to home queue later */
        pDesc->freeQueue = &pDevInfo->smallBufDescQueue;
        descQPushTail(&pDevInfo->smallBufDescQueue, pDesc, pDesc);
    }
    A_ATOMIC_SUB(NUM_SMALL_BUFF, &pDevInfo->freeDescCount);


    ASSERT(sizeof(DRV_BUF_DESC) % sizeof(A_UINT32) == 0);
    ASSERT((LARGE_BUFF_SIZE - sizeof(DRV_BUF_DESC) - LARGE_BUFF_COOKIE_SIZE) >= AR_BUF_SIZE);

    /*
     * For large buffers, allocate one at a time, add it to a linked list and attach to a descriptor.
     * If we don't get all all the buffers or descriptors, error out and the caller will free
     * everything previously allocated.
     */
    descQInit(&pDevInfo->largeBufDescQueue);
    for (i = 0; i < NUM_LARGE_BUFF; i++) {
        athNdisAllocMem(pDevInfo, LARGE_BUFF_SIZE, &physAddr, &pBufDesc);
        if (pBufDesc == NULL) {
            return A_NO_MEMORY;
        }
        pBufDesc->nextBufDesc  = NULL;
        pBufDesc->thisPhysAddr = physAddr;
        A_CORRUPTION_BUG_INIT(pBufDesc);

        if (pDevInfo->pLargebufMemory == NULL) {
            /* first buffer alloc'd */
            pDevInfo->pLargebufMemory = pBufDesc;
        } else {
            /* chain subsequent buffers */
            tpBufDesc->nextBufDesc = pBufDesc;
        }
        tpBufDesc = pBufDesc;

        pDesc = descQPopHead(&pDevInfo->emptyDescQueue);
        if (pDesc == NULL) {
            A_ATOMIC_SET(&pDevInfo->freeDescCount, 0);
            return A_NO_MEMORY;
        }

        A_ATOMIC_DEC(&pDevInfo->freeDescCount);

        pDesc->pBufferVirtPtr.ptr = (char *)pBufDesc + sizeof(struct drvBufDesc);
        pDesc->bufferPhysPtr      = physAddr + sizeof(struct drvBufDesc);
        pDesc->pOrigBufferVirtPtr = pDesc->pBufferVirtPtr.ptr;
        pDesc->pOrigbufferPhysPtr = pDesc->bufferPhysPtr;

        /* set default free queue - will be changed to home queue later */
        pDesc->freeQueue = &pDevInfo->largeBufDescQueue;
        descQPushTail(&pDevInfo->largeBufDescQueue, pDesc, pDesc);
    }

    /*
     * Allocate two additional buffers for software encryption/decryption
     * and link them into the above allocated largebuf memory area chain
     */
    athNdisAllocMem(pDevInfo, LARGE_BUFF_SIZE, &physAddr, (void **)&pBufDesc);
    if (pBufDesc == NULL) {
        return A_NO_MEMORY;
    }
    pBufDesc->nextBufDesc  = NULL;
    pBufDesc->thisPhysAddr = physAddr;
    A_CORRUPTION_BUG_INIT(pBufDesc);
    tpBufDesc->nextBufDesc = pBufDesc;   /* chain to above chain */
    tpBufDesc = pBufDesc;
    pDevInfo->virtEncryptBuf = (char *)pBufDesc + sizeof(struct drvBufDesc);
    pDevInfo->physEncryptBuf = physAddr + sizeof(struct drvBufDesc);

    athNdisAllocMem(pDevInfo, LARGE_BUFF_SIZE, &physAddr, (void **)&pBufDesc);
    if (pBufDesc == NULL) {
        return A_NO_MEMORY;
    }
    pBufDesc->nextBufDesc  = NULL;
    pBufDesc->thisPhysAddr = physAddr;
    A_CORRUPTION_BUG_INIT(pBufDesc);
    tpBufDesc->nextBufDesc = pBufDesc;   /* chain to above chain */
    tpBufDesc = pBufDesc;
    pDevInfo->virtDecryptBuf = (char *)pBufDesc + sizeof(struct drvBufDesc);
    pDevInfo->physDecryptBuf = physAddr + sizeof(struct drvBufDesc);

    return A_OK;
}

/**************************************************************************
 * createBuffandDesc - create a physically contiguous buffer and descriptor
 *
 * Create a physically contiguous data buffer and create a descriptor
 * to go with it.  An ATHEROS_DESC gets returned, pointer to data buffer
 * can be assertained from the descriptor. Since we maintain two queues
 * rxbuf and smallbuf, we return a desc/buf pair from the correct queue.
 *
 * RETURNS: OK if successful, ERROR otherwise
 */
A_STATUS
createBuffandDesc(WLAN_DEV_INFO *pDevInfo, A_UINT16 buffSize, ATHEROS_DESC **ppDesc)
{
    A_UINT32        buffPhysAddr;   /* physical address of the buffer */
    void            *pBuffVirtAddr; /* virtual address of the buffer */
    A_STATUS        status;         /* to check functions status */
    ATHEROS_DESC    *pDesc;         /* pointer to atheros descriptor */

    /* Get the physically contiguous buffer */
    if (buffSize <= SMALL_BUFF_SIZE) {
        /* get one from the small pool */
        A_SEM_LOCK((pDevInfo->smallBufDescQueue).qSem, WAIT_FOREVER);
        pDesc = descQPopHead(&pDevInfo->smallBufDescQueue);
        A_SEM_UNLOCK((pDevInfo->smallBufDescQueue).qSem);
    } else {
        /* get one from the large i.e. receive pool */
        ASSERT(buffSize <= AR_BUF_SIZE);
        A_SEM_LOCK((pDevInfo->largeBufDescQueue).qSem, WAIT_FOREVER);
        pDesc = descQPopHead(&pDevInfo->largeBufDescQueue);
        A_SEM_UNLOCK((pDevInfo->largeBufDescQueue).qSem);
    }

    if (!pDesc) {
        return A_NO_MEMORY;
    }

    /* clean up the descriptor */
    A_DESC_INIT(pDesc);

    /* Fill in the descriptor fields */
    pDesc->frameLength  = buffSize;
    pDesc->bufferLength = buffSize;

    pDesc->pOSHandle      = NULL;
    pDesc->OSbuffer       = NULL;
    pDesc->OSspecificInfo = 0;

    *ppDesc = pDesc;

    return A_OK;
}

/**************************************************************************
 * freeBuffandDesc - free buffer and descriptor pair
 *
 * Free the buffer and descriptor, created with createBuffandDesc(), back
 * to the queue from whence it came
 *
 * RETURNS: N/A
 */
void
freeBuffandDesc(WLAN_DEV_INFO *pDevInfo, ATHEROS_DESC *pDesc)
{
    NDIS_HANDLE     pHandle = pDevInfo->pOSHandle->NicAdapterHandle;
    PNDIS_PACKET    pPacket = pDesc->pOSHandle;
    WDC_BUFFER_DESC bufferDesc;

    /* Unlink it from reset of the chain */
    pDesc->pNextVirtPtr = NULL;

    if (pPacket) {
        
        ASSERT(pDesc->freeQueue != &pDevInfo->rxQueue);

        /* For deserialized driver, we always need call NdisMSendComplete
         * to complete the packet, if anything is wrong in send path, 
         * we need complete the packet in here.
         */

#ifdef Linux
        /* Instead of doing this we should ideally place it under OSPKTREF_FREE
         * but that affects code in other places */
        pDesc->pOSHandle = NULL;
#endif
        if (MP_RSRVD_GET_REF_COUNT(pPacket) == 0) {
            ATH_RELEASE_SPINLOCK(pDevInfo->sendLock);
            if (pDevInfo->NdisResetInProgress) {
                /*
                 * LW
                 * Deserialized and connection-oriented miniport drivers must 
                 * return NDIS_STATUS_REQUEST_ABORTED for any queued send packets
                 * from MiniportReset.
                 */
                NdisMSendComplete(pHandle, pPacket, NDIS_STATUS_REQUEST_ABORTED);
            } else {
                NdisMSendComplete(pHandle, pPacket, NDIS_STATUS_SUCCESS);
            }
            drainSendWaitQueue(pDevInfo);

            ATH_ACQUIRE_SPINLOCK(pDevInfo->sendLock);
            pDevInfo->localSta->stats.GoodTransmits ++;

            NdisInterlockedDecrement(&pDevInfo->pOSHandle->txFramesPended);
        }
    }

    if (pDesc->freeQueue == &pDevInfo->emptyDescQueue) {

        if (pDesc->ffRefHack) {
            ATHEROS_DESC *pRxDesc = (ATHEROS_DESC *)(pDesc->ffRefHack);

            pDesc->ffRefHack = 0;
            ASSERT(pDesc->OSbuffer == NULL);

            ASSERT(pRxDesc->freeQueue == &pDevInfo->rxQueue);
            ASSERT(pRxDesc->ffRefHack);
            if (!(--pRxDesc->ffRefHack)) {
                pRxDesc->ffRefHack = 1;
                A_SEM_LOCK((pRxDesc->freeQueue)->qSem, WAIT_FOREVER);
                descQPushTail(pRxDesc->freeQueue, pRxDesc, pRxDesc);
                A_SEM_UNLOCK((pRxDesc->freeQueue)->qSem);
            }
        }


        ASSERT(!pDesc->pOSHandle);

        memFreeDescriptor(pDevInfo, pDesc);

        return;
    }

    ASSERT((pDesc->freeQueue == &pDevInfo->rxQueue) ||
           (pDesc->freeQueue == &pDevInfo->largeBufDescQueue) ||
           (pDesc->freeQueue == &pDevInfo->smallBufDescQueue));

    if (pDesc->freeQueue != &pDevInfo->smallBufDescQueue) {
        A_CORRUPTION_BUG_CHECK(((DRV_BUF_DESC *)pDesc->pOrigBufferVirtPtr) - 1);
    }

    pDesc->pBufferVirtPtr.ptr = pDesc->pOrigBufferVirtPtr;
    pDesc->bufferPhysPtr      = pDesc->pOrigbufferPhysPtr;

    if (pDesc->freeQueue == &pDevInfo->rxQueue) {
        ASSERT(pDesc->ffRefHack);
        if (--pDesc->ffRefHack) {
            return;
        }
        pDesc->ffRefHack = 1;

        if (!pDevInfo->targetStopRec) {
            /* xxxxPGSxxxx - TODO - don't do this during detach time when target completes all rx buffers!!!! */
            /* Give host rx buffer back to target */
            /* xxxxPGSxxxx - TODO - need WDC_BUFFER_DESC management routines (especially for tx) */

            pDevInfo->targetRxPending++;

            // pDesc->pBufferVirtPtr.ptr   = NULL;
            // pDesc->bufferPhysPtr        = (A_UINT32) NULL;
            // XXX: status
            wdcRxDone(pDevInfo->targetHandle, pDesc->wdcRxInfo);
        }

        A_SEM_LOCK((pDesc->freeQueue)->qSem, WAIT_FOREVER);
        descQPushTail(pDesc->freeQueue, pDesc, pDesc);
        A_SEM_UNLOCK((pDesc->freeQueue)->qSem);
            /* buffer given to target, no need to push onto host rx queue */
        return;
    }

    A_SEM_LOCK((pDesc->freeQueue)->qSem, WAIT_FOREVER);
    descQPushTail(pDesc->freeQueue, pDesc, pDesc);
    A_SEM_UNLOCK((pDesc->freeQueue)->qSem);
}

/**************************************************************************
 * initGPIO
 *
 * Initialize GPIO pins that are specified in the registry
 *
 * RETURNS: A_OK if all well, else error code
 */
A_STATUS
initGPIO(WLAN_DEV_INFO *pDevInfo)
{
    A_UINT32    i;
    A_STATUS    status = A_OK;

    for (i = 0; i < NUM_GPIO_FUNCS; i++) {
        // TODO: 6 GPIO pins on all current h/w; should be retrieved from accessor func
        if (pDevInfo->staConfig.gpioPinFuncs[i] < 6) {
            pDevInfo->gpioFunc[i].enabled = TRUE;
            pDevInfo->gpioFunc[i].pin = pDevInfo->staConfig.gpioPinFuncs[i];
        } else {
            pDevInfo->gpioFunc[i].enabled = FALSE;
        }
    }

    status = initGpioFunc0(pDevInfo);
    status = initGpioFunc1(pDevInfo);

    return status;
}

/**************************************************************************
 * initGpioFunc0
 *
 * Initialize General Purpose I/O function 0
 *
 * Description: Setup the hardware for output on the GPIO pin for this function
 *
 * RETURNS: A_OK if all well, else error code
 */
A_STATUS
initGpioFunc0(WLAN_DEV_INFO *pDevInfo)
{
    A_STATUS    status = A_OK;

    if (pDevInfo->gpioFunc[0].enabled) {
        halGpioCfgOutput(pDevInfo, pDevInfo->gpioFunc[0].pin);
    } else {
        status = A_ENOTSUP;
    }

    return status;
}

/**************************************************************************
 * initGpioFunc1
 *
 * Initialize General Purpose I/O function 1
 *
 * Description: Setup the hardware for output on the GPIO pin for this function
 *
 * RETURNS: A_OK if all well, else error code
 */
A_STATUS
initGpioFunc1(WLAN_DEV_INFO *pDevInfo)
{
    A_STATUS    status = A_OK;

    if (pDevInfo->gpioFunc[1].enabled) {
        halGpioCfgOutput(pDevInfo, pDevInfo->gpioFunc[1].pin);
    } else {
        status = A_ENOTSUP;
    }

    return status;
}

/**************************************************************************
 * gpioFunc0
 *
 * General Purpose I/O function 0
 *
 * Description: Sets/clears the GPIO pin associated with this function
 * according to the input parm.  Could be used to light up an LED.
 *
 * RETURNS: A_OK if all well, else error code
 */
A_STATUS
gpioFunc0(WLAN_DEV_INFO *pDevInfo, A_BOOL enable)
{
    A_STATUS    status = A_OK;

    if (pDevInfo->gpioFunc[0].enabled) {
        halGpioSet(pDevInfo, pDevInfo->gpioFunc[0].pin, enable);
    } else {
        status = A_EINVAL;
    }

    return status;
}

/**************************************************************************
 * gpioFunc1
 *
 * General Purpose I/O function 1
 *
 * Description: Sets/clears the GPIO pin associated with this function
 * according to the input parm.  Is used to light up LEDs (active low).
 *
 * RETURNS: A_OK if all well, else error code
 */
A_STATUS
gpioFunc1(WLAN_DEV_INFO *pDevInfo, A_BOOL enable)
{
    A_STATUS    status = A_OK;

    if (pDevInfo->gpioFunc[1].enabled) {
        halGpioSet(pDevInfo, pDevInfo->gpioFunc[1].pin, !enable);
        pDevInfo->gpioFunc[1].state = enable;
    } else {
        status = A_EINVAL;
    }

    return status;
}

