/*************************************************************************
 * Copyright  2000-2003 Atheros Communications, Inc., All Rights Reserved
 *
 * Atheros and the Atheros logo and design are trademarks of Atheros
 * Communications, Inc.
 *
 * $Id: //depot/sw/branches/1.3_USB_LINUX_port/src/USB/wlan/host/sta/intercept.c#1 $
 *
 * Interceptor routines used for debugging, profiling, etc.
 *
 * Notes for use
 * -------------
 * Define INTERCEPT_ENABLED in the build configuration of your choosing.  This
 * will work in both debug and production builds, but DEBUG_PRINTF will need to
 * be defined in the latter in order to effectively see the profiling results.
 *
 * Profiling in enabled from driver load time.  All driver entry points are
 * currently tagged for profiling, although the code is structured to be flexible
 * enough so that one can add iceptEntry() and iceptExit() around any block of
 * code to profile it (as long as a new location is defined and space is allocated
 * in the array).
 *
 * To display the profiling output break into SoftICE and type "e dumpIt 1" (debug
 * build) or "e <xxxx> 1" (production build) where <xxxx> is the address of dumpIt
 * that is displayed during iceptInit().
 */

#include "wlantype.h"
#include "intercept.h"

#ifdef INTERCEPT_ENABLED

/*
 * The following is not to be included in NDIS miniports (WHQL tests will fail).
 * We include it here only as a special compile-time option.
 */
LARGE_INTEGER
KeQueryPerformanceCounter(PLARGE_INTEGER PerformanceFrequency);

typedef struct ProfileInfo {
    A_BOOL   in;
    A_UINT32 ticks;
    A_UINT32 ticksMax;
#define PROFILE_HISTORY 128
    A_UINT32 history[PROFILE_HISTORY];
    A_UINT32 historyIdx;
} PROFILE_INFO;

PROFILE_INFO profiles[MAX_ENTRY_POINTS];

char *profNames[MAX_ENTRY_POINTS] = {
    "STAPOLL",      // 0
    "OIDGET",       // 1
    "OIDSET",       // 2
    "SEND",         // 3
    "RETURNRX",     // 4
    "CHECKHANG",    // 5
    "SMETIMER",     // 6
    "CAL",          // 7
    "ISR",          // 8
    "DPC",          // 9
    "INITIALIZE",   // 10
    "HALT",         // 11
    "SHUTDOWN",     // 12
    "RESET",        // 13
    "RESETDONE",    // 14
    "PNP",          // 15
    "SLEEPTRACK",   // 16
    "LEAP",         // 17
    "MACSTOP",      // 18
    "HIBERNATE",    // 19
    "LED",          // 20
    "?????????"     // 21
};

/*
 * Set the following value to something non-zero from within the debugger
 * to dump the profiling statistics gathered up to that point, and to reset
 * the stored maximums.
 */
int dumpIt = 0;

/*
 * You can set the following trigger to some value (in microseconds).  If we
 * spend longer than the trigger time in the driver, the profiling code will
 * drop into SoftICE.
 */
A_UINT32 profileTrigger = 0xffffffff;

/*
 * We use this in lieu of ASSERT(0) as the profling code may be enabled in free
 * builds.
 */
#define profileEnterDebugger() __asm { int 3 };

#ifdef DEBUG
int profileDebugLevel = 0;
#define profilePrintf if (profileDebugLevel) uiPrintf
#else
#define profilePrintf
#endif

static A_UINT32
profileTickGet()
{
    LARGE_INTEGER li;
    LARGE_INTEGER li2;

    li = KeQueryPerformanceCounter(&li2);

    return li.u.LowPart / 4; // heuristically determined 4 ticks per us
}

static A_UINT32
profileDump(ICEPT_ENUM locn)
{
    PROFILE_INFO *pInfo;
    A_UINT32     i, max;

    ASSERT(locn < MAX_ENTRY_POINTS);

    pInfo = &profiles[locn];

    uiPrintf("Dump of %s, max = %d\n", profNames[locn], pInfo->ticksMax);

    for (i = 0; i < PROFILE_HISTORY; i++) {
        if (pInfo->history[i] == 0) {
            continue;
        }

        uiPrintf("idx: %3d -> %6d\n", i, pInfo->history[i]);
        i++;
    }

    max = pInfo->ticksMax;
    pInfo->ticksMax = 0;

    return max;
}

VOID
iceptInit()
{
    int i;

    profilePrintf(">iceptInit\n");

    /*
     * For optimized builds sometimes the symbols don't match the actual memory
     * location of the variable, so we print it out here so that one can type
     * in SoftICE, "e <addr> 1" instead of "e pleaseDump 1"
     */
    uiPrintf(">>> dumpIt located at 0x%08x\n", &dumpIt);

    for (i = 0; i < MAX_ENTRY_POINTS; i++) {
        int          j;
        PROFILE_INFO *pInfo = &profiles[i];

        pInfo->in         = FALSE;
        pInfo->ticks      = 0;
        pInfo->ticksMax   = 0;
        pInfo->historyIdx = 0;

        for (j = 0; j < PROFILE_HISTORY; j++) {
            pInfo->history[j] = 0;
        }
    }

    profilePrintf("<iceptInit\n");
}

VOID
iceptEntry(ICEPT_ENUM locn)
{
    PROFILE_INFO *pInfo;

    ASSERT(locn < MAX_ENTRY_POINTS);

    profilePrintf(">iceptEntry: %s\n", profNames[locn]);

    pInfo = &profiles[locn];

    /* Protect against re-entry */
    ASSERT(!pInfo->in);

    pInfo->ticks = profileTickGet();
    pInfo->in    = TRUE;

    profilePrintf("<iceptEntry\n");
}

VOID
iceptExit(ICEPT_ENUM locn)
{
    PROFILE_INFO *pInfo;
    A_UINT32     delta;

    ASSERT(locn < MAX_ENTRY_POINTS);

    pInfo = &profiles[locn];

    /* Ensure that iceptEntry() was called first */
    ASSERT(pInfo->in);

    delta = profileTickGet() - pInfo->ticks;

    profilePrintf(">iceptExit: %s, delta = %d\n", profNames[locn], delta);

    if (delta > pInfo->ticksMax) {
        pInfo->ticksMax = delta;
    }

    if (delta > profileTrigger) {
        profileEnterDebugger();
    }

    pInfo->history[pInfo->historyIdx] = delta;
    pInfo->historyIdx++;
    pInfo->historyIdx %= PROFILE_HISTORY;
    pInfo->in          = FALSE;

    profilePrintf("<iceptExit\n");
}

VOID
iceptPoll()
{
    A_UINT32 i, max = 0, locn;

    if (!dumpIt) {
        return;
    }

    for (i = 0; i < MAX_ENTRY_POINTS; i++) {
        A_UINT32     temp;
        PROFILE_INFO *pInfo = &profiles[i];

        if (pInfo->ticksMax == 0) {
            continue;
        }

        temp = profileDump(i);
        if (temp > max) {
            max  = temp;
            locn = i;
        }
    }

    uiPrintf("Overall max: %d @ %s\n", max, profNames[locn]);

    dumpIt = 0;

    profileEnterDebugger(); // Stop here to look at the output
}

#endif // INTERCEPT_ENABLED
