/************************************************************************
 * $Id: //depot/sw/branches/1.3_USB_LINUX_port/src/USB/wlan/host/sta/interrup.c#1 $
 * 
 * interrupt routines for NDIS driver
 *
 * Copyright  2000-2003 Atheros Communications, Inc., All Rights Reserved
 *
 * Atheros and the Atheros logo and design are trademarks of Atheros
 * Communications, Inc.
 *
 * Sample Code from Microsoft Windows 2000 Driver Development Kit is
 * used under license from Microsoft Corporation and was developed for
 * Microsoft by Intel Corp., Hillsboro, Oregon: Copyright (c) 1994-1997
 * by Intel Corporation.
 */

#include "wlantype.h"
#include "wlandrv.h"
#include "wlanext.h"
#include "ndisdrvext.h"
#include "wlanCapDe.h"
#include "wlanfragde.h"
#include "wlansmeext.h"
#include "stacserv.h"
#include "halApi.h"
#include "intercept.h"
#if defined(PCI_INTERFACE)
#include "arTransmit.h"

#define NUM_DPC_LOOPS   2

BOOLEAN synchronizedGetIsrReg(PVOID pContext)
{
    WLAN_DEV_INFO *pDev    = (WLAN_DEV_INFO *) pContext;
    pDev->isrValue         = pDev->globISRReg;
    pDev->globISRReg = 0;
    return TRUE;
}

/*****************************************************************************
 * Procedure:   athIsr (MiniportIsr in DDK)
 *
 * Description: This is the interrupt service routine.  It will check to see
 *              if there are any interrupts pending, and if there are, it will
 *              disable board interrupts and schedule a HandleInterrupt
 *              callback.
 *
 * Arguments:
 *  MiniportAdapterContext   The context value returned by the Miniport
 *                           when the adapter was initialized
 *
 * Result:
 *  InterruptRecognized          Returns TRUE if the interrupt belonged to
 *                               this adapter, FALSE otherwise.
 *  QueueMiniportHandleInterrupt Returns TRUE if we want a callback to
 *                               our DPC.
 */
ENTRY_FN_VOID(athIsr,
              (PBOOLEAN pIsRecognized, PBOOLEAN pQueueHandler, NDIS_HANDLE context),
              (pIsRecognized, pQueueHandler, context), ICEPT_ISR)
{
    WLAN_DEV_INFO   *pDev;
    HAL_INT_TYPE    isrValue, unmaskedIsrValue;
    A_UINT32        descQueueBitMask = 0;
    HAL_INT_TYPE    mask = 0;

    pDev = DEVINFO_FROM_CONTEXT(context);

    *pQueueHandler = FALSE;

    *pIsRecognized = halIsInterruptPending(HACK_TO_PARDEV(pDev));
    if (!(*pIsRecognized)) {
        /* Must not be our interrupt */
        return;
    }

    isrValue = halGetInterrupts(HACK_TO_PARDEV(pDev), &unmaskedIsrValue, &descQueueBitMask);

    /*
     * Do this after the call to halGetInterrupts() so that we clear any
     * interrupt condition, even in a zombie state.
     */
    if (!pDev->pOSHandle->openForBusiness) {
        return;
    }

    /* Check for suprise removal of our card */
    if (isrValue == HAL_INT_NOCARD) {
        uiPrintf("athIsr: Card Removed!\n");
        *pIsRecognized = FALSE;
        return;
    }

    if (isrValue & HAL_INT_GPIO) {
        /* rfkill interrupt from GPIO: flip the polarity of the GPIO interrupt */
        if (halGpioGet(pDev, pDev->rfSilent.gpioSelect) == pDev->rfSilent.polarity) {
            halGpioSetIntr(pDev, pDev->rfSilent.gpioSelect, !pDev->rfSilent.polarity);
        } else {
            halGpioSetIntr(pDev, pDev->rfSilent.gpioSelect, pDev->rfSilent.polarity);
        }
    }

    if (isrValue & HAL_INT_RXORN) {
        /*
         * On some HW, RXORN will continue to assert on each new arriving frame.
         * This would keep us from ever entering our DPC, so we must disable
         * the interrupt here in the ISR to avoid this condition.
         */
        pDev->localSta->stats.RcvDmaOverrunErrors++;
        mask |= HAL_INT_RXORN;
    }

    if (isrValue & HAL_INT_TXURN) {
        /*
         * After a frame started transmitting, the NIC was unable to get the rest of the
         * data from host memory in time before the transmitter needed it.  Adjust the tx
         * trigger up by half the difference between maximum and the current value.
         */
        uiPrintf("athIsr: Transmit Underrun!\n");
        halUpdateTxTrigLevel(HACK_TO_PARDEV(pDev), TRUE);
        isrValue &= ~HAL_INT_TXURN;
    }

    if (isrValue & HAL_INT_RX) {
        mask |= HAL_INT_RX | HAL_INT_TX;
    }

    if (isrValue & HAL_INT_TX) {
        mask |= HAL_INT_TX | HAL_INT_RX;
    }

    if (mask) {
        halDisableInterrupts(HACK_TO_PARDEV(pDev), mask);
    }

    if (isrValue & HAL_INT_TXDESC) {
            
        /* 
         * TODO :Queue Num is used to just pick the burst queue in the 
         * baseBSS eventually it has to reverse looked up to indentify 
         * the BSS it belongs to and then the burst queue of the BSS 
         * should be used here
         */
       if (IS_QUEUE_BIT_SET(descQueueBitMask, 
                             pDev->baseBss.burstQueue.halTxQueueNum))
       {
            QUEUE_DETAILS   *pQueue;
        
            ASSERT(pDev->staConfig.abolt & ABOLT_BURST);
            pQueue = &pDev->baseBss.burstQueue;
            /* Check to see if we have to trigger another burst, which is 
             * decided based on pending burst counts and if there is no 
             * pending operations on the Queue 
             */
            if (!pQueue->queuingBurst) {
                if (pQueue->qBurstCount > 0) {
                    pQueue->qBurstCount -= 1;
                    halStartTxDma(HACK_TO_PARDEV(pDev),pQueue->halTxQueueNum);
                }
            }
            pQueue->queuingBurst = FALSE;
            isrValue &= ~HAL_INT_TXDESC;
        }
    }

    /*
     * We check to see if any RX or TX ints are present in the ISR but not
     * actually asserting because they are masked.  This keeps us from dropping
     * interrupts and leaving completed frames on the RX or TX queues.
     *
     * If any masked or unmasked interrupts have not been handled in the ISR
     * we instruct the OS to queue up our DPC to process the rest of the
     * interrupts at a lower IRQL.
     */
    unmaskedIsrValue &= HAL_INT_RX | HAL_INT_TX;
    if (isrValue || unmaskedIsrValue) {
        pDev->globISRReg |= isrValue | unmaskedIsrValue;
        *pQueueHandler = TRUE;
    }
}

/*****************************************************************************
 * Procedure:   athHandleInterrupt (DPC)
 *
 * Description: This routine is queued by the ISR when some interrupt
 *              processing needs to be done.  It's main job is to call the
 *              interrupt processing code (process receives, cleanup completed
 *              transmits, etc). It will only be called with the adapter's
 *              interrupts masked. This is the DPC for this driver.
 *
 * Arguments:
 *  context     The context value returned by the
 *              Miniport when the adapter was initialized.
 */
ENTRY_FN_VOID(athHandleInterrupt, (NDIS_HANDLE context), (context), ICEPT_DPC)
{
    HAL_INT_TYPE    isrValue, enableMask = 0;
    WLAN_DEV_INFO   *pDev;
    int             i;
    A_STATUS        savedPowerMode;

    ASSERT(context);

    pDev = DEVINFO_FROM_CONTEXT(context);

    if (!pDev->pOSHandle->openForBusiness) {
        return;
    }

    /* After entering hibernate, ignore subsequent DPCs */
    if (pDev->powerMgmt.powerState == D3_STATE) {
        return;
    }

    /* Ensure HW is awake at least until DPC exits */
    ATH_ACQUIRE_SPINLOCK(pDev->lock);
    powerSet(pDev, WAKE_UP, SET_PWR_TO_3, FALSE, 0);
    ATH_RELEASE_SPINLOCK(pDev->lock);

    /*
     * We loop through NUM_DPC_LOOPS times to avoid some of the ISR->DPC
     * latency during heavy CPU load.
     */
    for (i = 0; i < NUM_DPC_LOOPS; i++) {
        /* Read and then clear the ISR value that the ISR might modify. */
        halDisableInterrupts(HACK_TO_PARDEV(pDev), HAL_INT_GLOBAL);
        NdisMSynchronizeWithInterrupt(&pDev->pOSHandle->Interrupt, 
            synchronizedGetIsrReg, (PVOID) pDev);
        isrValue = pDev->isrValue;
        halEnableInterrupts(HACK_TO_PARDEV(pDev), HAL_INT_GLOBAL);

        if (!isrValue) {
            break;
        }

        if (isrValue & HAL_INT_GPIO) {
            /* GPIO interrupt received for switching RF on or off. */
            if (pDev->rfSilent.polarity == 
                halGpioGet(pDev, pDev->rfSilent.gpioSelect))
            {
                /* Switch closed, turn off RF */
                uiPrintf("RF disable switch closed\n");
                athDrvRadioDisable(pDev, &pDev->rfSilent.hwRadioDisable);
                /* Radio is off and chips put to sleep */
            } else {
                /* switch opened, activate now. */
                 uiPrintf("RF disable switch opened\n");
                 athDrvRadioEnable(pDev, &pDev->rfSilent.hwRadioDisable);
            }
            /*
             * Either no further h/w access allowed (disable) or chip is now
             * awake to receive subsequent interrupts; time to leave
             */
            ATH_ACQUIRE_SPINLOCK(pDev->lock);
            powerSet(pDev, WAKE_UP, REDUCE_PWR_FROM_3, FALSE, 0);
            ATH_RELEASE_SPINLOCK(pDev->lock);
            return;
        }

        /*
         * RXORN is fatal for some HW, and merely unfortunate on others.  If it
         * is fatal, the HAL_INT_FATAL bit will be set in the value returned
         * from halGetInterrupts().
         */
        if (isrValue & HAL_INT_RXORN) {
            /*
             * RXORNs are not fatal anymore as updated in bug 9208, and can 
             * happen quite frequently on Venice/Hainan based MAC with 
             * Compression and Fast Frames Enabled 
             */ 
            enableMask |= HAL_INT_RXORN;
        }

        /* Check for fatal interrupts */
        if (isrValue & HAL_INT_FATAL) {
            uiPrintf("athHandleInterrupt: Fatal error!  Resetting...\n");
            //only reset can run at one time
            ATH_ACQUIRE_SPINLOCK(pDev->lock);
            ResetOnError(pDev, DONT_CLEAR_TX_Q, RESET_INC_CTR, DONT_WAIT);

            /* 
             * If master cycle abort went off, maybe returning cache line size
             * to its' original value will prevent recurrence 
             */
            uiPrintf("athHandleInterrupt: Reverting to previous cache line size\n");
            NdisWritePciSlotInformation(pDev->pOSHandle->NicAdapterHandle,
                                        0, PCI_CACHE_LINE_REGISTER,
                                        &pDev->staConfig.cacheLineSize, 0x1);
            powerSet(pDev, WAKE_UP, REDUCE_PWR_FROM_3, FALSE, 0);
            ATH_RELEASE_SPINLOCK(pDev->lock);
            return;
        }

        if ( (isrValue & HAL_INT_BMISS) &&
             (pDev->localSta->staState & STATE_CONNECTED) == STATE_CONNECTED )
        {
            /*
             * We missed some beacons. We assume the AP is either down or we
             * have moved out of range. We inform connection services to take
             * appropriate action.
             */
            ATH_ACQUIRE_SPINLOCK(pDev->lock);
            cservBmissNotify(pDev);
            ATH_RELEASE_SPINLOCK(pDev->lock);
        }

        if (isrValue & HAL_INT_RX) {
            ATH_ACQUIRE_SPINLOCK(pDev->lock);
            ProcessRXInterrupt(pDev);
            ATH_RELEASE_SPINLOCK(pDev->lock);
            enableMask |= HAL_INT_RX | HAL_INT_TX;
        }

        if (isrValue & HAL_INT_TX) {
            ATH_ACQUIRE_SPINLOCK(pDev->lock);
            arProcessTXInterrupt(HACK_TO_PARDEV(pDev));
            ATH_RELEASE_SPINLOCK(pDev->lock);
            enableMask |= HAL_INT_RX | HAL_INT_TX;
        }

        if (isrValue & HAL_INT_DTIM) {
// RE-INSERT WHEN H/W PROCESSING OF BEACONS IS INTEGRATED
//            powerTimIndication(pDev, FALSE, TRUE);
        }

    } /* for (i < NUM_DPC_LOOPS) */

    /* If we disabled some interrupts in the ISR, re-enable them here. */
    halEnableInterrupts(HACK_TO_PARDEV(pDev), enableMask);

    /* Restore power to former or desired state */
    ATH_ACQUIRE_SPINLOCK(pDev->lock);
    powerSet(pDev, WAKE_UP, REDUCE_PWR_FROM_3, FALSE, 0);
    ATH_RELEASE_SPINLOCK(pDev->lock);
}

#endif
