/*
 * $Id: //depot/sw/branches/1.3_USB_LINUX_port/src/USB/wlan/host/sta/leapsupp.c#3 $
 *
 * Copyright (c) 2002 Atheros Communications Inc., All Rights Reserved
 *
 * Support for LEAP on the client.
 */

#include "wlantype.h"
#include "wlandrv.h"
#include "wlanframe.h"
#include "wlanext.h"
#ifndef Linux
#include "ndisdrvext.h"
#endif
#include "stacserv.h"
#include "intercept.h"
#include "wlanCapDe.h"

#include "wlansmeext.h"

#include "eap.h"
#include "md5.h"
#include "md4.h"
#include "crypto/rc4.h"
#include "crypto/des.h"
#include "ccx.h"
#include "ccxdefs.h"
#include "leap.h"
#include "usbstub.h"

/*
 *  This is a supplicant implementation of the Cisco LEAP protocol.
 *  LEAP uses EAPOL-style frame formats to exchange MS-CHAP v2
 *  authentication messages.  It distributes key messages using
 *  key descriptor format 1 (RC4 encrypted).
 */

/* Prototypes */

static void
DesEncrypt(const unsigned char * clear,
           unsigned char * key, unsigned char * cipher);

static A_STATUS
challengeResponseCalc(const A_UINT8 *challenge, const A_UINT8 *passwdHash,
                      A_UINT8 *response);

static A_STATUS
leapSendEapolFrame(WLAN_DEV_INFO *pDev, EAP *pEapHeader, A_UINT8 EapolType,
                   void *pEapData, A_UINT16 dataLen, WLAN_MACADDR *pDestAddr,
                   WLAN_MACADDR *pSrcAddr);
static A_BOOL
isEapolValid(EAPOL_HEADER *, EAP *);

void
leapRestart(WLAN_DEV_INFO *pDevInfo, SIB_ENTRY *pSib, A_BOOL deAuth);

static void
hashPasswd(A_UINT8 *passwd, A_UINT32 passwdLen, A_UINT8 *digest, A_BOOL x);

/* XXX Globals that probably need relocating */

static const int maxRetries       = 3;
static A_BOOL    leapTimerPending = FALSE;

typedef struct {
    A_UCHAR singleHash[16];
    A_UCHAR doubleHash[16];
} LEAP_HASH;
static LEAP_HASH leapHash;

#define LEAPDBG_STATE    0x0001
#define LEAPDBG_CHATTER  0x0002

#ifdef DEBUG
static int leapDebug = 0;
#else
static const int leapDebug = 0;
#endif

#define toMSDefaultKey(x)               ((x) | 0x80000000)

#define ST_ENTRY                       \
        if (leapDebug & LEAPDBG_STATE) \
                uiPrintf("****LEAPstate = %d\n", pSib->leapCtx.leapState)

#define LEAP_CHAT \
        if (leapDebug & LEAPDBG_CHATTER) uiPrintf


/* passwdLen does NOT include a terminating NUL */

static void
MD4Hash(unsigned char * pwd, A_UINT32 pwdlen, unsigned char *out)
{
    MD4_CTX         md4Context;
    unsigned char   buffer[256 * 2];
    unsigned int    i;

    A_MEM_ZERO(buffer, sizeof(buffer));
    for (i=0; i < pwdlen; i++) {
        buffer[i] = *pwd++;
    }

    MD4Init(&md4Context);
    MD4Update(&md4Context, buffer, pwdlen);
    MD4Final(out, &md4Context);
}

/*
 *  Take an ASCII pwd, convert to unicode, then MD4 hash it.
 */
static void
ntpwdhash(unsigned char *pwd, A_UINT32 pwdlen, unsigned char * out )
{
    unsigned char   unicodepwd[256 * 2];
    unsigned int    i;

    A_MEM_ZERO(unicodepwd, sizeof(unicodepwd));

    for (i = 0; i < pwdlen; i++) {
        unicodepwd[i*2] = *pwd++;
    }

    MD4Hash(unicodepwd, (pwdlen * 2), out);
}

/*
 *  Called at driver load time, at passive level.
 */
A_STATUS
leapInit(WLAN_DEV_INFO *pDevInfo)
{
    ASSERT(pDevInfo->staConfig.leapEnabled == TRUE);
    ASSERT(pDevInfo->staConfig.leapInDriver == TRUE);

    /* Prepare passwd for use by zero padding */
    A_MEM_ZERO((char *)&leapHash, sizeof(leapHash));

    /* Calculate the single and double hashes */
    ntpwdhash(pDevInfo->staConfig.leapUserPasswd,
              pDevInfo->staConfig.leapUserPasswdLen,
              leapHash.singleHash);

    MD4Hash(leapHash.singleHash, sizeof(leapHash.singleHash),
            leapHash.doubleHash);

    leapTimerPending = FALSE;

    LEAP_CHAT("LEAP: leapInit:  timeout = %d\n",
              pDevInfo->staConfig.leapTimeout);

    return A_OK;
}

/*
 *  Called after assoc, whenever auth fails
 */
void
leapRestart(WLAN_DEV_INFO *pDevInfo, SIB_ENTRY *pSib, A_BOOL deAuth)
{
    LEAP_CHAT("--- LEAP restart ");
    if (pSib != NULL) {
        LEAP_CHAT("state = %x\n", pSib->leapCtx.leapState);
        if (deAuth) {
            /* Send frame to tell AP to go away */
            wlanMlmeDeauthRequest(pDevInfo, pSib->pOpBss, &pSib->macAddr,
                                  REASON_UNSPECIFIED, TRUE);
        }

        /* Destroy all transient state assoc with this AP */
        A_MEM_ZERO(&pSib->leapCtx, sizeof(pSib->leapCtx));
    }
    LEAP_CHAT("\n");

    /* Disarm timer */
    A_UNTIMEOUT(&pDevInfo->leapTimer);
    leapTimerPending = FALSE;
}

/*
 *  Take a DIX frame and push it into
 *  the LEAP state machine.  Free the buffer and desc at the end.
 */
void
leapReceive(WLAN_DEV_INFO *pDevInfo, ATHEROS_DESC *pDesc)
{
    LAN_FRAME_HEADER *pLanHeader = pDesc->pBufferVirtPtr.lanHeader;
    EAPOL_HEADER    *eapolIn, *eapolOut;
    EAP             *eapIn = NULL, *pEapOut;
    EAP             eapOut;
    A_UINT8         hwKey[16]; // max key len
    A_UINT8         digest[16];
    A_UINT8         savedKeySig[16];
    EAPOL_KEY       *eapolKeyMsg;
    int             i;
    A_UINT16        len;
    A_UINT8         buf[32];
    SIB_ENTRY       *pSib          = pDesc->pSrcSibEntry;
    CSERV_INFO      *pCsInfo       = pDevInfo->pCsInfo;
    ATHEROS_DESC    *pOutDesc      = NULL;
    LEAP_CHALLENGE  *leapChallenge = NULL;
    LEAP_CHALLRESP  *leapChallResp = NULL;
    A_UINT16        keyLen, pktLen;
    A_STATUS        ret;
    /* desc contains 1 byte of key material; allow for more */
    struct {
        NDIS_802_11_KEY  desc;
        A_UINT8          buf[16 - 1];
    } keyDesc;
    A_UINT8         sessionKeyBuf[16 +
                                  LEAP_CHALLENGE_LEN +
                                  LEAP_CHALLRESP_LEN +
                                  LEAP_CHALLENGE_LEN +
                                  LEAP_CHALLRESP_LEN];
    A_UINT64        replay64;

    ASSERT(be2cpu16(pLanHeader->lanTypeOrLen) == ETHERTYPE_EAPOL);

    if ((pDevInfo->localSta->staState & STATE_ASSOC) == 0) {
        LEAP_CHAT("LEAP:  no assoc\n");
        goto boogie;
    }

    /* Set up pointers to header boundaries of received frame */
    eapolIn = (EAPOL_HEADER *)(pLanHeader + 1);
    eapolIn->eapolBodyLength = be2cpu16(eapolIn->eapolBodyLength);
    if (eapolIn->eapolType == EAPOL_TYPE_PACKET) {
        eapIn = (EAP *)(eapolIn + 1);
        /* swap in one place */
        eapIn->eapLength = be2cpu16(eapIn->eapLength);
    }

    if (!isEapolValid(eapolIn, eapIn)) {
        leapRestart(pDevInfo, pSib, TRUE);
        goto boogie;
    }

    /* silently ignore this frame */
    if (eapIn && eapIn->eapLength > EAP_SUCCFAIL_LEN &&
        eapIn->eapType == EAP_TYPE_NOTIFICATION)
    {
        goto boogie;
    }

    pEapOut = &eapOut;

    /* XXX ignore eapIdentifiers smaller than the last one sent */

    /* Take immediate action on a couple of frame types */

    /*
     *  Bootstrap state machine when we get ident request.
     *  We flow into this state silently from any other since
     *  we always want to respond to an AP's ident requests.
     */
    if (eapIn && eapIn->eapCode == EAP_CODE_REQUEST &&
        eapIn->eapLength > EAP_SUCCFAIL_LEN &&
        eapIn->eapType == EAP_TYPE_IDENTITY)
    {
        LEAP_CHAT("+++++ LEAP set state = ST_IDENT_REQ\n");
        pSib->leapCtx.leapState = ST_IDENT_REQ;
        /*
         *  If we're re-keying, we will not have gone thru
         *  mlme.ASSOCIATE, so we need to set the
         *  timer here.
         */
        if (!leapTimerPending) {
            A_TIMEOUT(&pDevInfo->leapTimer, pDevInfo->staConfig.leapTimeout,
                      (A_UINT32)pDevInfo, 0);
            leapTimerPending = TRUE;
        }
    } else if (eapIn && eapIn->eapCode == EAP_CODE_REQUEST &&
               eapIn->eapLength > EAP_SUCCFAIL_LEN &&
               eapIn->eapType != EAP_TYPE_LEAP)
    {

        /* Got request with non-LEAP type, send EAP_NAK */
        pEapOut->eapCode       = EAP_CODE_RESPONSE;
        pEapOut->eapIdentifier = eapIn->eapIdentifier;
        pEapOut->eapLength     = cpu2be16(sizeof(EAP));
        pEapOut->eapType       = EAP_TYPE_NAK;

        /* indicate our desire */
        pEapOut->eapData[0]    = EAP_TYPE_LEAP;

        ret = leapSendEapolFrame(pDevInfo, &eapOut, EAPOL_TYPE_PACKET,
                                 pEapOut->eapData, 1,
                                 &pDevInfo->bssDescr->bssId,
                                 &pDevInfo->localSta->macAddr);
        if (ret != A_OK) {
            leapRestart(pDevInfo, pSib, TRUE);
        }

        LEAP_CHAT("LEAP: Got non-LEAP EAP frame...sending EAP_NAK\n");

        goto boogie;

    } else if (eapIn && eapIn->eapCode == EAP_CODE_FAILURE) {
        /*
         *  Push leapState back to init state.  An 802.1x authenticator
         *  will use these frames as its usual init, maybe LEAP
         *  will someday too.  Don't deauth.
         */
        pSib->leapCtx.leapState = ST_INITIALIZE;
        LEAP_CHAT("LEAP: Got EAP_FAILURE, changing state to ST_INITIALIZE\n");

        ccxRogueListAdd(pDevInfo, &pDevInfo->bssDescr->bssId,
                     DDP_FAILCODE_CHALLFROMAP);
        goto boogie;
    }

    switch (pSib->leapCtx.leapState) {

    case ST_IDENT_REQ:
        ST_ENTRY;
        /* Send EAP-Response/Identity */
        pEapOut->eapCode = EAP_CODE_RESPONSE;
        pEapOut->eapType = EAP_TYPE_IDENTITY;

        /* Send same ident as sent by authenticator */
        pEapOut->eapIdentifier = eapIn->eapIdentifier;
        pEapOut->eapLength     = cpu2be16(pDevInfo->staConfig.leapUserNameLen +
                                        EAP_HDRLEN);

        ret = leapSendEapolFrame(pDevInfo, pEapOut, EAPOL_TYPE_PACKET,
                                 pDevInfo->staConfig.leapUserName,
                                 (A_UINT16)pDevInfo->staConfig.leapUserNameLen,
                                 &pDevInfo->bssDescr->bssId,
                                 &pDevInfo->localSta->macAddr);
        if (ret != A_OK) {
            leapRestart(pDevInfo, pSib, TRUE);
            break;
        }

        LEAP_CHAT("LEAP:  Response/Indentity sent\n");
        pSib->leapCtx.leapState = ST_LEAP_CHALLENGE;

        break;

    case ST_LEAP_CHALLENGE:
        ST_ENTRY;

        if (eapIn == NULL || eapIn->eapCode != EAP_CODE_REQUEST ||
            eapIn->eapLength < (EAP_HDRLEN+LEAP_CHALLENGE_HDRLEN) ||
            eapIn->eapType != EAP_TYPE_LEAP)
        {
            break;
        }

        leapChallenge = (LEAP_CHALLENGE *)eapIn->eapData;

        if (leapChallenge->version != LEAP_VERSION ||
            leapChallenge->length != LEAP_CHALLENGE_LEN)
        {
            break;
        }

        A_BCOPY(leapChallenge->random,
                pSib->leapCtx.challengeToClient,
                sizeof(pSib->leapCtx.challengeToClient));

        /* Send EAP-Response/LEAP Challenge Resp */

        /*
         *  Using the random field in the incoming frame,
         *  Calculate LEAP Challenge Resp,
         *  render into pSib->leapCtx
         */
        if (challengeResponseCalc(leapChallenge->random, // incoming
                  leapHash.singleHash,
                  pSib->leapCtx.challengeToClientResponse) != A_OK)
        {
            leapRestart(pDevInfo, pSib, TRUE);
            break;
        }

        len = (A_UINT16)(LEAP_CHALLRESP_HDRLEN +
                         pDevInfo->staConfig.leapUserNameLen);

        pEapOut->eapCode = EAP_CODE_RESPONSE;
        pEapOut->eapType = EAP_TYPE_LEAP;

        /* Send same ident as sent by authenticator */
        pEapOut->eapIdentifier = eapIn->eapIdentifier;
        pEapOut->eapLength     = cpu2be16(EAP_HDRLEN + len);

        A_MEM_ZERO(sessionKeyBuf, sizeof(sessionKeyBuf));
        leapChallResp = (LEAP_CHALLRESP *)sessionKeyBuf;
        leapChallResp->version = LEAP_VERSION;
        leapChallResp->length  = LEAP_CHALLRESP_LEN;
        A_BCOPY(pSib->leapCtx.challengeToClientResponse,
                leapChallResp->response, LEAP_CHALLRESP_LEN);
        A_BCOPY(pDevInfo->staConfig.leapUserName, leapChallResp->user,
                pDevInfo->staConfig.leapUserNameLen);

        ret = leapSendEapolFrame(pDevInfo, pEapOut, EAPOL_TYPE_PACKET,
                                 leapChallResp, len,
                                 &pDevInfo->bssDescr->bssId,
                                 &pDevInfo->localSta->macAddr);
        if (ret != A_OK) {
            leapRestart(pDevInfo, pSib, TRUE);
            break;
        }

        LEAP_CHAT("LEAP:  Challenge Response sent\n");
        pSib->leapCtx.leapState = ST_EAP_SUCCESS;

        break;

    case ST_EAP_SUCCESS:
        ST_ENTRY;
        if (eapIn == NULL || eapIn->eapCode != EAP_CODE_SUCCESS ||
            eapIn->eapLength != EAP_SUCCFAIL_LEN)
        {
            break;
        }

        len = (A_UINT16)(LEAP_CHALLENGE_HDRLEN +
                         pDevInfo->staConfig.leapUserNameLen);

        pEapOut->eapCode       = EAP_CODE_REQUEST;
        pEapOut->eapType       = EAP_TYPE_LEAP;
        pEapOut->eapIdentifier = eapIn->eapIdentifier;
        pEapOut->eapLength     = cpu2be16(EAP_HDRLEN + len);

        A_MEM_ZERO(sessionKeyBuf, sizeof(sessionKeyBuf));
        leapChallenge = (LEAP_CHALLENGE *)sessionKeyBuf;
        leapChallenge->version = LEAP_VERSION;
        leapChallenge->length  = LEAP_CHALLENGE_LEN;

        /* Use some entropy (?) to create a random challenge to AS */
        /* XXX need to do this as a background task */
        for (i = 0; i < sizeof(leapChallenge->random); i++) {
            leapChallenge->random[i] = (A_UINT8)
                (halGetRandomSeed(pDevInfo) ^ tickGet()) & 0xFF;
            /*
             *  halGetRandomSeed() gives more entropy
             *  the longer you wait between calls.
             */
//                udelay(13 * 1000);
        }

        A_BCOPY(pDevInfo->staConfig.leapUserName, leapChallenge->user,
                pDevInfo->staConfig.leapUserNameLen);

        A_BCOPY(leapChallenge->random, pSib->leapCtx.challengeToRadius,
                sizeof(pSib->leapCtx.challengeToRadius));


        ret = leapSendEapolFrame(pDevInfo, pEapOut, EAPOL_TYPE_PACKET,
                                 leapChallenge, len,
                                 &pDevInfo->bssDescr->bssId,
                                 &pDevInfo->localSta->macAddr);
        if (ret != A_OK) {
            leapRestart(pDevInfo, pSib, TRUE);
            break;
        }

        LEAP_CHAT("LEAP:  Challenge sent\n");
        /* Send EAP-Request/LEAP Challenge */
        pSib->leapCtx.leapState = ST_LEAP_CHALL_RESP;

        break;

    case ST_LEAP_CHALL_RESP: {
        ST_ENTRY;

        if (eapIn == NULL || eapIn->eapCode != EAP_CODE_RESPONSE ||
            eapIn->eapLength < (EAP_HDRLEN+LEAP_CHALLRESP_HDRLEN) ||
            eapIn->eapType != EAP_TYPE_LEAP)
        {
            break;
        }

        leapChallResp = (LEAP_CHALLRESP *)eapIn->eapData;

        if (leapChallResp->version != LEAP_VERSION) {
            break;
        }

        /* Cancel timer */
        A_UNTIMEOUT(&pDevInfo->leapTimer);
        leapTimerPending = FALSE;

        /* Find the expected response value */
        if (challengeResponseCalc(pSib->leapCtx.challengeToRadius,
                                  leapHash.doubleHash,
                                  pSib->leapCtx.
                                  challengeToRadiusResponse) != A_OK)
        {
            /* Found a rogue AP...add to list */
            ccxRogueListAdd(pDevInfo, &pDevInfo->bssDescr->bssId,
                         DDP_FAILCODE_CHALLTOAP);
            leapRestart(pDevInfo, pSib, TRUE);
            break;
        }

        if (A_BCOMP(pSib->leapCtx.challengeToRadiusResponse,
                    leapChallResp->response,
                    LEAP_CHALLRESP_LEN) != 0)
        {
            LEAP_CHAT("AS failed authentication\n");
            leapRestart(pDevInfo, pSib, TRUE);
            break;
        }
        LEAP_CHAT("LEAP: auth SUCCESS\n");

        /* Assemble entropy, calculate hash */
        /* Let's copy to avoid any compiler alignment mistakes */
        A_MEM_ZERO(sessionKeyBuf, sizeof(sessionKeyBuf));
        A_BCOPY(leapHash.doubleHash, sessionKeyBuf, 16);
        A_BCOPY(pSib->leapCtx.challengeToRadius, &sessionKeyBuf[16],
                LEAP_CHALLENGE_LEN);
        A_BCOPY(pSib->leapCtx.challengeToRadiusResponse,
                &sessionKeyBuf[16+LEAP_CHALLENGE_LEN], LEAP_CHALLRESP_LEN);
        A_BCOPY(pSib->leapCtx.challengeToClient,
                &sessionKeyBuf[16+LEAP_CHALLENGE_LEN+LEAP_CHALLRESP_LEN],
                LEAP_CHALLENGE_LEN);
        A_BCOPY(pSib->leapCtx.challengeToClientResponse,
                &sessionKeyBuf[16 + LEAP_CHALLENGE_LEN + LEAP_CHALLRESP_LEN +
                     LEAP_CHALLENGE_LEN], LEAP_CHALLRESP_LEN);

        /* Render into our key buffer to save */
        md5_calc(pSib->leapCtx.leapSessionKey, sessionKeyBuf,
                 sizeof(sessionKeyBuf));

        A_MEM_ZERO(sessionKeyBuf, sizeof(sessionKeyBuf));
        /*
         *  cancel OS disconnect indicate timer if it
         *  was started for previous failure to associate
         */
        osIndicateConnectStatus(pDevInfo, TRUE); // notify OS upper layer
        pCsInfo->apConnAttemptCur = 0;

        /* No outgoing frame for this state */

        pSib->leapCtx.leapState = ST_EAPOL_KEYMSG;
        }
        break;
    case ST_EAPOL_KEYMSG:
        ST_ENTRY;

        if (eapolIn->eapolType != EAPOL_TYPE_KEYMSG) {
            LEAP_CHAT("LEAP: Got non-keymsg while waiting for keymsg\n");
            leapRestart(pDevInfo, pSib, TRUE);
            break;
        }

        /*
         *  When in this state, we're fully authenticated.  We'll
         *  stick in this state as long as we get key messages which
         *  we can correctly decrypt and authenticate.
         *  XXX We need to save the ReplayCounter away so that we
         *  can test for replay while we still have this same authentication;
         *  that is, the leapSessionKey has not changed.
         */

        eapolKeyMsg = (EAPOL_KEY *)(eapolIn + 1);

        if (eapolKeyMsg->keyType != EAPOL_KEY_TYPE_RC4) {
            leapRestart(pDevInfo, pSib, TRUE);
            break;
        }

        /* Authenticate the KeyMsg */

        /* Save a copy of the observed key signature */
        A_BCOPY(eapolKeyMsg->keySig, savedKeySig,
                sizeof(savedKeySig));

        /* Set keySig to a known value */
        A_MEM_ZERO(eapolKeyMsg->keySig, sizeof(eapolKeyMsg->keySig));

        pktLen = eapolIn->eapolBodyLength;

        /* XXX have to swap the bodylength back for the auth calc */
        eapolIn->eapolBodyLength = cpu2be16(eapolIn->eapolBodyLength);

        /*
         *  For unicast, LEAP does not send a key in the msg, so
         *  the length is only that of the keymsg hdr itself.
         */

        hmac_md5((A_UINT8 *)eapolIn, pktLen + sizeof(EAPOL_HEADER),
                 pSib->leapCtx.leapSessionKey,
                 sizeof(pSib->leapCtx.leapSessionKey), digest);

        if (A_BCOMP(digest, savedKeySig, sizeof(savedKeySig)) != 0) {
            LEAP_CHAT("LEAP: KeyMsg failed authentication\n");
            leapRestart(pDevInfo, pSib, TRUE);
            break;
        }

        keyLen = be2cpu16(eapolKeyMsg->keyLength);

        /* Check for replay attacks */
        replay64 = be2cpu64(eapolKeyMsg->keyReplayCtr);
        if (replay64 <= pSib->leapCtx.replayCtr) {
            leapRestart(pDevInfo, pSib, TRUE);
            break;
        }
        pSib->leapCtx.replayCtr = replay64;

        LEAP_CHAT("LEAP: key mesg authenticated.\n");

        A_MEM_ZERO(&keyDesc, sizeof(keyDesc));

        if (eapolKeyMsg->keyIndex & KEY_USAGE_UNICAST) {
            /*
             *  For unicast keys, we already have the FullLength
             *  Derived Key in pSib->leapCtx.leapSessionKey.  We just
             *  need to trim it to size.
             */
            A_MEM_ZERO(hwKey, sizeof(hwKey));
            A_BCOPY(pSib->leapCtx.leapSessionKey, hwKey,
                    keyLen);

            /* Plumb unicast key into hw */

            /* Cons up an NDIS key descriptor */
            keyDesc.desc.Length = sizeof(NDIS_802_11_KEY) - 1 + keyLen;
            keyDesc.desc.KeyIndex = toMSDefaultKey(
                eapolKeyMsg->keyIndex & ~KEY_USAGE_UNICAST);

            keyDesc.desc.KeyLength = keyLen;

            /* copy key over the bytes beginning with KeyMaterial */
            A_BCOPY(hwKey, keyDesc.desc.KeyMaterial, keyLen);

            LEAP_CHAT("LEAP: installing unicast key\n");

            if (athKeyPlumb(pDevInfo, &keyDesc.desc, ADD_KEY) != A_OK) {
                /* XXX This is prob not the right penalty
                 *  for a key we can't add */
                leapRestart(pDevInfo, pSib, TRUE);
                break;
            }

            /* Send the rogue AP info to this AP */
            ccxRogueListFlush(pDevInfo, &pDevInfo->bssDescr->bssId);

        } else {
            /* Decrypt key from KeyMsg */

            /* RC4 Key = keyMsgIV || FullLengthDerived Key */
            A_BCOPY(eapolKeyMsg->keyIV, buf, sizeof(eapolKeyMsg->keyIV));
            A_BCOPY(pSib->leapCtx.leapSessionKey,
                    &buf[sizeof(eapolKeyMsg->keyIV)],
                    sizeof(pSib->leapCtx.leapSessionKey));

            /* decrypt, actually */
            rc4_encrypt(hwKey, (A_UINT8 *)(eapolKeyMsg+1),
                        keyLen, buf, sizeof(eapolKeyMsg->keyIV) +
                        sizeof(pSib->leapCtx.leapSessionKey));

            /* Plumb Key into hw */

            /* Cons up an NDIS key descriptor */
            keyDesc.desc.Length = sizeof(NDIS_802_11_KEY) - 1 + keyLen;
            keyDesc.desc.KeyIndex = eapolKeyMsg->keyIndex;
            keyDesc.desc.KeyLength = keyLen;

            /* copy key over the bytes beginning with KeyMaterial */
            A_BCOPY(hwKey, keyDesc.desc.KeyMaterial, keyLen);

            LEAP_CHAT("LEAP: installing broadcast key\n");
            if (athKeyPlumb(pDevInfo, &keyDesc.desc, ADD_KEY) != A_OK) {
                /* XXX This is prob not the right
                 * penalty for a key we can't add */
                leapRestart(pDevInfo, pSib, TRUE);
                break;
            }
        }

        /* No outgoing frame to transmit */

        pSib->leapCtx.leapState = ST_EAPOL_KEYMSG; // stay here
        break;

    case ST_INITIALIZE:
        ST_ENTRY;
        break;

    default:
        ST_ENTRY;
        leapRestart(pDevInfo, pSib, TRUE);
        break;
    }

boogie:
    /* Free incoming buffer/desc */
    freeBuffandDesc(pDevInfo, pDesc);

    return;
}

/*
 *  Timeout service routine.
 *  Executed at dispatch level, never in context of DPC.
 */
ENTRY_FN_VOID(leapTimerSvc,(void *pContext), (pContext), ICEPT_LEAP)
{
    SIB_ENTRY     *pSib;
    WLAN_DEV_INFO *pDevInfo = DEV_INFO_FROM_CONTEXT(pContext);

    LEAP_CHAT("LEAP: leapTimerSvc called\n");
    if (NULL == pDevInfo) {
        return;
    }

    if (!pDevInfo->pOSHandle->openForBusiness) {
        return;
    }
  
    // make sure only one timer will be fired at one time
    ATH_ACQUIRE_SPINLOCK(pDevInfo->lock);

    powerSet(pDevInfo, WAKE_UP, SET_PWR_TO_3, FALSE, 0);

    ccxRogueListAdd(pDevInfo, &pDevInfo->bssDescr->bssId,
                 DDP_FAILCODE_AUTHTIMEOUT);

    pSib = sibEntryFind(pDevInfo, &pDevInfo->bssDescr->bssId,
                        &pDevInfo->baseBss);

    if (pSib && (pSib->staState & STATE_ASSOC)) {
        leapRestart(pDevInfo, pSib, TRUE);
    } else {
        leapRestart(pDevInfo, NULL, TRUE);
    }

    powerSet(pDevInfo, WAKE_UP, REDUCE_PWR_FROM_3, FALSE, 0);

    ATH_RELEASE_SPINLOCK(pDevInfo->lock);

    return;
}

/*
 *  Used to trigger LEAP from MLME-Assoc.Confirm SAP
 */
void
leapAssocCallback(WLAN_DEV_INFO *pDevInfo, SIB_ENTRY *pSib)
{
    ATHEROS_DESC *pOutDesc;
    A_STATUS      ret;

    LEAP_CHAT("LEAP: assoc callback\n");
    pSib->leapCtx.leapState = ST_INITIALIZE;

    ret = leapSendEapolFrame(pDevInfo, NULL, EAPOL_TYPE_START, NULL, 0,
                             &pDevInfo->bssDescr->bssId,
                             &pDevInfo->localSta->macAddr);

    if (ret != A_OK) {
        leapRestart(pDevInfo, pSib, TRUE);
        return;
    }

    LEAP_CHAT("LEAP: EAPOL Start sent\n");

    /* Set a timer for the total auth completion */
    A_TIMEOUT(&pDevInfo->leapTimer, pDevInfo->staConfig.leapTimeout,
              (A_UINT32)pDevInfo, 0);
    leapTimerPending = TRUE;

    return;
}

static A_STATUS
leapSendEapolFrame(WLAN_DEV_INFO *pDev, EAP *pEapHeader, A_UINT8 EapolType,
                   void *pEapData, A_UINT16 dataLen, WLAN_MACADDR *pDestAddr,
                   WLAN_MACADDR *pSrcAddr)
{
    ATHEROS_DESC     *pDesc;
    EAPOL_HEADER     *pEapolHdr;
    LAN_FRAME_HEADER *pHdr;
    A_UINT32         headroom, eapLen;
    A_UINT16         totalLen;
    A_STATUS         status;

    ASSERT((EapolType != EAPOL_TYPE_PACKET) || (pEapHeader != NULL));
    ASSERT(pDestAddr && pSrcAddr);

    /* Determine needed headroom for encapsulating the frame later. */
    headroom = sizeof(WLAN_DATA_MAC_HEADER3) + sizeof(EAPOL_SNAP_HDR) +
        sizeof(EAPOL_HEADER) + EAP_HDRLEN + MAX_IV_FIELD_SIZE;
    totalLen = (A_UINT16)headroom + dataLen + MAX_ICV_FIELD_SIZE;
    eapLen = dataLen;

    /* Get a buffer and descriptor pair */
    if (createBuffandDesc(pDev, totalLen, &pDesc) != A_OK) {
        return A_ERROR;
    }

    /* Provide sufficient room for headers. */
    bufferPopHdr(pDesc, headroom);

    /* Set the payload length */
    pDesc->bufferLength = dataLen;
    pDesc->frameLength  = dataLen;

    /* Fill in the payload data */
    if (dataLen > 0) {
        A_DRIVER_BCOPY(pEapData, pDesc->pBufferVirtPtr.byte, dataLen);
    }

    /* Push and Fill in EAP Header */
    if (EapolType == EAPOL_TYPE_PACKET) {
        bufferPushHdr(pDesc, EAP_HDRLEN);
        A_DRIVER_BCOPY(pEapHeader, pDesc->pBufferVirtPtr.byte, EAP_HDRLEN);
        eapLen += EAP_HDRLEN;
    }

    /* Push and Fill the dot1x header */
    bufferPushHdr(pDesc, sizeof(EAPOL_HEADER));
    pEapolHdr = (EAPOL_HEADER *)pDesc->pBufferVirtPtr.byte;
    pEapolHdr->eapolProtoVers  = EAPOL_PROTOVERS;
    pEapolHdr->eapolType       = EapolType;
    pEapolHdr->eapolBodyLength = cpu2be16(eapLen);

    /* Push space and fill the SNAP header */
    bufferPushHdr(pDesc, sizeof(LLC_SNAP_HEADER));
    A_BCOPY(&eapolSnap, pDesc->pBufferVirtPtr.llcSnapHeader,
            sizeof(LLC_SNAP_HEADER));

    totalLen = (A_UINT16)pDesc->frameLength;

    /* Push space for the 802.3 header. */
    bufferPushHdr(pDesc, sizeof(LAN_FRAME_HEADER));
    pHdr = pDesc->pBufferVirtPtr.lanHeader;

    /* Fill in the 802.3 header */
    A_MACADDR_COPY(pDestAddr, &pHdr->destAddr);
    A_MACADDR_COPY(pSrcAddr, &pHdr->srcAddr);
    pHdr->lanTypeOrLen = cpu2be16(totalLen);

    /*
     * HACK ALERT:
     * Station uses the basic virtual device only. This should be done in a
     * more layered fashion, but it's not.  Yet.
     */
    pDesc->pOpBss = &pDev->baseBss;

    status = wlanDataFrameTransmit(pDev, pDesc);

    return status;
}

static A_BOOL
isEapolValid(EAPOL_HEADER *eapolIn, EAP *eapIn)
{
    /* Frame protocol compliance checks */
    if (eapolIn->eapolProtoVers != EAPOL_PROTOVERS) {
        LEAP_CHAT("LEAP:  bad EAPOL version\n");
        return FALSE;
    }
#if 0
    /* XXX EAP-SUCCESS does not use the eapType field ==> is only 4 bytes */
    if (eapIn && eapolIn->eapolType == EAPOL_TYPE_PACKET &&
        eapolIn->eapolBodyLength < EAP_SUCCFAIL_LEN)
    {
        LEAP_CHAT("LEAP:  bad EAPOL length\n");
        return FALSE;
    }
#endif
    if (eapolIn->eapolType == EAPOL_TYPE_KEYMSG &&
        eapolIn->eapolBodyLength < sizeof(EAPOL_KEY))
    {
        return FALSE;
    }
    return TRUE;
}

void
leapAuthCallback(WLAN_DEV_INFO *pDevInfo, WLAN_MACADDR
                 *peerAddr, MLME_RESULT resultCode)
{
    if (pDevInfo->staConfig.leapEnabled &&
        resultCode == WLAN_UNSUPPORTED_ALG)
    {
        ccxRogueListAdd(pDevInfo, peerAddr, DDP_FAILCODE_INVALALG);
    }
}

/*
 *  mschap takes an 8-byte challenge string and 16-byte NT password hash
 *  and returns a 24-byte response string in response
 */
static A_STATUS
mschap(const char *pwdhash, const char *chall, char *response)
{
    unsigned char hash[21];

    A_MEM_ZERO(hash, 21);
    A_BCOPY(pwdhash, hash, 16);

    DesEncrypt(chall, hash + 14, response + 16);
    DesEncrypt(chall, hash +  7, response + 8);
    DesEncrypt(chall, hash +  0, response + 0);

    return A_OK;
}

static A_STATUS
challengeResponseCalc(const A_UINT8 *challenge, const A_UINT8 *passwdHash,
                      A_UINT8 *response)
{
    return mschap(passwdHash, challenge, response);
}

/*****************************************************************************
 * DES derived from ftp://ftp.psy.uq.oz.au/pub/Crypto/DES/libdes-x.xx.tar.gz
 *****************************************************************************/

static int
des_set_key(des_cblock * key, des_key_schedule schedule)
{
    register unsigned long c,d,t,s;
    register unsigned char *in;
    register unsigned long *k;
    register int i;

    k=(unsigned long *)schedule;
    in=(unsigned char *)key;

    c2l(in,c);
    c2l(in,d);

    /* do PC1 in 60 simple operations */
    /*  PERM_OP(d,c,t,4,0x0f0f0f0f);
        HPERM_OP(c,t,-2, 0xcccc0000);
        HPERM_OP(c,t,-1, 0xaaaa0000);
        HPERM_OP(c,t, 8, 0x00ff0000);
        HPERM_OP(c,t,-1, 0xaaaa0000);
        HPERM_OP(d,t,-8, 0xff000000);
        HPERM_OP(d,t, 8, 0x00ff0000);
        HPERM_OP(d,t, 2, 0x33330000);
        d=((d&0x00aa00aa)<<7)|((d&0x55005500)>>7)|(d&0xaa55aa55);
        d=(d>>8)|((c&0xf0000000)>>4);
        c&=0x0fffffff; */

    /* I now do it in 47 simple operations :-)
     * Thanks to John Fletcher (john_fletcher@lccmail.ocf.llnl.gov)
     * for the inspiration. :-) */
    PERM_OP (d,c,t,4,0x0f0f0f0f);
    HPERM_OP(c,t,-2,0xcccc0000);
    HPERM_OP(d,t,-2,0xcccc0000);
    PERM_OP (d,c,t,1,0x55555555);
    PERM_OP (c,d,t,8,0x00ff00ff);
    PERM_OP (d,c,t,1,0x55555555);
    d=  (((d&0x000000ff)<<16)| (d&0x0000ff00)     |
         ((d&0x00ff0000)>>16)|((c&0xf0000000)>>4));
    c&=0x0fffffff;

    for (i=0; i<ITERATIONS; i++)
    {
        if (shifts2[i])
        { c=((c>>2)|(c<<26)); d=((d>>2)|(d<<26)); }
        else
        { c=((c>>1)|(c<<27)); d=((d>>1)|(d<<27)); }
        c&=0x0fffffff;
        d&=0x0fffffff;
        /* could be a few less shifts but I am to lazy at this
         * point in time to investigate */
        s=      des_skb[0][ (c    )&0x3f                ]|
            des_skb[1][((c>> 6)&0x03)|((c>> 7)&0x3c)]|
            des_skb[2][((c>>13)&0x0f)|((c>>14)&0x30)]|
            des_skb[3][((c>>20)&0x01)|((c>>21)&0x06) |
                       ((c>>22)&0x38)];
        t=      des_skb[4][ (d    )&0x3f                ]|
            des_skb[5][((d>> 7)&0x03)|((d>> 8)&0x3c)]|
            des_skb[6][ (d>>15)&0x3f                ]|
            des_skb[7][((d>>21)&0x0f)|((d>>22)&0x30)];

        /* table contained 0213 4657 */
        *(k++)=((t<<16)|(s&0x0000ffff))&0xffffffff;
        s=     ((s>>16)|(t&0xffff0000));

        s=(s<<4)|(s>>28);
        *(k++)=s&0xffffffff;
    }
    return(0);
}

static unsigned char
Get7Bits(unsigned char * input, int startBit)
{
    register unsigned int       word;

    word  = (unsigned)input[startBit / 8] << 8;
    word |= (unsigned)input[startBit / 8 + 1];

    word >>= 15 - (startBit % 8 + 7);

    return word & 0xFE;
}


static int
des_encrypt(unsigned long * input, unsigned long * output, des_key_schedule ks)
{
    register unsigned long l,r,t,u;
    register int i;
    register unsigned long *s;

    l=input[0];
    r=input[1];

    IP(l,r,t);
    /* Things have been modified so that the initial rotate is
     * done outside the loop.  This required the
     * des_SPtrans values in sp.h to be rotated 1 bit to the right.
     * One perl script later and things have a 5% speed up on a sparc2.
     * Thanks to Richard Outerbridge <71755.204@CompuServe.COM>
     * for pointing this out. */
    t=(r<<1)|(r>>31);
    r=(l<<1)|(l>>31);
    l=t;

    /* clear the top bits on machines with 8byte longs */
    l&=0xffffffff;
    r&=0xffffffff;

    s=(unsigned long *)ks;
    /* I don't know if it is worth the effort of loop unrolling the
     * inner loop */
    for (i=0; i<32; i+=4) {
        D_ENCRYPT(l,r,i+0); /*  1 */
        D_ENCRYPT(r,l,i+2); /*  2 */
    }
    l=(l>>1)|(l<<31);
    r=(r>>1)|(r<<31);
    /* clear the top bits on machines with 8byte longs */
    l&=0xffffffff;
    r&=0xffffffff;

    FP(r,l,t);
    output[0]=l;
    output[1]=r;
    l=r=t=u=0;
    return(0);
}

static void
DesEncrypt(const unsigned char * clear, unsigned char * key,
           unsigned char * cipher)
{
    des_cblock          des_key;
    des_key_schedule    key_schedule;
    int                 i;
    unsigned long       l0,l1;
    unsigned char       *in,*out;
    unsigned long       ll[2];

    des_key[0] = Get7Bits(key,  0);
    des_key[1] = Get7Bits(key,  7);
    des_key[2] = Get7Bits(key, 14);
    des_key[3] = Get7Bits(key, 21);
    des_key[4] = Get7Bits(key, 28);
    des_key[5] = Get7Bits(key, 35);
    des_key[6] = Get7Bits(key, 42);
    des_key[7] = Get7Bits(key, 49);
    for (i=0; i<DES_KEY_SZ; i++) {
        key[i]=odd_parity[key[i]];
    }
    des_set_key(&des_key, key_schedule);

    in=(unsigned char *) clear;
    out=(unsigned char *) cipher;
    c2l(in,l0);
    c2l(in,l1);
    ll[0]=l0;
    ll[1]=l1;
    des_encrypt(ll,ll,key_schedule);
    l0=ll[0];
    l1=ll[1];
    l2c(l0,out);
    l2c(l1,out);
    l0=l1=ll[0]=ll[1]=0;
}
