/* $Id: //depot/sw/branches/1.3_USB_LINUX_port/src/USB/wlan/host/sta/linuxfunc.c#3 $
 * 
 *  Copyright (c) 2000-2003 Atheros Communications, Inc., All Rights Reserved
 *
 *  Linux equivalent of osfunc.c
 */

#include "wlantype.h"
#include "wlanext.h"
#include "wlansmeext.h"
#include "display.h"
#include "stacserv.h"
#include "intercept.h"
#include "ccx.h"

#define GET_TIMER_FROM_CONTEXT(_pContext)   (A_TIMER *) (_pContext)
#define IS_TIMER_QUEUED(pTimer)             (A_ATOMIC_READ(&(pTimer)->timerCount) > 0 ? 1: 0)
/*
 * Need to get rid of yucky extern.  Blech.
 */
extern void
staPollFunction(WLAN_DEV_INFO *);

#if defined(DEBUG_PRINTF)
void
drvPrintf(const char * format, ...)
{
    va_list argList;    /* holds the arguement list passed in */
    char    buffer[256];

    /* if have logging turned on then can also write to a file as needed */

    /* get the arguement list */
    va_start(argList, format);

    /*
     * using vprintf to perform the printing is the same as printf, only
     * it takes a va_list or arguements
     */
    vsprintf(buffer, format, argList);

    printk(buffer);
}
#endif

A_STATUS
athNdisAllocMem(
    WLAN_DEV_INFO *pDevInfo,
    A_UINT32      buffSize,
    A_UINT32      *pPhysAddress,
    void          **ppVirtAddress)
{

    (*ppVirtAddress) = A_DRIVER_MALLOC(buffSize);

    if (*ppVirtAddress == NULL) {
        *pPhysAddress = 0;      // NULL;
        return A_NO_MEMORY;
    }

    // Zero out the allocated space
    A_MEM_ZERO(*ppVirtAddress, buffSize);

    /* For usb device, we don't need physical address anymore*/
    *pPhysAddress = (A_UINT32)(*ppVirtAddress);

    return A_OK;
}

A_STATUS
athNdisFreeMem(
    WLAN_DEV_INFO *pDevInfo,
    A_UINT32      buffSize,
    A_UINT32      physAddress,
    void          *pVirtAddress)
{
    A_DRIVER_FREE(pVirtAddress,buffSize);
    return A_OK;
}

/* drvNdisTimerInit - Initialize Atheros timer in NDIS environment
 * This function stores parameters in the Atheros timer object and
 * and initializes NDIS timer object part of Atheros timer using NDIS
 * call.
 * Called by A_TIMER_INIT() macro.
 */
A_STATUS
drvLinuxTimerInit(
    WLAN_DEV_INFO   *pDevInfo,
    A_TIMER         *pTimer,    /* Pointer to timer object */
    void            *pFunc,     /* timer function */
    A_BOOL          repeat)     /* 0 if one shot timer, 1 if periodic */
{
    struct timer_list *osTimerObj = NULL;
    pTimer->pdevInfo   = pDevInfo;
    pTimer->pFunc      = pFunc;
    pTimer->repeat     = repeat;
    pTimer->state      = FALSE;
    A_ATOMIC_SET(&pTimer->timerCount,0);
    pTimer->param      = 0;

    if (pTimer->osTimerObj == NULL) {
        pTimer->osTimerObj = A_DRIVER_MALLOC(sizeof(struct timer_list));
        if (NULL == pTimer->osTimerObj) {
            return A_ERROR;
        }
    }
    osTimerObj = pTimer->osTimerObj;

    init_timer ((struct timer_list *)osTimerObj);
    osTimerObj->data = (unsigned long)pTimer;
    osTimerObj->function = athTimerHandler;

    return A_OK;
}

/* drvNdisTimerSet - Fire off the timer that is already initialized.
 * This function stores timer parameter in the fixed location in the
 * timer object and sets the NDIS timer with given timeout period value.
 * Called by A_TIMEOUT() macro.
 */
A_STATUS
drvLinuxTimerSet(A_TIMER *pTimer, A_UINT32 timeVal,
                A_UINT32 param, A_UINT32 oper)
{
    A_UINT32 TimeInJiffies;

    if (!(SYSTEMSTATE_VALID(pTimer->pdevInfo)) ||
        SYSTEMSTATE_CHECK(pTimer->pdevInfo, ATHUSB_SYSTEM_HALT)         ||
        SYSTEMSTATE_CHECK(pTimer->pdevInfo, ATHUSB_SYSTEM_SHUTDOWN))
    {
        uiPrintf("drvLinuxTimerSet: failed system state invalid\n");
        return A_ERROR;
    }

    TimeInJiffies     = (timeVal * HZ) / 1000;
    if (TimeInJiffies == 0) {
        TimeInJiffies = 1;
    }

    pTimer->param     = param;
    pTimer->operation = oper;
    pTimer->timeVal   = timeVal;
    pTimer->state     = FALSE;

    if (!IS_TIMER_QUEUED(pTimer)) {
        /*
         * Only add to the timer count if there is no
         * outstanding timers.
         */
        NdisInterlockedIncrement(&pTimer->timerCount);
    }

    /*
     * No longer use periodic timers, replaced with one-shot timers
     */
    mod_timer(pTimer->osTimerObj, jiffies + TimeInJiffies);

    return A_OK;
}

/* drvNdisTimerCancel - This function cancels a timer that is already
 * fired off. It doesn't know if the timer has expired, it cancels the timeout
 * anyway.
 * Called by A_UNTIMEOUT() macro.
 */
A_BOOL
drvLinuxTimerCancel(A_TIMER *pTimer)
{
    ASSERT(pTimer);

    if (pTimer->osTimerObj && pTimer->state == FALSE) {
        del_timer((struct timer_list *)pTimer->osTimerObj);
        pTimer->state = TRUE;
        NdisInterlockedDecrement(&pTimer->timerCount);
    }
    
    pTimer->operation = INVALID_TIMEOUT;

    return (TRUE);;
}

/* drvNdisTimerDelete - This function deletes a timer.
 * It doesn't know if the timer has expired, it cancels the timeout
 * anyway.
 * Called by A_DELETE_TIMER() macro.
 */
A_STATUS
drvLinuxTimerDelete(A_TIMER *pTimer)
{
    if (pTimer) {
        if (pTimer->osTimerObj) {
            if (pTimer->state == FALSE) {
                del_timer((struct timer_list *)pTimer->osTimerObj);
            }
            pTimer->state = TRUE;
            NdisInterlockedDecrement(&pTimer->timerCount);

            // delete the object now.
            A_DRIVER_FREE(pTimer->osTimerObj, sizeof(struct timer_list));
            pTimer->osTimerObj = NULL;          // 0;
        }
    }
    
    return A_OK;
}

void
athTimerHandler(PVOID pContext)
{
    WLAN_DEV_INFO *pDev;
    A_TIMER       *pTimer;

    ASSERT(pContext);

    pTimer = GET_TIMER_FROM_CONTEXT(pContext);
    pDev   = pTimer->pdevInfo;

    A_ATOMIC_DEC(&pTimer->timerCount);

    if (!pDev->pOSHandle->openForBusiness) {
        return;
    }

    if (FALSE == SYSTEMSTATE_VALID(pDev)) {
        return;
    }

    pTimer->pFunc((void *)pTimer->param);

    if (pTimer->repeat) {
        A_TIMEOUT(pTimer, pTimer->timeVal, pTimer->param, pTimer->operation);
    }
}

ENTRY_FN_VOID(mlmePollHandler,(void *pContext), (pContext), ICEPT_SMETIMER)
{
    WLAN_DEV_INFO *pDev = DEV_INFO_FROM_CONTEXT(pContext);
    
    ASSERT(pDev);
    
    if (!pDev->pOSHandle->openForBusiness) {
        return;
    }
    
    // make sure only one timer will be fired at one time
    ATH_ACQUIRE_SPINLOCK_IRQ(pDev->lock);
    mlmePoll(pDev);
    ATH_RELEASE_SPINLOCK_IRQ(pDev->lock);
}

ENTRY_FN_VOID(staPollHandler,(void *pContext), (pContext), ICEPT_STAPOLL)
{
    WLAN_DEV_INFO *pDevInfo = DEV_INFO_FROM_CONTEXT(pContext);

    ASSERT(pDevInfo);

    if (!pDevInfo->pOSHandle->openForBusiness) {
        return;
    }

    drainSendWaitQueue(pDevInfo);

    // Call the generic station polling function
    // make sure only one timer will be fired at one time
    ATH_ACQUIRE_SPINLOCK_IRQ(pDevInfo->lock);
    staPollFunction(pDevInfo);
    ATH_RELEASE_SPINLOCK_IRQ(pDevInfo->lock);
}

//
// Retrieve the number of ticks since the system was started.
//
A_UINT32
tickGet()
{
    return ((jiffies * 1000)/HZ);
}

void
osIndicateConnectStatus(WLAN_DEV_INFO *pDevInfo, A_BOOL connectionStatus)
{
    A_BOOL isIbss = (pDevInfo->bssDescr->bsstype == INDEPENDENT_BSS);

    ASSERT(pDevInfo->pCsInfo);

    // whether connecting or disconnecting, cancel pending
    // reporting of disconnect
    pDevInfo->pCsInfo->disconnectCur = 0; 

    if (connectionStatus == FALSE) {
        /*
         * If we are no longer connected and cckm is enabled we should
         * invalidate our KRK.  This will make sure the next association will
         * go through an entire dot1x authentication
         */
        ccxCckmKrkInvalidate(pDevInfo);

        /* 
         * If we are no longer connected, we should abort all pending frame
         */
        ATH_RELEASE_SPINLOCK(pDevInfo->lock);
        abortSendWaitQueue(pDevInfo);
        ATH_ACQUIRE_SPINLOCK(pDevInfo->lock);
    }

    // report if new status is TRUE or old status was true
    if (pDevInfo->pOSHandle->connectionStatus ||
        connectionStatus || pDevInfo->pOSHandle->postInit)
    {
        // avoid duplicate reporting same status
        // Inform the OS about change in link status
        if (!pDevInfo->NdisResetInProgress) {
            uiPrintf("osIndicateConnectStatus: Indicating %s\n",
                     connectionStatus ? "CONNECTED" : "DISCONNECTED");

            ATH_RELEASE_SPINLOCK(pDevInfo->lock);

#ifdef Linux
            {
                union iwreq_data iwreq;
                memset(&iwreq, 0, sizeof(iwreq));
            if (connectionStatus) {
                netif_carrier_on(pDevInfo->pOSHandle->NicAdapterHandle);
                memcpy(iwreq.addr.sa_data, pDevInfo->baseBss.bssId.octets,
                        ETH_LENGTH_OF_ADDRESS);
            } else {
                netif_carrier_off(pDevInfo->pOSHandle->NicAdapterHandle);
                memset(iwreq.addr.sa_data, 0, ETH_LENGTH_OF_ADDRESS);
            }
            wireless_send_event(pDevInfo->pOSHandle->NicAdapterHandle,
                    SIOCGIWAP, &iwreq, NULL);
            }
#endif

            ATH_ACQUIRE_SPINLOCK(pDevInfo->lock);

            osIndicateResourcesAvailable(pDevInfo);

            pDevInfo->pOSHandle->postInit = FALSE;
        }

        pDevInfo->pOSHandle->connectionStatus = connectionStatus;
        wdcSetLedState(pDevInfo->targetHandle,connectionStatus);

#ifdef ADHOC_2STA_B4_CONNECTED
        if (isIbss && connectionStatus) {
            cservStartDisconnectTimer(pDevInfo);
        }
#endif
    }
}

void
osIndicateAuth(WLAN_DEV_INFO *pDevInfo, ULONG flags,
               WLAN_MACADDR  *pBssid)
{
}

void
osIndicateResourcesAvailable(WLAN_DEV_INFO *pDevInfo)
{
    if (!pDevInfo->pOSHandle->resourceAvail && !pDevInfo->NdisHaltInProgress) {
        pDevInfo->pOSHandle->resourceAvail = TRUE;
    }
}

