/* parse.c
 *
 * Copyright © 2000-2003 Atheros Communications, Inc., All Rights Reserved
 *
 * Get config values from Windows Registry for NDIS driver.
 */

#include "wlantype.h"
#include "wlandrv.h"
#include "wlanext.h"
#ifdef Linux
#include "linuxext.h"
#endif  /* Linux */
#include "display.h"
#include "wlanchannel.h"
#include "stacserv.h"
#include "wlansmeext.h"

#ifdef DEBUG
extern int powerDebugLevel;
extern int rxDebugLevel;
int oidDebugLevel;
extern int mlmeDebugLevel;
int        lockDebugLevel;
#endif

/* Needed b/c the DDK doesn't have this call for NDIS drivers */
#if !NDIS_WDM
extern NTSTATUS
RtlUnicodeStringToInteger(PUNICODE_STRING, ULONG, PULONG);
#endif
/*
 * NDIS 4.0 Miniports have special requirements.  Define this here to make the
 * cleaner below
 */
#if NDIS40_MINIPORT
#define isNdis40 1
#else
#define isNdis40 0
#endif

/*
 * Some macros for ease of readability and organization
 */
#define FIELD_SIZE(_type, _field) (sizeof(((_type *)0)->_field))
#define NUM_REG_PARAM             (sizeof(paramTable) / sizeof(CONFIG_PARAM))

#define RT_ENUM_2_NDIS(_x)                     \
    ( isNdis40 ? NdisParameterString :         \
      (_x) == tDEC ? NdisParameterInteger :    \
      (_x) == tHEX ? NdisParameterHexInteger : \
      NdisParameterString )

#define OS(_x)      sOS,     FIELD_OFFSET(OS_DEV_INFO, _x),     FIELD_SIZE(OS_DEV_INFO, _x),     0
#define OS_ANSI(_x) sOS,     FIELD_OFFSET(OS_DEV_INFO, _x),     FIELD_SIZE(OS_DEV_INFO, _x),     FIELD_OFFSET(OS_DEV_INFO, _x##Buffer)
#define CFG(_x)     sCONFIG, FIELD_OFFSET(WLAN_STA_CONFIG, _x), FIELD_SIZE(WLAN_STA_CONFIG, _x), 0
#define CFG_LEN(_x) sCONFIG, FIELD_OFFSET(WLAN_STA_CONFIG, _x), FIELD_SIZE(WLAN_STA_CONFIG, _x), FIELD_OFFSET(WLAN_STA_CONFIG, _x##Len)
#define RV_DUMMY    0,       0,                                 0,                               0
#define GLOBAL(_x)  sNONE,   ((A_UINT32)&_x),                   sizeof(_x),                      0

#define RV(_regName, _type, _struct_info, _default, _min, _max) \
    { NDIS_STRING_CONST(#_regName), #_regName, _type, _struct_info, _default, _min, _max }

#define XRV(_regName, _type, _struct_info, _default, _min, _max, _defStr) \
    { NDIS_STRING_CONST(#_regName), #_regName, _type, _struct_info, _default, _min, _max, _defStr }

/*
 * Enum and struct definitions
 */

typedef enum {
    sOS,
    sCONFIG,
    sNONE
} STRUCT_ENUM;

typedef enum {
    tDEC,         // Decimal integer
    tHEX,         // Hexadecimal integer
    tSSID,        // SSID string
    tKEY,         // Security key string
    tSTRLEN,      // Generic string with an associated, but separate, length field
    tANSI,        // Generic ANSI string
    tBSSID,        // MAC ADDR
    tLEAPPW,
} REG_TYPE_ENUM;

typedef struct ConfigParam {
    NDIS_STRING   RegVarName;     // variable name text, NDIS-style
    char          *RegAscName;    // variable name text
    REG_TYPE_ENUM RegVarType;     // integer, string, etc.
    STRUCT_ENUM   StructureName;  // name of structure where the field is located
    UINT32        FieldOffset;    // offset to structure field to be loaded
    UINT          FieldSize;      // size (in bytes) of the field
    UINT          Field2Offset;   // second offset field used for strings with separate length storage
    UINT          Default;        // default value if not in registry
    UINT          Min;            // minimum value allowed
    UINT          Max;            // maximum value allowed
    char          *DefaultStr;    // defaults to NULL in initializer list below`
} CONFIG_PARAM;

/****************************************************************************
 * Registry Parameters Table
 *
 * This table contains a list of all of the configuration parameters
 * that the driver supports.   The driver will attempt to find these
 * parameters in the registry and use the registry value for these
 * parameters. If the parameter is not found in the registry, then the
 * default value is used.
 */
static CONFIG_PARAM paramTable[] = {
/*------------------------------------------------------------------------------
 *- Registry Name --------- Type -- Where to store it --------- Deflt Min  Max -
 *------------------------------------------------------------------------------*/

    /* OS-specific */
    RV(DriverDesc,          tANSI,  OS_ANSI(driverDesc),        0,    0,   0),
    RV(BusConfig,           tDEC,   CFG(busConfig),             0x80, 0,   0xffff),
    RV(MapRegisters,        tDEC,   OS(numMapRegisters),        256,  32,  MAX_MAP_REGISTERS),
    RV(descsToKeep,         tDEC,   CFG(descsToKeep),           96,   0,   128),

    /* Radio/HW */
    RV(rateCtrlEnable,      tDEC,   CFG(rateCtrlEnable),        1,    0,   1),
    RV(TriggerAdj,          tDEC,   CFG(txTrigThresh),          1000, 100, 32767),
    RV(CalibrationTime,     tDEC,   CFG(calibrationTime),       30,   0,   65535),
    RV(gpioPinFunc0,        tDEC,   CFG(gpioPinFuncs[0]),       0xff, 0,   0xff),
    RV(gpioPinFunc1,        tDEC,   CFG(gpioPinFuncs[1]),       0xff, 0,   0xff),
    RV(TransmitRate11a,     tDEC,   CFG(defaultRateIndex[WIRELESS_MODE_11a]),   0, 0, 7),
    RV(TransmitRate11b,     tDEC,   CFG(defaultRateIndex[WIRELESS_MODE_11b]),   0, 0, 3),
    RV(TransmitRate11g,     tDEC,   CFG(defaultRateIndex[WIRELESS_MODE_11g]),   0, 0, 11),
    RV(TransmitRate108g,    tDEC,   CFG(defaultRateIndex[WIRELESS_MODE_108g]),  0, 0, 7),
    RV(TransmitRateTurbo,   tDEC,   CFG(defaultRateIndex[WIRELESS_MODE_TURBO]), 0, 0, 7),
    RV(antennaSwitch,       tDEC,   CFG(diversityControl),      ANTENNA_CONTROLLABLE, ANTENNA_CONTROLLABLE, 2),

    /* 11g */
    RV(modeCTS,             tDEC,   CFG(modeCTS),               2,    0,   15),
    RV(rateCTS,             tDEC,   CFG(rateCTS),               4,    0,   15),
    RV(shortSlotTime,       tDEC,   CFG(shortSlotTime),         1,    0,   1),
    RV(gdraft5,             tDEC,   CFG(gdraft5),               0,    0,   1),
    RV(protectionType,      tDEC,   CFG(protectionType),        0,    0,   1),

    /* 802.11 spec stuff */
    RV(Ssid,                tSSID,  RV_DUMMY,                   0,    0,   0),
    RV(Ssid2,               tSSID,  RV_DUMMY,                   0,    0,   0),
    RV(Ssid3,               tSSID,  RV_DUMMY,                   0,    0,   0),
    RV(bssType,             tDEC,   CFG(bssType),               1,    1,   3),
    RV(beaconInterval,      tDEC,   CFG(beaconInterval),        100,  20,  1000),
    RV(FragThreshold,       tDEC,   CFG(userFragThreshold),     2346, 256, 32767),
    RV(RTSThreshold,        tDEC,   CFG(userRTSThreshold),      2346, 256, 32767),
    RV(HwTxRetries,         tDEC,   CFG(hwTxRetries),           4,    1,   15),
    RV(SwTxRetryScale,      tDEC,   CFG(swRetryMaxRetries),     6,    0,   15),
    RV(shortPreamble,       tDEC,   CFG(shortPreamble),         1,    0,   1),
    RV(atimWindow,          tDEC,   CFG(atimWindow),            0,    0,   1000),
    RV(cfpDuration,         tDEC,   CFG(cfpDuration),           0,    0,   1000),

    /* Regulatory */
    RV(NetBand,             tDEC,   CFG(NetBand),               MODE_SELECT_ALL, 0, 0xffff),
    RV(AdhocBand,           tDEC,   CFG(AdhocBand),             ATH_ADHOCBAND_AONLY, ATH_ADHOCBAND_BONLY, ATH_ADHOCBAND_108GONLY),
    RV(RD,                  tDEC,   CFG(RegDomain),             0xffff,     0,   0xffff),
    RV(tpc,                 tDEC,   CFG(tpScale),               0,     0,   4),
    XRV(ccode,              tANSI,  OS_ANSI(countryName),       0,     0,   0,    "US"),
    RV(adhocChannel,        tDEC,   CFG(adhocChannelFreq),      0,     0,   0xffff),
    RV(clist,               tANSI,  OS_ANSI(clist),             0,     0,   0),
    RV(ignore11dBeacon,     tDEC,   CFG(ignore11dBeacon),       0,     0,   1),
    RV(quietDuration,       tDEC,   CFG(quietDuration),         0,     0,   1000),
    RV(quietOffset,         tDEC,   CFG(quietOffset),           10,    0,   1000),
    RV(quietAckCtsAllow,    tDEC,   CFG(quietAckCtsAllow),      0,     0,   1),
    RV(extendedChanMode,    tDEC,   CFG(extendedChanMode),      1,     0,   1),
    RV(overRideTxPower,     tDEC,   CFG(overRideTxPower),       0,     0,   30),

    /* Security */
    RV(EncryptionAlg,       tDEC,   CFG(encryptionAlg),         ENCRYPTION_AES_OCB, ENCRYPTION_DISABLED, ENCRYPTION_AUTO),
    RV(privacyInvoked,      tDEC,   CFG(privacyInvoked),        0,     0,   1),
    RV(keyLength0,          tDEC,   CFG(keys[0].keyLength),     16,    5,   16),     // Length MUST COME BEFORE key
    RV(key0,                tKEY,   CFG(keys[0]),               0,     0,   0),
    RV(keyLength1,          tDEC,   CFG(keys[1].keyLength),     16,    5,   16),     // Length MUST COME BEFORE key
    RV(key1,                tKEY,   CFG(keys[1]),               0,     0,   0),
    RV(keyLength2,          tDEC,   CFG(keys[2].keyLength),     16,    5,   16),     // Length MUST COME BEFORE key
    RV(key2,                tKEY,   CFG(keys[2]),               0,     0,   0),
    RV(keyLength3,          tDEC,   CFG(keys[3].keyLength),     16,    5,   16),     // Length MUST COME BEFORE key
    RV(key3,                tKEY,   CFG(keys[3]),               0,     0,   0),
    RV(uniqKeyLength,       tDEC,   CFG(keys[UNIQ_KEY_INDEX].keyLength), 16, 5, 16), // Length MUST COME BEFORE key
    RV(uniqKey,             tKEY,   CFG(keys[UNIQ_KEY_INDEX]),  0,     0,   0),
    RV(defaultKey,          tDEC,   CFG(defaultKey),            0,     0,   4),
#ifdef WLAN_CONFIG_WPA
    RV(leapEnabled,         tDEC,   CFG(leapEnabled),           0,     0,   1),
    RV(leapUserName,        tSTRLEN,CFG_LEN(leapUserName),      0,     0,   0),
    RV(leapUserPasswdLen,   tDEC,   CFG(leapUserPasswdLen),     0,     0,   256),    // Length MUST COME BEFORE password
    RV(leapUserPasswd,      tLEAPPW,CFG(leapUserPasswd),        0,     0,   0),
    RV(leapTimeout,         tDEC,   CFG(leapTimeout),           30000, 500, 60000),
#endif /* WLAN_CONFIG_WPA */
    RV(CardCfgId,           tANSI,  OS_ANSI(cardCfgId),         0,     0,   0),
    RV(authType,            tDEC,   CFG(authType),              AUTH_TYPE_AUTO, AUTH_TYPE_OPEN, AUTH_TYPE_WPAPSK),
    RV(authTypeUseOnly,     tDEC,   CFG(authTypeUseOnly),       AUTH_TYPE_DEFAULT, AUTH_TYPE_DEFAULT, AUTH_TYPE_SHARED),
    RV(wpaEnabled,          tDEC,   CFG(wpaEnabled),            0,     0,   1),
    RV(mixedPrivacyAllow,   tDEC,   CFG(mixedPrivacyAllow),     0,     0,   1),

    /* Connection Services */
    RV(roamRssiA,           tDEC,   CFG(roamRssiA),             15,    0,   95),
    RV(roamRssiB,           tDEC,   CFG(roamRssiB),             24,    0,   95),
    RV(roamRssiBOnly,       tDEC,   CFG(roamRssiBOnly),         8,     0,   95),
    RV(roamRateA,           tDEC,   CFG(roamRateA),             24000, 0,   54000),
    RV(roamRateB,           tDEC,   CFG(roamRateB),             9000,  0,   11000),
    RV(roamRateBOnly,       tDEC,   CFG(roamRateBOnly),         5000,  0,   11000),
    RV(bssAgingPeriod,      tDEC,   CFG(bssAgingPeriod),        120,   0,   65535),
    RV(clearListOnScan,     tDEC,   CFG(clearListOnScan),       0,     0,   1),
    RV(bkScanEnable,        tDEC,   CFG(bkScanEnable),          1,     0,   1),
    RV(scanTimePreSleep,    tDEC,   CFG(scanTimePreSleep),      300,   0,   65535),
    RV(sleepTimePostScan,   tDEC,   CFG(sleepTimePostScan),     60,    0,   65535),
    RV(scanType,            tDEC,   CFG(scanType),              ANY_SCAN, ACTIVE_SCAN, ANY_SCAN),
    RV(noBeaconTimeout,     tDEC,   CFG(noBeaconTimeout),       3000,  0,   65535),
    RV(reAssocEnable,       tDEC,   CFG(reAssocEnable),         1,     0,   1),
    RV(prefBssid1,        tBSSID,   CFG(prefBssids[1]),         0,     0,   1),
    RV(prefBssid2,        tBSSID,   CFG(prefBssids[2]),         0,     0,   1),
    RV(prefBssid3,        tBSSID,   CFG(prefBssids[3]),         0,     0,   1),
    RV(prefBssid4,        tBSSID,   CFG(prefBssids[4]),         0,     0,   1),

    /* Power Management */
    RV(radioEnable,         tDEC,   CFG(radioEnable),           1,     0,   1),
    RV(SleepMode,           tDEC,   CFG(sleepMode),             0,     0,   2),
    RV(SleepTimePwrSave,    tDEC,   CFG(sleepTimePwrSave),      1000,  100, 65535),
    RV(SleepTimePerf,       tDEC,   CFG(sleepTimePerf),         200,   100, 65535),
    RV(AwakeTimePwrSave,    tDEC,   CFG(inactivityTimePwrSave), 200,   1,   65535),
    RV(AwakeTimePerf,       tDEC,   CFG(inactivityTimePerf),    6000,  1,   65535),
    RV(SleepLogEnable,      tDEC,   CFG(sleepLogEnable),        0,     0,   1),
    RV(SleepSampleTime,     tDEC,   CFG(sleepSampleInterval),   10,    10,  1000),
    RV(RollingAvgPeriod,    tDEC,   CFG(rollingAvgPeriod),      10,    1,   60),
    RV(overrideACstatus,    tDEC,   CFG(overrideACstatus),      0,     0,   1),
    RV(enable32KHzClock,    tDEC,   CFG(enable32KHzClock),      0,     0,   2),

    /* Windows Interface */
    RV(translateNullSsid,   tDEC,   CFG(translateNullSsid),     0,     0,   1),

    /* Miscellaneous */
#ifdef WME
    RV(WmeEnabled,                 tDEC,   CFG(WmeEnabled),             0,     0,   1),
#endif
    RV(abolt,               tDEC,   CFG(abolt),                 0xbf,  0,   0xffff),
    RV(removeNoGSubId,      tDEC,   CFG(removeNoGSubId),        0,     0,   1),
#if 0 /* TBD */
    RV(compProc,            tDEC,   CFG(compProc),              0,     0,   3),
    RV(compWinSize,         tDEC,   CFG(compWinSize),           4096,  256, 4096),
#endif
    RV(burstTime,           tDEC,   CFG(burstTime),             2,     0,   1023),
    RV(burstSeqThreshold,   tDEC,   CFG(burstSeqThreshold),     3,     2,   255),
    RV(clientName,         tANSI,   CFG(clientName),            0,     0,   1),
    RV(regCapBits,          tDEC,   CFG(regCapBits),            0,     0,   0x001f),

    /* CCX Knobs */
    RV(ccxRadioMeasEnable,   tDEC,  CFG(ccxRadioMeasEnable),    1,     0,   1),
    RV(ccxRmMaxOffChanTime,  tDEC,  CFG(ccxRmMaxOffChanTime),   40,    0,   255),
    RV(ccxRmOffChanTimeWhenBusy, tDEC,  CFG(ccxRmOffChanTimeWhenBusy),10, 0, 255),
#ifdef DEBUG
    RV(powerDebugLevel,     tDEC,   GLOBAL(powerDebugLevel),    0,     0,   256),
    RV(rxDebugLevel,        tDEC,   GLOBAL(rxDebugLevel),       0,     0,   256),
    RV(oidDebugLevel,       tDEC,   GLOBAL(oidDebugLevel),      0,     0,   256),
    RV(txDebugLevel,        tDEC,   GLOBAL(txDebugLevel),       0,     0,   256),
    RV(mlmeDebugLevel,      tDEC,   GLOBAL(mlmeDebugLevel),     0,     0,   256),
    RV(lockDebugLevel,      tDEC,   GLOBAL(lockDebugLevel),     0,     0,   9),
    RV(cservDebugLevel,     tDEC,   GLOBAL(cservDebugLevel),    0,     0,   9),
#endif
};


//-----------------------------------------------------------------------------
// Procedure:   ParseRegistryParameters
//
// Description: This routine will parse all of the parameters out of the
//              registry and store the values in the passed config structure.
//              Structure.  If the parameter is not present in the registry,
//              then the default value for the parameter will be placed into
//              the config structure.  This routine also checks the validity
//              of the parameter value.  If the value is out of range, the
//              default value will be used.
//-----------------------------------------------------------------------------
NDIS_STATUS
ParseRegistryParameters(WLAN_DEV_INFO *pDevInfo, NDIS_HANDLE ConfigHandle,
                        WLAN_STA_CONFIG *pConfig)
{
    NDIS_STATUS  status;
    CONFIG_PARAM *pParam;
    UINT         i;
    ULONG        value;
    PUCHAR       basePtr;
    PUCHAR       fieldPtr;
    PVOID        pAddr;
    UINT         Length;
    char         regName[32];
    OS_DEV_INFO  *pOsInfo = pDevInfo->pOSHandle;


    uiPrintf("> %s\n", __FUNCTION__);

    /* Initialize any values required before parsing the parameters */
    ssidArrayInit(&pConfig->cfgSsids);

    /* Loop through the registry values specified in the above array */
    for (i = 0, pParam = paramTable; i < NUM_REG_PARAM; i++, pParam++) {
        A_BOOL found;
        A_BOOL useDefault = FALSE;

        switch (pParam->StructureName) {
        case sOS:
            basePtr = (PUCHAR)pOsInfo;
            break;
        case sCONFIG:
            ASSERT(pConfig);
            basePtr = (PUCHAR)pConfig;
            break;
        case sNONE:
            basePtr = (PUCHAR)0;
            break;
        default:
            ASSERT(0);
        }

        fieldPtr = basePtr + pParam->FieldOffset;

        A_STRCPY(regName, pParam->RegAscName);

        /*
         * All of the registry parameters are stored as strings.
         * On NT 4, using NdisReadConfiguration with parameterType ==
         * NdisParameterInteger on a string will succeed (status wise), but
         * the parameter type returned will be string and the
         * buffer contents will be invalid.
         * To fix this, force NdisReadConfiguration to read all
         * parameters as strings, and then convert to integers as needed.
         */

        /* Get the configuration value for the parameter. */
        NdisReadConfiguration(&status, &pParamValue, ConfigHandle,
                              &pParam->RegVarName, RT_ENUM_2_NDIS(pParam->RegVarType));

        found = (status == NDIS_STATUS_SUCCESS);

        /* Process the registry value based on type */
        switch (pParam->RegVarType) {
        case tHEX:
        case tDEC:
            if (found) {
            } else {
                useDefault = TRUE;
            }

            if (useDefault) {
                /* A parameter wasn't present or was invalid */
                value = pParam->Default;
            }

            uiPrintf("%-25s %d\n", regName, value);

            /* Store away the value into its proper spot */
            switch (pParam->FieldSize) {
            case sizeof(UCHAR):
                *((PUCHAR)fieldPtr)  = (UCHAR)value;
                break;

            case sizeof(USHORT):
                *((PUSHORT)fieldPtr) = (USHORT)value;
                break;

            case sizeof(ULONG):
                *((PULONG)fieldPtr)  = (ULONG)value;
                break;

            default:
                /* Needs to be one of the sizes above */
                ASSERT(0);
                break;
            }
            break;

        default: {
            /* If it wasn't any of the above types, assume it's a string */
            ANSI_STRING ansiStr;
            CHAR        ansiBuf[133];

            /* Initialize our ANSI string */
            ansiStr.MaximumLength = sizeof(ansiBuf);
            ansiStr.Buffer        = ansiBuf;
            ansiStr.Length        = 0;

            if (pParam->DefaultStr) {
                A_STRCPY(ansiStr.Buffer, pParam->DefaultStr);
                ansiStr.Length = (USHORT)strlen(pParam->DefaultStr);
                ASSERT(ansiStr.Length < sizeof(ansiBuf));
            }


            /* Make the string in ansiStr printable (null-terminated) */
            ansiStr.Buffer[ansiStr.Length] = '\0';

            uiPrintf("%-25s %s\n", regName, ansiStr.Buffer);

            switch (pParam->RegVarType) {
            case tSSID:
                /*
                 * Only add the SSID if it was found in the registry.  If none
                 * are in the registry, then there will be no valid SSIDs added to
                 * the array, and autoconnection is thusly disabled.
                 */
                if (found) {
                    ssidArrayAdd(&pConfig->cfgSsids, ansiStr.Buffer, (A_UINT8)ansiStr.Length);
                } else {
                    ssidArrayAdd(&pConfig->cfgSsids, "", 0);
                }
                break;

            case tKEY: {
                WLAN_PRIV_RECORD *pKey = (WLAN_PRIV_RECORD *)fieldPtr;
                A_INT32         encryptedLen;

                ASSERT(pKey);
                /* Length of encrypted keys in registry is not simply 40, 104 or 128 bits. */
                encryptedLen = pKey->keyLength == 5 ? 8 : 16;

                asciiToKey(ansiStr.Buffer, ansiStr.Length, pKey, pKey->keyLength, encryptedLen);
                break;
            }

            case tLEAPPW: {
                /*
                 * LEAP Password is a special case as it needs to call
                 * asciiToPasswd() and return if things go badly.
                 */
                A_UINT32 *pLen = (A_UINT32 *)(basePtr + pParam->Field2Offset);

                ASSERT(pLen);
                if (!asciiToPasswd(ansiStr.Buffer, ansiStr.Length, fieldPtr,
                                   (A_UINT16)*pLen))
                {
                    uiPrintf("Could not convert LEAP passwd to ascii.\n");
                    return NDIS_ERROR_CODE_UNSUPPORTED_CONFIGURATION;
                }
                break;
            }

            case tSTRLEN: {
                /*
                 * Field2Offset points to a separate A_UINT32 length element
                 * associated with the string
                 */
                A_UINT32 *pLen = (A_UINT32 *)(basePtr + pParam->Field2Offset);

                ASSERT(pLen);
                A_STRNCPY(fieldPtr, ansiStr.Buffer, ansiStr.Length);
                *pLen = ansiStr.Length;
                break;
            }

            case tANSI: {
                /*
                 * Generic ANSI string handler.  These are NDIS-specific and
                 * should probably be moved to a separate routine that would
                 * copy the string values out of the WLAN_STA_CONFIG struct
                 * and into their proper OS-specific locations.
                 */
                ANSI_STRING *pStr = (ANSI_STRING *)fieldPtr;
                CHAR        *pBuf = (CHAR *)(basePtr + pParam->Field2Offset);

                ASSERT(pStr && pBuf);
                pStr->Length        = ansiStr.Length;
                pStr->MaximumLength = ansiStr.Length; // won't change later, anyway
                pStr->Buffer        = pBuf;
                A_STRNCPY(pBuf, ansiStr.Buffer, ansiStr.Length);
                pBuf[pStr->Length] = '\0';  // Null terminate the string
                break;
            }

            case tBSSID: {
                WLAN_MACADDR *pBssid = (WLAN_MACADDR *)fieldPtr;
                A_BOOL       goodAddr = FALSE;

                if (ansiStr.Length == ETH_LENGTH_OF_ADDRESS * 2) {
                    goodAddr = asciiToMacAddr(ansiStr.Buffer, pBssid);
                }

                if (goodAddr == FALSE) {
                    A_MACADDR_COPY(&nullMacAddr, pBssid);
                }
                break;
            }

            default:
                /* Shouldn't make it here */
                ASSERT(0);
            } // switch on type of string
        } // default
        } // switch on overall type
    } // for loop for each config parameter


    return NDIS_STATUS_SUCCESS;
}

