/*
 * $Id: //depot/sw/branches/1.3_USB_LINUX_port/src/USB/wlan/host/sta/powermgt.c#3 $
 *
 * Copyright (c) 2001-2003 Atheros Communications, Inc., All Rights Reserved
 *
 * This file contains the power management APIs and policies.
 *
 * One of these policies is that block sleep has priority over normal sleep.  This
 * allows, for instance, background scanning to proceed even though the NIC was
 * put in sleep due to lack of network activity.
 */

#include "wlantype.h"
#include "wlanproto.h"
#include "wlandrv.h"
#include "wlanDev.h"
#include "wlanext.h"
#include "wlansmeext.h"
#include "wlanSend.h"
#include "wlanchannel.h"
#include "stacserv.h"
#include "wlanCapDe.h"
#include "usbstub.h"

#ifdef DEBUG
int powerDebugLevel;
#define powerPrintf     if (powerDebugLevel > 1) uiPrintf
#define ASSERT_PWR_MGMT if (powerDebugLevel > 0) ASSERT
#else
#define powerPrintf(...)
#define ASSERT_PWR_MGMT ASSERT
#endif

// The following strings are used for debug print statements only; they're
// optimized out by the compiler in the production build.
static char *powerStrings[]  = {"D0", "D1", "D2", "D3"};
static char *sleepStrings[]  = {"AWAKE", "PENDING_SLEEP", "SLEEP",
                                "PENDING_BLOCKING_SLEEP", "BLOCKING_SLEEP"};
static char *eventStrings[]  = {"WAKE_UP", "HIBERNATE", "RADIO", "AC",
                                "FORCE_SLEEP", "FAKE_SLEEP"};
static char *valueStrings[]  = {"ENABLE", "ENABLED", "DISABLE", "DISABLED"};
static char *wakeupStrings[] = {"SET_PWR_TO_1", "REDUCE_PWR_FROM_1",
                                "SET_PWR_TO_2", "REDUCE_PWR_FROM_2",
                                "SET_PWR_TO_3", "REDUCE_PWR_FROM_3"};

// forward references
A_STATUS
drvPowerSet(WLAN_DEV_INFO *pDev, PWR_MGMT_DEVICE_POWER_TYPE DxRequest,
            A_BOOL D1PwrUp);

//-----------------------------------------------------------------------------
// Procedure:   powerInitialization
//
// Description: This routine initializes the power management timers, and sets
//              the mode of the device to Awake state if possible
//
// Returns: A_OK upon success, A_HARDWARE if the chip could not be placed into
//          the a power-up state.
//
//-----------------------------------------------------------------------------
A_STATUS
powerInitialization(WLAN_DEV_INFO *pDev)
{
    A_STATUS    status;

    powerPrintf("powerInitialization: Entry\n");

    pDev->powerMgmt.powerChangeTxDesc = 0;

    // Initialize for PS-Polls
    status = powerPSPollInit(pDev);
    if (status != A_OK) {
        return status;
    }

    switch (pDev->staConfig.sleepMode) {
    case POWERMGT_SLEEP:
        // Maximum powersave
        pDev->noActivityInterval       = pDev->staConfig.inactivityTimePwrSave /
                                         CSERV_TIME_STAPOLL;
        pDev->staConfig.listenInterval = pDev->staConfig.sleepTimePwrSave;
        break;
    case POWERMGT_PERF:
    case POWERMGT_CAM:
    default:
        // Normal powersave or Continuously Awake Mode (CAM)
        // Note that when in CAM, we still set these values for use in holding
        // off background scanning.  Setting these values should not cause a STA
        // in CAM mode to ever go to sleep.
        pDev->noActivityInterval       = pDev->staConfig.inactivityTimePerf /
                                         CSERV_TIME_STAPOLL;
        pDev->staConfig.listenInterval = pDev->staConfig.sleepTimePerf;
    }

    // Initialize the current countdown for inactivity, remove left-over sleep states,
    // believe that AP knows we're awake (no PM)
    pDev->noActivityCur = pDev->noActivityInterval;
    pDev->powerMgmt.forceSleep = DISABLE;
    pDev->powerMgmt.fakeSleep  = DISABLE;
    pDev->powerMgmt.apPmState  = FALSE;

    return status;
}


//-----------------------------------------------------------------------------
// Procedure:   powerSet
//
// Description: This routine attempts to change the power setting of the NIC on
//              the network, or in response to a system power event. It sends
//              out a power management packet if requested.
//
// Returns: A_OK upon success, A_PENDING if a notification was sent to the AP
//          and its' ACK is pending, A_ECANCELED if the request could not be
//          completed (a subsequent request may be successful), A_HARDWARE if
//          the chip could not be placed into the desired power setting.
//
//-----------------------------------------------------------------------------
A_STATUS
powerSet(WLAN_DEV_INFO *pDev, PWR_MGMT_EVENT_TYPE event, A_UINT32 value, A_BOOL notifyAP, void (*callBack)(void *, int, A_BOOL))
{
    A_UINT32 newState  = pDev->powerMgmt.powerState;
    A_UINT32 oldState  = pDev->powerMgmt.powerState;
    A_STATUS status    = A_OK;
    A_BOOL   D1PwrUp   = FALSE;
    A_BOOL   opPending = FALSE;

    powerPrintf("powerSet: current power/sleep states = %s/%s, request = %s/",
                powerStrings[pDev->powerMgmt.powerState],
                sleepStrings[pDev->powerMgmt.powerSleepSubstate],
                eventStrings[event]);

    // While in hibernate, allow only hibernate requests
    if ((pDev->powerMgmt.hibernate == ENABLED) && (event != HIBERNATE)) {
        return A_EINVAL;
    }

    // Validate and store event and value
    switch (event) {

    case HIBERNATE:
        powerPrintf("%s\n", valueStrings[value]);
        ASSERT_PWR_MGMT(value != pDev->powerMgmt.hibernate);
        if (value == pDev->powerMgmt.hibernate) {
            return status;
        }
        ASSERT_PWR_MGMT((value == ENABLE) || (value == DISABLE));
        if ((value != ENABLE) && (value != DISABLE)) {
            return A_EINVAL;
        }
        pDev->powerMgmt.hibernate = value;
        break;

    case RADIO_ONLINE:
        powerPrintf("%s\n", valueStrings[value]);
        ASSERT_PWR_MGMT(value != pDev->powerMgmt.radio);
        if (value == pDev->powerMgmt.radio) {
            return status;
        }
        ASSERT_PWR_MGMT((value == ENABLE) || (value == DISABLE));
        if ((value != ENABLE) && (value != DISABLE)) {
            return A_EINVAL;
        }
        pDev->powerMgmt.radio = value;
        break;

    case AC_ONLINE:
        powerPrintf("%s\n", valueStrings[value]);
        if (value == pDev->powerMgmt.onAC) {
            return status;
        }
        ASSERT_PWR_MGMT((value == ENABLE) || (value == DISABLE));
        if ((value != ENABLE) && (value != DISABLE)) {
            return A_EINVAL;
        }
        pDev->powerMgmt.onAC = value;
        break;

    case WAKE_UP:
        powerPrintf("%s\n", wakeupStrings[value]);
        if (value == SET_PWR_TO_1) {
            pDev->powerMgmt.internalAwake_1 = SET_PWR_TO_1;
            pDev->powerMgmt.internalAwakeRefCnt_1++;
        } else if (value == REDUCE_PWR_FROM_1) {
            ASSERT_PWR_MGMT(pDev->powerMgmt.internalAwakeRefCnt_1);
            if (pDev->powerMgmt.internalAwakeRefCnt_1) {
                pDev->powerMgmt.internalAwakeRefCnt_1--;
                if (pDev->powerMgmt.internalAwakeRefCnt_1 == 0) {
                    pDev->powerMgmt.internalAwake_1 = REDUCE_PWR_FROM_1;
                } else {
                    powerPrintf("powerSet: no change in state\n");
                    powerPrintf("powerSet: awake ref cnt_1 = %d\n", pDev->powerMgmt.internalAwakeRefCnt_1);
                    return status;
                }
            } else {
                return status;
            }
        } else if (value == SET_PWR_TO_2) {
            pDev->powerMgmt.internalAwake_2 = SET_PWR_TO_2;
            pDev->powerMgmt.internalAwakeRefCnt_2++;
        } else if (value == REDUCE_PWR_FROM_2) {
            ASSERT_PWR_MGMT(pDev->powerMgmt.internalAwakeRefCnt_2);
            if (pDev->powerMgmt.internalAwakeRefCnt_2) {
                pDev->powerMgmt.internalAwakeRefCnt_2--;
                if (pDev->powerMgmt.internalAwakeRefCnt_2 == 0) {
                    pDev->powerMgmt.internalAwake_2 = REDUCE_PWR_FROM_2;
                } else {
                    powerPrintf("powerSet: no change in state\n");
                    powerPrintf("powerSet: awake ref cnt_2 = %d\n", pDev->powerMgmt.internalAwakeRefCnt_2);
                    return status;
                }
            } else {
                return status;
            }
        } else if (value == SET_PWR_TO_3) {
            pDev->powerMgmt.internalAwake_3 = SET_PWR_TO_3;
            pDev->powerMgmt.internalAwakeRefCnt_3++;
        } else if (value == REDUCE_PWR_FROM_3) {
            ASSERT_PWR_MGMT(pDev->powerMgmt.internalAwakeRefCnt_3);
            if (pDev->powerMgmt.internalAwakeRefCnt_3) {
                pDev->powerMgmt.internalAwakeRefCnt_3--;
                if (pDev->powerMgmt.internalAwakeRefCnt_3 == 0) {
                    pDev->powerMgmt.internalAwake_3 = REDUCE_PWR_FROM_3;
                } else {
                    powerPrintf("powerSet: no change in state\n");
                    powerPrintf("powerSet: awake ref cnt_3 = %d\n", pDev->powerMgmt.internalAwakeRefCnt_3);
                    return status;
                }
            } else {
                return status;
            }
        } else {
            ASSERT_PWR_MGMT(0);
            return A_EINVAL;
        }
        break;

    case FAKE_SLEEP:
        powerPrintf("%s\n", valueStrings[value]);
        if (value == pDev->powerMgmt.fakeSleep) {
            return status;
        }
        ASSERT_PWR_MGMT((value == ENABLE) || (value == DISABLE));
        if ((value != ENABLE) && (value != DISABLE)) {
            return A_EINVAL;
        }

        if (value == ENABLE) {
            //Fix bug 8911. We need drain out fast frame queue before we send NULL data out.
            txFfFlush(pDev);           

            // Wake up for background scan.  Fake sleep implies we always tell AP
            // we're sleeping... but not really.
            if (notifyAP) {
                status = powerSendMgtFrame(pDev, TRUE);
                if (status == A_PENDING) {
                    pDev->powerMgmt.powerSleepSubstate = STATE_PENDING_BLOCKING_SLEEP;
                    opPending = TRUE;
                } else if (status == A_OK) {
                    pDev->powerMgmt.powerSleepSubstate = STATE_BLOCKING_SLEEP;
                }
            } else {
                pDev->powerMgmt.powerSleepSubstate = STATE_BLOCKING_SLEEP;
            }
        } else {
            // Background scan is done, restore awake state w/AP wake up
            if (notifyAP) {
                status = powerSendMgtFrame(pDev, FALSE);
                if ((status == A_PENDING) || (status == A_OK)) {
                    pDev->powerMgmt.powerSleepSubstate = STATE_AWAKE;
                }
            } else {
                pDev->powerMgmt.powerSleepSubstate = STATE_AWAKE;
            }
			// In any case, let NDIS send
            osIndicateResourcesAvailable(pDev);
        }
        if ((status != A_PENDING) && (status != A_OK)) {
            return status;
        }

        pDev->powerMgmt.fakeSleep = value;
        break;

    case FORCE_SLEEP:
        powerPrintf("%s\n", valueStrings[value]);
        if (value == pDev->powerMgmt.forceSleep) {
            return status;
        }
        ASSERT_PWR_MGMT((value == ENABLE) || (value == DISABLE));
        if ((value != ENABLE) && (value != DISABLE)) {
            return A_EINVAL;
        }
        pDev->powerMgmt.forceSleep = value;
        break;


    default:
        return A_EINVAL;
    }

    if (pDev->powerMgmt.hibernate == ENABLE) {
        // Entering hibernate or standby, prepare for full power-off
        if (notifyAP) {
            drvPowerSet(pDev, D0_STATE, FALSE);

            wlanMlmeDeauthRequest(pDev, &pDev->baseBss,
                                 &pDev->baseBss.bssId, REASON_AUTH_LEAVING, TRUE);

            pDev->localSta->staState &= ~ STATE_CONNECTED;
            pDev->localSta->transState = TSTATE_QUIET;

            // give them fair chance to get transmitted
            // busy wait for 10 milliseconds
            udelay(10000);
        }

        newState = D3_STATE;
        pDev->powerMgmt.hibernate = ENABLED;

    } else if (pDev->powerMgmt.internalAwake_3 == SET_PWR_TO_3) {
        newState = D0_STATE;

    } else if ((pDev->powerMgmt.radio == DISABLE) ||
               (pDev->powerMgmt.radio == DISABLED))
    {
        // GUI or switch turned off radio; sleep the MAC
        if (notifyAP) {
            drvPowerSet(pDev, D0_STATE, FALSE);

            wlanMlmeDeauthRequest(pDev, &pDev->baseBss,
                                 &pDev->baseBss.bssId, REASON_AUTH_LEAVING, TRUE);

            pDev->localSta->staState &= ~ STATE_CONNECTED;
            pDev->localSta->transState = TSTATE_QUIET;

            // give them fair chance to get transmitted
            // busy wait for 10 milliseconds
            udelay(10000);
        }

        newState = D2_STATE;
        pDev->powerMgmt.radio = DISABLED;

    } else if (pDev->powerMgmt.internalAwake_2 == SET_PWR_TO_2) {
        newState = D0_STATE;

    } else if (pDev->powerMgmt.fakeSleep == ENABLE) {
        newState = D0_STATE;

    } else if ((pDev->powerMgmt.onAC == ENABLE) ||
               (pDev->powerMgmt.onAC == ENABLED))
    {
        // AC cord plugged in; no power save at all
        newState = D0_STATE;
        pDev->powerMgmt.onAC = ENABLED;

    } else if (pDev->staConfig.sleepMode == POWERMGT_CAM) {
        newState = D0_STATE;

    } else if ((pDev->powerMgmt.forceSleep == ENABLE) ||
               (pDev->powerMgmt.forceSleep == ENABLED))
    {
        newState = D2_STATE;
        pDev->powerMgmt.forceSleep = ENABLED;

    } else if (pDev->powerMgmt.internalAwake_1 == SET_PWR_TO_1) {
        newState = D0_STATE;

    } else {
        // Default power state
        if (pDev->bssDescr->bsstype == INDEPENDENT_BSS) {
            newState = D0_STATE;

        } else {
            // If assoc, prepare for network sleep else stay awake
            if ((pDev->localSta->staState & STATE_CONNECTED) == STATE_CONNECTED) {
                if (!pDev->powerMgmt.apPmState) {
                    pDev->powerMgmt.powerSleepSubstate = STATE_AWAKE;
                }
                pDev->powerMgmt.sleepTimerEnable = TRUE;
                newState = D1_STATE;
                if ((pDev->powerMgmt.powerSleepSubstate != STATE_SLEEP) ||
                     pDev->powerMgmt.dtim)
                {
                    D1PwrUp = TRUE;
                }
            } else {
                newState = D0_STATE;
            }
        }
    }

    status = drvPowerSet(pDev, newState, D1PwrUp);
    powerPrintf("powerSet: new power/sleep states = %s/%s\n",
                powerStrings[pDev->powerMgmt.powerState],
                sleepStrings[pDev->powerMgmt.powerSleepSubstate]);
    powerPrintf("powerSet: awake ref cnt 1 = %d\n", pDev->powerMgmt.internalAwakeRefCnt_1);
    powerPrintf("powerSet: awake ref cnt 2 = %d\n", pDev->powerMgmt.internalAwakeRefCnt_2);
    powerPrintf("powerSet: awake ref cnt 3 = %d\n", pDev->powerMgmt.internalAwakeRefCnt_3);

    // If caller wanted a callback upon request completion and it's not done yet,
    // save caller's callback routine to use later, otherwise indicate it now
    // TODO: if and when there is more than one callback routine that is passed down (currently
    // only mlmePowerCallback()), the routine's address should be saved into and used from
    // the txDesc of the power mgt frame sent to the AP
    if (status == A_OK && callBack) {
        if (opPending) {
            pDev->powerMgmt.PowerRequestCallback = callBack;
        } else {
            (*callBack)(pDev, status, powerEnabled(pDev));
        }
    }
    
	return opPending ? A_PENDING : status;
}

A_BOOL
powerEnabled(WLAN_DEV_INFO *pDev)
{
    if ((pDev->powerMgmt.powerState == D2_STATE) ||
        (pDev->powerMgmt.powerState == D3_STATE))
    {
        return FALSE;
    }

    if ( pDev->powerMgmt.powerState == D1_STATE &&
         (pDev->powerMgmt.powerSleepSubstate == STATE_PENDING_SLEEP ||
          pDev->powerMgmt.powerSleepSubstate == STATE_SLEEP) )
    {
        return FALSE;
    }

    return TRUE;
}


A_STATUS
powerSendRequest(WLAN_DEV_INFO *pDev)
{
    A_STATUS status = A_OK;

    // There is tx request activity
    pDev->noActivityCur = pDev->noActivityInterval;

    if (pDev->powerMgmt.fakeSleep == ENABLE) {
        return A_NO_RESOURCE;
    }

    if (powerEnabled(pDev)) {
        return status;
    }

    switch (pDev->powerMgmt.powerState) {
    case D1_STATE:
        switch (pDev->powerMgmt.powerSleepSubstate) {
        case STATE_SLEEP:
        case STATE_PENDING_SLEEP:

            /* Set the device to awake now */
            wdcTargetSetPowerMode(pDev->targetHandle, TARGET_DEVICE_AWAKE, &status);
            if (status == A_OK) {
                pDev->powerMgmt.powerSleepSubstate = STATE_AWAKE;
            } else {
                status = A_NO_RESOURCE;
            }
            break;

        case STATE_BLOCKING_SLEEP:
        case STATE_PENDING_BLOCKING_SLEEP:
            status = A_NO_RESOURCE;
            break;

        default:
            break;
        }
        break;

    case D2_STATE:
    case D3_STATE:
        status = A_NO_RESOURCE;
        break;

    default:
        break;
    }

    return status;
}

/******************************************************************************
 * staFrameTransmit
 *
 * Prepare to send an 802.11 packet.  Assumes NIC power is enabled.
 *
 * RETURNS: A_OK or A_ERROR or A_NO_MEMORY
 */
static A_STATUS
staFrameTransmit(WLAN_DEV_INFO *pDevInfo, ATHEROS_DESC *pDesc)
{
    WLAN_MACADDR  *pDestAddr = &pDesc->pBufferVirtPtr.header->address1;
    QUEUE_DETAILS *pTxQueue;
    A_STATUS       ret;

    /*
     * set the dest SIB here! callers in the STA side
     * code aren't very religious about setting it
     */
    pDesc->pDestSibEntry = sibEntryFind(pDevInfo, pDestAddr, NULL);

    /*
     * Station uses the basic virtual device only - and
     * everything goes out the main tx queue
     */
    pDesc->pOpBss = &pDevInfo->baseBss;
    if (IS_CHAN_G(pDevInfo->staConfig.pChannel->channelFlags) &&
        pDesc->pDestSibEntry && 
        (pDesc->pDestSibEntry->wlanMode == STA_MODE_G) && 
        (pDevInfo->staConfig.abolt & ABOLT_BURST))
    {
        pTxQueue = &pDesc->pOpBss->burstQueue;
    }
    else {
#ifdef WME
        if (pDevInfo->staConfig.WmeEnabled) {
            /* for pwr mgt only */
            pTxQueue = &pDevInfo->baseBss.wmequeues[TXQ_ID_FOR_DATA];
        } else {
            pTxQueue = &pDevInfo->baseBss.txQueue;        
        }
#else
        pTxQueue = &pDevInfo->baseBss.txQueue;        
#endif
    }

    pDesc->targetTxQId = pTxQueue->targetQueueId;
    ret = wlanFrameTransmit(pDevInfo, pTxQueue, pDesc);

    return ret;
}

/******************************************************************************
 * powerFrameTransmit
 *
 * Prepare to send an 802.11 packet.  Called by internal management
 * routines only.  Turn on power to NIC if it's not enabled.
 *
 * RETURNS: OK or ERROR.
 */
A_STATUS
powerFrameTransmit(WLAN_DEV_INFO *pDev, ATHEROS_DESC *pDesc)
{
    A_STATUS status = A_OK;

    if (powerSendRequest(pDev) != A_OK)
    {
        freeBuffandDescChain(pDev, pDesc);
        return A_ECANCELED;
    }

    ASSERT(pDev->powerMgmt.powerState != D2_STATE);
    pDev->powerMgmt.powerSleepSubstate = STATE_AWAKE;

    // There is tx request activity
    pDev->noActivityCur = pDev->noActivityInterval;

    status = staFrameTransmit(pDev, pDesc);

    return status;
}

void
powerMgtPoll(WLAN_DEV_INFO *pDev)
{
    A_STATUS    status;

    // Can't go into network sleep for the following reasons:
    // 1. Not in power state that allows it
    // 2. One or more clients wants the NIC awake
    // 3. Not associated
    // 4. Power already off (for same or other reason)
    if ((pDev->powerMgmt.powerState != D1_STATE) ||
        (pDev->powerMgmt.internalAwakeRefCnt_1))
    {
        return;
    }

    if ((pDev->localSta->staState & STATE_CONNECTED) != STATE_CONNECTED ||
        !powerEnabled(pDev) ||
        !pDev->powerMgmt.sleepTimerEnable)
    {
        return;
    }

    // Inform AP of intention to sleep.  Will do so upon ACK reception.
    // If synchronously successful, AP now knows we're going to enter sleep.
    if (pDev->powerMgmt.powerSleepSubstate == STATE_AWAKE &&
        --pDev->noActivityCur <= 0)
    {
        status = powerSendMgtFrame(pDev, TRUE);
        if (status == A_PENDING) {
            pDev->powerMgmt.powerSleepSubstate = STATE_PENDING_SLEEP;
        } else if (status == A_OK) {
            pDev->powerMgmt.powerSleepSubstate = STATE_SLEEP;
        }
        pDev->noActivityCur = pDev->noActivityInterval;
    }
}

void
powerReceiveIndication(WLAN_DEV_INFO *pDev)
{
    // There is receive activity
    pDev->noActivityCur = pDev->noActivityInterval;

    if (pDev->powerMgmt.powerState != D1_STATE) {
        return;
    }

    if (pDev->powerMgmt.powerSleepSubstate == STATE_SLEEP) {

        A_STATUS status;
         /* Set the device to Sleep now */
        wdcTargetSetPowerMode(pDev->targetHandle, TARGET_DEVICE_AWAKE, &status);
        if (status == A_OK) {
            pDev->powerMgmt.powerSleepSubstate = STATE_AWAKE;
        }
    } else if (pDev->powerMgmt.powerSleepSubstate == STATE_PENDING_SLEEP) {
        pDev->powerMgmt.powerSleepSubstate = STATE_AWAKE;
        powerSendMgtFrame(pDev, FALSE);
    }
}

/******************************************************************************
 * powerTimIndication
 *
 * Inform power mgmt of TIM and DTIM values from the most current beacon.  If 
 * TIM is set, wake up if necessary, inform AP of same and stay awake until timeout.  
 * Use DTIM to stay awake until end of multi/broadcast traffic.
 *
 */
void
powerTimIndication(WLAN_DEV_INFO *pDev, A_BOOL isDirected, A_BOOL isMc)
{

    // If incoming CAB traffic stay awake until done
    pDev->powerMgmt.dtim = isMc;

    if (!isDirected) {
        return;
    }

    // Sync with AP's perspective of our sleep state
    pDev->powerMgmt.apPmState = TRUE;

    // Wake up if necessary. Cannot access hw in sleep
    if (pDev->powerMgmt.powerSleepSubstate == STATE_SLEEP ||
        pDev->powerMgmt.powerSleepSubstate == STATE_PENDING_SLEEP)
    {
        pDev->powerMgmt.powerSleepSubstate = STATE_AWAKE;
    }

    // Avoid messing up fake sleep
    if (pDev->powerMgmt.powerSleepSubstate == STATE_AWAKE) {
        // TIM indicates AP knows, or thinks, we've been asleep; tell it otherwise
        powerSendMgtFrame(pDev, FALSE);
        // There will be upcoming unicast receive activity. Make sure we don't fall
        // asleep before we get our frames from the AP.
        pDev->noActivityCur = pDev->noActivityInterval;
    }
}

void
powerCabDone(WLAN_DEV_INFO *pDev)
{
    pDev->powerMgmt.dtim = FALSE;
}

/******************************************************************************
 * powerSendMgtFrame
 *
 * Build a 802.11 null data frame with the Power Management bit set as
 * requested.  Power up the chip and send the frame.  Save the descriptor
 * pointer so we'll recognize it upon receipt of the ACK, allowing state update.
 *
 * RETURNS: Success: A_OK (we think AP already knows i.e. apPmState is as requested)
 *                   A_PENDING (notify AP, apPmState will be adjusted upon tx completion)
 *          Error:   A_NO_MEMORY or A_ECANCELED
 */
A_STATUS
powerSendMgtFrame(WLAN_DEV_INFO *pDev, A_BOOL PowerMgt)
{
    A_UINT16                frameLen;
    WLAN_DATA_MAC_HEADER3   *pFrame;    // Pointer to power mgt frame
    A_STATUS                status = A_PENDING;
    A_STATUS                pwrStatus;

    // Don't bother AP if already in the desired state but ensure chip's awake if requested
    // and if callback routine exists, call it once
    if (PowerMgt == pDev->powerMgmt.apPmState) {
        if (!PowerMgt) {

            wdcTargetSetPowerMode(pDev->targetHandle, TARGET_DEVICE_AWAKE, &pwrStatus);
            ASSERT(pwrStatus == A_OK);
        } 
        if (pDev->powerMgmt.PowerRequestCallback) {
            (*pDev->powerMgmt.PowerRequestCallback)(pDev, A_OK, powerEnabled(pDev));
            pDev->powerMgmt.PowerRequestCallback = 0;
        }
        powerPrintf("AP in correct state already\n");
        return A_OK;
    }

    if (pDev->powerMgmt.powerChangeTxDesc) {
        return A_EBUSY;
    }

    frameLen = sizeof(WLAN_DATA_MAC_HEADER3);

    if (createBuffandDesc(pDev, frameLen, &pDev->powerMgmt.powerChangeTxDesc) != 0) {
        return A_NO_MEMORY;
    }

    pFrame = pDev->powerMgmt.powerChangeTxDesc->pBufferVirtPtr.header3;
    dataFrameHdrFill(pDev, pFrame,
                     (WLAN_MACADDR *) &pDev->baseBss.bssId, SUBT_NODATA);

    pFrame->frameControl.pwrMgt = PowerMgt;

    wdcTargetSetPowerMode(pDev->targetHandle, TARGET_DEVICE_AWAKE, &pwrStatus);
    
    (pDev->powerMgmt.powerChangeTxDesc)->bIndicateXmitStatus = TRUE;
    status = staFrameTransmit(pDev, pDev->powerMgmt.powerChangeTxDesc);
    if (status != A_OK) {
        pDev->powerMgmt.powerChangeTxDesc = 0;
        status = A_ECANCELED;
    } else {
        status = A_PENDING;
    }

    return status;
}

/* dataFrameHdrFill - Fill management packet header fields
 *
 * DESCRIPTION
 * Copies information from device information, global info, etc. into
 * fields of data packet header.
 */
void
dataFrameHdrFill(WLAN_DEV_INFO          *pDev,
                 WLAN_DATA_MAC_HEADER3  *pFrame,
                 WLAN_MACADDR           *destAddr,
                 int                    subtype)
{
    /* Fill in Frame Control fields */
    A_MEM_ZERO((char *)&pFrame->frameControl, sizeof(pFrame->frameControl));
    pFrame->frameControl.fType    = FRAME_DATA;
    pFrame->frameControl.fSubtype = (A_UINT8) subtype;
    pFrame->frameControl.protoVer = PROTOCOL_VER_0;

    /* BSSID and Destination */
    if (pDev->bssDescr->bsstype == INFRASTRUCTURE_BSS) {
        /* Infrastructure mode, AP is the receiver */
        A_MACADDR_COPY(&pDev->baseBss.bssId, &pFrame->address1);
        A_MACADDR_COPY(destAddr, &pFrame->address3);
        pFrame->frameControl.ToDS = 1;
    } else {
        /* Adhoc mode, destination is receiver */
        A_MACADDR_COPY(destAddr, &pFrame->address1);
        A_MACADDR_COPY(&pDev->baseBss.bssId, &pFrame->address3);
        pFrame->frameControl.ToDS = 0;
    }

    /* Source address */
    A_MACADDR_COPY(&pDev->localSta->macAddr, &pFrame->address2);

    /* duration, seq control fields are not initialized here */

    WLAN_SET_FRAGNUM(pFrame->seqControl, 0);
}

//-----------------------------------------------------------------------------
// Procedure:   powerTxIndication
//
// Description: For each transmitted frame, take note of its' Power Management bit,
//              as it indicates most reliably what the AP believes is the sleeping
//              state of the STA.  This guides future decisions about whether it is
//              necessary to inform the AP about changes to the sleep state.
//              If the frame indicates an attempt to enter sleep, transition state
//              appropriately, and inform any requestor of the change of the result.
//              Note: assumes that the chip has been forced awake by an interrupt
//              or beacon reception, so will not actually sleep until the end of DPC.
//
// Arguments:   MiniportAdapterContext: Adapter Structure pointer
//              pktTransmitOK: TRUE if no error on transmit
//              pDesc: tx descriptor ptr for this frame
//
// Returns:     TRUE if this frame was generated internally to inform AP of PM change
//
//-----------------------------------------------------------------------------
A_BOOL
powerTxIndication(WLAN_DEV_INFO *pDev, A_BOOL pktTransmitOK, ATHEROS_DESC *pDesc)
{
    A_STATUS result = A_OK;
    A_BOOL   status = (pDesc == pDev->powerMgmt.powerChangeTxDesc);
    A_BOOL   pwrMgt = pDesc->pBufferVirtPtr.header->frameControl.pwrMgt;

    powerPrintf("powerTxIndication: Entry\n");

    if (status) {
        pDev->powerMgmt.powerChangeTxDesc = 0;
    }

    if (pDev->powerMgmt.apPmState == pwrMgt) {
        return status;
    }

    pDev->powerMgmt.apPmState = pktTransmitOK ? pwrMgt : !pwrMgt;
    result = A_OK;
    if (pwrMgt) {
        if (pDev->powerMgmt.powerSleepSubstate == STATE_PENDING_BLOCKING_SLEEP) {
            // Despite our frame's success or failure, we still want to go off
            // and scan
            pDev->powerMgmt.powerSleepSubstate = STATE_BLOCKING_SLEEP;
        } else {
            if (pktTransmitOK) {
                // Should only get here if already in D1_STATE
                if (pDev->powerMgmt.powerSleepSubstate == STATE_PENDING_SLEEP) {

                    A_STATUS pwrStatus;
                    /* Set the device to Sleep now */
                    wdcTargetSetPowerMode(pDev->targetHandle, TARGET_DEVICE_PWRSAVE, &pwrStatus);
                    if (pwrStatus == A_OK) {
                        // Set the state now
                        pDev->powerMgmt.powerSleepSubstate = STATE_SLEEP;
                    } else {
                        pDev->powerMgmt.powerSleepSubstate = STATE_AWAKE;
                    }
                    // No change for any other states, in particular STATE_AWAKE
                }
            } else {
                // Either requests to sleep or inform AP we're awake failed
                pDev->powerMgmt.powerSleepSubstate = STATE_AWAKE;
                result = A_ECOMM;
            }
        }
    } else {
        pDev->powerMgmt.powerSleepSubstate = STATE_AWAKE;
    }

    // While we're waiting for this packet to transmit, we may
    // have returned NDIS_STATUS_RESOURCES. Clear that condition.
    osIndicateResourcesAvailable(pDev);

    // If callback routine exists, call it once
    if (pDev->powerMgmt.PowerRequestCallback) {
        (*pDev->powerMgmt.PowerRequestCallback)(pDev, result, powerEnabled(pDev));
        pDev->powerMgmt.PowerRequestCallback = 0;
    }
    return status;
}


//-----------------------------------------------------------------------------
// Procedure: powerPSPollInit
//
// Description:  Create and initialize the PS Poll control frame.
//
//
// Arguments: MiniportAdapterContext (Adapter Structure pointer)
//
// Returns: A_STATUS
//
//-----------------------------------------------------------------------------
A_STATUS
powerPSPollInit(WLAN_DEV_INFO *pDev)
{

#if 1
/* xxxxPGSxxxx TODO - partition */
    return A_OK;

#else
    QUEUE_DETAILS       *pQueue = &pDev->baseBss.psPollQueue;
    HAL_TX_QUEUE_INFO   queueParam;
    ATHEROS_DESC        *pPSPollDesc;
    WLAN_FRAME_PSPOLL   *pPSPollBuffer;
    A_STATUS            status;

    // This code can get called again at reset time, so make sure we don't allocate
    // another PS-Poll descriptor; or allocate the PS-Poll queue again
    if (pQueue->pDescQueueHead == NULL) {
        // Allocate PS-Poll buffer and descriptor
        status = createBuffandDesc(pDev, sizeof(WLAN_FRAME_PSPOLL), &pQueue->pDescQueueHead);
        if (status != A_OK) {
            uiPrintf("powerPSPollInit: Not enough memory for PS-Poll ctrl frame!\n");
            return A_ERROR;
        }
    } else {
        halReleaseTxQueue(HACK_TO_PARDEV(pDev), pQueue->halTxQueueNum);
    }

    A_QUEUE_INFO_INIT(&queueParam, &pDev->baseBss.phyChAPs.ac[ACI_BE]);
    queueParam.mode       = TXQ_MODE_PSPOLL;
    queueParam.priority   = TXQ_ID_FOR_PSPOLL;
    pQueue->halTxQueueNum = halSetupTxQueue(HACK_TO_PARDEV(pDev), &queueParam);

    pPSPollDesc   = pQueue->pDescQueueHead;
    pPSPollBuffer = pPSPollDesc->pBufferVirtPtr.psPoll;

    // Initialize the PS-Poll fields
    A_MEM_ZERO((char *)pPSPollBuffer, sizeof(WLAN_FRAME_PSPOLL));

    // Fill in Frame Control fields
    pPSPollBuffer->frameCtrl.fType      = FRAME_CTRL;
    pPSPollBuffer->frameCtrl.fSubtype   = (A_UINT16) SUBT_PSPOLL;
    pPSPollBuffer->frameCtrl.FromDS     = 0;    // All control frames have FromDS
    pPSPollBuffer->frameCtrl.ToDS       = 0;    // and ToDS as 0
    pPSPollBuffer->frameCtrl.protoVer   = PROTOCOL_VER_0;   // 0 defined by standard
    pPSPollBuffer->frameCtrl.pwrMgt     = 1;    // Still in Power-Save
    pPSPollBuffer->frameCtrl.wep        = 0;    // No encryption
    pPSPollBuffer->frameCtrl.moreData   = 0;
    pPSPollBuffer->frameCtrl.moreFrag   = 0;    // no fragmentation for control frames

    // Real Assoc ID is set in powerPSPollEnable
    WLAN_SET_DURATION_AID(pPSPollBuffer->durationId, 0);

    // BSSID
    A_MACADDR_COPY(&pDev->baseBss.bssId, &pPSPollBuffer->bssId);

    // Transmitter address
    A_MACADDR_COPY(&pDev->localSta->macAddr, &pPSPollBuffer->transAddr);

    // Initialize hardware descriptor
    pPSPollDesc->hw.txControl.frameLength  = sizeof(WLAN_FRAME_PSPOLL) + FCS_FIELD_SIZE;
    pPSPollDesc->hw.txControl.bufferLength = sizeof(WLAN_FRAME_PSPOLL);
    pPSPollDesc->hw.txControl.more         = 0;

    // Initialize hardware for PS-Polls
    halSetupPSPollDesc(HACK_TO_PARDEV(pDev), pPSPollDesc);
    halSetTxDP(HACK_TO_PARDEV(pDev), pQueue->halTxQueueNum, pPSPollDesc->thisPhysPtr);

    return A_OK;
#endif

}

//-----------------------------------------------------------------------------
// Procedure: powerPSPollFree
//
// Description:
//
//
// Arguments: MiniportAdapterContext (Adapter Structure pointer)
//
// Returns: nothing
//
//-----------------------------------------------------------------------------
void
powerPSPollFree(WLAN_DEV_INFO *pDev)
{
    QUEUE_DETAILS *pQueue = &pDev->baseBss.psPollQueue;

    if (pQueue->pDescQueueHead) {
        freeBuffandDesc(pDev, pQueue->pDescQueueHead);
        pQueue->pDescQueueHead = NULL;
        halReleaseTxQueue(HACK_TO_PARDEV(pDev), pQueue->halTxQueueNum);
    }
}

//-----------------------------------------------------------------------------
// Procedure: powerHwResetNotify
//
// Description:
//
//
// Arguments: MiniportAdapterContext (Adapter Structure pointer)
//
// Returns: nothing
//
//-----------------------------------------------------------------------------
void
powerHwResetNotify(WLAN_DEV_INFO *pDev)
{
    if (!pDev->powerMgmt.needResyncToBeacon) {
        powerSetResyncToBeacon(pDev);
    }

    /*
     * forget any power change null frames we've sent! they
     * should have been freed up anyway
     */
    pDev->powerMgmt.powerChangeTxDesc = 0;

    /*
     * Setup the PS Poll descriptor again - our channel/bss
     * might have changed
     */
    powerPSPollInit(pDev);
}

//-----------------------------------------------------------------------------
// Procedure: powerSetResyncToBeacon
//
// Description: Notify Power Mgmt that we need to resynch to the next
//              beacon before going (back) to sleep.
void
powerSetResyncToBeacon(WLAN_DEV_INFO *pDev)
{
    /* Can't sleep anymore until we resync beacons */
    powerSet(pDev, WAKE_UP, SET_PWR_TO_1, FALSE, 0);

    pDev->powerMgmt.needResyncToBeacon = TRUE;
}


//-----------------------------------------------------------------------------
// Procedure: powerSleepHangCheck
//
// Description: Check if the hw has been sleeping too long! It could imply
//              that our sleep timers have gotten out of sync with the TSF;
//              force wake until the sleep timers get resynced
void
powerSleepHangCheck(WLAN_DEV_INFO *pDev)
{
    const A_UINT32 powerSleepHangCheckBmissThreshold = 5;
    A_UINT32       lastReceivedBeacon, missedBeacons;

    /*
     * nothing to worry about if the sw hasn't asked the hw to
     * go to sleep i.e. hw is not in the NETWORK_SLEEP state;
     * worry only when we're connected to an AP; nothing to do
     * if we're already aware of being out of sync
     */
    if ((pDev->powerMgmt.powerState != D1_STATE)
        || (pDev->powerMgmt.powerSleepSubstate != STATE_SLEEP)
        || (pDev->bssDescr->bsstype != INFRASTRUCTURE_BSS)
        || ((pDev->localSta->staState & STATE_CONNECTED) != STATE_CONNECTED)
        || pDev->powerMgmt.needResyncToBeacon)
    {
        return;
    }

    lastReceivedBeacon = TSF_TO_TU(pDev->localSta->lastBeaconTime);

    if (lastReceivedBeacon > pDev->powerMgmt.lastRememberedBeacon) {
        /* goodness: have seen a beacon since we were here last */
        pDev->powerMgmt.lastRememberedBeacon = lastReceivedBeacon;
        return;
    }

    /*
     * remember how many we've missed in lastRememberedBeacon
     * itself - this hack assumes that beacon period or sleep
     * duration is greater than our threshold below
     */
    ++pDev->powerMgmt.lastRememberedBeacon;
    missedBeacons = ((pDev->powerMgmt.lastRememberedBeacon - lastReceivedBeacon) *
                    CSERV_TIME_STAPOLL) / pDev->powerMgmt.sleepDuration;

    if (missedBeacons < powerSleepHangCheckBmissThreshold) {
        /* haven't crossed the threshold yet! */
        return;
    }

    /* Can't sleep anymore until we resync beacons */
    powerPrintf("powerSleepHangCheck: we've been sleeping too long! wake to resync timers!");
    powerSetResyncToBeacon(pDev);
}

//-----------------------------------------------------------------------------
// Procedure: drvPowerSet
//
// Description: If the requested state is different from the current state,
//              transitions the hardware accordingly and updates the state.
//
//              Special cases: entering and resuming from standby/hibernation(D3).
//              Entering shuts down the h/w and s/w and exits immediately.
//              Resuming restarts h/w and s/w thereby entering D0 temporarily,
//              but then will move to the state requested.  Subsequently it checks
//              for a change in the rf kill switch (if any) and calls routines
//              to update accordingly, which may call powerSet() and call this
//              routine again - but the recursion will end there since the old
//              state is no longer D3.
//
//
// Arguments: MiniportAdapterContext (Adapter Structure pointer)
//            DxRequest - Device power level requested:
//              D0 - Full awake, full power
//              D1 - Atheros network sleep (i.e. normal)
//              D2 - Atheros full sleep (i.e. forced)
//              D3 - Full power off (i.e. prepare for such) for standby/hibernation
//            D1PwrUp - TRUE if power is desired even though in network sleep mode
//
// Returns: A_OK normally
//          A_HARDWARE if hardware didn't respond
//
//-----------------------------------------------------------------------------
A_STATUS
drvPowerSet(WLAN_DEV_INFO *pDev, PWR_MGMT_DEVICE_POWER_TYPE DxRequest, A_BOOL D1PwrUp)
{
    A_UINT32    newState = pDev->powerMgmt.powerState;
    A_UINT32    oldState = pDev->powerMgmt.powerState;
    A_STATUS    status = A_OK;

    if (DxRequest == oldState) {
        return status;
    }

    powerPrintf("drvPowerSet: Current power = %d, requested power = %d, D1PwrUp = %d\n",
                oldState, DxRequest, D1PwrUp);

    // Entering hibernate/standby
    // Shut down the hardware and exit
    if (DxRequest == D3_STATE) {

        //
        // XXX: THIS CODE SHOULD BE COMMON WITH THE DRIVER SHUTDOWN and RF_DISABLE CODE
        //
        osIndicateConnectStatus(pDev, FALSE);

        ATH_RELEASE_SPINLOCK(pDev->lock);
        athShutdownDriver(pDev, HIBERNATE_EVENT);
        ATH_ACQUIRE_SPINLOCK(pDev->lock);

        pDev->powerMgmt.powerState = D3_STATE;

        return status;
    }

    // Resuming from hibernate/standby - restore chip and s/w state
    // with the expectation of entering D0 (at least temporarily)
    if (oldState == D3_STATE) {
        //
        // XXX: THIS CODE SHOULD BE COMMON WITH THE DRIVER STARTUP and RF_ENABLE CODE
        //
        // Unreset chips and restore state                

        pDev->NicResetInProgress = FALSE;

        // Write vendor-private area of PCI config space
        //WritePCISlotInfo(pDev, pDev->pOSHandle->PciSlotNumber);        

        wdcTargetSetPowerMode(pDev->targetHandle, TARGET_DEVICE_AWAKE, &status);
        ASSERT_PWR_MGMT(status == A_OK);
        
        NdisInterlockedIncrement(&pDev->disableEnableCounter);
        pDev->flushInProgress   =   FALSE;
        A_ATOMIC_SET (&pDev->numFlushCall, 0);

        wdcTargetEnable(pDev->targetHandle,NULL);

        startNdisTimers(pDev);

        StartReceiveUnit(pDev, (RX_UCAST  | RX_MCAST | RX_BCAST |
                            RX_BEACON | RX_PROBE_REQ));

        newState = D0_STATE;

        status = wlanDevReset(pDev, WLAN_STA_SERVICE, pDev->staConfig.pChannel, FALSE, pDev->turboPrimeInfo.turboPrimeAllowed);
        ASSERT_PWR_MGMT(status == A_OK);

        status = securityInitialization(pDev);
    
        // restore h/w encrypt engine setting
        setupSwEncryption(pDev);

        halEnableInterrupts(HACK_TO_PARDEV(pDev), HAL_INT_GLOBAL);

        RestartTransmit(pDev);
        StartReceiveUnit(pDev, 0);

        // clear the resources condition caused by power save or bkgnd scanning
        osIndicateResourcesAvailable(pDev);

#if !NDIS_WDM
        initGpioFunc0(pDev);
        initGpioFunc1(pDev);
#endif
    }

    // At this point we're either in D0, D1 or D2 states,
    // and will transition to a different one of the same group.

    switch (DxRequest) {
        case D2_STATE:
            wdcTargetSetPowerMode(pDev->targetHandle, TARGET_DEVICE_SLEEP, &status);

            if (status == A_OK ) {
                newState = D2_STATE;
            }
            break;

        case D1_STATE:
            wdcTargetSetPowerMode(pDev->targetHandle, D1PwrUp ? TARGET_DEVICE_AWAKE : TARGET_DEVICE_PWRSAVE, &status);

            if (status == A_OK ) {
                newState = D1_STATE;
            }
            break;

        case D0_STATE:
            wdcTargetSetPowerMode(pDev->targetHandle, TARGET_DEVICE_AWAKE, &status);

            if (status == A_OK ) {
                newState = D0_STATE;
            }
            break;

    default:
        break;
    }

    pDev->powerMgmt.powerState = newState;

    if (oldState == D3_STATE) {
        RF_SILENT_INFO *pInfo = &pDev->rfSilent;

        // Set radio according to current setting of GPIO switch (may
        // have changed while NIC was powered down)
        if (pInfo->eepEnabled) {
            if (pInfo->polarity == halGpioGet(pDev, pInfo->gpioSelect)) {
                // switch closed, turn off rf
                powerPrintf("RF disable switch closed\n");

                ATH_RELEASE_SPINLOCK(pDev->lock);
                athDrvRadioDisable(pDev, &pInfo->hwRadioDisable);
                ATH_ACQUIRE_SPINLOCK(pDev->lock);
            } else {
                // switch opened, activate now.
                powerPrintf("RF disable switch opened\n");

                ATH_RELEASE_SPINLOCK(pDev->lock);
                athDrvRadioEnable(pDev, &pInfo->hwRadioDisable);
                ATH_ACQUIRE_SPINLOCK(pDev->lock);
            }
        }
        // Set the LED if radio's enabled
        gpioFunc0(pDev, athDrvRadioStatus(pDev, FALSE));
        gpioFunc1(pDev, athDrvRadioStatus(pDev, FALSE));
    }

    return status;
}

