/*************************************************************************
 * $Id: //depot/sw/branches/1.3_USB_LINUX_port/src/USB/wlan/host/sta/receive.c#4 $
 *
 * Receive logic for the Atheros NDIS driver.
 *
 * Copyright © 2000-2003 Atheros Communications, Inc., All Rights Reserved
 *
 * Atheros and the Atheros logo and design are trademarks of Atheros
 * Communications, Inc.
 *
 * Sample Code from Microsoft Windows 2000 Driver Development Kit is
 * used under license from Microsoft Corporation and was developed for
 * Microsoft by Intel Corp., Hillsboro, Oregon: Copyright (c) 1994-1997
 * by Intel Corporation.
 */

#include "wlantype.h"
#include "wlandrv.h"
#include "wlanext.h"
#include "wlanCapDe.h"
#include "wlanfragde.h"
#include "wlanframe.h"
#include "wlansmeext.h"
#include "wlanDev.h"
#include "wlanReceive.h"
#include "stacserv.h"
#include "ccx.h"
#include "display.h"
#include "wdcApi.h"
#include "usbstub.h"
/*
 * Storage
 */

static const SNAP_HANDLER snapHandler[] = { AIRONET_HANDLER };
static const int numSnapHandlers = sizeof(snapHandler) / sizeof(snapHandler[0]);

#ifdef DEBUG
int rxDebugLevel;
#endif

/*
 * Definitions and Macros
 */

#define DESCS_TO_KEEP       32
#define ETHER_TYPE_OFFSET   (sizeof(WLAN_DATA_MAC_HEADER3) + CAP_TYPE_SIZE)

/*
 * Forward declarations
 */

static A_BOOL
isMcForUs(WLAN_DEV_INFO *pDev, WLAN_MACADDR *pAddr);

static INLINE A_STATUS
rxAllocateSib(WLAN_DEV_INFO *pDevInfo, WLAN_MACADDR *pSrcStaMacAddr,
              SIB_ENTRY **ppSibEntry);


/*
 * Implementation
 */

/*****************************************************************************
 * rxNullHandler
 *
 * A generic frame "handler" for SNAP types that we don't care to process.
 *
 */
A_STATUS
rxNullHandler(WLAN_DEV_INFO *pDev, ATHEROS_DESC *pDesc, A_UINT16 etherType)
{
    return A_ERROR;
}

/*****************************************************************************
 * staFrameReceive
 *
 * Called to perform any processing on DIX/802.3 frames before indicating them
 * up the stack.
 */
void
staFrameReceive(WLAN_DEV_INFO *pDevInfo, ATHEROS_DESC *pRxDesc)
{
    LAN_FRAME_HEADER *pLanHeader = pRxDesc->pBufferVirtPtr.lanHeader;
    WLAN_MACADDR     *pDestAddr  = &pLanHeader->destAddr;
    A_UINT32         frameLen    = pRxDesc->frameLength;
    WLAN_STATS       *pStats     = &pDevInfo->localSta->stats;
    WLAN_STA_CONFIG  *pConfig    = &pDevInfo->staConfig;
    PNDIS_PACKET     pPacket;
    NDIS_STATUS      status;

    rxPrintf("> %s\n", __FUNCTION__);

    /*
     * Drop frames with no data payload. Some protocol drivers don't
     * like it when we pass these up.
     */
    if (frameLen == ETHER_HDR_SIZE) {
        freeBuffandDescChain(pDevInfo, pRxDesc);
        rxPrintf("< %s: dropping zero-payload frame\n", __FUNCTION__);
        return;
    }

    /* Record statistics for this packet. */
    pStats->GoodReceiveBytes += frameLen;
    if (isGrp(pDestAddr)) {
        if (isBcast(pDestAddr)) {
            if (!(pDevInfo->NdisFilterReg & NDIS_PACKET_TYPE_BROADCAST)) {
                freeBuffandDescChain(pDevInfo, pRxDesc);
                rxPrintf("< %s: dropping filtered broadcast\n", __FUNCTION__);
                return;
            }
            pStats->RxBroadcastFrames++;
        } else {
            if (!(pDevInfo->NdisFilterReg & (NDIS_PACKET_TYPE_MULTICAST |
                  NDIS_PACKET_TYPE_ALL_MULTICAST)) ||
                 (!isMcForUs(pDevInfo, pDestAddr)))
            {
                freeBuffandDescChain(pDevInfo, pRxDesc);
                rxPrintf("< %s: dropping filtered or "
                         "others' multicast\n", __FUNCTION__);
                return;
            }
            pStats->MulticastReceives++;
        }
    } else {
        pStats->RxUnicastFrames++;
    }

#if defined(WLAN_CONFIG_LEAP) && defined(LEAP_IN_DRIVER)
    if (pConfig->leapEnabled && pConfig->leapInDriver &&
        be2cpu16(pLanHeader->lanTypeOrLen) == ETHERTYPE_EAPOL)
    {
        /* leapReceive will free the buffer and desc */
        leapReceive(pDevInfo, pRxDesc);

        rxPrintf("< %s: LEAP frame not indicated to NDIS\n", __FUNCTION__);
        return;
    }
#endif  /* WLAN_CONFIG_LEAP && LEAP_IN_DRIVER */

    /* Perform Atheros descriptor to NDIS packet conversion. */
    pPacket = (PNDIS_PACKET)athDescToOsPacket(pDevInfo, pRxDesc);
    ASSERT(pPacket);

#ifdef Linux
    netif_rx(pPacket);
    NdisInterlockedIncrement(&pDevInfo->totalRxIndicated);
#endif /* Linux */
    rxPrintf("< %s: NDIS packet indicated and %s\n", __FUNCTION__,
             status == NDIS_STATUS_PENDING ? "kept" : "returned");
}

void
hostRxCallback(
    IN  WLAN_DEV_INFO       *pDevInfo,
    IN  ATHEROS_DESC        *pDesc,
    IN  HOST_CONN_HANDLE    hostConnHandle,
    IN  RECEIVE_INFO        *pInfo)
{
    WLAN_DATA_MAC_HEADER3  *pHeader;
    WLAN_MACADDR           *pSrcAddr, *pDestAddr, *pBssid;
    SIB_ENTRY              *pSibEntry = NULL;
    SIB_ENTRY              *pLocalSta;
    WLAN_STATS             *pStats;
    A_BOOL                 isIbss;
    RX_STATUS_ENUM         status;
    A_BOOL                 isFromOurBss = FALSE;
    WLAN_PRIV_RECORD       *pKey = NULL;

    if (!pDevInfo->pOSHandle->openForBusiness) {
        if (pDesc) {
            freeBuffandDescChain(pDevInfo, pDesc);
        }
        return;
    }

    ASSERT(pDevInfo);
    ASSERT(pDesc);
    ASSERT(pInfo);

    rx2Printf("> %s\n", __FUNCTION__);

#if TBD
    ASSERT(pDevInfo->targetRxPending > 0);
#endif
    pDevInfo->targetRxPending--;

    if (pInfo->status == RX_STATUS_STOP_IN_PROGRESS) {
        freeBuffandDescChain(pDevInfo, pDesc);
        rxPrintf("< %s: Stop in progess\n", __FUNCTION__);
        return;
    }

    /* Fill in host descriptor rx status with receive info */
    pDesc->status.rx.dataLength     = (A_UINT16) pInfo->frameLength;
    pDesc->status.rx.timestamp.low  = pInfo->timestamp.low;
    pDesc->status.rx.timestamp.high = pInfo->timestamp.high;
    pDesc->status.rx.decryptError   = (A_UINT8)pInfo->decryptError;
    pDesc->status.rx.phyError       = (A_UINT8) pInfo->phyError;
    pDesc->status.rx.rate           = (A_UINT8) pInfo->rate;
    pDesc->status.rx.hwIndex        = (A_UINT8)pInfo->connIndex;
    pDesc->status.rx.rssi           = (A_INT8) pInfo->rssi;
    pDesc->status.rx.antenna        = (A_UINT8) pInfo->antenna;

    /* XXX Divy. Have to act on key cache miss */
    pDesc->status.rx.keyCacheMiss   = (A_UINT8) pInfo->keyCacheMiss;

    /* xxxxPGSxxxx - TODO  pInfo->channel not handled */

    /* xxxxPGSxxxx - HACK - make sure this is valid - fix when SIB stuff is done
     * Look at ar5212ProcessRxDesc() this function sets up pOpBss. This may need
     * to be brought into the host side (from target).
     */
    pDesc->pOpBss = &pDevInfo->baseBss;

    /* Initialize the pointed key */
    pDesc->pKey = NULL;

    pLocalSta = pDevInfo->localSta;
    pStats    = &pLocalSta->stats;
    isIbss    = (pDevInfo->bssDescr->bsstype == INDEPENDENT_BSS);

    if (pDevInfo->dropAllRxFrames || !wlanDevPresent(pDevInfo)) {
        /*
         * In order to change channels or halt the driver more
         * gracefully, we drop all received frames in software while
         * still allowing the hardware to receive.  This is to
         * circumvent the problem whereby disabling receive no longer
         * allows us to receive ACKs for transmitted frames,
         * causing all transmits to fail.
         */
        freeBuffandDescChain(pDevInfo, pDesc);
        rxPrintf("< %s: dropAllRxFrames == TRUE\n", __FUNCTION__);
        return;
    }

    status = pInfo->status;

    switch(status) {
    case RX_STATUS_CRC_ERR:
        pStats->RcvCrcErrors++;
        break;
    case RX_STATUS_PHY_ERR:
        pStats->RcvPhyErrors++;
        break;
    case RX_STATUS_DECRYPT_CRC_ERR:
        pStats->RcvDecryptCrcErrors++;
        break;
    case RX_STATUS_DECRYPT_MIC_ERR:
        pStats->RcvPhyErrors++;
        break;
    case RX_STATUS_DECOMP_ERR:
        pStats->RcvDecompCrcErrors++;
        break;
    case RX_STATUS_KEY_ERR:
        break;
    }

    if (status != RX_STATUS_OK && status != RX_STATUS_DECRYPT_MIC_ERR) {
        if (status == RX_STATUS_PHY_ERR &&
            pDevInfo->NicResetInProgress != TRUE)
        {
            PHY_ERR_REPORT(pDevInfo, pInfo, pDesc);
        }

        pStats->ReceiveErrors++;

        /*
         * Don't bother processing bad frames.
         * Decrypt error frames are passed up for use on the AP, but
         * the STA has no use for these frames.
         */
        freeBuffandDescChain(pDevInfo, pDesc);
        rxPrintf("< %s: dropping erroneous frame %x\n", __FUNCTION__, status);
        return;
    }

    pHeader = pDesc->pBufferVirtPtr.header3;
    pBssid  = pHeader->frameControl.FromDS ?
              &pHeader->address2 : &pHeader->address3;

    if (A_MACADDR_COMP(pBssid, &pDevInfo->baseBss.bssId) == 0) {
        isFromOurBss = TRUE;

        /*
         * HW will always receive all Probe Requests despite RX filter
         * settings to avoid them. Discard these directed (to the AP)
         * Probe Requests from other STAs. These do not indicate that
         * the AP is still alive.
         */
        if (!isIbss && pHeader->frameControl.fType == FRAME_MGT &&
            pHeader->frameControl.fSubtype == SUBT_PROBE_REQ)
        {
            rxPrintf("< %s: dropping non-IBSS Probe Request\n", __FUNCTION__);
            freeBuffandDescChain(pDevInfo, pDesc);
            return;
        }

        /* We received a frame from our BSS. It's alive!! */
#if NDIS_WDM
        /*
        * (LW) Attention !!!
        * This is temp hack, should revisit later
        */
        pDevInfo->noApActivity = FALSE;
#else
        HACK_TO_PARDEV(pDevInfo)->noApActivity = FALSE;
#endif
    }

    if (pDevInfo->NicResetInProgress) {
        /* We are in the middle of a reset so drop the frame. */
        rxPrintf("< %s: Reset in progress\n", __FUNCTION__);
        freeBuffandDescChain(pDevInfo, pDesc);
        return;
    }

    /* Filter frames that NDIS/protocols don't want */
    pDestAddr = &pHeader->address1;

    if ((pHeader->frameControl.fType == FRAME_DATA) && (!isFromOurBss)) {
        rxPrintf("Discarding Data frame not from our BSS\n");
        freeBuffandDescChain(pDevInfo, pDesc);
        return;
    }

    if (pHeader->frameControl.fType == FRAME_DATA && isGrp(pDestAddr) &&
        !pHeader->frameControl.moreData)
    {
        powerCabDone(pDevInfo);
    }

    /*
     * Filter out MC/BC packets that we originally sent to the AP and
     * have received after it resent them.
     * Filter out packets that use our own mac address as TA too
     */
    if ((pHeader->frameControl.FromDS &&
        A_MACADDR_COMP(&pHeader->address3, &pLocalSta->macAddr) == 0) ||
        A_MACADDR_COMP(&pHeader->address2, &pLocalSta->macAddr) == 0)
    {
        /* We are the original sender. Toss it out */
        freeBuffandDescChain(pDevInfo, pDesc);
        rx2Printf("< %s: dropping our own BC/MC frame\n", __FUNCTION__);
        return;
    }

    /*
     * Process the frame based on frame type. We may receive these
     * frames if we're in promiscuous mode.
     */
    switch (pHeader->frameControl.fType) {
    case FRAME_CTRL:
    case FRAME_RESERVED:
        freeBuffandDescChain(pDevInfo, pDesc);
        rxPrintf("< %s: dropping our own BC/MC frame\n", __FUNCTION__);
        return;
    case FRAME_MGT:
        /* LW
         * Give up managmenet frame in the middle of channel change
         */
        if (pDevInfo->rcvChangeInProgress == TRUE) {
            freeBuffandDescChain(pDevInfo, pDesc);
            return;
        }

        if (pHeader->frameControl.fSubtype == SUBT_PROBE_REQ) {
            /*
             * Original comment: See bug 1516.
             * WTH does it have to do with bug 1516?
             */
            if (isIbss) {
                pDesc->status.rx.dataLength -= FCS_FIELD_SIZE;
                mlmeMgtRxCompletion(pDevInfo, pDesc);
                return;
            } else {
                freeBuffandDescChain(pDevInfo, pDesc);
            }

            rxPrintf("< %s: dropping Probe Request\n", __FUNCTION__);
            return;
        }
        break;
    default:
        break;
    }

    A_CORRUPTION_BUG_CHECK(((DRV_BUF_DESC *)pDesc->pOrigBufferVirtPtr) - 1);

    /* Discard frames which contain no data. */
    if (pHeader->frameControl.fType == FRAME_DATA) {
        switch (pHeader->frameControl.fSubtype) {
        case SUBT_DATA:
#ifdef WME
        case SUBT_QOS:
#endif
        case SUBT_DATA_CFACK:
        case SUBT_DATA_CFPOLL:
        case SUBT_DATA_CFACK_CFPOLL:
            break;
        default:
            rxPrintf("Discarding Data frame with no data\n");
            freeBuffandDescChain(pDevInfo, pDesc);
            return;
        }
    }

    /* Get the sib entry for the source STA. */
    pSrcAddr = &pHeader->address2;

    /*
     * Validate the source address before allocating
     * the SIB entry.
     */
    if (isGrp(pSrcAddr) || (A_MACADDR_COMP(pSrcAddr, &nullMacAddr) == 0)) {
        rxPrintf("< %s: invalid src address (", __FUNCTION__);
        rxPrintMacAddress(*pSrcAddr);
        rxPrintf(")\n");
        freeBuffandDescChain(pDevInfo, pDesc);
        return;
    }

    if ((pDevInfo->bssDescr->bsstype == INFRASTRUCTURE_BSS) || 
        ((pDevInfo->bssDescr->bsstype == INDEPENDENT_BSS) &&
         ((pDevInfo->localSta->staState & STATE_JOINED) == STATE_JOINED) &&
         (pHeader->frameControl.fType == FRAME_DATA) &&
         (pHeader->frameControl.fSubtype == SUBT_DATA)))
    {
        status = rxAllocateSib(pDevInfo, pSrcAddr, &pSibEntry);
        if (A_OK != status) {
            freeBuffandDescChain(pDevInfo, pDesc);
            return;
        }
    }

    pDesc->pSrcSibEntry = pSibEntry;

    /*
     * Ignore duplicate frames only if unicast. BC/MC frames are not
     * retried.
     */
    if (pSibEntry && !isGrp(&pHeader->address1)) {
        /*
         * If it's a retry/duplicate of the last one received,
         * count it and then drop it.
         */
        if (pHeader->frameControl.retry) {
#ifdef WME
            A_UINT16 acc = 0;

            /* need to parse hdr's QoS_Cntl field to get queue number */
            A_RX_DESC_CHAIN_TO_TX(pDesc);
            if ((pHeader->frameControl.fType == FRAME_DATA) &&
                (pHeader->frameControl.fSubtype == SUBT_QOS))
            {
                A_UINT16 qosCntl;

                bufferPopHdr(pDesc, sizeof(WLAN_DATA_MAC_HEADER3));

                if (pHeader->frameControl.FromDS && pHeader->frameControl.ToDS) {
                    void    *tmp = bufferGetHdr(pDesc, sizeof(WLAN_MACADDR) + sizeof(QOS_CNTL));
                    A_UINT8 *tmpQosCntl = (A_UINT8 *)((WLAN_MACADDR *)tmp + 1);

                    qosCntl = (A_UINT16) ((*tmpQosCntl) + (*(tmpQosCntl + 1) << 8));
                    rxPrintf ("%s: WDS!!!!!   qosCntl = %d\n", __FUNCTION__, qosCntl);
                } else {
                    A_UINT8 *tmp = (A_UINT8 *) bufferGetHdr(pDesc, sizeof(QOS_CNTL));

                    qosCntl = (A_UINT16) ((*tmp) + (*(tmp+1) << 8));
                    rxPrintf ("qosCntl = %d\n",qosCntl);
                }

                acc = ((QOS_CNTL *)(&qosCntl))->tid;
                bufferPushHdr(pDesc, sizeof(WLAN_DATA_MAC_HEADER3));
            } else {
                acc = 0;   // default place to store seq numbers if QoS is not used
            }

            rxPrintf ("%s: acc = %d\n", __FUNCTION__, acc);

            if (pHeader->seqControl.data == pSibEntry->lastRxSeqControl[acc].data) {
                pStats->RcvDupSeqErrors++;

                if (pSibEntry->lastRxSeqControl[acc].data ==
                    pSibEntry->lastLastRxSeqControl[acc].data)
                {
                    pSibEntry->stats.MultipleRxDuplicates++;
                }

                pSibEntry->lastLastRxSeqControl[acc] = pSibEntry->lastRxSeqControl[acc];
                freeBuffandDescChain(pDevInfo, pDesc);
                rxPrintf("< %s: discarding duplicate frame\n", __FUNCTION__);
                return;
            }
        }

        pSibEntry->lastRxSeqControl[acc] = pHeader->seqControl;
    }
#else
            if (pHeader->seqControl.data == pSibEntry->lastRxSeqControl.data) {
                pStats->RcvDupSeqErrors++;

                if (pSibEntry->lastRxSeqControl.data ==
                    pSibEntry->lastLastRxSeqControl.data)
                {
                    pSibEntry->stats.MultipleRxDuplicates++;
                }

                pSibEntry->lastLastRxSeqControl = pSibEntry->lastRxSeqControl;
                freeBuffandDescChain(pDevInfo, pDesc);
                rxPrintf("< %s: discarding duplicate frame\n", __FUNCTION__);
                return;
            }
        }

        pSibEntry->lastRxSeqControl = pHeader->seqControl;
    }
#endif

    A_CORRUPTION_BUG_CHECK(((DRV_BUF_DESC *)pDesc->pOrigBufferVirtPtr) - 1);

    if (pSibEntry && pDevInfo->staConfig.privacyInvoked) {
        /* Check decrypt policy */
        status = rxKeyLookup(pDevInfo, pDesc);
        if (status == A_ERROR) {
            pStats->ReceiveErrors++;
            pSibEntry->stats.ReceiveErrors++;
            freeBuffandDescChain(pDevInfo, pDesc);
            rxPrintf("< %s: rxKeyLookup() failed\n", __FUNCTION__);
            return;
        }

        if (status == A_OK) {
            if (setupDecrypt(pDevInfo, pDesc) != A_OK) {
                freeBuffandDescChain(pDevInfo, pDesc);
                pStats->RcvDecipherErrors++;
                pSibEntry->stats.RcvDecipherErrors++;
                pStats->ReceiveErrors++;
                pSibEntry->stats.ReceiveErrors++;
                rxPrintf("< %s: setupDecrypt() failed\n", __FUNCTION__);
                return;
            }

            /*
             * setupDecrypt changes buffers on us, so we reinitialize
             * our header pointer.
             */
            pHeader = pDesc->pBufferVirtPtr.header3;
        }
    } else if (pDesc->pBufferVirtPtr.header->frameControl.wep) {
        /* Drop encr frames if no privacy */
        pStats->RcvWEPExcludedCount++;
        pStats->ReceiveErrors++;

        if (pSibEntry) {
            pSibEntry->stats.RcvWEPExcludedCount++;
            pSibEntry->stats.ReceiveErrors++;
        }

        freeBuffandDesc(pDevInfo, pDesc);
        rxPrintf("< %s: privacy disabled, "
                 "dropping encrypted frame\n", __FUNCTION__);
        return;
    }

#if DEBUG
        if ((pHeader->frameControl.fType == FRAME_DATA) &&
            (pHeader->frameControl.fSubtype == SUBT_DATA)) {

            ipDecode(pDevInfo, pDesc, TRUE);
        }
#endif

    /* Subtract off FCS length */
    pDesc->status.rx.dataLength -= FCS_FIELD_SIZE;

    rx2Printf("%s: Seq No.: %4d, Frag No.: %d\n", __FUNCTION__,
             WLAN_GET_SEQNUM(pHeader->seqControl),
             WLAN_GET_FRAGNUM(pHeader->seqControl));

    A_CORRUPTION_BUG_CHECK(((DRV_BUF_DESC *)pDesc->pOrigBufferVirtPtr) - 1);

    /* Defragment the frame, if required. */
    pDesc = wlanDeFrag(pDevInfo, pDesc);
    if (pDesc == NULL) {
        /*
         * The fragment was consumed into an in-process frame
         * reconstruction or dropped if bad.
         */
        return;
    }

    A_CORRUPTION_BUG_CHECK(((DRV_BUF_DESC *)pDesc->pOrigBufferVirtPtr) - 1);

    /* wlanDeFrag() changes buffers, so reinitialize our pointer. */
    pHeader = pDesc->pBufferVirtPtr.header3;

    if (pHeader->frameControl.fType == FRAME_MGT) {
        if (isFromOurBss &&
            pHeader->frameControl.fSubtype == SUBT_BEACON)
        {
            /*
             * The connection's RSSI is based off of RSSI from beacons.
             * Beacons are always sent at the same rate and are always
             * received, and so should be a good indication of RSSI.
             */
            pStats->rssi = A_RSSI_LPF(pStats->rssi, pInfo->rssi);

            /* Update global MIB items, again, this time read beacon count too. */
            halMibControl(pDevInfo, UPDATE_SW_ALL);
        }

        /* mlmeMgtRxCompletion() is responsible for freeing the buf and desc. */
        mlmeMgtRxCompletion(pDevInfo, pDesc);
        rx2Printf("< %s: management frames not indicated to NDIS\n", __FUNCTION__);
        return;
    }

    /*
     * Notify Power Mgmt of unicast/multicast rx activity. If we've
     * been in a power-down state, power up. If we got this frame
     * because of a PS-POLL, send a Null Data frame to tell the AP
     * we are now awake.
     * Don't wake up for broadcast frames.
     */
    if (!isBcast(&pHeader->address1)) {
        powerReceiveIndication(pDevInfo);
    }

    /*
     * See corresponding comment in send.c to understand why the
     * authentication-based frame class concept is currently not
     * supported for the IBSS case.
     */
    if ((pDevInfo->bssDescr->bsstype == ANY_BSS) ||
        ((pDevInfo->bssDescr->bsstype == INDEPENDENT_BSS) &&
         ((!(pDevInfo->localSta->staState & STATE_JOINED)) ||
          (A_MACADDR_COMP(&pHeader->address3,
                          &pDevInfo->baseBss.bssId) != 0))))
    {
        freeBuffandDescChain(pDevInfo, pDesc);
        rxPrintf("< %s: dropping frame -- not our BSSID\n", __FUNCTION__);
        return;
    }

    if (pDevInfo->bssDescr->bsstype == INFRASTRUCTURE_BSS &&
        pHeader->frameControl.FromDS &&
        (pDevInfo->localSta->staState & (STATE_AUTH | STATE_ASSOC)) !=
        (STATE_AUTH | STATE_ASSOC))
    {
        /* We are not in the right state, cannot receive packets */
        if (!isGrp(&pHeader->address1)) {
            /* received class 3 frame when not in state 3, send deauth */
            wlanMlmeDeauthRequest(pDevInfo, &pDevInfo->baseBss,
                              &pHeader->address2, REASON_CLASS3, TRUE);
        }

        freeBuffandDescChain(pDevInfo, pDesc);
        rxPrintf("< %s: rx'd class 3 frame, not in that state\n", __FUNCTION__);
        return;
    }

    /* Record our receive rate only if it's not a BC/MC frame. */
    if (!isGrp(pDestAddr)) {
        pStats->RcvRate =
            pDevInfo->baseBss.pRateTable->info[pInfo->rate].rateKbps;
    }

    if (pDevInfo->bssDescr->bsstype == INDEPENDENT_BSS) {
        cservAdHocTrafficReceived(pDevInfo);
    }

    wlanDataFrameReceive(pDevInfo, pDesc);

    if (!pDevInfo->dropAllRxFrames) {
        PHY_ERR_POLL_FUNCTION(pDevInfo);
    }

    /* Keep track of activity to stave off background scan if we're busy */
    cservUpdateLastActivityTime(pDevInfo);

    rx2Printf("< %s\n", __FUNCTION__);
}

static INLINE A_STATUS
rxAllocateSib(WLAN_DEV_INFO *pDevInfo, WLAN_MACADDR *pSrcStaMacAddr,
              SIB_ENTRY **ppSibEntry)
{
    SIB_ENTRY       *pSibEntry = *ppSibEntry = NULL;
    CONN_ATTRIBUTES connAttribs;


    if (pDevInfo->bssDescr->bsstype == INFRASTRUCTURE_BSS &&
     0 != A_MACADDR_COMP(&pDevInfo->bssDescr->bssId, pSrcStaMacAddr))
    {
     return A_OK;
    }

    if ((pDevInfo->bssDescr->bsstype == INDEPENDENT_BSS) &&
         !(pDevInfo->localSta->staState & STATE_JOINED) )
    {
        return A_OK;
    }

    /* Search the SIB table for the frame's sender. */
    pSibEntry = sibEntryFind(pDevInfo, pSrcStaMacAddr, NULL);
    if ((pSibEntry == NULL) && (pDevInfo->bssDescr->bsstype == INDEPENDENT_BSS)) {

        /* Allocate an entry, do permission check, and add the entry */
        pSibEntry = sibEntryAlloc(pDevInfo, pSrcStaMacAddr,
                                  &pDevInfo->baseBss);
        if (pSibEntry == NULL) {
            /*
             * We just drop the frame at this level.  SIB
             * overflow will be handled by sibEntryAlloc(),
             * unless things are really bad.
             */
            return A_ERROR;

        }

        sibEntryAdd(pSibEntry);

        /* New connection => Create target side connection */
        A_MEM_ZERO(&connAttribs, sizeof(connAttribs));
        connAttribs.keytype             = PRIV_KEY_TYPE_NULL;
        connAttribs.compression         = 0;                    /* TODO */
        connAttribs.longPreambleOnly    = 0;                    /* TODO */
        connAttribs.currentTxRateAnt    = 0;                    /* TODO */
        connAttribs.serviceType         = pSibEntry->serviceType;
        wlanRateSet2WdcRateSet(&pSibEntry->workingRateSet  , &connAttribs.workingRateSet);
        connAttribs.wlanMode            = pSibEntry->wlanMode;

        wdcCreateConnection(pDevInfo->targetHandle,
                            &pSibEntry->macAddr,
                            pSibEntry->wdcConnId,
                            0,
                           &connAttribs);
        NdisInterlockedIncrement(&pDevInfo->numFlushCall);
        wdcFlush(pDevInfo->targetHandle, pDevInfo->disableEnableCounter);

        /*
         * So far only WEP is supported for ad hoc.
         * Install the default WEP key if necessary.
         */
        if (pDevInfo->staConfig.privacyInvoked) {
            WLAN_PRIV_RECORD    *pKey;
            WLAN_MACADDR        nullMacAddr;

            A_MEM_ZERO(&nullMacAddr, sizeof(nullMacAddr));

            pKey = &pDevInfo->keyTable[pDevInfo->staConfig.defaultKey];
            pSibEntry->pPriv = pKey;
            wlanInstallConnectionKey(pDevInfo,
                                     pSibEntry,
                                     &nullMacAddr,
                                     pKey);
        }
    }

    *ppSibEntry = pSibEntry;

    return A_OK;
}


/*****************************************************************************
 * SetupReceiveQueues
 *
 * Description: Setup the Receive Frame Area (RFA) at INIT time.   This
 *              includes setting up each RFD in the RFA, and linking all of the
 *              RFDs together.
 *              Also set up our buffers for NDIS 5 and multiple receive indications
 *              through a packet array.
 *
 *              Note: NDIS buffer desc allocs are done after defrag
 *
 * Arguments:
 *   pDevInfo   ptr to DevInfo object instance
 *
 * Returns:     NDIS_STATUS_RESOURCES    if any allocation request failed
 *              NDIS_STATUS_SUCCESS      otherwise
 */
NDIS_STATUS
SetupReceiveQueues(IN WLAN_DEV_INFO *pDev)
{
    OS_DEV_INFO  *posInfo = pDev->pOSHandle;
    ATHEROS_DESC *pDesc;
    NDIS_STATUS  status;
    int          i;

    ASSERT(posInfo);

    rxPrintf("> %s\n", __FUNCTION__);

    /*
     * Set up a pool of data for us to build our packet array out of
     * for indicating groups of packets to NDIS.
     * This could be quite the memory hog, but makes management
     * of the pointers associated with asynchronous memory allocation
     * easier.
     */
    NdisAllocatePacketPool(&status, &posInfo->rxPacketPool, 2 * MAX_RX_PACKETS,
                           PROTOCOL_RESERVED_SIZE_IN_PACKET);
    if (status != NDIS_STATUS_SUCCESS) {
        rxPrintf("< %s: NdisAllocatePacketPool() failed!\n", __FUNCTION__);
        return status;
    }

    /*
     * Set up our pool of buffer descriptors...
     * We will have at most 1 per packet, so we allocate as
     * many buffers as we have packets.
     * NdisAllocateBufferPool() supposedly always returns success...
     */
    NdisAllocateBufferPool(&status, &posInfo->rxBufferPool, 2 * MAX_RX_PACKETS);
    if (status != NDIS_STATUS_SUCCESS) {
        return status;
    }

    /*
     * Assuming that Atheros descriptors and large buffers are already allocated,
     * pull them out, hookup NDIS packets and place them on the receive queue.
     */
    descQInit(&pDev->rxQueue);
    for (i = 0; i < MAX_RX_PACKETS; i++) {
        pDesc = descQPopHead(&pDev->largeBufDescQueue);
        if (!pDesc) {
            rxPrintf("< %s: Ran out of descriptors!\n", __FUNCTION__);
            return NDIS_STATUS_RESOURCES;
        }

        ASSERT(pDesc->pOSHandle == NULL);

        pDesc->pBufferVirtPtr.ptr = NULL;
        pDesc->bufferPhysPtr      = (A_UINT32) NULL;
        pDesc->pNextVirtPtr       = NULL;
        pDesc->freeQueue          = &pDev->rxQueue;
        pDesc->ffRefHack          = 1;
        pDesc->wdcRxInfo          = NULL;
        /* Push on host rx queue */
        descQPushTail(&pDev->rxQueue, pDesc, pDesc);
    }

    A_ATOMIC_SET(&pDev->totalRxIndicated, 0);
    pDev->targetRxPending      = 0;
    pDev->targetStopRec = FALSE;

    rxPrintf("< %s\n", __FUNCTION__);

    return NDIS_STATUS_SUCCESS;
}


/*****************************************************************************
 * StartReceiveUnit
 *
 * Sets the filter as indicated and mark software state to allow RX.
 *
 * Arguments:
 *      pDevInfo    ptr to DevInfo object instance
 *      newFilter   new value for rx filter.
 */
VOID
StartReceiveUnit(IN WLAN_DEV_INFO *pDevInfo, IN A_UINT32 newFilter)
{
#if TBD
    ASSERT(pDevInfo->targetRxPending == 0);
#endif

    pDevInfo->targetStopRec = FALSE;
    pDevInfo->dropAllRxFrames = FALSE;

    /*
     * Restore the filter and then set any additional
     * bits requested
     */
    wlanRxFilter(pDevInfo, 0, RX_FILTER_RESTORE);
    if (newFilter) {
        wlanRxFilter(pDevInfo, newFilter, RX_FILTER_SET);
    }

    /* xxxxPGSxxxx - TODO - where does this go? host/target?  ANI related */
    PHY_ERR_RESET(pDevInfo);
}

/*****************************************************************************
 * StopReceive - Routine to stop receive hardware
 */
VOID
StopReceive(WLAN_DEV_INFO *pDevInfo)
{
    rxPrintf("> %s\n", __FUNCTION__);

    /* The following should always be set before calling this fn */
    ASSERT(pDevInfo->dropAllRxFrames);

    /* Set flag to prevent host passing rx buffers to target */
    pDevInfo->targetStopRec = TRUE;

#if TBD
    ASSERT(pDevInfo->targetRxPending == 0);
#endif

    rxPrintf("< %s\n", __FUNCTION__);
}

/*****************************************************************************
 * StopSwReceive - Stops processing of received frames in the driver
 */
VOID
StopSwReceive(WLAN_DEV_INFO *pDev)
{
    rxPrintf("> %s\n", __FUNCTION__);

    pDev->dropAllRxFrames = TRUE;

    rxPrintf("< %s\n", __FUNCTION__);
}

/*****************************************************************************
 * isMcForUs - Filter multicast frames that are not wanted
 *
 * Description:
 * If we're not receiving all m/c pkts, filter such a pkt
 * through the NDIS-supplied list.
 * Note: Promiscuous mode is not currently supported
 */
A_BOOL
isMcForUs(WLAN_DEV_INFO *pDev, WLAN_MACADDR *pAddr)
{
    UINT i;

    if (pDev->NdisFilterReg & NDIS_PACKET_TYPE_ALL_MULTICAST) {
        return TRUE;
    }

    for (i = 0; i < pDev->mcTableCount; i++) {
        if (A_MACADDR_COMP(pAddr, &pDev->mcTable[i]) == 0) {
            return TRUE;
        }
    }

    return FALSE;
}


